/*
 * Copyright (c) 1990 Paul Pomes
 * Copyright (c) 1990 University of Illinois Board of Trustees
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of Illinois, Urbana.  In addition, redistribution
 * and use must conform to the terms listed in the Copying file in
 * this directory.
 *
 * The name of the University may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char rcsid[] = "@(#)$Header: /usr/local/src/cso/oed2/RCS/FindTag.c,v 2.0.0.1 90/03/30 15:27:14 paul Exp $";
#endif /* lint */

#include	"oed2.h"

/*
 * FindTag -- Locate a tag in the master TagList
 *
 *	Given a character string of a tag found in the text, do a binary
 *	search of the lexicographically sorted master TagList for the
 *	matching tag.  Associated with each tag in TagList is the numeric
 *	value of the tag (see oed2.h), and what actions should occur when
 *	the tag is encountered in the text.  Actions are usually font changes
 *	or setting of a SkipTag variable.  The return value, in the case
 *	where the tag is found, is to a copy of the matching tag.  This
 *	prevents munging TagList by the calling routine.
 *
 *	parameters:
 *		Str -- char pointer to tag desired
 *	returns:
 *		Pointer to a copy of the selected TAG struct in TagList or
 *		NULL if no match is found.
 *	side effects:
 *		Prints error messages for unknown tags.
 *	deficiencies:
 *
 */

/*
 * This list must be sorted in lexicographic order using the second
 * (character string) field.  Use sort +1.
 */
static struct TAG	TagList[] = {
/*
  Format Mode	    S_ALL	   S_ETYM	    S_DEF	   S_QUOTE
Value	String	Start	End	Start	End	Start	End	Start	End
 */

TAG_NS,	"#",	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,
TAG_A,	"A",	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,
TAG_BL,	"BL",	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,
TAG_CF,	"CF",	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,
TAG_CO,	"CO",	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_CU,	"CU",	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_D,	"D",	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,
TAG_DAT, "DAT",	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,
TAG_DEF, "DEF",	NO_OP,	NO_OP,	SKIP,	SKIP,	NO_OP,	NO_OP,	SKIP,	SKIP,
TAG_E,	"E",	NO_OP,	NEWL,	NO_OP,	NEWL,	NO_OP,	NEWL,	NO_OP,	NEWL,
TAG_EQ,	"EQ",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_ET,	"ET",	L_SQ,	R_SQ,	L_SQ,	R_SQ,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_ETN, "ETN",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_FS,	"FS",	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_HG,	"HG",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_HL,	"HL",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_HO,	"HO",	UP,	DOWN,	UP,	DOWN,	UP,	DOWN,	UP,	DOWN,
TAG_IL,	"IL",	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,
TAG_IPH, "IPH",	PAREN,	PAREN,	PAREN,	PAREN,	PAREN,	PAREN,	PAREN,	PAREN,
TAG_IPR, "IPR",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_L,	"L",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_LB,	"LB",	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,
TAG_LF,	"LF",	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_LQ,	"LQ",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_MF,	"MF",	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_MPR, "MPR",	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_NB,	"NB",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_OLD, "OLD#", SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_PQP, "PQP",	PARA,	NO_OP,	PARA,	NO_OP,	PARA,	NO_OP,	PARA,	NO_OP,
TAG_PS,	"PS",	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,
TAG_PSA, "PSA",	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,
TAG_Q,	"Q",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_QP,	"QP",	PARA,	NO_OP,	SKIP,	SKIP,	SKIP,	SKIP,	PARA,	NO_OP,
TAG_RX,	"RX",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SN,	"S0",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SN,	"S1",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SN,	"S2",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SN,	"S3",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SN,	"S4",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SN,	"S5",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SN,	"S6",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SN,	"S7",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SN,	"S8",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SE,	"SE",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SF,	"SF",	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,
TAG_SN,	"SN",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SQ,	"SQ",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SSE, "SSE",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_SSSE, "SSSE", NO_OP, NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_ST,	"ST",	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_T,	"T",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_VD,	"VD",	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,
TAG_VF,	"VF",	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,
TAG_VL,	"VL",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_W,	"W",	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,
TAG_XDAT, "XDAT", FONTB, FONTP,	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,
TAG_XIL, "XIL",	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,
TAG_XL,	"XL",	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,
TAG_XR,	"XR",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_b,	"b",	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,	FONTB,	FONTP,
TAG_bch, "bch", NO_OP, NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_bka, "bka",	FONTIP,	FONTPS,	FONTIP,	FONTPS,	FONTIP,	FONTPS,	FONTIP,	FONTPS,
TAG_bkn, "bkn",	FONTIP,	FONTPS,	FONTIP,	FONTPS,	FONTIP,	FONTPS,	FONTIP,	FONTPS,
TAG_chem, "chem", NO_OP, NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_col, "col",	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_cols, "cols", SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_dn,	"dn",	SLASH,	NO_OP,	SLASH,	NO_OP,	SLASH,	NO_OP,	SLASH,	NO_OP,
TAG_form, "form", NO_OP, NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_fr,	"fr",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_frac, "frac", NO_OP, NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_gk,	"gk",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_i,	"i",	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,	FONTI,	FONTP,
TAG_in,	"in",	DOWN,	UP,	DOWN,	UP,	DOWN,	UP,	DOWN,	UP,
TAG_just, "just", SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_ln,	"ln",	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,
TAG_note, "note", PARA, NO_OP,	PARA,	NO_OP,	PARA,	NO_OP,	PARA,	NO_OP,
TAG_nu,	"nu",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_ov,	"ov",	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,	NO_OP,
TAG_p,	"p",	PARA,	NO_OP,	PARA,	NO_OP,	PARA,	NO_OP,	PARA,	NO_OP,
TAG_r,	"r",	FONTR,	FONTP,	FONTR,	FONTP,	FONTR,	FONTP,	FONTR,	FONTP,
TAG_sc,	"sc",	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,	S_C,
TAG_su,	"su",	UP,	DOWN,	UP,	DOWN,	UP,	DOWN,	UP,	DOWN,
TAG_table, "table", SKIP, SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP,	SKIP
};

#define		TAG_TOP	(sizeof (TagList) / sizeof (struct TAG))

struct TAG *
FindTag (Str)
char	*Str;
{
	int			top, bot, mid;	/* binary search variables */
	int			result;		/* compare result in search */
	static struct TAG	TagCopy;	/* copy of matching tag */

	bot = 0;
	top = TAG_TOP - 1;

	if (tTd(3,5))
		printf ("FindTag: Searching for tag %s\n", Str);

	/* Tags falling below or above the list are unknown */
	if (strcmp (Str, TagList[bot].String) < 0
	 || strcmp (Str, TagList[top].String) > 0) {
		if (tTd(3,2))
			printf ("FindTag: Tag %s out of list range\n", Str);
		return (TAGNULL);
	}

	/* Special case of a match on the top of list */
	if (strcmp (Str, TagList[top].String) == 0) {
		bcopy ((char *)&TagList[top], (char *)&TagCopy,
			sizeof (struct TAG));
		return (&TagCopy);
	}

	/* Divide and conquer in log(n) tempo */
	for (;;) {
		mid = (top + bot) / 2;
		if ((result = strcmp (Str, TagList[mid].String)) == 0) {
			bcopy ((char *)&TagList[mid], (char *)&TagCopy,
				sizeof (struct TAG));
			return (&TagCopy);
		}
		if (mid == top || mid == bot)
			return (TAGNULL);
		else if (result > 0) {
			bot = mid;
		}
		else {
			top = mid;
		}
	}
	/* NOTREACHED */
} /* FindTag */
