/*
** EXECUTE.C - This is the ODBC sample driver code for
** executing SQL Commands.
**
**	This code is furnished on an as-is basis as part of the ODBC SDK and is
**	intended for example purposes only.
**
*/

//	-	-	-	-	-	-	-	-	-
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "gorta.h"

#include "states.h"

#include <msql.h>

//	-	-	-	-	-	-	-	-	-
static RETCODE ExecuteStmt(HSTMT   hstmt);
static void Localise(char **);

//	Execute a prepared SQL statement

RETCODE SQL_API SQLExecute(
	HSTMT	hstmt)		// statement to execute.
{
	LPSTMT lpstmt;
	RETCODE retcode;
	int iCount = 0;
	struct PBindInfo *pBind;
	int iBinding;

	lpstmt = hstmt;
	if (!lpstmt)
		goto Exit;
	assert(lpstmt->lMagic == STATEMENT_MAGIC);

	/* TBD
	 * Free old error information.
	 */
	assert(lpstmt->pcPrepared);

    /* TBD
     * Check that all parameters have been bound. Also check to see if run
     * time parameters are required.
     */
	
	/*
	 * Reset lpstmt->iCurrent. This will control the number of parameters
	 * to be passed through SQLPutData(). 1 will be the first parameter, so
	 * 0 is considered to be uninitialised.
	 */
	lpstmt->iPBCurrent = 0;
	
	/*
	 * If any parameters are required at execution time, cannot perform the
	 * statement. It will be done throught SQLPutData() and SQLParamData().
	 */
	for (iBinding=0; iBinding<lpstmt->iPBindings; iBinding++)
	{
		pBind = &(lpstmt->pPBindings[iBinding]);
		if ((*pBind->plActualLen) == (long)SQL_DATA_AT_EXEC)
		{
			retcode = SQL_NEED_DATA;
			goto Exit;
		}
	}
	
	retcode = ExecuteStmt(hstmt);
	
Exit:
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	Performs the equivalent of SQLPrepare, followed by SQLExecute.
//
//	Execute a command directly.
//	Call msqlQuery, msqlStoreResult
RETCODE SQL_API SQLExecDirect(
	HSTMT     hstmt,
	UCHAR FAR *szSqlStr,
	SDWORD    cbSqlStr)
{
	LPSTMT lpstmt;
	RETCODE retcode;
	int iBinding;
	struct PBindInfo *pBind;
	int iParams; /* Count the number of parameters to be bound */

	lpstmt = hstmt;
	if (!lpstmt)
		goto Exit;
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
	
	/* TBD
	 * Free old error information.
	 */
	if (lpstmt->pcPrepared)
		free(lpstmt->pcPrepared);
	
	if (cbSqlStr == SQL_NTS)
	{
		lpstmt->pcPrepared = strdup(szSqlStr);
	}
	else
	{
		assert(cbSqlStr > 0);
		lpstmt->pcPrepared = malloc((int)cbSqlStr + 1);
		memcpy(lpstmt->pcPrepared, szSqlStr, (int)cbSqlStr);
		lpstmt->pcPrepared[cbSqlStr] = '\0';
	}

	/*
	 * Count the number of parameters in the statement
	 */
	iParams = 0;
	{
		char *pMark;
		pMark = strchr(lpstmt->pcPrepared, '?');
		while (pMark)
		{
			iParams++;
			pMark = strchr(pMark + 1, '?');
		}
	}
	lpstmt->iPBindings = iParams;

	/*
	 * If any parameters are required at execution time, cannot perform the
	 * statement. It will be done throught SQLPutData() and SQLParamData().
	 */
	for (iBinding=0; iBinding<lpstmt->iPBindings; iBinding++)
	{
		pBind = &(lpstmt->pPBindings[iBinding]);
		if ((*pBind->plActualLen) == (long)SQL_DATA_AT_EXEC)
		{
			retcode = SQL_NEED_DATA;
			goto Exit;
		}
	}
	
	retcode = ExecuteStmt(hstmt);
	free(lpstmt->pcPrepared);
	lpstmt->pcPrepared = NULL;
	
Exit:
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	Returns the SQL string as modified by the driver.

RETCODE SQL_API SQLNativeSql(
	LPDBC      lpdbc,
	UCHAR FAR *szSqlStrIn,
	SDWORD     cbSqlStrIn,
	UCHAR FAR *szSqlStr,
	SDWORD     cbSqlStrMax,
	SDWORD FAR *pcbSqlStr)
{
	return SQL_SUCCESS;
}

//	-	-	-	-	-	-	-	-	-

//	Supplies parameter data at execution time.	Used in conjuction with
//	SQLPutData.

//	Retrieves the Current parameter being processed in the prepared statement.
RETCODE SQL_API SQLParamData(
	HSTMT	hstmt,
	PTR FAR *prgbValue)
{
	RETCODE retcode = SQL_SUCCESS;
	LPSTMT lpstmt;
	int iParam;
	struct PBindInfo *pBind;

	lpstmt = hstmt;
	if (!lpstmt)
		goto Exit;
	assert(lpstmt->lMagic == STATEMENT_MAGIC);

	/*
	 * The current parameter is defined by lpstmt->iPBCurrent.
	 * This function winds that parameter on to the next one.
	 * If there are more parameters required, then the SQL_NEED_DATA
	 * is returned. Otherwise the statement is formatted and executed.
	 *
	 * The parameter numbers are offset by one. iCurrent == 1 for array
	 * element 0.
	 */
	for (iParam = lpstmt->iPBCurrent + 1; iParam <= lpstmt->iPBindings; iParam++)
	{
		if (*(lpstmt->pPBindings[iParam - 1].plActualLen) == (long)SQL_DATA_AT_EXEC)
		{
			lpstmt->iPBCurrent = iParam;
			pBind = &(lpstmt->pPBindings[iParam - 1]);
			*prgbValue = pBind->pvdBuffer;
			retcode = SQL_NEED_DATA;
			goto Exit;
		}
	}
	
	/*
	 * If we reach this point, then all parameters have been bound. The statement
	 * can be executed.
	 */
	retcode = ExecuteStmt(hstmt);
	/*
	 * TBD - if executed due to SQLExecute() then free the buffer.
	 */

	retcode = SQL_SUCCESS;
Exit:	
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	Supplies parameter data at execution time.	Used in conjunction with
//	SQLParamData.

RETCODE SQL_API SQLPutData(
	HSTMT   hstmt,
	PTR     rgbValue,
	SDWORD  cbValue)
{
	RETCODE retcode = SQL_SUCCESS;
	LPSTMT lpstmt;
	struct PBindInfo *pBind;

	lpstmt = hstmt;
	if (!lpstmt)
		goto Exit;
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
	
	pBind = &(lpstmt->pPBindings[lpstmt->iPBCurrent - 1]);
	/* TBD - process more that one SQLPutData call per parameter */

	/* TBD Ascertain whether the data buffer needs to be malloced and 
	 * copied or whether the application buffer can be used.
	 */	
	pBind->iRTLen = cbValue;
	pBind->pvdRTBuffer = rgbValue;

Exit:
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	This is not implemented, but we pretend it is to support the CORE
//	specification.
RETCODE SQL_API SQLCancel(
	HSTMT	hstmt)	// Statement to cancel.
{
	/* TBD - This function should free up any resources bound for run
	 * time parameters - I think.
	 */
	return SQL_SUCCESS;
}

//	-	-	-	-	-	-	-	-	-
/*
 * This function is an internal function to execute a statement, once
 * all the statement parameters have been passed. It could be called from
 * SQLExecute(), SQLExecDirect() or SQLParamData().
 */
static RETCODE ExecuteStmt(HSTMT   hstmt)
{
	LPSTMT lpstmt;
	m_result *msqlMResult; 
	int iRv;
	RETCODE retcode;
	char szErrorMsg[160];
	LPEI lpeiTmp;
	char *pcSection, *pcMark;
	int iCount = 0;
	struct PBindInfo *pBind;
	int iOffset;
	char *szExecBuf;

	lpstmt = hstmt;
	
	if (!lpstmt)
		goto Exit;
	assert(lpstmt->lMagic == STATEMENT_MAGIC);

	szExecBuf = malloc(2048);	/* TBD - check max length of MSQL request */
	assert(szExecBuf);	
	
	/*
	 * If there is previously a result then clear it.
	 */
	if (lpstmt->pvdMResult)
	{
		msqlMResult = lpstmt->pvdMResult;
		msqlFreeResult(msqlMResult);
		lpstmt->pvdMResult = NULL;
	}

	if (!lpstmt->iPBindings)
	{
		/*
		 * No parameters. Statement can be executed.
		 */
		strcpy(szExecBuf, lpstmt->pcPrepared);
	}
	else
	{
		/*
	 	 * Replace in the buffer all the ? with the parameter bindings
	 	 */
		pcSection = lpstmt->pcPrepared;
		pcMark = strchr(pcSection, '?');
		if (pcMark)
			*pcMark = '\0';
		strcpy(szExecBuf, pcSection);
		if (pcMark)
			*pcMark = '?';		
		iOffset = strlen(szExecBuf);
		while (pcMark)
		{
			pcSection = pcMark + 1;
			iCount++;	/* iCount now set to the parameter number */

			pBind = &(lpstmt->pPBindings[iCount - 1]);
			switch (pBind->iSqlType)
			{
			case SQL_CHAR:
			case SQL_VARCHAR:
				if (pBind->plActualLen && *(pBind->plActualLen) == SQL_NTS)
					iOffset += sprintf(&(szExecBuf[iOffset]), "'%s'", pBind->pvdBuffer);
				else if  (pBind->plActualLen && *(pBind->plActualLen) > 0)
					iOffset += sprintf(&(szExecBuf[iOffset]), "'%.*s'", 
						(int)(*pBind->plActualLen), pBind->pvdBuffer);
				else if (pBind->plActualLen && *(pBind->plActualLen) == SQL_NULL_DATA)
					iOffset += sprintf(&(szExecBuf[iOffset]), "NULL", pBind->pvdBuffer);
				else if (pBind->plActualLen && *(pBind->plActualLen) == SQL_DATA_AT_EXEC)
					iOffset += sprintf(&(szExecBuf[iOffset]), "'%.*s'",	(int)pBind->iRTLen,
						pBind->pvdRTBuffer);
				else
					assert(0);
				break;
			case SQL_INTEGER:
				if  (pBind->plActualLen && *(pBind->plActualLen) > 0)
					iOffset += sprintf(&(szExecBuf[iOffset]), "%ld", *(long *)pBind->pvdBuffer);
				else if (pBind->plActualLen && *(pBind->plActualLen) == SQL_NULL_DATA)
					iOffset += sprintf(&(szExecBuf[iOffset]), "NULL");
				else if (pBind->plActualLen && *(pBind->plActualLen) == SQL_DATA_AT_EXEC)
					iOffset += sprintf(&(szExecBuf[iOffset]), "%ld", *(long *)pBind->pvdRTBuffer); 
				else
					iOffset += sprintf(&(szExecBuf[iOffset]), "%ld", *(long *)pBind->pvdBuffer);
				break;
			case 0:
				/* Parameter not bound ? */
				assert(0);
			default:
				assert(0);
			}
			pcMark = strchr(pcSection, '?');
			if (pcMark)
				*pcMark = '\0';
			strcpy(&(szExecBuf[iOffset]), pcSection);
			iOffset += strlen(pcSection);
			if (pcMark)
				*pcMark = '?';		
		}
	}
	
	Localise(&(szExecBuf));
	iRv = msqlQuery(lpstmt->lpdbc->iSocketFd, szExecBuf);
	if (iRv != 0)
	{
		retcode = SQL_ERROR;
		lpeiTmp = malloc(sizeof(EI));
		if (!lpeiTmp)
		{
			retcode = SQL_ERROR;
			goto Exit;
		}
		lpeiTmp->pNext = lpstmt->lpei;
		lpeiTmp->lErrorCode=-1;
		strncpy(lpeiTmp->szSqlState, MSQL_GENERAL_ERROR, 6);
/*
 * To try to find problem with access support, temporarily replace
 * this return value with "S0002".
 */
		strncpy(lpeiTmp->szSqlState, "S0002", 6); 
		msqlErrorText(szErrorMsg);						/* MSQL Extension */
		sprintf(lpeiTmp->szErrorMessage, "[MSQLODBC][MSQL]%.*s", 160, szErrorMsg);
		lpstmt->lpei=lpeiTmp;
		goto Exit;
	}
	
	msqlMResult = msqlStoreResult();
	
	/*
	 * In the MSQL view of the world the current row is the first row
	 * after a query is performed. However in the ODBC view the current
	 * row does not exist until the row is fetchd. Then it becomes the
	 * current row. In order to implement this I set the row cursor to
	 * NULL. In the first fetch() the row cursor is set to the first
	 * row result.
	 */
	if (msqlMResult)
		msqlMResult->cursor = NULL;
    
    lpstmt->pvdMResult = msqlMResult;
    retcode = SQL_SUCCESS;
	
Exit:
	free(szExecBuf);
	return(retcode);
}

/*
 * This function converts the syntax passed from the source, to a syntax
 * acceptable to Mini SQL. The parameter is a pointer to a pointer to a 
 * malloced buffer.
 */	
static void
Localise(char **ppcBuffer)
{	
	char *szExecBuf = *ppcBuffer;
	
	/* A little preprocessing on the SQL. Microsoft query puts
	 * carriage returns characters into the request. MSQL doesn't like these.
	 * Simply remove them.
	 */
	{
		char *pc;
		while (pc = strchr(szExecBuf, '\r'))
			memmove(pc, pc+1, strlen(pc));
	}


#if 0
	For version 2.10.0.5 I changed the grammar in the msqld on UNIX. This 
	is easier than writing a grammar here.
		
	/*
	 * Need to convert the SQL Syntax - 'IS NULL' to the acceptable syntax
	 * '= NULL'. I hope this doesn't keep getting to be more and more work.
	 * May need to slot in a YACC Grammar here if we are not careful. 
	 */
	{
		/* This is a cheap and nasty solution. I assume that the words IS
		 * NULL or IS NOT NULL cannot occur within a quoted argument.
		 * If you want more put it in the back end.
		 */
		char *pcIS; /* Pointer to the occurance of 'IS' in the sentance */
		
		while ((pcIS = strstr(szExecBuf, "IS")) || (pcIS = strstr(szExecBuf, "is")) ||
			(pcIS = strstr(szExecBuf, "Is")) || (pcIS = strstr(szExecBuf, "iS")))
		{
			/* Check if the next word is 'NOT'. */
			char *pcNext;
			int bIsNot = 0;

			/* Ensure that the IS is a free standing word */
			if ((*(pcIS -1) != ' ' && *(pcIS -1) != '\t' && *(pcIS -1) != '\n') ||
				(*(pcIS +2) != ' ' && *(pcIS +2) != '\t' && *(pcIS +2) != '\n'))
			{
				szExecBuf = pcIS + 1;	/* Search for next occurrance */
				continue;
			}
			
			pcNext = strtok(pcIS, " \t\n"); /* IS */
			pcNext = strtok(NULL, " \t\n"); /* Next word */
			if (!pcNext)
				break;

			if (toupper(pcNext[0]) == 'N' && toupper(pcNext[1]) == 'O' &&
				toupper(pcNext[2]) == 'T' && pcNext[3] == '\0')
			{
				/* Found is not */
				bIsNot = 1;
				pcNext = strtok(NULL, " \t\n");
			}
			/* Expecting NULL */
			if (toupper(pcNext[0]) == 'N' && toupper(pcNext[1]) == 'U' &&
				toupper(pcNext[2]) == 'L' && toupper(pcNext[3]) == 'L' &&
				pcNext[4] == '\0')
			{
				/* Found the word NULL */
				if (bIsNot)
				{
					strcpy(pcIS, "<> NULL ");
					memmove(pcIS + 8, pcNext+5, strlen(pcNext+5)+1);
				}
				else
				{
					strcpy(pcIS, "= NULL ");
					memmove(pcIS + 7, pcNext+5, strlen(pcNext+5)+1);
				}
				szExecBuf = pcIS;				
			}
			szExecBuf = pcIS + 1;	/* Search for next occurrance */
		}
	}
#endif
}

