/*  File: dump.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
      Dumps the whole database, creating a dialog graph  
      or a single object in an already open file.  

      Dumps all visible B-classes (not X), and those A classes
      having a dumpFunction declared in quovadis.wrm
 * Exported functions:
     dumpAll(void), dumpKey(KEY,FILE *)
 * HISTORY:
 * Last edited: Mar 18 16:20 1992 (mieg)
 * * Oct 23 19:08 1991 (mieg): Suppresed dump of X classes (i.e. Keyword)
 * Created: Wed Oct 23 19:08:38 1991 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "lex.h"

#include "a.h"
#include "bs.h"
#include "dump.h"
#include "display.h"
#include "graph.h"
#include "pick.h"
#include "session.h"
#include "mytime.h"

/*************************************************************************/
/******************************************************************/

static FILE* f = 0 ;
DumpFunc dumpFunc[256] ;
DumpFunc asnDumpFunc[256] ;

/*********************************************************/

static BOOL  dumpPossible(int class)
{ if( pickList[class & 255].type == 'A')
    return dumpFunc[class] ? TRUE : FALSE ;

  return
    !pickXref(class) && pickVisible(class) ;
}

/*********************************************************/

static BOOL  asnDumpPossible(int class)
{ if( pickList[class & 255].type == 'A')
    return 
      asnDumpFunc[class] ? TRUE : FALSE ;

  return
    !pickXref(class) && lexMax(class) ;
}

/*********************************************************/

static BOOL dumpArray(Array a, int class)
{ 
  return
    class>0 &&
      class<256 &&    
	dumpFunc[class] &&
	  dumpFunc[class] (f, 0 ,a) ;
}

/*********************************************************/

static BOOL asnDumpArray(Array a, int class)
{ 
  return
    class>0 &&
      class<256 &&    
	asnDumpFunc[class] &&
	  asnDumpFunc[class] (f, 0, a) ;
}

/*********************************************************/

static BOOL  dumpKey2(KEY key, int class)
{
  BOOL result = TRUE ; /* optimist default */
  OBJ obj ;
  Array a ;
  char type = pickType(key) ;

  if(f)
    fprintf (f,"%s : \"%s\"\n",	   
	     className(key), name(key)) ;

  if(iskey(key) == 2)   /* object is not empty */
    switch (type)
      {
      case 'A':    
	if(a = uArrayGet(key, -1,0))  /* No check on pArray size */
	  { result = dumpArray(a, class) ;
	    arrayDestroy(a) ;
	  }
	break ;
      case 'B':
	if( obj = bsCreate (key))
	  { if(f && KEYKEY(key))
	      bsAceDump (obj,f) ;
	    bsDestroy (obj) ;
	  }
	break ;
      }
  if(f)
    fputc ('\n',f) ;	/* leave a blank line at end of object */
  return result ;
}

/*********************************************************/

static BOOL  asnDumpKey2(KEY key, int class, BOOL previousAsn)
{
  BOOL result = FALSE ;
  OBJ obj ;
  Array a ;
  char type = pickType(key) ;


  if(!f)
    return FALSE ;

  if(iskey(key) == 2)   /* object is not empty */
    switch (type)
      {
      case 'A':    
	if(a = uArrayGet(key, -1, 0))  /* No check on pArray size */
	  { result = asnDumpArray(a, class) ;
	    arrayDestroy(a) ;
	  }
	break ;
      case 'B':
	if( obj = bsCreate (key))
	  { if(f && KEYKEY(key))
	      { result = TRUE ;
		if(previousAsn)
		  fprintf(f, " ,\n") ;
		bsAsnDump (obj,f) ;
	      }
	    bsDestroy (obj) ;
	  }
	break ;
      }
 
  return result ;
}

/*********************************************/
/************* public routine ****************/
/*********************************************/

BOOL asnDumpKey(KEY key, FILE* ff, BOOL previousAsn) 
{
  if(iskey(key) )
    if(asnDumpPossible(class(key)))
      { f = ff ;
	return asnDumpKey2(key, class(key), previousAsn) ;
      }
    else
      messout ("No asnDumper available for object %s", name(key)) ;
  
  return FALSE ;
}

/******************************************/

BOOL dumpKey(KEY key, FILE* ff) 
{
  if(iskey(key) )
    if(dumpPossible(class(key)))
      { f = ff ;
	return dumpKey2(key, class(key)) ;
      }
    else
      messout ("No dumper available for object %s", name(key)) ;
  
  return FALSE ;
}

/*********************************************/
/*********************************************/
