/*  File: gmapdisp.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: Display of the genetic map
 * Exported functions:
       gMapDisplay
 * HISTORY:
 * Last edited: Apr 22 20:07 1992 (mieg)
 * * Mar  4 03:25 1992 (rd): added simplest form of chromosome banding
 * * Jan  6 13:11 1992 (rd): corrected green box scroll bug
 * * Dec 19 (neil): change "Hilite" to "Highlight" (button);
                    change "Show All Objects" to "Unhighlight, Revert" (menu);
                    make action of Hide and Highlight treat bit flags the same; use keySetFind in both;
                    re-highlight when unselecting a highlit object;
                    on "Unhighlight, Revert", take away current-object highlighting as well;
 * * Dec 13 16:48 1991 (mieg): Added topMargin
 * * Dec 11 17:24 1991 (mieg): Added drag gene and private gMaps
 * * Nov 29 14:49 1991 (mieg): Added Locus display
 * Created: Fri Oct 18 20:25:36 1991 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "keyset.h"
#include "graph.h"
#include "key.h"
#include "lex.h"
#include "bs.h"
#include "a.h"
#include "systags.wrm"
#include "classes.wrm"
#include "sysclass.wrm"
#include "tags.wrm"
#include "session.h"
#include "disptype.wrm"
#include "display.h"
#include "query.h"
#include "wormtype.h"

typedef struct LOOKSTUFF
  { int   magic;        /* == MAGIC */
    KEY   key ;
    int   activeBox ;
    int	  rearNameBox ;
    unsigned int flag; /*<<--*/
    float centre ;
    float mag ;		/* gMap units * mag = window coords */
    Array segs,      	/* array of SEG's from the obj */
          boxIndex ;    /* if >256 a SEG, else something special */
    struct
      { float fac,offset ;
      }   cursor ;
  } *LOOK ;

#define MAGIC  837648

				/* look->flag flags */
#define FLAG_REVERT_DISPLAY 0x00000001

static void gMapDestroy (void) ;
static void gMapRecalculate(void) ;
static void gMapPick (int box, double x , double y) ;
static void gMapMiddleDown (double x, double y) ;
static void gMapKbd (int k) ;
static void gMapDraw (LOOK look, KEY key) ;
static Array gMapConvert (KEY chrom) ;
static void wholeChromosome (void);
static void gMapResize (void) ;
static void gMapSelect (LOOK look, int box) ;
static void gMapFollow (LOOK look, double x , double y) ;
static void gMapShowAll(void) ;
static void gMapHide(void) ;
static void gMapHighlight(void) ;
static void setNeighbourKeys (KEY newAbove, KEY newBelow) ;
static void drawNeighbourKeys (BOOL isOn) ;
static int  order(void *a, void *b) ;     /* for arraySort() call */
static void gMapLoadMap (void) ;
static void gMapSaveMap (void) ;
static void gMapFindCentre(LOOK look) ;
static void gMapFindMag(LOOK look) ;
static void gMapGotoCmap (void) ;

static MENUOPT gMapMenu[] =
            { graphDestroy, "Quit",
		help,"Help",
		graphPrint,"Print",
		displayPreserve,"Preserve",
		gMapRecalculate,"Recalculate",
		gMapHide,"Show Selected Objects",
		gMapHighlight,"Highlight Selected Objects",
		gMapShowAll,"Unhighlight, Revert",
		gMapLoadMap, "Load Private Map",
		gMapSaveMap, "Save Private Map",
		gMapGotoCmap, "Physical Chromo Map",
		0, 0
            } ;

static int minLiveBox ;
static int nx, ny, topMargin = 2 ;
static float ny2 ;
#define GMAP2GRAPH(look,x) (look->mag * ((x) - look->centre) + ny2 + topMargin)
#define GRAPH2GMAP(look,x) (((x) - ny2 - topMargin) / look->mag + look->centre)
static int xRear, xScale, x2p, x3p, xGenes ;

typedef struct
  { KEY key ;
    float x, dx ;
    unsigned int flag ; /* I use shift right operators */
  } SEG ;
#define segFormat "k2fi"

static void gMapInsert2p(KEY key) ;
static void gMapInsert3p(KEY key) ;
static void contig2genes(SEG seg) ;
static void rearrangement2genes (SEG seg1) ;

SEG foreignSeg ;
				/* seg->flag flags  i use the leftest octet to store colour*/
#define FLAG_CLONED		0x00000001
#define FLAG_MARKER_GENE	0x00000002
#define FLAG_RELATED		0x00000004
#define FLAG_STRESSED		0x00000008
#define FLAG_ANTI_RELATED	0x00000010
#define FLAG_GMAPPED_CLONE	0x00000020
#define FLAG_FOREIGN_SEG	0x00000040
#define FLAG_CONTIG_START	0x00000080
#define FLAG_CONTIG_END		0x00000100
#define FLAG_HIDE		0x00000200
#define FLAG_HIGHLIGHT		0x00000400
#define FLAG_CENTROMERE		0x00000800
#define FLAG_DARK_BAND		0x00001000
#define FLAG_NOR		0x00002000

#define LOOKGET(name)     LOOK look ; \
                          if (!graphAssFind (gMapDisplay,&look)) \
		            messcrash ("graph not found in %s",name) ; \
			  if (!look) \
                            messcrash ("%s received a null pointer",name) ; \
                          if (look->magic != MAGIC) \
                            messcrash ("%s received a wrong pointer",name)

/**********************************/
     /* Recalculate all gMaps */
void gMapMakeAll(void)
{
  KEY chromosome = 0 ;	/* 0 primes lexNext() */
  Array segs ;

  if(!isWriteAccess())
    { messout("Sorry, you do not have Write Access");
      return ;
    }

  if (lexNext (_VChromosome,&chromosome)) /* skip the model */
    while (lexNext (_VChromosome,&chromosome))
      {
	if (segs = gMapConvert (chromosome))
	  arrayDestroy (segs) ;
      }
}

/**********************************/

BOOL gMapGetPos(KEY from, KEY *chromo, float *x1, float *x2)
{ OBJ obj ;
  BOOL result = FALSE ;

  if(!from || !(obj = bsCreate (from)))
    return FALSE ;
      
  if (bsGetKey (obj,_gMap,chromo)
      && bsGetData (obj,_bsRight,_Float, x1))
    { if (!isGene(from) && bsGetData (obj, _bsRight, _Float, x2))
	*x1 = 0.5 * (*x2 + *x1) ;
      result = TRUE ;
    }
  bsDestroy (obj) ;
  return result ;
}

/**********************************/

BOOL gMapDisplay (KEY key, KEY from, BOOL isOldGraph)
{
  LOOK oldlook, look=(LOOK)messalloc(sizeof(struct LOOKSTUFF)) ;
  KEY  gMapkey, curr = 0 ;
  float x2 ;

  look->magic = MAGIC;

  look->flag=0; /**<<--*/

  if (key && class(key) != _VgMap && class(key) != _VChromosome)
    { from = key ;
      key = 0 ;
    }

  look->centre = look->mag = 99999 ;   	/* centre on parent object if poss */
  if(from)
    if(class(from) == _VCalcul)
      { look->centre = (float)(KEYKEY(from))/1000.0 - 1000 ;
	from = 0 ;
      }
    else if (gMapGetPos(from, &key, &(look->centre),&x2 ))
      curr = from ;
    else
      { display (from, 0, TREE) ;
	goto abort ;
      }

  if (!key)
    goto abort ;

  switch ((int)class(key))
    {
    case _VgMap :
      gMapkey = key ;
      if (look->segs = arrayGet(gMapkey,SEG, segFormat))
	break;
      key = KEYMAKE(_VChromosome,KEYKEY(gMapkey)) ;	/* drop through */
    case _VChromosome : 
      if (lexReClass(key, &gMapkey,_VgMap) &&
	  (look->segs = arrayGet(gMapkey,SEG, segFormat)))
	break ;
      if (!(look->segs = gMapConvert (key)))
                   /* Displays translocations etc as tree */
	{ display (key,from,TREE) ;
	  goto abort ;
	}
      break ;
      if (!lexReClass(key, &gMapkey,_VgMap))
	messcrash("gMap undefined for key %s", name(key) ) ;
    default:
      messout ("I cannot gMap such objects") ;
      goto abort ;
    }
  look->key = gMapkey ;

  look->boxIndex = arrayCreate (64,SEG*) ;
  look->activeBox = 0 ;

    /* Try to keep same magnification */
  if (isOldGraph &&
      graphAssFind (gMapDisplay,&oldlook) &&
      oldlook &&
      oldlook->magic == MAGIC &&
      oldlook->key == look->key)
    look->mag = oldlook->mag ;
  
  if (isOldGraph)
    { 
      graphRetitle (name (key)) ;
      gMapDestroy () ;
      graphAssRemove (gMapDisplay) ;
    }
  else 
    { if (!displayCreate (GMAP))
	goto abort ;
      
      graphRetitle (name(key)) ;
      graphRegister (RESIZE,(GraphFunc)gMapResize) ;
      graphRegister (DESTROY, gMapDestroy) ;
      graphRegister (PICK, (GraphFunc)gMapPick) ;
      graphRegister (MIDDLE_DOWN, (GraphFunc)gMapMiddleDown) ;
      graphRegister (KEYBOARD, (GraphFunc)gMapKbd) ;
      graphMenu (gMapMenu) ;
    }

  graphAssociate (gMapDisplay,look) ;

  if (look->centre == 99999 )
     gMapFindCentre(look) ;
  if (look->mag == 99999 )
     gMapFindMag(look) ;
  
  gMapDraw (look, curr) ;

  return TRUE ;

abort :
  messfree (look) ;
  return FALSE ;
}

/************************************************************/
/***************** Registered routines *********************/

static void gMapDestroy (void)
{
  LOOKGET("gMapDestroy") ;

  arrayDestroy(look->segs) ;
  arrayDestroy(look->boxIndex) ;

  look->magic = 0 ;
  messfree (look) ;
}

/*************************************************************/

static void gMapRecalculate(void)
{
  KEY chrom, curr ;
  LOOKGET("gMapRecalculate") ;
  
  if (lexReClass(look->key,&chrom,_VChromosome))
    { if (look->activeBox)
	curr = arr(look->boxIndex,look->activeBox,SEG*)->key ;
      else 
	curr = 0 ;
      arrayDestroy (look->segs) ;
      if (look->segs = gMapConvert (chrom))
	gMapDraw (look, curr) ;
      else
	{ messout ("Sorry, I have to abandon this gMap") ;
	  graphDestroy () ;
	}
    }
}

/******** coordinated dragging of pos and scale boxes **********/

static int posBox,scaleBox ;
static LOOK lookDrag ;

static void scaleBoxDrag (float *x, float *y, BOOL isDone)
{
  static BOOL isDragging = FALSE ;
  static float newCentre, yCentre ;
  
  if (isDragging)
    graphXorBox (posBox,xScale-0.5,yCentre) ;
  else
    isDragging = TRUE ;
    
   *x = 0 ;
 
  newCentre = lookDrag->cursor.offset + 
    		(*y - 3 - topMargin) / lookDrag->cursor.fac ;
  yCentre = GMAP2GRAPH(lookDrag,newCentre) ;

  if (isDone)
    { lookDrag->centre = newCentre ;
      gMapDraw (lookDrag, 0) ;
      isDragging = FALSE ;
    }
  else
    graphXorBox (posBox,xScale-0.5,yCentre) ;
}

static void posBoxDrag (float *x, float *y, BOOL isDone)
{
  static BOOL isDragging = FALSE ;
  static float newCentre,yScaleBox ;

  if (isDragging)
    graphXorBox (scaleBox,0,yScaleBox) ;
  else
    isDragging = TRUE ;

  *x = xScale-0.5 ;

  newCentre = GRAPH2GMAP(lookDrag,*y) ;
  yScaleBox = 3 + topMargin + 
    (newCentre - lookDrag->cursor.offset) * lookDrag->cursor.fac ;
  if (isDone)
    { lookDrag->centre = newCentre ;
      gMapDraw (lookDrag, 0) ;
      isDragging = FALSE ;
    }
  else
    graphXorBox (scaleBox,0,yScaleBox) ;
}

static void makeDragBoxes (void)
{
  float x1,x2,y1,y2 ;

  graphBoxDraw (posBox, WHITE, WHITE) ;
  graphBoxDim (posBox, &x1, &y1, &x2, &y2) ;
  posBox = graphBoxStart() ;
  graphLine (xScale-0.5, y1, xScale-0.5, y2) ;
  graphLine (200, y1, 200, y2) ; /* to make box wide */
  graphBoxEnd () ;
  graphBoxDraw (posBox, BLACK, TRANSPARENT) ;

  graphBoxDraw (scaleBox, WHITE, WHITE) ;
  graphBoxDim (scaleBox, &x1, &y1, &x2, &y2) ;
  scaleBox = graphBoxStart() ;
  graphFillRectangle (5.25, y1, 5.75, y2) ;
  graphLine (0, y1, 0, y2) ;	/* to make box wide */
  graphBoxEnd () ;
  graphBoxDraw (scaleBox, BLACK, TRANSPARENT) ;
}

/***********************************/

static void gMapPick (int box,  double x , double y) 
{
  int i ;
  LOOKGET("gMapPick") ;

  if (!box)
    return ;
  if ((i = (int)arr(look->boxIndex,box,SEG*)) > 256)	/* a SEG */
    if (box == look->activeBox)         /* a second hit - follow it */
      gMapFollow (look,x,y) ;
    else
      gMapSelect (look,box) ;
  else switch (i)		/* a control box */
    {
    case 4 :
      lookDrag = look ;
      makeDragBoxes () ;
      graphBoxDrag (scaleBox,scaleBoxDrag) ;
      break ;
    case 5 :
      lookDrag = look ;
      makeDragBoxes () ;
      graphBoxDrag (posBox,posBoxDrag) ;
      break ;
    }
}

/********* use middle button for cursor **********/

static double oldy , oldDy , oldx;
static BOOL dragFast ;
#define DRAGFASTLIMIT 6

static void gMapMiddleDrag (double x, double y) 
{
  if(dragFast)
    { graphXorLine (0, oldy - oldDy, DRAGFASTLIMIT, oldy - oldDy) ;
      graphXorLine (0, oldy + oldDy, DRAGFASTLIMIT, oldy + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, oldy, nx, oldy) ;

  oldy = y ;

  if(dragFast)
    { oldDy *= exp ((x - oldx) / 25.) ;
      oldx = x ;
      graphXorLine (0, y - oldDy, DRAGFASTLIMIT, y - oldDy) ;
      graphXorLine (0, y + oldDy, DRAGFASTLIMIT, y + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, y, nx, y) ;
}

static void gMapMiddleUp (double x, double y) 
{  float x1,x2,y1,y2 ;
  LOOKGET("gMapMiddleUp") ;

  if(dragFast)
    { graphBoxDim (scaleBox, &x1, &y1, &x2, &y2) ;
      look->mag *= (y2 - y1) / (2. * oldDy) ;
      look->centre = look->cursor.offset + 
	(y - (y2 - y1) * .5 - 3.05 - topMargin ) / look->cursor.fac ;
    }
  else
   look->centre +=  (y - .5  - ny2 - topMargin) / look->mag ;
  gMapDraw (look, 0) ;
}

static void gMapMiddleDown (double x, double y) 
{  float x1,x2,y1,y2 ;
  LOOKGET("gMapMiddleDown") ;


  graphBoxDim (scaleBox, &x1, &y1, &x2, &y2) ;
  oldDy = (y2 - y1) / 2. ;

  lookDrag = look ;
  dragFast = x < DRAGFASTLIMIT ? TRUE : FALSE ;

  if(dragFast)
    { graphXorLine (0, y - oldDy, DRAGFASTLIMIT, y - oldDy) ;
      graphXorLine (0, y + oldDy, DRAGFASTLIMIT, y + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, y, nx, y) ;
   
  oldx = x ;
  oldy = y ;
  graphRegister (MIDDLE_DRAG,(GraphFunc) gMapMiddleDrag) ;	/* must redo */
  graphRegister (MIDDLE_UP,(GraphFunc) gMapMiddleUp) ;
}

/*****************/

static void gMapKbd (int k)
{
  int  box ;
  LOOKGET("gMapKbd") ;

  if (!look->activeBox)
    return ;

  box = look->activeBox ;
  switch (k)
    {
    case UP_KEY :
      if (box > minLiveBox)
	--box ;
      break ;
    case DOWN_KEY :
      if (box < arrayMax(look->boxIndex) - 1)
	++box ;
      break ;
    }
  if (look->activeBox != box)
    { gMapSelect (look,box) ;
      gMapFollow (look,0.,0.) ;
    }
  else
    gMapSelect (look,box) ;
}

/**********************************/

static void gMapResize (void)
{
  LOOKGET("gMapResize") ;

  gMapDraw (look, 0) ;
}

/***************************/

static void gMapGotoCmap (void)
{
  KEY chrom ;
  LOOKGET("gMapGotoCmap") ;

  if (lexReClass (look->key,&chrom,_VChromosome))
    display (chrom, 0, CMAP) ;
}

/*****************************************************************/
/****************** Drawing routines *****************************/

static BOOL  gMapGetCurrentPos(LOOK look, KEY key, float *y)
{ 
  int i ;
  SEG *seg ;

  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;
      if (seg->key == key)
	{ *y = seg->x ;
	  return TRUE ;
	}
    }
  return FALSE ;
}

/******************/

static int gMapGet2p(LOOK look, KEY key, float *y1, float *y2, float *distancep, float *errorp)
{ OBJ Datum = bsCreate(key) ;
  KEY gene1, gene2 ; int result = 1 ;
  float c, yg1, yg2 ;

  if(!Datum)
    return FALSE ;
  if(!bsGetKey(Datum,_Gene1,&gene1) ||
     !bsGetKey(Datum,_Gene2,&gene2) ||
     !gMapGetCurrentPos(look, gene1, &yg1) ||
     !gMapGetCurrentPos(look, gene2, &yg2) 
     )
    goto abort ;
  
  if(yg2 < yg1)
    { c = yg2 ; yg2 = yg1 ; yg1 = c ; }
  *y1 = yg1 ; *y2 = yg2 ;
  *distancep = *errorp = 0 ;
  if (bsGetData(Datum,_Distance,_Float,distancep))
   { result++ ;
     if (bsGetData(Datum,_Error,_Float,errorp))
       result++ ;
   }
  bsDestroy(Datum) ;
  return result ;

 abort:
  bsDestroy(Datum) ;
  return 0 ;
}

static void gMapInsert2p(KEY key) 
{ int n ;
  SEG *seg ; 
  float y1, y2, dist, delta , c , max , min ;
  LOOKGET("Insert2p") ;
  n = arrayMax(look->segs) ;
  if(!gMapGet2p(look, key, &y1, &y2,&dist,&delta))
    return ;
  seg = arrayp(look->segs,n,SEG) ;

  seg->key = key ;
  seg->flag = FLAG_RELATED ;
  seg->x = c = (y1 + y2 ) / 2 ; 
  min = max = y1 ;
  if(y1>y2) 
    min = y2 ;
  else max = y2 ;
 
  if(c-dist/2 -delta/4 < min)
    min = c - dist/2 - delta/4 ;
  if(c+dist/2 +delta/4 > max)
    max = c + dist/2 + delta/4 ;

  seg->dx = max - min ;
}

/******************/

static BOOL  gMapGet3p(LOOK look, KEY key, int *cs, float *y1, float *y2, float *y3)
{ OBJ Datum = bsCreate(key) ;
  KEY gene1, gene2, gene3 ;

  if(!Datum)
    return FALSE ;
  if(!bsGetKey(Datum,_Gene1,&gene1) ||
     !bsGetKey(Datum,_Gene2,&gene2) ||
     !bsGetKey(Datum,_Gene3,&gene3) ||
     !gMapGetCurrentPos(look, gene1, y1) ||
     !gMapGetCurrentPos(look, gene2, y2) ||
     !gMapGetCurrentPos(look, gene3, y3) 
     )
    goto abort ;
 
  *cs = 1 ;
  if(bsFindTag(Datum,_ABC))
    { if ( *y1 <= *y2 && *y1 <= *y3)
	*cs = ( *y2 <=  *y3 ) ? 1 : 2  ;
    else if ( *y2 <= *y1 && *y2 <= *y3)
      *cs = ( *y1 <=  *y3 ) ? 3 : 4  ;
    else if ( *y3 <= *y1 && *y3 <= *y2)
      *cs = ( *y1 <=  *y2 ) ? 5 : 6  ;
    }
  else if(bsFindTag(Datum,_AB_C))
    { if ( *y1 <= *y2 && *y1 <= *y3)
	*cs = ( *y2 <=  *y3 ) ? 1 : 2  ;
    else if ( *y2 <= *y1 && *y2 <= *y3)
      *cs = ( *y1 <=  *y3 ) ? 1 : 4  ;
    else if ( *y3 <= *y1 && *y3 <= *y2)
      *cs = ( *y1 <=  *y2 ) ? 1 : 6  ;
    }
  if(bsFindTag(Datum,_A_BC))
    { if ( *y1 <= *y2 && *y1 <= *y3)
	*cs = ( *y2 <=  *y3 ) ? 1 : 1 ;
    else if ( *y2 <= *y1 && *y2 <= *y3)
      *cs = ( *y1 <=  *y3 ) ? 3 : 1  ;
    else if ( *y3 <= *y1 && *y3 <= *y2)
      *cs = ( *y1 <=  *y2 ) ? 5 : 6  ;
    }

  bsDestroy(Datum) ;
  return TRUE ;

 abort:
  bsDestroy(Datum) ;
  return FALSE ;
}

static void gMapInsert3p(KEY key) 
{ int n , cs ;
  SEG *seg ; 
  float y1, y2, y3 , min, max ;
  LOOKGET("Insert3p") ;
  n = arrayMax(look->segs) ;
  if(!gMapGet3p(look, key, &cs, &y1, &y2,&y3))
    return ;
  seg = arrayp(look->segs,n,SEG) ;

  seg->key = key ;
  seg->flag = FLAG_RELATED ;
 
  min = max = y1 ;
  if(y2 < min)
    min = y2 ;
  if(y3 < min)
    min = y3 ;
  if(y2 > max)
    max = y2 ;
  if(y3 > max)
    max = y3 ;
  
  seg->x = (y1 + y2 ) / 2 ; 
  seg->dx = max - min ;
}

static void  gMapDraw2p(LOOK l, KEY key,float x) 
{
  float c, yg1, yg2, dist, delta ;
  int found = gMapGet2p(l, key, &yg1,&yg2,&dist,&delta) ;

  if (!found)
    return ;
 
  if(yg2 < yg1)
    { c = yg2 ; yg2 = yg1 ; yg1 = c ; }
  c = ( yg1 + yg2) / 2 ;
  delta /= 4 ;
  
  if (found < 3)  /* Distance or error unknown  */
    { graphColor(YELLOW) ;
      graphFillRectangle (x,GMAP2GRAPH(l, c + dist/2) , x,GMAP2GRAPH(l,c - dist/2)) ;
      graphColor(BLACK) ;
    }
     /* join the 2 genes */
  graphLine (x,GMAP2GRAPH(l, c + dist/2) , x,GMAP2GRAPH(l,c - dist/2)) ;
 
  if (found == 3  )  /* error is known */
    { /* a red rectangle from true to best positions */
      graphColor(RED) ;
      
      graphFillRectangle(x-.2,GMAP2GRAPH(l,yg1),x+.2,GMAP2GRAPH(l,c - dist/2)) ;
      graphFillRectangle(x-.2,GMAP2GRAPH(l,yg2),x+.2,GMAP2GRAPH(l,c + dist/2)) ;
      
      graphColor(BLACK) ;
      graphRectangle(x-.2,GMAP2GRAPH(l,yg1),x+.2,GMAP2GRAPH(l,c - dist/2)) ;
      graphRectangle(x-.2,GMAP2GRAPH(l,yg2),x+.2,GMAP2GRAPH(l,c + dist/2)) ;


      /* a green error box around best position, hence hiding the red box */
      graphColor(GREEN) ;
      graphFillRectangle(x-.25,GMAP2GRAPH(l,c - dist/2 - delta),
			 x+.25,GMAP2GRAPH(l,c - dist/2 + delta)) ;
      graphFillRectangle(x-.25,GMAP2GRAPH(l,c + dist/2 - delta),
			 x+.25,GMAP2GRAPH(l,c + dist/2 + delta)) ;
      
      graphColor(BLACK) ;
      graphRectangle(x-.25,GMAP2GRAPH(l,c - dist/2 - delta),
		     x+.25,GMAP2GRAPH(l,c - dist/2 + delta)) ;
      graphRectangle(x-.25,GMAP2GRAPH(l,c + dist/2 - delta),
		     x+.25,GMAP2GRAPH(l,c + dist/2 + delta)) ;
    }

  graphPointsize(.7) ;
  graphPoint(x,GMAP2GRAPH(l,yg1)) ;
  graphPoint(x,GMAP2GRAPH(l,yg2)) ;

}

/*******************/

static void upArrow(LOOK l, float x, float y) 
{
  graphLine(x, GMAP2GRAPH(l, y), x + .3, GMAP2GRAPH(l, y) - .8 ) ;
  graphLine(x, GMAP2GRAPH(l, y), x - .3, GMAP2GRAPH(l, y) - .8 ) ;
}

static void downArrow(LOOK l, float x, float y) 
{
  graphLine(x, GMAP2GRAPH(l, y), x + .3, GMAP2GRAPH(l, y) + .8 ) ;
  graphLine(x, GMAP2GRAPH(l, y), x - .3, GMAP2GRAPH(l, y) + .8 ) ;
}

static void redArrows(LOOK l, float x, float y1, float y2)
{ 
  graphColor(RED) ;
  graphFillRectangle(x-.3,GMAP2GRAPH(l,y1),x+.3,GMAP2GRAPH(l,y2)) ;
  graphColor(BLACK) ;
  
  downArrow(l, x, y1) ;
  upArrow(l, x, y2) ;
  graphLine (x,GMAP2GRAPH(l, y1) , x,GMAP2GRAPH(l, y2)) ;
}

/*******************/

static void  gMapDraw3p(LOOK l, KEY key,float x) 
{
  int cs ;
  float  yg1, yg2, yg3 ;

  if(!gMapGet3p(l, key, &cs , &yg1,&yg2,&yg3))
    return ;
 
     /* join the 3 pairs of  genes */
  graphLine (x,GMAP2GRAPH(l, yg1) , x,GMAP2GRAPH(l, yg2)) ;
  graphLine (x,GMAP2GRAPH(l, yg2) , x,GMAP2GRAPH(l, yg3)) ;
  graphLine (x,GMAP2GRAPH(l, yg3) , x,GMAP2GRAPH(l, yg1)) ;

    /* a red rectangle from true to best positions */
  graphPointsize(.7) ;
  
  switch (cs)
    {
    case 1: case 6: /* abc or cba */
      graphPoint(x,GMAP2GRAPH(l,yg1)) ;
      graphPoint(x,GMAP2GRAPH(l,yg2)) ;
      graphPoint(x,GMAP2GRAPH(l,yg3)) ;
      break ;
      
    case 2:    /* acb */
      graphPoint(x,GMAP2GRAPH(l,yg1)) ;
      redArrows(l, x, yg3, yg2) ;
      break ;
      
    case 3:    /* bac */
      graphPoint(x,GMAP2GRAPH(l,yg3)) ;
      redArrows(l, x, yg2, yg1) ;
      break ;
      
    case 4:    /* bca */
      graphPoint(x,GMAP2GRAPH(l,yg1)) ;
      redArrows(l, x, yg2, yg3) ;
      break ;
      
    case 5:    /* cab */
      graphPoint(x,GMAP2GRAPH(l,yg3)) ;
      redArrows(l, x, yg1, yg2) ;
      break ;
    }
}

/************* first a mini package for bumping text *************/

	/* This one turned on its side w.r.t. pMapdisp version.
	   Coords are text coords, but bumping is more complicated,
	   because we are given both x and y.  We want to preserve y if
	   possible, so go right in x, then down in y.
	*/

typedef struct bumpStruct
  { int n ;
    float *bottom ;
    float curr ;
  } *BUMP ;

static BUMP bumpCreate (int ncol)
{
  int i ;
  BUMP bump = (BUMP) messalloc (sizeof (struct bumpStruct)) ;

  bump->n = ncol ;
  bump->bottom = (float*) messalloc (ncol*sizeof (float)) ;
  for (i = 0 ; i < ncol ; ++i)
    bump->bottom[i] = -100000000.0 ;
  bump->curr = -100000.0 ;
  return bump ;
}

static void bumpDestroy (BUMP bump)
{
  messfree (bump->bottom) ;
  messfree (bump) ;
}

static void bumpGene (BUMP bump, char *text, double x, double y)
{
  int i,j ;
  int len = strlen(text) + 1 ;	/* +1 gives space after gene */
  
  if (x < 0) 
    x = 0 ;
  if (x+len+20 > bump->n)
    x = bump->n - len - 20 ;
  if (y < bump->curr)
    y = bump->curr ;

  while (TRUE)
    { for (i = x, j = len ; i < bump->n ; ++i)	/* always start at x */
	{ if (bump->bottom[i] > y)
	    j = len ;
	  else if (!--j)		/* have found a gap */
	    break ;
	}
      if (!j)	
	{ graphText (text, xGenes + i-len, y - .5) ;
	  for (j = 0 ; j < len ; j++)	/* advance bump */
	    bump->bottom[i-j] = y+1 ;
	  bump->curr = y ;
	  break ;			/* done */
	}
      y += 1 ;	/* try next row down */
      x = 5 ;
    }
}

/*** bump package for rearrangement lines is just like for pMap ***/

#define xRightEnd (xScale - 4)

static Array bumpR = 0, bump2 = 0, bump3 = 0 ;

static void yBumpReset (Array *bumpp)
{
  int i ;
  *bumpp = arrayReCreate(*bumpp,50, float) ;
  for (i = 0 ; i <arrayMax(*bumpp) ; ++i)
    arr(*bumpp, i, float) = -100000000.0 ;
}

static void bumpRear (BOOL doDraw, LOOK look, Array bump, KEY key, float y1, float y2)
{
  int i, min ;
  float x , y ;

  min = 0 ;
  y = 1000000 ;
  for (i = 0 ; i <arrayMax(bump) ; ++i)
    if (arr(bump, i, float) < y)
      { min = i ;
	y = arr(bump, i, float) ;
      }
 
  if (y >= y1)	/* overlap, extend bumpArray */
    array(bump, min = i, float) = y2 ;
 
  if(bump == bumpR)
    x = xRear ;
  else if(bump == bump2)
    x = x2p ;
  else if(bump == bump3)
    x = x3p ;
  x +=  min ;
  if (y1 < topMargin) 
    y1 = topMargin ;
  if (y2 >  ny  - 0.5) 
    y2 = ny  - 0.5 ;

  if (doDraw)
    switch(class(key))
    {
    case _VRearrangement:
      if (y1 > topMargin) 
	graphLine (x-0.25, y1, x+0.25, y1) ; 
      if (y2 < ny  - 0.5) 
	graphLine (x-0.25, y2, x+0.25, y2) ; 

      if (isDeficiency(key))
	graphLine (x, y1, x, y2) ;
      else if (isDuplication(key))
	{ graphLine (x-0.2, y1, x-0.2, y2) ;
	  graphLine (x+0.2, y1, x+0.2, y2) ;
	}
      break ;
    case _V2_point_data:
      gMapDraw2p(look, key, x) ;
      break ;
    case _V3_point_data:
      gMapDraw3p(look, key, x) ;
      break ;
    }
  

  array(bump, min, float) = y2 ;
}

/**************************************************/

static void drawScale (LOOK look)
{
  float cutoff = 5 / look->mag ;
  float unit = 0.01 ;
  float subunit = 0.001 ;
  float x,y ;

  while (unit < cutoff)
    { unit *= 2 ;
      subunit *= 5 ;
      if (unit >= cutoff)
	break ;
      unit *= 2.5 ;
      if (unit >= cutoff)
	break ;
      unit *= 2 ;
      subunit *= 2 ;
    }

  x = look->centre - (ny2-2)/look->mag ;
  x = unit * (((x>=0)?1:0) + (int)(x/unit)) ;
  while ((y = GMAP2GRAPH(look, x)) < ny  - 1)
    { 
      graphLine (xScale-1.5,y,xScale-0.5,y) ;
      if(unit >= 1)
	graphText (messprintf ("%-4.0f",x),xScale,y-.5) ;
      else if(unit >= .1)
	graphText (messprintf ("%-4.1f",x),xScale,y-.5) ;
      else if(unit >= .01)
	graphText (messprintf ("%-4.2f",x),xScale,y-.5) ;
      else if(unit >= .001)
	graphText (messprintf ("%-4.3f",x),xScale,y-.5) ;
      x += unit ;
    }

  x = look->centre - (ny2-2)/look->mag ;
  x = subunit * (((x>=0)?1:0) + (int)(x/subunit)) ;
   while ((y = GMAP2GRAPH(look, x)) < ny  - 1)
    { 
      graphLine (xScale-1.0,y,xScale-0.5,y) ;
      x += subunit ;
    }

  graphLine (xScale-0.5,2 + topMargin ,xScale-0.5,ny - 0.5 ) ;

  array(look->boxIndex,posBox=graphBoxStart(),SEG*) = (SEG*)5 ;
  graphFillRectangle (xScale-1,ny2 + topMargin ,xScale,ny2+1+topMargin) ;
  graphBoxEnd () ;
  graphBoxDraw (posBox,DARKGREEN,TRANSPARENT) ;
}

static void drawChromosome (LOOK look)
{
  float y, dy, pageful, x0, xn, x ;
  int i ;
  SEG* seg ;
  BOOL isBands = FALSE ;

  x0 = xn = 0.0 ;
  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    if (class(arrp(look->segs,i,SEG)->key) == _VGene ||
	class(arrp(look->segs,i,SEG)->key) == _VLocus)
      { y = arrp(look->segs,i,SEG)->x ;
	if (y < x0) x0 = y ;
	if (y > xn) xn = y ;
      }
  if (xn == x0)
    { messout ("No genes on this chromosome") ;
      xn = x0 + 0.1 ;		/* to avoid a crash */
    }
  look->cursor.fac = (ny - topMargin - 6.0) / (xn - x0) ;
  pageful = (ny - topMargin - 3) / look->mag ;	/* gMap units */
  look->cursor.offset = x0 + 0.5*pageful ;

  graphTextHeight (0.75) ;
  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;
      if (seg->flag & FLAG_MARKER_GENE)
	{ y = 3 + topMargin +  (seg->x - x0) * look->cursor.fac ;
	  graphText (name(seg->key),0.5,y-0.25) ;
	}
      if (class(seg->key) == _VChrom_Band)
	{ isBands = TRUE ;
	  y = 3 + topMargin +  (seg->x - x0) * look->cursor.fac ;
	  dy =  0.5 * seg->dx * look->cursor.fac ;
	  if (seg->flag & FLAG_CENTROMERE)
	    { graphColor (WHITE) ;
	      graphFillRectangle (4.5, y-dy, 6.5, y+dy) ;
	      graphColor (BLACK) ;
	      graphLine (4.5, y-dy, 6.5, y+dy) ;
	      graphLine (6.5, y-dy, 4.5, y+dy) ;
	      graphLine (4.5, y-dy, 6.5, y-dy) ;
	      graphLine (6.5, y+dy, 4.5, y+dy) ;
	    }	      
	  else 
	    { if (seg->flag & FLAG_DARK_BAND)
		graphColor (DARKGRAY) ;
	      else if (seg->flag & FLAG_NOR)
		graphColor (LIGHTGRAY) ;
	      else
		graphColor (WHITE) ;
	      if (seg->flag >> 28)
		graphColor((seg->flag  >> 28) + WHITE) ;
	      graphFillRectangle (4.5, y-dy, 6.5, y+dy) ;
	      graphColor (BLACK) ;
	      graphRectangle (4.5, y-dy, 6.5, y+dy) ;
	      graphText (name(seg->key),0.5,y-0.25) ;
	    }
	}
    }
  graphTextHeight (0) ;

  x = isBands ? 7.5 : 5.5 ;

  graphFillRectangle (x-0.25,3 + topMargin ,x+0.25,ny - 3) ;

  array(look->boxIndex,scaleBox=graphBoxStart(),SEG*) = (SEG*)4 ;
  y = 3 + topMargin + (look->centre - look->cursor.offset) * look->cursor.fac ;
  graphRectangle (x-0.5, y, x+0.5, y + pageful*look->cursor.fac) ;
  graphBoxEnd () ;
  graphBoxDraw (scaleBox,DARKGREEN,GREEN) ;

  graphColor (DARKGRAY) ;
  graphLine (x, y, xScale-0.5, 2.5 + topMargin ) ;
  graphLine (x, y + pageful*look->cursor.fac, xScale-0.5, ny - 1) ;

  graphColor (BLACK) ;
}

/*************************************************************/
/******** gMap show intersect with active keyset *************/
 
static void gMapShowAll(void)
{
  int i, n ;
  SEG *seg ;
 
  LOOKGET("gMapShowAll") ;

  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  n = FLAG_STRESSED | FLAG_RELATED | FLAG_ANTI_RELATED | FLAG_HIDE | FLAG_HIGHLIGHT ;
  while(i--)
    { seg->flag &= ~n ;  /* bitwise NOT */
      seg++ ;
    }
  look->flag|=FLAG_REVERT_DISPLAY; /*<<--flag for unhighlighting current box*/
  gMapDraw (look, 0) ;
  look->flag&= ~FLAG_REVERT_DISPLAY; /*<<--only place where I use this flag*/
}

/**********/

static void gMapHide(void)
{
  int i, j ;
  SEG *seg ;
  void *dummy ;
  KEYSET keySet = 0 ;
 
  LOOKGET("gMapHide") ;

  if(!keySetActive(&keySet, &dummy))
    { messout("First select a keySet window, thank you.") ;
      return ;
    }
  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  while(i--)
    { seg->flag &= ~FLAG_HIDE ;  /* bitwise NOT */
      if (!keySetFind(keySet, seg->key, &j)) /*<<--*/
	seg->flag |= FLAG_HIDE ;
      seg++ ;
    }
  gMapDraw (look, 0) ;
}

/**********/

static void gMapHighlight(void)
{
  int i, j;
  SEG *seg ;
  void *dummy ;
  KEYSET keySet = 0 ;
 
  LOOKGET("gMapHighlight") ;

  if (!keySetActive(&keySet, &dummy))
    { messout("First select a keySet window, thank you.") ;
      return ;
    }
  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  while(i--)
    { seg->flag &= ~FLAG_HIGHLIGHT; /*<<--*/
      if (keySetFind (keySet, seg->key, &j)) /*<<--*/
	seg->flag |= FLAG_HIGHLIGHT ;
      seg++ ;
    }
  gMapDraw (look, 0) ;
}


/****** gMap evaluation *************/
    /* JTM 19 3 91 */
static void stressedGenes(void)
{
  int i, n ;
  KEY key ;
  SEG *seg ;
  KEYSET all2p , genes, genes1, genes2, dfDup, delData, all3p, genes3,
         deletions, relations, contig, clone ;
  LOOKGET("stressedGenes") ;

  seg = arr(look->boxIndex,look->activeBox,SEG*) ;
  if(!seg)
    return ;
  key = seg->key ;
  if(class(key) == _VContig)
    { contig2genes(*seg) ;
      gMapDraw(look, 0) ;
      return ;
    }
  if(class(key) == _VRearrangement)
    { rearrangement2genes(*seg) ;
      gMapDraw(look, 0) ;
      return ;
    }
  if(class(key) != _VGene)
    { messout("First pick a gene") ;
      return ;
    }
  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  n = FLAG_STRESSED | FLAG_RELATED | FLAG_ANTI_RELATED ;
  while(i--)
    { seg->flag &= ~n ;  /* bitwise NOT */
      seg++ ;
    }

  all2p = queryKey(key,">2Point") ;
  genes1 =  query(all2p,">Gene1") ;
  genes2 =  query(all2p,">Gene2") ;
  genes = keySetOR(genes1,genes2) ;		    
  keySetDestroy(genes1) ;
  keySetDestroy(genes2) ;
  i = keySetMax(all2p) ;
  while(i--)
    gMapInsert2p(keySet(all2p,i)) ;
  keySetDestroy(all2p) ;

  dfDup = queryKey(key,">Df_Dup") ;
  delData = query(dfDup,"Results = *deletes*") ;
  deletions = query(delData,">Rearrangement") ;
  genes1 = keySetOR(genes,deletions) ;
  keySetDestroy (genes) ;
  keySetDestroy (delData) ;
  keySetDestroy (deletions) ;

  delData = query(dfDup,"Results = *includes*") ;
  deletions = query(delData,">Rearrangement") ;
  genes2 = keySetOR(genes1,deletions) ;
  keySetDestroy (genes1) ;
  keySetDestroy (delData) ;
  keySetDestroy (deletions) ;

  clone = queryKey(key,">Clone");  
  contig = query(clone,">pMap");  
  relations = keySetOR(genes2,contig) ;
  keySetDestroy(clone) ;
  keySetDestroy(contig) ;
  keySetDestroy(genes2) ;

  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  while(i--)
    { if(keySetFind(relations,seg->key,&n))
	seg->flag |= FLAG_RELATED ;
      seg++ ;
    }
  keySetDestroy(relations) ;

  all3p = queryKey(key,">3Point") ;
  genes1 =  query(all3p,">Gene1") ;
  genes2 =  query(all3p,">Gene2") ;
  genes3 =  query(all3p,">Gene3") ;
  genes = keySetOR (genes1, genes2) ;
  keySetDestroy(genes1) ;
  keySetDestroy(genes2) ;
  genes2 = keySetOR (genes, genes3) ;
  keySetDestroy (genes) ;
  keySetDestroy (genes3) ;
  i = keySetMax(all3p) ;
  while(i--)
    gMapInsert3p(keySet(all3p,i)) ;
  keySetDestroy (all3p) ;
  keySetDestroy (all3p) ;

  delData = query(dfDup,"Results = \"*does not delete*\"") ;
  deletions = query(delData,">Rearrangement") ;
  genes1 = keySetOR(genes2,deletions) ;
  keySetDestroy (genes2) ;
  keySetDestroy (delData) ;
  keySetDestroy (deletions) ;

  delData = query(dfDup,"Results = \"*does not include*\"") ;
  deletions = query(delData,">Rearrangement") ;
  relations = keySetOR(genes1,deletions) ;
  keySetDestroy (genes1) ;
  keySetDestroy (delData) ;
  keySetDestroy (deletions) ;

  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  while(i--)
    { if(keySetFind(relations,seg->key,&n))
	seg->flag |= FLAG_ANTI_RELATED ;
      seg++ ;
    }
  keySetDestroy(relations) ;
  gMapDraw (look, 0) ;
}

/****** contig handling *************/
    /* JTM 20-4-91 */
static void contig2genes(SEG seg1)
{
  register int i ; 
  int n ;
  SEG *seg ;
  KEY chromo ;
  KEYSET  genes ;
  KEY contig = seg1.key ; 
  LOOKGET("contig2genes") ;

  if(!look || !look->segs || !arrayMax(look->segs))
    return ;
  chromo = KEYMAKE(_VChromosome,KEYKEY(look->key)) ;
  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  n = FLAG_STRESSED | FLAG_RELATED | FLAG_ANTI_RELATED ;

  while(i--)
    { seg->flag &= ~n ;  /* bitwise NOT */
      seg++ ;
    }

  genes  = queryKey(chromo,
		    messprintf(
			       ">Gene Clone ; >Clone Contig = %s ;>Gene",
			       name(contig))) ;

  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  while(i--)
    { if(keySetFind(genes,seg->key,&n))
	seg->flag |= FLAG_RELATED ;
      seg++ ;
    }

  seg1.flag |= FLAG_STRESSED ;
  keySetDestroy(genes) ;

  gMapDraw (look, 0) ;
}

/*****************************************************/

static void rearrangement2genes (SEG seg1)
{
  register int i ; 
  int n ;
  SEG *seg ;
  KEYSET   genes ;
  KEY rear = seg1.key ;
  LOOKGET("rearrangement2genes") ;

  if (!look || !look->segs || !arrayMax(look->segs))
    return ;
  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  n = FLAG_STRESSED | FLAG_RELATED | FLAG_ANTI_RELATED ;
  while(i--)
    { seg->flag &= ~n ;  /* bitwise NOT */
      seg++ ;
    }

   if (isDeficiency(rear))
    genes = queryKey(rear,
		     ">Df_Dup ;  A_deletes_B ; >Gene") ; 
  else if (isDuplication(rear))
    genes = queryKey(rear,
		     ">Df_Dup ;  A_includes_B ; >Gene") ; 

  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  while(i--)
    { if(keySetFind(genes,seg->key,&n))
	seg->flag |= FLAG_RELATED ;
      seg++ ;
    }

  keySetDestroy(genes) ;
  
  if (isDeficiency(rear))
    genes = queryKey(rear,
		     ">Df_Dup ; A_does_not_delete_B ; >Gene") ; 
  else if (isDuplication(rear)) 
    genes = queryKey(rear,
		     ">Df_Dup ; A_does_not_include_B ; >Gene") ; 

  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  while(i--)
    { if(keySetFind(genes,seg->key,&n))
	seg->flag |= FLAG_ANTI_RELATED ;
      seg++ ;
    }

  keySetDestroy(genes) ;

  seg1.flag |= FLAG_STRESSED ;
  gMapDraw(look, 0) ;
}

/****** geneDragging evaluation *************/
    /* JTM 6 12 91 */

static KEY draggedGene = 0 ;
static void gMapMiddleUpGene (double x, double y) 
{  SEG *seg ;
  LOOKGET("gMapMiddleUp") ;

   seg = arr(look->boxIndex,look->activeBox,SEG*) ;
   seg->x = look->centre + (y - ny2 - topMargin )  / look->mag ;
   seg->dx = 2 * oldDy  / look->mag ;

   arraySort(look->segs,order) ;
   graphRegister (MIDDLE_DOWN, (GraphFunc)gMapMiddleDown) ;
   gMapDraw (look, draggedGene) ;
}

/************/

static void gMapMiddleDragGene (double x, double y) 
{
    graphXorLine (DRAGFASTLIMIT, oldy - oldDy , nx, oldy - oldDy) ;
    if(oldDy) 
      graphXorLine (DRAGFASTLIMIT, oldy + oldDy , nx, oldy + oldDy) ;
 
    oldy = y ;
    if(oldx != -99999)
      oldDy *= exp ((x - oldx) / 25.) ;
    oldx = x ;
    graphXorLine (DRAGFASTLIMIT, y - oldDy , nx, y - oldDy) ;
     if(oldDy) 
       graphXorLine (DRAGFASTLIMIT, y + oldDy , nx, y + oldDy) ;
}

/************/

static void gMapDragGene(void)
{ 
  KEY key ;
  float y ;
  SEG *seg ;
  LOOKGET("dragGene") ;

  seg = arr(look->boxIndex,look->activeBox,SEG*) ;
  if(!seg)
    { messout("First pick a gene or a rearrangement") ;
      return ;
    }

  key = seg->key ;
  y = GMAP2GRAPH(look,seg->x) ;
  oldDy = seg->dx * look->mag / 2 ;

  oldx = -99999 ;

  if( class(key) != _VGene &&
     class(key) != _VLocus &&
     class(key) != _VRearrangement)
    { messout("First pick a gene or a locus or a rearrangement") ;
      return ;
    }

  draggedGene = key ;
  lookDrag = look ;
  graphXorLine (DRAGFASTLIMIT, y - oldDy , nx, y - oldDy) ;
  if(oldDy) 
    graphXorLine (DRAGFASTLIMIT, y + oldDy , nx, y + oldDy) ;
  oldy = y ;


  graphRegister (MIDDLE_DOWN, (GraphFunc) gMapMiddleDragGene) ;
  graphRegister (MIDDLE_DRAG,(GraphFunc) gMapMiddleDragGene) ;
  graphRegister (MIDDLE_UP,(GraphFunc) gMapMiddleUpGene) ;
}

/****** magnification control ******/
  /* JTM 18-6 Recenters the whole chromo correctly.
   * For arabidopsis, centre in positive coordinates somewhere,
   * For C.elegans we defualt the value to zero.
   */

static void gMapFindMag(LOOK look) 
{ SEG *seg = arrp(look->segs, arrayMax(look->segs) - 1 , SEG) ;

  graphFitBounds (&nx,&ny) ;
  ny2 = 0.5 * (ny - topMargin) ; 

  if (seg->key != _Centre)
    messout ("Please recalculate this genetic map") ;
  if (seg->flag)
    look->mag = (2*ny2-5)/(seg->flag) ; /* User announce length */ 
  else
    look->mag = 10 ; /* Default */
}


static void gMapFindCentre(LOOK look) 
{ SEG *seg = arrp(look->segs, arrayMax(look->segs) - 1 , SEG) ;

  if (seg->key != _Centre)
    messout ("Please recalculate this genetic map") ;
  
  look->centre = seg->x ;
}

static void wholeChromosome (void)
{ SEG *seg ;
  LOOKGET("wholeChromosome") ; 

  seg = arrp(look->segs, arrayMax(look->segs) - 1 , SEG) ;

  if (seg->key != _Centre)
    messout ("Please recalculate this genetic map") ;
  
  look->mag = (2*ny2-5)/(seg->dx) ;
  look->centre = seg->x ;

  gMapDraw (look, 0) ;
}

static void zoomIn (void)
{ LOOKGET("zoomIn") ; look->mag *= 2 ; gMapDraw (look, 0) ;}

static void zoomOut (void)
{ LOOKGET("zoomOut") ; look->mag /= 2 ; gMapDraw (look, 0) ;}

static MENUOPT buttonOpts[] = {
  wholeChromosome, "Whole", 
  zoomIn, "Zoom In",
  zoomOut, "Zoom Out",
  stressedGenes,"GMap Data",
  gMapHide,"Restrict",
  gMapHighlight,"Highlight",
  gMapShowAll,"All",
  gMapDragGene,"Drag Gene",
  0, 0} ;

/**************************************/

static int order(void *a, void *b)     /* for arraySort() call */
{
  SEG *seg1 = (SEG*)a, *seg2 = (SEG*)b ;
  int t1 = class(seg1->key), t2 = class(seg2->key) ;
  float diff = seg1->x - seg2->x ;

  if(seg1->key == _Centre)  /* centre is last, used in gMapFindCentre */
    return +1 ;
  if(seg2->key == _Centre)
    return -1 ;

  if ((t1 == _VClone || t1 == _VContig) &&
      (t2 != _VClone && t2 != _VContig))
    return -1 ;
  else if ((t2 == _VClone || t2 == _VContig) &&
	   (t1 != _VClone && t1 != _VContig))
    return 1 ;

  if (diff > 0)
    return 1 ;
  else if (diff < 0)
    return -1 ;
  else
    return 0 ;
}

/****************/

static void gMapGetMargins (LOOK look)
{
  int   i, table;
  float y1, y2 ;
  SEG   *seg ;
  BOOL  isBands = FALSE ;

  if (foreignSeg.key)
    arrayInsert (look->segs, &foreignSeg, order) ;

  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;
 
      if(seg->flag & FLAG_HIDE)
	continue ;
      y1 = GMAP2GRAPH(look,seg->x - seg->dx/2) ;
      y2 = GMAP2GRAPH(look,seg->x + seg->dx/2) ;
      if (y2 > topMargin && y1 < ny - 1)
	{ 
	  table = class(seg->key) ;
	  if (seg == &foreignSeg)
	    table = _VGene ;
	  switch (table)
	    { 
	    case _VRearrangement:
	      bumpRear (FALSE, look, bumpR, seg->key, y1+0.5, y2+0.5) ;
	      break ;
	    case _V2_point_data:
	      if (seg->flag & FLAG_RELATED)
		bumpRear (FALSE, look,bump2, seg->key, y1+0.5, y2+0.5) ;
	      break ;
	    case _V3_point_data:
	      if (seg->flag & FLAG_RELATED)
		bumpRear (FALSE, look,bump3, seg->key, y1+0.5, y2+0.5) ;
	      break ;
	    case _VChrom_Band:
	      isBands = TRUE ;
	      break ;
	    }
	}
    }
  xRear = 8 ;
  if (isBands)
    xRear += 2 ;
  xScale = xRear + arrayMax(bumpR) + 4 ;
  if (isBands)
    xScale += 1 ;
  x2p = xScale + 6 ;
  x3p = x2p + (arrayMax(bump2) ? arrayMax(bump2) + 2 : 0 ) ;
  xGenes = x3p + (arrayMax(bump3) ? arrayMax(bump3) + 2 : 0 ) ;
}

/**************************************************************/

static void gMapDraw (LOOK look, KEY curr)
{
  int   i, ibox, table;
  float y, y1, y2 ;
  SEG   *seg, *contigSeg = 0 ;
  BUMP  bump = bumpCreate (64) ;
  
  if (!curr && look->activeBox)
    curr = arr(look->boxIndex,look->activeBox, SEG*)->key ;

  graphClear () ;
  graphBoxDraw (0,BLACK,WHITE) ;
  graphFitBounds (&nx,&ny) ;

  if (ny < 10 + topMargin)
    { messout ("Sorry, this window is too small for a genetic gMap") ;
      return ;
    }
  ny2 = 0.5 * (ny - topMargin) ; 

  look->boxIndex = arrayReCreate (look->boxIndex,50,SEG*) ;
  
  yBumpReset (&bumpR) ; 
  yBumpReset (&bump2) ;
  yBumpReset (&bump3) ;

  gMapGetMargins(look) ;

  yBumpReset (&bumpR) ; 
  yBumpReset (&bump2) ;
  yBumpReset (&bump3) ;

  look->activeBox = 0 ;
  look->rearNameBox = 0 ;
  arrayMax (look->boxIndex) = 0 ;

  graphText (name (look->key),1,0.5) ;

  graphText ("NB Contig ends are extrapolated", 50, 2.5) ;
  graphText ("by a global linear fit on the contig", 50, 3.5) ;

  drawScale (look) ;
  drawChromosome (look) ;   /* must come at end of preamble boxes*/
  minLiveBox = arrayMax(look->boxIndex) ;

  graphColor (DARKGRAY) ;
  graphFillRectangle (xScale-2.6,2 + topMargin ,xScale-2.3,ny - 0.5) ;
  graphColor (BLACK) ;

  if (foreignSeg.key)
    arrayInsert (look->segs, &foreignSeg, order) ;

  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;

      if(seg->key == _Centre)
	continue ;
      if(seg->flag & FLAG_HIDE)
	continue ;
      y1 = GMAP2GRAPH(look,seg->x - seg->dx/2) ;
      y2 = GMAP2GRAPH(look,seg->x + seg->dx/2) ;
      if (y2 > topMargin  && y1 < ny - 1)
	{ 
	  table = class(seg->key) ;
	  if (contigSeg && table != _VClone) /* safe end contig */
	    { graphBoxEnd() ;
	      contigSeg = 0 ;
	    }

	  if (seg->flag & FLAG_CONTIG_START) /* arrows outside box */
	    { y = GMAP2GRAPH(look,seg->x) ;
	      graphLine (xScale-3.1, y+0.4, xScale-2.5, y) ;
	      graphLine (xScale-1.9, y+0.4, xScale-2.5, y) ;
	    }
	  else if (seg->flag & FLAG_CONTIG_END)
	    { y = GMAP2GRAPH(look,seg->x) ;
	      graphLine (xScale-3.1, y+0.6, xScale-2.5, y+1.0) ;
	      graphLine (xScale-1.9, y+0.6, xScale-2.5, y+1.0) ;
	      continue ;	/* done with this seg */
	    }
		  
	  ibox = graphBoxStart() ;
	  array(look->boxIndex,ibox,SEG*) = seg ;
	  if (seg == &foreignSeg)
	    table = _VGene ;
	  switch (table)
	    { 
	    case _VGene:
	    case _VLocus:
	      y = (y1+y2)/2 ;
	      if (y < topMargin + 2 || y > ny - 1)
		break ;
	      bumpGene (bump,name(seg->key),20*seg->dx,y) ;
	      if (seg->flag & FLAG_STRESSED)
		graphBoxDraw (ibox,BLACK,RED) ;
	      else if (seg->flag & FLAG_RELATED)
		graphBoxDraw (ibox,BLACK,GREEN) ;
	      else if (seg->flag & FLAG_ANTI_RELATED)
		graphBoxDraw (ibox,BLACK,LIGHTBLUE) ;
	      else if (seg->flag & FLAG_CLONED)
		graphBoxDraw (ibox,BLACK,YELLOW) ;
              else
	        graphBoxDraw (ibox,BLACK,WHITE) ;
	      break ;
	    case _VClone:
	      graphRectangle (xScale-3.0,y1-0.1,xScale-2,y1+0.1) ;
	      if (seg->flag & FLAG_STRESSED)
		graphBoxDraw (ibox,BLACK,RED) ;
	      else
		graphBoxDraw (ibox,BLACK,YELLOW) ;
	      break ;
	    case _VRearrangement:
	      bumpRear (TRUE, look, bumpR, seg->key, y1+0.5, y2+0.5) ;
	      if (seg->flag & FLAG_STRESSED)
		graphBoxDraw (ibox,BLACK,RED) ;
	      else if (seg->flag & FLAG_RELATED)
		graphBoxDraw (ibox,BLACK,GREEN) ;
	      else if (seg->flag & FLAG_ANTI_RELATED)
		graphBoxDraw (ibox,BLACK,LIGHTBLUE) ;
              else
		graphBoxDraw (ibox,BLACK,TRANSPARENT) ;
	      break ;
	    case _V2_point_data:
	      if (seg->flag & FLAG_RELATED)
		bumpRear (TRUE, look, bump2, seg->key, y1+0.5, y2+0.5) ;
	      break ;
	    case _V3_point_data:
	      if (seg->flag & FLAG_RELATED)
		bumpRear (TRUE, look, bump3, seg->key, y1+0.5, y2+0.5) ;
	      break ;
	    case _VContig:
	      if (!(seg->flag & FLAG_CONTIG_START))
		messerror ("Screwup with drawing contig ends") ;
	      y1 = GMAP2GRAPH(look,seg->x) ;
	      if (y1 > ny - 1)
		  break ;
	      contigSeg = seg ;
	      if (y1 < topMargin + 1.5)
		y1 = topMargin + 1.5 ;
	      graphLine(xScale-3.0, y1+0.5, xScale-3.0, y2+0.5) ;
	      graphLine(xScale-2.0, y1+0.5, xScale-2.0, y2+0.5) ;
	      if (seg->flag & FLAG_STRESSED)
		graphBoxDraw (ibox,BLACK,RED) ;
	      else if (seg->flag & FLAG_RELATED)
		graphBoxDraw (ibox,BLACK,LIGHTRED) ;
              else
		graphBoxDraw (ibox,BLACK,YELLOW) ;
	      goto skipBoxEnd ;
	    case _VChrom_Band:
	      graphRectangle (xScale-5.0,y1+0.5,xScale-3.5,y2+0.5) ;
	      if (seg->flag & FLAG_DARK_BAND)
		graphBoxDraw (ibox, BLACK, DARKGRAY) ;
	      else if (seg->flag & FLAG_NOR)
		graphBoxDraw (ibox, BLACK, LIGHTGRAY) ;
	      else
		graphBoxDraw (ibox, BLACK, WHITE) ;
	      break ;
	    default:
	      messout ("Where do I put %s in a gMap",
		       name(seg->key) ) ; break;
	    }
	  graphBoxEnd() ;
	  if (seg->flag & FLAG_HIGHLIGHT)
	    graphBoxDraw (ibox, BLACK, MAGENTA) ;
	skipBoxEnd:
	  if (seg->key == curr && !(look->flag & FLAG_REVERT_DISPLAY)) gMapSelect(look, ibox);
	}
    }
  drawNeighbourKeys (TRUE) ;
                 /* Draw buttons last to be on top of graph */
  graphButtons (buttonOpts, 5, 0.5, nx) ; 
  graphRedraw () ;
  bumpDestroy (bump) ;
}

/**********************************/

static Array gMapConvert (KEY chrom)
{
  KEY gMap, gene, key, locus, clone, contig, rearrangement, band , col ;
  OBJ Chrom, Gene, Locus, Cluster, Rearrangement, Contig, Band ;
  Array segs ;
  SEG *seg;
  float x, dx, x1, x2 , xmin = 9999999, xmax = -999999 ;
  int i = 0;

  if (!(Chrom = bsCreate (chrom)))
    { messout ("No data for chromosome %s",name(chrom)) ;
      return 0 ;
    }
  if(bsFindTag(Chrom, _Non_graphic))
    { bsDestroy(Chrom) ;
      return 0 ;
    }
  if (lexReClass(chrom, &gMap,_VgMap) &&
      !lexlock (gMap))
    { messout ("Sorry, %s is locked (being processed elsewhere)",
	       name(gMap)) ; 
      return 0 ;
    }

  segs = arrayCreate (128,SEG) ;

  if (bsGetKey (Chrom,_Gene,&gene))
    do if (Gene = bsCreate (gene))
      { 
	x = 10000.0 ; dx = 0 ;
	if (bsGetKey (Gene,_gMap,&key)
	    && bsGetData (Gene,_bsRight,_Float,&x))
	  bsGetData (Gene,_bsRight,_Float,&dx) ;
	else if (bsGetKey (Gene,_In_cluster,&key) 
		 && (Cluster = bsCreate (key)))
	  { if (bsGetKey (Cluster,_gMap,&key) && (key == chrom) && 
		bsGetData (Cluster,_bsRight,_Float,&x))
	      bsGetData (Cluster,_bsRight,_Float,&dx) ;
	    bsDestroy (Cluster) ;
	  }

	if (x != 10000.0)
	  { seg = arrayp(segs,i++,SEG) ;
	    seg->key = gene ;
	    seg->x = x ;
	    seg->dx = dx ;
	    seg->flag = 0 ;
	    if (bsFindTag (Gene,_Marker_gene))
	      seg->flag |= FLAG_MARKER_GENE ;
	    if (bsGetKey (Gene,_Clone,&clone))
	      { seg->flag |= FLAG_CLONED ;
		seg = arrayp(segs,i++,SEG) ;
		seg->key = clone ;
		seg->x = x ;
		seg->dx = 0 ;
		seg->flag = 0 ;
   	/*	if (bsFindTag (Gene, _Mapping_data))  */
		  seg->flag |= FLAG_GMAPPED_CLONE ;
	      }
	    if (x < xmin)
	      xmin = x ;
	    if(x>xmax)
	      xmax = x ;
	  }    
	bsDestroy (Gene) ;
      }
    while(bsGetKey(Chrom,_bsDown,&gene)) ;

  if (bsGetKey (Chrom,_Locus,&locus))
    do if (Locus = bsCreate (locus))
      { 
	x = 10000.0 ; dx = 0 ;
	if (bsGetKey (Locus,_gMap,&key)
	    && bsGetData (Locus,_bsRight,_Float,&x))
	  bsGetData (Locus,_bsRight,_Float,&dx) ;

	if (x != 10000.0)
	  { seg = arrayp(segs,i++,SEG) ;
	    seg->key = locus ;
	    seg->x = x ;
	    seg->dx = dx ;
	    seg->flag = 0 ;
	    if (bsGetKey (Locus,_Clone,&clone))
	      { seg->flag |= FLAG_CLONED ;
		seg = arrayp(segs,i++,SEG) ;
		seg->key = clone ;
		seg->x = x ;
		seg->dx = 0 ;
		seg->flag = 0 ;
		seg->flag |= FLAG_GMAPPED_CLONE ;
	      }
	    if (x < xmin)
	      xmin = x ;
	    if(x>xmax)
	      xmax = x ;
	  }    
	bsDestroy (Locus) ;
      }
    while(bsGetKey (Chrom,_bsDown,&locus)) ;

  if (bsGetKey(Chrom,_Rearrangement,&rearrangement))
    do if (Rearrangement = bsCreate(rearrangement))
      {
	if (bsGetKey (Rearrangement,_gMap,&key) && (key == chrom) && 
	    bsGetData (Rearrangement,_bsRight,_Float,&x1) &&
	    bsGetData (Rearrangement,_bsRight,_Float,&x2))
	  { seg = arrayp(segs,i++,SEG) ;
	    seg->key = rearrangement ;
	    seg->x = x = (x1 + x2) / 2 ;
	    seg->dx = x2 - x1 ;
	    if (seg->dx < 0)
	      seg->dx = -seg->dx ;
	    seg->flag = 0 ;
	    if (x < xmin)
	      xmin = x ;
	    if(x>xmax)
	      xmax = x ;
	  }

	bsDestroy (Rearrangement) ;
      }
    while(bsGetKey(Chrom,_bsDown,&rearrangement)) ;

  if (bsGetKey(Chrom,_Contig,&contig))
    do if (Contig = bsCreate (contig))
      {
	if (bsGetKey (Contig,_gMap,&key) && (key == chrom) && 
	    bsGetData (Contig,_bsRight,_Float,&x1) &&
	    bsGetData (Contig,_bsRight,_Float,&x2))
	  { seg = arrayp(segs,i++,SEG) ;
	    seg->key = contig ;
	    seg->x = x1 ;
	    seg->dx = 2*(x2 - x1) ;
	    seg->flag = FLAG_CONTIG_START ;
	    seg = arrayp (segs,i++,SEG) ;
	    seg->key = contig ;
	    seg->x = x2 ;
	    seg->dx = 2*(x2 - x1) ;
	    seg->flag = FLAG_CONTIG_END ;
	  }

	bsDestroy (Contig) ;
      }
    while (bsGetKey (Chrom,_bsDown,&contig)) ;

  if (bsGetKey (Chrom, _Chrom_Band, &band))
    do if (Band = bsCreate (band))
      { if (bsGetKey (Band,_gMap,&key)
	    && bsGetData (Band,_bsRight,_Float,&x1)
	    && bsGetData (Band,_bsRight,_Float,&x2))
	  { seg = arrayp(segs,i++,SEG) ;
	    seg->key = band ;
	    seg->x = 0.5*(x1+x2) ;
	    seg->dx = x2 - x1 ;
	    seg->flag = 0 ;
	    if (bsFindTag (Band,_Centromere))
	      seg->flag |= FLAG_CENTROMERE ;
	    if (bsFindTag (Band,_Dark))
	      seg->flag |= FLAG_DARK_BAND ;
	    if (bsFindTag (Band,_NOR))
	      seg->flag |= FLAG_NOR ;
	    if (bsFindTag (Band,_Colour) &&
		bsGetKeyTags(Band, _bsRight, &col))
	      seg->flag |= ((col - _WHITE) & 15  ) << 28;
	    
	    if (x1 < xmin)
	      xmin = x1 ;
	    if (x2 > xmax)
	      xmax = x2 ;
	  }    
	bsDestroy (Band) ;
      }
    while (bsGetKey (Chrom, _bsDown, &band)) ;


  if (gMap)
    lexunlock(gMap) ;

  if (arrayMax(segs))
    { float centre = 0 ;
				/* Add in the value of the centre and extension */
      seg = arrayp(segs,arrayMax(segs),SEG) ;
      seg->key = _Centre ;
      seg->x = 0 ; seg->flag = 0 ;
      if (bsGetData (Chrom, _Centre, _Float, &centre))
	{ seg->x = centre ;
	  if (bsGetData (Chrom, _bsRight, _Float, &centre))
	    seg->flag  = centre ; /* scandalous abuse of seg->flag by jtm */
	}
     	
      seg->dx = xmax - xmin > 2 ? xmax - xmin : 2. ;
    
      arraySort (segs,order) ;
      lexaddkey(name(chrom), &gMap, _VgMap) ;
      arrayStore (gMap,segs,segFormat) ;
    }
  else
    arrayDestroy(segs) ;
  
  bsDestroy (Chrom) ;
  return segs ;
}

/*********************************************************/

static void gMapSelect (LOOK look, int box)
{
  SEG *seg ;
  static char rearNameText[20] ;

  if (box >= arrayMax(look->boxIndex) || box < minLiveBox)
    return ;

  if (look->activeBox)
    { seg = arr(look->boxIndex,look->activeBox,SEG*) ;
      switch (class (seg->key))
	{
	case _VGene:
	case _VLocus:
	  graphBoxDraw(look->activeBox, BLACK,
            (seg->flag & FLAG_HIGHLIGHT)? MAGENTA:
	    (seg->flag & FLAG_CLONED)? YELLOW: WHITE
          );
	  break ;
	case _VClone:
	case _VContig:
	  graphBoxDraw (look->activeBox, BLACK, (seg->flag & FLAG_HIGHLIGHT)? MAGENTA: YELLOW);
	  break ;
	case _VRearrangement:
	  graphBoxDraw (look->activeBox, BLACK, (seg->flag & FLAG_HIGHLIGHT)? MAGENTA: WHITE);
	  if (look->rearNameBox)
	    graphBoxDraw (look->rearNameBox, WHITE, (seg->flag & FLAG_HIGHLIGHT)? MAGENTA: WHITE) ;
	  break ;
	}
    }

  look->activeBox = box ;

  seg = arr(look->boxIndex,look->activeBox,SEG*) ;
  switch (class (seg->key))
    {
    case _VGene:
    case _VLocus:
      graphBoxDraw (look->activeBox, WHITE, RED) ;
      break ;
    case _VClone:
    case _VContig:
      graphBoxDraw (look->activeBox, BLACK, RED) ;
      break ;
    case _VRearrangement:
      graphBoxDraw (look->activeBox, WHITE, RED) ;
      if (!look->rearNameBox)
	{ look->rearNameBox = graphBoxStart () ;
	  graphTextPtr (rearNameText, 7, 3, xRightEnd-7) ;
	  graphBoxEnd () ;
	}
      strcpy (rearNameText, name (seg->key)) ;
      graphBoxDraw (look->rearNameBox, WHITE, RED) ;
    }
}

static void gMapFollow (LOOK look, double x, double y)
{ SEG *seg = arr(look->boxIndex,look->activeBox,SEG*) ;
  SEG *seg1, *seg2 ;
  KEY from = look->key, contig, key ;
  float y1, y2, c1, c2 ;
  int i, x1, x2 ;
  OBJ Clone ;

   switch(class(seg->key))
     {
     case _VRearrangement:
     case _V2_point_data:
     case _V3_point_data:
     case _VGene:
     case _VLocus:
       display (seg->key,from,TREE) ;
       break ;
     case _VContig:
       contig = seg->key ;
       y1 = seg->x ; 
       if (GMAP2GRAPH(look,y1) < 1.5)
	 y1 = look->centre + (1.5 - ny2)/look->mag ;
       y2 = seg->x + seg->dx/2 ;
       y = y/look->mag + y1 ;
 /* strategy: find 2 gMapped clones around or by y and interpolate */
       seg1 = seg2 = 0 ; 
       c1 = c2 = 0 ;
       for (i = 0 ; i < arrayMax(look->segs) ; ++i)
	 { seg = arrp(look->segs,i,SEG) ;
	   if (seg->x > y2)
	     break ;
  /* one could be more restrictive when creating  FLAG_GMAPPED_CLONE */
	   if (seg->x >= y1 && (seg->flag & FLAG_GMAPPED_CLONE))
	     { Clone = bsCreate (seg->key) ;
	       if (bsGetKey (Clone, _pMap, &key) 
		   &&  key == contig
		   &&  bsGetData (Clone, _bsRight, _Int, &x1)
		   &&  bsGetData (Clone, _bsRight, _Int, &x2))
		 { seg1 = seg2 ; seg2 = seg ;
		   c1 = c2 ; c2 = 0.5*(x1+x2) ;
		   if (seg1 && seg->x > y)
		     break ;
		 }
	       bsDestroy (Clone) ;
	     }
	 }
       drawNeighbourKeys (FALSE) ;
       setNeighbourKeys (0,0) ;
       if (!seg2)
	 messout ("Sorry: contig has no genetic gMap data") ;
       else if (!seg1)
	 { messout ("Sorry: only one gMapped clone in that contig \
- putting you there") ;
	   display (contig, seg2->key, 0) ;
	 }
       else
	 { x1 = 0x800000L + c1 +
	     (c2 - c1) * (y - seg1->x) / (seg2->x - seg1->x) ;
	   from = KEYMAKE (_VCalcul,x1) ;
/*
	   printf ("interpolating between %s and %s - %d\n",
   name(seg1->key),name(seg2->key),x1) ;
*/
	   setNeighbourKeys (seg1->key, seg2->key) ;
	   drawNeighbourKeys (TRUE) ;
	   display (contig, from, 0) ;
	 }
       break ;

/* what follows is JTM's version for the linear fit */
/*
  double mx, p = 0 ;
  int n ;
  OBJ Contig ;
       mx = look->mag * seg->dx ;
       if (y >= 0 && y <= mx)
	 p = y /mx ;
       if (Contig = bsCreate(seg->key))
	 { if (bsGetData(Contig,_Length,_Int,&n))
	     { n = n*p/4096 ;
	       from = KEYMAKE(_VCalcul,n) ;
	     }
	   bsDestroy(Contig) ;
	 }
       display (seg->key,from,0) ;
       break ;
*/
     default:
       display (seg->key,from,0) ;
       break ;
     }
}

/******************************/

void pMapToGMap (KEY contig, KEY from, int y)
{
  KEY gMap, chrom, key ;
  OBJ Contig, Clone ;
  SEG *seg, *seg1, *seg2 ;
  float c1, c2 ;
  int i, x1, x2 , contigStart, contigEnd ;
  Array segs ;
  BOOL inContig = FALSE ;

  		/* find the correct chromosome */
  if (!(Contig = bsCreate(contig)))
    { messout ("Can't open contig object") ;
      return ;
    }
  if (!bsGetKey (Contig, _gMap, &chrom) || !chrom)
    { bsDestroy (Contig) ;
      messout("This contig is not assigned to a chromosome") ;
      return ;
    }
  bsGetData (Contig,_pMap,_Int,&contigStart) ;
  bsGetData (Contig,_bsRight,_Int,&contigEnd) ;
  bsDestroy (Contig) ;

		/* next check if it is there already */
  gMap = KEYMAKE(_VgMap, KEYKEY(chrom)) ;
  segs = arrayGet (gMap, SEG,segFormat) ;
  if(!segs) 
    { messout("Please recalculate the genetic gMap of chromosome %s",
	      name(chrom)) ;
      return ;
    }
  if(from && class(from) != _VClone)
    for (i = arrayMax(segs) ; i-- ;)
      if (arr(segs,i,SEG).key == from)
	{ display (gMap, from, 0) ;
	  arrayDestroy (segs) ;
	  return ;
	}

 /* strategy: find 2 gMapped clones around or by y and interpolate */
  seg1 = seg2 = 0 ; 
  c1 = c2 = 0 ;
  for (i = 0 ; i < arrayMax(segs) ; ++i)
    { seg = arrp(segs,i,SEG) ;

      if (seg->key == contig)
	if (!inContig)              	/* start of contig */
	  { inContig = TRUE ;
	    seg2 = seg ;
	    c2 = contigStart ;
	    continue ;
	  }
	else 
	 { inContig = TRUE ;      	/* end of contig */
	    seg1 = seg2 ; seg2 = seg ;
	    c1 = c2 ; c2 = contigEnd ;
	    break ;
	  }
      if (seg->flag & FLAG_GMAPPED_CLONE)
	{ Clone = bsCreate (seg->key) ;
	  if (bsGetKey (Clone, _pMap, &key) 
	      &&  key == contig
	      &&  bsGetData (Clone, _bsRight, _Int, &x1)
	      &&  bsGetData (Clone, _bsRight, _Int, &x2))
	    { seg1 = seg2 ; seg2 = seg ;
	      c1 = c2 ; c2 = 0.5*(x1+x2) ;
	    }
	  bsDestroy (Clone) ;
	  if (seg1 && c2 > y)
	    break ;
	}
    }

  foreignSeg.key = 0 ;
  foreignSeg.flag = FLAG_FOREIGN_SEG ;
  foreignSeg.dx = 0 ;
  if (!seg2)
    { messout ("Sorry: contig has no genetic gMap data") ;
      setNeighbourKeys (0, 0) ;
    }
  else if (!seg1)
    { messout ("Sorry: only one gMapped clone in that contig \
- putting you there") ;
      foreignSeg.key = seg2->key ;
      foreignSeg.x = seg2->x ;
      setNeighbourKeys (seg2->key, 0) ;
      display (gMap, seg2->key, 0) ;
    }
  else
    { float x = seg1->x + (seg2->x - seg1->x) * (y - c1) / (c2 - c1) ;
      setNeighbourKeys (seg1->key, seg2->key) ;
      from = KEYMAKE(_VCalcul, 1000.0*(x + 1000.0)) ;
      display (gMap, from, 0) ;
    }
  arrayDestroy (segs) ;
}

/*****************************/

static KEY aboveKey, belowKey ;

static void setNeighbourKeys (KEY newAbove, KEY newBelow)
{
  aboveKey = newAbove ;
  belowKey = newBelow ;
}

static void drawNeighbourKeys (BOOL isOn)
{
  KEY key ;
  int i ;
  LOOKGET ("drawNeighbourKeys") ;

  for (i = minLiveBox ; i < arrayMax(look->boxIndex) ; ++i)
    { key = arr (look->boxIndex,i,SEG*)->key ;
      if (key == aboveKey || key == belowKey)
	if (isOn)
	  graphBoxDraw (i, DARKBLUE, DARKBLUE) ;
	else
	  graphBoxDraw (i, BLACK, YELLOW) ;
      else if (class(key) != _VContig && 
	       class(key) != _VClone)
	break ;
    }
}



/**************************************************************/
/*********  Private Maps **************************************/

static LOOK lookLoad = 0 ;
static Graph lookLoadGraph = 0 ;

static void gMapSaveMap(void)
{ KEY key = 0 ;

  LOOKGET("gMapSaveMap") ;
 
  if(!isWriteAccess())
    { messout("Sorry, you do not have Write Access");
      return ;
    }

  while (!key && graphPrompt ("Give a name","","t"))
    if (!lexaddkey 
	(messprintf("gMapPrivate-%s-%s",
		    name(look->key), freeword()) , &key,_VgMap) &&
	!graphQuery ("Overwrite existing keyset ?"))
      return ;
  
  if (!key)
    return ;
      
  if (!lexlock(key))
    { messout ("Sorry, key is locked by someone else") ;
      return ;
    }
  arrayStore (key,look->segs,segFormat) ;
  lexunlock (key) ;
}

/******************/

static void gMapLoadMap2(KEY key)
{ Array segs  = 0 ;

  if(!key || class(key) != _VgMap ||
     strncmp(name(key),"gMapPrivate-",12)  )
    { messout("%s is not a private genetic map, sorry") ;
      return ;
    }
  if (segs = arrayGet(key, SEG,segFormat))
    { arrayDestroy(lookLoad->segs) ;
      lookLoad->segs = segs ;
      if(!graphActivate(lookLoadGraph))
	messcrash("Graph confusion in gMapDraw") ;
 
      gMapDraw(lookLoad, 0) ;
    }
}

/**************/

static void gMapLoadMap(void)
{
  extern void ksetShowSelection(KEYSET ks) ; /*<<--just to keep the compiler quiet: should sort it out better*/
  KEYSET ks ;
  LOOKGET("gMapLoadMap") ;

  ks = query(0, messprintf(">?gMap  gMapPrivate-%s-*",
			name(look->key) )) ;
  if(!ks || !keySetMax(ks))
    {
      messout("Sorry, no private version of the %s is available.",
	      name(look->key)) ;
      keySetDestroy(ks) ;
      return ;
    }
  lookLoad = look ;
  lookLoadGraph = graphActive() ;
  displayBlock((BlockFunc) gMapLoadMap2, 
	       "Chose a private map from the selection window") ;
  ksetShowSelection(ks) ;
}

/***************************************************************/
/********** code for general stuff in wormtype.h ***********/

#include <ctype.h>

BOOL isDeficiency(KEY def)
{ char *cp = name(def) ;
  return ((islower(*cp) && (*(cp+1)=='D') && (*(cp+2)=='f'))
	  || (islower(*cp) && islower(*(cp+1))
	      && (*(cp+2)=='D') && (*(cp+3)=='f'))) ;
}

BOOL isDuplication(KEY def)
{ char *cp = name(def) ;
  return ((islower(*cp) && (*(cp+1)=='D') && (*(cp+2)=='p'))
	  || (islower(*cp) &&  islower(*(cp+1))
	      && (*(cp+2)=='D') && (*(cp+3)=='p'))) ;
}

BOOL isTranslocation(KEY def)
{ char *cp = name(def) ;
  return ((islower(*cp) && (*(cp+1)=='T'))
	  || (islower(*cp) &&  islower(*(cp+1)) && (*(cp+2)=='T'))) ;
}

BOOL isGene(KEY def)
{ char *cp = name(def) ;
  return 
    islower(*cp++)
      && islower(*cp++)
	&& islower(*cp++)
	  && (*cp == '-') ;
}

/************* end of file ****************/
