/*  File: fmapdisp.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 **      Shows and Edits feature fMaps                            **
 * Exported functions: fMapDisplay, fMapMakeAll, fMapActive
   plus communications with dnacpt.c: displayDna, displayDnaPicture,
   getMessage, intronsExons, Raise nd SetStatus, sequenceLength
 * HISTORY:
 * Last edited: Apr  2 17:46 1992 (mieg)
 * * Mar 11 19:34 1992 (rd): rewrote fmapIntronsExons to work properly
	also origin shift to object and originButton, posConvert...
 * * Jan  6 19:45 1992 (rd): fixed scrolling by picking green box
 * * Dec 13 15:05 1991 (mieg): added topMargin
 * * Nov  5 12:01 1991 (rd): added Analyze to main menu, fixed selection bug
 * * Oct 24 13:05 1991 (mieg): I shortened the intron segs by 2 bases
 * * Oct 21 13:32 1991 (mieg): I adjusted fMapCentre in an artrificial way
 * * Oct 18 20:30 1991 (mieg): Changing the original scale in fmapdisplay
 * Created: Fri Oct 18 20:14:17 1991 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "keyset.h"
#include "graph.h"
#include "key.h"
#include "lex.h"
#include "bs.h"
#include "a.h"
#include "dna.h"
#include "systags.wrm"
#include "classes.wrm"
#include "sysclass.wrm"
#include "tags.wrm"
#include "disptype.wrm"
#include "display.h"
#include "query.h"
#include "pick.h"
#include "session.h"
#include "wormtype.h"

typedef struct LOOKSTUFF
  { int   magic;        /* == MAGIC */
    KEY   seqKey, mapKey , dnaKey ;
    int   min, max, origin, length, position, frame, 
          activeBox, dnaBox, homolBox , sequenceBox, proteinBox , 
          frameBox, lengthBox, originBox, segNameBox;
    char  homolText[40] , originBuffer[24],
          lengthBuffer[12], frameBuffer[10],
    	  segNameBuffer[16], segX1Buffer[12], segX2Buffer[12] ;
    Graph graph ;
    float centre , mag , topDnaBox ; /* fMap units * mag = window coords */
    Array segs,      	 /* array of SEG's from the obj */
          boxIndex,      /* if >minLiveBox, a SEG, else something special */
          dnaArray ,
          protein,
          colors ;
    int showStatus ;
    BOOL isRna ;
    int proteinWidth, dnaWidth, fWidth ;
    struct
      { float fac,offset ;
      }   cursor ;
  } *LOOK ;

#define MAGIC  837649

#define xScale 17

int sequenceLength (KEY seq) ;	/* global */

static void fMapDestroy (void) ;
/* static void fMapRecalculate (void) ; */
static void fMapPick (int box, double x , double y) ;
static void fMapMiddleDown (double x , double y) ;
static void fMapKbd (int k) ;
static void fMapDraw (LOOK look, KEY key) ;
static Array fMapConvert (KEY chrom, KEY fMap) ;     
static void fMapResize (void) ;
static void fMapSelectBox (LOOK look, int box) ;
static void fMapFollow (LOOK look, double x , double y) ;
static void fMapCentre (LOOK look, KEY key, KEY from) ;
static void fMapClearDNA (void) ;
static BOOL fMapFindDNA(LOOK look) ;
static void findUnit(LOOK look, float *u, float *sub) ;
static void localDnaAnalyze(void) ;
static void fMapSetLength(void) ;
static void fMapShiftOrigin(void) ;
static void fMapOriginButton(void) ;

extern void emblDump (void) ;
static MENUOPT fMapMenu[] =
            { graphDestroy, "Quit",
		help,"Help",
		graphPrint,"Print",
		localDnaAnalyze,"Analyze",
		displayPreserve,"Preserve",
/*		fMapRecalculate,"Recalculate", */
		fMapClearDNA, "Clear DNA",
		emblDump, "EMBL dump",
		0, 0
            } ;

static int minLiveBox ;
static int graphWidth, graphHeight , fMargin , topMargin = 4 ;
static int halfGraphHeight;
#define FMAP2GRAPH(look,x) (look->mag * ((x) - look->centre) + halfGraphHeight + topMargin)

typedef enum { MASTER, SEQUENCE_UP, SEQUENCE_DOWN, 
	       TITLE, cDNA, PROPERTY, TEXT, GENERAL, CDS,
	       EXON_UP, EXON_DOWN, INTRON_UP, INTRON_DOWN,
	       FEATURE_UP, FEATURE_DOWN, FEATURE_TAG,
	       TRANSPOSON_UP, TRANSPOSON_DOWN,
	       PEP_HOMOL, DNA_HOMOL, HOMOL_2 } Subtype ;

/* note that for seg's with the same x1, type determines order */

typedef struct
  { KEY key, parent ;
    Subtype type ;
    int x1,x2 ;
    float f ;
  } SEG ;
#define segFormat "2k3if"

#define FLAG_CLONED 0x01

#define STATUS_DNA 0x01
#define STATUS_PROTEIN 0x02
#define STATUS_COORDS 0x04
#define STATUS_TEXT_FEATURES 0x08
#define STATUS_GRAPH_FEATURES 0x10

BOOL fMapDisplay (KEY key, KEY from, BOOL isOldGraph) ;
#define LOOKGET(name)     LOOK look ; \
                          if (!graphAssFind (fMapDisplay,&look)) \
		            messcrash ("graph not found in %s",name) ; \
			  if (!look) \
                            messcrash ("%s received a null pointer",name) ; \
                          if (look->magic != MAGIC) \
                            messcrash ("%s received a wrong pointer",name)

/**********************************/
     /* Recalculate all fMaps */
void fMapMakeAll(void)
{
  KEY  fMapKey , seqKey = 0 ;	/* 0 primes lexNext() */
  Array segs ;

  if(!isWriteAccess())
    { messout("Sorry, you do not have Write Access");
      return ;
    }
  return ;  /* We dont want to save f map arrays */
  if(lexNext(_VSequence, &seqKey))
    while(lexNext(_VSequence, &seqKey))
      { lexaddkey(name(seqKey),&fMapKey, _VfMap) ;
	segs = fMapConvert (seqKey, fMapKey) ;
	arrayDestroy (segs) ;
      }
}

/***************************************************/
/******** Communication toolKit ********************/
static LOOK selectedfMap = 0 ; 

static void fMapUnselect (void)
{
  if (selectedfMap && selectedfMap->magic == MAGIC)
    { Graph hold = graphActive () ;
      graphActivate (selectedfMap->graph) ;
      graphBoxDraw (1,WHITE,WHITE) ;
      graphActivate (hold) ;
    }
  selectedfMap = 0 ;
}

/***************/

static void fMapSelect (LOOK look)
{
  if (selectedfMap && selectedfMap != look)
    fMapUnselect() ;
  selectedfMap = look ;
  graphBoxDraw (1,BLACK,LIGHTRED) ;
}

/***************/

BOOL fMapActive(Array *dnaArrayp, Array *proteinArrayp,
		Array *dnaColp, KEY *seqKeyp, int *frame, void** lookp)
{
  if (selectedfMap && selectedfMap->magic == MAGIC
      && graphExists(selectedfMap->graph)
      && fMapFindDNA(selectedfMap) )
    { *lookp = selectedfMap ;
      *seqKeyp = selectedfMap->seqKey ;
      *dnaArrayp = selectedfMap->dnaArray ;
      *proteinArrayp = selectedfMap->protein ;
      *dnaColp = selectedfMap->colors ;
      *frame = selectedfMap->frame ;
      return TRUE ;
    }

  selectedfMap = 0 ;
  *lookp = 0 ; 
  *seqKeyp = 0 ; *dnaArrayp = 0 ; *dnaColp = 0 ; *frame = 0 ;
  return FALSE ;
}

/***************************************************************/
/***************************************************************/

static void fMapDisplaySequenceProtein 
  			(LOOK look,BOOL isProtein, Array dna,
			 BOOL isRna, int margin, int width)
{  
  int i, j , ii , triple = isProtein ? 3 : 1 ,
      coor = look->showStatus & STATUS_COORDS ? 6 : 0 ;
  int nx, ny = graphHeight ,  colorBox , start ;
  int n = arrayMax(dna) , oldcolor = -1 , color ;
  int xOrigin = look->origin ;
  register char *cp ;
  register int *clp ;
  char letter[2] ;
  float subunit , y ;
  int frame = look->frame ;
  char *decodeChar = isRna ? rnaDecodeChar : dnaDecodeChar ;

/* subunit is the number of bases I will try to write on each line */
  subunit = .4/look->mag ;
  subunit = 3 * (int)subunit ;

  while(look->mag*subunit < 1.002)
    subunit += 3 ; /* so successive lines will not overlap */
    
 /* nx is the number of bases i will actually write */
  nx = ((int)(width/3))*3 ;  /* <= subunit */
  nx -= coor ;
  if(triple * nx < 3)
    return ;
  if(triple*nx > subunit)
    nx = subunit/triple ;
 

  start = look->centre - (halfGraphHeight-2)/look->mag ;
  if(start<0)
    start = 0 ;
  start = ((int)(start/subunit)) * subunit ;
  frame = (frame + 6) % 3  ; /* needed because -2 % 3 = -2 not 1 ! */
  start += frame ; 
  start /= triple ;
  
  letter[1] = 0 ;
  j = 3 ;

/*  sprintf(look->lengthBuffer, "%-10d",triple * arrayMax(dna)) ; */

  graphBoxStart() ;		/* background box */
  oldcolor = BLACK ;
  graphColor(BLACK) ;

  for(i = start ; 
      i<n && ((y = FMAP2GRAPH(look, triple*i + (isProtein ? frame + 1 : 0)) + .5) < ny-1);
      i += subunit/triple  )
    {  
      cp = arrp(dna, 0, char) + i ;
      clp = arrp(look->colors, 0, int ) + ( isProtein ? triple*i + frame + 1 : i) ;
      oldcolor = BLACK ;
      graphBoxStart() ;        /* line box */
      if(coor)    /* +1 since biologists ignore the existence of the zero */
	graphText(messprintf("%-6d ",i+1-(xOrigin/triple)),margin, y) ; 
	
      for(ii=0;ii<nx && i+ii < n ;ii++, cp++, clp+= triple)
	{
	  color = *clp ;
	  if (color != oldcolor)
	    { if (oldcolor != BLACK)
		{ graphBoxEnd() ;	/* color box */
		  graphBoxDraw(colorBox, BLACK, oldcolor) ;
		}
	      oldcolor = color ;
	      if (color != BLACK)
		colorBox = graphBoxStart() ;  /* color box */
	    }
	  *letter = isProtein ?
	    *cp :  decodeChar[*cp & 15] ;
	  graphText(letter, margin + coor + ii, y) ;
	}
      if (oldcolor != BLACK) 
	{ graphBoxEnd() ;	/* color box */
	  graphBoxDraw(colorBox, BLACK, oldcolor) ;
	}
      if(triple*nx < subunit)
	graphText("...", margin+coor+ii,y) ;
      graphBoxEnd() ;  /* line box */
    }
  graphBoxEnd() ;  /* background box */
} 

/***************/

static void fMapDisplaySequence(LOOK look)
{ 
  graphBoxClear(look->proteinBox) ;
  graphBoxClear(look->sequenceBox) ;
  switch (look->showStatus & (STATUS_DNA | STATUS_PROTEIN))
    {
    case 0:
      return ;
      break ;
    case STATUS_DNA:
      look->sequenceBox = graphBoxStart() ;		/* background box */
      fMapDisplaySequenceProtein(look, FALSE, look->dnaArray,look->isRna,
				 xScale + 10, look->dnaWidth) ;
      graphBoxEnd() ;  /* background box */
      graphBoxDraw(look->sequenceBox,BLACK,WHITE) ; 
      break ;
    case STATUS_PROTEIN:
      look->proteinBox = graphBoxStart() ;		/* background box */
      if(arrayMax(look->protein))
	fMapDisplaySequenceProtein(look, TRUE, look->protein,look->isRna,
				   xScale + 10, look->proteinWidth) ;
      graphBoxEnd() ;  /* background box */
      graphBoxDraw(look->sequenceBox,BLACK,WHITE) ; 
      break ;
    default:			/* both */
      look->sequenceBox = graphBoxStart() ;		/* background box */
      fMapDisplaySequenceProtein(look, FALSE, look->dnaArray,look->isRna,
				 xScale + 14 + look->proteinWidth, look->dnaWidth) ;
      if(arrayMax(look->protein))
       { look->proteinBox = graphBoxStart() ;		/* background box */
	 fMapDisplaySequenceProtein(look, TRUE, look->protein,look->isRna,
				    xScale + 10 , look->proteinWidth) ;
	 graphBoxEnd() ;  /* background box */
       }
      graphBoxEnd() ;  /* background box */
      graphBoxDraw(look->sequenceBox,BLACK,WHITE) ; 
      break ;
    }
}

/***************/
  /* to display the DNA as a yelllow rectangle with colored marks */
void fMapDisplayDNApicture (LOOK look)
{ 
  int start, end, i, i1, box, col, old , older ,ny = graphHeight ;
  float x1, x2 ;

  start = look->centre - (halfGraphHeight-2)/look->mag ;
  if(start<0) 
    start = 0 ;
  end = look->centre + (halfGraphHeight-2)/look->mag ;
  if (end > arrayMax(look->colors))
    end = arrayMax(look->colors) ;

  graphBoxClear(look->dnaBox) ;
  box = graphBoxStart() ;  /* see fMapDraw */

  x1 = FMAP2GRAPH(look, 0)  + .5 ;
  x2 = FMAP2GRAPH(look, end) + .5 ;
  if (x1 < 2 + topMargin)
    x1 = 2 + topMargin ;
  if (x2 > ny - 2)
    x2 = ny - 2 ;

  look->topDnaBox = x1 ;
  graphRectangle (xScale-3.2,x1,xScale-2.,x2) ;

  for (i = i1 = start , old = older = BLACK ; i <= end ; i++)
    { if (i < end)
	col = arr(look->colors,i,int) ;
      else 
	col = BLACK ;
      if (col != old)
	{ if (old != BLACK)
	    { if (old != older)
		{ older = old ;
		  switch (old)
		    { 
		    case LIGHTRED: graphColor(RED) ; break ;
		    case LIGHTBLUE: graphColor(BLUE) ; break ;
		    case LIGHTGREEN: graphColor(GREEN) ; break ;
		    default: graphColor(old) ;
		    }
		}
	      x1 = FMAP2GRAPH(look, i1) + .5 ;
	      x2 = FMAP2GRAPH(look, i + 1) + .5 ;
	      graphFillRectangle(xScale - 3.2, x1 ,
				 xScale - 2., x2 ) ;
	    }
	  old = col ;
	  i1 = i ;
	}
    }

  x1 = FMAP2GRAPH(look, 0)  + .5 ;
  if (x1 < 2 + topMargin)
    x1 = 2 + topMargin ;
  x2 = FMAP2GRAPH(look, look->origin) + .5 ;
  if (x1 < x2)
    { graphColor(BLUE) ;
      graphFillRectangle(xScale - 3.2, x1 ,
			 xScale - 2.7, x2 ) ;
    }
 
  x1 = FMAP2GRAPH(look, look->length + look->origin) + .5 ;
  x2 = FMAP2GRAPH(look, end) + .5 ;
  if (x2 > ny - 2)
    x2 = ny - 2 ;
  if (x1 < x2)
    { graphColor(BLUE) ;
      graphFillRectangle(xScale - 3.2, x1 ,
			 xScale - 2.7, x2 ) ;
    }
	
  graphBoxEnd() ;
  look->dnaBox = box ;
  graphBoxDraw(box,BLACK,YELLOW) ;
}

/***************/

static Array fMapRecursiveFindDNA(KEY seq, LOOK look)
{  
  OBJ obj ;
  Array dnaArray = 0 ;
  Array aa = 0, suba = 0 ;
  KEY subSeq ;
  int pos1, pos2, i, n ;
  char *cp, *cq ;
 
  if(!seq || 
     !(obj = bsCreate (seq)))
    return 0 ;

  if(bsFindTag(obj,_RNA))
    look->isRna = TRUE ;
  
      /* seq itself has a DNA sequence */
  if(bsGetKey (obj,_DNA,&(look->dnaKey)) )
    { bsDestroy(obj) ;
      return dnaGet(look->dnaKey) ;
    }
 
  aa = arrayCreate(12, BSunit) ;

   /* else Recontruct from subsequences */

  if(! bsFindTag(obj,_Contains) ||
     ! bsFlatten(obj, 4, aa))
    goto abort ;
  bsDestroy(obj) ;

  dnaArray = arrayCreate(1000, char) ;

  for (i=0; i < arrayMax(aa); i+= 4)
    { pos1 = arr(aa,i+2,BSunit).i ;
      pos2 = arr(aa,i+3,BSunit).i ;
      subSeq = arr(aa,i+1,BSunit).k ;
      if (pos1 && pos2 && 
	  (suba = fMapRecursiveFindDNA(subSeq, look)))
	{ if (pos1 <= pos2)
	    { n = pos2 - pos1 + 1 ;
	      if (arrayMax(suba) != n)
		messout ("Subsequence length error for %s", 
			 name(subSeq)) ;
	      else
		{ 
		  cp = arrayp(dnaArray, pos2-1, char) ;
		  cq = arrp(suba, n-1, char) ;
		  while (n--)
		    { if (*cp && *cp != *cq)
			{ messout
  ("Overlapping subsequence %s of %s does  not match aanother one",
    name(subSeq),  name(seq)) ;
			  goto abort ;
			}
		      else
			*cp-- = *cq-- ;
		    }
		}
	    }
	  else
	    { n = pos1 - pos2 + 1 ;
	      if (arrayMax(suba) != n)
		messout ("Subsequence length error for %s", 
			 name(subSeq)) ;
	      else
		{ cp = arrayp(dnaArray, pos1-1, char) ; /* Make room */
		  cq = arrp(suba, 0, char) ;
		  while (n--)
		    { if (*cp && *cp != *cq)
			{ messout
  ("Overlapping subsequence %s of %s does  not match aanother one",
    name(subSeq),  name(seq)) ;
			  goto abort ;
			}
		      else
			*cp-- = *cq++ ;
		    }
		}
	    }
	  arrayDestroy (suba) ;
	}
    }
  arrayDestroy(aa) ;		 
  return dnaArray ;

 abort:
  arrayDestroy(aa) ;
  arrayDestroy(suba) ;
  bsDestroy(obj) ;
  return dnaArray ;
}

/***************/

static BOOL fMapFindDNA(LOOK look)
{ int *ip, i ;
  if(arrayExists(look->dnaArray))
    return TRUE ;
 
  arrayDestroy(look->dnaArray) ;
  arrayDestroy(look->protein) ;
  arrayDestroy(look->colors) ;
  
  if(look->dnaArray  = fMapRecursiveFindDNA(look->seqKey, look))
    { i = arrayMax(look->dnaArray) ;
      look->colors = arrayCreate(i,int) ;
      look->protein = arrayCreate(i/3 + 2,char) ;
      look->origin = 0 ;
      look->length = arrayMax(look->dnaArray) ;
      arrayMax(look->colors) = i ;
      ip = arrp(look->colors,0,int) ;
      while(i--) 
	*ip++ = BLACK ;
      return TRUE ;
    }
  return FALSE ;
}

/**********************************/

BOOL fMapDisplay (KEY key, KEY from, BOOL isOldGraph)
{
  LOOK look ;
  KEY  fMapkey, seq = 0, parent = 0 ;
  OBJ  obj, Seq ;
 
  if (class(key) == _VSequence)
    seq = key ;
  else if (class(key) == _VfMap)
    lexword2key(name(key),&seq,_VSequence) ;
  else if (class(key) == _VDNA)
    lexword2key(name(key),&seq,_VSequence) ;
  else if (obj = bsCreate (key))
    { bsGetKey (obj, _Sequence, &seq) ;
      bsDestroy (obj) ;
    }

  if (!(seq && (Seq = bsCreate(seq))))
    return FALSE ;

  while (bsGetKey (Seq, _Source, &parent)) /* recurse up */
    { bsDestroy (Seq) ;
      if (!(Seq = bsCreate(parent)))
	break ;
      seq = parent ;
    }
  if (Seq)
    bsDestroy (Seq) ;

  if(!sequenceLength(seq))  /* In this case the drawing would be meaningless */
    { display(seq,0,TREE) ;
      return FALSE ;
    }

  look=(LOOK)messalloc(sizeof(struct LOOKSTUFF)) ;
  look->magic = MAGIC ;
  look->seqKey = seq ;

  lexaddkey (name(seq), &fMapkey, _VfMap) ;
/*
  if (!(look->segs = arrayGet (fMapkey, SEG, segFormat)))
*/
  if(!(look->segs = fMapConvert (seq, fMapkey)))
    goto abort ;
     

  look->mapKey = fMapkey ;
  look->boxIndex = arrayCreate (64,int) ;
  look->activeBox = 0 ;
  look->showStatus = STATUS_TEXT_FEATURES ;  /* Show features */
  *look->homolText = 0 ;

  if (isOldGraph)
    { graphRetitle (name (key)) ;
      fMapDestroy () ;
      graphClear () ;
      graphGoto (0,0) ;
      graphAssRemove (fMapDisplay) ;
    }
  else 
    { look->graph = displayCreate (FMAP) ;
      if (!look->graph)
	goto abort ;

      graphRetitle (name(key)) ;
      graphHelp ("Feature-map") ;
      graphRegister (RESIZE, (GraphFunc)fMapResize) ;
      graphRegister (DESTROY, fMapDestroy) ;
      graphRegister (PICK,(GraphFunc) fMapPick) ;
      graphRegister (MIDDLE_DOWN,(GraphFunc) fMapMiddleDown) ;
      graphRegister (KEYBOARD, (GraphFunc)fMapKbd) ;
      graphRegister (MESSAGE_DESTROY, displayUnBlock) ;
      graphMenu (fMapMenu) ;
    }

  graphAssociate (fMapDisplay,look) ;
  look->graph = graphActive() ;

  fMapCentre(look, key, from) ;	/* centre on key if possible and choose magnification */

  look->showStatus = STATUS_TEXT_FEATURES ;

  fMapDraw (look, key) ;
  fMapSelect(look) ; 

  return TRUE ;

abort :
  messfree (look) ;
  return FALSE ;
}

/************************************************************/
/***************** Registered rourtines *********************/

static void fMapDestroy (void)
{
  LOOKGET(fMapDestroy) ;

  arrayDestroy(look->segs) ;
  arrayDestroy(look->boxIndex) ;
  arrayDestroy(look->colors) ;
  arrayDestroy(look->dnaArray) ;
  arrayDestroy(look->protein) ;

  if(look == selectedfMap)
    selectedfMap = 0 ;
  look->magic = 0 ;
  messfree (look) ;
}

/*************************************************************/
/*
static void fMapRecalculate(void)
{
  KEY curr = 0 ;
  LOOKGET("fMapRecalculate") ;
  
  if (look->seqKey)
    { if (look->activeBox)
	curr =  arrp(look->segs,
	 arr(look->boxIndex,look->activeBox,int),SEG)->parent ;
      if (!class(curr))
	curr = 0 ;	// avoids key being a tag
      arrayDestroy (look->segs) ;
      if (look->segs = fMapConvert (look->seqKey, look->mapKey))
	fMapDraw (look, curr) ;
      else
	{ messout ("Sorry, I have to abandon this fMap") ;
	  graphDestroy () ;
	}
    }
}
*/
/******** coordinated dragging of pos and scale boxes **********/

static int posBox,scaleBox ;
static LOOK lookDrag ;

static void scaleBoxDrag (float *x, float *y, BOOL isDone)
{
  static BOOL isDragging = FALSE ;
  static float newCentre, yCentre, oldX ;

  if (isDragging)
    graphXorBox (posBox,xScale-0.5,yCentre) ;
  else
    { oldX = *x ;
      isDragging = TRUE ;
    }

  *x = oldX ;

  newCentre = lookDrag->cursor.offset + 
    		(*y - 3 - topMargin) / lookDrag->cursor.fac ;
  yCentre = FMAP2GRAPH(lookDrag,newCentre) ;

  if (isDone)
    { lookDrag->centre = newCentre ;
      fMapDraw (lookDrag, 0) ;
      isDragging = FALSE ;
    }
  else
    graphXorBox (posBox,xScale-0.5,yCentre) ;
}

static void posBoxDrag (float *x, float *y, BOOL isDone)
{
  static BOOL isDragging = FALSE ;
  static float newCentre,yScaleBox ;

  if (isDragging)
    graphXorBox (scaleBox,0,yScaleBox) ;
  else
    isDragging = TRUE ;

  *x = xScale-0.5 ;

  newCentre = lookDrag->centre + 
    (*y - halfGraphHeight - topMargin) / lookDrag->mag ;
  yScaleBox = 3 + topMargin + 
    (newCentre - lookDrag->cursor.offset) * lookDrag->cursor.fac ;
  if (isDone)
    { lookDrag->centre = newCentre ;
      fMapDraw (lookDrag, 0) ;
      isDragging = FALSE ;
    }
  else
    graphXorBox (scaleBox,0,yScaleBox) ;
}

static void makeDragBoxes (void)
{
 float x1,x2,y1,y2 ;
  int ny21 = halfGraphHeight + topMargin ;

  graphBoxDraw (posBox, WHITE, WHITE) ;
  posBox = graphBoxStart() ;
  graphLine (xScale-0.5,ny21 ,xScale-0.5,ny21+1) ;
  graphLine (200,ny21,200,ny21+1) ;	/* to make box wide */
  graphBoxEnd () ;
  graphBoxDraw (posBox, BLACK, TRANSPARENT) ;

  graphBoxDraw (scaleBox, WHITE, WHITE) ;
  graphBoxDim (scaleBox, &x1, &y1, &x2, &y2) ;
  scaleBox = graphBoxStart() ;
  graphFillRectangle (5.25, y1, 5.75, y2) ;
  graphLine (0, y1, 0, y2) ;	/* to make box wide */
  graphBoxEnd () ;
  graphBoxDraw (scaleBox, BLACK, TRANSPARENT) ;
}

/***********************************/

static void fMapPick(int box, double x, double y)
{
  KEY key ;
  LOOKGET("fMapPick") ;
  
  if (look != selectedfMap)
    fMapSelect(look) ; 
  if (!box)
    return ;
  if (box == look->dnaBox)
    box = 4 ;
  if (box >= minLiveBox) 
    if (box == look->activeBox)         /* a second hit - follow it */
      fMapFollow (look,x,y) ;
    else
      fMapSelectBox (look,box) ;
  else  if (box == look->originBox)
    graphTextEntry (look->originBuffer,0,0,0,0) ;
  else  if (box == look->lengthBox)
    graphTextEntry (look->lengthBuffer,0,0,0,0) ;
  else switch (box)		/* a control box */
    {
    case 2 :
      lookDrag = look ;
      makeDragBoxes () ;
      graphBoxDrag (posBox,posBoxDrag) ;
      break ;
    case 3 :
      lookDrag = look ;
      makeDragBoxes () ;
      graphBoxDrag (scaleBox,scaleBoxDrag) ;
      break ;
    case 4:   /* DNA box */
      look->position = (y - .5 + look->topDnaBox - halfGraphHeight - topMargin) / look->mag 
	+ look->centre - look->origin + 1 ;
      localDnaAnalyze() ;
      break ;
    case 5:
      display(look->seqKey,0,TREE) ;
      break ;
    case 6:
      freeforcecard (look->homolText) ;
      if (freecheck ("wii") && 
	  lexword2key(freeword(),&key,_VSequence))
	{ int pos1, pos2 ;
	  freeint (&pos1) ; freeint (&pos2) ;
	  display (key, KEYMAKE(_VCalcul, (pos1+pos2)/20), 0) ;
	}
    }
}

/********* use middle button for cursor **********/

static double oldy , oldDy , oldx;
static BOOL dragFast ;
#define DRAGFASTLIMIT 6

static void fMapMiddleDrag (double x, double y) 
{
  if(dragFast)
    { graphXorLine (0, oldy - oldDy, DRAGFASTLIMIT, oldy - oldDy) ;
      graphXorLine (0, oldy + oldDy, DRAGFASTLIMIT, oldy + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, oldy, graphWidth, oldy) ;

  oldy = y ;

  if(dragFast)
    { oldDy *= exp ((x - oldx) / 25.) ;
      oldx = x ;
      graphXorLine (0, y - oldDy, DRAGFASTLIMIT, y - oldDy) ;
      graphXorLine (0, y + oldDy, DRAGFASTLIMIT, y + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, y, graphWidth, y) ;
}

static void fMapMiddleUp (double x, double y) 
{  float x1,x2,y1,y2 ;
  LOOKGET("fMapMiddleUp") ;

  if(dragFast)
    { graphBoxDim (scaleBox, &x1, &y1, &x2, &y2) ;
      look->mag *= (y2 - y1) / (2. * oldDy) ;
      look->centre = look->cursor.offset + 
	(y - (y2 - y1) * .5 - 3.05 - topMargin) / look->cursor.fac ;
    }
  else
   look->centre = look->centre + (y - .5  - halfGraphHeight - topMargin) / look->mag ;
  fMapDraw (look, 0) ;
}

static void fMapMiddleDown (double x, double y) 
{  float x1,x2,y1,y2 ;
  LOOKGET("fMapMiddleDown") ;


  graphBoxDim (scaleBox, &x1, &y1, &x2, &y2) ;
  oldDy = (y2 - y1) / 2. ;
  
  lookDrag = look ;
  dragFast = x < DRAGFASTLIMIT ? TRUE : FALSE ;

  if(dragFast)
    { graphXorLine (0, y - oldDy, DRAGFASTLIMIT, y - oldDy) ;
      graphXorLine (0, y + oldDy, DRAGFASTLIMIT, y + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, y, graphWidth, y) ;
   
  oldx = x ;
  oldy = y ;
  graphRegister (MIDDLE_DRAG,(GraphFunc) fMapMiddleDrag) ;	/* must redo */
  graphRegister (MIDDLE_UP, (GraphFunc)fMapMiddleUp) ;
}

/***********************************/

static void fMapKbd (int k)
{
  int  box ;
  LOOKGET(fMapKbd) ;

   if(look != selectedfMap)
    fMapSelect(look) ; 
  if (!look->activeBox)
    return ;

  box = look->activeBox ;
  switch (k)
    {
    case UP_KEY :
      if (box > minLiveBox)
	--box ;
      break ;
    case DOWN_KEY :
      if (box < arrayMax(look->boxIndex) - 1)
	++box ;
      break ;
    }
  fMapSelectBox (look,box) ;
  if (look->activeBox != box)
    fMapFollow (look,0.,0.) ;
}

/**********************************/

static void fMapResize (void)
{
  LOOKGET(fMapResize) ;

  fMapDraw (look, 0) ;
}


/*****************************************************************/
/****************** Drawing routines *****************************/

/************* first a mini package for bumping text *************/

	/* Copied from gmapdisp.c
	*/

typedef struct bumpStruct
  { int n ;
    float *bottom ;
    float curr ;
  } *BUMP ;

static BUMP bumpCreate (int ncol)
{
  int i ;
  BUMP bump = (BUMP) messalloc (sizeof (struct bumpStruct)) ;

  bump->n = ncol ;
  bump->bottom = (float*) messalloc (ncol*sizeof (float)) ;
  for (i = 0 ; i < ncol ; ++i)
    bump->bottom[i] = -100000000.0 ;
  bump->curr = 2 + topMargin ;
  return bump ;
}

static void bumpDestroy (BUMP bump)
{
  messfree (bump->bottom) ;
  messfree (bump) ;
}

static void bumpWrite (BUMP bump, char *text, int x, int y)
{
  int i,j ;
  int len = strlen(text) ;
  LOOKGET(bumpWrite) ;
  
  y = FMAP2GRAPH(look,y) ;
  if (y < bump->curr)
    y = bump->curr ;

  len += 1 ;
  if (x+len > bump->n)
    x -= 10 ;
  if (x < 0)
    x = 0 ;
  if (x+len > bump->n)
    len = bump->n - x ;

  while (TRUE)
    { for (i = x, j = len ; i < bump->n ; ++i)	/* always start at x */
	{ if (bump->bottom[i] > y)
	    j = len ;
	  else if (!--j)		/* have found a gap */
	    break ;
	}
      if (!j)
	{ graphText (text, fMargin + i-len, y) ;
	  for (j = 0 ; j < len ; j++)	/* advance bump */
	    bump->bottom[i-j] = y+1 ;
	  bump->curr = y ;
	  break ;			/* done */
	}
      y += 1 ;	/* try next row down */
    }
}

/**************************************************/

static void findUnit(LOOK look, float *u, float *sub)
{
  float cutoff = 5 / look->mag ;
  float unit = 1 ;
  float subunit  ;

  while(unit*10 > cutoff)
    unit /= 10. ;
  subunit = unit/10 ;
  while (unit < cutoff)
    { unit *= 2 ;
      subunit *= 5 ;
      if (unit >= cutoff)
	break ;
      unit *= 2.5 ;
      if (unit >= cutoff)
	break ;
      unit *= 2 ;
      subunit *= 2 ;
    }
  *u = unit ; *sub = subunit ;
}

/**************************************************/

static void drawScale (LOOK look)
{
  float unit, subunit ;
  float x,y , x0 = look->origin ;
  int ny = graphHeight ;

  findUnit(look, &unit, &subunit) ;

  x = look->centre - (halfGraphHeight-2)/look->mag - x0 ;
  x = unit * (((x>=0)?1:0) + (int)(x/unit)) ;
  while ((y = FMAP2GRAPH(look, x + x0)) < ny  - 1)
    { graphLine (xScale-1.5,y+0.4,xScale-0.5,y+0.4) ;
      graphText (messprintf ("%d",(int)(x+0.5)),xScale,y) ;
      x += unit ;
    }

  x = look->centre - (halfGraphHeight-2)/look->mag - x0 ;
  x = subunit * (((x>=0)?1:0) + (int)(x/subunit)) ;
  while ((y = FMAP2GRAPH(look, x + x0)) < ny  - 1)
    { graphLine (xScale-1.0,y+0.4,xScale-0.5,y+0.4) ;
      x += subunit ;
    }

  graphLine (xScale-0.5,2 + topMargin , xScale-0.5, ny-0.5) ;

  array(look->boxIndex,posBox=graphBoxStart(),SEG*) = (SEG*)5 ;
  graphFillRectangle (xScale-1,halfGraphHeight + topMargin,
		      xScale,halfGraphHeight+1 + topMargin) ;
  graphBoxEnd () ;
  graphBoxDraw (posBox,DARKGREEN,TRANSPARENT) ;
}

static void drawChromosome (LOOK look)
{
  float y,pageful,x0, xx ;
  int i ;
  int ny = graphHeight ;
  SEG *seg = arrp(look->segs,0,SEG) ;

  x0 = seg->x1  ;
         /* Draw complete black Line */
  xx = (seg->x2 - seg->x1)  ;
  if(xx<=0) xx = 1. ;

  look->cursor.fac = (ny - topMargin - 6.0) / xx ;
  graphFillRectangle (5.25,3+topMargin,5.75,ny-3) ;

/*  array(look->boxIndex,scaleBox=graphBoxStart(),SEG*) = (SEG*)4 ; */


        /* Draw Green scaleBox */
  pageful = (ny - topMargin - 3) / look->mag ;	
  look->cursor.offset = x0 + 0.5*pageful ;
  y = 3 + topMargin + (look->centre - look->cursor.offset) * look->cursor.fac ;

  scaleBox=graphBoxStart() ;
  graphRectangle (5, y, 6, y + pageful*look->cursor.fac) ;
  graphBoxEnd () ;
  graphBoxDraw (scaleBox,DARKGREEN,GREEN) ;

       /* Draw oblique line outside of boxes */ 
  graphColor (DARKGRAY) ;
  graphLine (5.5, y, xScale - 1, 2.5 + topMargin) ;
  graphLine (5.5, y + pageful*look->cursor.fac, xScale-1, ny-1) ;

  graphColor (BLACK) ;

     /* Position main Markers */
  graphTextHeight (0.75) ;
  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;
      if (class(seg->key) == _VGene)
	{ y = 2.96 + topMargin + (seg->x1 + 2*seg->x2 - 3*x0) * look->cursor.fac / 3.  ;
	  graphText (name(seg->key),0.5,y) ;
	}
    }
  graphTextHeight (0) ;
}

/*****************************************************************/
/*****************************************************************/
/****** magnification control ******/

static void wholeSequence (void)
{ float  su ;
  float x0, xdiff ;
   LOOKGET("wholeSequence") ; 

  graphFitBounds (&graphWidth,&graphHeight) ;
  halfGraphHeight = 0.5*(graphHeight - topMargin) ;
  
  x0 = arr(look->segs,0,SEG).x2 +  arr(look->segs,0,SEG).x1 ;
  xdiff = 1.05 * arr(look->segs,0,SEG).x2 -  arr(look->segs,0,SEG).x1 ;

  look->centre = x0 / 2 ;  
  look->mag = (graphHeight- topMargin - 5)/xdiff ;

  su = .4/look->mag ;
  su = 3 * (int)su ;
  if(su<3)
    su = 3 ;
/*
  if (su>50)
    look->showStatus = STATUS_TEXT_FEATURES ;
*/
  fMapDraw (look, 0) ;
}

static void zoomIn (void)
{ float su ;
  LOOKGET("zoomIn") ;

  su = .4/look->mag ;
  su = 3 * (int)su ;
/*
  if(su<24 && fMapFindDNA(look))
    { look->showStatus |= (STATUS_DNA | STATUS_PROTEIN) ;
      localDnaAnalyze() ;
    }
*/
  if(su >1)
    look->mag *= 2 ;

  if(su<3)
    su = 3 ;

  fMapDraw (look, 0) ;
}

static void zoomOut (void)
{ float su ;
  LOOKGET("zoomOut") ; 

  look->mag /= 2 ; 

  su = .4/look->mag ;
  su = 3 * (int)su ;
  if(su<3)
    su = 3 ;
/*
  if(su>50)
    look->showStatus = STATUS_TEXT_FEATURES ;
*/
  fMapDraw (look, 0) ;
}

/*
static void pageUp (void)
{
  int ny = graphHeight ;
  LOOKGET("pageUp") ; 
  look->centre -= (ny*0.8)/look->mag ; fMapDraw (look, 0) ;
 }

static void pageDown (void)
{ int ny = graphHeight ;
  LOOKGET("pageDown") ; 
  look->centre += (ny*0.8)/look->mag ; fMapDraw (look, 0) ;
}
*/
/*************************************/
extern void   dnaAnalyse(void) ;
static void localDnaAnalyze(void)
{
  LOOKGET("localDnaAnalyse") ;

  fMapSelect(look) ;
  if(!fMapFindDNA(look))
    messout("Sorry, the DNA sequence is not in this database") ;
  else
    dnaAnalyse() ; 
}

/*************************************/

static void fMapToggleDna(void)
{ int status ;
  LOOKGET("fMapToggleDna") ;
  
  status = look->showStatus ;
  if (status & STATUS_DNA)
    { look->showStatus ^= STATUS_DNA ;
      fMapDraw(look,0) ;
    }
  else
    if (!fMapFindDNA(look))
      {
/*
	 if(look->seqKey)
	    display(look->seqKey, 0, TREE) ;
*/
	messout ("Sorry, the DNA sequence is not in this database") ;
      }
    else
      { look->showStatus ^= STATUS_DNA ;
	fMapDraw(look,0) ;
      }
}

/*************************************/

static void fMapToggleProtein(void)
{ int status ;
  LOOKGET("fMapToggleProtein") ;
  
  
  status = look->showStatus ;
  if (status & STATUS_PROTEIN)
    { look->showStatus ^= STATUS_PROTEIN ;
      fMapDraw(look,0) ;
    }
  else
    if (!fMapFindDNA(look))
      messout ("Sorry, the sequence is not in this database") ;
    else
      { look->showStatus ^= STATUS_PROTEIN ;
	dnaAnalyse() ;  /* needed to set up dnacpt->frame ! */
	dnacptTranslate2(look->dnaArray, look->protein, 
			look->colors, look->frame) ; /* does display */
      }
}

/*************************************/

static void fMapToggleTextFeatures(void)
{
  LOOKGET("fMapToggleTextFeatures") ;

  look->showStatus ^= STATUS_TEXT_FEATURES ;
  fMapDraw(look,0) ;
}

/*************************************/

static void fMapToggleCoordinates(void)
{
  LOOKGET("fMapToggleCoordinates") ;

  look->showStatus ^= STATUS_COORDS ;
  fMapDraw(look,0) ;
}

/**************************************/

static MENUOPT buttonOpts[] = {
  wholeSequence, "Whole",
  zoomIn, "Zoom In",
  zoomOut, "Zoom Out",
  fMapToggleTextFeatures, "Features",
  fMapToggleDna, "DNA",
  fMapToggleProtein, "Translation",
  fMapToggleCoordinates, "Coordinates",
  localDnaAnalyze,"Analyze",
  0, 0} ;

/**************************************/

static int fMapOrder(void *a, void *b)     
{
  SEG *seg1 = (SEG*)a, *seg2 = (SEG*)b ;
  int diff ;

  if(seg1->type == MASTER)
    if(seg2->type == MASTER)
      return 0 ;
    else
      return -1 ;

  if(seg2->type == MASTER)
    return 1 ;

  diff = seg1->x1 - seg2->x1 ;
  if (diff > 0)
    return 1 ;
  else if (diff < 0)
    return -1 ;


  diff = seg1->type - seg2->type ;
  if (diff > 0)
    return 1 ;
  else if (diff < 0)
    return -1 ;

  return 0 ;
}

/*****************************************************************/

int sequenceLength (KEY seq)
{
  int len = 0 ;
  OBJ obj ;
  LOOK look ;

  look=(LOOK)messalloc(sizeof(struct LOOKSTUFF)) ;
  look->magic = MAGIC ;
  look->seqKey = seq ;
 
  if (fMapFindDNA(look))
    { len = arrayMax(look->dnaArray) ;

      if(isWriteAccess() &&
	 (obj =bsUpdate(seq) ) )
	{ bsAddData(obj,_Length,_Int,&len) ;
	  bsSave(obj) ;
	}

      arrayDestroy (look->dnaArray) ;
    }
  messfree(look) ;

  return len ;
}

/*****************************************************************/

static void fMapCentre (LOOK look, KEY key, KEY from)
		/* need key and from for Calcul positioning */
{ 
  int i, nx, ny ;
  SEG *seg ;
  
  graphFitBounds (&graphWidth,&graphHeight) ;
  halfGraphHeight = 0.5*(graphHeight - topMargin) ;
 
 
  nx = graphWidth ; ny = graphHeight - topMargin - 5 ;

  if (key)
    for (i = 0 ; i < arrayMax (look->segs) ; ++i)
      { seg = arrp (look->segs, i, SEG) ;
	if (seg->type == SEQUENCE_UP ||
	    seg->type == SEQUENCE_DOWN)
	  { if (seg->key == key)
	      break ;
	    if (from && seg->key == from)
	      break ;
	  }
      }
  else 
    i = arrayMax(look->segs) ;
  

  if (i == arrayMax(look->segs))
    { seg = arrp (look->segs, 0, SEG) ;
      look->centre = 0.5 * (seg->x1 + seg->x2) ;
      look->mag = ny/(seg->x2 - seg->x1 + 1.0) ;
    }
  else if (class(from) == _VCalcul)
    { i = KEYKEY(from)*10 ;
      if(i < seg->x2 - seg->x1 + 1)
	{ 
	  if (seg->type == SEQUENCE_UP)
	    look->centre = seg->x2 - i ;
	  else
	    look->centre = seg->x1 + i ;
	}
      else
	look->centre = 0.5 * (seg->x1 + seg->x2) ;
      look->mag = ny/(seg->x2 - seg->x1 + 1.0) ;

       /* Go to rather high magnification 
      look->mag = (ny-5)/ 5000.0 ; */
    }
  else
    { look->centre = 0.5 * (seg->x1 + seg->x2) ;
      look->mag = ny/(seg->x2 - seg->x1 + 1.0) ;
      if (i > 1)
	look->mag *= 0.5 ;
    }

  if (look->mag <= 0)  /* safeguard */
    look->mag = 0.05 ;
}

/***************/
#define INTRON_COLOR YELLOW
#define EXON_COLOR LIGHTGREEN

   /* Splicing subroutine */
Array  fMapGetMessage (void *vv, BOOL *positiveStrand)
{ LOOK look = (LOOK) vv ;
  int   i, j, jCode = 0, cds1=0, cds2=0 ;
  SEG   *seg ;
  KEY parent ;
  Array dna = look->dnaArray , cDna ;
 
  if (look && look->magic != MAGIC)
    messcrash ("fMapGetMessage called with corrupted handle") ;
  
  seg = arrp(look->segs, arr(look->boxIndex,look->activeBox,int), SEG) ;
  if (!seg)
    messcrash ("Richard owes Jean a pint") ;

  parent = seg->parent ;
  
  if (!iskey(parent) || class(seg->parent) != _VSequence)
    { messout ("Please pick a sequence first") ;
      return 0 ;
    }
  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;
      if (seg->parent == parent)
	switch (seg->type)
	  {
	  case CDS:
	    cds1 = seg->x1 - 1 ; /* Plato strikes again */
	    cds2 = seg->x2 ;
	    break ;
	  case SEQUENCE_DOWN:
	    *positiveStrand = TRUE ;
	    break ;
	  case SEQUENCE_UP:
	    *positiveStrand = FALSE ;
	    break ;
	  default:
	    break;
	  }
    }
  if (cds1 == cds2)
    { messout ("Sorry, no coding sequence in %s", name(parent)) ;
      return 0 ;
    }

  cDna = arrayCreate(10000, char) ; /* wild guess */
  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;
      if (seg->parent == parent)
      switch (seg->type)
	{			/* pick cDna */
	case EXON_DOWN: case EXON_UP:
	  for (j = seg->x1 - 1 ; j < seg->x2 ; ++j)
	    if (j >= cds1 && j < cds2)
	      array(cDna,jCode++,char) = arr(dna,j,char) ;
	  break ;
	default:
	  break;
	}
      if (j - 1 > arrayMax(look->colors))
	messcrash ("Length over flow %d > %d in fMapcDna",
		   j, arrayMax(look->colors)) ;
    }

  return cDna ;
}

/********************/

void fMapIntronsExons (void *vv, Array prot)
		/* prot should be the protein sequence of the seq corresponding
		   to the activebox, reversed if UP.  This routine recalculates
		   Arrays look->colors and look->protein, then redraws */
{ 
  LOOK	look = (LOOK) vv ;
  int   i, j, jProt = 0, jCode = 0, colour, *ip ;
  int	length, cds1, cds2 ;
  char	*cp, aa ; 
  Array protein ;
  Array colors = look->colors ;
  SEG   *seg ;
  KEY 	parent ;
  BOOL  isUp = FALSE ;

  if (!look || look->magic != MAGIC ||
      !look->dnaArray || !look->colors)
    return ;
  
  if (look && look->magic != MAGIC)
    messcrash("fMapIntronsExons called with corrupted handle") ;
  if (look && !graphActivate (look->graph))
    { messout ("fMapIntronsExons lost its graph") ;
      return ;
    }

  graphPop() ;

  seg = arrp(look->segs, arr(look->boxIndex,look->activeBox,int), SEG) ;
  parent = seg->parent ;

  for (i = arrayMax(colors), ip = arrp(colors,0,int) ; i-- ;)
    *ip++ = BLACK ;

  protein = look->protein = arrayReCreate(look->protein, 
					  arrayMax(look->dnaArray)/3 + 3, char) ;
  arrayMax(protein) = arrayMax(look->dnaArray)/3 + 1 ;
  for (i = arrayMax(protein), cp = arrp(protein,0,char) ; i-- ;)
    *cp++ = ' ' ;

  length = 0 ;
  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;
      if (seg->parent == parent) switch (seg->type)
	{
	case EXON_UP:
	  isUp = TRUE ;
	case EXON_DOWN:
	  length += seg->x2 - seg->x1 + 1 ;
	case INTRON_UP: case INTRON_DOWN:
          if (seg->x2 > arrayMax(colors))
	    messcrash ("Length overflow  %d > %d in fMapIntronsExons",
		       seg->x2,  arrayMax(colors)) ;
	  break ;
	case CDS:
	  cds1 = seg->x1 - 1 ;
	  cds2 = seg->x2 ;
	  break ;
	default:
	  break;
	}
    }
  if (length % 3)
    { messout ("Coding length is %d mod 3", length % 3) ;
      if (isUp)
	jCode = -(length % 3) ;
    }
  
  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;
      if (seg->parent == parent) switch (seg->type)
	{        /* chose translation frame */
	case EXON_DOWN: case EXON_UP:
	  colour = MAGENTA ;
	  for (j = seg->x1 - 1 ; j < seg->x2 ; ++j)
	    if (j >= cds1 && j < cds2)
	      { switch (jCode % 3)
		  {
		  case 0:
		    if (jProt < arrayMax(prot)) 
		      aa = arr(prot,jProt++,char) ;
		    if (aa == '*')
		      colour = RED ;
		    else
		      colour = LIGHTGREEN ;
		    break ;
		  case 1: 
		    arr(protein,j/3,char) = aa ; /* preserved over introns */
		    break ;
		  }
		arr(colors,j,int) = colour ;
		++jCode ;
	      }
	  for (j = jCode % 3 ; j ; --j)
	    arr(colors,seg->x2-j,int) = MAGENTA ;
	  break ;
	case INTRON_DOWN: case INTRON_UP:
	  for (j = seg->x1 - 1 ; j < seg->x2 ; ++j)
	    if (j >= cds1 && j < cds2)
	      { arr(colors,j,int) = YELLOW ;
		if (arr(protein,(j+1)/3,char) == ' ' )
		  arr(protein,(j+1)/3,char) = '-' ;
	      }
	  break ;
	default:
	  break;
	}
    }
  fMapDraw (look,0) ;
}

/*************************************************************************/
/*************************************************************************/

static void fMapShiftOrigin(void)
{
  KEY key ;
  int i,x0 ;
  LOOKGET("fMapShiftOrigin") ;

  fMapSelect(look) ;

  if (lexword2key (look->originBuffer, &key, _VSequence))
    { for (i = 0 ; i < arrayMax(look->segs) ; ++i)
	if (arrp(look->segs,i,SEG)->key == key)
	  { look->origin = arrp(look->segs,i,SEG)->x1 - 1 ;
	    goto done ;
	  }
      messout ("Sorry, sequence %s is not being displayed in this graph",
	       name(key)) ;
      return ;
    }
  else if (sscanf(look->originBuffer,"%d",&x0))
    { look->origin +=  x0 - 1 ;  /* -1 Plato dixit  */
      strcpy (look->originBuffer, "1") ;
    }
  else
    { messout ("Give either a sequence name or a position in the "
	       "current units.") ;
      return ;
    }

 done:
  fMapDraw(look,0) ; /* since i want to rewrite the coordinate system */
}

static void fMapDoPickOrigin (KEY key)
{
  int i ;
  LOOKGET("fMapDoPickOrigin") ;

  fMapSelect(look) ;

  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    if (arrp(look->segs,i,SEG)->key == key)
      { look->origin = arrp(look->segs,i,SEG)->x1 - 1 ;
	fMapDraw(look,0) ; /* since i want to rewrite the coordinate system */
	graphUnMessage () ;
	return ;
      }
  messout ("Sorry, sequence %s is not being displayed in this graph",
	   name(key)) ;
}

static void fMapOriginButton (void)
{
  displayBlock (fMapDoPickOrigin, 
		"If the object is being displayed on this graph the origin"
		"will be set to the start of that sequence.\n"
		"Remove this message to cancel.") ;
}

/**********************************/

static void fMapSetLength(void)
{
  int x0 , org ;
  LOOKGET("fMapSetLength") ;

   if(!fMapFindDNA(look))
    { messout("Sorry, the DNA sequence is not in this database") ;
      goto abort ;
    }

  fMapSelect(look) ;
  if (!sscanf(look->lengthBuffer,"%d",&x0))
    while(TRUE)
      {
	if (!graphPrompt("New active length in base pair unit:", "0", "i"))
	  return ;
	else
	  if (freeint(&x0) && x0 > 0)
	    break ;
	messout("Please type a positive integer") ;
      }
  
  org = look->origin ;
  if (org >= arrayMax(look->dnaArray) &&
      x0 + org < 0)
    { messout ("Note that these limits %d, %d are off the actual sequence: 0, %d",
	       org, org + x0, arrayMax(look->dnaArray)) ;
    }
    
  strncpy(look->lengthBuffer, messprintf("%d", x0), 7) ;
  look->length = x0 ;
  fMapDraw(look,0) ; /* since i want to hide some of the chromosome */
  return ;

 abort:
  strncpy(look->lengthBuffer, messprintf("%d", look->length), 7) ;
}

/****************************************************************/
/****************************************************************/

static void fMapDrawScalesEtc (LOOK look)
{
  int	i, nx, ny ;
     /* change the color of frameBox */
  int color[] = {LIGHTBLUE, LIGHTGREEN, YELLOW} ;
  nx = graphWidth ; ny = graphHeight ;

  arrayMax (look->boxIndex) = 0 ;

  graphColor(BLACK) ;
  graphBoxStart () ; 
    graphText ("Selected fMap",1.0,.5) ; 
  graphBoxEnd () ;
  if (look == selectedfMap)
    graphBoxDraw (1,BLACK,LIGHTRED) ;
  else
    graphBoxDraw (1,WHITE,WHITE) ;

  drawScale (look) ;
  drawChromosome (look) ;

  look->dnaBox = i = graphBoxStart() ;
  graphRectangle (xScale-3.2,2 + topMargin,xScale-2.,ny-2) ;
  graphBoxEnd() ;
  graphBoxDraw (i,BLACK,YELLOW) ;
  graphColor (BLACK) ;

  look->segNameBox = graphBoxStart() ;
  graphTextPtr (look->segNameBuffer, 1, 2, 13) ;
  graphTextPtr (look->segX1Buffer, 1, 3, 6) ;
  graphTextPtr (look->segX2Buffer, 8, 3, 6) ;
  graphBoxEnd() ;
  graphBoxDraw (look->segNameBox, BLACK, LIGHTBLUE) ;

  look->homolBox = i = graphBoxStart() ;
  graphTextPtr (look->homolText, xScale+10, ny-1, 40) ;
  graphBoxEnd() ;
  graphBoxDraw (i, BLACK, WHITE) ;

  look->frameBox = i = graphBoxStart() ;
  graphTextPtr (look->frameBuffer, 17, .5, 8) ;
  graphBoxEnd() ;
 
   { int f = look->frame ; 
      if(f>=0)
	strcpy (look->frameBuffer, messprintf("Frame %d", 1 + look->frame)) ;
      else if ( f != -3)
	strcpy (look->frameBuffer, messprintf("Frame %d", -1 + look->frame)) ;
      else if ( f == -3)
	strcpy (look->frameBuffer, messprintf("Frame %d", -1 )) ;
    }

  graphBoxDraw (i, BLACK, color[(3 + look->frame) % 3]) ;

  graphButton ("Origin:", fMapOriginButton, 27.5, .3) ;
  if (!strlen(look->originBuffer))
    strcpy (look->originBuffer,"1") ;
  look->originBox =
    graphTextEntry (look->originBuffer, 14, 36, .5, 
		    fMapShiftOrigin) ;

  graphText ("Active length: ", 52, .5) ;
  if (!strlen(look->lengthBuffer))
    strcpy(look->lengthBuffer,"0") ;
  look->lengthBox =
    graphTextEntry (look->lengthBuffer, 8, 66.5, .5, 
		    fMapSetLength) ;

  minLiveBox = 1 + graphBoxStart() ;
  graphBoxEnd() ;
}

/****************************************************************/

static void fMapDrawGraphFeatures (LOOK look)
{
  int	i, box, nx, ny ;
  float x, y1, y2 ;
  SEG	*seg ;

  fMargin = graphWidth - look->fWidth -2 ;
  nx = graphWidth ; ny = graphHeight ;

  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;

      y1 = FMAP2GRAPH(look,seg->x1) ;
      y2 = FMAP2GRAPH(look,seg->x2+1) ;	/* to cover full base */
      if (y2 > 2 + topMargin && y1 < ny-1)
	{ 
	  if (y1 < 2 + topMargin)
	    y1 = 2 + topMargin ;
	  switch (seg->type)
	    { 
	    case TRANSPOSON_UP:
	    case TRANSPOSON_DOWN:
	      if (!seg->parent)
		break ;
	      if (seg->type == TRANSPOSON_UP)
		x = xScale - 4.5 ;
	      else
		x = xScale + 5.5 ;
	      box = graphBoxStart () ;
	      array(look->boxIndex,box,int) = i ;
	      graphRectangle (x-0.2, y1, x+0.2, y2) ;
	      graphBoxEnd() ;
	      graphBoxDraw (box, BLUE, WHITE) ;
	      break ;

	    case EXON_UP:
	    case EXON_DOWN:
	      if (seg->type == EXON_UP)
		x = xScale - 6 ;
	      else
		x = xScale + 7 ;
	      box = graphBoxStart () ;
	      array(look->boxIndex,box,int) = i ;
	      graphRectangle (x-0.5, y1, x+0.5, y2) ;
	      graphBoxEnd() ;
	      graphBoxDraw (box, BLUE, WHITE) ;
	      break ;

	    case INTRON_UP:
	    case INTRON_DOWN:
	      if (seg->type == INTRON_UP)
		x = xScale - 6 ;
	      else
		x = xScale + 7 ;
	      box = graphBoxStart () ;
	      array(look->boxIndex,box,int) = i ;
	      y2 = FMAP2GRAPH(look,seg->x2+1) ;	/* to match existing box */
	      graphLine (x, y1, x+0.5, y1) ; /* for clean appearance */
	      graphLine (x, y1, x+0.5, 0.5*(y1+y2)) ;
	      graphLine (x, y2, x+0.5, 0.5*(y1+y2)) ;
	      graphBoxEnd() ;
	      graphBoxDraw (box, BLUE, WHITE) ;
	      break ;

	    case FEATURE_UP:
	    case FEATURE_DOWN:
	      if (seg->type == FEATURE_UP)
		x = xScale - 7.5 ;
	      else
		x = xScale + 8 ;
	      box = graphBoxStart() ;
	      array(look->boxIndex,box,int) = i ;
	      graphRectangle (x, y1, x+0.5, y2) ;
	      graphBoxEnd() ;
	      graphBoxDraw (box, DARKRED, WHITE) ;
	      break;
	    
	    default:
	      break ;
	      
	    }
	}
    }
}

/****************************************************************/

static void fMapDrawTextFeatures (LOOK look)
{
  int	i, box, nx, ny ;
  Stack textStack = stackCreate(100) ;
  float x, y1, y2 ;
  SEG	*seg ;
  BUMP  bump ;
  OBJ   obj ;
  char  *text ;

  if(look->fWidth < 3)
    return ;
  fMargin = graphWidth - look->fWidth + 1 ;
  nx = graphWidth ; ny = graphHeight ;
  bump = bumpCreate (look->fWidth) ;

  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;

      y1 = FMAP2GRAPH(look,seg->x1) ;
      y2 = FMAP2GRAPH(look,seg->x2+1) ;	/* to cover full base */
      if (y2 > 2 + topMargin  && y1 < ny-1)
	{ 
	  switch (seg->type)
	    { 
	    case TITLE:
	    case cDNA:
	    case PROPERTY:
	      box = graphBoxStart();
	      array(look->boxIndex,box,int) = i ;
	      bumpWrite (bump,name(seg->key), 0, seg->x1) ;
	      graphBoxEnd() ;
	      break ;

	    case TEXT:
	      if (!(obj = bsCreate (seg->key)))
		break ;
	      if (bsFindTag (obj,seg->key) &&
		  bsGetData (obj, _bsRight, _Text, &text))
		{ box = graphBoxStart () ;
		  array(look->boxIndex,box,int) = i ;
		  bumpWrite (bump,text, 5, seg->x1) ;
		  graphBoxEnd() ;
		}
	      bsDestroy (obj) ;
	      break;

	    case FEATURE_TAG:
	      box = graphBoxStart() ;
	      array(look->boxIndex,box,int) = i ;
	      bumpWrite (bump,name(seg->key),10,seg->x1) ;
	      graphBoxEnd() ;
	      break ;
	    
	    case DNA_HOMOL:
	    case PEP_HOMOL:
	      box = graphBoxStart() ;
	      array(look->boxIndex,box,int) = i ;
	      x = fMargin + 10 + 30.0 / seg->f ;
	      graphRectangle (x, y1, x+1, y2) ;
	      graphBoxEnd() ;
	      graphBoxDraw (box, (seg->type==DNA_HOMOL) 
			           ? DARKGREEN : DARKBLUE, WHITE) ;
	      break;

	    case MASTER: 
	    case HOMOL_2: 
	      break ;

	    case GENERAL:
	      box = graphBoxStart() ;
	      array(look->boxIndex,box,int) = i ;
	      stackClear (textStack) ;
	      catText (textStack, className(seg->key)) ;
	      catText (textStack, ": ") ;
	      catText (textStack, name(seg->key)) ;
	      bumpWrite (bump,stackText(textStack,0),10,seg->x1) ; 
	      graphBoxEnd() ;
	      if (class(seg->key) == _VClone)
		graphBoxDraw(box,BLACK,YELLOW) ;
	      else
		graphBoxDraw(box,BLACK,WHITE) ;
	      break ;
	
	    default:
	      break ;
	    }
	}
    }

  bumpDestroy (bump) ;
  stackDestroy (textStack) ;
}

/**************/

static void findWidths(LOOK look)
{ int nx , x ;
  float subunit ;
  int 
    isD =  (look->showStatus & STATUS_DNA) ? 1 : 0 ,
    isP =  ((look->showStatus & STATUS_PROTEIN) &&
	    look->protein && arrayMax(look->protein) )
      ? 1 : 0 ,
    isF =  (look->showStatus & STATUS_TEXT_FEATURES) ? 1 : 0 ,
    isC =  (look->showStatus & STATUS_COORDS) ? 1 : 0 ;
  int
    nn = 3 * isD + isP + 2 * isF ;

  graphFitBounds (&graphWidth,&graphHeight) ;
  halfGraphHeight = 0.5*(graphHeight - topMargin) ;
  
  if(!nn)
    return ;

  if(!isP)  /* protein missing */
    look->showStatus &= ~(STATUS_PROTEIN) ;  /* tilda not */


  nx = graphWidth 
    - xScale - 12              /* To the right of scales etc */
    - 8 * isC * (isD + isP) ;  /* space for coordinates */

  subunit = .4/look->mag ;  /* see fMapDisplaySequenceProtein */
  subunit = 3 * (int)subunit ;

  while (look->mag*subunit < 1.002)
    subunit += 3 ; /* so successive lines will not overlap */

  x = nx/nn - 1  > subunit ? subunit : nx/nn - 1 ;

  look->dnaWidth = isD * ( 3 * x + 8 * isC ) ;
  look->proteinWidth = isP * ( x + 8  * isC ) ;
  look->fWidth = isF * (nx - look->dnaWidth - look->proteinWidth
    - 4 * (isD + isP) ) ;  /* intervals between displays */
  if (look->fWidth < 0)
    look->showStatus ^= STATUS_TEXT_FEATURES ;
}

/**************/

static void fMapDraw (LOOK look, KEY curr)
{
  int i, currIndex = 0 ;

  if (graphActivate(look->graph))
    graphPop() ;
  else
    return ;
     
  findWidths(look) ;
  graphClear () ;
  look->sequenceBox = 0 ;
  look->proteinBox = 0 ;
  look->dnaBox = 0 ;
 
  if (graphHeight < 10 || graphWidth < xScale+20)
    { messout ("Sorry, this window is too small for a sequence display") ;
      return ;
    }

  if (curr)
    { for (i = minLiveBox ; i < arrayMax(look->boxIndex) ; ++i)
	if (arrp(look->segs,arr(look->boxIndex,i,int),SEG)->key == curr)
	  { currIndex = i ;
	    break ;
	  }
    }
  else if (look->activeBox)
    currIndex = arr(look->boxIndex, look->activeBox, int) ;

  fMapDrawScalesEtc(look) ;	/* sets minLiveBox */

  fMapDrawGraphFeatures(look) ;
  if (look->showStatus & STATUS_TEXT_FEATURES)
    fMapDrawTextFeatures(look) ;

  if (look->colors)
    { fMapDisplayDNApicture(look) ;
      fMapDisplaySequence(look) ;
    }

  graphButtons (buttonOpts, 16, 2.5, graphWidth) ; /* To have them on top */

  look->activeBox = 0 ;
  if (currIndex)
    for (i = minLiveBox ; i < arrayMax(look->boxIndex) ; ++i)
      if (arr(look->boxIndex, i, int) == currIndex)
	{ fMapSelectBox (look,i) ;
	  break ;
	}

  graphRedraw() ;
  fMapSelect(look) ;
}

/***************/
  /* to display the DNA icon and the sequence */
void fMapDisplayDNA (void *vv)
{  LOOK look = (LOOK) vv ;

   if (!look || look->magic != MAGIC || 
       !graphActivate (look->graph))
     messcrash ("fMapDisplayDNA was passed a bad LOOK structure") ;

   fMapDraw (look, 0) ;
}

/**********************************/

int fMapSetFrame(int f)
{ 
  LOOK  look = 0 ; int old = -5 ; /*impossible frame */

  if (selectedfMap && selectedfMap->magic == MAGIC
      && graphExists(selectedfMap->graph))
    look = selectedfMap ;
  if (look)
    { old = look->frame ;
      look->frame = f ;
    }
  return 
    old  ;
}

/**********************************/

int fMapSetStatus (int s)
{ 
  LOOK  look = 0 ;

  if (selectedfMap && selectedfMap->magic == MAGIC
      && graphExists(selectedfMap->graph))
    look = selectedfMap ;
  if (!look)
    return 0 ;
  if (s != -1 && look->showStatus != s)
    { look->showStatus = s ;
      graphActivate(look->graph) ;
      fMapDraw(look,0) ;
    }
  return look->showStatus ;
}

/**********************************/

void fMapRaiseStatus(int s)
{ 
  LOOK  look = 0 ;

  if (selectedfMap && selectedfMap->magic == MAGIC
      && graphExists(selectedfMap->graph))
    look = selectedfMap ;
  if(look && s != -1)
    look->showStatus |= s ;
}

/**********************************/

void fMapFindOrigin(void *vv, int *origin, int *length)
{ LOOK look = (LOOK) vv ;

  if (look->magic != MAGIC )
    messcrash("look confusion in fMapShiftOrigin") ;
  *origin = look->origin ;
  *length = look->length ;
}

/*****************************************************************/
/*****************************************************************/

static void fMapClearDNA (void)
{ 
  int   i , *ip ;

  LOOKGET("fMapClearDNA") ;
  
  if(look->colors)
    for (i = arrayMax(look->colors), ip = arrp(look->colors,0,int) ; i-- ;)
      *ip++ = BLACK ;

  look->showStatus = 0 ;
  fMapDraw(look,0) ;
}

/*****************************************************************/
/*****************************************************************/

static Associator assScore = 0 ;

float relScore (KEY method, float score)
{
  OBJ obj ;
  float min = 0.0 ;

  if (!method)
    return 1.0 ;

  if (!assScore)
    assScore = assCreate () ;
  if (!assFind (assScore, (void*)method, &min) &&
      (obj = bsCreate (method)))
    { if (bsGetData (obj, _Min_score, _Float, &min))
	assInsert (assScore, (void*)method, *(void**)&min) ;
      bsDestroy (obj) ;
    }

  if (!min)
    return score ;
  else if (score < min)
    return 1.0 ;
  else
    return score/min ;
}

float unRelScore (KEY method, float score)
{
  OBJ obj ;
  float min = 0.0 ;

  if (!method)
    return score ;

  if (!assScore)
    assScore = assCreate () ;
  if (!assFind (assScore, (void*)method, &min) &&
      (obj = bsCreate (method)))
    { if (bsGetData (obj, _Min_score, _Float, &min))
	assInsert (assScore, (void*)method, *(void**)&min) ;
      bsDestroy (obj) ;
    }

  if (!min)
    return score ;
  else
    return score*min ;
}

/*****************************/

static BOOL pcDown ;
static int pcStart, pcEnd ;
static Array pcArray = 0 ;

static void posConvertInit (SEG *seg, Array exons, BOOL isDown)
{
  int i ;

  pcDown = isDown ;
  pcStart = seg->x1 - 1 ;
  pcEnd = seg->x2 + 1 ;
  pcArray = arrayReCreate(pcArray,20,int) ;
  for (i = 0 ; i < arrayMax(exons)-2 ; i+=2)
    { array(pcArray,i,int) = arr(exons,i+1,BSunit).i ;
      array(pcArray,i+1,int) = 
	arr(exons,i+2,BSunit).i - arr(exons,i+1,BSunit).i - 1 ;
    }
}

static void posConvert (SEG *seg, int pos1, int pos2)
{
  int i ;

  for (i = 0 ; i < arrayMax(pcArray) ; i += 2)
    if (pos1 > arr(pcArray, i, int))
      pos1 += arr(pcArray, i+1, int) ;
    else
      break ;

  for (i = 0 ; i < arrayMax(pcArray) ; i += 2)
    if (pos2 > arr(pcArray, i, int))
      pos2 += arr(pcArray, i+1, int) ;
    else
      break ;
	
  if (pcDown)
    { seg->x1 = pos1 + pcStart ;
      seg->x2 = pos2 + pcStart ;
    }
  else
    { seg->x1 = pcEnd - pos2 ;
      seg->x2 = pcEnd - pos1 ;
    }
}

static Array fMapConvert (KEY seq, KEY fMap)
{
  KEY	key, parent, title, tag ;
  char	*text ;
  Array segs ;
  BSunit *u ;
  SEG	*seg, *seg1, *seg2 ;
  OBJ	obj ;
  int	i, nsegs = 0, iseg, pos1, pos2, tmp ;
  float score ;
  BOOL	isDown, isExons ;
  static Array units = 0 ;
  static Array flatA = 0 ;	/* NB used by posConvert */

/* Need POS_TO_SEG1 since posConvert only valid after posConvertInit */
#define POS_TO_SEG1	if (isDown) \
                          { seg1->x1 = seg->x1 + pos1 - 1 ; \
			    seg1->x2 = seg->x1 + pos2 - 1 ; \
			  } \
                        else \
			  { seg1->x1 = seg->x2 - pos2 + 1 ; \
			    seg1->x2 = seg->x2 - pos1 + 1 ; \
			  }

  if (!(obj = bsCreate (seq)))
    { messout ("No data for sequence %s",name(seq)) ;
      return 0 ;
    }
  if (!lexlock (fMap))
    { messout ("Sorry, %s is locked (being processed elsewhere)",
	       name(fMap)) ; 
      return 0 ;
    }

  segs = arrayCreate (128,SEG) ;
  if (!units)
    units = arrayCreate (256, BSunit) ;

				/* first seg is master */
  seg1 = arrayp(segs,nsegs++,SEG) ;
  seg1->key = seg1->parent = 0 ;
  seg1->x1 = 1 ;
  if (!(seg1->x2 = sequenceLength(seq)))
    seg1->x2 = 10000 ;		/* arbitrary default */
  seg1->type = MASTER ;
				/* second is self */
  seg2 = arrayp(segs,nsegs++,SEG) ;
  seg2->key = seg2->parent = seq ;
  seg2->x1 = 1 ;
  seg2->x2 = seg1->x2 ;
  seg2->type = SEQUENCE_DOWN ;
  bsDestroy (obj) ;

  for (iseg = 1 ; iseg < nsegs ; ++iseg)
/* recurses through all sequences since they get added in turn */
    { seg = arrp(segs,iseg,SEG) ;
      if (seg->type == SEQUENCE_DOWN)
	isDown = TRUE ;
      else if (seg->type == SEQUENCE_UP)
	isDown = FALSE ;
      else
	continue ;		/* only interested in sequences */
      if (!(obj = bsCreate (seg->key)))
	continue ;
      parent = seg->key ;
				/* first create subsequences */
      if (bsFindTag (obj, _Contains) && bsFlatten (obj, 4, units))
	{ for (i = 0 ; i < arrayMax(units) ; i += 4)
	    { u = arrp(units,i,BSunit) ;
	      seg1 = arrayp (segs,nsegs++,SEG) ;
	      seg1->key = seg1->parent = u[1].k ;
	      if (!u[2].i || !u[3].i)
		{ messerror ("Coords of subsequence %s missing in %s",
			     name(seg1->key), name(seg->key)) ;
		  --nsegs ;
		}
	      else 
		{ pos1 = u[2].i ; pos2 = u[3].i ;
		  seg1->type = (pos2 > pos1) ? SEQUENCE_DOWN : SEQUENCE_UP ;
		  if (pos1 > pos2)
		    { tmp = pos2 ; pos2 = pos1 ; pos1 = tmp ;}
		  POS_TO_SEG1 ;
		}
	    }
	}

      if (bsFindTag (obj, _Transposon))
	seg->type = isDown ? TRANSPOSON_DOWN : TRANSPOSON_UP ;

         /* source exons */
      pos1 = 0 ;
      flatA = arrayReCreate(flatA, 24, BSunit) ;
      if (bsFindTag (obj, _Source_Exons) && bsFlatten (obj, 2, flatA))
	for (i = 0 ; i < arrayMax(flatA) ; i += 2)
	  { pos2 = arr(flatA, i, BSunit).i ;
	    if (pos1)
	      { seg1 = arrayp (segs,nsegs++,SEG) ;
		POS_TO_SEG1 ;
		seg1->key = 0 ;
		seg1->parent = parent ;
		seg1->x1++ ;  /* Because the bsTree gives the coordinates of the exon */
		seg1->x2-- ;
		seg1->type = (isDown) ? INTRON_DOWN : INTRON_UP ;
	      }
	    pos1 = pos2 ;
	    pos2 = arr(flatA, i+1, BSunit).i ;
	    if (pos2)
	      { seg1 = arrayp (segs,nsegs++,SEG) ;
		POS_TO_SEG1 ;
		seg1->key = 0 ;
		seg1->parent = parent ;
		seg1->type = (isDown) ? EXON_DOWN : EXON_UP ;
	      }
	    else
	      messerror ("Missing pos2 for exon in %s", name(seg->key)) ;
	    pos1 = pos2 ;
	    isExons = TRUE ;
	  }
      else
	{ arrayMax (flatA) = 0 ;
	  array (flatA, 0, BSunit).i = 1 ;
	  array (flatA, 1, BSunit).i = seg->x2 - seg->x1 + 1 ;
	  isExons = FALSE ;
	}

      posConvertInit (seg, flatA, isDown) ;

	  /* text identifying sequence */
      seg1 = arrayp (segs,nsegs++,SEG) ;
      *seg1 = *seg ;
      if (bsGetKey (obj, _Title, &title))
	seg1->key = title ;	/* else = parent implicitly */
      seg1->type = TITLE ;

         /* cDNA */
      if (bsFindTag (obj, _cDNA))
	{ seg1 =  arrayp (segs,nsegs++,SEG) ;
	  *seg1 = *seg ;
	  seg1->key = _cDNA ;
	  seg1->type = cDNA ;
	}

      if (bsFindTag (obj, _CDS))
	{ seg1 = arrayp (segs,nsegs++,SEG) ;
	  seg1->key = _CDS ;
	  seg1->type = CDS ;
	  seg1->parent = parent ;
	  pos1 = 1 ; 
	  pos2 = seg->x2 - seg->x1 + 1 ;
	  bsGetData (obj, _bsRight, _Int, &pos1) ;
	  bsGetData (obj, _bsRight, _Int, &pos2) ;
	  posConvert (seg1, pos1, pos2) ;
	  if (!isExons)
	    { seg2 = arrayp (segs,nsegs++,SEG) ;
	      *seg2 = *seg1 ;
	      seg2->key = 0 ;
	      seg2->type = isDown ? EXON_DOWN : EXON_UP ;
	    }
	}

         /* Properties */
      if (bsGetKey (obj, _Properties, &key))
	switch (key)
	  {
	  case _Coding: 
	    break ;
	  case _Transcript:
	    bsGetKey (obj, _bsRight, &key) ; /* fall through */
	  default:
	    seg1 = arrayp (segs, nsegs++, SEG) ;
	    *seg1 = *seg ;
	    seg1->type = PROPERTY ;
	    seg1->key = key ;
	    if (bsGetData (obj, _bsRight, _Text, &text))
	      { seg1 =  arrayp (segs,nsegs++,SEG) ;
		*seg1 = *seg ;
		seg1->type = TEXT ;
		seg1->key = key ;
	      }
	  }

  	 /* remarks, clones etc */
      if (bsFindTag (obj, _General) && 
	  bsFlatten (obj, 2, units))
	for (i = 0 ; i < arrayMax(units)/2 ; ++i)
	  { seg1 = arrayp(segs,nsegs++,SEG) ;
	    *seg1 = *seg ;
	    seg1->key = arr(units, 2*i+1, BSunit).k ;
	    if (arr(units, 2*i, BSunit).k == _Foreign_Reference)
	      seg1->type = TEXT ;
	    else
	      seg1->type = GENERAL ;
	  }
      
	 /* features */
      if (bsFindTag(obj, _Features) &&
	  bsFlatten(obj, 4, flatA))
	for(i=0; i<arrayMax(flatA); i+=4)
	  { tag  = arr(flatA,i, BSunit).k ; 
	    pos1 = arr(flatA,i+1, BSunit).i ; 
	    pos2 = arr(flatA,i+2, BSunit).i ; 
	    /* fix of an inconsistency of the model
	       which has a single int, nov 1 91 
	       */
	    if (tag == _TSL_site)
	      pos2 = pos1 ;
	    key  = arr(flatA,i+3, BSunit).k ; 
	    if( class(key) != _VText ||
	       ! iskey(key))
	      key = tag ;
	    if(!pos2)
	      pos2 = pos1 ;
	    
	    seg1 =  arrayp (segs,nsegs++,SEG) ;
	    seg1->type = isDown ? FEATURE_DOWN : FEATURE_UP ;
	    seg1->key = 0 ;
	    posConvert (seg1, pos1, pos2) ;
	    seg2 =  arrayp (segs,nsegs++,SEG) ;
	    seg2->type = FEATURE_TAG ;
	    seg2->key = key ;
	    seg2->x1 = seg1->x1 ; seg2->x2 = seg1->x2 ;
	    seg2->parent = seg1->parent = nsegs ;
	    /* so they are a distinct pair when picked */
	  }
      
	 /* Homologies */
      if (bsFindTag (obj, _DNA_homol) && bsFlatten (obj, 7, units))
	for (i = 0 ; i < arrayMax(units)/7 ; ++i)
	  { u = arrayp(units, 7*i, BSunit) ;
	    score = relScore (u[1].k, u[2].f) ;
	    if (score < 1.0)
	      score = 1.0 ;
	    seg1 = arrayp (segs,nsegs++,SEG) ;
	    seg1->type = DNA_HOMOL ;
	    seg1->key = parent ;
	    seg1->parent = nsegs ;
	    seg1->f = score ;
	    pos1 = u[3].i ; pos2 = u[4].i ;
	    posConvert (seg1, pos1, pos2) ;
	    seg2 = arrayp (segs,nsegs++,SEG) ;
	    seg2->key = u[0].k ;
	    seg2->x1 = u[5].i ; seg2->x2 = u[6].i ;
	    seg2->type = HOMOL_2 ;
	    seg2->parent = seg1->parent ;
	    seg2->f = u[1].f ;	/* gross abuse - a KEY */
	  }
      
      if (bsFindTag (obj, _Pep_homol) && bsFlatten (obj, 7, units))
	for (i = 0 ; i < arrayMax(units)/7 ; ++i)
	  { u = arrayp(units, 7*i, BSunit) ;
	    score = relScore (u[1].k, u[2].f) ;
	    if (score < 1.0)
	      score = 1.0 ;
	    seg1 = arrayp (segs,nsegs++,SEG) ;
	    seg1->type = PEP_HOMOL ;
	    seg1->key = parent ;
	    seg1->parent = nsegs ;
	    seg1->f = score ;
	    pos1 = u[3].i ; pos2 = u[4].i ; 
	    posConvert (seg1, pos1, pos2) ;
	    seg2 = arrayp (segs,nsegs++,SEG) ;
	    seg2->key = u[0].k ;
	    seg2->x1 = u[5].i ; seg2->x2 = u[6].i ;
	    seg2->type = HOMOL_2 ;
	    seg2->parent = seg1->parent ;
	    seg2->f = u[1].f ;
	  }

      bsDestroy(obj) ;
    }

  arraySort (segs, fMapOrder) ;

/* arrayStore (fMap,segs, segFormat) ; *//* RMD comment out until we fix the structure */
  lexunlock(fMap) ;

  return segs ;
}

/*********************************************************/

#define NORMAL_FCOL BLACK
#define NORMAL_BCOL WHITE
#define SELECT_FCOL BLACK
#define SELECT_BCOL LIGHTRED
#define RELATED_BCOL LIGHTBLUE

static void fMapSelectBox (LOOK look, int box)
{ 
  int i ;
  SEG *seg, *seg2 ;
  KEY parent ;
  char *goodName ;

  if (box >= arrayMax(look->boxIndex) || box < minLiveBox)
    return ;

  if (look->activeBox)
    { seg = arrp(look->segs,
		 arr(look->boxIndex,look->activeBox,int),SEG) ;
      if (seg->type == DNA_HOMOL || seg->type == PEP_HOMOL)
	graphBoxDraw (look->homolBox, BLACK, WHITE) ;
      parent = seg->parent ;
      if (parent)
	for (i = minLiveBox ; i < arrayMax(look->boxIndex) ; ++i)
	  { seg = arrp(look->segs,arr(look->boxIndex,i,int),SEG) ;
	    if (seg->parent == parent)
	      switch (seg->type)
		{
		case SEQUENCE_UP: case SEQUENCE_DOWN:
		case EXON_UP: case EXON_DOWN:
		case INTRON_UP: case INTRON_DOWN:
		  graphBoxDraw (i,BLUE,NORMAL_BCOL) ;
		  break ;
		default:
		  graphBoxDraw (i,NORMAL_FCOL,NORMAL_BCOL) ;
		  break ;
		}
	  }

    }

  look->activeBox = box ;
  
  seg = arrp(look->segs,
	     arr(look->boxIndex,look->activeBox,int),SEG) ;

  goodName = iskey(seg->key) ? name(seg->key) : 
    ( seg->parent && iskey(seg->parent) ? name(seg->parent) : "") ; 
  strncpy (look->segNameBuffer, goodName, 13) ;
  strncpy (look->segX1Buffer, 
	   messprintf ("%d", seg->x1 - look->origin), 6) ;
  strncpy (look->segX2Buffer, 
	   messprintf ("%d", seg->x2 - look->origin), 6) ;
  graphBoxDraw (look->segNameBox, BLACK, LIGHTBLUE) ;

  parent = seg->parent ;
  switch (seg->type)
    {
    case SEQUENCE_UP: case SEQUENCE_DOWN:
    case EXON_UP: case EXON_DOWN:
    case INTRON_UP: case INTRON_DOWN:
      graphBoxDraw (look->activeBox,BLUE,SELECT_BCOL) ;
      break ;
    case DNA_HOMOL:
    case PEP_HOMOL:
      for (i = 0 ; i < arrayMax(look->segs) ; ++i)
	{ seg2 = arrp(look->segs, i, SEG) ;
	  if (seg2->parent == parent && 
	      seg2->type == HOMOL_2)
	    break ;
	}
      if (i < arrayMax(look->segs))
	{ strncpy (look->homolText, 
		   messprintf ("%s %d %d %s %f",
			       name(seg2->key),
			       seg2->x1, seg2->x2,
			       name(*(KEY*)&seg2->f),
			       unRelScore(*(KEY*)&seg2->f,
					  seg->f)),
		   39) ;
	  graphBoxDraw (look->homolBox, 
			BLACK, RELATED_BCOL) ;
	}
				/* fall through */
    default:
      graphBoxDraw (box,BLACK,SELECT_BCOL) ;
      break ;
    }

  if (parent)
    for (i = minLiveBox ; i < arrayMax(look->boxIndex) ; i++)
      { seg = arrp(look->segs,arr(look->boxIndex,i,int),SEG) ;
	if (seg->parent == parent)
	  switch (seg->type)
	    {
	    case SEQUENCE_UP: case SEQUENCE_DOWN:
	    case EXON_UP: case EXON_DOWN:
	    case INTRON_UP: case INTRON_DOWN:
	      graphBoxDraw (i,BLUE,RELATED_BCOL) ;
	      break ;
	    default:
	      graphBoxDraw (i,BLACK,RELATED_BCOL) ;
	      break ;
	    }
      }
}

/**************************************/

static void fMapFollow (LOOK look, double x, double y)
{
  SEG* seg = arrp(look->segs,arr(look->boxIndex,look->activeBox,int),SEG) ;
  KEY key = seg->key ;
  int table = class(key) ;

/*     jtm, i do not understand why we should not follow on double select homology
if (seg->type == DNA_HOMOL || seg->type == PEP_HOMOL)
    { return ;			// should go to homol2
    }
*/

  if (!table || table == _VText)
    { key = seg->parent ;
      table = class(key) ;
    }

  switch (table)
    {
    case _VSequence:
      display (key, look->seqKey, TREE) ;
      break ;
    case 0:
      break ;
    default:
      display (key, look->seqKey, 0) ;
      break ;
    }
}

/****************************/
/****************************/
