/*  File: cmapdisp.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: physical unit display of a chromosome
 	based on gmap
 * Exported functions: cmapDisplay()
 * HISTORY:
 * Last edited: Apr  2 18:04 1992 (mieg)
 * Created: Thu Jan  9 22:54:23 1992 (rd)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "keyset.h"
#include "graph.h"
#include "key.h"
#include "lex.h"
#include "bs.h"
#include "a.h"
#include "systags.wrm"
#include "classes.wrm"
#include "sysclass.wrm"
#include "tags.wrm"
#include "session.h"
#include "disptype.wrm"
#include "display.h"
#include "query.h"
#include "wormtype.h"
#include "grid.h"

/* the basic units are Mb - left end is 0 - calculate length from contigs
*/

typedef struct LOOKSTUFF
  { int   magic ;	/* == MAGIC */
    KEY   key ;		/* a chromosome */
    int   activeBox ;
    unsigned int flag ;
    float centre, mag ;	
    float max ;
    Array segs,      	/* info on things to be drawn */
          boxes,	/* a BOX structure of each box */
          neighbours ;	/* 0-terminated lists of neighbours */
    Associator ctgAss ;
  } *LOOK ;

#define MAGIC 361718

typedef struct
  { KEY key ;
    float x1, x2 ;
    float p1, p2 ;
    int flag ;
  } SEG ;

typedef struct
  { SEG* seg ;		/* pointer OK: segs fixed when drawing */
    int  neigh ;	/* index in look->neighbours */
    int  col ;
  } BOX ;

/* p1, p2 are used for contigs: cmap <-> pmap
                   for in_situ data for in_situ clones
*/

				/* seg->flag flags */
#define FLAG_HIGHLIGHT		0x0001
#define FLAG_PROCESSED		0x0002
#define FLAG_INSITU		0x0004
#define FLAG_HYB		0x0008
#define FLAG_DELETING		0x0010

#define LOOKGET(name)     LOOK look ; \
                          if (!graphAssFind (cMapDisplay,&look)) \
		            messcrash ("graph not found in %s",name) ; \
			  if (!look) \
                            messcrash ("%s received a null pointer",name) ; \
                          if (look->magic != MAGIC) \
                            messcrash ("%s received a wrong pointer",name)

static void cMapDestroy (void) ;
static void cMapRecalculate(void) ;
static void cMapPick (int box, double x , double y) ;
static void cMapDragCursor (float *x, float *y, BOOL isDone) ;
static void cMapMiddleDown (double x, double y) ;
static void cMapDrag (double x, double y) ;
static void cMapUp (double x, double y) ;
static void cMapDraw (LOOK look, KEY key) ;
static BOOL cMapConvert (LOOK look) ;
static void cMapResize (void) ;
static void cMapSelect (LOOK look, int box) ;
static void cMapFollow (LOOK look, double x, double y) ;
static void cMapHighlight (void) ;
static void cMapAdd (void) ;
static void cMapSubtract (void) ;
static void cMapWhole (void) ;
static int  cMapOrder(void *a, void *b) ; /* for arraySort() call */
static void cdnaPaperDisplay (void) ;
static void cMapChangeSymbolSize (void) ;
static void bumpSloppySet (void) ;

static MENUOPT cMapMenu[] =
              { graphDestroy, "Quit",
		help,"Help",
		graphPrint,"Print",
		displayPreserve,"Preserve",
		cMapRecalculate,"Recalculate",
		cMapHighlight,"Highlight Selected Objects",
		cMapAdd,"Add Selected Objects",
		cMapSubtract,"Subtract Selected Objects",
		cdnaPaperDisplay,"Full Genome Distribution",
		cMapChangeSymbolSize,"Change Symbol Size",
		bumpSloppySet,"Change bump sloppiness",
		0, 0
	      } ;

static int	cursorBox ;

extern BOOL gMapGetPos(KEY from, KEY *chromo, float *x1, float *x2) ;

/**********************/

BOOL cMapDisplay (KEY key, KEY from, BOOL isOldGraph)
{
  LOOK look=(LOOK)messalloc(sizeof(struct LOOKSTUFF)) ;
  float x1, x2 ;

  if (key && class(key) != _VChromosome)
    { from = key ;
      key = 0 ;
    }
  if (from && !gMapGetPos(from, &key, &x1, &x2))
    if (key)
      from = 0 ;
    else
      display (from, 0, TREE) ;
  
  if (!key)
    goto abort ;

  look->activeBox = 0 ;
  look->magic = MAGIC;
  look->flag = 0 ;
  look->max = 15.0 ;		/* arbitrary min */
  look->key = key ;

  if (!cMapConvert (look))
    goto abort ;
  
  if (isOldGraph)
    { graphRetitle (name (key)) ;
      cMapDestroy () ;
      graphAssRemove (cMapDisplay) ;
    }
  else 
    { if (!displayCreate(CMAP))
	goto abort ;
      graphRetitle (name(key)) ;
      
      graphRegister (RESIZE,(GraphFunc)cMapResize) ;
      graphRegister (DESTROY, cMapDestroy) ;
      graphRegister (PICK, (GraphFunc)cMapPick) ;
      graphRegister (MIDDLE_DOWN, (GraphFunc)cMapMiddleDown) ;
      graphRegister (MIDDLE_DRAG, (GraphFunc) cMapDrag) ;
      graphRegister (MIDDLE_UP, (GraphFunc) cMapUp) ;
      graphMenu (cMapMenu) ;
    }

  graphAssociate (cMapDisplay,look) ;

  cMapWhole () ;	/* sets ->centre, ->mag and calls Draw() */
  return TRUE ;

abort :
  messfree (look) ;
  return FALSE ;
}

/*********************/

static void cMapDestroy (void)
{
  LOOKGET("cMapDestroy") ;

  arrayDestroy(look->segs) ;
  arrayDestroy(look->boxes) ;

  look->magic = 0 ;
  messfree (look) ;
}

/**********************************/

static void cMapResize (void)
{
  LOOKGET("cMapResize") ;

  cMapDraw (look, 0) ;
}

/***********************************/

static void cMapPick (int box, double x, double y) 
{
  LOOKGET("cMapPick") ;

  if (!box)
    return ;

  if (box == cursorBox)
    graphBoxDrag (cursorBox, cMapDragCursor) ;
  else if (box == look->activeBox)
    cMapFollow (look, x, y) ;
  else
    cMapSelect (look, box) ;
}

/*********************/

static void cMapClear (void)
{
  KEY curr ;
  LOOKGET("cMapClear") ;
  
  if (look->activeBox && arrp(look->boxes,look->activeBox,BOX)->seg)
    curr = arrp(look->boxes,look->activeBox,BOX)->seg->key ;
  else 
    curr = 0 ;
  cMapConvert (look) ;
  cMapDraw (look, curr) ;
}

static void cMapRecalculate (void)
{
  KEY cacheKey ;
  LOOKGET("cMapRecalculate") ;

  if (lexReClass (look->key, &cacheKey, _VpMap))
    arrayKill (cacheKey) ;

  cMapClear () ;
}

static void cMapSelect (LOOK look, int box) 
{
  int i, j ;

  if (look->activeBox)
    { graphBoxDraw (look->activeBox, DARKBLUE, LIGHTBLUE) ;
      if (i = arrp(look->boxes, box, BOX)->neigh)
	while (j = arr(look->neighbours, j, int))
	  graphBoxDraw (j, DARKBLUE, LIGHTBLUE) ;
    }
  if (arrp(look->boxes, box, BOX)->seg)
    { look->activeBox = box ;
      graphBoxDraw (box, DARKBLUE, RED) ;
      if (i = arrp(look->boxes, box, BOX)->neigh)
	while (j = arr(look->neighbours, j, int))
	  graphBoxDraw (j, DARKBLUE, LIGHTBLUE) ;
    }
  else
    look->activeBox = 0 ;
}

static void cMapFollow (LOOK look, double x, double y)
{
  display (arrp(look->boxes, look->activeBox, BOX)->seg->key,
		 look->key, 0) ;
}

/**************************************************/
/**************** drawing info ********************/

static int	nx, ny ;	/* window dimensions */
static float	yCentre ;	/* ny/2 */
static float	yLength ;	/* length of picture */
static float	topMargin = 2 ;	/* space at top for buttons etc */
static float	bottomMargin = 1 ; /* space at bottom */
static float	xCursor = 5 ;	/* midline of mini-chromosome */
static float	xScale = 10 ;	/* scale bar text LHS */
static float	xContig = 14 ;	/* midline of contig boxes */
static int	xItem = 16 ;	/* left edge of item field */
static int	xInSitu = 30 ;	/* for insitu clones */

static float	symbolSize = 1.0 ; /* size of little squares */

static BOOL isPapDisp = FALSE ;

#define MAP2GRAPH(look,x) \
  (yCentre  +  look->mag * ((x) - look->centre))
#define GRAPH2MAP(look,x) \
  (((x) - yCentre) / look->mag + look->centre)

/********* start off with some utility routines ********/

static void getNxNy(void)
{
  graphFitBounds (&nx, &ny) ;
  yLength = (ny - topMargin - bottomMargin) ;
  yCentre = topMargin + 0.5*yLength ;
}

/***************************************/

static void cMapWhole (void)
{ 
  LOOKGET("cMapWhole") ; 
  getNxNy() ;
  look->centre = 0.5 * look->max ;
  look->mag = yLength / look->max ;
  cMapDraw (look, 0) ;
}

static void cMapZoomIn (void)
{ 
  LOOKGET("cMapZoomIn") ; 
  look->mag *= 2 ; 
  cMapDraw (look, 0) ;
}

static void cMapZoomOut (void)
{ 
  LOOKGET("cMapZoomOut") ; 
  look->mag /= 2 ; 
  cMapDraw (look, 0) ;
}

static void cMapChangeSymbolSize (void)
{
  if (graphPrompt ("Change size of little square symbols to",
		   messprintf ("%f", symbolSize), "fz"))
    freefloat (&symbolSize) ;
}

/**************************************************************/
/***************** dragging code - middle button **************/

static double	oldy, oldDy, oldx;
static BOOL	dragFast ;
#define DRAGFASTLIMIT xScale

static void cMapDrag (double x, double y) 
{
  if (dragFast)
    { graphXorLine (0, oldy - oldDy, DRAGFASTLIMIT, oldy - oldDy) ;
      graphXorLine (0, oldy + oldDy, DRAGFASTLIMIT, oldy + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, oldy, nx, oldy) ;

  oldy = y ;

  if (dragFast)
    { oldDy *= exp ((x - oldx) / 25.) ;
      oldx = x ;
      graphXorLine (0, y - oldDy, DRAGFASTLIMIT, y - oldDy) ;
      graphXorLine (0, y + oldDy, DRAGFASTLIMIT, y + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, y, nx, y) ;
}

static void cMapUp (double x, double y) 
{ 
  float x1,x2,y1,y2 ;
  LOOKGET("cMapUp") ;

  if (dragFast)
    { graphBoxDim (cursorBox, &x1, &y1, &x2, &y2) ;
      look->mag *= (y2 - y1) / (2. * oldDy) ;
      look->centre = look->max * (y - topMargin) / yLength ;
    }
  else
    look->centre +=  (y - 0.5 - yCentre) / look->mag ;
  cMapDraw (look, 0) ;
}

static void cMapMiddleDown (double x, double y) 
{  
  float x1,x2,y1,y2 ;
  LOOKGET("cMapMiddleDown") ;

  getNxNy () ; 

  graphBoxDim (cursorBox, &x1, &y1, &x2, &y2) ;
  oldDy = (y2 - y1) / 2. ;

  dragFast = (x < DRAGFASTLIMIT) ? TRUE : FALSE ;

  if(dragFast)
    { graphXorLine (0, y - oldDy, DRAGFASTLIMIT, y - oldDy) ;
      graphXorLine (0, y + oldDy, DRAGFASTLIMIT, y + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, y, nx, y) ;
   
  oldx = x ;
  oldy = y ;
  graphRegister (MIDDLE_DRAG, (GraphFunc) cMapDrag) ;
  graphRegister (MIDDLE_UP, (GraphFunc) cMapUp) ;
}

static void cMapDragCursor (float *x, float *y, BOOL isDone)
{
  if (isDone)
    { float x1,y1,x2,y2 ;
      LOOKGET("cMapDragCursor") ;

      getNxNy() ;
      graphBoxDim (cursorBox, &x1, &y1, &x2, &y2) ;
      look->centre = look->max * \
	((*y + 0.5*(y2-y1)) - topMargin) / yLength ;
      cMapDraw (look, 0) ;
    }
  else
    *x = xCursor - 0.5 ;
}

/*****************************************************************/
/************* bump package based on gmap text *************/

typedef struct bumpStruct
  { int n ;
    float *bottom ;
  } *BUMP ;

static BUMP bumpCreate (int ncol)
{
  int i ;
  BUMP bump = (BUMP) messalloc (sizeof (struct bumpStruct)) ;

  bump->n = ncol ;
  bump->bottom = (float*) messalloc (ncol*sizeof (float)) ;
  for (i = 0 ; i < ncol ; ++i)
    bump->bottom[i] = -100000000.0 ;
  return bump ;
}

static void bumpDestroy (BUMP bump)
{
  messfree (bump->bottom) ;
  messfree (bump) ;
}

static float bumpSloppy = 0.0 ;

static void bumpItem (BUMP bump, int wid, float height, 
		                 int *px, float *py)
				/* works by resetting x, y */
{
  int i, j ;
  int x = *px ;
  float ynew, y = *py ;
  
  if (x+wid+20 > bump->n)
    x = bump->n - wid - 20 ;
  if (x < 0) 
    x = 0 ;

  ynew = y ;

  while (TRUE)
    { for (i = x, j = wid ; i < bump->n ; ++i)	/* always start at x */
	{ if (bump->bottom[i] > y + bumpSloppy)
	    j = wid ;
	  else 
	    { if (bump->bottom[i] > ynew)
		ynew = bump->bottom[i] ;
	      if (!--j)		/* have found a gap */
		break ;
	    }
	}
      if (!j)	
	{ for (j = 0 ; j < wid ; j++)	/* advance bump */
	    bump->bottom[i-j] = ynew+height ;
	  *px = i-wid ;
	  *py = ynew ;
	  return ;
	}
      y += 1 ;	/* try next row down */
    }
}

static void bumpSloppySet (void)
{
  if (graphPrompt ("Set sloppiness of bump system - 0 for nosloppy",
		   messprintf ("%f", bumpSloppy), "fz"))
    freefloat (&bumpSloppy) ;
}

/**************************************************/
/**************************************************/

static void drawScale (LOOK look)
{
  float cutoff = 5 / look->mag ;
  float unit = 0.01 ;
  float subunit = 0.001 ;
  float x, y, start, end, oldxScale = xScale ;

  while (unit < cutoff)
    { unit *= 2 ;
      subunit *= 5 ;
      if (unit >= cutoff)
	break ;
      unit *= 2.5 ;
      if (unit >= cutoff)
	break ;
      unit *= 2 ;
      subunit *= 2 ;
    }

  if (isPapDisp)
    { graphTextHeight (0.8) ;
      xScale = xContig-0.5 ;
    }

  start = GRAPH2MAP(look, topMargin) ;
  if (start < 0)
    { start = 0 ;
/*      graphLine (xScale-1.5, MAP2GRAPH(look,start), 
		 xScale-0.5, MAP2GRAPH(look,start)) ; */
    }
  end = GRAPH2MAP(look, ny-bottomMargin) ;
  if (end > look->max)
    { end = look->max ;
/*      graphLine (xScale-1.5, MAP2GRAPH(look,end),
		 xScale-0.5, MAP2GRAPH(look,end)) ; */
    }
      
  x = unit * (((start > 0) ? 1 : 0) + (int)(start/unit)) ;
  while (x <= end)
    { y = MAP2GRAPH(look, x) ;
      if (isPapDisp)
	{
	  graphLine (xScale-1.5,y,xScale-0.5,y) ;
	  if (unit >= 1)
	    graphText (messprintf ("%4.0f",x),xScale-4.2,y-.3) ;
	  else if (unit >= .1)
	    graphText (messprintf ("%4.1f",x),xScale-4.2,y-.3) ;
	  else if (unit >= .01)
	    graphText (messprintf ("%4.2f",x),xScale-4.2,y-.3) ;
	  else if (unit >= .001)
	    graphText (messprintf ("%4.3f",x),xScale-4.2,y-.3) ;
	}
      else
	{
	  graphLine (xScale-1.5,y,xScale-0.5,y) ;
	  if(unit >= 1)
	    graphText (messprintf ("%-4.0f",x),xScale,y-.5) ;
	  else if(unit >= .1)
	    graphText (messprintf ("%-4.1f",x),xScale,y-.5) ;
	  else if(unit >= .01)
	    graphText (messprintf ("%-4.2f",x),xScale,y-.5) ;
	  else if(unit >= .001)
	    graphText (messprintf ("%-4.3f",x),xScale,y-.5) ;
	}
      x += unit ;
    }

  if (!isPapDisp)
    { x = subunit * (((start>=0)?1:0) + (int)(start/subunit)) ;
      while (x <= end)
	{ y = MAP2GRAPH(look,x) ;
	  graphLine (xScale-1.0,y,xScale-0.5,y) ;
	  x += subunit ;
	}
    }

  graphLine (xScale-0.5, MAP2GRAPH(look,start), 
	     xScale-0.5, MAP2GRAPH(look,end)) ;

  if (isPapDisp)
    { xScale = oldxScale ;
      graphTextHeight (0.0) ;
    }
}

static void drawChromosome (LOOK look)
{
  int i ;

#define MAP2CHROM(x) \
  (topMargin + yLength * (x) / look->max)

  graphFillRectangle (xCursor - 0.25, topMargin, 
		      xCursor + 0.25, ny - bottomMargin) ;

  cursorBox = graphBoxStart() ;
  arrayp(look->boxes,cursorBox,BOX)->seg = 0 ;
  graphRectangle (xCursor - 0.5, 
		  MAP2CHROM(GRAPH2MAP(look, topMargin)), 
		  xCursor + 0.5, 
		  MAP2CHROM(GRAPH2MAP(look, ny - bottomMargin))) ;
  graphBoxEnd () ;
  graphBoxDraw (cursorBox,DARKGREEN,GREEN) ;

  graphColor (DARKGRAY) ;
  graphLine (xCursor, MAP2CHROM(GRAPH2MAP(look,topMargin)), 
	     xScale-0.5, topMargin ) ;
  graphLine (xCursor, MAP2CHROM(GRAPH2MAP(look,ny - bottomMargin)),
	     xScale-0.5, ny - bottomMargin) ;
  graphColor (BLACK) ;

  graphTextHeight (0.75) ;
  for (i = 0 ; i <= 10 ; ++i)
    graphText (messprintf ("%3d%%",10*i),
	       1, topMargin + yLength * i / 10.0 - 0.25) ;
  graphTextHeight (0) ;
}

/***********************************************/

static void cMapDrawBoxFinish (int ibox, BOX *box)
{
  graphBoxEnd () ;
  if (box->seg && box->seg->flag & FLAG_HIGHLIGHT)
    graphBoxDraw (ibox, BLACK, MAGENTA) ;
  else
    graphBoxDraw (ibox, BLACK, box->col) ;
}

static void cMapAddNeigh (LOOK look, BOX *box, int ibox)
{
  if (!box->neigh)
    { array(look->neighbours, arrayMax(look->neighbours), int) = 0 ;
      box->neigh = arrayMax(look->neighbours) ;
    }
  array(look->neighbours, arrayMax(look->neighbours), int) = ibox ;
}

/***********************************************/

static MENUOPT buttonOpts[] = {
  cMapWhole, "Whole", 
  cMapZoomIn, "Zoom In",
  cMapZoomOut, "Zoom Out",
  cMapClear, "Clear",
  0, 0} ;

static void cMapDraw (LOOK look, KEY curr)
{
  int iseg, ibox ;
  SEG *seg ;
  float screenMin, screenMax ;
  int x, xs ;
  float y, y1s, y2s ;
  BUMP keyBump, inSituBump ;
  BOX *box ;

  look->neighbours = arrayReCreate (look->neighbours, 32, int) ;
  look->boxes = arrayReCreate (look->boxes, 64, BOX) ;
  look->activeBox = 0 ;

  if (!isPapDisp)
    { graphClear () ;
      graphColor (BLACK) ;
      getNxNy () ;
      if (yLength < 6)
	{ messout ("Sorry, this window is too small for a chromo map") ;
	  return ;
	}
      drawScale (look) ;
      drawChromosome (look) ;
      graphText (name(look->key), 1, 0.7) ;
    }
  

  screenMin = GRAPH2MAP(look, topMargin) ;
  screenMax = GRAPH2MAP(look, ny - bottomMargin) ;

  keyBump = bumpCreate ((nx-xItem)/symbolSize) ;
  inSituBump = bumpCreate (nx-xInSitu) ;

  for (iseg = 0 ; iseg < arrayMax(look->segs) ; ++iseg)
    { seg = arrp(look->segs, iseg, SEG) ;
      if (seg->x2 < screenMin || seg->x1 > screenMax)
	continue ;
      ibox = graphBoxStart () ;
      box = arrayp(look->boxes, ibox, BOX) ;
      box->seg = seg ;
      box->neigh = 0 ;
      switch (class(seg->key))
	{
	case _VContig:
	  graphRectangle (xContig - 0.5, MAP2GRAPH(look, seg->x1), 
			  xContig + 0.5, MAP2GRAPH(look, seg->x2)) ;
	  box->col = YELLOW ;
	  cMapDrawBoxFinish (ibox, box) ;
	  break ;
	case _VClone:
	  x = 0 ;
	  y = MAP2GRAPH(look,0.5*(seg->x1+seg->x2)) ;
	  if (isPapDisp)
	    { bumpItem (keyBump, 2, 0.025, &x, &y) ;
	      graphLine (xItem + x, y, xItem + x + 2, y) ;
	    }
	  else
	    { bumpItem (keyBump, 1, symbolSize * 0.75, &x, &y) ;
	      graphRectangle (xItem + x*symbolSize, y, 
			      xItem + (x+0.8)*symbolSize, y + 0.6*symbolSize) ;
	    }
	  box->col = LIGHTBLUE ;
	  cMapDrawBoxFinish (ibox, box) ;
	  if (!isPapDisp && (seg->flag & FLAG_INSITU))
	    { xs = 0 ;
	      y1s = MAP2GRAPH(look, look->max * seg->p1) ;
	      y2s = MAP2GRAPH(look, look->max * seg->p2) ;
	      bumpItem (inSituBump, 1, y2s - y1s + 1.0, &xs, &y1s) ;
	      xs += xInSitu ;
	      graphLine (xs, y1s, x + xItem + 0.8, y) ;
	      graphLine (xs, y2s, x + xItem + 0.8, y + 0.6) ;
	      ibox = graphBoxStart() ;
	      cMapAddNeigh (look, box, ibox) ; /* to old box */
	      box = arrayp(look->boxes, ibox, BOX) ;
	      box->seg = seg ;
	      box->neigh = 0 ;
	      cMapAddNeigh (look, box, ibox-1) ;
	      graphRectangle (xs, y1s, xs + 0.5, y2s) ;
	      box->col = LIGHTBLUE ;
	      cMapDrawBoxFinish (ibox, box) ;
	    }
	  break ;
	case _VGene:
	  x = xContig - 2.0 - 0.5*strlen(name(seg->key)) ;
	  y = MAP2GRAPH(look,0.5*(seg->x1+seg->x2)-0.3) ;
	  graphTextHeight (0.7) ;
	  graphText (name(seg->key), x, y) ;
	  graphTextHeight (0.0) ;
	  box->col = WHITE ;
	  cMapDrawBoxFinish (ibox, box) ;
	  break ;
	default:
	  graphBoxEnd() ;
	}
    }

  bumpDestroy (keyBump) ;
  bumpDestroy (inSituBump) ;

  if (!isPapDisp)
    graphButtons (buttonOpts, 5, 0.5, nx) ; /* at end of mapDraw() */
  else
    drawScale (look) ;

  graphRedraw () ;
}

/*********************************************************/
/********************* conversion code *******************/

#define PMAP_FAC  (2048.0 / 1000000.0)

static int cMapOrder(void *a, void *b) /* for arraySort() call */
{
  SEG *seg1 = (SEG*)a, *seg2 = (SEG*)b ;
  float diff = seg1->x1 - seg2->x1 ;

  if (diff > 0)
    return 1 ;
  else if (diff < 0)
    return -1 ;
  else
    return 0 ;
}

/********************/

typedef struct
  { int pmap ;
    float vmap ;
    SEG *seg ;
    KEY key ;
  } BLIT ;

static int blitOrder(void *a1, void *a2) /* for arraySort() call */
{
  BLIT *b1 = (BLIT*)a1, *b2 = (BLIT*)a2 ;
  float diff = b1->vmap - b2->vmap ;

  if (diff > 0) return 1 ;
  else if (diff < 0) return -1 ;
  else return 0 ;
}

static BOOL cMapConvert (LOOK look)
{
  KEY contig, clone, key ;
  OBJ Chrom, Contig, Clone ;
  int i, min, max, x1, x2 ;
  float vaxmap, total = 0.0, end ;
  SEG *seg ;
  int iseg = 0 ;
  BLIT *blit, *blitOld ;
  int iblit = 0 ;
  static Array blits = 0 ;

  if (!(Chrom = bsCreate (look->key)))
    return FALSE ;

  if (lexReClass (look->key, &key, _VpMap))
    { arrayDestroy (look->segs) ;
      if (look->segs = arrayGet (key, SEG, "k4fi"))
	{ end = 0.0 ;
	  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
	    { seg = arrp(look->segs, i, SEG) ;
	      if (seg->x2 > end) end = seg->x2 ;
	    }
	  look->max = end ;
	  return TRUE ;
	}
    }

  printf ("\nPhysical map of chromosome %s\n", name(look->key)) ;

  look->segs = arrayReCreate (look->segs, 64, SEG) ;
  blits = arrayReCreate (blits,64,BLIT) ;

  if (bsGetKey (Chrom, _Contig, &contig)) do
    { if (!(Contig = bsCreate (contig)))
	continue ;
      seg = arrayp (look->segs, iseg, SEG) ;
      seg->key = contig ;
      seg->flag = 0 ;
      min = 1000000 ;
      max = -1000000 ;
      if (bsGetKey (Contig, _Clone, &clone)) do
	{ if (!(Clone = bsCreate (clone)))
	    continue ;
	  if (bsGetKey (Clone, _pMap, &key) &&
	      bsGetData (Clone, _bsRight, _Int, &x1) &&
	      bsGetData (Clone, _bsRight, _Int, &x2))
	    { if (x1 < min) min = x1 ;
	      if (x2 > max) max = x2 ;
	      if (bsGetData (Clone, _Vaxmap, _Float, &vaxmap))
		{ blit = arrayp (blits, iblit, BLIT) ;
		  blit->pmap = 0.5 * (x1 + x2) ;
		  blit->vmap = vaxmap ;
		  blit->seg = seg ;
		  blit->key = clone ;
		  ++iblit ;
		}
	    }
	  bsDestroy (Clone) ;
	} while (bsGetKey (Contig, _bsDown, &clone)) ;
      seg->p1 = min ;
      seg->p2 = max ;
      seg->x1 = 0.0 ;
      if (max > min)
	seg->x2 = (max-min) * PMAP_FAC ;
      else
	seg->x2 = 0.0 ;
      total += seg->x2 ;
      printf ("Contig %s\t%8.4f\n", name(contig), seg->x2) ;
      bsDestroy (Contig) ;
      ++iseg ;
    } while (bsGetKey (Chrom, _bsDown, &contig)) ;

  bsDestroy (Chrom) ;

  printf ("Total length of %d contigs is %8.4f\n", iseg, total) ;

  arraySort (blits, blitOrder) ;
  blitOld = 0 ;
  end = 0.0 ;
  for (iblit = 0 ; iblit < arrayMax(blits) ; ++iblit)
    { blit = arrp(blits, iblit, BLIT) ;
      seg = blit->seg ;
      printf ("%4.1f\t%s\t%d\t%s\n", blit->vmap, 
	      name(seg->key), blit->pmap, name(blit->key)) ;
      if ((!blitOld || blitOld->seg != seg) &&
	  !(seg->flag & FLAG_PROCESSED))
        { end += 0.1 ;
	  seg->x1 = end ;
	  end += seg->x2 ;
	  seg->x2 = end ;
	  end += 0.1 ;
	  seg->flag |= FLAG_PROCESSED ;
	}
      blitOld = blit ;
    }
  look->max = end ;
  
  arraySort (look->segs, cMapOrder) ;
  lexaddkey (name(look->key), &key, _VpMap) ;
  arrayStore (key, look->segs, "k4fi") ;
  return TRUE ;
}

static void cMapAdd (void)	/* add keys from active keyset */
{ 
  int i, j, k, x1, x2 ;
  int iseg, ictg ;
  SEG *seg, *ctgSeg ;
  void *dummy ;
  KEY key, contig, clone ;
  OBJ obj, Yac, Clone ;
  KEYSET keySet = 0 ;
  static Array units = 0 ;
  static Array pos, map ;
  GRIDMAP *grid ;
 
  LOOKGET("cMapAdd") ;

  if (!keySetActive(&keySet, &dummy))
    { messout("First select a keySet window, thank you.") ;
      return ;
    }

  units = arrayReCreate (units, 32, BSunit) ;
  pos = arrayReCreate (pos, 16, GRIDMAP) ;
  map = arrayReCreate (map, 8, GRIDMAP) ;
  look->ctgAss = assReCreate (look->ctgAss) ;

  for (iseg = 0 ; iseg < arrayMax(look->segs) ; ++iseg)
    { seg = arrp(look->segs, iseg, SEG) ;
      if (class(seg->key) == _VContig)
	assInsert (look->ctgAss, (void*)seg->key, (void*)iseg) ;
    }

  iseg = arrayMax (look->segs) ;
  for (i = 0 ; i < keySetMax(keySet) ; ++i)
    { key = arr(keySet, i, KEY) ;
      switch (class (key))
	{
	case _VGene:
	  if (obj = bsCreate (key))
	    { if (bsGetKey (obj, _Clone, &clone) && (Clone = bsCreate (clone)))
		{ if (bsGetKey (Clone, _pMap, &contig) &&
		      bsGetData (Clone, _bsRight, _Int, &x1) &&
		      bsGetData (Clone, _bsRight, _Int, &x2) &&
		      assFind (look->ctgAss, (void*)contig, &ictg))
		    { seg = arrayp (look->segs, iseg, SEG) ;
		      ctgSeg = arrp (look->segs, ictg, SEG) ;
		      seg->key = key ;
		      seg->x1 = ctgSeg->x1 + PMAP_FAC*(x1-ctgSeg->p1) ;
		      seg->x2 = ctgSeg->x1 + PMAP_FAC*(x2-ctgSeg->p1) ;
		      seg->flag = 0 ;
		      ++iseg ;
		    }
		  bsDestroy (Clone) ;
		}
	      bsDestroy (obj) ;
	    }
	  break ;
	case _VClone:
	  if (obj = bsCreate (key))
	    { if (bsGetKey (obj, _pMap, &contig) &&
		  bsGetData (obj, _bsRight, _Int, &x1) &&
		  bsGetData (obj, _bsRight, _Int, &x2) &&
		  assFind (look->ctgAss, (void*)contig, &ictg))
		{ seg = arrayp (look->segs, iseg, SEG) ;
		  ctgSeg = arrp (look->segs, ictg, SEG) ;
		  seg->key = key ;
		  seg->x1 = ctgSeg->x1 + PMAP_FAC*(x1-ctgSeg->p1) ;
		  seg->x2 = ctgSeg->x1 + PMAP_FAC*(x2-ctgSeg->p1) ;
		  seg->flag = 0 ;
		  if (bsGetData (obj, _In_Situ, _Int, &x1) &&
		      bsGetData (obj, _bsRight, _Int, &x2))
		    { seg->flag |= FLAG_INSITU ;
		      seg->p1 = x1/100.0 ; /* NB p is a float, x an int */
		      seg->p2 = x2/100.0 ;
		    }
		  ++iseg ;
		}
	      else if (bsFindTag (obj, _Hybridizes_to) &&
		       bsFlatten (obj, 2, units))
		{ arrayMax(pos) = k = 0 ;
		  for (j = 1 ; j < arrayMax(units) ; j += 2)
		    if (Yac = bsCreate(arr(units,j,BSunit).k))
		      { if (bsGetKey (Yac, _pMap, &contig) &&
			    bsGetData (Yac, _bsRight, _Int, &x1) &&
			    bsGetData (Yac, _bsRight, _Int, &x2) &&
			    assFind (look->ctgAss, 
				     (void*)contig, &ictg))
			  { grid = arrayp (pos, k, GRIDMAP) ;
			    grid->ctg = contig ;
			    grid->x = 0.5 * (x1+x2) ;
			    ++k ;
			  }
			bsDestroy (Yac) ;
		      }
		  if (arrayMax(pos))
		    { gridCluster (pos, map) ;
		      for (k = 0 ; k < arrayMax(map) ; ++k)
			{ grid = arrp (map, k, GRIDMAP) ;
			  if (!assFind (look->ctgAss, 
					(void*)grid->ctg, &ictg))
			    continue ;
			  if (!grid->x)
			    invokeDebugger () ;
			  seg = arrayp (look->segs, iseg, SEG) ;
			  ctgSeg = arrp (look->segs, ictg, SEG) ;
			  seg->key = key ;
			  seg->x1 = ctgSeg->x1 + 
			    PMAP_FAC*(grid->x - ctgSeg->p1) ;
			  seg->x2 = seg->x1 ; 
			  seg->flag = FLAG_HYB ;
			  ++iseg ;
			}
		    }
		}
	      bsDestroy (obj) ;
	    }
	  break ;
	}
    }

  arraySort (look->segs, cMapOrder) ;

  if (!isPapDisp)
    cMapDraw (look, 0) ;
}

static void cMapSubtract (void) 
{
  int i, j;
  SEG *seg, *seg2 ;
  void *dummy ;
  KEYSET keySet = 0 ;
 
  LOOKGET("cMapSubtract") ;

  if (!keySetActive(&keySet, &dummy))
    { messout("First select a keySet window, thank you.") ;
      return ;
    }
  for (seg = arrp(look->segs,0,SEG), i = arrayMax(look->segs) ; i-- ; seg++)
    if (keySetFind (keySet, seg->key, &j))
      seg->flag |= FLAG_DELETING ;
  
  for (seg = seg2 = arrp(look->segs,0,SEG), i = arrayMax(look->segs) ; i-- ; seg++)
    if (!(seg->flag & FLAG_DELETING))
      *seg2++ = *seg ;
  arrayMax(look->segs) = seg2 - arrp(look->segs,0,SEG) + 1 ;

  cMapDraw (look, 0) ;
}

static void cMapHighlight (void)
{
  int i, j ;
  SEG *seg ;
  void *dummy ;
  KEYSET keySet = 0 ;
 
  LOOKGET("cMapHighlight") ;

  if (!keySetActive(&keySet, &dummy))
    { messout("First select a keySet window, thank you.") ;
      return ;
    }
  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  while(i--)
    { seg->flag &= ~FLAG_HIGHLIGHT ;
      if (keySetFind (keySet, seg->key, &j))
	seg->flag |= FLAG_HIGHLIGHT ;
      seg++ ;
    }
  cMapDraw (look, 0) ;
}

/************* special function for cDNA paper display **********/

static void cdnaPaperDisplay (void)
{
  static char* chroms[] = {"I", "II", "III", "IV", "V", "X"} ;
  KEY key ;
  int i, xOff ;
  LOOKGET ("cdnaPaperDisplay") ;

  graphClear () ;
  graphColor (BLACK) ;
  isPapDisp = TRUE ;
  nx = 100 ;
  ny = 100 ;
  look->mag = 1.0 ;
  for (i = 0 ; i < 6 ; ++i)
    { lexword2key (chroms[i], &key, _VChromosome) ;
      look->key = key ;
      if (!cMapConvert (look))
	messcrash ("lost a chromosome!") ;
      cMapAdd () ;		/* get contents of current keyset */
      look->centre = 0.0 ;
      yCentre = 3.5 + 22 * (i / 3) ;
      xOff = -7 + 20 * (i % 3) ;
      xContig += xOff ; xItem += xOff ;
      graphTextHeight (1.5) ;
      graphText (chroms[i], xContig-7, MAP2GRAPH(look,-1)) ;
      graphTextHeight (0.0) ;
      cMapDraw (look, 0) ;
      xContig -= xOff ; xItem -= xOff ;
    }

  isPapDisp = FALSE ;
}
