/*  File: treedisp.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: display/editor for generic tree objects
 * Exported functions:
 	treeDisplay
 * HISTORY:
 * Last edited: Apr  2 20:40 1992 (mieg)
 * * Feb 11 13:09 1992 (mieg): Introduced graphCompletionEntry
 * Created: Wed Oct 16 17:41:46 1991 (rd)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "array.h"
#include "bs_.h"
#include "bs.h"
#include "dump.h"
#include "graph.h"
#include "key.h"
#include "lex.h"
#include "systags.wrm"
#include "sysclass.wrm"
#include "biblio.h"
#include "session.h"
#include "display.h"
#include "disptype.wrm"
#include "pick.h"

extern BOOL ACEDB_SU ;
typedef struct LOOKSTUFF
  { int   magic ;        /* == MAGIC */
    KEY   key ;
    OBJ   objet ;
    Graph graph ;
    int   activebox ;
    Array content ;      /* content of each box: BS */
    Associator bs2box ;
    BS    bsWait ;	/* for use during update */
    char* text ;	/* for typing in things */
    int class ;         /* for completion mechanism */
  } *LOOK ;


#define MAGIC  27182

static FREEOPT treeMenu[]=
            { 7,"Tree display",
             99,"Quit",
             98,"Help",
             13,"Print",
              9,"Update",
	     23,"Biblio",
              5,"Tree",
	     14,"Text dump"
            } ;

static FREEOPT updateMenu[]=
            { 5,"Tree Edit",	/* changed to 7 if ACEDB_SU */
             99,"Quit",
             98,"Help",
	      3,"Save",
	      7,"Add Comment",
	      10,"Cancel",
	      8,"Delete",
	      11,"Edit"
            } ;

extern void  lookBiblio	       (OBJ objet, KEY k) ;

static void  lookDestroy       (void) ;
static void  lookMenu 	       (KEY k) ;
static void  lookPick          (int k) ;
static void  lookKbd           (int k) ;
static void  lookDraw          (LOOK look);
static void  updatePick        (int k) ;
static void  updateKbd	       (int k) ;
static void  defuse            (BS bs) ;
static void  addComment        (char *string) ;
static void  addData           (char *string) ;
static void  addKeyText        (char *string) ;
static void  addKey            (KEY key) ;
static void  editEntry	       (char *text) ;
static void  treeDump          (LOOK look) ;

#define LINE_LENGTH 40
#define NORMAL_FCOL BLACK
#define NORMAL_BCOL WHITE
#define SELECT_FCOL WHITE
#define SELECT_BCOL BLACK

#define LOOKGET(name)     LOOK look ; \
     			  if (!graphAssFind (treeDisplay,&look)) \
		            messcrash ("%s can't find graph",name) ; \
			  if (!look) \
                            messcrash ("%S received a null pointer",name) ; \
                          if (look->magic != MAGIC) \
                            messcrash ("%s received a wrong pointer",name)

/**************************************************************/

BOOL treeDisplay (KEY key, KEY from, BOOL isOldGraph)
{
  LOOK look=(LOOK)messalloc(sizeof(struct LOOKSTUFF));

  look->magic = MAGIC;
  look->key = key;
  if (pickType (key) != 'B' || !(look->objet = bsCreate(key)))
    goto abort ;
  look->content = arrayCreate(32,BS) ;

  if (isOldGraph)
    { lookDestroy () ;
      graphClear () ;
      graphGoto (0,0) ;
      graphRetitle (name (key)) ;
    }
  else
    { if (!displayCreate (TREE)) 
	goto abort ;
      graphRetitle (name(key)) ;
      graphRegister (DESTROY, lookDestroy) ;
      graphRegister (PICK, (GraphFunc)lookPick) ;
      graphRegister (KEYBOARD,(GraphFunc) lookKbd) ;
      graphFreeMenu (lookMenu, treeMenu) ;
    }

  look->graph = graphActive() ;
  graphAssociate (treeDisplay,look) ;

  look->content = arrayCreate(32,BS);
  look->bs2box = assCreate () ;

  look->activebox = 0 ;
  lookDraw (look) ;

  return TRUE ;

abort :
  free (look) ;
  return FALSE ;
}

/************************************************************/

static void lookDestroy (void)
{
  LOOKGET("lookDestroy") ;

  if (isCacheModified(look->objet->x) &&
      graphQuery (messprintf("%s not saved, Save ?",name(look->key))))
    { defuse (look->objet->root) ;
      bsSave (look->objet);
    }
  else
    bsDestroy (look->objet);

  arrayDestroy (look->content) ;
  assDestroy (look->bs2box) ;
  if (look->text)
    messfree (look->text) ;

  look->magic = 0 ;
  free (look) ;

  graphAssRemove (treeDisplay) ;
}

/*********************************************************/

static void lookPick (int box)
{
  LOOKGET("lookPick") ;

  displayPreserve () ;

  if (!box)
    return ;
  else if ( box >= arrayMax (look->content))
    { messerror("LookPick received a wrong box number");
      return ;
    }

  if (box == look->activebox)         /* a second hit - follow it */
    { 
      KEY key = arr(look->content,look->activebox,BS)->key ;
      
      if (key == _Pick_me_to_call)
	externalDisplay(look->key) ;
      else
	display (key,look->key,0) ;
    }
  else                              /* change now */
    {
      if (look->activebox > 0)
	graphBoxDraw(look->activebox, NORMAL_FCOL, NORMAL_BCOL);
      look->activebox = box ;
      graphBoxDraw(look->activebox, SELECT_FCOL, SELECT_BCOL);
    }
}

/*****************/

static void lookKbd (int k)
{
  BS bs ;
  int box ;
  LOOKGET("lookKbd") ;

  if (isDisplayBlocked())
    return ;

  displayPreserve () ;

  if (!look->activebox) return ;
  graphBoxDraw(look->activebox, NORMAL_FCOL, NORMAL_BCOL);
  if (!(bs = arr(look->content,look->activebox,BS)))
    return ;
  switch (k)
    {
    case LEFT_KEY :
      while (bs->up->down == bs)
	bs = bs->up ;
      if (bs->up->up)
	bs = bs->up ;
      break ;
    case RIGHT_KEY :
      if (bs->right)
	bs = bs->right ;
      break ;
    case DOWN_KEY :
      if (bs->down)
	bs = bs->down ;
      else while (bs->up->down == bs)
	bs = bs->up ;
      break ;
    case UP_KEY :
      if (bs->up->down == bs)
	bs = bs->up ;
      else while (bs->down)
	bs = bs->down ;
      break ;
    case SPACE_KEY :
      if (bs->right)
	bs = bs->right ;
      else if (bs->down)
	bs = bs->down ;
      else
	while (bs->up->up)
	  { while (bs->up->down == bs)
	      bs = bs->up ;
	    if (!bs->up->up)
	      break ;
	    bs = bs->up ;
	    if (bs->down)
	      { bs = bs->down ;
		break ;
	      }
	  }
      break ;
    }

  assFind (look->bs2box, bs, &box) ;
  graphBoxDraw(box, SELECT_FCOL, SELECT_BCOL);
  if (box != look->activebox &&
      class(bs->key) && iskey(bs->key) == 2)
    display (bs->key,look->key,0) ;

  look->activebox = box ;
}

/*************************************************************/

void treeUpdate (void)
{
  LOOKGET("lookMenu") ;

  if(!isWriteAccess())
    { messout("Sorry, you do not have Write Access");
      return ;
    }
  if (!KEYKEY(look->key))
    { messout ("Sorry, you can not update models this way") ;
      return ;
    }
  if (class(look->key) == _VSession)
    { messout ("Sorry, you can not update a session object") ;
      return ;
    }
  bsDestroy (look->objet) ;
  if (!(look->objet = bsUpdate (look->key)))
    { look->objet = bsCreate (look->key) ;
      lookDraw (look) ;
      return ;
    }
  cacheMark (look->objet->x) ;
  if (ACEDB_SU)
    { updateMenu->key = 7 ;
      bsFuseModel (look->objet) ;
    }
  look->activebox = 0 ;
  lookDraw (look) ;
  graphRegister (PICK,(GraphFunc) updatePick) ;
  graphRegister (KEYBOARD, (GraphFunc)updateKbd) ;
  graphFreeMenu (lookMenu, updateMenu) ;
  graphHelp ("Update") ;
}

static void lookMenu (KEY k)
{
  BS bs ;
  LOOKGET("lookMenu") ;

  displayPreserve () ;	/* if at end then check gActive is correct first */

  switch ((int) k)
    {

    case 3 :                        /* save */
      if (look->bsWait)
	{ messout ("Finish what you have started first, or cancel") ;
	  return ;
	}
      defuse (look->objet->root) ;
      bsSave (look->objet) ;
      look->objet = bsCreate (look->key) ;
      look->activebox = 0 ;
      lookDraw (look) ;
      graphRegister (PICK,(GraphFunc) lookPick) ;
      graphRegister (KEYBOARD,(GraphFunc) lookKbd) ;
      graphFreeMenu (lookMenu, treeMenu) ;
      graphHelp ("Tree") ;
      break ;
      
    case 5 :
      if (!look->activebox)
	{ messout ("You must first select a node") ;
	  return ;
	}
      bs = arr(look->content,look->activebox,BS) ;
      if (iskey(bs->key) == 2 )
	{ 
	  if(pickType(bs->key) == 'B')
	    display(bs->key, look->key, TREE) ;
	  else
	    messout("Cannot display %s as a tree", name(bs->key)) ;
	}
      else
	messout("No data associated to %s", name(bs->key)) ;
	
      break ;

    case 7 :		/* add comment */
      if (look->bsWait)
	{ messout ("Finish what you have started first, or cancel") ;
	  return ;
	}
      if (!look->activebox)
	{ messout ("You must first select a node to add to (single-click)") ;
	  return ;
	}
      bs = arr(look->content,look->activebox,BS) ;
      if (bs->size & 1)
	{ messout (
 "You must add comments to the original object (black and white stuff)") ;
	  return ;
	}
      while (class(bs->key) == _VComment ||
	     class(bs->key) == _VUserComment)
	bs = bs->up ;	/* don't add comments to comments */
      bs->size |= 8 ;	/* flag for lookDraw() */
      look->bsWait = bs ;
      lookDraw (look) ;
      break ;

    case 8 :		/* delete */
      if (look->bsWait)
	{ messout ("Finish what you have started first, or cancel") ;
	  return ;
	}
      if (!look->activebox)
	{ messout ("You must first select a node to prune (single-click)") ;
	  return ;
	}
      bs = arr(look->content,look->activebox,BS) ;
      if (bs->size & 1)
	{ messout ("You can only delete parts of the original object") ;
	  return ;
	}
      look->objet->curr = bs ;
      look->objet->modCurr = (bs->bt) ? bs->bt->bsm : 0 ;
      defuse (look->objet->root) ;
      bsRemove (look->objet) ;
      bsFuseModel (look->objet) ;
      lookDraw (look) ;
      break ;

    case 9 :		/* update */
      treeUpdate () ;
      break ;

    case 10 :		/* cancel operation */
      if (!look->bsWait)
	return ;
      if (look->bsWait->size & 16)
	displayUnBlock() ;		/* clears display block */
      look->bsWait->size &= ~60 ;   /* clears edit and fake node flags */
      look->bsWait = 0 ;
      lookDraw (look) ;
      break ;

    case 11:		/* edit */
      if (look->bsWait)
	{ messout ("Finish what you have started first, or cancel") ;
	  return ;
	}
      if (!look->activebox)
	{ messout ("You must first select a node to edit (single-click)") ;
	  return ;
	}
      bs = arr(look->content,look->activebox,BS) ;
      if (bs->size & 1)
	{ messout ("You can only edit parts of the original object") ;
	  return ;
	}
      if (bsIsTag(bs))
	{ messout ("You can not edit tags - only object names or data") ;
	  return ;
	}
      look->bsWait = bs ;
      bs->size |= 32 ;
      lookDraw (look) ;
      break ;

    case 13 :
      graphPS ("tree") ; break ;

    case 14:
      treeDump (look) ; break ;

    case 23 :
      biblioKey (look->key) ; break ;

    case 98 :
      help() ; break ;

    case 99 :
      graphDestroy () ; break ;
    }
}

/****************************************************/

static void treeDump (LOOK look)
{
  FILE *fil ;
  static char dirName[80] = "." ;
  static char fileName[24] = "object" ;

  if (!(fil = filqueryopen (dirName, fileName, "txt", "w")))
    { messout ("failed to open text dump file") ;
      return ;
    }
  fprintf (fil,"// data dumped from tree display\n\n") ;
  dumpKey (look->key, fil) ;
  fclose (fil) ;
  messout ("Object %s written to %s/%s\n", 
	   name(look->key), dirName, fileName) ;
}

/*************************************************************/
/*************** drawing package *****************************/
/*******************************************************/

static LOOK drawLook ;
static int xmax ;
static BOOL isModel ;

extern int ksetClassComplete(char *text, int len, int class) ;
static void classComplete(char *cp, int len)
{
  static previousClass = -1 ;
  
  if (!drawLook->class)
    return ;
  if (previousClass != drawLook->class)
    ksetClassComplete(0,0,0) ;
  else  
    ksetClassComplete(cp, len, drawLook->class) ;
  previousClass = drawLook->class ;
}

static int drawTextEntry (int class, int len, int x, int y, void (*fn)(char*))
{
  int box = graphCompletionEntry (classComplete, drawLook->text, len, x, y, fn) ;
  array (drawLook->content, box, BS) = 0 ;
  if (x + len > xmax)
    xmax = x + len ;
  drawLook->class = class ;
  return y+1 ;
}

static char* drawBStext (BS bs)
{
  if (isModel)
    return name (bs->key) ;

  if (bs->key <= _LastC)
    return bsText(bs) ;
  else if (bs->key == _Int)
    return messprintf ("%d", bs->n.i) ;
  else if (bs->key == _Float)
    return messprintf ("%g", bs->n.f) ;
  else
    return name (bs->key) ;
}

char *expandText (KEY key)
{
  KEY tag = pickExpandTag(key) ;
  OBJ obj ;
  char *text = 0 ;

  if ( tag &&
      (obj = bsCreate(key)))
    { if (bsGetKeyTags (obj,tag,0))
	text = drawBStext (obj->curr) ;
      bsDestroy (obj) ;
    }

  return text ;
}

static int drawBS (BS bs, int x, int y)	/* recursive routine */
{
  int yMe ;
  int xPlus = 0 ;
  char *text, *cp ;
  int box ;

  if (bs->size & 4)
    { y = drawTextEntry (0, 32, x, y, addData) ;
    }
  else if (bs->size & 16)
    { y = drawTextEntry (class(bs->key), 255, x, y, addKeyText) ;
      displayBlock (addKey,0) ;
    }

  if (bs->size & 64)
    text = name (bs->key - 1 ) ;  /* use data tag names in model sections */
  else if (bs->size & 1)
    text = name (bs->key) ;  /* use data tag names in model sections */
  else if (!(text = expandText (bs->key)))
    text = drawBStext (bs) ;

  yMe = y ;

  if (bs->size & 32)		/* edit entry */
    { strcpy (drawLook->text, text) ;
      y = drawTextEntry (class(bs->key), 255, x, y, editEntry) ;
    }
  else
    { box = graphBoxStart() ;
      array (drawLook->content, box, BS) = bs ;
      assInsert (drawLook->bs2box, bs, (void*) box) ;
      uLinesText (text, LINE_LENGTH) ;
      if (cp = uNextLine(text))
	{ graphText (cp,x,yMe++) ;	/* write out this node */
	  if (strlen(cp) > xPlus)
	    xPlus = strlen(cp) ;
	}
      while (cp = uNextLine(text))	/* indent following lines */
	{ graphText (cp,x+2,yMe++) ;
	  if (strlen(cp)+2 > xPlus)
	    xPlus = strlen(cp)+2 ;
	}
      graphBoxEnd() ;
    }

  if (box == drawLook->activebox)
    graphBoxDraw (box, WHITE, (bs->size&3) ? 
		  ((bs->size & 64) ? GREEN : BLUE) : 
		              (class(bs->key) == _VUserComment) ?
		                             DARKGRAY : BLACK) ;
  else
    graphBoxDraw (box, BLACK, (bs->size&3) ? 
		  ((bs->size & 64) ? LIGHTGREEN : CYAN) : 
		              (class(bs->key) == _VUserComment) ?
		                             LIGHTGRAY : WHITE) ;

  xPlus += x ;
  if (xPlus > xmax)
    xmax = xPlus ;
  xPlus = xPlus + 6 - xPlus%4 ;

  if (bs->size & 8)	/* add comment box */
    { y = drawTextEntry (_VText, 255,xPlus,y,addComment) ;
    }

  if (bs->right)
    y = drawBS (bs->right, xPlus, y) ;  /* to the right at same y */

  if (yMe > y)

    y = yMe ;

  if (bs->down)
    y = drawBS (bs->down, x, y) ;		/* below at new y location */

  return y ;
}

static void lookDraw (LOOK look)
{
  int ymax = 1 ;

  graphClear () ;
  arrayMax(look->content) = 0 ;
  assClear (look->bs2box) ;

  graphText (name(look->key),0,0) ;

  isModel = (KEYKEY(look->key) == 0) ;

  if (look->objet->root && look->objet->root->right)
    { drawLook = look ;
      if (!drawLook->text)
	drawLook->text = (char *) messalloc (256) ;
      else
	memset(drawLook->text,0,(mysize_t)256) ;
      xmax = 0 ;
      ymax = drawBS (look->objet->root->right,2,2) ;
    }

  graphTextBounds (xmax,ymax) ;
  graphRedraw() ;

  if (look->activebox >= arrayMax(look->content))
    look->activebox = 0 ;

  isModel = FALSE ;
}

/************************************************************/
/************************************************************/

/* Updating:
   The plan is to build a new tree containing the union of the model
   and the current object.  We will rebuild every time an action changes the
   structure.  
   Parts of the model not yet represented in the object will be coloured
   blue.  The XREF, UNIQUE etc bits of the model will not be shown. Class
   slots (?Class) will appear in cyan at the bottom of each column of objects,
   so you can add new members (or replace if unique).
   slots (#Class) will not be shown.
   If you double-click on a cyan tag it will be added to your object.
   If you double click on a (cyan) ?Class then you will be prompted for a name,
   but you can also pick something from that class elsewhere in the system.  If
   you give a name that is a new lex entry it will ask for confirmation before
   adding.
   In update mode you can not display any new objects, nor get the biblio
   etc.

   We will use bs->size to hold the information about whether an entry is
   in the model or not.  Adding and subtracting the model is done by two
   routines bsFuseModel() in bssubs.c and defuse() below.
   Codes for ->size are:
   	1	part of model
	2	unique part of original tree
	   so both 1 and 2 are painted cyan
	4	fake node to add data (TextEntry)
	8	fake node to add comment (TextEntry)
	16	fake node to add key (TextEntry and DisplayBlock)
	32      edit current node - rather than enter a new one
	64	special node to add Type (TextEntry and DisplayBlock)
	   all from 4 on are in fact mutually exclusive - there
	   should be at most one at any one time (set to bsWait).

   We disable the keyboard routine (used for data entry), and enable a new
   pick routine.  The menu should change to contain Save (returns to standard
   form), Delete, and Add comment.  We should replace Save in the standard
   menu by Update.  Perhaps the standard menu could also contain Add comment?
*/

static void fixTag (OBJ obj, BS bs) ;
static void fixType (OBJ obj, BS bs) ;

static void updatePick (int box)
{
  BS bs ;

  LOOKGET("updatePick") ;

  displayPreserve () ;
  if (!box)
    return ;
  else if (box >= arrayMax (look->content))
    { messerror("updatePick received a wrong box number %d",box);
      return ;
    }

  if (box == look->activebox)         /* a second hit - follow it */
    { bs = arr(look->content,look->activebox,BS) ;
      if (!bs)	      /* not a bs box, e.g. inside a textEntry */
	return ;
      if (look->bsWait)
	{ messout ("Finish what you are doing first (or cancel)") ;
	  return ;
	}
      if (isDisplayBlocked())	/* from different update window */
	{ messout ("First deal with the other object you are waiting on") ;
	  return ;
	}
      if (bs->size & 64)        /* a type */
	{ fixType (look->objet,bs) ;
	  lookDraw (look) ;
	}
      else if (bsIsTag(bs))
	{ fixTag (look->objet,bs) ;
	  lookDraw (look) ;
	}
      else if (!(bs->size & 1)) /* in original object */
	{ look->bsWait = bs ;
	  bs->size |= 32 ;	/* edit flag */
	  lookDraw (look) ;
	}
      else if (bsIsTag(bs->up) || !(bs->up->size & 1))
	{ if (bsIsTag(bs->up) && (bs->up->size & 1))
	    fixTag (look->objet,bs->up) ;
	  look->bsWait = bs ;
	  if (class(bs->key))
	    bs->size |= 16 ;	/* signal to lookDraw */
	  else
	    bs->size |= 4 ;
	  lookDraw (look) ;
	}
      else
        messout (
"Sorry - you must be next to a tag or an existing part of the object") ;
    }
  else                              /* change highlighted entry */
    {
      if (look->activebox > 0 &&
	  (bs = arr(look->content,look->activebox,BS)))
	if (bs->size & 3)
	  graphBoxDraw (look->activebox, BLACK,
			((bs->size & 64) ? LIGHTGREEN : CYAN) ) ;
	else
	  graphBoxDraw (look->activebox, BLACK, WHITE) ;
      look->activebox = box ;
      bs = arr(look->content,box,BS) ;
      if (!bs)		/* not a bs box */
	look->activebox = 0 ;
      else if (bs->size & 3)
	graphBoxDraw (look->activebox, WHITE, 
		      ((bs->size & 64) ? GREEN : BLUE) ) ;
      else
	graphBoxDraw (look->activebox, WHITE, BLACK) ;
    }
}

static void updateKbd (int k)
{
  BS bs ;
  int box ;
  LOOKGET("updateKbd") ;

  if (!look->activebox)
    return ;
  if (!(bs = arr(look->content,look->activebox,BS)))
    return ;

  switch (k)
    {
    case RETURN_KEY:
      updatePick (look->activebox) ;
      break ;
    case LEFT_KEY :
      while (bs->up->down == bs)
	bs = bs->up ;
      if (bs->up->up)
	bs = bs->up ;
      break ;
    case RIGHT_KEY :
      if (bs->right)
	bs = bs->right ;
      break ;
    case DOWN_KEY :
      if (bs->down)
	bs = bs->down ;
      else while (bs->up->down == bs)
	bs = bs->up ;
      break ;
    case UP_KEY :
      if (bs->up->down == bs)
	bs = bs->up ;
      else while (bs->down)
	bs = bs->down ;
      break ;
    case '\t': case SPACE_KEY :
      if (bs->right)
	bs = bs->right ;
      else if (bs->down)
	bs = bs->down ;
      else
	while (bs->up->up)
	  { while (bs->up->down == bs)
	      bs = bs->up ;
	    if (!bs->up->up)
	      break ;
	    bs = bs->up ;
	    if (bs->down)
	      { bs = bs->down ;
		break ;
	      }
	  }
      break ;
    default: return ;
    }

  if (assFind (look->bs2box, bs, &box)
      && box != look->activebox)
    updatePick (box) ;
}

/********************************/

static void defuse (BS bs)	/* recursive */
{
  BS temp ;

  bs->size = 0 ;

  while (bs->right && bs->right->size & 1)
    { temp = bs->right ;
      bs->right = bs->right->down ;
      if (bs->right)
	bs->right->up = bs ;
      temp->up = 0 ;
      temp->down = 0 ;
      bsTreePrune (temp) ;
    }
  if (bs->right)
    defuse (bs->right) ;

  while (bs->down && bs->down->size & 1)
    { temp = bs->down ;
      bs->down = bs->down->down ;
      if (bs->down)
	bs->down->up = bs ;
      temp->up = 0 ;
      temp->down = 0 ;
      bsTreePrune (temp) ;
    }
  if (bs->down)
    defuse (bs->down) ;
}

/***********************************/

static void fixTag (OBJ obj, BS bs)
{
  while (bs->size & 1)	/* code for belonging to model */
    { bs->size &= ~1 ;
      while (bs->up && bs->up->down == bs)	/* go to top of column */
	bs = bs->up ;
      if (bs->up)		/* go to entry in previous column */
	bs = bs->up ;
    }
}

/***********************************/

static void fixType (OBJ obj, BS bs)
{ extern void bsSubFuseType(OBJ obj, BS bs) ;

  fixTag(obj, bs->up) ;
  bsSubFuseType(obj, bs) ;
}

/*********************************/

static void addComment (char* text)
{
  LOOKGET("addComment") ;

  if (!look->bsWait)
    messcrash ("screwup in addComment - no bs to add to") ;

  look->objet->curr = look->bsWait ;	/* point to add to */
  if (ACEDB_SU)
    bsAddComment (look->objet, text, 'C') ;
  else
    bsAddComment (look->objet, text, 'U') ;
  
  look->bsWait->size &= ~8 ;
  look->bsWait = 0 ;
  lookDraw (look) ;
}

static void addKey (KEY key)
{
  BS bs, bsm ;
  OBJ obj ;
  LOOKGET("addKey") ;

  if (!key)		/* cancellation */
    return ;

  if (!(bs = look->bsWait))
    messcrash ("screwup in addKey - bs is missing") ;

  if (pickXref(class(key)))	/* comments come via editEntry */
    { obj = look->objet ;
      if (!obj->xref) obj->xref = stackCreate (64) ;
      push (obj->xref, key, KEY) ;
      push (obj->xref, (_Quoted_in), KEY) ;
    }
  else 
    { if(!bs->bt || !(bsm = bs->bt->bsm))
	messcrash ("screwup in addKey - bsm is missing") ;
      look->objet->curr = bs ;
      look->objet->modCurr = bsm ;
      if (!bsTypeCheck (look->objet,key))
	{ messout ("Sorry - that key is not in the right class") ;
	  return ;
	}
    }

  bs->key = key ;
  bs->size &= ~1 ;			/* clear "model" flag */
  defuse (look->objet->root) ;	/* to correctly redisplay model */
  bsFuseModel (look->objet) ;
  ++look->activebox ;

  bs->size &= ~48 ; /* must do this since display block is freed on return */
  look->bsWait = 0 ;
  lookDraw (look) ;
}

static void addKeyText (char *text)
{ 
  BS bs ;
  KEY key ;
  int table ;
  LOOKGET("addKeyText") ;
  
  if (!(bs = look->bsWait))
    messcrash ("Screwup in addKeyText - bs missing") ;

  table = class(bs->key) ;
  if (!lexword2key(text,&key,table))
    if (table != _VText && !graphQuery ("Unknown name - do you want to create a new object? "))
      { while(*text) 
	  *text++ = 0 ;  /* necessary to set totally to 0 for graphTextEntry */
	return ;
      }
    else
      lexaddkey (text,&key,table) ;

  display (key,0,0) ;		/* calls addKey via displayBlock() */
}

static void addData (char* text)
{
  BS bs ;
  int i ;
  float f ;
  LOOKGET("addData") ;

  if (!(bs = look->bsWait))
    messcrash ("screwup in addData - no bs to add to") ;

  if (bs->key < _LastC)
    { bs->bt->cp = (char*) messalloc (strlen (text) + 1) ;
      strcpy (bs->bt->cp, text) ;
    }
  else if (bs->key == _Int)
    if (sscanf (text,"%d",&i))
      bs->n.i = i ;
    else
      { messout ("Sorry - not a good integer") ; return ; }
  else if (bs->key == _Float)
    if (sscanf (text,"%f",&f))
      bs->n.f = f ;
    else
      { messout ("Sorry - not a good floating point number") ; return ; }
  else
    messcrash ("Bad data type %d in treedisp addData",bs->key) ;

  bs->size &= ~37 ;		/* only get here if we added OK */
  look->bsWait = 0 ;
  defuse (look->objet->root) ; /* to correctly redisplay model */
  bsFuseModel (look->objet) ;
  ++look->activebox ;
  lookDraw (look) ;
}

/*******************/

static void editEntry (char *text)
{
  BS bs, bsm ;
  KEY key ;
  OBJ obj ;
  int table ;
  LOOKGET("editEntry") ;

  if (!(bs = look->bsWait))
    messcrash ("screwup in editEntry - no bs to edit") ;
  if (bs->key < _LastN)
    addData (text) ;
  else
    { table = class(bs->key) ;
      if (!lexword2key(text,&key,table))
	if (!pickXref(table) && !graphQuery ("Unknown name - do you want to create a new object? "))
	  return ;
	else
	  lexaddkey (text,&key,table) ;

      if (pickXref(table))	/* especially comments! */
	{ obj = look->objet ;
	  if (!obj->xref) obj->xref = stackCreate (64) ;
	  push (obj->xref, bs->key, KEY) ;
	  push (obj->xref, (_Quoted_in | DELETE_BIT), KEY) ;
	}
      else 
	{ if (!bs->bt || !(bsm = bs->bt->bsm))
	    messcrash ("screwup in editEntry - bsm is missing") ;
	  if (bsm->n.key)		/* must delete XREF */
	    { obj = look->objet ;
	      if (!obj->xref) obj->xref = stackCreate (64) ;
	      push (obj->xref, bs->key, KEY) ;
	      push (obj->xref, (KEYKEY(bsm->n.key) | DELETE_BIT), 
		    KEY) ;
	    }
	}

      addKey (key) ;
    }
}

/************************************************************/
/************************************************************/
