/*  File: ksetdisp.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: Manipulate Key Sets.
      keyset windows have their own menus and  KEYSET. When they 
      are destroyed they (VERY IMPORTANT) destroy the keyset.
 * Exported functions:
 * HISTORY:
 * Last edited: Apr  2 18:34 1992 (mieg)
 * * Nov  5 18:27 1991 (mieg): I introduced keySetAlpha
 * * Oct 23 18:53 1991 (mieg): changed name dump to a keySetDump
 * Created: Wed Oct 23 18:52:00 1991 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "lex.h"
#include "graph.h"
#include "array.h"
#include "key.h"
#include "keyset.h"
#include "display.h"
#include "dump.h"

#include "a.h"
#include "bs.h"
#include "sysclass.wrm"
#include "session.h"     
#include "query.h"
#include "pick.h"
#include "biblio.h"

 extern void Bdump (KEY key) ;

	/* keySetShow () displays a KEYSET (a sorted array of KEY's) in 
	   multicolumn format, allowing the user to pick and move around 
	   with arrow keys.  Objects are displayed with default displaytype.

	   To initialise the system, call with handle==0.  It returns the new 
	   handle.  To reuse the graph call first with (0,handle) to
	   kill the alphabetic keyset. The aim of this is to
	   allow the user to change the displayed KEYSET.

	   keySetCurrent() allows the user to set the key directly (a bit
	   like pick).  It takes the key as argument, not the box number.
	   If the requested key is not in the list it returns 0, else the key.
	   If k==0 it returns the current key.

	   Although we can save to permanent objects (by the menu Save option)
	   and recover from them (via display() and keySetDispCreate()),
	   a keyset display window is NEVER attached to an object - it always
	   deals in copies of the permanent keyset.  This allows much greater
	   flexibility in changing the contents of a window, avoids tangles
	   with the cache, and is absolutely necessary.
	*/

#define NCOL 3
#define COLWIDTH 16
#define MAXOBS 1000

#define MAGIC 187641

typedef struct LOOKstruct
  { int   magic ;
    KEYSET keySet , keySetAlpha ;
    KEY   curr ;
    Graph graph ;
    Stack pages ;
    BOOL  isBlocking ;
    int   base, top ;		/* top is 1 + index of last displayed key */
  } *LOOK ;

static FREEOPT menuOptions[] =
            {19,"Key Set ?",
             99,"Quit",
             98,"Help",		
	    -1,"",
/*	     5,"Sort",	*/
             6,"Add Keys",
	     2,"Copy",
             1,"Save",
	    -1,"",
	    20,"AND",
	    21,"OR",
	    22,"XOR",
	    -1,"",
	     3,"Ace Dump",
	    31,"Asn Dump",
	     8,"Name Dump",
	     7,"FastA Dump",
	    -1,"",
            40,"Tree",
	     4,"Biblio",
            41,"Debug"
            } ;

static void localDestroy (void) ;
static void localPick (int box) ;
static void localKbd (int k) ;
static void localMenu (KEY k) ;
static void pageUp(void), pageDown(void) ;
static void* keySetShow2 (KEYSET keySet, LOOK look) ;
static void ksetAddKey (KEY key) ;
static void ksetDeleteKey (KEY key) ;
static void ksetUnBlock (void) ;

extern int keySetAlphaOrder(void *a,void *b) ;

#define LOOKGET(name)     LOOK look ; \
       			  if (!graphAssFind (keySetShow,&look)) \
		            messcrash ("graph not found in %s", name) ; \
			  if (!look) \
                            messcrash ("%s received a null pointer",name) ; \
                          if (look->magic != MAGIC) \
                            messcrash ("%s received a wrong pointer",name)

static LOOK selectedKeySet = 0 ;

/******************************/
/** fake biblio functions   **/
/** for model free kernel   **/
/******************************/

#ifndef BIBLIO

void biblioKey (KEY key) 
{ messout ("Sorry : Recompile with flag BIBLIO") ;
  return ;
}
void biblioKeySet (char *title, KEYSET s) 
{ messout ("Sorry : Recompile with flag BIBLIO") ;
  return ;
}

#endif

/******************************/

static void pageUp(void)
{
  LOOKGET ("pageUp") ;
  if (look->base)
    { look->base = pop(look->pages,int) ;
      keySetShow2 (look->keySet, look) ;
    }
  else
   messout ("Already on the top page!") ;
}

static void pageDown(void)
{
  LOOKGET ("pageDown") ;
  if (look->top < keySetMax(look->keySet))
    { push (look->pages,look->base,int) ;
      look->base = look->top ;
      keySetShow2 (look->keySet, look) ;
    }
  else
   messout ("Already on the bottom page!") ;
}

static void localResize (void)
{
  int curr ;
  LOOKGET("localResize") ;
  curr = look->curr ;
  keySetShow2 (look->keySet, look) ;
  if (curr > 1 && curr < look->top - look->base + 2)
    { look->curr = curr ;
      graphBoxDraw (curr, WHITE, BLACK) ;
    }
}

static void* keySetShow2 (KEYSET keySet, LOOK look)
{  
  int	i, ix, iy, dx, bad = 0 ,
     max = ( keySet ? keySetMax(keySet) : 0 ) ;
  char	*cp ;
  int   nx, ny ;

  graphClear () ;
  if (!look->isBlocking)	/* so as not to hide message */
    graphPop() ;
  graphFitBounds (&nx, &ny) ;

  if (look->base >= max)
    { stackClear (look->pages) ;
      look->base = 0 ;
    }
  iy = look->base ? 2 : 0 ;

  graphBoxStart () ; 
    graphText ("Selected KeySet",14,iy) ; 
  graphBoxEnd () ;
  if (look == selectedKeySet)
    graphBoxDraw (1,BLACK,LIGHTRED) ;
  else
    graphBoxDraw (1,WHITE,WHITE) ;

  ++iy ;
  ix = 0 ;
  if (max)
    { 
      if(!look->keySetAlpha)
        look->keySetAlpha = keySetAlphaHeap(keySet, look->base + 12 * ny) ;
      else
	if(keySetMax(look->keySetAlpha) < keySetMax(look->keySet) &&
	   keySetMax(look->keySetAlpha) < look->base + 3 * ny)
	  { keySetDestroy(look->keySetAlpha) ;
	    look->keySetAlpha 
	      = keySetAlphaHeap(look->keySet, keySetMax(look->keySet)) ;
	  }
    }

  for (i = look->base ; i < max ; i++)
    { if (iy >= ny-2)
	break ;
      cp = 0 ;
     if(!nextName(arr(look->keySetAlpha,i,KEY), &cp) || *cp == '\177')
       { bad ++ ;
	 continue ;
       }
      dx = 1 + (1 + strlen (cp)) / COLWIDTH ;
      if (ix && ix+dx > NCOL)
	{ ++iy ; ix = 0 ; }
      graphBoxStart () ;
      graphText (cp,ix*COLWIDTH, iy) ;
      graphBoxEnd () ;
      ix += dx ;
      if (ix >= NCOL)
	{ ++iy ; ix = 0 ; }
    }
  look->top = i ;
  if (look->base)
    graphButton ("Page up", pageUp, 19, 0.5) ;
  if (look->top < max)
    graphButton ("Page Down", pageDown, 19, ny - 1.5) ;

  look->keySet = keySet ;
  look->curr = 0 ;

  iy = look->base ? 2 : 0 ;
  graphText (messprintf ("%d items", max - bad),0,iy) ;

  graphRedraw () ;

  return look ;
}

void* keySetShow (KEYSET keySet, void* handle)
{
  LOOK  look = handle ;

  if (look && look->magic != MAGIC)
    messcrash("keySetShow called with corrupted handle") ;
  if (look && !graphActivate (look->graph))
    messout ("keySetShow lost its graph - taking over this one") ;

  if (!look)
    { look = (LOOK) messalloc (sizeof (struct LOOKstruct)) ;
      look->magic = MAGIC ;
      look->graph = graphActive() ;
      look->pages = stackCreate(32) ;
      graphRegister (DESTROY, localDestroy) ;
      graphRegister (MESSAGE_DESTROY, ksetUnBlock) ;
      graphRegister (PICK, (GraphFunc)localPick) ;
      graphRegister (KEYBOARD,(GraphFunc) localKbd) ;
      graphRegister (RESIZE, (GraphFunc)localResize) ;
      graphFreeMenu (localMenu, menuOptions) ;

      graphAssociate (keySetShow,look) ;
      if (!selectedKeySet)
	selectedKeySet = look ;
    }
  else
    { 
      if (look->magic != MAGIC)
	messcrash ("keySetShow called with bad handle") ;
      look->curr = 0 ;
      look->base = 0 ;
      stackClear(look->pages) ;
      keySetDestroy(look->keySetAlpha) ;
    }

  return keySetShow2(keySet, look) ;
}

/***********************************/

BOOL keySetDisplay (KEY key, KEY from, BOOL isOldGraph)
{
  KEYSET kset ;
  void*  handle ;

  if (!(kset = arrayGet (key,KEY,"k")))
    return FALSE ;

  if (!isOldGraph)
    displayCreate (DtKeySet) ;
  else
    localDestroy () ;
  graphRetitle (name(key)) ;
  
  handle = keySetShow (kset, 0) ;

  return (handle != 0) ;
}

/***********************************/

/* BEWARE: look->curr (a box) runs 2..keySetMax+1, not 0..max-1 */
/* box 0 is the background, 1 is the "selected" message */

KEY keySetCurrent (void *handle, KEY k)
{
  int i ;
  LOOK look = (LOOK) handle ;

  if (look->magic != MAGIC)
    messcrash ("keySetCurrent called with bad look handle") ;
  if (!k)
    return keySet(look->keySetAlpha,look->curr - 2 + look->base) ;
  if (!keySetFind (look->keySetAlpha,k,&i))
    return 0 ;
  if (i < look->base)
    { while ((look->base = pop(look->pages,int)) >= i) ;
      keySetShow2 (look->keySet, look) ;
    }
  else if (look->curr > 1)
    graphBoxDraw (look->curr,BLACK,WHITE) ;
  if (i < look->top)
    { look->curr = i-look->base+2 ;
      graphBoxDraw (look->curr,WHITE,BLACK) ;
      graphGoto (0,1+i%NCOL) ;
    }
    
  return k ;
}

/*************************************/
/*************************************/

static void localDestroy (void)
{
  LOOKGET("keySetDestroy") ;

  look->magic = 0 ;

  keySetDestroy (look->keySet) ;
  keySetDestroy (look->keySetAlpha) ;
    
  messfree (look) ;

  graphAssRemove (keySetShow) ;
}

/*************************************/

       /* selects keyset for mix, query etc */
void keySetSelect ()
{
  LOOKGET ("keySetSelect") ;

  if (    selectedKeySet 
      &&  selectedKeySet != look
      && selectedKeySet->magic == MAGIC
      && graphExists(selectedKeySet->graph))
    { Graph hold = graphActive () ;
      graphActivate (selectedKeySet->graph) ;
      
      graphBoxDraw (1,WHITE,WHITE) ;   /* box 1 is the message */
      graphActivate (hold) ;
    }
  graphBoxDraw (1,BLACK,LIGHTRED) ;
  selectedKeySet = look ;
}

/***************/

BOOL keySetActive(KEYSET *setp, void** lookp) 
{
  if (selectedKeySet && selectedKeySet->magic == MAGIC
      && graphExists(selectedKeySet->graph)
      && arrayExists(selectedKeySet->keySet))
    { *lookp = selectedKeySet ;
      * setp = selectedKeySet->keySet ;
      return TRUE ;
    }
  selectedKeySet = 0 ;
  *setp = 0 ; *lookp = 0 ;
  return FALSE ;
}

/***************/

/* BEWARE: look->curr (a box) runs 2..keySetMax+1, not 0..max-1 */
/* box 0 is the background, 1 is the "selected" message */

static void localPick (int box)
{
  LOOKGET("keySetPick") ;

  if (look != selectedKeySet)
    keySetSelect () ; 
  if (box <= 1)
    return ;
  if (box == look->curr)                  /* double click */
    { KEY key = keySet(look->keySetAlpha, 
		       look->curr - 2 + look->base) ;
      if (look->isBlocking)
	ksetDeleteKey (key) ;
      else 
	display (key,0,0) ;
    }
  else
    { if (look->curr > 1)
        graphBoxDraw (look->curr, BLACK, WHITE) ;
      look->curr = box ;
      graphBoxDraw (look->curr, WHITE, BLACK) ;
    }
}

/***********************************/

static void keySetMix(LOOK query, int operator)
{
  KEYSET s ;
  LOOK  q = selectedKeySet ;
  
  switch(operator)
    {
    case 20:
      s = keySetAND(q->keySet,query->keySet) ;
      break ;
    case 21:
      s = keySetOR(q->keySet,query->keySet) ;
      break ;
    case 22:
      s = keySetXOR(q->keySet,query->keySet) ;
      break ;
    default:
      messcrash("keySetMix received wrong operator %d",operator) ;
    }
  arrayDestroy(query->keySet) ;
  query->keySet = s ;
/*   unSelect(selectedKeySet) ; */
  keySetDestroy(query->keySetAlpha) ;
  keySetShow2(query->keySet,query ) ;
}

/***********************************/

static void localKbd (int k)
{
  float x0,y0,x1,y1,x2,y2 ;
  int dispMax, dispMin ;
  int max ;
  LOOKGET("keySetPick") ;

  if (look->curr < 2)
    return ;

  max = keySetMax (look->keySet) ;
  dispMin = 2 ;
  dispMax = 1 + look->top - look->base ;
  
  graphBoxDraw(look->curr, BLACK, WHITE);
  switch (k)
    {
    case LEFT_KEY :
      if (look->curr > dispMin)
	--look->curr ;
      else if (look->base)
	{ pageUp () ;
	  look->curr = look->top - look->base + 1 ;
	}
      break ;
    case RIGHT_KEY :
      if (look->curr < dispMax)
	++look->curr ;
      else if (look->top < max)
	{ pageDown () ;
	  look->curr = 2 ;
	}
      break ;
    case DOWN_KEY :
      graphBoxDim (look->curr,&x0,&y0,&x2,&y2) ;
      while (look->curr < dispMax &&
	     (y1 < y0+0.5 || x2 < x0))
	graphBoxDim (++look->curr,&x1,&y1,&x2,&y2) ;
      if ((y1 < y0 + 0.5 || x2 < x0) && look->top < max)
	{ pageDown() ;
	  look->curr = 2 ;
	}
      break ;
    case UP_KEY :
      graphBoxDim (look->curr,&x0,&y0,&x2,&y2) ;
      y1 = y0 ;
      while (look->curr > 2 && (y1 > y0-0.5 || x1 > x0))
	graphBoxDim (--look->curr,&x1,&y1,&x2,&y2) ;
      if ((y1 > y0-0.5 || x1 > x0) && look->base)
	{ pageUp () ;
	  look->curr = look->top - look->base + 1 ;
	}
      break ;
    case HOME_KEY :
      if (look->base)
	{ while (look->base = pop(look->pages,int)) ;
	  keySetShow2 (look->keySet, look) ;
	}
      look->curr = 2 ;
      break ;
/*    case END_KEY :
      if (look->top < max)
	{ ???
	if(look->keySetAlpha &&
	keySetMax(look->keySetAlpha) < keySetMax(look->keySet))
	{ keySetDestroy(look->keySetAlpha) ;
	look->keySetAlpha = keySetAlphaHeap(keySet, keySetMax(look->keySet)) ;
	}
	keySetShow2 (look->keySet, look) ;
	}
      look->curr = max - look->page*200 + 1 ;
      break ;
*/
    default:
      graphBoxDraw(look->curr, WHITE, BLACK) ;
      return ;
    }
  graphBoxDraw (look->curr, WHITE, BLACK) ;
  display (keySet(look->keySetAlpha,look->curr + look->base - 2),0,0) ;
}

/***************************/

static void ksetAddKey (KEY key)
{
  LOOKGET("ksetAddKey") ;

  keySetInsert(look->keySet,key) ;
  keySetDestroy(look->keySetAlpha) ;

  keySetShow2(look->keySet,look) ;

  displayRepeatBlock () ;
}

/***************************/

static void ksetDeleteKey (KEY key)
{
  LOOKGET("ksetAddKey") ;

  keySetRemove(look->keySet,key) ;
  keySetDestroy(look->keySetAlpha) ;

  keySetShow2(look->keySet,look) ;

  displayRepeatBlock () ;
}

/*****************************/

static void ksetUnBlock (void)
{
  LOOKGET("ksetUnBlock") ;

  if (look->isBlocking)
    { look->isBlocking = FALSE ;
      displayUnBlock() ;
    }
}

/*************************************/

extern BOOL asnDumpKey(KEY key, FILE* fil, BOOL previousAsn) ;
extern void dumpKeySetFastA (KEYSET keySet) ; /* in dnasubs.c */
extern BOOL keySetDump(FILE *f, Stack s1, Array s) ; /* in keysetdump.c */

static void localMenu (KEY k)
{
  Graph  g ;
  KEY	 key = 0 ;
  int    i ;
  BOOL   previousAsn = FALSE ;
  FILE   *fil ;
  static char fileName[24],dirName[80] ;
  LOOKGET("localMenu") ;

  switch (k)
    {
    case 99: 
      graphDestroy () ; 
      break ;
    case 98: 
      helpOn("Selection") ;
      break ;
    case 1:	/* save */
      if(!isWriteAccess())
	{ messout("Sorry, you do not have Write Access");
	  return ;
	}

      while (!key && graphPrompt ("Give a name","","t"))
	if (!lexaddkey (freeword(),&key,_VKeySet) &&
	    !graphQuery ("Overwrite existing keyset ?"))
	  return ;
      
      if (!key)
	break ;
      
      if (!lexlock(key))
	{ messout ("Sorry, key is locked by someone else") ;
	  break ;
	}
      arrayStore (key,look->keySet,"k") ;
      lexunlock (key) ;

      break ;
    case 2:	/* copy */
      g = graphActive () ;
      displayCreate(DtKeySet) ;
      graphRetitle ("Copied keyset") ;
      keySetShow (keySetCopy (look->keySet),0) ;
      graphActivate (g) ;
      break ;
    case 4:	/* biblio */
      if (keySetMax(look->keySet) > keySetMax(look->keySetAlpha))
	{ keySetDestroy(look->keySetAlpha) ;
	  look->keySetAlpha = 
	    keySetAlphaHeap(look->keySet, keySetMax(look->keySet)) ;
	}
      biblioKeySet ("Biblio for keyset",look->keySetAlpha) ;
      break ;
    case 3:	/* .ace dump */
      if (!(fil = filqueryopen (dirName, fileName, "ace", "w")))
	{ messout ("failed to open ace dump file") ;
	  return ;
	}
      fprintf (fil,"// data dumped from keyset display\n\n") ;
      if (keySetMax(look->keySet) > keySetMax(look->keySetAlpha))
	{ keySetDestroy(look->keySetAlpha) ;
	  look->keySetAlpha = 
	    keySetAlphaHeap(look->keySet, keySetMax(look->keySet)) ;
	}
      for (i = 0 ; i < keySetMax(look->keySetAlpha) ; ++i)
	dumpKey (keySet(look->keySetAlpha,i), fil) ;
      fclose (fil) ;
      break ;
    case 31:	/* .asn dump */
      if (!(fil = filqueryopen (dirName, fileName, "asn", "w")))
	{ messout ("failed to open asn dump file") ;
	  return ;
	}
      previousAsn = FALSE ;
      fprintf (fil,"ACeDB-data ::= \n{") ;
      if (keySetMax(look->keySet) > keySetMax(look->keySetAlpha))
	{ keySetDestroy(look->keySetAlpha) ;
	  look->keySetAlpha = 
	    keySetAlphaHeap(look->keySet, keySetMax(look->keySet)) ;
	}
      for (i = 0 ; i < keySetMax(look->keySetAlpha) ; ++i)
	{ previousAsn |=   /* OR equal */
	    asnDumpKey (keySet(look->keySetAlpha,i), fil, previousAsn) ;
	}
      fprintf(fil, "\n}\n") ;
      fclose (fil) ;
      break ;
    case 8:	/* name dump */
      if (!(fil = filqueryopen (dirName, fileName, "ace", "w")))
	{ messout ("failed to open ace dump file") ;
	  return ;
	}
      fprintf(fil,"KeySet : \"%s\"\n",fileName) ;
      if (keySetMax(look->keySet) > keySetMax(look->keySetAlpha))
	{ keySetDestroy(look->keySetAlpha) ;
	  look->keySetAlpha = 
	    keySetAlphaHeap(look->keySet, keySetMax(look->keySet)) ;
	}
      keySetDump(fil,0 ,look->keySetAlpha) ;
      fclose (fil) ;
      break ;
    case 7:     /* FastA format sequence dump */
      if (keySetMax(look->keySet) > keySetMax(look->keySetAlpha))
	{ keySetDestroy(look->keySetAlpha) ;
	  look->keySetAlpha = 
	    keySetAlphaHeap(look->keySet, keySetMax(look->keySet)) ;
	}
      dumpKeySetFastA (look->keySetAlpha) ;
      break ;
    case 5:	/* sort */
      keySetDestroy(look->keySetAlpha) ;
      look->keySetAlpha = 
	keySetAlphaHeap(look->keySet, keySetMax(look->keySet)) ;
      keySetShow2(look->keySet,look) ;
      break ;
    case 6:     /* Add key */
      look->isBlocking = TRUE ;
      displayBlock (ksetAddKey,
	"Double-clicking oustide your own window will add entries, "
	"inside will delete them.  "
	"This will continue until you remove this message") ;
      break ;
    case 20: case 21: case 22:     /* JTM mix */
      if (selectedKeySet && selectedKeySet->magic == MAGIC)
	keySetMix(look, k) ;
      else
	messout
	  ("%s%s",
	   "First select a key Set window by picking ",
	   "its background with the left mouse button") ;
      break ;
    case 40:     /* JTM Tree */    /* Display Type  1 is TREE */
      if (look->curr < 2)
	break ;
      if(pickType(keySet(look->keySetAlpha,look->curr-2+look->base)) == 'B')
	display (keySet(look->keySetAlpha,look->curr-2+look->base),0,1) ;
      break ;
    case 41:     /* JTM Hexadecimal dump of disk blocks */
      if (look->curr < 2)
	break ;
      Bdump(keySet(look->keySetAlpha,look->curr-2+look->base)) ;
      break ;
    default: 
      break ;
    }
}

/*************************************/
/*************************************/
