/*  File: bsdumps.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 * Exported functions:
 * HISTORY:
 * Last edited: Mar 26 14:40 1992 (mieg)
 * * Mar 11 00:04 1992 (rd): asn dumps
 * Created: Sun Feb  9 21:02:19 1992 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "bs_.h"
#include "bs.h"
#include "lex.h"
#include "pick.h"
#include "systags.wrm"
#include "sysclass.wrm"
#include "chrono.h"

/**************************************************************/

void bsDump (OBJ obj)
{ bsTreeDump(obj->root) ;
}

/********** ace dump package for BS trees *************/

/* RMD 31/12/90
   Strategy is to do a depth first search of the tree.
   When a leaf is found: if it is a tag, write it out,
   if not, pop back onto a temporary stack until one is found,
   then run forward again to write it out.
*/

static Stack aceDumpWorkStack,aceDumpTempStack ;	/* used in recursion */
static FILE  *dumpFile ;
static Stack aceDumpText ;

static void bsAceDumpCell (BS bs, BS bsm, BOOL dumpAll)
{
  char *cp ;
  static BOOL first ;

#define addText(xx) catText(aceDumpText,xx)
#define LINE_LENGTH 70

  if (!bs)
    return ;

  push (aceDumpWorkStack, bs, BS) ;

  bsModelMatch (bs, &bsm) ;

  if (bs->right)
    { if (bsm && bsm->right && bsIsType(bsm->right))
	bsAceDumpCell (bs->right, bsModelRight(bsm), TRUE) ;
      else
	bsAceDumpCell (bs->right, bsm ? bsm->right : 0, dumpAll) ;
    }
  else
    { stackClear (aceDumpTempStack) ;
      while (!stackEmpty (aceDumpWorkStack))
	{ bs = pop (aceDumpWorkStack, BS) ;
	  push (aceDumpTempStack, bs, BS) ;
	  if (!dumpAll && bsIsTag(bs))
	    break ;
	}
      stackClear (aceDumpText) ;
      first = TRUE ;
      while (!stackEmpty (aceDumpTempStack))
	{ bs = pop (aceDumpTempStack, BS) ;
	  push (aceDumpWorkStack, bs, BS) ;
	  if (class(bs->key))
	    { if (class(bs->key) == _VComment)
		addText (" -C") ;
	      if (class(bs->key) == _VUserComment)
		addText (" -U") ;
	      addText(" \"") ; 
	      addText(name(bs->key)) ; 
	      addText("\"") ;
	    }
	  else if (bs->key == _Int)
	    addText(messprintf (" %d", bs->n.i)) ;
	  else if (bs->key == _Float)
	    addText (messprintf (" %f", bs->n.f)) ;
	  else if (bs->key <= _LastC)
	    { addText(" \"") ; 
	      addText(bsText(bs)) ; 
	      addText("\"") ;
	    }
	  else			/* a tag */
	    { if (first)
		{ addText(name(bs->key)) ; addText("\t") ; }
	      else
		{ addText(" ") ; addText(name(bs->key)) ; }
	    }
	  first = FALSE ;
	}
      uLinesText (stackText(aceDumpText,0), LINE_LENGTH) ;
      fprintf (dumpFile, "%s", uNextLine(stackText(aceDumpText,0))) ;
      while (cp = uNextLine(stackText(aceDumpText,0)))
	fprintf (dumpFile, "\\\n	%s", cp) ;
      fputc ('\n',dumpFile) ;
    }				/* after all this, bs is as it was! */

  pop (aceDumpWorkStack,BS) ;

  if (bs->down)
    bsAceDumpCell (bs->down, bsm, dumpAll) ;
}

void bsAceDump (OBJ obj, FILE *fil)
{
  dumpFile = fil ;
  aceDumpTempStack = stackCreate (16) ;
  aceDumpWorkStack = stackCreate (16) ;
  aceDumpText = stackCreate(64) ;

  bsAceDumpCell (obj->root->right, bsModelRoot(obj)->right, FALSE) ;

  stackDestroy(aceDumpTempStack) ;
  stackDestroy(aceDumpWorkStack) ;
  stackDestroy(aceDumpText) ;
}

/***************************************************************************/

static void asnSpace(int level)
{
  int i = 3*level ;
  if (i>0)
    while(i--)
      fputc(' ',dumpFile) ;
}

char *asnCleanUp (char *text)	/* convert '_' to '-' */
{
  static char buf[128] ;
  char *cp ;

  if (strlen(text) > 110)
    messcrash ("Sorry, asnCleanUp has hard limit on tag length of 110") ;

  strcpy (buf , "acedb-") ;
  strcat (buf , text) ;
  for (cp = buf ; *cp ; ++cp)
    if (*cp == '_')
      *cp = '-' ;
    else
      *cp = freelower(*cp) ;

  return buf ;
}

/**************/
static char* asnFloat(float x)
{
  int i, k , n = 0 ;
  double y = x > 0 ? x : -x ;
  
  if (!x)
    return "{ 0, 10, 1 }" ;

  i = log10(y) ;
  if (i > 5)
    for (k = 5 ; k < i ; k++)
      { n++ ; y /= 10 ; }
  else
    for (k = i ; k < 5 ; k++)
      { n-- ; y *= 10 ; }
  
  return messprintf("{ %c%d, 10, %d}",  x > 0 ? ' ' : '-', (int) y, n) ;
}

static void bsAsnDumpCell (BS bs, BS bsm, int level)
{
  static char *tag ;
  static int tagLevel ;
  BS bsmOld, bsDown ;
  BOOL comma , openBracket  = FALSE ;

  if (!bsModelMatch (bs, &bsm))
    return ;

  if (!bsIsTag (bsm))		/* only at top of column */
    { if (bsm->n.key & UNIQUE_BIT)
	fprintf (dumpFile, " value") ;
      else
	{ openBracket  = TRUE ;
	  fprintf (dumpFile, " values { ") ;
	}
    }
  bsmOld = bsm ;

 work:
  bsDown = bs->down ;
  fprintf (dumpFile,"\n") ;
  asnSpace (level) ;

  if (bsIsTag(bs))
    { tag = asnCleanUp (name(bs->key)) ;
      tagLevel = level ;
      fprintf (dumpFile, "%s { ", tag) ;
      comma = FALSE ;
    }
  else 
    { if (bs->key == _Int)
	fprintf (dumpFile, " { %s-%d %d", tag, level-tagLevel, bs->n.i) ;
      else if (bs->key == _Float)
	fprintf (dumpFile, " { %s-%d %s", tag, level-tagLevel, asnFloat(bs->n.f)) ;
      else if (bs->key <= _LastC)
	fprintf (dumpFile, " { %s-%d \"%s\"", tag, level-tagLevel, uBrokenText(bsText(bs),80)) ;
      else if (class(bs->key))
	fprintf (dumpFile, " { %s-%d \"%s\"", tag, level-tagLevel, uBrokenText(name(bs->key),80)) ;
      comma = TRUE ;
    }
  bs = bs->right ;
  if (bs && bsIsComment (bs))
    { fprintf (dumpFile, " comment { \"%s\"", name(bs->key)) ;
      while ((bs = bs->down) && bsIsComment (bs))
	{ fputc (',', dumpFile) ;
	  asnSpace (level+6) ; 
	  fprintf (dumpFile, " \"%s\"", name (bs->key)) ;
	}
      fprintf (dumpFile, " }") ;
      comma = TRUE ;
    }
  if (bs)
    { if (comma)
	fputc (',', dumpFile) ;
      comma = TRUE ;
      bsAsnDumpCell (bs, bsModelRight(bsm), level+1) ;
    }
  if (!comma)
    fprintf (dumpFile, " filler TRUE ") ;    
  fprintf (dumpFile, " }") ;

  if (bsDown)
    { bs = bsDown ;
      if (!bsModelMatch (bs, &bsm))
	bsm = 0 ;		/* flag to return, but must check bracket */
      if (openBracket && bsIsTag(bs))
	{ openBracket  = FALSE ;
	  fprintf (dumpFile, " }") ;
	}
      if (bsm)
	{ fputc (',', dumpFile) ;
	  goto work ;
	}
    }
  if (openBracket)
    fprintf (dumpFile, " }") ;
}

/********/

void bsAsnDump (OBJ obj, FILE *fil)
{
  dumpFile = fil ;

  fprintf (dumpFile, "\n   %s", asnCleanUp(className(obj->key))) ;
  fprintf (dumpFile, "\n    { acedb-id \"%s\"", name(obj->key)) ;
  if (obj->root->right)
    { fputc (',', dumpFile) ; 
      bsAsnDumpCell (obj->root->right, obj->modCurr->right, 2) ;
    }
  fprintf (dumpFile, "\n    }") ;
}

/*********************/

static void bsAsnDumpModelCell (BS bsm, int level)
{ 
  char *mytext ;
  static char *tag ;
  static int tagLevel ;

  asnSpace (level) ;

  if (bsIsType(bsm))
    fprintf (dumpFile, "type Acedb-%s OPTIONAL\t-- Type %s",
	     asnCleanUp(className(bsm->key)), className(bsm->key)) ;
  else 
    { if (bsIsTag(bsm))
	{ tag = asnCleanUp (name (bsm->key)) ;
	  tagLevel = level ;
	  fprintf (dumpFile, "%s SEQUENCE\t\t\t-- Tag %s\n",
		   tag, name(bsm->key)) ;
	  asnSpace (level) ; fprintf (dumpFile, " { ") ;
	}
      else 
	{ if (bsm->key == _Int)
	    mytext = "INTEGER,\t\t-- Int" ;
          else if (bsm->key == _Float)
	    mytext = "REAL,\t\t -- Float" ;
	  else if (bsm->key <= _LastC)
	    mytext = "VisibleString,\t-- Text" ;
	  else if (class(bsm->key) == _VText)
	    mytext = "VisibleString,\t-- Key ?Text" ;
	  else if (class(bsm->key))
	    mytext = messprintf ("VisibleString,\t-- Key ?%s",
				 className (bsm->key)) ;
	  else
	    { messout ("Unknown bsAsnDumpModelCell : %d %s",
		       bsm->key, name(bsm->key)) ;
	      mytext = "VisibleString,\t-- *** BAD ACEDB MODEL" ;
	    }
	  if (bsm->n.key & UNIQUE_BIT)
	    fprintf(dumpFile, "value SEQUENCE\n") ;
	  else
	    fprintf(dumpFile, "values SET OF SEQUENCE\n") ;
	  asnSpace(level) ; fprintf( dumpFile, " { %s-%d %s\n", 
				    tag, level-tagLevel, mytext) ;
	  asnSpace(level+1) ;
	}
      if (bsm->right && bsm != bsm->right)  /* a fix for REPEAT, ANY */
	{ fprintf (dumpFile, 
		   "comment Acedb-Comment OPTIONAL, filler BOOLEAN OPTIONAL,\n") ;   
	  bsAsnDumpModelCell (bsm->right, level+1) ;
	}
      else
	fprintf (dumpFile, "comment Acedb-Comment OPTIONAL, filler BOOLEAN OPTIONAL ") ;   
      fprintf (dumpFile, "\n") ;
      asnSpace(level) ; fprintf (dumpFile, " } OPTIONAL") ;
    }
  
  if (bsm->down)
    { fprintf (dumpFile, ",\n") ; 
      bsAsnDumpModelCell (bsm->down, level) ;
    }
}

/*********************/

void  bsAsnDumpModel (FILE *f, int class)
{ OBJ obj ; KEY k = 1 ;
      /* with KEYKEY = 0, the model is not created */
      /* with bsCreate it may not be if iskey(class,1) < 2 */
  for (k = 1 ; k < lexMax(class) ; k++)  
    if (obj = bsCreate(KEYMAKE(class, k)))
      break ;
  if (!obj)
    return ;
  dumpFile = f ;
  fprintf (f, "\nAcedb-%s ::=  SEQUENCE", asnCleanUp(pickClass2Word(class))) ;
  fprintf (f, "\n { acedb-id VisibleString,\t\t-- Name\n") ;
  bsAsnDumpModelCell (obj->modCurr->right, 1) ;
  fprintf (f, "\n    }") ;
  bsDestroy (obj) ;    
}

/***************************************************************************/
/***************************************************************************/
