/*  File: asubs.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 **  Creates saves and updates permanent Arrays and Stacks
 * Exported functions:
 * HISTORY:
 * Last edited: Apr  3 18:37 1992 (mieg)
 * Created: Wed Jan 22 15:38:55 1992 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "a_.h"
#include "a.h"
#include "lex.h"
#include "block.h"
#include "chrono.h"
#include "pick.h"

/**************************************************************/
 /* expected format looks like "f2v3k" "3i" etc. */
int sizeOfFormattedArray(char *format)
{ int n = 0 , m = 1 ,
    i = sizeof(int),  f = sizeof(float) , 
    v = sizeof(void *) , k = sizeof(KEY) ;
  char *cp = format ;
  
  if (!cp)
    messcrash("sizeOfFormattedArray recieved a null pointer") ;
  cp-- ;
  while (*++cp)
    {
      if (*cp >= '0' && *cp <= '9' )  /* repetition factor */
	if ( m > 1 || *(cp - 1) == '1')
	  m = 10 * m + (*cp - '0') ; 
	else  /* distinguish "c" "1c" "11c" */
	  m = (*cp - '0') ;
      else
	switch(*cp)
	  {
	  case 'c': n += m; m = 1 ; break ; /* single char */
	  case 'k': n += m * k ; m = 1 ; break ;  /* key */
	  case 'i': n += m * i ; m = 1 ; break ;  /* integer */
	  case 'f': n += m * f ; m = 1 ; break ;  /* float */
	  case 'v': n += m * v ; m = 1 ; break ;  /* void pointer */
	  default:
	    messcrash("Unknown format %s at %s in sizeOfFormattedArray", 
		      format, cp) ;
	  }
    }
  if (m != 1)
    messcrash("uncomplete format %s in sizeOfFormattedArray", format) ;
  
#if defined(NEXT)
  i = 2 ;   /* NEXT seems to align on half words. */
#endif   
  return  n == 1 ? n : ( (n + i - 1) / i) * i ; /* word boundaries */
}
	
/**************************************************************/

BOOL dumpFormattedArray (FILE *f, Array a, char *format)
{ int i , m = 1 ;
  char *cp = format ,  *cq ;
  KEY key ;

  if (!arrayExists(a) || !f)
    messcrash("null a or f in dumpFormattedArray") ;
 
  for (i = 0 ; i < arrayMax(a) ; i++)
    { fputc('\n', f) ;
      cq = a->base + i * a->size ;
      cp = format - 1 ;
      while(*++cp)
	{
	  if (*cp >= '0' && *cp <= '9' )  /* repetition factor */
	    if ( m > 1 || *(cp - 1) == '1')
	      m = 10 * m + (*cp - '0') ; 
	    else  /* distinguish "c" "1c" "11c" */
	      m = (*cp - '0') ;
	  else
	    switch (*cp)
	      {
	      case 'k':
		while(m--)
		  {
		    key = *(KEY*) cq ;
		    fprintf(f, " %s : \"%s\" ",
			    className(key),
			    name(key)) ;
		    cq += sizeof(KEY) ;
		  }
		m = 1 ;
		break ;
	      case 'v':
		messcrash("dumpFormattedArray tries to dump a pointer") ;
		break ;
	      case 'i':
		while (m--)
		  { 
		    fprintf(f, " %d ", *(int*)cq) ;
		    cq += sizeof(int) ;
		  }
		m = 1 ;
		break ;
	      case 'f':
		while (m--)
		  {
		    fprintf(f, " %f ", *(float*)cq) ;
		    cq += sizeof(float) ;
		  }
		m = 1 ;
		break ;
	      case 'c':
		while (m--)    /* note that I use no carriage return */
		  fputc(*cq++, f) ;
		m = 1 ;
		break ;
	      default:
		messcrash("Unknown format %s at %s in dumpFormattedArray", 
			  format, cq ) ;
	      }
	}
    }
  fputc('\n', f) ;
  fputc('\n', f) ;	    
  return TRUE ;
}
	
/**************************************************************/
        /* No check if size = -1 */
Array uArrayGet(KEY key, int size, char *format)
{
 Array a;
 BP bp ;
 int i = 0, j;

 if (pickType(key) != 'A')
    messcrash ("Attempt to arrayGet a non-array key %s:%s",
	       className(key), name(key)) ;
 if(!iskeyold(key))
   return 0;
 blockpinn(key,&bp);

 if (!format)
   format="" ;
 if (size != -1 && size != sizeOfFormattedArray(format))
   messcrash("Actual size of array  %d != size %d of format %s",
	     size, sizeOfFormattedArray(format), format) ;
	      
 if (bp->hat.h.type != 'A')
   messcrash("arrayGet(%s) read a block of h.type %c",
	     name(key),bp->hat.h.type);
 if (size != -1 && bp->hat.size != size)
   messcrash("arrayGet(%s) read an array of incorrect size %d != %d",
	     name(key),bp->hat.size,size);
 a =  uArrayCreate(bp->hat.max,bp->hat.size);
 arrayMax(a) = bp->hat.max;
 while (TRUE)
   {
     j = bp->hat.localmax ;
     memcpy(a->base + i*a->size, bp->n, j*a->size) ;
     i += j ;
     if (!blockNext(&bp))
       break ;
   }
 blockunpinn(key);
 if(i != arrayMax(a))
   messcrash("Size inconsistency in arrayGet (%s)",
	     name(key));

 return a;
}

/**********************************************************/

void arrayStore(KEY key, Array a, char* format)
{
 BP bp;
 int i = 0, j = arrayExists(a) ? a->max : 0 , local , k;
 extern int blockCheck(void) ;

 if (pickType(key) != 'A')
   messcrash ("Attempt to arrayStore a non-array key %s:%s",
	       className(key), name(key)) ;

 if (!format)
   format="" ;
 if (a->size != sizeOfFormattedArray(format))
   messcrash("Actual size of array  %d != size %d of format %s",
	     a->size, sizeOfFormattedArray(format), format) ;
	      
 if (j)
   { blockpinn (key, &bp);

     local = AMAX / a->size ;
     lexUnsetStatus (key, EMPTYSTATUS) ;
     while (TRUE)
       {
	 bp->hat.h.key = key;
	 bp->hat.h.type = 'A'; 
	 bp->hat.size = a->size;
	 bp->hat.max  = a->max;

	 k = bp->hat.localmax =  (j>local) ? local : j ;
	 memcpy(bp->n, a->base + i*a->size, k*a->size) ;
	 if (k < local)
	   memset(bp->n + k*a->size, 0, (local-k)*a->size) ;

	 j -= k ;
	 i += k ;
	 if(!j) 
	   {
	     blockSetEnd(bp) ;
	     break ;
	   }
	 blockSetNext(&bp) ;
       }
     blockrewrite(key) ;
   }
 else
   arrayKill(key) ;
}

/**********************************************************/
  /* it is assumed here that array has 
     no entry in the objcache
     */
void arrayKill(KEY key)
{
  BP bp;
  
  if (pickType(key) != 'A')
    messcrash ("Attempt to arrayKill a non-array key %s:%s",
	       className(key), name(key)) ;

  if (iskeyold(key)) 
    { blockpinn(key, &bp); /* unpinned by blockSetEmpty */
      blockSetEmpty(bp) ;
      lexkill(key) ;      
    }
}

/*********************************************/
/*********************************************/

void stackStore(KEY key, Stack s)
{
 if(!stackExists(s))
   messcrash("stackStore received a non initialised stack");
 arrayMax(s->a) = stackMark(s) ;
 arrayStore(key, s->a,"c") ;
}

/*********************************************/

Stack stackGet(KEY key)
{
  Stack s ;
  Array a = arrayGet(key, char,"c") ;
  
  if(!a)
    return 0 ;
  
  if(a->size != sizeof(char))
    messcrash("stackGet read a non char array") ;
  
  s = stackCreate(32);
  arrayDestroy(s->a) ;
  s->a = a ;
  s->pos = s->ptr = s->a->base + arrayMax(s->a) ;
  s->safe = s->a->base + s->a->dim - 8 ; 
  return s ;
}

/**********************************************************/
/**********************************************************/





