/*  File: lexsubs.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 * Exported functions:
 * HISTORY:
 * Last edited: Apr 22 16:08 1992 (mieg)
 * * Jan 10 00:54 1992 (mieg): added lexAlias, eliminated lastmodif
 * * Dec 11 12:22 1991 (mieg): lexcleanup to remove spaces in addkey
 * * Nov  5 21:28 1991 (mieg): introduced the hash package
 * Created: Tue Nov  5 21:28:40 1991 (mieg)
 *-------------------------------------------------------------------
 */

      /***************************************************************/
      /***************************************************************/
      /**  File lexsubs.c :                                         **/
      /**  Handles as Arrays  the lexique of the ACeDB program.     **/
      /***************************************************************/
      /***************************************************************/
      /*                                                             */
      /*  Many    routines are public :                              */
      /*     avail,show, make/dump,  2define,                        */
      /*     iskey,lock/unlock, iskeylocked,                         */
      /*     KEY2LEX,key2word/word2key, addkey, kill,                */
      /*     randomkey,lexNext and lexstrcmp.                        */
      /*                                                             */
      /*  There are 256 vocabularies, the first 128 are generic      */
      /*  and common to all implementations of the base, the others  */
      /*  can be handled freely by every user.                       */
      /*                                                             */
      /*  Voc[0] contains the flow control dialog, the others the    */
      /*  names of the genes, alleles, bits of the map and other     */
      /*  objects, and the keywords describing the phenotypes.       */
      /*                                                             */
      /*  Each word is coded by a key. A key is 4 bytes long, the    */
      /*  first byte is the vocabulary number, thus, the maximum     */
      /*  number of entry per voc. is 16M. Each key may then refer   */
      /*  to a disk address (or 0 if the word is not the name of an  */
      /*  object or a list), and to a cache control area if the      */
      /*  object is loaded.                                          */
      /*                                                             */
      /*  Lexshow is for debugging.                                  */
      /*  Lexmake is called on entering the session, it reads in the */
      /*  lexiques from disk and return 0 if ok. LexSave writes      */
      /*  them back to disk. Note that the lexique is not saved on   */
      /*  every modification, therefore, if a crash occurs during    */
      /*  a session, the lexiques must be reconstructed from the     */
      /*  disk.                                                      */
      /*                                                             */
      /*  iskey returns 0 on unknown keys, 1 if the key is empty     */
      /*  or 2 if an object is associated to this key.               */
      /*  lock prevents the simultaneous updating by 2 processes     */
      /*  it is invoked by BSlock_make                               */
      /*                                                             */
      /*  Name make a key into a word, and word2key a word    */
      /*  into a key if the word is known.  Lexaddkey(w,k,t) adds    */
      /*  a new word w and sets its key k in vocabulary t.           */
      /*                                                             */
      /*  Lexstrcmp is a case unsensitive strcmp.                    */
      /*                                                             */
      /*         R.Durbin & J.Thierry-Mieg.                          */
      /*                    last modified  15/9/90 by JTM.           */
      /*                                                             */
      /***************************************************************/

#include <ctype.h>

#define ARRAY_NO_CHECK		/*  gross abuse of array bounds here */
				/* Let us exercice our priviledges */

#include "acedb.h"
#include "lex_bl_.h"
#include "array.h"
#include "a.h"
#include "lex.h"
#include "sysclass.wrm"
#include "key.h"
#include "disk.h"    /*call to diskprepare in lexinit*/
#include "display.h"
#include "pick.h"
#include "chrono.h"
#include "session.h"

#define MAXLEX   ((1<<24) - 1)          /*number of entries per vocabulary <=FFFFFF*/
#define MAXVOCAB ((1<<24) - 1)     /*number of char in a vocabulary<max(LEXOFFSET)*/
#define MAXTABLE 256             /*number of vocabulary tables <= 256*/
                                 /* watchout 256 meaning MAXTABLE used in coresubs */
                                 /* and also in parse.c , maybe elsewhere */
                                 /* and corresponds to the first byte inside KEY */
static Array  Lexi [MAXTABLE];
static KEYSET LexHashTable[MAXTABLE] ;
static int    nClass[MAXTABLE] ;  /* number of bits for hasher */
static Stack Voc  [MAXTABLE];
static BOOL  lexIsRead[MAXTABLE] ;
static char* nakedName(KEY key) ;  /* Does not follow aliases */
static int vocmodif[MAXTABLE];
static int lexhmodif[MAXTABLE];
static int leximodif[MAXTABLE];

#define si ((U_Int)sizeof(LEXI))

static void lexvocread(int table) ;
static void lexvocwrite(int table);
static void lexiread(int table) ;
static void lexiwrite(int table);
static void lexhread(int table) ;
static void lexhwrite(int table);
static void lexdefine2voc0(int pass, char *name, char *ending) ;

static void lexHashInsert(int class, KEY key) ;
 void lexReHashClass(int class) ;

#define PRINT FALSE	/* set TRUE to show save progress */

/**************************************************************/

int lexavail(unsigned long *vocnum,
             unsigned long *keynum,unsigned long *vocspace)
{
 register  int t=256;
 unsigned long n=0,v=0,k=0;
 while(t--)
  if(Lexi[t])
      { n++;
        v+=(unsigned long) stackTextMax(Voc[t]) ;
        k+=(unsigned long) arrayMax(Lexi[t]) ;
        }

  *vocspace=v; *vocnum=n;*keynum=k;

  return(0);
}

/*************************************************/

void lexInit(void)
{
  register int t = MAXTABLE ;

  while(t--)
    {
      vocmodif[t] = lexhmodif[t] = leximodif[t] = FALSE ;
      Voc[t] = 0 ; Lexi[t] = LexHashTable[t] = 0 ;
      nClass[t] = 0 ;
      lexIsRead[t] = FALSE ;
    }

 /*  diskprepare()  will be called implicitly on first disk access */

  lexdefine2voc0(0,"wspec/systags","wrm") ;
  lexdefine2voc0(1,"wspec/tags","wrm") ;
  lexIsRead[0] = lexIsRead[1] = TRUE ;
  lexReHashClass(0) ;
  lexReHashClass(1) ;
}

/*************************************************/
  /* To be used only from the session manager */
void lexClear(void)
{
  register int t ;
  
  for(t=3;t< MAXTABLE; t++)
    { vocmodif[t] = lexhmodif[t] = leximodif[t] = FALSE ;
      stackDestroy(Voc[t]) ;
      arrayDestroy(Lexi[t]) ;
      keySetDestroy(LexHashTable[t]) ;
      nClass[t] = 0 ;
      lexIsRead[t] = FALSE ;
    }
}

/*************************************************/

static void lexReadTable(int t)
{
  if(lexIsRead[t])
    return ;
  lexIsRead[t] = TRUE ;

  if(*pickClass2Word(t))
    { lexvocread(t) ;
      if(Voc[t])  /* could be 0 if class is new */
	{ mainActivity(
	    messprintf("Reading in class %s",
		       pickClass2Word(t))) ;
	  lexiread(t) ;
	  lexhread(t) ;
	  mainActivity(0) ;
	}
      else  /* Reinitialise this lexique */
	{ KEY key ;
	  arrayDestroy(Lexi[t]) ;
	  keySetDestroy(LexHashTable[t]) ;
	  nClass[t] = 0 ;
	  lexaddkey(0,&key,t) ;
	}
     }
}

/*************************************************/

void lexRead(void)
{int t=0 ;
                /* table 0 is parsed from tags.wrm
		   table 1 is created by hand
                   table 2 is the session chooser
                   table 3 is the Voc lexique
                   table 4 is the BAT chooser
                */
   for(t=3;t< 5; t++)
       lexReadTable(t) ;
}

/*************************************************/

static int nEntries(int t)
{
  if (t<0 || t>MAXTABLE)
    return 0 ;
  if (!lexIsRead[t])
    lexReadTable(t) ;
  return
    Lexi[t] ? arrayMax(Lexi[t]) : 0 ;
}

/*************************************************/

void lexReadGlobalTables(void)
{ lexIsRead[1] = lexIsRead[0] = TRUE ;
  lexiread(1) ;
  lexReHashClass(1) ;

   /* For compatibility with databases prior to release 1.4 */
  { KEY key ; 
    lexaddkey("_lexh1",&key,1) ;
    lexaddkey("_lexh2",&key,1) ;
    lexaddkey("_lexh3",&key,1) ;
  }

  diskPrepare() ;  /* Reads the Global Bat */
  lexReadTable(2) ;
}

/**************************************************************/
void lexmark(int tt)    /*enables blocksubs to modify leximodif
                        * when changing a disk address
                        * note that the memory address is never
                        * saved to disk, nor the locks
                        * and that only lexaddkey and  lexRename
			* touch lexh and voc
                        */
{
  leximodif[tt] = TRUE ; 
}

/******************************************************************/
      /*writes the modified lexiques to disk*/
      /*public ; called by saveAll() and dosomethingelse()*/

BOOL  lexSave(void) 
{
  int t=MAXTABLE ;
  BOOL j = FALSE ;

  while(t--)  /* do NOT save lex[0]. That one is written as tags.wrm */
                   /*t = 1 Global voc */
    {             
      if(t==1)
	lexiwrite(t);
      else if(t>1)
	if(Lexi[t])
	  {
	    lexvocwrite(t);
	    lexiwrite(t);
	    lexhwrite(t);
	  }
    }
  return j ? TRUE : FALSE ;
}

/********************************************/

static void lexvocread(int t)
{ KEY key ;
  
  lexaddkey (messprintf("_voc%d",t),&key,t>3 ? _VVoc : _VGlobal ) ;
  stackDestroy(Voc[t]) ;
  Voc[t] = stackGet(key) ;
  vocmodif[t] = FALSE ;
}

/**************************************************************/

static void lexvocwrite(int t)
{
  KEY key ;

  if (!isWriteAccess())
    return ;

  if(!vocmodif[t])
    return ;
  if(PRINT)
    printf("\n   writingVoc %d",t);
  
  lexaddkey(messprintf("_voc%d",t),&key,t>3 ? _VVoc : _VGlobal ) ;
  stackStore(key,Voc[t]) ;
  
  vocmodif [t] = FALSE ;
}

/**************************************************************/

static void lexhread(int t)
{
  KEY key ;

  lexaddkey(messprintf("_lexh%d",t),&key,t>3 ? _VVoc : _VGlobal ) ;
  keySetDestroy(LexHashTable[t]) ;
  LexHashTable[t] = arrayGet(key, KEY,"k") ;
  if (!LexHashTable[t])
    lexReHashClass(t) ;
  else
    { int n = arrayMax(LexHashTable[t]) , nBits = 0 ;
      while ((1 << ++nBits) < n) ;
      if(n != (1<<nBits))
	messcrash("Wrong size in lexhread %d", t) ;
      nClass[t] = nBits ;
    }

  lexhmodif [t] = FALSE ;
}

/**************************************************************/

static void lexhwrite(int t)
{
 /*  Commented out since i want to change the hashText 
     routine till it is really good

 KEY key ;
  
 
  if (!isWriteAccess())
    return ;

  if(!lexhmodif[t])
    return ;
  if(PRINT)
    printf("\n   writinglexh %d",t);

  lexaddkey(messprintf("_lexh%d",t),&key,t>3 ? _VVoc : _VGlobal ) ;
  arrayStore(key,LexHashTable[t]) ;
*/  
  lexhmodif[t] = FALSE ;
  return ;
}

/**************************************************************/

static void lexiread(int t)
{
  KEY key ;
  register LEXI* lxi ;
  register int j ;
  Array dummy ;
  
  lexaddkey(messprintf("_lexi%d",t),&key,t>3 ? _VVoc : _VGlobal ) ;
  dummy = arrayGet(key, LEXI,lexiFormat) ;
  arrayDestroy(Lexi[t]) ;
  Lexi[t] = dummy ;
  if (!Lexi[t])
    messcrash("Lexi[%d] not found",t);

  lxi=arrp(Lexi[t], 0, LEXI);
  j = arrayMax(Lexi[t]);
  while(j--)
    { lxi   ->addr=(ALLOCP)NULL;
      lxi   ->cache =(void *)NULL;
      lxi   ->lock &= ~LOCKSTATUS ; /* NOT 3, to zero the last 2 bits */
      lxi++;
    }
  leximodif[t] = FALSE ;
}

/**************************************************************/
static void lexiwrite(int t)
{ 
  KEY key ;

  if (!isWriteAccess())
    return ;

  lexaddkey(messprintf("_lexi%d",t),&key,t>3 ? _VVoc : _VGlobal ) ;

  if(!leximodif[t])
    return ;
  if(PRINT)
    printf("\n   writinglexi %3d,  %8d  entries.",
	   t, arrayMax(Lexi[t]));  
  leximodif[t] = FALSE ;  /* must come before the write */

  if(PRINT)
    printf("address %d.", KEY2LEX(key)->diskaddr);  
  arrayStore(key,Lexi[t],lexiFormat) ;
}

/******************************************************************/

void lexOverLoad(KEY key,DISK disk) 
{  KEY2LEX(key)->diskaddr = disk ;
   KEY2LEX(key)->cache = 0 ;
   KEY2LEX(key)->addr = 0 ;
}

/******************************************************************/

DISK lexDisk(KEY key)
{  return KEY2LEX(key)->diskaddr ;
}

/******************************************************************/
   /* Remove leadind and trailing spaces and non ascii */
static char *lexcleanup(char *cp)
{  char *cq = cp ;

   while(*cp == ' ')
     cp++ ;
   cq = cp + strlen(cp) ;
   while(cq > cp)
     if (*--cq == ' ')
       *cq = 0 ;
   else
     break ;
 /*  may be we should also have this
#include <ctype.h>
  cq = cp - 1 ;
   while(*++cq)
     if (!isascii(*cq))   or may be isgraph ?
       { *cq = 0 ;
	 break ;
       }
 */  
   return cp ;
}

/******************************************************************/

static void lexdefine2voc0(int pass, char *fileName, char * ending)
{
  FILE * myf;
  KEY key,key2;
  int table=0, i;
  int  badline, line = 0 ;
  char
    *cp,
    cutter=' ',
    tag[80] ;
  
  if(!(myf=filopen(fileName,ending,"r"),myf)) return;
  freeread(myf);  /*to skip the file header*/

  badline = 0;
  while(freeread(myf))
    { line++ ;
      cp=freewordcut(" ",&cutter);        /* get #define*/
      
      if(!cp || strcmp(cp,"#define"))
	{ if(cp && !strcmp(cp,"/*GLOBAL_LEXIQUES"))
	    { if(!pass) table = 1 ;
	      continue ;
	    }
	  if(badline++ == 25)
	    messcrash("File %s is messed up before line %d, sorry",
		      line ) ;
	  else 
	    continue;
	}
      cp = freewordcut(" ",&cutter) ;
      if(!cp || *cp != '_')
	messcrash("Tag %s should start by an underscore",
		  cp ? cp : "No tag") ;
      cp++ ;

      if(strlen(cp)>75)
	messcrash("Tag %s is too long",cp) ;
      strcpy (tag, cp) ;

      if(!freeint(&i))
	messcrash
	  ( "The keyvalue of tag %s is missing in %s.%s, I quit",
	   cp,fileName,ending);
      key = i ;

      if(lexword2key(tag,&key2,table) && key != key2)
	messcrash
	  ("The old %d and new %d values of tag %s differ.",
	   key2, key, tag) ;
      i = nEntries(table) ;
      if(!i) i = 1 ; /* Because a zeroth key is created by lexaddkey */
      i = KEYMAKE(table,i) ;
      if(i < key)
	{ for(;i < key; i++)
	    lexaddkey(messprintf("__sys%d",i),&key2,table) ;
	  lexaddkey(tag,&key,table);
	}
      else if(i == key)
	lexaddkey(tag,&key,table);
      else
	{ cp = name(KEYMAKE(key,table)) ;
	  if(strcmp(cp,messprintf("__sys%d",key)))
	    messcrash("Tag %s = %d tries to overwrite tag %s",
		      tag, key, cp) ;
	  /* Else I am overwriting a dummy tag */
	  array(Lexi[0], (int) key, LEXI).nameoffset 
	    = stackMark(Voc[0]) ;
	  pushText(Voc[0], lexcleanup(tag) ) ;

	  leximodif[0] = lexhmodif[0] = vocmodif[0] = TRUE ;
	}
    }
  fclose(myf);
}


/********************************************/
static LEXP myKey2Lex = 0 ;
                      /* Returns 0 if kk is unknown,
                       * 1 if kk is just a vocabulary entry
                       * 2 if kk corresponds to a full object
		       *
		       * sets myKey2Lex
                       */
int isNakedKey(KEY kk)
{
  KEY k = KEYKEY(kk);
  int t = class(kk);
  if (!kk || (k >= nEntries(t)) )
    return 0 ;

  myKey2Lex = arrp(Lexi[t], k, LEXI);

  return
    ( myKey2Lex->cache || 
     myKey2Lex->addr || 
     myKey2Lex->diskaddr)  ?  2  :  1  ;
}

/********************************************/
   /* prevent the formation of alias loops */
BOOL lexAliasLoop(KEY old, KEY new)
{
  KEY kk = new ;
  int result ;
  
  while (TRUE)
    { result = isNakedKey(kk) ;
      if (!result ||
	  ! (myKey2Lex->lock & ALIASSTATUS))
	break ;
      kk = (KEY) myKey2Lex->diskaddr ;
      if (kk == old)
	 /* to alias old to new would create a loop */
	return TRUE ;
    }
  return FALSE ;
}

/********************************************/
   /* Iterates through the alias list */
int iskey(KEY kk)
{
  KEY start = kk ;
  int result ;
  
  while (TRUE)
    { result = isNakedKey(kk) ;
      if (!result ||
	  ! (myKey2Lex->lock & ALIASSTATUS))
	break ;
      kk = (KEY) myKey2Lex->diskaddr ;
      if (kk == start)
	messcrash ("Loop in alias list of Key %d : %s",
		   kk, nakedName(kk) ) ;
    }
  return result ;
}

/********************************************/

                      /* Returns TRUE
                       * if kk has a primary cache or a disk address
                       */
BOOL iskeyold(KEY kk)
{
  return
    iskey(kk) ?
      (  (myKey2Lex->addr || myKey2Lex->diskaddr) ? TRUE : FALSE ) :
	FALSE ;
}
/********************************************/

LEXP KEY2LEX(KEY kk)
{
 return
   iskey(kk) ?  myKey2Lex : 0 ;
}

/********************************************/

BOOL lexlock(KEY key)
{      /* code partially  duplicated in cacheLock*/
 LEXP q=KEY2LEX(key);
  if(q)
    {
      if(q->lock & LOCKSTATUS)
	messerror("Double locking  of %s",
		  name(key));
      else
	{
	  q->lock |= LOCKSTATUS; return TRUE;
	}
    }
  return FALSE;
}

/********************************************/

void lexunlock(KEY key)
{
 LEXP q=KEY2LEX(key);
  if(   q
     && (q->lock & LOCKSTATUS) )
    q->lock &= ~LOCKSTATUS; 
  else
    messcrash("Unbalanced unlock of %s",
	      name(key));
}

/********************************************/
  /* alteration of LEXPRIVATESTATUS flags would 
     foul the lex system */
  /* code duplicated in cacheUpdate*/
void lexSetStatus(KEY key, unsigned char c) 
{     
  LEXP q=KEY2LEX(key);
  if(q)
      q->lock |= (c & ~LEXPRIVATESTATUS) ;
   
  else
    messcrash("lexUnsetStatus called with bad key %s",
	      name(key));
}

/********************************************/
  
  /* code duplicated in cacheUpdate*/
void lexUnsetStatus(KEY key, unsigned char c) 
{     
  LEXP q=KEY2LEX(key);
  if(q)
      q->lock &= (~c) | LEXPRIVATESTATUS ;
   
  else
    messcrash("lexUnsetStatus called with bad key %s",
	      name(key));
}

/********************************************/

unsigned char lexGetStatus(KEY key)
{     
 LEXP q=KEY2LEX(key);
 if(q)
   return q->lock ;
  else
    messcrash("lexGetStatus called with bad key %s",
	      name(key));
  return 0 ;  /* for compiler happiness */
}

/********************************************/
   /* Avoid the alias system, used by word2key and hasher */
static char * nakedName(KEY kk)  
{     
 return
   isNakedKey(kk)                ?
    stackText(Voc[class(kk)],
	myKey2Lex->nameoffset)
                            :
    "\177\177(NULL KEY)"        ;     
}

/********************************************/

char * name(KEY kk)  
{     
 return
   iskey(kk)                ?
    stackText(Voc[class(kk)],
	myKey2Lex->nameoffset)
                            :
    "\177\177(NULL KEY)"        ;     
}

/********************************************/
  /* Iterates through the alias list */
BOOL nextName(KEY key, char **cpp)  
{ static LEXP lxp = 0 ;
  static KEY lastKey = 0 , kk ;
  
  if ( *cpp && key != lastKey)
    messcrash ("nextName %s called out of context", nakedName(key)) ;
 
  if (!*cpp)    /* initialise */
    kk = lastKey = key ;
  else         /* loop */
    { if (!lxp || !(lxp->lock & ALIASSTATUS)) /* shift once */
	return FALSE ;
      kk = (KEY) lxp->diskaddr ;
    }

  while (TRUE)
    { if (!isNakedKey(kk))
	return FALSE ;
      if (!(myKey2Lex->lock & EMPTYSTATUS))   /* if empty */
	break ;
      if (!(myKey2Lex->lock & ALIASSTATUS))  /* loop again */
	return FALSE ;
      kk = (KEY) myKey2Lex->diskaddr ;
    }
  lxp = myKey2Lex ;                         /* used by shift once */
  *cpp = nakedName(kk) ;
  return TRUE ;
}

/********************************************/

BOOL  lexReClass(KEY key,KEY *kp, int t)
{ return 
    lexword2key(name(key),kp,t) ;
}

/**********************************************/

static BOOL colonParse (char* note, char **class, char **item)
{
  char *cp ;

  for (cp = note ; *cp != ':' ; ++cp) 
    if (!*cp)
      return FALSE ;

  *cp = 0 ;
  *class = note ;
  *item = ++cp ;
  return TRUE ;
}

BOOL lexClassKey(char *text, KEY *kp)
{ int class ;
  char *cName, *kName ;
  *kp = 0 ;
  return
    colonParse (text, &cName, &kName) &&
      (class = pickWord2Class (cName)) &&
	(*(kName - 1) = ':') &&     /* Restore text */
	lexword2key (kName, kp, class) ;
}

/********************************************/
/********************************************/
/************ Hash Package ******************/

                     /* Hashing system */
#define SIZEOFint 8*sizeof(int)
#define ROTATION 13
#define LEFTOVER SIZEOFint - ROTATION

unsigned int hashText(char *cp, int n, int rotate)
{
  unsigned int i, h = 0 , j = 0, leftover = SIZEOFint - rotate ;

  chrono("hashText") ;  
  while(*cp)
    { h = freeupper(*cp++) ^  /* XOR*/
	( ( h >> leftover ) | (h << rotate)) ; 
    }
  /* compress down to n bits */
  for(j = h, i = n ; i< SIZEOFint; i+=n)
    j ^= (h>>i) ;
  chronoReturn() ;
  return
     j &  ( (1<<n) - 1  )  ;
}
      
/***************************************************/

 void lexReHashClass(int class)
{
  static KEYSET hashTable ;
  KEY key = 0 ; int n ;

  chrono("lexReHashClass") ;
  hashTable  = LexHashTable[class] ;
  if(!hashTable) 
    { n = 7 ;  /* chose size */
      while((1 << ++n) * .3  < lexMax(class)) ;
      nClass[class] = n ; 
    }
  else
  { if (keySetMax(hashTable) != 1<<nClass[class])
      messcrash
	("Class %s, arrayMax(hashTable[%d] = %d != 2^(nClass = %d)",
	 pickClass2Word(class), n, nClass[class]) ;
    n = ++nClass[class] ;
    keySetDestroy(hashTable) ;
  }
  LexHashTable[class] = arrayCreate(1<<n, KEY) ;
  keySet(LexHashTable[class], ( 1 << n ) - 1) = 0 ; /* make room */
  
  key = arrayMax(Lexi[class]) ;
  while(key--)
    lexHashInsert(class, key) ;
  chronoReturn() ;
}

/****************************************************/
   /* should only be called if !lexword2key */
   /* key must be just the KEYKEY part */
static int NII = 0 , NIC = 0 , NWC = 0 , NWF = 0 , NWG = 0 ;
static void lexHashInsert(int class, KEY key)
{
  static KEYSET hashTable ;
  unsigned int h ;
  int n , nn , rotate = 5 ;
  KEY *kp, *kpMin, *kpMax ;
  char *cp ;

  chrono("lexHashInsert") ;
  hashTable  = LexHashTable[class] ;
  if(arrayMax(Lexi[class]) > .34 * keySetMax(hashTable))
    lexReHashClass(class) ;

  cp = nakedName(KEYMAKE(class,key)) ;
  if (!*cp)
    return ;  /* may occur after a rename */
  h = hashText(cp, nClass [class], rotate) ;
  n = arrayMax(hashTable) ;
  nn = nClass[class] ;

  kp = arrp(hashTable, h , KEY) ;
  kpMin = arrp(hashTable, 0 , KEY) ;
  kpMax = arrp(hashTable, n -1, KEY) ;
 
  while ( n--)  /* circulate thru table, starting at h */
    { if(!*kp)      /* found empty slot, do insert */
	{ *kp = key ;
	  NII++ ;
	  chronoReturn() ;
	  return ;
	}
        /* circulate */
      if (rotate > 1)
	{ h = hashText(cp, nn, --rotate) ;
	  kp = arrp(hashTable, h , KEY) ;
	}
      else
	{ if (kp == kpMax)
	    kp = kpMin ;
	else
	  kp++ ;
	}
      NIC++ ;
    }
  messcrash("Hash table full, lexhashInsert is bugged") ;
}

/*
  I comment out this function 
  it needs plot so prevents acequery froom linking

  Note that i am not sure that it coincides with the 
  current strategy in hashInsert and word2key

#include "plot.h"
void lexHashTest(int class)
{
  static KEYSET hashTable ;
  unsigned int h ;
  int n, nn , nn1 , rotate ;
  KEY *kp, *kpMin, *kpMax ;
  int empty = 0 , full = 0 , j ;
  Array histo = arrayCreate(200,int) ;
  char *cp ;

  hashTable  = LexHashTable[class] ;
  if(!hashTable || !arrayMax(hashTable))
    {  printf("empty table\n") ;
       return ;
     }
  
  kpMin = arrp(hashTable, 0 , KEY) ;
  kpMax = arrp(hashTable, n -1, KEY) ;
  nn = nClass[class] ;
  nn1 = n = arrayMax(hashTable) ;
  for(h=0; h <nn; h++)
    { 
      kp = arrp(hashTable, h , KEY) ;
      rotate = 5 ;
      j = 0 ;
      if (!*kp)
	empty++ ;
      else
	{ full++ ;
	  cp = nakedName(KEYMAKE(class, *kp)) ;
	  n = nn ;
	  while ( n--)  
	    { if(!*kp)      
		{
		 array(histo,j,int)++ ;
		 break ;
		}

	      if (rotate > 1)
		{ h = hashText(cp, nn,--rotate) ;
		  kp = arrp(hashTable, h , KEY) ;
		}
	      else
		{ if (kp == kpMax)
		    kp = kpMin ;
		else
		  kp++ ;
		}
	      j++ ;
	    }
	}
    }
  plotHisto
    (messprintf("Hash correl %s", pickClass2Word(class)), histo) ;

}
*/


/********************************************/

static void lexHardRename(KEY key, char *new)
{
  char *old = nakedName(key) , *cpMin, *cpMax, *cp ;
  int class = class(key) ;
  Stack voc = Voc[class] ;

  if (!iskey(key))
    messcrash("lexHardRename received a bad key") ;
  
  cp = old ;
  while(*cp) cp++ ; /* search for a zero */
  while(!*cp) cp++ ; /* search for first non zero */
  cpMax = cp - 1 ;
  
  cp = old - 1 ;
  while(!*cp) cp-- ; /* search for first non zero */
  cpMin = cp + 2 ;

  if (cpMin < stackText(voc,0) ||
      cpMax >= stackText(voc,stackTextMax(voc)) ||
      cpMax - cpMin > strlen(new))
    { 
      cp = old ;
      while(*cp) *cp++ = 0 ;
      myKey2Lex->nameoffset = stackMark(voc) ;
      pushText(voc, lexcleanup(new)) ;
    }
  else
    while(*cpMin++ = *new++) ;

  if (myKey2Lex->lock & EMPTYSTATUS)
    { myKey2Lex->lock &= ~EMPTYSTATUS ;
      leximodif[class] = TRUE ;
    }
  vocmodif[class] = TRUE ;
  lexReHashClass(class) ;
}
  
/********************************************/

static void lexMakeAlias(KEY key, KEY alias, BOOL keepOldName)
{
  KEY *kp ;
  iskey(key) ;
  
  kp = (KEY *) (&(myKey2Lex->diskaddr)) ;
  *kp = alias ;
  myKey2Lex->lock = ALIASSTATUS ;  
  if (!keepOldName)
    { char *cp = nakedName(key) ;
      while (*cp) *cp++ = 0 ;
      myKey2Lex->lock |= EMPTYSTATUS ;
      lexReHashClass(class(key)) ;
    }
  iskey(alias) ;
  myKey2Lex->lock &= ~EMPTYSTATUS ;
  leximodif[class(key)] = TRUE ;
}
 
/********************************************/
  /* Returns TRUE if newName is accepted */
  /* if keepOldName, the 2 names will be used in the future,
   * so we need 2 keys anyway,
   * If not, we would rather keep a single key except if the old 
   * and new one allready exist because they may be referenced from
   * elsewhere.
   */

BOOL lexAlias(KEY key, char* newName, BOOL doAsk, BOOL keepOldName)
{
  char *cp, *oldName = name(key) ;
  int  class = class(key) ;
  KEY  newKey ;

  if(!isWriteAccess())
    { messerror("Sorry, lexAlias called without Write Access");
      return FALSE ;
    }

  if (!iskey(key))
    { messerror ("lex Rename called on unknown key %d", key) ;
      return FALSE ;
    }

  if (name(key) != nakedName(key))
    { if (doAsk)
	messout("%s is already aliased to %s, sorry, I cannot proceed",
		nakedName(key), name(key)) ;
      return FALSE ;
    }

  newName = lexcleanup(newName) ;

  if (!strcmp(oldName, newName)) /* Identity */
    return TRUE ;

  if (!lexstrcmp(oldName, newName)) /* Equivalence and certainly same length */
    { strcpy (oldName, newName) ;
      vocmodif[class] = TRUE ;
      return TRUE ; /* Do not rehash */
    }

/***** RMD 4/3/92 to allow hard renaming only *****/
/***** rewrote hard rename because it was bugged when I tested it *****/
/***** all the cpMin etc stuff seemed nonstandard and asking for trouble *****/
/***** couldn't see any reason to unset EMTPYSTATUS *****/

  if (lexword2key (newName, &newKey, class))
    { messout ("Sorry, fusing objects is not operational yet") ;
      return FALSE ;
    }
  else if (keepOldName)
    { messout ("Sorry, aliasing objects names is operational yet") ;
      return FALSE ;
    }
  else
    { if (strlen(newName) > strlen(oldName))
	{ for (cp = oldName ; *cp ;)
	    *cp++ = 0 ;
	  myKey2Lex->nameoffset = stackMark(Voc[class]) ;
	  leximodif[class] = TRUE ;	/* this seems to be necessary */
	  pushText (Voc[class], lexcleanup(newName)) ;
	}
      else
	{ for (cp = oldName ; *cp ;)
	    *cp++ = 0 ;
	  strcpy (oldName, newName) ;
	}

      vocmodif[class] = TRUE ;
      lexReHashClass(class) ;
      return TRUE ;
    }

/***************************************************/

       /* Hence on, operation is non trivial and requires unlocked objects */
  if (lexiskeylocked(key))
    { messout("Sorry, alias fails because %s is locked elsewhere",
	      name(key)) ;
      return FALSE ;
    }

  if (lexword2key(newName, &newKey, class))
      {                 /* I have to fuse the entries */
      if(lexAliasLoop(key, newKey))
	{ if (doAsk)
	   messout("Sorry, aliassing %s to %s would create a loop.",
		      oldName, newName) ;
	  return FALSE ;
	}
     }
  else     /* New key does not yet exists */
    { if (keepOldName)     
	lexaddkey(newName, &newKey, class) ;
      else
	{ lexHardRename(key, newName) ;
	  return TRUE ;
	}
    }

  if (lexiskeylocked(newKey))
    { if (doAsk)
	messout("Sorry, alias fails because %s is locked elsewhere",
	      name(newKey)) ;
      return FALSE ;
    }
  
  saveAll() ;  /* to empty the caches before manipulating LEXI */

  if ( iskey(key) == 2 &&
      iskey(newKey) == 1 )
    {  /* Just shift things over, but key must be changed on disk */
       LEXP newLex = KEY2LEX(newKey) ,
            lex = KEY2LEX(key) ;
       newLex->cache = 0 ; lex->cache = 0 ;
       blockAlias(key, newKey) ;  /* edit the key value on disk */
       newLex->addr = lex->addr ;
       newLex->diskaddr = lex->diskaddr ;
       lex->addr = 0 ;
       lex->diskaddr = 0 ;
       saveAll() ;
     }
  else if ( iskey(key) == 2 &&
      iskey(newKey) == 2 )
    {
      if (pickType(key) != 'B')
	{ if (doAsk)
	    messout (" I don t know how to Fuse non TREE objects, sorry") ;
	  return FALSE ;
	}
      if (doAsk)
	{ if(!graphQuery
	     (messprintf("%s and %s both exist.\n,"
			 "In case of conflict i will rather keep the data of %s.\n\n"
			 "Do you want to procees",
			 name(key), name(newKey), name(newKey))))
	    return FALSE ;
	}
      messout("dumpFuseKeys(key, newKey) ; not yet coded sorry") ;
    }

 
  lexMakeAlias(key, newKey, keepOldName) ;
  
  return TRUE ;
}
  
/********************************************/
/********************************************/
static KEY *KP ;
BOOL lexword2key(char *cp,KEY *key,int class)
                                    /* Case unsensitive  */
                            /*given a word *cp, sets its key in *ip */
                            /*search done only in the t lexique */
                            /*returns TRUE if found, FALSE if not */
                            /* No completion performed */
{
  unsigned int h ;
  int n , nn , rotate = 5 ;
  KEY *kp, *kpMin, *kpMax ;
  KEYSET hashTable ;
  int t = class ;
  LEXI *lexi ;
  char *voc ;


  if ( t<0 || t>=MAXTABLE )
    messcrash("lexword2key called on impossible class %d", class) ;

  if(!lexIsRead[class])
    lexReadTable(class) ;

  hashTable = LexHashTable [class] ;
  if ( !cp || !*cp || !nClass[class] )
    { *key = 0;    
      chronoReturn() ;
      return FALSE ;
    }

  chrono("lexword2key") ;
  cp = lexcleanup(cp) ;  /* remove spaces etc */

  voc = stackText(Voc[class],0) ;
  lexi = arrp(Lexi[class], 0 ,LEXI) ;  
  h = hashText(cp, nn = nClass [class], rotate) ;
  n = arrayMax(hashTable) ;
  kp = arrp(hashTable, h , KEY) ;
  kpMin = arrp(hashTable, 0 , KEY) ;
  kpMax = arrp(hashTable, n -1, KEY) ;
  while ( n--)  /* circulate thru table, starting at h */
    { if(!*kp)      /* found empty slot, cp is unknown */
	{ *key = 0 ;
	  KP = kp ;  /* private to be used by lexaddkey */
	  chronoReturn() ;
	  NWG++ ;
	  return FALSE ;
	}
      if( !lexstrcmp(cp, voc +  (lexi + *kp)->nameoffset)) 
	{ *key = KEYMAKE(class,*kp) ;
	  chronoReturn() ;
	  NWF++ ;
	  return TRUE ;   /* found */
	}
        /* circulate */
      if (rotate > 1 )
	{ h = hashText(cp, nn, --rotate) ;
	  kp = arrp(hashTable, h , KEY) ;
	}
      else
	{ if (kp == kpMax)
	    kp = kpMin ;
	else
	  kp++ ;
	}
      NWC++ ;
    }
  chronoReturn() ;
  KP = 0 ;
  return FALSE ;   /* Whole table scanned without success */
}

/**************************************************************************/
                 /* To obtain the next key in KEY  order        */
                 /* or the first entry if *key=0.               */
                 /* Updtates the value of *key,                 */
                 /* Returns 0 if the vocabulary is empty        */
                 /* or if *key is its last entry, 1 otherwise.  */
 
KEY lexLastKey(int class)
{
  KEY key, k = 0 ;
  
  while(lexNext(class,&k))
    key = k ;

  return key ;
}
 
/********/

BOOL lexNext(int class, KEY *k)
{
 static int i = -1 ;

 if(!lexIsRead[class])
   lexReadTable(class) ;

 if(!*k)       /* Find beginning of lexique */
    { i = -1;
      if(!class)
	{ i = 1 ; *k = i ; return TRUE ; }    /* prevents looping on class 0 */
    }

  if (Lexi[class]                              /* lexique non-empty */
     && (!*k || KEYKEY(*k) == i))  /* in order */
    while (i+1 < arrayMax(Lexi[class]))    /* not at end of lexique */
      {
	*k = KEYMAKE(class, ++i) ;
	if (!(arrp(Lexi[class], i, LEXI)-> lock & EMPTYSTATUS))
	 return TRUE;
      }

  *k = 0 ; i = 0  ;
  return FALSE ;
}

/********************************************/
void lexkill(KEY kk)      /*deallocates a key*/
{
 iskey(kk) ;

 chrono("lexkill") ;

 /*  saveAll() ;  must be done before  to empty the cache 
     but not here because lexkill is used by sessionControl 
  */
 if(myKey2Lex->diskaddr)
   { myKey2Lex->diskaddr = 0 ;
     leximodif[class(kk)] = TRUE ;
   }
 myKey2Lex->addr = 0 ;
 if ( ! (myKey2Lex->lock & EMPTYSTATUS ))
   { myKey2Lex->lock |= EMPTYSTATUS;
     leximodif[class(kk)] = TRUE ;
   }
 myKey2Lex->cache = 0 ;
 chronoReturn() ;
 return;
}
/***********************************/

/* Correctly sorts anything containing integers */
/* lexReanme relies on the fact that lexstrcmp must fail
 *  if the length differ
 */
int lexstrcmp(char *a,char *b)
{ register char c,d,*p,*q;

  chrono("lexstrcmp") ;
  while (*a)
    {                /* Bond7 should come before Bond68 */
      if (isdigit(*a) && isdigit(*b))
	{
	  for (p = a ; isdigit(*p) ; ++p) ;
	  for (q = b ; isdigit(*q) ; ++q) ;
	  if (p-a > q-b) { chronoReturn() ; return 1 ; }  /* the longer number is the bigger */
	  if (p-a < q-b) { chronoReturn() ;return -1 ; }
	  while (isdigit (*a))
	    { if (*a > *b) { chronoReturn() ; return 1 ; }
	      if (*a++ < *b++) { chronoReturn() ; return -1 ; }
	    }
	}
      else
        { if((c=freeupper(*a++))>(d=freeupper(*b++))) { chronoReturn() ; return 1 ; }
	  if(c<d) { chronoReturn() ; return -1 ; }
	}
    }
 if (!*b) { chronoReturn() ; return 0 ;}
 { chronoReturn() ;  return -1 ; }
 }

/**************************************************************/

BOOL  lexaddkey(char *cp,KEY *kptr,int t)
                   /*add to the t lexique the word *cp */
                   /*returns TRUE if added, FALSE if known */
{
 KEY k;
 LEXI ai;

 chrono("lexaddkey") ;

 KP = 0 ;
  if(cp && lexword2key (cp,kptr,t))
    { chronoReturn() ;
      return FALSE ;
    }
				/*word allready known*/
 if(!lexIsRead[t])
   { lexReadTable(t) ;
     KP = 0 ;
   }

                                /* initialise the lexique */
 if (!Lexi[t])
   { int nBits = 8 ;
     Voc[t] = stackCreate(1 << (nBits + 2)) ;
     Lexi[t] = arrayCreate(1 << (nBits - 1), LEXI);
     LexHashTable[t] = arrayCreate(1<<nBits, KEY);
     keySet(LexHashTable[t], (1<<nBits) - 1) = 0 ; /* make room */
     nClass[t] = nBits ;
  
     if (t && (pickType(KEYMAKE(t,0)) == 'B'))
       lexaddkey (messprintf ("?%s", pickClass2Word(t)), &k, t) ;
     if (lexaddkey("\177\176(NULL KEY)",&k,t) &&
	 KEY2LEX(k) )
       KEY2LEX(k)-> lock = EMPTYSTATUS ;  /* i.e. never display it */
     KP = 0 ;
   }
 
 if (!cp || !*cp) /* call with cp=0 used for initialisation */
   {
     *kptr = KEYMAKE(t,0);
     chronoReturn() ;
     return TRUE ;
   }

 k = (KEY) arrayMax(Lexi[t]) ;
 if ((k >= MAXLEX) || ((stackTextMax(Voc[t]) + strlen(cp)) > MAXVOCAB)) {
    messcrash("Lexique %d full",t);
 }
      /*create a lex record*/
 ai.nameoffset = stackMark(Voc[t]) ;
 ai.lock=0;
 ai.diskaddr=NULLDISK;
 ai.addr=NULL;
 ai.cache = NULL;

      /* add the word to the end of vocabulary */
 pushText(Voc[t], lexcleanup(cp)) ;
         /*add an entry to the end of lex */
  array(Lexi[t], (int) k, LEXI) = ai ; 
/* note that Voc and Lexi must be updated first 
   for lexHashInsert to work */

 if(KP)
   { chrono("Found KP") ; chronoReturn() ;
     *KP = k ;
     /* direct insertion , be carefull if you touch the code */
     /* KP is inherited from the last call too lexword2key */
     if(arrayMax(Lexi[t]) > .34 * keySetMax(LexHashTable[t]))
       lexReHashClass(t) ;
   }
 else
   lexHashInsert(t, k) ;
 
 leximodif[t] = lexhmodif[t] = vocmodif[t] = TRUE ;
  *kptr = KEYMAKE(t,k) ;
  chronoReturn() ;
  return TRUE;
}

/*************************************************/

int lexMax(int t)
{
  return nEntries(t);
}

/*************************************************/

int vocMax(int t)
{
  return Voc[t] ? stackMax(Voc[t]) : 0 ;
}

/*************************************************/
/*************************************************/



