/*  File: model.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 **  Reads the models of the ACeDB program.                   
 * Exported functions:
     readModels
 * HISTORY:
 * Last edited: Mar 26 13:13 1992 (mieg)
 * * Feb  2 11:46 1992 (mieg): hardCrash routine modification
 * Created: Wed Nov  6 13:27:39 1991 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "array.h"
#include "lex.h"
#include "pick.h"
#include "bs_.h"
#include "bs.h"     /* bsCreate etc */
#include "session.h"
extern int getuid(void) , geteuid(void) ;  /* Unix */ 
extern void saveAll(void) ;
extern void wormClose(void) ;
extern sessionHardCrash(char* s) ;   /* If model read failure exit immediatly */
#define MTEST FALSE

/************************************************************************/

static int line ;

static int getLine (Array aNode, int *plevel)	/* returns FALSE at eof */
{
  static FILE *fil ;
  static Array levelCount ;
  static Stack sword ;		/* holds arbitrarily long string */
  char c,*word ;
  KEY dummy ;
  int iNode = 0,
      counter = 0,
      inword = FALSE,
      eow = FALSE,
      eol,level,table ;

  if (!fil)
    { if (!(fil =  filopen ("wspec/models","wrm","r")))
	messcrash ("Can't find model file") ;
      sword = stackCreate (64) ;
      levelCount = arrayCreate (8,int) ;
      array(levelCount,0,int) = 0 ;
      line = 1 ;
    }

  arrayMax(aNode) = 0 ; 
  for (eol = FALSE ; !eol && !feof(fil) ; ++counter)
    { switch (c = getc (fil))
	{
	case '\t' : 
	  while (++counter%8) ;
	  --counter ; eow = TRUE ; break ;
        case '/' :
          while((c = getc(fil)) && c != '\n' && c != EOF ) ;
	  /* empty line and fall through */
	case '\n' : case EOF :
	  eol = TRUE ; eow = TRUE ; break ;
	case ' ' :
	  eow = TRUE ; break ;
        default :
	  if (!inword)
	    { if (!iNode)	/* find level */
		{ for (level = 0 ; level < arrayMax(levelCount) ; level++)
		    if (arr(levelCount,level,int) >= counter)
		      break ;
		  if (level == arrayMax(levelCount) ||
		      arr(levelCount,level,int) > counter)
		    sessionHardCrash
		      (messprintf
		       ("Model file: tree screwup line %d level %d",line,level)) ;
		  *plevel = level ;
		}
	      array(levelCount,level,int) = counter ;
	      ++level ;
	      stackClear (sword) ;
	      inword = TRUE ; eow = FALSE ;
	    }
	  push (sword,c,char) ;
	}
      if (inword && eow)			/* end of word */
	{ push(sword,0,char) ;
	  word = stackText(sword,0) ;
	  if (lexword2key (word,arrayp(aNode,iNode,KEY),0) != 1)
	    { 
	      if ( ( *word == '?' || *word == '#') &&  /* Class pointer or type */
		  ( table = pickWord2Class(word+1)))
		{ array(aNode,iNode,KEY) = KEYMAKE(table,*word == '#' ? 1 : 0) ;
		  if(!lexMax(table))  /* New class being defined */
		    lexaddkey(0,&dummy,table) ;
		}
	      else
		sessionHardCrash
		  (messprintf
		   ("Model file : unknown item {%s} line %d",
		    word,line)) ;
	    }
	  ++iNode ;
	  inword = FALSE ;
	}
    }

if (MTEST)
  { fprintf (stderr,"%d : ",*plevel) ;
    for (iNode = 0 ; iNode < arrayMax(aNode) ; ++iNode)
      fprintf (stderr,"%s ",name(arr(aNode,iNode,KEY))) ;
    fprintf (stderr,"\n") ;
  }

  if (feof (fil))
    { fclose (fil) ;
      fil = 0 ;
      arrayDestroy (levelCount) ;
      stackDestroy (sword) ;
      return FALSE ;
    }

  ++line ;
  return TRUE ;
}

/******************************************************************/

  /* updates the models 
   * called by update() and readModels() 
   */
extern BOOL READING_MODELS ;   /* in bssubs.c */
BOOL getNewModels (void)
{
  Array aNode,aDown ;
  KEY   key ;
  OBJ   obj = 0 ;
  BS    bs ;
  int   i,level ;
  int   table ;

  if(!isWriteAccess())
    { messout("Sorry, you do not have Write Access");
      return FALSE ;
    }

  aNode = arrayCreate (4,KEY) ;
  aDown = arrayCreate (8,BS) ;		/* available down pointers */
  READING_MODELS = TRUE ;


  while (getLine (aNode,&level))
    if (arrayMax(aNode))
      { i = 0 ;
	if (!obj)	      	/* make it - 1st node is object key */
	  { key = arr(aNode,i++,KEY) ;
	    if (level > 0)
	      sessionHardCrash (messprintf (
"New model %s starts inset at line %d - perhaps an extra white line?", 
					 name(key), line)) ;
	    table = class(key) ;
	    obj = bsUpdate (key) ;
	    if (!obj)
	      sessionHardCrash (messprintf (
		"Can not form model for %s at line %d",
					 name(key), line)) ;
	    bs = obj->root ;
	    bsTreePrune(bs->right); /* Kill old version of model */
	    bs->right = BSalloc() ;  /* very first addition is right */
	    bs->right->up = bs ;
	    bs = bs->right ;
	    bs->key = arr (aNode,i++,KEY) ;
	    ++level ; /* hocus pocus */
	    array(aDown,level++,BS) = bs ;
	  }
	else if (level > arrayMax(aDown))
	  sessionHardCrash (messprintf (
		   "Model screwup: impossible level line %d",
				     line)) ;
	else
	  { bs = arr(aDown,level,BS) ;
	    bs->down = BSalloc() ;		/* first addition is down */
	    bs->down->up = bs ;
	    bs = bs->down ;
	    bs->key = arr (aNode,i++,KEY) ;
	    array(aDown,level++,BS) = bs ;
	  }
	
	while (i < arrayMax(aNode))
	  { bs->right = BSalloc () ;	/* rest are right */
	    bs->right->up = bs ;
	    bs = bs->right ;
	    bs->key = arr (aNode,i++,KEY) ;
	    array(aDown,level++,BS) = bs ;
	  }
	arrayMax(aDown) = level ;
      }
    else if (obj)	/* close it */
      { 
	if (MTEST)
	  bsTreeDump (obj->root) ;
	cacheMark (obj->x) ;
	bsSave (obj) ;
	bsMakePaths (table) ;
	obj = 0 ;
      }
  
  if (obj)
    { cacheMark (obj->x) ;
      bsSave (obj) ;
      bsMakePaths (table) ;
    }

  READING_MODELS = FALSE ;
  return TRUE ;
}

/*******************************************************/
/***** Authorisation checks ****************************/

 void readModels (void)
{
  FILE *f ;

  /* Only the acedb administrator is entitled to proceed
   * I check this by trying to open passwd.wrm, a
   * In the install documentation, it is stated that
   * this file should be set chmod 644
   */

         /* First you must own the running process */

  if ( (getuid() != geteuid()) 
      || (!(f = filopen("wspec/passwd","wrm","a"))) ) 
    { messout
	("%s%s","Sorry, only the administrator owning ",
	 "the $(ACEDB) directory can modify the models");
      fclose(f) ;
      return ;
    }
  fclose(f) ;

  if(thisSession.session != 1 && isWriteAccess())
    if (graphQuery("You did not save the current session: should I ?"))
	sessionClose(TRUE) ;
      else
	return ;

  sessionWriteAccess() ;
  
  if(!isWriteAccess())
    { messout("%s %s",
	      "Sorry, you do not have Write Access,",
	      "I cannot update the models.") ;
      return ;
    }

  if(thisSession.session != 1 && !graphQuery
     (messprintf
      ("%s%s%s",
       "Watch out ! An erroneous modification of the models ",
       "may screwup the system.  ",
       "Do you want to proceed"))) 
    return ;

  getNewModels() ;
}

/********************************************************/
/********************************************************/

