/*  File: freesubs.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: free format input - record based
 * Exported functions: lots - see regular.h
 * HISTORY:
 * Last edited: Apr  3 15:32 1992 (mieg)
 * Created: Sun Oct 27 18:16:01 1991 (rd)
 *-------------------------------------------------------------------
 */
 
#include "regular.h"
#include "array.h"
#include <ctype.h>
 
static int isInteractive = TRUE ;      /* can get set FALSE somehow ? */
 
#define MAXSTREAM 8
#define MAXNPAR 8
typedef struct
  { FILE *fil ;
    char *text ;
    char special[24] ;
    int npar ;
    int parMark[MAXNPAR] ;
    int line ;
  } STREAM ;
static STREAM   stream[MAXSTREAM] ;
static int	streamlevel ;
static FILE	*currfil ;	/* either currfil or currtext is 0 */
static char	*currtext ;	/* the other is the current source */
static Stack    parStack ;
static int	maxcard = 1024 ;
static  char    *card, *word, *cardEnd, *pos ;
static Associator filAss ;
 
#define _losewhite    while (*pos == ' '|| *pos == '\t') ++pos
#define _stepover(x)  (*pos == x && ++pos)
#define _FREECHAR     (currfil ? fgetc (currfil) : *currtext++)
 
/************************************/
 
void freeinit ()
{ static BOOL isInitialised = FALSE ;
  
  if (!isInitialised)
    { streamlevel = 0 ;
      currtext = 0 ;
      stream[streamlevel].fil = currfil = stdin ;
      stream[streamlevel].text = 0 ;
      freespecial ("\n;/%\\@$") ;
      card = messalloc (maxcard) ;
      cardEnd = &card[maxcard-1] ;
      pos = card ;
      word = messalloc (maxcard) ;
      filAss = assCreate () ;
      parStack = stackCreate (128) ;
      isInitialised = TRUE ;
    }
}

/*******************/

static void freeExtend (char **pin)
{	/* only happens when getting card */
  char *oldCard = card ;

  maxcard *= 2 ;
  card = messalloc (maxcard) ;
  memcpy (card, oldCard, maxcard/2) ;
  cardEnd = &card[maxcard-1] ;
  *pin += (card - oldCard) ;
  word = messalloc (maxcard) ;
}
 
/********************/

static char special[256] ;

void freespecial (char* text)
{
  if (!text)
    messcrash ("freespecial received 0 text") ;
  if (strlen(text) > 23)
    messcrash ("freespecial received a string longer than 23") ;
  if (text != stream[streamlevel].special)
    strcpy (stream[streamlevel].special, text) ;
  memset (special, 0, (mysize_t) 256) ;
  while (*text)
    special [((int) *text++) & 0xFF] = TRUE ;
  special[0] = TRUE ;
  special[EOF] = TRUE ;		/* better have these to ensure streams terminate! */
}
 
/********************/
 
void freeforcecard (char *string)
{
  freesettext (string, "") ;
  freespecial ("") ;
  freecard (0) ;
}
 
/********************/
 
BOOL freecard (int level)	/* returns FALSE when streamlevel drops below level */
{ 
  char *in,ch,*cp ;
  int kpar ;
  int isecho = FALSE ;		/* could reset sometime? */
  FILE *fil ;

restart :
  if (level > streamlevel)
    return FALSE ;

  if (isecho)
    printf (!currfil ? "From text >" : "From file >") ;
  in = card ; --in ;
 
  while (TRUE)
    { if (++in >= cardEnd)
	freeExtend (&in) ;

	/*
	 *  The next test was added to avoid reading past the EOF
	 *  even when *in == (char) EOF would fail.
	 */

	if (currfil && feof (currfil)) {
	    if (streamlevel)
	      { if (currfil)
		  fclose (currfil) ;
		for (kpar = stream[streamlevel].npar ; kpar-- ;)
		  popText (parStack) ;
		--streamlevel ;
		currfil = stream[streamlevel].fil ;
		currtext = stream[streamlevel].text ;
		freespecial (stream[streamlevel].special) ;
	      }
	    goto got_line;
	}

	/*
	 *  Please notice that feof(currfil) may be true
	 *  even if (char) EOF is not matched !!!
	 *  So the fgetc(currfil) may never give the true
	 *  EOF indication. Relying on the next macro
	 *  combined with switch using `char *in' is a bug.
	 *
	 *  #define _FREECHAR     (currfil ? fgetc (currfil) : *currtext++)
	 */
      *in = _FREECHAR ;
      if (special[((int) *in) & 0xFF] && *in != '$' && *in != '@' )
	switch (*in)
	  {
	  case '\n':		/* == '\x0a' */
	  case ';':		/* card break for multiple commands on one line */
	    goto got_line ;
/*	  case (char) EOF:	*/
	  case '\0':
	    if (streamlevel)
	      { if (currfil)
		  fclose (currfil) ;
		for (kpar = stream[streamlevel].npar ; kpar-- ;)
		  popText (parStack) ;
		--streamlevel ;
		currfil = stream[streamlevel].fil ;
		currtext = stream[streamlevel].text ;
		freespecial (stream[streamlevel].special) ;
	      }
	    goto got_line;
	  case '/':		/* // means start of comment */
	    if ((ch = _FREECHAR) == '/')
	      { while ((ch = _FREECHAR) != '\n' && ch != EOF) ;
		goto got_line ;
	      }
	    else
	      { if (isecho) putchar (*in) ;
		if (currfil)                     /* push back ch */
		  ungetc (ch, currfil) ;
		else
		  --currtext ;
	      }
	    break ;
	  case '%':		/* possible parameter */
	    if (isdigit (ch = _FREECHAR) &&
		(kpar = ch-49) >= 0 && kpar < stream[streamlevel].npar)
	      { cp = stackText (parStack, stream[streamlevel].parMark[kpar]) ;
		for (--in ; *cp ; ++cp)
		  { *++in = *cp ;
		    if (isecho)
		      putchar (*in) ;
		  }
	      }
	    else
	      { if (isecho) putchar (*in) ;
		if (currfil)
		  ungetc (ch, currfil) ;
		else
		  --currtext ;
	      }
	    break ;
	  case '\\':		/* escapes next character - continuation at EOL */
	    *in = _FREECHAR ;
	    if (*in == '\n')
	      { if (isInteractive && !streamlevel)  /* JTM guess */
		  printf ("  Continuation >") ;
		stream[streamlevel].line ++ ;  /* JTM */
		--in ;
	      }
	    else if (*in == '"') /* restore backslash for freeword to read */
	      { *in = '\\' ;
		*++in = '"' ;
	      }
	    break ;
	  default:
	    messout ("freesubs got unrecognized special character 0x%x = %c\n",
		     *in, *in) ;
	  }
      else
	{ if (!isprint(*in))
	    --in ;
	  else if (isecho)	/* write it out */
	    putchar (*in) ;
	}
    }				/* while TRUE loop */
  messout ("inaccessible statement in freecard()") ;
 
got_line:
  stream[streamlevel].line ++ ;
  *in = 0 ;
  if (isecho)
    putchar ('\n') ;
  pos = card ;
  _losewhite ;
  if (_stepover ('@') && special['@'])        /* command file */
    { if (fil = freeopen ("commands","cmd","r"))
	freesetfile (fil, pos) ;
      goto restart ;
    }
  if (_stepover ('$') && special['$'])        /* shell command */
    { system (pos) ;
      goto restart ;
    }

  return TRUE ;
}
 
/************************************************/
 
BOOL freeread (FILE *fil)                /* reads card from fil */
{
  char *in = card ;
  char ch ;
  int  *line ;
  
  if (!assFind (filAss, fil, &line))
    { line = (int*) messalloc (sizeof (int)) ;
      assInsert (filAss, fil, line) ;
    }
 
  --in ;
  while (TRUE)
    { ++in ;
      if (in >= cardEnd)
	freeExtend (&in) ;
      switch (*in = getc(fil))
        {
	case '\n' :
	  ++*line ;
	case (char) EOF :
	  goto got_line ;
	case '/' :		/* // means start of comment */
	  if ((ch = getc (fil)) == '/')
	    { while (getc(fil) != '\n' && !feof(fil)) ;
	      ++*line ;
	      if (in > card)	/* // at start of line ignores line */
		goto got_line ;
	      else
		--in ; /* in = 0   unprintable, so backstepped */
	    }
	  else
	    ungetc (ch,fil) ;
	  break ;
	case '\\' :		/* escape next character */
	  *in = getc(fil) ;
	  if (*in == '\n')	/* continuation */
	    { ++*line ;
	      while (isspace (*in = getc(fil))) ;    /* remove whitespace */
	    }
	  else if (*in == '"')	/* restore backslash for freeword to read */
	    { *in = '\\' ; 
	      *++in = '"' ;
	    }
	  /* NB fall through - in case next char is nonprinting */
	default:
	  if (!isprint (*in))	/* ignore control chars, e.g. \x0d */
	    --in ;
	}
    }
  messout ("card overflow in freeread()") ;
 
got_line :
  *in = 0 ;
  pos = card ;
  _losewhite ;
  if (feof(fil))
    { assRemove (filAss, fil) ;
      messfree (line) ;
    }
  return *pos || !feof(fil) ;
}

int freeline (FILE *fil)
{ int *line ;

  if (assFind (filAss, fil, &line))
    return *line ;
  else
    return 0 ;
}
 
int freestreamline (int level)
{ 
  return   
    stream[streamlevel].line  ;
}
 
/********************************************/


static void freenewstream (char *parms)
{
  int kpar ;

  stream[streamlevel].fil = currfil ;
  stream[streamlevel].text = currtext ;
  if (++streamlevel == MAXSTREAM)
    messcrash ("MAXSTREAM overflow in freenewstream") ;
  strcpy (stream[streamlevel].special, stream[streamlevel-1].special) ;

  pos = parms ;			/* abuse freeword() to get parms */

  for (kpar = 0 ; kpar < MAXNPAR && freeword () ; kpar++) /* read parameters */
    { stream[streamlevel].parMark[kpar] = stackMark (parStack) ;
      pushText (parStack, word) ;
    }

  stream[streamlevel].npar = kpar ;
  stream[streamlevel].line  = 1 ;
  pos = card ;			/* restore pos to start of blank card */
  *card = 0 ;
}
 
int freesettext (char *string, char *parms)
{
  freenewstream (parms) ;

  currfil = 0 ;
  currtext = string ;

  return streamlevel ;
}

int freesetfile (FILE *fil, char *parms)
{
  freenewstream (parms) ;

  currfil = fil ;
  currtext = 0 ;

  return streamlevel ;
}

/************************************************/
/* freeword(), freewordcut() and freestep() are the only calls that
     directly move pos forward -- all others act via freeword().
   freeback() moves pos back one word.
*/
 
char *freeword ()
 
 {char *cw ;
 
  _losewhite ;             /* needed in case of intervening freestep() */
  if (_stepover ('"'))
   {for (cw = word ; *pos ; *cw++ = *pos++)
      { if (_stepover('"'))
	  break ;
        if (_stepover('\\'))	/* skip next char, but break if end of line */
	  if (!*pos)
	    break ;
	  else
	    ++pos ;
      }
   }
  else
    for (cw = word ; isgraph (*pos) && !_stepover(',') ; *cw++ = *pos++) ;
 
  _losewhite ;
  *cw = 0 ;
  return *word ? word : 0 ;
 }
 
/************************************************/
 
char *freewordcut (char *cutset, char *cutter)
        /* Moves along card, looking for a character from cut, which is a
           0-terminated char list of separators.
           Returns everything up to but not including the first match.
           pos is moved one char beyond the character.
           *cutter contains the char found, or if end of card is reached, 0.
        */
 {char *cc,*cw ;
 
  for (cw = word ; *pos ; *cw++ = *pos++)
    for (cc = cutset ; *cc ; ++cc)
      if (*cc == *pos)
        goto wcut ;
wcut:
  *cutter = *pos ;
  if (*pos)
    ++pos ;
  _losewhite ;
  *cw = 0 ;
  return *word ? word : 0 ;
 }
 
/************************************************/
 
void freeback (void)    /* goes back one word - inefficient but reliable */
 
 {char *now = pos ;
  char *old = pos ;
 
  pos = card ; _losewhite ;
  while  (pos < now)
   {old = pos ;
    freeword () ;
   }
  pos = old ;
 }
 
/************************************************/
 
BOOL freeint (int *p)
 
 {char *keep = pos ;
 int  old = *p ;
  char dummy ;
 
  if (freeword () && (sscanf (word,"%d%c",p,&dummy) == 1))
                        /* dummy trick checks no unmatched characters */
    return (TRUE) ;
  else
   {pos = keep ;
    *p = old ;
    return (FALSE) ;
   }
 }
 
/*****************************/
 
BOOL freefloat (float *p)
 
 {char *keep = pos ;
  float old = *p ;
  char dummy ;
 
  if (freeword () && (sscanf (word,"%f%c",p,&dummy) == 1))
    return (TRUE) ;
  else
   {pos = keep ;
    *p = old ;
    return (FALSE) ;
   }
 }
 
/**************************************************/
 
BOOL freedouble (double *p)
 
 {char *keep = pos ;
  double old = *p ;
  char dummy ;
 
  if (freeword () && (sscanf (word,"%lf%c",p,&dummy) == 1))
    return (TRUE) ;
  else
   {pos = keep ;
    *p = old ;
    return (FALSE) ;
   }
 }
 
/**************************************************/
 
FILE *freeopen (char *defname, char *ending, char *spec)
 
 {FILE *result ;
  char nambuf[128],*cp,*np ;
  int ifend = FALSE ;
 
  cp = freeword () ? word : defname ;
  np = nambuf ;
  while (*cp)
   {if (*cp == '.')
      ifend = TRUE ;
    *np++ = *cp++ ;
   }
  if (!ifend)
   {*np++ = '.' ;
    strcpy (np,ending) ;
   }
  else
    *np = 0 ;
 
  if (result = fopen (nambuf,spec),result)
    return result ;
  else
   {messout ("Failed to open %s",nambuf) ;
    return FALSE ;
   }
 }
 
/*************************************************/
 
static int ambiguouskey;
 
BOOL freekey (KEY *kpt, FREEOPT *options)
{
  char  *keep = pos ;

  if (!freeword())
    return FALSE ;

  ambiguouskey = FALSE ;
  if (freekeymatch (word, kpt, options))
    return TRUE;
 
  if (ambiguouskey)
    messout ("Keyword %s is ambiguous",word) ;
  else if (word[0] != '?')
    messout ("Keyword %s does not match",word) ;
 
  pos = keep ;
  return FALSE ;
}
 
/*****************/
 
BOOL freekeymatch (char *cp, KEY *kpt, FREEOPT *options)
{
  char  *io,*iw ;
  int   nopt = (int)options->key ;
  KEY   key ;
 
  ambiguouskey = FALSE;
  if (!nopt || !cp)
    return FALSE ;
 
  while (TRUE)
    { iw = cp ;
      io = (++options)->text ;

      while (freeupper (*iw++) == freeupper(*io++))
      if (!*iw)
        goto foundit ;
      if (!--nopt)
        return FALSE ;
    }
 
foundit :
  key = options->key ;
 
  while (--nopt)                /* check that later options are different */
    { io = (++options)->text ;
      iw = word ;
      while (freeupper (*iw++) == freeupper (*io++))
        if (!*iw)
          { ambiguouskey = TRUE;
            return FALSE ;
          }
    }
 
  *kpt = key ;
  return TRUE ;
}
 
/***************************************************/
 
BOOL freeselect (KEY *kpt, FREEOPT *options)     /* like the old freemenu */
{
  if (isInteractive)
    printf ("%s > ",options[0].text) ;
  freecard (0) ;                       /* just get a card */
  if (isInteractive)
    while (freestep ('?'))            /* write out options list */
      { int i ;
	for (i = 1 ; i <= options[0].key ; i++)
	  printf ("  %s\n",options[i].text) ;
	printf ("%s > ",options[0].text) ;
	freecard (0) ;
      }
  return freekey (kpt,options) ;
}
 
/**************************************/
 
BOOL freequery (char *query)
{
  if (isInteractive)
    { int retval, answer = 0 ;
      printf ("%s (y or n) ",query) ;
      answer = getchar () ;
      retval = (answer == 'y' || answer == 'Y') ;
      while (answer != '\n')
        answer = getchar () ;
      return retval ;
    }
  else
    return TRUE ;
}
 
/**********/
 
BOOL freeprompt (char *prompt, char *dfault, char *fmt)
{ 
  if (isInteractive)
    printf("%s ? > ",prompt);
  freecard (0) ;                       /* just get a card */
  if (freecheck (fmt))
    return TRUE ;
  else
    { messout ("input mismatch : format '%s' expected, card was\n%s",
	       fmt,card) ;
      return FALSE ;
   }
}
 
/*************************************/
 
int freefmtlength (char *fmt)
 
 {char *cp ;
  int length = 0 ;
 
  if (isdigit(*fmt))
   {sscanf (fmt,"%d",&length) ;
    return length ;
   }
 
  for (cp = fmt ; *cp ; ++cp)
    switch (*cp)
     {
case 'i' : case 'f' : case 'd' : length += 8 ; break ;
case 'w' : length += 32 ; break ;
case 't' : length += 80 ; break ;
case 'o' :
      if (*++cp)
        messcrash ("'o' can not end free format %s",fmt) ;
      length += 2 ; break ;
     }
 
  if (!length)
    length = 40 ;
  return length ;
 }
 
/****************/
 
BOOL freecheck (char *fmt)
        /* checks that whatever is in card fits specified format
           note that 't' format option changes card by inserting a '"' */
 {char *keep = pos ;
  union {int i ; float r ; double d ;}
          target ;
  char *fp,*start ;
  int nquote = 1 ;
 
  for (fp = fmt ; *fp ; ++fp)
    switch (*fp)
     {
case 'w' : if (freeword ()) break ; else goto retFALSE ;
case 'i' : if (freeint (&target.i)) break ; else goto retFALSE ;
case 'f' : if (freefloat (&target.r)) break ; else goto retFALSE ;
case 'd' : if (freedouble (&target.d)) break ; else goto retFALSE ;
case 't' :      /* must insert '"' and double any remaining '"'s */
      for (start = pos ; *pos ; ++pos)
        if (*pos == '"')
          ++nquote ;
      for ( ; pos >= start ; --pos)
       {*(pos + nquote) = *pos ;
        if (*pos == '"')
          *(pos + --nquote) = '"' ;
       }
      *start = '"' ;
      goto retTRUE ;
case 'z' : if (freeword ()) goto retFALSE ; else goto retTRUE ;
case 'o' :
      if (!*++fp) messcrash ("'o' can not end free format %s",fmt) ;
      freestep (*fp) ; break ;
default :
      if (!isdigit(*fp) && !isspace(*fp))
        messout ("unrecognised char %d = %c in free format %s",*fp,*fp,fmt) ;
     }
 
retTRUE :
  pos = keep ; return TRUE ;
retFALSE :
  pos = keep ; return FALSE ;
 }
 
/************************ little routines ************************/
 
BOOL freestep (char x)
 {return (*pos && freeupper (*pos) == x && pos++) ;
 }
 
void freenext (void)
 {_losewhite ;
 }
 
char FREE_UPPER[] =
{ 0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,
  16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,
  32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,
  48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,
  64,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,
  80,81,82,83,84,85,86,87,88,89,90,91,92,93,94,95,
  96,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,
  80,81,82,83,84,85,86,87,88,89,90,123,124,125,126,127
} ;

char FREE_LOWER[] =
{  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15, 
  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,  
  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,  
  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,  
  64,  97,  98,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122,  91,  92,  93,  94,  95,
  96,  97,  98,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127
} ;

char* freepos (void)		/* cheat to give pos onwards */
{ return pos ;
}

char freechar (void)
{ return  *pos ? *pos++ : 0 ;
}

/*********** end of file *****************/
