/*  File: display.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 **       Class sensitive object display                           **
          Shares and reuse display windows.
 * Exported functions:
   display 
     displays a Key according to diplayType 
     if displayType==0, it defaults according to classes.wrm.
     Each display type reuses its own window until that window has 
       been preserved with displayPreserve.

   displayBlock
     puts up a message over the current active window and blocks 
       the display system until a key has been picked somewhere.
     the rest of the program still runs.
     The registered function receives as parameter the picked key 
       and, implicitly, the graph it was picked from, which is 
       active.

   displayUnBlock
     Clears out.

   displayRepeatBlock
     Reactivates the current blocking function.

   isDisplayBlocked
     should be called by clients to prevent access to blocked displays

   displayPreserve
     preserves the object, so that the next object of that class
     will not reuse its window.
 * HISTORY:
 * Last edited: Apr  2 13:48 1992 (mieg)
 * Created: Sun Dec  1 17:29:54 1991 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "lex.h"
#include "graph.h"
#include "display.h"
#include "pick.h"

#define MAXNDISPLAY 16
static int nDisplays = 0 ;
static int maxDisplayType = 0 ;
static Graph previous[64] ;  /* Hard limit on display types */
DisplayFunc displayFunc[64] ;  /* Hard limit on display types */
static Graph blockGraph = 0 ;
static BOOL isRepeatBlock = FALSE ;
static BlockFunc blockFunc ;

/*************************************************/
    /* displayType is declared int,
       not displayType to avoid recompilations.
       */

BOOL display (KEY key, KEY from, int displayType)
{
  Graph g = graphActive() ;	/* save */
  BOOL gReuse ;
    
  if (graphActivate (blockGraph))
    { (*blockFunc) (key) ;
      if (isRepeatBlock)
	isRepeatBlock = FALSE ;
      else
	{ if (graphActivate (blockGraph))
	    graphUnMessage () ;
	  blockGraph = 0 ;
	}
      graphActivate (g) ;	
      return FALSE ;
    }

  if (nDisplays >= MAXNDISPLAY)
    { messout ("Too many displayed objects - delete some first") ;

      return FALSE ;
    }
  
  if (!displayType)
    displayType = pickDisplayType (key) ;

  if (!displayType) /* check before previous[] */
    return FALSE ;    /* not displayable */

  if (!KEYKEY(key) && pickType(key) == 'B')
    displayType = 1  ; /* 1 must be TREE for the  model */
  
  if(displayType>63)
    messcrash
      ("%s%s",
       "You are tresspassing a hard limit of 63 display types",
       "Modify the file display.c and recompile the kernel") ;
	      

  if (g = previous[displayType]) 
    { if (graphActivate(g))
	{ gReuse = TRUE ;
	}
      else
	{ previous[displayType] = 0 ;
	  gReuse = FALSE ;
	}
    }
  else 
    gReuse = FALSE;

  if(displayType > maxDisplayType)
    maxDisplayType = displayType ;

  if(displayType &&
     displayFunc[displayType] &&
     displayFunc[displayType](key,from,gReuse))
    { previous[displayType] = graphActive() ;
      graphPop () ;
      return TRUE ;
    }
  else
    return FALSE ;
}
  
/**********************************/

void displayPreserve (void)
{
  Graph g = graphActive() ;
  int i = maxDisplayType + 1 ;

  while (i--)
    if (previous[i] == g)
      previous[i] = 0 ;
}

/**************************************/

BOOL displayBlock (BlockFunc func, char *message)
{
  if (blockGraph)
    return FALSE ;
  blockGraph = graphActive() ;
  isRepeatBlock = FALSE ;
  blockFunc = func ;
  graphMessage (messprintf ("%s\n\n%s",
    "Select an object by double-clicking as "
    "though you were going to display it.",
    message ?  message : "")) ;
  return TRUE ;
}

/**************************************/

void displayRepeatBlock (void)
{
  isRepeatBlock = TRUE ;
}

/**************************************/

void displayUnBlock(void)
{
  if (blockGraph)		/* prevent callback recursion */
    { blockGraph = 0 ;
      graphUnMessage () ;
    }
  isRepeatBlock = FALSE ;
  blockFunc = 0 ;
}

/**************************************/

BOOL isDisplayBlocked (void) 
{
  return (blockGraph != 0) ;
}

/******************************************/
/******************************************/


