/*  File: arraysub.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 **  Arbitrary length arrays, stacks, associators,            **
 **    line breaking and all sorts of other goodies.          **
 **  Prerequisites: messubs.c and regular.h                   **
 * Exported functions:
 **  See Header file: array.h (includes lots of macros)       **
 * HISTORY:
 * Last edited: Mar 26 13:13 1992 (mieg)
 * * Jan 21 16:25 1992 (mieg): messcrash in uArrayCreate(size<=0)
 * * Dec 17 11:40 1991 (mieg): stackTokeniseTextOn() tokeniser.
 * * Dec 12 15:45 1991 (mieg): Stack magic and stackNextText
 * Created: Thu Dec 12 15:43:25 1989 (mieg)
 *-------------------------------------------------------------------
 */

   /* Warning : if you modify Array or Stack structures or 
      procedures in this file or array.h, you may need to modify 
      accordingly the persistent array package asubs.c.
   */

#include "regular.h"
#include "array.h"

/******** tells how much system stack used *********/

char *stackorigin ;

unsigned int stackused (void)
{ char x ;
  if (!stackorigin)          /* ideally should set in main() */
    stackorigin = &x ;
  return stackorigin - &x ;        /* MSDOS stack grows down */
}

/************ Array : class to implement variable length arrays ************/

static int nArrays = 0 ;

Array uArrayCreate (int n, int size)
{ static int id = 0 ;
  Array new = (Array) messalloc (sizeof (struct ArrayStruct)) ;

  if (size <= 0)
    messcrash("negative size %d in uArrayCreate", size) ;
  if (n < 1)
    n = 1 ;
  new->base = (char *) messalloc (n*size) ;
  new->dim = n ;
  new->max = 0 ;
  new->size = size ;
  new->id = ++id ;
  new->magic = ARRAY_MAGIC ;
  nArrays++ ;
  return new ;
}

/**************/

Array uArrayReCreate (Array a, int n, int size)
{ if (!arrayExists(a))
    return  uArrayCreate(n, size) ;

  if(a->size != size)
    messcrash("Type  missmatch in uArrayRecreate, you should always "
	      "call recreate using the same type") ;

  memset(a->base,0,(mysize_t)(a->dim*size)) ;
  if (n < 1)
    n = 1 ;
  if (a->dim < n)
    arrayExtend(a, n) ;

  a->max = 0 ;
  return a ;
}

/**************/

void uArrayDestroy (Array a)
{
  if(a && a->magic == ARRAY_MAGIC)
    {
      messfree (a->base) ;
      a->magic = 0 ;
      messfree (a) ;
      nArrays-- ;
    }
}

/**************/

int arrayNumber (void)
{ return nArrays ;
}

/******************************/

void arrayExtend (Array a, int n)
{
  char *new ;

  if (!a || n < a->dim)
    return ;
  while (n >= a->dim)
    { /*
	if (a->dim >= MAXINT>>1)
	messcrash ("Array extends over MAXINT") ;
      */	/* RMD I agree we should make this work */
      a->dim *= 2 ;
    }
  if (!(new = (char *) calloc (a->dim,a->size)))
    messcrash ("calloc failure expanding array %lx to %d",a,a->dim) ;
  memcpy (new,a->base,a->size*a->max) ;
  free (a->base) ;
  a->base = new ;
}

/***************/

char *uArray (Array a, int i)
{
  if (i >= a->max)
    { if (i >= a->dim)
        arrayExtend (a,i) ;
      a->max = i+1 ;
    }
#ifdef ARRAY_CHECK
  if (i < 0)
    messcrash ("referencing array element %d < 0", i) ;
#endif
  return a->base + i*a->size ;
}

/***************/

char *uArrCheck (Array a, int i)
{
  if (i >= a->max || i < 0)
    messcrash ("array index %d out of bounds [0,%d]",
	       i, a->max - 1) ;
  return a->base + i*a->size ;
}

/**************/

Array arrayCopy (Array a)
{
 Array b ;

  if (a && a->size)
    {
      b = uArrayCreate (a->max, a->size) ;
      memcpy(b->base, a->base, a->max * a->size);
      b->max = a->max ;
      return b;
    }
 else
   return 0 ;
}

/**************/

void arrayCompress(Array a)
{
  int i, j, k , as ;
  char *x, *y, *ab  ;
  
  if (!a || !a->size || arrayMax(a) < 2 )
    return ;

  ab = a->base ; 
  as = a->size ;
  for (i = 1, j = 0 ; i < arrayMax(a) ; i++)
    { x = ab + i * as ; y = ab + j * as ;
      for (k = a->size ; k-- ;)		
	if (*x++ != *y++) 
	  goto different ;
      continue ;
      
    different:
      if (i != ++j)
	{ x = ab + i * as ; y = ab + j * as ;
	  for (k = a->size ; k-- ;)	 
	    *y++ = *x++ ;
	}
    }
  arrayMax(a) = j + 1 ;
}

/****************/

void arraySort (Array a, int (*order)())
{
  unsigned int n = a->max,
        s = a->size ;
  void *v = a->base ;

  qsort(v, n, s, order) ;
}

/***********************************************************/


BOOL arrayIsEntry (Array a, int i, void *s)
{
  char *cp = uArray(a,i), *cq = s ;
  int j = a->size;

  while (j--)
    if (*cp++ != *cq++) 
      return FALSE ;
  return TRUE;
}

/***********************************************/
       /* Finds Entry s from Array  a
        * sorted in ascending order of order()
        * If found, returns TRUE and sets *ip
        * if not, returns FALSE and sets *ip one step left
        */

BOOL arrayFind(Array a, void *s, int *ip, int (* order)())
{

  int i = 0 , j = arrayMax(a), k;

  if(!j || order(s,uArray(a,0))<0)
    { *ip =-1; return FALSE;}   /* not found */
  if( order(s,uArray(a,--j))>0 )
    { *ip = j; return FALSE;}
  if (arrayIsEntry(a,j,s))
    { *ip = j; return TRUE;}
  if (arrayIsEntry(a,i,s))
    { *ip = i; return TRUE;}
  while(TRUE)
    { k = i + ((j-i) >> 1) ; /* midpoint */
      if (arrayIsEntry(a,k,s))
	{*ip = k; return TRUE; }
      order(s,uArray(a,k)) > 0 ?
	(i = k) : (j = k) ;
      if (i == (j-1) )
        break;
    }
  *ip = i ;
  return FALSE;
}

/**************************************************************/
       /* Removes Entry s from Array  a
        * sorted in ascending order of order()
        */

BOOL arrayRemove (Array a, void * s, int (* order)())
{
  int i;

  if (arrayFind(a, s, &i,order))
    {
      /* memcpy would be faster but regions overlap
       * and memcpy is said to fail with some compilers
       */
      char *cp = uArray(a,i),  *cq = cp + a->size ;
      int j = (arrayMax(a) - i)*(a->size) ;
      while(j--)
	*cp++ = *cq++;

      arrayMax(a) --;
      return TRUE;
    }
  else

    return FALSE;
}

/**************************************************************/
       /* Insert Segment s in Array  a
        * in ascending order of s.begin
        */

BOOL arrayInsert(Array a, void * s, int (*order)())
{
  int i, j;

  if (arrayFind(a, s, &i,order))
    return FALSE;  /* no doubles */
  
  j = arrayMax(a) + 1;
  uArray(a,j-1) ; /* to create space */

	/* avoid memcpy for same reasons as above */
  {
    char* cp = uArray(a,j - 1) + a->size - 1,  *cq = cp - a->size ;
    int k = (j - i - 1)*(a->size);
    while(k--)
      *cp-- = *cq--;
    
    cp = uArray(a,i+1); cq = (char *) s; k = a->size;
    while(k--)
      *cp++ = *cq++;
  }
  return TRUE;
}

/********* Stack : arbitrary Stack class - inherits from Array **********/

Stack stackCreate (int n)               /* n is initial size */
{
  Stack s = (Stack) messalloc (sizeof (struct StackStruct)) ;

  s->magic = STACK_MAGIC ;
  s->a = arrayCreate (n,char) ;
  s->pos = s->ptr = s->a->base ;
  s->safe = s->a->base + s->a->dim - 16 ; /* safe to store objs up to size 8 */
  return s ;
}

Stack stackReCreate (Stack s, int n)               /* n is initial size */
{
  if (!stackExists(s))
    return stackCreate(n) ;

  s->a = arrayReCreate (s->a,n,char) ;
  s->pos = s->ptr = s->a->base ;
  s->safe = s->a->base + s->a->dim - 16 ; /* safe to store objs up to size 8 */
  return s ;
}

Stack arrayToStack(Array a)
{ Stack s ;
  int n ;
  if (!arrayExists(a) || a->size != 1 )
    return 0 ;

  n = arrayMax(a) ;
  s = stackCreate(n  + 32) ;
  
  s->a = arrayCreate (n,char) ;
  memcpy(s->a->base, a->base, n) ;

  s->pos = s->a->base ;
  s->ptr = s->a->base + n ;
  s->safe = s->a->base + s->a->dim - 16 ; /* safe to store objs up to size 8 */
  while ((long)s->ptr % 4)
    *(s->ptr)++ = 0 ;   
  return s ;
}

void uStackDestroy (Stack s)
{
  if(s && s->magic == STACK_MAGIC)
    { arrayDestroy (s->a) ;
      s->magic = 0 ;
      free (s) ;
    }
}

void stackExtend (Stack s, int n)
{
  int ptr = s->ptr - s->a->base,
      pos = s->pos - s->a->base ;
  s->a->max = s->a->dim ;	/* since only up to ->max copied over */
  arrayExtend (s->a,ptr+n+16) ;	/* relies on arrayExtend mechanism */
  s->ptr = s->a->base + ptr ;
  s->pos = s->a->base + pos ;
  s->safe = s->a->base + s->a->dim - 16 ;
}

int stackMark (Stack s)
{ return (s->ptr - s->a->base) ;
}

void stackCursor (Stack s, int mark)
{ s->pos = s->a->base + mark ;
}

void pushText (Stack s, char* text)
{
  while (s->ptr + strlen(text) > s->safe)
    stackExtend (s,strlen(text)+1) ;
  while (*(s->ptr)++ = *text++) ;
  while ((long)s->ptr % 4)
    *(s->ptr)++ = 0 ;   
}

char* popText (Stack s)
{
  char *base = s->a->base ;

  while (s->ptr > base && !*--(s->ptr)) ;
  while (s->ptr >= base && *--(s->ptr)) ;
  return ++(s->ptr) ;
}

void catText (Stack s, char* text)
{
  while (s->ptr + strlen(text) > s->safe)
    stackExtend (s,strlen(text)+1) ;
  *s->ptr = 0 ;
  while (*s->ptr == 0 && s->ptr >= s->a->base)
    s->ptr -- ;
  s->ptr ++ ;
  while (*(s->ptr)++ = *text++) ;
  while ((long)s->ptr % 4)
    *(s->ptr)++ = 0 ;   
}

char* stackNextText (Stack s)
{ char *text = s->pos ;
  if (text>= s->ptr)
    return 0 ;  /* JTM, so while stackNextText makes sense */
  while (*s->pos++) ;
  while ((long)s->pos % 4)
    ++s->pos ;
  return text ;
}

/*********/
     /* Push text in stack s, after breaking it on delimiters */
     /* You can later access the tokens with command
	while (token = stackNextText(s)) work on your tokens ;
	*/
void  stackTokeniseTextOn(Stack s, char *text, char *delimiters)
{
  char *cp, *cq , *cend, *cd, old ;
  int i, n ;

  if(!stackExists(s) || !text || !delimiters)
    messcrash("stackTextOn received some null parameter") ;

  n = strlen(delimiters) ;
  cp = cq  = text ;
  while(TRUE)
    {
      while(*cp == ' ')
	cp++ ;
      cq = cp ;
      old = 0 ;
      while(*cq)
	for (cd = delimiters, i = 0 ; i < n ; cd++, i++)
	  if (*cd == *cq)
	   { old = *cq ;
	     *cq = 0 ;
	     break ;
	    }
	  else
	    cq++ ;
      cend = cq ;
      if(cend > cp)
	while(*--cend == ' ')
	  *cend = 0 ;
      if (cend > cp)
	pushText(s,cp) ;
      if(!old)
	{ stackCursor(s, 0) ;
	  return ;
	}
      cp = ++cq ;
    }
}

void stackClear(Stack s)
{ if (stackExists(s))
    { s->pos = s->ptr = s->a->base;
      s->a->max = 0;
    }
}

/****************** routines to set text into lines ***********************/

static char *linesText ;
static Array lines ;
static Array textcopy ;
static int kLine, popLine ;

/**********/

int uLinesText (char *text, int width)
{
  char *cp,*bp ;
  int i ;
  int nlines = 0 ;
  int length = strlen (text) ;
  int safe = length + 2*(length/width + 1) ;
  static int isFirst = TRUE ;

  if (isFirst)

    { isFirst = FALSE ;
      lines = arrayCreate(16,char*) ;
      textcopy = arrayCreate(128,char) ;
    }

  linesText = text ;
  array(textcopy,safe,char) = 0 ;   /* ensures textcopy is long enough */

  if (!*text)
    { nlines = popLine = kLine = 0 ;
      array(lines,0,char*) = 0 ;
      return 0 ;
    }

  bp = text ;
  cp = textcopy->base ;
  for (nlines = 0 ; *bp ; ++nlines)
    { array(lines,nlines,char*) = cp ;
      for (i = 0 ; (*cp = *bp) && *cp != '\n' ; ++i, ++cp, ++bp)
        if (i == width)           /* back up to last space */
          { while (i--)
	      { --bp ; --cp ;
		if (*cp == ' ')
		  goto eol ;
	      }
	    cp += width ;             /* no spaces in whole line ! */
	    bp += width ;
	    break ;
	  }
eol:  if (!*cp)
        break ;
      ++bp ;
      if (*cp != '\n')
	++cp ;
      *cp++ = 0 ;
    }
  kLine = 0 ;                                   /* reset for uNextLine() */
  popLine = ++nlines ;
  array(lines,nlines,char*) = 0 ;            /* 0 terminate */

  return nlines ;
 }

char *uNextLine (char *text)
 {
   if (text != linesText)
     messout ("Warning : uNextLine being called with bad context") ;
   return array(lines,kLine++,char*) ;
 }

char *uPopLine (char *text)
 {
   if (text != linesText)
     messout ("Warning : uPopLine being called with bad context") ;
   if (popLine)
     return array(lines,--popLine,char*) ;
   else return 0;
 }

char **uBrokenLines (char *text, int width)
{
  uLinesText (text,width) ;
  return (char**)lines->base ;
}

char *uBrokenText (char *text, int width)
{
  char *cp ;

  uLinesText (text,width) ;
  uNextLine (text) ;
  while (cp = uNextLine (text))
    *(cp-1) = '\n' ;
  return arrp(textcopy,0,char) ;
}

/*************************************************************/


/* Associator package is for associating pairs of pointers. 
   Assumes that an "in" item is non-zero.
   Implemented as a hash table of size 2^m + 1.  The last entry is a
   sentinel element that holds 0 (to avoid testing for the end of 
   the array in the loop.  The hash function is just the last m bits
   of the pointer of the object shifted right by 2 (since the last 
   two bits are always 0).  
   Grabbed from Steve Om's sather code by Richard Durbin.
   Package is a priveledged user of Array's - to clear.
   User has access to structure member ->n = # of pairs
*/

#define VSIZE (sizeof(void*))

Associator assCreate (void)
{ static int nAss = 0 ;
  Associator a = (Associator) messalloc (sizeof (struct AssStruct)) ;

  a->magic = ASS_MAGIC ;
  a->id = ++nAss ;
  a->n = 0 ;
  a->m = 8 ;
  a->in = (void**) messalloc ((a->m+1)*VSIZE) ;
  a->out = (void**) messalloc ((a->m+1)*VSIZE) ;
  a->mask = 7 ;
  return a ;
}

/********************/

Associator assReCreate (Associator a)
{
  if (!assExists(a))
    return assCreate() ;

  a->n = 0 ;
  memset(a->in,0,(mysize_t)((a->m + 1)*VSIZE)) ;
  memset(a->out,0,(mysize_t)((a->m + 1)*VSIZE)) ;
  a->mask = 7 ;
  return a ;
}

/********************/

void uAssDestroy (Associator a)
{
  if (!assExists(a)) return ;

  messfree (a->in) ;
  messfree (a->out) ;
  messfree (a) ;
}

/*******************/

void assClear (Associator a)
{
  if (!assExists(a)) return ;
  
  a->n = 0 ;
  memset (a->in,0,a->m*VSIZE) ;
  memset (a->out,0,a->m*VSIZE) ;
}

/*******************/

BOOL uAssFind (Associator a, void* xin, void** pout)
	/* if found, updates *pout and returns TRUE, else returns FALSE	*/
{ register int	 hash ;
  register void* test ;
  
  if (!assExists(a) || !xin) return FALSE ;

  hash = ((unsigned int)xin >> 2) & a->mask ;
new_start:
  while (TRUE)
    { test = a->in[hash] ;
      if (test == xin)
	{ if (pout)
	    *pout = a->out[hash] ;
	  return TRUE ;
	}
      if (!test)
	break ;
      ++hash ;
    }
  if (hash == a->m)
    { hash = 0 ;
      goto new_start ;
    }
  return FALSE ;
}

/********************/

static void assDouble (Associator a)
{
  int newsize = a->m*2 ;
  void **old_in = a->in, **old_out = a->out ;
  int i,hash ;
  void* xin ;

  a->mask = (a->mask << 1) + 1 ;
  a->in  = (void**) messalloc ((newsize+1)*VSIZE) ;
  a->out = (void**) messalloc ((newsize+1)*VSIZE) ;

  for (i = 0 ; i < a->m ; ++i)
    if (xin = old_in[i])
      { hash = ((unsigned int)xin >> 2) & a->mask ;
	while (TRUE) 
	  { if (!a->in[hash])
	      { a->in[hash] = xin ;
		a->out[hash] = old_out[i] ;
		break ;
	      }
	    ++hash ;
	    if (hash == newsize)
	      hash = 0 ;
	  }
      }
  messfree (old_in) ;
  messfree (old_out) ;
  a->m = newsize ;
}

/********************/

BOOL assInsert (Associator a, void* xin, void* xout)
     /* if already there returns FALSE, else inserts and returns TRUE */
{
  int hash ;
  void *test ;

  if (!assExists(a) || !xin) return FALSE ;

  if (a->n * 2 >= a->m)
    assDouble (a) ;

  hash = ((unsigned int)xin >> 2) & a->mask ;
  while (TRUE)
    { test = a->in[hash] ;
      if (!test)
	{ a->in[hash] = xin ;
	  a->out[hash] = xout ;
	  ++a->n ;
	  return TRUE ;
	}
      if (test == xin)		/* already there */
	return FALSE ;
      if (++hash == a->m)
	hash = 0 ;
    }
}

/*****************/

BOOL assRemove (Associator a, void* xin)
	/* if found, removes entry and returns TRUE, else returns FALSE	*/
{ 
  int hash, hole, index ;
  void* test ;

  if (!assExists(a) || !xin) return FALSE ;

  hash = ((unsigned int)xin >> 2) & a->mask ;
  while (TRUE)
    { test = a->in[hash] ;
      if (!test)
	return FALSE ;		/* not in table */
      if (test == xin)
	{ hole = hash ;
	  a->in[hash] = a->out[hash] = 0 ;
	  --a->n ;
	  break ;
	}
      if (++hash == a->m)
	hash = 0 ;
    }

  index = hole ;		/* check hole not in the way */
  while (TRUE)
    { if (++index == a->m)
	index = 0 ;
      if (!a->in[index])
	return TRUE ;		/* done */
      hash = ((unsigned int)a->in[index] >> 2) & a->mask ;
      if (((hash <= hole) && (hole < index || index < hash))
	  || ((hole < index) && (index < hash)))
	{ a->in[hole] = a->in[index] ;
	  a->out[hole] = a->out[index] ;
	  hole = index ;
	  a->in[index] = 0 ;
	  a->out[index] = 0 ;
	}
    }
}

/*******************/

void assDump (Associator a)
{
  int i ;

  if (!assExists(a)) return ;

  fprintf (stderr,"Associator %lx : %d pairs\n",(unsigned long)a,a->n) ;
  for (i = 0 ; i < a->m ; ++i)
    if (a->in[i])
      fprintf (stderr,"%lx - %lx\n",
	       (unsigned long) a->in[i],
	       (unsigned long) a->out[i]) ;
}

/*********************/

BOOL uAssNext (Associator a, void* *pin, void* *pout)
{	/* lets you step through all members of the table */
  static int i ;
  static Associator as ;

  if(!assExists(a))
     messcrash("uAssNext received a non existing associator") ;
  if (!*pin)
    { as = a ;
      i = 0 ;
    }
  else if (as != a)
    { messerror ("Wrong associator in call to assNext()") ;
      return FALSE ;
    }
  else if (*pin != a->in[i])
    { messerror ("Non-consecutive call to assNext()") ;
      return FALSE ;
    }
  else
    ++i ;

  while (i < a->m)
    { if (a->in[i])
	{ *pin = a->in[i] ;
	  *pout = a->out[i] ;
	  return TRUE ;
	}
      ++i ;
    }
  return FALSE ;
}

/********* end of file **********/
