/*
alarm - Put up cute windows to warn of impending events.
Inspired by Apollo alarm_server -private.
Written in a coding frenzy on the last day of 1990 by:
    Jim Rees, University Of Michigan IFS Project
*/

#include <stdio.h>
#include <sys/types.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Mailbox.h>
#include <X11/Xaw/Cardinals.h>

#include "bm"

#define SLEEPTIME	(1 * 60)	/* How often to check */
#define BACKTIME	(60 * 60)	/* How far back to go for old alarms on startup */
#define FWDTIME		(5 * 60)	/* How far ahead to go for new alarms */
#define BSIZE	1024

char def_alarms[] = "%s/user_data/alarms";
char dmbufname[] = "`node_data/paste_buffers/default.txt";

static String fallback_resources[] = {
    "*BiffWindow.geometry:	+0+0",
    "*BiffWindow*font:		fixed",

    "*BiffWindow.box.orientation:	vertical",
    "*BiffWindow*vSpace:	0",
    "*BiffWindow*hSpace:	0",

    "*AlarmWindow.geometry:	+54+0",
    "*AlarmWindow*font:		-adobe-new century schoolbook-bold-r-normal--24-240-75-75-p-149-iso8859-1",
    "*AlarmWindow.box.orientation: horizontal",

    "*xbutton.label:		X",
    "*dmbutton.label:		dm",
    NULL,
};

int bflag, mflag, xflag;
char *afilename;
int sleeptime = SLEEPTIME;
char *buf;

int do_alarms(), destroy(), xtodm(), dmtox();

extern XtAppContext _XtDefaultAppContext();
extern char *malloc(), *getenv(), *index(), *strdup();

main(ac, av)
int ac;
char *av[];
{
	char *home;
	Display *display;
	XtAppContext app_con;
	Widget toplevel, box, xbutton, dmbutton;

	buf = malloc(BSIZE);

	while (ac > 1 && av[1][0] == '-') {
		switch (av[1][1]) {
		case 'f':
			afilename = strdup(av[2]);
			ac--;
			av++;
			break;
		case 'u':
			sleeptime = atoi(av[2]);
			ac--;
			av++;
			break;
		case 'b':
			bflag = 1;
			break;
		case 'm':
			mflag = 1;
			break;
		case 'x':
			xflag = 1;
			break;
		}
		ac--;
		av++;
	}

	umask(0);
	home = getenv("HOME");
	if (home == NULL) {
		fprintf(stderr, "There's no place like $HOME\n");
		exit(1);
	}
	if (afilename == NULL) {
		sprintf(buf, def_alarms, home);
		afilename = strdup(buf);
	}

	XtToolkitInitialize();

	/* This is just like XtCreateApplicationContext() except that it
	   makes the new app_con the default.  We need this because the
	   mailbox widget depends on the default app_con being set.  */

	app_con = _XtDefaultAppContext();
	XtAppSetFallbackResources(app_con, fallback_resources);
	display = XtOpenDisplay(app_con, NULL, NULL, "alarm", NULL, 0, &ac, av);
	if (display == NULL) {
		fprintf(stderr, "Can't open display\n");
		exit(1);
	}

	if (mflag || xflag) {
		toplevel = XtAppCreateShell("biff", "BiffWindow", applicationShellWidgetClass, display, NULL, 0);
		box = XtCreateManagedWidget("box", boxWidgetClass, toplevel, NULL, 0);
	}
	if (mflag)
		XtCreateManagedWidget("mailbox", mailboxWidgetClass, box, NULL, 0);
	if (xflag) {
		xbutton = XtCreateManagedWidget("xbutton", commandWidgetClass, box, NULL, 0);
		XtAddCallback(xbutton, XtNcallback, xtodm, display);
		dmbutton = XtCreateManagedWidget("dmbutton", commandWidgetClass, box, NULL, 0);
		XtAddCallback(dmbutton, XtNcallback, dmtox, display);
	}

	XtRealizeWidget(toplevel);

	do_alarms(display, 0);
	XtAppMainLoop(app_con);
}

do_alarms(display, id)
Display *display;
XtIntervalId id;
{
	static time_t nexttime;
	time_t t, thistime;
	FILE *f;
	char astring[100];
	static int ntimes;

	if (nexttime == 0) {
		time(&t);
		nexttime = t - t % sleeptime;
		/* First time through, display old alarms */
		thistime = nexttime - BACKTIME - FWDTIME;
	} else
		thistime = nexttime;
	nexttime += sleeptime;

	f = fopen(afilename, "r");
	if (f != NULL) {
		while (fgets(astring, sizeof astring, f) != NULL)
			parse_alarm(astring, thistime, nexttime, display, (ntimes == 0));
		fclose(f);
	} else
		perror(afilename);

	ntimes++;
	time(&t);
	if (nexttime > t)
		XtAppAddTimeOut(XtDisplayToApplicationContext(display),
		    (nexttime - t) * 1000, do_alarms, display);
	else
		do_alarms(display, 0);
}

parse_alarm(s, thistime, nexttime, display, startup)
char *s;
time_t thistime, nexttime;
int startup;
{
	time_t at;
	char *cp;
	int n;

	if (s[0] == '#')
		return;
	n = strlen(s) - 1;
	if (s[n] == '\n')
		s[n] = '\0';
	cp = index(s, '\t');
	if (cp == NULL) {
		if (startup) {
			sprintf(buf, "Missing tab in \"%s\"\n", s);
			alarm_window(display, buf);
		}
		return;
	}
	*cp++ = '\0';
	at = getdate(s, NULL);
	if (at == (time_t) -1) {
		if (startup) {
			sprintf(buf, "Cannot parse date string %s\n", s);
			alarm_window(display, buf);
		}
		return;
	}
	at -= FWDTIME;
	if (thistime <= at && at < nexttime)
		alarm_window(display, cp);
}

alarm_window(display, text)
Display *display;
char *text;
{
	Widget toplevel, box, button;
	Arg args[2];
	char name[32];
	static int an;
	static Pixmap pm;

	sprintf(name, "alarm%d", an++);
	toplevel = XtAppCreateShell(name, "AlarmWindow", applicationShellWidgetClass, display, NULL, 0);

	box = XtCreateManagedWidget("box", boxWidgetClass, toplevel, args, 0);

	XtSetArg(args[0], XtNwidth, bm_width + 8);
	XtSetArg(args[1], XtNheight, bm_height + 8);
	button = XtCreateManagedWidget("destroy", commandWidgetClass, box, args, 2);
	XtAddCallback(button, XtNcallback, destroy, toplevel);

	XtSetArg(args[0], XtNlabel, text);
	XtCreateManagedWidget(text, labelWidgetClass, box, args, 1);
	XtRealizeWidget(toplevel);

	if (an == 1)
		pm = XCreateBitmapFromData(display, XtWindow(button), bm_bits, bm_width, bm_height);
	XtSetArg(args[0], XtNbitmap, pm);
	XtSetValues(button, args, 1);

	if (bflag)
		XBell(display, 0);
}

destroy(w, s, callData)
Widget w, s;
XtPointer callData;
{
	XtDestroyWidget(s);
}

create_biff_box(display)
Display *display;
{
	Widget toplevel;

	toplevel = XtAppCreateShell("biff", "Alarm", applicationShellWidgetClass, display, NULL, 0);
	XtCreateManagedWidget("mailbox", mailboxWidgetClass, toplevel, NULL, 0);
	XtRealizeWidget(toplevel);
}

xtodm(w, display, d)
Widget w;
Display *display;
XtPointer d;
{
	int fd, n;
	char *s;

	s = XFetchBuffer(display, &n, 0);
	if (n == 0)
		return;
	if ((fd = creat(dmbufname, 0777)) >= 0) {
		write(fd, s, n);
		close(fd);
	}
	XFree(s);
}

dmtox(w, display, d)
Widget w;
Display *display;
XtPointer d;
{
	int fd, n;
	char *s = NULL, *sx;

	if ((fd = open(dmbufname, 0)) < 0)
		goto getout;
	if ((n = lseek(fd, 0, 2)) <= 0)
		goto getout;
	if ((s = malloc(n)) == NULL)
		goto getout;
	lseek(fd, 0, 0);
	if ((n = read(fd, s, n)) <= 0)
		goto getout;
	XStoreBuffer(display, s, n, 0);

getout:
	if (fd >= 0)
		close(fd);
	if (s != NULL)
		free(s);
}
