#include <OI/oi.H>

/*
 *	interactive menus
 *
 *	This program introduces the following menu concepts:
 *		- setting the main menu in an application window.
 *		- associating menus with menu cells.
 *		- associating dialog boxes with menu cells.
 *		- setting callbacks on menu cells.
 *		- creating menu cells with icons.
 *
 *	Please refer to the README file for more information on demos
 *	which provide a simpler introduction to menus, as well as demos
 *	which show how to recursively build menus (a really neat demo).
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_init()
 *			OI_begin_interaction()
 *			OI_fini()
 *		- OI_d_tech
 *			set_associated_object()
 *		- OI_app_window
 *			oi_create_app_window()
 *			set_main_menu();
 *		- OI_menu
 *			general information on menus
 *		- OI_menu_cell
 *			general information on menu cells
 *		- OI_button_menu
 */



/*
 *	The OI_cell_spec structure is documented in the OI_menu_cell man page.
 *	For review, the structure is:
 *		struct OI_cell_spec {
 *			char			*namp ;		- ptr to name string for cell
 *			char			*labelp ;	- ptr to string for cell label
 *			OI_action_fn		*fnp ;		- function to execute when cell fires
 *			OI_callback		*objp ;		- ptr to user obj if callback is a 
 *									C++ member function
 *			OI_action_memfn		*memfnp ;	- member func to execute when cell fires
 *			void			*argp ;		- ptr to user arg to pass to callback
 *			OI_menu_cell_type	type;		- Icon type as defined in defs.H
 *			struct	_OI_menu_spec	*menu_ptr;	- Pointer to walking menu
 *			OI_number		wid ;		- width of pixmap
 *			OI_number		ht ;		- height of pixmap
 *			OI_number		dpth ;		- depth of pixmap
 *		};
 */

static void	set_color( OI_menu_cell *, void *, OI_number );


/*
 *	define the `colors' menu.
 *	Note that we are setting the same callback for each cell to the c-routine `set_color'.
 *	Notice, also, that we are using the argument fields to convey color information to the
 *	set_color callback.
 */
static OI_cell_spec
colors[] =
{
	{"red","red",		(OI_action_fnp) set_color,NULL,NULL_PMF,(void *)"sienna",OI_TEXT_CELL,NULL,0,0,0},
	{"orange","orange",	(OI_action_fnp) set_color,NULL,NULL_PMF,(void *)"coral",OI_TEXT_CELL,NULL,0,0,0},
	{"yellow","yellow",	(OI_action_fnp) set_color,NULL,NULL_PMF,(void *)"greenyellow",OI_TEXT_CELL,NULL,0,0,0},
	{"green","green",	(OI_action_fnp) set_color,NULL,NULL_PMF,(void *)"limegreen",OI_TEXT_CELL,NULL,0,0,0},
	{"blue","blue",		(OI_action_fnp) set_color,NULL,NULL_PMF,(void *)"turquoise",OI_TEXT_CELL,NULL,0,0,0},
	{"plum","plum",		(OI_action_fnp) set_color,NULL,NULL_PMF,(void *)"maroon",OI_TEXT_CELL,NULL,0,0,0},
	{"orchid","orchid",	(OI_action_fnp) set_color,NULL,NULL_PMF,(void *)"violetred",OI_TEXT_CELL,NULL,0,0,0},
	{"brown","brown",	(OI_action_fnp) set_color,NULL,NULL_PMF,(void *)"wheat",OI_TEXT_CELL,NULL,0,0,0},
};


/*
 *	define `topics' menu.
 */
static OI_cell_spec
topics[] =
{
	{"colors","colors",NULL,NULL,NULL_PMF,NULL,OI_TEXT_CELL,NULL,0,0,0},
	{"files","files",NULL,NULL,NULL_PMF,NULL,OI_TEXT_CELL,NULL,0,0,0},
};

/*
 *	define the application window's primary (main) menu.
 *
 *	this menu currently contains only one cell, Exit, onto which we are specifying a 
 *	special action function:
 *		OI_end_interaction
 *	As documented in the OIIntro man pages, OI_end_interaction has the special property
 *	of terminating interaction with the user, thus causing the original OI_begin_interaction
 *	to return.
 */
static OI_cell_spec
title[] = 
{
	{"exitButton","Exit",(OI_action_fnp)(&OI_end_interaction),NULL,NULL_PMF,NULL,OI_TEXT_CELL,NULL,0,0,0},
};

void
main (int argc, char**argv)
{
	OI_connection		*conp ;
	OI_app_window		*wp ;
	OI_menu			*main_menu ;
	OI_menu			*topic_menu ;
	OI_menu			*color_menu ;
	OI_menu_cell		*cellp ;
	OI_file_dialog_box	*fdbx ;

	
	/*
	 *	Open a connection.
	 */
	if (conp = OI_init(&argc, argv, "HelloWorld")) {
		/*
		 *	create the application window.
		 *	set layout to row major.
		 */
		wp = oi_create_app_window("main",200,100,"Main Window") ;
		wp->set_layout( OI_layout_row );

		/*
		 *	Create a horizontal button menu, and make it the application
		 *	window's main menu.  Setting a menu as the application windows
		 *	main menu usually looks best in motif, in color.
		 */
		main_menu = oi_create_button_menu( "mainMenu", OI_n_cells(title), title, OI_HORIZONTAL );
		wp->set_main_menu( main_menu );

		/*
		 *	Next, create a vertical button menu containing various topics of interest,
		 *	and layout this object within the application window.
		 */
		topic_menu = oi_create_button_menu( "topics", OI_n_cells(topics), topics, OI_VERTICAL, "topics" );
		topic_menu->layout_associated_object( wp, 1, 1, OI_ACTIVE );

		/*
		 *	Next, create a menu listing available colors, and attach this menu as a submenu
		 *	off of the `variousTopics' menu `color' cell.
		 *
		 *	IMPORTANT POINTS:
		 *		1. We determine which cell to attach the submenu to by using the
		 *			OI_d_tech::descendant member function.
		 *		2. We associate the color menu with it's parent cell by using
		 *			set_associated_object, and NOT layout_associated_object.
		 *		3. The sub menu is associated as OI_ACTIVE_NOT_DISPLAYED.
		 */
		color_menu = oi_create_button_menu( "color_sub_menu", OI_n_cells(colors), colors, OI_VERTICAL );
		color_menu->set_associated_object( topic_menu->descendant( "colors" ), 1, 1, OI_ACTIVE_NOT_DISPLAYED );

		/*
		 *	Next, let's create a file dialog box, and associate the dialog box
		 *	off of the `variousTopics' menu `files' cell.
		 *
		 *	again, note the following points:
		 *		1. We determine which cell to attach the submenu to by using the
		 *			OI_d_tech::descendant member function.
		 *		2. We associate the dialog box with it's parent cell by using
		 *			set_associated_object, and NOT layout_associated_object.
		 *		3. The dialog box is associated as OI_ACTIVE_NOT_DISPLAYED.
		 */
		fdbx = oi_create_file_dialog_box( "file_dialog_box", ".", "*" );
		fdbx->set_associated_object( topic_menu->descendant( "files" ), 1, 1, OI_ACTIVE_NOT_DISPLAYED );

		/*
		 *	Next, let's dynamically add a cell to the `topics' menu.
		 *	Instead of another OI_TEXT_CELL, this cell will be an OI_ICON_CELL.
		 *
		 *	IMPORTANT POINTS:
		 *		1. When adding a cell, ALWAYS use menu->add_cell,
		 *			NEVER attempt to set_associate_object or layout_associated_object.
		 *		2. to determine the cell position, use menu->num_cells() to determine the
		 *			number of cells in the menu.
		 *
		 */
		if ((cellp = oi_create_menu_cell( "icons", "../bitmaps/pps-logo.xbm", NULL, NULL, OI_ICON_CELL ) )) {
			topic_menu->add_cell( cellp, topic_menu->num_cells() );
		}
		/*
		 *	Associate another file_dialog_box with this ICON_CELL.  Setup the file_dialog_box
		 *	to automatically list the icon directory.
		 */
		fdbx = oi_create_file_dialog_box( "icon_dialog_box", "../bitmaps", "*" );
		fdbx->set_associated_object( cellp, 1, 1, OI_ACTIVE_NOT_DISPLAYED );

		/*
		 *	display main window.
		 *	begin interaction.
		 */
		wp->set_associated_object(wp->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE) ;
		OI_begin_interaction() ;
	}

	/*
	 *	Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini() ;
}


/*
 *	set color
 *	
 *	This is the C callback function for the color menu cells.
 *	When a cell is selected, both the cell, and the parent cell
 *	containing the colors pull-right will have their foreground
 *	and background colors set.
 *
 */
static void
set_color( OI_menu_cell * cellp, void * argp, OI_number )
{
	OI_d_tech	*objp ;
	if (cellp->selected()) {
		cellp->set_fg_color( (char *) argp );
		cellp->set_bkg_color( (char *) cellp->name() );
		if (objp = cellp->ancestor( "colors" )) {
			objp->set_fg_color( (char *) argp );
			objp->set_bkg_color( (char *) cellp->name() );
		}
	}
}
