/******************************************************************************
*******************************************************************************
*
*   Copyright (c) 1989, 1991, 1992 by Digital Equipment Corporation
*
*   Permission to use, copy, modify, and distribute this software for any
*   purpose and without fee is hereby granted, provided that the above
*   copyright notice and this permission notice appear in all copies, and that
*   the name of Digital Equipment Corporation not be used in advertising or
*   publicity pertaining to distribution of the document or software without
*   specific, written prior permission.
*
*   Digital Equipment Corporation makes no representations about the
*   suitability of the software described herein for any purpose.  It is
*   provided "as is" without express or implied warranty.
*  
*  DEC is a registered trademark of Digital Equipment Corporation
*  DIGITAL is a registered trademark of Digital Equipment Corporation
*  X Window System is a trademark of the Massachusetts Institute of Technology
*
*******************************************************************************
******************************************************************************/

/*
 * File:	xcd.c
 * Author:	Mark Clement
 * Date:	April 4, 1991
 *
 * Description:
 *   A sample Compact Disk front panel program.
 *
 * Audit Trail
 *
 * version 2.1
 * Modified: 	Mark Longo
 * Date:	January 14, 1992
 *
 * added version argumen (-v).
 * 
 * Fixed a timing bug that caused the last track to cutoff early when
 * scan mode had been used.  Fixed an obscure selection highlighting bug.
 * Added a cuter bitmap icon for the iconified display.  Put in a
 * "better" copyright notice.
 *
 * no version label
 * Modified: 	Mark Longo
 * Date:	January 2, 1992
 *
 * Added "scan" function to advance or reverse disk position a few
 * seconds at a time.  Re-designed the status display to now include track
 * index display and an option menu for remaining/elapsed/total.  Now get
 * select highlight background color from widget armColor resource.  Many
 * fixes to "|<<" function.  Fixed bug where entering shuffle mode with a CD
 * of over 24 tracks hung the program.  Fixed a bunch of other small but
 * annoying shuffle problems
 *
 * Modified: 	Mark Longo
 * Date:	December 10, 1991
 *
 * Add virtual device layer interface to view the device as having
 * idealized commands.  Added selection buttons so that a specific track
 * can be directly selected.  Also, the selection buttons show the play
 * sequence and the track now playing.  Now get default volume level,
 * select button size and color from the UID file.
 * 	
 * Changed play_track() to play all tracks between the current track
 * and the last track if we aren't in shuffle mode.  This gets rid of
 * the stop/reset cycle that interrupts play between tracks.  This is
 * especially desirable for CD's with tracks that run right into each
 * other.
 * 
 * Massive overhaul with much simpler data structures and processing.
 * Fixed many design and implementation bugs.
 * 
 */

/*
 * Notes on the RRD42 CDROM PLAYER device:
 *
 * As a point of interest, the drive thinks the first track is #1, not #0.
 * It's convenient for us if the first track is track #0 so that we
 * can use the track number to index an array of size [total_tracks].
 * For this reason, the table of contents structure always thinks the
 * current track is one less than the hardware does.  This mismatch is
 * known only to the play_track() function, which fixes the mismatch
 * at the last moment when a play command is issued to the device.
 *
 * A note from the original source:
 *   A bug in the CDROM firmware or the SCSI software hangs the CDROM if
 *   you issue a stop and play command without waiting a bit, so when
 *   going play to stop to play again we sleep a second.  (I haven't
 *   observed this myself, but maybe my CDROM firmware has been fixed -ml)
 * 
 */

#include <stdio.h>
#include <time.h>

#include <Mrm/MrmAppl.h>	/* Motif Toolkit */
#include "cdutil.h"		/* Device specific includes */

/*
 * Global structures
 *
 * There are two primary global structures (not inlcuding the hardware
 * structure).  These are named "status" and "toc".
 *
 * "status":
 * 	the status structure defined immediately below contains
 * 	device state relevant to this program.  The status structure is
 * 	reloaded every time we call poll_device().  poll_device() polls
 * 	the device via ioctl() and updates much of the status
 * 	structure and the some of the toc structure.
 *
 * "toc":
 * 	The table of contents structures are laid out as follows.
 *
 *	toc +-------+
 *	    | entry |----> +--------------+
 *	    | ...   |      | track_number |
 *	    | ...   |	   | track_length |
 *	    +-------+	   +--------------+
 *	    		   | track_number |
 *	    		   | track_length |
 *	    		   +--------------+
 *	    		   | 		  |
 *	    		    . . . . . . .
 *	    		   |		  |
 *	    		   +--------------+
 *
 * 	The toc structure holds info relevant to the entire CD (eg:
 * 	total_tracks, etc).  toc.entry points to allocated space
 * 	for track structures which are accessed by indexing (ie:
 * 	toc.entryi[i].track_number.  The next CD track for playing is
 * 	ALWAYS selected by incrementing the index.  The index wraps to
 * 	zero when it gets to the last valid track strcuture.
 *
 *      If the track numbers in the array are numbered sequentially,
 *      then the tracks play in sequence.  Shuffle play is implemented
 *      by numbering the valid toc.entry's in random sequence.  The
 *      tracks then play in random sequence as the array index
 *      increments in linear fashion.
 *
 */

#define	VERSION "2.1"

typedef struct _cdstat
{
  char *devname;
  char *progname;

  int state;		/* Current state of the CDROM */

#define IDLE 		0
#define STOPPED 	1
#define PLAYING 	2
#define PAUSED 		3
#define EJECTED 	4

  int flags;		/* Flags to control the CDROM */

#define CD_EXCLUSIVE 	0x01	/* To mask out the timeout during changes */
#define CD_REPEAT_TRACK	0x02	/* Repeat the current track */
#define CD_REPEAT_DISK 	0x04	/* Repeat the whole disk */
#define CD_PREVENT 	0x08	/* Prevent removal */
#define CD_DEVICE_OPEN	0x10	/* Device is open */
#define CD_SHUFFLE	0x20	/* the play list is shuffled */

#define CD_REMAINING	0x40	/* display time remaining */
#define CD_ELAPSED	0x80	/* display time elapsed*/
#define CD_TOTAL	0x100	/* display total time only */
#define CD_TIME_MASK	0x1C0	/* timer mask bits */
  
  int current_track;	/* reported current track playing (start = 1) */
  int current_index;	/* reported current index playing */
  int current_seconds;	/* current seconds into the track */
  int current_total_seconds;	/* current total seconds into the disc */
  int currentVolume;   	/* current volume setting */
	
  XtIntervalId timer;	/* To update the time each second, we set
			 * a timer.  timer holds the handle to allow us
			 * to turn off the timer before its expiration
			 */

  XtIntervalId scan_timer;	/* Same for scanning if active */
  enum {BACK, AHEAD, OFF} scan_direction;
  int scan_stride;		/* Current scanning skip distance */

#define SCAN_BITE  	950	/* ms to play each scan interval */
#define MIN_SCAN_STRIDE	5	/* Starting secs to skip per scan interval */
#define MAX_SCAN_STRIDE	20	/* Final secs to skip per scan interval */

} statusRec, *statusPtr;

statusRec status;

/*
 * table of contents structures
 */

typedef struct _tocEntry
{
  int track_number;		/* Track number of this TOC entry */
  int track_address;		/* start adrs in secs (used for SCAN mode) */
  int track_length; 		/* length in secs */
} tocEntryRec, *tocEntryPtr;

typedef struct _toc
{
  tocEntryPtr entry;	/* pointer to an array of tocEntryRec */
  int current;		/* the number of the last toc.entry to be played */
  int last;		/* the number of the last track on the disk */
  int total_time;	/* the total seconds on the CD */
  int total_tracks;	/* the total tracks on the CD */
} tocRec, *tocPtr;

tocRec toc;

/*
 * Declare space to hold Table of Contents read from CDROM hardware
 */

struct cd_toc_head_and_entries glob_toc;

static MrmHierarchy	s_MrmHierarchy;		/* MRM database hierarch id */
static char		*vec[]={"xcd.uid"}; 	/* MRM database file */
static char		*icon_name[]={"xcd.xbm"}; /* Icon bitmap file */

/*
 * Declare exported routines
 */

static void play_button_activate(),
	  stop_button_activate(),
	  pause_button_activate(),
	  prev_track_button_activate(),
	  next_track_button_activate(),
          scan_back_button_arm(),
          scan_ahead_button_arm(),
          scan_button_disarm(),
 	  shuffle_button_activate(),
	  repeat_button_activate(),
	  eject_button_activate(),
	  quit_button_activate(),
	  prevent_button_activate(),
	  select_button_activate(),
	  volume_slider_activate(),
          status_menu_activate(),
	  create_cb();

/*
 * Set up registration structure.  These strings will be associated with
 * the corresponding routines and made visible to the uil commands.
 */

static MrmRegisterArg	regvec[] = {
	{"play_button_activate", (caddr_t)play_button_activate},
	{"stop_button_activate", (caddr_t)stop_button_activate},
	{"pause_button_activate", (caddr_t)pause_button_activate},
	{"prev_track_button_activate", (caddr_t)prev_track_button_activate},
	{"next_track_button_activate", (caddr_t)next_track_button_activate},
        {"scan_back_button_arm", (caddr_t)scan_back_button_arm},
        {"scan_ahead_button_arm", (caddr_t)scan_ahead_button_arm},
        {"scan_button_disarm", (caddr_t)scan_button_disarm},
	{"shuffle_button_activate", (caddr_t)shuffle_button_activate},
	{"repeat_button_activate", (caddr_t)repeat_button_activate},
	{"eject_button_activate", (caddr_t)eject_button_activate},
	{"quit_button_activate", (caddr_t)quit_button_activate},
	{"prevent_button_activate", (caddr_t)prevent_button_activate},
	{"select_button_activate", (caddr_t)select_button_activate},
	{"volume_slider_activate", (caddr_t)volume_slider_activate},
	{"status_menu_activate", (caddr_t)status_menu_activate},
	{"create_cb", (caddr_t)create_cb}
	};

static MrmCount		 regnum = sizeof(regvec) / sizeof(MrmRegisterArg);

/*
 * Define the indices into the widget array.  When the widgets are created,
 * the create callback will fill in the corresponding entry in the widget
 * array for use when the widget characteristics need to be changed.
 */

#define k_prevent_id 		0
#define k_stop_id 		1
#define k_play_id 		2
#define k_pause_id 		3
#define k_eject_id		4
#define k_repeat_id 		5
#define k_shuffle_id 		6
#define k_trackNum_id		7
#define k_trackTime_id		8
#define k_discTime_id		9
#define k_indexNum_id		10
#define k_selectPad_id 		11
#define k_selectButton_id 	12

#define NWIDGET 	13
static Widget primaryWidgets[NWIDGET];

/* status widget ID's */

#define REMAINING	1
#define ELAPSED		2
#define	TOTAL		3

/* selection widget array */

static Widget selectWidgets[MAXTRACKS];
static int selectTags[MAXTRACKS];  /* ptrs to these ints passed by select cb's */

static Widget toplevel, xcdmain;

/*
 * these arrays are indexed by the widget ID numbers and are used to
 * switch from the active icon to the passive icon in the
 * lightMainButtons() routine.
 */

static char *passiveIconName[] =
{
  "allowIcon",
  "passiveStopIcon",
  "passivePlayIcon",
  "passivePauseIcon",
  "passiveEjectIcon",
  "",
  "",
  ""
};  

static char *activeIconName[] =
{
  "preventIcon",
  "activeStopIcon",
  "activePlayIcon",
  "activePauseIcon",
  "activeEjectIcon",
  "",
  "",
  ""
};  

/*
 * non-standard returning functions
 */

void poll_device(), stop_timer(), start_timer(), timeout_proc();
void start_scan_timer(), stop_scan_timer(), scan_timeout();
void updateStatusDisplay(), updateSelectDisplay();
void setupTOC(), clearTOC();
void play_track(), prev_track(), next_track();
void stop_device(), pause_device(), resume_device(), eject_device();
void set_volume();

/*
 * externals...
 */

extern char *optarg;
extern int opterr;

/*
 * global to this file
 */

static Display *display;

/* select button literals */

static Pixel highlightColorFG, highlightColorBG;
static Pixel lowlightColorFG, lowlightColorBG;
static int *pSelectKeyHeight, *pSelectKeyWidth, *pSelectKeysPerRow;

/************************************************************************
 *
 * 	main()... the buck starts here
 * 	
 ************************************************************************/

int
main(argc, argv)
     int argc;
     char **argv;
{
     static MrmCode class;
     XtAppContext app_context;
     int i, n;
     int *pDefaultVolume;
     int dtype;
     time_t salt;
     Pixel buttonBackground;
     Arg argList[16];
     char *pC, *arg, c;
     char *getenv();

     bzero(&status, sizeof(statusRec));

     pC = rindex(argv[0], '/');
     status.progname = pC ? &(pC[1]) : argv[0];

     while((c = getopt(argc, argv, "f:v")) != -1)
       {
	 switch(c)
	   {
	   case 'f':
	     status.devname = optarg;
	     break;
	    
	   case 'v':
	     printf("%s: version %s\n", status.progname, VERSION);
	     exit(0);
	   }
       }

     if (!status.devname)
       {
	 status.devname = getenv("CDROM");
       }

     if (status.devname)
       {
	 if(open_dev() == 0)
	   {
	     status.state = IDLE;
	   }

	 else
	   {
	     status.state = EJECTED;
	   }
       }
     else
       {
	 fprintf (stderr, "%s: Please specify a device name using the '-f device' option\nor set the CDROM environment variable.\n", status.progname);
	 exit (-1);
       }

     /*
      *  Initialize the window stuff
      */

     MrmInitialize();		/* always do this 1st */
     XtToolkitInitialize();	/* ...then this. */

     app_context = XtCreateApplicationContext();
     display = XtOpenDisplay(app_context, NULL, status.progname, "Xcd",
                            NULL, 0, &argc, argv);

     if (display == NULL)
       {
	 fprintf(stderr, "%s:  Can't open X display\n", argv[0]);
	 exit(1);
       }

     /*
      * get the toplevel widget ID
      */
     
     n = 0;
     XtSetArg(argList[n], XmNallowShellResize, False);
     n++;
     XtSetArg(argList[n], XmNsensitive, True);
     n++;

     toplevel = XtAppCreateShell(status.progname, "Xcd",
				 applicationShellWidgetClass,
				 display, argList, n);

     /*
      *  Define the Mrm hierarchy (only 1 file)
      */

     if (MrmOpenHierarchy (1,		    /* number of files (1) */
			   vec, 	    /* filename(s) */
			   NULL,	    /* os_ext_list */
			   &s_MrmHierarchy) /* ptr to returned id */
	 != MrmSUCCESS)
       {
	 fprintf (stderr, "can't open the UID file hierarchy\n");
       }

     /*
      * fetch the application's icon name from the UID file and set the
      * value in the toplevel widget.
      */
     
     XtSetArg(argList[0], XmNiconName, "k_iconName");
     MrmFetchSetValues(s_MrmHierarchy, toplevel, argList, 1);

     /*
      *  Set the default volume level.  If you don't initialize it
      *  it reverts to the maximum level (ouch!).
      */

     MrmFetchLiteral(s_MrmHierarchy, "k_defaultVolume", NULL, &pDefaultVolume,
		     &dtype);

     status.currentVolume = *pDefaultVolume;

     /*
      * fetch the literals pertaining to the select buttons from the
      * UID file.
      */
     
     MrmFetchLiteral(s_MrmHierarchy, "k_selectKeyHeight", NULL,
		     &pSelectKeyHeight, &dtype);
     MrmFetchLiteral(s_MrmHierarchy, "k_selectKeyWidth", NULL,
		     &pSelectKeyWidth, &dtype);
     MrmFetchLiteral(s_MrmHierarchy, "k_selectKeysPerRow", NULL,
		     &pSelectKeysPerRow, &dtype);

     /*
      * Register the names and addresses of the callback routines so
      * that the resource manager can bind them to their names in the
      * UID file.
      */

     if (MrmRegisterNames(regvec, regnum) != MrmSUCCESS)
       fprintf(stderr, "can't register names\n");

     /*
      *  create the main widget and the kids
      */

     if (MrmFetchWidget(s_MrmHierarchy,
			"xcdMain",	/* name of main window widget */
			toplevel,	/* top widget ID from Xtoolkit */
			&xcdmain,
			&class)
	 != MrmSUCCESS)
       fprintf(stderr, "can't fetch interface\n");

     /*
      * Initialize display values, etc.
      */

     status.flags &= ~(CD_REPEAT_DISK | CD_REPEAT_TRACK);

     XtSetArg(argList[0], XmNlabelPixmap, "repeatOffIcon");
     MrmFetchSetValues(s_MrmHierarchy, primaryWidgets[k_repeat_id],
		       argList, 1);
     
     status.flags &= ~CD_SHUFFLE;

     XtSetArg(argList[0], XmNlabelPixmap, "serialPlayIcon");
     MrmFetchSetValues(s_MrmHierarchy, primaryWidgets[k_shuffle_id],
		       argList, 1);

    /*
     * get the selection button colors
     */
     
     n = 0;
     XtSetArg(argList[n], XmNforeground, &lowlightColorFG);
     n++;
     XtSetArg(argList[n], XmNbackground, &lowlightColorBG);
     n++;
     XtSetArg(argList[n], XmNarmColor, &highlightColorBG);
     n++;
     XtGetValues(primaryWidgets[k_selectButton_id], argList, n);
     
     highlightColorFG = lowlightColorFG;
     
     XtUnmanageChild(primaryWidgets[k_selectButton_id]);

     /*
      * set the time display defaults
      */

     status.flags |= CD_REMAINING;
     
     /*
      * Make the toplevel widget "manage" the main window (or whatever the
      * the UIL defines as the topmost widget).  This will cause it to be
      * "realized" when the toplevel widget is "realized"
      */

     XtManageChild(xcdmain);

     /*
      * Realize the toplevel widget.  This will cause the entire "managed"
      * widget hierarchy to be displayed
      */

     XtRealizeWidget(toplevel);

     init_icon(display, toplevel);

     /*
      * init the random number generator in case the user shuffles
      */

     salt = time();
     srandom((int)salt);

     /*
      * set up the display for initial dipiction
      */
     
     toc.entry = (tocEntryPtr) malloc(MAXTRACKS * sizeof(tocEntryRec));

     poll_device();	/* toc.entry[] must be malloc'd b4 u get here */

     updateStatusDisplay(); 

     switch (status.state)
       {
       case PLAYING:
       case PAUSED:

	 setupTOC();
	 lightMainButton(k_play_id);
	 lightSelectButton(status.current_track - 1);
	 start_timer(toplevel);
	 break;

       case IDLE:
	 
	 stop_device();

       case STOPPED:

	 setupTOC();
	 lightMainButton(k_stop_id);	 
	 set_volume(status.currentVolume);
	 break;
	 
       case EJECTED:

	 clearTOC();
	 break;
	 
       default:
	 break;
       }
	   
     /*
      * Loop and process events
      */

     XtAppMainLoop(app_context);

     /* UNREACHABLE */

     return(0);
     
} /* main */

/*************************************************************************
 *************************************************************************
 *************************************************************************
 *
 * 	MOTIF CALLBACKS...
 *
 *************************************************************************
 *************************************************************************
 *************************************************************************/

/************************************************************************
 *
 * 	play_button_activate()... If a disk is inserted, play it.
 *
 ************************************************************************
 *
 * If the disk is paused, issue a resume command, otherwise restart
 * play at the current track.  If the drive was stopped or previously
 * ejected, rebuild the table of contents.  (This restores default
 * order after a shuffle).
 */

static void
play_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    switch (status.state)
      {
      case PAUSED:
	resume_device();
	break;

      case EJECTED:
      case IDLE:
      case STOPPED:

	/*
	 * PLAY might be pressed after a new CD was inserted so we
	 * need to call poll_device() to get the new TOC.
	 */

	poll_device();

	if (status.state == EJECTED)	/* still ejected? */
	  return;
	
	setupTOC();
	play_track();
	break;
	
      case PLAYING:
      default:
	return;
      }
    
} /* play_button_activate */

/************************************************************************
 *
 *	select_button_activate()...
 *
 ************************************************************************/

static void
select_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    switch (status.state)
      {
      case IDLE:
      case EJECTED:

	setupTOC();
	break;
	
      case PAUSED:
      case STOPPED:
      case PLAYING:
	break;

      default:
	return;
      }
	  
    toc.current = (int)*tag;
    play_track();

} /* select_button_activate */

/************************************************************************
 *
 * prev_track_button_activate()... play previous track (or current track)
 *
 ***********************************************************************/
 
static void
prev_track_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    switch (status.state)
      {
      case EJECTED:
      case IDLE:
	
	/* handle a newly inserted disk */

	poll_device();

	if (status.state == EJECTED)	/* still ejected? */
	  return;
	
	setupTOC();
	break;
	
      case STOPPED:
      case PLAYING:
	break;

      case PAUSED:
      default:
	return;
      }
	
    prev_track();
    
} /* prev_track_button_activate */

/************************************************************************
 *
 * 	next_track_button_activate()... play the next track
 *
 ************************************************************************/

static void
next_track_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    switch (status.state)
      {
      case EJECTED:
      case IDLE:

	/* handle a newly inserted disk */

	poll_device();

	if (status.state == EJECTED)	/* still ejected? */
	  return;
	
	setupTOC();
	play_track();
	break;
	
      case STOPPED:
      case PLAYING:
	next_track();
	break;
	
      case PAUSED:
      default:
	return;
      }

} /* next_track_button_activate */

/************************************************************************
 *
 * scan_back_button_arm()... enter scanning backward state
 *
 ***********************************************************************/
 
static void
scan_back_button_arm(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    switch (status.state)
      {
      case PLAYING:
	stop_timer();		/* stop the usual timeout */
	status.scan_direction = BACK;
	status.scan_stride = MIN_SCAN_STRIDE;
	start_scan_timer(widget);
	break;

      case EJECTED:
      case IDLE:
      case STOPPED:
      case PAUSED:
      default:
	return;
      }
    
} /* scan_back_button_arm */

/************************************************************************
 *
 * scan_ahead_button_arm()... enter scanning ahead state
 *
 ***********************************************************************/
 
static void
scan_ahead_button_arm(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    switch (status.state)
      {
      case PLAYING:
	stop_timer();		/* stop the usual timeout */
	status.scan_direction = AHEAD;
	status.scan_stride = MIN_SCAN_STRIDE;
	start_scan_timer(widget);
	break;

      case EJECTED:
      case IDLE:
      case STOPPED:
      case PAUSED:
      default:
	return;
      }
    
} /* scan_ahead_button_arm */

/************************************************************************
 *
 * scan_button_disarm()... exit scanning state
 *
 ***********************************************************************/
 
static void
scan_button_disarm(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    switch (status.state)
      {
      case IDLE:
      case PLAYING:
	stop_scan_timer();
	status.scan_direction = OFF;
	status.scan_stride = MIN_SCAN_STRIDE;
	start_timer(widget);		/* resume the usual timer */
	
	poll_device();			/* get new latest numbers... */
	updateStatusDisplay();		/* ...and print them */
	break;

      case EJECTED:
      case STOPPED:
      case PAUSED:
      default:
	return;
      }
    
} /* scan_button_disarm */

/************************************************************************
 *
 * 	stop_button_activate()... stop the CD
 *
 ************************************************************************/

static void
stop_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    poll_device();
  
    switch (status.state)
      {
      case PLAYING:
      case PAUSED:

	stop_device();
	break;
	
      case IDLE:
	setupTOC();
	stop_device();
	break;

      case EJECTED:
      case STOPPED:
      default:
	return;
      }
	
} /* stop_button_activate */

/************************************************************************
 *
 * 	pause_button_activate()... pause if currently playing, else resume
 *
 ************************************************************************/

static void
pause_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    switch (status.state)
      {
      case PLAYING:
	pause_device();
	break;
	
      case PAUSED:
	resume_device();
	break;
	
      case EJECTED:
      case IDLE:
      case STOPPED:
      default:
	return;
      }
    
} /* pause_button_activate */

/************************************************************************
 *
 * 	shuffle_button_activate()... shuffle the play list and start play
 *
 ************************************************************************/

static void
shuffle_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    Arg argList[16];

    switch (status.state)
      {
      case IDLE:
      case PLAYING:
      case STOPPED:

	if (status.flags & CD_SHUFFLE)
	  {
	    status.flags &= ~CD_SHUFFLE;
	    XtSetArg(argList[0], XmNlabelPixmap, "serialPlayIcon");
	  }
	else
	  {
	    status.flags |= CD_SHUFFLE;
	    XtSetArg(argList[0], XmNlabelPixmap, "shufflePlayIcon");
	  }

	/* update the widget to use the new icon */
	
	MrmFetchSetValues(s_MrmHierarchy, primaryWidgets[k_shuffle_id],
			  argList, 1);

	setupTOC();
	play_track();
	break;
	
      case PAUSED:
      case EJECTED:
      default:
	return;
      }

} /* shuffle_button_activate */

/************************************************************************
 *
 * 	repeat_button_activate()... handle repeat state changes
 *
 ************************************************************************/

static void
repeat_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    Arg argList[16];

    if (status.flags & CD_REPEAT_TRACK)
      {
	status.flags &= ~(CD_REPEAT_DISK | CD_REPEAT_TRACK);
	XtSetArg(argList[0], XmNlabelPixmap, "repeatOffIcon");
      }
    else if (status.flags & CD_REPEAT_DISK)
      {
	status.flags &= ~CD_REPEAT_DISK;
	status.flags |= CD_REPEAT_TRACK;
	XtSetArg(argList[0], XmNlabelPixmap, "repeatTrackIcon");
      }
    else
      {
	status.flags |= CD_REPEAT_DISK;
	XtSetArg(argList[0], XmNlabelPixmap, "repeatDiskIcon");
      }
    
    MrmFetchSetValues(s_MrmHierarchy, primaryWidgets[k_repeat_id], argList, 1);

} /* repeat_button_activate */

/************************************************************************
 *
 * 	eject_button_activate()... eject the CD
 *
 ************************************************************************/

static void
eject_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    poll_device();

    if (status.state == EJECTED || status.flags & CD_PREVENT)
      return;
    
    eject_device();

} /* eject_button_activate */

/************************************************************************
 *
 * 	quit_button_activate()...  bye!
 *
 ************************************************************************/

static void
quit_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    exit(0);
}

/************************************************************************
 *
 *	prevent_button_activate()... toggle prevent state
 *
 ***********************************************************************
 *
 * Either allow or prevent removal of the caddy.  Also update the 
 * Icon on the button to show the current state.
 */

static void
prevent_button_activate(widget, tag, callback_data)
     Widget	widget;
     char	*tag;
     XmAnyCallbackStruct *callback_data;
{
    Arg argList[1];

    if (status.flags & CD_PREVENT)
      {
	status.flags &= ~CD_PREVENT;
	AllowRemoval();
	XtSetArg(argList[0], XmNlabelPixmap, passiveIconName[k_prevent_id]);
      }
    else
      {
	status.flags |= CD_PREVENT;
	PreventRemoval();
	XtSetArg(argList[0], XmNlabelPixmap, activeIconName[k_prevent_id]);
      }

    MrmFetchSetValues(s_MrmHierarchy, primaryWidgets[k_prevent_id],
		      argList, 1);

} /* prevent_button_activate */

/************************************************************************
 *
 * volume_slider_activate()... change the volume level
 *
 ************************************************************************/

static void
volume_slider_activate(widget, tag, scale)
     Widget	widget;
     char	*tag;
     XmScaleCallbackStruct *scale;
{
    set_volume(scale->value);
}

/************************************************************************
 *
 *	status_menu_activate()... status option selection
 *
 ************************************************************************/

static void
status_menu_activate(widget, id, callback_data)
     Widget	widget;
     char	*id;
     XmAnyCallbackStruct *callback_data;
{
    status.flags &= ~(CD_TIME_MASK);

    switch (*id)
      {
      case REMAINING:
	status.flags |= CD_REMAINING;
	break;
	
      case ELAPSED:
	status.flags |= CD_ELAPSED;
	break;

      case TOTAL:
	status.flags |= CD_TOTAL;
	break;
	
      default:
	break;
      }

    updateStatusDisplay();
    
} /* status_menu_activate */

/************************************************************************
 *
 *	create_cb()... called when widgets startup
 *
 ************************************************************************/

static void
create_cb(w, id, reason)
     Widget w;
     int *id;
     unsigned long *reason;
{
    if (*id < NWIDGET)
      primaryWidgets[*id] = w;
}

/***********************************************************************
 ***********************************************************************
 ***********************************************************************
 *
 * 	UTILITY ROUTINES...
 *
 ***********************************************************************
 ***********************************************************************
 ***********************************************************************/

/************************************************************************
 *
 * 	updateStatusDisplay()... update the changing time numbers
 *
 ************************************************************************
 *
 * All of the states being serviced below make this routine a disgusting
 * mess.  The object of the mess is to update the running track/index and
 * time displays only when we have to (ie: when they change).  We go to this
 * trouble because when you update a time label the display has a tendency
 * to blink, which is damn annoying.
 */

#define printLabel(string, widget) \
    xstr = XmStringLtoRCreate(string, ""); \
    XtSetArg(argList[0], XmNlabelString, xstr); \
    XtSetValues(widget, argList, 1); \
    XmStringFree(xstr)

void
updateStatusDisplay()
{
    static int lastIndex = 0;
    static int lastTrack = 0;
    static int lastFlags = 0;
    
    char disc[64], track[64], trackNum[64], indexNum[64];
    XmString xstr;
    Arg argList[16];
    
    status.flags |= CD_EXCLUSIVE; 	/* no interruptions... */

    /*
     * if the disc is stopped, put the totals on the board
     */

    if (status.state == PLAYING ||  status.state == PAUSED)
      {
	/*
	 * update the status display's time and track info
	 */
	
	switch (status.flags & CD_TIME_MASK)
	  {
	  case CD_REMAINING:
	    sprintf(track, "%02d:%02d",
		    (toc.entry[toc.current].track_length -
		     status.current_seconds) /60,
		    (toc.entry[toc.current].track_length -
		     status.current_seconds) %60);
	    sprintf(disc, "%02d:%02d",
		    (toc.total_time - status.current_total_seconds)/60,
		    (toc.total_time - status.current_total_seconds)%60);
	    break;

	  case CD_ELAPSED:
	    sprintf(track, "%02d:%02d", status.current_seconds/60,
		    status.current_seconds%60);
	    sprintf(disc, "%02d:%02d", status.current_total_seconds/60,
		    status.current_total_seconds%60);
	    break;	
	
	  case CD_TOTAL:
	    sprintf(track, "%02d:%02d",
		    toc.entry[toc.current].track_length/60,
		    toc.entry[toc.current].track_length%60);
	    sprintf(disc, "%02d:%02d", toc.total_time/60, toc.total_time%60);
	    break;
	  }

	/*
	 * update the running times if we're in REMAINING or ELAPSED mode,
	 * OR if this is the first time through in TOTAL mode, OR if
	 * TOTAL mode and the track number just changed.
	 */

	if (status.flags & CD_TOTAL)
	  {
	    if (!(lastFlags & CD_TOTAL))
	      {
		printLabel(track, primaryWidgets[k_trackTime_id]);
		printLabel(disc, primaryWidgets[k_discTime_id]);
	      }
	    else if (lastTrack != toc.entry[toc.current].track_number + 1)
	      {
		printLabel(track, primaryWidgets[k_trackTime_id]);
	      }
	  }

	 if (status.flags & (CD_REMAINING | CD_ELAPSED))
	   {
	     /*
	      * If in SHUFFLE mode, then disc time remaining or elapsed
	      * makes no sense, so display total disc time instead.
	      */
	    
	     if (status.flags & CD_SHUFFLE)
	       {
		 if (!(lastFlags & CD_SHUFFLE))
		   {
		     sprintf(disc, "%02d:%02d", toc.total_time/60,
			     toc.total_time%60);
		     printLabel(disc, primaryWidgets[k_discTime_id]);
		   }
		 printLabel(track, primaryWidgets[k_trackTime_id]);
	       }
	     else
	       {
		 printLabel(track, primaryWidgets[k_trackTime_id]);
		 printLabel(disc, primaryWidgets[k_discTime_id]);
	       }
	   }

	/*
	 * if the track number/index changed, update it
	 */

	if (lastTrack != toc.entry[toc.current].track_number + 1)
	  {
	    sprintf(trackNum, "%02d", toc.entry[toc.current].track_number + 1);
	    printLabel(trackNum, primaryWidgets[k_trackNum_id]);
	    printLabel(track, primaryWidgets[k_trackTime_id]);
	  }

	if (lastIndex != status.current_index)
	  {
	    sprintf(indexNum, "%02d", status.current_index);
	    printLabel(indexNum, primaryWidgets[k_indexNum_id]);
	  }

	lastFlags = status.flags;
	lastIndex = status.current_index;
	lastTrack = toc.entry[toc.current].track_number + 1;
      }

    /*
     * if the device isn't in run mode, then display the totals or nothing.
     * When the device is stopped this code is not continually called
     * so the display won't blink, therefore we don't bother to check
     * if the below display work is redundant.
     */
    
    else
      {
	if (status.state == STOPPED || status.state == IDLE)
	  {
	    sprintf(disc, "%02d:%02d", toc.total_time/60, toc.total_time%60);
	  }
	else
	  {
	    strcpy(disc, "00:00");
	  }

	strcpy(track, "00:00");
	strcpy(trackNum, "00");
	strcpy(indexNum, "00");

	printLabel(trackNum, primaryWidgets[k_trackNum_id]);
	printLabel(indexNum, primaryWidgets[k_indexNum_id]);
	printLabel(track, primaryWidgets[k_trackTime_id]);
	printLabel(disc, primaryWidgets[k_discTime_id]);

	lastTrack = 0;
	lastIndex = 0;
	lastFlags = status.flags;
      }

    status.flags &= ~CD_EXCLUSIVE;
    
} /* updateStatusDisplay */

/**********************************************************************
 *
 * 	updateSelectDisplay()... 
 * 	
 **********************************************************************
 *
 * relabel the select buttons
 * 
 * If there's now a different number of tracks than the last time we
 * did this, the disk must've changed.  If there aren't enough select
 * button widgets to meet demand, we make some more.
 */

void
updateSelectDisplay()
{
    static int buttonCount = 0;
    MrmCode class;
    Arg argList[16];
    XmString xstr;
    char buf[64], number[10];
    int i, n, x, y, height, width;
    int xCount = 0;
    int yCount = 0;

    /*
     * display the new button picture
     */

    XtUnmanageChildren(selectWidgets, buttonCount);

    if (toc.total_tracks == 0)
      {
	return;
      }
    
    /*
     * If we don't now have as many buttons as total_tracks, I guess the
     * disk changed.  Go make more buttons.
     */

    if (toc.total_tracks > buttonCount)
      {
	for (i = buttonCount; i < toc.total_tracks; i++)
	  {
	    /* build a new name string for the new widget */

	    strcpy(buf, "selectButton");
	    sprintf(number, "%d", i);
	    strcat(buf, number);

	    selectWidgets[i] = 0;

	    /* create the new widget by copying selectButton1 */
	    
	    MrmFetchWidgetOverride(s_MrmHierarchy,
				   "selectButtons",
				   primaryWidgets[k_selectPad_id],
				   buf,
				   NULL, NULL,  /* no arglist to pass */
				   &selectWidgets[i], &class);

	    /*
	     * setup the the new widget's callback to pass a pointer
	     * to its button number at activate time
	     */
	    
	    selectTags[i] = i;

	    XtAddCallback(selectWidgets[i], XmNactivateCallback,
			  select_button_activate, (caddr_t) &selectTags[i]);

	    buttonCount++;
	  }
      }
    
    /*
     * relabel each button with the number of the track to be played when
     * the button is pushed, and position each button within the selection
     * frame.
     */
    
    for (i = 0; i < toc.total_tracks; i++)
      {
	n = 0;
	
	sprintf(buf, "%d", toc.entry[i].track_number + 1);
        xstr = XmStringLtoRCreate(buf, "");
	XtSetArg(argList[n], XmNlabelString, xstr);
	n++;

	if (yCount == 0)
	  {
	    XtSetArg(argList[n], XmNtopAttachment, XmATTACH_FORM);
	    n++;
	  }
	else
	  {
	    XtSetArg(argList[n], XmNtopAttachment, XmATTACH_WIDGET);
	    n++;
	    XtSetArg(argList[n], XmNtopWidget,
		     selectWidgets[i - *pSelectKeysPerRow]);
	    n++;
	  }

	if (xCount == 0)
	  {
	    XtSetArg(argList[n], XmNleftAttachment, XmATTACH_FORM);
	    n++;
	  }
	else
	  {
	    XtSetArg(argList[n], XmNleftAttachment, XmATTACH_WIDGET);
	    n++;
	    XtSetArg(argList[n], XmNleftWidget, selectWidgets[i-1]);
	    n++;
	  }

	XtSetValues(selectWidgets[i], argList, n);

	/*
	 * X Bug?
	 * 
	 * I'm not sure why but if you don't explicitly load the button
	 * size here using a *separate* XtSetValues call from that above,
	 * the buttons get drawn only big enough to fit the label text
	 * (even if you init defaults in the UID file's "selectButton1"
	 * template).
	 */
	
	n = 0;
	XtSetArg(argList[n], XmNwidth, *pSelectKeyWidth);
	n++;
	XtSetArg(argList[n], XmNheight, *pSelectKeyHeight);
	n++;

	XtSetValues(selectWidgets[i], argList, n);

	/*
	 * keep track of next relative button position
	 */

	if (++xCount >= *pSelectKeysPerRow)
	  {
	    xCount = 0;
	    yCount++;
	  }	    
      }

    /*
     * adjust the dimensions of the selectKeyPad Form widget to exactly
     * hold all the buttons in the new disk.
     */

    width = (*pSelectKeyWidth * *pSelectKeysPerRow);
    height = (toc.total_tracks / *pSelectKeysPerRow) * *pSelectKeyHeight;

    if (toc.total_tracks % *pSelectKeysPerRow)
      height += *pSelectKeyHeight;

    XtResizeWidget(primaryWidgets[k_selectPad_id], width, height, NULL, NULL);

    XtManageChildren(selectWidgets, toc.total_tracks);

} /* updateSelectDisplay */

/**********************************************************************
 *
 * 	lightMainButton()...
 *
 **********************************************************************
 *
 * Highlight the main button passed in.  Unhighlight the last button
 * that was passed in, if any.
 *
 * By convention, the static int "buttonLit" contains the widget id of
 * the last button we lit up.  It contains -1 if no buttons are currently
 * lit.
 *
 * The calling convention is that an input value of -1 causes the
 * currently lit up track (if any) to be unhighlighted.
 */

lightMainButton(buttonToLight)
int buttonToLight;
{
    static int buttonLit = -1;
    Arg argList[16];
    int n;

    /*
     * if the button to be lit is already lit up, we got nothing to do!
     */
    
    if (buttonToLight == buttonLit)
      {
	return;
      }

    /*
     * if there's a button currently highlighted, shut it off.
     */

    if (buttonLit != -1)
      {
	XtSetArg(argList[0], XmNlabelPixmap, passiveIconName[buttonLit]);
	MrmFetchSetValues(s_MrmHierarchy, primaryWidgets[buttonLit],
			  argList, 1);

	buttonLit = -1;
      }

    /*
     * if there's a button to be highlighted, light it up.
     */
    
    if (buttonToLight != -1)
      {
	XtSetArg(argList[0], XmNlabelPixmap, activeIconName[buttonToLight]);
	MrmFetchSetValues(s_MrmHierarchy, primaryWidgets[buttonToLight],
			      argList, 1);

	buttonLit = buttonToLight;
      }
    
} /* lightMainButton */

/**********************************************************************
 *
 * 	lightSelectButton()...
 * 	
 **********************************************************************
 *
 * Highlight the selection button that corresponds to the currently
 * selected track.  Unlight the previous selection, if there is one.
 *
 * By convention, the static int "buttonLit" contains the number
 * of the last track we lit up.  It contains -1 if no tracks are
 * currently lit.
 *
 * The calling convention is that an input value of -1 causes the
 * currently lit up track (if any) to be unhighlighted.
 */

lightSelectButton(buttonToLight)
    int buttonToLight;
{
    static int buttonLit = -1;
    Arg argList[16];
    int n;

    if (buttonToLight == buttonLit)
      {
	return;
      }
	
    /*
     * if there's a track currently highlighted, shut it off.
     */
    
    if (buttonLit != -1)
      {
	n = 0;
	XtSetArg(argList[n], XmNbackground, lowlightColorBG);
	n++;
	XtSetArg(argList[n], XmNforeground, lowlightColorFG);
	n++;

	if (selectWidgets[buttonLit])
	  XtSetValues(selectWidgets[buttonLit], argList, n);

	buttonLit = -1;
      }

    /*
     * if there's a track to be highlighted, light it up.
     */
    
    if (buttonToLight != -1)
      {
	n = 0;
	XtSetArg(argList[n], XmNforeground, highlightColorFG); 
	n++;
	XtSetArg(argList[n], XmNbackground, highlightColorBG);
	n++;

	if (selectWidgets[buttonToLight])
	  XtSetValues(selectWidgets[buttonToLight], argList, n);

	buttonLit = buttonToLight;
      }

} /* lightSelectButton */

/************************************************************************
 *
 * 	setupTOC()... build the table of contents
 *
 ************************************************************************
 *
 * Build a table of contents from hardware information.  The hardware
 * information must have been gathered during poll_device(), so be sure
 * poll_device() gets called before you come here.
 *
 * The table of contents is built in random order if shuffle is active,
 * sequential order otherwise.
 *
 * This routine also updates the select and status displays according
 * to the new TOC info.
 */

#define CD_DIRECTORY_TIME 2

void
setupTOC()
{
    int i, j, n;

    if (status.state == EJECTED)
      return;

    /*
     * build the toc entries for either shuffle or serial order
     */
    
    toc.total_time = CD_DIRECTORY_TIME;
    
    if (status.flags & CD_SHUFFLE)
      {
	long shuffleTrack;
	int gotNum[MAXTRACKS];

	bzero(gotNum, MAXTRACKS * sizeof(int));
    	i = 0;
	
	while (i < toc.total_tracks)
	  {
	    shuffleTrack = random() % toc.total_tracks;
	
	    if (gotNum[shuffleTrack] == 0)
	      {
		gotNum[shuffleTrack] = -1;

		toc.entry[i].track_number = shuffleTrack;
		toc.entry[i].track_length = getTrackLength(shuffleTrack);
		toc.total_time += toc.entry[i].track_length;

		i++;	/* only increment when we find a new track */
	      }
	  }
      }
    else
      {
	for (i = 0; i < toc.total_tracks; i++)
	  {
	    toc.entry[i].track_number = i;
	    toc.entry[i].track_length = getTrackLength(i);
	    toc.total_time += toc.entry[i].track_length;
	  }
      }

    /*
     * Compute and record the start address of each track in seconds.
     * We need this to know when the "scan" function crosses tracks.
     */

    n = CD_DIRECTORY_TIME;
    
    for (i = 0; i < toc.total_tracks; i++)
      {
	for (j = 0; j < toc.total_tracks; j++)
	  {
	    if (toc.entry[j].track_number == i)
	      {
		toc.entry[j].track_address = n;
		n += toc.entry[j].track_length;
		continue;
	      }
	  }
      }

    toc.current = 0;

    updateSelectDisplay();	/* unmap the select buttons */
    
} /* setupTOC */

/************************************************************************
 *
 * 	clearTOC()... zero out the table of contents
 *
 ************************************************************************/

void
clearTOC()
{
    bzero(toc.entry, MAXTRACKS * sizeof(tocEntryRec));
    
    toc.current = 0;
    toc.last = 0;
    toc.total_tracks = 0;
    toc.total_time = 0;

} /* clearTOC */

/*
 *  Define the application icon image bitmap.
 */

#define cd_icon_width 36
#define cd_icon_height 36
static char cd_icon_bits[] = {
   0xff, 0xff, 0xff, 0xff, 0x0f, 0xff, 0x1f, 0xc0, 0xff, 0x0f, 0xff, 0xc3,
   0x1f, 0xfe, 0x0f, 0xff, 0x38, 0xe0, 0xf8, 0x0f, 0x7f, 0x86, 0x0f, 0xf3,
   0x0f, 0x3f, 0x71, 0x70, 0xe4, 0x0f, 0x9f, 0x0c, 0x87, 0xc9, 0x0f, 0x4f,
   0xe2, 0x38, 0x92, 0x0f, 0x27, 0x19, 0xc2, 0x24, 0x0f, 0x97, 0xc4, 0x1d,
   0x49, 0x0f, 0x53, 0x32, 0x60, 0x52, 0x0e, 0x4b, 0x89, 0x8f, 0x94, 0x0e,
   0x2b, 0x65, 0x30, 0xa5, 0x0e, 0xa9, 0x14, 0x47, 0xa9, 0x0c, 0xa5, 0xd2,
   0x5f, 0x2a, 0x0d, 0x95, 0xca, 0x9f, 0x4a, 0x0d, 0x55, 0xea, 0xbf, 0x52,
   0x0d, 0x55, 0xe9, 0xbf, 0x54, 0x0d, 0x55, 0xea, 0xbf, 0x52, 0x0d, 0x95,
   0xca, 0x9f, 0x4a, 0x0d, 0xa5, 0xd2, 0x5f, 0x2a, 0x0d, 0xa9, 0x14, 0x47,
   0xa9, 0x0c, 0x2b, 0x65, 0x30, 0xa5, 0x0e, 0x4b, 0x89, 0x8f, 0x94, 0x0e,
   0x53, 0x32, 0x60, 0x52, 0x0e, 0x97, 0xc4, 0x1d, 0x49, 0x0f, 0x27, 0x19,
   0xc2, 0x24, 0x0f, 0x4f, 0xe2, 0x38, 0x92, 0x0f, 0x9f, 0x0c, 0x87, 0xc9,
   0x0f, 0x3f, 0x71, 0x70, 0xe4, 0x0f, 0x7f, 0x86, 0x0f, 0xf3, 0x0f, 0xff,
   0x38, 0xe0, 0xf8, 0x0f, 0xff, 0xc3, 0x1f, 0xfe, 0x0f, 0xff, 0x1f, 0xc0,
   0xff, 0x0f, 0xff, 0xff, 0xff, 0xff, 0x0f, 0xff, 0xff, 0xff, 0xff, 0x0f};


init_icon(display, toplevel)
     Display  	*display;
     Widget toplevel;
{
     Window window;
     Arg argList[1];
     Pixmap cd_icon;
	
     window = XtWindow(toplevel);
     cd_icon = XCreateBitmapFromData(display, window,
				     cd_icon_bits,
				     cd_icon_width,
				     cd_icon_height);

     XtSetArg(argList[0], XmNiconPixmap, cd_icon);
     XtSetValues(toplevel, argList, 1);
    
} /* init_icon */

/************************************************************************
 *
 * open_dev()	Open the physical device
 *
 ************************************************************************
 *
 * The open will not complete until the CD is inserted into the drive.
 * This routine is provided to allow the program to be started before the
 * CD is inserted and later allow the open to complete.
 *
 * Returns:
 * 	0 = success
 *	1 = failure
 */

open_dev()
{
    int fd = 0;

    /* Attempt to open the CD-ROM device */

    if (status.devname)
      {
	if ((status.flags & CD_DEVICE_OPEN) == 0)
	  {
	    if ((fd = open(status.devname, O_RDONLY)) >= 0)
	      {
		set_fd(fd); 	/* Set the fd for the util routines */
		status.flags |= CD_DEVICE_OPEN;
		return(0);
	    } 
	  }
      }
    return(-1);
    
} /* open_dev */

/***********************************************************************
 ***********************************************************************
 ***********************************************************************
 *
 * 	DEVICE LAYER...
 *
 ***********************************************************************
 ***********************************************************************
 ***********************************************************************
 *
 * This layer presents a virtual device interface to the rest of the
 * program.  The routines here provide abstractions that are hardware-like
 * in that they are atoimic functions, yet they may not be implemented in
 * the hardware.  In short, this interface presents an idealized view of
 * device command/control functions.
 *
 * The routines are:
 *
 * play_track() - begin play at a specific track
 * stop_device() - stop the play
 * pause_device() - pause the play
 * resume_play() - resume play after a pause command
 * next_track() - play the track following the currently playing one
 * prev_track() - play the track preceeding the currently playing one
 * eject_device() - barf out the CD
 * poll_device() - update the status and toc structs from device info
 * set_volume() - set (scale) the volume level
 *
 * Button highlighting for the main buttons and the selection buttons
 * is done in this layer.
 *
 */

/************************************************************************
 *
 *	play_track()... play the selected track
 *
 ***********************************************************************/

void
play_track()
{
    status.flags |= CD_EXCLUSIVE;
    status.state = PLAYING;
    
    if (status.flags & CD_SHUFFLE)
      {
	PlayTrack(toc.entry[toc.current].track_number + 1,
		  toc.entry[toc.current].track_number + 1);
      }
    else
      {
	PlayTrack(toc.entry[toc.current].track_number + 1,
		  toc.entry[toc.last].track_number + 1);
      }	
    
    /*
     * reset volume after every play command or else it defaults to the
     * maximum value (ouch!)
     */

    set_volume(status.currentVolume);

    /*
     * update the lights
     */
    
    lightMainButton(k_play_id);
    lightSelectButton(toc.current);

    poll_device();     /* get new numbers for the status struct */

    updateStatusDisplay();
    
    if (status.timer == 0)
	start_timer(primaryWidgets[k_play_id]);
    
    status.flags &= ~CD_EXCLUSIVE;
    
} /* play_track */

/************************************************************************
 *
 * 	next_track()... select the next CD track
 *
 ************************************************************************/

void
next_track() 
{
    if (toc.current < toc.last)
      toc.current++;
    else
      toc.current = 0;
    
    play_track();
     
} /* next_track */

/************************************************************************
 *
 * 	prev_track()... select the previous track
 *
 ************************************************************************
 *
 * On a real CD player, the 1st time you punch " |<< " the current
 * track starts over.  If you quickly punch it again, the previous track
 * plays and so on.  To emmulate that function, if the track has
 * played more than 2 seconds we assume you are punching " |<< " the 1st
 * time in the current command sequence and restart play at the current
 * track.  Else, we play the previous track.
 */

void
prev_track()
{
    if (status.current_seconds < 2)
      {
	if (toc.current)
	  --toc.current;
	else
	  toc.current = toc.last;
      }
    
    play_track();
    
} /* prev_track */

/************************************************************************
 *
 * 	pause_device()... pause the CD in play, if any
 *
 ************************************************************************/

void
pause_device() 
{
    status.flags |= CD_EXCLUSIVE;
    
    stop_timer();
    lightMainButton(k_pause_id);

    PausePlay();

    status.state = PAUSED;
    status.flags &= ~CD_EXCLUSIVE;
  
} /* pause_device */

/************************************************************************
 *
 * 	resume_device()... start playing agin following a pause
 *
 ************************************************************************/

void
resume_device() 
{
    if (status.state == PAUSED)
      {
	status.flags |= CD_EXCLUSIVE;

	if (status.timer == 0)
	  start_timer(primaryWidgets[k_pause_id]);

	status.state = PLAYING;
	lightMainButton(k_play_id);

	ResumePlay();

	status.flags &= ~CD_EXCLUSIVE;
      }
  
} /* resume_device */

/************************************************************************
 *
 * 	stop_device()... stop the player
 *
 ************************************************************************/

void
stop_device() 
{
    status.flags |= CD_EXCLUSIVE;

    status.state = STOPPED;
    toc.current = 0;

    status.current_track = 1;
    status.current_seconds = 0;
    status.current_index = 0;

    lightMainButton(k_stop_id);
    lightSelectButton(-1);

    StopUnit();

    stop_timer();

    updateStatusDisplay();
    updateSelectDisplay();

    status.flags &= ~CD_EXCLUSIVE;

} /* stop_device */

/************************************************************************
 *
 * 	eject_device()... barf up a CD
 *
 ************************************************************************/

void
eject_device() 
{
    status.flags |= CD_EXCLUSIVE;

    status.state = EJECTED;

    status.current_track = 1;
    status.current_seconds = 0;
    status.current_index = 0;

    clearTOC();

    lightMainButton(k_eject_id);
    lightSelectButton(-1);

    updateSelectDisplay();	/* unmap the select buttons */
    updateStatusDisplay();
    
    stop_timer();
    EjectUnit();
    
    status.flags &= ~CD_EXCLUSIVE;

} /* eject_device */

/********************************************************************
 *
 * 	set_volume()... device independant volume setting
 *
 ********************************************************************
 *
 * the input volume value from the volumeSlider widget is in the range
 * 0-100.  The hardware supports volume settings 0-255 so we scale the
 * input values to the device specific range.
 *
 * Also, there are a few inherent problems with the hardware's volume scale.
 * First, in most cases, volume settings below 128 are barely audible, so
 * the hardware's effective effective "real" volume range is from 128 to 255,
 * even though the full 0-255 range is legal. Second, volume deltas appear
 * non-linear to the human ear (ie: your ear thinks a volume change of 5 units
 * is less than the same 5 unit change heard at high volume levels).
 *
 * We solve both these problems by scaling volume slider values of 1-50 to
 * actual values of 128-219 while scaling volume slider values of 51-100 to
 * a smaller range of 220-255.  Zero always scales to zero.  Not the slickest
 * scaling algorithm, but it works.
 */

void
set_volume(sliderValue)
int sliderValue;
{
    int scaledValue;
    float percent;

    status.currentVolume = sliderValue;
    scaledValue = sliderValue;
    
    if (sliderValue)
      {
	if (sliderValue <= 50)
	  {
	    percent = scaledValue / (float)50;
	    scaledValue = (percent * 91) + 128;
	  }
	else
	  {
	    percent = (sliderValue - 50) / (float)50;
	    scaledValue = (percent * 35) + 220;
	  }
      }
    
    /* same volume for right and left */

    status.flags |= CD_EXCLUSIVE;

    SetVolume(scaledValue, scaledValue);

    status.flags &= ~CD_EXCLUSIVE;

} /* set_volume */

/************************************************************************
 *
 * 	start_timer()... start the timer
 *
 ************************************************************************/

void
start_timer(w)
     Widget w;
{
	status.timer = XtAppAddTimeOut(XtWidgetToApplicationContext(w), 
				       950, timeout_proc, w);
}

/************************************************************************
 *
 *	stop_timer()... stop the timer
 *
 ************************************************************************/

void
stop_timer()
{
    if (status.timer != 0)
      XtRemoveTimeOut(status.timer);

    status.timer = 0;
    
} /* stop_timer */

/************************************************************************
 *
 * 	timeout_proc()... timeout procedure
 *
 ************************************************************************
 *
 * This routine is called every second.
 *
 * It updates the highlighting of the select buttons to display the
 * current track, the numbers in the status display window, and handles
 * the various repeat modes.
 *
 * NOTE:  When the caddy is ejected, the timeout is stopped to avoid
 * 	  polling the CDROM.  When the CDROM is polled during it's
 * 	  ejected state, an error message is sent to the log file, which
 * 	  should be avoided.  The timeout is started again when a play
 * 	  command is executed.
 */

static void
timeout_proc(w, t)
     Widget w;
     XtIntervalId *t;
{
    int i;

    if ((status.flags & CD_EXCLUSIVE) == 0)
      {
	poll_device();
	updateStatusDisplay();
		    
	switch (status.state)
	  {
	  case IDLE:

	    /*
	     * if the player is idle we've either finished the disk or
	     * we're in shuffle mode and between tracks.  Handle the
	     * various repeat modes.
	     */
	    
	    if (toc.current >= toc.last)
	      {
		if (status.flags & CD_REPEAT_DISK)
		  {
		    next_track();
		  }
		else if (status.flags & CD_REPEAT_TRACK)
		  {
		    play_track();
		  }
		else
		  {
		    stop_device();
		    return;
		  }
	      }
	    else
	      {
		if (status.flags & CD_REPEAT_TRACK)
		    play_track();
		else
		    next_track();
	      }

	    break;
	    
	  case PLAYING:
	    
	    /*
	     * if not in shuffle mode and...
	     * if the track now playing no longer matches the the
	     * track that the TOC thinks is playing...
	     */

	    if (!(status.flags & CD_SHUFFLE))
	      {
		if (toc.entry[toc.current].track_number !=
		    status.current_track-1)
		  {
		    if (status.flags & CD_REPEAT_TRACK)
		      {
			play_track();
		      }
		    else
		      {
			/*
			 * fix toc.current and the select button highlighting
			 * to reflect new reality.
			 */
			for (i = 0; i < toc.total_tracks; i++)
			  {
			    if (toc.entry[i].track_number ==
				status.current_track-1)
			      toc.current = i;
			  }
			lightSelectButton(toc.current);
		      }
		  }
	      }
	    break;
	    
	  case STOPPED:
	  case EJECTED:
	    return;
		
	  default:
	    break;
	  }
      }

    /*
     * set the timer to fire again.
     */

    status.timer = XtAppAddTimeOut(XtWidgetToApplicationContext(w), 
				   950, timeout_proc, w);
    
} /* timeout_proc */

/***********************************************************************
 *
 *	start_scan_timer()...
 *
 **********************************************************************/

void
start_scan_timer(w)
     Widget w;
{
    status.scan_timer = XtAppAddTimeOut(XtWidgetToApplicationContext(w),
					SCAN_BITE, scan_timeout, w);

} /* start_scan_timer */

/***********************************************************************
 *
 *	stop_scan_timer()...
 *
 **********************************************************************/

void
stop_scan_timer()
{
    if (status.scan_timer != 0)
	XtRemoveTimeOut(status.scan_timer);
    
    status.scan_timer = 0;

  } /* stop_scan_timer */

/***********************************************************************
 *
 *	scan_timout()...
 *
 **********************************************************************
 *
 * This routine implements the audio cues in scan mode.  You enter
 * scan mode by holding down a scan button (">>" or "<<") and exit
 * when you release the button.  When you press a scan button a
 * "scan_timer" is started which causes this routine to get called
 * every "SCAN_BITE" milleseconds.
 *
 * When this routine runs, it restarts play at a point "status.scan_stride"
 * seconds from the current position.  Play then continues until the timer
 * fires and this routine runs again, restarting play at the next position.
 *
 * There is a crude accelerator that causes the stride to get one
 * second larger up to MAX_SCAN_STRIDE seconds, each time this routine runs.
 *
 */

static void
scan_timeout(w, t)
     Widget w;
     XtIntervalId *t;
{
    union cd_address startTime, endTime;
    int seconds;
    int newTrack = FALSE;
    
    start_scan_timer(w);
    
    if ((status.flags & CD_EXCLUSIVE) == 0)
      {
	poll_device();

	/*
	 * compute the absolute address of the new play point expressed in
	 * seconds.  This calculation accounts for the possibility that
	 * we're in shuffle mode with it's non-contiguous tracks.
	 */

	startTime.msf.f_units = 0;
	
	switch(status.scan_direction)
	  {
 	  case BACK:
	      
	    if (status.current_total_seconds - status.scan_stride <
		toc.entry[toc.current].track_address)
	      {
		if (toc.current)
		  --toc.current;
		else
		  toc.current = toc.last;
		
		seconds = (toc.entry[toc.current].track_address +
			   toc.entry[toc.current].track_length) -
			     (status.scan_stride - status.current_seconds);

		newTrack = TRUE;
	      }
	    else
	      {
		seconds = status.current_total_seconds - status.scan_stride;
	      }

	    startTime.msf.m_units = seconds / 60;
	    startTime.msf.s_units = seconds % 60;
	    break;

	  case AHEAD:

	    if ((status.current_seconds + status.scan_stride) >=
		toc.entry[toc.current].track_length)
	      {
		seconds = (status.current_seconds + status.scan_stride) - 
                          toc.entry[toc.current].track_length;
		
		if (toc.current == toc.last)
		  toc.current = 0;
		else
		  toc.current++;

		seconds += toc.entry[toc.current].track_address;

		newTrack = TRUE;
	      }
	    else
	      {
		seconds = status.current_total_seconds + status.scan_stride;
	      }

	    startTime.msf.m_units = seconds / 60;
	    startTime.msf.s_units = seconds % 60;
	    break;

	  case OFF:
	    return;
	  }

	/*
	 * set play end interval. (we reissue the play command in the
	 * button_disarm callback).
	 */
	
	endTime.msf.f_units = 0;

	if (status.flags & CD_SHUFFLE)
	  {
	    seconds = (toc.entry[toc.current].track_address +
		       toc.entry[toc.current].track_length);
	  }
	else
	  {
	    seconds = toc.total_time;
	  }
	    
	endTime.msf.s_units = seconds % 60;
	endTime.msf.m_units = seconds / 60;

	PlayTime(startTime, endTime);

	/*
	 * If you don't set volume, it defaults to the wrong value
	 */

	set_volume(status.currentVolume);	

	/*
	 * Primitive scan speed acceleration; the longer you hold down
	 * the scan button, the more music we jump over between samples.
	 */

	if (status.scan_stride < MAX_SCAN_STRIDE)
	  status.scan_stride += 2;

	if (newTrack)
	  {
	    lightSelectButton(toc.current);
	    poll_device();	/* get new numbers for display update */
	  }
      }

    updateStatusDisplay();

} /* scan_timeout */

/************************************************************************
 *
 *	poll_device()... read status from the hardware
 *
 ************************************************************************
 *
 * This routine put status info from the hardware into the global
 * "status" structure.
 *
 * It also loads toc.total_tracks and toc.last
 */

void
poll_device()
{
    struct cd_subc_information sci;
    struct cd_subc_header *sch;
    struct cd_subc_position *scp;
    struct cd_playback_status ps;
    struct cd_toc_header *th;
    struct cd_toc_entry *tocentp;      /* Pointer to array of entries	*/
    union cd_address *abaddr, *caddr;
    int csecaddr, absecaddr;
    int i;
    
    sch = &sci.sci_header; 
    scp = &sci.sci_scp;  	
    abaddr = &scp->scp_absaddr;

    /*
     * If the device is not currently open, try the open again
     */

    if ((status.flags & CD_DEVICE_OPEN) == 0)
      {
	if (open_dev())
	  {
	    status.state = EJECTED;
	    clearTOC();
	    return;
	  }
	status.flags |= CD_DEVICE_OPEN;
      }
    
    tocentp = glob_toc.cdte;
    th = &glob_toc.cdth;
    
    /*
     * Set address format to minutes, seconds, frame
     */

    SetAddrFormatMsf();

    /*
     * go fetch the current playback status from the device
     */
    
    if (GetPlaybackStatus(&ps))
      {
	/* The disk must have been removed */

	status.state = EJECTED;
	status.current_track = 1;
	status.current_seconds = 0;
	status.current_index = 0;

	clearTOC();
	lightMainButton(k_eject_id);
	lightSelectButton(-1);
	updateSelectDisplay();	/* unmap the select buttons */
	stop_timer();
	return;
      }

    /*
     * poll the device for status and update out own status info.
     */
    
    GetPlayPosition(&sci);
	
    status.current_track = scp->scp_track_number;
    status.current_index = scp->scp_index_number;
	
    if(GetTOC(&glob_toc) == 0)
      {
	toc.total_tracks = (th->th_ending_track - th->th_starting_track) + 1;
	toc.last = toc.total_tracks - 1;
      }
    else
      {
	stop_device();
	clearTOC();
      }
    
    /*
     * if the device is in play mode...
     */

    if (ps.ps_audio_status == PS_PLAY_IN_PROGRESS)
      {
	status.state = PLAYING;

	/*
	 * calculate the number of seconds we're into the current track...
	 */
	
	for (i = 0, caddr = 0; i < toc.total_tracks; i++)
	  {
	    if (tocentp[i].te_track_number == status.current_track)
	      { 
		caddr = &tocentp[i].te_absaddr;
		break;
	      }
	  }

	/*
	 * caddr is address of beginning of current track.
	 * abaddr is address of current position.
	 */

	if (caddr != 0)
	  {
	    csecaddr = (caddr->msf.m_units * 60) + caddr->msf.s_units;
	    absecaddr = (abaddr->msf.m_units * 60) + abaddr->msf.s_units;
	    status.current_seconds = absecaddr - csecaddr;
	    status.current_total_seconds = absecaddr;
	  }
      }
    else
      {
	status.state = IDLE;
      }

} /* poll_device */

/************************************************************************
 *
 * 	getTrackLength()... get the length of a specific track #
 *
 ************************************************************************
 *
 * This routine reads the hardware structure so poll_device() must have
 * been called sometime after the disk was inserted but before you get
 * here. 
 */

int
getTrackLength(trackNum)
     int trackNum;
{
    struct cd_toc_entry *toc_entry;	/* Pointer to array of entries	*/
    struct cd_address *thisaddrp, *nextaddrp;	
    int secs;

    toc_entry = glob_toc.cdte;
    
    thisaddrp = &toc_entry[trackNum].te_absaddr;
    nextaddrp = &toc_entry[trackNum + 1].te_absaddr;

    /*
     * using the absolute address of the current track and the next
     * track, calculate the number of seconds in the current track.
     */

    secs = ((nextaddrp->msf.m_units * 60) + nextaddrp->msf.s_units) - 
           ((thisaddrp->msf.m_units * 60) + thisaddrp->msf.s_units);

    return(secs);
    
} /* getTrackLength */

