/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 sample.c
 *	Date:	 3/29/90
 *	Version: 1.28
 *
 *	sample.c - Code to magnify the sample record.
 *
 *	Print the headers and display the static data and other interesting
 *	data from the "first" record.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)sample.c	1.28 (monitor) 3/29/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of record.h to monitor.h.
 *
 * Feb. 15, 1989 -- arr
 *
 *	Added limits the width of strings that are printed and more
 *	data to the sample display.
 *
 * Feb. 17, 1989 -- arr
 *
 *	Moved f_first(), f_last(), f_sample() from functions.c.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <curses.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#include <net/if.h>
#include <netinet/in.h>

#include "include.h"
#include "monitor.h"
#include "extern.h"

/*
 *	Default display.
 */
extern	int	lines ;		/* length of screen being used */
extern	int	cols ;		/* width of screen being used */
extern	WINDOW	*wp ;

/*
 *	Functions that don't return an (int).
 */
char	*ctime(),
	*strncpy(),
	*str_arch(),
	*str_data(),
	*format_time() ;

static	mag_ticks = 0 ;

/*
 *	Display the changing data.
 *
 *	(date and elapsed time)
 *
 *	Because all of the magnify functions print the same first
 *	two pieces of infromation, I've moved the common code into
 *	their own functions.
 *
 *	ARGSUSED
 */
magnify_sample()
{
	sample_body((double)(sample.mon_ticks - mag_ticks)/first.mon_hz);

	mag_ticks = sample.mon_ticks ;
}

#define	SAMPLE_LINES	(20)

/*
 *	This is arbitrarily chosen number that accounts for the 
 *	member headers and a couple of character at the end of the
 *	screen.
 */
#define	SAMPLE_WIDTH	(17)

/*
 *	Print the header for the sample record.
 *
 *	ARGSUSED
 */
open_sample()
{
	char	*newline, *strchr() ;
	char	boot_time[M_DATESIZE+1], start_time[M_DATESIZE+1] ;
	int	width ;

	sample_header() ;

	lines = SAMPLE_LINES ;
	width = cols - SAMPLE_WIDTH ;

	boot_time[M_DATESIZE] = '\0' ;
	start_time[M_DATESIZE] = '\0' ;

	/*
	 *	A quick little bandage for old data files.
	 */
	if((newline = strchr(first.mon_version, '\n')) != NULL )
		*newline = '\0' ;

	(void)strncpy(boot_time, ctime((long *)&first.mon_boot.tv_sec), M_DATESIZE) ;
	(void)strncpy(start_time, ctime((long *)&first.mon_sdate), M_DATESIZE) ;

	/*
	 *	Take the new lines off the dates.
	 */
	if((newline = strchr(boot_time, '\n')) != NULL )
		*newline = '\0' ;

	if((newline = strchr(start_time, '\n')) != NULL )
		*newline = '\0' ;

	wprintw(wp, "Version:      %-.*s\n", width, first.mon_version);
	wprintw(wp, "Boot Time:    %-.*s\n", width, boot_time) ;
	wprintw(wp, "Start date:   %-.*s\n", width, start_time);
	wprintw(wp, "Monitor:      V%d.%d\n", major(first.mon_record),
		minor(first.mon_record)) ;

	wprintw(wp, "Arch.:        %-.*s\n", width, str_arch(first.mon_arch)) ;
	wprintw(wp, "Hostname:     %-.*s\n", width, first.mon_hostname);
	wprintw(wp, "Sleep time:   %d seconds\n", first.mon_sleep);
	wprintw(wp, "# of CPU's:   %d\n", first.mon_cpu) ;
	wprintw(wp, "# of Disks:   %d\n", first.mon_disk) ;
	wprintw(wp, "# of Netif's: %d\n", first.mon_netif) ;
	wprintw(wp, "# of BUF's:   %d\n", first.mon_buf) ;
	wprintw(wp, "Physmem:      %d Kb.\n", pgtok(first.mon_physmem)) ;
	wprintw(wp, "Hz:           %d\n", first.mon_hz);
	wprintw(wp, "NBPG:         %d\n", first.mon_nbpg) ;
	wprintw(wp, "DEV_BSIZE:    %d\n", first.mon_bsize) ;
	wprintw(wp, "Options:      <%-.*s>\n", width, str_data(first.mon_options));
}

/*
 *	The common code for all magnify headers.
 */
sample_header()
{
	wclear(wp) ;

	wmove(wp, 0, 0) ;

	wprintw(wp, "Elapsed time:\n");
	wprintw(wp, "Current time:\n\n");
}

#define	BODY_OFFSET	(14)
#define	DATE_WIDTH	(24)

/*
 *	The common code to print the elapsed time and current
 *	current time.
 */
sample_body(etime)
double	etime ;
{
	char	 current_time[M_DATESIZE], *newline, *strchr() ;

	current_time[M_DATESIZE] = '\0' ;

	(void)strncpy(current_time, ctime((long *)&sample.mon_timestamp),
		M_DATESIZE) ;

	if((newline = strchr(current_time, '\n')) != NULL )
		*newline = '\0' ;

	wmove(wp, 0, BODY_OFFSET) ;
	wprintw(wp, "%s\n", format_time(etime));

	wmove(wp, 1, BODY_OFFSET) ;
	wprintw(wp, "%*.*s", DATE_WIDTH, DATE_WIDTH, current_time) ;
}

/*
 *	Turn an elapsed time into a readable string.
 */
char	*format_time(etime)
double	etime ;
{
	static	 char	buf[BUFSIZ] ;
	double	 fract ;
	register seconds, minutes, hours, days ;

	seconds = etime ;
	fract = (etime - seconds) + (seconds % 60) ;

	days = seconds / (24 * 60 * 60) ;
	seconds %= (24 * 60 * 60) ;
	hours = seconds / (60 * 60) ;
	seconds %= (60 * 60) ;
	minutes = seconds / 60 ;

	if( days )
		(void)sprintf(buf, "%d %2d:%02.2d:%05.2f",
			days, hours, minutes, fract) ;
	else if( hours )
		(void)sprintf(buf, "%d:%02.2d:%05.2f", hours, minutes, fract) ;
	else if( minutes )
		(void)sprintf(buf, "%d:%05.2f", minutes, fract) ;
	else
		(void)sprintf(buf, "%.2f", fract) ;

	return buf ;
}

/*
 *	Dump the contents of the FIRST record.
 */
f_first(p)
struct mon_first *p ;
{
	char	*strchr(), *newline ;

	if((p->mon_flag & MON$M_VALID) == 0 )
		return ;

	printf("%s.\n", records[MON$C_FIRST].string) ;

	/*
	 *	If the version string has a newline in it, replace it
	 *	with a NULL.  Otherwise force a NULL at the end of the
	 *	allocated space.
	 */
	if((newline = strchr(p->mon_version, '\n')) != NULL )
		*newline = '\0' ;
	else
		p->mon_version[MON$S_VERSION] = '\0' ;

	printf("\tversion:  %s\n", p->mon_version);
	printf("\tboot:     %s", ctime((long *)&p->mon_boot.tv_sec)) ;
	printf("\tsdate:    %s", ctime((long *)&p->mon_sdate));
	printf("\tmonitor:  V%d.%d\n", major(p->mon_record),
		minor(p->mon_record)) ;

	printf("\tarch:     %s\n", str_arch(p->mon_arch)) ;
	printf("\thostname: %s\n", p->mon_hostname);
	printf("\tsleep:    %d seconds\n", p->mon_sleep);
	printf("\tcpu:      %d\n", p->mon_cpu);
	printf("\tdisk:     %d\n", p->mon_disk);
	printf("\tnetif:    %d\n", p->mon_netif);
	printf("\tbuf:      %d\n", p->mon_buf);
	printf("\tphysmem:  %d Kb.\n", pgtok(p->mon_physmem));
	printf("\thz:       %d\n", p->mon_hz);
	printf("\tnbpg:     %d\n", p->mon_nbpg) ;
	printf("\tbsize:    %d\n", p->mon_bsize) ;
	printf("\tpid:      %d\n", p->mon_pid);

	printf("\toptions:  <%s>\n", str_data(p->mon_options));
}

/*
 *	Print the contents of the "last" record.
 */
f_last(p)
struct mon_last *p ;
{
	if((p->mon_flag & MON$M_VALID) == 0 )
		return ;

	printf("%s.\n", records[MON$C_LAST].string) ;

	printf("\tedate:   %s", ctime((long *)&p->mon_edate));
	printf("\tsignal:  %d\n", p->mon_signal);
	printf("\tcode:    %d\n", p->mon_code);

	printf("\tusage:\n") ;
	print_rusage(&p->mon_usage, "\t\t") ;
}

/*
 *	Print the contents of the sample record.
 */
f_sample(p)
struct mon_sample *p ;
{
	if((p->mon_flag & MON$M_VALID) == 0 )
		return ;

	printf("%s.\n", records[MON$C_SAMPLE].string) ;

	printf("\ttimestamp: %s", ctime((long *)&p->mon_timestamp));
	printf("\tdatalen:   %d\n", p->mon_datalen);
	printf("\tticks:     %d ticks\n", p->mon_ticks) ;
}

char	*str_arch(arch)
int	arch ;
{
	if( arch == MON$C_VAX )
		return "VAX" ;
	else if( arch == MON$C_MIPS )
		return "MIPS" ;
	else
		return "Unknown archtecture." ;
}
