/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 os.c
 *	Date:	 3/29/90
 *	Version: 1.30
 *
 *	os.c - Functions to magnify general O/S info (paging, memory,
 *	processes, etc).  Also contains the code for calculating
 *	changes in tty's.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)os.c	1.30 (monitor) 3/29/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 * 
 * Jan. 6, 1989 -- arr
 *
 *	?
 *
 * Feb. 15, 1989 -- arr
 *
 *	Turned pgtok() macro into a function.  I had to do this because
 *	the page size on the VAX and MIPS are different.  In order to
 *	have data be portable between architectures, I have to store
 *	the page size in the first record.
 *
 * Feb. 26, 1989 -- arr
 *
 *	Move the FORK functions and data to their own file.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <curses.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#include <net/if.h>
#include <netinet/in.h>

#include "include.h"
#include "options.h"
#include "monitor.h"
#include "extern.h"

/*
 *	Each magnify function has to remember when it was last run
 *	so that it can maintain the correct elapsed time.
 *
 *	These will need to be external and have to be reset for
 *	session being replayed.
 */
int	proc_ticks = 0,		/* clock ticks at previous sample */
	memory_ticks = 0,
	sample_ticks = 0,
	paging_ticks = 0 ;

/*
 *	Default display.
 */
extern	int	lines ;		/* length of the screen being used. */
extern	WINDOW	*wp ;

/*
 *	Other globals, defined in screen.c unless otherwise noted.
 */
extern	pid_ticks[],		/* clock ticks at previous sample: in pid.c */
	soft_faults[],
	hard_faults[],
	in_block[],
	out_block[],
	send_message[],
	rcv_message[] ;

/*
 *	Constants for the placement of the memory data.
 */
#define	MEMORY_PHYS	(3)
#define	MEMORY_X	(23)

#define	MEMORY_LINES	(16)
#define	FREE_LINES	(6)

/*
 *	Print the headers and static information for the memory
 *	and freemem records.  If we are printing the freemem
 *	record then we'll return after printing the "Free Memory".
 *
 *	Lines will initial set for displaying free memory, but
 *	reset if all memory is being displayed.
 */
open_memory(op)
OPTION	*op ;
{
	sample_header() ;

	wmove(wp, MEMORY_PHYS, 0) ;

	lines = FREE_LINES ;

	wprintw(wp, "Physical memory:       %10d Kb.\n", pgtok(first.mon_physmem));
	wprintw(wp, "Free Memory:           %10d Kb.\n\n", 0);

	if( op->opt_free )
		return ;

	lines = MEMORY_LINES ;

	wprintw(wp, "Total virtual memory:  %10d Kb.\n", 0);
	wprintw(wp, "Active virtual memory: %10d Kb.\n", 0);
	wprintw(wp, "Total real memory:     %10d Kb.\n", 0);
	wprintw(wp, "Active real memory:    %10d Kb.\n\n", 0);

	wprintw(wp, "Total virtual text:    %10d Kb.\n", 0);
	wprintw(wp, "Active virtual text:   %10d Kb.\n", 0);
	wprintw(wp, "Total real text:       %10d Kb.\n", 0);
	wprintw(wp, "Active real text:      %10d Kb.\n", 0);
}

/*
 *	The magnify function for memory.
 */
magnify_memory(op)
OPTION	*op ;
{
	register free = op->opt_free ? mon_free.mon_freemem : memory.mon_free ;

	sample_body((double)(sample.mon_ticks - memory_ticks)/first.mon_hz) ;
	memory_ticks = sample.mon_ticks ;

	wmove(wp, MEMORY_PHYS + 1, MEMORY_X) ;
	wprintw(wp, "%10d", pgtok(free));

	if( op->opt_free )
		return ;

	wmove(wp, MEMORY_PHYS + 3, MEMORY_X) ;
	wprintw(wp, "%10d", pgtok(memory.mon_vm));
	wmove(wp, MEMORY_PHYS + 4, MEMORY_X) ;
	wprintw(wp, "%10d", pgtok(memory.mon_avm));
	wmove(wp, MEMORY_PHYS + 5, MEMORY_X) ;
	wprintw(wp, "%10d", pgtok(memory.mon_rm));
	wmove(wp, MEMORY_PHYS + 6, MEMORY_X) ;
	wprintw(wp, "%10d", pgtok(memory.mon_arm));

	wmove(wp, MEMORY_PHYS + 8, MEMORY_X) ;
	wprintw(wp, "%10d", pgtok(memory.mon_vmtxt));
	wmove(wp, MEMORY_PHYS + 9, MEMORY_X) ;
	wprintw(wp, "%10d", pgtok(memory.mon_avmtxt));
	wmove(wp, MEMORY_PHYS + 10, MEMORY_X) ;
	wprintw(wp, "%10d", pgtok(memory.mon_rmtxt));
	wmove(wp, MEMORY_PHYS + 11, MEMORY_X) ;
	wprintw(wp, "%10d", pgtok(memory.mon_armtxt));
}

/*
 *	The page data is divided up into 5 areas.  These are the
 *	Y origins and X offsets for that data.
 */
#define	PAGE_AREA_1	(3)
#define	PAGE_AREA_3	(11)
#define	PAGE_AREA_4	(18)
#define	PAGE_OFF_1	(25)
#define	PAGE_OFF_2	(58)
#define	PAGE_OFF_3	(37)
#define	PAGE_OFF_4	(25)
#define	PAGE_OFF_5	(64)

/*
 *	ARGSUSED
 */
magnify_paging(op)
OPTION	*op ;
{
	sample_body((double)(sample.mon_ticks - paging_ticks)/first.mon_hz) ;
	paging_ticks = sample.mon_ticks ;

	/*
	 *	Area 1 - Page faults, pages free by daemon, reclaims,
	 *	deficit and free memory.
	 */
	wmove(wp, PAGE_AREA_1, PAGE_OFF_1) ;
	wprintw(wp, "%10d", mon_page.mon_faults);

	wmove(wp, PAGE_AREA_1 + 1, PAGE_OFF_1) ;
	wprintw(wp, "%10d", mon_page.mon_dfree);

	wmove(wp, PAGE_AREA_1 + 2, PAGE_OFF_1) ;
	wprintw(wp, "%10d", mon_page.mon_intrans);

	wmove(wp, PAGE_AREA_1 + 3, PAGE_OFF_1) ;
	wprintw(wp, "%10d", mon_page.mon_pgrec);

	wmove(wp, PAGE_AREA_1 + 4, PAGE_OFF_1) ;
	wprintw(wp, "%10d", mon_page.mon_fastpgrec);

	wmove(wp, PAGE_AREA_1 + 5, PAGE_OFF_1) ;
	wprintw(wp, "%10d", mon_page.mon_deficit);

	wmove(wp, PAGE_AREA_1 + 6, PAGE_OFF_1) ;
	wprintw(wp, "%10d", pgtok(mon_page.mon_freemem)) ;

	/*
	 *	Area 2 - Pages swapped in/out, pageins/pageouts and
	 *	swaps.
	 */
	wmove(wp, PAGE_AREA_1, PAGE_OFF_2) ;
	wprintw(wp, "%10d", mon_page.mon_pswpin);

	wmove(wp, PAGE_AREA_1 + 1, PAGE_OFF_2) ;
	wprintw(wp, "%10d", mon_page.mon_pswpout);

	wmove(wp, PAGE_AREA_1 + 2, PAGE_OFF_2) ;
	wprintw(wp, "%10d", mon_page.mon_pgin);

	wmove(wp, PAGE_AREA_1 + 3, PAGE_OFF_2) ;
	wprintw(wp, "%10d", mon_page.mon_pgout);

	wmove(wp, PAGE_AREA_1 + 4, PAGE_OFF_2) ;
	wprintw(wp, "%10d", mon_page.mon_swpin);

	wmove(wp, PAGE_AREA_1 + 5, PAGE_OFF_2) ;
	wprintw(wp, "%10d", mon_page.mon_swpout);

	/*
	 *	Area 3 - Misc. paging stats.
	 */
	wmove(wp, PAGE_AREA_3, PAGE_OFF_3) ;
	wprintw(wp, "%10d", mon_page.mon_pgfrec);

	wmove(wp, PAGE_AREA_3 + 1, PAGE_OFF_3) ;
	wprintw(wp, "%10d", mon_page.mon_scan);

	wmove(wp, PAGE_AREA_3 + 2, PAGE_OFF_3) ;
	wprintw(wp, "%10d", mon_page.mon_rev);

	wmove(wp, PAGE_AREA_3 + 3, PAGE_OFF_3) ;
	wprintw(wp, "%10d", mon_page.mon_seqfree);

	wmove(wp, PAGE_AREA_3 + 4, PAGE_OFF_3) ;
	wprintw(wp, "%10d", mon_page.mon_xsfrec);

	wmove(wp, PAGE_AREA_3 + 5, PAGE_OFF_3) ;
	wprintw(wp, "%10d", mon_page.mon_xifrec);

	/*
	 *	Area 4 - 
	 */
	wmove(wp, PAGE_AREA_4, PAGE_OFF_4) ;
	wprintw(wp, "%10d", mon_page.mon_zfod);

	wmove(wp, PAGE_AREA_4 + 1, PAGE_OFF_4) ;
	wprintw(wp, "%10d", mon_page.mon_vrfod);

	wmove(wp, PAGE_AREA_4 + 2, PAGE_OFF_4) ;
	wprintw(wp, "%10d", mon_page.mon_exfod);

	/*
	 *	Area 5 - 
	 */
	wmove(wp, PAGE_AREA_4, PAGE_OFF_5) ;
	wprintw(wp, "%10d", mon_page.mon_nzfod);

	wmove(wp, PAGE_AREA_4 + 1, PAGE_OFF_5) ;
	wprintw(wp, "%10d", mon_page.mon_nvrfod);

	wmove(wp, PAGE_AREA_4 + 2, PAGE_OFF_5) ;
	wprintw(wp, "%10d", mon_page.mon_nexfod);
}

/*
 *	Offsets of the various field titles.
 */
#define	PAGE_TITLE_1	(0)
#define	PAGE_TITLE_2	(40)
#define	PAGE_TITLE_3	(0)
#define	PAGE_TITLE_4	(0)
#define	PAGE_TITLE_5	(40)

#define	PAGE_LINES	(22)

/*
 *	Print and format the heading for the paging info.
 * 
 *	ARGSUSED
 */
open_paging(op)
OPTION	*op ;
{
	sample_header() ;

	lines = PAGE_LINES ;

	wmove(wp, PAGE_AREA_1, 0) ;
	wprintw(wp, "Total page faults:");
	wmove(wp, PAGE_AREA_1, PAGE_TITLE_2) ;
	wprintw(wp, "Pages swapped in:");

	wmove(wp, PAGE_AREA_1 + 1, 0) ;
	wprintw(wp, "Pages freed by daemon:");
	wmove(wp, PAGE_AREA_1 + 1, PAGE_TITLE_2) ;
	wprintw(wp, "Pages swapped out:");

	wmove(wp, PAGE_AREA_1 + 2, 0) ;
	wprintw(wp, "Intransit page Faults:");
	wmove(wp, PAGE_AREA_1 + 2, PAGE_TITLE_2) ;
	wprintw(wp, "Pageins:");

	wmove(wp, PAGE_AREA_1 + 3, 0) ;
	wprintw(wp, "Page reclaims:");
	wmove(wp, PAGE_AREA_1 + 3, PAGE_TITLE_2) ;
	wprintw(wp, "Pageouts:");

	wmove(wp, PAGE_AREA_1 + 4, 0) ;
	wprintw(wp, "Fast reclaims in locore:");
	wmove(wp, PAGE_AREA_1 + 4, PAGE_TITLE_2) ;
	wprintw(wp, "Swap in's:");

	wmove(wp, PAGE_AREA_1 + 5, 0) ;
	wprintw(wp, "Short term deficit:");
	wmove(wp, PAGE_AREA_1 + 5, PAGE_TITLE_2) ;
	wprintw(wp, "Swap out's:");

	wmove(wp, PAGE_AREA_1 + 6, 0) ;
	wprintw(wp, "Free Memory in Kb.:") ;

	wmove(wp, PAGE_AREA_3, 0);
	wprintw(wp, "Page reclaims from free list:\n");
	wprintw(wp, "Pages scanned by clock daemon:\n");
	wprintw(wp, "Revolutions of the clock hand:\n");
	wprintw(wp, "Sequential process pages freed:\n");
	wprintw(wp, "Swap text pages found in free list:\n");
	wprintw(wp, "Inode text pages found in free list:\n");

	/*
	 *	Area 4 - 
	 */
	wmove(wp, PAGE_AREA_4, 0) ;
	wprintw(wp, "Zero fill page faults:");

	wmove(wp, PAGE_AREA_4 + 1, 0) ;
	wprintw(wp, "File fill page faults:");

	wmove(wp, PAGE_AREA_4 + 2, 0) ;
	wprintw(wp, "a.out fill page faults:");

	/*
	 *	Area 5 - 
	 */
	wmove(wp, PAGE_AREA_4, PAGE_TITLE_5) ;
	wprintw(wp, "Zero fill pages created:");

	wmove(wp, PAGE_AREA_4 + 1, PAGE_TITLE_5) ;
	wprintw(wp, "File fill pages created:");

	wmove(wp, PAGE_AREA_4 + 2, PAGE_TITLE_5) ;
	wprintw(wp, "a.out fill pages created:");
}

#define	PROC_ORIGIN	(3)
#define	PROC_OFFSET	(18)

#define	PROC_LINES	(9)

/*
 *	Functions for doing the process counts.
 *
 *	ARGSUSED
 */
open_process(op)
OPTION	*op ;
{
	sample_header() ;

	lines = PROC_LINES ;

	wprintw(wp, "Run Queue:\n");
	wprintw(wp, "Disk Wait:\n");
	wprintw(wp, "Page Wait:\n");
	wprintw(wp, "Sleeping in core:\n");
	wprintw(wp, "Swapped jobs:");
}

/*
 *	ARGSUSED
 */
magnify_process(op)
OPTION	*op ;
{
	sample_body((double)(sample.mon_ticks - proc_ticks)/first.mon_hz) ;
	proc_ticks = sample.mon_ticks ;

	wmove(wp, PROC_ORIGIN, PROC_OFFSET) ;
	wprintw(wp, "%3d", mon_proc.mon_rq);

	wmove(wp, PROC_ORIGIN + 1, PROC_OFFSET) ;
	wprintw(wp, "%3d", mon_proc.mon_dw);

	wmove(wp, PROC_ORIGIN + 2, PROC_OFFSET) ;
	wprintw(wp, "%3d", mon_proc.mon_pw);

	wmove(wp, PROC_ORIGIN + 3, PROC_OFFSET) ;
	wprintw(wp, "%3d", mon_proc.mon_sl);

	wmove(wp, PROC_ORIGIN + 4, PROC_OFFSET) ;
	wprintw(wp, "%3d", mon_proc.mon_sw);
}

struct	tot_disk {
	int	t_ticks ;	/* clock ticks at previous sample */
	int	t_xfer,
		t_seek,
		t_wds,
		t_time ;
} ;

struct	tot_disk	t_disk[DK_NDRIVE] ;

/*
 *	Initialize whatever static counters need to be initialized.
 * 
 *	Previously only fork and tty were initialized.  Since I made
 *	disk structures static, I added code to initialize those.  I 
 *	have also included the delta functions for disk.  I later moved
 *	the fork setup code into fork.c.  Eventually I'll do the same
 *	with ttys.
 */
init_statics(op)
OPTION	*op ;
{
	if( op->opt_fork )
		setup_fork() ;

	if( op->opt_tty )
		init_tty_data() ;

	if( op->opt_free || op->opt_memory )
		memory_ticks = 0 ;

	if( op->opt_page )
		paging_ticks = 0 ;

	if( op->opt_sample )
		sample_ticks = 0 ;

	if( op->opt_proc )
		proc_ticks = 0 ;

	if( op->opt_pid ) {
		pid_ticks[CHILD]    = pid_ticks[SELF] = 0 ;
		soft_faults[CHILD]  = soft_faults[SELF] = 0 ;
		hard_faults[CHILD]  = hard_faults[SELF] = 0 ;
		in_block[CHILD]     = in_block[SELF] = 0 ;
		out_block[CHILD]    = out_block[SELF] = 0 ;
		send_message[CHILD] = send_message[SELF] = 0 ;
		rcv_message[CHILD]  = rcv_message[SELF] = 0 ;
	}

	if( op->opt_disk )
		bzero((char *)t_disk, n_disk * sizeof(struct tot_disk));
}

/*
 *	Calculate the change in the disk I/O stats.
 *
 *	The sample time (ticks) is maintained for each disk in the
 *	list so that a future feature will be easier to impliment
 *	(add and subtract data).
 *
 *	Note: This module is here because it keeps previous sample
 *	data local.
 */
double	delta_disk(index, ticks)
register index, ticks ;
{
	register tmp ;
	double	 etime ;

	etime = (double)(ticks - t_disk[index].t_ticks)/first.mon_hz ;
	t_disk[index].t_ticks = ticks ;

	tmp = disk[index].mon_xfer ;
	      disk[index].mon_xfer -= t_disk[index].t_xfer ;
	      t_disk[index].t_xfer = tmp ;

	tmp = disk[index].mon_seek ;
	      disk[index].mon_seek -= t_disk[index].t_seek ;
	      t_disk[index].t_seek = tmp ;

	tmp = disk[index].mon_wds ;
	      disk[index].mon_wds -= t_disk[index].t_wds ;
	      t_disk[index].t_wds = tmp ;

	tmp = disk[index].mon_time ;
	      disk[index].mon_time -= t_disk[index].t_time ;
	      t_disk[index].t_time = tmp ;

	return etime ;
}

/*
 *	This function assumes that 1024 and the value of NBPG are
 *	multiples, depending on their relative values.  I believe
 *	this is a safe assumption on a binary computer.
 */
pgtok(x)
int	x ;
{
	if( first.mon_nbpg < 1024 )
		return (x) / (1024 / first.mon_nbpg) ;
	else
		return (x) * (first.mon_nbpg / 1024) ;
}

/*
 *	Might as well make this one a function also...
 *
 *	A function for turning number of sectors into one Kilobyte
 *	units.  It uses the block size constant.
 */
btok(x)
int	x ;
{
	if( first.mon_bsize < 1024 )
		return (x) / (1024 / first.mon_bsize) ;
	else
		return (x) * (first.mon_bsize / 1024) ;
}
