/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 options.c
 *	Date:	 3/29/90
 *	Version: 1.71
 *
 *	options.c - This file contains functions for processing the command 
 *	line to set various available options.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)options.c	1.71 (monitor) 3/29/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 *
 * 28-June-1988 -- arr
 *
 *	Put the various ..._set data structures into one bit
 *	field.
 *
 *	Make a "security" check in the -file and -output options.
 *	This uses a global variable called "secure" that may be
 *	patched with adb(1).
 *
 * 29-June-1988 -- arr
 *
 *	Compress the switches in display_option() and collect_option()
 *	to use as much common code as possible.
 *
 *	Change the way OPT_READ was handled to use the replay fuctions
 *	instead of it's input functions.
 *
 *	Further compressed display_option() and collect_option() so
 *	that they went away completely.  Changed the handling of
 *	pid collection so that it goes though the OPT_LIVE code.
 *
 *	Remove the references to OPT_RATE.  Replace most of the
 *	switch tables in set_display() and set_collect() with a
 *	loop using the display_table and collect_table arrays.
 *
 *  4-July-1988 -- arr
 *
 *	Remove the restriction that let monitor only collect data
 *	on processes when the opt_pid option was turned on.
 *
 *	Fixed the error message that says the display was changed
 *	to SCREEN.  It previously said that the COLLECTION mode was
 *	changed to SCREEN.
 *
 * Jan. 15 1989 -- arr
 *
 *	Turned on SWAP space data collection.
 *
 * Jan. 16, 1989 -- arr
 *
 *	Turn on collection of buffer and namei cache.
 *
 * Jan. 18, 1989 -- arr
 *
 *	Added "all" option and made it the default for data collection.
 *	Also removed ALLOW_NAMES and PROC_ONLY #ifdef's.
 *
 * Dec. 28, 1989 -- arr
 *
 *	Added support for fractional seconds.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#include <net/if.h>
#include <netinet/in.h>

#include "options.h"
#include "monitor.h"
#include "include.h"
#include "extern.h"

/*
 *	Functions that don't return (int).
 */
void	exit() ;

/*
 *	Yet another convient macro...
 */
#define	MACRO(s, code, bit, type)	{ s, sizeof(s)-1, code, bit, type, }

/*
 *	Bit encodings used locally.
 */
#define	OPT_NEXTWORD	01

/*
 *	No relationships between the indexs and the fields should be 
 *	assumed.  Gaps have been left in the OPT_ number that aren't
 *	reflected in the size of the table.
 */
static struct tag {
	char	*opt_string ;		/* the name of the option */
	int	opt_length,		/* the length of the string */
		opt_code,		/* The OPT_... code */
		opt_bit,		/* The MON$M_ bit for the data option */
		opt_type ;		/* The option type */
} array[] = {
/*
 *	Display modes (0-7).
 */
	MACRO("save",	 OPT_SAVE,	0, OPT_DISPLAY),
	MACRO("screen",	 OPT_SCREEN,	0, OPT_DISPLAY),
	MACRO("window",	 OPT_WINDOW,	0, OPT_RFFU),
#ifdef	ALLOW_NOP
	MACRO("nop",	 OPT_NOP,	0, OPT_DISPLAY),
#else
	MACRO("nop",	 OPT_NOP,	0, OPT_NOSUP),
#endif
	MACRO("print",	 OPT_PRINT,	0, OPT_RFFU),
	MACRO("read",	 OPT_READ,	0, OPT_DISPLAY),
/*
 *	Collection modes (8-15).
 */
	MACRO("live",	 OPT_LIVE,	0, OPT_COLLECT),
	MACRO("replay",	 OPT_REPLAY,	0, OPT_COLLECT),
	MACRO("names",	 OPT_NAMES,	0, OPT_COLLECT),
/*
 *	Data selection (16-39).
 */
	MACRO("cpu",	 OPT_CPU,	MON$M_CPU, OPT_DATA),
	MACRO("tty",	 OPT_TTY,	MON$M_TTY, OPT_DATA),
	MACRO("disk",	 OPT_DISK,	MON$M_DISK, OPT_DATA),
	MACRO("free",	 OPT_FREE,	MON$M_FREE, OPT_DATA),
	MACRO("fork",	 OPT_FORK,	MON$M_FORK, OPT_DATA),
	MACRO("page",	 OPT_PAGE,	MON$M_PAGE, OPT_DATA),
	MACRO("process", OPT_PROC,	MON$M_PROC, OPT_DATA),
	MACRO("swap",	 OPT_SWAP,	MON$M_SWAP, OPT_DATA),
	MACRO("users",	 OPT_USER,	MON$M_USER, OPT_DATA),
	MACRO("netif",	 OPT_NETIF,	MON$M_NETIF, OPT_DATA),
	MACRO("memory",	 OPT_MEMORY,	MON$M_MEMORY, OPT_DATA),
	MACRO("loadave", OPT_LOADAVE,	MON$M_LOADAVE, OPT_DATA),
	MACRO("io", 	 OPT_IO,	MON$M_IO, OPT_DATA),
	MACRO("mon", 	 OPT_MON,	MON$M_MON, OPT_DATA),
	MACRO("all",	 OPT_ALL,	MON$M_ALL, OPT_DATA),
	MACRO("pid",	 OPT_PID,	MON$M_PID, OPT_DATA),
	MACRO("os",	 OPT_OS,	MON$M_OS, OPT_DATA),
	MACRO("buf",	 OPT_BUF,	MON$M_BUF, OPT_DATA),
	MACRO("nfs",	 OPT_NFS,	MON$M_NFS, OPT_RFFU),
	MACRO("namei",	 OPT_NAMEI,	MON$M_NAMEI, OPT_DATA),
	MACRO("kmem",	 OPT_KMEM,	MON$M_KMEM, OPT_DATA),
/*
 *	Command line options (40-...).
 */
	MACRO("-file",	  OPT_FILE,	OPT_NEXTWORD,	OPT_MINUS),
	MACRO("-total",	  OPT_TOTAL,	0,		OPT_MINUS),
	MACRO("-output",  OPT_OUTPUT,	OPT_NEXTWORD,	OPT_MINUS),
	MACRO("-sleep",	  OPT_SLEEP,	OPT_NEXTWORD,	OPT_MINUS),
#ifdef	DEBUG
	MACRO("-debug",	  OPT_DEBUG,	0,		OPT_MINUS),
#endif
	MACRO("-prompt",  OPT_PROMPT,	OPT_NEXTWORD,	OPT_MINUS),
	MACRO("-help",	  OPT_HELP,	0,		OPT_MINUS),
	MACRO("-version", OPT_VERSION,	0,		OPT_MINUS),
	MACRO("-magnify", OPT_MAGNIFY,  OPT_NEXTWORD,	OPT_MINUS),
	MACRO("-sample",  OPT_SAMPLE,	0,		OPT_MINUS),
	MACRO("-kernel",  OPT_KERNEL,   OPT_NEXTWORD,   OPT_MINUS),
};

/*
 *	Conditionally compile in the debugging code.  If the code is
 *	compiled in, debugging messages will be printed if the "-d"
 *	option is used on the command line.
 */
#ifdef	DEBUG
int	debug = 0 ;
#endif

/*
 *	A flag which can be used to disable the ability to set
 *	the data file and screen dump names from the command line.
 *
 *	It needs to somewhere that abd can get to it.  In a object
 *	code distribution this would live in a data file that could
 *	be configured as necessary.
 */
int	secure = 1 ;

/*
 *	Various local data structures.
 *
 *	module - The module name passed to the error message functions.
 *	set - A bit field used to keep track of what options have been set.
 */
static char *module = "options" ;
static struct set_bits set ;

/*
 *	The "default" display functions.  These will be used in replay
 *	to remember what the default display mode was.
 */
int	(*def_display)(),
	(*def_redraw)(),
	(*def_close)(),
	(*def_open)(),
	(*def_help)() ;

/*
 *	A macro for getting the number of elements in a static
 *	table.
 */
#define	SIZE(x)	(sizeof(x)/sizeof(x[0]))

/*
 *	The various available functions.  First the "display"
 *	functions.
 */
int	screen(), open_screen(), close_screen(), redraw_screen(), help_screen() ;
int	open_read(), close_read(), display_read() ;
int	save(), open_save(), close_save() ;

struct display_table display_table[] = {
	OPT_SCREEN, 1, 2,
	screen, open_screen, close_screen, redraw_screen, help_screen,

	OPT_READ, 0, 2,
	display_read, open_read, close_read, NULL, NULL,

	OPT_SAVE, 0, 600,
	save, open_save, close_save, NULL, NULL,

	OPT_NOP, 0, 2,
	NULL, NULL, NULL, NULL, NULL,
};

/*
 *	Next the data collection functions.
 */
int	replay(), open_replay(), close_replay() ;
int	names(), open_names(), close_names() ;
int	live(), open_live(), close_live() ;

struct collect_table collect_table[] = {
	OPT_LIVE,   live,         open_live,   close_live,
	OPT_REPLAY, replay,       open_replay, close_replay,
	OPT_NAMES,  names,        open_names,  close_names,
};

/*
 *	Look at the argument vector to set the various options.  Default
 *	options will also be set here.
 */
parse(argc, argv, op)
int	argc ;
char	**argv ;
OPTION	*op ;
{
	int	i, mcc ;
	char	*next_word, *value, *strchr() ;

	/*
	 *	Initialize the options structure.
	 */
	init_options(op);

	/*
	 *	Do any special command name processing.
	 */
	command_name(argv[0], op) ;

	/*
	 *	Loop through the arguments.
	 */
	for(i = 1; i < argc; i++) {
		/*
		 *	See if the argument has an '=' in it and remember
		 *	the right part of the equal.
		 */
		if((value = strchr(argv[i], '=')) != NULL ) {
			*value = '\0' ;
			value++ ;
		}

		/*
		 *	Look in the option table to see if the argument
		 *	matches.  Return a "magick cookie code" which is
		 *	index of the table entry.
		 */
		mcc = search_options(argv[i]) ;

		switch( array[mcc].opt_type ) {
		case OPT_DISPLAY:
			op->display_mode = array[mcc].opt_code ;
			set.display = 1 ;
			break ;
		case OPT_COLLECT:
			op->collect_mode = array[mcc].opt_code ;
			set.collect = 1 ;
			break ;
		case OPT_DATA:
			data_option(op, mcc, value) ;
			break ;
		case OPT_MINUS:
			/*
			 *	Get the next argument if one is there.
			 */
			next_word = (i + 1) == argc ? NULL : argv[i+1] ;
			i += minus_option(op, mcc, next_word) ;
			break ;
		case OPT_RFFU:
			info("Reserved for future use: %s.\n", module, argv[i]) ;
			break ;
		case OPT_NOSUP:
		default:
			info("Unsupported option: %s.\n", module, argv[i]);
			exit(-1) ;
			break ;
		}
	}

	/*
	 *	verify that the various combinations of options are
	 *	reasonable and change those that aren't.
	 */
	verify_options(op);

	/*
	 *	Setup the functions based on the display and collection
	 *	modes chosen.
	 */
	set_collect(op) ;
	set_display(op) ;

#ifdef	DEBUG
	if( debug )
		print_option(op) ;
#endif

	def_display = op->display ;
	def_redraw = op->redraw_display ;
	def_close = op->close_display ;
	def_open = op->open_display ;
	def_help = op->help_display ;
}

/*
 *	Verify that the options are set reasonably.  The first pass of 
 *	verify_options() will set the options according to the various 
 *	relationships between the display mode and collection mode.
 */
verify_options(op)
OPTION	*op ;
{
	/*
	 *	Anytime OPT_NAMES is used as a collection option
	 *	then set the display mode op OPT_NOP and return.
	 *	The set_display() and set_collect() will ensure
	 *	that everything is setup correctly.
	 */
	if( op->collect_mode == OPT_NAMES ) {
		op->display_mode = OPT_NOP ;
		set.collect = 1 ;
		set.display = 1 ;

		return ;
	}

	/*
	 *	If both the MEMORY and FREE data options are turned on
	 *	turn off FREE.
	 */
	if( op->opt_memory && op->opt_free )
		op->opt_free = 0 ;

	/*
	 *   If the display mode is SAVE...
	 *
	 *	If the collection mode is not set, then use LIVE.
	 *
	 *	or if the collection mode is REPLAY change the display 
	 *	mode to SCREEN.
	 */
	if( op->display_mode == OPT_SAVE ) {
		if( set.collect == 0 ) {
			op->collect_mode = OPT_LIVE ;
			set.collect = 1 ;
		}
		else if( op->collect_mode == OPT_REPLAY ) {
			info("Display mode changed to SCREEN.\n", module) ;
			op->display_mode = OPT_SCREEN ;
			set.display = 1 ;
		}
		else
			;	/* don't care */
	}

	/*
	 *	If the display mode hasn't been set, use SAVE if the collection
	 *	mode is LIVE and has been set and a data file has been set.
 	 *	Otherwise use SCREEN.
	 *
	 *	Since the default collection mode is LIVE we only want to
	 *	set the display mode to SAVE if a data file has been set AND
	 *	LIVE was selected on the command line.
	 */
	if( set.display == 0 ) {
		if( op->collect_mode == OPT_LIVE && set.collect && set.file ) {
			op->display_mode = OPT_SAVE ;
			set.display = 1 ;
		}
		else {
			op->display_mode = OPT_SCREEN ;
			set.display = 1 ;
		}
	}

	/*
	 *	Whenever a collection mode hasn't been set, use REPLAY if
	 *	a data file has been specified, otherwise use LIVE as the
	 *	collection mode.
	 */
	if( set.collect == 0 ) {
		set.collect = 1 ;

		if( set.file )
			op->collect_mode = OPT_REPLAY ;
		else
			op->collect_mode = OPT_LIVE ;
	}

	/*
	 *	If we haven't chosen a data option turn on a default.
	 */
	if( op->collect_mode == OPT_LIVE && set.data == 0 ) {
		set.data = 1 ;
		op->opt_data |= MON$M_ALL ;
	}
}

/*
 *	Based on the collection and display modes set the corresponding
 *	functions.  Panic if we see an unexpected collection or display
 *	mode.
 */
set_display(op)
OPTION	*op ;
{
	int	i ;

	for(i = 0; i < SIZE(display_table); i++) {
		if( display_table[i].code == op->display_mode ) {
			op->display = display_table[i].display ;
			op->help_display = display_table[i].help ;
			op->open_display = display_table[i].open ;
			op->close_display = display_table[i].close ;
			op->redraw_display = display_table[i].redraw ;

			op->opt_inter = display_table[i].interactive ;

			if( !op->sleep_set )
				op->opt_sleep = display_table[i].sleep ;

			return ;
		}
	}

	panic("Unexpected display mode!\n", module) ;
}

/*
 *	Based on the collection and display modes set the corresponding
 *	functions.  Panic if we see an unexpected collection or display
 *	mode.
 */
set_collect(op)
OPTION	*op ;
{
	int	i ;

	for(i = 0; i < SIZE(collect_table); i++) {
		if( collect_table[i].code == op->collect_mode ) {
			op->collect = collect_table[i].collect ;
			op->open_collect = collect_table[i].open ;
			op->close_collect = collect_table[i].close ;

			break ;
		}
	}

	/*
	 *	Do some option specific processing.
	 */
	switch( op->collect_mode ) {
	case OPT_LIVE:
		break ;
	case OPT_REPLAY:
		/*
		 *	When the display mode is OPT_READ set the
		 *	sleep time to zero (0) unless the user has
		 *	set the time from the command line.
		 */
		if( !op->sleep_set && op->display_mode == OPT_READ ) {
			op->opt_sleep = 0 ;
			op->sleep_set = 1 ;
		}

		break ; 
	case OPT_NAMES:
		/*
		 *	Recognize that is a valid collection otherwise
		 *	it would panic.
		 */
		break ;
	default:
		panic("Unexpected collection mode!\n", module) ;
	}
}

/*
 *	Initialize the options structures.
 */
init_options(op)
OPTION	*op ;
{
	op->opt_data = 0 ;
	op->opt_sample = 1 ;

	op->opt_flag = 0 ;

	op->opt_file = "monitor.dat" ;
	op->opt_output = "screen.dump" ;
	op->opt_prompt = "monitor> " ;
	op->opt_kernel = "/vmunix" ;
	op->opt_magnify = 0 ;

	op->display_mode = OPT_SCREEN;
	op->collect_mode = OPT_LIVE ;

	set.display = 0 ;
	set.collect = 0 ;
	set.file = 0 ;
	set.data = 0 ;
}

/*
 *	Process "-" preceded options.  The value returned by this
 *	function is the number of command line arguments to skip.
 */
minus_option(op, mcc, next_word)
int	mcc ;
char	*next_word ;
OPTION	*op ;
{
	int	skip = 0 ;
	char	*value, *strchr() ;
	double	atof(), sleep_time ;

	if((array[mcc].opt_bit & OPT_NEXTWORD) && next_word == NULL ) {
		info("This option requires an additional argument: %s.\n",
			module, array[mcc].opt_string) ;
		return skip ;
	}

	if( array[mcc].opt_bit & OPT_NEXTWORD )
		skip = 1 ;

	switch( array[mcc].opt_code ) {
#ifdef	DEBUG
	case OPT_DEBUG:
		debug++ ;
		break ;
#endif
	case OPT_SLEEP:				/* sleep */
		sleep_time = atof(next_word) ;

		op->opt_sleep = (int)sleep_time ;
		op->opt_fract = sleep_time - op->opt_sleep ;
		op->sleep_set = 1 ;

		break ;
	case OPT_FILE:				/* data file */
		if( secure ) {
			op->opt_file = next_word ;

			if( strcmp(next_word, "-") == 0 )
				op->opt_standard = 1 ;

			set.file = 1 ;
		}
		else {
			info("This option isn't secure: %s.\n", module, "-file") ;
			info("Using default: %s.\n", module, op->opt_file) ;
		}

		break ;
	case OPT_OUTPUT:			/* screen/print dump file */
		if( secure )
			op->opt_output = next_word ;
		else {
			info("This option isn't secure: %s.\n", module, "-output") ;
			info("Using default: %s.\n", module, op->opt_output) ;
		}

		break ;
	case OPT_KERNEL:
		op->opt_kernel = next_word ;
		break ;
	case OPT_PROMPT:			/* prompt */
		op->opt_prompt = next_word ;
		break ; 
	case OPT_HELP:				/* help */
		help();
		exit(0);
	case OPT_VERSION:			/* version string */
		show_version() ;
		exit(0) ;
	case OPT_TOTAL:				/* turn on disk totals */
		op->opt_total = 1 ;
		break ;
	case OPT_SAMPLE:			/* magnify sample record */
		op->opt_mag_sample = 1 ;
		break ;
	case OPT_MAGNIFY:			/* use a magnify function */
		if((value = strchr(next_word, '=')) != NULL ) {
			*value = '\0' ;
			value++ ;
		}

		mcc = search_options(next_word) ;

		if( mcc == -1 || array[mcc].opt_type == OPT_RFFU )
			info("Data option not supported.\n", module) ; 
		else {
			op->opt_magnify |= array[mcc].opt_bit ;

			data_option(op, mcc, value) ;
		}

		break ;
	default:
		panic("Unexpected option!\n", module) ;
		break ;
	}

	return skip ;
}

/*
 *	Do any special processing needed for data options.
 */
data_option(op, mcc, value)
OPTION	*op ;
int	mcc ;
char	*value ;
{
	struct list *list = 0 ;

	switch( array[mcc].opt_code ) {
	case OPT_CPU:
		list = &cpu_list ;
		break ;
	case OPT_DISK:
		list = &disk_list ;
		break ;
	case OPT_NETIF:
		list = &netif_list ;
		break ;
	case OPT_PID:
		op->opt_proc_id = atoi(value) ;
		break ;
	default:
		break ;
	}

	if( list )
		build_value(value, (struct list *)list) ;

	set.data = 1 ;

	op->opt_data |= array[mcc].opt_bit ;
}

/*
 *	Search "array" for an option that matches word.  Abbreviations
 *	of option string are recognized.  If there is more than one
 *	one match then "word" is considered ambiguous and a message
 *	is printed.  If "word" is longer than option string the match
 *	will fail.
 *
 *	This function returns the index of the entry or -1.
 */
search_options(word)
char	*word ;
{
	register i, length, found = -1, n_found = 0 ;

	length = strlen(word) ;

	for(i = 0; i < sizeof(array)/sizeof(array[0]); i++) {
		if( length > array[i].opt_length )
			continue ;

		if( strncmp(word, array[i].opt_string, length) == 0 ) {
			n_found++ ;
			found = i ;
		}
	}

	if( n_found > 1 ) {
		info("The argument \"%s\" is ambiguous.  Matching options are:\n", module, word);
		for(i = 0; i < sizeof(array)/sizeof(array[0]); i++)
			if( strncmp(word, array[i].opt_string, length) == 0 )
				fprintf(stderr, "\t%s\n", array[i].opt_string) ;

		return -1 ;
	}
	else
		return found ;
}

/*
 *	Restore the display to their defaults.
 */
restore_display(op)
OPTION	*op ;
{
	op->redraw_display = def_redraw ;
	op->close_display = def_close ;
	op->open_display = def_open ;
	op->help_display = def_help ;
	op->display = def_display ;
}

#ifdef	DEBUG

char	*str_data(),
	*str_opt_flags() ;

/*
 *	Translate the integer value of a collection or display
 *	mode into the corresponding string.
 */
char	*search_name(mode)
int	mode ;
{
	register i ;

	for(i = 0; i < sizeof(array)/sizeof(array[0]) ; i++)
		if( mode == array[i].opt_code )
			return array[i].opt_string ;

	return "(null)" ;
}

/*
 *	Print the contents of the option structure for debugging.
 */
print_option(op)
OPTION	*op ;
{
	fprintf(stderr, "Contents of option structure:\n") ;
	fprintf(stderr, "\tdata:    <%s>\n", str_data(op->opt_data)) ;
	fprintf(stderr, "\tflags:   <%s>\n", str_opt_flags(op->opt_flag)) ;
	fprintf(stderr, "\tdisplay: %s\n", search_name(op->display_mode));
	fprintf(stderr, "\tcollect: %s\n", search_name(op->collect_mode));
	fprintf(stderr, "\tproc_id: %d\n", op->opt_proc_id) ;
	fprintf(stderr, "\tsleep:   %d\n", op->opt_sleep) ;
	fprintf(stderr, "\tfile:    %s\n", op->opt_file) ;
	fprintf(stderr, "\tprompt:  %s\n", op->opt_prompt) ;
	fprintf(stderr, "\toutput:  %s\n", op->opt_output) ;
	fprintf(stderr, "\tmagnify: <%s>\n", str_data(op->opt_magnify)) ;
}
#endif

extern	char	*program_name ;

/*
 *	Eventually this will do command name processing.  For
 *	example if "monitor" is invoked as "names" the names 
 *	option will be selected automatically. 	Currently this
 *	function only sets the program name for the error messages.
 *
 *	ARGSUSED
 */
command_name(s, op)
char	*s ;
OPTION	*op ;
{
	program_name = s ;
}
