/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 netif.c
 *	Date:	 3/29/90
 *	Version: 1.39
 *
 *	netif.c - Read network interface device names from kernal.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)netif.c	1.39 (monitor) 3/29/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 *
 * Feb. 15, 1989 -- arr
 *
 *	Correctly set mon_flag with MON$M_VALID.
 *
 * Week ending Feb. 25, 1989 -- arr
 *
 *	Reformatted netif magnification screen.  Also moved f_netif()
 *	from functions.c to netif.c.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <curses.h>

#include <sys/types.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#include <net/if.h>
#include <netinet/in.h>

#include "include.h"
#include "monitor.h"
#include "options.h"
#include "extern.h"

/*
 *	Module name for error functions.
 */
static	char	*module = "netif" ;

/*
 *	Functions that don't return (int).
 */
char	*ctime(),
	*calloc(),
	*strcpy(),
	*sprintf(),
	*strrchr(),
	*str_netif() ;

/*
 *	Default display.
 */
extern	int	lines ;			/* length of screen being used. */
extern	int	cols ;			/* width of screen being used */
extern	WINDOW	*wp ;

/*
 *	Walk through the list of network interfaces and setup
 *	the array for monitor to use.
 */
net_devices()
{
	struct	ifnet		ifnet ;
	struct	mon_netif	*ifp ;
	char	device[BUFSIZ] ;
	off_t	firstif, ip ;
	unsigned n = 0 ; 

	/*
	 *	Make sure there is a list of network
	 *	interfaces to read and if there is
	 *	read the address of the first one.
	 */
	if((ip = (off_t)namelist[NM_IFNET].n_value) == 0 )
		return ;

	readk((long)ip, (char *)&firstif, sizeof(firstif));

	/*
	 *	Make a quick pass of the list to count the
	 *	number of interfaces and allocate space for
	 *	mon_netif structures.
	 */
	for(ip = firstif; ip; ip = (off_t)ifnet.if_next ) {
		readk((long)ip, (char *)&ifnet, sizeof(ifnet));
		n++ ;
	}

	if((netif = (struct mon_netif *)calloc(n, MON$S_NETIF)) == NULL )
		fatal("Can't allocate space for netif structures: %s.\n", module);

	for(ip = firstif; ip; ip = (off_t)ifnet.if_next ) {
		/*
		 *	Read the ifnet structure and the interface
		 *	name.  Using the name and unit number see if
		 *	this is an "interesting" interface.
		 */
		readk((long)ip, (char *)&ifnet, sizeof(ifnet));

		readk((long)ifnet.if_name, (char *)device, IFNAMSIZ);

		if( find_value(&netif_list, device, ifnet.if_unit) == 0 )
			continue ;

		ifp = netif + n_netif ;

		/*
		 *	Initialize static data.
		 */
		ifp->mon_type   = MON$C_NETIF ;
		ifp->mon_flag   = MON$M_VALID ;
		ifp->mon_length = MON$S_NETIF ;
		ifp->mon_ifnet  = (struct ifnet *)ip ;
		ifp->mon_unit   = ifnet.if_unit ;
		ifp->mon_index  = ifp->mon_if = n_netif++ ;

		(void)strcpy(ifp->mon_name, device);
		ifp->mon_name[IFNAMSIZ] = '\0' ;

		/*
		 *	Initialize the first pass of dynamic data.
		 */
		ifp->mon_collisions = ifnet.if_collisions ;
		ifp->mon_ipackets = ifnet.if_ipackets ;
		ifp->mon_opackets = ifnet.if_opackets ;
		ifp->mon_ierrors = ifnet.if_ierrors ;
		ifp->mon_oerrors = ifnet.if_oerrors ;
	}
}

#define	NETIF_OFFSET	(12)
#define	NETIF_X		(15)
#define	NETIF_Y		(3)

#define	NETIF_BOOT	(NETIF_Y+4)
#define	NETIF_SAMPLE	(NETIF_Y+12)

#define	NETIF_LINES 	(21)
#define	NETIF_WIDTH	(12)

/*
 *	Setup for netif magnify function. 
 *
 *	ARGSUSED
 */
open_netif(op)
OPTION	*op ;
{
	lines = NETIF_LINES ;

	sample_header() ;

	wmove(wp, NETIF_Y, 0) ;
	wprintw(wp, "Interface\n\nSince boot (packets/sec.):\n\n") ;
	wprintw(wp, "Input\nInput errors\nOutput\nOutput errors\nCollisions\n\n") ;
	wprintw(wp, "Sample rate (packets/sec.):\n\n") ;
	wprintw(wp, "Input\nInput errors\nOutput\nOutput errors\nCollisions") ;
}

/* 
 *	Network interface magnify function.
 *
 *	ARGSUSED
 */
magnify_netif(op)
OPTION	*op ;
{
	register i, x = NETIF_X, limit ;
	char	 name[DEV_SIZE+11] ;
	double	 delta_netif(), etime ;

	limit = (cols - (NETIF_X + 2)) / NETIF_WIDTH ;

	for(i = 0; i < n_netif && limit ; i++) {
		if((netif[i].mon_flag & MON$M_VALID) == 0 )
			continue ;
		else
			limit-- ;

		/*
		 *	Build a string for the name.
		 */
		(void)sprintf(name, "%s%d", netif[i].mon_name,
			netif[i].mon_unit) ;

		wmove(wp, NETIF_Y, x) ;
		wprintw(wp, "  %10s", name);

		/*
		 *	The elapsed time since boot.
		 */
		etime = sample.mon_ticks / first.mon_hz ;

		/*
		 *	This is where all the "since boot" rates will
		 *	be printed.
		 */
		wmove(wp, NETIF_BOOT, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_ipackets / etime) ;

		wmove(wp, NETIF_BOOT+1, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_ierrors / etime) ;

		wmove(wp, NETIF_BOOT+2, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_opackets / etime) ;

		wmove(wp, NETIF_BOOT+3, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_oerrors / etime) ;

		wmove(wp, NETIF_BOOT+4, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_collisions / etime) ;

		/*
		 *	The elasped time since the last sample.
		 */
		etime = delta_netif(netif[i].mon_index, sample.mon_ticks);

		/*
		 *	Then do all the sample rates.
		 */
		wmove(wp, NETIF_SAMPLE, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_ipackets / etime) ;

		wmove(wp, NETIF_SAMPLE+1, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_ierrors / etime) ;

		wmove(wp, NETIF_SAMPLE+2, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_opackets / etime) ;

		wmove(wp, NETIF_SAMPLE+3, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_oerrors / etime) ;

		wmove(wp, NETIF_SAMPLE+4, x) ;
		wprintw(wp, "  %10.2f", netif[i].mon_collisions / etime) ;

		x += NETIF_OFFSET ;
	}

	sample_body(etime) ;
}

/*
 *	Function to dump a NETIF record.
 */
f_netif(p)
struct mon_netif *p ;
{
	if((p->mon_flag & MON$M_VALID) == 0 )
		return ;

	printf("%s.\n", records[MON$C_NETIF].string) ;

	printf("\tname:       %s%d\n", p->mon_name, p->mon_unit);
	printf("\tif_flags:   <%s>\n", str_netif(p->mon_if_flags));
	printf("\tipackets:   %d\n", p->mon_ipackets);
	printf("\tierrors:    %d\n", p->mon_ierrors);
	printf("\topackets:   %d\n", p->mon_opackets);
	printf("\toerrors:    %d\n", p->mon_oerrors);
	printf("\tcollisions: %d\n", p->mon_collisions);
	printf("\taddr:       %x\n", p->mon_ifnet);
}
