/*
 *	Author:  Alan Rollow, EIS/CXO, Digital Equipment Corp.
 *	File:	 names.c
 *	Date:	 8/21/90
 *	Version: 1.23
 *
 *	names.c - Functions to read kernel names.
 *
 *	Some of these functions duplicate code that is elsewhere in
 *	monitor, but because modification of the original function
 *	would make it unwieldy and look like it had been hacked,
 *	I have rewritten the code here.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)names.c	1.23 (monitor) 8/21/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 * 22-Sept-1988 -- arr
 *
 *	Fix bogus use of sizeof(DEV_SIZE) in MASSBUS code.
 *
 * 29-November-1988 -- arr
 *
 *	Add #ifdef's for MIPS version.
 *
 * 27-December-1988 -- arr
 *
 *	Added code to give more detail about MIPS based systems.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>
#include <sys/config.h>
#include <sys/cpudata.h>

#ifdef	V4_ULTRIX
#	include <sys/namei.h>
#endif
#ifdef	mips
#	include <sys/fixpoint.h>
#endif

#include <net/if.h>
#include <netinet/in.h>

#include <machine/pte.h>
#include <machine/param.h>

#ifdef	V4_ULTRIX
#	ifdef	vax
#		include <io/mba/vax/mbavar.h>
#		include <io/mba/vax/mbareg.h>
#		include <io/uba/ubavar.h>
#	elif	mips
#		include <io/uba/ubavar.h>
#		include <machine/cpuconf.h>
#		include "mbavar.h"
#		include "mbareg.h"
#	endif
#else
#	ifdef	vax
#		include <vaxmba/mbavar.h>
#		include <vaxmba/mbareg.h>
#		include <vaxuba/ubavar.h>
#	elif	mips
#		include <machine/cpuconf.h>
#		include "mbavar.h"
#		include "mbareg.h"
#		include "ubavar.h"
#	endif
#endif

#include "pgtok.h"
#include "include.h"
#include "options.h"

#ifndef	MAXCPU
#	define	MAXCPU	(32)
#endif

/*
 *	This is the set of kernel symbol used just by the
 *	names code.  By only using the ones needed the nlist(3)
 *	call should be faster.
 */
#define	NAMES_ADPT		(0)
#define	NAMES_BOOT		(1)
#define	NAMES_CPU		(2)
#define	NAMES_CPUDATA		(3)
#define	NAMES_CPU_SUBTYPE	(4)
#define	NAMES_CPU_SYSTYPE	(5)
#define	NAMES_DK_XFER		(6)
#define	NAMES_IFNET		(7)
#define	NAMES_MAXCPU		(8)
#define	NAMES_MBDINIT		(9)
#define	NAMES_MBSINIT		(10)
#define	NAMES_PHYSMEM		(11)
#define	NAMES_UBDINIT		(12)

struct nlist names_nm[] = {
	{ "_config_adpt" },
	{ "_boottime" },
	{ "_cpu" },
	{ "_cpudata" },
	{ "_cpu_subtype" },
	{ "_cpu_systype" },
	{ "_dk_xfer" },
	{ "_ifnet" },
	{ "_maxcpu" },
	{ "_mbdinit" },
	{ "_mbsinit" },
	{ "_physmem" },
	{ "_ubdinit" },
	{ 0 }
} ;


/*
 *	Data objects declared elsewhere, but used locally.
 */
extern	int	kmem ;
extern  long	dk_xfer[DK_NDRIVE] ;
extern	char	dk_map[DK_NDRIVE] ;

/*
 *	Taken from <sys/devio.h>.
 */
#define	DEV_SIZE	8

/*
 *	Alias the names of the CPUdata structure members.
 */
#ifdef	V4_ULTRIX
#	define	c_state		cpu_state
#	define	c_cptime	cpu_cptime
#	define	c_ident		cpu_num
#endif

/*
 *	Functions which don't return (int).
 */
char	*str_cpuid(),
	*str_netif(),
	*str_state(),
	*strcpy(),
	*calloc() ;

void	nlist(),
	free() ;

/*
 *	Module name for error functions.
 */
static	char	*module = "names" ;

/*
 *	Get the configuration information from the expect places.
 *	Someday we may need the option structure.
 *
 *	ARGSUSED
 */
names(op)
OPTION	*op ;
{
	register i ;

	/*
	 *	Clear the dk_map[].
	 */
	for(i = 0; i < DK_NDRIVE; i++)
		dk_map[i] = 0 ;

	/*
	 *	Print out information about the CPU(s).
	 */
	cpu_names() ;

	/*
	 *	Devices that are UNIBUS style.  This includes
	 *	all sorts of stuff that don't even get close to
	 *	a UNIBUS, but the name is historical.
	 */
	uba_names() ;

	/*
	 *	Get the names of MASSBUS devices.
	 */
	mba_names() ;

	/*
	 *	We'll play tricks with the _dk_xxx data structures
	 *	to get names of missing disks.
	 */
	scsi_names() ;

	/*
	 *	Print the names and stats on network interfaces.
	 */
	netif_names() ;

	return MON_EXIT ;
}

/*
 *	Set up the names_nm and open the kernel so we can get
 *	to the names.  Someday we may need the option structure.
 *
 *	ARGSUSED
 */
open_names(op)
OPTION	*op ;
{
	/*
	 *	Initialize the names_nm, nlist(x) doesn't return
	 *	anything to signify an error.
	 */
	(void)nlist(op->opt_kernel, names_nm) ;

	/*
	 *	Open memory.
	 */
	if((kmem = open("/dev/kmem", O_RDONLY, 0)) == -1 )
		fatal("Can't open /dev/kmem: %s.\n", module) ;
}

/*
 *	Try to close /dev/kmem.  I hope it works :-)
 *	Someday we may need the option structure.
 *
 *	ARGSUSED
 */
close_names(op)
{
	if( close(kmem) == -1 )
		fatal("Can't close /dev/kmem: %s.\n", module) ;
}

/*
 *	Print something about the CPU.  This is for pre V2.0 systems.
 */
static	cpu_names()
{
	int	sid, physmem, mon_maxcpu ;
	long	subtype = -1, sid_addr ;
	caddr_t	addr ;
	struct timeval boot ;

#ifdef	vax
	sid_addr = (long)names_nm[NAMES_CPU].n_value ;
#elif	mips
	sid_addr = (long)names_nm[NAMES_CPU_SYSTYPE].n_value ;
#endif

	sid = getsid(sid_addr) ;

	/*
	 *	Get the amount of physical memory.
	 */
	if((addr = (caddr_t)names_nm[NAMES_PHYSMEM].n_value) == 0 ) {
		warning("Can't get physical memory size.\n", module) ;
		return ;
	}
	else
		readk((long)addr, (char *)&physmem, sizeof(physmem)) ;

	/*
	 *	Get the boot time.
	 */
	if((addr = (caddr_t)names_nm[NAMES_BOOT].n_value) == 0 ) {
		warning("Can't get boot time.\n", module) ;
		return ;
	}
	else
		readk((long)addr, (char *)&boot, sizeof(boot)) ;

#ifdef	vax
	/*
	 *	Get the CPU subtype.
	 */
	if((addr = (caddr_t)names_nm[NAMES_CPU_SUBTYPE].n_value) == 0 )
		subtype = -1 ;
	else
		readk((long)addr, (char *)&subtype, sizeof(long)) ;
#elif	mips
	subtype = -1 ;
#endif

	printf("#\n") ;
	printf("#\tCPU information.\n") ;
	printf("#\n") ;

	printf("%s with %d bytes of memory.\n", str_cpuid(sid, subtype),
		physmem * NBPG) ;
	printf("The system was booted on %24.24s\n", ctime((long *)&boot.tv_sec)) ;
	uptime(boot.tv_sec) ;

	/*
	 *	Find out how many CPU's there should be.
	 */
	if( names_nm[NAMES_MAXCPU].n_value == 0 )
		mon_maxcpu = MAXCPU ;
	else
		mon_maxcpu = get_word((long)names_nm[NAMES_MAXCPU].n_value) ;

	print_cpu_data(mon_maxcpu) ;
}

/*
 *	Print something about each cpu.  I'd like to see state, ident and
 *	and time in CPU states.
 */
print_cpu_data(mon_maxcpu)
int	mon_maxcpu ;
{
	struct cpudata mon_cpudata, *cpuaddr, *array[MAXCPU] ;
	int	i, state ;
	double	total ;

	/*
	 *	Initialize the cpudata array.
	 */
	for(i = 0; i < MAXCPU; i++)
		array[i] = 0 ;

	/*
	 *	Get the starting point of cpudata structures.  In
	 *	V4.0 this is an array of addresses, before that
	 *	it is the address of the first structure.
	 */
	if((cpuaddr = (struct cpudata *)names_nm[NAMES_CPUDATA].n_value) == 0 ) {
		warning("The CPU data is not available.\n", module) ;
		return ;
	}

	/*
	 *	Build an array of the addresses.
	 */
#ifdef	V4_ULTRIX
	readk((long)cpuaddr, array, sizeof(array[0]) * MAXCPU) ;
#else
	for(i = 0; i < mon_maxcpu; i++)
		array[i] = cpuaddr + i ;
#endif

	/*
	 *	Then for each member in the array that has a non-
	 *	zero address, print information about that CPU.
	 */
	for(i = 0; i < MAXCPU; i++) {
		if( array[i] == 0 )
			continue ;

		readk((long)array[i], (char *)&mon_cpudata,
			sizeof(mon_cpudata)) ;

	/*
	 *	The compiled for the MIPS set the CPU state to "run".  
	 *	Hopefully this will be fixed in a future release.
	 */
#ifdef	mips
		mon_cpudata.c_state = CPU_RUN ;
#endif
		for(state = 0, total = 0; state < CPUSTATES; state++)
			total += mon_cpudata.c_cptime[state] ;

		if( total == 0 )
			total = 1 ;

		printf("CPU #%d (%x): %.0f%% user, %.0f%% nice, %.0f%% system, %.0f%% idle; state: <%s>\n",
			i, mon_cpudata.c_ident,
			100. * mon_cpudata.c_cptime[CP_USER] / total,
			100. * mon_cpudata.c_cptime[CP_NICE] / total,
			100. * mon_cpudata.c_cptime[CP_SYS]  / total,
			100. * mon_cpudata.c_cptime[CP_IDLE] / total,
			str_state(mon_cpudata.c_state)) ;
	}
}

/*
 *	Print out the MASSBUS device and slave names.
 */
static	mba_names()
{
	struct mba_device device, *md ;
	struct mba_slave  slave,  *ms ;
	struct mba_driver driver ;
	char	dname[DEV_SIZE+1], sname[DEV_SIZE+1] ;
	int	header_printed = 0 ;

	dname[DEV_SIZE] = '\0' ;
	sname[DEV_SIZE] = '\0' ;

	/*
	 *	Make sure this is a reasonable place to start.
	 */
	if((md = (struct mba_device *)names_nm[NAMES_MBDINIT].n_value) == 0 )
		return ;


	for( ; ; ) {
		/*
		 *	Read the device structure.
		 */
		readk((long)md, (char *)&device, sizeof(device)) ;

		/*
		 *	If this is the last one in the list, stop.
		 */
		if( device.mi_driver == 0 )
			break ;

		if( !header_printed ) {
			printf("#\n#\tMASSBUS information.\n#\n") ;
			header_printed = 1 ;
		}

		/*
		 *	Read the driver structure and the device name.
		 */
		readk((long)device.mi_driver, (char *)&driver, sizeof(driver)) ;

		if( driver.md_dname == 0 )
			(void)strcpy(dname, "master") ;
		else
			readk((long)driver.md_dname, (char *)dname, DEV_SIZE) ;

		/*
		 *	Print the information out and go to the next
		 *	next device structure.
		 */
		printf("%s%d on mba%d drive %d %s alive.\n",
			dname, device.mi_unit,
			device.mi_mbanum, device.mi_drive,
			device.mi_alive ? "is" : "is not") ;

		/*
		 *	Mark this device in the dk_map[].
		 */
		if( device.mi_dk >= 0 && device.mi_dk < DK_NDRIVE )
			dk_map[device.mi_dk] = 1 ;

		md++ ;
	}

	/*
	 *	See if the beginning of the slave list is a reasonable
	 *	place to start.
	 */
	if((ms = (struct mba_slave *)names_nm[NAMES_MBSINIT].n_value) == 0 )
		return ;

	for( ; ; ) {
		/*
		 *	Read the slave structure.
		 */
		readk((long)ms, (char *)&slave, sizeof(slave)) ;

		/*
		 *	If this is the last one, stop.
		 */
		if( slave.ms_driver == 0 )
			break ;

		/*
		 *	Read the driver structure, device name and
		 *	slave name.
		 */
		readk((long)slave.ms_driver, (char *)&driver, sizeof(driver)) ;

		if( driver.md_dname == 0 )
			(void)strcpy(dname, "master") ;
		else
			readk((long)driver.md_dname, (char *)dname, DEV_SIZE) ;

		if( driver.md_sname == 0 )
			(void)strcpy(sname, "slave") ;
		else
			readk((long)driver.md_sname, (char *)sname, DEV_SIZE) ;

		/*
		 *	Print the information out and go to the
		 *	next one.
		 */
		printf("%s%d on %s%d slave %d %s alive.\n",
			sname, slave.ms_slave,
			dname, slave.ms_ctlr, slave.ms_slave,
			slave.ms_alive ? "is" : "is not") ;

		ms++ ;
	}
}

/*
 *	Print out information about network devices.
 */
static	netif_names()
{
	off_t	firstif, ip ;
	struct ifnet ifnet ;
	char	interface[IFNAMSIZ+1] ;

	/*
	 *	Make sure there is a list...
	 */
	if((ip = (off_t)names_nm[NAMES_IFNET].n_value) == 0 )
		return ;

	interface[IFNAMSIZ] = '\0' ;

	printf("#\n") ;
	printf("#\tNetwork interface information.\n") ;
	printf("#\n") ;

	/*
	 *	Read the address of the first one.
	 */
	readk((long)ip, (char *)&firstif, sizeof(firstif)) ;

	ip = firstif ;

	while( ip ) {
		/*
		 *	Read the ifnet structure and the
		 *	interface name.
		 */
		readk((long)ip, (char *)&ifnet, sizeof(ifnet)) ;

		if( ifnet.if_name == 0 )
			(void)strcpy(interface, "ifnet") ;
		else
			readk((long)ifnet.if_name, (char *)interface, IFNAMSIZ) ;

		/*
		 *	Print something interesting about
		 *	the interface.
		 */
		print_if(interface, &ifnet) ;

		/*
		 *	Goto the next one.
		 */
		ip = (off_t)ifnet.if_next ;
	}
}

/*
 *	Print extended information about network devices.
 */
static	print_if(name, ifp)
char	*name ;
struct ifnet *ifp ;
{
	register total = 0, n ;

	total = ifp->if_ipackets + ifp->if_opackets + 
		ifp->if_ierrors + ifp->if_oerrors + 
		ifp->if_collisions ;

	printf("%s%d is <%s>, with:\n", name, ifp->if_unit,
		str_netif(ifp->if_flags)) ;

	if( n = ifp->if_ipackets )
		printf("%10d (%.0f%%) input packets\n", n, 100.0 * n/total) ;

	if( n = ifp->if_ierrors )
		printf("%10d (%.0f%%) input errors\n", n, 100.0 * n/total) ;

	if( n = ifp->if_opackets )
		printf("%10d (%.0f%%) output packets\n", n, 100.0 * n/total) ;

	if( n = ifp->if_oerrors )
		printf("%10d (%.0f%%) output errors\n", n, 100.0 * n/total) ;

	if( n = ifp->if_collisions )
		printf("%10d (%.0f%%) collisions\n", n, 100.0 * n/total) ;
}

/*
 *	Print information about UNIBUS or Q-BUS devices.
 */
static	uba_names()
{
	struct config_adpt *p, adpt ;
	char	parent[BUFSIZ] ;
	int	i, header_printed = 0 ;

	if((p = (struct config_adpt *)names_nm[NAMES_ADPT].n_value) == 0 )
		return ;

	if( names_nm[NAMES_UBDINIT].n_value == 0 )
		return ;

	build_table((struct uba_device *)names_nm[NAMES_UBDINIT].n_value) ;

	for(i = 0; ; i++) {
		if((p + i) == 0 )
			break ;

		readk((long)(p + i), (char *)&adpt, sizeof(adpt)) ;

		if( adpt.p_name == 0 )
			break ;

		if( adpt.c_ptr == 0 )
			continue ;

		if( !header_printed ) {
			printf("#\n#\tUNIBUS, Q-BUS, VAXBI information.\n#\n") ;
			header_printed = 1 ;
		}

		/*
		 *	Read the parent adapter name.
		 */
		readk((long)adpt.p_name, parent, BUFSIZ) ;

		switch( adpt.c_type ) {
		case 'A':
			print_iobus(&adpt, parent, adpt.p_num) ;
			break ;
		case 'D':
			print_device((struct uba_device *)adpt.c_ptr,
				parent, adpt.p_num) ;
			break ;
		case 'C':
			print_controller((struct uba_ctlr *)adpt.c_ptr,
				parent, adpt.p_num) ;
			break ;
		default:
			printf("unknown bus type: %c\n", adpt.c_type) ;
			break ;
		} 
	}

	free_table() ;
}

print_iobus(ap, parent, unit)
struct config_adpt *ap ;
char	*parent ;
int	unit ;
{
	char	controller[BUFSIZ] ;

	if( ap->c_num < 0 )
		return ;

	if( ap->c_name == 0 )
		(void)strcpy(controller, "iobus") ;
	else
		readk((long)ap->c_name, controller, BUFSIZ) ;

	if( strcmp(parent, "nexus") == 0 )
		printf("%s%d is a nexus.\n", controller, ap->c_num) ;
	else
		printf("%s%d at %s%d %s alive.\n",
			controller, ap->c_num, parent, unit,
			ap->c_ptr == (char *)CONFIG_ALIVE ? "is" : "is not") ;
}

/*
 *	Print information about a controller and return its name.
 */
print_controller(p, parent, unit)
struct uba_ctlr *p ;
char	*parent ;
int	unit ;
{
	struct uba_ctlr ctlr ;
	static char controller[DEV_SIZE+1] ;

	controller[DEV_SIZE] = '\0' ;

	/*
	 *	Read the controller structure.
	 */
	readk((long)p, (char *)&ctlr, sizeof(ctlr)) ;

	/*
	 *	Read the driver structure and controller name.
	 */
	if( ctlr.um_ctlrname == 0 )
		(void)strcpy(controller, "ctlr") ;
	else
		readk((long)ctlr.um_ctlrname, (char *)controller, DEV_SIZE) ;

	/*
	 *	Print something about the controller.
	 */
	printf("%s%d at %s%d %s alive.\n", controller, ctlr.um_ctlr,
		parent, unit, ctlr.um_alive ? "is" : "is not") ;

	find_devices(p, controller, ctlr.um_ctlr) ;

	return ;
}

/*
 *	Print device information.
 */
print_device(p, parent, unit)
struct uba_device *p ;
char	*parent ;
int	unit ;
{
	char	dev_name[DEV_SIZE+1] ;
	struct uba_device device ;
	
	dev_name[DEV_SIZE] = '\0' ;

	/*
	 *	Read the device structure.
	 */
	readk((long)p, (char *)&device, sizeof(struct uba_device)) ;
	
	/*
	 *	Read the device name.
	 */
	if( device.ui_devname == 0 )
		(void)strcpy(dev_name, "device") ;
	else
		readk((long)device.ui_devname, dev_name, DEV_SIZE) ;

	if( device.ui_mi )
		printf("%s%d at %s%d %s %d %s alive.\n", dev_name, 
			device.ui_unit, parent, unit, 
			device.ui_dk == -1 ? "slave" : "drive", device.ui_slave, 
			device.ui_alive ? "is" : "is not") ;
	else
		printf("%s%d at %s%d %s alive.\n", dev_name, device.ui_unit, 
			parent, unit, device.ui_alive ? "is" : "is not") ;

	if( device.ui_dk >= 0 && device.ui_dk < DK_NDRIVE )
		dk_map[device.ui_dk] = 1 ;
}

/*
 *	Build of table of "interesting" device information.
 */
struct device_table {
	struct uba_device *d_ptr ;	/* address of device structure */
	struct uba_ctlr   *c_ptr ;	/* address of controller */
} ;

struct device_table *uba_device = NULL ;
int	n_devices = 0 ;

find_devices(p, name, unit)
struct uba_ctlr *p ;
char	*name ;
int	unit ;
{
	int	i ;

	for(i = 0; i < n_devices; i++) {
		if( uba_device[i].c_ptr == p )
			print_device(uba_device[i].d_ptr, name, unit) ;
	}
}

build_table(p)
struct uba_device *p ;
{
	struct uba_device device ;
	unsigned i ;

	for(i = 0; ; i++) {
		readk((long)(p + i), (char *)&device, sizeof(device)) ;

		if( device.ui_driver == 0 )
			break ;
	}

	n_devices = i ;

	if((uba_device = (struct device_table *)calloc(i, sizeof(struct device_table))) == NULL )
		fatal("Can't allocate space for device table: %s.\n", module) ;

	for(i = 0; i < n_devices; i++) {
		readk((long)(p + i), (char *)&device, sizeof(device)) ;

		uba_device[i].d_ptr = p + i ;
		uba_device[i].c_ptr = device.ui_mi ;
	}
}

free_table()
{	
	free((char *)uba_device) ;
}

/*
 *	Given a boot-time print the time the system has been up.
 */
uptime(boot)
long	boot ;
{
	long	time(), now ;
	int	days, hours, minutes, comma = 0 ;

	now = time((long *)0) ;

	if( now <= boot )
		printf("which is unreasonable, since it is now %24.24s.\n",
			ctime((long *)&now)) ;
	else {
		minutes = (now - boot) / 60 ;
		days = minutes / (24 * 60) ;
		minutes %= (24 * 60) ;
		hours = minutes / 60 ;
		minutes %= 60 ;

		printf("and has been up for") ;

		if( days ) {
			comma++ ;
			printf(" %d day%s", days, days == 1 ? "" : "s") ;
		}

		if( hours ) {
			if( comma++ )
				putchar(',') ;

			printf(" %d hour%s", hours, hours == 1 ? "" : "s") ;
		}

		if( minutes ) {
			if( comma++ )
				putchar(',') ;

			printf(" %d minute%s", minutes, minutes == 1 ? "" : "s") ;
		}

		if( days == 0 && hours == 0 && minutes == 0 )
			printf(" less than a minute.\n") ;
		else
			printf(".\n") ;
	}
}

/*
 *	Some versions (V3.x on DECstations) don't store the device 
 *	configuration, relying on a non-zero number transfers in the 
 *	_dk_xfer[] array.  We'll fake something something to at least
 *	acknowledge that there are other devices out there on the
 *	SCSI.
 */
scsi_names()
{
	register i, scsi_header_printed = 0 ;

	/*
	 *	Read the transfer list.
	 */
	readk((long)names_nm[NAMES_DK_XFER].n_value, (char *)dk_xfer, 
		sizeof(dk_xfer)) ;

	for(i = 0; i < DK_NDRIVE; i++) {
		if( dk_map[i] || dk_xfer[i] == 0 )
			continue ;

		if( scsi_header_printed == 0 ) {
			printf("#\n#\tOther Disk or SCSI devices.\n#\n") ;
			scsi_header_printed = 1 ;
		}

		printf("dk%d is alive.\n", i) ;
	}
}
