/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 magnify.c
 *	Date:	 3/29/90
 *	Version: 1.35
 *
 *	magnify.c - The general work of the magnify function.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)magnify.c	1.35 (monitor) 3/29/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 *
 * Jan. 15, 1989 -- arr
 *
 *	Move mag_array[] to tables.c
 *
 * Feb. 10, 1989 -- arr
 *
 *	Changed RECTYPE3 to RECONF.
 *
 * Feb. 14, 1989 -- arr
 *
 *	Added arrow key support.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <curses.h>
#include <stdio.h>

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <limits.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#include <net/if.h>
#include <netinet/in.h>

#include "include.h"
#include "options.h"
#include "monitor.h"
#include "extern.h"

/*
 *	Default display.
 */
extern	int	lines ;			/* length of screen being used. */
extern	WINDOW	*wp ;

/*
 *	Module name for error functions.
 */
static	char	*module = "magnify" ;

#define	Y_ORIG	2

/*
 *	This lives in tables.c
 */
extern struct magnify_table mag_array[] ;

/*
 *	This is the display function being used before magnify was
 *	called.  Magnify current assumes that the screen function
 *	is the only place that magnify will be called from.  This
 *	is not necessarily reasonable.
 *
 *	"Current_function" is used to remember what the current magnify
 *	function is.  When there is no current magnify function
 *	it should be -1.
 */
static	(*display_function)() = 0 ;
static	(*redraw_function)() = 0 ;
static	(*help_function)() = 0 ;
static	current_function = -1 ;

/*
 *	Inquire of the user what he wants to magnify and select
 *	the approproiate output function.
 */
magnify(op)
OPTION	*op ;
{
	register i, mask ;
	int	 choices[MON$N_RECORDS], n_choices = 0 ;

/*
 *	Build a table which will map the indexs used by the
 *	simple menu into mag_array indexs.  If there is 
 *	only one entry in the table then use that function.
 */
	for(i = 0; i < sizeof(op->opt_data) * CHAR_BIT; i++)
		if((op->opt_data & (1 << i)) && mag_array[i].mag_open )
			choices[n_choices++] = i ;

/*
 *	There isn't anything to magnify.
 */
	if( n_choices == 0 )
		return ;

/*
 *	Magnify the one choice.  The loop already verified that a
 *	function exists, so select_function WILL succede.
 */
	if( n_choices == 1 ) {
		if( select_function(1 << choices[0], op) == 0 )
			panic("Function doesn't really exist!\n", module) ;

		return ;
	}

/*
 *	Run the menu until one of the options is selected.
 */
	mask = magnify_menu(op, n_choices, choices) ;

/*
 *	Once again, we KNOW that there is a magnify function
 *	available if the mask is non-zero.
 */
	if( mask && select_function(mask, op) == 0 )
		panic("Function doesn't really exist!\n", module) ;
}

/*
 *	A simple menu to select the magnify function from
 *	those available.  This function will either return
 *	a mask useable by select_function or zero.
 */
magnify_menu(op, n_choices, choices)
int	n_choices ;
int	*choices ;
OPTION	*op ;
{
	int	lasty, command, y = Y_ORIG, save_lines ;
	FILE	*fp, *fopen() ;

	/*
	 *	Save the number of lines used by the previous
	 *	display function.
	 */
	save_lines = lines ;

	lasty = Y_ORIG + n_choices - 1 ;

	redraw_magnify(op->opt_data, y) ;

	for( ; ; ) {
		switch( command = get_command()) {
		case G_HELP:
			help_magnify() ;
			/* yes, there isn't a break here. */
		case G_REDRAW:
			redraw_magnify(op->opt_data, y) ;
			break ;
		case G_DUMP:
			if((fp = fopen(op->opt_output, "a")) == NULL ) {
				warning("Can't open output file: %s (%s).\n",
					module, op->opt_output) ;
				continue ;
			}

			dump_screen(fp) ;

			if( fclose(fp) == EOF )
				warning("Can't close dump file: %s.\n", module) ;

			/*
			 *	Clear the bottom line.
			 */
			wmove(wp, LINES - 1, 0) ;
			wclrtoeol(wp) ;

			/*
			 *	Redraw the cursor.
			 */
			wmove(wp, y, 0) ;
			wprintw(wp, "==>") ;

			break ;
		case G_UP:
		case G_DOWN:
			wmove(wp, y, 0) ;
			wprintw(wp, "   ");

			if( command == G_DOWN )
				y = (y == lasty) ? Y_ORIG : y + 1 ;
			else
				y = (y == Y_ORIG) ? lasty : y - 1 ;

			wmove(wp, y, 0) ;
			wprintw(wp, "==>") ;

			break ;
		case G_SELECT:
			lines = save_lines ;
			return 1 << choices[y - Y_ORIG] ;
		case G_EXIT:
			lines = save_lines ;
			return 0 ;
		default:
			break ;
		}

		wrefresh(wp) ;
	}
}

/*
 *	Read characters and return a command based on the 
 *	input.  It tries to do the reasonable thing when
 *	it sees an ESC.  Currently it works for <HELP>,
 *	<SELECT>, the UP-ARROW and DOWN-ARROW.  It should
 *	be possible to generalize for the main loop input
 *	parser.
 */
get_command()
{
	register command, ch1, ch2 ;

	switch( getch() ) {
	case 'h':
	case '?':
		command = G_HELP ;
		break ;
	case 'r':
	case 12:
		command = G_REDRAW ;
		break ;
	case 'd':
		command = G_DUMP ;
		break ;
	case 'j':
		command = G_DOWN ;
		break ;
	case 'k':
		command = G_UP ;
		break ;
	case 's':
		command = G_SELECT ;
		break ;
	case 'e':
	case 'q':
		command = G_EXIT ;
		break ;
	case '\033':
		command = getch() ;

		if( command != '[' && command != 'O' )
			command = G_UNKNOWN ;
		else {
			switch( getch()) {
			case 'A':
				command = G_UP ;
				break ;
			case 'B':
				command = G_DOWN ;
				break ;
			case 'C':
				command = G_UNKNOWN ;
				break ;
			case 'D':
				command = G_UNKNOWN ;
				break ;
			case '4':
				if( getch() == '~' )
					command = G_SELECT ;
				else
					command = G_UNKNOWN ;

				break ;
			case '2':
				ch1 = getch() ;
				ch2 = getch() ;

				if( ch1 == '8' && ch2 == '~' )
					command = G_HELP ;
				else
					command = G_UNKNOWN ;

				break ;
			default:
				command = G_UNKNOWN ;
				break ;
			}
		}

		break ;
	default:
		command = G_UNKNOWN ;
		break ;
	}

	return command ;
}
/*
 *	Draw the initial menu.
 */
redraw_magnify(flags, y)
int	flags, y ;
{
	int	i ;

	lines = Y_ORIG ;

	wclear(wp) ;

	wmove(wp, 0, 0) ;
	
	wprintw(wp, "Type '?' or 'h' for help.") ;

	wmove(wp, Y_ORIG, 0);

/*
 *	Print out which are available.
 */
	for(i = 0 ; i < sizeof(flags) * CHAR_BIT; i++)
		if((flags & (1 << i)) && mag_array[i].mag_open ) {
			wprintw(wp, "    %s\n", records[i].string);
			lines++ ;
		}

	/*
	 *	Draw the pointer.
	 */
	wmove(wp, y, 0) ;
	wprintw(wp, "==>") ;

	wrefresh(wp) ;
}


/*
 *	The documentation says the error flag is -1.  Experience
 *	shows that the error flag is equal to 0.
 */
#define	FFS_ERROR	0

/*
 *	Given a bit mask of possible magnify available, check
 *	each bit to see if there is a corresponding magnify
 *	function and set it up if there is.  If there isn't
 *	a function available return 0, otherwise a non-zero
 *	value.
 */
select_function(mask, op)
int	mask ;
OPTION	*op ;
{
	int	bit ;

	/*
	 *  1.  Turn off the control records, except sample.
	 */
	mask &= ~(MON$M_FIRST | MON$M_LAST | MON$M_RECONF) ;

	/*
	 *	Until the mask is zero or a magnify function is
	 *	found get the first bit set and check for the
	 *	function.
	 */
	for( ; ; ) {
		if((bit = ffs(mask)) == FFS_ERROR )
			return 0 ;

		bit -= 1 ;

		if( mag_array[bit].mag_display )
			break ;

		mask &= ~(1 << bit) ;
	}

	/*
	 *	Remember the display functions the first time
	 *	magnify is called.
	 */
	if( display_function == 0 ) {
		display_function = op->display ;
		redraw_function = op->redraw_display ;
		help_function = op->help_display ;
	}

	/*
	 *	Setup the new display functions.
	 */
	op->display = mag_array[bit].mag_display ;
	op->open_display = mag_array[bit].mag_open ;
	op->redraw_display = mag_array[bit].mag_open ;
	op->help_display = mag_array[bit].mag_help ;

	/*
	 *	Save the index of the new display functions.
	 */
	current_function = bit ;

	return bit ;
}

/*
 *	Undo the effect of a magnify.
 *
 *	To prevent the various fields from overflowing we'll run
 *	the delta_...() functions.
 */
unmagnify(op)
OPTION	*op ;
{
	if( display_function == 0 )
		return ;

	op->display = display_function ;
	op->redraw_display = redraw_function ;
	op->help_display = help_function ;

	if( op->redraw_display )
		(*op->redraw_display)(op) ;

	display_function = redraw_function = help_function = 0 ;
	current_function = -1 ;
}

/*
 *	Help function for the magnify menu.
 */
help_magnify()
{
	wclear(wp) ;

	wmove(wp, 0, 0) ;
	wprintw(wp, "Help - Magnify menu.\n\n") ;

	wprintw(wp, "\t'j' - Move the cursor down.\n") ;
	wprintw(wp, "\t'k' - Move the cursor up.\n") ;
	wprintw(wp, "\t'd' - Dump a copy of the screen.\n") ;
	wprintw(wp, "\t's' - Select a record.\n") ;
	wprintw(wp, "\t'r' - Redraw the screen.\n") ;
	wprintw(wp, "\t'h' - This help screen.\n") ;
	wprintw(wp, "\t'?' - This help screen.\n") ;
	wprintw(wp, "\t'e' - Return to the previous display.\n\n") ;

	continue_prompt() ;
}
