/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 live.c
 *	Date:	 8/21/90
 *	Version: 1.60
 *
 *	live.c - This file contains code for collecting system data.
 *	It will fill the structure (or structures) provided based 
 *	on the "options" argument.
 *
 *	Derived from:
 *
 *		@(#)gather.c	1.17 5/24/86
 */
#ifndef	lint
static	char	SccsId[] = "@(#)live.c	1.60 (monitor) 8/21/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 *
 * 29-June-1988 -- arr
 *
 *	Added code from pid.c to setup for process data collection.
 *	This changed open_live(), close_live() and added setup_proc()
 *	to live().
 *
 *	Added a call to collect_pid() if opt_pid is turned on.
 *
 *	Added a check on opt_data to see if it was necessary to
 *	call collect_kernel().
 *
 *	Dust of some acculated lint.
 *
 *  1-July-1988 -- arr
 *
 *	Allow live() to collect other data when opt_pid is turned
 *	on.  Live() will return MON_EXIT whenever collect_pid()
 *	does.
 *
 * 29-November-1988 -- arr
 *
 *	Fix the load average code to work on the PMAX.
 *
 *  2-December-1988 -- arr
 *
 *	Fix disk data collection code to work on the PMAX.
 *
 *  Jan. 5, 1989 -- arr
 *
 *	Fixed collection of paging data to understand the individual
 *	fields.
 *
 * Jan. 15, 1989 -- arr
 *
 *	Run the swap space collection functions.
 *
 * Feb. 10, 1989 -- arr
 *
 *	Added a case for BUF records to the switch that calculates
 *	the size of a record.
 *
 * Feb. 15, 1989 -- arr
 *
 *	Correctly set mon_flag with MON$M_VALID (in FIRST and SAMPLE).
 *
 * March 11, 1989 -- arr
 *
 *	Changed needed for change to mon_fork structure in monitor.h.
 *
 * April 28, 1989 -- arr
 *
 *	Added a new method of collecting the number of clock ticks
 *	the system has been running.  This was added in anticipation
 *	of SMP Ultrix.  It's currently turned by the USE_GETTIMEOFDAY
 *	constant.
 *
 * November 1989 -- arr
 *
 *	Changes for CPU data structures on SMP systems.
 *
 * Dec. 25, 1989 -- arr
 *
 *	Added a data structure needed to fully resolve all the
 *	different ways of tracking devices.
 *
 *	Cleaned up namelist setup and checking for private namelists.
 *
 * Dec. 27, 1989 -- arr
 *
 *	Move tty namelist check and collection to tty.c.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#ifdef	V4_ULTRIX
#	include <sys/namei.h>
#endif
#ifdef	mips
#	include <sys/fixpoint.h>
#endif

#include <net/if.h>
#include <netinet/in.h>

#include "include.h"
#include "options.h"
#include "monitor.h"
#include "extern.h"

/*
 *	The list of data options that collect_kernel() will
 *	want to look at.
 */
#define	MON$M_KERNEL	(MON$M_CPU | MON$M_TTY | MON$M_DISK | MON$M_FORK | \
			 MON$M_FREE | MON$M_PAGE | MON$M_PROC | MON$M_MEMORY | \
			 MON$M_LOADAVE | MON$M_NAMEI )

/*
 *	The module name for the error functions.
 */
static	char	*module = "live" ;

/*
 *	Declarations for function that don't return (int).
 */
char	*strcpy() ;

void	exit(),
	nlist() ;

long	time() ;

/*
 *	The record version is declared and maintained in version.c.
 *	This is the only other file that uses it.
 */
extern	dev_t	record_version ;

/*
 *	These get used from open_live() and close_live().
 */
extern	int	mem, swap ;

/*
 *	Most of the kernel data will be read directly into the
 *	mon_... structures.  However some of it needs to shuffled
 *	off to a variety of places.  These are the objects for that
 *	info.
 */
struct vmtotal	 Total ;
struct vmmeter	 Sum, Rate ;
struct nchstats  Namei ;
struct forkstat  Fork ;
int		 free_memory ;
long		 dk_time[DK_NDRIVE] ;
long		 dk_seek[DK_NDRIVE] ;
long		 dk_xfer[DK_NDRIVE] ;
long		 dk_wds[DK_NDRIVE] ;
/*
 *	This could be done as a bitmap.  It is used to keep
 *	track of what disks have been found.  It's kept here
 *	because this is where data for the other dk_xxx[]
 *	arrays are.
 */
char		 dk_map[DK_NDRIVE] ;

/*
 *	Space for the load average.
 */
#ifdef	vax
	double	avenrun[MON$N_LOADAVE] ;
#elif 	mips
	fix	avenrun[MON$N_LOADAVE] ;
#endif

/*
 *	Initialize the static data, verify the options vs. the 
 *	namelist to see what can really be collected.  We also 
 *	make one pass on the kernel and run the delta functions 
 *	for disks, netif, forks and ttys. 
 */
open_live(op)
OPTION	*op ;
{
	/*
	 *	Initialize the namelist.  If the data collection
	 *	functions with private namelist are the only ones
	 *	used this doesn't need to be called.  I'll have to
	 *	work on that.
	 */
	nlist(op->opt_kernel, namelist);	/* no error returned */

	/*
	 *	Open /dev/kmem.
	 */
	if((kmem = open("/dev/kmem", O_RDONLY, 0)) == -1 )
		fatal("Can't open /dev/kmem: %s.\n", module);

	/*
	 *	If we're collecting data on processes, open /dev/mem
	 *	and /dev/drum.
	 */
	if( op->opt_pid ) {
		if((mem = open("/dev/mem", O_RDONLY, 0)) == -1 )
			fatal("Can't open /dev/mem: %s.\n", module);

		if((swap = open("/dev/drum", O_RDONLY, 0)) == -1 )
			fatal("Can't open /dev/drum: %s.\n", module);
	}

	/*
	 *	Open the system who file (/etc/wtmp, but it should be kept in
	 *	memory somewhere).  If we can't open the file for some reason
	 *	print an error message (in open_whofile()) and turn off the
	 *	option.
	 */
	if( op->opt_user )
		if( open_whofile() == -1 )
			op->opt_user = 0 ;

	/*
	 *	Check the contents of the namelist and turn off the options
	 *	that we can't access information for.
	 */
	check_namelist(op) ;

	/*
	 *	Do the setup stuff for swap space.  This also checks
	 *	the namelist.
	 */
	if( op->opt_swap )
		setup_swap(op) ;

	/*
	 *	Do the setup stuff for swap space.  This also checks
	 *	the namelist.
	 */
	if( op->opt_buf )
		setup_buf() ;

	/*
	 *	Collect the static info.
	 */
	collect_static(op);

	/*
	 *	Set all the fields in the first record.
	 */
	setup_first(op) ;

	/*
	 *	By now we know how many total structures we need.  We only
	 *	need these for some display functions.  We don't need the
	 *	data for save() and will return.
	 */
	if( op->display_mode == OPT_SAVE )
		return ;
	else
		allocate_totals(op) ;
}

/*
 *	Collect data out of the running kernel.
 */
live(op)
OPTION	*op ;
{
	register struct mon_netif *ip ;
	register i, rc = MON_NORMAL ;
	struct ifnet ifnet ;

	/*
	 *	Read the kernel data used by all data options.
	 */
	collect_common(op) ;

	if( op->opt_pid )
		rc = collect_pid(op) ;

	/*
	 *	Collect the swap space data.
	 */
	if( op->opt_swap )
		collect_swap() ;

	/*
	 *	Collect the buffer cache data.
	 */
	if( op->opt_buf )
		collect_buffer_cache() ;

	/*
	 *	Read the data specific kernel data.
	 */
	if( op->opt_data & MON$M_KERNEL )
		collect_kernel(op) ;

	/*
	 *	Get the number of users.
	 */
	if( op->opt_user )
		mon_user.mon_user = users() ;

	/*
	 *	Get the CPU information.
	 */
	if( op->opt_cpu ) {
		/*
		 *	This hasn't moved in V4.0.
		 */
		cpu[0].mon_pdma = Sum.v_pdma ;

		/*
		 *	In V4.0 this is in the cpudata structure.
		 */
#ifndef	V4_ULTRIX
		cpu[0].mon_trap = Sum.v_trap ;
		cpu[0].mon_intr = Sum.v_intr ;
		cpu[0].mon_syscall = Sum.v_syscall ;
#ifdef	mips
		cpu[0].mon_swtch = Sum.v_swtch ;	
#endif
#endif

		/*
		 *	These are still in the vmmeter structure
		 *	in V4.0, but aren't there on the VAX.
		 */
#ifdef	vax
		cpu[0].mon_tlbpid = 0 ;
		cpu[0].mon_soft = 0 ;
#endif
#ifdef	mips
		cpu[0].mon_tlbpid = Sum.v_tlbpid ;
		cpu[0].mon_soft = Sum.v_soft ;
#endif
		live_cpu() ;
	}

	/*
	 *	The loop here will collect the network interface 
 	 *	dynamic data.
	 */
	for(i = 0; i < n_netif; i++) {
		ip = netif + i ;

		readk((long)ip->mon_ifnet, (char *)&ifnet, sizeof(ifnet));

		ip->mon_if_flags   = ifnet.if_flags ;
		ip->mon_ipackets   = ifnet.if_ipackets ;
		ip->mon_ierrors    = ifnet.if_ierrors ;
		ip->mon_opackets   = ifnet.if_opackets ;
		ip->mon_oerrors    = ifnet.if_oerrors ;
		ip->mon_collisions = ifnet.if_collisions ;
	}

	/*
	 *	Collect the disk data from the approiate places.
	 */
	for(i = 0; i < n_disk; i++) {
		register dk = disk[i].mon_dk ;

		disk[i].mon_time = dk_time[dk] ;
		disk[i].mon_seek = dk_seek[dk] ;
		disk[i].mon_xfer = dk_xfer[dk] ;
		disk[i].mon_wds = dk_wds[dk] ;
	}

	/*
	 *	Start moving around the collected dynamic data.
	 */
	if( op->opt_free )
		mon_free.mon_freemem = free_memory ;

	if( op->opt_loadave ) {
		for(i = 0; i < MON$N_LOADAVE; i++) {
#ifdef	vax
			loadave.mon_loadave[i] = avenrun[i] ;
#endif
#ifdef	mips
			loadave.mon_loadave[i] = FIX_TO_DBL(avenrun[i]) ;
#endif
		}
	}

	if( op->opt_memory ) {
		memory.mon_vm     = Total.t_vm ;
		memory.mon_avm    = Total.t_avm ;
		memory.mon_rm     = Total.t_rm ;
		memory.mon_arm    = Total.t_arm ;
		memory.mon_vmtxt  = Total.t_vmtxt ;
		memory.mon_avmtxt = Total.t_avmtxt ;
		memory.mon_rmtxt  = Total.t_rmtxt ;
		memory.mon_armtxt = Total.t_armtxt ;
		memory.mon_free   = Total.t_free ;
	}

	if( op->opt_proc ) {
		mon_proc.mon_rq = Total.t_rq ;
		mon_proc.mon_dw = Total.t_dw ;
		mon_proc.mon_pw = Total.t_pw ;
		mon_proc.mon_sl = Total.t_sl ;
		mon_proc.mon_sw = Total.t_sw ;
	}

	if( op->opt_page ) {
		mon_page.mon_freemem = free_memory ;

		mon_page.mon_pswpin  = Rate.v_pswpin ;
		mon_page.mon_pswpout = Rate.v_pswpout ;
		mon_page.mon_pgin    = Rate.v_pgin ;
		mon_page.mon_pgout   = Rate.v_pgout ;
		mon_page.mon_pgpgin  = Rate.v_pgpgin ;
		mon_page.mon_pgpgout = Rate.v_pgpgout ;

		mon_page.mon_intrans = Rate.v_intrans ;
		mon_page.mon_pgrec   = Rate.v_pgrec ;
		mon_page.mon_xifrec  = Rate.v_xifrec ;
		mon_page.mon_xsfrec  = Rate.v_xsfrec ;

		mon_page.mon_zfod    = Rate.v_zfod ;
		mon_page.mon_nexfod  = Rate.v_nexfod ;
		mon_page.mon_nvrfod  = Rate.v_nvrfod ;
		mon_page.mon_exfod   = Rate.v_exfod ;
		mon_page.mon_vrfod   = Rate.v_vrfod ;
		mon_page.mon_nzfod   = Rate.v_nzfod ;

		mon_page.mon_pgfrec    = Rate.v_pgfrec ;
		mon_page.mon_scan      = Rate.v_scan ;
		mon_page.mon_fastpgrec = Rate.v_fastpgrec ;
		mon_page.mon_faults    = Rate.v_faults ;
		mon_page.mon_rev       = Rate.v_rev ;
		mon_page.mon_seqfree   = Rate.v_seqfree ;
		mon_page.mon_dfree     = Rate.v_dfree ;

		mon_page.mon_swpin  = Rate.v_swpin ;
		mon_page.mon_swpout = Rate.v_swpout ;
	}

	if( op->opt_fork ) {
		mon_fork.mon_fork       = Fork.cntfork ;
		mon_fork.mon_fork_size  = Fork.sizfork ;
		mon_fork.mon_vfork      = Fork.cntvfork ;
		mon_fork.mon_vfork_size = Fork.sizvfork ; 
	}

	if( op->opt_namei ) {
		namei.mon_goodhits  = Namei.ncs_goodhits ;
		namei.mon_miss      = Namei.ncs_miss ;
		namei.mon_pass2     = Namei.ncs_pass2 ;
		namei.mon_2passes   = Namei.ncs_2passes ;
#ifdef	V4_ULTRIX
		namei.mon_badhits   = 0 ;
		namei.mon_falsehits = 0 ;
		namei.mon_long      = Namei.ncs_too_long ;
#else
		namei.mon_badhits   = Namei.ncs_badhits ;
		namei.mon_falsehits = Namei.ncs_falsehits ;
		namei.mon_long      = Namei.ncs_long ;
#endif
	}

	/*
 	 *	MON_EXIT is an error condition for opt_collect functions.
	 */
	return rc ;
}

/*
 *	Free all the space allocated by init_ and generally clean up.
 */
close_live(op)
OPTION	*op ;
{
	if( op->opt_buf )
		finish_buf() ;

	if( op->opt_swap )
		free_swap() ;

	if( op->opt_user )
		close_whofile() ;

	if( op->display_mode != OPT_SAVE )
		free_totals(op);

	if( close(kmem) == -1 )
		fatal("Can't close /dev/kmem: %s.\n", module);

	if( op->opt_pid ) {
		if( close(mem) == -1 )
			fatal("Can't close /dev/mem: %s.\n", module);

		if( close(swap) == -1 )
			fatal("Can't close /dev/drum: %s.\n", module);
	}
}

/*
 *	Gather the static data (names of devices, physmem, etc).
 */
collect_static(op)
OPTION	*op ;
{
	if( op->opt_disk )
		collect_disk(op) ;

	if( op->opt_cpu )
		config_cpus() ;		/* CPU's for state info */

	if( op->opt_netif )
		net_devices();		/* network interfaces */

	/*
	 *	If we're collecting data on processes, do some setup
	 *	here.
	 */
	if( op->opt_pid )
		setup_proc(op) ;
}

/*
 *	This is where we check the namelist values to determine what
 *	is available.  If a needed namelist entry is 0, they we
 *	turn off the approiate option.
 *
 *	The real question is when and why a namelist entry might be
 *	zero.  Under what circumstances is this to be expected and
 *	and if unexpected should we panic()?
 *
 *	The current version of this code check for things which
 *	I believe might not be there.
 */
check_namelist(op)
OPTION	*op ;
{
	/*
	 *	See if the per CPU data is available.
	 */
	if( namelist[NM_CPUDATA].n_value == 0 ) {
		info("The CPU data is not available.\n", module) ;
		op->opt_cpu = 0 ;
	}

	/*
	 *	We'll use dk_busy to determine if the disk I/O info
	 *	is available.
	 */
	if( namelist[NM_DK_BUSY].n_value == 0 ) {
		info("Disk I/O data is not available.\n", module);
		op->opt_disk = 0 ;
	}

	/*
	 *	Between V3.1 and V4.0 the tty namelist is different
	 *	enough we'll put the code in it's own function.
	 */
	check_tty_namelist(op) ;

	if( namelist[NM_IFNET].n_value == 0 ) {
		info("Network interface data not available.\n", module);
		op->opt_netif = 0 ;
	}

	/*
	 *	Make sure the namei stats are available.
	 */
	if( namelist[NM_NAMEI].n_value == 0 ) {
                info("Namei statistics are not available.\n", module);
		op->opt_namei = 0 ;
        }
}

/*
 *	Setup the first record.
 */
setup_first(op)
OPTION	*op ;
{
	char	*newline, *strchr() ;

	/*
	 *	The easy stuff.
	 */
	first.mon_options = op->opt_data ;
	first.mon_sleep   = op->opt_sleep ;
	first.mon_flag    = MON$M_VALID ;

	first.mon_cpu     = n_cpu ;
	first.mon_disk    = n_disk ;
	first.mon_netif   = n_netif ;
	first.mon_buf     = n_buf ;

	first.mon_pid 	  = getpid();
	first.mon_sdate	  = time((long *)0);
	first.mon_record  = record_version ;
	first.mon_arch    = architecture ;
	first.mon_nbpg    = NBPG ;
	first.mon_bsize   = DEV_BSIZE ;

	/*
	 *	Read in the version string.
	 */
	readk((long)namelist[NM_VERSION].n_value, (char *)first.mon_version,
		MON$S_VERSION);

	first.mon_version[MON$S_VERSION] = '\0' ;

	if((newline = strchr(first.mon_version, '\n')) != NULL )
		*newline = '\0' ;

	/*
	 *	Read in "_hz".
	 */
	first.mon_hz = get_word((long)namelist[NM_HZ].n_value) ;

	/*
	 *	Read in the amount of physical memory.
	 */
	first.mon_physmem = get_word((long)namelist[NM_PHYSMEM].n_value) ;

	/*
	 *	Read the boot time.
	 */
	readk((long)namelist[NM_BOOT].n_value, (char *)&first.mon_boot,
		sizeof(first.mon_boot)) ;
	
	/*
	 *	Get the hostname.  We are going to use a normal 
	 *	system call here...
	 */
	if( gethostname(first.mon_hostname, MON$S_HOSTNAME) == -1 )
		(void)strcpy(first.mon_hostname, "Who am I?");

	first.mon_hostname[MON$S_HOSTNAME] = '\0' ;
}

/*
 *	Collect the common data needed for each sample.
 */
collect_common(op)
OPTION	*op ;
{
	/*
	 *	Collect the timestamp.
	 */
	sample.mon_timestamp = time((long *)0) ;

	/*
	 *	Find size of the data sample.
	 */
	sample.mon_datalen = sample_size(op->opt_data) ;

	sample.mon_flag  = MON$M_VALID ;

	/*
	 *	The display functions need to maintain the elapsed 
	 *	time between samples.  To do this we'll find the
	 *	and store the number of clock ticks that have occured
	 *	since the system booted.  The functions that need to
	 *	keep track of clock ticks will use this number.
	 *
	 *	There are three ways of doing this, two of which
	 *	are below.  The third assumes that the processor
	 *	with the most clock ticks represents how long the
	 *	system has been up.  I won't use it.
	 */

 	/*
	 *	This method uses the current time, the boottime and
	 *	and the clock speed to figure out how many clock
	 *	ticks have happened.  It stands a fair chance of
	 *	working on many versions of Ultrix, but is new and
	 *	untested.
	 */
	sample.mon_ticks = get_ticks() ;
}

/*
 *	Number of microseconds in a second.
 */
#define	MICROSEC	(1000000)

/*
 *	This uses the boottime and clock speed from the first
 *	record to find out how many clock ticks the system has
 *	been up.
 */
get_ticks()
{
	register ticks ;
	register struct timeval *np, *bp ;
	struct timeval now ;
	struct timezone tz ;
	double   parts ;

	np = &now ;
	bp = &first.mon_boot ;

	/*
	 *	Get the current time.
 	 */
	if( gettimeofday(np, &tz) == -1 )
		fatal("gettimeofday(2) failed: %s.\n", module) ;

	ticks = (np->tv_sec - bp->tv_sec) * first.mon_hz ;

	if( np->tv_usec < bp->tv_usec ) {
		ticks -= first.mon_hz ;
		parts = bp->tv_usec - np->tv_usec ;
	}
	else
		parts = np->tv_usec - bp->tv_usec ;

	parts /= MICROSEC ;

	return ticks + (int)(parts * first.mon_hz) ;
}

/*
 *	Read most of the kernel data.  The only exceptions are the
 *	_cpudata structures and the network interface data.
 */
collect_kernel(op)
OPTION	*op ;
{
	/*
	 *	Read the data from the kernel that's used by more
	 *	than one data option.
 	 */
	if( op->opt_free || op->opt_page )
		free_memory = get_word((long)namelist[NM_FREE].n_value) ;

	if( op->opt_memory || op->opt_proc )
		readk((long)namelist[NM_TOTAL].n_value, (char *)&Total,
			sizeof(Total)) ;

	/*
	 *	The cpu and page options require the contents of the
	 *	_sum structure.
	 */
	if( op->opt_cpu || op->opt_page )
		readk((long)namelist[NM_SUM].n_value, (char *)&Sum, sizeof(Sum)) ;

	/*
	 *	Read the disk data.
	 */
	if( op->opt_disk ) {
		readk((long)namelist[NM_DK_WDS].n_value, (char *)dk_wds,
			sizeof(dk_wds)) ;
		readk((long)namelist[NM_DK_XFER].n_value, (char *)dk_xfer, 
			sizeof(dk_xfer)) ;
		readk((long)namelist[NM_DK_SEEK].n_value, (char *)dk_seek, 
			sizeof(dk_seek)) ;
		readk((long)namelist[NM_DK_TIME].n_value, (char *)dk_time, 
			sizeof(dk_time)) ;
	}

	if( op->opt_tty )
		collect_tty() ;

	if( op->opt_loadave ) {
		readk((long)namelist[NM_LOADAVE].n_value, (char *)avenrun,
			sizeof(avenrun)) ;
	}

	if( op->opt_fork ) {
		readk((long)namelist[NM_FORK].n_value, (char *)&Fork,
			sizeof(Fork)) ;
	}

	/*
	 *	Read the _rate and _deficit data for the page option.
	 */
	if( op->opt_page ) {
		readk((long)namelist[NM_RATE].n_value, (char *)&Rate, sizeof(Rate)) ;

		mon_page.mon_deficit = get_word((long)namelist[NM_DEFICIT].n_value) ;
	}

	/*
	 *	Read _nchstats.
	 */
	if( op->opt_namei )
		readk((long)namelist[NM_NAMEI].n_value, (char *)&Namei,
                	sizeof(Namei)) ;

}

/*
 *	Find just the size of a data sample.  This is a stripped
 *	version of the code in iovec.c.
 */
sample_size(data)
int	data ;
{
	register int i, size = 0 ;

	for(i = 0; i < sizeof(struct opt_data_bits) * NBBY; i++) {
		if((data & (1 << i)) == 0 )
			continue ;

		switch( i ) {
		case MON$C_DISK:
			size += (MON$S_DISK * n_disk) ;
			break ;
		case MON$C_NETIF:
			size += (MON$S_NETIF * n_netif) ;
			break ;
		case MON$C_CPU:
			size += (MON$S_CPU * n_cpu) ;
			break ;
		case MON$C_BUF:
			size += (MON$S_BUF * n_buf) ;
			break ;
		default:
			size += records[i].size ;
			break ;
		}
	}

	return size ;
}
