/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 cpu.c
 *	Date:	 8/22/90
 *	Version: 1.60
 *
 *	cpu.c - Figure out how many CPU's there are and allocate 
 *	an array large enough for all of them.
 *
 *	I only thought this was complicated when we added multi-
 *	processor support, but SMP is a real bear.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)cpu.c	1.60 (monitor) 8/22/90" ;
#endif

/*
 * Modification History
 *
 * 27-June-1988 - arr
 *
 *	Moved the string data for bits into a new source file
 *	string_data.c.
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 *
 *	Use a data table instead of switch table to determine the
 *	cpu name.
 *
 * Beginning Nov. 16, 1989 -- arr
 *
 *	Fix support for SMP.  To complicate matters the SMP
 *	support keeps changing, but this varient should be
 *	stable and still work on pre-V4.0 with #ifdef's.
 *
 * Dec. 25, 1989 -- arr
 *
 *	Change the way getsid works and is used.
 *
 * Feb. 16, 1989 -- arr
 *
 *	Added mon_name to CPU record and moved f_cpu() from functions.c
 *
 * Feb. 15, 1989 -- arr
 *
 *	Correctly set mon_flag with MON$M_VALID.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 * Aug. 22, 1990 -- arr
 *
 *	Fix a problem in magnify_cpu() where the number of system
 *	calls and interrupts is too high.
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <curses.h>

#include <sys/types.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/dk.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/time.h>
#include <sys/proc.h>

#include <sys/cpudata.h>

#include <net/if.h>
#include <netinet/in.h>

#ifdef	vax
#	include <machine/cpu.h>
#elif	mips
#	include <machine/cpuconf.h>
#endif

#include "monitor.h"
#include "include.h"
#include "options.h"
#include "extern.h"

/*
 *	Aliases for CPudata structure names changed in V4.0.
 */
#ifdef	V4_ULTRIX
#	define	c_state		cpu_state
#	define	c_cptime	cpu_cptime
#	define	c_switch	cpu_switch
#endif

/*
 *	Set aside enough space for the array of CPU pointers.
 */
#ifndef	MAXCPU
#	define	MAXCPU	(DK_NDRIVE)
#endif
/*
 *	Module name for error functions.
 */
static	char	*module = "cpu" ;

/*
 *	Functions that don't return (int).
 */
char	*calloc(),
	*strncpy(),
	*str_state(),
	*str_cpuid(),
	*str_generic() ;

void	delta_syscall_intr() ;

/*
 *	Default display.
 */
extern	int	lines ;			/* length of screen being used */
extern	WINDOW	*wp ;

/*
 *	Determine what version of Ultrix we are on and set up the
 *	CPU list accordingly.
 */
config_cpus()
{
	register state, i ;
	struct cpudata *array[MAXCPU], *cpuaddr, mon_cpudata ;
	int	subtype ;
	long	sid_addr ;

	/*
	 *	Initialize the array of cpudata indexes.
	 */
	for(i = 0; i < MAXCPU; i++)
		array[i] = 0 ;

	/*
	 *	Arrange to use the right namelist entry for the
	 *	"sid".
	 */
#ifdef	vax
	sid_addr = (long)namelist[NM_CPU].n_value ;
#endif
#ifdef	mips
	sid_addr = (long)namelist[NM_CPU_SYSTYPE].n_value ;
#endif

	if( namelist[NM_MAXCPU].n_value )
		maxcpu = get_word((long)namelist[NM_MAXCPU].n_value) ;
	else
		maxcpu = MAXCPU ;

	if( maxcpu > MAXCPU ) {
		info("The CPU array isn't big enough for the number of CPUs in the system.\n", module) ;
		info("Some CPUs may be missing.\n", module) ;
		maxcpu = MAXCPU ;
	}

	if((cpu = (struct mon_cpu *)calloc((unsigned)maxcpu, MON$S_CPU)) == NULL )
		fatal("Can't allocate space for CPU structures: %s.\n", module);

	/*
	 *	Get the CPU subtype.
	 */
	if( namelist[NM_CPU_SUBTYPE].n_value )
		readk((long)namelist[NM_CPU_SUBTYPE].n_value, (char *)&subtype,
			sizeof(subtype)) ;

	/*
	 *	Before V4.0 this was address of the array itself.  In
	 *	V4.0 it is the address of an array of addresses of
	 *	CPU data structures. So we have arrange to do the
	 *	right thing.
	 *
	 *	In V4.0 we'll read the array of addresses directly,
	 *	otherwise we'll fake it.
	 *	
	 */
	cpuaddr = (struct cpudata *)namelist[NM_CPUDATA].n_value ;

#ifdef	V4_ULTRIX
	readk((long)cpuaddr, array, sizeof(array[0]) * MAXCPU) ;
#else
	for(i = 0; i < maxcpu; i++)
		array[i] = cpuaddr + i ;
#endif

	/*
	 *	For every cpu in the array {
	 *
	 *		If ( we don't want this cpu ) continue ;
	 * 
	 *		save the data for this cpu.
	 *	}
	 */
	for(i = 0, n_cpu = -1; i < MAXCPU; i++) {
		if( find_value(&cpu_list, "", (int)n_cpu) == 0 )
			continue ;

		if( array[i] == 0 )
			continue ;

		n_cpu++ ;

		readk((long)array[i], (char *)&mon_cpudata,
			sizeof(mon_cpudata));

		cpu[n_cpu].mon_type    = MON$C_CPU ;
		cpu[n_cpu].mon_length  = MON$S_CPU ;
		cpu[n_cpu].mon_ident   = getsid(sid_addr) ;
		cpu[n_cpu].mon_index   = n_cpu ;
		cpu[n_cpu].mon_where   = (caddr_t)array[i] ;
		cpu[n_cpu].mon_subtype = subtype ;

#ifdef	V4_ULTRIX
		cpu[n_cpu].mon_cpu     = mon_cpudata.cpu_num ;
		cpu[n_cpu].mon_state   = mon_cpudata.cpu_state ;
#else
#	ifdef	vax
		cpu[n_cpu].mon_state   = mon_cpudata.c_state ;
#	endif
#	ifdef	mips
		cpu[n_cpu].mon_state   = CPU_RUN ;
#	endif
		cpu[n_cpu].mon_cpu     = i ;
#endif

		if( cpu[n_cpu].mon_state & CPU_RUN )
			cpu[n_cpu].mon_flag |= MON$M_VALID ;
		else
			continue ;

/*
 *	On V4.0 we can get lots more things from cpudata structure
 *	that we have get elsewhere in earlier version.
 */
#if	defined(V4_ULTRIX) || defined(vax)
		cpu[n_cpu].mon_swtch  = mon_cpudata.c_switch ;
#endif

#ifdef	V4_ULTRIX
		cpu[n_cpu].mon_syscall = mon_cpudata.cpu_syscall ;
		cpu[n_cpu].mon_trap    = mon_cpudata.cpu_trap ;
		cpu[n_cpu].mon_intr    = mon_cpudata.cpu_intr ;
#endif

		for(state = 0; state < CPUSTATES; state++)
			cpu[n_cpu].mon_time[state] = mon_cpudata.c_cptime[state] ;

		(void)strncpy(cpu[n_cpu].mon_name, 
			str_cpuid(cpu[n_cpu].mon_ident, cpu[n_cpu].mon_subtype),
			MON$S_CPUNAME);
	}

	n_cpu++ ;
}

/*
 *	The function(s) to collect data from a running system.  The are two
 *	ways of collecting the data depending on the value of maxcpu.
 */

/*
 *	The loop here will collect the per CPU data.  Another
 *	function sets up the linked list of CPU's.  It will have
 *	worry about the true implimentation of multiple CPU machines.
 */
live_cpu()
{
	register i, state ;
	struct   cpudata mon_cpudata ;

	/*
 	 *	If the list is empty don't bother.
	 */
	if( cpu == (struct mon_cpu *)NULL )
		return ;

	/*
	 *	Collect the current data foreach cpu.  Do this even
	 *	if the CPU isn't running because we want to get the
	 *	CPU state.
	 */
	for(i = 0; i < n_cpu; i++) {
		readk((long)cpu[i].mon_where, (char *)&mon_cpudata,
			sizeof(mon_cpudata));

		for(state = 0; state < CPUSTATES; state++)
			cpu[i].mon_time[state] = mon_cpudata.c_cptime[state] ;

#if	defined(V4_ULTRIX) || defined(vax)
		cpu[i].mon_swtch = mon_cpudata.c_switch ;
#endif

#ifdef	V4_ULTRIX
		cpu[i].mon_syscall = mon_cpudata.cpu_syscall ;
		cpu[i].mon_intr    = mon_cpudata.cpu_intr ;
		cpu[i].mon_trap    = mon_cpudata.cpu_trap ;
#endif
	}
}

#define	HALF	(COLS / 2)

#define	CPU_INTR	 (3)
#define CPU_PDMA	 (4)
#define	CPU_OFFSET	 (25)
#define	CPU_STATE	 (8)
#define	CPU_OFFSET_STATE (5)

#define	CPU_BASE_LINES	(9)		/* + 1 per CPU */

/*
 *	Print headers for the CPU magnify function.
 *
 *	ARGSUSED
 */
open_cpu(op)
OPTION	*op ;
{
	register i ;

	lines = CPU_BASE_LINES ;

	sample_header() ;

	wprintw(wp, "Calls to trap():\n") ;
	wprintw(wp, "Calls to syscall():\n") ;
	wprintw(wp, "Calls to new_tlbpid():\n") ;

	wmove(wp, CPU_INTR, HALF) ;
	wprintw(wp, "Device Interrupts:") ;

	wmove(wp, CPU_INTR+1, HALF) ;
	wprintw(wp, "Pseudo-dma interrupts:") ;

	wmove(wp, CPU_INTR+2, HALF) ;
	wprintw(wp, "Software interrupts:") ;

	wmove(wp, CPU_STATE-1, 0);
#ifdef	V4_ULTRIX
	wprintw(wp, "CPU:  %%user   %%nice %%system   %%idle  swtch   intr  scall  state\n");
#else
	wprintw(wp, "CPU:  %%user   %%nice %%system   %%idle  swtch  state\n");
#endif

	if( cpu == NULL )
		return ;

	for(i = 0; i < first.mon_cpu; i++, lines++) {
		wmove(wp, CPU_STATE+i, 0) ;
		wprintw(wp, "#%d", cpu[i].mon_cpu);

		if((CPU_STATE + i) == (LINES - 1))
			break ;
	}
}

/*
 *	We'll use this later...
 */
#define	DELTA(current, previous)	{ tmp = current ; \
					  current -= previous ; \
					  previous = tmp ; \
					}
/* 
 *	"Magnify" the CPU information.
 *
 *	ARGSUSED
 */
magnify_cpu(op)
OPTION	*op ;
{
	double	 total_time, delta_cpu(), etime ;
	register i ;
	int	 total_syscall, total_intr ;

	delta_syscall_intr(&total_syscall, &total_intr) ;

	for(i = 0; i < first.mon_cpu; i++) {
		if((cpu[i].mon_flag & MON$M_VALID) == 0 )
			continue ;

		etime = delta_cpu(i, sample.mon_ticks, &total_time) ;

		if( i == 0 ) {
			wmove(wp, CPU_INTR, CPU_OFFSET) ;
			wprintw(wp, "%8.1f", cpu[0].mon_trap / etime);

			wmove(wp, CPU_INTR+1, CPU_OFFSET) ;
			wprintw(wp, "%8.1f", total_syscall / etime);

			wmove(wp, CPU_INTR+2, CPU_OFFSET) ;
			wprintw(wp, "%8.1f", cpu[0].mon_tlbpid / etime);

			wmove(wp, CPU_INTR, HALF+CPU_OFFSET) ;
			wprintw(wp, "%8.1f", total_intr / etime);

			wmove(wp, CPU_INTR+1, HALF+CPU_OFFSET) ;
			wprintw(wp, "%8.1f", cpu[0].mon_pdma / etime);

			wmove(wp, CPU_INTR+2, HALF+CPU_OFFSET) ;
			wprintw(wp, "%8.1f", cpu[0].mon_soft / etime);
		}

		wmove(wp, CPU_STATE + i, CPU_OFFSET_STATE) ;
#ifdef	V4_ULTRIX
		wprintw(wp, "%6.0f  %6.0f  %6.0f  %6.0f  %5.0f  %5.0f  %5.0f  <%s>",
#else
		wprintw(wp, "%6.0f  %6.0f  %6.0f  %6.0f  %5.0f  <%s>",
#endif
			100. * cpu[i].mon_time[0] / total_time,
			100. * cpu[i].mon_time[1] / total_time,
			100. * cpu[i].mon_time[2] / total_time,
			100. * cpu[i].mon_time[3] / total_time,
			cpu[i].mon_swtch / etime,
#ifdef	V4_ULTRIX
			cpu[i].mon_intr / etime,
			cpu[i].mon_syscall / etime,
#endif
			str_state(cpu[i].mon_state));

		wclrtoeol(wp) ;

		if((CPU_STATE + i) == (LINES - 1))
			break ;
	}

	sample_body(etime) ;
}

/*
 *	Local version of a system call to get the contents of the SID
 *	register.  This function constructs a number which looks like
 *	the contents of a CPU SID register.
 *
 *	It is a feature of the MicroVAX architecture that this function
 *	will always return the correct SID.
 *
 *	It tries real hard to do the right thing on VAX and RISC
 *	systems for V3.x and V4.0.
 */
getsid(addr)
long	addr ;
{
/*
 *	The union used to construct an SID look-alike.
 */
#ifdef	vax
	union	cpusid cpu_sid ;
#endif

/*
 *	The architecture dependent code.
 */
#ifdef	vax
	cpu_sid.cpuany.cp_type = get_word(addr) ;

	return cpu_sid.cpusid ;
#endif

#ifdef	mips
	return get_word(addr) ;
#endif
}

/*
 *	Functions and data to take the CPU state bits and turn them into
 *	something "meaningful".
 */
extern struct bit_names state_names[] ;

char	*str_state(state)
int	state ;
{
	static	char	buf[BUFSIZ] ;

	return str_generic(state, buf, state_names) ;
}

extern struct cpu_type cpu_types[] ;

/*
 *	Functions and data to take the CPU id and turn it into something
 *	"meaningful".  Another part of monitor takes the cpu type and 
 *	turns it into something that looks a little like the CPU id.
 */
char	*str_cpuid(ident, subtype)
int	ident ;
long	subtype ;
{
#ifdef	vax
	union cpusid zounds ;
#endif
	int	type ;
	struct cpu_type *sp, *tp ;

	tp = &cpu_types[0] ;

/*
 *	This is the VAX version of how to turn the system
 *	id into a system name.
 */
#ifdef	vax
	zounds.cpusid = ident ;
	type = zounds.cpuany.cp_type ;
#endif

/*
 *	On the mips we can get everything from the cpu_systype.
 *	This will ignore the value of subtype passed to us.
 */
#ifdef	mips
	type    = GETCPUTYPE(ident) ;
	subtype = GETSYSTYPE(ident) ;
#endif

	while( tp->string != NULL ) {
		if( tp->type == type ) {
			/*
			 *	If there isn't a subtype list return
			 *	the string for the type.
			 */
			if((sp = tp->subtype) == NULL )
				return tp->string ;

			/*
			 *	Else...  Walk the the subtype list
			 *	looking for a matching subtype.
			 */
			while( sp->string != NULL ) {
				if( sp->type == subtype )
					return sp->string ;

				sp++ ;
			}

			/*
			 *	If there isn't a subtype match return
			 *	the type string.
			 */
			return tp->string ;
		}

		tp++ ;
	}

	return "Unknown CPU type" ;
}

/*
 *	Function to dump the CPU record.
 */
f_cpu(p)
struct mon_cpu *p ;
{
	register state ;

	if((p->mon_flag & MON$M_VALID) == 0 )
		return ;

	printf("%s.\n", records[MON$C_CPU].string) ;

	printf("\tname:    %s\n", p->mon_name) ;
	printf("\tident:   0x%x\n", p->mon_ident, p->mon_ident) ;
	printf("\tsubtype: 0x%x\n", p->mon_subtype, p->mon_subtype) ;

	for(state = 0 ; state < CPUSTATES; state++)
		printf("\t[%d]:     %d ticks\n", state, p->mon_time[state]);

	printf("\tstate:   <%s>\n", str_state(p->mon_state)) ;
	printf("\tcpu:     %d\n", p->mon_cpu) ;
	printf("\tindex:   %d\n", p->mon_index) ;
	printf("\tswtch:   %u\n", p->mon_swtch) ;
	printf("\ttrap:    %u\n", p->mon_trap) ;
	printf("\tsyscall: %u\n", p->mon_syscall) ;
	printf("\tintr:    %u\n", p->mon_intr) ;
	printf("\tpdma:    %u\n", p->mon_pdma) ;
	printf("\ttlbpid:  %u\n", p->mon_tlbpid) ;
	printf("\tsoft:    %u\n", p->mon_soft) ;
}
