/*
 *  Copyright (C) 2017 - This file is part of libecc project
 *
 *  Authors:
 *      Ryad BENADJILA <ryadbenadjila@gmail.com>
 *      Arnaud EBALARD <arnaud.ebalard@ssi.gouv.fr>
 *      Jean-Pierre FLORI <jean-pierre.flori@ssi.gouv.fr>
 *
 *  Contributors:
 *      Nicolas VIVET <nicolas.vivet@ssi.gouv.fr>
 *      Karim KHALFALLAH <karim.khalfallah@ssi.gouv.fr>
 *
 *  This software is licensed under a dual BSD and GPL v2 license.
 *  See LICENSE file at the root folder of the project.
 */
#ifndef __EC_SELF_TESTS_CORE_H__
#define __EC_SELF_TESTS_CORE_H__

#include <libecc/libsig.h>

/* A test is fully defined by the attributes pointed in this structure. */
typedef struct {
	/* Test case name */
	const char *name;

	/* Curve params */
	const ec_str_params *ec_str_p;

	/* Private key */
	const u8 *priv_key;
	u8 priv_key_len;

	/* Function returning a fixed random value */
	int (*nn_random) (nn_t out, nn_src_t q);

	/* Hash function */
	hash_alg_type hash_type;

	/* Message */
	const char *msg;
	u32 msglen;

	/* Expected signature and associated length */
	ec_alg_type sig_type;
	const u8 *exp_sig;
	u8 exp_siglen;

	/* Optional ancillary data */
	const u8 *adata;
	u16 adata_len;
} ec_test_case;

/* ECDH test case */
typedef struct {
	/* Test case name */
	const char *name;

	/* ECDH type */
	ec_alg_type ecdh_type;

	/* Curve params */
	const ec_str_params *ec_str_p;

	/* Our private key */
	const u8 *our_priv_key;
	u8 our_priv_key_len;

	/* Peer public key */
	const u8 *peer_pub_key;
	u8 peer_pub_key_len;

	/* Our expected public key */
	const u8 *exp_our_pub_key;
	u8 exp_our_pub_key_len;

	/* Expected shared secret */
	const u8 *exp_shared_secret;
	u8 exp_shared_secret_len;
} ecdh_test_case;

/*******************************************************************
 ************** ECDSA tests ****************************************
 *******************************************************************/
#if (defined(WITH_HASH_SHA3_224) || defined(WITH_HASH_SHA3_256) || \
     defined(WITH_HASH_SHA3_384) || defined(WITH_HASH_SHA3_512))

/*
 * This test message is the 1600 bits message used by NIST in its
 * test vectors for SHA3. We reuse it for sig/verif test vectors
 * using SHA3.
 */
static const u8 sha3_1600_bit_msg[] = {
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
	0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3
};
#endif

#ifdef WITH_SIG_ECDSA
#ifdef WITH_HASH_SHA3_224
#ifdef WITH_CURVE_SECP224R1
#define ECDSA_SHA3_224_SECP224R1_SELF_TEST

/* ECDSA secp224r1 test vectors */

static int ecdsa_nn_random_secp224r1_sha3_224_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp224r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0xb7, 0x63, 0x8a, 0x22, 0xe5, 0xee, 0xe7, 0x76,
		0xa4, 0x25, 0x3f, 0x97, 0xe8, 0x35, 0x1a, 0x3c,
		0x94, 0xb0, 0x19, 0x37, 0x66, 0x2a, 0x0e, 0xbd,
		0xd6, 0xe7, 0xae, 0xe0
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp224r1_sha3_224_test_vectors_priv_key[] = {
	0xc3, 0x1a, 0x15, 0x2d, 0x2c, 0x62, 0x40, 0x21,
	0x80, 0x87, 0x55, 0xb2, 0xcb, 0xdf, 0x80, 0xd3,
	0x09, 0xe3, 0x6a, 0xea, 0xc1, 0xcc, 0xd9, 0xdb,
	0x82, 0xb1, 0xd7, 0xc3
};

static const u8 ecdsa_secp224r1_sha3_224_test_vectors_expected_sig[] = {
	0x56, 0xa6, 0x93, 0xe1, 0x20, 0x0a, 0xa8, 0xe3,
	0xd2, 0x78, 0x07, 0xba, 0x5c, 0xe7, 0xb8, 0xf7,
	0xdf, 0x24, 0xa6, 0x65, 0x1e, 0xb8, 0x70, 0x6e,
	0x65, 0x52, 0x49, 0xd4,
	0x14, 0xd0, 0x4b, 0x6b, 0xb9, 0x32, 0x58, 0x7d,
	0xc0, 0xc6, 0x1c, 0xa1, 0x44, 0xf8, 0x4b, 0xf4,
	0x6e, 0x7a, 0xbd, 0xcf, 0xa7, 0xa7, 0xa1, 0xee,
	0x56, 0x06, 0x9a, 0x37
};

static const ec_test_case ecdsa_secp224r1_sha3_224_test_case = {
	.name = "ECDSA-SHA3_224/secp224r1",
	.ec_str_p = &secp224r1_str_params,
	.priv_key = ecdsa_secp224r1_sha3_224_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp224r1_sha3_224_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_secp224r1_sha3_224_test_vector,
	.hash_type = SHA3_224,
	.msg = (const char *)sha3_1600_bit_msg,
	.msglen = sizeof(sha3_1600_bit_msg),
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp224r1_sha3_224_test_vectors_expected_sig,
	.exp_siglen =
		sizeof(ecdsa_secp224r1_sha3_224_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP224R1 */
#endif /* WITH_HASH_SHA3_224 */

#ifdef WITH_HASH_SHA3_256
#ifdef WITH_CURVE_SECP256R1
#define ECDSA_SHA3_256_SECP256R1_SELF_TEST

/* ECDSA secp256r1 test vectors */

static int ecdsa_nn_random_secp256r1_sha3_256_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp256r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp256r1_sha3_256_test_vectors_priv_key[] = {
	0xDC, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 ecdsa_secp256r1_sha3_256_test_vectors_expected_sig[] = {
	0xCB, 0x28, 0xE0, 0x99, 0x9B, 0x9C, 0x77, 0x15,
	0xFD, 0x0A, 0x80, 0xD8, 0xE4, 0x7A, 0x77, 0x07,
	0x97, 0x16, 0xCB, 0xBF, 0x91, 0x7D, 0xD7, 0x2E,
	0x97, 0x56, 0x6E, 0xA1, 0xC0, 0x66, 0x95, 0x7C,

	0x1d, 0x5d, 0x46, 0x09, 0xa2, 0xf9, 0x69, 0xa1,
	0x90, 0xeb, 0x6b, 0x84, 0x51, 0xdd, 0x43, 0x0d,
	0x65, 0x07, 0x10, 0x4d, 0xb6, 0x46, 0x61, 0x68,
	0xec, 0x7a, 0x73, 0xdb, 0x8c, 0x96, 0xe9, 0x1b
};

static const ec_test_case ecdsa_secp256r1_sha3_256_test_case = {
	.name = "ECDSA-SHA3_256/secp256r1",
	.ec_str_p = &secp256r1_str_params,
	.priv_key = ecdsa_secp256r1_sha3_256_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp256r1_sha3_256_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_secp256r1_sha3_256_test_vector,
	.hash_type = SHA3_256,
	.msg = (const char *)sha3_1600_bit_msg,
	.msglen = sizeof(sha3_1600_bit_msg),
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp256r1_sha3_256_test_vectors_expected_sig,
	.exp_siglen =
		sizeof(ecdsa_secp256r1_sha3_256_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP256R1 */
#endif /* WITH_HASH_SHA3_256 */

#ifdef WITH_HASH_SHA3_512
#ifdef WITH_CURVE_SECP256R1
#define ECDSA_SHA3_512_SECP256R1_SELF_TEST

/*
 * ECDSA secp256r1 test vector using SHA3_512. This test vector is intended
 * to test truncation (right shift) step in signature and verification
 * code when the output size of the hash function is larger than the
 * bit size of q (order of the group generated by the base point).
 */

static int ecdsa_nn_random_secp256r1_sha3_512_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp256r1_sha3_512_test_vectors_priv_key[] = {
	0xDC, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 ecdsa_secp256r1_sha3_512_test_vectors_expected_sig[] = {
	0xcb, 0x28, 0xe0, 0x99, 0x9b, 0x9c, 0x77, 0x15,
	0xfd, 0x0a, 0x80, 0xd8, 0xe4, 0x7a, 0x77, 0x07,
	0x97, 0x16, 0xcb, 0xbf, 0x91, 0x7d, 0xd7, 0x2e,
	0x97, 0x56, 0x6e, 0xa1, 0xc0, 0x66, 0x95, 0x7c,

	0x88, 0x48, 0x1a, 0x4f, 0x5b, 0xd4, 0xf6, 0x8f,
	0xcf, 0xe0, 0xb3, 0x15, 0xc4, 0xd4, 0x0a, 0xf9,
	0xd5, 0x31, 0x76, 0x97, 0xae, 0x29, 0x45, 0x47,
	0xef, 0x7c, 0x8f, 0x01, 0x41, 0x24, 0xea, 0xff
};

static const ec_test_case ecdsa_secp256r1_sha3_512_test_case = {
	.name = "ECDSA-SHA3_512/secp256r1",
	.ec_str_p = &secp256r1_str_params,
	.priv_key = ecdsa_secp256r1_sha3_512_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp256r1_sha3_512_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_secp256r1_sha3_512_test_vector,
	.hash_type = SHA3_512,
	.msg = (const char *)sha3_1600_bit_msg,
	.msglen = sizeof(sha3_1600_bit_msg),
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp256r1_sha3_512_test_vectors_expected_sig,
	.exp_siglen =
		sizeof(ecdsa_secp256r1_sha3_512_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP256R1 */
#endif /* WITH_HASH_SHA3_512 */

#ifdef WITH_HASH_SHA3_384
#ifdef WITH_CURVE_SECP384R1
#define ECDSA_SHA3_384_SECP384R1_SELF_TEST

/* ECDSA secp384r1 test vectors */

static int ecdsa_nn_random_secp384r1_sha3_384_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp384r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0xB4, 0xB7, 0x4E, 0x44, 0xD7, 0x1A, 0x13, 0xD5,
		0x68, 0x00, 0x3D, 0x74, 0x89, 0x90, 0x8D, 0x56,
		0x4C, 0x77, 0x61, 0xE2, 0x29, 0xC5, 0x8C, 0xBF,
		0xA1, 0x89, 0x50, 0x09, 0x6E, 0xB7, 0x46, 0x3B,
		0x85, 0x4D, 0x7F, 0xA9, 0x92, 0xF9, 0x34, 0xD9,
		0x27, 0x37, 0x62, 0x85, 0xE6, 0x34, 0x14, 0xFA
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp384r1_sha3_384_test_vectors_priv_key[] = {
	0x0B, 0xEB, 0x64, 0x66, 0x34, 0xBA, 0x87, 0x73,
	0x5D, 0x77, 0xAE, 0x48, 0x09, 0xA0, 0xEB, 0xEA,
	0x86, 0x55, 0x35, 0xDE, 0x4C, 0x1E, 0x1D, 0xCB,
	0x69, 0x2E, 0x84, 0x70, 0x8E, 0x81, 0xA5, 0xAF,
	0x62, 0xE5, 0x28, 0xC3, 0x8B, 0x2A, 0x81, 0xB3,
	0x53, 0x09, 0x66, 0x8D, 0x73, 0x52, 0x4D, 0x9F
};

static const u8 ecdsa_secp384r1_sha3_384_test_vectors_expected_sig[] = {
	0xfb, 0x01, 0x7b, 0x91, 0x4e, 0x29, 0x14, 0x94,
	0x32, 0xd8, 0xba, 0xc2, 0x9a, 0x51, 0x46, 0x40,
	0xb4, 0x6f, 0x53, 0xdd, 0xab, 0x2c, 0x69, 0x94,
	0x80, 0x84, 0xe2, 0x93, 0x0f, 0x1c, 0x8f, 0x7e,
	0x08, 0xe0, 0x7c, 0x9c, 0x63, 0xf2, 0xd2, 0x1a,
	0x07, 0xdc, 0xb5, 0x6a, 0x6a, 0xf5, 0x6e, 0xb3,

	0xbe, 0x9b, 0x4f, 0x2b, 0x7b, 0x02, 0xca, 0xd3,
	0xd0, 0xb4, 0x7c, 0xf6, 0x20, 0xfa, 0x26, 0xae,
	0x24, 0x6c, 0x95, 0x86, 0x1d, 0xbe, 0x6b, 0x5d,
	0xc5, 0xb3, 0xb3, 0xfb, 0xea, 0x96, 0x4f, 0xef,
	0x48, 0xb7, 0x4b, 0x07, 0x4a, 0xdd, 0x6a, 0xc8,
	0x8f, 0x4f, 0x3f, 0x9d, 0xdd, 0x3a, 0xa4, 0x1c
};

static const ec_test_case ecdsa_secp384r1_sha3_384_test_case = {
	.name = "ECDSA-SHA3_384/secp384r1",
	.ec_str_p = &secp384r1_str_params,
	.priv_key = ecdsa_secp384r1_sha3_384_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp384r1_sha3_384_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_secp384r1_sha3_384_test_vector,
	.hash_type = SHA3_384,
	.msg = (const char *)sha3_1600_bit_msg,
	.msglen = sizeof(sha3_1600_bit_msg),
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp384r1_sha3_384_test_vectors_expected_sig,
	.exp_siglen =
		sizeof(ecdsa_secp384r1_sha3_384_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP384R1 */
#endif /* WITH_HASH_SHA3_384 */

#ifdef WITH_HASH_SHA3_512
#ifdef WITH_CURVE_SECP521R1
#define ECDSA_SHA3_512_SECP521R1_SELF_TEST

/* ECDSA secp521r1 test vectors */

static int ecdsa_nn_random_secp521r1_sha3_512_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp521r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0x00, 0xC1, 0xC2, 0xB3, 0x05, 0x41, 0x9F, 0x5A,
		0x41, 0x34, 0x4D, 0x7E, 0x43, 0x59, 0x93, 0x3D,
		0x73, 0x40, 0x96, 0xF5, 0x56, 0x19, 0x7A, 0x9B,
		0x24, 0x43, 0x42, 0xB8, 0xB6, 0x2F, 0x46, 0xF9,
		0x37, 0x37, 0x78, 0xF9, 0xDE, 0x6B, 0x64, 0x97,
		0xB1, 0xEF, 0x82, 0x5F, 0xF2, 0x4F, 0x42, 0xF9,
		0xB4, 0xA4, 0xBD, 0x73, 0x82, 0xCF, 0xC3, 0x37,
		0x8A, 0x54, 0x0B, 0x1B, 0x7F, 0x0C, 0x1B, 0x95,
		0x6C, 0x2F
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp521r1_sha3_512_test_vectors_priv_key[] = {
	0x00, 0x65, 0xFD, 0xA3, 0x40, 0x94, 0x51, 0xDC,
	0xAB, 0x0A, 0x0E, 0xAD, 0x45, 0x49, 0x51, 0x12,
	0xA3, 0xD8, 0x13, 0xC1, 0x7B, 0xFD, 0x34, 0xBD,
	0xF8, 0xC1, 0x20, 0x9D, 0x7D, 0xF5, 0x84, 0x91,
	0x20, 0x59, 0x77, 0x79, 0x06, 0x0A, 0x7F, 0xF9,
	0xD7, 0x04, 0xAD, 0xF7, 0x8B, 0x57, 0x0F, 0xFA,
	0xD6, 0xF0, 0x62, 0xE9, 0x5C, 0x7E, 0x0C, 0x5D,
	0x54, 0x81, 0xC5, 0xB1, 0x53, 0xB4, 0x8B, 0x37,
	0x5F, 0xA1
};

static const u8 ecdsa_secp521r1_sha3_512_test_vectors_expected_sig[] = {
	0x01, 0x54, 0xfd, 0x38, 0x36, 0xaf, 0x92, 0xd0,
	0xdc, 0xa5, 0x7d, 0xd5, 0x34, 0x1d, 0x30, 0x53,
	0x98, 0x85, 0x34, 0xfd, 0xe8, 0x31, 0x8f, 0xc6,
	0xaa, 0xaa, 0xb6, 0x8e, 0x2e, 0x6f, 0x43, 0x39,
	0xb1, 0x9f, 0x2f, 0x28, 0x1a, 0x7e, 0x0b, 0x22,
	0xc2, 0x69, 0xd9, 0x3c, 0xf8, 0x79, 0x4a, 0x92,
	0x78, 0x88, 0x0e, 0xd7, 0xdb, 0xb8, 0xd9, 0x36,
	0x2c, 0xae, 0xac, 0xee, 0x54, 0x43, 0x20, 0x55,
	0x22, 0x51,
	0x00, 0x15, 0xeb, 0xfd, 0xad, 0xf3, 0xf8, 0x00,
	0x11, 0x51, 0x47, 0x9e, 0xec, 0xf4, 0x19, 0x53,
	0x01, 0x56, 0xc2, 0x85, 0x62, 0x69, 0xc5, 0x24,
	0x4c, 0x7a, 0xbd, 0xac, 0x36, 0xab, 0x45, 0xd0,
	0xdc, 0xf5, 0xa5, 0xb5, 0x33, 0xff, 0xff, 0xa2,
	0xa8, 0xbe, 0xea, 0xea, 0xda, 0x04, 0xc1, 0xc5,
	0xc8, 0x1e, 0x99, 0xa1, 0xd7, 0x88, 0xc9, 0x24,
	0x03, 0xe1, 0x78, 0x4a, 0x82, 0xcd, 0xd5, 0xe5,
	0xfb, 0x27
};

static const ec_test_case ecdsa_secp521r1_sha3_512_test_case = {
	.name = "ECDSA-SHA3_512/secp521r1",
	.ec_str_p = &secp521r1_str_params,
	.priv_key = ecdsa_secp521r1_sha3_512_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp521r1_sha3_512_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_secp521r1_sha3_512_test_vector,
	.hash_type = SHA3_512,
	.msg = (const char *)sha3_1600_bit_msg,
	.msglen = sizeof(sha3_1600_bit_msg),
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp521r1_sha3_512_test_vectors_expected_sig,
	.exp_siglen =
		sizeof(ecdsa_secp521r1_sha3_512_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP521R1 */
#endif /* WITH_HASH_SHA3_512 */

#ifdef WITH_HASH_SHA224
#ifdef WITH_CURVE_SECP192R1
#define ECDSA_SHA224_SECP192R1_SELF_TEST

/* ECDSA secp192r1 w/ sha224 test vectors to test truncation steps */

static int ecdsa_nn_random_secp192r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0xFA, 0x6D, 0xE2, 0x97, 0x46, 0xBB, 0xEB, 0x7F,
		0x8B, 0xB1, 0xE7, 0x61, 0xF8, 0x5F, 0x7D, 0xFB,
		0x29, 0x83, 0x16, 0x9D, 0x82, 0xFA, 0x2F, 0x4E
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp192r1_test_vectors_priv_key[] = {
	0x1A, 0x8D, 0x59, 0x8F, 0xC1, 0x5B, 0xF0, 0xFD,
	0x89, 0x03, 0x0B, 0x5C, 0xB1, 0x11, 0x1A, 0xEB,
	0x92, 0xAE, 0x8B, 0xAF, 0x5E, 0xA4, 0x75, 0xFB
};

static const u8 ecdsa_secp192r1_test_vectors_expected_sig[] = {
	0x88, 0x50, 0x52, 0x38, 0x0F, 0xF1, 0x47, 0xB7,
	0x34, 0xC3, 0x30, 0xC4, 0x3D, 0x39, 0xB2, 0xC4,
	0xA8, 0x9F, 0x29, 0xB0, 0xF7, 0x49, 0xFE, 0xAD,

	0x66, 0x63, 0xF2, 0x78, 0x36, 0x98, 0x7E, 0xED,
	0x45, 0x85, 0x82, 0xD1, 0xF4, 0x43, 0x29, 0x3A,
	0x5E, 0xD8, 0x88, 0x49, 0xB2, 0xFC, 0x5C, 0xD9
};

static const ec_test_case ecdsa_secp192r1_test_case = {
	.name = "ECDSA-SHA224/secp192r1",
	.ec_str_p = &secp192r1_str_params,
	.priv_key = ecdsa_secp192r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp192r1_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_secp192r1_test_vector,
	.hash_type = SHA224,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp192r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_secp192r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP192R1 */
#endif /* WITH_HASH_SHA224 */

#ifdef WITH_HASH_SHA224
#ifdef WITH_CURVE_SECP224R1
#define ECDSA_SHA224_SECP224R1_SELF_TEST

/* ECDSA secp224r1 test vectors */

static int ecdsa_nn_random_rfc4754_secp224r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp224r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0xb7, 0x63, 0x8a, 0x22, 0xe5, 0xee, 0xe7, 0x76,
		0xa4, 0x25, 0x3f, 0x97, 0xe8, 0x35, 0x1a, 0x3c,
		0x94, 0xb0, 0x19, 0x37, 0x66, 0x2a, 0x0e, 0xbd,
		0xd6, 0xe7, 0xae, 0xe0
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp224r1_test_vectors_priv_key[] = {
	0xc3, 0x1a, 0x15, 0x2d, 0x2c, 0x62, 0x40, 0x21,
	0x80, 0x87, 0x55, 0xb2, 0xcb, 0xdf, 0x80, 0xd3,
	0x09, 0xe3, 0x6a, 0xea, 0xc1, 0xcc, 0xd9, 0xdb,
	0x82, 0xb1, 0xd7, 0xc3
};

static const u8 ecdsa_secp224r1_test_vectors_expected_sig[] = {
	0x56, 0xa6, 0x93, 0xe1, 0x20, 0x0a, 0xa8, 0xe3,
	0xd2, 0x78, 0x07, 0xba, 0x5c, 0xe7, 0xb8, 0xf7,
	0xdf, 0x24, 0xa6, 0x65, 0x1e, 0xb8, 0x70, 0x6e,
	0x65, 0x52, 0x49, 0xd4,
	0x4e, 0x49, 0xa0, 0x37, 0x9d, 0x04, 0x4f, 0x34,
	0x82, 0xe5, 0x68, 0x32, 0x45, 0x84, 0x23, 0xcf,
	0x0a, 0xef, 0xef, 0xe9, 0x08, 0x1b, 0x6b, 0xc7,
	0x80, 0x5e, 0x18, 0xea
};

static const ec_test_case ecdsa_secp224r1_test_case = {
	.name = "ECDSA-SHA224/secp224r1",
	.ec_str_p = &secp224r1_str_params,
	.priv_key = ecdsa_secp224r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp224r1_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_rfc4754_secp224r1_test_vector,
	.hash_type = SHA224,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp224r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_secp224r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP224R1 */
#endif /* WITH_HASH_SHA224 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_SECP256R1
#define ECDSA_SHA256_SECP256R1_SELF_TEST

/* ECDSA secp256r1 test vectors */

static int ecdsa_nn_random_rfc4754_secp256r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp256r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp256r1_test_vectors_priv_key[] = {
	0xDC, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 ecdsa_secp256r1_test_vectors_expected_sig[] = {
	0xCB, 0x28, 0xE0, 0x99, 0x9B, 0x9C, 0x77, 0x15,
	0xFD, 0x0A, 0x80, 0xD8, 0xE4, 0x7A, 0x77, 0x07,
	0x97, 0x16, 0xCB, 0xBF, 0x91, 0x7D, 0xD7, 0x2E,
	0x97, 0x56, 0x6E, 0xA1, 0xC0, 0x66, 0x95, 0x7C,
	0x86, 0xFA, 0x3B, 0xB4, 0xE2, 0x6C, 0xAD, 0x5B,
	0xF9, 0x0B, 0x7F, 0x81, 0x89, 0x92, 0x56, 0xCE,
	0x75, 0x94, 0xBB, 0x1E, 0xA0, 0xC8, 0x92, 0x12,
	0x74, 0x8B, 0xFF, 0x3B, 0x3D, 0x5B, 0x03, 0x15
};

static const ec_test_case ecdsa_secp256r1_test_case = {
	.name = "ECDSA-SHA256/secp256r1",
	.ec_str_p = &secp256r1_str_params,
	.priv_key = ecdsa_secp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp256r1_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_rfc4754_secp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_secp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_SECP256R1
#define ECDSA_SHA512_SECP256R1_SELF_TEST

/*
 * ECDSA secp256r1 test vector using SHA512. This test vector is intended
 * to test truncation (right shift) step in signature and verification
 * code when the output size of the hash function is larger than the
 * bit size of q (order of the group generated by the base point).
 */

static int ecdsa_nn_random_secp256r1_sha512_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp256r1_sha512_test_vectors_priv_key[] = {
	0xDC, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 ecdsa_secp256r1_sha512_test_vectors_expected_sig[] = {
	0xcb, 0x28, 0xe0, 0x99, 0x9b, 0x9c, 0x77, 0x15,
	0xfd, 0x0a, 0x80, 0xd8, 0xe4, 0x7a, 0x77, 0x07,
	0x97, 0x16, 0xcb, 0xbf, 0x91, 0x7d, 0xd7, 0x2e,
	0x97, 0x56, 0x6e, 0xa1, 0xc0, 0x66, 0x95, 0x7c,
	0x78, 0x84, 0x35, 0x83, 0x06, 0xaf, 0xd1, 0xc8,
	0xa2, 0xa5, 0xf8, 0xee, 0x73, 0xe4, 0x22, 0x6f,
	0x1f, 0x6a, 0x14, 0xec, 0xaf, 0x2d, 0xe3, 0x0e,
	0xde, 0x1e, 0xb4, 0xe9, 0xe2, 0xf6, 0x8d, 0x3a
};

static const ec_test_case ecdsa_secp256r1_sha512_test_case = {
	.name = "ECDSA-SHA512/secp256r1",
	.ec_str_p = &secp256r1_str_params,
	.priv_key = ecdsa_secp256r1_sha512_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp256r1_sha512_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_secp256r1_sha512_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp256r1_sha512_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_secp256r1_sha512_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP256R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_SECP384R1
#define ECDSA_SHA384_SECP384R1_SELF_TEST

/* ECDSA secp384r1 test vectors */

static int ecdsa_nn_random_rfc4754_secp384r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp384r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0xB4, 0xB7, 0x4E, 0x44, 0xD7, 0x1A, 0x13, 0xD5,
		0x68, 0x00, 0x3D, 0x74, 0x89, 0x90, 0x8D, 0x56,
		0x4C, 0x77, 0x61, 0xE2, 0x29, 0xC5, 0x8C, 0xBF,
		0xA1, 0x89, 0x50, 0x09, 0x6E, 0xB7, 0x46, 0x3B,
		0x85, 0x4D, 0x7F, 0xA9, 0x92, 0xF9, 0x34, 0xD9,
		0x27, 0x37, 0x62, 0x85, 0xE6, 0x34, 0x14, 0xFA
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp384r1_test_vectors_priv_key[] = {
	0x0B, 0xEB, 0x64, 0x66, 0x34, 0xBA, 0x87, 0x73,
	0x5D, 0x77, 0xAE, 0x48, 0x09, 0xA0, 0xEB, 0xEA,
	0x86, 0x55, 0x35, 0xDE, 0x4C, 0x1E, 0x1D, 0xCB,
	0x69, 0x2E, 0x84, 0x70, 0x8E, 0x81, 0xA5, 0xAF,
	0x62, 0xE5, 0x28, 0xC3, 0x8B, 0x2A, 0x81, 0xB3,
	0x53, 0x09, 0x66, 0x8D, 0x73, 0x52, 0x4D, 0x9F
};

static const u8 ecdsa_secp384r1_test_vectors_expected_sig[] = {
	0xFB, 0x01, 0x7B, 0x91, 0x4E, 0x29, 0x14, 0x94,
	0x32, 0xD8, 0xBA, 0xC2, 0x9A, 0x51, 0x46, 0x40,
	0xB4, 0x6F, 0x53, 0xDD, 0xAB, 0x2C, 0x69, 0x94,
	0x80, 0x84, 0xE2, 0x93, 0x0F, 0x1C, 0x8F, 0x7E,
	0x08, 0xE0, 0x7C, 0x9C, 0x63, 0xF2, 0xD2, 0x1A,
	0x07, 0xDC, 0xB5, 0x6A, 0x6A, 0xF5, 0x6E, 0xB3,
	0xB2, 0x63, 0xA1, 0x30, 0x5E, 0x05, 0x7F, 0x98,
	0x4D, 0x38, 0x72, 0x6A, 0x1B, 0x46, 0x87, 0x41,
	0x09, 0xF4, 0x17, 0xBC, 0xA1, 0x12, 0x67, 0x4C,
	0x52, 0x82, 0x62, 0xA4, 0x0A, 0x62, 0x9A, 0xF1,
	0xCB, 0xB9, 0xF5, 0x16, 0xCE, 0x0F, 0xA7, 0xD2,
	0xFF, 0x63, 0x08, 0x63, 0xA0, 0x0E, 0x8B, 0x9F
};

static const ec_test_case ecdsa_secp384r1_test_case = {
	.name = "ECDSA-SHA384/secp384r1",
	.ec_str_p = &secp384r1_str_params,
	.priv_key = ecdsa_secp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp384r1_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_rfc4754_secp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_secp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_SECP521R1
#define ECDSA_SHA512_SECP521R1_SELF_TEST

/* ECDSA secp521r1 test vectors */

static int ecdsa_nn_random_secp521r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp521r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0x00, 0xC1, 0xC2, 0xB3, 0x05, 0x41, 0x9F, 0x5A,
		0x41, 0x34, 0x4D, 0x7E, 0x43, 0x59, 0x93, 0x3D,
		0x73, 0x40, 0x96, 0xF5, 0x56, 0x19, 0x7A, 0x9B,
		0x24, 0x43, 0x42, 0xB8, 0xB6, 0x2F, 0x46, 0xF9,
		0x37, 0x37, 0x78, 0xF9, 0xDE, 0x6B, 0x64, 0x97,
		0xB1, 0xEF, 0x82, 0x5F, 0xF2, 0x4F, 0x42, 0xF9,
		0xB4, 0xA4, 0xBD, 0x73, 0x82, 0xCF, 0xC3, 0x37,
		0x8A, 0x54, 0x0B, 0x1B, 0x7F, 0x0C, 0x1B, 0x95,
		0x6C, 0x2F
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_secp521r1_test_vectors_priv_key[] = {
	0x00, 0x65, 0xFD, 0xA3, 0x40, 0x94, 0x51, 0xDC,
	0xAB, 0x0A, 0x0E, 0xAD, 0x45, 0x49, 0x51, 0x12,
	0xA3, 0xD8, 0x13, 0xC1, 0x7B, 0xFD, 0x34, 0xBD,
	0xF8, 0xC1, 0x20, 0x9D, 0x7D, 0xF5, 0x84, 0x91,
	0x20, 0x59, 0x77, 0x79, 0x06, 0x0A, 0x7F, 0xF9,
	0xD7, 0x04, 0xAD, 0xF7, 0x8B, 0x57, 0x0F, 0xFA,
	0xD6, 0xF0, 0x62, 0xE9, 0x5C, 0x7E, 0x0C, 0x5D,
	0x54, 0x81, 0xC5, 0xB1, 0x53, 0xB4, 0x8B, 0x37,
	0x5F, 0xA1
};

static const u8 ecdsa_secp521r1_test_vectors_expected_sig[] = {
	0x01, 0x54, 0xFD, 0x38, 0x36, 0xAF, 0x92, 0xD0,
	0xDC, 0xA5, 0x7D, 0xD5, 0x34, 0x1D, 0x30, 0x53,
	0x98, 0x85, 0x34, 0xFD, 0xE8, 0x31, 0x8F, 0xC6,
	0xAA, 0xAA, 0xB6, 0x8E, 0x2E, 0x6F, 0x43, 0x39,
	0xB1, 0x9F, 0x2F, 0x28, 0x1A, 0x7E, 0x0B, 0x22,
	0xC2, 0x69, 0xD9, 0x3C, 0xF8, 0x79, 0x4A, 0x92,
	0x78, 0x88, 0x0E, 0xD7, 0xDB, 0xB8, 0xD9, 0x36,
	0x2C, 0xAE, 0xAC, 0xEE, 0x54, 0x43, 0x20, 0x55,
	0x22, 0x51, 0x01, 0x77, 0x05, 0xA7, 0x03, 0x02,
	0x90, 0xD1, 0xCE, 0xB6, 0x05, 0xA9, 0xA1, 0xBB,
	0x03, 0xFF, 0x9C, 0xDD, 0x52, 0x1E, 0x87, 0xA6,
	0x96, 0xEC, 0x92, 0x6C, 0x8C, 0x10, 0xC8, 0x36,
	0x2D, 0xF4, 0x97, 0x53, 0x67, 0x10, 0x1F, 0x67,
	0xD1, 0xCF, 0x9B, 0xCC, 0xBF, 0x2F, 0x3D, 0x23,
	0x95, 0x34, 0xFA, 0x50, 0x9E, 0x70, 0xAA, 0xC8,
	0x51, 0xAE, 0x01, 0xAA, 0xC6, 0x8D, 0x62, 0xF8,
	0x66, 0x47, 0x26, 0x60
};

static const ec_test_case ecdsa_secp521r1_test_case = {
	.name = "ECDSA-SHA512/secp521r1",
	.ec_str_p = &secp521r1_str_params,
	.priv_key = ecdsa_secp521r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_secp521r1_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_secp521r1_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_secp521r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_secp521r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP521R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_BRAINPOOLP256R1
#define ECDSA_SHA256_BRAINPOOLP256R1_SELF_TEST

/* ECDSA Brainpoolp256r1 test vectors */

static int ecdsa_nn_random_brainpoolp256r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp256r1 signature
	 */
	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_brainpoolp256r1_test_vectors_priv_key[] = {
	0x00, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 ecdsa_brainpoolp256r1_test_vectors_expected_sig[] = {
	0xa3, 0xfa, 0x53, 0x9a, 0xc2, 0xcf, 0xfb, 0xd5,
	0xc5, 0xad, 0xb6, 0x64, 0x8c, 0xb3, 0xb5, 0xe3,
	0x6a, 0x08, 0x7d, 0xcc, 0xd5, 0xda, 0xae, 0x8a,
	0x05, 0x87, 0xac, 0x37, 0x88, 0x78, 0x79, 0xb5,
	0xa7, 0xff, 0x72, 0xa9, 0xd8, 0x5c, 0x6e, 0xdd,
	0x48, 0x56, 0x2e, 0x8c, 0xd8, 0xf7, 0x6d, 0xab,
	0xe3, 0xdb, 0xc3, 0x96, 0x05, 0x69, 0xdf, 0x5d,
	0x13, 0xf9, 0x83, 0x5c, 0xf4, 0xca, 0x72, 0x3b
};

static const ec_test_case ecdsa_brainpoolp256r1_test_case = {
	.name = "ECDSA-SHA256/brainpoolp256r1",
	.ec_str_p = &brainpoolp256r1_str_params,
	.priv_key = ecdsa_brainpoolp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_brainpoolp256r1_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_brainpoolp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_brainpoolp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_brainpoolp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_BRAINPOOLP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_BRAINPOOLP384R1
#define ECDSA_SHA384_BRAINPOOLP384R1_SELF_TEST

/* ECDSA Brainpoolp384r1 test vectors */

static int ecdsa_nn_random_brainpoolp384r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp384r1 signature
	 */
	const u8 k_buf[] = {
		0x87, 0x77, 0x2f, 0x94, 0x81, 0x26, 0xe2, 0x38,
		0x04, 0xec, 0x30, 0x71, 0x34, 0x94, 0x74, 0x84,
		0x2c, 0x21, 0x08, 0x6b, 0xf2, 0x91, 0xfb, 0x33,
		0x02, 0xc2, 0x54, 0x82, 0x6c, 0x14, 0xc0, 0xa8,
		0x5d, 0x66, 0x03, 0x0b, 0xe9, 0x0b, 0xfe, 0x51,
		0xd4, 0x53, 0x0a, 0x55, 0xfd, 0x61, 0x51, 0x21
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_brainpoolp384r1_test_vectors_priv_key[] = {
	0x22, 0xde, 0x1b, 0x4e, 0x46, 0xdf, 0x58, 0x37,
	0xe0, 0x08, 0xa2, 0x67, 0x7a, 0xc6, 0x25, 0x6a,
	0xf3, 0x0e, 0xd1, 0xd9, 0x58, 0xe8, 0xa4, 0xee,
	0xd5, 0x02, 0x0a, 0xef, 0xbb, 0xf7, 0x90, 0x6e,
	0x35, 0x99, 0x43, 0xda, 0x81, 0xcc, 0x37, 0xdf,
	0xcf, 0x03, 0x83, 0x64, 0xf3, 0x96, 0x13, 0x5a
};

static const u8 ecdsa_brainpoolp384r1_test_vectors_expected_sig[] = {
	0x84, 0xa4, 0xe3, 0xa5, 0xdb, 0x1a, 0x93, 0x1f,
	0x77, 0x3e, 0x55, 0xfb, 0xa3, 0x69, 0xc3, 0x6d,
	0x21, 0x92, 0xb4, 0x7d, 0xf8, 0x5d, 0x05, 0x20,
	0x80, 0x23, 0xdb, 0xaa, 0x15, 0x5f, 0xfe, 0xcc,
	0xf9, 0xc0, 0x8f, 0xf1, 0xba, 0x79, 0xe0, 0x13,
	0x1e, 0xb4, 0x2c, 0x13, 0xa8, 0x17, 0x99, 0xd6,

	0x06, 0x9e, 0x3e, 0x95, 0x40, 0x80, 0xba, 0xef,
	0xfe, 0x3b, 0x73, 0xb1, 0xb9, 0x35, 0xa5, 0x60,
	0x27, 0x21, 0xa1, 0xd0, 0x0e, 0xe1, 0x12, 0xc0,
	0x86, 0x0b, 0xfc, 0xea, 0x93, 0x37, 0x33, 0xca,
	0x48, 0xd3, 0xae, 0xd8, 0xf7, 0x02, 0xdc, 0x86,
	0x2b, 0x18, 0x2b, 0x12, 0x8c, 0x8f, 0x3a, 0x4c,
};

static const ec_test_case ecdsa_brainpoolp384r1_test_case = {
	.name = "ECDSA-SHA384/brainpoolp384r1",
	.ec_str_p = &brainpoolp384r1_str_params,
	.priv_key = ecdsa_brainpoolp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_brainpoolp384r1_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_brainpoolp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_brainpoolp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_brainpoolp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_BRAINPOOLP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_BRAINPOOLP512R1
#define ECDSA_SHA512_BRAINPOOLP512R1_SELF_TEST

/* ECDSA Brainpoolp512r1 test vectors */

static int ecdsa_nn_random_brainpoolp512r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp512r1 signature
	 */
	const u8 k_buf[] = {
		0x00, 0xC1, 0xC2, 0xB3, 0x05, 0x41, 0x9F, 0x5A,
		0x41, 0x34, 0x4D, 0x7E, 0x43, 0x59, 0x93, 0x3D,
		0x73, 0x40, 0x96, 0xF5, 0x56, 0x19, 0x7A, 0x9B,
		0x24, 0x43, 0x42, 0xB8, 0xB6, 0x2F, 0x46, 0xF9,
		0x37, 0x37, 0x78, 0xF9, 0xDE, 0x6B, 0x64, 0x97,
		0xB1, 0xEF, 0x82, 0x5F, 0xF2, 0x4F, 0x42, 0xF9,
		0xB4, 0xA4, 0xBD, 0x73, 0x82, 0xCF, 0xC3, 0x37,
		0x8A, 0x54, 0x0B, 0x1B, 0x7F, 0x0C, 0x1B, 0x95
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_brainpoolp512r1_test_vectors_priv_key[] = {
	0x01, 0xA3, 0x40, 0x94, 0x51, 0xDC, 0xAB, 0x0A,
	0x0E, 0xAD, 0x45, 0x49, 0x51, 0x12, 0xA3, 0xD8,
	0x13, 0xC1, 0x7B, 0xFD, 0x34, 0xBD, 0xF8, 0xC1,
	0x20, 0x9D, 0x7D, 0xF5, 0x84, 0x91, 0x20, 0x59,
	0x77, 0x79, 0x06, 0x0A, 0x7F, 0xF9, 0xD7, 0x04,
	0xAD, 0xF7, 0x8B, 0x57, 0x0F, 0xFA, 0xD6, 0xF0,
	0x62, 0xE9, 0x5C, 0x7E, 0x0C, 0x5D, 0x54, 0x81,
	0xC5, 0xB1, 0x53, 0xB4, 0x8B, 0x37, 0x5F, 0xA1
};

static const u8 ecdsa_brainpoolp512r1_test_vectors_expected_sig[] = {
	0x34, 0x83, 0xdd, 0x62, 0x64, 0xab, 0x50, 0xc3,
	0x7c, 0x01, 0x89, 0x96, 0xf2, 0x18, 0x3f, 0x31,
	0x19, 0xb4, 0xd5, 0x8a, 0x71, 0xba, 0x96, 0xee,
	0x4f, 0x1d, 0x69, 0x0d, 0x06, 0xff, 0x04, 0xe7,
	0x62, 0xc3, 0x7c, 0x87, 0xac, 0xcb, 0x1b, 0xcb,
	0xc6, 0xe1, 0x50, 0x57, 0x7f, 0xd9, 0xa5, 0xd5,
	0xfc, 0x64, 0xc9, 0xaa, 0x21, 0x2f, 0x43, 0xac,
	0x36, 0xeb, 0xaa, 0xb1, 0x10, 0x5b, 0xe0, 0x37,
	0xa9, 0x70, 0xb3, 0xd3, 0x80, 0x1d, 0x51, 0x3d,
	0x38, 0x9d, 0x9c, 0x80, 0x76, 0xd9, 0x99, 0x6f,
	0xad, 0x0f, 0xc7, 0xf8, 0x67, 0x40, 0xc2, 0x28,
	0x1c, 0xe3, 0xfb, 0xae, 0x71, 0x82, 0xd3, 0xe8,
	0x5c, 0xb2, 0x24, 0xec, 0x88, 0xb5, 0x81, 0x2a,
	0xb1, 0x15, 0x32, 0xfd, 0x01, 0x7c, 0x94, 0xed,
	0xd7, 0x8c, 0xa6, 0x32, 0x1e, 0x24, 0x94, 0xa7,
	0x38, 0xd0, 0xea, 0xc8, 0x34, 0x05, 0xb9, 0x9b
};

static const ec_test_case ecdsa_brainpoolp512r1_test_case = {
	.name = "ECDSA-SHA512/brainpoolp512r1",
	.ec_str_p = &brainpoolp512r1_str_params,
	.priv_key = ecdsa_brainpoolp512r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_brainpoolp512r1_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_brainpoolp512r1_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_brainpoolp512r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_brainpoolp512r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP512R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_FRP256V1
#define ECDSA_SHA256_FRP256V1_SELF_TEST

/* ECDSA frp256v1 test vectors */

static int ecdsa_nn_random_frp256v1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for frp256v1 signature
	 */
	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecdsa_frp256v1_test_vectors_priv_key[] = {
	0x00, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 ecdsa_frp256v1_test_vectors_expected_sig[] = {
	0x22, 0xb1, 0xd0, 0x0d, 0xc8, 0x28, 0xa5, 0xcb,
	0xa2, 0x64, 0x31, 0x9f, 0x05, 0x93, 0x3a, 0xc8,
	0xed, 0xe6, 0x32, 0x3d, 0x69, 0xd0, 0x2b, 0x31,
	0x4e, 0x10, 0xca, 0xa5, 0xf1, 0x3b, 0x29, 0x96,
	0xbd, 0xef, 0x27, 0xe6, 0x3c, 0x16, 0xee, 0x46,
	0xf8, 0x14, 0x22, 0x9d, 0x3c, 0xf2, 0x3f, 0x08,
	0xc7, 0x96, 0x26, 0x13, 0x55, 0xe7, 0xd4, 0x4c,
	0x9d, 0xd9, 0xd6, 0xba, 0x08, 0x57, 0x1f, 0xd8
};

static const ec_test_case ecdsa_frp256v1_test_case = {
	.name = "ECDSA-SHA256/frp256v1",
	.ec_str_p = &frp256v1_str_params,
	.priv_key = ecdsa_frp256v1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecdsa_frp256v1_test_vectors_priv_key),
	.nn_random = ecdsa_nn_random_frp256v1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECDSA,
	.exp_sig = ecdsa_frp256v1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecdsa_frp256v1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_FRP256V1 */
#endif /* WITH_HASH_SHA256 */
#endif /* WITH_SIG_ECDSA */

/*******************************************************************
 ************** ECKDSA tests ***************************************
 *******************************************************************/

#ifdef WITH_SIG_ECKCDSA
#ifdef WITH_HASH_SHA224
#ifdef WITH_CURVE_SECP224R1
#define ECKCDSA_SHA224_SECP224R1_SELF_TEST

/* ECKCDSA secp224r1 test vectors */

static int eckcdsa_nn_random_iso14888_3_secp224r1_test_vector(nn_t out,
							      nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x76, 0xA0, 0xAF, 0xC1, 0x86, 0x46, 0xD1, 0xB6,
		0x20, 0xA0, 0x79, 0xFB, 0x22, 0x38, 0x65, 0xA7,
		0xBC, 0xB4, 0x47, 0xF3, 0xC0, 0x3A, 0x35, 0xD8,
		0x78, 0xEA, 0x4C, 0xDA
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_secp224r1_test_vectors_priv_key[] = {
	0x56, 0x2A, 0x6F, 0x64, 0xE1, 0x62, 0xFF, 0xCB,
	0x51, 0xCD, 0x47, 0x07, 0x77, 0x4A, 0xE3, 0x66,
	0x81, 0xB6, 0xCE, 0xF2, 0x05, 0xFE, 0x5D, 0x43,
	0x91, 0x29, 0x56, 0xA2
};

static const u8 eckcdsa_secp224r1_test_vectors_expected_sig[] = {
	0xEE, 0xA5, 0x8C, 0x91, 0xE0, 0xCD, 0xCE, 0xB5,
	0x79, 0x9B, 0x00, 0xD2, 0x41, 0x2D, 0x92, 0x8F,
	0xDD, 0x23, 0x12, 0x2A, 0x1C, 0x2B, 0xDF, 0x43,
	0xC2, 0xF8, 0xDA, 0xFA, 0xAE, 0xBA, 0xB5, 0x3C,
	0x7A, 0x44, 0xA8, 0xB2, 0x2F, 0x35, 0xFD, 0xB9,
	0xDE, 0x26, 0x5F, 0x23, 0xB8, 0x9F, 0x65, 0xA6,
	0x9A, 0x8B, 0x7B, 0xD4, 0x06, 0x19, 0x11, 0xA6
};

static const ec_test_case eckcdsa_secp224r1_test_case = {
	.name = "ECKCDSA-SHA224/secp224r1",
	.ec_str_p = &secp224r1_str_params,
	.priv_key = eckcdsa_secp224r1_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_secp224r1_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_iso14888_3_secp224r1_test_vector,
	.hash_type = SHA224,
	.msg = "This is a sample message for EC-KCDSA implementation validation.",
	.msglen = 64,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_secp224r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(eckcdsa_secp224r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP224R1 */
#endif /* WITH_HASH_SHA224 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_SECP224R1
#define ECKCDSA_SHA256_SECP224R1_SELF_TEST

/* ECKCDSA secp224r1 w/ SHA-256 test vectors (specific for truncation test) */

static int eckcdsa_nn_random_iso14888_3_secp224r1_sha256_test_vector(nn_t out,
								     nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0xEE, 0xC7, 0x9D, 0x8D, 0x46, 0x48, 0xDF, 0x3A,
		0x83, 0x2A, 0x66, 0xE3, 0x77, 0x55, 0x37, 0xE0,
		0x00, 0xCC, 0x9B, 0x95, 0x7E, 0x13, 0x19, 0xC5,
		0xDB, 0x9D, 0xD4, 0xF7
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_secp224r1_sha256_test_vectors_priv_key[] = {
	0x61, 0x58, 0x58, 0x27, 0x44, 0x9D, 0xBC, 0x0E,
	0xC1, 0x61, 0xB2, 0xCF, 0x85, 0x75, 0xC9, 0xDF,
	0x14, 0x9F, 0x41, 0xDD, 0x02, 0x89, 0xBE, 0x4F,
	0xF1, 0x10, 0x77, 0x3D
};

static const u8 eckcdsa_secp224r1_sha256_test_vectors_expected_sig[] = {
	0x64, 0xB4, 0x9E, 0x97, 0x7E, 0x65, 0x34, 0xF8,
	0x77, 0xCB, 0x68, 0xA3, 0x80, 0x6F, 0x6A, 0x98,
	0x93, 0x11, 0xCE, 0xAA, 0x8A, 0x64, 0xA0, 0x55,
	0x80, 0x77, 0xC0, 0x4B,

	0xAF, 0xF2, 0x3D, 0x40, 0xB1, 0x77, 0x95, 0x11,
	0x51, 0xBE, 0x32, 0xF6, 0x56, 0x1B, 0x1B, 0x73,
	0x9E, 0x3E, 0x8F, 0x82, 0x2C, 0xC5, 0x2D, 0x4C,
	0xB3, 0x90, 0x9A, 0x93
};

static const ec_test_case eckcdsa_secp224r1_sha256_test_case = {
	.name = "ECKCDSA-SHA256/secp224r1",
	.ec_str_p = &secp224r1_str_params,
	.priv_key = eckcdsa_secp224r1_sha256_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_secp224r1_sha256_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_iso14888_3_secp224r1_sha256_test_vector,
	.hash_type = SHA256,
	.msg = "This is a sample message for EC-KCDSA implementation validation.",
	.msglen = 64,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_secp224r1_sha256_test_vectors_expected_sig,
	.exp_siglen =
		sizeof(eckcdsa_secp224r1_sha256_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP224R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_SECP256R1
#define ECKCDSA_SHA256_SECP256R1_SELF_TEST

/* ECKCDSA secp256r1 test vectors */

static int eckcdsa_nn_random_iso14888_3_secp256r1_test_vector(nn_t out,
							      nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x71, 0xB8, 0x8F, 0x39, 0x89, 0x16, 0xDA, 0x9C,
		0x90, 0xF5, 0x55, 0xF1, 0xB5, 0x73, 0x2B, 0x7D,
		0xC6, 0x36, 0xB4, 0x9C, 0x63, 0x81, 0x50, 0xBA,
		0xC1, 0x1B, 0xF0, 0x5C, 0xFE, 0x16, 0x59, 0x6A
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_secp256r1_test_vectors_priv_key[] = {
	0x90, 0x51, 0xA2, 0x75, 0xAA, 0x4D, 0x98, 0x43,
	0x9E, 0xDD, 0xED, 0x13, 0xFA, 0x1C, 0x6C, 0xBB,
	0xCC, 0xE7, 0x75, 0xD8, 0xCC, 0x94, 0x33, 0xDE,
	0xE6, 0x9C, 0x59, 0x84, 0x8B, 0x35, 0x94, 0xDF
};

static const u8 eckcdsa_secp256r1_test_vectors_expected_sig[] = {
	0x0e, 0xdd, 0xf6, 0x80, 0x60, 0x12, 0x66, 0xee,
	0x1d, 0xa8, 0x3e, 0x55, 0xa6, 0xd9, 0x44, 0x5f,
	0xc7, 0x81, 0xda, 0xeb, 0x14, 0xc7, 0x65, 0xe7,
	0xe5, 0xd0, 0xcd, 0xba, 0xf1, 0xf1, 0x4a, 0x68,
	0x9b, 0x33, 0x34, 0x57, 0x66, 0x1c, 0x7c, 0xf7,
	0x41, 0xbd, 0xdb, 0xc0, 0x83, 0x55, 0x53, 0xdf,
	0xbb, 0x37, 0xee, 0x74, 0xf5, 0x3d, 0xb6, 0x99,
	0xe0, 0xa1, 0x77, 0x80, 0xc7, 0xb6, 0xf1, 0xd0
};

static const ec_test_case eckcdsa_secp256r1_test_case = {
	.name = "ECKCDSA-SHA256/secp256r1",
	.ec_str_p = &secp256r1_str_params,
	.priv_key = eckcdsa_secp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_secp256r1_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_iso14888_3_secp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "This is a sample message for EC-KCDSA implementation validation.",
	.msglen = 64,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_secp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(eckcdsa_secp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_SECP384R1
#define ECKCDSA_SHA384_SECP384R1_SELF_TEST

/* ECKCDSA secp384r1 test vectors */

static int eckcdsa_nn_random_secp384r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x08, 0x16, 0x2e, 0xf8, 0x24, 0xd2, 0xd5, 0x11,
		0x4e, 0x08, 0x61, 0xf3, 0x93, 0xb4, 0x6f, 0xb6,
		0x02, 0xa4, 0x95, 0xa2, 0xca, 0x17, 0x7f, 0x47,
		0xda, 0x4e, 0x6f, 0x2d, 0x23, 0xa1, 0x85, 0xc5,
		0xb8, 0x8f, 0x1b, 0x7e, 0x2c, 0xba, 0x8d, 0x77,
		0x37, 0x16, 0x09, 0x3a, 0xf5, 0x97, 0x9a, 0x7e
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_secp384r1_test_vectors_priv_key[] = {
	0x4f, 0xdb, 0xf2, 0xe7, 0x4e, 0x56, 0x81, 0x4e,
	0x4c, 0x2f, 0x40, 0x87, 0x90, 0x9c, 0xe9, 0xc4,
	0x73, 0x1b, 0xff, 0x06, 0x1a, 0x78, 0x82, 0x73,
	0x84, 0x1b, 0xf5, 0x68, 0xce, 0x63, 0x21, 0x3e,
	0x47, 0x04, 0x04, 0xea, 0x3e, 0xb6, 0x78, 0xc5,
	0x00, 0xba, 0x37, 0x92, 0xf2, 0x4a, 0x7a, 0x51
};

static const u8 eckcdsa_secp384r1_test_vectors_expected_sig[] = {

	0xd0, 0x37, 0x26, 0x47, 0x44, 0xe3, 0x8a, 0x38,
	0xca, 0xcc, 0xdc, 0x8f, 0xcb, 0x08, 0xd8, 0xcb,
	0x52, 0x18, 0x7d, 0x38, 0x3c, 0xc0, 0xb6, 0x29,
	0xf6, 0x27, 0x71, 0xd9, 0x3f, 0x5a, 0xfc, 0xd3,
	0xc1, 0x44, 0xc1, 0x53, 0xef, 0x06, 0xd9, 0x9a,
	0xc9, 0x20, 0x58, 0x81, 0xde, 0x03, 0xd1, 0x71,
	0xf1, 0xff, 0xc1, 0x43, 0xb9, 0x4e, 0x44, 0xdc,
	0xec, 0xd1, 0x61, 0xfd, 0xc5, 0x27, 0x0a, 0x32,
	0x71, 0x3e, 0x59, 0x38, 0xfb, 0x35, 0x60, 0x0e,
	0xdd, 0xd0, 0x0c, 0x63, 0x89, 0xfb, 0x33, 0xb0,
	0x3e, 0x77, 0xd2, 0xf7, 0x69, 0x4a, 0x04, 0x08,
	0x5a, 0xf9, 0x85, 0x31, 0xdf, 0xe6, 0xf1, 0x73
};

static const ec_test_case eckcdsa_secp384r1_test_case = {
	.name = "ECKCDSA-SHA384/secp384r1",
	.ec_str_p = &secp384r1_str_params,
	.priv_key = eckcdsa_secp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_secp384r1_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_secp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "This is a sample message for EC-KCDSA implementation validation.",
	.msglen = 64,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_secp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(eckcdsa_secp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_SECP256R1
#define ECKCDSA_SHA512_SECP256R1_SELF_TEST

/*
 * ECKCDSA secp256r1 test vector using SHA512. This test vector is intended
 * to test truncation (left shift) steps in signature and verification
 * code when the output size of the hash function is larger than the
 * bit size of q (order of the group generated by the base point).
 */

static int eckcdsa_nn_random_secp256r1_sha512_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_secp256r1_sha512_test_vectors_priv_key[] = {
	0xDC, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 eckcdsa_secp256r1_sha512_test_vectors_expected_sig[] = {
	0x70, 0x02, 0x1d, 0x6d, 0x5a, 0x5e, 0x99, 0x59,
	0x43, 0x18, 0xa3, 0xba, 0x8d, 0x27, 0xa5, 0x39,
	0xa5, 0x73, 0x76, 0x80, 0x73, 0xc8, 0x4d, 0xa0,
	0xdb, 0xcc, 0xf9, 0x24, 0x9a, 0xfe, 0x78, 0x54,

	0xb6, 0x97, 0x28, 0xe1, 0x7b, 0x9e, 0x85, 0xb0,
	0x30, 0x05, 0x8f, 0x28, 0x54, 0xbf, 0x34, 0xd1,
	0xdd, 0x1d, 0xf6, 0x7a, 0x02, 0x64, 0x23, 0x76,
	0x9e, 0xb4, 0xfd, 0x0f, 0x4c, 0x2e, 0x15, 0x6b
};

static const ec_test_case eckcdsa_secp256r1_sha512_test_case = {
	.name = "ECKCDSA-SHA512/secp256r1",
	.ec_str_p = &secp256r1_str_params,
	.priv_key = eckcdsa_secp256r1_sha512_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_secp256r1_sha512_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_secp256r1_sha512_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_secp256r1_sha512_test_vectors_expected_sig,
	.exp_siglen =
		sizeof(eckcdsa_secp256r1_sha512_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP256R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_SECP521R1
#define ECKCDSA_SHA512_SECP521R1_SELF_TEST

/* ECKCDSA secp521r1 test vectors */

static int eckcdsa_nn_random_secp521r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x01, 0xb6, 0xfa, 0xf4, 0x5f, 0xb8, 0x6f, 0x57,
		0x9c, 0x4b, 0x54, 0xae, 0xc7, 0xa7, 0xf2, 0x95,
		0x6e, 0x4d, 0x93, 0x94, 0xb3, 0x62, 0x82, 0x0a,
		0xa5, 0x3f, 0x1b, 0x6c, 0x07, 0x3b, 0xbe, 0xc0,
		0x24, 0x33, 0x14, 0xdb, 0x11, 0xb6, 0x20, 0x21,
		0x35, 0x50, 0x20, 0xe8, 0x05, 0xad, 0x8e, 0x6d,
		0x0e, 0xa2, 0x7c, 0x04, 0x35, 0x9a, 0xd3, 0xf8,
		0x44, 0xd6, 0x5b, 0xbb, 0x73, 0xb7, 0x91, 0xd5,
		0x9a, 0x1c
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_secp521r1_test_vectors_priv_key[] = {
	0x01, 0x65, 0xe1, 0xf1, 0xf2, 0xe7, 0xae, 0x6b,
	0x50, 0x2a, 0x19, 0x4e, 0xf0, 0x3a, 0xd0, 0x50,
	0xe3, 0xa5, 0xae, 0x31, 0xfc, 0xb4, 0xf7, 0x0d,
	0x58, 0xf5, 0xc6, 0xdf, 0x28, 0x66, 0x71, 0x7f,
	0x44, 0x09, 0xe7, 0x78, 0x02, 0x4d, 0xe5, 0x2b,
	0xef, 0xc9, 0x6d, 0xb9, 0xfb, 0xba, 0x21, 0x30,
	0x6d, 0x12, 0x5a, 0x27, 0xcf, 0xcd, 0xce, 0x91,
	0x00, 0x09, 0x79, 0x63, 0xbf, 0x0c, 0x40, 0xe6,
	0x96, 0xdc
};

static const u8 eckcdsa_secp521r1_test_vectors_expected_sig[] = {
	0x90, 0x52, 0xbb, 0xc4, 0xdf, 0xe8, 0x80, 0xfb,
	0x25, 0xcb, 0xa7, 0xa7, 0x47, 0xa0, 0xd0, 0xe9,
	0x0e, 0x10, 0x63, 0x5b, 0xe1, 0xc2, 0x03, 0xc0,
	0x5e, 0x6d, 0x1c, 0x7d, 0x3d, 0x07, 0x1c, 0xe0,
	0x5a, 0xd7, 0xd5, 0x1c, 0x38, 0xe6, 0xa5, 0x7a,
	0x85, 0x58, 0x01, 0x79, 0xb0, 0x32, 0x2e, 0x66,
	0xdc, 0x95, 0x7f, 0xe6, 0xd3, 0xf3, 0xa3, 0x0a,
	0x05, 0x37, 0xfb, 0xf0, 0xd1, 0x24, 0xb4, 0x49,
	0x01, 0x36, 0x43, 0x43, 0xdf, 0xd8, 0x52, 0x64,
	0x15, 0xe0, 0x25, 0x79, 0xd5, 0xe1, 0x11, 0xee,
	0x5a, 0x8e, 0x50, 0xf3, 0x41, 0x90, 0x3d, 0x5f,
	0x28, 0xa0, 0x78, 0xac, 0x17, 0x82, 0xfe, 0x7d,
	0xbf, 0xf7, 0xd8, 0xcc, 0x3f, 0x51, 0x17, 0x76,
	0x59, 0x37, 0xd4, 0x92, 0xc3, 0x6a, 0x00, 0xb3,
	0x0b, 0x70, 0xb1, 0xdb, 0xd3, 0x95, 0x34, 0x02,
	0x2a, 0xa6, 0x50, 0x7a, 0xfd, 0xe6, 0x15, 0xbe,
	0xec, 0xba
};

static const ec_test_case eckcdsa_secp521r1_test_case = {
	.name = "ECKCDSA-SHA512/secp521r1",
	.ec_str_p = &secp521r1_str_params,
	.priv_key = eckcdsa_secp521r1_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_secp521r1_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_secp521r1_test_vector,
	.hash_type = SHA512,
	.msg = "This is a sample message for EC-KCDSA implementation validation.",
	.msglen = 64,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_secp521r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(eckcdsa_secp521r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP521R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_BRAINPOOLP256R1
#define ECKCDSA_SHA256_BRAINPOOLP256R1_SELF_TEST

/* ECKCDSA brainpoolp256r1 test vectors */

static int eckcdsa_nn_random_brainpoolp256r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x09, 0x8a, 0x5b, 0x7c, 0xfa, 0x31, 0x7b, 0x79,
		0x6a, 0xf4, 0x46, 0xc4, 0x0e, 0x3e, 0xb5, 0x28,
		0x79, 0x03, 0x42, 0x2c, 0x56, 0x62, 0x49, 0x78,
		0xd9, 0x02, 0xc5, 0x9b, 0x7a, 0x92, 0xfe, 0x9b
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_brainpoolp256r1_test_vectors_priv_key[] = {
	0xa1, 0x45, 0x67, 0x88, 0x20, 0x4e, 0x69, 0xba,
	0xf3, 0x84, 0x88, 0xca, 0xe7, 0x77, 0x63, 0xca,
	0xde, 0xe5, 0xd0, 0xd9, 0x1a, 0xa0, 0xe5, 0xe6,
	0x0d, 0x19, 0xb0, 0x0d, 0x3a, 0x67, 0xed, 0x48
};

static const u8 eckcdsa_brainpoolp256r1_test_vectors_expected_sig[] = {
	0xec, 0xc8, 0x2e, 0x85, 0x66, 0x90, 0x05, 0xaf,
	0x07, 0x7c, 0x67, 0xde, 0xba, 0xea, 0x8e, 0xbb,
	0x33, 0x26, 0x14, 0xe6, 0x61, 0xef, 0xc7, 0xff,
	0xd1, 0xf4, 0x52, 0xe1, 0xc4, 0x42, 0x23, 0xcd,

	0x2f, 0x53, 0xb0, 0x9d, 0xdf, 0x97, 0xa8, 0x22,
	0x42, 0x55, 0x29, 0x65, 0xe1, 0xa2, 0x09, 0x27,
	0x8b, 0xe1, 0x4d, 0xde, 0x16, 0xfd, 0x54, 0xbc,
	0x16, 0xf1, 0xf0, 0x9d, 0x28, 0xe7, 0xfd, 0x11,
};

static const ec_test_case eckcdsa_brainpoolp256r1_test_case = {
	.name = "ECKCDSA-SHA256/brainpoolp256r1",
	.ec_str_p = &brainpoolp256r1_str_params,
	.priv_key = eckcdsa_brainpoolp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_brainpoolp256r1_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_brainpoolp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "brainpoolP256r1",
	.msglen = 15,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_brainpoolp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(eckcdsa_brainpoolp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_BRAINPOOLP384R1
#define ECKCDSA_SHA384_BRAINPOOLP384R1_SELF_TEST

/* ECKCDSA brainpoolp384r1 test vectors */

static int eckcdsa_nn_random_brainpoolp384r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x07, 0x0f, 0x21, 0xec, 0xa6, 0xef, 0x53, 0xd8,
		0x81, 0x75, 0x33, 0xe8, 0x9c, 0xb6, 0xb7, 0x4f,
		0xb8, 0x2f, 0x4c, 0x23, 0x00, 0x80, 0x8c, 0xab,
		0x19, 0x9e, 0x6f, 0xfe, 0x96, 0xe6, 0x1d, 0x29,
		0x22, 0x7d, 0x76, 0x43, 0x3c, 0x34, 0x33, 0xc3,
		0x4d, 0x18, 0x5a, 0x8f, 0x01, 0x21, 0x38, 0x5b
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_brainpoolp384r1_test_vectors_priv_key[] = {
	0x21, 0x24, 0x1d, 0x99, 0x52, 0xcb, 0x23, 0x67,
	0x21, 0xcd, 0x02, 0xd4, 0xd5, 0xe0, 0xfd, 0xbe,
	0x22, 0xd7, 0xab, 0x69, 0xaf, 0xc1, 0x22, 0x12,
	0x16, 0xc6, 0x09, 0x96, 0x05, 0x14, 0xec, 0x3b,
	0xe6, 0x5f, 0x52, 0x73, 0xdd, 0x06, 0x9f, 0xdf,
	0x63, 0x7e, 0xd9, 0x5c, 0x6c, 0xe5, 0x4f, 0x08
};

static const u8 eckcdsa_brainpoolp384r1_test_vectors_expected_sig[] = {
	0xe4, 0x9a, 0x76, 0x4e, 0xeb, 0x13, 0x1e, 0xa1,
	0xba, 0x0c, 0xef, 0xa5, 0xd6, 0x79, 0xa1, 0xf9,
	0x63, 0x5b, 0x93, 0x41, 0xa6, 0xb1, 0x3c, 0x40,
	0xbb, 0xd5, 0x5c, 0x38, 0x20, 0xe5, 0x09, 0xfa,
	0x15, 0x8c, 0x5a, 0x1a, 0xf1, 0xb9, 0x29, 0x26,
	0x8b, 0xe1, 0x06, 0xbe, 0xb2, 0x08, 0xc4, 0xc9,

	0x3b, 0xa4, 0x36, 0xa6, 0x8e, 0x1e, 0xe9, 0x25,
	0x97, 0xdd, 0x12, 0x5e, 0x6e, 0xfb, 0xe5, 0x28,
	0x18, 0x47, 0x41, 0x75, 0xdb, 0x2f, 0x47, 0xa5,
	0xd0, 0xd0, 0x2b, 0xd7, 0xc0, 0x81, 0x43, 0xf1,
	0xc3, 0xe9, 0xbf, 0x2f, 0xf1, 0x4c, 0x18, 0x46,
	0xff, 0xbc, 0xdf, 0x39, 0x93, 0x88, 0x59, 0x6b
};

static const ec_test_case eckcdsa_brainpoolp384r1_test_case = {
	.name = "ECKCDSA-SHA384/brainpoolp384r1",
	.ec_str_p = &brainpoolp384r1_str_params,
	.priv_key = eckcdsa_brainpoolp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_brainpoolp384r1_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_brainpoolp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "brainpoolP384r1",
	.msglen = 15,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_brainpoolp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(eckcdsa_brainpoolp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_BRAINPOOLP512R1
#define ECKCDSA_SHA512_BRAINPOOLP512R1_SELF_TEST

/* ECKCDSA brainpoolp512r1 test vectors */

static int eckcdsa_nn_random_brainpoolp512r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x46, 0x3d, 0x92, 0x92, 0x3f, 0xe3, 0x37, 0xd8,
		0xe5, 0xfc, 0x5a, 0xbe, 0x76, 0xf6, 0x3d, 0x9e,
		0xec, 0x9a, 0xaf, 0x57, 0x48, 0x7d, 0xef, 0xe1,
		0xfa, 0xe1, 0x76, 0x5d, 0x7e, 0xf7, 0x2f, 0x5d,
		0x10, 0x64, 0xe1, 0xbe, 0x65, 0x68, 0x5c, 0x08,
		0x40, 0x59, 0x7e, 0x98, 0x79, 0x75, 0x1b, 0xda,
		0xc8, 0x42, 0xcf, 0xbb, 0xb0, 0x8e, 0x5c, 0x42,
		0x8d, 0x1c, 0xa6, 0xe7, 0x63, 0x4b, 0x37, 0xea
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_brainpoolp512r1_test_vectors_priv_key[] = {
	0xa9, 0xeb, 0x22, 0xd4, 0x1a, 0x8a, 0xa1, 0x0d,
	0xdf, 0xec, 0xd3, 0x1f, 0x93, 0xc8, 0xaf, 0x9a,
	0x38, 0x7f, 0xf5, 0x29, 0xb9, 0x81, 0xeb, 0x8e,
	0x3a, 0x73, 0xbe, 0x90, 0x70, 0x51, 0x99, 0x50,
	0xc2, 0x8d, 0x81, 0xce, 0x31, 0x84, 0x79, 0x67,
	0x18, 0xbf, 0xe4, 0x39, 0x17, 0x8c, 0x38, 0xe9,
	0xa8, 0x10, 0x99, 0xf7, 0xd2, 0x60, 0x7c, 0xbf,
	0x53, 0xa7, 0x12, 0xc8, 0xb8, 0xb3, 0x35, 0xcb
};

static const u8 eckcdsa_brainpoolp512r1_test_vectors_expected_sig[] = {
	0xc0, 0x0d, 0x74, 0xbe, 0x19, 0x03, 0x45, 0xe3,
	0xd9, 0x73, 0x90, 0xfc, 0xcf, 0xa1, 0x27, 0x1d,
	0x5e, 0x8d, 0x8f, 0xfb, 0x15, 0x31, 0x49, 0xf6,
	0xa7, 0xc5, 0x69, 0xa8, 0x78, 0x63, 0x7b, 0xf5,
	0xcd, 0x02, 0x78, 0xf9, 0x2b, 0xce, 0x8c, 0x64,
	0x4c, 0xd3, 0xd4, 0x25, 0xc9, 0x50, 0x14, 0x34,
	0xfe, 0xdb, 0x1e, 0xc8, 0xf6, 0x55, 0x60, 0x79,
	0xce, 0x51, 0xf8, 0xa4, 0xcc, 0x2e, 0xa9, 0xe3,

	0x45, 0xd8, 0xcf, 0x97, 0x8c, 0x15, 0x58, 0xbd,
	0x11, 0xba, 0xbe, 0x41, 0x97, 0xe1, 0x06, 0x19,
	0x0d, 0x60, 0x58, 0x53, 0x5e, 0xbd, 0xf4, 0x00,
	0x06, 0xbd, 0xf2, 0x2b, 0x11, 0x2c, 0x87, 0x7e,
	0xe4, 0x4c, 0xa2, 0xdc, 0xf9, 0x0a, 0xfe, 0x21,
	0x04, 0xbd, 0xf6, 0xd8, 0xb9, 0x07, 0xaa, 0x4c,
	0x75, 0xbd, 0x2e, 0x55, 0x65, 0xa3, 0xca, 0x8f,
	0xae, 0xb1, 0x10, 0x7e, 0x85, 0x39, 0xe2, 0x1f
};

static const ec_test_case eckcdsa_brainpoolp512r1_test_case = {
	.name = "ECKCDSA-SHA512/brainpoolp512r1",
	.ec_str_p = &brainpoolp512r1_str_params,
	.priv_key = eckcdsa_brainpoolp512r1_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_brainpoolp512r1_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_brainpoolp512r1_test_vector,
	.hash_type = SHA512,
	.msg = "brainpoolP512r1",
	.msglen = 15,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_brainpoolp512r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(eckcdsa_brainpoolp512r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP512R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_FRP256V1
#define ECKCDSA_SHA256_FRP256V1_SELF_TEST

/* ECKCDSA frp256v1 test vectors */

static int eckcdsa_nn_random_frp256v1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x71, 0xB8, 0x8F, 0x39, 0x89, 0x16, 0xDA, 0x9C,
		0x90, 0xF5, 0x55, 0xF1, 0xB5, 0x73, 0x2B, 0x7D,
		0xC6, 0x36, 0xB4, 0x9C, 0x63, 0x81, 0x50, 0xBA,
		0xC1, 0x1B, 0xF0, 0x5C, 0xFE, 0x16, 0x59, 0x6A
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 eckcdsa_frp256v1_test_vectors_priv_key[] = {
	0x90, 0x51, 0xA2, 0x75, 0xAA, 0x4D, 0x98, 0x43,
	0x9E, 0xDD, 0xED, 0x13, 0xFA, 0x1C, 0x6C, 0xBB,
	0xCC, 0xE7, 0x75, 0xD8, 0xCC, 0x94, 0x33, 0xDE,
	0xE6, 0x9C, 0x59, 0x84, 0x8B, 0x35, 0x94, 0xDF
};

static const u8 eckcdsa_frp25v1_test_vectors_expected_sig[] = {
	0x37, 0x58, 0xd0, 0x27, 0x18, 0x11, 0x16, 0xe9,
	0x2b, 0xaf, 0x70, 0xbf, 0x16, 0x48, 0x64, 0x34,
	0x9e, 0x75, 0x60, 0x3e, 0x89, 0xa4, 0xc6, 0xc1,
	0xbd, 0xb8, 0x01, 0x52, 0x2d, 0xca, 0x20, 0x31,
	0x2d, 0x20, 0x6f, 0x3e, 0x70, 0x5e, 0x5e, 0xfc,
	0x32, 0xd4, 0x3c, 0xb0, 0xe1, 0xf7, 0xb2, 0xc7,
	0x0c, 0xc2, 0xb0, 0x26, 0x93, 0xcb, 0x69, 0xb7,
	0x29, 0xb7, 0x97, 0x5c, 0x67, 0xd6, 0xa3, 0xe6
};

static const ec_test_case eckcdsa_frp256v1_test_case = {
	.name = "ECKCDSA-SHA256/frp256v1",
	.ec_str_p = &frp256v1_str_params,
	.priv_key = eckcdsa_frp256v1_test_vectors_priv_key,
	.priv_key_len = sizeof(eckcdsa_frp256v1_test_vectors_priv_key),
	.nn_random = eckcdsa_nn_random_frp256v1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECKCDSA,
	.exp_sig = eckcdsa_frp25v1_test_vectors_expected_sig,
	.exp_siglen = sizeof(eckcdsa_frp25v1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_FRP256V1 */
#endif /* WITH_HASH_SHA256 */
#endif /* WITH_SIG_ECKCDSA */

/*******************************************************************
 ************** ECSDSA tests ***************************************
 *******************************************************************/

#ifdef WITH_SIG_ECSDSA
#ifdef WITH_HASH_SHA224
#ifdef WITH_CURVE_SECP224R1
#define ECSDSA_SHA224_SECP224R1_SELF_TEST

/* ECSDSA secp224r1 test vectors */

static int ecsdsa_nn_random_secp224r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x89, 0x99, 0x80, 0x6e, 0x0f, 0x68, 0x10, 0x9d,
		0x31, 0x56, 0x1b, 0x41, 0xbf, 0x1a, 0x56, 0x6e,
		0xfb, 0x13, 0x2c, 0x3e, 0xd7, 0x1e, 0x29, 0x5e,
		0x5d, 0x9b, 0x2e, 0x68
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecsdsa_secp224r1_test_vectors_priv_key[] = {
	0xda, 0x71, 0x2d, 0x9b, 0x6f, 0xdb, 0xf7, 0x8f,
	0x92, 0x51, 0xcc, 0x3a, 0xe4, 0xc7, 0xce, 0xd3,
	0x87, 0x1a, 0x19, 0x9e, 0xf5, 0x91, 0xd1, 0x23,
	0xa9, 0x36, 0x44, 0x57
};

static const u8 ecsdsa_secp224r1_test_vectors_expected_sig[] = {
	0x41, 0x69, 0xd3, 0xec, 0x93, 0x7b, 0x15, 0xce,
	0x15, 0x34, 0x53, 0x8d, 0x37, 0xe8, 0xc5, 0x7e,
	0x12, 0x1f, 0xd5, 0x9f, 0x90, 0x1b, 0x5e, 0xf8,
	0xd0, 0x0d, 0x62, 0x04,
	0xf0, 0xfe, 0xbe, 0x28, 0x36, 0xb0, 0x87, 0xec,
	0x44, 0x98, 0x94, 0x7a, 0x6c, 0x4c, 0x8a, 0xea,
	0x5a, 0x5a, 0x19, 0x8a, 0xce, 0x95, 0x2f, 0x75,
	0x07, 0x37, 0x23, 0x1b
};

static const ec_test_case ecsdsa_secp224r1_test_case = {
	.name = "ECSDSA-SHA224/secp224r1",
	.ec_str_p = &secp224r1_str_params,
	.priv_key = ecsdsa_secp224r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecsdsa_secp224r1_test_vectors_priv_key),
	.nn_random = ecsdsa_nn_random_secp224r1_test_vector,
	.hash_type = SHA224,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECSDSA,
	.exp_sig = ecsdsa_secp224r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecsdsa_secp224r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP224R1 */
#endif /* WITH_HASH_SHA224 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_SECP256R1
#define ECSDSA_SHA256_SECP256R1_SELF_TEST

/* ECSDSA secp256r1 test vectors */

static int ecsdsa_nn_random_iso14888_3_secp256r1_test_vector(nn_t out,
							     nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0xDE, 0x7E, 0x0E, 0x5E, 0x66, 0x3F, 0x24, 0x18,
		0x34, 0x14, 0xB7, 0xC7, 0x2F, 0x24, 0x54, 0x6B,
		0x81, 0xE9, 0xE5, 0xF4, 0x10, 0xBE, 0xBF, 0x26,
		0xF3, 0xCA, 0x5F, 0xA8, 0x2F, 0x51, 0x92, 0xC8
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecsdsa_secp256r1_test_vectors_priv_key[] = {
	0x52, 0x02, 0xA3, 0xD8, 0xAC, 0xAF, 0x69, 0x09,
	0xD1, 0x2C, 0x9A, 0x77, 0x4C, 0xD8, 0x86, 0xF9,
	0xFB, 0xA6, 0x11, 0x37, 0xFF, 0xD3, 0xE8, 0xE7,
	0x6A, 0xED, 0x36, 0x3F, 0xB4, 0x7A, 0xC4, 0x92
};

static const u8 ecsdsa_secp256r1_test_vectors_expected_sig[] = {
	0x5A, 0x79, 0xA0, 0xAA, 0x9B, 0x24, 0x1E, 0x38,
	0x1A, 0x59, 0x4B, 0x22, 0x05, 0x54, 0xD0, 0x96,
	0xA5, 0xF0, 0x9F, 0xA6, 0x28, 0xAD, 0x9A, 0x33,
	0xC3, 0xCE, 0x43, 0x93, 0xAD, 0xE1, 0xDE, 0xF7,
	0x5C, 0x0E, 0xB7, 0x8B, 0x67, 0xA5, 0x13, 0xC3,
	0xE5, 0x3B, 0x26, 0x19, 0xF9, 0x68, 0x55, 0xE2,
	0x91, 0xD5, 0x14, 0x1C, 0x7C, 0xD0, 0x91, 0x5E,
	0x1D, 0x04, 0xB3, 0x47, 0x45, 0x7C, 0x96, 0x01
};

static const ec_test_case ecsdsa_secp256r1_test_case = {
	.name = "ECSDSA-SHA256/secp256r1",
	.ec_str_p = &secp256r1_str_params,
	.priv_key = ecsdsa_secp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecsdsa_secp256r1_test_vectors_priv_key),
	.nn_random = ecsdsa_nn_random_iso14888_3_secp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECSDSA,
	.exp_sig = ecsdsa_secp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecsdsa_secp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_SECP384R1
#define ECSDSA_SHA384_SECP384R1_SELF_TEST

/* ECSDSA secp384r1 test vectors */

static int ecsdsa_nn_random_iso14888_3_secp384r1_test_vector(nn_t out,
							     nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x8A, 0x29, 0xE7, 0x72, 0x35, 0x7B, 0xBA, 0x6F,
		0x5C, 0x9E, 0xA7, 0x65, 0xD5, 0x08, 0x2B, 0x9B,
		0xC7, 0xA7, 0x4C, 0x33, 0xE9, 0xD9, 0x4D, 0x49,
		0xFB, 0x2C, 0x9D, 0x3B, 0x52, 0x3A, 0x82, 0x16,
		0x96, 0x82, 0xEC, 0xF1, 0x6F, 0x1D, 0x06, 0x26,
		0x90, 0x42, 0xF3, 0xAF, 0x04, 0x4B, 0x4D, 0xE8
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecsdsa_secp384r1_test_vectors_priv_key[] = {
	0x7E, 0x49, 0x14, 0xFE, 0x4B, 0x61, 0x70, 0x89,
	0xF9, 0xFE, 0x80, 0xAD, 0x91, 0x3D, 0x55, 0x30,
	0xEE, 0xC4, 0xF3, 0x0B, 0xCA, 0x1A, 0xD6, 0x6E,
	0x7B, 0x5A, 0xEA, 0xCF, 0x29, 0xD2, 0xF5, 0x67,
	0xD9, 0xA8, 0xF4, 0xA4, 0x55, 0x2A, 0x1A, 0x1A,
	0xF3, 0xE0, 0xB6, 0xD0, 0xA4, 0x9D, 0xFC, 0xC8
};

static const u8 ecsdsa_secp384r1_test_vectors_expected_sig[] = {
	0xF9, 0x07, 0x55, 0x3B, 0xB5, 0xC7, 0xDE, 0x02,
	0x9A, 0x2A, 0x56, 0x70, 0x78, 0xDF, 0xF9, 0xB8,
	0x03, 0xEC, 0x64, 0x96, 0x0D, 0x75, 0xBA, 0x73,
	0xA8, 0x55, 0x90, 0xAC, 0xC0, 0xAC, 0x44, 0x79,
	0xAC, 0x52, 0xE5, 0x1D, 0x56, 0x91, 0xFC, 0xB0,
	0x69, 0xDC, 0x5C, 0xD2, 0x4E, 0x0B, 0xCE, 0xC7,
	0x0B, 0x9D, 0x66, 0xD5, 0xDE, 0x70, 0xFA, 0xA8,
	0xB3, 0x56, 0x34, 0xA3, 0x7B, 0x33, 0xC2, 0xC4,
	0x60, 0xB8, 0xDC, 0x0B, 0xD4, 0xC8, 0x74, 0x5B,
	0xB8, 0x4D, 0xC1, 0x5C, 0xA8, 0x57, 0x0B, 0x07,
	0x92, 0x58, 0xF9, 0x77, 0xDA, 0x8B, 0x40, 0x61,
	0xF3, 0xDA, 0x6E, 0xBD, 0x7C, 0x42, 0x9A, 0x89
};

static const ec_test_case ecsdsa_secp384r1_test_case = {
	.name = "ECSDSA-SHA384/secp384r1",
	.ec_str_p = &secp384r1_str_params,
	.priv_key = ecsdsa_secp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecsdsa_secp384r1_test_vectors_priv_key),
	.nn_random = ecsdsa_nn_random_iso14888_3_secp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECSDSA,
	.exp_sig = ecsdsa_secp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecsdsa_secp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_SECP521R1
#define ECSDSA_SHA512_SECP521R1_SELF_TEST

/* ECSDSA secp521r1 test vectors */

static int ecsdsa_nn_random_secp521r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp521r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0x01, 0xad, 0xb3, 0xfd, 0xcf, 0x67, 0x5e, 0x25,
		0x3c, 0xfe, 0xac, 0x56, 0x1f, 0x21, 0x67, 0x67,
		0xa4, 0xba, 0xa7, 0x90, 0xac, 0x68, 0x1c, 0x74,
		0xf3, 0xdd, 0x83, 0x33, 0x5c, 0x9b, 0x37, 0x41,
		0xc8, 0x88, 0x77, 0x77, 0x33, 0xf7, 0xc7, 0xea,
		0x15, 0xd4, 0x5f, 0xa0, 0x7f, 0xba, 0x60, 0x39,
		0x4b, 0x3e, 0x16, 0xcd, 0xce, 0x1b, 0x02, 0xef,
		0xbc, 0x59, 0xba, 0xb0, 0xbd, 0x77, 0xa6, 0x50,
		0x07, 0xd4
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecsdsa_secp521r1_test_vectors_priv_key[] = {
	0x01, 0x82, 0x1f, 0x88, 0xbd, 0xe6, 0x75, 0x1c,
	0x06, 0x29, 0xfb, 0x75, 0x3b, 0xb1, 0xc7, 0xec,
	0x07, 0xbe, 0x64, 0x82, 0x88, 0x4a, 0x77, 0x4e,
	0x18, 0x3e, 0x1d, 0xcf, 0x95, 0x49, 0x4f, 0x81,
	0x7e, 0x71, 0x5b, 0xfe, 0x1d, 0xe3, 0x17, 0x24,
	0xc2, 0x93, 0x24, 0x67, 0xb1, 0x39, 0x90, 0x02,
	0x4c, 0x25, 0xc5, 0x54, 0x13, 0xbc, 0xe1, 0x24,
	0xa5, 0x35, 0x83, 0xf7, 0x3a, 0x86, 0xed, 0xc4,
	0x39, 0xb0
};

static const u8 ecsdsa_secp521r1_test_vectors_expected_sig[] = {
	0xe7, 0x3a, 0xe5, 0x72, 0x50, 0x29, 0x9b, 0x19,
	0x26, 0xbc, 0x79, 0x8a, 0x84, 0x6b, 0xd3, 0xbb,
	0x29, 0x15, 0x9d, 0x0d, 0xcb, 0x50, 0x87, 0xbb,
	0xd1, 0x72, 0xbd, 0x4a, 0x4b, 0x42, 0x3f, 0xb8,
	0xf1, 0x8b, 0xc2, 0x30, 0x1c, 0xa3, 0x1b, 0x94,
	0xef, 0x1a, 0xa0, 0x87, 0x71, 0x81, 0x0b, 0x37,
	0xa5, 0xd5, 0x4f, 0x0c, 0xde, 0xf1, 0xb1, 0xee,
	0xb0, 0x7b, 0x6c, 0xca, 0x35, 0xcf, 0xb2, 0x04,
	0x00, 0xba, 0xbf, 0x93, 0x78, 0xea, 0x91, 0x23,
	0x57, 0x49, 0xe1, 0x05, 0x52, 0x92, 0x2a, 0xff,
	0xb3, 0x32, 0x51, 0x6f, 0x59, 0xd9, 0xe1, 0x78,
	0x61, 0xfe, 0xb5, 0x43, 0x37, 0x40, 0xf3, 0xb8,
	0x18, 0x8c, 0x95, 0x2e, 0xd6, 0xb3, 0xea, 0x1a,
	0x37, 0x48, 0xfb, 0xa5, 0x77, 0xba, 0x69, 0xe2,
	0x12, 0x6b, 0x2c, 0xc9, 0x4e, 0x4c, 0x59, 0xe7,
	0x86, 0xe2, 0xfa, 0xdd, 0x5b, 0x79, 0xab, 0x15,
	0x5c, 0xf6
};

static const ec_test_case ecsdsa_secp521r1_test_case = {
	.name = "ECSDSA-SHA512/secp521r1",
	.ec_str_p = &secp521r1_str_params,
	.priv_key = ecsdsa_secp521r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecsdsa_secp521r1_test_vectors_priv_key),
	.nn_random = ecsdsa_nn_random_secp521r1_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECSDSA,
	.exp_sig = ecsdsa_secp521r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecsdsa_secp521r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP521R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_BRAINPOOLP256R1
#define ECSDSA_SHA256_BRAINPOOLP256R1_SELF_TEST

/* ECSDSA Brainpoolp256r1 test vectors */

static int ecsdsa_nn_random_brainpoolp256r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp256r1 signature
	 */
	const u8 k_buf[] = {
		0x29, 0xa5, 0xc2, 0x64, 0xba, 0x76, 0x37, 0x9d,
		0x86, 0x49, 0x8a, 0x64, 0x16, 0xfc, 0x7f, 0xba,
		0x9d, 0x4f, 0x62, 0x75, 0x64, 0xc6, 0x98, 0xab,
		0x4d, 0x95, 0xd1, 0x90, 0x6c, 0x8c, 0x61, 0xe4
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecsdsa_brainpoolp256r1_test_vectors_priv_key[] = {
	0xa9, 0x35, 0x71, 0x33, 0x4a, 0xc3, 0x2b, 0x50,
	0x26, 0x8d, 0xdc, 0xa0, 0x95, 0x23, 0x89, 0x3a,
	0x8f, 0x29, 0x89, 0xa9, 0x4f, 0x9f, 0x44, 0xa9,
	0x1b, 0x77, 0x43, 0xf7, 0xe1, 0x45, 0xae, 0xb7
};

static const u8 ecsdsa_brainpoolp256r1_test_vectors_expected_sig[] = {
	0x0e, 0x7a, 0xf5, 0x0b, 0xf4, 0xe0, 0x8b, 0xf8,
	0x51, 0x00, 0x44, 0x24, 0xee, 0x9d, 0x65, 0x02,
	0xfc, 0xd1, 0x16, 0x4e, 0xe3, 0xd9, 0x9a, 0x00,
	0xa8, 0x4f, 0xd5, 0xdb, 0x81, 0x48, 0x00, 0xeb,
	0x64, 0x7a, 0x24, 0xe6, 0x07, 0xb6, 0xfc, 0x09,
	0xd8, 0x8b, 0x1b, 0x57, 0x2c, 0xfc, 0x4c, 0xe2,
	0x9e, 0x25, 0xfa, 0xe1, 0x43, 0x1f, 0x0d, 0xfa,
	0x58, 0x6b, 0xd1, 0x6d, 0xee, 0xff, 0x92, 0xb6
};

static const ec_test_case ecsdsa_brainpoolp256r1_test_case = {
	.name = "ECSDSA-SHA256/brainpoolp256r1",
	.ec_str_p = &brainpoolp256r1_str_params,
	.priv_key = ecsdsa_brainpoolp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecsdsa_brainpoolp256r1_test_vectors_priv_key),
	.nn_random = ecsdsa_nn_random_brainpoolp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECSDSA,
	.exp_sig = ecsdsa_brainpoolp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecsdsa_brainpoolp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_BRAINPOOLP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_BRAINPOOLP384R1
#define ECSDSA_SHA384_BRAINPOOLP384R1_SELF_TEST

/* ECSDSA Brainpoolp384r1 test vectors */

static int ecsdsa_nn_random_brainpoolp384r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp384r1 signature
	 */
	const u8 k_buf[] = {
		0x4f, 0x71, 0xb2, 0x3e, 0x7f, 0xbb, 0x47, 0xa8,
		0x8d, 0xaf, 0xdb, 0xea, 0xdc, 0x78, 0x14, 0x50,
		0xa9, 0x3b, 0xc8, 0x4b, 0x5b, 0xd4, 0xd5, 0x38,
		0x6b, 0x1e, 0x19, 0x5f, 0x5c, 0x29, 0xe7, 0x9f,
		0x9e, 0xfe, 0x66, 0x1d, 0x02, 0x7b, 0xef, 0xbb,
		0xb1, 0xca, 0x1d, 0xa2, 0xfe, 0xa2, 0xab, 0x75
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecsdsa_brainpoolp384r1_test_vectors_priv_key[] = {
	0x71, 0xa8, 0x85, 0x7e, 0x32, 0x14, 0x5f, 0x02,
	0x90, 0x02, 0xbe, 0xdc, 0x68, 0x77, 0x78, 0x1d,
	0xd9, 0x34, 0x6d, 0x5f, 0x4f, 0x7d, 0x0f, 0x96,
	0xe3, 0x1d, 0x6f, 0x41, 0x42, 0x36, 0x37, 0x1d,
	0x42, 0x2e, 0x97, 0xa9, 0x21, 0xc5, 0x04, 0x9c,
	0xb4, 0xbf, 0x49, 0x46, 0x8a, 0x99, 0x0a, 0x16
};

static const u8 ecsdsa_brainpoolp384r1_test_vectors_expected_sig[] = {
	0x97, 0x27, 0x3d, 0x1b, 0x4a, 0x56, 0xe5, 0x55,
	0x77, 0xe0, 0x9c, 0xd4, 0xe0, 0x78, 0x2f, 0x6e,
	0x8d, 0x8b, 0x2e, 0xe0, 0xde, 0xef, 0x84, 0x44,
	0xd1, 0x3a, 0xc3, 0x5c, 0xe5, 0xf3, 0x5f, 0xae,
	0x49, 0xc2, 0x6a, 0x31, 0x26, 0xb4, 0x44, 0x88,
	0xc1, 0x4e, 0x3e, 0xcf, 0x78, 0x4c, 0xd8, 0xe5,

	0x1b, 0x71, 0x79, 0x37, 0x89, 0x3a, 0x77, 0x90,
	0xa8, 0x80, 0x01, 0x19, 0x87, 0xb6, 0x47, 0x1e,
	0x4b, 0xb4, 0x56, 0xc9, 0x08, 0x79, 0x32, 0x29,
	0x2f, 0x28, 0x35, 0x09, 0xcb, 0x39, 0x6c, 0x68,
	0x13, 0x84, 0xcc, 0xb0, 0xc3, 0x97, 0x72, 0x86,
	0x4c, 0xb5, 0x18, 0x5e, 0xaa, 0x59, 0xf4, 0x3d
};

static const ec_test_case ecsdsa_brainpoolp384r1_test_case = {
	.name = "ECSDSA-SHA384/brainpoolp384r1",
	.ec_str_p = &brainpoolp384r1_str_params,
	.priv_key = ecsdsa_brainpoolp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecsdsa_brainpoolp384r1_test_vectors_priv_key),
	.nn_random = ecsdsa_nn_random_brainpoolp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECSDSA,
	.exp_sig = ecsdsa_brainpoolp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecsdsa_brainpoolp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_BRAINPOOLP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_BRAINPOOLP512R1
#define ECSDSA_SHA512_BRAINPOOLP512R1_SELF_TEST

/* ECSDSA Brainpoolp512r1 test vectors */

static int ecsdsa_nn_random_brainpoolp512r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp512r1 signature
	 */
	const u8 k_buf[] = {
		0x65, 0xe7, 0x6d, 0xe3, 0x2e, 0x77, 0x21, 0x67,
		0x3a, 0x99, 0x1d, 0x95, 0xda, 0x6f, 0x7e, 0x0a,
		0x5a, 0x72, 0x72, 0xf8, 0xde, 0x4a, 0xae, 0xfe,
		0xd9, 0x7b, 0x49, 0x07, 0xda, 0x0e, 0xe8, 0x4d,
		0x66, 0xdd, 0x51, 0x8e, 0x6e, 0x75, 0xd3, 0x28,
		0x54, 0x08, 0xaf, 0x82, 0xeb, 0xf2, 0x5a, 0x5e,
		0x3c, 0x14, 0x9b, 0x6e, 0xba, 0xba, 0x2b, 0x7b,
		0xdf, 0x6d, 0x26, 0x0e, 0x3a, 0xea, 0xae, 0xbc
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecsdsa_brainpoolp512r1_test_vectors_priv_key[] = {
	0x20, 0xe6, 0xe2, 0x08, 0x67, 0xf7, 0x29, 0x0c,
	0x04, 0xfc, 0x00, 0x7e, 0x77, 0x34, 0x9e, 0x30,
	0x09, 0xaa, 0xc0, 0x3d, 0xfa, 0xf0, 0xd2, 0x56,
	0x70, 0x86, 0x73, 0x86, 0xca, 0xce, 0xdb, 0xcd,
	0x55, 0x3d, 0xd0, 0x1b, 0x86, 0xdd, 0x50, 0x17,
	0xe1, 0xdc, 0x45, 0xa1, 0xa5, 0xae, 0xd3, 0x80,
	0x72, 0xef, 0x04, 0xe7, 0x9f, 0x36, 0xc3, 0xe7,
	0x84, 0xd5, 0xdf, 0x80, 0x39, 0xc2, 0x5d, 0x18
};

static const u8 ecsdsa_brainpoolp512r1_test_vectors_expected_sig[] = {
	0x7f, 0xf5, 0x12, 0x0a, 0xaa, 0x8f, 0x7a, 0x31,
	0xe9, 0xaf, 0x44, 0x4b, 0xdf, 0x97, 0x1e, 0x80,
	0x0c, 0xb8, 0x3c, 0x4c, 0x45, 0x39, 0xa1, 0xa6,
	0x51, 0x53, 0xc2, 0x32, 0x48, 0x62, 0x7a, 0x12,
	0xf5, 0xf3, 0xb1, 0xd6, 0xdf, 0x07, 0x78, 0x14,
	0xa0, 0x53, 0xcb, 0x6e, 0xef, 0x07, 0x20, 0xc3,
	0x9f, 0xf2, 0x34, 0x20, 0x31, 0x21, 0x5e, 0xea,
	0x55, 0xd7, 0xff, 0xd6, 0x4f, 0x18, 0xc2, 0x64,

	0x45, 0xdb, 0xd2, 0xae, 0xf5, 0x0c, 0x14, 0x8b,
	0xad, 0xaa, 0x2d, 0x1e, 0x04, 0x06, 0xaf, 0xb9,
	0x1f, 0x17, 0x20, 0x3f, 0x29, 0x8c, 0xc6, 0x63,
	0xb6, 0xf9, 0xc4, 0x56, 0x4f, 0xa7, 0x76, 0x31,
	0xe5, 0x46, 0x6b, 0x86, 0x80, 0x59, 0xdf, 0x01,
	0x27, 0x38, 0x0f, 0x17, 0x8c, 0x59, 0xcc, 0xad,
	0x8c, 0x63, 0xe4, 0xa6, 0x48, 0x84, 0x88, 0x95,
	0x0e, 0x6f, 0xa6, 0x2a, 0xbf, 0x4b, 0xff, 0x30
};

static const ec_test_case ecsdsa_brainpoolp512r1_test_case = {
	.name = "ECSDSA-SHA512/brainpoolp512r1",
	.ec_str_p = &brainpoolp512r1_str_params,
	.priv_key = ecsdsa_brainpoolp512r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecsdsa_brainpoolp512r1_test_vectors_priv_key),
	.nn_random = ecsdsa_nn_random_brainpoolp512r1_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECSDSA,
	.exp_sig = ecsdsa_brainpoolp512r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecsdsa_brainpoolp512r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP512R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_FRP256V1
#define ECSDSA_SHA256_FRP256V1_SELF_TEST

/* ECSDSA frp256v1 test vectors */

static int ecsdsa_nn_random_frp256v1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for frp256v1 signature
	 */
	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecsdsa_frp256v1_test_vectors_priv_key[] = {
	0x00, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 ecsdsa_frp256v1_test_vectors_expected_sig[] = {
	0x9e, 0x57, 0x81, 0x73, 0xf9, 0x7b, 0xbe, 0xb1,
	0xdc, 0xc4, 0x4f, 0xe6, 0xc1, 0x7a, 0x40, 0x9e,
	0x22, 0xad, 0x13, 0xa1, 0xa9, 0x69, 0x14, 0x62,
	0x0d, 0x08, 0x76, 0x0f, 0xe0, 0xd7, 0xc6, 0x75,

	0x9b, 0x3a, 0xb9, 0xd5, 0xb6, 0x7f, 0xf2, 0xb9,
	0xaa, 0xe8, 0xc5, 0xed, 0xf9, 0x82, 0xd4, 0xf7,
	0x6d, 0x7c, 0x61, 0x36, 0x56, 0xe7, 0x73, 0xaa,
	0x1a, 0xab, 0xa6, 0x30, 0x90, 0x1d, 0x56, 0xa6,
};

static const ec_test_case ecsdsa_frp256v1_test_case = {
	.name = "ECSDSA-SHA256/frp256v1",
	.ec_str_p = &frp256v1_str_params,
	.priv_key = ecsdsa_frp256v1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecsdsa_frp256v1_test_vectors_priv_key),
	.nn_random = ecsdsa_nn_random_frp256v1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECSDSA,
	.exp_sig = ecsdsa_frp256v1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecsdsa_frp256v1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_FRP256V1 */
#endif /* WITH_HASH_SHA256 */
#endif /* WITH_SIG_ECSDSA */

/*******************************************************************
 ************** ECOSDSA tests **************************************
 *******************************************************************/

#ifdef WITH_SIG_ECOSDSA
#ifdef WITH_HASH_SHA224
#ifdef WITH_CURVE_SECP224R1
#define ECOSDSA_SHA224_SECP224R1_SELF_TEST

/* ECOSDSA secp224r1 test vectors */

static int ecosdsa_nn_random_secp224r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0xc1, 0xbe, 0x08, 0xf7, 0xa6, 0xda, 0x95, 0xea,
		0xda, 0xc9, 0x2b, 0x7a, 0x8d, 0xfa, 0x4b, 0x8c,
		0x0b, 0x09, 0x9a, 0xf3, 0xa3, 0x0a, 0x23, 0xb5,
		0xa7, 0x6a, 0xa4, 0xc5
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecosdsa_secp224r1_test_vectors_priv_key[] = {
	0xfd, 0x22, 0xee, 0x55, 0x5f, 0x4a, 0xb5, 0x40,
	0x1b, 0xbd, 0x53, 0x65, 0xec, 0x8a, 0xec, 0x37,
	0xf3, 0xb4, 0xcb, 0xe4, 0xeb, 0xdd, 0x55, 0x40,
	0xad, 0x92, 0x29, 0xf0
};

static const u8 ecosdsa_secp224r1_test_vectors_expected_sig[] = {
	0x48, 0x33, 0x6d, 0x3d, 0x8d, 0xa5, 0x3b, 0x5d,
	0xff, 0xca, 0x3e, 0x36, 0x24, 0x7b, 0xeb, 0x07,
	0x0c, 0x32, 0x53, 0x21, 0xf3, 0x7e, 0xb3, 0xf7,
	0xe3, 0xb1, 0xaa, 0xb3,
	0x28, 0x39, 0xe8, 0x80, 0xbe, 0x1a, 0x2e, 0xd8,
	0x31, 0x4f, 0x85, 0x2d, 0xf6, 0x6b, 0x1a, 0xc0,
	0xb7, 0x22, 0x89, 0xa9, 0xf8, 0x70, 0xd5, 0x19,
	0xbb, 0x80, 0x11, 0x16
};

static const ec_test_case ecosdsa_secp224r1_test_case = {
	.name = "ECOSDSA-SHA224/secp224r1",
	.ec_str_p = &secp224r1_str_params,
	.priv_key = ecosdsa_secp224r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecosdsa_secp224r1_test_vectors_priv_key),
	.nn_random = ecosdsa_nn_random_secp224r1_test_vector,
	.hash_type = SHA224,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECOSDSA,
	.exp_sig = ecosdsa_secp224r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecosdsa_secp224r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP224R1 */
#endif /* WITH_HASH_SHA224 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_SECP256R1
#define ECOSDSA_SHA256_SECP256R1_SELF_TEST

/* ECOSDSA secp256r1 test vectors */

static int ecosdsa_nn_random_iso14888_3_secp256r1_test_vector(nn_t out,
							      nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0xDE, 0x7E, 0x0E, 0x5E, 0x66, 0x3F, 0x24, 0x18,
		0x34, 0x14, 0xB7, 0xC7, 0x2F, 0x24, 0x54, 0x6B,
		0x81, 0xE9, 0xE5, 0xF4, 0x10, 0xBE, 0xBF, 0x26,
		0xF3, 0xCA, 0x5F, 0xA8, 0x2F, 0x51, 0x92, 0xC8
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecosdsa_secp256r1_test_vectors_priv_key[] = {
	0x52, 0x02, 0xA3, 0xD8, 0xAC, 0xAF, 0x69, 0x09,
	0xD1, 0x2C, 0x9A, 0x77, 0x4C, 0xD8, 0x86, 0xF9,
	0xFB, 0xA6, 0x11, 0x37, 0xFF, 0xD3, 0xE8, 0xE7,
	0x6A, 0xED, 0x36, 0x3F, 0xB4, 0x7A, 0xC4, 0x92
};

static const u8 ecosdsa_secp256r1_test_vectors_expected_sig[] = {
	0xD7, 0xFB, 0x81, 0x35, 0xD8, 0xEA, 0x45, 0xE8,
	0xFB, 0x3C, 0x90, 0x59, 0xF1, 0x46, 0xE2, 0x63,
	0x0E, 0xF4, 0xBD, 0x51, 0xC4, 0x00, 0x6A, 0x92,
	0xED, 0xB4, 0xC8, 0xB0, 0x84, 0x99, 0x63, 0xFB,
	0xB4, 0x6D, 0x15, 0x25, 0x37, 0x9E, 0x02, 0xE2,
	0x32, 0xD9, 0x79, 0x28, 0x26, 0x5B, 0x72, 0x54,
	0xEA, 0x2E, 0xD9, 0x78, 0x13, 0x45, 0x43, 0x88,
	0xC1, 0xA0, 0x8F, 0x62, 0xDC, 0xCD, 0x70, 0xB3
};

static const ec_test_case ecosdsa_secp256r1_test_case = {
	.name = "ECOSDSA-SHA256/secp256r1",
	.ec_str_p = &secp256r1_str_params,
	.priv_key = ecosdsa_secp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecosdsa_secp256r1_test_vectors_priv_key),
	.nn_random = ecosdsa_nn_random_iso14888_3_secp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECOSDSA,
	.exp_sig = ecosdsa_secp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecosdsa_secp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_SECP384R1
#define ECOSDSA_SHA384_SECP384R1_SELF_TEST

/* ECOSDSA secp384r1 test vectors */

static int ecosdsa_nn_random_iso14888_3_secp384r1_test_vector(nn_t out,
							      nn_src_t q)

{
	int ret, cmp;
	const u8 k_buf[] = {
		0x8A, 0x29, 0xE7, 0x72, 0x35, 0x7B, 0xBA, 0x6F,
		0x5C, 0x9E, 0xA7, 0x65, 0xD5, 0x08, 0x2B, 0x9B,
		0xC7, 0xA7, 0x4C, 0x33, 0xE9, 0xD9, 0x4D, 0x49,
		0xFB, 0x2C, 0x9D, 0x3B, 0x52, 0x3A, 0x82, 0x16,
		0x96, 0x82, 0xEC, 0xF1, 0x6F, 0x1D, 0x06, 0x26,
		0x90, 0x42, 0xF3, 0xAF, 0x04, 0x4B, 0x4D, 0xE8
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecosdsa_secp384r1_test_vectors_priv_key[] = {
	0x7E, 0x49, 0x14, 0xFE, 0x4B, 0x61, 0x70, 0x89,
	0xF9, 0xFE, 0x80, 0xAD, 0x91, 0x3D, 0x55, 0x30,
	0xEE, 0xC4, 0xF3, 0x0B, 0xCA, 0x1A, 0xD6, 0x6E,
	0x7B, 0x5A, 0xEA, 0xCF, 0x29, 0xD2, 0xF5, 0x67,
	0xD9, 0xA8, 0xF4, 0xA4, 0x55, 0x2A, 0x1A, 0x1A,
	0xF3, 0xE0, 0xB6, 0xD0, 0xA4, 0x9D, 0xFC, 0xC8
};

static const u8 ecosdsa_secp384r1_test_vectors_expected_sig[] = {
	0x27, 0xD2, 0xF5, 0xB9, 0x62, 0xA3, 0xAC, 0xF6,
	0x39, 0x0A, 0x47, 0x18, 0xEA, 0x54, 0x0D, 0xA7,
	0x96, 0x12, 0xA6, 0x0E, 0xAA, 0x15, 0xBE, 0xBB,
	0x00, 0xB9, 0xE1, 0x66, 0x57, 0x83, 0xF7, 0xC7,
	0x91, 0xCC, 0xAC, 0x42, 0x2C, 0xEE, 0x81, 0x5A,
	0x9C, 0x5D, 0xA3, 0x67, 0x8A, 0xC8, 0xD1, 0xF0,
	0x22, 0xCC, 0x89, 0xCE, 0xB9, 0xE6, 0xBE, 0x84,
	0x15, 0xCC, 0x14, 0xB3, 0x99, 0xBC, 0x66, 0xE6,
	0xF3, 0xA2, 0x1E, 0x5B, 0xA3, 0x8E, 0x09, 0xA6,
	0xDE, 0x8D, 0xE6, 0x70, 0xA1, 0x45, 0xC0, 0xE4,
	0x74, 0xD5, 0xCC, 0x88, 0xBE, 0x88, 0x78, 0xF0,
	0x12, 0x3C, 0xC6, 0x62, 0x25, 0xA1, 0xBA, 0x12
};

static const ec_test_case ecosdsa_secp384r1_test_case = {
	.name = "ECOSDSA-SHA384/secp384r1",
	.ec_str_p = &secp384r1_str_params,
	.priv_key = ecosdsa_secp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecosdsa_secp384r1_test_vectors_priv_key),
	.nn_random = ecosdsa_nn_random_iso14888_3_secp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECOSDSA,
	.exp_sig = ecosdsa_secp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecosdsa_secp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_SECP521R1
#define ECOSDSA_SHA512_SECP521R1_SELF_TEST

/* ECOSDSA secp521r1 test vectors */

static int ecosdsa_nn_random_secp521r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp521r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0x01, 0x73, 0x03, 0x00, 0x16, 0x30, 0x79, 0x94,
		0x9b, 0xb6, 0xe3, 0x60, 0x3c, 0x96, 0xf6, 0xf0,
		0x90, 0x23, 0xa6, 0xf7, 0x48, 0xc5, 0xa6, 0xfc,
		0x41, 0xec, 0x39, 0xf5, 0x24, 0x04, 0x59, 0x59,
		0xa8, 0xfe, 0x4c, 0xc1, 0x9b, 0x01, 0x47, 0x5a,
		0xac, 0xc5, 0xf3, 0x92, 0xab, 0x50, 0x60, 0x1f,
		0x39, 0x8b, 0xc3, 0xb8, 0x01, 0x4a, 0x63, 0x91,
		0x1b, 0xce, 0x30, 0xe7, 0xcc, 0x0f, 0x3d, 0x82,
		0x3b, 0x05
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecosdsa_secp521r1_test_vectors_priv_key[] = {
	0x01, 0x08, 0xb9, 0x4c, 0x21, 0x92, 0x06, 0xb8,
	0xd8, 0x92, 0x16, 0x90, 0xa5, 0x15, 0x76, 0xae,
	0x70, 0x3d, 0x46, 0x4b, 0x71, 0xbd, 0xf5, 0x68,
	0xf9, 0xf6, 0x70, 0x92, 0x95, 0x67, 0x85, 0x80,
	0x55, 0xb9, 0x16, 0xdb, 0xce, 0x5d, 0xb5, 0x6a,
	0x25, 0xb7, 0x91, 0x33, 0x25, 0xf5, 0x14, 0xf7,
	0x2a, 0x22, 0xbe, 0x4e, 0xfb, 0x4d, 0xb7, 0x9e,
	0x2f, 0x01, 0xb5, 0x71, 0xa9, 0x3f, 0x21, 0x79,
	0x82, 0xc5
};

static const u8 ecosdsa_secp521r1_test_vectors_expected_sig[] = {
	0xbd, 0x54, 0xe8, 0xd7, 0x27, 0x1c, 0xcc, 0x1e,
	0x0a, 0xdd, 0xc3, 0x33, 0x8a, 0x99, 0xeb, 0x94,
	0x5e, 0x65, 0x40, 0x18, 0x05, 0x6f, 0xaf, 0x7e,
	0xc7, 0x6c, 0x0a, 0xa6, 0xb4, 0xd2, 0x62, 0x89,
	0x00, 0xe0, 0x5c, 0x05, 0x24, 0x95, 0xa0, 0x91,
	0x7d, 0x80, 0x85, 0x14, 0xd9, 0x1e, 0x41, 0x42,
	0xc5, 0xae, 0x53, 0x06, 0x24, 0x31, 0xe2, 0x31,
	0x05, 0x8d, 0xb5, 0x19, 0x2e, 0x35, 0x90, 0xe3,

	0x00, 0x52, 0x7b, 0xab, 0x84, 0xc9, 0xaf, 0x15,
	0xcf, 0xa7, 0xd9, 0x25, 0x29, 0xdc, 0x15, 0xd2,
	0x17, 0x45, 0xee, 0x04, 0x88, 0x91, 0xf5, 0xe0,
	0x6a, 0x06, 0x65, 0xe5, 0xcb, 0x16, 0x2e, 0x62,
	0x2a, 0x0a, 0x62, 0xc3, 0x12, 0x23, 0xcc, 0x21,
	0x6d, 0x8f, 0xfe, 0xc3, 0x3a, 0xcd, 0xdd, 0xe5,
	0x0d, 0x58, 0xef, 0xe3, 0x87, 0x89, 0x3a, 0x5c,
	0xbc, 0xe2, 0xea, 0x07, 0x6e, 0x3b, 0x7c, 0xe7,
	0x95, 0x00,
};

static const ec_test_case ecosdsa_secp521r1_test_case = {
	.name = "ECOSDSA-SHA512/secp521r1",
	.ec_str_p = &secp521r1_str_params,
	.priv_key = ecosdsa_secp521r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecosdsa_secp521r1_test_vectors_priv_key),
	.nn_random = ecosdsa_nn_random_secp521r1_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECOSDSA,
	.exp_sig = ecosdsa_secp521r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecosdsa_secp521r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP521R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_BRAINPOOLP256R1
#define ECOSDSA_SHA256_BRAINPOOLP256R1_SELF_TEST

/* ECOSDSA Brainpoolp256r1 test vectors */

static int ecosdsa_nn_random_brainpoolp256r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp256r1 signature
	 */
	const u8 k_buf[] = {
		0x3f, 0x80, 0xde, 0x27, 0x3e, 0xef, 0xc9, 0x53,
		0xc9, 0x25, 0xab, 0x53, 0xdd, 0xba, 0xd4, 0xb3,
		0x23, 0x98, 0xb7, 0xcc, 0x39, 0xb1, 0xda, 0xe4,
		0xbe, 0x68, 0x37, 0xd4, 0xd9, 0x9e, 0xa0, 0x49
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecosdsa_brainpoolp256r1_test_vectors_priv_key[] = {
	0xa8, 0x44, 0x22, 0x5a, 0xf8, 0xae, 0xb6, 0xa7,
	0x66, 0x10, 0x93, 0xe5, 0x44, 0xd8, 0x47, 0xa0,
	0x63, 0x82, 0x08, 0xcf, 0xc6, 0x4f, 0x6e, 0xa8,
	0x1c, 0xbb, 0x0d, 0xbb, 0xcf, 0x16, 0x4d, 0xa2
};

static const u8 ecosdsa_brainpoolp256r1_test_vectors_expected_sig[] = {
	0xc1, 0x5a, 0x0e, 0xd2, 0x9c, 0x39, 0x3b, 0x04,
	0x14, 0xaa, 0x40, 0xe5, 0x1c, 0xe6, 0x45, 0x28,
	0x19, 0xa9, 0x7f, 0x66, 0x1b, 0x4f, 0x11, 0x50,
	0x13, 0x8f, 0xc4, 0xe6, 0x96, 0x47, 0xe1, 0xa4,

	0x1c, 0x26, 0x9e, 0x5e, 0xd2, 0x9d, 0x38, 0x1a,
	0xc4, 0xa0, 0xed, 0x86, 0xe4, 0x73, 0x68, 0x46,
	0xa4, 0x34, 0x72, 0x6a, 0xb5, 0x36, 0x0d, 0x3a,
	0x7e, 0x2b, 0x1c, 0x44, 0x0a, 0x69, 0xfe, 0xde
};

static const ec_test_case ecosdsa_brainpoolp256r1_test_case = {
	.name = "ECOSDSA-SHA256/brainpoolp256r1",
	.ec_str_p = &brainpoolp256r1_str_params,
	.priv_key = ecosdsa_brainpoolp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecosdsa_brainpoolp256r1_test_vectors_priv_key),
	.nn_random = ecosdsa_nn_random_brainpoolp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECOSDSA,
	.exp_sig = ecosdsa_brainpoolp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecosdsa_brainpoolp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_BRAINPOOLP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_BRAINPOOLP384R1
#define ECOSDSA_SHA384_BRAINPOOLP384R1_SELF_TEST

/* ECOSDSA Brainpoolp384r1 test vectors */

static int ecosdsa_nn_random_brainpoolp384r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp384r1 signature
	 */
	const u8 k_buf[] = {
		0x4c, 0x0f, 0xa0, 0xfb, 0xd2, 0xec, 0x92, 0x0f,
		0x70, 0x32, 0xb2, 0xd5, 0xa4, 0xb0, 0xe5, 0xb0,
		0x88, 0xc1, 0x70, 0x6a, 0xe1, 0x4a, 0xc7, 0x49,
		0xc3, 0x14, 0x8a, 0x09, 0x9d, 0x3f, 0x0e, 0xde,
		0xe6, 0xf9, 0x73, 0x36, 0x76, 0x3b, 0xa7, 0x53,
		0x4c, 0xf5, 0x29, 0xa3, 0xd6, 0xe4, 0x60, 0x3e
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecosdsa_brainpoolp384r1_test_vectors_priv_key[] = {
	0x84, 0x27, 0xc5, 0xc6, 0xb1, 0x95, 0x1a, 0x87,
	0x4a, 0x00, 0x02, 0xfc, 0xeb, 0x8b, 0x8f, 0x46,
	0x53, 0x2a, 0x82, 0xff, 0xac, 0x72, 0x78, 0x70,
	0x6a, 0x04, 0xe6, 0x25, 0x6b, 0x0f, 0xbd, 0x6e,
	0x10, 0xab, 0x74, 0x9d, 0xee, 0x3e, 0x02, 0x7d,
	0xc5, 0x2e, 0x3c, 0x55, 0x25, 0xc0, 0x1e, 0x7d
};

static const u8 ecosdsa_brainpoolp384r1_test_vectors_expected_sig[] = {
	0xf8, 0x8e, 0x7c, 0x39, 0xcf, 0x98, 0xa5, 0x7d,
	0xb1, 0x6f, 0x3b, 0xea, 0x31, 0x61, 0xd5, 0x8a,
	0x9d, 0x90, 0xfe, 0x4b, 0xa8, 0xc0, 0x71, 0x65,
	0x93, 0x63, 0x34, 0x99, 0x0b, 0x06, 0xaf, 0x5e,
	0x86, 0xb1, 0x10, 0x53, 0x39, 0xfd, 0xf1, 0x52,
	0xda, 0x28, 0x9f, 0x6b, 0x4c, 0xd4, 0x20, 0x43,

	0x69, 0x4e, 0x23, 0x7e, 0xc4, 0x5a, 0x83, 0x22,
	0x61, 0x53, 0xfd, 0x08, 0xc7, 0x5e, 0xdc, 0x69,
	0xc0, 0x22, 0x3f, 0x59, 0x8b, 0xf1, 0x9e, 0x8d,
	0x07, 0xb0, 0xb2, 0x38, 0x8a, 0xe2, 0xb7, 0x79,
	0x46, 0xda, 0xa2, 0x39, 0x4d, 0x7a, 0xce, 0x38,
	0x2a, 0xc6, 0xbe, 0xda, 0x21, 0x67, 0x81, 0xbf
};

static const ec_test_case ecosdsa_brainpoolp384r1_test_case = {
	.name = "ECOSDSA-SHA384/brainpoolp384r1",
	.ec_str_p = &brainpoolp384r1_str_params,
	.priv_key = ecosdsa_brainpoolp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecosdsa_brainpoolp384r1_test_vectors_priv_key),
	.nn_random = ecosdsa_nn_random_brainpoolp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECOSDSA,
	.exp_sig = ecosdsa_brainpoolp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecosdsa_brainpoolp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_BRAINPOOLP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_BRAINPOOLP512R1
#define ECOSDSA_SHA512_BRAINPOOLP512R1_SELF_TEST

/* ECOSDSA Brainpoolp512r1 test vectors */

static int ecosdsa_nn_random_brainpoolp512r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp512r1 signature
	 */
	const u8 k_buf[] = {
		0x70, 0x82, 0xf3, 0x1a, 0xb1, 0x06, 0x2d, 0x5b,
		0xb8, 0x16, 0x87, 0xd5, 0x65, 0x34, 0x74, 0xc3,
		0x53, 0x23, 0x0a, 0xc0, 0x09, 0xde, 0xd1, 0x44,
		0xad, 0x47, 0x86, 0xda, 0x89, 0xbf, 0x33, 0x0e,
		0x34, 0xd2, 0x12, 0x40, 0x44, 0xa5, 0x8b, 0x08,
		0x5a, 0x7e, 0x70, 0xbc, 0xfd, 0xed, 0x72, 0x03,
		0x07, 0x66, 0xc5, 0x41, 0x49, 0x1a, 0xda, 0x73,
		0x34, 0x3e, 0x7d, 0xf9, 0x0f, 0x0a, 0x4e, 0x38
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecosdsa_brainpoolp512r1_test_vectors_priv_key[] = {
	0x9d, 0xf5, 0xaa, 0xf5, 0xdb, 0xc9, 0x8d, 0x6b,
	0x55, 0xf2, 0xa0, 0x8c, 0xd7, 0xea, 0x7f, 0x7b,
	0x19, 0xdb, 0xc7, 0x8c, 0x28, 0x0d, 0x76, 0xd5,
	0xd1, 0xd2, 0x71, 0xf1, 0x73, 0xc3, 0xca, 0x1e,
	0xf8, 0xfa, 0x96, 0x87, 0x10, 0xd6, 0x14, 0x06,
	0xff, 0xce, 0x74, 0x89, 0x34, 0x27, 0x19, 0x57,
	0x5d, 0x8a, 0x9e, 0x3f, 0x0a, 0xcc, 0xf4, 0x53,
	0xfa, 0x51, 0x5a, 0x8d, 0xcd, 0x3f, 0x5a, 0xf6
};

static const u8 ecosdsa_brainpoolp512r1_test_vectors_expected_sig[] = {
	0xb2, 0x08, 0xeb, 0xa4, 0xf6, 0x87, 0x2b, 0xd6,
	0xcd, 0xef, 0xad, 0x6e, 0x72, 0x50, 0xb0, 0xb1,
	0x43, 0xb1, 0x4d, 0xfd, 0xf6, 0x10, 0xe8, 0xb5,
	0x95, 0x60, 0xec, 0xe6, 0xf7, 0x03, 0x56, 0xf6,
	0xac, 0xc0, 0x69, 0x57, 0xb5, 0x31, 0x30, 0x60,
	0x0f, 0x58, 0x3f, 0x6a, 0x8d, 0xf8, 0x1d, 0x4b,
	0xb0, 0xe3, 0x05, 0xe8, 0xdb, 0x5a, 0xee, 0x7f,
	0x15, 0xdb, 0xa8, 0xac, 0xbc, 0x3e, 0xba, 0x5d,

	0x79, 0xc7, 0xad, 0xc8, 0x4a, 0x09, 0xd2, 0x84,
	0x30, 0x9b, 0x09, 0xdc, 0xe3, 0x79, 0x69, 0x33,
	0x4b, 0xb5, 0x3e, 0x57, 0x4c, 0x57, 0x1f, 0xf7,
	0x41, 0x48, 0xeb, 0x27, 0xf9, 0xcc, 0xb9, 0x9c,
	0x23, 0xd7, 0xed, 0xd2, 0x7e, 0x46, 0xb6, 0xd9,
	0x1e, 0x59, 0xc3, 0xb5, 0x62, 0x66, 0x6a, 0x88,
	0xab, 0x3c, 0x66, 0x48, 0x96, 0x57, 0xbb, 0xd4,
	0xd7, 0x4c, 0xac, 0x91, 0x59, 0xfc, 0xb2, 0x08
};

static const ec_test_case ecosdsa_brainpoolp512r1_test_case = {
	.name = "ECOSDSA-SHA512/brainpoolp512r1",
	.ec_str_p = &brainpoolp512r1_str_params,
	.priv_key = ecosdsa_brainpoolp512r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecosdsa_brainpoolp512r1_test_vectors_priv_key),
	.nn_random = ecosdsa_nn_random_brainpoolp512r1_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECOSDSA,
	.exp_sig = ecosdsa_brainpoolp512r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecosdsa_brainpoolp512r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP512R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_FRP256V1
#define ECOSDSA_SHA256_FRP256V1_SELF_TEST

/* ECOSDSA frp256v1 test vectors */

static int ecosdsa_nn_random_frp256v1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for frp256v1 signature
	 */
	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecosdsa_frp256v1_test_vectors_priv_key[] = {
	0x00, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 ecosdsa_frp256v1_test_vectors_expected_sig[] = {
	0xae, 0xc2, 0x3f, 0x82, 0x0f, 0x4c, 0x50, 0x80,
	0xf4, 0xcd, 0x15, 0xd1, 0xd4, 0x7d, 0x68, 0x51,
	0x5e, 0x98, 0xb2, 0x85, 0xe2, 0x6f, 0x6a, 0x08,
	0x61, 0x46, 0x4b, 0x7e, 0x2b, 0xe0, 0xa4, 0x61,

	0xd7, 0xa4, 0x22, 0x5f, 0x29, 0x79, 0x8c, 0x78,
	0x97, 0x2f, 0x1b, 0xce, 0x6b, 0x35, 0x97, 0x7d,
	0x85, 0x9b, 0x7e, 0x22, 0x9b, 0xde, 0xa4, 0x19,
	0xc9, 0xbe, 0x26, 0x35, 0x9e, 0x54, 0xea, 0x14
};

static const ec_test_case ecosdsa_frp256v1_test_case = {
	.name = "ECOSDSA-SHA256/frp256v1",
	.ec_str_p = &frp256v1_str_params,
	.priv_key = ecosdsa_frp256v1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecosdsa_frp256v1_test_vectors_priv_key),
	.nn_random = ecosdsa_nn_random_frp256v1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECOSDSA,
	.exp_sig = ecosdsa_frp256v1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecosdsa_frp256v1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_FRP256V1 */
#endif /* WITH_HASH_SHA256 */
#endif /* WITH_SIG_ECOSDSA */

/*******************************************************************
 ************** ECFSDSA tests **************************************
 *******************************************************************/

#ifdef WITH_SIG_ECFSDSA
#ifdef WITH_HASH_SHA224
#ifdef WITH_CURVE_SECP224R1
#define ECFSDSA_SHA224_SECP224R1_SELF_TEST

/* ECFSDSA secp224r1 test vectors */

static int ecfsdsa_nn_random_secp224r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x42, 0xcf, 0xfe, 0x41, 0xf5, 0x01, 0xb4, 0x99,
		0x45, 0xe3, 0xd1, 0x03, 0xe9, 0x00, 0x48, 0xb9,
		0x6f, 0x47, 0xee, 0xb3, 0xb7, 0x37, 0xdc, 0x6a,
		0x79, 0xc7, 0xa4, 0x2e
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecfsdsa_secp224r1_test_vectors_priv_key[] = {
	0xda, 0xd8, 0x0f, 0xfb, 0x64, 0x35, 0x6b, 0xd5,
	0x3b, 0x66, 0x3b, 0x6e, 0x20, 0xa8, 0x08, 0x94,
	0xb7, 0x93, 0x9a, 0xde, 0x21, 0xc7, 0x1e, 0xb8,
	0x09, 0xf7, 0xb5, 0x39
};

static const u8 ecfsdsa_secp224r1_test_vectors_expected_sig[] = {
	0x55, 0x49, 0x93, 0x3b, 0xb2, 0x5d, 0x21, 0xc6,
	0x10, 0x84, 0xc1, 0xa4, 0x92, 0xed, 0x95, 0x30,
	0x0b, 0x50, 0x78, 0xd9, 0xa9, 0x5c, 0xac, 0x43,
	0xfd, 0xeb, 0x94, 0x9f, 0x32, 0xc2, 0xa1, 0xb1,
	0x67, 0x56, 0x3a, 0x6b, 0xe9, 0xdd, 0x77, 0x4e,
	0xa6, 0x5a, 0x29, 0xb1, 0x9f, 0xf3, 0xd8, 0x3b,
	0xdc, 0xa4, 0xf0, 0x3b, 0x7b, 0xc5, 0x27, 0x67,

	0xb4, 0xe9, 0xf2, 0x03, 0x1a, 0x98, 0xbe, 0x55,
	0xf1, 0xa9, 0xb3, 0x8b, 0x23, 0x31, 0x98, 0x6c,
	0x3a, 0xfd, 0x97, 0x2a, 0xed, 0xcd, 0x80, 0x5a,
	0x3c, 0x6e, 0x3a, 0x8c
};

static const ec_test_case ecfsdsa_secp224r1_test_case = {
	.name = "ECFSDSA-SHA224/secp224r1",
	.ec_str_p = &secp224r1_str_params,
	.priv_key = ecfsdsa_secp224r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecfsdsa_secp224r1_test_vectors_priv_key),
	.nn_random = ecfsdsa_nn_random_secp224r1_test_vector,
	.hash_type = SHA224,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECFSDSA,
	.exp_sig = ecfsdsa_secp224r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecfsdsa_secp224r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP224R1 */
#endif /* WITH_HASH_SHA224 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_SECP256R1
#define ECFSDSA_SHA256_SECP256R1_SELF_TEST

/* ECFSDSA secp256r1 test vectors */

static int ecfsdsa_nn_random_iso14888_3_secp256r1_test_vector(nn_t out,
							      nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x89, 0x4D, 0xEA, 0xB4, 0x4D, 0x88, 0x45, 0x0F,
		0xE8, 0xDA, 0xC6, 0x63, 0xF0, 0xE5, 0x86, 0x50,
		0x31, 0xE8, 0x75, 0xBA, 0x22, 0x4C, 0x06, 0x01,
		0x3C, 0x53, 0xD0, 0xE3, 0x01, 0x09, 0xC2, 0x07
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecfsdsa_secp256r1_test_vectors_priv_key[] = {
	0xAC, 0xCA, 0x7F, 0x0D, 0xD3, 0xAC, 0x53, 0x5F,
	0x48, 0x9B, 0x34, 0x0F, 0x6B, 0xD7, 0xF5, 0x03,
	0x61, 0xB0, 0xEE, 0x09, 0x5A, 0xE6, 0xA2, 0x28,
	0x9A, 0x6A, 0xB3, 0x29, 0x23, 0x81, 0x23, 0xE5
};

static const u8 ecfsdsa_secp256r1_test_vectors_expected_sig[] = {
	0xAF, 0x31, 0x2F, 0xBD, 0x77, 0x92, 0x12, 0x5C,
	0x5C, 0xDF, 0xBA, 0x69, 0xE6, 0xD3, 0x69, 0x90,
	0x0A, 0xCE, 0x9A, 0x70, 0xBA, 0x65, 0x3F, 0xFF,
	0xBD, 0x91, 0x40, 0xE0, 0x00, 0x79, 0xFA, 0xE8,
	0xB7, 0xCE, 0xC5, 0x70, 0x16, 0xA0, 0xB9, 0x7A,
	0xA0, 0x69, 0xD5, 0x4E, 0x0D, 0xA9, 0x5E, 0x45,
	0xFB, 0x50, 0xB6, 0x77, 0x1F, 0xB6, 0x9F, 0x53,
	0xFE, 0xF0, 0x0F, 0xC8, 0xB0, 0x0E, 0x1F, 0xEC,
	0x25, 0x84, 0x70, 0x40, 0x23, 0x04, 0xBC, 0x2D,
	0xB4, 0x4F, 0x3B, 0x2A, 0x20, 0xC0, 0x8F, 0xF2,
	0xA6, 0x4F, 0x56, 0x6B, 0xAA, 0x2E, 0xB7, 0xBF,
	0x37, 0xE1, 0x61, 0x9B, 0x6A, 0xE0, 0x98, 0x44
};

static const ec_test_case ecfsdsa_secp256r1_test_case = {
	.name = "ECFSDSA-SHA256/secp256r1",
	.ec_str_p = &secp256r1_str_params,
	.priv_key = ecfsdsa_secp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecfsdsa_secp256r1_test_vectors_priv_key),
	.nn_random = ecfsdsa_nn_random_iso14888_3_secp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECFSDSA,
	.exp_sig = ecfsdsa_secp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecfsdsa_secp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_SECP384R1
#define ECFSDSA_SHA384_SECP384R1_SELF_TEST

/* ECFSDSA secp384r1 test vectors */

static int ecfsdsa_nn_random_iso14888_3_secp384r1_test_vector(nn_t out,
							      nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x51, 0xC5, 0xB8, 0xB2, 0xE5, 0x9C, 0xF7, 0x8F,
		0x54, 0xE7, 0x7C, 0xDB, 0x0B, 0x2E, 0x26, 0x69,
		0xB6, 0x6B, 0x32, 0x84, 0x8B, 0x7B, 0x53, 0x78,
		0x01, 0x48, 0x3D, 0xE2, 0x39, 0x42, 0x27, 0x45,
		0x6F, 0x49, 0x30, 0xC8, 0x53, 0xFB, 0xFF, 0x6C,
		0x58, 0xFA, 0x6E, 0x1C, 0xC5, 0xD9, 0x74, 0x66
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecfsdsa_secp384r1_test_vectors_priv_key[] = {
	0x95, 0xA4, 0xD2, 0x57, 0xA7, 0x29, 0x8C, 0x66,
	0x10, 0xA3, 0x75, 0x58, 0x78, 0x50, 0x36, 0xDE,
	0x6F, 0x9F, 0xB9, 0x97, 0x73, 0x5C, 0x07, 0x6B,
	0x8C, 0x8A, 0x18, 0xB2, 0xAA, 0xAC, 0x31, 0x42,
	0x50, 0x7A, 0x25, 0x60, 0x3D, 0x7C, 0x95, 0xF9,
	0xE5, 0xF0, 0x30, 0x7E, 0xC5, 0xA5, 0x6D, 0x7E
};

static const u8 ecfsdsa_secp384r1_test_vectors_expected_sig[] = {
	0x29, 0xB7, 0x85, 0xAB, 0x7F, 0xAC, 0x1F, 0x80,
	0xF6, 0x4C, 0xE2, 0xF2, 0xD8, 0x8A, 0xBA, 0x8F,
	0xE6, 0x10, 0x3B, 0x25, 0x65, 0xFE, 0xB5, 0xAE,
	0x82, 0xFB, 0x6C, 0xF5, 0x8F, 0x8C, 0xE1, 0xE7,
	0x3F, 0x7A, 0x8D, 0x68, 0x3B, 0xEA, 0xBD, 0x24,
	0xB7, 0xEA, 0x78, 0xF6, 0x01, 0x3C, 0x9F, 0x70,
	0xDE, 0x94, 0xFA, 0xD1, 0x72, 0x81, 0xF4, 0xA6,
	0x0A, 0xA2, 0x54, 0x23, 0xE9, 0xD8, 0x71, 0x22,
	0x96, 0x54, 0x02, 0x19, 0x52, 0x39, 0xB8, 0x39,
	0xFC, 0xD4, 0x4C, 0xDF, 0x54, 0x5B, 0xF7, 0x4E,
	0x13, 0x00, 0xC9, 0xF5, 0x7F, 0xFC, 0x83, 0x0D,
	0xEF, 0xD9, 0x7B, 0x66, 0xD5, 0x7E, 0x9D, 0x07,

	0xD1, 0x85, 0x80, 0x62, 0xC5, 0x50, 0x4E, 0x21,
	0x78, 0x52, 0x39, 0x26, 0x42, 0x3F, 0xDD, 0x83,
	0x99, 0xA8, 0xBA, 0x2B, 0x85, 0xBF, 0x45, 0x85,
	0x3F, 0x8E, 0x04, 0xBF, 0x20, 0x44, 0x15, 0x16,
	0xE7, 0x1A, 0x78, 0xB0, 0x9C, 0x7A, 0x7E, 0xE6,
	0x20, 0xB7, 0xF5, 0x37, 0xE6, 0xC1, 0xDE, 0xEE
};

static const ec_test_case ecfsdsa_secp384r1_test_case = {
	.name = "ECFSDSA-SHA384/secp384r1",
	.ec_str_p = &secp384r1_str_params,
	.priv_key = ecfsdsa_secp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecfsdsa_secp384r1_test_vectors_priv_key),
	.nn_random = ecfsdsa_nn_random_iso14888_3_secp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECFSDSA,
	.exp_sig = ecfsdsa_secp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecfsdsa_secp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_SECP521R1
#define ECFSDSA_SHA512_SECP521R1_SELF_TEST

/* ECFSDSA secp521r1 test vectors */

static int ecfsdsa_nn_random_secp521r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for secp521r1 signature
	 * test vectors from RFC4754
	 */
	const u8 k_buf[] = {
		0x01, 0xbc, 0x24, 0x62, 0x6a, 0xd6, 0x10, 0x3c,
		0x72, 0xdc, 0x63, 0xb6, 0x8c, 0xe4, 0xcd, 0xb2,
		0xf0, 0x63, 0x39, 0x9c, 0x69, 0x69, 0x84, 0xc6,
		0x2f, 0x8c, 0x58, 0x5f, 0x12, 0x7d, 0x47, 0x71,
		0x6e, 0x0e, 0x9d, 0xd7, 0x2b, 0x27, 0x28, 0xfb,
		0xab, 0xed, 0xe4, 0x12, 0x0b, 0xcf, 0x83, 0x5a,
		0x9a, 0x32, 0xf1, 0x3e, 0x65, 0x1d, 0x18, 0x2a,
		0x69, 0x13, 0xfd, 0xeb, 0x9d, 0x79, 0xf5, 0xb0,
		0xbf, 0xe5
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecfsdsa_secp521r1_test_vectors_priv_key[] = {
	0x00, 0x9c, 0x0d, 0x38, 0xdb, 0x40, 0x76, 0x55,
	0x37, 0x79, 0xd9, 0x4a, 0x8f, 0x4f, 0x74, 0x2d,
	0x32, 0xd0, 0xf9, 0xf2, 0x9b, 0xea, 0x27, 0x60,
	0x47, 0x03, 0xcb, 0xe7, 0x34, 0x9e, 0x5f, 0x14,
	0x52, 0x5b, 0xfe, 0x57, 0x41, 0x50, 0x31, 0x52,
	0x0b, 0x19, 0xf7, 0x68, 0x70, 0x1a, 0x57, 0x0b,
	0x62, 0xa6, 0x4e, 0xc1, 0xc0, 0xd0, 0xe7, 0xa0,
	0x84, 0x94, 0x37, 0xfc, 0xf6, 0x7b, 0x56, 0x12,
	0x53, 0xc3
};

static const u8 ecfsdsa_secp521r1_test_vectors_expected_sig[] = {
	0x00, 0x43, 0xf7, 0x90, 0x80, 0xe4, 0xff, 0x8c,
	0x91, 0x44, 0x93, 0xad, 0xbb, 0x35, 0x3d, 0x78,
	0xbb, 0x15, 0x5d, 0x95, 0xde, 0x51, 0x9b, 0x78,
	0xe2, 0xa1, 0xfa, 0x34, 0x70, 0x0b, 0x45, 0x04,
	0x6a, 0xd2, 0xea, 0x99, 0xa7, 0x0c, 0xf8, 0x37,
	0x7a, 0xcf, 0x72, 0x89, 0x4b, 0xa3, 0xa5, 0x8c,
	0x5d, 0x50, 0xcb, 0x01, 0x3e, 0x3a, 0x3e, 0x19,
	0xc8, 0xcb, 0xf2, 0xca, 0xcf, 0x1e, 0x28, 0x47,
	0xcb, 0x92, 0x01, 0x70, 0x49, 0x0a, 0x35, 0xde,
	0xfd, 0x32, 0xc6, 0xd6, 0x08, 0x2a, 0xc5, 0x24,
	0x70, 0x51, 0x3f, 0x03, 0x93, 0x92, 0x71, 0xfb,
	0x8b, 0xa8, 0xbe, 0x1a, 0xe1, 0xc2, 0x64, 0xf9,
	0x86, 0x78, 0x5b, 0xb8, 0xd0, 0xa1, 0xcc, 0xd0,
	0xc8, 0xf6, 0x93, 0xb7, 0xff, 0x92, 0xe3, 0x72,
	0x20, 0xe5, 0x1c, 0x15, 0x80, 0xb7, 0x4d, 0xda,
	0x1c, 0x90, 0x79, 0x5b, 0x52, 0x5b, 0x23, 0x5b,
	0x1e, 0x0e, 0xc8, 0xb7,

	0x01, 0x0f, 0x45, 0xec, 0x7c, 0x8b, 0xe0, 0x8f,
	0x42, 0xb4, 0x4f, 0xe1, 0x7c, 0xde, 0x1e, 0xe4,
	0x9f, 0xa7, 0x47, 0xcb, 0x4a, 0x23, 0x42, 0xb7,
	0x72, 0x26, 0x9b, 0xc5, 0x96, 0x16, 0x8f, 0x82,
	0xa4, 0xc3, 0x44, 0x71, 0x1c, 0xdb, 0xf7, 0x20,
	0x3f, 0x45, 0x19, 0x93, 0x00, 0xb7, 0xc5, 0xc6,
	0xb1, 0xb6, 0xa1, 0x67, 0x9e, 0xe9, 0xba, 0x88,
	0xe6, 0x30, 0x13, 0x13, 0x85, 0xaf, 0x2d, 0x5d,
	0x82, 0xce
};

static const ec_test_case ecfsdsa_secp521r1_test_case = {
	.name = "ECFSDSA-SHA512/secp521r1",
	.ec_str_p = &secp521r1_str_params,
	.priv_key = ecfsdsa_secp521r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecfsdsa_secp521r1_test_vectors_priv_key),
	.nn_random = ecfsdsa_nn_random_secp521r1_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECFSDSA,
	.exp_sig = ecfsdsa_secp521r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecfsdsa_secp521r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_SECP521R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_BRAINPOOLP256R1
#define ECFSDSA_SHA256_BRAINPOOLP256R1_SELF_TEST

/* ECFSDSA Brainpoolp256r1 test vectors */

static int ecfsdsa_nn_random_brainpoolp256r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp256r1 signature
	 */
	const u8 k_buf[] = {
		0x94, 0x06, 0x79, 0x01, 0x57, 0xb2, 0x15, 0x15,
		0x35, 0x11, 0x37, 0xbc, 0xe0, 0x42, 0x8f, 0xc3,
		0xde, 0x3a, 0xe9, 0x86, 0xa2, 0xf3, 0xab, 0xac,
		0xaf, 0x0e, 0xa5, 0xcb, 0x24, 0x49, 0x19, 0x37
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecfsdsa_brainpoolp256r1_test_vectors_priv_key[] = {
	0x17, 0x86, 0x00, 0xbb, 0xf4, 0xf6, 0xe8, 0x92,
	0xbe, 0x80, 0x67, 0xc5, 0xd4, 0xdb, 0xd3, 0x90,
	0x67, 0x6c, 0xb8, 0x57, 0xc1, 0x68, 0x31, 0x0b,
	0x87, 0x3a, 0xdc, 0xe4, 0x74, 0xd3, 0xcf, 0x2d
};

static const u8 ecfsdsa_brainpoolp256r1_test_vectors_expected_sig[] = {
	0xa7, 0xf1, 0xc8, 0xcb, 0x02, 0x9f, 0x01, 0xda,
	0xb8, 0xf5, 0x68, 0x74, 0x57, 0x2c, 0xf0, 0xdb,
	0x75, 0x2f, 0x95, 0x8b, 0xb5, 0x23, 0x53, 0xd9,
	0x2d, 0x87, 0xec, 0x9f, 0x59, 0x53, 0x12, 0x3e,
	0x5e, 0x2a, 0x30, 0x64, 0x9b, 0xb8, 0xa2, 0x5f,
	0x8f, 0xcb, 0xe1, 0xd9, 0x7f, 0xb4, 0x98, 0x7d,
	0x2d, 0xad, 0xd8, 0x1b, 0x05, 0xdd, 0x8b, 0x94,
	0xbe, 0x1a, 0xc5, 0xc9, 0x14, 0x05, 0x09, 0x52,

	0x2d, 0xf7, 0x9a, 0x80, 0xaa, 0xa9, 0x57, 0x82,
	0xff, 0x96, 0x44, 0x62, 0xce, 0xd9, 0x68, 0xf5,
	0x11, 0xc6, 0xb5, 0x1a, 0x8e, 0xb9, 0x44, 0xd1,
	0xf9, 0xe7, 0xbc, 0xb3, 0xd0, 0xd8, 0x98, 0xc4
};

static const ec_test_case ecfsdsa_brainpoolp256r1_test_case = {
	.name = "ECFSDSA-SHA256/brainpoolp256r1",
	.ec_str_p = &brainpoolp256r1_str_params,
	.priv_key = ecfsdsa_brainpoolp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecfsdsa_brainpoolp256r1_test_vectors_priv_key),
	.nn_random = ecfsdsa_nn_random_brainpoolp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECFSDSA,
	.exp_sig = ecfsdsa_brainpoolp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecfsdsa_brainpoolp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_BRAINPOOLP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_BRAINPOOLP384R1
#define ECFSDSA_SHA384_BRAINPOOLP384R1_SELF_TEST

/* ECFSDSA Brainpoolp384r1 test vectors */

static int ecfsdsa_nn_random_brainpoolp384r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp384r1 signature
	 */
	const u8 k_buf[] = {
		0x20, 0x89, 0x69, 0xde, 0xcd, 0xd0, 0x61, 0xd9,
		0xee, 0x13, 0xae, 0xdc, 0xbe, 0xd0, 0xd3, 0x3c,
		0x6e, 0xe7, 0xf7, 0x57, 0xd7, 0xfd, 0xe4, 0xb2,
		0x1e, 0xbc, 0xbd, 0xd5, 0x21, 0x57, 0x86, 0xf5,
		0x28, 0x3e, 0x65, 0x81, 0x46, 0x92, 0xbd, 0xc0,
		0x38, 0x82, 0x52, 0xce, 0x58, 0x3d, 0xf5, 0x77
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecfsdsa_brainpoolp384r1_test_vectors_priv_key[] = {
	0x1c, 0x5b, 0xdb, 0xfa, 0xf3, 0x7e, 0x68, 0x87,
	0xd3, 0xf9, 0x10, 0xa6, 0x45, 0x14, 0x4c, 0x95,
	0xe6, 0xcc, 0x48, 0xe7, 0x8c, 0xfa, 0x6b, 0x05,
	0xac, 0xf5, 0x0a, 0xcd, 0x51, 0x38, 0xba, 0xba,
	0xcd, 0x26, 0x01, 0x50, 0x8a, 0x86, 0x48, 0xd3,
	0x87, 0x82, 0x37, 0xb1, 0xca, 0xf1, 0xd8, 0xcf
};

static const u8 ecfsdsa_brainpoolp384r1_test_vectors_expected_sig[] = {
	0x7e, 0xa5, 0xf5, 0x97, 0x64, 0x8a, 0x55, 0x0c,
	0x5a, 0xb1, 0xaa, 0x74, 0xd4, 0x40, 0x42, 0xdb,
	0xf8, 0xaf, 0xf2, 0x71, 0xad, 0x14, 0xaf, 0xc3,
	0x89, 0x0c, 0xe4, 0x62, 0x2d, 0x91, 0xdb, 0x94,
	0x87, 0xa0, 0xbf, 0x3a, 0xf0, 0x55, 0xf1, 0xf6,
	0xaa, 0x60, 0x8c, 0x83, 0x62, 0xe2, 0x7f, 0xf5,
	0x49, 0xc4, 0xe6, 0xa7, 0x6c, 0xbc, 0x03, 0xac,
	0xb0, 0x7d, 0x6b, 0x75, 0x25, 0xe5, 0x5e, 0xfd,
	0x97, 0xdd, 0x5c, 0xec, 0x09, 0x55, 0x0b, 0x58,
	0x47, 0x0a, 0x9b, 0x7e, 0x07, 0xe9, 0x5d, 0x4b,
	0x86, 0xf6, 0xfa, 0xe3, 0xe8, 0xb7, 0x66, 0xa9,
	0xca, 0x40, 0xe7, 0x4a, 0xc4, 0x44, 0x79, 0xe7,

	0x3d, 0x33, 0x34, 0xaf, 0x71, 0x29, 0x4c, 0x1c,
	0xef, 0x28, 0x2f, 0xdb, 0xbf, 0xf2, 0x4e, 0x7f,
	0xde, 0xf0, 0xb9, 0xe2, 0x4c, 0xaa, 0xdf, 0x56,
	0x52, 0xcc, 0x1e, 0xb5, 0x33, 0x6e, 0x29, 0x8e,
	0x29, 0x8f, 0x2c, 0x5b, 0x0f, 0x2f, 0x1f, 0x8c,
	0xb8, 0xd8, 0x2a, 0x8f, 0x12, 0x5b, 0x08, 0xc9
};

static const ec_test_case ecfsdsa_brainpoolp384r1_test_case = {
	.name = "ECFSDSA-SHA384/brainpoolp384r1",
	.ec_str_p = &brainpoolp384r1_str_params,
	.priv_key = ecfsdsa_brainpoolp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecfsdsa_brainpoolp384r1_test_vectors_priv_key),
	.nn_random = ecfsdsa_nn_random_brainpoolp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECFSDSA,
	.exp_sig = ecfsdsa_brainpoolp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecfsdsa_brainpoolp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_BRAINPOOLP384R1 */
#endif /* WITH_HASH_SHA384 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_BRAINPOOLP512R1
#define ECFSDSA_SHA512_BRAINPOOLP512R1_SELF_TEST

/* ECFSDSA Brainpoolp512r1 test vectors */

static int ecfsdsa_nn_random_brainpoolp512r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for brainpoolp512r1 signature
	 */
	const u8 k_buf[] = {
		0x5f, 0x84, 0x57, 0xa0, 0xbc, 0xd3, 0x49, 0xd9,
		0x7d, 0x0a, 0xc7, 0x95, 0x49, 0x49, 0x43, 0x03,
		0x06, 0x37, 0x2b, 0x63, 0xfc, 0x73, 0x14, 0xaa,
		0xe0, 0x23, 0x4d, 0x96, 0x66, 0x53, 0x46, 0xea,
		0xa7, 0x8a, 0x60, 0x05, 0xa6, 0x0f, 0xc4, 0x16,
		0x11, 0x4c, 0xbc, 0x02, 0x67, 0x6d, 0x49, 0xe6,
		0xea, 0x92, 0x21, 0xcd, 0xe7, 0xd7, 0x2a, 0x45,
		0xf7, 0xc2, 0xf5, 0x07, 0xfe, 0x39, 0x65, 0x52
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecfsdsa_brainpoolp512r1_test_vectors_priv_key[] = {
	0xaa, 0x85, 0x90, 0x2b, 0xdc, 0x3a, 0x89, 0xb9,
	0xa2, 0xa6, 0x12, 0x80, 0xb2, 0xd5, 0xd4, 0xea,
	0xb0, 0xdd, 0x69, 0xb6, 0x45, 0xd0, 0x2b, 0xfd,
	0xb1, 0xa6, 0xa9, 0x36, 0x39, 0xff, 0x27, 0x1e,
	0xf3, 0x52, 0x9f, 0xbc, 0xca, 0x45, 0xe5, 0x7b,
	0x46, 0x3a, 0x3c, 0xe3, 0x88, 0x90, 0x44, 0xeb,
	0x2f, 0x6d, 0x09, 0x80, 0x47, 0x50, 0xe7, 0xf9,
	0x43, 0x10, 0x80, 0xe0, 0xb0, 0x5c, 0x2b, 0xab
};

static const u8 ecfsdsa_brainpoolp512r1_test_vectors_expected_sig[] = {
	0x3e, 0xe9, 0x03, 0x40, 0x75, 0x7a, 0x19, 0xae,
	0x24, 0x2f, 0xf9, 0x89, 0x77, 0xb8, 0x23, 0x54,
	0xf3, 0x7e, 0x68, 0xda, 0x32, 0xcc, 0x80, 0xf7,
	0x83, 0x5f, 0x1f, 0xd6, 0xf9, 0x64, 0x62, 0x64,
	0x0f, 0xc2, 0xbb, 0x6e, 0x83, 0x3e, 0x2e, 0xd6,
	0x29, 0x37, 0x1f, 0x43, 0xd3, 0x83, 0xa9, 0x9b,
	0x62, 0x25, 0x68, 0x7f, 0xd5, 0xf9, 0x9b, 0x07,
	0xfb, 0x4b, 0x95, 0x30, 0x5f, 0x25, 0xa5, 0x4d,
	0x58, 0x5a, 0xf5, 0x95, 0xd6, 0x07, 0x92, 0xa4,
	0xdf, 0x01, 0xf0, 0x42, 0x2f, 0x39, 0x94, 0x9c,
	0xf9, 0x8b, 0xe4, 0x0e, 0x9f, 0xb3, 0x8b, 0x40,
	0xc4, 0xaa, 0x2d, 0x49, 0x36, 0xf7, 0x0f, 0x71,
	0x7e, 0x30, 0x3e, 0xe7, 0xba, 0x0f, 0x44, 0x28,
	0xa4, 0xfd, 0x18, 0x9d, 0x88, 0xe8, 0x5c, 0x5e,
	0x69, 0xf8, 0x7b, 0xb7, 0x44, 0x44, 0x0a, 0x41,
	0xfd, 0x79, 0x37, 0x14, 0x54, 0x49, 0x6b, 0xc1,

	0x29, 0x0a, 0x96, 0x8b, 0x0d, 0x46, 0xbe, 0x9c,
	0x5e, 0x3e, 0x1c, 0x1e, 0xa3, 0x09, 0xb3, 0x03,
	0x10, 0x81, 0x6b, 0x08, 0x01, 0xab, 0x8f, 0x20,
	0x0e, 0xf7, 0x7c, 0x86, 0xfb, 0xf4, 0xfc, 0xc3,
	0xeb, 0x43, 0x8f, 0xf9, 0xd7, 0x6e, 0xc0, 0x62,
	0x2e, 0xbc, 0x37, 0xa4, 0xbe, 0x14, 0x12, 0x26,
	0xc2, 0xa4, 0x40, 0xd6, 0x90, 0x8c, 0xe4, 0x73,
	0x0c, 0x5c, 0xf3, 0xbb, 0x7b, 0xf7, 0x6c, 0x7f
};

static const ec_test_case ecfsdsa_brainpoolp512r1_test_case = {
	.name = "ECFSDSA-SHA512/brainpoolp512r1",
	.ec_str_p = &brainpoolp512r1_str_params,
	.priv_key = ecfsdsa_brainpoolp512r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecfsdsa_brainpoolp512r1_test_vectors_priv_key),
	.nn_random = ecfsdsa_nn_random_brainpoolp512r1_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECFSDSA,
	.exp_sig = ecfsdsa_brainpoolp512r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecfsdsa_brainpoolp512r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP512R1 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_FRP256V1
#define ECFSDSA_SHA256_FRP256V1_SELF_TEST

/* ECFSDSA frp256v1 test vectors */

static int ecfsdsa_nn_random_frp256v1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	/*
	 * Fixed ephemeral private key for frp256v1 signature
	 */
	const u8 k_buf[] = {
		0x9E, 0x56, 0xF5, 0x09, 0x19, 0x67, 0x84, 0xD9,
		0x63, 0xD1, 0xC0, 0xA4, 0x01, 0x51, 0x0E, 0xE7,
		0xAD, 0xA3, 0xDC, 0xC5, 0xDE, 0xE0, 0x4B, 0x15,
		0x4B, 0xF6, 0x1A, 0xF1, 0xD5, 0xA6, 0xDE, 0xCE
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecfsdsa_frp256v1_test_vectors_priv_key[] = {
	0x00, 0x51, 0xD3, 0x86, 0x6A, 0x15, 0xBA, 0xCD,
	0xE3, 0x3D, 0x96, 0xF9, 0x92, 0xFC, 0xA9, 0x9D,
	0xA7, 0xE6, 0xEF, 0x09, 0x34, 0xE7, 0x09, 0x75,
	0x59, 0xC2, 0x7F, 0x16, 0x14, 0xC8, 0x8A, 0x7F
};

static const u8 ecfsdsa_frp256v1_test_vectors_expected_sig[] = {
	0x22, 0xb1, 0xd0, 0x0d, 0xc8, 0x28, 0xa5, 0xcb,
	0xa2, 0x64, 0x31, 0x9f, 0x05, 0x93, 0x3a, 0xc8,
	0xed, 0xe6, 0x32, 0x3d, 0x69, 0xd0, 0x2b, 0x31,
	0x4e, 0x10, 0xca, 0xa5, 0xf1, 0x3b, 0x29, 0x96,
	0x52, 0xe2, 0xef, 0x4e, 0x46, 0x9f, 0x08, 0x63,
	0xfb, 0x3c, 0xf4, 0xc5, 0x18, 0xb0, 0x5f, 0xc4,
	0xbb, 0x1d, 0xaf, 0xb7, 0xd2, 0xc4, 0x30, 0x7e,
	0xda, 0x41, 0xef, 0xac, 0x88, 0x8f, 0x0c, 0x7d,

	0x9b, 0x3a, 0xb9, 0xd5, 0xb6, 0x7f, 0xf2, 0xb9,
	0xaa, 0xe8, 0xc5, 0xed, 0xf9, 0x82, 0xd4, 0xf7,
	0x6d, 0x7c, 0x61, 0x36, 0x56, 0xe7, 0x73, 0xaa,
	0x1a, 0xab, 0xa6, 0x30, 0x90, 0x1d, 0x56, 0xa6
};

static const ec_test_case ecfsdsa_frp256v1_test_case = {
	.name = "ECFSDSA-SHA256/frp256v1",
	.ec_str_p = &frp256v1_str_params,
	.priv_key = ecfsdsa_frp256v1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecfsdsa_frp256v1_test_vectors_priv_key),
	.nn_random = ecfsdsa_nn_random_frp256v1_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECFSDSA,
	.exp_sig = ecfsdsa_frp256v1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecfsdsa_frp256v1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_FRP256V1 */
#endif /* WITH_HASH_SHA256 */
#endif /* WITH_SIG_ECFSDSA */

/*******************************************************************
 ************** ECGDSA tests **************************************
 *******************************************************************/
#ifdef WITH_SIG_ECGDSA
#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_BRAINPOOLP192R1
#define ECGDSA_SHA256_BRAINPOOLP192R1_SELF_TEST

/* ECGDSA brainpoolp192r1 test vectors */

static int ecgdsa_nn_random_iso14888_3_brainpoolp192r1_test_vector(nn_t out,
								   nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x5A, 0x96, 0x62, 0x60, 0x96, 0x28, 0x8C, 0xC4,
		0x69, 0xF1, 0x70, 0x4E, 0xC0, 0x5F, 0x44, 0xD1,
		0xEC, 0x18, 0xBD, 0x32, 0xCE, 0xB0, 0x2D, 0x5B,
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecgdsa_brainpoolp192r1_test_vectors_priv_key[] = {
	0x40, 0xF9, 0x5B, 0x49, 0xA3, 0xB1, 0xBF, 0x55,
	0x31, 0x1A, 0x56, 0xDF, 0xD3, 0xB5, 0x06, 0x1E,
	0xE1, 0xDF, 0x64, 0x39, 0x84, 0xD4, 0x1E, 0x35,
};

static const u8 ecgdsa_brainpoolp192r1_test_vectors_expected_sig[] = {
	0xA0, 0x0B, 0x0A, 0xA2, 0x5D, 0xB6, 0xAB, 0x5C,
	0x21, 0xB8, 0x63, 0x00, 0xD9, 0xBC, 0x99, 0xF5,
	0x6E, 0x9D, 0xD1, 0xB7, 0xF1, 0xDC, 0x47, 0x74,
	0x63, 0x46, 0x35, 0xEF, 0x81, 0x32, 0x47, 0xD7,
	0x20, 0x24, 0x5C, 0x94, 0x09, 0xFB, 0x20, 0xA2,
	0x67, 0xC5, 0x60, 0xC8, 0x8E, 0xB2, 0xB0, 0x7B,
};

static const ec_test_case ecgdsa_brainpoolp192r1_test_case = {
	.name = "ECGDSA-SHA256/brainpoolp192r1",
	.ec_str_p = &brainpoolp192r1_str_params,
	.priv_key = ecgdsa_brainpoolp192r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecgdsa_brainpoolp192r1_test_vectors_priv_key),
	.nn_random = ecgdsa_nn_random_iso14888_3_brainpoolp192r1_test_vector,
	.hash_type = SHA256,
	.msg = "brainpoolP192r1",
	.msglen = 15,
	.sig_type = ECGDSA,
	.exp_sig = ecgdsa_brainpoolp192r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecgdsa_brainpoolp192r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP192R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA224
#ifdef WITH_CURVE_BRAINPOOLP224R1
#define ECGDSA_SHA224_BRAINPOOLP224R1_SELF_TEST

/* ECGDSA brainpoolp224r1 test vectors */

static int ecgdsa_nn_random_iso14888_3_brainpoolp224r1_test_vector(nn_t out,
								   nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x5B, 0x60, 0x4F, 0x2C, 0x35, 0xED, 0x04, 0x01,
		0xFC, 0xA3, 0x1E, 0x88, 0x0C, 0xB5, 0x5C, 0x2A,
		0x74, 0x56, 0xE7, 0x1A, 0x5C, 0xBA, 0xA8, 0xDF,
		0x2F, 0xC0, 0x3C, 0xA9
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecgdsa_brainpoolp224r1_test_vectors_priv_key[] = {
	0x7E, 0x75, 0xBC, 0x2C, 0xD5, 0x73, 0xB3, 0x8A,
	0xED, 0x09, 0x77, 0xAD, 0x61, 0x17, 0x63, 0xDD,
	0x57, 0xFB, 0x29, 0xB2, 0x20, 0x88, 0x33, 0x44,
	0xB8, 0x1D, 0xF0, 0x37
};

static const u8 ecgdsa_brainpoolp224r1_test_vectors_expected_sig[] = {
	0x60, 0xFB, 0xB2, 0xB1, 0x5F, 0x05, 0x5C, 0xD1,
	0xD4, 0x82, 0xED, 0x6D, 0xC5, 0x06, 0x9C, 0x8F,
	0x62, 0x4A, 0x34, 0x05, 0xB6, 0x7D, 0x11, 0xB3,
	0xB6, 0x5E, 0x02, 0x34, 0x5A, 0x05, 0x0F, 0x05,
	0xAF, 0x0B, 0x10, 0x6B, 0xA3, 0xF1, 0x46, 0x96,
	0xE6, 0x16, 0x2C, 0xA4, 0x6F, 0xBA, 0xBD, 0x2C,
	0x14, 0x44, 0x19, 0xDB, 0xB5, 0xBF, 0xBD, 0xC0
};

static const ec_test_case ecgdsa_brainpoolp224r1_test_case = {
	.name = "ECGDSA-SHA224/brainpoolp224r1",
	.ec_str_p = &brainpoolp224r1_str_params,
	.priv_key = ecgdsa_brainpoolp224r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecgdsa_brainpoolp224r1_test_vectors_priv_key),
	.nn_random = ecgdsa_nn_random_iso14888_3_brainpoolp224r1_test_vector,
	.hash_type = SHA224,
	.msg = "brainpoolP224r1",
	.msglen = 15,
	.sig_type = ECGDSA,
	.exp_sig = ecgdsa_brainpoolp224r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecgdsa_brainpoolp224r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP224R1 */
#endif /* WITH_HASH_SHA224 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_BRAINPOOLP256R1
#define ECGDSA_SHA256_BRAINPOOLP256R1_SELF_TEST

/* ECGDSA brainpoolp256r1 test vectors */

static int ecgdsa_nn_random_iso14888_3_brainpoolp256r1_test_vector(nn_t out,
								   nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x0E, 0x64, 0x21, 0x27, 0x2D, 0xDA, 0xB9, 0xC2,
		0x07, 0xB1, 0x19, 0xBD, 0xD1, 0x0C, 0x03, 0x86,
		0x10, 0x05, 0x75, 0x2E, 0xEA, 0xBB, 0x3A, 0xC9,
		0x75, 0x13, 0x04, 0x1A, 0xDE, 0x62, 0x86, 0xD9
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecgdsa_brainpoolp256r1_test_vectors_priv_key[] = {
	0x52, 0xB9, 0x29, 0xB4, 0x02, 0x97, 0x43, 0x7B,
	0x98, 0x97, 0x3A, 0x2C, 0x43, 0x7E, 0x8F, 0x03,
	0xA2, 0x31, 0xEB, 0x61, 0xE0, 0xCD, 0x38, 0xFD,
	0xAD, 0x80, 0x2F, 0x00, 0xD5, 0x5A, 0x13, 0xA3
};

static const u8 ecgdsa_brainpoolp256r1_test_vectors_expected_sig[] = {
	0x82, 0x93, 0x49, 0xe3, 0xb6, 0xe1, 0xf3, 0xe5,
	0x15, 0xeb, 0x95, 0x81, 0xbe, 0x0f, 0x95, 0x8d,
	0xcc, 0xaa, 0xa6, 0xb6, 0x8d, 0x83, 0xba, 0x77,
	0x01, 0xdd, 0x7a, 0x08, 0x67, 0xe4, 0x4e, 0xa7,
	0x3d, 0xc2, 0xf1, 0x03, 0x29, 0x6a, 0x79, 0x3e,
	0x50, 0xdc, 0x22, 0x66, 0x65, 0x74, 0x70, 0xa4,
	0x0d, 0x2c, 0x9e, 0xa1, 0xca, 0x79, 0x7d, 0xea,
	0x61, 0x00, 0x42, 0xb7, 0x73, 0x0b, 0xbd, 0xce
};

static const ec_test_case ecgdsa_brainpoolp256r1_test_case = {
	.name = "ECGDSA-SHA256/brainpoolp256r1",
	.ec_str_p = &brainpoolp256r1_str_params,
	.priv_key = ecgdsa_brainpoolp256r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecgdsa_brainpoolp256r1_test_vectors_priv_key),
	.nn_random = ecgdsa_nn_random_iso14888_3_brainpoolp256r1_test_vector,
	.hash_type = SHA256,
	.msg = "brainpoolP256r1",
	.msglen = 15,
	.sig_type = ECGDSA,
	.exp_sig = ecgdsa_brainpoolp256r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecgdsa_brainpoolp256r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP256R1 */
#endif /* WITH_HASH_SHA256 */

#ifdef WITH_HASH_SHA384
#ifdef WITH_CURVE_BRAINPOOLP384R1
#define ECGDSA_SHA384_BRAINPOOLP384R1_SELF_TEST

/* ECGDSA brainpoolp384r1 test vectors */

static int ecgdsa_nn_random_brainpoolp384r1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;

	const u8 k_buf[] = {
		0x39, 0x1c, 0x0c, 0xff, 0xe3, 0xaa, 0x7e, 0x95,
		0x47, 0xd2, 0x3e, 0xe9, 0x70, 0x36, 0x12, 0x55,
		0x15, 0xce, 0x74, 0xd6, 0xfe, 0x55, 0x44, 0x07,
		0x17, 0x79, 0x64, 0xfd, 0xee, 0xce, 0x9d, 0x88,
		0xb3, 0xeb, 0xdc, 0xed, 0x5e, 0xe6, 0x34, 0xf3,
		0x0c, 0x3a, 0x74, 0xaa, 0xad, 0x54, 0x57, 0x0a
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecgdsa_brainpoolp384r1_test_vectors_priv_key[] = {
	0x70, 0x74, 0x93, 0xc8, 0x39, 0x9a, 0x07, 0x95,
	0xd1, 0xa3, 0xd2, 0x36, 0xae, 0x60, 0xa5, 0x46,
	0x86, 0xec, 0xc5, 0x7c, 0x1e, 0xc5, 0x51, 0xef,
	0x3f, 0x63, 0x19, 0x3a, 0x09, 0xd9, 0xb5, 0x9c,
	0xae, 0x50, 0x04, 0xee, 0x2d, 0x4d, 0x48, 0x71,
	0xf8, 0x5a, 0xcb, 0x1a, 0x60, 0x36, 0xbb, 0x68
};

static const u8 ecgdsa_brainpoolp384r1_test_vectors_expected_sig[] = {
	0x3b, 0x34, 0xbe, 0xb7, 0xa6, 0x9e, 0x07, 0x6d,
	0xd9, 0x6f, 0x2d, 0x71, 0x2a, 0xfe, 0xe2, 0xf2,
	0x72, 0xa4, 0xd7, 0xad, 0xdf, 0x29, 0x1c, 0xa7,
	0x99, 0xed, 0x62, 0x49, 0xa9, 0xbb, 0x4e, 0xcd,
	0xcb, 0xe4, 0x44, 0x88, 0x31, 0xc5, 0x49, 0xb7,
	0x8f, 0xc7, 0x2b, 0xa5, 0x5a, 0x34, 0x66, 0xa6,

	0x79, 0xa0, 0x47, 0x25, 0x3d, 0xad, 0x09, 0x80,
	0xe1, 0x3f, 0x99, 0xf3, 0x20, 0x4a, 0xd9, 0x7b,
	0xfe, 0x2c, 0xa2, 0x09, 0xdd, 0x33, 0xbe, 0xd5,
	0x7f, 0xe0, 0xb5, 0x1d, 0x52, 0x1d, 0x2e, 0x61,
	0xb2, 0x78, 0x81, 0x40, 0x90, 0x57, 0x02, 0xe5,
	0x1a, 0xdf, 0x44, 0x84, 0x9f, 0x0c, 0x4b, 0xce
};

static const ec_test_case ecgdsa_brainpoolp384r1_test_case = {
	.name = "ECGDSA-SHA384/brainpoolp384r1",
	.ec_str_p = &brainpoolp384r1_str_params,
	.priv_key = ecgdsa_brainpoolp384r1_test_vectors_priv_key,
	.priv_key_len = sizeof(ecgdsa_brainpoolp384r1_test_vectors_priv_key),
	.nn_random = ecgdsa_nn_random_brainpoolp384r1_test_vector,
	.hash_type = SHA384,
	.msg = "brainpoolP384r1",
	.msglen = 15,
	.sig_type = ECGDSA,
	.exp_sig = ecgdsa_brainpoolp384r1_test_vectors_expected_sig,
	.exp_siglen = sizeof(ecgdsa_brainpoolp384r1_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_BRAINPOOLP384R1 */
#endif /* WITH_HASH_SHA384 */
#endif /* WITH_SIG_ECGDSA */

/*******************************************************************
 ************** ECRDSA tests ***************************************
 *******************************************************************/
/*
 * NOTE: ISO/IEC 14888-3 standard seems to diverge from the existing implementations
 * of ECRDSA when treating the message hash, and from the examples of certificates provided
 * in RFC 7091 and draft-deremin-rfc4491-bis. While in ISO/IEC 14888-3 it is explicitely asked
 * to proceed with the hash of the message as big endian, the RFCs derived from the Russian
 * standard expect the hash value to be treated as little endian when importing it as an integer
 * (this discrepancy is exhibited and confirmed by test vectors present in ISO/IEC 14888-3, and
 * by X.509 certificates present in the RFCs). This seems (to be confirmed) to be a discrepancy of
 * ISO/IEC 14888-3 algorithm description that must be fixed there.
 *
 * In order to be conservative, libecc uses the Russian standard behavior as expected to be in line with
 * other implemetations, but keeps the ISO/IEC 14888-3 behavior if forced/asked by the user using
 * the USE_ISO14888_3_ECRDSA toggle. This allows to keep backward compatibility with previous versions of the
 * library if needed.
 *
 */

#ifdef WITH_SIG_ECRDSA

#ifndef USE_ISO14888_3_ECRDSA

/*
 * All the follwing test vectors are extracted from draft-deremin-rfc4491-bis RFC draft
 * (https://datatracker.ietf.org/doc/html/draft-deremin-rfc4491-bis) where certificates signed with
 * GOST R 34.10-2012 using GOST R 34.11-2012 (256 and 512 bits) are provided. We can use them
 * as test vectors with the message being the TBS (To Be Signed) blob of the certificate.
 */

#ifdef WITH_HASH_STREEBOG256
#ifdef WITH_CURVE_GOST256
#define ECRDSA_STREEBOG256_GOST256_SELF_TEST

static int ecrdsa_nn_random_rfc4491_bis_1_GOST_256bits_curve_test_vector(nn_t out, nn_src_t q){
	int ret, cmp;
	const u8 k_buf[] = {
		0x77, 0x10, 0x5c, 0x9b, 0x20, 0xbc, 0xd3, 0x12, 0x28, 0x23, 0xc8, 0xcf,
		0x6f, 0xcc, 0x7b, 0x95, 0x6d, 0xe3, 0x38, 0x14, 0xe9, 0x5b, 0x7f, 0xe6,
		0x4f, 0xed, 0x92, 0x45, 0x94, 0xdc, 0xea, 0xb3,
	};
        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_vector_priv_key[] = {
	0x7a, 0x92, 0x9a, 0xde, 0x78, 0x9b, 0xb9, 0xbe, 0x10, 0xed,
	0x35, 0x9d, 0xd3, 0x9a, 0x72, 0xc1, 0x1b, 0x60, 0x96, 0x1f,
	0x49, 0x39, 0x7e, 0xee, 0x1d, 0x19, 0xce, 0x98, 0x91, 0xec,
	0x3b, 0x28,
};
static const u8 ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_vector_expected_sig[] = {
	0x41, 0xaa, 0x28, 0xd2, 0xf1, 0xab, 0x14, 0x82, 0x80, 0xcd,
	0x9e, 0xd5, 0x6f, 0xed, 0xa4, 0x19, 0x74, 0x05, 0x35, 0x54,
	0xa4, 0x27, 0x67, 0xb8, 0x3a, 0xd0, 0x43, 0xfd, 0x39, 0xdc,
	0x04, 0x93,
	0x4d, 0x53, 0xf0, 0x12, 0xfe, 0x08, 0x17, 0x76, 0x50, 0x7d,
	0x4d, 0x9b, 0xb8, 0x1f, 0x00, 0xef, 0xdb, 0x4e, 0xef, 0xd4,
	0xab, 0x83, 0xba, 0xc4, 0xba, 0xcf, 0x73, 0x51, 0x73, 0xcf,
	0xa8, 0x1c,
};
static const unsigned char ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_vector_message[] = {
	0x30, 0x81, 0xdb, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x0a, 0x30,
	0x0a, 0x06, 0x08, 0x2a, 0x85, 0x03, 0x07, 0x01, 0x01, 0x03, 0x02, 0x30,
	0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x07,
	0x45, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x30, 0x20, 0x17, 0x0d, 0x30,
	0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a,
	0x18, 0x0f, 0x32, 0x30, 0x35, 0x30, 0x31, 0x32, 0x33, 0x31, 0x30, 0x30,
	0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06,
	0x03, 0x55, 0x04, 0x03, 0x13, 0x07, 0x45, 0x78, 0x61, 0x6d, 0x70, 0x6c,
	0x65, 0x30, 0x66, 0x30, 0x1f, 0x06, 0x08, 0x2a, 0x85, 0x03, 0x07, 0x01,
	0x01, 0x01, 0x01, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x85, 0x03, 0x02, 0x02,
	0x23, 0x00, 0x06, 0x08, 0x2a, 0x85, 0x03, 0x07, 0x01, 0x01, 0x02, 0x02,
	0x03, 0x43, 0x00, 0x04, 0x40, 0x0b, 0xd8, 0x6f, 0xe5, 0xd8, 0xdb, 0x89,
	0x66, 0x8f, 0x78, 0x9b, 0x4e, 0x1d, 0xba, 0x85, 0x85, 0xc5, 0x50, 0x8b,
	0x45, 0xec, 0x5b, 0x59, 0xd8, 0x90, 0x6d, 0xdb, 0x70, 0xe2, 0x49, 0x2b,
	0x7f, 0xda, 0x77, 0xff, 0x87, 0x1a, 0x10, 0xfb, 0xdf, 0x27, 0x66, 0xd2,
	0x93, 0xc5, 0xd1, 0x64, 0xaf, 0xbb, 0x3c, 0x7b, 0x97, 0x3a, 0x41, 0xc8,
	0x85, 0xd1, 0x1d, 0x70, 0xd6, 0x89, 0xb4, 0xf1, 0x26, 0xa3, 0x13, 0x30,
	0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
	0x05, 0x30, 0x03, 0x01, 0x01, 0xff,
};
static const ec_test_case ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_case = {
	.name="ECRDSA-STREEBOG256/GOST-256-curve (RFC4491)",
	.ec_str_p = &GOST_256bits_curve_str_params,
	.priv_key = ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_vector_priv_key,
	.priv_key_len = sizeof(ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_vector_priv_key),
	.nn_random = ecrdsa_nn_random_rfc4491_bis_1_GOST_256bits_curve_test_vector,
	.hash_type = STREEBOG256,
	.msg = (const char*)ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_vector_message,
	.msglen = sizeof(ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_vector_message),
	.sig_type = ECRDSA,
	.exp_sig = ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_vector_expected_sig,
	.exp_siglen = sizeof(ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_vector_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_GOST256 */
#endif /* WITH_HASH_STREEBOG256 */

#ifdef WITH_HASH_STREEBOG256
#ifdef WITH_CURVE_GOST_R3410_2012_256_PARAMSETA
#define ECRDSA_STREEBOG256_GOST256_PARAMSETA_SELF_TEST

static int ecrdsa_nn_random_rfc4491_bis_2_GOST_256bits_curve_test_vector(nn_t out, nn_src_t q){
	int ret, cmp;
	const u8 k_buf[] = {
		0x27, 0x10, 0x5c, 0x9b, 0x20, 0xbc, 0xd3, 0x12, 0x28, 0x23,
		0xc8, 0xcf, 0x6f, 0xcc, 0x7b, 0x95, 0x6d, 0xe3, 0x38, 0x14,
		0xe9, 0x5b, 0x7f, 0xe6, 0x4f, 0xed, 0x92, 0x45, 0x94, 0xdc,
		0xea, 0xb3,
	};
        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_vector_priv_key[] = {
	0x3a, 0x92, 0x9a, 0xde, 0x78, 0x9b, 0xb9, 0xbe, 0x10,
	0xed, 0x35, 0x9d, 0xd3, 0x9a, 0x72, 0xc1, 0x0b, 0x87,
	0xc8, 0x3f, 0x80, 0xbe, 0x18, 0xb8, 0x5c, 0x04, 0x1f,
	0x43, 0x25, 0xb6, 0x2e, 0xc1,
};
static const u8 ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_vector_expected_sig[] = {
	0x1d, 0x0e, 0x1d, 0xa5, 0xbe, 0x34, 0x7c, 0x6f, 0x1b,
	0x52, 0x56, 0xc7, 0xae, 0xac, 0x20, 0x0a, 0xd6, 0x4a,
	0xc7, 0x7a, 0x6f, 0x5b, 0x3a, 0x0e, 0x09, 0x73, 0x18,
	0xe7, 0xae, 0x6e, 0xe7, 0x69,
	0x14, 0x0b, 0x4d, 0xa9, 0x12, 0x4b, 0x09, 0xcb, 0x0d,
	0x5c, 0xe9, 0x28, 0xee, 0x87, 0x42, 0x73, 0xa3, 0x10,
	0x12, 0x94, 0x92, 0xec, 0x0e, 0x29, 0x36, 0x9e, 0x3b,
	0x79, 0x12, 0x48, 0x57, 0x8c,
};
static const unsigned char ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_vector_message[] = {
	0x30, 0x81, 0xd3, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x0a, 0x30,
	0x0a, 0x06, 0x08, 0x2a, 0x85, 0x03, 0x07, 0x01, 0x01, 0x03, 0x02, 0x30,
	0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x07,
	0x45, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x30, 0x20, 0x17, 0x0d, 0x30,
	0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a,
	0x18, 0x0f, 0x32, 0x30, 0x35, 0x30, 0x31, 0x32, 0x33, 0x31, 0x30, 0x30,
	0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06,
	0x03, 0x55, 0x04, 0x03, 0x13, 0x07, 0x45, 0x78, 0x61, 0x6d, 0x70, 0x6c,
	0x65, 0x30, 0x5e, 0x30, 0x17, 0x06, 0x08, 0x2a, 0x85, 0x03, 0x07, 0x01,
	0x01, 0x01, 0x01, 0x30, 0x0b, 0x06, 0x09, 0x2a, 0x85, 0x03, 0x07, 0x01,
	0x02, 0x01, 0x01, 0x01, 0x03, 0x43, 0x00, 0x04, 0x40, 0x74, 0x27, 0x95,
	0xd4, 0xbe, 0xe8, 0x84, 0xdd, 0xf2, 0x85, 0x0f, 0xec, 0x03, 0xea, 0x3f,
	0xaf, 0x18, 0x44, 0xe0, 0x1d, 0x9d, 0xa6, 0x0b, 0x64, 0x50, 0x93, 0xa5,
	0x5e, 0x26, 0xdf, 0xc3, 0x99, 0x78, 0xf5, 0x96, 0xcf, 0x4d, 0x4d, 0x0c,
	0x6c, 0xf1, 0xd1, 0x89, 0x43, 0xd9, 0x44, 0x93, 0xd1, 0x6b, 0x9e, 0xc0,
	0xa1, 0x6d, 0x51, 0x2d, 0x2e, 0x12, 0x7c, 0xc4, 0x69, 0x1a, 0x63, 0x18,
	0xe2, 0xa3, 0x13, 0x30, 0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13,
	0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff,
};
static const ec_test_case ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_case = {
	.name="ECRDSA-STREEBOG256/GOST-256-curve-paramsetA (RFC4491)",
	.ec_str_p = &gost_R3410_2012_256_paramSetA_str_params,
	.priv_key = ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_vector_priv_key,
	.priv_key_len = sizeof(ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_vector_priv_key),
	.nn_random = ecrdsa_nn_random_rfc4491_bis_2_GOST_256bits_curve_test_vector,
	.hash_type = STREEBOG256,
	.msg = (const char*)ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_vector_message,
	.msglen = sizeof(ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_vector_message),
	.sig_type = ECRDSA,
	.exp_sig = ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_vector_expected_sig,
	.exp_siglen = sizeof(ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_vector_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_GOST_R3410_2012_256_PARAMSETA */
#endif /* WITH_HASH_STREEBOG256 */

#ifdef WITH_HASH_STREEBOG512
#ifdef WITH_CURVE_GOST512
#define ECRDSA_STREEBOG512_GOST512_SELF_TEST

static int ecrdsa_nn_random_rfc4491_bis_1_GOST_512bits_curve_test_vector(nn_t out, nn_src_t q){
	int ret, cmp;
	const u8 k_buf[] = {
		0x03, 0x59, 0xe7, 0xf4, 0xb1, 0x41, 0x0f, 0xea, 0xcc,
		0x57, 0x04, 0x56, 0xc6, 0x80, 0x14, 0x96, 0x94, 0x63,
		0x12, 0x12, 0x0b, 0x39, 0xd0, 0x19, 0xd4, 0x55, 0x98,
		0x6e, 0x36, 0x4f, 0x36, 0x58, 0x86, 0x74, 0x8e, 0xd7,
		0xa4, 0x4b, 0x3e, 0x79, 0x44, 0x34, 0x00, 0x60, 0x11,
		0x84, 0x22, 0x86, 0x21, 0x22, 0x73, 0xa6, 0xd1, 0x4c,
		0xf7, 0x0e, 0xa3, 0xaf, 0x71, 0xbb, 0x1a, 0xe6, 0x79,
		0xf1,
	};
        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_vector_priv_key[] = {
	0x0B, 0xA6, 0x04, 0x8A, 0xAD, 0xAE, 0x24, 0x1B, 0xA4,
	0x09, 0x36, 0xD4, 0x77, 0x56, 0xD7, 0xC9, 0x30, 0x91,
	0xA0, 0xE8, 0x51, 0x46, 0x69, 0x70, 0x0E, 0xE7, 0x50,
	0x8E, 0x50, 0x8B, 0x10, 0x20, 0x72, 0xE8, 0x12, 0x3B,
	0x22, 0x00, 0xA0, 0x56, 0x33, 0x22, 0xDA, 0xD2, 0x82,
	0x7E, 0x27, 0x14, 0xA2, 0x63, 0x6B, 0x7B, 0xFD, 0x18,
	0xAA, 0xDF, 0xC6, 0x29, 0x67, 0x82, 0x1F, 0xA1, 0x8D,
	0xD4,
};
static const u8 ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_vector_expected_sig[] = {
	0x2f, 0x86, 0xfa, 0x60, 0xa0, 0x81, 0x09, 0x1a, 0x23,
	0xdd, 0x79, 0x5e, 0x1e, 0x3c, 0x68, 0x9e, 0xe5, 0x12,
	0xa3, 0xc8, 0x2e, 0xe0, 0xdc, 0xc2, 0x64, 0x3c, 0x78,
	0xee, 0xa8, 0xfc, 0xac, 0xd3, 0x54, 0x92, 0x55, 0x84,
	0x86, 0xb2, 0x0f, 0x1c, 0x9e, 0xc1, 0x97, 0xc9, 0x06,
	0x99, 0x85, 0x02, 0x60, 0xc9, 0x3b, 0xcb, 0xcd, 0x9c,
	0x5c, 0x33, 0x17, 0xe1, 0x93, 0x44, 0xe1, 0x73, 0xae,
	0x36,
	0x41, 0x57, 0x03, 0xd8, 0x92, 0xf1, 0xa5, 0xf3, 0xf6,
	0x8c, 0x43, 0x53, 0x18, 0x9a, 0x7e, 0xe2, 0x07, 0xb8,
	0x0b, 0x56, 0x31, 0xef, 0x9d, 0x49, 0x52, 0x9a, 0x4d,
	0x6b, 0x54, 0x2c, 0x2c, 0xfa, 0x15, 0xaa, 0x2e, 0xac,
	0xf1, 0x1f, 0x47, 0x0f, 0xde, 0x7d, 0x95, 0x48, 0x56,
	0x90, 0x3c, 0x35, 0xfd, 0x8f, 0x95, 0x5e, 0xf3, 0x00,
	0xd9, 0x5c, 0x77, 0x53, 0x4a, 0x72, 0x4a, 0x0e, 0xee,
	0x70,
};
static const unsigned char ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_vector_message[] = {
	0x30, 0x82, 0x01, 0x16, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x0b,
	0x30, 0x0a, 0x06, 0x08, 0x2a, 0x85, 0x03, 0x07, 0x01, 0x01, 0x03, 0x03,
	0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
	0x07, 0x45, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x30, 0x20, 0x17, 0x0d,
	0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
	0x5a, 0x18, 0x0f, 0x32, 0x30, 0x35, 0x30, 0x31, 0x32, 0x33, 0x31, 0x30,
	0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e,
	0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x07, 0x45, 0x78, 0x61, 0x6d, 0x70,
	0x6c, 0x65, 0x30, 0x81, 0xa0, 0x30, 0x17, 0x06, 0x08, 0x2a, 0x85, 0x03,
	0x07, 0x01, 0x01, 0x01, 0x02, 0x30, 0x0b, 0x06, 0x09, 0x2a, 0x85, 0x03,
	0x07, 0x01, 0x02, 0x01, 0x02, 0x00, 0x03, 0x81, 0x84, 0x00, 0x04, 0x81,
	0x80, 0xe1, 0xef, 0x30, 0xd5, 0x2c, 0x61, 0x33, 0xdd, 0xd9, 0x9d, 0x1d,
	0x5c, 0x41, 0x45, 0x5c, 0xf7, 0xdf, 0x4d, 0x8b, 0x4c, 0x92, 0x5b, 0xbc,
	0x69, 0xaf, 0x14, 0x33, 0xd1, 0x56, 0x58, 0x51, 0x5a, 0xdd, 0x21, 0x46,
	0x85, 0x0c, 0x32, 0x5c, 0x5b, 0x81, 0xc1, 0x33, 0xbe, 0x65, 0x5a, 0xa8,
	0xc4, 0xd4, 0x40, 0xe7, 0xb9, 0x8a, 0x8d, 0x59, 0x48, 0x7b, 0x0c, 0x76,
	0x96, 0xbc, 0xc5, 0x5d, 0x11, 0xec, 0xbe, 0x77, 0x36, 0xa9, 0xec, 0x35,
	0x7f, 0xf2, 0xfd, 0x39, 0x93, 0x1f, 0x4e, 0x11, 0x4c, 0xb8, 0xcd, 0xa3,
	0x59, 0x27, 0x0a, 0xc7, 0xf0, 0xe7, 0xff, 0x43, 0xd9, 0x41, 0x94, 0x19,
	0xea, 0x61, 0xfd, 0x2a, 0xb7, 0x7f, 0x5d, 0x9f, 0x63, 0x52, 0x3d, 0x3b,
	0x50, 0xa0, 0x4f, 0x63, 0xe2, 0xa0, 0xcf, 0x51, 0xb7, 0xc1, 0x3a, 0xdc,
	0x21, 0x56, 0x0f, 0x0b, 0xd4, 0x0c, 0xc9, 0xc7, 0x37, 0xa3, 0x13, 0x30,
	0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
	0x05, 0x30, 0x03, 0x01, 0x01, 0xff
};
static const ec_test_case ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_case = {
	.name="ECRDSA-STREEBOG512/GOST-512-curve (RFC4491)",
	.ec_str_p = &GOST_512bits_curve_str_params,
	.priv_key = ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_vector_priv_key,
	.priv_key_len = sizeof(ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_vector_priv_key),
	.nn_random = ecrdsa_nn_random_rfc4491_bis_1_GOST_512bits_curve_test_vector,
	.hash_type = STREEBOG512,
	.msg = (const char*)ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_vector_message,
	.msglen = sizeof(ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_vector_message),
	.sig_type = ECRDSA,
	.exp_sig = ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_vector_expected_sig,
	.exp_siglen = sizeof(ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_vector_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_GOST512 */
#endif /* WITH_HASH_STREEBOG512 */


#else /* !defined(USE_ISO14888_3_ECRDSA) */
/*
 * Unlike other signature algorithms (defined in ISO 14888-3:2015) for
 * which tests vectors are based on secp* or brainpoolp* curves, EC-RDSA
 * tests vectors are based on custom curves. As a side note, the 256
 * curve (not the random signature param and key) appear in RFC 5832 and
 * RFC 7091, but discrepancies exist in the algorithm with the ISO/IEC 14888-3
 * description and test vectors (see above).
 */

#ifdef WITH_HASH_SHA256
#ifdef WITH_CURVE_GOST256
#define ECRDSA_SHA256_GOST256_SELF_TEST

/* First, ECRDSA test vector on a 256-bit GOST curve */

static int ecrdsa_nn_random_iso14888_3_GOST_256bits_curve_test_vector(nn_t out,
								      nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x77, 0x10, 0x5C, 0x9B, 0x20, 0xBC, 0xD3, 0x12,
		0x28, 0x23, 0xC8, 0xCF, 0x6F, 0xCC, 0x7B, 0x95,
		0x6D, 0xE3, 0x38, 0x14, 0xE9, 0x5B, 0x7F, 0xE6,
		0x4F, 0xED, 0x92, 0x45, 0x94, 0xDC, 0xEA, 0xB3
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecrdsa_GOST_256bits_curve_test_vectors_priv_key[] = {
	0x7A, 0x92, 0x9A, 0xDE, 0x78, 0x9B, 0xB9, 0xBE,
	0x10, 0xED, 0x35, 0x9D, 0xD3, 0x9A, 0x72, 0xC1,
	0x1B, 0x60, 0x96, 0x1F, 0x49, 0x39, 0x7E, 0xEE,
	0x1D, 0x19, 0xCE, 0x98, 0x91, 0xEC, 0x3B, 0x28
};

static const u8 ecrdsa_GOST_256bits_curve_test_vectors_expected_sig[] = {
	0x41, 0xAA, 0x28, 0xD2, 0xF1, 0xAB, 0x14, 0x82,
	0x80, 0xCD, 0x9E, 0xD5, 0x6F, 0xED, 0xA4, 0x19,
	0x74, 0x05, 0x35, 0x54, 0xA4, 0x27, 0x67, 0xB8,
	0x3A, 0xD0, 0x43, 0xFD, 0x39, 0xDC, 0x04, 0x93,
	0x0A, 0x7B, 0xA4, 0x72, 0x2D, 0xA5, 0x69, 0x3F,
	0x22, 0x9D, 0x17, 0x5F, 0xAB, 0x6A, 0xFB, 0x85,
	0x7E, 0xC2, 0x27, 0x3B, 0x9F, 0x88, 0xDA, 0x58,
	0x92, 0xCE, 0xD3, 0x11, 0x7F, 0xCF, 0x1E, 0x36
};

static const ec_test_case ecrdsa_GOST_256bits_curve_test_case = {
	.name = "ECRDSA-SHA256/GOST-256-curve",
	.ec_str_p = &GOST_256bits_curve_str_params,
	.priv_key = ecrdsa_GOST_256bits_curve_test_vectors_priv_key,
	.priv_key_len =
		sizeof(ecrdsa_GOST_256bits_curve_test_vectors_priv_key),
	.nn_random =
		ecrdsa_nn_random_iso14888_3_GOST_256bits_curve_test_vector,
	.hash_type = SHA256,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECRDSA,
	.exp_sig = ecrdsa_GOST_256bits_curve_test_vectors_expected_sig,
	.exp_siglen =
		sizeof(ecrdsa_GOST_256bits_curve_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_GOST256 */
#endif /* WITH_HASH_SHA256 */

/* NOTE: the test vectors present in the GOST 34.10-2012 standard and in the
 * associated RFC7091 do not provide messages as input, they instead provide the
 * value of the raw result of the hash function reduced modulo q. This is kinf of
 * useless here as we want to test the full GOST stack (i.e. hash digest plus curves),
 * hence we use the PyGOST Python library to generate some test vectors.
 */
#ifdef WITH_HASH_STREEBOG256
#ifdef WITH_CURVE_GOST256
#define ECRDSA_STREEBOG256_GOST256_SELF_TEST
static int ecrdsa_nn_random_pygostlib_1_GOST_256bits_curve_test_vector(nn_t out, nn_src_t q){
	int ret, cmp;
	const u8 k_buf[] = {
		0x4c, 0xe0, 0xe1, 0x2a, 0x2a, 0x35, 0x82, 0xa2, 0x1b, 0xe0,
		0xe7, 0x3f, 0xaf, 0xf2, 0xe2, 0xdb, 0x0c, 0xc2, 0x04, 0x80,
		0x33, 0x86, 0x36, 0xa2, 0x75, 0xcd, 0x12, 0xee, 0x0e, 0x3b,
		0x7a, 0xa7,
	};
        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 ecrdsa_pygostlib_1_GOST_256bits_curve_test_vector_priv_key[] = {
	0x34, 0xce, 0x5e, 0x59, 0xef, 0x00, 0x78, 0x53, 0x06,
	0xe8, 0x4a, 0xfd, 0x53, 0x47, 0xab, 0xe5, 0x68, 0x92,
	0x4e, 0xfe, 0x26, 0x32, 0x11, 0x1d, 0x05, 0x6a, 0xf9,
	0x1c, 0x4d, 0x7d, 0x59, 0x9a,
};
static const u8 ecrdsa_pygostlib_1_GOST_256bits_curve_test_vector_expected_sig[] = {
	0x43, 0x3f, 0x32, 0xf8, 0xcb, 0xdb, 0x53, 0x3d, 0x1e,
	0x8f, 0x5f, 0x5b, 0xfa, 0xc8, 0x67, 0x90, 0x5d, 0x01,
	0x22, 0xa7, 0x40, 0x5d, 0x06, 0xa2, 0x06, 0xac, 0x6a,
	0xfb, 0x42, 0xf9, 0x9c, 0x66, 0x4a, 0xa8, 0xea, 0x8f,
	0x5c, 0xbc, 0x88, 0xf3, 0x3b, 0x66, 0x9f, 0x0d, 0xa6,
	0x21, 0xa9, 0xe2, 0x1c, 0xba, 0xcc, 0xca, 0x91, 0x97,
	0xe6, 0xdf, 0xcb, 0x14, 0x9a, 0x99, 0x26, 0xf2, 0xd7,
	0xd2,
};
static const char ecrdsa_pygostlib_1_GOST_256bits_curve_test_vector_message[] = {
	0x61, 0x62, 0x63,
};
static const ec_test_case ecrdsa_pygostlib_1_GOST_256bits_curve_test_case = {
	.name="ECRDSA-STREEBOG256/GOST-256-curve 1",
	.ec_str_p = &GOST_256bits_curve_str_params,
	.priv_key = ecrdsa_pygostlib_1_GOST_256bits_curve_test_vector_priv_key,
	.priv_key_len = sizeof(ecrdsa_pygostlib_1_GOST_256bits_curve_test_vector_priv_key),
	.nn_random = ecrdsa_nn_random_pygostlib_1_GOST_256bits_curve_test_vector,
	.hash_type = STREEBOG256,
	.msg = ecrdsa_pygostlib_1_GOST_256bits_curve_test_vector_message,
	.msglen = sizeof(ecrdsa_pygostlib_1_GOST_256bits_curve_test_vector_message),
	.sig_type = ECRDSA,
	.exp_sig = ecrdsa_pygostlib_1_GOST_256bits_curve_test_vector_expected_sig,
	.exp_siglen = sizeof(ecrdsa_pygostlib_1_GOST_256bits_curve_test_vector_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

static int ecrdsa_nn_random_pygostlib_2_GOST_256bits_curve_test_vector(nn_t out, nn_src_t q){
	int ret, cmp;
	const u8 k_buf[] = {
		0x1b, 0x91, 0xc9, 0xc8, 0xf3, 0x3d, 0x16, 0x2f, 0xe0,
		0x97, 0xf2, 0x8e, 0x1d, 0x8a, 0x52, 0xab, 0x8f, 0x31,
		0x91, 0x55, 0x08, 0xf7, 0x1c, 0x80, 0x65, 0xac, 0x50,
		0x61, 0xff, 0x20, 0x07, 0x07,
	};
        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 ecrdsa_pygostlib_2_GOST_256bits_curve_test_vector_priv_key[] = {
	0x36, 0xf5, 0x26, 0x39, 0x79, 0x87, 0x88, 0x83, 0x06,
	0x56, 0x86, 0xd8, 0x7e, 0x04, 0xf0, 0x68, 0xbb, 0xf5,
	0x54, 0x18, 0xbf, 0xbf, 0x02, 0x1f, 0xa3, 0x0e, 0x9c,
	0xfb, 0xa7, 0x8d, 0x63, 0xef,
};
static const u8 ecrdsa_pygostlib_2_GOST_256bits_curve_test_vector_expected_sig[] = {
	0x24, 0x28, 0x81, 0x56, 0xe2, 0xaa, 0xb4, 0xd7, 0xbd,
	0x33, 0x2c, 0x50, 0x39, 0xe0, 0x84, 0x8a, 0x85, 0x2a,
	0xfd, 0xab, 0x36, 0x35, 0xea, 0x1e, 0xdd, 0xd0, 0x28,
	0xf3, 0x34, 0x5d, 0x68, 0xcf, 0x24, 0xaf, 0x4b, 0xe3,
	0x86, 0xc6, 0x95, 0x9c, 0xfb, 0xd5, 0x0e, 0x2e, 0x11,
	0x17, 0x65, 0x09, 0x18, 0xea, 0xe9, 0x3c, 0x60, 0x67,
	0x3f, 0xe2, 0xa2, 0xe9, 0xde, 0x33, 0xe7, 0xff, 0x79,
	0x0c,
};
static const char ecrdsa_pygostlib_2_GOST_256bits_curve_test_vector_message[] = {
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64,
};
static const ec_test_case ecrdsa_pygostlib_2_GOST_256bits_curve_test_case = {
	.name="ECRDSA-STREEBOG256/GOST-256-curve 2",
	.ec_str_p = &GOST_256bits_curve_str_params,
	.priv_key = ecrdsa_pygostlib_2_GOST_256bits_curve_test_vector_priv_key,
	.priv_key_len = sizeof(ecrdsa_pygostlib_2_GOST_256bits_curve_test_vector_priv_key),
	.nn_random = ecrdsa_nn_random_pygostlib_2_GOST_256bits_curve_test_vector,
	.hash_type = STREEBOG256,
	.msg = ecrdsa_pygostlib_2_GOST_256bits_curve_test_vector_message,
	.msglen = sizeof(ecrdsa_pygostlib_2_GOST_256bits_curve_test_vector_message),
	.sig_type = ECRDSA,
	.exp_sig = ecrdsa_pygostlib_2_GOST_256bits_curve_test_vector_expected_sig,
	.exp_siglen = sizeof(ecrdsa_pygostlib_2_GOST_256bits_curve_test_vector_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_GOST256 */
#endif /* WITH_HASH_STREEBOG256 */

#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_GOST512
#define ECRDSA_SHA512_GOST512_SELF_TEST

/* Then, ECRDSA test vector on a GOST 512-bit curve */

static int ecrdsa_nn_random_iso14888_3_GOST_512bits_curve_test_vector(nn_t out,
								nn_src_t q)
{
	int ret, cmp;
	/*
	 * Current version of ISO 14888-3:2015 has a bad k value in its
	 * test vectors. The value of k is higher than q!!!! Instead of
	 * reducing the wrong value each time, let's change that false
	 * value for a correct one (i.e. the reduced one mod q).
	 *

	 const u8 k_buf[] = {
	 0xC5, 0x73, 0xF6, 0xB3, 0x01, 0xD9, 0x9C, 0x24,
	 0xC4, 0x22, 0xA4, 0x27, 0x1E, 0x9E, 0xC9, 0x3B,
	 0xAE, 0xAA, 0x6E, 0xEF, 0x0D, 0xE8, 0x24, 0x77,
	 0xD8, 0xB7, 0x39, 0x1F, 0x9F, 0x67, 0x90, 0xD9,
	 0xDD, 0xE5, 0x14, 0x6F, 0x02, 0xEC, 0xA5, 0x67,
	 0x2C, 0x38, 0xFC, 0x80, 0x9C, 0xF4, 0xCA, 0x88,
	 0x93, 0x7C, 0x4B, 0x3A, 0x39, 0x36, 0xAD, 0xF9,
	 0x90, 0x8F, 0x79, 0x6C, 0x86, 0xC0, 0x5C, 0x43
	 };

	 */

	const u8 k_buf[] = {
		0x3b, 0x10, 0x9d, 0x0f, 0x05, 0xd9, 0x54, 0x96,
		0x1a, 0x08, 0x57, 0x30, 0x48, 0x3e, 0xec, 0x3a,
		0x8a, 0x54, 0x45, 0x89, 0x0e, 0x76, 0x06, 0x6e,
		0x2e, 0xe0, 0x41, 0x0c, 0x33, 0xc1, 0xee, 0x1e,
		0x8d, 0x86, 0xb9, 0x71, 0x6c, 0xb1, 0x2f, 0xd8,
		0xf9, 0x18, 0x43, 0xc2, 0xc3, 0x6c, 0x82, 0xa4,
		0xe2, 0x9f, 0xff, 0x5e, 0xbc, 0xef, 0x22, 0xcd,
		0xe4, 0x06, 0x23, 0x89, 0x76, 0xf2, 0x8e, 0x85
	};

	ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
	ret = nn_cmp(out, q, &cmp); EG(ret, err);

	ret = (cmp >= 0) ? -1 : 0;

err:
	return ret;
}

static const u8 ecrdsa_GOST_512bits_curve_test_vectors_priv_key[] = {
	0x0B, 0xA6, 0x04, 0x8A, 0xAD, 0xAE, 0x24, 0x1B,
	0xA4, 0x09, 0x36, 0xD4, 0x77, 0x56, 0xD7, 0xC9,
	0x30, 0x91, 0xA0, 0xE8, 0x51, 0x46, 0x69, 0x70,
	0x0E, 0xE7, 0x50, 0x8E, 0x50, 0x8B, 0x10, 0x20,
	0x72, 0xE8, 0x12, 0x3B, 0x22, 0x00, 0xA0, 0x56,
	0x33, 0x22, 0xDA, 0xD2, 0x82, 0x7E, 0x27, 0x14,
	0xA2, 0x63, 0x6B, 0x7B, 0xFD, 0x18, 0xAA, 0xDF,
	0xC6, 0x29, 0x67, 0x82, 0x1F, 0xA1, 0x8D, 0xD4
};

static const u8 ecrdsa_GOST_512bits_curve_test_vectors_expected_sig[] = {
	0x13, 0xc5, 0x65, 0x57, 0xe3, 0x00, 0x89, 0x8b,
	0xf6, 0xc9, 0x1a, 0x08, 0xaf, 0x0c, 0xaf, 0x80,
	0x10, 0x46, 0xa2, 0xdc, 0x58, 0xcf, 0x7e, 0x84,
	0xa1, 0x5d, 0xa3, 0xb6, 0x89, 0xc0, 0xeb, 0x29,
	0x73, 0xf5, 0xbe, 0x70, 0x27, 0xdb, 0xdd, 0x77,
	0xbc, 0xe5, 0xd3, 0x37, 0x6a, 0xd5, 0x79, 0x3c,
	0x21, 0x31, 0x57, 0x85, 0xaa, 0x6d, 0x25, 0x36,
	0xa2, 0x0c, 0x91, 0x58, 0x14, 0xf2, 0xad, 0xdc,
	0x32, 0xC0, 0xB1, 0x5B, 0xE3, 0x67, 0x58, 0x3B,
	0xB3, 0xFA, 0xEF, 0xF1, 0x49, 0xAF, 0x87, 0xD1,
	0x18, 0xBF, 0x18, 0xE1, 0x34, 0x87, 0xE0, 0xC6,
	0xAB, 0x75, 0x80, 0xB8, 0x62, 0xEC, 0x10, 0x4A,
	0x41, 0xEC, 0x9A, 0x5F, 0xB1, 0x7B, 0x0E, 0x0E,
	0xDB, 0xCF, 0xFD, 0x92, 0x0D, 0x6F, 0x62, 0x7E,
	0x70, 0x4A, 0x82, 0xCC, 0x53, 0x41, 0x27, 0xF6,
	0x44, 0xFD, 0xC9, 0x58, 0x98, 0x4D, 0xDC, 0xA0
};

static const ec_test_case ecrdsa_GOST_512bits_curve_test_case = {
	.name = "ECRDSA-SHA512/GOST-512-curve",
	.ec_str_p = &GOST_512bits_curve_str_params,
	.priv_key = ecrdsa_GOST_512bits_curve_test_vectors_priv_key,
	.priv_key_len =
		sizeof(ecrdsa_GOST_512bits_curve_test_vectors_priv_key),
	.nn_random =
		ecrdsa_nn_random_iso14888_3_GOST_512bits_curve_test_vector,
	.hash_type = SHA512,
	.msg = "abc",
	.msglen = 3,
	.sig_type = ECRDSA,
	.exp_sig = ecrdsa_GOST_512bits_curve_test_vectors_expected_sig,
	.exp_siglen =
		sizeof(ecrdsa_GOST_512bits_curve_test_vectors_expected_sig),
	.adata = NULL,
	.adata_len = 0
};
#endif /* WITH_CURVE_GOST512 */
#endif /* WITH_HASH_SHA512 */

#ifdef WITH_HASH_STREEBOG512
#ifdef WITH_CURVE_GOST512
#define ECRDSA_STREEBOG512_GOST512_SELF_TEST

static int ecrdsa_nn_random_pygostlib_1_GOST_512bits_curve_test_vector(nn_t out, nn_src_t q){
	int ret, cmp;
	const u8 k_buf[] = {
		0x03, 0xc3, 0xcb, 0xa3, 0x26, 0xc7, 0xdd, 0x44,	0x8e,
		0x98, 0xa1, 0x03, 0x37, 0x71, 0x4e, 0xf7, 0xa7, 0x9c,
		0xc1, 0x06, 0x30, 0x34, 0xb6, 0xcf, 0x63, 0x0c, 0x1d,
		0xe7, 0x7d, 0x80, 0xe0, 0x95, 0xc5, 0x7c, 0x5b, 0xa3,
		0x78, 0x01, 0x99, 0xde, 0x1f, 0x7c, 0xb0, 0x5b, 0x5b,
		0x08, 0x41, 0xd2, 0xcd, 0x88, 0x60, 0x8e, 0x75, 0xa8,
		0x8f, 0xe4, 0x9f, 0xee, 0xe8, 0xfd, 0xc2, 0x9f, 0x8b,
		0xa6,
	};
        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 ecrdsa_pygostlib_1_GOST_512bits_curve_test_vector_priv_key[] = {
	0x0c, 0x18, 0x44, 0xa6, 0x1c, 0xbb, 0x08, 0xb7, 0xa1,
	0x86, 0x32, 0x1e, 0xf5, 0x53, 0x97, 0x7d, 0xef, 0x40,
	0x87, 0x7e, 0xd8, 0x98, 0xfb, 0xf6, 0xfd, 0x3e, 0xab,
	0xe4, 0xcf, 0xcf, 0xd3, 0x04, 0x21, 0xe5, 0xbf, 0xcb,
	0x73, 0xce, 0xd7, 0x35, 0x98, 0xa5, 0x4f, 0x9d, 0x9e,
	0x4b, 0x55, 0x83, 0xd6, 0x04, 0x65, 0x3c, 0xb3, 0x0a,
	0xbb, 0xe0, 0x56, 0x96, 0x3f, 0x08, 0xaf, 0x9a, 0x5b, 0x4f,
};
static const u8 ecrdsa_pygostlib_1_GOST_512bits_curve_test_vector_expected_sig[] = {
	0x0e, 0xf7, 0xf8, 0xcb, 0x41, 0x86, 0xce, 0xb5, 0x86,
	0x43, 0x53, 0x15, 0xec, 0x67, 0x48, 0xfe, 0x20, 0x58,
	0x35, 0x57, 0x51, 0xd3, 0xfe, 0x25, 0xe2, 0x42, 0xad,
	0x96, 0x58, 0xae, 0x55, 0x11, 0xef, 0xb3, 0x3f, 0x93,
	0x2d, 0x71, 0x0c, 0xea, 0xc9, 0xd7, 0x4e, 0x34, 0x98,
	0xde, 0x85, 0xc9, 0xf1, 0x31, 0x52, 0xfe, 0xf4, 0x65,
	0xa6, 0x57, 0x05, 0x78, 0x4d, 0x6d, 0x45, 0x40, 0x09,
	0x4b, 0x1f, 0x2f, 0xc0, 0x2b, 0x79, 0xc5, 0x85, 0x1a,
	0xc1, 0xc1, 0x4b, 0xaa, 0x63, 0x58, 0xab, 0x07, 0x6e,
	0x0b, 0xc7, 0xfd, 0x28, 0x98, 0xe9, 0x88, 0x90, 0x6c,
	0xd7, 0x14, 0x02, 0x30, 0x14, 0x8c, 0xa7, 0xe7, 0x24,
	0x9a, 0x64, 0x83, 0x63, 0x9c, 0x4f, 0x18, 0xf5, 0x84,
	0x7e, 0xce, 0x03, 0xad, 0x3f, 0x9f, 0x26, 0x2f, 0xcb,
	0xaa, 0x3c, 0x55, 0xf7, 0xee, 0xd2, 0x91, 0xb3, 0xe7,
	0xeb, 0x4b,
};
static const char ecrdsa_pygostlib_1_GOST_512bits_curve_test_vector_message[] = {
	0x61, 0x62, 0x63,
};
static const ec_test_case ecrdsa_pygostlib_1_GOST_512bits_curve_test_case = {
	.name="ECRDSA-STREEBOG512/GOST-512-curve 1",
	.ec_str_p = &GOST_512bits_curve_str_params,
	.priv_key = ecrdsa_pygostlib_1_GOST_512bits_curve_test_vector_priv_key,
	.priv_key_len = sizeof(ecrdsa_pygostlib_1_GOST_512bits_curve_test_vector_priv_key),
	.nn_random = ecrdsa_nn_random_pygostlib_1_GOST_512bits_curve_test_vector,
	.hash_type = STREEBOG512,
	.msg = ecrdsa_pygostlib_1_GOST_512bits_curve_test_vector_message,
	.msglen = sizeof(ecrdsa_pygostlib_1_GOST_512bits_curve_test_vector_message),
	.sig_type = ECRDSA,
	.exp_sig = ecrdsa_pygostlib_1_GOST_512bits_curve_test_vector_expected_sig,
	.exp_siglen = sizeof(ecrdsa_pygostlib_1_GOST_512bits_curve_test_vector_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

static int ecrdsa_nn_random_pygostlib_2_GOST_512bits_curve_test_vector(nn_t out, nn_src_t q){
	int ret, cmp;
	const u8 k_buf[] = {
		0x15, 0x56, 0x79, 0x4e, 0xed, 0x00, 0x7c, 0xdc, 0xc0,
		0xc1, 0x3f, 0xb3, 0x6b, 0xa3, 0xa3, 0x00, 0xdd, 0x16,
		0xce, 0x6d, 0x83, 0xf6, 0x49, 0xab, 0x8d, 0x7a, 0x06,
		0x38, 0x34, 0xf0, 0x8b, 0xaf, 0xbc, 0x4e, 0x2a, 0x30,
		0x25, 0xee, 0xa2, 0x59, 0x5a, 0xc6, 0x79, 0x57, 0x84,
		0xc9, 0x15, 0x65, 0xed, 0xcb, 0x33, 0x42, 0x8a, 0x7a,
		0xd0, 0xe7, 0x87, 0xfe, 0x3c, 0x16, 0xd1, 0xb5, 0x50,
		0x8a,
	};
        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 ecrdsa_pygostlib_2_GOST_512bits_curve_test_vector_priv_key[] = {
	0x32, 0xb5, 0xda, 0xed, 0x49, 0x2e, 0x13, 0xc5, 0x8a, 0xb5, 0xa1, 0x41,
	0x9e, 0x01, 0x2f, 0x0c, 0x69, 0xc1, 0x4e, 0xf0, 0xcf, 0x84, 0xce, 0x81,
	0x4f, 0x76, 0x3b, 0x6f, 0xac, 0xd5, 0x7c, 0xd9, 0x42, 0xfe, 0x47, 0xe5,
	0x13, 0x38, 0xbd, 0xdd, 0xd5, 0x91, 0xe0, 0x38, 0xa6, 0x77, 0x17, 0x89,
	0x85, 0x9c, 0x55, 0x54, 0xa3, 0xd0, 0x94, 0x6d, 0x20, 0xb4, 0xd3, 0xfd,
	0x5c, 0xad, 0x17, 0xbc,
};
static const u8 ecrdsa_pygostlib_2_GOST_512bits_curve_test_vector_expected_sig[] = {
	0x06, 0xfa, 0x71, 0xf0, 0xf9, 0x5b, 0xab, 0x4f, 0x08, 0x5b, 0x81, 0x44,
	0x32, 0x3a, 0x79, 0xa3, 0x9f, 0xb6, 0x72, 0x7c, 0x5b, 0x14, 0x48, 0xcb,
	0x2b, 0x42, 0xc9, 0x75, 0xb8, 0x35, 0x63, 0xf0, 0x6f, 0x81, 0x02, 0x78,
	0xd3, 0x2d, 0x7b, 0x04, 0x91, 0x11, 0xb0, 0x69, 0xa0, 0xba, 0x62, 0x47,
	0x3c, 0x72, 0x29, 0xef, 0x1b, 0xc2, 0x5c, 0xe5, 0x94, 0x4a, 0xf4, 0xbb,
	0x64, 0xf8, 0x68, 0xb9, 0x2e, 0x29, 0xa4, 0x50, 0xd2, 0x38, 0xae, 0xc2,
	0x8f, 0x6c, 0x3f, 0xcf, 0x2b, 0x13, 0x5b, 0xf9, 0x73, 0x6a, 0xf8, 0xac,
	0x13, 0xff, 0xff, 0x60, 0x19, 0xbd, 0x89, 0xae, 0x4e, 0x29, 0xb8, 0x07,
	0xea, 0xa9, 0xe3, 0xf1, 0x78, 0xa2, 0x4f, 0x07, 0x3d, 0xd8, 0x6b, 0x9f,
	0x2e, 0xb7, 0x2d, 0x96, 0xd2, 0xea, 0xae, 0x68, 0x38, 0xdb, 0x0d, 0x42,
	0xa7, 0x73, 0xf5, 0x5b, 0xbd, 0x3c, 0xe8, 0x84,
};
static const char ecrdsa_pygostlib_2_GOST_512bits_curve_test_vector_message[] = {
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61,
	0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62,
	0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63,
	0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	0x61, 0x62, 0x63, 0x64,
};
static const ec_test_case ecrdsa_pygostlib_2_GOST_512bits_curve_test_case = {
	.name="ECRDSA-STREEBOG512/GOST-512-curve 2",
	.ec_str_p = &GOST_512bits_curve_str_params,
	.priv_key = ecrdsa_pygostlib_2_GOST_512bits_curve_test_vector_priv_key,
	.priv_key_len = sizeof(ecrdsa_pygostlib_2_GOST_512bits_curve_test_vector_priv_key),
	.nn_random = ecrdsa_nn_random_pygostlib_2_GOST_512bits_curve_test_vector,
	.hash_type = STREEBOG512,
	.msg = ecrdsa_pygostlib_2_GOST_512bits_curve_test_vector_message,
	.msglen = sizeof(ecrdsa_pygostlib_2_GOST_512bits_curve_test_vector_message),
	.sig_type = ECRDSA,
	.exp_sig = ecrdsa_pygostlib_2_GOST_512bits_curve_test_vector_expected_sig,
	.exp_siglen = sizeof(ecrdsa_pygostlib_2_GOST_512bits_curve_test_vector_expected_sig),
	.adata = NULL,
	.adata_len = 0
};

#endif /* WITH_CURVE_GOST512 */
#endif /* WITH_HASH_STREEBOG512 */

#endif /* defined(USE_ISO14888_3_ECRDSA) */

#endif /* WITH_SIG_ECRDSA */

/*******************************************************************
 ************** SM2 tests ******************************************
 *******************************************************************/

/*
 * Unlike other signature algorithms (defined in ISO 14888-3:2015) for
 * which tests vectors are based on secp* or brainpoolp* curves, SM2
 * tests vectors are based on custom curves.
 */

#ifdef WITH_SIG_SM2
#ifdef WITH_HASH_SM3

#ifdef WITH_CURVE_SM2P256TEST
#define SM2_SM3_CURVE_SM2P256TEST

/* First, SM2 test vector on a 256-bit SM2 TEST curve */

static int sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vector(nn_t out,
								      nn_src_t
								      q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x6C, 0xB2, 0x8D, 0x99, 0x38, 0x5C, 0x17, 0x5C,
		0x94, 0xF9, 0x4E, 0x93, 0x48, 0x17, 0x66, 0x3F,
		0xC1, 0x76, 0xD9, 0x25, 0xDD, 0x72, 0xB7, 0x27,
		0x26, 0x0D, 0xBA, 0xAE, 0x1F, 0xB2, 0xF9, 0x6F
	};
        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}

static const u8 sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_priv_key[] = {
	0x12, 0x8B, 0x2F, 0xA8, 0xBD, 0x43, 0x3C, 0x6C,
	0x06, 0x8C, 0x8D, 0x80, 0x3D, 0xFF, 0x79, 0x79,
	0x2A, 0x51, 0x9A, 0x55, 0x17, 0x1B, 0x1B, 0x65,
	0x0C, 0x23, 0x66, 0x1D, 0x15, 0x89, 0x72, 0x63
};

static const u8 sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_expected_sig[] = {
	0x40, 0xF1, 0xEC, 0x59, 0xF7, 0x93, 0xD9, 0xF4,
	0x9E, 0x09, 0xDC, 0xEF, 0x49, 0x13, 0x0D, 0x41,
	0x94, 0xF7, 0x9F, 0xB1, 0xEE, 0xD2, 0xCA, 0xA5,
	0x5B, 0xAC, 0xDB, 0x49, 0xC4, 0xE7, 0x55, 0xD1,
	0x6F, 0xC6, 0xDA, 0xC3, 0x2C, 0x5D, 0x5C, 0xF1,
	0x0C, 0x77, 0xDF, 0xB2, 0x0F, 0x7C, 0x2E, 0xB6,
	0x67, 0xA4, 0x57, 0x87, 0x2F, 0xB0, 0x9E, 0xC5,
	0x63, 0x27, 0xA6, 0x7E, 0xC7, 0xDE, 0xEB, 0xE7
};

static const char sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_msg[] = "message digest";

static const char sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_adata[] = "ALICE123@YAHOO.COM";

static const ec_test_case sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_case = {
	.name = "SM2-SM3/SM2P256TEST",
	.ec_str_p = &sm2p256test_str_params,
	.priv_key = sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_priv_key,
	.priv_key_len = sizeof(sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_priv_key),
	.nn_random = sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vector,
	.hash_type = SM3,
	.msg = sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_msg,
	.msglen = 14,
	.sig_type = SM2,
	.exp_sig = sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_expected_sig,
	.exp_siglen = sizeof(sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_expected_sig),
	.adata = (const u8*)sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_vectors_adata,
	.adata_len = 18,
};
#endif /* WITH_CURVE_SM2P256TEST */

#ifdef WITH_CURVE_SM2P256V1
#define SM2_SM3_CURVE_SM2P256V1

/* SM2 test vector on a 256-bit SM2 standardized curve */

static int sm2_nn_random_sm2p256v1_test_vector(nn_t out, nn_src_t q)
{
	int ret, cmp;
	const u8 k_buf[] = {
		0x59, 0x27, 0x6E, 0x27, 0xD5, 0x06, 0x86, 0x1A, 0x16,
		0x68, 0x0F, 0x3A, 0xD9, 0xC0, 0x2D, 0xCC, 0xEF, 0x3C,
		0xC1, 0xFA, 0x3C, 0xDB, 0xE4, 0xCE, 0x6D, 0x54, 0xB8,
		0x0D, 0xEA, 0xC1, 0xBC, 0x21,
	};
        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}

static const u8 sm2_nn_random_sm2p256v1_test_vectors_priv_key[] = {
	0x39, 0x45, 0x20, 0x8F, 0x7B, 0x21, 0x44, 0xB1, 0x3F, 0x36, 0xE3,
	0x8A, 0xC6, 0xD3, 0x9F, 0x95, 0x88, 0x93, 0x93, 0x69, 0x28, 0x60,
	0xB5, 0x1A, 0x42, 0xFB, 0x81, 0xEF, 0x4D, 0xF7, 0xC5, 0xB8,
};

static const u8 sm2_nn_random_sm2p256v1_test_vectors_expected_sig[] = {
	0xF5, 0xA0, 0x3B, 0x06, 0x48, 0xD2, 0xC4, 0x63, 0x0E, 0xEA, 0xC5,
	0x13, 0xE1, 0xBB, 0x81, 0xA1, 0x59, 0x44, 0xDA, 0x38, 0x27, 0xD5,
	0xB7, 0x41, 0x43, 0xAC, 0x7E, 0xAC, 0xEE, 0xE7, 0x20, 0xB3, 0xB1,
	0xB6, 0xAA, 0x29, 0xDF, 0x21, 0x2F, 0xD8, 0x76, 0x31, 0x82, 0xBC,
	0x0D, 0x42, 0x1C, 0xA1, 0xBB, 0x90, 0x38, 0xFD, 0x1F, 0x7F, 0x42,
	0xD4, 0x84, 0x0B, 0x69, 0xC4, 0x85, 0xBB, 0xC1, 0xAA,
};

static const u8 sm2_nn_random_sm2p256v1_test_vectors_adata[] = {
	0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x31, 0x32, 0x33,
	0x34, 0x35, 0x36, 0x37, 0x38,
};

static const u8 tmp_msg_sm2_sm2p256v1[] = {
	0x6D, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x20, 0x64, 0x69, 0x67, 0x65, 0x73, 0x74,
};
static const ec_test_case sm2_nn_random_sm2p256v1_test_case = {
	.name = "SM2-SM3/SM2P256V1",
	.ec_str_p = &sm2p256v1_str_params,
	.priv_key = sm2_nn_random_sm2p256v1_test_vectors_priv_key,
	.priv_key_len = sizeof(sm2_nn_random_sm2p256v1_test_vectors_priv_key),
	.nn_random = sm2_nn_random_sm2p256v1_test_vector,
	.hash_type = SM3,
	.msg = (const char *)tmp_msg_sm2_sm2p256v1,
	.msglen = sizeof(tmp_msg_sm2_sm2p256v1),
	.sig_type = SM2,
	.exp_sig = sm2_nn_random_sm2p256v1_test_vectors_expected_sig,
	.exp_siglen = sizeof(sm2_nn_random_sm2p256v1_test_vectors_expected_sig),
	.adata = sm2_nn_random_sm2p256v1_test_vectors_adata,
	.adata_len = sizeof(sm2_nn_random_sm2p256v1_test_vectors_adata),
};
#endif /* WITH_CURVE_SM2P256TEST */


#endif /* WITH_HASH_SM3 */
#endif /* WITH_SIG_SM2 */


/*******************************************************************
 ************** EdDSA tests ****************************************
 *******************************************************************/
#ifdef WITH_SIG_EDDSA25519
#ifdef WITH_HASH_SHA512
#ifdef WITH_CURVE_WEI25519
#define EDDSA25519_SHA512_WEI25519_SELF_TEST

#include "ed25519_test_vectors.h"

#define EDDSA25519PH_SHA512_WEI25519_SELF_TEST

#include "ed25519ph_test_vectors.h"

#define EDDSA25519CTX_SHA512_WEI25519_SELF_TEST

#include "ed25519ctx_test_vectors.h"

#endif /* WITH_CURVE_WEI25519 */
#endif /* WITH_HASH_SHA512 */
#endif /* WITH_SIG_EDDSA */

#ifdef WITH_SIG_EDDSA448
#ifdef WITH_HASH_SHAKE256
#ifdef WITH_CURVE_WEI448
#define EDDSA448_SHAKE256_WEI448_SELF_TEST

#include "ed448_test_vectors.h"

#define EDDSA448PH_SHAKE256_WEI448_SELF_TEST

#include "ed448ph_test_vectors.h"

#endif /* WITH_CURVE_WEI448 */
#endif /* WITH_HASH_SHAKE256 */
#endif /* WITH_SIG_EDDSA */

/*********** Deterministic ECDSA tests taken from RFC6979 **************/
#ifdef WITH_SIG_DECDSA

#include "decdsa_test_vectors.h"

#endif /* WITH_SIG_DECDSA */

#if defined(WITH_SIG_BIGN) && defined(WITH_CURVE_BIGN256V1)

/* NOTE: tests taken from STB 34.101.45-2013 */
#include "bign_test_vectors.h"

#endif

#if defined(WITH_SIG_DBIGN) && defined(WITH_CURVE_BIGN256V1)

/* NOTE: tests taken from STB 34.101.45-2013 */
#include "dbign_test_vectors.h"

#endif

#if defined(WITH_SIG_BIP0340) && defined(WITH_HASH_SHA256) && defined(WITH_CURVE_SECP256K1)

/* NOTE: tests taken from:
 * https://github.com/bitcoin/bips/blob/master/bip-0340/test-vectors.csv
 */
#include "bip0340_test_vectors.h"

#endif

/* ADD curve test vectors header here */
/* XXX: Do not remove the comment above, as it is
 * used by external tools as a placeholder to add or
 * remove automatically generated code.
 */

/* Dummy empty test case to avoid empty array
 * when no test case is defined
 */
static const ec_test_case dummy_test_case = {
	.name = "Dummy SIGN",
	.ec_str_p = NULL,
	.priv_key = NULL,
	.priv_key_len = 0,
	.nn_random = NULL,
	.hash_type = UNKNOWN_HASH_ALG,
	.msg = NULL,
	.msglen = 0,
	.sig_type = UNKNOWN_ALG,
	.exp_sig = NULL,
	.exp_siglen = 0,
	.adata = NULL,
	.adata_len = 0
};

/* List of all test cases */

static const ec_test_case *ec_fixed_vector_tests[] = {
	/* ECDSA */
#ifdef ECDSA_SHA224_SECP224R1_SELF_TEST
	&ecdsa_secp224r1_test_case,
#endif
#ifdef ECDSA_SHA256_SECP256R1_SELF_TEST
	&ecdsa_secp256r1_test_case,
#endif
#ifdef ECDSA_SHA512_SECP256R1_SELF_TEST
	&ecdsa_secp256r1_sha512_test_case,
#endif
#ifdef ECDSA_SHA384_SECP384R1_SELF_TEST
	&ecdsa_secp384r1_test_case,
#endif
#ifdef ECDSA_SHA512_SECP521R1_SELF_TEST
	&ecdsa_secp521r1_test_case,
#endif
#ifdef ECDSA_SHA256_BRAINPOOLP256R1_SELF_TEST
	&ecdsa_brainpoolp256r1_test_case,
#endif
#ifdef ECDSA_SHA384_BRAINPOOLP384R1_SELF_TEST
	&ecdsa_brainpoolp384r1_test_case,
#endif
#ifdef ECDSA_SHA512_BRAINPOOLP512R1_SELF_TEST
	&ecdsa_brainpoolp512r1_test_case,
#endif
#ifdef ECDSA_SHA256_FRP256V1_SELF_TEST
	&ecdsa_frp256v1_test_case,
#endif
#ifdef ECDSA_SHA3_224_SECP224R1_SELF_TEST
	&ecdsa_secp224r1_sha3_224_test_case,
#endif
#ifdef ECDSA_SHA3_256_SECP256R1_SELF_TEST
	&ecdsa_secp256r1_sha3_256_test_case,
#endif
#ifdef ECDSA_SHA3_512_SECP256R1_SELF_TEST
	&ecdsa_secp256r1_sha3_512_test_case,
#endif
#ifdef ECDSA_SHA3_384_SECP384R1_SELF_TEST
	&ecdsa_secp384r1_sha3_384_test_case,
#endif
#ifdef ECDSA_SHA3_512_SECP521R1_SELF_TEST
	&ecdsa_secp521r1_sha3_512_test_case,
#endif
#ifdef ECDSA_SHA224_SECP192R1_SELF_TEST
	&ecdsa_secp192r1_test_case,
#endif
	/* ECKCDSA */
#ifdef ECKCDSA_SHA224_SECP224R1_SELF_TEST
	&eckcdsa_secp224r1_test_case,
#endif
#ifdef ECKCDSA_SHA256_SECP224R1_SELF_TEST
	&eckcdsa_secp224r1_sha256_test_case,
#endif
#ifdef ECKCDSA_SHA256_SECP256R1_SELF_TEST
	&eckcdsa_secp256r1_test_case,
#endif
#ifdef ECKCDSA_SHA384_SECP384R1_SELF_TEST
	&eckcdsa_secp384r1_test_case,
#endif
#ifdef ECKCDSA_SHA512_SECP256R1_SELF_TEST
	&eckcdsa_secp256r1_sha512_test_case,
#endif
#ifdef ECKCDSA_SHA512_SECP521R1_SELF_TEST
	&eckcdsa_secp521r1_test_case,
#endif
#ifdef ECKCDSA_SHA256_BRAINPOOLP256R1_SELF_TEST
	&eckcdsa_brainpoolp256r1_test_case,
#endif
#ifdef ECKCDSA_SHA384_BRAINPOOLP384R1_SELF_TEST
	&eckcdsa_brainpoolp384r1_test_case,
#endif
#ifdef ECKCDSA_SHA512_BRAINPOOLP512R1_SELF_TEST
	&eckcdsa_brainpoolp512r1_test_case,
#endif
#ifdef ECKCDSA_SHA256_FRP256V1_SELF_TEST
	&eckcdsa_frp256v1_test_case,
#endif
	/* ECSDSA */
#ifdef ECSDSA_SHA224_SECP224R1_SELF_TEST
	&ecsdsa_secp224r1_test_case,
#endif
#ifdef ECSDSA_SHA256_SECP256R1_SELF_TEST
	&ecsdsa_secp256r1_test_case,
#endif
#ifdef ECSDSA_SHA384_SECP384R1_SELF_TEST
	&ecsdsa_secp384r1_test_case,
#endif
#ifdef ECSDSA_SHA512_SECP521R1_SELF_TEST
	&ecsdsa_secp521r1_test_case,
#endif
#ifdef ECSDSA_SHA256_BRAINPOOLP256R1_SELF_TEST
	&ecsdsa_brainpoolp256r1_test_case,
#endif
#ifdef ECSDSA_SHA384_BRAINPOOLP384R1_SELF_TEST
	&ecsdsa_brainpoolp384r1_test_case,
#endif
#ifdef ECSDSA_SHA512_BRAINPOOLP512R1_SELF_TEST
	&ecsdsa_brainpoolp512r1_test_case,
#endif
#ifdef ECSDSA_SHA256_FRP256V1_SELF_TEST
	&ecsdsa_frp256v1_test_case,
#endif
	/* ECOSDSA */
#ifdef ECOSDSA_SHA224_SECP224R1_SELF_TEST
	&ecosdsa_secp224r1_test_case,
#endif
#ifdef ECOSDSA_SHA256_SECP256R1_SELF_TEST
	&ecosdsa_secp256r1_test_case,
#endif
#ifdef ECOSDSA_SHA384_SECP384R1_SELF_TEST
	&ecosdsa_secp384r1_test_case,
#endif
#ifdef ECOSDSA_SHA512_SECP521R1_SELF_TEST
	&ecosdsa_secp521r1_test_case,
#endif
#ifdef ECOSDSA_SHA256_BRAINPOOLP256R1_SELF_TEST
	&ecosdsa_brainpoolp256r1_test_case,
#endif
#ifdef ECOSDSA_SHA384_BRAINPOOLP384R1_SELF_TEST
	&ecosdsa_brainpoolp384r1_test_case,
#endif
#ifdef ECOSDSA_SHA512_BRAINPOOLP512R1_SELF_TEST
	&ecosdsa_brainpoolp512r1_test_case,
#endif
#ifdef ECOSDSA_SHA256_FRP256V1_SELF_TEST
	&ecosdsa_frp256v1_test_case,
#endif
	/* ECFSDSA */
#ifdef ECFSDSA_SHA224_SECP224R1_SELF_TEST
	&ecfsdsa_secp224r1_test_case,
#endif
#ifdef ECFSDSA_SHA256_SECP256R1_SELF_TEST
	&ecfsdsa_secp256r1_test_case,
#endif
#ifdef ECFSDSA_SHA384_SECP384R1_SELF_TEST
	&ecfsdsa_secp384r1_test_case,
#endif
#ifdef ECFSDSA_SHA512_SECP521R1_SELF_TEST
	&ecfsdsa_secp521r1_test_case,
#endif
#ifdef ECFSDSA_SHA256_BRAINPOOLP256R1_SELF_TEST
	&ecfsdsa_brainpoolp256r1_test_case,
#endif
#ifdef ECFSDSA_SHA384_BRAINPOOLP384R1_SELF_TEST
	&ecfsdsa_brainpoolp384r1_test_case,
#endif
#ifdef ECFSDSA_SHA512_BRAINPOOLP512R1_SELF_TEST
	&ecfsdsa_brainpoolp512r1_test_case,
#endif
#ifdef ECFSDSA_SHA256_FRP256V1_SELF_TEST
	&ecfsdsa_frp256v1_test_case,
#endif
	/* ECGDSA */
#ifdef ECGDSA_SHA256_BRAINPOOLP192R1_SELF_TEST
	&ecgdsa_brainpoolp192r1_test_case,
#endif
#ifdef ECGDSA_SHA224_BRAINPOOLP224R1_SELF_TEST
	&ecgdsa_brainpoolp224r1_test_case,
#endif
#ifdef ECGDSA_SHA256_BRAINPOOLP256R1_SELF_TEST
	&ecgdsa_brainpoolp256r1_test_case,
#endif
#ifdef ECGDSA_SHA384_BRAINPOOLP384R1_SELF_TEST
	&ecgdsa_brainpoolp384r1_test_case,
#endif
	/* ECRDSA */
#ifndef USE_ISO14888_3_ECRDSA
#ifdef ECRDSA_STREEBOG256_GOST256_SELF_TEST
	&ecrdsa_rfc4491_bis_1_GOST_256bits_curve_test_case,
#endif
#ifdef ECRDSA_STREEBOG256_GOST256_PARAMSETA_SELF_TEST
	&ecrdsa_rfc4491_bis_2_GOST_256bits_curve_test_case,
#endif
#ifdef ECRDSA_STREEBOG512_GOST512_SELF_TEST
	&ecrdsa_rfc4491_bis_1_GOST_512bits_curve_test_case,
#endif
/**/
#else /* defined(USE_ISO14888_3_ECRDSA) */
#ifdef ECRDSA_SHA256_GOST256_SELF_TEST
	&ecrdsa_GOST_256bits_curve_test_case,
#endif
#ifdef ECRDSA_SHA512_GOST512_SELF_TEST
	&ecrdsa_GOST_512bits_curve_test_case,
#endif
#ifdef ECRDSA_STREEBOG256_GOST256_SELF_TEST
	&ecrdsa_pygostlib_1_GOST_256bits_curve_test_case,
	&ecrdsa_pygostlib_2_GOST_256bits_curve_test_case,
#endif
#ifdef ECRDSA_STREEBOG512_GOST512_SELF_TEST
	&ecrdsa_pygostlib_1_GOST_512bits_curve_test_case,
	&ecrdsa_pygostlib_2_GOST_512bits_curve_test_case,
#endif
#endif /* defined(USE_ISO14888_3_ECRDSA) */
#ifdef SM2_SM3_CURVE_SM2P256TEST
	&sm2_nn_random_iso14888_3_SM2_256bits_test_curve_test_case,
#endif
#ifdef SM2_SM3_CURVE_SM2P256V1
	&sm2_nn_random_sm2p256v1_test_case,
#endif
	/* EdDSA Ed25519 */
#ifdef EDDSA25519_SHA512_WEI25519_SELF_TEST
	EDDSA25519_SHA512_WEI25519_ALL_TESTS()
#endif
	/* EdDSA Ed25519ph */
#ifdef EDDSA25519PH_SHA512_WEI25519_SELF_TEST
	EDDSA25519PH_SHA512_WEI25519_ALL_TESTS()
#endif
	/* EdDSA Ed25519ctx */
#ifdef EDDSA25519CTX_SHA512_WEI25519_SELF_TEST
	EDDSA25519CTX_SHA512_WEI25519_ALL_TESTS()
#endif
	/* EdDSA Ed448 */
#ifdef EDDSA448_SHAKE256_WEI448_SELF_TEST
	EDDSA448_SHAKE256_WEI448_ALL_TESTS()
#endif
	/* EdDSA Ed448PH */
#ifdef EDDSA448PH_SHAKE256_WEI448_SELF_TEST
	EDDSA448PH_SHAKE256_WEI448_ALL_TESTS()
#endif
#ifdef DECDSA_SHA224_SECP192R1_SELF_TEST_0
	&decdsa_rfc6979_SECP192R1_SHA224_0_test_case,
#endif /* DECDSA_SHA224_SECP192R1_SELF_TEST_0 */
#ifdef DECDSA_SHA256_SECP192R1_SELF_TEST_0
	&decdsa_rfc6979_SECP192R1_SHA256_0_test_case,
#endif /* DECDSA_SHA256_SECP192R1_SELF_TEST_0 */
#ifdef DECDSA_SHA384_SECP192R1_SELF_TEST_0
	&decdsa_rfc6979_SECP192R1_SHA384_0_test_case,
#endif /* DECDSA_SHA384_SECP192R1_SELF_TEST_0 */
#ifdef DECDSA_SHA512_SECP192R1_SELF_TEST_0
	&decdsa_rfc6979_SECP192R1_SHA512_0_test_case,
#endif /* DECDSA_SHA512_SECP192R1_SELF_TEST_0 */
#ifdef DECDSA_SHA224_SECP192R1_SELF_TEST_1
	&decdsa_rfc6979_SECP192R1_SHA224_1_test_case,
#endif /* DECDSA_SHA224_SECP192R1_SELF_TEST_1 */
#ifdef DECDSA_SHA256_SECP192R1_SELF_TEST_1
	&decdsa_rfc6979_SECP192R1_SHA256_1_test_case,
#endif /* DECDSA_SHA256_SECP192R1_SELF_TEST_1 */
#ifdef DECDSA_SHA384_SECP192R1_SELF_TEST_1
	&decdsa_rfc6979_SECP192R1_SHA384_1_test_case,
#endif /* DECDSA_SHA384_SECP192R1_SELF_TEST_1 */
#ifdef DECDSA_SHA512_SECP192R1_SELF_TEST_1
	&decdsa_rfc6979_SECP192R1_SHA512_1_test_case,
#endif /* DECDSA_SHA512_SECP192R1_SELF_TEST_1 */
#ifdef DECDSA_SHA224_SECP256R1_SELF_TEST_0
	&decdsa_rfc6979_SECP256R1_SHA224_0_test_case,
#endif /* DECDSA_SHA224_SECP256R1_SELF_TEST_0 */
#ifdef DECDSA_SHA256_SECP256R1_SELF_TEST_0
	&decdsa_rfc6979_SECP256R1_SHA256_0_test_case,
#endif /* DECDSA_SHA256_SECP256R1_SELF_TEST_0 */
#ifdef DECDSA_SHA384_SECP256R1_SELF_TEST_0
	&decdsa_rfc6979_SECP256R1_SHA384_0_test_case,
#endif /* DECDSA_SHA384_SECP256R1_SELF_TEST_0 */
#ifdef DECDSA_SHA512_SECP256R1_SELF_TEST_0
	&decdsa_rfc6979_SECP256R1_SHA512_0_test_case,
#endif /* DECDSA_SHA512_SECP256R1_SELF_TEST_0 */
#ifdef DECDSA_SHA224_SECP256R1_SELF_TEST_1
	&decdsa_rfc6979_SECP256R1_SHA224_1_test_case,
#endif /* DECDSA_SHA224_SECP256R1_SELF_TEST_1 */
#ifdef DECDSA_SHA256_SECP256R1_SELF_TEST_1
	&decdsa_rfc6979_SECP256R1_SHA256_1_test_case,
#endif /* DECDSA_SHA256_SECP256R1_SELF_TEST_1 */
#ifdef DECDSA_SHA384_SECP256R1_SELF_TEST_1
	&decdsa_rfc6979_SECP256R1_SHA384_1_test_case,
#endif /* DECDSA_SHA384_SECP256R1_SELF_TEST_1 */
#ifdef DECDSA_SHA512_SECP256R1_SELF_TEST_1
	&decdsa_rfc6979_SECP256R1_SHA512_1_test_case,
#endif /* DECDSA_SHA512_SECP256R1_SELF_TEST_1 */
#ifdef DECDSA_SHA224_SECP384R1_SELF_TEST_0
	&decdsa_rfc6979_SECP384R1_SHA224_0_test_case,
#endif /* DECDSA_SHA224_SECP384R1_SELF_TEST_0 */
#ifdef DECDSA_SHA256_SECP384R1_SELF_TEST_0
	&decdsa_rfc6979_SECP384R1_SHA256_0_test_case,
#endif /* DECDSA_SHA256_SECP384R1_SELF_TEST_0 */
#ifdef DECDSA_SHA384_SECP384R1_SELF_TEST_0
	&decdsa_rfc6979_SECP384R1_SHA384_0_test_case,
#endif /* DECDSA_SHA384_SECP384R1_SELF_TEST_0 */
#ifdef DECDSA_SHA512_SECP384R1_SELF_TEST_0
	&decdsa_rfc6979_SECP384R1_SHA512_0_test_case,
#endif /* DECDSA_SHA512_SECP384R1_SELF_TEST_0 */
#ifdef DECDSA_SHA224_SECP384R1_SELF_TEST_1
	&decdsa_rfc6979_SECP384R1_SHA224_1_test_case,
#endif /* DECDSA_SHA224_SECP384R1_SELF_TEST_1 */
#ifdef DECDSA_SHA256_SECP384R1_SELF_TEST_1
	&decdsa_rfc6979_SECP384R1_SHA256_1_test_case,
#endif /* DECDSA_SHA256_SECP384R1_SELF_TEST_1 */
#ifdef DECDSA_SHA384_SECP384R1_SELF_TEST_1
	&decdsa_rfc6979_SECP384R1_SHA384_1_test_case,
#endif /* DECDSA_SHA384_SECP384R1_SELF_TEST_1 */
#ifdef DECDSA_SHA512_SECP384R1_SELF_TEST_1
	&decdsa_rfc6979_SECP384R1_SHA512_1_test_case,
#endif /* DECDSA_SHA512_SECP384R1_SELF_TEST_1 */
#ifdef DECDSA_SHA224_SECP521R1_SELF_TEST_0
	&decdsa_rfc6979_SECP521R1_SHA224_0_test_case,
#endif /* DECDSA_SHA224_SECP521R1_SELF_TEST_0 */
#ifdef DECDSA_SHA256_SECP521R1_SELF_TEST_0
	&decdsa_rfc6979_SECP521R1_SHA256_0_test_case,
#endif /* DECDSA_SHA256_SECP521R1_SELF_TEST_0 */
#ifdef DECDSA_SHA384_SECP521R1_SELF_TEST_0
	&decdsa_rfc6979_SECP521R1_SHA384_0_test_case,
#endif /* DECDSA_SHA384_SECP521R1_SELF_TEST_0 */
#ifdef DECDSA_SHA512_SECP521R1_SELF_TEST_0
	&decdsa_rfc6979_SECP521R1_SHA512_0_test_case,
#endif /* DECDSA_SHA512_SECP521R1_SELF_TEST_0 */
#ifdef DECDSA_SHA224_SECP521R1_SELF_TEST_1
	&decdsa_rfc6979_SECP521R1_SHA224_1_test_case,
#endif /* DECDSA_SHA224_SECP521R1_SELF_TEST_1 */
#ifdef DECDSA_SHA256_SECP521R1_SELF_TEST_1
	&decdsa_rfc6979_SECP521R1_SHA256_1_test_case,
#endif /* DECDSA_SHA256_SECP521R1_SELF_TEST_1 */
#ifdef DECDSA_SHA384_SECP521R1_SELF_TEST_1
	&decdsa_rfc6979_SECP521R1_SHA384_1_test_case,
#endif /* DECDSA_SHA384_SECP521R1_SELF_TEST_1 */
#ifdef DECDSA_SHA512_SECP521R1_SELF_TEST_1
	&decdsa_rfc6979_SECP521R1_SHA512_1_test_case,
#endif /* DECDSA_SHA512_SECP521R1_SELF_TEST_1 */
#if defined(WITH_SIG_BIGN) && defined(WITH_CURVE_BIGN256V1)
	BIGN_ALL_TESTS()
#endif
#if defined(WITH_SIG_DBIGN) && defined(WITH_CURVE_BIGN256V1)
	DBIGN_ALL_TESTS()
#endif
#if defined(WITH_SIG_BIP0340) && defined(WITH_HASH_SHA256) && defined(WITH_CURVE_SECP256K1)
	BIP0340_ALL_TESTS()
#endif

	/* Dummy empty test case to avoid empty array
	 * when no test case is defined */
	&dummy_test_case,

/* ADD curve test case here */
/* XXX: Do not remove the comment above, as it is
 * used by external tools as a placeholder to add or
 * remove automatically generated code.
 */
};

#define EC_FIXED_VECTOR_NUM_TESTS \
	(sizeof(ec_fixed_vector_tests) / sizeof(ec_fixed_vector_tests[0]))


/* Dummy empty test case to avoid empty array
 * when no test case is defined
 */
static const ecdh_test_case ecdh_dummy_test_case = {
	.name = "Dummy ECDH",
	.ecdh_type = UNKNOWN_ALG,
	.ec_str_p = NULL,
	.our_priv_key = NULL,
	.our_priv_key_len = 0,
	.peer_pub_key = NULL,
	.peer_pub_key_len = 0,
	.exp_our_pub_key = NULL,
	.exp_our_pub_key_len = 0,
	.exp_shared_secret = NULL,
	.exp_shared_secret_len = 0,
};

/*******************************************************************
 ************** ECCCDH tests ***************************************
 *******************************************************************/
#ifdef WITH_ECCCDH
/* NOTE: these tests are taken from the NIST CAVS 14.1 test suite
 * on curves P-192 P-224 P-256 P-384 P-521
 */
#include "ecccdh_test_vectors.h"
#endif /* WITH_ECCCDH */

#ifdef WITH_X25519
/* NOTE: tests taken from RFC7748 and https://tools.ietf.org/id/draft-ietf-ipsecme-safecurves-03.xml */
#include "x25519_test_vectors.h"
#endif /* WITH_X25519 */

#ifdef WITH_X448
/* NOTE: tests taken from RFC7748 */
#include "x448_test_vectors.h"
#endif /* WITH_X448 */

#if defined(WITH_ECCCDH) || defined(WITH_X25519) || defined(WITH_X448)
static const ecdh_test_case *ecdh_fixed_vector_tests[] = {
#ifdef ECCCDH_SECP192R1_SELF_TEST_0
	&ecccdh_SECP192R1_0_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_0 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_1
	&ecccdh_SECP192R1_1_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_1 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_2
	&ecccdh_SECP192R1_2_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_2 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_3
	&ecccdh_SECP192R1_3_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_3 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_4
	&ecccdh_SECP192R1_4_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_4 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_5
	&ecccdh_SECP192R1_5_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_5 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_6
	&ecccdh_SECP192R1_6_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_6 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_7
	&ecccdh_SECP192R1_7_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_7 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_8
	&ecccdh_SECP192R1_8_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_8 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_9
	&ecccdh_SECP192R1_9_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_9 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_10
	&ecccdh_SECP192R1_10_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_10 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_11
	&ecccdh_SECP192R1_11_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_11 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_12
	&ecccdh_SECP192R1_12_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_12 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_13
	&ecccdh_SECP192R1_13_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_13 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_14
	&ecccdh_SECP192R1_14_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_14 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_15
	&ecccdh_SECP192R1_15_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_15 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_16
	&ecccdh_SECP192R1_16_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_16 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_17
	&ecccdh_SECP192R1_17_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_17 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_18
	&ecccdh_SECP192R1_18_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_18 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_19
	&ecccdh_SECP192R1_19_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_19 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_20
	&ecccdh_SECP192R1_20_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_20 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_21
	&ecccdh_SECP192R1_21_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_21 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_22
	&ecccdh_SECP192R1_22_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_22 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_23
	&ecccdh_SECP192R1_23_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_23 */
#ifdef ECCCDH_SECP192R1_SELF_TEST_24
	&ecccdh_SECP192R1_24_test_case,
#endif /* ECCCDH_SECP192R1_SELF_TEST_24 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_0
	&ecccdh_SECP224R1_0_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_0 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_1
	&ecccdh_SECP224R1_1_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_1 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_2
	&ecccdh_SECP224R1_2_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_2 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_3
	&ecccdh_SECP224R1_3_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_3 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_4
	&ecccdh_SECP224R1_4_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_4 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_5
	&ecccdh_SECP224R1_5_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_5 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_6
	&ecccdh_SECP224R1_6_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_6 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_7
	&ecccdh_SECP224R1_7_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_7 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_8
	&ecccdh_SECP224R1_8_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_8 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_9
	&ecccdh_SECP224R1_9_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_9 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_10
	&ecccdh_SECP224R1_10_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_10 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_11
	&ecccdh_SECP224R1_11_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_11 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_12
	&ecccdh_SECP224R1_12_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_12 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_13
	&ecccdh_SECP224R1_13_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_13 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_14
	&ecccdh_SECP224R1_14_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_14 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_15
	&ecccdh_SECP224R1_15_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_15 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_16
	&ecccdh_SECP224R1_16_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_16 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_17
	&ecccdh_SECP224R1_17_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_17 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_18
	&ecccdh_SECP224R1_18_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_18 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_19
	&ecccdh_SECP224R1_19_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_19 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_20
	&ecccdh_SECP224R1_20_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_20 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_21
	&ecccdh_SECP224R1_21_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_21 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_22
	&ecccdh_SECP224R1_22_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_22 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_23
	&ecccdh_SECP224R1_23_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_23 */
#ifdef ECCCDH_SECP224R1_SELF_TEST_24
	&ecccdh_SECP224R1_24_test_case,
#endif /* ECCCDH_SECP224R1_SELF_TEST_24 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_0
	&ecccdh_SECP256R1_0_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_0 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_1
	&ecccdh_SECP256R1_1_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_1 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_2
	&ecccdh_SECP256R1_2_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_2 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_3
	&ecccdh_SECP256R1_3_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_3 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_4
	&ecccdh_SECP256R1_4_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_4 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_5
	&ecccdh_SECP256R1_5_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_5 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_6
	&ecccdh_SECP256R1_6_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_6 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_7
	&ecccdh_SECP256R1_7_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_7 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_8
	&ecccdh_SECP256R1_8_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_8 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_9
	&ecccdh_SECP256R1_9_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_9 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_10
	&ecccdh_SECP256R1_10_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_10 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_11
	&ecccdh_SECP256R1_11_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_11 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_12
	&ecccdh_SECP256R1_12_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_12 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_13
	&ecccdh_SECP256R1_13_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_13 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_14
	&ecccdh_SECP256R1_14_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_14 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_15
	&ecccdh_SECP256R1_15_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_15 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_16
	&ecccdh_SECP256R1_16_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_16 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_17
	&ecccdh_SECP256R1_17_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_17 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_18
	&ecccdh_SECP256R1_18_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_18 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_19
	&ecccdh_SECP256R1_19_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_19 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_20
	&ecccdh_SECP256R1_20_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_20 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_21
	&ecccdh_SECP256R1_21_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_21 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_22
	&ecccdh_SECP256R1_22_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_22 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_23
	&ecccdh_SECP256R1_23_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_23 */
#ifdef ECCCDH_SECP256R1_SELF_TEST_24
	&ecccdh_SECP256R1_24_test_case,
#endif /* ECCCDH_SECP256R1_SELF_TEST_24 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_0
	&ecccdh_SECP384R1_0_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_0 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_1
	&ecccdh_SECP384R1_1_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_1 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_2
	&ecccdh_SECP384R1_2_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_2 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_3
	&ecccdh_SECP384R1_3_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_3 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_4
	&ecccdh_SECP384R1_4_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_4 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_5
	&ecccdh_SECP384R1_5_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_5 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_6
	&ecccdh_SECP384R1_6_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_6 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_7
	&ecccdh_SECP384R1_7_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_7 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_8
	&ecccdh_SECP384R1_8_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_8 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_9
	&ecccdh_SECP384R1_9_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_9 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_10
	&ecccdh_SECP384R1_10_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_10 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_11
	&ecccdh_SECP384R1_11_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_11 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_12
	&ecccdh_SECP384R1_12_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_12 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_13
	&ecccdh_SECP384R1_13_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_13 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_14
	&ecccdh_SECP384R1_14_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_14 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_15
	&ecccdh_SECP384R1_15_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_15 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_16
	&ecccdh_SECP384R1_16_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_16 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_17
	&ecccdh_SECP384R1_17_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_17 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_18
	&ecccdh_SECP384R1_18_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_18 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_19
	&ecccdh_SECP384R1_19_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_19 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_20
	&ecccdh_SECP384R1_20_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_20 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_21
	&ecccdh_SECP384R1_21_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_21 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_22
	&ecccdh_SECP384R1_22_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_22 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_23
	&ecccdh_SECP384R1_23_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_23 */
#ifdef ECCCDH_SECP384R1_SELF_TEST_24
	&ecccdh_SECP384R1_24_test_case,
#endif /* ECCCDH_SECP384R1_SELF_TEST_24 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_0
	&ecccdh_SECP521R1_0_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_0 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_1
	&ecccdh_SECP521R1_1_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_1 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_2
	&ecccdh_SECP521R1_2_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_2 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_3
	&ecccdh_SECP521R1_3_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_3 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_4
	&ecccdh_SECP521R1_4_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_4 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_5
	&ecccdh_SECP521R1_5_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_5 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_6
	&ecccdh_SECP521R1_6_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_6 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_7
	&ecccdh_SECP521R1_7_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_7 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_8
	&ecccdh_SECP521R1_8_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_8 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_9
	&ecccdh_SECP521R1_9_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_9 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_10
	&ecccdh_SECP521R1_10_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_10 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_11
	&ecccdh_SECP521R1_11_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_11 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_12
	&ecccdh_SECP521R1_12_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_12 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_13
	&ecccdh_SECP521R1_13_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_13 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_14
	&ecccdh_SECP521R1_14_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_14 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_15
	&ecccdh_SECP521R1_15_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_15 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_16
	&ecccdh_SECP521R1_16_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_16 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_17
	&ecccdh_SECP521R1_17_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_17 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_18
	&ecccdh_SECP521R1_18_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_18 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_19
	&ecccdh_SECP521R1_19_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_19 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_20
	&ecccdh_SECP521R1_20_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_20 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_21
	&ecccdh_SECP521R1_21_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_21 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_22
	&ecccdh_SECP521R1_22_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_22 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_23
	&ecccdh_SECP521R1_23_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_23 */
#ifdef ECCCDH_SECP521R1_SELF_TEST_24
	&ecccdh_SECP521R1_24_test_case,
#endif /* ECCCDH_SECP521R1_SELF_TEST_24 */
#if defined(WITH_X25519) && defined(WITH_CURVE_WEI25519)
	&x25519_WEI25519_0_test_case,
	&x25519_WEI25519_1_test_case,
	&x25519_WEI25519_2_test_case,
	&x25519_WEI25519_3_test_case,
#endif
#if defined(WITH_X448) && defined(WITH_CURVE_WEI448)
	&x448_WEI448_0_test_case,
	&x448_WEI448_1_test_case,
#endif

	/* Dummy empty test case to avoid empty array
	 * when no test case is defined */
	&ecdh_dummy_test_case,
};
#endif /* !(defined(WITH_ECCCDH) || defined(WITH_X25519) || defined(WITH_X448)) */

#define ECDH_FIXED_VECTOR_NUM_TESTS \
        (sizeof(ecdh_fixed_vector_tests) / sizeof(ecdh_fixed_vector_tests[0]))

/*
 * A fixed test can fail in various ways. The way we report the failure
 * to the caller is by returning a non-zero value, in which we encode
 * some informations on the failure: curve, sig alg, hash alg, operation
 * (key import, signature, sig comparison, verification). Those 4 pieces
 * of information are each encoded on 8 bits in that order on the 28 LSB
 * of the return value. The function below produces a meaningful negative
 * return value in that specific format.
 */
typedef enum {
	TEST_KEY_IMPORT_ERROR = 1,
	TEST_SIG_ERROR = 2,
	TEST_SIG_COMP_ERROR = 3,
	TEST_VERIF_ERROR = 4,
	TEST_ECDH_ERROR = 5,
	TEST_ECDH_COMP_ERROR = 6,
} test_err_kind;

static int encode_error_value(const ec_test_case *c, test_err_kind failed_test, u32 *err_val)
{
	ec_curve_type ctype;
	ec_alg_type stype = c->sig_type;
	hash_alg_type htype = c->hash_type;
	test_err_kind etype = failed_test;
	int ret;

	MUST_HAVE((c != NULL) && (err_val != NULL), ret, err);

	ret = ec_get_curve_type_by_name(c->ec_str_p->name->buf,
					c->ec_str_p->name->buflen, &ctype); EG(ret, err);

	*err_val = (((u32)ctype << 24) |
		    ((u32)stype << 16) |
		    ((u32)htype <<  8) |
		    ((u32)etype));
	ret = 0;

err:
	return ret;
}

static inline int ecdh_encode_error_value(const ecdh_test_case *c, test_err_kind failed_test, u32 *err_val)
{
	ec_curve_type ctype;
	ec_alg_type stype = c->ecdh_type;
	test_err_kind etype = failed_test;
	int ret;

	MUST_HAVE((c != NULL) && (err_val != NULL), ret, err);

	ret = ec_get_curve_type_by_name(c->ec_str_p->name->buf,
					c->ec_str_p->name->buflen, &ctype); EG(ret, err);

	*err_val = (((u32)ctype << 24) |
		    ((u32)stype << 16) |
		    ((u32)0 <<  8) |
		    ((u32)etype));
	ret = 0;

err:
	return ret;
}

int perform_known_test_vectors_test(const char *sig, const char *hash, const char *curve);
int perform_random_sig_verif_test(const char *sig, const char *hash, const char *curve);
int perform_performance_test(const char *sig, const char *hash, const char *curve);

#endif /* __EC_SELF_TESTS_CORE_H__ */
