/*  AutoDesk Animator .COL file converter.
    Written by David Leadbetter
        email: davidl@cs.uq.oz.au

    Converts between AA Pro .COL format, and AA format (which is compatible
    with the WGT .PAL format), and vice versa.

    NOTE: It is not necessary to specify the conversion direction, as
          this is detected automatically by the code.


    AA & AAPro .COL file formats (in brief):
    =======================================

    Each of the AA formats holds a 256 colour palette as the RGB components.

    RGB Colour component resolution:
        AA .COL files:    0..63  (64 values)  (262144 possible colours)
        AAPro .COL files: 0..255 (256 values) (16777216 possible colours)

    Each colour component is stored as one byte in the .COL file:
        AA .COL files:    768 bytes (only the lower 6 bits are used)
        AAPro .COL files: 776 bytes (includes 8 byte header).

    This code was originally written and compiled using Borland C++ v2.0
    (Donations of Borland C++ v3.1 welcomed ;-)
*/

#include <stdio.h>
#include <stdlib.h>

typedef unsigned char aapro_col[776]; /* The array to hold the AAPro .COL data */
typedef unsigned char aa_col[768];    /* The array to hold the AA .COL data */
typedef enum col_type { STD, PRO } col_type;

/* The AAPro .COL file header data */
unsigned char aapro_header[] = {8, 3, 0, 0, 35, 177, 0, 0};

int check_aa_col_values(aa)
	aa_col *aa;
/*
    Checks that all RGB colour component values in the AA .COL file
    are valid (ie in the range 0..63 decimal)
    Used to verify a valid AA .COL file.
    NOTE: since the AAPro .COL header contains the character dec 177 we
    will never confuse file types :-)
*/
{
	int count;

	for (count = 0; count < 768; count++)
		if ((*aa)[count] >= 64)
			return 1;
	return 0;
}

int check_aapro_col_header(aapro)
	aapro_col *aapro;
/*
    Checks the AAPro header.
    Used to verify a valid AAPro .COL file.
    NOTE: since the AA .COL files only contain characters 0..63 decimal,
    and the AAPro .COL header contains dec 177, we will never confuse
    file types :-)
*/
{
	int count;

	for (count = 0; count < 8; count++)
		if ((*aapro)[count] != aapro_header[count])
			return 1;
	return 0;
}

aa_col *read_aa_col(infile)
	FILE *infile;
/* Reads in an AA .COL file. */
{
	aa_col *result;
	int num;

	result = (aa_col *) malloc(sizeof(aa_col));
	num = fread(*result, 1, 768, infile);
	if ((num != 768) || (check_aa_col_values(result) != 0)) {
		free(*result);
		return NULL;
	}
	return result;
}

aapro_col *read_aapro_col(infile)
	FILE *infile;
/* Reads in an AAPro .COL file */
{
	aapro_col *result;
	int num;

	result = (aapro_col *) malloc(sizeof(aapro_col));
	num = fread(*result, 1, 776, infile);
	if ((num != 776) || (check_aapro_col_header(*result) != 0)) {
		free(*result);
		return NULL;
	}
	return result;
}

int write_aa_col(outfile, aa)
	FILE *outfile;
	aa_col *aa;
/* Writes an AA .COL to file. */
{
	int num;

	num = fwrite(*aa, 1, 768, outfile);
	if (num != 768) {
		return 1;
	}
	return 0;
}

int write_aapro_col(outfile, aapro)
	FILE *outfile;
	aa_col *aapro;
/* Writes an AAPro .COL to file. */
{
	int num;

	num = fwrite(*aapro, 1, 776, outfile);
	if (num != 776) {
		return 1;
	}
	return 0;
}

aapro_col *aa2aapro(aa)
	aa_col *aa;
/*
    Coverts an AA .COL file to an AAPro .COL file.
    First we prepend the AAPro .COL header;
    Then we multiply each RGB colour component value by 4
    (increasing the resolution from 64 to 256).
*/

{
	aapro_col *result;
	int count;

	result = (aapro_col *) malloc(sizeof(aapro_col));
	memcpy(*result, aapro_header, 8);
	for (count = 0; count < 768; count++) {
		(*result)[count+8] = (*aa)[count] * 4;
	}
	return result;
}

aa_col *aapro2aa(aapro)
	aapro_col *aapro;
/*
    Convert an AAPro .COL file to an AA .COL file.
    We overlook the AAPro .COL header, and divide each RGB colour
    component by 4 (decreasing the resolution from 256 to 64).
*/
{
	aa_col *result;
	int count;

	result = (aa_col *) malloc(sizeof(aa_col));
	for (count = 8; count < 776; count++) {
		(*result)[count-8] = (int)((*aapro)[count] / 4);
	}
	return result;
}

int main(int argc, char **argv)
{
	FILE *file;
	aapro_col *aap;
	aa_col *aa;
	col_type source;
	int result;

	if (argc < 2 || argc > 3) {
		fprintf(stderr, "Usage: col_cvt source [dest]\n");
		return(1);
	}


	if ((file = fopen(argv[1], "rb")) == NULL) {
		fprintf(stderr, "Error opening file %s for input.\n",argv[1]);
		exit(-1);
	}

	aa = read_aa_col(file);
	if (aa != NULL)
		source = STD;
	else {
		rewind(file);
		aap = read_aapro_col(file);
		if (aap != NULL)
			source = PRO;
		else {
			fprintf(stderr, "Input file is not a valid .COL file.\n");
			exit(-1);
		}
	}
	fclose(file);


	if ((file = fopen(argv[argc-1], "wb")) == NULL) {
		fprintf(stderr, "Error opening file %s for output.\n",argv[argc-1]);
		exit(-1);
	}
	switch (source) {
		case STD:
			aap = aa2aapro(aa);
			result = write_aapro_col(file, aap);
			printf("Conversion complete (AA --> AAPRO)\n");
			break;
		case PRO:
			aa = aapro2aa(aap);
			result = write_aa_col(file, aa);
			printf("Conversion complete (AAPRO --> AA)\n");
			break;
		default:
			fprintf(stderr,"Internal error - invalid value assigned to source file type!\n");
			exit(-1);
	}
	fclose(file);


	return result;
}
