#include <adv3.h>
#include <en_us.h>

#include <storyguide.h>


/* The game begins with the prologue active. */
modify StoryModerator
    initStoryPoints = [prologue]
;


/* The prologue, up until the player closes her eyes. */
prologue: StoryPoint
    /* 
     *   The prologue is ready to start as soon as the game is, and once it's 
     *   finished it can never become active again.
     */
    checkActive() 
    {
        return !completed_;
    }
    
    /* The prologue is over when the player closes her eyes. */
    checkCompleted() 
    {
        return !eyes.isOpen;
    }
    
    /* This is always followed by the "counting" scene. */
    nextStoryPointPool = [counting]
    
    /* 
     *   Only a few actions are allowed during the prologue - saying yes or 
     *   no, closing your eyes, counting, and so on. System actions are okay,
     *   of course.
     *
     *   TODO: Allow GOODBYE and travel as well, for premature endings?
     */
    allowAction(action)
    {
        /* Don't allow the PC to talk to Yvonne or Tiana at this stage. */
        if (gDobj is in (yvonne, tiana))
        {
            return nil;
        }
        
        /* 
         *   If the action is a system action, saying yes or no, counting, or
         *   talking to Emma about Sardines or Hide and Seek, allow it.
         */
        if (action.actionOfKind(SystemAction) || 
            action.actionOfKind(YesAction) || 
            action.actionOfKind(NoAction) || 
            action.actionOfKind(CountAction) || 
            action.actionOfKind(CountToAction) ||
            (gTopic && gTopic.getBestMatch is in (tGame, tHnS)))
        {
            return true;
        }
        
        /* 
         *   If the action is closing your eyes (but not closing anything 
         *   else) allow it.
         */
        if (action.actionOfKind(CloseAction) && action.getDobj() == eyes)
        {
            return true;
        }
        
        /* Everything is else is out. */
        return nil;
    }
    
    /* 
     *   At the end of the prologue, show the title screen and move the 
     *   player, Tiana and Yvonne into darkness.
     */
    doPostCmdCode()
    {
        /* Tell the player to count. */
        "Now count --- to 50, remember. I'm going to hide. Don't 
        peek!<.p>";
        
        /* 
         *   Move all the characters except Emma into Darkness. In case the 
         *   player closes her eyes in the middle of a conversation, set 
         *   Emma's conversation node to nil.
         */
        [me, yvonne, tiana].forEach({x: x.moveIntoForTravel(dark)});
        emma.setConvNode(nil);
    }
;


/* The counting scene */
counting: StoryPoint
    /* It begins when the player is in darkness... */
    checkActive() {
        return gPlayerChar.isIn(dark);
    }
    
    /* And it finishes when the countdown gets to 50. */
    checkCompleted()
    {
        return (countedTo == 50);
    }
    
    /* The next scene after this is "searching". */
    nextStoryPointPool = [searching]
    
    /* 
     *   During counting, Yvonne and Tiana will count even if you don't join 
     *   in. This keeps track of where they are up to.
     */
    countedTo = 0
    
    /* Increment the counter by 7. */
    incrementCounter() {
        countedTo += 7;
    }
    
    /*
     *   As long as this scene lasts, the others will be counting to 50, even
     *   if you don't join in.
     */
    doContCmdCode(preCmd)
    {
        /* 
         *   Begin or continue counting - but only once per turn, so check 
         *   preCmd first. Also, if we've already finished counting, don't 
         *   bother.
         */
        if (preCmd && countedTo != 50)
        {
            doCount();
        }
    }
    
    
    /* When the scene ends, end the count. */
    doPostCmdCode()
    {
        /* end the count, unless the player already has by counting herself */
        if (!gPlayerChar.hasCounted)
        {
            endCount();
        }
    }
    
    /* do the count */
    doCount()
    {
        /* 
         *   if countedTo is still 0, tell the player that the others are
         *   beginning to count
         */
        if (countedTo == 0)
        {
            "<.p><<yvonne.theName>> and <<tiana.theName>> begin to count: 
            ";
        }
        
        /* print the count - three numbers */
        "<.p><q>";
        continueCount();
        "</q><.p>";
        
        /* bump up countedTo to the next ten */
        incrementCounter();
    }
    
    /* count out three numbers */
    continueCount()
    {
        /* capitalise the first number */
        "\^";
        
        /* loop through three times */
        for (local i = 0; i < 3; i++)
        {
            "<<spellInt(++countedTo)>>...";
            if (i < 2)
                " ";
        }
    }
    
    /* count out the final three numbers and open the player's eyes */
    endCount()
    {
        /* say the final numbers */
        "<q>Forty-eight... forty-nine... fifty!</q><.p>";
        countedTo = 50;
        
        /* 
         *   if Emma isn't already in her hiding spot, put her there and 
         *   cancel her agenda items for this scene
         */
        if (!emma.isIn(bushNearCubby))
        {
            /* stick Emma in her hiding spot */
            emma.enterHidingSpot(bushNearCubby);
            
            /* cancel agenda items */
            forEachInstance(AgendaItem, new function(x)
            {
                if (x.getActor() == emma && x.storyPoint == counting)
                {
                    x.isDone = true;
                }
            });
        }
        
        /* 
         *   If the player's eyes are not already open, open them (which 
         *   moves her and the others into the backyard). Otherwise, 
         *   announce that Tiana and Yvonne open their eyes.
         */
        if (eyes.isOpen())
        {
            "<<yvonne.theName>> and <<tiana.theName>> look around. ";
            [tiana, yvonne].forEach(
                {x: x.lastAction = LookAction.createActionInstance});
        }
        else
        {
            counting.openEyes();
        }
    }
    
    /* 
     *   Several actions will make the PC open her eyes, and they all have a 
     *   common subset of results, so collect those into a method.
     */
    openEyes()
    {
        /* make the status bar active */
        gameMain.showStatus = true;
        
        /* make the eyes open and announce this */
        eyes.makeOpen(true);
        "<.p>You open your eyes<<gPlayerChar.isCheating ? ' and glance around
            cautiously' : ''>>.<.p>";
        
        /* 
         *   Show the title screen, unless the PC has already opened her eyes
         *   and then closed them again.
         */
        if (!eyes.haveBeenOpen)
        {
            gameMain.showTitles();
            eyes.haveBeenOpen = true;
        }
        
        /* set Yvonne and Tiana's ActorStates to searching */
        tiana.setCurState(tianaSearching);
        yvonne.setCurState(yvonneSearching);
        
        /* move everyone (except Emma) back into the backyard */
        [tiana, yvonne, me].forEach({x: x.moveIntoForTravel(lawnWest)});
        gPlayerChar.lookAround(true);
    }
;


/* 
 *   The searching scene, which will always take place after counting has 
 *   finished.
 */
searching: StoryPoint
    /* There are no special requirements for this scene to begin. */
    checkActive()
    {
        return true;
    }
    
    /* 
     *   The only scene that can follow this one is the playing-hide-and-seek
     *   scene.
     */
    nextStoryPointPool = [playingHnS]
    
    /* And this scene ends when playingHnS is ready to begin. */
    checkCompleted()
    {
        return (playingHnS.checkActive());
    }
            
    doPreCmdCode()
    {
        /* 
         *   If Yvonne is cheating (or nobody is cheating), she heads off 
         *   towards Emma's hiding place immediately.
         *
         *   By default, this uses up a turn for her, causing her to do 
         *   nothing on the next turn. That's not what I want, so decrement 
         *   her nextRunTime counter to make the library think she hasn't 
         *   done anything yet.
         */
        if (yvonne.isCheating || !gPlayerChar.isCheating)
        {
            yvonne.executeAgenda();
            yvonne.nextRunTime--;
        }
        
        /* 
         *   If the PC is *not* cheating, then Tiana heads off in her own 
         *   direction immediately as well. I realise this makes no sense in 
         *   terms of character motivation, since Tiana has no way of 
         *   knowing whether the player is cheating or not.
         */
        if (!gPlayerChar.isCheating)
        {
            tiana.executeAgenda();
            tiana.nextRunTime--;
        }
    }
    
    /* what happens when you lose the game */
    loseGame()
    {
        "Emma wriggles out of the bush, pushing past you. <q>You 
        were ages, what took you so long?</q> she demands. 
        <q>Doesn't matter. You're last, you lose, losers smell like 
        dirty shoes.\ I'm bored of this game now, let's go inside. 
        I'm thirsty.</q><.p>
        The others scamper off to the house, ignoring you.";
        endGame('You have lost');
    }
;



/* 
 *   The playing-hide-and-seek scene, which takes place after the PC agrees 
 *   with one of the NPCs to play Hide and Seek instead of Sardines. This is 
 *   a scene in its own right rather than a sudden ending to give the PC a 
 *   chance to get all the NPCs involved in the game.
 */
playingHnS: StoryPoint
    /* becomes active when the PC is playing Hide and Seek */
    checkActive()
    {
        return (gPlayerChar.isPlayingHnS);
    }
    
    
    /* inviting a third person to play with you */
    invitationToPlay()
    {
        "You say, <q>Me and<<tiana.isProperName ? ' Tiana ' : '... um, 
            her,</q> pointing towards {the tiana/her}, <q>'>>are going to play 
        Hide and Seek. ";
        
        if (gPlayerChar.roleInHnS == roleCounting)
            "I'm going to count, and she's going to hide. ";
        else
            "She's going to count, and I'm going to hide. ";
        
        "Do you want to join in?</q><.p>";
    }
    
    
    /* ending for playing Hide and Seek with Tiana only */
    playingWithTiana()
    {
        "<q>What's taking you two so long?</q> Emma demands. <q>We've
        been waiting ages, and you're <i>still</i> looking?</q><.p>
        <q>We're not playing that game anymore,</q> you announce,
        <q>we're playing Hide and Seek instead.</q><.p>
        Emma scowls. <q>You never said you were playing something
        else! You didn't even ask if we wanted to play, or 
        anything! That's not fair, we didn't even get a 
        say!</q> Before you can say anything else, she adds,
        <q>But we don't want to play with you anyway. Come on,
        Yvonne, let's go do something fun on our own.</q><.p>
        She and Yvonne spend the rest of the afternoon sulking in
        the cubby house, while you and Tiana <<tiana.learnNameInPassing>>
        play Hide and Seek. By the time you have to go home, Tiana has
        an invitation to come over to your place next weekend. ";
        
        endGame('You have made a friend');
    }
    
    
    /* ending for playing Hide and Seek with Yvonne and Tiana */
    playingWithYvonne()
    {
        if (emma.knowsEveryoneIsPlayingHnS)
        {
            "Tiana <<tiana.learnNameInPassing>> and Yvonne play 
            several games of Hide and Seek with you, but 
            Emma doesn't reappear. It's not until you're driving away that you 
            see her again, staring out of her bedroom
            window at your car. It's hard to tell, but you think her 
            eyes look a little red.<.p>
            Next weekend you and Tiana are going to visit Yvonne 
            instead. She's promised to show you her pet budgies. ";
            
            endGame('Emma has lost');
        }
        else
        {
            "<q>What's taking you all so long?</q> she demands. <q>I
            was waiting ages and ages, and you didn't find me.</q><.p>
            <q>We decided to play Hide and Seek instead,</q> Yvonne
            announces. <q>We didn't want to play that other 
            game.</q><.p>
            Emma stares at her. <q>But what about me?</q><.p>
            Tiana <<tiana.learnNameInPassing>> shuffles uncomfortably. 
            <q>We thought you wouldn't want to play,</q> she says, 
            <q>you just wanted to play your game.</q><.p>
            <q>But you didn't even ask! You just left me...</q>
            Emma's voice trails off. <q>I do want to play,</q>
            she adds. She looks at Yvonne, but Yvonne says
            nothing.<.p>
            There's a long silence. Finally Emma says, <q>I really
            do want to play. Please can I?</q> ";
            
            emma.moveIntoForTravel(gPlayerChar.getOutermostRoom);
            emma.makePosture(standing);
            emma.isHidden = nil;
            emma.knowsEveryoneIsPlayingHnS = true;
            
            /* 
             *   This doesn't end the game immediately - first there's a 
             *   final ConvNode to negotiate. Does Emma get to play or not?
             */
            emma.initiateConversation(nil, emmaCanPlayHnS);
        }
    }
;
