/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*--------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: May 1 1987
----------------------------------------------------------------------------*/

#ifdef BFLY					  /* butterfly */
# ifdef BSIM
#  include      "bflysim.h"			  /* butterfly sim */
# else
#  include      "bflycontrol.h"			  /* butterfly control */
# endif
#else
# include       "uniproc.h"			  /* uniprocessor */
#endif

#ifdef BFLY
  extern struct nametable *table_list; 
  extern int no_tables;
#endif

/*********************************************
 
 unit Names

*********************************************/
/*---------------------------------------------------------------------------
  Names units.  As well as
  naming a single unit, this function can name a vector or 2-D array of
  units.  The name may then be used during simulation from the command
  interface, and may also be used during network construction.
  name is a pointer to the character string name to be given.   type
   is the type of name - SCALAR, VECTOR, or ARRAY.   index is
  the index of the unit to be named, or the first unit in the vector or
  array.   length is the number of units if it is a VECTOR, and the
  number of columns if it is an ARRAY, and is undefined for SCALAR.
  depth is the number of rows for an ARRAY, and is undefined for
  SCALAR and VECTOR.  A name which is specified as VECTOR or ARRAY will
  apply that name to the unit with the index specified, and to the
  requisite number of units following it in the unit array. 
----------------------------------------------------------------------------*/
				
NameUnit(name,type,index,length,depth)
    char *name;
    int type,index,length,depth;

{
#ifdef BFLY
  int off,newtab,oldtab;
#endif
  int s,finish;
  Index *indices, tabind;			  /* Index is char for uni */

BFLYCATCH

  if (!(type == SCALAR || type == VECTOR || type == ARRAY))
    LOGfprintf(stderr,"Naming failed, unknown type in NameUnit: %d\n\n",type);
  else
    if ((indices = EnterName(name,type,index,length, /* enter in name table */
			     depth,&tabind)) == NULL) /* final param not */
      LOGfprintf(stderr,
	      "Name Unit: could not enter name in name table: %s\n",
	      name);
    else
      {						  /* needed for uniproc */
	switch (type)
	  {
	  case SCALAR:
	    finish = index+1;
	    break;
	  case VECTOR:
	    finish = index + length;
	    break;
	  case ARRAY:
	    finish = index + (length*depth);
	    break;
	  default:
	    finish = index;
	    LOGfprintf(stderr,
		       "Not scalar, vector or array type in NameUnit: %d\n",
		       index);
	  }
	if (finish > LastUnit)
	  {
	    LOGfprintf(stderr,"Unit index out of range.. last is %d\n",
		       NoUnits-1);
	    AlterName(name,STRING_SYM,0,0,0,&tabind);
	    return 0;
	  }
#ifndef BFLY					  /* uniprocessor: simple! */

	for(s = index;s < finish;s++)
	  UnitList[s].name = indices;		  /* save pointer in unit */
      }

#else						  /* butterfly version */

        oldtab = SIM_UNIT(index);		  /* adjust index table(s) */
	indices = (Index *)Map_Obj(table_list[oldtab].index,0,RW_rw_);
	for(s = index;s < finish;s++)
	  {
	    if((newtab = SIM_UNIT(s)) != oldtab)
	      {
		Unmap_Obj(table_list[oldtab].index,indices);
		indices = (Index *)Map_Obj(table_list[newtab].index,0,RW_rw_);
		oldtab = newtab;
	      }
	    off = OFFSET_UNIT(s);
	    indices[off].table = tabind.table;
	    indices[off].offset = tabind.offset; /* */
	  }
	Unmap_Obj(table_list[oldtab].index,indices);
      }
#endif						  /* end butterfly version */
  return 0;
BFLYTHROW("NameUnit",0)
}
/*---------------------------------------------------------------------------
  Returns the index of the unit with the given name.  If the name is
  that of a VECTOR, then  column  gives offset of the unit within
  the vector.  If the name is that of an ARRAY, then  column  and
   row  give the column and row of the unit within the array.  If
  the name is not that of a unit, or either of the indices are out of
  range, then the function returns -1.
----------------------------------------------------------------------------*/

NameToInd(name, offset, row)
    char *name;
    int offset,row;
{
  NameDesc nte;
  
BFLYCATCH

  if (FindName(name,&nte) == NULL)
    return -1;
  switch (nte.type)
    {
    case SCALAR:
      return nte.index;
    case VECTOR:
      if (offset >= nte.size)
	{
	  LOGfprintf(stderr,"index out of range: %s[%d]\n",name,offset);
	  return -1;
	}
      return (nte.index + offset);
    case ARRAY:
      if (offset >= nte.size || row >= nte.length)
	{
	  LOGfprintf(stderr,"index out of range: %s[%d][%d]\n",
		     row,name,offset);
	  return -1;
	}
      return (nte.index + row*nte.size + offset);
    default:
      LOGfprintf(stderr,"%s not a unit name; type is %d\n",name,nte.type);
      return -1;
    }
BFLYTHROW("NameToInd",0)
}

/*---------------------------------------------------------------------------
  If  name  is the name of a unit type, a unit, a site, a
  function, a unit state, a set, or an unused name in the name table,
  NameToType returns a pointer to a volatile string detailing the type
  of name, e.g. ``unit vector name''.  If the name is not found, it
  returns a pointer to the volatile string ``unknown name''.
----------------------------------------------------------------------------*/

char * NameToType(name)
     char * name;

{
  NameDesc nte;
  static char buff[80];
  char buf[32];
  char * str;

  if (FindName(name,&nte) == NULL)
    str = "unknown name";
  else
    switch (nte.type)
      {
      case SCALAR:
	str = "unit name";
	break;
      case VECTOR:
	str = "unit vector name";
	break;
      case ARRAY:
	str = "unit array name";
	break;
      case SET_SYM:
	str = "set name";
	break;
      case STATE_SYM:
	str = "state name";
	break;
      case STRING_SYM:
	str = "unused name";
	break;
      case TYPE_SYM:
	str = "unit type name";
	break;
      case FUNC_SYM:
	str = "function name";
	break;
      case SITE_SYM:
	str = "site name";
	break;
      case DATA_SYM:
	str = "variable name";
	break;
      case CODE_SYM:
	str = "code unit name";
	break;
      default:
	(void) sprintf(buf,"unknown type (%d) of name",nte.type);
	str = buf;
      }
  strcpy(buff,str);
  return (buff);
}

/*---------------------------------------------------------------------------
  Returns a pointer to a volitile string containing name of the unit
  with index  u , or  **NO NAME**  if
  the unit has not been given a name.  If the name is that of a VECTOR
  or ARRAY, the name has the form  name[offset]  or 
  name[row][column] .
----------------------------------------------------------------------------*/

char * IndToName(ind)
    int ind;

{
  static char	buff[80];
  NameDesc nte;
  int j,k;
  char * nptr;

BFLYCATCH
#ifdef BFLY					  /* butterfly */
  Index * indices;
  int tab,offset;
  NameDesc *table;
						  /* get table entry */
  if (OFFSET_UNIT(ind) > NoUnits - 1)
    return NULL;
  indices = (Index *) Map_Obj(table_list[SIM_UNIT(ind)].index,0,RW_rw_);
  tab = indices[OFFSET_UNIT(ind)].table;
  offset = indices[OFFSET_UNIT(ind)].offset;
  Unmap_Obj(table_list[SIM_UNIT(ind)].index,indices);
  if((tab + offset) == 0)
    {
      (void) sprintf(buff,"%s","**NO NAME**");
      return buff;
    }
  table = (NameDesc *) Map_Obj(table_list[tab].table,0,RW_rw_);
  nte = *(table+offset);	/* copy name table entry */
  Unmap_Obj(table_list[tab].table,table);

#else						  /* uniprocessor */

  if (ind > NoUnits - 1)			  /* get table entry */
    return NULL;
  if ((nptr = UnitList[ind].name) == NULL)
    return "**NO NAME**";
  if (!FindName(nptr,&nte))
	return "?";

#endif

  switch (nte.type)				  /* all versions */
    {
    case SCALAR:
      return nte.name;
    case VECTOR:
      {
	(void) sprintf(buff, "%s[%d]", nte.name, ind - nte.index);
	return buff;
      }
    case ARRAY:
      {
	k = (ind - nte.index)/nte.size;
	j = (ind - nte.index) % nte.size;
	(void) sprintf(buff,"%s[%d][%d]", nte.name, k, j);
	return buff;
      }
    default:
      LOGfprintf(stderr,
		 "Unit %d holds pointer to non-unit name: %s\n",ind,nte.name);
    }
  return NULL;
BFLYTHROW("IndToName",0)
}

/****************************************************

 state Names

****************************************************/


/***** DeclareState *****/
/*---------------------------------------------------------------------------
  Associates a name with a state number.  num must be in the range 0 to 99.
----------------------------------------------------------------------------*/

DeclareState(name,num)
    char *name;
    int num;
{
    NameDesc look;
    Index * nptr,tabind;

BFLYCATCH
    if(num < 0 || num >= NoStates)
      {
	LOGfprintf(stderr, "DeclareState: state index must be in 0..%d\n",NoStates);
	return;
      }

    if (FindName(name,&look) == NULL || look.type == STRING_SYM)
      {				/* name not in use yet */
	if (StateNames[num] != NULL)
	  {
	    LOGfprintf(stderr,"Warning: deleting state %s\n",
		       StateNames[num]);
	    AlterName(StateNames[num],STRING_SYM,0,0,0,&tabind);
	  }				/* previous name no longer a state */
      }
    else			/* name in use */
      if (look.type == STATE_SYM)
	if (look.index == num)	/* already mapped */
	  return;
	else
	  {			/* mapped to different value */
	    LOGfprintf(stderr,"Warning: changing value for state %s from %d to %d\n",name,look.index,num);
	    StateNames[look.index] = NULL;
	    if (StateNames[num] != NULL)
	      {
		LOGfprintf(stderr,"Warning: deleting state %s\n",
			   StateNames[num]);
		AlterName(StateNames[num],STRING_SYM,0,0,0,&tabind);
	      }
	  }
      else			/* name already usef for something else */
	{
	  LOGfprintf(stderr,"Error: Declaring %s a state name, but already known as %s\n",name,NameToType(name));
	  return;
	}

    nptr = EnterName(name,STATE_SYM,num,0,0,&tabind);
    StateCount++;
	
#ifndef BFLY			/* uniprocessor */
    StateNames[num] = nptr;
#else				/* butterfly */
    StateNames[num] = StoreString(name);
#endif
BFLYTHROW("DeclareState",0)
}

/***** NameToState *****/
/*---------------------------------------------------------------------------
  Passed a state  name , returns the state number, or -1
  if the  name  is not that of a state.
----------------------------------------------------------------------------*/


NameToState(name)
    char *name;
{
    NameDesc look;

BFLYCATCH
    if (FindName(name,&look) == NULL)		/* no such name */
      return -1;
    if (look.type == STATE_SYM)
      return (look.index);
    return -1;
BFLYTHROW("NameToState",0)
} /* NameState */


/***** StateToName *****/
/*---------------------------------------------------------------------------
   Passed a state number, returns the name.  If there is no predefined
   mapping, the name is the string version of the number (volatile).
----------------------------------------------------------------------------*/

char *StateToName(snum)
    int snum;
{
    static char tname[10];

BFLYCATCH
    if(snum < 0 || snum >= NoStates || StateNames[snum] == NULL)
      {
        (void) sprintf(tname,"%d\0",snum);
        return &tname[0];
      }  
    return StateNames[snum];
BFLYTHROW("StateToName",0);
} /* StateName */

/* Set name functions */

/***** NameSet *****/
/*---------------------------------------------------------------------------
   Passed a set name, returns the number.  Returns -1 if no such name.
----------------------------------------------------------------------------*/

NameToSet(name)
    char *name;
{
    NameDesc look;

BFLYCATCH
    if (FindName(name,&look) == NULL)		/* no such name */
      return -1;
    if (look.type == SET_SYM)
      return (look.index);
    return -1;
BFLYTHROW("NameToSet",0)
} /* NameToSet */


/***** SetToName *****/
/*---------------------------------------------------------------------------
   Passed a set number, returns the name.  If there is no predefined
   mapping, the name is the string version of the number (volatile).
----------------------------------------------------------------------------*/
 
char *SetToName(snum)
    int snum;
{
    static char tname[10];

BFLYCATCH
    if(snum < 0 || snum >= NoSets || SetNames[snum] == NULL)
      {
        (void) sprintf(tname,"%d\0",snum);
        return &tname[0];
      }  
    return SetNames[snum];
BFLYTHROW("StateToName",0);
} /* SetToName */

/****************************************************

  Item table management

****************************************************/
static int NoItems = 0;		/* greatest item index used to date */
static int MaxNoItems = 100;	/* maximum number of items (can increase) */
static MappingTable * si_item_Names = NULL;

MappingTable * NameToItem(name,item,type)
     char * name;
     MappingTable * item;
     int type;

{
  NameDesc nte;

  if (FindName(name,&nte) == NULL)
    return NULL;
  if (nte.index < 0 || nte.index > NoItems || (nte.type != type && type >= 0))
    return NULL;
  *item = *(si_item_Names + nte.index);
  return item;
}

MappingTable * PointerToItem(ptr,item,type)
     char * ptr;
     MappingTable * item;
     int type;

{
  register int i;
  register MappingTable * titem;
  NameDesc nte;

  for (i = 0, titem = si_item_Names;
       i <= NoItems &&
       (titem->name == NULL || titem->item.fileinfo != ptr ||
	(FindName(titem->name,&nte)->type != type && nte.type >= 0) );
       i++, titem++);
  if (i > NoItems)
    return NULL;
  *item = *titem;
  return item;
}

MappingTable * IndexToItem(index,item)
     int index;
     MappingTable * item;

{
  if (index < 0 || index > NoItems)
    return NULL;
  *item = *(si_item_Names + index);
  return item;
}


/***** NameToFunc *****/
/*---------------------------------------------------------------------------
  If  name  is the name of a user function, or a simulator command
  function, then NameToFunc returns a pointer to the function, otherwise
  NULL.
----------------------------------------------------------------------------*/

func_ptr NameToFunc(name)
     char    *name;

{
    MappingTable mte;

BFLYCATCH
    if (NameToItem(name,&mte,FUNC_SYM) == NULL)
      return NULL;
    else
      return (mte.item.func);
BFLYTHROW("NameToFunc",0)
}

/***** FuncToName *****/
/*---------------------------------------------------------------------------
  If  fncp  is a pointer to a user function, e.g a
  unit function, or to a simulator command function, then FuncToName
  returns a pointer to the name of the function, otherwise NULL.
----------------------------------------------------------------------------*/

char * FuncToName(fncp)
     func_ptr fncp;
{
    MappingTable mte;
BFLYCATCH
  if (PointerToItem(fncp,&mte,FUNC_SYM) == NULL || mte.name == NULL)
    return "**NO NAME**";	/* return valid pointer! */
  else
    return (mte.name);
BFLYTHROW("FuncToName",0)
}

/***** InitFuncNames *****/
/*---------------------------------------------------------------------------
  called from the mainline to set up the function name/pointer mapping.
  Calls functions build by makebind.
----------------------------------------------------------------------------*/
 
si_InitFuncNames()

{
BFLYCATCH
#ifdef BSIM			/* no commands interface on sims */
  si_AddFuncToTable("NullFunc",NullFunc);
#else
  si_cmd_bind_func();
#endif
  si_user_bind_func();
BFLYTHROW("InitFuncNames",0)
}

AddItemToTable(name,ptr,type,prev,ptrlist,codeindx)
     char * name, * ptrlist;	/* ptrlist not really char *, but only */
     int * ptr;			/* passed on as param to another proc */
     int type, prev, codeindx;

{
  NameDesc nte, * ntp, ntf;
  int retval;
  register int i;
  register int j;
  register char *optr;
  register char *nptr;
  Index *iptr;
  MappingTable * new_item_Names, mte, *mtp;

BFLYCATCH
  if (prev < 0)
    retval = 0 - codeindx;
  else
    retval = 0 - prev;
  if (si_item_Names == NULL)	/* make table if first time called */
    {
      si_item_Names = (MappingTable *) si_malloc (MaxNoItems *
					    sizeof(MappingTable));
      for (i = 0; i < MaxNoItems; i++)
	(si_item_Names+i)->name = NULL;	/* initialize */
    }
  if (!((ntp = FindName(name,&nte)) == NULL || ntp->type == STRING_SYM ||
      ntp->type == type))
    {
      LOGfprintf(stderr,"Unable to add %s to name table as function/variable,already used as %s\n",name,NameToType(name));
      return(retval);
    }
  if (ntp != NULL && ntp->type == type)
    { 
      if (ntp->size < 0)
	{
	  LOGfprintf(stderr,"Attempt to reenter library function %s in item table, can't handle.\n",name);
	  return(retval);
	}
      LOGfprintf(stderr,"!! references to %s in %s will get old version of %s\n",name, IndexToItem(ntp->size,&mte)->name, name);
      for (i = 0,mtp = si_item_Names+ntp->size; /* code unit containing item */
	   mtp->next != ntp->index; /* follow chain to this item */
	   mtp = si_item_Names+mtp->next)
	if (FindName(mtp->name,&ntf)->type == FUNC_SYM)
	  i++;			/* count functions */
      mtp->next = (si_item_Names+ntp->index)->next; /* remove from chain */
      if (mtp->next < 0 && i == 0) /* no functions left in old code unit */
	{
	  LOGfprintf(stderr,"No accessible functions left in codeunit %s, deleting it...\n",IndexToItem(ntp->size,&mte)->name);
	  si_RemoveCodeUnit(si_FreeCodeUnit(ntp->size));
	}
      i = ntp->index;
    }
  else
    {
      for (i = 0; i < MaxNoItems &&
	   (si_item_Names+i)->name != NULL; i++);
      if (i >= MaxNoItems)
	{			/* make a bigger table, copy and update */
	  LOGfprintf(stderr,"Expanding item table...\n");
	  new_item_Names = (MappingTable *) si_malloc ((MaxNoItems + 200) *
						    sizeof (MappingTable));
	  j = MaxNoItems * sizeof(MappingTable); /* no. bytes to copy */
	  for (optr = (char *) si_item_Names, /* copy old table to new */
	       nptr = (char *) new_item_Names, i = 0;
	       i++ < j;
	       *nptr++ = *optr++);
	  free((char *)si_item_Names);	/* free up old table */
	  si_item_Names = new_item_Names;
	  for (i = 0; i < 200; i++) /* initialize extra items in new block */
	    (si_item_Names+MaxNoItems+i)->name = NULL;
	  i = MaxNoItems;
	  MaxNoItems+=200;
	}
    }
  if (i > NoItems)
    NoItems = i;
  if (type == FUNC_SYM)
    si_ChangeFuncPointers(name,ptr,ptrlist);/*redo ptrs in units,sites,links */
#ifndef BSIM			/* bfly control and sim enter globally */
  iptr = AlterName(name,type,i,codeindx,0,&nte); /* iptr is Index* */
#endif
#ifdef BFLY			/* bfly control and bfly sim enter locally */
  nptr = StoreString(name);	/* nptr is char* */
#else
  nptr = iptr;
#endif
  (si_item_Names+i)->item.intval = ptr;
  (si_item_Names+i)->next = prev;
  (si_item_Names+i)->name = nptr;
  return(i);			/* positive */
}

AddItemToChain(itemindx, chainindx)
     int itemindx, chainindx;

{
  NameDesc nte;

  (si_item_Names+itemindx)->next = chainindx;
  if (!FindName((si_item_Names+itemindx)->name,&nte))
  {
	LOGfprintf(stderr, "AddItemToChain(%d,%d): FindName(%s) failed!\n",
		itemindx, chainindx, (si_item_Names+itemindx)->name);
	return;
  }
  AlterName(nte.name,nte.type,nte.index,itemindx,0,&nte);
}

DeleteItemChain(index)
     int index;

{
  register int i;
  MappingTable * mtp;
  NameDesc nte;

  for (i = index;
       i >= 0;
       i = mtp->next)
    {
      mtp = si_item_Names+i;
      AlterName(mtp->name,STRING_SYM,0,0,0,&nte); /* free nametable */
      mtp->name = NULL;	/* free up maptable entry */
    }
}








