/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*--------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: May 1 1987
----------------------------------------------------------------------------*/
/*                            MAIN LINE CODE                                */


#include <stdio.h>
#include <ctype.h>				  /* token predicates */
#include <varargs.h>

#ifdef BFLY
#  include "bflycontrol.h"
#else
#  include <sys/file.h>
#  include "uniproc.h"
#  include <signal.h>
#  include <sgtty.h>
#endif
#include "lex.h"
#include "version.h"

char * calloc();
char * strcat();
char * strcpy();

FILE * GetCmdFile();
static command_reader(), read_line(), do_line();

#ifdef BFLY					  /* butterfly flush */
#  define BFLUSH fflush(stdout); Sleep(1000);
#else
#  define BFLUSH
#endif

int si_Running = FALSE;		/* set to true in command_reader */
				/* set to false in Cmd_return */

/** set by Lex, the command parser **/

static char YYarg[MAX_COMMAND_LEN];		  /* text matched */
static char command_line[MAX_COMMAND_LEN];	  /* command line characters */
static int current;				  /* index into command_line */

static FILE *infile;				  /* input file */
static int reading = 0;				  /* 1 if reading from file */
static FILE *newin[MAXFILES];			  /* array of read file ptrs */
static next_index = 0;				  /* index into above array */
static int saved = 0;				  /* if not 0, UnLex value */

static Interrupt = 0;				  /* set if interrupt occurs
						     during guarded code */
#ifndef BFLY
  static int	term_fd;			  /* used by stty and gtty */
  static struct sgttyb	term_state;		  /* to store state of term */
#endif

#define GETACHAR (command_line[current++])	  /* gets one command char */
#define PUTACHAR (current -= 1)			  /* "unget" one character */


/***** Lex *****/
/*
  Lex puts the next token in YYarg ( a character string )
  and returns type of token found.  Command arrives in char string.
*/

static int Lex(command_line)
     char * command_line;
{
  int i,c;
#ifdef mips
  /* this is disgusting but we'll just do this to get it to compile for now */
  static char myeof = (char) EOF;
#else
#define myeof EOF
#endif
  
BFLYCATCH
  if(saved)
    {   					  /* token is in saved */
      i = saved;				  /* note saved token  */
      saved = 0;				  /* reset saved to false */
      return i;					  /* return token */
    }

  while((c = GETACHAR) == ' ' || c == '\t');	  /* ignore whitespace */
  if(c == '\n' || c == '\0') return END_LINE;	  /* return if newline */
  if(c == myeof) return EOF;			  /* or end-of-file */
  if(c == '?')
    {
      YYarg[0] = '?';				  /* fill in YYarg */
      YYarg[1] = '\0';				  /* terminate string */
      return QUESTION_MARK;			  /* give help */
    }
  if(c == '"')
    {						  /* get quoted string */
      for(i = 0;				  /* !save opening quote */
	((YYarg[i] = GETACHAR) != '"') &&	  /* read to closing quote */
	YYarg[i] != '\n';i++);			  /* or newline */
    YYarg[i] = '\0';				  /* add end of string char */
    return STRING;				  /* return type */
    }
  else
    {						  /* read to whitespace */
      for(i = 1,YYarg[0] = c;			  /* save first char */
	(c = GETACHAR) != ' ' && c != '\n' &&	  /* read to space or nl */
	  c != '\0' && c != '\t' && c != myeof;	  /* or eos or tab or eof */
	i++)
	YYarg[i] = c;				  /* and save */
      YYarg[i] = '\0';				  /* add end of string char */
      PUTACHAR;					  /* read one too many */
      return TOKEN;				  /* return type */
    }
BFLYTHROW("Lex (control)",0)
}

/**** UnLex ****/
/*
    Next call to Lex will return tok.
*/
static UnLex(tok)
{
    saved = tok;
}

#ifndef BFLY					  /* terminal handling for */
						  /* uniprocessor version */
/* function call when output pipe is broken */
static BrkPipe()
{
  if ((Dispf = popen("cat >/dev/null", "w")) == NULL)
    perror("sim");
}

/*---------------------------------------------------------------------------
   function called when interrupt during run (^C).  Tries to enter the
   command interface, but if during guarded code, processing is delayed.
----------------------------------------------------------------------------*/

static AsyncInt()

{
  int save;
  unsigned int saverr;

/*  char c; */
/*  while ((c = getc(infile)) != '\n' && c != EOF);*/
  if (Guarded)
    {
      LOGfprintf(stdout,
		 "\nInterrupt during guarded code... will be processed asap\n");
      Interrupt = TRUE;
    }
  else
    {
      Interrupt = FALSE;
      if(stty(term_fd, &term_state))
	perror("Cannot restore terminal modes");
      LOGfprintf(stdout,"\n\nEntering command interface.. quit to continue\n");
      save = LogCmd;
      saverr = Errors;
      Errors = 0;
      LogCmd = FALSE;
      Debug++;
      debug_command_reader("interrupt");
      if (Debug > 0)
	Debug--;
      LogCmd = save;
      Errors = saverr;
/*      ungetc(c,infile); */
      LOGfprintf(stdout,"exiting interrupt interface... hit return key twice");
      fflush(stdout);
    }
}

/*---------------------------------------------------------------------------
   Guard  simply increments  Guarded  and exits.  
  Release  decrements  Guarded  and checks if an interrupt has
  occurred since the previous call to  Guard .  If so, and if 
  Guarded  is zero, the interrupt processing routine is called to
  enter the interrupt interface.  The fact that  Guarded  is
  incremented and decremented by each matched pair of calls to  Guard
   and  Release  means that these calls may be nested, and
  interrupts will only be processed when the outermost level is reached.
----------------------------------------------------------------------------*/

Guard()

{
  Guarded++;
}

Release()

{
  Guarded--;
  if (Interrupt && Guarded == 0)
    {
      LOGfprintf(stdout,"\nProcessing delayed interrupt...\n");
      AsyncInt();
    }
}

#endif						  /* end of uniproc. stuff */

/***** main *****/
/*---------------------------------------------------------------------------
                              MAIN LINE CODE

This is the mainline routine.  It is either called from the <main> function
in si_main.c or from some other system.
----------------------------------------------------------------------------*/

simulator_setup(argc,argv)
int argc;
char ** argv;

{
    FILE *nfp;
    func_ptr func;

BFLYCATCH
#ifdef BFLY
    Set_Process_Hang(NULL,TRUE);		  /* ???? */
    PipeFlag = FALSE;		/* no piping yet implemented */
#else
    Guard();
    si_InitFiles();				  /* initialise log,chk,net */
#endif

    si_Program = argv[0];		/* save program name for time-stamping */
    LOGfprintf(stdout,"\nRochester Connectionist Simulator (version %s patchlevel %s%s)\n\n", RCS_VERSIONSTRING, RCS_PATCHLEVEL, RCS_LOCALPATCHLEVEL);
    Format++;
    LOGfprintf(stdout,"Copyright (C) 1989 University of Rochester\n\
This software comes with ABSOLUTELY NO WARRANTY; for details, type \
`status warranty'.  This is free software, and you are welcome to \
redistribute it under certain conditions; type `status copying' for \
details.\n\n");
    Format--;

#ifdef BFLY
    LOGfprintf(stdout,"Butterfly ");

  catch
    Names = (char **) si_calloc(NameSize,sizeof(char *));
#endif

#ifdef FSIM
    LOGfprintf(stdout,"floating point version\n");
#else
    LOGfprintf(stdout,"integer version\n");
#endif
#ifdef TSIM
    LOGfprintf(stdout,"with propagation delay\n\n");
#else
    LOGfprintf(stdout,"no propagation delay\n\n");
#endif

    infile = stdin;
    StateNames = (char **) si_calloc(NoStates,sizeof(char *));
#ifndef BFLY
    si_InitFuncNames();				  /* establish name to
						     function bindings */
    si_init_make();
#endif
    LOGfprintf(stdout,"Debugging turned on, not in Auto-Fix mode\n");

#ifdef BFLY
  onthrow
    when(TRUE)
      {
	LOGfprintf(stderr,"control in local initialization: %s\n",throwtext);
	exit(1);
      }
  endcatch;
    
    setup(argc,argv);
    si_InitFuncNames();		/* name table required; init in setup */
#else
    Release();
    if(argc >= 2)
      {
	if (*argv[1]!='-')
	  {
	    if ((nfp=fopen(argv[1],"r")) == NULL)
	      {
		LOGfprintf(stdout,"can't open file %s\n",argv[1]);
		exit(0);
	      }
	    else 
	      {
		NetLoad(nfp);
	      }
	  }
      }

    if((term_fd = open("/dev/tty", O_RDWR, 0)) < 0)
	perror("Could not open terminal");
    if(gtty(term_fd, &term_state))
	perror("Cannot save terminal modes");
    signal(SIGPIPE, BrkPipe);
    signal(SIGINT, AsyncInt);
  if ((func=NameToFunc("gi_start"))!=NULL)
    func(argc,argv);		/* start up graphics stuff */
  else
    CmdFile = GetCmdFile();	/* no files on butterfly */
#endif    
  }

simulator_run()

{
  func_ptr func;

#ifndef BFLY
  if ((func=NameToFunc("gi_transfer"))==NULL)
    command_reader();
  else
    func();
#else
    command_reader();
#endif
}

simulator_quit()

{
  func_ptr func;

#ifndef BFLY
  if ((func=NameToFunc("gi_quit"))!=NULL)
    func();
#endif
}


/*---------------------------------------------------------------------------
  This is the function called by the  Graphics Interface  to pass
  a command to the simulator.   cmd_line  is a character string
  containing the command to be executed.   extern_command_reader 
  executes the command and returns a message string if any output to 
  stderr  occurred via  LOGfprintf  during the command, or NULL
  if there was no such output.
----------------------------------------------------------------------------*/

char * extern_command_reader(cmd_line)		  /* called externally */
     char * cmd_line;				  /* command string */

{
  static char * GiErrMsg = "Check simulator window for information";

  si_GiErrCnt = 0;					  /* expect no error */
  do
    if (infile != stdin)			  /* ie reading cmd file */
      {
	read_line();				  /* read from infile */
	do_line(command_line);			  /* and execute command */
      }
    else					  /* just do this command */
      do_line(cmd_line);			  /* could be a read */
  while (infile != stdin);
  if (si_GiErrCnt != 0)
    return (GiErrMsg);				  /* return error message */
  else
    return NULL;				  /* no error message */
}

/*---------------------------------------------------------------------------
  Loop forever, reading a line and executing the command.  Exit is done in
  Cmd_quit, or elsewhere
----------------------------------------------------------------------------*/

static command_reader()

{
BFLYCATCH
  si_Running = TRUE;
  while (si_Running == TRUE)
    {
      if(!reading)
	LOGfprintf(stdout,"-> ");
      read_line();				  /* from infile */
      do_line(command_line);
    }
BFLYTHROW("command_reader (control)",0)
}

/*---------------------------------------------------------------------------
  Reads in a line and puts it in the buffer commmand.
----------------------------------------------------------------------------*/

static read_line()

{
  int i,c;
  
BFLYCATCH
  for(i = 0;i < MAX_COMMAND_LEN - 1;i++)
    {
      command_line[i] = c = getc(infile);
      if (LogCmd && c != EOF)
	putc(c,CmdFile);
      if (Logging && c != EOF)
	putc(c,LogFile);
      if(i == 0 && c == ' ')
	i--;
      if(c == '\n')
	if(i == 0)
	  {
	    LOGfprintf(stdout,"-> ");
	    i--;
	  }
	else
	  break;
      if(c == EOF) break;
    }
  if(i == MAX_COMMAND_LEN - 1)
    {
      LOGfprintf(stdout,"line too long (max %d characters): input ignored\n",
	     MAX_COMMAND_LEN);
      command_line[0] = '\0';
    }
  else
    command_line[i+1] = '\0';
BFLYTHROW("read_line (control)",0)
}

/*---------------------------------------------------------------------------
  Call the function whose pointer is the argument, with an argc-argv
  structure as parameters, built with calls to Lex.
----------------------------------------------------------------------------*/

static CallFunc(func,cmd_line)
     int  (* func)();
     char * cmd_line;

{
  char *argv[NUMARGS];
  int   argc = 0;

BFLYCATCH
  do
    {
      argv[argc] = (char *) si_malloc(strlen(YYarg) + 1);
      strcpy(argv[argc++], YYarg);
    }
  while(Lex(cmd_line) != END_LINE);
  func(argc, argv);
#ifdef BFLY
/*  LOGfprintf(stdout,"freeing up args after %s\n",FuncToName(func));*/
#endif
  while (argc > 0)
    free(argv[--argc]);				  /* free up arg storage */
  return;
BFLYTHROW("CallFunc (control)",0)
}  

/*---------------------------------------------------------------------------
  Using the command line read in by read_line, call Lex to get the first
  token.  Expect a command name, but others are possible.
----------------------------------------------------------------------------*/

static do_line(cmd_line)
     char * cmd_line;

{
    int token, i, j;
    func_ptr func;
    char * cmd;
    MappingTable mte;
    char scmd[MAX_COMMAND_LEN];

BFLYCATCH
    current = 0;
    (void) sprintf(scmd, "Cmd_");
    switch(token= Lex(cmd_line))
      {
      case TOKEN:				  /* should be command */
	strcat(scmd,YYarg);			  /* all commands prefix */
	if ((func = NameToFunc(scmd)) == NULL)	  /* can't find command */
	  {
	    LOGfprintf(stdout,"Cannot find command `%s' : input ignored\n", YYarg);
	    if (NameToFunc(YYarg) != NULL)	  /* not command function */
	      LOGfprintf(stdout,"The function `%s' exists; use `call' to access it\n",
		     YYarg);
	  }
	else
	  CallFunc(func,cmd_line);		  /* call command function */
	break;
      case STRING:				  /* unexpected */
	LOGfprintf(stdout,"? Command expected, quoted string found: %s\n", YYarg);
	break;
      case END_LINE : break;			  /* ignore blank lines */
      case EOF :
	if(infile == stdin)			  /* if reading from tty */
	  {
#ifdef BFLY
	    exit(0);
#else
	    clearerr(stdin);			  /* more reading to do */
	    Cmd_quit(1,(char**)NULL);		  /* save & exit to shell */
#endif
	  }
	else					  /* must be reading file */
	  CloseCmdFile();	                  /* so close it */
	break;
      case QUESTION_MARK:
	PipeBegin();
	LOGfprintf(Dispf,"Commands are:\n");
	for (i = j = 0; IndexToItem(i,&mte) != NULL; i++)
	  if ((cmd = mte.name) != NULL && cmd[0]=='C' && cmd[1]=='m' &&
	      cmd[2]=='d' && cmd[3]=='_' && cmd[5] != '\0')
	    if (j++%5 == 0)
	      {
		si_termitem(Dispf);
		si_additem(Dispf,"",cmd+4,(((j-1)%5)*15)+4,4);
	      }
	    else
	      si_additem(Dispf,"",cmd+4,(((j-1)%5)*15)+4,4);
	si_termitem(Dispf);
	LOGfprintf(Dispf,"Type `help <command>' or `<command> ?' for more information\n");
	PipeEnd();
	break;
      default :					  /* unknown */
	LOGfprintf(stdout,"Unknown command: %s(%d) ('?' for help)\n",YYarg,token);
	while(Lex(cmd_line) != END_LINE);		  /* ignore to end of line */
      }
BFLYTHROW("do_line (control)",0)
}

/*---------------------------------------------------------------------------
  Close the current command file.  Reset infile to be the one that opened it,
  or stdin.
----------------------------------------------------------------------------*/

CloseCmdFile()

{
BFLYCATCH
  fclose(newin[--next_index]);			  /* close file */
  if (next_index != 0)				  /* if from within file */
    infile = newin[next_index-1];		  /* reset to enclosing file */
  else				  
    {
      infile = stdin;				  /* reset to terminal */
      LogCmd = TRUE;
      reading = FALSE;
    }
BFLYTHROW("CloseCmdFile",0)
}

/*---------------------------------------------------------------------------
  Open a file to read commands from.
----------------------------------------------------------------------------*/

ReadCmdFile(fname)
     char * fname;


{
  FILE * fopen_obj();

BFLYCATCH
  if( (newin[next_index] =
#ifdef BFLY
       fopen_obj(fname,"r")
#else
       fopen(fname,"r")
#endif
       ) == NULL)
    LOGfprintf(stdout,"can't open file %s\n",fname);
  else 
    {
      infile = newin[next_index++];
      reading = TRUE;
      LogCmd = FALSE;
    }
BFLYTHROW("ReadCmdFile",0)
}

/***** PipeBegin *****/
/*
   sets up the file stream Dispf for output;  piped to command in PipeCommand
   if PipeFlag == 1; sent to standard out if not.
*/

PipeBegin()
{
BFLYCATCH
    if(PipeFlag && Dispf == stdout)
      {
	fflush(stdout);
        if((Dispf = popen(PipeCommand,"w")) != NULL)
	  return;
        else
	  LOGfprintf(stdout,"ERROR: cannot pipe to %s\n",PipeCommand);
      }
    Dispf = stdout;
BFLYTHROW("PipeBegin",0)
}

/***** PipeEnd *****/
/*
   closes the file stream Dispf if connected to pipe
*/

PipeEnd()
{        
BFLYCATCH
    if(PipeFlag && Dispf != stdout)
      pclose(Dispf);
    Dispf = stdout;
BFLYTHROW("PipeEnd",0)
}

#define LINE_LENGTH 75				  /* formatted line length */
#define INDENT 3				  /* formatted line indent */

/*---------------------------------------------------------------------------
An augmented-restricted version of fprintf.  The string str is written
to file fp, with substitution of the arguments in order for %s, %d, %f.
If Logging is TRUE, the string is written to the Log file as well as to
fp.  If fp is stderr then a message count for the Graphics Interface is
incremented.  If Format is TRUE, the function formats the string into
lines no longer than 75 characters, and indents all the lines thus
formed by 3 spaces.
----------------------------------------------------------------------------*/

LOGfprintf(va_alist)
va_dcl
{
  FILE *fp;
  register char *str;
  va_list argp;

  int helpcount = 0;				/* for formatting */
  char tbf[10],wbf[1024];			/* temp and word buffers */
  char bf[2048];				/* accumulate output chars*/
  int i,j,k;
  char c;

  va_start(argp);
  fp = va_arg(argp, FILE*);
  str = va_arg(argp, char*);

BFLYCATCH
#ifdef BFLY
/*    LockStdin();*/
#endif

  bf[0] = '\0';
  if (Format)
    {
      (void) sprintf(wbf,"%s","   ");
      j = INDENT;
    }
  else
    j = 0;
  for (i = 0; (c = str[i]) != '\0'; i++)	/* for each char in str */
    switch (c)
      {
      case '\n':
	if (Format)
	  {
	    if (helpcount + j > LINE_LENGTH - 1)
	      strcat(bf,"\n   ");
	    wbf[j++] = c;
	    wbf[j++] = '\0';
	    (void) strcat(bf,wbf);		/* add word to output */
	    j = 0;
	    helpcount = 0;
	  }
	else
	    {
	      wbf[j++] = c;
	      wbf[j++] = '\0';
	      (void) strcat(bf,wbf);		/* add word to output */
	      j = 0;
	    }
	break;
      case '\t':
	if (Format)
	  for (j--; helpcount%8 != 0; helpcount++)
	    wbf[j++] = ' ';
	else
	  wbf[j++] = c;
	break;
      case '%':
	switch(str[++i])
	  {
	  case 's':
	    (void) strcpy(wbf+j, va_arg(argp, char*));
	    j += strlen(wbf+j);
	    break;
	  case 'd':
	    (void) sprintf(wbf+j, "%d", va_arg(argp, int));
	    j += strlen(wbf+j);
	    break;
	  case 'x':
	    (void) sprintf(wbf+j, "%x", va_arg(argp, int));
	    j += strlen(wbf+j);
	    break;
	  case 'f':
	    (void) sprintf(wbf+j, "%f", va_arg(argp, double));
	    j += strlen(wbf+j);
	    break;
	  case 'g':
	    (void) sprintf(wbf+j, "%g", va_arg(argp, double));
	    j += strlen(wbf+j);
	    break;
	  case '0': case '1': case '2': case '3': case '4': case '5':
	  case '6': case '7': case '8': case '9': case 'h': case 'l':
	    tbf[0] = '%'; tbf[1] = str[i++];
	    tbf[2] = str[i]; tbf[3] = '\0';
	    (void) sprintf(wbf+j,tbf,va_arg(argp, int));
	    j +=strlen(wbf+j);
	    break;
	  case 'c':
	    wbf[j++] = va_arg(argp, int);
	    break;
	  case '%':
	    wbf[j++] = '%';
	    break;
	  default:
	    fprintf(stderr,
		    "Unknown print option in LOGfprintf: %%%c\n",str[i]);
	  }
	break;
      default:
	if (c == ' ')
	  if (Format)
	    {
	      if (helpcount + j > LINE_LENGTH - 1)
		{
		  strcat(bf,"\n   ");
		  helpcount = 0;
		}
	      if (!(j == 0 && helpcount == 0))
		{
		  wbf[j++] = c;
		  wbf[j++] = '\0';
		  (void) strcat(bf,wbf);	/* add word to output */
		  helpcount += strlen(wbf);
		  j = 0;
		}
	    }
	  else
	    {
	      wbf[j++] = c;
	      wbf[j++] = '\0';
	      (void) strcat(bf,wbf);		/* add word to output */
	      j = 0;
	    }
	else
	  if (Format)
	    {
	      if (helpcount == 0)
		{
		  for (k = 0; k < INDENT; k++)
		    wbf[j++] = ' ';
		  helpcount = 1;
		}
	      wbf[j++] = c;
	    }
	  else
	    wbf[j++] = c;
    }
  *(wbf+j) = '\0';
  (void) strcat(bf,wbf);			/* add anything left */
  fprintf(fp,"%s",bf);				/* print to given file */
#ifdef BFLY
  fflush(fp);
#else
  if (Logging)
    fprintf(LogFile,"%s",bf);			/* print to log file */
#endif
 if (fp == stderr)				/* for GI */
   si_GiErrCnt += 1;
#ifdef BFLY
/*    UnlockStdin();*/
#endif

  va_end(argp);

BFLYTHROW("LOGfprintf",0)
}

/***** UserWait *****/
/*-------------------------------------------------------------------------
   prints the provided prompt, then reads a single character in cbreak
   mode (not on Butterfly).  Returns the character, mapped to lower case.
---------------------------------------------------------------------------*/

#ifdef BFLY

UserWait(str)
  char * str;

{
  int c;

BFLYCATCH
  LOGfprintf(stdout,str);
  c = getchar();
  if (c != '\n')
    while (getchar() != '\n');
  if (c >= 'A' && c <= 'Z')
    c += ('a' - 'A');		/* map to lower case */
  return (c);
BFLYTHROW("UserWait",0)
}

#else

UserWait(prompt)
    char *prompt;
{
    struct sgttyb old,new;
    int ans;
 
    ioctl(0,TIOCGETP,&old);             /* set cbreak */
    new = old;
    new.sg_flags |= CBREAK;
    ioctl(0,TIOCSETP,&new);
    printf("%s",prompt);
    fflush(stdout);
    ans = getchar();
    ioctl(0,TIOCSETP,&old);             /* restore */
    putchar('\n');
    if(isupper(ans)) ans = tolower(ans);
    putchar('\n');
    return ans;
}

#endif
