/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*--------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: May 1 1987
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <sys/file.h>
#include "uniproc.h"

#define NO     1
#define OVER   2
#define APPEND 3
#define MAX_CMD_LEN 180

/* the following macro saves user-typed characters in the log and cmd files */
#define LOGGETC(c)  if ((c = getc(stdin)) != EOF) \
  { \
    if (LogCmd) \
      putc(c,CmdFile); \
    if (Logging) \
      putc(c,LogFile); \
  } else

static FILE * netfile = NULL;
static FILE * chkfile = NULL;
static FILE * logfile = NULL;
static FILE * cmdfile = NULL;
static char logname[80];
static char chkname[80];
static char netname[80];
static char cmdname[80];
static int logcnt = 0;
static int chkcnt = 0;
static int netcnt = 0;

static int confirm();

char * getwd();

/*---------------------------------------------------------------------------
  Called from mainline on startup to initialise file names.
----------------------------------------------------------------------------*/

si_InitFiles()

{
  logname[0] = '\0';
  chkname[0] = '\0';
  netname[0] = '\0';
  cmdname[0] = '\0';
}

/*---------------------------------------------------------------------------
  Confirms or replaces the default file name.  name is the default name.
  str is the file use: log, net, chk, cmd.  type is a hack for the command
  file because it is already open, -1 for command files.  Opens the file and
  returns a pointer to it.  Can overwrite or append to existing files.
----------------------------------------------------------------------------*/

static FILE * askfilename(name,str,type)
     char * name;
     char * str;
     int type;

{
  int i,j,c;
  char cmd[MAX_CMD_LEN];
  FILE * fp = NULL;
#ifdef mips
  /* this is disgusting but we'll just do this to get it to compile for now */
  static char myeof = (char) EOF;
#else
#define myeof EOF
#endif

  do
    {
      LOGfprintf(stdout,"%s file name [default: %s] > ",str,name);
      for(i = 0;i < MAX_CMD_LEN - 1;i++)
	{
	  LOGGETC(c);
	  cmd[i] = c;				  /* save in buffer */
	  if(i == 0 && (c == ' ' || c == '\t'))	  /* if initial whitespace */
	    i--;				  /* then ignore */
	  if(c == '\n' || c == EOF)
	    break;				  /* end of command */
	  if (i == MAX_CMD_LEN - 1)		  /* chomp excess */
	    i--;
	}
      if(i == MAX_CMD_LEN - 1)
	{
	  printf("line too long (max %d characters): input ignored\n",
		 MAX_CMD_LEN);
	  i = 0;
	}
      else
	{
	  if (cmd[0] == '\n' || cmd[0] == myeof)
	    if (type == -1)			  /* cmd file - hack */
	      return NULL;
	    else
	      strcpy(cmd,name);
	  else
	    cmd[i] = '\0';			  /* terminate string */
	  if ((fp = fopen(cmd,"r")) == NULL)	  /* open for reading? */
	    if ((fp = fopen(cmd,"w")) == NULL)	  /* open for writing */
	      LOGfprintf(stderr,"Cannot open file %s\n",cmd); /*  */
	    else
	      ;					  /* open ok for writing */
	  else					  /*  */
	    {					  /* open for reading */
	      switch (j = confirm("Overwrite file ",cmd)) /* overwrite ? */
		{
		case OVER:			  /* yes */
		  fclose(fp);
		  fp = fopen(cmd,"w");
		  break;
		case NO:			  /* no */
		  fclose(fp);
		  fp = NULL;
		  break;
		case APPEND:			  /* append */
		  fclose(fp);
		  fp = fopen(cmd,"a");
		  break;
		default:
		  LOGfprintf(stderr,
			  "Internal error: invalid return code from confirm: %d\n",
		    j);
		} /* switch */
	    }/* if */
	} /* if */
    } /* do */
  while (fp == NULL);
  strcpy(name,cmd);			  /* copy into file name */
  return (fp);
}

/*---------------------------------------------------------------------------
  Used to confirm a file name.  str is a message, name the file name.
  Returns OVER (overwrite), NO (don't use), or APPEND (append to existing).
----------------------------------------------------------------------------*/

static int confirm(str,name)
     char * str;
     char * name;

{
  int done = FALSE;
  int c;

  do
    {
      LOGfprintf(stdout,"%s%s (y,n,a[ppend]) ? ",str,name);
      do
	LOGGETC(c);
      while (c == ' ' || c == '\t');
      if (c == 'y' || c == 'Y')
	done = OVER;
      else
	if (c == 'n' || c == 'N')
	  done = NO;
	else
	  if (c == 'a' || c == 'A')
	    done = APPEND;
      do
	LOGGETC(c);
      while (c != '\n' && c != EOF);
    }
  while (!done && c != EOF);
  return (done);
}

/*---------------------------------------------------------------------------
  General routine for opening files for logging, checkpointing, saving,
  and command file moving.  fname is the suggested name. fp is the existin
  file pointer for the file, if not NULL then it is returned.  ext is the
  extension for the file type, e.g. cmd, log, net, chk.  cnt is the count
  to append to the extension, so name becomes: fname.ext.cnt.  Returns a
  pointer to the opened file.
----------------------------------------------------------------------------*/
  
static FILE * GetFile(fname,fp,ext,cnt)
     char * fname;
     FILE ** fp;
     char * ext;
     int * cnt;

{
  int i, j;

  if ((*fp) != NULL)
    return (*fp);
  if (fname[0] == '\0')		/* generate name from process id */
    (void) sprintf(fname, "%s%ld.%s.%d", "run", getpid(), ext, ++(*cnt));
  else
    {
      for (i = 0, j = strlen(fname);
	   i < j && fname[i] != '.';
	   i++);		/* find . if there */
      if (i == j && (*cnt) >= 0)		/* if no . */
	(void) sprintf(fname+i,".%s.%d\0",ext,++(*cnt)); /* append type and count */
    }
  return (*fp = askfilename(fname,ext,*cnt));
}

/*---------------------------------------------------------------------------
  For asking yes/no questions.  returns 1 for yes, 0 for no.
----------------------------------------------------------------------------*/

static int askyn(str,name)
     char * str;
     char * name;

{
  int done = FALSE;
  int c;

  do
    {
      LOGfprintf(stdout,"%s%s (y,n) ? ",str,name);
      do
	LOGGETC(c);
      while (c == ' ' || c == '\t');
      if (c != '\n' && c != EOF)
	{
	  if (c == 'y' || c == 'Y')
	    done = OVER;
	  else
	    if (c == 'n' || c == 'N')
	      done = NO;
	  do
	    LOGGETC(c);
	  while (c != '\n' && c != EOF);
	}
    }
  while (!done && c != EOF);
  return (done - 1);
}

/*---------------------------------------------------------------------------
  Opens a checkpoint file using GetFile.  Returns pointer to the file.
----------------------------------------------------------------------------*/

FILE * GetChkFile(fname)
     char * fname;

{
  if (fname != NULL)
    {
      strcpy(chkname,fname);
      chkcnt = 0;
    }
  return (GetFile(chkname,&chkfile,"chk",&chkcnt));
}

/*---------------------------------------------------------------------------
  Opens a file for logging the keyboard input only, and returns a
  descriptor to it.  The file name is of the form  run????.cmd.\# 
  where ???? is the process ID of the current process, and \# is an
  integer.
----------------------------------------------------------------------------*/

FILE * GetCmdFile()

{
  FILE * fp;
  
  LogCmd = TRUE;
  (void) sprintf(cmdname, "%s%ld.%s", "run", getpid(), "cmd");
  if ((fp = fopen(cmdname,"w")) == NULL)	  /* open for writing */
    {
      LOGfprintf(stderr,"Cannot open file %s to log commands\n",cmdname);
      LogCmd = FALSE;
    }
  return (fp);
}

/*---------------------------------------------------------------------------
  Closes the file opened with  GetCmdFile , and asks the user if
  it should be saved.  If the answer is yes, prompts for a name for the
  file and does a UNIX environment call to  mv  it to that name.
  Otherwise it does a UNIX environment call to  rm  the file.
----------------------------------------------------------------------------*/

SaveCmdFile()

{
  char cmd[160];
  char  fname[80];
  int i = -1;

  LogCmd = FALSE;
  fclose(CmdFile);
  if (askyn("Save command file",""))
    {
      strcpy(fname,cmdname);
      if (GetFile(fname,&cmdfile,"cmd",&i) != NULL)
	{
	  fclose(cmdfile);
	  (void) sprintf(cmd,"mv %s %s",cmdname,fname);
	  system(cmd);
	}
    }
  else
    {
      (void) sprintf(cmd,"rm %s",cmdname);
      system(cmd);
    }
}  

/*---------------------------------------------------------------------------
  Opens a log file using GetFile.  Returns pointer to the file.  If there
  is already a log file open, asks for confirmation, etc.
----------------------------------------------------------------------------*/

FILE * GetLogFile(fname)
     char * fname;

{
  if (fname == NULL)
    return (GetFile(logname,&logfile,"log",&logcnt));
  else
    if (logfile != NULL)
      if (askyn("Close existing log file ",logname))
	{
	  fclose(logfile);
	  logfile = NULL;
	  strcpy(logname,fname);
	  logcnt = 0;
	  return (GetFile(logname,&logfile,"log",&logcnt));
	}
      else
	return (logfile);
    else
      {
	strcpy(logname,fname);
	logcnt = 0;
	return (GetFile(logname,&logfile,"log",&logcnt));
      }
}

/*---------------------------------------------------------------------------
  Opens a file for saving the network using GetFile.  Returns pointer to
  the file.
----------------------------------------------------------------------------*/

FILE * GetNetFile(fname)
     char * fname;

{
  if (fname != NULL)
    {
      strcpy(netname,fname);
      netcnt = 0;
    }
  return (GetFile(netname,&netfile,"net",&netcnt));
}

/*---------------------------------------------------------------------------
  The next four functions are called when the simulator "thinks" a log file
  should (maybe) be started or terminated.
----------------------------------------------------------------------------*/

AskLogOn ()

{
  if (!Logging)
    if (askyn("Do you want a log file",""))
      LogOn();
}


AskLogOff ()

{
  if (Logging)
    if (askyn("Close the log file",""))
      LogOff();
}


LogOn()

{
  if (Logging)
    LOGfprintf(stderr,"Logging is already turned on!\n");
  else
    {
      LogFile = GetLogFile((char*)NULL);
      Logging = TRUE;
    }
}


LogOff()

{
  if (Logging)
    {
      fclose(LogFile);
      LogFile = NULL;
      logfile = NULL;
      Logging = FALSE;
    }
  else
    LOGfprintf(stderr,"Logging is already turned off!\n");
}

/*---------------------------------------------------------------------------
  Closes the checkpoint file, if one is open.
----------------------------------------------------------------------------*/

CloseChkFile()

{
  if (chkfile != NULL)
    {
      fclose(chkfile);
      chkfile = NULL;
    }
  else
    LOGfprintf(stderr,"Internal error: can't close checkpoint file, not open\n");
}

/*---------------------------------------------------------------------------
  Closes the network saving file if one is open.
----------------------------------------------------------------------------*/

CloseNetFile()

{
  if (netfile != NULL)
    {
      fclose(netfile);
      netfile = NULL;
    }
  else
    LOGfprintf(stderr,"Internal error: can't close net file, not open\n");
}




