/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*--------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: October 30th 1987
----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------
  This file provides functions to dynamically link in object files and
  reset simulator function pointers accordingly.  All global functions
  and data in the object files are made availible at the interface
----------------------------------------------------------------------------*/

/* some of the following "#ifdef mips" MIGHT be better off as "#ifdef ultrix" */
#ifdef mips
#include <sys/exec.h>
#include <nlist.h>
#else
#include <a.out.h>
#endif

#include <ctype.h>
#include <sys/file.h>
#include <sys/types.h>

#include "uniproc.h"

#ifdef mips
#define NAME_OFFSET 0
#define TEXT_ADDR " "
#else
#define NAME_OFFSET 1
#define TEXT_ADDR "-T 0"
#endif

#define SIMLIB "-lbp"
				/* simulator libraries to be searched */
				/* basic library always in program */

typedef struct l_e
{
  char * name;
  char type;			/* 'C' for data, 'T' for funcs */
  item_ptr item;
  struct l_e * next;
} list_elt, * list_elt_ptr;

extern char * si_Program;       /* holds executable name for time-stamp */
char * si_DynamicAddress;	/* address of space malloced for dyn. code */
static char * basefile = NULL; /* symbol table basis file for linking */
static char basefilename[128];

static RestoreCodeUnit(), make_binding_file();

list_elt_ptr si_FreeCodeUnit(ind)
     int ind;

{				/* save old pointers in case of failure */
  NameDesc nte;
  int index;
  list_elt_ptr flist, *eptr;
  MappingTable mte, *mtp;

  if (ind > 0 && FindName((IndexToItem(ind,&mte)->name),&nte) != NULL &&
      nte.size >= 0 && nte.index != nte.size)
    {				/* first item is not code unit */
      AddItemToChain(nte.size,nte.index); /* so add the code unit to front */
      ind = nte.size;
    }
  flist = (list_elt_ptr) si_malloc (sizeof(list_elt));
  eptr = &(flist);
  for (index = ind;
       index >= 0;
       index = mtp->next)
    {				/* save copy of item table entries */
      mtp = IndexToItem(index,&mte);
      (*eptr) = (list_elt_ptr) si_malloc (sizeof(list_elt));
      (*eptr)->name = (char *) si_malloc (strlen(mtp->name)+1);
      strcpy((*eptr)->name,mtp->name);
      (*eptr)->item.intval = mtp->item.intval;
      if (!FindName(mtp->name,&nte))
	  LOGfprintf(stderr, "si_FreeCodeUnit(%d): FindName(%s) failed!\n",
		ind, mtp->name);
      (*eptr)->type = (char) nte.type;
      eptr = &((*eptr)->next);
    }
  (*eptr) = NULL;		/* null terminated list */
  DeleteItemChain(ind);		/* free up item table entries */
  return(flist);
}

int si_Dynamic_Link(objfile,add,libs,verbose)
     char * objfile, * libs;
     int add,verbose;

{
  char faslfile[64],faslfile2[64],tmpname2[64],tmpname3[64],bndfile[64];
  FILE *fp;
  char tempfilename[32];
  char tempfilename2[32];
  char command[512];
  char * bindaddr;
  int maptableindex,count,codesize, bindsize;
  NameDesc nte;
  list_elt_ptr  flist;
  struct exec header;
  struct nlist nl[2];

  if (add)
    {
      LOGfprintf(stderr,"Incorporation into base code not implemented\n");
      return -1;
    }
  (void) sprintf(tempfilename, "/tmp/fasltemp%d.cd", getpid());
  (void) sprintf(tempfilename2, "/tmp/fasl2temp%d.cd", getpid());
  (void) sprintf(faslfile, "%s.o", objfile); /* user's object file */
  (void) sprintf(faslfile2,"/tmp/fasltemp%d.o", getpid()); /* after link with libs */
  (void) sprintf(tmpname2, "/tmp/fbndtemp%d.nm", getpid());
  (void) sprintf(tmpname3, "/tmp/fbndtemp%d.c", getpid());
  (void) sprintf(bndfile, "/tmp/fbndtemp%d.o", getpid());
  if (basefile == NULL)
    {
      basefile=si_Program;	/* initial symbol table basis this prog */
      (void) sprintf(basefilename,"/tmp/%dbasefile",getpid());	/* after incorp. */
    }
  if (!(FindName(faslfile,&nte) == NULL || nte.type == STRING_SYM ||
	nte.type == CODE_SYM))
    {
      LOGfprintf(stderr,"Name `%s' already in use, rename and recompile your file\n",
	     faslfile);
      return(-1);
    }
				/* GET ITEM NAMES, ITEM CODE SIZE */
				/* first command (load) resolves references
				   to library functions.  Second command (nm)
				   gets the names of all the global funcs,
				   including the library funcs pulled in.
				   Third command incrementally loads user
				   code into a junk file using the current
				   program as basis; this is just so we can
				   open the junk file and find out how much
				   code space to malloc when we really load.
				 */

  (void) sprintf(command,
	  "%s -g %s > %s; %s -d -N -x -A %s %s %s -o %s %s -L%s/lib %s -lm -lc",
	  NM,			/* name extractor to use */
	  faslfile,tmpname2,    /* use nm to get names from fasl to tmp */
	  LD,                   /* linker to use */
	  basefile,		/* program file as symbol table basis*/
	  TEXT_ADDR,		/* default address to link at */
	  faslfile,		/* code to load */
	  tempfilename,		/* file to load into */
	  libs,			/* user supplied switches */
	  SIMINC,		/* simulator library directory */
	  SIMLIB);		/* simulator library names */
  LOGfprintf(stderr,"Finding functions and variables in %s...\n",faslfile);
  if (verbose == TRUE)
    LOGfprintf(stderr,"%s\n",command);
  if (system(command) != 0)	/* load just to get text and data sizes */
    {
      LOGfprintf(stderr,"Name extraction or linkage editor failed.\n");
      goto getout;
    }
  /* MAKE BINDING.c FILE */
  if ((count = make_binding_file(tmpname2,tmpname3,faslfile)) < 0)
    {				/* make binding .c file */
      LOGfprintf(stderr,"unable to make binding file\n");
      goto getout;
    }

				/* GET CODE AND DATA SIZE, MALLOC SPACE
				   for user code */

  if ((fp = fopen(tempfilename, "r")) == NULL)
    {
      LOGfprintf(stderr,"Unable to open %s\n",tempfilename);
      goto getout;
    }
  fread((char *) &header, sizeof(header), 1, fp);
  codesize = header.a_text+header.a_data+header.a_bss;
  si_DynamicAddress = (char *) valloc (codesize);
  fclose(fp);
				/* RESOLVE REFERENCES */
				/* incrementally load user code into a temp
				   file using the current program file as 
				   the basis for resolving references.  Load
				   at the address just malloced.  Resolve all
				   library references.
				*/

				/* compile the file of code created to add the
				   user functions and variables to the item
				   table.  Load it into a junk file at a junk
				   address, so we can open the junk file and
				   find out how much space to malloc.
				 */
  (void) sprintf(command,
	  "%s -d -N -x -A %s -T %x %s -o %s %s -L%s/lib %s -lm -lc; xyz=`pwd`; cd /tmp; %s -c -I%s/include %s; cd $xyz; %s -d -N -x -A %s %s %s -o %s -lm -lc",
	  LD,			/* linker to use */
	  basefile,		/* program file as symbol table basis*/
	  si_DynamicAddress,	/* address to load into */
	  faslfile,		/* code file to load */
	  tempfilename,		/* file to load into */
	  libs,			/* user supplied switches */
	  SIMINC,		/* simulator library directory */
	  SIMLIB,		/* simulator library names */
				/* next are for compile and load */
	  CC,			/* compiler to use */
	  SIMINC,		/* simulator include directory */
	  tmpname3,		/* compile tmpname3 file into bndfile */
	  LD,			/* linker to use */
	  tempfilename,		/* file with user code loaded as ref basis */
	  TEXT_ADDR,		/* default address to link at */
	  bndfile,		/* load binding .o file */
	  tempfilename2);	/* resulting code file */
  LOGfprintf(stderr,"Loading %d bytes at address 0x%x...\n",
		codesize, si_DynamicAddress);
  if (verbose == TRUE)
    LOGfprintf(stderr,"%s\n",command);
  if (system(command) != 0)	/* this is the real load */
    {
      LOGfprintf(stderr,"The linkage editor or compiler failed.\n");
      free(si_DynamicAddress);
      goto getout;
    }
				/* READ IN CODE */
  if ((fp = fopen(tempfilename, "r")) == NULL)
    {
      LOGfprintf(stderr,"Unable to open %s\n",tempfilename);
      free(si_DynamicAddress);
      goto getout;
    }
  fread((char *)&header, sizeof(header), 1, fp);
  if (codesize != header.a_text+header.a_data+header.a_bss)
    {
      LOGfprintf(stdout,"? differing user code sizes\n");
      codesize = header.a_text+header.a_data+header.a_bss+4;
      LOGfprintf(stdout,"Loading %d bytes of user code and data\n",codesize);
    }
#ifdef mips
  fseek(fp, N_TXTOFF(header.ex_f, header.ex_o), 0); /* seek to text start */
#endif
  fread(si_DynamicAddress, codesize, 1, fp);
  fclose(fp);

				/* GET CODE AND DATA SIZE, MALLOC SPACE
				   for binding code */
  if ((fp = fopen(tempfilename2, "r")) == NULL)
    {
      LOGfprintf(stderr,"Unable to open %s\n",tempfilename2);
      free(si_DynamicAddress);
      goto getout;
    }
  fread((char*)&header, sizeof(header), 1, fp);
  fclose(fp);
  bindsize = header.a_text+header.a_data+header.a_bss;
  bindaddr = (char *) valloc (bindsize);

				/* LOAD AND RESOLVE BINDING CODE*/
				/* Load it into a temporary file
				   using the previous temporary file (which
				   contains the current program and the
				   new user code) as basis for resolving
				   references.  Load at the address just
				   malloced.
				 */
  (void) sprintf(command,
	  "%s -d -N -x -A %s -T %x %s -o %s -lm -lc",
	  LD,			/* linker to use */
	  tempfilename,		/* file with user code loaded as ref basis */
	  bindaddr,		/* where to load into */
	  bndfile,		/* load binding .o file */
	  tempfilename2);	/* resulting code file */
  LOGfprintf(stderr,"Entering %d functions and variables in item table...\n",
	     count);
  if (verbose == TRUE)
    LOGfprintf(stderr,"%s\n",command);
  if (system(command) != 0)
    {
      LOGfprintf(stderr,"The compiler or linkage editor failed.\n");
      free(si_DynamicAddress);
      free(bindaddr);
      goto getout;
    }

#ifdef mips
  nl[0].n_name = "_si_dynamic_bind"; /*  name of binding function */
  nl[1].n_name = NULL;
#else
  nl[0].n_un.n_name = "_si_dynamic_bind"; /*  name of binding function */
  nl[1].n_un.n_name = NULL;
#endif
  if (nlist(tempfilename2,nl) != 0)
    {
      LOGfprintf(stderr,"Unable to find binding function\n");
      free(si_DynamicAddress);
      free(bindaddr);
      goto getout;
    }
				/* READ IN BINDING CODE */
  if ((fp = fopen(tempfilename2, "r")) == NULL)
    {
      LOGfprintf(stderr,"Unable to open %s\n",tempfilename2);
      free(si_DynamicAddress);
      free(bindaddr);
      goto getout;
    }
  fread((char*)&header, sizeof(header), 1, fp);
  if (bindsize != header.a_text+header.a_data+header.a_bss)
    {
      LOGfprintf(stdout,"? differing binding code sizes\n");
      bindsize = header.a_text+header.a_data+header.a_bss+4;
      LOGfprintf(stdout,"Loading %d bytes of binding code and data\n",bindsize);
    }
#ifdef mips
  fseek(fp, N_TXTOFF(header.ex_f, header.ex_o), 0); /* seek to text start */
#endif
  fread(bindaddr, bindsize, 1, fp);
  fclose(fp);

  if (FindName(faslfile,&nte) != NULL && nte.type == CODE_SYM)
    flist = si_FreeCodeUnit(nte.index);	/* FREE UP OLD ITEM TABLE ENTRIES */
  else
    flist = NULL;
  maptableindex = (* (int(*)())nl[0].n_value)(flist);
  /*maptableindex = ((func_ptr) header.a_entry)(flist);*/
				/* EXECUTE BINDING CODE */
				/* just one function in the binding code
				   file, so the entry point will be for
				   that function.  It calls AddItemToTable
				   repeatedly to put the items in the table.
				   The parameter flist is a list of the
				   old items, so unit function pointers can
				   be changed.
				*/
  free(bindaddr);		/* free up code space used to bind functions */
  if (maptableindex < 0)	/* binding failed */
    RestoreCodeUnit(flist,maptableindex);
  else
    si_RemoveCodeUnit(flist);	/* remove previous code items */
  if (add == TRUE && maptableindex >= 0) /* permanently incorporate in symbol table */
    {
      (void) sprintf(command,"mv %s %s; rm -f /tmp/f*%d.*", /* save basis file */
	      tempfilename,basefilename,getpid()); /* delete others */
      if (verbose == TRUE)
	LOGfprintf(stdout,"%s\n",command);
      system(command);
      basefile= basefilename;
    }
  else 
    {
      (void) sprintf(command,"rm -f /tmp/f*%d.*",getpid());
      if (verbose == TRUE)
	LOGfprintf(stdout,"%s\n",command); /* delete files */
      system(command);
    }
  return(maptableindex);

 getout:
  (void) sprintf(command,"rm -f /tmp/f*%d.*",getpid());
  if (verbose == TRUE)
    LOGfprintf(stdout,"%s\n",command); /* delete files */
  system(command);
  return(-1);
}

static RestoreCodeUnit(flist,maptableindex)
     list_elt_ptr flist;
     int maptableindex;

{
  list_elt_ptr *eptr,*tptr, tmpflist;
  int i;

  LOGfprintf(stderr,"Mapping failed, restoring previous state\n");
  free(si_DynamicAddress);	/* free code space */
  tmpflist = si_FreeCodeUnit(0-maptableindex); /* some unit func pointers */
  if (flist != NULL)		/* may have changed */
    {				/* restore previous pointers */
      for (eptr = &(flist->next), i = -1; /* first is code unit item */
	   (*eptr) != NULL; /* destroy as we go */
	   tptr = &((*eptr)->next), free ((*eptr)->name),
	   free((char*)(*eptr)), eptr = tptr)
	i = AddItemToTable((*eptr)->name,(*eptr)->item.intval,
			      (*eptr)->type,i,tmpflist);
      AddItemToTable(flist->name, /* restore old code unit */
			flist->item.intval,
			flist->type,i,NULL);
      free(flist->name);
      free((char*)flist);
    }
  for (eptr = &tmpflist;	/* destroy list created above */
       (*eptr) != NULL;		/* destroy as we go */
       tptr = &((*eptr)->next), free ((*eptr)->name),
       free((char*)(*eptr)), eptr = tptr);

}

si_RemoveCodeUnit(flist)
     list_elt_ptr flist;

{
  list_elt_ptr *eptr,*tptr;

  if (flist != NULL)
    {
      for (eptr = &(flist->next); /* first is code unit item */
	   (*eptr) != NULL;	/* destroy as we go */
	   tptr = &((*eptr)->next), free ((*eptr)->name),
	   free((char*)(*eptr)), eptr = tptr);
      free(flist->item.fileinfo);	/* free up old code space */
      free(flist->name);
      free((char*)flist);
    }
}

/* |||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
 *   The following are format strings for output.  The different sections
 *   have nothing special about them, they are sectioned the way they
 *   are for programming convenience
 * \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\///////////////////////////////////// */
	char	*sec_3 = "si_dynamic_bind(flist)\n   char * flist;\n\n{\nint curindex = -1, codeindx;\nextern char * si_DynamicAddress;";
	char	*sec_4 = "    if ((curindex = AddItemToTable(\"%s\",%s,%d,curindex,flist,codeindx)) < 0)\n      goto getout;\n";
	char	*sec_5 = "  getout:  return(curindex);\n}\n";
	char	*sec_6 = "    if ((curindex = AddItemToTable(\"%s\",&%s,%d,curindex,flist,codeindx)) < 0)\n      goto getout;\n";
        char	*sec_7 = "    if ((codeindx = curindex = AddItemToTable(\"%s\",%s,%d,-1,flist,-1)) < 0)\n      goto getout;\n    else curindex = -1;\n";
        char    *sec_8 = "    AddItemToChain(codeindx,curindex);\n    return(codeindx);\n";
/* //////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\ */

static make_binding_file(nmfname,bindfname,codefilename)
     char * nmfname;
     char * bindfname;
     char * codefilename;

{
  int c, size;
  char buf[120];
  FILE * nm_fp, * bind_fp;
  int cnt = 0;
  list_elt_ptr flist, * eptr, * tptr;
  
  flist = (list_elt_ptr) si_malloc (sizeof(list_elt));
  eptr = &(flist);
  if ((nm_fp = fopen(nmfname,"r")) == NULL)
    {
      LOGfprintf(stderr,"unable to open %s",nmfname);
      return(-1);
    }
  bind_fp = fopen(bindfname, "w");
  c = getc(nm_fp);
  while (c != EOF)
    {
      if (isxdigit(c))
	{
	  ungetc(c,nm_fp);
	  fscanf(nm_fp,"%x",&size);
	  getc(nm_fp); c = getc(nm_fp);	/* get type of name */
	  fscanf(nm_fp,"%s",buf);	/* this is the name */
	  if ((size == 4 && c == 'C') || c == 'T')
	    {
	      (*eptr) = (list_elt_ptr) si_malloc (sizeof(list_elt));
	      (*eptr)->name = (char *) si_malloc (strlen(buf));
	      (*eptr)->type = (char) c;
	      strcpy((*eptr)->name,buf+NAME_OFFSET);
	      eptr = &((*eptr)->next);
	      fprintf(bind_fp, "extern int %s", buf+NAME_OFFSET);
	      if (c == 'C')
		fprintf(bind_fp,";\n");
	      else
		fprintf(bind_fp,"();\n");
	      cnt++;
	    }
	}
      for(c = getc(nm_fp);c != '\n' && c != EOF;c = getc(nm_fp));
      if (c != EOF)
	c = getc(nm_fp);	/* get first char on new line */
    }
  fclose(nm_fp);
  (*eptr) = NULL;		/* NULL terminated list */
  fprintf(bind_fp,"%s",sec_3);
  fprintf(bind_fp, sec_7, codefilename, "si_DynamicAddress", CODE_SYM);
  for (eptr = &(flist); (*eptr) != NULL; /* destroy as we go */
       tptr = &((*eptr)->next), free ((*eptr)->name),
       free((char*)(*eptr)), eptr = tptr)
    if ((*eptr)->type == 'T')	/* function item*/
      fprintf(bind_fp, sec_4, (*eptr)->name,(*eptr)->name, FUNC_SYM);
    else			/* data item */
      fprintf(bind_fp, sec_6, (*eptr)->name,(*eptr)->name, DATA_SYM);
  fprintf(bind_fp, "%s", sec_8);
  fprintf(bind_fp, sec_5);
  fclose(bind_fp);
  return(cnt);
}

/*---------------------------------------------------------------------------
  Change unit, site and link function pointers.  <name> is the function name,
  <new> is the new pointer to the function.
----------------------------------------------------------------------------*/

static func_ptr findfunc(flist,name)
     list_elt_ptr flist;
     char * name;

{
  register list_elt_ptr lptr;

  for (lptr = flist; lptr != NULL; lptr = lptr->next)
    if (lptr->type == (char) FUNC_SYM && !(strcmp(lptr->name,name)))
      return (lptr->item.func);
  return NULL;
}

#define NOT_FOUND 1
#define UNIT_FUNC 2
#define SITE_FUNC 3

si_ChangeFuncPointers(name, new, flist)
     char * name;
     func_ptr new;
     list_elt_ptr flist;

{
  register Unit * up;
  register Site * sp;
  register Link * lp;
  register int count, ftype;
  func_ptr old;
  
  if ((old = findfunc(flist,name)) == NULL)/* get previous pointer if exists */
    if ((old = NameToFunc(name)) == NULL) /* maybe in different code file */
      return;			/* if not being replaced, no worries */
  for (count = 0, up = &UnitList[0], ftype = NOT_FOUND;
       ftype == NOT_FOUND && count < NoUnits;	/* unit,site,link functions */
       count++, up++)
    if (up->unit_f == old)		/* first unit function */
      ftype = UNIT_FUNC;
    else
      {
	sp = up->sites;
	while (sp != NULL && ftype == NOT_FOUND)
        if (sp->site_f == old)		/* first site function */
          ftype = SITE_FUNC;
        else
          sp = sp->next;
      }

  switch (ftype)
    {
    case UNIT_FUNC:
      for (up--, count--; count < NoUnits; count++, up++)
        if (up->unit_f == old)
	  {
	    up->unit_f = new;
	    ftype = 0 - UNIT_FUNC;
	  }
      break;
    case SITE_FUNC:
      for (up--, count--; count < NoUnits; count++, up++)
        if (up->sites != NULL)
          for (sp = up->sites; sp != NULL; sp = sp->next)
            if (sp->site_f == old)
	      {
		sp->site_f = new;
		ftype = 0 - SITE_FUNC;
	      }
      break;
    case NOT_FOUND:			/* link function or nothing */
      for (count = 0, up = &UnitList[0]; count < NoUnits; count++, up++)
        if (up->sites != NULL)
          for (sp = up->sites; sp != NULL; sp = sp->next)
            if (sp->inputs != NULL)
              for (lp = sp->inputs; lp != NULL; lp = lp->next)
                if (lp->link_f == old)
		  {
		    lp->link_f = new;
		    ftype = 0 - NOT_FOUND;
		  }
    }

  switch (ftype)
    {
    case 0 - UNIT_FUNC:
      LOGfprintf(stderr,"Reset unit function pointers");
      break;
    case 0 - SITE_FUNC:
      LOGfprintf(stderr,"Reset site function pointers");
      break;
    case 0 - NOT_FOUND:
      LOGfprintf(stderr,"Reset link function pointers");
    }
  
  if (ftype < 0)
    if (new == (func_ptr) NullFunc)
      LOGfprintf(stderr," for %s to null function\n",name);
    else
      LOGfprintf(stderr," for %s to new definition\n",name);
}

list_elt_ptr si_NullFuncPointers(flist)
     list_elt_ptr flist;

{
  register list_elt_ptr tp;

  for (tp = flist; tp != NULL; tp = tp->next)
    if (tp->type == FUNC_SYM)
      si_ChangeFuncPointers(tp->name,NullFunc,flist);
  return flist;
}
