/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*----------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: May 1 1987
----------------------------------------------------------------------------*/

/* The following commands run on the butterfly control node and the
   uniprocessor versions.  For simulation, dynamic activity.
   Read the help User Manual for specification of each command.
   Read the help information at the end of each function for a description
*/

#ifdef BFLY
#  include "bflycontrol.h"
#else
#  include "uniproc.h"
#  include <sys/time.h>
#  include <sys/types.h>
extern time_t time();
#endif
#include <ctype.h>

extern char *strcat();

#include "lex.h"

extern char *NameToType();

extern int si_Running;

#ifndef BFLY
Cmd_return(argc,argv)
     int argc;
     char ** argv;

{
  func_ptr func;

  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc != 1)
    goto synerror;
  if ((func = NameToFunc("gi_return_to_caller")) != NULL)
    func();
  else
    si_Running = FALSE;
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command returns to whatever invoked the simulator, normally UNIX.  If the simulator was called by some other program, control is returned to that program.\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: return\n\n");
  return 0;
}

Cmd_compile(argc,argv)
     int argc;
     char ** argv;

{
  char tmpname1[64], cmd[256], flags1[32], flags2[32];

#ifdef FSIM
  (void) sprintf(flags1,"%s","-DFSIM");
#else
  flags1[0] = '\0';		/* null string */
#endif
#ifdef TSIM
  (void) sprintf(flags2,"%s","-DTSIM");
#else
  flags2[0] = '\0';		/* null string */
#endif
  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc < 2 || argc > 3)
    goto synerror;
  (void) sprintf(tmpname1, "%s.c", argv[1]);
  if (argc == 2)
    (void) sprintf(cmd,"%s -c %s %s -I%s/include %s",
	    CC,			/* compiler to use*/
	    flags1,flags2,
	    SIMINC,tmpname1);
  else
    (void) sprintf(cmd,"%s -c %s %s %s -I%s/include %s",
	    CC,			/* compiler to use*/
	    flags1,flags2,
	    argv[2],SIMINC,tmpname1);
  LOGfprintf(stderr,"%s\n",cmd);
  if (system(cmd)!= 0)	/* compile user file */
    LOGfprintf(stderr,"The compilation failed.\n");
  else
    LOGfprintf(stderr,"The compilation succeeded.\n");
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command compiles a file of C code.  The filename should be given without the .c extension (so to compile `myfuncs.c' use the command: `compile myfuncs').  The resulting object file will be named `filename.o'.\n");
  LOGfprintf(Dispf,"You may give your own switches to the compilation in a string as the third parameter.  For example, if you want to specify an include directory, you might use: `compile myfile \"-I~/mydir -O\"'\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: compile <filename> [\"<switches>\"]\n\n");
  return 0;
}

Cmd_loadcode(argc,argv)
     int argc;
     char ** argv;

{
  char * libs;
  int verbose;

  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc < 2 || argc > 4)
    goto synerror;
  if (argc > 2)
    switch (Lex(argv[2]))
      {
      case ALL:
	libs = "";
	verbose = TRUE;
	break;
      default:
	libs = argv[2];
	if (argc > 3)
	  if (Lex(argv[3]) == ALL)
	    verbose = TRUE;
	  else
	    goto synerror;
	else
	  verbose = FALSE;
      }
  else
    {
      libs = "";
      verbose = FALSE;
    }
  if (si_Dynamic_Link(argv[1], FALSE,libs,verbose) < 0)
    {				/* link in user code, not permanently */
      LOGfprintf(stderr,"dynamic linking failed\n");
      return(0);
    }
  LOGfprintf(stderr,"%s.o linked successfully\n",argv[1]);
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command dynamically loads an object file (.o) into the running system.  The filename should be given without the .o extension.  The command is intended for loading of user functions, in particular unit, site and link functions, and parameters.  Libraries to be searched may be given as a third parameter, in the usual format for the UNIX `ld' command.  See below for examples.\n");
  LOGfprintf(Dispf,"The unit of code is an object file, NOT a function.  Thus each code unit is known by the name of the object file (.o).  Loading a code unit will cause any previously loaded code unit with the same name to be destroyed.  Thus functions and data loaded with this command may be redefined.\n");
  LOGfprintf(Dispf,"References in the new code unit to functions in a code unit loaded or reloaded previously CANNOT be resolved.  Therefore all code units must be completely self contained except for references to functions which are part of the simulator or which were loaded when the simulation executable was made (i.e. which were loaded by the `makesim' UNIX level command).\n");
  LOGfprintf(Dispf,"All unit, site and link functions are updated to the latest version of their functions.  All global functions and 4 byte UNINITIALISED variables are put in the item table, so you can `call' the functions and modify the variables (see `value' command).  Generally integers and floats are 4 byte variables.\n");
  LOGfprintf(Dispf,"You may specify library directories and libraries to be searched as a third parameter, a string.  The -L and -l switches to `ld' may be given in this string (or in fact any other switches).  For example, you might issue the command `loadcode mycode \"-L~/lib -lneuro\"'.\n");
  LOGfprintf(Dispf,"To see the shell commands issued to load your code, include the keyword `all' at the end of your command, i.e. after the codefile or the switches if you used any.\n");
  LOGfprintf(Dispf,"Example: you have an empty simulator, a file of unit functions (`func.c') and a file of network building functions (`build.c').  Compile these files using the `compile' command.  The network building code makes reference to the unit functions, so they must be combined with the `combine' command before they can be loaded.  Combine them into file `comb.o' (use `combine build func - comb').  Load `comb.o' into the simulator - `loadcode comb'.\n\n");
  LOGfprintf(Dispf,"Now you can call your network building function(s).  If after experimenting you need to change your unit functions, simply edit `funcs.c', recompile (`compile funcs') and reload (`loadcode funcs').  The new unit functions will be installed.  If you want to rebuild the network, or add a piece of network, the unit functions must again be combined with the network building code before loading.\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: loadcode <filename> [<\"libraries>\"] [<all>]\n\n");
  return 0;
}

Cmd_combine(argc,argv)
     int argc;
     char ** argv;

{
  int curpos = 0;
  char cmd[512], command[512];
  char * flags;

  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc < 5)
    goto synerror;

  Curarg = 1;
  while (argc > Curarg && Lex(argv[Curarg]) != MINUS)
    {
      (void) sprintf(cmd+curpos,"%s.o ", argv[Curarg-1]);
      curpos += strlen(argv[Curarg-1])+3;
    }
  if (Curarg + 1 != argc)	/* last is flags */
    if (Curarg+2 != argc)	/* syntax error */
      goto synerror;
    else
      flags = argv[Curarg+1];
  else
    flags = "";
  if (strlen(flags) > 100)
    {
      LOGfprintf(stderr,"Maximum 100 characters in flags string, ignoring command\n");
      return 0;
    }
  (void) sprintf(cmd+curpos,"-o %s.o",argv[Curarg]); /* output file */
  (void) sprintf(command,"ld -r %s ",flags);
  (void) sprintf(command+strlen(flags)+7,"%s",cmd);
  LOGfprintf(stderr,"%s\n",command);
  if (system(command)!= 0)	/* compile user file */
    LOGfprintf(stderr,"File linking failed.\n");
  else
    LOGfprintf(stderr,"File linking succeeded.\n");
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command combines two or more object (.o) files into a single object file, so as to be able to use the `loadcode' command with network construction and activation functions in different files.  Switches may be provide to the shell `ld' command as an optional last parameter.\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: combine <file> <file> [<file>]* - <dest file> [<\"switches>\"]\n\n");
  return 0;

}   

/******************* DESIGN ISSUES NOT RESOLVED ************************
Cmd_incorporate(argc,argv)
     int argc;
     char ** argv;

{
  char * libs;
  int verbose;

  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc < 2 || argc > 4)
    goto synerror;
  if (argc > 2)
    switch (Lex(argv[2]))
      {
      case ALL:
	libs = "";
	verbose = TRUE;
	break;
      default:
	libs = argv[2];
	if (argc > 3)
	  if (Lex(argv[3]) == ALL)
	    verbose = TRUE;
	  else
	    goto synerror;
	else
	  verbose = FALSE;
      }
  else
    {
      libs = "";
      verbose = FALSE;
    }
  if (si_Dynamic_Link(argv[1], TRUE,libs,verbose) < 0)
    {		
      LOGfprintf(stderr,"dynamic linking failed\n");
      return(0);
    }
  LOGfprintf(stderr,"%s.o linked successfully\n",argv[1]);
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command is similar to the `loadcode' command, except that a code unit loaded with this command becomes incorporated into the base simulator.  This means that other code units loaded later may make references to functions and variables in the code unit(s) loaded with this command.\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: incorporate <filename> [<\"libraries>\"] [<all>]\n\n");
  return 0;
}
*********************************************************************/

Cmd_value(argc,argv)
     int argc;
     char ** argv;

{
  NameDesc nte;
  MappingTable mte;
  int tok;

  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc < 2)
    goto synerror;
  if (FindName(argv[1],&nte) == NULL || nte.type != DATA_SYM)
    {
      LOGfprintf(stderr,"Second item `%s' not the name of a data item\n",argv[1]);
      goto synerror;
    }
  if (IndexToItem(nte.index,&mte) == NULL)
    {
      LOGfprintf(stderr,"Internal error: Name table corrupted. \n");
      return 0;
    }
  if ((tok = Lex(argv[2])) == INT)
    *(mte.item.intval) = Yyint;
  else
    if (tok == FLOAT)
      *(mte.item.floatval) = Yyfloat;
    else
      if (tok == HELP)		/* show value */
	if (argc > 3)		/* float value */
	  LOGfprintf(stdout,"Value of %s is %f\n",argv[1],
		     *(mte.item.floatval));
	else			/* integer value */
	  LOGfprintf(stdout,"Value of %s is %d\n",argv[1],
		     *(mte.item.intval));
      else
        {			/* syntax error */
	  LOGfprintf(stderr,"Third item `%s' not an integer, floating point number or `?'.\n",argv[2]);
	  goto synerror;
	}
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command shows or sets the value of a C global variable in a section of code loaded with the `loadcode' commands.  Note that only 4 byte UNINITIALISED variables are known, i.e. integers and floats.  `value x 3' or `value x 3.0' will set x to 3 or 3.0 depending on whether it is a floating point or integer parameter.  You MUST use the correct form, or the value will be set incorrectly.\n\n");
  LOGfprintf(Dispf,"`value x ?' prints the value of x as an integer, `value x ? float' prints the value as a floating point number.  Type conversion is NOT automatic, the default is integer and you must indicate floating point explicitly if required.  Any variables which have an initialiser will not be known (see the C manual for a definition of `initialiser').\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: value <data name> <value | ? [float]>\n\n");
  return 0;
}

Cmd_whatis(argc,argv)
     int argc;
     char ** argv;

{
  NameDesc nte;

  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc != 2)
    goto synerror;
  
  if (FindName(argv[1],&nte) == NULL)
    LOGfprintf(stderr,"%s is not in the name table\n",argv[1]);
  else
    LOGfprintf(stderr,"%s is a %s\n",argv[1],NameToType(argv[1]));
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command prints a message as to what kind of object (unit, site, function, variable, etc) the simulator associates with the given name\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: whatis name\n\n");
  return 0;
}
  
Cmd_whereis(argc,argv)
     int argc;
     char ** argv;

{
  NameDesc nte;
  MappingTable mte;

  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc != 2)
    goto synerror;
  
  if (FindName(argv[1],&nte) == NULL)
    LOGfprintf(stderr,"%s is not in the name table\n",argv[1]);
  else
    if (nte.type != FUNC_SYM && nte.type != DATA_SYM)
      LOGfprintf(stderr,"%s is not a function or variable\n",argv[1]);
    else
      if (nte.size < 0)
	LOGfprintf(stderr,"%s is defined in the simulator executable %s\n",
		   argv[1], si_Program);
      else
	LOGfprintf(stderr,"%s is defined in code unit %s\n",
		   argv[1], IndexToItem(nte.size,&mte)->name);
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command prints the name of the code unit where the given function or variable is defined currently\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: whereis <function-name | variable-name>\n\n");
  return 0;
}

Cmd_deletecode(argc,argv)
     int argc;
     char ** argv;

{
  NameDesc nte;
  char objectname[128];

  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc != 2)
    goto synerror;
  (void) sprintf(objectname,"%s.o",argv[1]);
  if (FindName(objectname,&nte) == NULL)
    LOGfprintf(stderr,"%s not in name table\n",objectname);
  else
    if (nte.type != CODE_SYM)
      LOGfprintf(stderr,"%s is not a code unit, is a %s\n",
		 objectname,NameToType(objectname));
    else
      si_RemoveCodeUnit(si_NullFuncPointers(si_FreeCodeUnit(nte.index)));
  return 0;
  
 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command deletes a code unit, removing all functions and variables defined in that code unit.  If any of the functions are in use as unit, site or link functions, the unit, site and link pointers are set to point to the null function (NullFunc) instead.\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: abort\n\n");
  return 0;
}
  
Cmd_abort(argc,argv)
     int argc;
     char ** argv;

{
  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc != 1)
    goto synerror;
  abort();
  
 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command closes all files and then signals IOT, usually resulting in termination with a core dump to file `core', which  may  be used for debugging.\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: abort\n\n");
  return 0;
}
  
Cmd_AllocateUnits(argc,argv)
     int argc;
     char ** argv;

{
  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc != 2 || Lex(argv[1]) != INT)
    goto synerror;
  AllocateUnits(Yyint);
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"AllocateUnits is used to create data space for the units.  The parameter is the number of units which are to be created.  Auto-allocation occurs if not enough units are availible.  If Auto-allocation has to be done, then the same number of units as were created at the last call to AllocateUnits are created.\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: AllocateUnits <number>\n\n");
  return 0;
}
#endif

Cmd_NameUnit(argc,argv)
     int argc;
     char ** argv;

{
  int index, width = 1, depth = 0, type, lx;

  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc < 4 || argc > 6)
    goto synerror;
  switch (argc)
    {
    case 6:
      if (Lex(argv[5]) != INT)
	goto synerror;
      else
	depth = Yyint;
    case 5:
      if (Lex(argv[4]) != INT)
	goto synerror;
      else
	width = Yyint;
    case 4:
      if (Lex(argv[3]) != INT)
	goto synerror;
      else
	index = Yyint;
    }
  if ((lx = Lex(argv[2])) != SCAL && lx != VECT && lx != ARR)
    goto synerror;
  else
    if (lx == SCAL)
      type = SCALAR;
    else
      if (lx == VECT)
	type = VECTOR;
      else
	type = ARRAY;
  NameUnit(argv[1],type,index,width,depth);
  return 0;
  
 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"NameUnit is used to give a name to one or more units.  It takes the same arguments as the simulator function of the same name, i.e. a <name>, a <type>, the <index> of the first unit to which the <name> is to be applied, the <width> (if a vector or array), and <depth> if an array.  The possible types are: scalar, vector, array.  For a scalar name, the name is applied to a single unit.  For a vector name, the name is applied to <width> units starting with <index>.  For an array name, the name is applied to <width*depth> units starting with <index>.\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: NameUnit <name> <scalar|vector|array> <index> [<width> [<depth]]\n\n");
  return 0;
}
     
  
Cmd_MakeUnit(argc,argv)
     int argc;
     char ** argv;

{
  data_type data;
  int istate,state;
  pot_type ipot, pot;
  Output out;
  char * type;
  func_ptr func;
  int i,lx;

BFLYCATCH
  ipot = pot = data = out = istate = state = 0;
  type = "NullType";
  func = (func_ptr)NullFunc;
  switch (argc)
    {
    case 9:
      if (Lex(argv[8]) != INT)
	goto synerror;
      state = Yyint;
    case 8:
      if (Lex(argv[7]) != INT)
	goto synerror;
      istate = Yyint;
    case 7:
      if ((lx = Lex(argv[6])) == INT)
	out = Yyint;
      else
	if (lx == FLOAT)
	  out = Yyfloat;
	else
	  goto synerror;
    case 6:
      if ((lx = Lex(argv[5])) == INT)
        data = Yyint;
      else
	if (lx == FLOAT)
	  data = Yyfloat;
	else
	  goto synerror;
    case 5:
      if ((lx =Lex(argv[4])) == INT)
	pot = Yyint;
      else
	if (lx == FLOAT)
	  pot = Yyfloat;
	else
	  goto synerror;
    case 4:
      if ((lx =Lex(argv[3])) == INT)
	ipot = Yyint;
      else
	if (lx == FLOAT)
	  ipot = Yyfloat;
	else
	  goto synerror;
    case 3:
      if (!strcmp(argv[2],"NULL"))
	func = (func_ptr)NullFunc;
      else
	if ((func = NameToFunc(argv[2])) == NULL)
	  {
	    LOGfprintf(stderr,
		  "Second argument not recognized as a function name: %s\n",
		       argv[2]);
	    return 0;
	  }
    case 2:
      if (Lex(argv[1]) == HELP)
	goto helpinfo;
      type = argv[1];
    case 1:
      break;
    default:
      goto synerror;
    }
#ifdef BFLY			/* butterfly control node */
  i = MakeUnit(type,func,ipot,data,out,istate,state);
#else				/* uniprocessor version */
  i = si_DebugMakeUnit(type,func,ipot,pot,data,out,istate,state);
#endif
  if (i < 0)
    LOGfprintf(stdout,"Unit not made\n");
  else
    LOGfprintf(stdout,"Made unit %d\n",i);
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"MakeUnit is the simulator command to make one unit.  It takes the same arguments as the simulator function to make a unit, namely type, function, initial potential, potential, data, output, initial state, and state, in that order.  The first parameter, type, is simply stored in the name table by the simulator for display purposes.  You can use any name that is not already in use (i.e. not a name of a set, state, function or unit(s)).  The second parameter, function, is the unit function, which must be known to the simulator (i.e. a library function or a function in your files, or NULL for the function which does nothing).\n\n");
  LOGfprintf(Dispf,"The remaining arguments are integer values for the various fields of the unit. Initial potential is the unit potential after a reset.  Potential is the current unit potential.  Data is for you to use as you wish.  Output is the current unit output.  Initial state is the unit state after a reset.  State is the current unit state.\n\n");
  LOGfprintf(Dispf,"Defaults for the numeric arguments are zero. Default for the function is the function NullFunc (does nothing).  Default for type is the name NullType.  You must specify all arguments up to the last one whose value you don't want to default (counting from left to right). Debugging is automatically switched on for the duration of this command.\n\n");

 synerror:
	Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: MakeUnit <type> <function> <ipot> <pot> <data> <out> <istate> <state>\n");
  LOGfprintf(Dispf,"\tmissing arguments will default (numeric ones to 0)\n");
  return 0;
BFLYTHROW("Cmd_MakeUnit",0)
}

Site * si_DebugAddSite();

Cmd_AddSite(argc,argv)
     int argc;
     char ** argv;

{
  int unit, lx;
  data_type data;
  char * name;
  func_ptr func;
  Site * sp;

BFLYCATCH
  data = 0;
  unit = -1;
  name = "NullSite";
  func = (func_ptr)NullFunc;
  switch (argc)
    {
    case 5:
      if ((lx = Lex(argv[4])) == INT)
	data = Yyint;
      else
	if (lx == FLOAT)
	  data == Yyfloat;
	else
	  goto synerror;
    case 4:
      if (!strcmp(argv[3],"NULL"))
	func = (func_ptr)NullFunc;
      else
	if ((func = NameToFunc(argv[3])) == NULL)
	  {
	    LOGfprintf(stderr,
		   "Third argument not recognized as a function name: %s\n",
		       argv[3]);
	    return 0;
	  }
    case 3:
      name = argv[2];
    case 2:
      if (Lex(argv[1]) == HELP)
	goto helpinfo;
      if (Lex(argv[1]) != INT)
	goto synerror;
      unit = Yyint;
      break;
    case 1:
      LOGfprintf(stdout,"You must specify a unit number to attach the site to\n");
    default:
      goto synerror;
    }

#ifdef BFLY			/* butterfly control node */
  sp = AddSite(unit,name,func,data);
#else				/* uniprocessor version */
  sp = si_DebugAddSite(unit,name,func,data);
  if (sp == NULL)
    LOGfprintf(stderr,"Site not made\n");
#endif
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"AddSite is the simulator command to add a site to a unit.  It takes the same arguments as the simulator function to add a site, namely: unit index, site name, site function, and site data.  A site with the given name and function is attached to the unit with the given index, and the data field is set as given.  You can use any name that is not already in use (i.e. not a name of a set, state, function, type or unit(s)).  The site function must be known to the simulator (i.e. a library function or a function in your files or NULL for the function that does nothing).  The unit index must be that of an existing unit.  Data is for you to use as you wish.\n\n");
  LOGfprintf(Dispf,"Defaults exist for site name, site function and data are NullSite, NullFunc and 0 respectively.  You MUST specify the unit to which the site is to be attached, and all arguments up to the last one you wish to specify (counting left to right).  Debugging is automatically switched on for the duration of this command.\n"); 
 synerror:
	Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: AddSite <unit> <sitename> <function> <data>\n");
  LOGfprintf(Dispf,"\tmissing arguments will default (numeric ones to 0)\n");
  return 0;
BFLYTHROW("Cmd_AddSite",0)
}

Link * si_DebugMakeLink();

Cmd_MakeLink(argc,argv)
     int argc;
     char ** argv;

{
  int from, to, lx;
  data_type data;
  weight_type weight;
  char * name;
  func_ptr func;
  Link * ip;

BFLYCATCH
  from = to = weight = data = 0;
  name = "nullsite";
  func = (func_ptr)NullFunc;
  switch (argc)
    {
    case 7:
      if (!strcmp(argv[6],"NULL"))
	func = (func_ptr)NullFunc;
      else
	if ((func = NameToFunc(argv[6])) == NULL)
	  {
	    LOGfprintf(stderr,
		 "Sixth argument not recognized as a function name: %s\n",
		       argv[6]);
	    return 0;
	  }
    case 6:
      if ((lx = Lex(argv[5])) == INT)
	data = Yyint;
      else
	if (lx == FLOAT)
	  data == Yyfloat;
	else
	  goto synerror;
    case 5:
      if ((lx = Lex(argv[4])) == INT)
	weight = Yyint;
      else
	if (lx == FLOAT)
	  weight = Yyfloat;
	else
	  goto synerror;
    case 4:
      name = argv[3];
    case 3:
      if (Lex(argv[2]) != INT)
	goto synerror;
      to = Yyint;
    case 2:
      if (Lex(argv[1]) == HELP)
	goto helpinfo;
      if (Lex(argv[1]) != INT)
	goto synerror;
      from = Yyint;
    case 1:
      break;
    default:
      goto synerror;
    }

#ifdef BFLY			/* butterfly control node */
  ip = MakeLink(from,to,name,weight,data,func);
#else				/* uniprocessor version */
  ip = si_DebugMakeLink(from,to,name,weight,data,func);
  if (ip == NULL)
    LOGfprintf(stderr,"Link not made\n");
#endif
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"MakeLink is the simulator command to make a link from one unit to another. It takes the same arguments as the simulator function to make a link, namely: source unit index, target unit index, site name (on target unit), link weight, link data and link function.  A link from the source unit to the named site on the target unit is made with function, weight and data as given.  The link function must be known to the simulator (i.e. a library function or a function in your files, or NULL for the function which does nothing).\n\n");
  LOGfprintf(Dispf,"The weight you give is scaled down by the simulator by a factor of 1000, so 500 corresponds to a real weight of 0.5 Data is for you to use as you wish.\n\n");
  LOGfprintf(Dispf,"Defaults exist for function (NullFunc), weight (0) and data (0).  You MUST specify the source and target unit indices, and the name of the site on the target unit, and all arguments up to the last one you wish to specify (counting from left to right).  Debugging is automatically switched on for the duration of this command.\n");

 synerror:
	Format = FALSE;
  LOGfprintf(Dispf,
	  "\nUsage: MakeLink <from> <to> <site> <weight> <data> <function>\n");
  fprintf(Dispf,"\tmissing arguments will default (numeric ones to 0)\n");
  return 0;
BFLYTHROW("Cmd_MakeLink",0)
}

#ifndef BFLY
Cmd_deletelinks(argc,argv)
     int argc;
     char ** argv;

{
    char * sitename;
    int ul,uh,us,usind;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;

    if (argc < 4) goto synerror;

    Curarg=1;				  /* first command argument */
    while(argc > Curarg)
      {
	if(!GetUnits(argc,argv)) return 0;
	if (argc <= Curarg) goto synerror;
	ul = Ulow; uh = Uhigh; us = Uset; usind = Usetind;
	if(!GetUnits(argc,argv)) return 0;
	if (argc <= Curarg) goto synerror;
	Lex(argv[Curarg]);
	sitename = Yyarg;
	
#ifdef BCNTL
	Set_Cmd_Numbs(9,ul,uh,us,usind,Uset,Usetind);
	Set_Cmd_Names(1,sitename); /* see weight command for implementation */
	Send_Range_Cmd(DELINK_C,Ulow,Uhigh);
#else
	DeleteLinks(ul,uh,us,usind,Ulow,Uhigh,Uset,Usetind,sitename);
#endif
      }
    return 0;

  helpinfo:
  Format = TRUE;					  /* print detailed help */
    LOGfprintf(Dispf,"The deletelinks command deletes one or more links.  Deleted links are put on the free list and used by the next MakeLink command.  The first unit index is the originating unit.  The second unit index is the receiving unit, and the sitename is the name of the site on the receiving unit to which the link is attached.\n\nMany links may be deleted with one command.  The unit specifications may be given in any of the usual ways (`help UnitId' for details).  The site name may be `all', meaning all sites at the destination unit(s).  To delete all the links, use `deletelinks all all all'.  To delete all links from units in set `source' to site `excite' on units 24 through 48, use `deletelinks source 24 - 48 excite'.\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: deletelinks <From-UnitID> <To-UnitID> <To-site>\n\n");
    return 0;
BFLYTHROW("Cmd_DeleteLink",0)
} /* weight */

Cmd_deletesites(argc,argv)
     int argc;
     char ** argv;

{
    char * sitename;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;

    if (argc < 3) goto synerror;

    Curarg=1;				  /* first command argument */
    while(argc > Curarg)
      {
	if(!GetUnits(argc,argv)) return 0;
	if (argc <= Curarg) goto synerror;
	Lex(argv[Curarg]);
	sitename = Yyarg;

#ifdef BCNTL
	Set_Cmd_Numbs(9,ul,uh,us,usind,Uset,Usetind);
	Set_Cmd_Names(1,sitename); /* see weight command for implementation */
	Send_Range_Cmd(DELSITE_C,Ulow,Uhigh);
#else
	DeleteSites(Ulow,Uhigh,Uset,Usetind,sitename);
#endif
      }
    return 0;

  helpinfo:
  Format = TRUE;		/* print detailed help */
    LOGfprintf(Dispf,"The deletesites command deletes one or more sites.  Delete sites are returned to the free list for use by the next AddSite command.  Links arriving at the site(s) are also deleted.  The unit index is that of the unit to which the site is attached.  The sitename is the name of the site.\n\nMany sites may be deleted with one command.  The unit specifications may be given in any of the usual ways (`help UnitId' for details).  The site name may be `all', meaning all sites on the unit(s).  To delete all the sites on all units, use `deletesites all all'.  To delete all sites named `excite' on units 35 through 70, use `deletesites 35 - 70 excite'.\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: deletesites <UnitID> <sitename>\n\n");
    return 0;
BFLYTHROW("Cmd_deletesites",0)
} /* weight */

Cmd_scavenge(argc,argv)		/* ???????????????? */
     int argc;
     char ** argv;

{
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc > 1)
    goto synerror;

  ScavengeLinks();
  ScavengeNames();
#ifdef TSIM
  ScavengeBuffers();
#endif

  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command attempts to recover unused space.  It can take a while if your network is large.  In the propagation delay version, it will reduce the length of the output buffers to the minimum needed, which may speed up simulation considerably.\n\n");

 synerror:
  Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: scavenge\n\n");

  return 0;
}
#endif

/***** quit *****/

Cmd_quit(argc,argv)		/* ???????????????? */
     int argc;
     char ** argv;

{
BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc > 1)
    goto synerror;
#ifndef BFLY
  if (Logging)
    LogOff();
  if (LogCmd)
    SaveCmdFile();
#endif
  exit(0);

 helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The quit command returns you to UNIX.  If logging is turned on, the log file will be closed.  Before returning you to UNIX, the simulator will ask you if you want to save the command file (the list of commands you have typed during this session). \n\nThis command has a different effect if used from a higher level interface.  From a debug interface (prompt `debug[n]>') it will return you to the previous level interface, but only if you have fixed all the errors that put you at this level.  From the interrupt interface (prompt `interrrupt[n]>' it will cause whatever operation was interrupted to continue at the point at which it was interrupted.\n\n");
  LOGfprintf(Dispf,"You can exit to UNIX from any level by typing control_D.\n");

 synerror:
	Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: quit\n\n");

  return 0;
BFLYTHROW("Cmd_quit",0)
}

Cmd_q(argc,argv)
     int argc;
     char ** argv;

{Cmd_quit(argc,argv);
}

/***** call *****/

Cmd_call(argc,argv)		/* ????????????????????? */
     int argc;
     char ** argv;

{
    func_ptr func;

BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if(argc < 2)
    goto helpinfo;
  else
    {
      func = NameToFunc(argv[1]);
      if(func == NULL)
	{
	  LOGfprintf(stderr,"cannot find function %s\n",argv[1]);
	  EAT;
	}
      else
	func(argc-1,++argv);
    }
  return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The call command is used to call any function in your code (so long as it was not declared static).  The syntax is given below.  The command simply calls the function with the arguments as given, no checking is done to see that you have given the right number and type of arguments.  Therefore missing arguments will get random values.\n");

	Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: call <function name> [arg1 arg2 ...]\n");

  return 0;
BFLYTHROW("Cmd_call",0)
}

/***** Pause *****/

Cmd_pause(argc,argv)
     int argc;
     char ** argv;
     
{
BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if(argc != 2)
    goto synerror;
  else
    switch(Lex(argv[1]))
      {
      case ON :
	Pause = 1;
	break;
      case OFF :
	Pause = 0;
	break;
      default :
	LOGfprintf(stderr,"\nUsage: pause ON | OFF\n\n");
	EAT;
      }
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The pause command allows you to turn pausing during simulation on or off. If pausing is turned on, then whenever the Show set is listed (determined by the `show' command), the simulator will pause and wait for you to tell it to continue.  This allows you to examine any displayed information before continuing with the simulation.\n");

 synerror:
	Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: pause <on|off>\n\n");

  return 0;
BFLYTHROW("Cmd_pause",0)
} /* Pause */


#ifndef BFLY			/* ******************notice this!!!!! */

/***** load *****/

Cmd_load(argc,argv)		/* ???????????????? */
     int argc;
     char ** argv;

{
    FILE *fp;

BFLYCATCH
    if ((argc == 2) && (Lex(argv[1]) == HELP))
      goto helpinfo;
    if(argc != 2)
      goto synerror;
    else
      if ((fp =
#ifdef BFLY
	   fopen_obj(argv[1],"r")
#else
	   fopen(argv[1],"r")
#endif
	   ) == NULL)
	LOGfprintf(stderr,"cannot open %s\n\n",argv[1]);
      else
	{
	  NetLoad(fp);
	  fclose(fp);
	}
    return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The load command is used to load an empty simulator with a network stored in a file.  If you have made or allocated any units you cannot use this command.  The network you load should have been saved to file with the `save' command.\n");

  synerror:
	Format = FALSE;
      LOGfprintf(Dispf,"\nUsage: load [file name]\n\n");

    return 0;
BFLYTHROW("Cmd_load",0)
} /* load */

#endif

/***** echo *****/

Cmd_e(argc,argv)
     int argc;
     char ** argv;

{Cmd_echo(argc,argv);
}

Cmd_echo(argc,argv)
    int argc;
    char ** argv;

{
BFLYCATCH
  switch(argc)
    {
    case 1:
      if (Echo)
	LOGfprintf(stdout,"Progress echoed every %d steps\n\n",EchoStep);
      else
	LOGfprintf(stdout,"echo turned off\n");
      break;
    case 2:
      switch(Lex(argv[1]))
	{
	case INT:
	  EchoStep = Yyint;
	  Echo = TRUE;
	  break;
	case ON:
	  Echo = TRUE;
	  break;
	case OFF:
	  Echo = FALSE;
   break;
 case HELP:
   goto helpinfo;
 default:
   LOGfprintf(stdout,"Unknown echo option: %s\n\n",argv[1]);
   EAT;
   goto synerror;
 }
      break;
    default:
      goto synerror;
    }
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The echo command allows you to turn echoing on and off and to set the step count for echoing.  If echoing is turned on, then the simulator will issue a message every so many steps, where the number of steps is set by you using this command (default is every step).\n");

 synerror:
	Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: echo [StepCount | on | off]\n\n");

  return 0;
BFLYTHROW("Cmd_echo",0)
} /* echo */


#ifndef BFLY
/***** shell *****/
    /* ????????????????????? */
Cmd_shell(argc,argv)
    int argc;
    char ** argv;

{
    char command[MAX_COMMAND_LEN];
    char *getenv();
    int i;

BFLYCATCH
    if ((argc == 2) && (Lex(argv[1]) == HELP))
      goto helpinfo;
    command[0] = '\0';
    for(i = 1; i < argc; i++)
      {
	(void) strcat(command,argv[i]);
	(void) strcat(command," ");
      }
    if(command[0] != '\0')     /* specific command */
      system(command);
    else       /* call the shell */
      system(getenv("SHELL"));
    return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The shell command is used to execute a UNIX shell command or to put you into a UNIX shell.  Issuing the shell command without any arguments puts you into a shell, and you return to the simulator when you exit that shell.  If you issue the shell command with arguments, the argument string is simply interpreted as a UNIX shell command and executed.  For instance, if you wanted to find out what files are in the directory you are currently in, you would type `shell ls'.\n");

    Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: shell [<shell command> [<arguments>]]\n\n");

    return 0;
BFLYTHROW("Cmd_shell",0)
} /* shell */
#endif

/***** go *****/
    /* ???????????????? */
Cmd_g(argc,argv)
     int argc;
     char ** argv;

{Cmd_go(argc,argv);
}

Cmd_go(argc,argv)
    int argc;
    char ** argv;

{
    int no_steps;
    int timed = FALSE;
    int start, finish;

#ifndef BFLY
    struct timeval tp;
    struct timezone tzp;
#endif

BFLYCATCH
    switch(argc)
      {
      case 1:
 no_steps = 1;
 break;
      case 2:
 switch(Lex(argv[1]))
   {
   case INT:
     no_steps = Yyint;
     break;
   case CLOCK:
     no_steps = 1;
     timed = TRUE;
     break;
   case HELP:
     goto helpinfo;
   default:
     LOGfprintf(stdout,"Illegal number of steps: %s\n\n",argv[1]);
     EAT;
     return 0;
   }
 break;
      case 3:
 if ((Lex(argv[1]) == CLOCK) && (Lex(argv[2]) == INT))
   {
     no_steps = Yyint;
     timed = TRUE;
   }
 else
   goto synerror;
 break;
      default:
 goto synerror;
      }

#ifdef BFLY
    if(timed) start = rtc;
    Step(no_steps);
    if(timed)
      {
 finish = rtc;
 LOGfprintf(stdout,"time = %ld seconds\n",finish - start);
 fflush(stdout);
      }
#else
    if (timed)
      {
 gettimeofday(&tp,&tzp);
 start = tp.tv_sec;
      }
    Step(no_steps);
    if (timed)
      {
 gettimeofday(&tp,&tzp);
 finish = tp.tv_sec;
 LOGfprintf(stdout,"time = %ld seconds\n",finish - start);
      }
#endif

    return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The go command causes one or more steps to be simulated.  The simple version expects one argument, the number of steps to be simulated.  You may also ask for the simulator to time the steps, by typing `go clock n', where n is the number of steps you want timed.  The timer operates in increments of one second, thus you will want to time at least several steps to get an idea of simulation speed.\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: go [ [clock] <StepCount>]\n");

  return 0;
BFLYTHROW("Cmd_go",0)
} /* go */



/***** show *****/
				/* ????????????????????? */
Cmd_show(argc,argv)
    int argc;
    char ** argv;

{
    int i,lx;
    register int u;
    register Unit * up;

BFLYCATCH
    switch(argc)
      {
      case 1:
	ShowUnits();
	if(Pause)
	  UserWait("any character to continue");
	return 0;
      case 2:
	switch(Lex(argv[1]))
	  {
	  case ON:
	    Show = TRUE;
	    break;
	  case OFF:
	    Show = FALSE;
	    break;
	  case SET:
	    LOGfprintf(stdout,"Show sets are:\n");
	    for (i = 0; i < LastSet; i++)
	      if(ShowSets>>i & 1)
		LOGfprintf(stdout,"\t%s\n",SetNames[i]);
	    break;
	  case POT:
	    LOGfprintf(stderr,"\nUsage: show pot <UnitId>\n\n");
	    break;
	  case PLUS:
	    LOGfprintf(stderr,"\nUsage: show + <UnitId>\n\n");
	    break;
	  case MINUS:
	    LOGfprintf(stderr,"\nUsage: show - <UnitId>\n\n");
	    break;
	  case STEP:
	    LOGfprintf(stdout,"Show every %d steps\n",ShowStep);
	    if(Show)
	      LOGfprintf(stdout,"(Show is on)\n\n");
	    else
	      LOGfprintf(stdout,"(Show is off)\n\n");
	    break;
	  case HELP:
	    goto helpinfo;
	  default:
	    LOGfprintf(stderr,"Unknown show option: %s\n\n",argv[1]);
	  }
	return 0;
      default:
	switch(Lex(argv[1]))
	  {
	  case STEP :
	    switch(Lex(argv[2]))
	      {
	      case INT:
		ShowStep = Yyint;
		break;
	      default:
		LOGfprintf(stderr,"Unknown SHOW STEP option: %s\n\n",argv[2]);
	      }
	    break;
	  case POT :
	    switch(Lex(argv[2]))
	      {
	      case INT :
		ShowPot = Yyint;
		break;
	      case FLOAT :
		ShowPot = Yyfloat;
		break;
	      default:
		LOGfprintf(stdout,"Unknown potential value: %s\n\n",argv[2]);
	    }
#ifdef BFLY
	    Send_Sim_Cmd(SHOWPOT_C,ShowPot);
#endif
	    break;
	  case PLUS :
	    Curarg = 2;				  /* for GetUnits */
	    while(argc > Curarg)
	      {
		if(!GetUnits(argc,argv))
		  return 0;
#ifdef BFLY
		Set_Cmd_Numbs(2,Uset,Usetind);
		Send_Range_Cmd(SHOWADD_C,Ulow,Uhigh);
#else
		FOR_UNITS_P(u,up)
		  SetFlagP(up,SHOW_FLAG);
#endif
	      }
	    break;
	  case MINUS :
	    Curarg = 2;				  /* for GetUnits */
	    while(argc > Curarg)
	      {
		if(!GetUnits(argc,argv))
		  return 0;
#ifdef BFLY
		Set_Cmd_Numbs(2,Uset,Usetind);
		Send_Range_Cmd(SHOWDEL_C,Ulow,Uhigh);
#else
		FOR_UNITS_P(u,up)
		  UnsetFlagP(up,SHOW_FLAG);
#endif

	      }
	    break;
	  case SET :
	    lx = Lex(argv[2]);
	    if(lx == PLUS || lx == MINUS)
	      {
		NameDesc *look,nte;
		int si;

		if(argc == 3)
		  {
		    LOGfprintf(stderr,"\nUsage: show set [+|-] <SetName>\n\n");
		    return 0;
		  }
		look = FindName(argv[3],&nte);
		if(look == NULL || look->type != SET_SYM)
		  {
		    LOGfprintf(stderr,"%s not a set\n",argv[3]);
		    break;
		  }
		si = look->index;
		ShowSets &= (1<<si);
#ifdef BFLY
		Set_Cmd_Numbs(2,TRUE,si);
		if (lx == PLUS)
		  Send_Range_Cmd(SHOWADD_C,0,NoUnits-1);
		else
		  Send_Range_Cmd(SHOWDEL_C,0,NoUnits-1);
#else
		if(lx == PLUS)
		  ShowSets |= 1<<si;
		else
		  ShowSets &= ~(1<<si);
#endif
		break;
	      }
	    else
	      {
		LOGfprintf(stdout,"\nUsage: show set [(+|-) <set name>]\n\n");
		break;
	      }
	  default:
	    LOGfprintf(stderr,"Unknown SHOW option %s\n\n",argv[1]);
	  }
      }
    return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The show command is used to control what information is displayed during execution.  With it you can turn showing on or off, cause selected units or sets of units to be added to those displayed, require that units whose activation (potential) is over a limit be displayed, set the number of simulation steps between displays, and immediately show all units meeting the requirements previously established.  If showing is turned on, then every n steps (n is set by you with `show step n') all units in the show list are displayed.\n\n");
    LOGfprintf(Dispf,"You may add/subtract units to the list with `show +|- <unit-id>', where a unit-id may be any of the standard ways of specifying one or more units (try `help UnitId' for information on this).  You may add a set to the display list with `show set +|- <set name>'.  In this case all the units in the set are displayed, even if the set changes during simulation.  You can use this feature to track interesting units by adding and subtracting them from the set in the unit functions.\n\n");
    LOGfprintf(Dispf,"You may set the threshold potential for displaying with `show pot value' where value is your threshold.  In the floating point simulator the value may be a real or integer.  If you type `show set' the simulator will tell you which sets of units are in the display list.  If you simply type `show' then all the units in the show list will be displayed.  Use `show on|off' to turn the show option on or off.  Unit displays are in the abbreviated form.\n");

	Format = FALSE;
    LOGfprintf(Dispf,"\n\tUsage: show [on | off]\n\tshow <step | pot > <value>\n\tshow set [(+|-) <set name>]\n\n");
    return 0;

BFLYTHROW("Cmd_show",0)
} /* show */

#ifndef BFLY			/* ****************notice this !!!!! */

/***** read ******/
/*
   Reads commands from named file.
*/

Cmd_read(argc,argv)
    int argc;
    char ** argv;

{
BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if(argc != 2)
    goto synerror;
  ReadCmdFile(argv[1]);				  /* in main.c */
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The read command is used to execute simulator commands in a file.  The command can be nested - in other words a file of commands that is being read in may contain a read command to read a different file.  The maximum nesting depth is %d.  A command file of each session is automatically produced so that you can rerun the session easily.\n",MAXFILES);

 synerror:
	Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: read <file name>\n\n");

  return 0;
BFLYTHROW("Cmd_read",0)
} /* read */


/***** save ******/
/*
   Saves network to file
*/
				/* ????????????????????????? */

Cmd_save(argc,argv)
    int argc;
    char ** argv;

{
    extern FILE* GetNetFile();

BFLYCATCH
    if ((argc == 2) && (Lex(argv[1]) == HELP))
      goto helpinfo;
    switch (argc)
      {
      case 1:
	NetSave(GetNetFile((char*)NULL));
	CloseNetFile();
	break;
      case 2:
	NetSave(GetNetFile(argv[1]));
	CloseNetFile();
	break;
      default:
	goto synerror;
      }
    return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The save command is used to save a network to file.  The structure (pattern of units and links) and state (values of weights, potentials, etc) is saved. The network may be reloaded into an empty simulator with the `load' command. Related commands are `checkpoint' and `restore', which checkpoint and restore just the state to file.\n\n");
    LOGfprintf(Dispf,"You may specify a file name for the save file, if you do not you will be prompted for one.  The convention is that all save file names are of the form `name.net.n', where n is an automatically incremented integer, although you may use any file name you like.\n");

  synerror:
    Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: save [<file name>]\n\n");

    return 0;
BFLYTHROW("Cmd_save",0)
}

#ifndef BFLY
Cmd_log(argc,argv)		/* no logging on butterfly */
     int argc;
     char ** argv;
		
{
  switch(argc)
    {
    case 1:
      if (Logging)
	LOGfprintf(stdout,"Log is on\n");
      else
	LOGfprintf(stdout,"Log is off\n");
      break;
    case 2:
      switch (Lex(argv[1]))
	{
	case ON:
	  LogOn();
	  break;
	case OFF:
	  LogOff();
	  break;
	case HELP:
	  goto helpinfo;
	default:
	  EAT;
	  goto synerror;
	}
      break;
    default:
      goto synerror;
    }
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The log command allows you to switch session logging on or off, or to examine whether logging is switched on or off.  `log on|off' will switch on or off, `log' will report whether logging is on or off.  When you switch logging on, you will be prompted for a file name.  The convention is that all log files are of the form `name.log.n' where n is an automatically incremented integer, although you may use any file name you like.\n\n");
  LOGfprintf(Dispf,"Switching logging off will close the current logfile, and if you switch logging again you will be able to commence a new log file or append to any existing file (including the previous log file).\n\n");

 synerror:
  Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: log [<on|off>]\n");
  return 0;
}
#endif

/***** pipe ******/
/*
   controls piping of output through other programs
*/

Cmd_pipe(argc,argv)
    int argc;
    char ** argv;

{
  int i;

BFLYCATCH
  switch(argc)
    {
    case 1:
      LOGfprintf(stdout,"pipe command is: %s\n",PipeCommand);
      if (PipeFlag)
	LOGfprintf(stdout,"pipe is on\n");
      else
	LOGfprintf(stdout,"pipe is off\n");
      break;
    case 2:
      switch(Lex(argv[1]))
	{
	case ON:
	  PipeFlag = TRUE;
	  break;
	case OFF:
	  PipeFlag = FALSE;
	  break;
	case HELP:
	  goto helpinfo;
	default :
	  strcpy(PipeCommand,argv[1]);
	}
      break;
    default:
      PipeCommand[0] = '\0';
      for(i = 1; i < argc; i++)
	{
	  (void) strcat(PipeCommand,argv[i]);
	  (void) strcat(PipeCommand," ");
	}
    }
  return 0;

 helpinfo:
  Format = TRUE;
   LOGfprintf(Dispf,"The pipe command controls whether or not output to the terminal is piped through another process.  The simulator maintains a current pipe command (or process) to which piping will happen if it is switched on.  You may reset the pipe command with `pipe <command>' - the default is the UNIX `more' command.  You may switch piping on or off with `pipe on|off'.  You may find out the current pipe command and whether piping is on or off with simply `pipe'.\n\n");
   LOGfprintf(Dispf,"If piping is on the output of a unit/link listing or a unit display is piped through the command, but all other output is sent normally.  For example, you may save a listing of all the links in a file `links.lst' by setting the pipe command with `pipe cat > links.lst', switching piping on with `pipe on', listing the links with `list links', and switching the pipe off with `pipe off'.  Beware!  Each simulator command which outputs via a pipe (list, display and shell) closes the pipe before returning to you.  This means that if the pipe is set as above, each command will overwrite the file `links.lst'.\n");

  Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: pipe [on | off | <command>]\n\n");

  return 0;
BFLYTHROW("Cmd_pipe",0)
} /* pipe */

#endif				/* ****************notice this !!!!!! */

/***** print *****/


/*
   doprints rest of line; used in files loaded with read
*/
static doprint(pause,argc,argv)
    int pause;
    int argc;
    char ** argv;
{
  int i = 1;

BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if(argc < 2)
    goto synerror;
  for (i = 1; i < argc; i++)
    LOGfprintf(stdout,"%s ",argv[i]);  
  if(pause)
    UserWait("\nPAUSE - any char to continue");
  return;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The print and printpause commands are primarily for use in simulator command files (see the `read' command).  When the simulator is reading from a command file it does not echo the commands on the screen.  The print command simply sends its arguments to the terminal, thus if inserted judiciously in a command file, the print commands will enable you to monitor progress in reading the file.  The printpause command gives you greater flexibility - the message is printed on the screen and the simulator waits for you to respond before continuing with reading the file.\n\n");
  LOGfprintf(Dispf,"For instance, if `printpause ended stage 1' is in the command file being read, you will see:\n\nended stage one Pause - any char to continue'\n\nand the simulator will wait for your response.  At this stage you could do something fancy like hit control_C.  This will put you in the interrupt interface which will allow you to examine the state of the network, and when you exit the interrupt interface, the simulator will continue as if you had responded normally to its pause message.\n");

 synerror:
  Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: print[pause] <message>\n");

  return;
BFLYTHROW("doprint",0)
} /* doprint */

Cmd_print(argc,argv)
    int argc;
    char ** argv;

{  doprint(0,argc,argv);
 }

Cmd_printpause(argc,argv)
    int argc;
    char ** argv;

{  doprint(1,argc,argv);
 }

/*****	deleteset *****/
/*
*/
Cmd_deleteset(argc,argv)	/* ?????????????and all following */
    int argc;
    char ** argv;

{
BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if(argc != 2)
    goto synerror;
  else
    DeleteSet(argv[1]);
  return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"This command deletes a set.  All units in the set are removed from the set and the name is reset to be unused\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: deleteset <set name>\n\n");
  return 0;
BFLYTHROW("Cmd_deletset",0)
}


/*****	addset *****/
/*
*/
Cmd_addset(argc,argv)
    int argc;
    char ** argv;

{
    int created=0;
    char * set;

BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc < 3)
    goto synerror;
  if(!IsSet(set = argv[1]))
    {
      if (DeclareSet(set) == -1)
	return 0;
      created = 1;
    }
  Curarg = 2;					  /* for GetUnits */
  if (GetUnits(argc,argv))
    if (Uset)
      {
	if (created)
	  DeleteSet(set);
	LOGfprintf(stderr,"Use unionset!\n");
      }
    else
      AddToSet(set,Ulow,Uhigh);
  else
    {
      if(created)
	DeleteSet(set);
      LOGfprintf(stderr,"Command ignored!\n\n");
    }
  return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"This command adds one or more units to a set.  The units may be specified using any of the normal methods (try `help UnitId' for information on this). If the set does not already exist it is created.\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: addset <set name> <UnitId>\n\n");

    return 0;
BFLYTHROW("Cmd_addset",0)
}

/*****	remset *****/
/*
*/
Cmd_remset(argc,argv)
    int argc;
    char ** argv;

{
    int created=0;
    char * set;

BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc < 3)
    goto synerror;
  if(!IsSet(set = argv[1]))
    {
      LOGfprintf(stderr,"%s is not a set name!\n",argv[1]);
      return 0;
    }
  Curarg = 2;					  /* for GetUnits */
  if(GetUnits(argc,argv))
    if (Uset)
      LOGfprintf(stderr,"Use diffset!\n");
    else
      RemFromSet(set,Ulow,Uhigh);
  else
    goto synerror;
  return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"This command removes one or more units from a set.  The units may be specified using any of the normal methods (try `help UnitId' for information on this).\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: remset <set name> <UnitId>\n\n");

    return 0;
BFLYTHROW("Cmd_remset",0)
}

/*****  unionset *****/
/*
*/
Cmd_unionset(argc,argv)
    int argc;
    char ** argv;

{
    int created=0;
    char * set;

BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc != 4)
    goto synerror;
  if(!IsSet(set = argv[1]))
    {
      if(DeclareSet(set) == -1)
	return 0;
      created = 1;
    }
  if (!UnionSet(set,argv[2],argv[3]))
    {
      if(created)
	DeleteSet(set);
      LOGfprintf(stderr,"Command ignored!\n\n");
    }
  return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"This command assigns the union of the second and third sets to the first set. All units which are in either the second set or the third set or both are put in the first (answer) set.  If the answer set does not yet exist, it is created.  If the answer set exists, then any units in the set which are not in the union of the second and third sets are removed from it.\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,
	       "\nUsage: unionset <answer set name> <set name> <set name>\n\n");

  return 0;
BFLYTHROW("Cmd_unionset",0)
}

/*****	intersectset *****/
/*
*/
Cmd_intersectset(argc,argv)
    int argc;
    char ** argv;

{
    int created=0;
    char * set;

BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc != 4)
    goto synerror;
  if(!IsSet(set = argv[1]))
    {
      if(DeclareSet(set) == -1)
	return 0;
      created = 1;
    }
  if (!IntersectSet(set,argv[2],argv[3]))
    {
      if(created)
	DeleteSet(set);
      LOGfprintf(stderr,"Command ignored!\n\n");
    }
  return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"This command assigns the intersection of the second and third sets to the first set.  All units which are in either the second set or the third set or both are put in the first (answer) set.  If the answer set does not yet exist, it is created.  If the answer set exists, then any units in the set which are notin the intersection of the second and third sets are removed from it.\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,
	       "\nUsage: intersectset <answer set name> <set name> <set name>\n\n");

  return 0;
BFLYTHROW("Cmd_intersectset",0)
}

/*****	doinverseset *****/
/*
*/
Cmd_inverseset(argc,argv)
     int argc;
     char ** argv;

{
    int created=0;
    char * set;

BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc != 4)
    goto synerror;
  if(!IsSet(set = argv[1]))
    {
      if(DeclareSet(set) == -1)
	return 0;
      created = 1;
    }
  if (!InverseSet(set,argv[2]))
    {
      if(created)
	DeleteSet(set);
      LOGfprintf(stderr,"Command ignored!\n\n");
    }
  return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"This command assigns the inverse of the second set to the first set.  All units which are not in the second set are put in the first (answer) set.  If the answer set does not yet exist, it is created.  If the answer set exists, then any units in the set which are not in the inverse of the second set are removed from it.\n\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,
	       "\nUsage: inverseset <answer set name> <set name>\n\n");

  return 0;
BFLYTHROW("Cmd_inverseset",0)
}


/*****	diffset *****/
/*
*/
Cmd_diffset(argc,argv)
    int argc;
    char ** argv;

{
    int created=0;
    char * set;

BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc != 4)
    goto synerror;
  if(!IsSet(set = argv[1]))
    {
      if(DeclareSet(set) == -1)
	return 0;
      created = 1;
    }
  if (!DifferenceSet(set,argv[2],argv[3]))
    {
      if(created)
	DeleteSet(set);
      LOGfprintf(stderr,"Command ignored!\n\n");
    }
  return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"This command assigns the difference of the second and third sets to the first set.  All units which are in the second set but not the third set are put in the first (answer) set.  If the answer set does not yet exist, it   created.  If the answer set exists, then any units in the set which are notin the difference of the second and third sets are removed from it.  The difference of the second and third sets is the set of units which are in the second set but not in the third set.\n");

  synerror:
	Format = FALSE;
    LOGfprintf(Dispf,
	 "\nUsage: differenceset <answer set name> <set name> <set name>\n\n");

  return 0;
BFLYTHROW("Cmd_diffset",0)
}

/*****	sync *****/
/*
   Sets execution style: synchronous - every unit updated each time step,
                                       outputs updated after all done.
			 asynchronous- every unit updated each time step,
			               immediate output update, random order
			 fair async  - fixed fraction updated each time step,
			               chosen at random from all units,
				       outputs updated after each time step,
				       guaranteed all done at least once
				       after limit number of steps.
*/

Cmd_sync(argc,argv)
     int argc;
     char ** argv;

{
BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc > 1)
    goto synerror;
  else
    {
      SyncFlag = SYNC;
#ifdef BFLY			/* notify sims */
      Send_Sim_Cmd(SYNC_C,0,"");
#endif
    }
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The sync command sets the simulation style to be synchronous.  This means that at each step every unit is updated, with the new output values being held back until every unit has updated.  This is the fastest execution style.\n");

 synerror:
	Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: sync\n\n");

  return 0;
BFLYTHROW("Cmd_sync",0)
}

Cmd_async(argc,argv)
    int argc;
    char ** argv;

{
    int gettime;

BFLYCATCH
  SyncFlag = ASYNC;
  switch(argc)
    {
    case 1:
#ifdef BFLY
      gettime = rtc;
#else
      gettime = (int) time((time_t *)0);
#endif
      srandom(gettime);
      LOGfprintf(stdout,"random() seeded with time: %d\n",gettime);
      break;
    case 2:
      switch(Lex(argv[1]))
	{
	case INT:
	  srandom(Yyint);
	  gettime = Yyint;
	  LOGfprintf(stdout,"random() seeded with %d\n",Yyint);
	  break;
	case HELP:
	  goto helpinfo;
	default:
	  EAT;
	  goto synerror;
	}
      break;
    default:
      goto synerror;
    }
#ifdef BFLY			/* notify sims */
    Send_Sim_Cmd(ASYNC_C,gettime,"");
#endif
    return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The async command sets the execution style to be asynchronous.  This means that at each step every unit is updated, in pseudo-random order, with the new output value for a unit available immediately the unit has updated. This command may take an integer argument, the integer will then be used to seed the random number generator.  If you do not give a seed number, the system time will be used.  Seeding the random number generator with the same number on different occasions will ensure that the same sequence of numbers is generated.  This enables repetition of sessions run in asynchronous mode.\n");

  synerror:
    Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: async [<integer>]\n\n");
    return 0;
BFLYTHROW("Cmd_async",0)
}

Cmd_fsync(argc,argv)
     int argc;
     char ** argv;

{
  int fract, limit, gettime;

BFLYCATCH
  SyncFlag = FAIRASYNC;
  switch (argc)
    {
    case 1:
      if (ExecFract == 0)
	{
	  LOGfprintf(stderr,
		     "Execution percentage not set\n");
	  goto synerror;
	}
      else
	LOGfprintf(stdout,"Execution Percentage : %d%%, Execution Limit : %d steps\n",
	       ExecFract,ExecLimit);
      break;
    case 2:
      if (Lex(argv[1]) == HELP)
	goto helpinfo;
      else
	goto synerror;
    case 4:			/* this case is here so if falls into case 3 */
      if (Lex(argv[3]) == INT)
	{
	  srandom(Yyint);
	  gettime = Yyint;
	}
      else
	{
	  EAT;
	  goto synerror;
	}
    case 3:
      if (Lex(argv[1]) == INT)
	if (Yyint > 0 && Yyint <= 100)
	  fract = Yyint;
	else
	  {
	    LOGfprintf(stderr,
		    "Error: execution percentage not in range 1 to 100\n");
	    return 0;
	  }
      else
	{
	  EAT;
	  goto synerror;
	}
      if (Lex(argv[2]) == INT)
	if (Yyint > 0)
	  limit = Yyint;
	else
          {
	    LOGfprintf(stderr,
		    "Error: execution step limit must be greater than zero\n");
	    return 0;
	  }
      else
	{
	  EAT;
	  goto synerror;
	}
      if (fract * limit < 100)
	LOGfprintf(stdout,"??? In %d steps, at %d%% per step, cannot run all units!\n",
	       limit, fract);
      ExecFract = fract;			  /* set in globals */
      ExecLimit = limit;
      LOGfprintf(stdout,"Execution percentage set to %d, limit set to %d steps\n",
	     fract,limit);
      if (argc == 3)
	{
#ifdef BFLY
	  gettime = rtc;
#else
	  gettime = (int) time((time_t *)0);
#endif
	  srandom(gettime);
	}
#ifdef BFLY
      Set_Cmd_Numbs(3,gettime,ExecFract,ExecLimit);
      Send_Sim_Cmd(FSYNC_C,gettime,"");
#endif
      LOGfprintf(stdout,"random() seeded with: %d\n",gettime);
      break;
    default:
      goto synerror;
    }
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The fsync command sets the simulation style to be fair asynchronous.  In addition you may use it to set the execution percentage and the execution limit and to seed the random number generator.  Fair asynchronous means that at each step a percentage of the units are updated, picked pseudo-randomly from all the units, with the proviso that after a limiting number of steps every unit will have been updated at least once.\n\n");
  LOGfprintf(Dispf,"You may set the percentage and limit values with `fsync percentage limit'.  You may also seed the random number generator, if you do not it will be seeded with the system time. Seeding the random number generator with the same number on different occasions will ensure that the same sequence of numbers is generated.  This enables repetition of sessions run in fair asynchronous mode.n\n");

 synerror:
  Format = FALSE;
  LOGfprintf(Dispf,
	  "\nUsage: fsync <execfrac> <execlimit> [<random seed>]\n\n");
  return 0;
BFLYTHROW("Cmd_fsync",0)
}

Cmd_reset(argc,argv)
     int argc;
     char ** argv;

{
BFLYCATCH
  if ((argc == 2) && (Lex(argv[1]) == HELP))
    goto helpinfo;
  if (argc > 1)
    goto synerror;

#ifdef BFLY
  Send_Sim_Cmd(RESET_C,0,"");
  Clock = 0;
#else
  Reset();
#endif
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The reset commands resets the potential and state of all units to their original values (as specified when they were made with MakeUnit).  It also resets the outputs of all units to be zero.  It does not reset weights or any other parameters.\n");

 synerror:
	Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: reset \n\n");
BFLYTHROW("Cmd_reset",0)

 return 0;
}

#ifndef BFLY			/* ***************notice this !!!!!!!!! */

/* trash */
/*
  trash existing network, reset data structures
*/

Cmd_restart(argc,argv)
     int argc;
     char ** argv;

{
  if (argc == 2 && Lex(argv[1]) == HELP)
    goto helpinfo;
  if (argc != 1)
    goto synerror;
  TrashNetwork();
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"This command cleans out all the simulator data structures except the function table and the state names, so network construction can begin again from scratch.  The graphics display is also wiped clean.\n");
  Format = FALSE;

 synerror:
  LOGfprintf(Dispf,"\nUsage: restart\n\n");
  return 0;
}

/***** checkpoint ******/
/*
   Saves network to file
*/

Cmd_checkpoint(argc,argv)
    int argc;
    char ** argv;

{
    extern FILE* GetChkFile();

BFLYCATCH
    if ((argc == 2) && (Lex(argv[1]) == HELP))
      goto helpinfo;
    switch (argc)
      {
      case 1:
	NetCheckpoint(GetChkFile((char *)NULL));
	CloseChkFile();
	break;
      case 2:
	NetCheckpoint(GetChkFile(argv[1]));
	CloseChkFile();
	break;
      default:
	LOGfprintf(stderr,"\nUsage: checkpoint [<filename>]\n\n");
      }
    return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The checkpoint command is used to save a the state of a network to file.  The state consists of the values of unit parameters (e.g. weights, potentials, etc), the current sets and unit state names . The network state may be restored `restore' command.  Related commands are `save' and `load', which save and reload the structure (pattern of units and links) of the network as well.\n\n");
    LOGfprintf(Dispf,"You may specify a file name for the checkpoint file, if you do not you will be prompted for one.  The convention is that all checkpoint file names are of the form `name.chk.n', where n is an automatically incremented integer, although you may use any file name you want.\n");

    Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: checkpoint [<file name>]\n\n");

    return 0;

BFLYTHROW("Cmd_checkpoint",0)
} /* checkpoint */

/***** restore *****/

Cmd_restore(argc,argv)
     int argc;
     char ** argv;

{
    FILE *fp;

BFLYCATCH
    if ((argc == 2) && (Lex(argv[1]) == HELP))
      goto helpinfo;
    if (argc != 2)
      goto synerror;
    if ((fp =
#ifdef BFLY
	 fopen_obj(argv[1],"r")
#else
	 fopen(argv[1],"r")
#endif
	 ) == NULL)
      LOGfprintf(stderr,"cannot open %s\n\n",argv[1]);
    else
      {
	RestoreNetwork(fp);
	fclose(fp);
      }
    return 0;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The restore command restores the state of a network from a file made with the checkpoint command.  This command simply restores the state of the network (i.e. the weights, potentials, etc), it does NOT rebuild the network.  That means the network must already have been built.  The command will check that the file used for restoration was made from the same simulator with the same network.  If you have recompiled your simulator, but have the same network in it, ignore the warning message.  It will also issue a warning if you are restoring from a checkpoint file made during a previous session.\n");

  synerror:
    Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: load <file name>\n\n");

    return 0;
BFLYTHROW("Cmd_restore",0)
} /* load */

/***** debug *****/

Cmd_debug(argc,argv)
    int argc;
    char ** argv;

{
BFLYCATCH
  switch(argc)
    {
    case 1:
      if (Debug)
	{
	  LOGfprintf(stdout,"in debug mode\n");
	  if (AutoFix)
	    LOGfprintf(stdout,"in auto-fix mode\n");
	  else
	    LOGfprintf(stdout,"not in auto-fix mode\n");
	}
      else
	LOGfprintf(stdout,"not in debug mode\n");
      return 0;
    case 2:
      switch(Lex(argv[1]))
	{
	case ON:
	  Debug = 1;
	  AskLogOn();
	  return 0;
	case OFF:
	  Debug = 0;
	  AskLogOff();
	  return 0;
	case HELP:
	  goto helpinfo;
        default:
	  goto synerror;
	}
      break;
    case 3:
      if (Lex(argv[1]) == AUTO)
	switch(Lex(argv[2]))
	  {
	  case ON:
	    AutoFix = TRUE;
	    return 0;
	  case OFF:
	    AutoFix = FALSE;
	    return 0;
	  }
    }
  return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The debug command is used to switch network construction debugging on and off, and to switch automatic error correction on and off.  `debug on|off' will control whether debugging is operative.  `debug auto on|off' will control whether automatic error correction is in operation.\n\n");
  LOGfprintf(Dispf,"Debugging only applies to network construction, not to simulation.  If debugging is switchd on, then anytime you (or a function you call) tries to make a unit, site or link, or to name a unit, the simulator will check that the values you have given are suitable.  For instance, if you specify a function for a unit that is not known to the simulator, it will issue an error message.  If automatic correctiion (Auto-Fix) is switched on, the simulator will substitute what it thinks is a suitable value, and continues.\n\n");
LOGfprintf(Dispf,"If Auto-Fix is off, you will be thrown into a debug interface (level 1 or above) and asked to fix the errors yourself.  The `set' command is used to change incorrect values.  You will not be permitted to leave the debug interface until all values are within range.  However you can type `set default all' to reset all incorrect values to simulator defaults, or `ignore' to cause the simulator to not make the unit, site, link or name.\n\n");
  LOGfprintf(Dispf,"If debugging is turned off, absolutely NO CHECKING is done - you will get a core dump if anything is wrong during construction.  It is best to build and test your network with debug switched on until you are sure the build function is correct.\n\n");
  LOGfprintf(Dispf,"If you have debugging and Auto-Fix switched on, and errors or warnings are occurring rapidly, you may hit control_C which will put you in the interrupt interface whence you may examine the network, switch Auto-Fix off, etc.  If you happen to hit control_C when the network is not in a safe state, the interrupt will be delayed.\n\n");
  LOGfprintf(Dispf,"Simply typing `debug' will get you the current debug settings.\n");

 synerror:
  Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: debug [[auto] <on | off>]\n\n");
  return 0;
BFLYTHROW("Cmd_debug",0)
} /* pipe */

#endif                                    /* ****************notice this !!*/

/* end of commands for butterfly control and uniprocessor versions */

#ifndef BFLY

/* commands available only on uniprocessor version - none do anything.
*/

/***** load *****/

/*ARGSUSED*/
Debug_Cmd_load(argc,argv)
     int argc;
     char ** argv;

{
  LOGfprintf(stderr,"Cannot load a network while debugging!\n");
  return 0;
} /* load */

/***** go *****/

Debug_Cmd_g(argc,argv)
     int argc;
     char ** argv;

{Debug_Cmd_go(argc,argv);
}

/*ARGSUSED*/
Debug_Cmd_go(argc,argv)
    int argc;
    char ** argv;

{
  LOGfprintf(stderr,
	  "Cannot simulate network while debugging network construction!\n");
  return 0;
} /* go */

/***** read ******/
/*
   Reads commands from named file.
*/

/*ARGSUSED*/
Debug_Cmd_read(argc,argv)
    int argc;
    char ** argv;

{
  LOGfprintf(stderr,"Cannot read commands from file while debugging!\n");
  return 0;
} /* read */


/* end of uniprocessor only commands */

#endif				/* #ifndef BFLY */

#ifdef BCNTL

/* the following commands run on the butterfly control version only */

Cmd_talk(argc,argv)
     int argc;
     char ** argv;

{
  talk();
}

Cmd_rcall(argc,argv)
     int argc;
     char ** argv;

{
  int who,what,arg,c, index, argc = 0;
  char func[40],
  *arg_buff[NUMARGS];
  
  /*** who to call; either "all" or sim number */

BFLYCATCH  
  if( (who = Lex(argv[1])) == ALL)
    who = EVERY_N;
  else
    if(who == INT)
      who = Yyint;
    else
      {
	fprintf(stderr,"\nUsage: rcall <who> <function>\n\n");
	fflush(stderr);
	EAT;
	return;
      }
  
  /*name of function to call; simple minded error check: function
    name could be the same as some key word, but not a number     */
  if((what = Lex(argv[2])) == INT || what == FLOAT || what == END_LINE)
    {
      fprintf(stderr,"\nUsage: rcall <who> <function>\n\n");
      fflush(stderr );
      EAT;
      return;
    }
  else
    Rcall(who,argc-1,argv+1);
BFLYTHROW("Cmd_rcall",0)
}

/* end of butterfly control version only commands */

#endif				/* #ifdef BCNTL */
