/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains routines that set up, access and change the
 * mode panel and its associated data structures
 *
 * Kenton Lynne
 *******************************************************************/

#include "macros.h"
#include "externs.h"
#include "panel.h"
#include "control_panel.h"

/* form cursor for text mode */
static short tcursor_image[16] = {
#include "icon/tcursor"
};
DEFINE_CURSOR_FROM_IMAGE(t_cursor,15,10,PIX_SRC ^ PIX_DST,tcursor_image);
static struct cursor *tcursor = &t_cursor;

/* forn cursor for draw mode */
static short dcursor_image[16] = {
#include "icon/dcursor"
};
DEFINE_CURSOR_FROM_IMAGE(d_cursor,HOTX,HOTY,PIX_SRC ^ PIX_DST,dcursor_image);
static struct cursor *dcursor = &d_cursor;

static Panel mode_panel;

Panel_item
  gi_log_item,
  gi_logfile_item;

static Panel_item 
  mode_item;

static char cur_logfile[MAX_ITEM_LENGTH+1];

/* static procedures forward referenced */
static mode_proc(),
       log_proc();

/*************************************************************************/
gi_make_mode_panel()
{
/* sets up the mode panel
 */
  struct toolsw *mode_panel_sw;

  /* create the mode panel itself */
  if ((mode_panel_sw = panel_create(gi_tool,
                                    PANEL_HEIGHT, CNTL_HEIGHT,
                                    PANEL_WIDTH, CNTL_WIDTH,
                                    PANEL_LABEL_BOLD, TRUE,
                                    0)) == NULL)
  {
    fputs("Can't create mode panel\n", stderr);
    exit(1);
  }

  mode_panel = mode_panel_sw->ts_data;

  mode_item = panel_create_item
              (mode_panel, PANEL_CHOICE,
               PANEL_LABEL_STRING, "MODE:",
               PANEL_FEEDBACK, PANEL_INVERTED,
               PANEL_LABEL_X, PANEL_CU(0),
               PANEL_LABEL_Y, 5,
               PANEL_CHOICE_STRINGS, " Main ",
                                     " Link ",
                                     " Text ",
                                     " Draw ",
                                     " Custom ",
                                     0,
               PANEL_CHOICE_YS, 5, 0,
               PANEL_CHOICE_XS, PANEL_CU(6), 
                                PANEL_CU(12), 
                                PANEL_CU(18), 
                                PANEL_CU(24), 
                                PANEL_CU(30), 0,
               PANEL_NOTIFY_PROC, mode_proc,
               0);

  /* items  for logging file */
  gi_log_item = panel_create_item
              (mode_panel, PANEL_CHOICE,
               PANEL_DISPLAY_LEVEL, PANEL_CURRENT,
               PANEL_LABEL_STRING, "LOG:",
               PANEL_LABEL_X, gi_log_start_x, 
               PANEL_LABEL_Y, 5,
               PANEL_VALUE, TRUE,
               PANEL_CHOICE_STRINGS, "off",
                                        "on",
                                        0,
               PANEL_NOTIFY_PROC, log_proc,
               0);
  gi_logfile_item  = panel_create_item
              (mode_panel, PANEL_TEXT,
               PANEL_VALUE_STORED_LENGTH, MAX_ITEM_LENGTH,
               PANEL_VALUE_DISPLAY_LENGTH, MAX_ITEM_LENGTH,
               PANEL_LABEL_X, PANEL_CU(9)+gi_log_start_x, 
               PANEL_LABEL_Y, 5, 
               PANEL_LABEL_BOLD, FALSE,
               PANEL_LABEL_STRING, ":",
               PANEL_VALUE, DFT_LOG_FILE_NAME,
               0);
}

/*************************************************************************/
static hide_main_items()
{
/* non-display all the items normally displayed
 * only during Main mode
 */
  int i;

  for (i=MAIN_ITEM_FIRST; i<=MAIN_ITEM_LAST; i++)
  {
    panel_set(gi_mitem[i],PANEL_SHOW_ITEM,FALSE,0);
  }
}

/*************************************************************************/
static hide_link_items()
{
/* non-display all the items normally displayed
 * only during LINK mode
 */
  int i;

  for (i=LINK_ITEM_FIRST; i<=LINK_ITEM_LAST; i++)
  {
    panel_set(gi_litem[i],PANEL_SHOW_ITEM,FALSE,0);
  }
}

/*************************************************************************/
static hide_text_items()
{
/* non-display all the items normally displayed
 * only during TEXT mode
 */
  int i;

  for (i=TEXT_ITEM_FIRST; i<=TEXT_ITEM_LAST; i++)
  {
    panel_set(gi_titem[i],PANEL_SHOW_ITEM,FALSE,0);
  }
}

/*************************************************************************/
static hide_draw_items()
{
/* non-display all the items normally displayed
 * only during DRAW mode
 */
   int i;

   for (i=DRAW_ITEM_FIRST; i<=DRAW_ITEM_LAST; i++)
   {
     panel_set(gi_ditem[i],PANEL_SHOW_ITEM,FALSE,0);
   }
}

/*************************************************************************/
static hide_cust_items()
{
/* non-display all the items normally displayed
 * only during CUSTOM mode
 */
   int i;

   for (i=CUST_ITEM_FIRST; i<=CUST_ITEM_LAST; i++)
   {
     panel_set(gi_citem[i],PANEL_SHOW_ITEM,FALSE,0);
   }
}

/*************************************************************************/
static show_main_items()
{
/* Redisplay all the items normally displayed
 * during Main mode
 */
   int i;

   for (i=MAIN_ITEM_FIRST; i<=MAIN_ITEM_LAST; i++)
   {
     if (i!=TARGET_ITEM
     || gi_cur_unit_what==LINKIN 
     || gi_cur_unit_what==LINKOUT)
       panel_set(gi_mitem[i],PANEL_SHOW_ITEM,TRUE,0);
   }

   /* set the caret to the first item on the panel */
   panel_set(gi_control_panel, PANEL_CARET_ITEM, gi_mitem[WHO_ITEM], 0);
}

/*************************************************************************/
static show_link_items()
{
/* Redisplay all the items normally displayed
 * during LINK mode
 */
   int i;

   for (i=LINK_ITEM_FIRST; i<=LINK_ITEM_LAST; i++)
   {
      panel_set(gi_litem[i],PANEL_SHOW_ITEM,TRUE,0);
   }

   /* set the caret to the first item on the panel */
   panel_set(gi_control_panel, PANEL_CARET_ITEM, gi_litem[LSITE_ITEM], 0);
}

/*************************************************************************/
static show_text_items()
{
/* Redisplay all the items normally displayed
 * during TEXT mode
 */
   int i;

   for (i=TEXT_ITEM_FIRST; i<=TEXT_ITEM_LAST; i++)
   {
     panel_set(gi_titem[i],PANEL_SHOW_ITEM,TRUE,0);
   }
   panel_set(gi_control_panel, PANEL_CARET_ITEM, gi_titem[TFONT1_ITEM], 0);
}

/*************************************************************************/
static show_draw_items()
{
/* Redisplay all the items normally displayed
 * during DRAW mode
 */
   int i;

   for (i=DRAW_ITEM_FIRST; i<=DRAW_ITEM_LAST; i++)
   {
     panel_set(gi_ditem[i],PANEL_SHOW_ITEM,TRUE,0);
   }
}

/*************************************************************************/
static show_cust_items()
{
/* Redisplay all the items normally displayed
 * during CUSTOM mode
 */
   int i;

   for (i=CUST_ITEM_FIRST; i<=CUST_ITEM_LAST; i++)
   {
       panel_set(gi_citem[i],PANEL_SHOW_ITEM,TRUE,0);
   }

   /* move the caret to the first prompt on the panel */
   panel_set(gi_control_panel, PANEL_CARET_ITEM, gi_citem[CLBDN1_ITEM], 0);
}

/*************************************************************************/
/*ARGSUSED*/
static mode_proc(item, value, event)
   Panel_item item;
   int value;
   struct inputevent *event;
{
/* called when the user switches modes by clicking over
 * one of the mode buttons, this routine sets up the
 * proper control panel prompts for the new mode and
 * does any necessary cleanup from the mode being left 
 */

  /* display mouse button prompts for new mode in message window */
  switch (value)
  {
    case MODE_MAIN: 
      gi_put_message("show info | move | mark link target");
      break;
    case MODE_LINK:
      gi_put_message("show info | move | mark link target");
      break;
    case MODE_TEXT:
      gi_put_message( "mark text start | move | delete text");
      break;
    case MODE_DRAW:
      gi_put_message("mark vertex | move | delete drawing"); 
      break;
    case MODE_CUSTOM:
      gi_put_message("show info | move | mark link target");
      break;
  }

  /* if mode has not changed, just return */
  if (value==gi_mode) return;

  /* non-display the old mode control panel items */
  switch (gi_mode)
  {
      case MODE_MAIN: 
           hide_main_items();
           break;

      case MODE_LINK:   
           hide_link_items();

           /* if leaving link mode: make sure everything is reshown */
           gi_reshow_flag |= RESHOW_ALL;
           break;

      case MODE_TEXT:
           hide_text_items();
	   break;

      case MODE_DRAW:
           hide_draw_items();
	   break;

      case MODE_CUSTOM:
           hide_cust_items();
	   break;
   }

  /* if entering link mode, make sure everything is reshown */
  if (value==MODE_LINK) gi_reshow_flag |= RESHOW_ALL;

  /* set new mode */
  gi_mode = value;

  /* set up the correct input mask for new mode */
  win_setinputmask(gi_gfx->gfx_windowfd, &gi_mask[gi_mode],
                  (struct inputmask *)NULL, WIN_NULLLINK);

  /* show the control panel prompts for new mode */
  switch (gi_mode)
  {
    case MODE_MAIN: 
      show_main_items();
      win_setcursor(gi_gfx->gfx_windowfd, gi_cursor);
      break;
    case MODE_LINK:
      show_link_items();
      win_setcursor(gi_gfx->gfx_windowfd, gi_cursor);
      break;
    case MODE_TEXT:
      show_text_items();
      win_setcursor(gi_gfx->gfx_windowfd, tcursor);
      break;
    case MODE_DRAW:
      show_draw_items();
      win_setcursor(gi_gfx->gfx_windowfd, dcursor);
      break;
    case MODE_CUSTOM:
      show_cust_items();
      win_setcursor(gi_gfx->gfx_windowfd, gi_cursor);
      break;
  }

  /* if mode changes demands it, reshow the entire screen */
  if (gi_reshow_flag & RESHOW_ALL) 
    gi_reshow();

}

/*************************************************************************/
/*ARGSUSED*/
static log_proc(item, value, event)
   Panel_item item;
   int value;
   struct inputevent *event;
{
/* called when the user clicks over the "LOG:"
 * prompt to switch logging off or on
 */

   /* if logging is currently on, flush the buffer */
   if (gi_log_on && gi_log_fp) fflush(gi_log_fp);

   /* set up current log value */
   gi_log_on = value;

}

/*************************************************************************/
gi_log(string)
 char *string;
{
/* Logs the string to the current log file making sure
 * to open and close the currnet log file as needed.
 * Assumes gi_log_on is TRUE
 */

  char *log_name;

  /* get logfile name from panel */
  log_name = gi_strip((char *) panel_get_value(gi_logfile_item));

  /* if user has changed the name of the logging file 
     close the old log file and NULL the log file ptr
  */

  if (gi_log_fp)
  {
    if (strcmp(cur_logfile,log_name))
    {
      /* new log file specified - close old one */
      fclose(gi_log_fp);
      gi_log_fp = NULL;
    }
  }

  /* if the log file ptr is null, attempt to open the
     user requested logging file
  */

  if (gi_log_fp==NULL) 
  {
    /* check that a file name is not vacuous */ 
    if (strlen(log_name)<1)
    {
      gi_put_error("Missing log filename");   
      return(ERROR);
    }
    
    /* attempt to open the file */
    if ((gi_log_fp=fopen(log_name,"w"))==NULL)
    {
      gi_put_error("Unable to open log file");   
      return(ERROR);
    }
    
    /* remember this name as the current logging file */

    strcpy(cur_logfile,log_name);
  }

  /* write out the requested string to the current log file */

  fprintf(gi_log_fp,"%s\n",string);
  return(OK);
}

/*************************************************************************/
gi_log_info(at_x, at_y, col)
  int at_x, at_y, col;
{
/* create an info command to show info for
 * a unit at particular display coordinates
 */
  char log_string[MAX_CMD_LENGTH+1];
  
  /* build the info command in the log string */

  (void) sprintf(log_string,"%s %s %1d %1d %1d", 
                     GI_CMD, INFO_CMD_SHORT, 
                     at_x, at_y, col);

  /* write the log string to the log file */

  gi_log(log_string);
}
  
/*************************************************************************/
gi_log_delete(at_x, at_y)
  int at_x, at_y;
{
/* creates the move command that would delete
 * an object at the indicated display coordinates 
 */

  char log_string[MAX_CMD_LENGTH+1];
  
  /* build the delete command in the log string */

  (void) sprintf(log_string,"%s %s %1d %1d", 
                     GI_CMD, DEL_CMD_SHORT, 
                     at_x, at_y);

  /* write the log string to the log file */

  gi_log(log_string);
}

/*************************************************************************/
gi_log_move(from_x, from_y, to_x, to_y)
  int from_x, from_y, to_x, to_y;
{
/* creates the move command that would move
 * an object at the "from" display coordinates to
 * the "to" display coordinates
 */

  char log_string[MAX_CMD_LENGTH+1];
  
  /* build the move command in the log string */

  (void) sprintf(log_string,"%s %s %1d %1d %1d %1d", 
                     GI_CMD, MOVE_CMD_SHORT, 
                     from_x, from_y, to_x, to_y);

  /* write the log string to the log file */

  gi_log(log_string);
}

/*************************************************************************/
gi_log_text(ptr)
  struct txobj *ptr;
{
/* creates a text command that would create the
 * text object passed by the caller and then
 * writes that command to the current log file
 */
  struct saved_fonts *fptr;
  char *lptr, *tptr;
  char log_string[MAX_CMD_LENGTH+1];
  char buf[MAX_TEXT_LENGTH+1];
  
  /* initialize log_string to "gi t " */

  (void) sprintf(log_string,"%s %s ",GI_CMD, TEXT_CMD_SHORT);

  lptr = buf;

  /* copy text to a buffer surrouding it
     by double quotes and doubling any
     quotes actually in the string itself 
  */

  tptr = ptr->text_start;
  *lptr++ = DQUOTE;
  
  while (*tptr != '\0')
  {
    *lptr++ = *tptr;
    if (*tptr++ == DQUOTE)
      *lptr++ = DQUOTE;
  }
  *lptr++ = DQUOTE;
  *lptr = '\0';
  
  /* append the text to the log string */

  strcat(log_string,buf);

  /* append the display coordinates */

  (void) sprintf(buf," %1d %1d ",ptr->x_pos,ptr->y_pos);
  strcat(log_string,buf);

  /* if the specified font is not the default,
     search the chain of fonts for the name
     of the correct font and append it to
     the log string
  */

  if (ptr->text_font!=gi_dft_font)
  {
    /* search chain for fonts */
    for (fptr=gi_font_head; fptr!=NULL; fptr=fptr->next)
    {
      if (fptr->font_ptr==ptr->text_font) 
      {
        /* this is the one: append the font name */
        strcat(log_string,fptr->font_name);
        break;
      }
    }
  }
  
  gi_log(log_string);
}
  
/*************************************************************************/
gi_log_draw(ptr)
  struct drobj *ptr;
{
/* creates a draw command that would create the
 * drawn object passed by the caller and then
 * writes that command to the current log file
 */
  int i, *vptr;
  char log_string[MAX_CMD_LENGTH+1];
  char buf[MAX_TEXT_LENGTH+1];

  /* initialize string to "gi d" */

  (void) sprintf(log_string,"%s %s ",GI_CMD, DRAW_CMD_SHORT);

  /* append the number of vertices argument  or the  */
  /* indication that this is a bounding box */
  if (ptr->flag & BOUND_BOX)
    (void) sprintf(buf,"%1s",BOUND_BOX_STG);
  else
    (void) sprintf(buf,"%1d",ptr->num_vertices);

  strcat(log_string,buf);

  /* append all the vertex coordinates */

  for (i=0, vptr=ptr->vptr; i<2*ptr->num_vertices; i++, vptr++)
  {
    (void) sprintf(buf," %1d",*vptr);
    strcat(log_string,buf);
  }

  /* write the assembled string to the log file */

  gi_log(log_string);

}

