/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/************************************************************************
 * Graphics Interface
 * ------------------
 * This file contains miscellaneous routines used throughout the
 * Graphics Interface package of the Rochester Connections
 * Simulator.
 *
 * Note: because these routines are called from routines in
 *       other files, most of them should be declared static
 *       and beginning with the "gi_" prefix
 *
 * Kenton Lynne
 ***********************************************************************/

#include "macros.h"
#include "externs.h"
#include "control_panel.h"

/*************************************************************************/
gi_conv_coord(string,val) 
  char *string;
  int *val;
{
/* convertes the ASCII string passed by the caller into
 * an integer display coordinate, checking that the string
 * is indeed a number and that the coordinate it represents
 * is within range
 */

  if (gi_check_num(string)==ERROR)
    return(ERROR);
  else if(abs((*val=atoi(string))) > MAX_COORD)
    return(ERROR);
  return(OK);
}
    
/*************************************************************************/
gi_change_origin(x_displ, y_displ)
  int x_displ, y_displ;
{
/* updates the global values of the
 * origin and extent of the current window
 * and writes the updated value to the
 * control panel   
 */
  int new_origin_x, new_origin_y, new_extent_x, new_extent_y;
  
  /* calculate what new origin and extent coordinates would be */
  new_origin_x = gi_origin_x + x_displ;
  new_origin_y = gi_origin_y + y_displ;
  new_extent_x = gi_extent_x + x_displ;
  new_extent_y = gi_extent_y + y_displ;

  /* check that none are outside of display space */
  if (abs(new_origin_x) > MAX_COORD
  || abs(new_origin_y) > MAX_COORD
  || abs(new_extent_x) > MAX_COORD
  || abs(new_extent_y) > MAX_COORD)
  {
    gi_put_error("Cannot move display window outside display space");
  }
  else
  {
    /* update the official origin and extent dimensions */
    gi_origin_x = new_origin_x;
    gi_origin_y = new_origin_y;
    gi_extent_x = new_extent_x;
    gi_extent_y = new_extent_y;
  }

  /* write out the origin coordinates to the control_panel */
  gi_update_origin();
}

/*************************************************************************/
gi_do_steps(steps_left, upd_steps)
  int steps_left, upd_steps;
{
/* runs the simulator the indicated number
 * of steps, causing a reshow at the
 * update rate requested
 */

  while (steps_left > 0)
  {
    /* do a reshow at the end irregardless */
    if (upd_steps > steps_left)
      upd_steps = steps_left;

    /* have the simulator run the required number of simulation steps */
    gi_sim_go(upd_steps);

    /* indicate that all values and links may have changed */
    gi_reshow_flag |= ALL_VALS_NEEDED + ALL_LINKS_NEEDED;
    gi_off_display_ok = FALSE;

    /* reshow the screen */
    gi_reshow();
    steps_left -= upd_steps;
  }
}

/***********************************************************************/
gi_erase_grobj(ptr)
   struct grobj *ptr;
{
/* deletes a grobj put taking it off the chain it is
 * on and freeing its memory space
 */
  if (ptr!=NULL)
  {
    /* if this unit was the target, set target pointer off */
    if (ptr==gi_last_tagged)
       gi_last_tagged = NULL;
 
    /* if this unit is on the display and reshow all hasn't yet been set
       then erase the object now (avoiding need for possible reshow)
    */
    if (!(gi_reshow_flag & RESHOW_ALL) && gi_overlap_grobj(ptr))
      gi_display_grobj(ptr,PIX_CLR);
 
    /* remove object from chain and free storage */
    ptr->prev->next = ptr->next;
    ptr->next->prev = ptr->prev;
    free(ptr);
    return(OK);
  }
  else
    return(WARN);
}

/*************************************************************************/
gi_erase_drobj(ptr)
  struct drobj *ptr;
{
/* removes a drawn object from the chain and
 * returns the storage to the system
 */
  ptr->next->prev = ptr->prev;
  ptr->prev->next = ptr->next;

  free(ptr);
}
 
/*************************************************************************/
gi_erase_txobj(ptr)
  struct txobj *ptr;
{
/* removes a text object from the chain and
 * returns the storage to system
 */
  ptr->next->prev = ptr->prev;
  ptr->prev->next = ptr->next;

  free(ptr);
}

/*************************************************************************/
gi_save_vertex (ptr,x,y)
  struct drobj *ptr;
  int x, y;
{
/* saves the x and y coordinates of the vertex in
 * vertex_space and using the ptr to the graphic object
 * that the vertices are part of in case a new
 * piece of vertex space has to be allocated
 * Also updates the *_most coordinates
 */
  int *vertex_ptr;
  int i;
 
  if (gi_vertex_start > gi_last_vertex-1 || gi_vertex_start==NULL)
  {
    /* need to get more vertex space */
    if ((gi_vertex_start=(int *) malloc(VERTEX_SPACE_SIZE*sizeof(int)))==NULL)
    {
       gi_put_error("Cannot allocate more vertex space");
       return(ERROR);
    }
    gi_last_vertex = gi_vertex_start + VERTEX_SPACE_SIZE - 1;
    vertex_ptr = gi_vertex_start;
 
    /* copy any partial object to new area and update object pointer */
    for (i=0; i<ptr->num_vertices; i++)
    {
      *gi_vertex_start++ = *ptr->vptr++;
      *gi_vertex_start++ = *ptr->vptr++;
    }
    ptr->vptr = vertex_ptr;
  }  
 
  /* save the vertices away into vertex space */
  *gi_vertex_start++ = x;
  *gi_vertex_start++ = y;
 
  /* increment the number of vertices */
  ptr->num_vertices++;
 
  /* update the *_most values that describe the rectangle
     containing this object
  */
  if (x > ptr->right_most) ptr->right_most = x;
  else if (x < ptr->left_most) ptr->left_most = x;
 
  if (y > ptr->bottom_most) ptr->bottom_most = y;
  else if (y < ptr->top_most) ptr->top_most = y;

  return(OK);
}

/*************************************************************************/
gi_save_char(some_char,len)
  char some_char;
  int len;
{
/* saves the indicated character in string space at the
 * offset indicated by len.
 *
 * Note: if string_space becomes depleted, this routine
 *       will allocate more space and then recalculate
 *       the stringspace pointer to point into the
 *       new area and then move any partial string to it
 */

  char *char_ptr;
  int i;

  char_ptr = gi_string_start + len;

  /* if we are out of string space, get some more */
  if (char_ptr > gi_last_char || gi_last_char==NULL)
  {
    /* need to get more string space */
    if ((char_ptr= (char *) malloc(STRING_SPACE_SIZE))==NULL)
    {
       gi_put_error("Cannot allocate more string space");
       return /*(ERROR)*/;
    }

    /* copy any partial string to new area */
    for (i=0; i<len; i++) *char_ptr++ = *gi_string_start++;
    gi_string_start = char_ptr - len;
    gi_last_char = gi_string_start + STRING_SPACE_SIZE - 1;
  }
 
  /* write character to string space */
  *char_ptr = some_char;
}

/*************************************************************************/
gi_update_origin()
{
/* updates the origin coordinates on the
 * control panel
 */

  char origin_buf[2*MAX_NUM_LENGTH+2];

  (void) sprintf(origin_buf,"%1d %1d", gi_origin_x, gi_origin_y);
  panel_set(gi_origin_item, PANEL_LABEL_STRING, origin_buf, 0);
  panel_set(gi_reshow_o_item, PANEL_VALUE, origin_buf, 0);
}

/*************************************************************************/
gi_update_clock()
{
/* updates the clock field on the
 * control panel
 */

  char clock_buf[2*MAX_NUM_LENGTH+1];

  (void) sprintf(clock_buf,"%1d", gi_sim_get_clock());
  panel_set(gi_clock_item, PANEL_LABEL_STRING, clock_buf, 0);
}

/*************************************************************************/
gi_contains(bptr, top, bot, left, right)
  struct drobj *bptr;
  int top, bot, left, right;
{
/* given a rectangle in display space returns TRUE if
 * it is entirely contained within the bounding box
 * that is assumed to be pointed to by bptr
 * otherwise returns FALSE
 */
   if (bot > bptr->bottom_most
   ||  top < bptr->top_most
   ||  left < bptr->left_most
   ||  right > bptr->right_most)
     return(FALSE);
   else
     return(TRUE);
}

/*************************************************************************/
gi_bind_bind(bptr1, bptr2)
  struct drobj *bptr1, *bptr2;
{
/* given pointers to two bounding boxes, 
 * returns TRUE if the second overlaps the first
 * but is not contained entirely within it
 */
   /* first check if they overlap at all */
   if (bptr2->bottom_most < bptr1->top_most
   ||  bptr2->top_most > bptr1->bottom_most
   ||  bptr2->left_most > bptr1->right_most
   ||  bptr2->right_most < bptr1->left_most)
     return(FALSE);

   /* now check that the second isn't contained within the first */
   else if (gi_contains(bptr1,
                        bptr2->top_most,
                        bptr2->bottom_most,
                        bptr2->left_most,
                        bptr2->right_most))
     return(FALSE);

   /* finally check that the first isn't contained within the second */
   else if (gi_contains(bptr2,
                        bptr1->top_most,
                        bptr1->bottom_most,
                        bptr1->left_most,
                        bptr1->right_most))
     return(FALSE);

   /* passed all checks, they must overlap */
   else
     return(TRUE);
}

/*************************************************************************/
gi_overlap(top, bot, left, right)
  int top, bot, left, right;
{
/* given a rectangle in display space returns TRUE if
 * any part of it overlaps the current display space window
 * (that is origin and extent)
 * otherwise returns FALSE
 */
   if (bot < gi_origin_y
   ||  top > gi_extent_y
   ||  left > gi_extent_x
   ||  right < gi_origin_x)
     return(FALSE);
   else
     return(TRUE);
}

/*************************************************************************/
gi_overlap_txobj(ptr)
  struct txobj *ptr;
{
/* returns whether or not the indicated text object
 * overlaps with the current window
 * note : assumes origin_x, origin_y, extent_x and extent_y are correct
 *        (everything in display space coordinates)
 */

   return(gi_overlap(ptr->y_pos-ptr->text_font->pf_defaultsize.y,
                  ptr->y_pos,
                  ptr->x_pos,
                  ptr->x_pos+(ptr->text_font->pf_defaultsize.x*ptr->text_len)));
}

/*************************************************************************/
gi_overlap_grobj(ptr)
  struct grobj *ptr;
{
/* returns whether or not the indicated grobj 
 * overlaps with the current window
 * note : assumes origin_x, origin_y, extent_x and extent_y are correct
 *        (everything in display space coordinates)
 */

   return(gi_overlap(ptr->y_pos,
                  ptr->y_pos+ptr->gptr->size_y,
                  ptr->x_pos,
                  ptr->x_pos+ptr->gptr->size_x));
}

/*************************************************************************/
gi_overlap_drobj(ptr)
  struct drobj *ptr;
{
/* returns whether or not the indicated drawn object
 * overlaps with the current window
 * note : assumes origin_x, origin_y, extent_x and extent_y are correct
 *        (everything in display space coordinates)
 */

   return(gi_overlap(ptr->top_most,
                  ptr->bottom_most,
                  ptr->left_most,
                  ptr->right_most));
}

/*************************************************************************/
gi_display_drobj(ptr,op)
  struct drobj *ptr;
  int op;
{
/* display the line drawing indicated by the pointer
 * assumes that at least part of the object is within
 * the window
 */

  int *next_coord;
  int i, from_x, from_y, to_x, to_y;

  /* set up the first vertex */
  to_x = *ptr->vptr;
  to_y = *(ptr->vptr+1);
  next_coord = ptr->vptr + 2;

  /* loop through drawing lines between each new vertex */
  for (i=1; i<ptr->num_vertices; i++)
  {
    from_x = to_x;
    from_y = to_y;
    to_x = *next_coord++;
    to_y = *next_coord++;
    pw_vector (gi_gfx->gfx_pixwin,
	       from_x-gi_origin_x, from_y-gi_origin_y, 
               to_x-gi_origin_x, to_y-gi_origin_y,
	       op,1);
  }
}

/*************************************************************************/
gi_display_grobj(ptr,op)
  struct grobj *ptr;
  int op;
{
/* display the graphic object indicated by pointer
 * using the raster op requested
 * note: assumes that grobj is currently within
 *       the display window and that all fields are
 *       properly set
 */
     pw_write(gi_gfx->gfx_pixwin,
                ptr->x_pos-gi_origin_x,
                ptr->y_pos-gi_origin_y,
                ptr->gptr->size_x,
                ptr->gptr->size_y,
                op, 
                ptr->gptr->pix_ptr[ptr->icon_index], 
                0,
                0);
}

/*************************************************************************/
gi_display_txobj(ptr,op)
  struct txobj *ptr;
  int op;
{
/* display the text object indicated by pointer
 * using the raster op requested
 * note: assumes that txobj is currently within
 *       the display window
 */

   pw_text(gi_gfx->gfx_pixwin,
	   ptr->x_pos-gi_origin_x,
	   ptr->y_pos-gi_origin_y,
	   op,
	   ptr->text_font, 
	   ptr->text_start);
}

/*************************************************************************/
static insert_drobj(old_ptr, new_ptr)
  struct drobj *old_ptr, *new_ptr;
{
/* inserts the new text object in front of the old
 */

  /* set up the pointers in the object to be inserted */
  new_ptr->next = old_ptr->next;
  new_ptr->prev = old_ptr;

  /* now update the pointers of the old object */
  old_ptr->next = new_ptr;
  new_ptr->next->prev = new_ptr;
} 

/*************************************************************************/
static insert_txobj(old_ptr, new_ptr)
  struct txobj *old_ptr, *new_ptr;
{
/* inserts the new text object in front of the old
 */

  /* set up the pointers in the object to be inserted */
  new_ptr->next = old_ptr->next;
  new_ptr->prev = old_ptr;

  /* now update the pointers of the old object */
  old_ptr->next = new_ptr;
  new_ptr->next->prev = new_ptr;
} 

/*************************************************************************/
struct drobj *gi_remember_drobj(x1, y1, x2, y2, flag)
  int x1, y1, x2, y2;
  short flag;
{
/* given the first two vertices of a drawable object
 * builds and chains a new object filling in
 * all the relevant parameters
 */
 
  struct drobj *ptr;

  /* get some space for this object */
  if ((ptr=(struct drobj *) malloc(sizeof(struct drobj)))==NULL)
  {
    gi_put_error("Malloc out of space");
    return(NULL);
  }

  /* set up the default values */
  ptr->flag = flag;
  ptr->num_vertices = 0;
  ptr->vptr = gi_vertex_start;
  ptr->right_most = ptr->left_most = x1;
  ptr->top_most = ptr->bottom_most = y1;

  /* save the two vertices in vertex space */
  gi_save_vertex(ptr,x1,y1);
  gi_save_vertex(ptr,x2,y2);

  /* insert it at the front of the chain */
  insert_drobj(&gi_draw_head,ptr);
  return(ptr);
}

/*************************************************************************/
char *gi_save_string(string)
  char *string;
{
/* saves a string in string space 
 * and returns a pointer to it
 */
  int len;
  char *ptr;

  /* save the actual characters in stringspace */
  for (len=0; *string!='\0'; string++, len++) gi_save_char(*string,len);

  /* append the end-of-string character */
  gi_save_char('\0',len);

  /* set the pointer to the beginning of this string */
  ptr = gi_string_start;

  /* update the string start ptr to the next available character space */
  gi_string_start += len+1;

  return(ptr);
}

/*************************************************************************/
struct txobj *gi_remember_txobj(x_coord, y_coord, len, font)
  int x_coord, y_coord;
  short len;
  struct pixfont *font;
{
/* given the starting x and y coordinates of
 * a text string currently on the display 
 * (and saved in stringspace) this routine
 * builds a text object for that string and
 * links it into the text chain
 */
 
  struct txobj *ptr;

   /* get some space for this text object */
   if ((ptr=(struct txobj *) malloc(sizeof(struct txobj)))==NULL)
   {
     gi_put_error("Malloc out of space");
     return(NULL);
   }

   /* fill in the fields as requested */
   ptr->x_pos = x_coord;
   ptr->y_pos = y_coord;
   ptr->text_start = gi_string_start;
   ptr->text_len = len;
   ptr->text_font = font;
   ptr->flag = 0;

   /* insert into the chain */
   insert_txobj(&gi_text_head,ptr);

   /* now add the null and update the pointer into string space */
   gi_save_char('\0',len); 
   gi_string_start += (len + 1);
   return(ptr);
}
   
/*************************************************************************/
gi_move_drobj(ptr,x_offset, y_offset, erase_op, write_op)
  struct drobj *ptr;
  int x_offset, y_offset, erase_op, write_op;
{
/* moves the indicated drawn object by erasing it on
 * the screen, updating all its vertex positions by
 * the indicated x and y offsets and then redisplaying it
 * note: if erase_op is -1 (ERROR), the object will
 *       NOT be erased before moving it
 */
  int i;

  /* erase the old image */
  if (erase_op != -1) gi_display_drobj(ptr,erase_op);
 
  /* update the coordinates of the vertices */
  for (i=0; i<ptr->num_vertices; i++)
  {
    *(ptr->vptr+(2*i)) += x_offset;
    *(ptr->vptr+(2*i)+1) += y_offset;
  } 

  /* update the top/bottom/left/right fields */
  ptr->top_most += y_offset;
  ptr->bottom_most += y_offset;
  ptr->left_most += x_offset;
  ptr->right_most += x_offset;

  /* display object at new position */
  if (gi_overlap_drobj(ptr))
    gi_display_drobj(ptr,write_op);
}

/*************************************************************************/
gi_move_grobj(ptr,x_offset, y_offset, erase_flag)
  struct grobj *ptr;
  int x_offset, y_offset, erase_flag;
{
/* moves the indicated grobj object by erasing it on
 * the screen, updating its position on the display by
 * the indicated x and y offsets and then redisplaying it
 * (if erase_flag is FALSE (0), object is NOT erased 
 * NOTE: since this may be called via command interface
 *       it must check and rechain element as appropriate
 *       if it moves on/off the display
 */
  int op;
   
  /* erase previous position if on the display */
  if (erase_flag && ptr->flag & ON_DISPLAY && ptr->flag & DISPLAYED)
  {
    if (gi_mode!=MODE_LINK)
      gi_display_grobj(ptr,PIX_SRC ^ PIX_DST);
    else /* link mode */
    {
      /* if current target its reverse imaged */
      if (ptr==gi_cur_target_ptr)
        op = PIX_NOT(PIX_SRC ^ PIX_DST);
      else
        op = PIX_SRC ^ PIX_DST;
      gi_display_link(ptr,op);
    }
  }

 
  /* update new position */
  ptr->x_pos += x_offset;
  ptr->y_pos += y_offset;
 
  /* redisplay object at new position
     and update chain and flags if necessary
  */
  if (gi_overlap_grobj(ptr))
  { 
    /* if not prev on the display, set flags and move to display chain */
    if (!(ptr->flag & ON_DISPLAY))
    {
      ptr->flag |= ON_DISPLAY+DISPLAYED;
      gi_move_grobj_chain(&gi_marker,ptr,ptr);
    }
 
    /* display the object (possiblyin link mode) */
    if (gi_mode!=MODE_LINK)
      gi_display_grobj(ptr,PIX_SRC ^ PIX_DST);
    else
    {
      /* if current target: reverse image */
      if (ptr==gi_cur_target_ptr)
        op = PIX_NOT(PIX_SRC ^ PIX_DST);
      else
        op = PIX_SRC ^ PIX_DST;
      gi_display_link(ptr,op);
    }
  }

  else /* not on the display */
  {
    /* if had been on the display, reset flags and move to off_display */
    if (ptr->flag & ON_DISPLAY)
    {
      ptr->flag &= ~(ON_DISPLAY+DISPLAYED);
      gi_move_grobj_chain(&gi_off_display,ptr,ptr);
    }
  }
}

/*************************************************************************/
gi_move_txobj(ptr,x_offset, y_offset, erase_flag)
  struct txobj *ptr;
  int x_offset, y_offset, erase_flag;
{
/* moves the indicated text object by erasing it on
 * the screen, updating all its position on the display by
 * the indicated x and y offsets and then redisplaying it
 * note: if erase_flag is 0 (FALSE) object will NOT be
 *       erased prior to moving it
 */

  /* erase object at its old position */
  if (erase_flag) gi_display_txobj(ptr,PIX_SRC ^ PIX_DST);
 
  /* update its display coordinates */
  ptr->x_pos += x_offset;
  ptr->y_pos += y_offset;
 
  /* write out to new position */
  if (gi_overlap_txobj(ptr))
    gi_display_txobj(ptr,PIX_SRC ^ PIX_DST);
}

/*************************************************************************/
gi_move_bound(x_offset, y_offset, interactive_flag)
  int x_offset, y_offset, interactive_flag;
{
/* goes through all the object chains looking for
 * objects with the MOVING flag on, moves them
 * by the indicated offsets and then redisplays
 * them
 */

  struct grobj *gptr, *next_ptr;
  struct drobj *dptr;
  struct txobj *tptr;
  
  /* move all unit objects on the display chain */
  if (gi_marker.flag & MOVING)
  {
    gi_marker.flag &= ~MOVING;
    for (gptr=gi_marker.next; gptr!=&gi_marker; gptr=next_ptr)
    {
      /* need to save forward pointer for this object since
         it could be changed by call to gi_move_grobj
         */
      next_ptr = gptr->next;
      if (gptr->flag & MOVING)
      {
        gi_move_grobj(gptr,x_offset,y_offset,FALSE);
        gptr->flag &= ~MOVING;
      }
    }
  }

  /* move all unit objects on the off_display chain */
  if (gi_off_display.flag & MOVING)
  {
    gi_off_display.flag &= ~MOVING;
    for (gptr=gi_off_display.next; gptr!=&gi_off_display; gptr=next_ptr)
    {
      /* need to save forward pointer for this object since
         it could be changed by call to gi_move_grobj
         */
      next_ptr = gptr->next;
      if (gptr->flag & MOVING)
      {
        gi_move_grobj(gptr,x_offset,y_offset,FALSE);
        gptr->flag &= ~MOVING;
      }
    }
  }

  /* move all text objects */
  if (gi_text_head.flag & MOVING)
  {
    gi_text_head.flag &= ~MOVING;
    for (tptr=gi_text_head.next; tptr!=&gi_text_head; tptr=tptr->next)
    {
      if (tptr->flag & MOVING)
      {
        gi_move_txobj(tptr,x_offset,y_offset,FALSE);
        tptr->flag &= ~MOVING;
      }
    }
  }

  /* move all the drawn objects */
  if (gi_draw_head.flag & MOVING)
  {
    /* set off the header MOVING flag */
    gi_draw_head.flag &= ~MOVING;

    /* locate all MOVING objects on the chain */
    for (dptr=gi_draw_head.next; dptr!=&gi_draw_head; dptr=dptr->next)
    {
      if (dptr->flag & MOVING)
      {
        /* set off the MOVING flag */
        dptr->flag &= ~MOVING;
        
        /* check if bounding box and interactive move */
        if (dptr->flag & BOUND_BOX && interactive_flag)
        {
          /* just redisplay bound_box where it currently is */
          gi_display_drobj(dptr, PIX_SRC);
        }
        else   /* just move object to new position */
          gi_move_drobj(dptr,x_offset,y_offset,-1,PIX_SRC);
      }
    }
  }
}
    
/*************************************************************************/
gi_mark_bound(bptr, interactive_flag)
  struct drobj *bptr;  /* points to bounding box */
  int interactive_flag;
{
/* finds, marks as MOVING, and blanks (if displayed) all
 * objects (except bounding boxes) contained within the bounding box
 * note: this routine may call itself recursively
 * note: initeractive_flag should be on if this move is being
 *       done interactively
 */

  struct grobj *gptr;
  struct drobj *dptr;
  struct txobj *tptr;

  /* mark the bounding box as MOVING */
  bptr->flag |= MOVING;
  gi_draw_head.flag |= MOVING;
  
  /* blank out if not interactive mode */
  if (!interactive_flag && gi_overlap_drobj(bptr))
    gi_display_drobj(bptr,PIX_CLR);

  /* see if bounding box is on the display */
  if (gi_overlap_drobj(bptr))
  {
    /* search the on_display chain for unit objects */
    for (gptr=gi_marker.next; gptr!=&gi_marker; gptr=gptr->next)
    {
      /* check if unit is not yet marked and is within the bound box */
      if (!(gptr->flag & MOVING)
          && gi_contains(bptr,
                         gptr->y_pos,
                         gptr->y_pos+gptr->gptr->size_y,
                         gptr->x_pos,
                         gptr->x_pos+gptr->gptr->size_x))
      {
        /* mark as MOVING and blank it from display */
        gptr->flag |= MOVING;
        gi_display_grobj(gptr,PIX_SRC ^ PIX_DST);

        /* mark the chain as containing a MOVING object */
        gi_marker.flag |= MOVING;
      }
    }
  }

  /* if necessary, search off_display chain for unit objects */
  if (bptr->top_most < gi_origin_y
      || bptr->bottom_most > gi_extent_y
      || bptr->left_most < gi_origin_x
      || bptr->right_most > gi_extent_x)
  {
    for (gptr=gi_off_display.next; gptr!=&gi_off_display; gptr=gptr->next)
    {
      /* check if unit is not yet marked and is within the bound box */
      if (!(gptr->flag & MOVING)
          && gi_contains(bptr,
                         gptr->y_pos,
                         gptr->y_pos+gptr->gptr->size_y,
                         gptr->x_pos,
                         gptr->x_pos+gptr->gptr->size_x))
      {
        gptr->flag |= MOVING;
        gi_off_display.flag |= MOVING;

      }
    }
  }

  /* now search for text objects that are contained */
  for (tptr=gi_text_head.next; tptr!=&gi_text_head; tptr=tptr->next)
  {
    if (!(tptr->flag & MOVING)
        && gi_contains(bptr,
              tptr->y_pos-tptr->text_font->pf_defaultsize.y,
              tptr->y_pos,
              tptr->x_pos,
              tptr->x_pos+(tptr->text_font->pf_defaultsize.x*tptr->text_len)))
    {
      /* mark as MOVING and, if necessary, blank from display */
      tptr->flag |= MOVING;
      gi_text_head.flag |= MOVING;
      if (gi_overlap_txobj(tptr))
        gi_display_txobj(tptr,PIX_SRC ^ PIX_DST);
    }
  }

  /* now search for drawn objects that are contained */
  for (dptr=gi_draw_head.next; dptr!=&gi_draw_head; dptr=dptr->next)
  {
    /* check that object has not already been marked */
    if (!(dptr->flag & MOVING))
    {
      /* check if this is an overlapping bounding box */
      if ((dptr->flag & BOUND_BOX) && gi_bind_bind(bptr,dptr))
      {
        /* recursive call to mark objects within this new bounding box */
        gi_mark_bound(dptr,interactive_flag);
      }

      /* if not, just check for regular contained objects */
      else if (gi_contains(bptr,
                           dptr->top_most,
                           dptr->bottom_most,
                           dptr->left_most,
                           dptr->right_most))
      {
        /* mark as MOVING and blank out if on display (and not bound) */
        dptr->flag |= MOVING;
        if (gi_overlap_drobj(dptr) 
            && (!interactive_flag || (!(dptr->flag & BOUND_BOX))))
          gi_display_drobj(dptr,PIX_CLR);
      }
    }
  }
}
     
/*************************************************************************/
gi_marker_here(x,y)
  int x, y;
{
/* returns TRUE if the marker is currently at the
 * passed coordinates, else false
 */
  if (x >= gi_marker.x_pos 
  && x < (gi_marker.x_pos+UNITSIZE)
  && y >= gi_marker.y_pos 
  && y < (gi_marker.y_pos+UNITSIZE))
    return(TRUE);
  else
    return(FALSE);
}

/*************************************************************************/
struct drobj *gi_find_drobj(xpos, ypos)
  int xpos, ypos;
{
/* given display coordinates, attempts
 * to return the pointer to the drawn object near 
 * those coordinates --
 * otherwise returns NULL
 */

  struct drobj *ptr;
  int i;

  for (ptr=gi_draw_head.next; ptr!=&gi_draw_head; ptr=ptr->next)
  {
    /* check all vertices in this object so see if any are nearby */
    for (i=0; i<ptr->num_vertices; i++)
    {
      /* allow a tolerance of 3 pixels either way */
      if (abs(xpos-*(ptr->vptr+(i*2))) < 4
      && abs(ypos-*(ptr->vptr+(i*2)+1)) < 4)
      {
        return(ptr);
      }
    }
  }
  return(NULL);
}

/*************************************************************************/
struct txobj *gi_find_txobj(xpos, ypos)
  int xpos, ypos;
{
/* given display coordinates, attempts
 * to return the pointer to the text object near that
 * those coordinates otherwise returns NULL
 */

  struct txobj *ptr;

  for (ptr=gi_text_head.next; ptr!=&gi_text_head; ptr=ptr->next)
  {
    if (xpos >= ptr->x_pos
       && ypos <= ptr->y_pos
       && ypos>= ptr->y_pos - ptr->text_font->pf_defaultsize.y
       && xpos <= ptr->x_pos + (ptr->text_len*ptr->text_font->pf_defaultsize.x)
       )
       return(ptr);
  }
  return(NULL);
}

/*************************************************************************/
struct pixfont *gi_get_font(string)
  char *string;
{
/* set up cur_font to point to the current font as
 * indicated by the character string passed it
 * returns the pointer to the font
 */
 
  struct pixfont *new_font;
  struct saved_fonts *ptr;
  char buf[MAX_FONT_LENGTH+1];
 
  /* check if it is the default */
  if (string==NULL || strcmp(string,DFT_STRING)==0)
  {
    /* default font (loaded at initialization time) */
    return(gi_dft_font);
  }
  /* see if font already loaded in */
  for (ptr=gi_font_head; ptr!=NULL; ptr=ptr->next)
  {
    if (strcmp(string,ptr->font_name)==0)
    {
      return(ptr->font_ptr);
    }
  }  
 
  /* font is not currently loaded so attempt to load it now */
 
  buf[0] = '\0';  /* null out string buffer */
  if (string[0]!='/')
  {
    /* string does not begin with a "/", use default directory */
    strcpy(buf,FONT_DIR_STRING);
  }

  if ((new_font=pf_open(strcat(buf,string)))==NULL)
  {
    gi_put_error("Cannot open requested font");
    return(NULL);
  }

  /* save font information away : get space for new element  */
  if ((ptr=(struct saved_fonts *) malloc(sizeof(struct saved_fonts)))==NULL)
  {
    gi_put_error("malloc failure for get_font");
    return(NULL);
  }
 
  /* save string and font pointer in new element and add to chain */
  ptr->font_name = gi_save_string(string);
  ptr->font_ptr = new_font;
  ptr->next = gi_font_head;
  gi_font_head = ptr;
 
  /* return the new font just saved */
  return(new_font);
}

/*************************************************************************/
struct grobj *gi_find_xy(x,y)
{
/* given display coordinates returns the pointer to the
 * graphic object (currently on the display)
 * "near" those coordinates
 * if none, then returns NULL
 */
 
  struct grobj *ptr;

  ptr = gi_marker.next;
  while (ptr!=&gi_marker)
  {
    if (x >= ptr->x_pos 
    && x < (ptr->x_pos+ptr->gptr->size_x) 
    && y >= ptr->y_pos 
    && y < (ptr->y_pos+ptr->gptr->size_y) )
    {
      return(ptr);
    }
    ptr=ptr->next;
  } 

   return(NULL);
}

/*************************************************************************/
struct grobj *gi_find_grobj(x,y)
  int x, y;
{
/* finds the grobj at position x y on
 * either the display and off_display chains
 */

  struct grobj *ptr;

  /* look for it first on the display */
  if ((ptr=gi_find_xy(x,y))==NULL)
  {
    /* if not found on the display, search off_display chain */
    ptr = gi_off_display.next;
    while (ptr!=&gi_off_display)
    {
      if (x >= ptr->x_pos && 
          x < (ptr->x_pos+ptr->gptr->size_x) &&
          y >= ptr->y_pos &&
          y < (ptr->y_pos+ptr->gptr->size_y) )
        {
          return(ptr);
        }
        ptr=ptr->next;

     } 
     /* didn't find it anywhere, return NULL */
     ptr = NULL;
  }
  return(ptr);
}

/*************************************************************************/
static update_marker()
{
/* updates the position of the marker
 * on the control panel
 */
  static char markx[MAX_NUM_LENGTH+1], marky[MAX_NUM_LENGTH+1];

  (void) sprintf(markx,"%1d",gi_marker.x_pos);
  (void) sprintf(marky,"%1d",gi_marker.y_pos);
  panel_set_value(gi_mitem[WHEREX_ITEM],markx);
  panel_set_value(gi_mitem[WHEREY_ITEM],marky);
}

/*************************************************************************/
gi_move_marker(x,y)
   int x, y;
{
/* move marker from its current position to
 * the requested display space coordinates
 */

   /* clear current area if currently on display and being shown */
   if (!(gi_marker.flag & NOSHOW) && gi_overlap_grobj(&gi_marker))
     gi_display_grobj(&gi_marker,PIX_SRC ^ PIX_DST);

   /* set up new coordinates for marker */
   gi_marker.x_pos = x;
   gi_marker.y_pos = y;

   /* write new position to control panel */
   update_marker();
  
   /* write out marker to new position if on display and showable */
   if (!(gi_marker.flag & NOSHOW) && gi_overlap_grobj(&gi_marker))
     gi_display_grobj(&gi_marker,PIX_SRC ^ PIX_DST);
}

/*************************************************************************/
char *gi_strip(string) 
  char *string;
{
/* given a string, it strips off
 * any leading and trailing blanks or other
 * and returns the original pointer to that string
 * note: allows for a null pointer
 */
  int i;
  char *ptr;

  if (string)
  {
    /* first get rid of leading blanks */
    while (*string==BLANK) string++;
 
    /* if anything left, get rid of trailing blanks */
    if (i=strlen(string))
    {
      /* set ptr to end of string */
      ptr = string + i;
      /* replace blanks with nulls */
      while (*(++ptr)==BLANK) *ptr = '\0';
    }
  }
  return(string);
}

/*************************************************************************/
gi_check_num(string)
  char *string;
{
/* checks if a character string
 * contains only (signed or unsigned) integers
 */
  int signok=TRUE; 

  /* if null string, return a warning */
  if (string==NULL || *string=='\0') return(WARN);

  /* search string for signed number */
  for (; *string!='\0'; string++)
  {
     /* allow a sign in only the first position */
     if (signok)
     {
       if (*string!='-' && *string!='+' &&
          (*string<'0' || *string>'9') )
           return(ERROR);
       else
         signok = FALSE;
     }
     /* after first position, allow only numbers */
     else
     {
       if (*string<'0' || *string>'9') 
         return(ERROR);
     }
  }
  return(OK);
}

