/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains routines for setting up the control panel
 * and procedures called from that panel for the Graphics Interface
 * for the Rochester Connectionist Simulator.
 *
 * Note: variables defined here that can be accessed globally
 *       are externed in "control_panel.h"
 *
 * Kenton Lynne
 ********************************************************************/

#include "macros.h"
#include "externs.h"
#include "panel.h"
#include "cmd_panel.h"

/* get the icons for for the default icon images */
#include "dft.icons"

/* the following arrays hold the default aspect icons */

static struct pixrect *square[DFT_NRANGES+1]={
      &sq0, &sq1, &sq2, &sq3, &sq4,
      &sq5, &sq6, &sq7, &sq8,
      &sq9, &sq10, &sq11, &sq12,
      &sq13, &sq14, &sq15, &sq16};

static struct pixrect *triangle[DFT_NRANGES+1]={
      &tr0, &tr1, &tr2, &tr3, &tr4,
      &tr5, &tr6, &tr7, &tr8,
      &tr9, &tr10, &tr11, &tr12,
      &tr13, &tr14, &tr15, &tr16};
 
static struct pixrect *circle[DFT_NRANGES+1]={
      &cr0, &cr1, &cr2, &cr3, &cr4,
      &cr5, &cr6, &cr7, &cr8,
      &cr9, &cr10, &cr11, &cr12,
      &cr13, &cr14, &cr15, &cr16};
 
static struct pixrect *pentagon[DFT_NRANGES+1]={
      &pt0, &pt1, &pt2, &pt3, &pt4,
      &pt5, &pt6, &pt7, &pt8,
      &pt9, &pt10, &pt11, &pt12,
      &pt13, &pt14, &pt15, &pt16};
 
static struct pixrect *diamond[DFT_NRANGES+1]={
      &dm0, &dm1, &dm2, &dm3, &dm4,
      &dm5, &dm6, &dm7, &dm8,
      &dm9, &dm10, &dm11, &dm12,
      &dm13, &dm14, &dm15, &dm16};
 
static struct pixrect *grey[DFT_NRANGES+1]={
      &gy0, &gy1, &gy2, &gy3, &gy4,
      &gy5, &gy6, &gy7, &gy8,
      &gy9, &gy10, &gy11, &gy12,
      &gy13, &gy14, &gy15, &gy16};
 
/* set up the icon structures for the default icons */
struct gricons
     gi_square   = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, square},
     gi_circle   = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, circle},
     gi_triangle = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, triangle},
     gi_pentagon = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, pentagon},
     gi_diamond  = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, diamond},
     gi_grey     = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, grey};

/* static procedures in this file that are forward referenced */
static Panel_setting                
       but_proc(),
       font_proc();

static what_proc(),
       how_proc(),
       where_proc(),
       show_proc(),
       change_proc(),
       erase_proc(),
       ldir_proc(),
       lhow_proc(),
       dtype_proc(),
       custlog_proc(),
       go_proc(),
       dump_proc(),
       reshow_proc(),
       target_proc(),
       quit_proc();

Panel gi_control_panel;
  
/* panel items that can be globally accessed */
Panel_item 
  gi_mitem[MAIN_ITEM_LAST+1],
  gi_litem[LINK_ITEM_LAST+1],
  gi_titem[TEXT_ITEM_LAST+1],
  gi_ditem[DRAW_ITEM_LAST+1],
  gi_citem[CUST_ITEM_LAST+1],
  gi_clock_item,
  gi_origin_item,
  gi_reshow_o_item;

/* panel items only accessed here */
static Panel_item 
  go_steps_item,
  go_update_item,
  dump_image_item;

/*****************************************************************/
gi_make_control_panel()
{
/* initial definition and set up of the control panel 
 */
    struct toolsw *control_panel_sw;


     /* make the control panel itself */
     if ((control_panel_sw = panel_create(gi_tool,
            PANEL_HEIGHT, CNTL_HEIGHT,
            PANEL_WIDTH, CNTL_WIDTH,
            PANEL_LABEL_BOLD, TRUE,
            0)) == NULL)
     {
       fputs("Can't create control panel\n", stderr);
       exit(1);
     }
     gi_control_panel = control_panel_sw->ts_data;

     /* define all the panel items */

     /* items only displayed during MAIN mode */
     gi_mitem[WHO_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_LABEL_STRING, "WHO",
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 10,
                  PANEL_VALUE_X, TAB1,
                  PANEL_VALUE_Y, 10,
                  PANEL_VALUE, "0",
                  0);
     gi_mitem[HOWMANY_ITEM] = panel_create_item
                (gi_control_panel, PANEL_MESSAGE,
                 PANEL_LABEL_STRING, "HOW",
                 PANEL_LABEL_X, 5,
                 PANEL_LABEL_Y, 40,
                 0);
     gi_mitem[HOWMANY_ITEM2] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE, "all",
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH,
                  PANEL_LABEL_STRING, "MANY",
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 53,
                  PANEL_VALUE_X, TAB1,
                  PANEL_VALUE_Y, 45,
                  0);
     gi_mitem[WHAT_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_CHOICE,
                  PANEL_LABEL_STRING, "WHAT",
                  PANEL_FEEDBACK, PANEL_INVERTED,
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 80,
                  PANEL_CHOICE_STRINGS, "Potential",
                                        "Output",
                                        "State",
                                        "Data",
                                        "Link/in",
                                        "Link/out",
                                        0,
                  PANEL_CHOICE_XS, TAB1,TAB1+100,TAB1,TAB1+100,TAB1,TAB1+100,0,
                  PANEL_CHOICE_YS, 80, 80, 95, 95, 110, 110, 0,
                  PANEL_NOTIFY_PROC, what_proc,
                  0);
     gi_mitem[LRANGE_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, "from:",
                  PANEL_LABEL_X, TAB1,
                  PANEL_LABEL_Y, 125,
                  PANEL_VALUE, LRANGE_POT,
                  0);
     gi_mitem[HRANGE_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, "to:",
                  PANEL_LABEL_X, TAB1+100,
                  PANEL_LABEL_Y, 125,
                  PANEL_VALUE, HRANGE_POT,
                  0);
     gi_mitem[TARGET_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, "target:",
                  PANEL_LABEL_X, TAB1,
                  PANEL_LABEL_Y, 140,
                  PANEL_VALUE, "",
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_mitem[HOW_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_CHOICE,
                  PANEL_LABEL_STRING, "HOW",
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 170,
                  PANEL_FEEDBACK, PANEL_MARKED,
                  PANEL_CHOICE_IMAGES, square[DFT_NRANGES],
                                       circle[DFT_NRANGES],
                                       triangle[DFT_NRANGES],
                                       pentagon[DFT_NRANGES],
                                       diamond[DFT_NRANGES],
                                       grey[DFT_NRANGES],
                                       &your_icon,
                                       0,
                  PANEL_CHOICE_XS, 
                     TAB1,TAB1+30,TAB1+60,TAB1+90,TAB1+120,TAB1+150,TAB1,0,
                  PANEL_CHOICE_YS, 
                     170,170,170,170,170, 170,195,0,
                  PANEL_MARK_XS,
                     TAB1,TAB1+30,TAB1+60,TAB1+90,TAB1+120,TAB1+150,TAB1,0,
                  PANEL_MARK_YS, 
                     188,188,188,188,188, 188,213,0,
                  PANEL_MARK_IMAGES, &underline, 0,
                  PANEL_NOMARK_IMAGES, 0,
                  PANEL_NOTIFY_PROC, how_proc,
                  0);
     gi_mitem[SUBIMAGE_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_LABEL_X, TAB1+20,
                  PANEL_LABEL_Y, 200,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, "name:",
                  PANEL_VALUE, "",
                  0);
     gi_mitem[WHERE_ITEM] = panel_create_item
                (gi_control_panel, PANEL_MESSAGE,
                 PANEL_LABEL_STRING, "WHERE",
                 PANEL_LABEL_X, 5,
                 PANEL_LABEL_Y, 225,
                 0);
     gi_mitem[WHEREX_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH,
                  PANEL_LABEL_STRING, "start x:",
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_X, TAB1,
                  PANEL_LABEL_Y, 225,
                  PANEL_VALUE, "5",
                  PANEL_NOTIFY_PROC,where_proc,
                  0);
     gi_mitem[WHEREY_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH,
                  PANEL_LABEL_STRING, "y:",
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_X, TAB1+110,
                  PANEL_LABEL_Y, 225,
                  PANEL_VALUE, "5",
                  PANEL_NOTIFY_PROC,where_proc,
                  0);
     gi_mitem[SPACEX_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH-2,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH-2,
                  PANEL_LABEL_STRING, "space x:",
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_X, TAB1,
                  PANEL_LABEL_Y, 245,
                  PANEL_VALUE, "5",
                  0);
     gi_mitem[SPACEY_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH-2,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH-2,
                  PANEL_LABEL_STRING, "y:",
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_X, TAB1+110,
                  PANEL_LABEL_Y, 245,
                  PANEL_VALUE, "5",
                  0);
     gi_mitem[NUMCOLS_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH-3,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH-3,
                  PANEL_LABEL_STRING, "units per row:",
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_X, TAB1,
                  PANEL_LABEL_Y, 265,
                  PANEL_VALUE, "max",
                  0);
     gi_mitem[SHOW_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_BUTTON,
                  PANEL_LABEL_IMAGE, panel_button_image
                                        (gi_control_panel,
                                         "SHOW",     
                                         8,
                                         NULL),          
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 310,
                  PANEL_NOTIFY_PROC, show_proc,
                  0);
     gi_mitem[CHANGE_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_BUTTON,
                  PANEL_LABEL_IMAGE, panel_button_image
                                        (gi_control_panel,
                                         "CHANGE",     
                                         8,
                                         NULL),          
                  PANEL_LABEL_X, 85,
                  PANEL_LABEL_Y, 310,
                  PANEL_NOTIFY_PROC, change_proc,
                  0);
     gi_mitem[ERASE_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_BUTTON,
                  PANEL_LABEL_IMAGE, panel_button_image
                                        (gi_control_panel,
                                         "ERASE",     
                                         8,
                                         NULL),          
                  PANEL_LABEL_X, 165,
                  PANEL_LABEL_Y, 310,
                  PANEL_NOTIFY_PROC, erase_proc,
                  0);

     /* items displayed only during LINK mode */

     gi_litem[LSITE_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_LABEL_STRING, "TARGET",
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 10,
                  PANEL_VALUE_X, TAB1+5,
                  PANEL_VALUE_Y, 10,
                  PANEL_VALUE, ANY_SITE,
                  PANEL_NOTIFY_PROC, target_proc,
                  PANEL_SHOW_ITEM, FALSE,
                  0);

     gi_litem[LHOW_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_CHOICE,
                  PANEL_LABEL_STRING, "HOW",
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 40,
                  PANEL_FEEDBACK, PANEL_MARKED,
                  PANEL_CHOICE_IMAGES, &same_icon,
                                       square[DFT_NRANGES],
                                       circle[DFT_NRANGES],
                                       triangle[DFT_NRANGES],
                                       pentagon[DFT_NRANGES],
                                       diamond[DFT_NRANGES],
                                       0,
                  PANEL_CHOICE_XS, 
                     TAB1+5,TAB1+35,TAB1+65,TAB1+95,TAB1+125,TAB1+155,0,
                  PANEL_CHOICE_YS, 45, 0,
                  PANEL_MARK_XS,
                     TAB1+5,TAB1+35,TAB1+65,TAB1+95,TAB1+125,TAB1+155,0,
                  PANEL_MARK_YS, 63, 0,
                  PANEL_MARK_IMAGES, &underline, 0,
                  PANEL_NOMARK_IMAGES, 0,
                  PANEL_NOTIFY_PROC, lhow_proc,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     
     gi_litem[LDIR_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_CHOICE,
                  PANEL_LABEL_STRING, "DIRECTION",
                  PANEL_FEEDBACK, PANEL_INVERTED,
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 100,    
                  PANEL_CHOICE_STRINGS, " Link/in ",
                                        " Link/out ",
                                        0,
                  PANEL_CHOICE_XS, 80, 150, 0,
                  PANEL_CHOICE_YS, 100, 0,
                  PANEL_NOTIFY_PROC, ldir_proc,
                  PANEL_SHOW_ITEM, FALSE,
                  0);

     gi_litem[LLRANGE_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, "from:",
                  PANEL_LABEL_X, TAB1,
                  PANEL_LABEL_Y, 75,
                  PANEL_VALUE, LRANGE_LINKIN,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_litem[LHRANGE_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_NUM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_NUM_LENGTH,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, "to:",
                  PANEL_LABEL_X, TAB1+100,
                  PANEL_LABEL_Y, 75,
                  PANEL_VALUE, HRANGE_LINKIN,
                  PANEL_SHOW_ITEM, FALSE,
                  0);

    /* items displayed only during TEXT mode */
     gi_titem[TFONT1_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_LABEL_STRING, "FONT",
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 10,
                  PANEL_VALUE_X, TAB1,
                  PANEL_VALUE_Y, 10,
                  PANEL_VALUE, DFT_STRING,
                  PANEL_NOTIFY_PROC, font_proc,
                  PANEL_NOTIFY_LEVEL, PANEL_ALL,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_titem[TFONT2_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE_X, TAB1,
                  PANEL_VALUE_Y, 30,
                  PANEL_VALUE, "",
                  PANEL_NOTIFY_PROC, font_proc,
                  PANEL_NOTIFY_LEVEL, PANEL_ALL,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_titem[TFONT3_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, 
                        MAX_FONT_LENGTH-2*MAX_ITEM_LENGTH,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE_X, TAB1,
                  PANEL_VALUE_Y, 50,
                  PANEL_VALUE, "",
                  PANEL_SHOW_ITEM, FALSE,
                  0);

     /* items displayed only during DRAW mode */
     gi_ditem[DTYPE_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_CHOICE,
                  PANEL_LABEL_STRING, "TYPE",
                  PANEL_FEEDBACK, PANEL_INVERTED,
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 10,    
                  PANEL_CHOICE_STRINGS, " Lines ",
                                        " Boxes ",
                                        " Bounding Box ",
                                        0,
                  PANEL_CHOICE_XS, TAB1, 0,
                  PANEL_CHOICE_YS, 10, 30, 50, 0,
                  PANEL_NOTIFY_PROC, dtype_proc,
                  PANEL_SHOW_ITEM, FALSE,
                  0);

    /* items only displayed during CUSTOM mode */
     /* left button down */
     gi_citem[CLBDN1_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &lbut_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT1Y+OFF_MOUSE,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT1Y+OFF_LINE1,
                  PANEL_VALUE_STORED_LENGTH, CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, NULL_STRING,
                  PANEL_NOTIFY_PROC, but_proc,
                  PANEL_NOTIFY_LEVEL, PANEL_ALL,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_citem[CLBDN2_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &butdn_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT1Y+OFF_MOUSE+16,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT1Y+OFF_LINE2,
                  PANEL_VALUE_STORED_LENGTH, MAX_CUST_TEXT_LEN-CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, "",
                  PANEL_SHOW_ITEM, FALSE,
                  0);

     /* middle button down */
     gi_citem[CMBDN1_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &mbut_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT2Y+OFF_MOUSE,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT2Y+OFF_LINE1,
                  PANEL_VALUE_STORED_LENGTH, CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, NULL_STRING,
                  PANEL_NOTIFY_PROC, but_proc,
                  PANEL_NOTIFY_LEVEL, PANEL_ALL,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_citem[CMBDN2_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &butdn_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT2Y+OFF_MOUSE+16,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT2Y+OFF_LINE2,
                  PANEL_VALUE_STORED_LENGTH, MAX_CUST_TEXT_LEN-CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, "",
                  PANEL_SHOW_ITEM, FALSE,
                  0);

     /* right button down */
     gi_citem[CRBDN1_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &rbut_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT3Y+OFF_MOUSE,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT3Y+OFF_LINE1,
                  PANEL_VALUE_STORED_LENGTH, CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, NULL_STRING,
                  PANEL_NOTIFY_PROC, but_proc,
                  PANEL_NOTIFY_LEVEL, PANEL_ALL,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_citem[CRBDN2_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &butdn_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT3Y+OFF_MOUSE+16,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT3Y+OFF_LINE2,
                  PANEL_VALUE_STORED_LENGTH, MAX_CUST_TEXT_LEN-CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, "",
                  PANEL_SHOW_ITEM, FALSE,
                  0);

     /* left button up */
     gi_citem[CLBUP1_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &lbut_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT4Y+OFF_MOUSE,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT4Y+OFF_LINE1,
                  PANEL_VALUE_STORED_LENGTH, CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, NULL_STRING,
                  PANEL_NOTIFY_PROC, but_proc,
                  PANEL_NOTIFY_LEVEL, PANEL_ALL,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_citem[CLBUP2_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &butup_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT4Y+OFF_MOUSE+16,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT4Y+OFF_LINE2,
                  PANEL_VALUE_STORED_LENGTH, MAX_CUST_TEXT_LEN-CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, "",
                  PANEL_SHOW_ITEM, FALSE,
                  0);

     /* middle button up */
     gi_citem[CMBUP1_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &mbut_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT5Y+OFF_MOUSE,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT5Y+OFF_LINE1,
                  PANEL_VALUE_STORED_LENGTH, CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, NULL_STRING,
                  PANEL_NOTIFY_PROC, but_proc,
                  PANEL_NOTIFY_LEVEL, PANEL_ALL,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_citem[CMBUP2_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &butup_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT5Y+OFF_MOUSE+16,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT5Y+OFF_LINE2,
                  PANEL_VALUE_STORED_LENGTH, MAX_CUST_TEXT_LEN-CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, "",
                  PANEL_SHOW_ITEM, FALSE,
                  0);

     /* right button up */
     gi_citem[CRBUP1_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &rbut_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT6Y+OFF_MOUSE,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT6Y+OFF_LINE1,
                  PANEL_VALUE_STORED_LENGTH, CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, NULL_STRING,
                  PANEL_NOTIFY_PROC, but_proc,
                  PANEL_NOTIFY_LEVEL, PANEL_ALL,
                  PANEL_SHOW_ITEM, FALSE,
                  0);
     gi_citem[CRBUP2_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_IMAGE, &butup_icon,
                  PANEL_LABEL_X, BUTX,
                  PANEL_LABEL_Y, BUT6Y+OFF_MOUSE+16,
		  PANEL_VALUE_X, LINEX,
		  PANEL_VALUE_Y, BUT6Y+OFF_LINE2,
                  PANEL_VALUE_STORED_LENGTH, MAX_CUST_TEXT_LEN-CUST1_LEN,
                  PANEL_VALUE_DISPLAY_LENGTH, CUST1_LEN,
                  PANEL_VALUE, "",
                  PANEL_SHOW_ITEM, FALSE,
                  0);

    gi_citem[CLOG_ITEM] = panel_create_item
                 (gi_control_panel, PANEL_BUTTON,
                  PANEL_LABEL_IMAGE, panel_button_image
                                        (gi_control_panel,
                                         "LOG DEFINITIONS",     
                                         18,
                                         NULL),          
                  PANEL_LABEL_X, TAB1,
                  PANEL_LABEL_Y, BUT6Y+60,
                  PANEL_NOTIFY_PROC, custlog_proc,
                  PANEL_SHOW_ITEM, FALSE,
                  0);

    /* items displayed during all modes */
    panel_create_item
                (gi_control_panel, PANEL_MESSAGE,
                 PANEL_LABEL_STRING, 
                 "---------------------------------",
                 PANEL_LABEL_X, 2,
                 PANEL_LABEL_Y, 335,
                 0);

    panel_create_item
                (gi_control_panel, PANEL_MESSAGE,
                 PANEL_LABEL_STRING, 
                 "Clock=",
                 PANEL_LABEL_X, 5,
                 PANEL_LABEL_Y, 350,
                 0);

    gi_clock_item = panel_create_item
                (gi_control_panel, PANEL_MESSAGE,
                 PANEL_LABEL_STRING, 
                 "0",
                 PANEL_LABEL_X, PANEL_CU(6)+5,
                 PANEL_LABEL_Y, 350,
                 0);

    panel_create_item
                (gi_control_panel, PANEL_MESSAGE,
                 PANEL_LABEL_STRING, 
                 "Origin=",
                 PANEL_LABEL_X, TAB1+50,
                 PANEL_LABEL_Y, 350,
                 0);

    gi_origin_item = panel_create_item
                (gi_control_panel, PANEL_MESSAGE,
                 PANEL_LABEL_STRING, 
                 "0 0",
                 PANEL_LABEL_X, PANEL_CU(7)+TAB1+50,
                 PANEL_LABEL_Y, 350,
                 0);
    
    panel_create_item
                (gi_control_panel, PANEL_MESSAGE,
                 PANEL_LABEL_STRING, 
                 "---------------------------------",
                 PANEL_LABEL_X, 2,
                 PANEL_LABEL_Y, 365,
                 0);

    panel_create_item
                 (gi_control_panel, PANEL_BUTTON,
                  PANEL_LABEL_IMAGE, panel_button_image
                                        (gi_control_panel,
                                         "GO",     
                                         6,
                                         NULL),          
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 385,
                  PANEL_NOTIFY_PROC, go_proc,
                  0);
     go_steps_item = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, 5,
                  PANEL_VALUE_DISPLAY_LENGTH, 5,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, "number steps:",
                  PANEL_LABEL_X, 70,
                  PANEL_LABEL_Y, 380,
                  PANEL_VALUE, "1",
                  0);
     go_update_item = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, 4,
                  PANEL_VALUE_DISPLAY_LENGTH, 4,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, "update steps:",
                  PANEL_LABEL_X, 70,
                  PANEL_LABEL_Y, 395,
                  PANEL_VALUE, "1",
                  0);
    panel_create_item
                 (gi_control_panel, PANEL_BUTTON,
                  PANEL_LABEL_IMAGE, panel_button_image
                                        (gi_control_panel,
                                         "DUMP",     
                                         6,
                                         NULL),          
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 425,
                  PANEL_NOTIFY_PROC, dump_proc,
                  0);
     dump_image_item = panel_create_item
                 (gi_control_panel, PANEL_TEXT,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, ":",
                  PANEL_LABEL_X, 70,
                  PANEL_LABEL_Y, 425,
                  PANEL_VALUE_DISPLAY_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE_STORED_LENGTH, MAX_ITEM_LENGTH,
                  PANEL_VALUE, DFT_IMAGE_FILE_NAME,
                  0);
    panel_create_item
                 (gi_control_panel, PANEL_BUTTON,
                  PANEL_LABEL_IMAGE, panel_button_image
                                        (gi_control_panel,
                                         "RESHOW",     
                                         6,
                                         NULL),          
                  PANEL_LABEL_X, 5,
                  PANEL_LABEL_Y, 465,
                  PANEL_NOTIFY_PROC, reshow_proc,
                  0);
    gi_reshow_o_item = panel_create_item
                  (gi_control_panel, PANEL_TEXT,
                  PANEL_VALUE_STORED_LENGTH, 12,
                  PANEL_VALUE_DISPLAY_LENGTH, 12,
                  PANEL_LABEL_BOLD, FALSE,
                  PANEL_LABEL_STRING, ":",
                  PANEL_LABEL_X, 70,
                  PANEL_LABEL_Y, 465,
                  PANEL_VALUE, "0 0",
                  0);     
    panel_create_item
                 (gi_control_panel, PANEL_BUTTON,
                  PANEL_LABEL_IMAGE, panel_button_image
                                        (gi_control_panel,
                                         "QUIT",     
                                         4,
                                         NULL),          
                  PANEL_LABEL_X, 190,
                  PANEL_LABEL_Y, 465,
                  PANEL_NOTIFY_PROC, quit_proc,
                  0);
}

/*****************************************************************/
/*ARGSUSED*/
static quit_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/* called when the QUIT button is pressed
 */
  tool_done(gi_tool);
}


/*****************************************************************/
/*ARGSUSED*/
static how_proc(item, value, event)
   Panel_item item;
   int value;
   struct inputevent *event;
{
/* called when the user selects an default icon
 * on the HOW prompt (main mode)
 */

  /* set up the global current value of the icon selected */
  gi_cur_image_value = value + 1;

  /* if user has selected own icons, move caret to field */
  if (gi_cur_image_value > NUM_STD_ICONS)
  { 
    panel_set(gi_control_panel, PANEL_CARET_ITEM, gi_mitem[SUBIMAGE_ITEM], 0);
  }
}

/*****************************************************************/
/*ARGSUSED*/
static lhow_proc(item, value, event)
   Panel_item item;
   int value;
   struct inputevent *event;
{
/* called when the user selects an icon on the
 * HOW prompt (link mode)
 */

  /* if we are switching from the user's "same" icons to
     one of the defaults or vice versa, set the 
     gi_new_link_parms flag so that display_grobj_link
     will calculate a new link_index for each unit
  */
   if ((!value && gi_cur_link_image)
   || (!gi_cur_link_image && value))
     gi_new_link_parms = TRUE;

  /* based on the user's selection, set up the global link icon */
  switch (value)  
  {
    case 0: gi_cur_link_image = NULL;
            break;

    case 1: gi_cur_link_image = square;
            break;
   
    case 2: gi_cur_link_image = circle;
            break;
   
    case 3: gi_cur_link_image = triangle;
            break;
   
    case 4: gi_cur_link_image = pentagon;
            break;
   
    case 5: gi_cur_link_image = diamond;
            break;
   }

   /* force a reshow of all displayed units */
   gi_reshow_flag |= RESHOW_ALL;
   gi_reshow();
}

/*****************************************************************/
/*ARGSUSED*/
static dtype_proc(item, value, event)
   Panel_item item;
   int value;
   struct inputevent *event;
{
/* called when the user selects LINES or BOXES on
 * the TYPE prompt (draw mode)
 */
 
  gi_draw_type = value;
}

/*****************************************************************/
/*ARGSUSED*/
static ldir_proc(item, value, event)
   Panel_item item;
   int value;
   struct inputevent *event;
{
/* called when the user selects LINK/IN or LINK/OUT on
 * the direction prompt (link mode)
 */
   gi_link_direction = value;
}

/****************************************************************/
/*ARGSUSED*/
static what_proc(item, value, event)
   Panel_item item;
   int value;
   struct inputevent *event;
{
/* called when the user selects the aspect to be displayed
 * on the WHAT prompt (main mode)
 */

  int show=FALSE; 
  char *lrange, *hrange;

  /* set up the global variable indicating current WHAT choice */
  switch(value)
  {
    case 0: gi_cur_unit_what = POT;
            lrange = LRANGE_POT;
            hrange = HRANGE_POT;
            break;

    case 1: gi_cur_unit_what = OUTPUT;
            lrange = LRANGE_OUT;
            hrange = HRANGE_OUT;
            break;

    case 2: gi_cur_unit_what = STATE;
            lrange = LRANGE_STATE;
            hrange = HRANGE_STATE;
            break;

    case 3: gi_cur_unit_what = DATA;
            lrange = LRANGE_DATA;
            hrange = HRANGE_DATA;
            break;

    case 4: gi_cur_unit_what = LINKIN;
            lrange = LRANGE_LINKIN;
            hrange = HRANGE_LINKIN;
            show = TRUE;
            break;

    case 5: gi_cur_unit_what = LINKOUT;
            lrange = LRANGE_LINKOUT;
            hrange = HRANGE_LINKOUT;
            show = TRUE;
            break;
  }

  /* display the target prompt if LINKIN or LINKOUT */
  panel_set(gi_mitem[TARGET_ITEM],PANEL_SHOW_ITEM,show,0);

  /* set the default low and high range values */
  panel_set_value(gi_mitem[LRANGE_ITEM],lrange);
  panel_set_value(gi_mitem[HRANGE_ITEM],hrange);
}


/****************************************************************/
static where_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/* called when the user enters a value in the
 * x: or y: fields of the WHERE prompt
 */
   int x_coord, y_coord;
   char stringx[MAX_NUM_LENGTH+1], stringy[MAX_NUM_LENGTH+1];

   /* move the marker to reflect the new coordinates */
   strcpy(stringx,(char *) panel_get_value(gi_mitem[WHEREX_ITEM]));
   strcpy(stringy,(char *) panel_get_value(gi_mitem[WHEREY_ITEM]));
   if ((gi_check_num(stringx)==OK) && (gi_check_num(stringy)==OK))
   {
     y_coord = atoi(stringy);
     x_coord = atoi(stringx);
     gi_move_marker(x_coord,y_coord);
   }

   /* do default keystroke processing */
   panel_text_notify(item,event);
}

/****************************************************************/
static Panel_setting but_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/* called when the user enters a text into
 * the button field of the control panel in custom mode this
 * controls the smooth movement from one line to the
 * next when the current line is filled
 */
 
   Panel_setting ret_val; 
   Panel_item cur_item;
   static char buf[] = " ";

   /* get item user is currently typing within */
   cur_item = (Panel_item) panel_get(gi_control_panel, PANEL_CARET_ITEM, 0);

   /* do normal keystroke processing */
   ret_val = panel_text_notify(item,event);

   /* if this next char will overflow item, move caret to next line */
   if (ret_val==PANEL_INSERT 
   && strlen((char *) panel_get_value(cur_item))
   == CUST1_LEN)
   {
     /* move caret to next line */
     panel_advance_caret(gi_control_panel);

     /* make this character the value of next item */
     cur_item = (Panel_item) panel_get(gi_control_panel, PANEL_CARET_ITEM, 0);
     buf[0] = event->ie_code;
     panel_set_value(cur_item,buf);
     
     /* ignore the character (since we've already processed it) */
     ret_val = PANEL_NONE;
   }

   /* return panel_text_notify return value */
   return(ret_val);
}

/****************************************************************/
static Panel_setting font_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/* called when the user enters a text into
 * the FONT field of the control panel in Text mode
 * controls the smooth movement from one line to the
 * next when the current line is filled
 */
 
   Panel_setting ret_val; 
   Panel_item cur_item;
   static char buf[] = " ";

   /* get item user is currently typing within */
   cur_item = (Panel_item) panel_get(gi_control_panel, PANEL_CARET_ITEM, 0);

   /* do normal keystroke processing */
   ret_val = panel_text_notify(item,event);

   /* if this next char will overflow item, move caret to next line */
   if (ret_val==PANEL_INSERT 
   && strlen((char *) panel_get_value(cur_item))
   == MAX_ITEM_LENGTH)
   {
     /* more caret to next line */
     panel_advance_caret(gi_control_panel);

     /* make this character the value of next item */
     cur_item = (Panel_item) panel_get(gi_control_panel, PANEL_CARET_ITEM, 0);
     buf[0] = event->ie_code;
     panel_set_value(cur_item,buf);
     
     /* ignore the character (since we've already processed it) */
     ret_val = PANEL_NONE;
   }

   /* return panel_text_notify return value */
   return(ret_val);
}

/****************************************************************/
static target_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/* called when the user enters a value 
 * target field of in Link mode
 */
  int target;

   /* do default keystroke processing */
   panel_text_notify(item,event);

   /* get target unit from panel prompt */
   target = gi_get_target((char *) panel_get_value(gi_litem[LSITE_ITEM]));

   /* check that unit index was OK */
   if (target < 0)
   {
     gi_put_error("Invalid target");
   }

   else
   {
     gi_set_new_target(NULL,target);
     gi_reshow();
   }
}

/****************************************************************/
/*ARGSUSED*/
static custlog_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/* called when the LOG DEFINITIONS button is pressed
 * and creates a "button" command that would
 * define the same setting
 */
  char buf[MAX_CMD_LENGTH+1];
  int i, index;

  /* if logging isn't enabled, put out error message */
  if (!gi_log_on)
  {
    gi_put_error("Logging is not on: request ignored");
    return;
  }
  
  /* build a command in the buffer for each button  
     and log it to the requested log file
   */ 
  for (i=0; i<6; i++)
  {
    index = CLBDN1_ITEM + (2*i);
    (void) sprintf(buf,"%s %s %1d \"%s%s\"",GI_CMD,BUT_CMD_SHORT,i+1,
            (char *) panel_get_value(gi_citem[index]),
            (char *) panel_get_value(gi_citem[index+1]));
    gi_log(buf);
  }
  
  gi_put_message("Button definitions have been logged");
}

/****************************************************************/
/*ARGSUSED*/
static go_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/* called when the GO button is pressed
 */
   char buf[MAX_CMD_LENGTH+1];
   char *steps, *update;
   int upd_steps=1, steps_left=1;

   /* get steps and update steps args from panel */
   steps = gi_strip((char *) panel_get_value(go_steps_item));
   update = gi_strip((char *) panel_get_value(go_update_item));
   
   /* verify that values are numeric (or null)  */
   if (gi_check_num(steps)==ERROR || gi_check_num(update)==ERROR)
   {
     gi_put_error("Invalid steps or update specified");
   }
   else
   {
     /* if not null, extract numeric values for args */
     if (steps && strlen(steps))
       steps_left = atoi(steps);
     if (update && strlen(update))
       upd_steps = atoi(update);

     /* step the simulator the requested number of steps */
     gi_do_steps(steps_left,upd_steps);

     /* build and log the "go" command if logging enabled */
     if (gi_log_on)
     {
       (void) sprintf(buf,"%s %s %s %s",GI_CMD,GO_CMD_SHORT,steps,update);
       gi_log(buf);
     }
     
     gi_put_message("Go completed successfully");
   }
}

/****************************************************************/
/*ARGSUSED*/
static reshow_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/* called when the RESHOW button is pressed
 */
   int new_x, new_y;
   char buf[20];
   char logbuf[30];

   /* determine if the origin has been changed by the user */
   strcpy(buf,gi_strip((char *) panel_get_value(gi_reshow_o_item)));
   if (sscanf(buf,"%d %d", &new_x, &new_y)!=2)
   {
    gi_put_error("Illegal Origin specification");
     return /*(ERROR)*/;
   }

   /* check that coordinates are within range */
   if (abs(new_x)>MAX_COORD || abs(new_y)>MAX_COORD)
   {
     gi_put_error("Origin specification is out of range");
     return /*(ERROR)*/;
   }

   /* if the coordinates have changed, perform the update */
   if (new_x!=gi_origin_x || new_y!=gi_origin_y)
   {
      gi_change_origin(new_x-gi_origin_x,new_y-gi_origin_y); 
      gi_update_grobj_chain(TRUE, TRUE);
   }

   /* set flag to make sure gi_reshow clears screen and
      writes all objects
   */
   gi_reshow_flag |= RESHOW_ALL+CLEAR_NEEDED;
   gi_reshow();

   /* if logging enabled, create and log the command */
   if (gi_log_on) 
   {
     (void) sprintf(logbuf,"%s %s %s",GI_CMD,RESHOW_CMD_SHORT,buf);
     gi_log(logbuf);
   }
}

/****************************************************************/
/*ARGSUSED*/
static dump_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/* called when the DUMP button is pressed
 */
   char imagefilename[MAX_ITEM_LENGTH+1];
   FILE *raster_file;

   /* get the filename from the prompts */
   strcpy(imagefilename,
          gi_strip((char *) panel_get_value(dump_image_item)));
  
   /* check for a valid file name */
   if (strlen(imagefilename)==0)
   {
     gi_put_error("Please specify image filename");
   }
   else
   {
     /* attempt to open the image file */
     if ((raster_file=fopen(imagefilename,"w"))==NULL)
     {
       gi_put_error("Unable to open image file");
     }
     else
     {
       /* write the retained window to the file */
       if (pr_dump(gi_gfx->gfx_pixwin->pw_prretained,
                   raster_file,
                   NULL,
                   RT_STANDARD,
                   TRUE)==PIX_ERR)
  
        {
          gi_put_error("pr_dump failed");
        }
        else
        {
          fclose(raster_file);
          gi_put_message("Image dump successful");
        }
      }
   } 
}

/****************************************************************/
/*ARGSUSED*/
static show_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/*
 * called when the SHOW button is
 * pressed, this routine builds a command line (in command[])
 * from the panel values that have been set and
 * passes it to the command processor for 
 * execution
 */

   char buf1[MAX_CMD_LENGTH+1];
   char buf2[MAX_CMD_LENGTH+1];
   char *args[MAX_ARGS];
   int numargs;

   /* read panel prompts into buf1 to make up a show command */        
   set_args(SHOW_CMD_SHORT,buf1);

   /* parse the just created command into args array and buf2 */
   numargs = gi_parse_cmd(buf1,args,buf2);

   /* execute the show command */
   gi_do_show(numargs,args);

   /* if display panel needs to be updated call gi_reshow to do it */
   if (gi_reshow_flag) 
     gi_reshow();

   /* if logging enabled, then log the original command */
   if (gi_log_on) 
     gi_log(buf1);

   /* show the command just executed on the command panel */
   panel_set_value(gi_cmd_prev_item, buf1);

}

/****************************************************************/
/*ARGSUSED*/
static change_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/*
 * called when the CHANGE button is
 * pressed, this routine builds a command line (in command[])
 * from the panel values that have been set and
 * passes it to the command processor for
 * execution
 */
   char buf1[MAX_CMD_LENGTH+1];
   char buf2[MAX_CMD_LENGTH+1];
   char *args[MAX_ARGS];
   int numargs;

   /* build a change command in buf1 from the panel prompts */
   set_args(CHANGE_CMD_SHORT,buf1);

   /* parse the command into args array and buf2 */
   numargs = gi_parse_cmd(buf1,args,buf2);

   /* execute the change command */
   gi_do_change(numargs,args);

   /* if display display needs to be updated, call gi_reshow to do it */
   if (gi_reshow_flag) 
     gi_reshow();

   /* if logging enabled, log the original command */
   if (gi_log_on) 
     gi_log(buf1);

   /* show the command just executed on the command panel */
   panel_set_value(gi_cmd_prev_item, buf1);
}

/****************************************************************/
/*ARGSUSED*/
static erase_proc(item, event)
   Panel_item item;
   struct inputevent *event;
{
/*
 * called when the ERASE button is
 * pressed, this routine builds a command line (in command[])
 * from the panel values that have been set and
 * passes it to the erase command processor for 
 * execution
 */
   char buf1[MAX_CMD_LENGTH+1];
   char buf2[MAX_CMD_LENGTH+1];
   char *args[MAX_ARGS];
   int numargs;

   /* build an erase command in buf1 from the panel prompts */
   set_args(ERASE_CMD_SHORT,buf1);

   /* parse the command into args array and buf2 */
   numargs = gi_parse_cmd(buf1,args,buf2);

   /* execute the erase command */
   gi_do_erase(numargs,args);

   /* if logging enabled, log this command */
   if (gi_log_on) 
     gi_log(buf1);

   /* show the command just executed on the command panel */
   panel_set_value(gi_cmd_prev_item, buf1);
}

/****************************************************************/
static set_args(cmd, arg_buf)
   char *cmd, *arg_buf;
{
/* reads the values from the control panel
 * input areas, converts and puts them into
 * the arg_buf (which it assumes is long enough
 * to hold the maximum sized command)
 */
   char buf[MAX_CMD_LENGTH+1];
   char *what_stg;

   int i;

   /* initialize the buffer to "gi" */
   strcpy(arg_buf,GI_CMD);

   /* add the 1st argument: "s", "e" or "c" */
   add_arg(arg_buf,cmd);

   /* add the WHO field as next argument */
   add_arg(arg_buf,(char *) panel_get_value(gi_mitem[WHO_ITEM]));

   /* add the HOWMANY field as next arguement */
   add_arg(arg_buf,(char *) panel_get_value(gi_mitem[HOWMANY_ITEM2]));

   /* if not erase command: add the appropriate WHAT arg */
   if (strcmp(cmd,ERASE_CMD_SHORT)) /* not the erase command */
   {
      switch (gi_cur_unit_what)
      {
        case POT: 
             what_stg = POT_STG; break;

        case OUTPUT: 
             what_stg = OUTPUT_STG; break;

        case STATE: 
             what_stg = STATE_STG; break;

        case DATA: 
             what_stg = DATA_STG; break;

        case LINKIN: 
             what_stg = LINKIN_STG; break;

        case LINKOUT: 
             what_stg = LINKOUT_STG; break;

      }
      add_arg(arg_buf,what_stg);

      /* add the range arguments */
      add_arg(arg_buf,(char *) panel_get_value(gi_mitem[LRANGE_ITEM]));
      add_arg(arg_buf,(char *) panel_get_value(gi_mitem[HRANGE_ITEM]));
                     
      /* if LINKIN or LINKOUT, add the target item (else "0") */
      if (gi_cur_unit_what==LINKIN || gi_cur_unit_what==LINKOUT)
      {
        add_arg(arg_buf,(char *) panel_get_value(gi_mitem[TARGET_ITEM]));
      }
      else
        add_arg(arg_buf,"0");
      
      /* get the icon value (for default icons) or users icon file name */
      if (gi_cur_image_value <= NUM_STD_ICONS)
         (void) sprintf(buf,"%1d\0",gi_cur_image_value);
      else
         strcpy(buf,(char *) panel_get_value(gi_mitem[SUBIMAGE_ITEM]));
      add_arg(arg_buf,buf);
       
      /* if this is the show command, add positioning arguments */
      if (strcmp(cmd,CHANGE_CMD_SHORT)) 
      {
        for (i=WHEREX_ITEM; i<=NUMCOLS_ITEM; i++)
          add_arg(arg_buf, (char *) panel_get_value(gi_mitem[i]));
      }
   }
}

/****************************************************************/
static add_arg(string, arg)
  char *string, *arg;
{
/* given a string and and an argument
 * to add to it, this concatenates the
 * two strings putting a blank inbetween
 */
  if (arg) /* make sure arg not null */
  {
    strcat(string," ");
    strcat(string,arg);
  }
}
