/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

#include	"sim.h"
#include 	<math.h>
#ifdef	FSIM
#	define	BP_ONE	1.0
#	define	CAST	float
#	define	BP_ZERO	0.0
#else
#	define	BP_ONE	1000
#	define	CAST	int
#	define	BP_ZERO	0
#endif

#include "bp.h"

float BPmomentum = 0.5;
float BPlearn = 1.0;
float BPtemperature = 1.0;

/** UFone sets the output of the unit to 1 (1000).  Used for bias. **/

FLINT
UFone(up)
Unit	*up;
{
	SetOutput(UnitIndex(up), BP_ONE);
}

/** UFfire sets the output of the unit to 0 unless it is a positive number,
	in which case it subtracts 1 from it.  Used for fire unit.
**/

FLINT
UFfire(up)
Unit	*up;
{
	if(up->output <= BP_ZERO)
		SetOutput(UnitIndex(up), BP_ZERO);
	else
		SetOutput(UnitIndex(up), up->output - 1);
}

/** UFcontrol keeps track of the module.  The unit is initialy set in the
	QUIET state.  If the $fire site gets activation it turns to the
	forward state.  In this state, each time UFcontrol runs it increments
	the output and potential of the unit (to indicate what level is being
	run).  It then turns off the NO_UNIT_FUNC_FLAG for each unit in the
	layer that is being run (it is up to the unit functions to turn this
	back on when they are done).  If it is the second time on the output
	layer (up->data) then the state flips to the reverse state.  In this 
	state the output and potential are adjusted slightly differently.
	When the first level has been completed the state is switched back
	to QUIET.
**/

FLINT
UFcontrol(up)
Unit	*up;
{
	Site	*sp;
	char	*n;
	NameDesc	nd;
	register int	i, index, n_units;

	if(up->state == QUIET)
	{	for(sp = up->sites; strcmp(sp->name, "$fire"); sp = sp->next)
			if(!sp)
			{	LOGfprintf(stderr, "UFcontrol: Can't find $fire site in Control unit!!!\n");
				abort();
			}
		if(sp->value)
			up->state = FWD;
		else
			return;
	}
	switch(up->state) {
	case FWD:
		if(up->data != up->output)
		{	++up->output;
			++up->potential;
			break;
		}
	case REV:
		if(up->potential == 1)
		{	for(sp = up->sites; strcmp(sp->name, "$fire"); sp = sp->next)
			if(!sp)
			{	LOGfprintf(stderr, "UFcontrol: Can't find $fire site in Control unit!!!\n");
				abort();
			}
			if(sp->value)
			{	up->state = FWD;
				up->potential = up->output = (CAST)1;
			}
			else
			{	up->state = QUIET;
				up->output = up->potential = BP_ZERO;
			}
			break;
		}
		if(up->data == up->output)
		{	up->output *= (CAST)-1;
			up->state = REV;
		} else
		{	++up->output;
			--up->potential;
		}
		break;
	default:
		LOGfprintf(stderr, "UFcontrol: Control unit in bad state...look at core\n");
		abort();
	}
	n = (char *)malloc(strlen(up->name) + 4);
	(void) sprintf(n, "%s(%d)", up->name + 5, (int)up->potential);
	n_units = FindName(n, &nd)->size;
	index = nd.index;
	for(i = 0; i < n_units; i++)
		UnsetFlag(index + i, NO_UNIT_FUNC_FLAG);
	free(n);
}		

/** SFerror is the site function for the $error sites of the $output units.
	It compares the input (from a $teach unit) to the $output unit's
	output and puts the difference in the data field of the $learn
	site of the $output unit, as well as setting the site value to
	this difference (not necessary).  This allows the $output unit
	function's error processing to be the same as for $hidden units.
**/

FLINT
SFerror(up, sp)
Unit	*up;
Site	*sp;
{
	sp->value = up->sites->data  = *(sp->inputs->value) - up->output;
}

/** SFbpsigmoid totals up inputs and puts it through the sigmoid function
	to get result.
**/

/*ARGSUSED*/
FLINT
SFbpsigmoid(up,sp)
Unit	*up;
Site	*sp;
{
    FLINT 	sum = BP_ZERO;
    Link	*lp;

    for(lp = sp->inputs;lp != NULL; lp = lp->next)
	sum += (*(lp->value) * lp->weight)/BP_ONE;
    /* table lookup eventually */
    sp->value = (CAST)((1.0/(1.0 + exp(((double)-(sum/BPtemperature))/BP_ONE)))*BP_ONE);
}

/** UFh_o is the unit function for both $output and $hidden layers.  If
	BP_FORWARD_FLAG is set then simply sets its output and potential to value
	from $learn site.  Otherwise, in reverse state, it calculates
	necessary weight change for links, from information in the $learn
	site's data field (passed down from above), and changes those weights
	(there is a momentum factor that takes into acount the last weight
	change).  It then passes weight change information down to the units
	that are linked to its $learn site by modifying their $learn site's
	data fields.

	In either state (forward or reverse), after whatever specific 
	executions, the state is flipped, the NO_UNIT_FUNC_FLAG is set,
	and the data field of its $learn site is nullified so new error
	data can be recieved.

	NOTE: The bit-position of BP_FORWARD_FLAG might conflict with some other
	flag defined in another package.
**/

FLINT
UFh_o(up)
Unit	*up;
{
	FLINT	delta, nudelta, deltaw;
	Link	*lp;

/**-----------------**
 ** activation code **
 **-----------------**/
	if(TestFlagP(up, BP_FORWARD_FLAG))
	{	up->potential = up->output = up->sites->value;
		BPendfwd(up);
	} else
/**------------------------**
 ** error-propagation code **
 **------------------------**/
	{	
		delta = ((up->sites->data) * (up->output) *
			 (BP_ONE - up->output))/(BP_ONE * BP_ONE);
		nudelta = delta * BPlearn;
		for(lp = up->sites->inputs; lp != NULL; lp = lp->next)
		{	deltaw = ((nudelta * *(lp->value))/BP_ONE) 
					+ BPmomentum * lp->data;
			lp->weight += deltaw;
			lp->data = deltaw;
			*(FLINT *)(lp->link_f) += (delta * lp->weight)/BP_ONE;
		}
		BPendrev(up);
	}
}

BPendfwd(up)
Unit	*up;
{
	UnsetFlagP(up, BP_FORWARD_FLAG);
	SetFlagP(up, NO_UNIT_FUNC_FLAG);
}

BPendrev(up)
Unit	*up;
{
	SetFlagP(up, BP_FORWARD_FLAG);
	SetFlagP(up, NO_UNIT_FUNC_FLAG);
	up->sites->data = BP_ZERO;
}
