static char rcsid[] = "$Header: names.c,v 1.4 87/11/03 17:11:36 goddard Exp $";
/*----------------------------------------------------------------------------

  The Rochester Connectionist Simulator.  Copyright Univesity of Rochester.
  Author: Nigel Goddard
  Date: May 1 1987

----------------------------------------------------------------------------*/

#pragma segment SEGMBP

#ifdef BFLY                                       /* butterfly */
# ifdef BSIM
#  include      "bflysim.h"                       /* butterfly sim */
# else
#  include      "bflycontrol.h"                   /* butterfly control */
# endif
#else
# include       "Stupido:rochester:include:uniproc.h"       /* uniprocessor */
#endif


#ifdef BFLY
  extern struct nametable *table_list; 
  extern int no_tables;
#endif

/*********************************************
 
 unit Names

*********************************************/
/*---------------------------------------------------------------------------
  Names units.  As well as
  naming a single unit, this function can name a vector or 2-D array of
  units.  The name may then be used during simulation from the command
  interface, and may also be used during network construction.
  name is a pointer to the character string name to be given.   type
   is the type of name - SCALAR, VECTOR, or ARRAY.   index is
  the index of the unit to be named, or the first unit in the vector or
  array.   length is the number of units if it is a VECTOR, and the
  number of columns if it is an ARRAY, and is undefined for SCALAR.
  depth is the number of rows for an ARRAY, and is undefined for
  SCALAR and VECTOR.  A name which is specified as VECTOR or ARRAY will
  apply that name to the unit with the index specified, and to the
  requisite number of units following it in the unit array. 
----------------------------------------------------------------------------*/
                                
NameUnit(name,type,index,length,depth)
    char *name;
    int type,index,length,depth;

{
  int s,off,oldtab,newtab,finish;
  Index *indices, tabind;                         /* Index is char for uni */

BFLYCATCH

  if (!(type == SCALAR || type == VECTOR || type == ARRAY))
    LOGfprintf(stderr,"Naming failed, unknown type in NameUnit: %d\n\n",type);
  else
    if ((indices = EnterName(name,type,index,length, /* enter in name table */
                             depth,&tabind)) == NULL) /* final param not */
      LOGfprintf(stderr,
              "Name Unit: could not enter name in name table: %s\n",
              name);
    else
      {                                           /* needed for uniproc */
        switch (type)
          {
          case SCALAR:
            finish = index+1;
            break;
          case VECTOR:
            finish = index + length;
            break;
          case ARRAY:
            finish = index + (length*depth);
            break;
          default:
            finish = index;
            LOGfprintf(stderr,
                       "Not scalar, vector or array type in NameUnit: %d\n",
                       index);
          }
        if (finish > LastUnit)
          {
            LOGfprintf(stderr,"Unit index out of range.. last is %d\n",
                       NoUnits-1);
            AlterName(name,STRING_SYM,0,0,0,&tabind);
            return 0;
          }
#ifndef BFLY                                      /* uniprocessor: simple! */

        for(s = index;s < finish;s++)
          UnitList[s].name = indices;             /* save pointer in unit */
      }

#else                                             /* butterfly version */

        oldtab = SIM_UNIT(index);                 /* adjust index table(s) */
        indices = (Index *)Map_Obj(table_list[oldtab].index,0,RW_rw_);
        for(s = index;s < finish;s++)
          {
            if((newtab = SIM_UNIT(s)) != oldtab)
              {
                Unmap_Obj(table_list[oldtab].index,indices);
                indices = (Index *)Map_Obj(table_list[newtab].index,0,RW_rw_);
                oldtab = newtab;
              }
            off = OFFSET_UNIT(s);
            indices[off].table = tabind.table;
            indices[off].offset = tabind.offset; /* */
          }
        Unmap_Obj(table_list[oldtab].index,indices);
      }
#endif                                            /* end butterfly version */
BFLYTHROW("NameUnit",0)
}
/*---------------------------------------------------------------------------
  Returns the index of the unit with the given name.  If the name is
  that of a VECTOR, then  column  gives offset of the unit within
  the vector.  If the name is that of an ARRAY, then  column  and
   row  give the column and row of the unit within the array.  If
  the name is not that of a unit, or either of the indices are out of
  range, then the function returns -1.
----------------------------------------------------------------------------*/

NameToInd(name, offset, row)
    char *name;
    int offset,row;
{
  NameDesc nte;
  
BFLYCATCH

  if (FindName(name,&nte) == NULL)
    return -1;
  switch (nte.type)
    {
    case SCALAR:
      return nte.index;
    case VECTOR:
      if (offset >= nte.size)
        {
          LOGfprintf(stderr,"index out of range: %s[%d]\n",name,offset);
          return -1;
        }
      return (nte.index + offset);
    case ARRAY:
      if (offset >= nte.size || row >= nte.length)
        {
          LOGfprintf(stderr,"index out of range: %s[%d][%d]\n",
                     row,name,offset);
          return -1;
        }
      return (nte.index + row*nte.size + offset);
    default:
      LOGfprintf(stderr,"%s not a unit name; type is %d\n",name,nte.type);
      return -1;
    }
BFLYTHROW("NameToInd",0)
}

/*---------------------------------------------------------------------------
  If  name  is the name of a unit type, a unit, a site, a
  function, a unit state, a set, or an unused name in the name table,
  NameToType returns a pointer to a volatile string detailing the type
  of name, e.g. ``unit vector name''.  If the name is not found, it
  returns a pointer to the volatile string ``unknown name''.
----------------------------------------------------------------------------*/

char * NameToType(name)
     char * name;

{
  NameDesc nte;
  static char buff[80];
  char buf[32];
  char * str;

  if (FindName(name,&nte) == NULL)
    str = "unknown name";
  else
    switch (nte.type)
      {
      case SCALAR:
        str = "unit name";
        break;
      case VECTOR:
        str = "unit vector name";
        break;
      case ARRAY:
        str = "unit array name";
        break;
      case SET_SYM:
        str = "set name";
        break;
      case STATE_SYM:
        str = "state name";
        break;
      case STRING_SYM:
        str = "unused name";
        break;
      case TYPE_SYM:
        str = "unit type name";
        break;
      case FUNC_SYM:
        str = "function name";
        break;
      case SITE_SYM:
        str = "site name";
        break;
      case DATA_SYM:
        str = "variable name";
        break;
      case CODE_SYM:
        str = "code unit name";
        break;
      default:
        str = (char *) sprintf(buf,"unknown type (%d) of name",nte.type);
      }
  strcpy(buff,str);
  return (buff);
}

/*---------------------------------------------------------------------------
  Returns a pointer to a volitile string containing name of the unit
  with index  u , or  **NO NAME**  if
  the unit has not been given a name.  If the name is that of a VECTOR
  or ARRAY, the name has the form  name[offset]  or 
  name[row][column] .
----------------------------------------------------------------------------*/

char * IndToName(ind)
    int ind;

{
  static char   buff[80];
  NameDesc nte,*table;
  int j,k,tab,offset;
  char * nptr;
  Index * indices;

BFLYCATCH
#ifdef BFLY                                       /* butterfly */
                                                  /* get table entry */
  if (OFFSET_UNIT(ind) > NoUnits - 1)
    return NULL;
  indices = (Index *) Map_Obj(table_list[SIM_UNIT(ind)].index,0,RW_rw_);
  tab = indices[OFFSET_UNIT(ind)].table;
  offset = indices[OFFSET_UNIT(ind)].offset;
  Unmap_Obj(table_list[SIM_UNIT(ind)].index,indices);
  if((tab + offset) == 0)
    {
      sprintf(buff,"%s","**NO NAME**");
      return buff;
    }
  table = (NameDesc *) Map_Obj(table_list[tab].table,0,RW_rw_);
  nte = *(table+offset);        /* copy name table entry */
  Unmap_Obj(table_list[tab].table,table);

#else                                             /* uniprocessor */

  if (ind > NoUnits - 1)                          /* get table entry */
    return NULL;
  if ((nptr = UnitList[ind].name) == NULL)
    return "**NO NAME**";
  FindName(nptr,&nte);

#endif

  switch (nte.type)                               /* all versions */
    {
    case SCALAR:
      return nte.name;
    case VECTOR:
      {
        sprintf(buff,
                "%s[%d]", nte.name, ind - nte.index);
        return buff;
      }
    case ARRAY:
      {
        k = (ind - nte.index)/nte.size;
        j = (ind - nte.index) % nte.size;
        sprintf(buff,"%s[%d][%d]", nte.name,
                k, j);
        return buff;
      }
    default:
      LOGfprintf(stderr,
                 "Unit %d holds pointer to non-unit name: %s\n",ind,nte.name);
    }
  return NULL;
BFLYTHROW("IndToName",0)
}

/****************************************************

 state Names

****************************************************/


/***** DeclareState *****/
/*---------------------------------------------------------------------------
  Associates a name with a state number.  num must be in the range 0 to 99.
----------------------------------------------------------------------------*/

DeclareState(name,num)
    char *name;
    int num;
{
    NameDesc look;
    Index * nptr,tabind;

BFLYCATCH
    if(num < 0 || num >= NoStates)
      {
        LOGfprintf("DeclareState: state index must be in 0..%d\n",NoStates);
        return;
      }

    if (FindName(name,&look) == NULL || look.type == STRING_SYM)
      {                         /* name not in use yet */
        if (StateNames[num] != NULL)
          {
            LOGfprintf(stderr,"Warning: deleting state %s\n",
                       StateNames[num]);
            AlterName(StateNames[num],STRING_SYM,0,0,0,&tabind);
          }                             /* previous name no longer a state */
      }
    else                        /* name in use */
      if (look.type == STATE_SYM)
        if (look.index == num)  /* already mapped */
          return;
        else
          {                     /* mapped to different value */
            LOGfprintf(stderr,"Warning: changing value for state %s from %d to %d\n",name,look.index,num);
            StateNames[look.index] = NULL;
            if (StateNames[num] != NULL)
              {
                LOGfprintf(stderr,"Warning: deleting state %s\n",
                           StateNames[num]);
                AlterName(StateNames[num],STRING_SYM,0,0,0,&tabind);
              }
          }
      else                      /* name already usef for something else */
        {
          LOGfprintf(stderr,"Error: Declaring %s a state name, but already known as %s\n",name,NameToType(name));
          return;
        }

    nptr = EnterName(name,STATE_SYM,num,0,0,&tabind);
    StateCount++;
        
#ifndef BFLY                    /* uniprocessor */
    StateNames[num] = nptr;
#else                           /* butterfly */
    StateNames[num] = StoreString(name);
#endif
BFLYTHROW("DeclareState",0)
}

/***** NameToState *****/
/*---------------------------------------------------------------------------
  Passed a state  name , returns the state number, or -1
  if the  name  is not that of a state.
----------------------------------------------------------------------------*/


NameToState(name)
    char *name;
{
    int i;
    NameDesc look;

BFLYCATCH
    if (FindName(name,&look) == NULL)           /* no such name */
      return -1;
    if (look.type == STATE_SYM)
      return (look.index);
    return -1;
BFLYTHROW("NameToState",0)
} /* NameState */


/***** StateToName *****/
/*---------------------------------------------------------------------------
   Passed a state number, returns the name.  If there is no predefined
   mapping, the name is the string version of the number (volatile).
----------------------------------------------------------------------------*/

char *StateToName(snum)
    int snum;
{
    static char tname[10];

BFLYCATCH
    if(snum < 0 || snum >= NoStates || StateNames[snum] == NULL)
      {
        sprintf(tname,"%d\0",snum);
        return &tname[0];
      }  
    return StateNames[snum];
BFLYTHROW("StateToName",0);
} /* StateName */

/* Set name functions */

/***** NameSet *****/
/*---------------------------------------------------------------------------
   Passed a set name, returns the number.  Returns -1 if no such name.
----------------------------------------------------------------------------*/

NameToSet(name)
    char *name;
{
    int i;
    NameDesc look;

BFLYCATCH
    if (FindName(name,&look) == NULL)           /* no such name */
      return -1;
    if (look.type == SET_SYM)
      return (look.index);
    return -1;
BFLYTHROW("NameToSet",0)
} /* NameToSet */


/***** SetToName *****/
/*---------------------------------------------------------------------------
   Passed a set number, returns the name.  If there is no predefined
   mapping, the name is the string version of the number (volatile).
----------------------------------------------------------------------------*/
 
char *SetToName(snum)
    int snum;
{
    static char tname[10];

BFLYCATCH
    if(snum < 0 || snum >= NoSets || SetNames[snum] == NULL)
      {
        sprintf(tname,"%d\0",snum);
        return &tname[0];
      }  
    return SetNames[snum];
BFLYTHROW("StateToName",0);
} /* SetToName */

/****************************************************

  Item table management

****************************************************/
static int NoItems = 0;         /* greatest item index used to date */
static int MaxNoItems = 100;    /* maximum number of items (can increase) */
static MappingTable * si_item_Names = NULL;

MappingTable * NameToItem(name,item,type)
     char * name;
     MappingTable * item;
     int type;

{
  NameDesc nte;

  if (FindName(name,&nte) == NULL)
    return NULL;
  if (nte.index < 0 || nte.index > NoItems || (nte.type != type && type >= 0))
    return NULL;
  *item = *(si_item_Names + nte.index);
  return item;
}

MappingTable * PointerToItem(ptr,item,type)
     char * ptr;
     MappingTable * item;
     int type;

{
  register int i;
  register MappingTable * titem;
  NameDesc nte;

  for (i = 0, titem = si_item_Names;
       i <= NoItems &&
       (titem->name == NULL || titem->item.fileinfo != ptr ||
        (FindName(titem->name,&nte)->type != type && nte.type >= 0) );
       i++, titem++);
  if (i > NoItems)
    return NULL;
  *item = *titem;
  return item;
}

MappingTable * IndexToItem(index,item)
     int index;
     MappingTable * item;

{
  if (index < 0 || index > NoItems)
    return NULL;
  *item = *(si_item_Names + index);
  return item;
}


/***** NameToFunc *****/
/*---------------------------------------------------------------------------
  If  name  is the name of a user function, or a simulator command
  function, then NameToFunc returns a pointer to the function, otherwise
  NULL.
----------------------------------------------------------------------------*/

func_ptr NameToFunc(name)
     char    *name;

{
    int i;
    char *n;
    MappingTable mte;

BFLYCATCH
    if (NameToItem(name,&mte,FUNC_SYM) == NULL)
      return NULL;
    else
      return (mte.item.func);
BFLYTHROW("NameToFunc",0)
}

/***** FuncToName *****/
/*---------------------------------------------------------------------------
  If  fncp  is a pointer to a user function, e.g a
  unit function, or to a simulator command function, then FuncToName
  returns a pointer to the name of the function, otherwise NULL.
----------------------------------------------------------------------------*/

char * FuncToName(fncp)
     func_ptr fncp;
{
    int i;
    char * name;
    MappingTable mte;
    NameDesc nte;
BFLYCATCH
  if (PointerToItem(fncp,&mte,FUNC_SYM) == NULL || mte.name == NULL)
    return "**NO NAME**";       /* return valid pointer! */
  else
    return (mte.name);
BFLYTHROW("FuncToName",0)
}

/***** InitFuncNames *****/
/*---------------------------------------------------------------------------
  called from the mainline to set up the function name/pointer mapping.
  Calls functions build by makebind.
----------------------------------------------------------------------------*/
 
si_InitFuncNames()

{
  int cmdno = 0, userno = 0;

BFLYCATCH
#ifdef BSIM                     /* no commands interface on sims */
  si_AddFuncToTable("NullFunc",NullFunc);
#else
  si_cmd_bind_func();
#endif
  si_user_bind_func();
BFLYTHROW("InitFuncNames",0)
}

AddItemToTable(name,ptr,type,prev,ptrlist,codeindx)
     char * name, * ptrlist;    /* ptrlist not really char *, but only */
     int * ptr;                 /* passed on as param to another proc */
     int type, prev, codeindx;

{
  NameDesc nte, * ntp, ntf;
  int retval;
  register int i;
  register int j;
  register char *optr;
  register char *nptr;
  Index *iptr;
  MappingTable * new_item_Names, mte, *mtp;

BFLYCATCH
  if (prev < 0)
    retval = 0 - codeindx;
  else
    retval = 0 - prev;
  if (si_item_Names == NULL)    /* make table if first time called */
    {
      si_item_Names = (MappingTable *) si_malloc (MaxNoItems *
                                            sizeof(MappingTable));
      for (i = 0; i < MaxNoItems; i++)
        (si_item_Names+i)->name = NULL; /* initialize */
    }
  if (!((ntp = FindName(name,&nte)) == NULL || ntp->type == STRING_SYM ||
      ntp->type == type))
    {
      LOGfprintf(stderr,"Unable to add %s to name table as function/variable,already used as %s\n",name,NameToType(name));
      return(retval);
    }
  if (ntp != NULL && ntp->type == type)
    { 
      if (ntp->size < 0)
        {
#ifndef LINDA
          LOGfprintf(stderr,"Attempt to reenter library function %s in item table, can't handle.\n",name);
#endif
          return(retval);
        }
      LOGfprintf(stderr,"!! references to %s in %s will get old version of %s\n",name, IndexToItem(ntp->size,&mte)->name, name);
      for (i = 0,mtp = si_item_Names+ntp->size; /* code unit containing item */
           mtp->next != ntp->index; /* follow chain to this item */
           mtp = si_item_Names+mtp->next)
        if (FindName(mtp->name,&ntf)->type == FUNC_SYM)
          i++;                  /* count functions */
      mtp->next = (si_item_Names+ntp->index)->next; /* remove from chain */
      if (mtp->next < 0 && i == 0) /* no functions left in old code unit */
        {
          LOGfprintf(stderr,"No accessible functions left in codeunit %s, deleting it...\n",IndexToItem(ntp->size,&mte)->name);
          si_RemoveCodeUnit(si_FreeCodeUnit(ntp->size));
        }
      i = ntp->index;
    }
  else
    {
      for (i = 0; i < MaxNoItems &&
           (si_item_Names+i)->name != NULL; i++);
      if (i >= MaxNoItems)
        {                       /* make a bigger table, copy and update */
          LOGfprintf(stderr,"Expanding item table...\n");
          new_item_Names = (MappingTable *) si_malloc ((MaxNoItems + 200) *
                                                    sizeof (MappingTable));
          j = MaxNoItems * sizeof(MappingTable); /* no. bytes to copy */
          for (optr = (char *) si_item_Names, /* copy old table to new */
               nptr = (char *) new_item_Names, i = 0;
               i++ < j;
               *nptr++ = *optr++);
          free(si_item_Names);  /* free up old table */
          si_item_Names = new_item_Names;
          for (i = 0; i < 200; i++) /* initialize extra items in new block */
            (si_item_Names+MaxNoItems+i)->name = NULL;
          i = MaxNoItems;
          MaxNoItems+=200;
        }
    }
  if (i > NoItems)
    NoItems = i;
  if (type == FUNC_SYM)
    si_ChangeFuncPointers(name,ptr,ptrlist);/*redo ptrs in units,sites,links */
#ifndef BSIM                    /* bfly control and sim enter globally */
  iptr = AlterName(name,type,i,codeindx,0,&nte); /* iptr is Index* */
#endif
#ifdef BFLY                     /* bfly control and bfly sim enter locally */
  nptr = StoreString(name);     /* nptr is char* */
#else
  nptr = iptr;
#endif
  (si_item_Names+i)->item.intval = ptr;
  (si_item_Names+i)->next = prev;
  (si_item_Names+i)->name = nptr;
  return(i);                    /* positive */
}

AddItemToChain(itemindx, chainindx)
     int itemindx, chainindx;

{
  NameDesc nte;

  (si_item_Names+itemindx)->next = chainindx;
  FindName((si_item_Names+itemindx)->name,&nte);
  AlterName(nte.name,nte.type,nte.index,itemindx,0,&nte);
}

DeleteItemChain(index)
     int index;

{
  register int i;
  MappingTable * mtp;
  NameDesc nte;

  for (i = index;
       i >= 0;
       i = mtp->next)
    {
      mtp = si_item_Names+i;
      AlterName(mtp->name,STRING_SYM,0,0,0,&nte); /* free nametable */
      mtp->name = NULL; /* free up maptable entry */
    }
}

/*           ------ New stuff starts here ----- Mac version --------                 */

int Debug_Cmd_help();
int Debug_Cmd_ignore();
int Debug_Cmd_set();
int Debug_Cmd_q();
int Debug_Cmd_quit();
/*  int Debug_Cmd_debug();	*/
int Debug_Cmd_read();
int Debug_Cmd_go();
int Debug_Cmd_g();
int Debug_Cmd_load();
/*  int Cmd_rcall();	*/
/*  int Cmd_talk();	*/
int Cmd_debug();
int Cmd_restore();
int Cmd_checkpoint();
int Cmd_restart();
int Cmd_reset();
int Cmd_fsync();
int Cmd_async();
int Cmd_sync();
int Cmd_diffset();
int Cmd_inverseset();
int Cmd_intersectset();
int Cmd_unionset();
int Cmd_remset();
int Cmd_addset();
int Cmd_deleteset();
int Cmd_printpause();
int Cmd_print();
int Cmd_pipe();
int Cmd_log();
int Cmd_save();
int Cmd_read();
int Cmd_show();
int Cmd_go();
int Cmd_g();
int Cmd_shell();
int Cmd_echo();
int Cmd_e();
int Cmd_load();
int Cmd_pause();
int Cmd_call();
int Cmd_q();
int Cmd_quit();
int Cmd_scavenge();
int Cmd_deletesites();
int Cmd_deletelinks();
int Cmd_MakeLink();
int Cmd_AddSite();
int Cmd_MakeUnit();
int Cmd_NameUnit();
int Cmd_AllocateUnits();
int Cmd_abort();
int Cmd_deletecode();
int Cmd_whereis();
int Cmd_whatis();
int Cmd_value();
/* int Cmd_incorporate();  */
int Cmd_loadcode();
int Cmd_compile();
int Cmd_return();
int Cmd_status();
int Cmd_disp();
int Cmd_d();
int Cmd_list();
int Cmd_l();
int Cmd_weight();
int Cmd_w();
/* int Cmd_delay();  */
int Cmd_lfunc();
int Cmd_sfunc();
int Cmd_ufunc();
int Cmd_state();
int Cmd_out();
int Cmd_o();
int Cmd_pot();
int Cmd_p();
int Cmd_help();
int Cmd_h();


si_cmd_bind_func()
{
    AddItemToTable("NullFunc",NullFunc,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Debug_Cmd_help",Debug_Cmd_help,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Debug_Cmd_ignore",Debug_Cmd_ignore,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Debug_Cmd_set",Debug_Cmd_set,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Debug_Cmd_q",Debug_Cmd_q,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Debug_Cmd_quit",Debug_Cmd_quit,FUNC_SYM,-1,NULL,-1);
/*    AddItemToTable("Debug_Cmd_debug",Debug_Cmd_debug,FUNC_SYM,-1,NULL,-1);	*/
    AddItemToTable("Debug_Cmd_read",Debug_Cmd_read,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Debug_Cmd_go",Debug_Cmd_go,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Debug_Cmd_g",Debug_Cmd_g,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Debug_Cmd_load",Debug_Cmd_load,FUNC_SYM,-1,NULL,-1);
/*    AddItemToTable("Cmd_rcall",Cmd_rcall,FUNC_SYM,-1,NULL,-1);	*/
/*    AddItemToTable("Cmd_talk",Cmd_talk,FUNC_SYM,-1,NULL,-1);	*/
    AddItemToTable("Cmd_debug",Cmd_debug,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_restore",Cmd_restore,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_checkpoint",Cmd_checkpoint,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_restart",Cmd_restart,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_reset",Cmd_reset,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_fsync",Cmd_fsync,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_async",Cmd_async,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_sync",Cmd_sync,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_diffset",Cmd_diffset,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_inverseset",Cmd_inverseset,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_intersectset",Cmd_intersectset,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_unionset",Cmd_unionset,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_remset",Cmd_remset,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_addset",Cmd_addset,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_deleteset",Cmd_deleteset,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_printpause",Cmd_printpause,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_print",Cmd_print,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_pipe",Cmd_pipe,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_log",Cmd_log,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_save",Cmd_save,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_read",Cmd_read,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_show",Cmd_show,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_go",Cmd_go,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_g",Cmd_g,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_shell",Cmd_shell,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_echo",Cmd_echo,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_e",Cmd_e,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_load",Cmd_load,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_pause",Cmd_pause,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_call",Cmd_call,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_q",Cmd_q,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_quit",Cmd_quit,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_scavenge",Cmd_scavenge,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_deletesites",Cmd_deletesites,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_deletelinks",Cmd_deletelinks,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_MakeLink",Cmd_MakeLink,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_AddSite",Cmd_AddSite,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_MakeUnit",Cmd_MakeUnit,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_NameUnit",Cmd_NameUnit,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_AllocateUnits",Cmd_AllocateUnits,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_abort",Cmd_abort,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_deletecode",Cmd_deletecode,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_whereis",Cmd_whereis,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_whatis",Cmd_whatis,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_value",Cmd_value,FUNC_SYM,-1,NULL,-1);
/*    AddItemToTable("Cmd_incorporate",Cmd_incorporate,FUNC_SYM,-1,NULL,-1);	*/
    AddItemToTable("Cmd_loadcode",Cmd_loadcode,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_compile",Cmd_compile,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_return",Cmd_return,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_status",Cmd_status,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_disp",Cmd_disp,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_d",Cmd_d,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_list",Cmd_list,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_l",Cmd_l,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_weight",Cmd_weight,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_w",Cmd_w,FUNC_SYM,-1,NULL,-1);
/* #ifdef TSIM
	AddItemToTable("Cmd_delay",Cmd_delay,FUNC_SYM,-1,NULL,-1);
#endif  */
    AddItemToTable("Cmd_lfunc",Cmd_lfunc,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_sfunc",Cmd_sfunc,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_ufunc",Cmd_ufunc,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_state",Cmd_state,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_out",Cmd_out,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_o",Cmd_o,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_pot",Cmd_pot,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_p",Cmd_p,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_help",Cmd_help,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("Cmd_h",Cmd_h,FUNC_SYM,-1,NULL,-1);
    
}

/* ------------------------------------------------------------------------- */

int build();
int UFcolor();

FLINT SiteValue();
FLINT LFsimple();
FLINT SFprod();
FLINT SFxor();
FLINT SFand();
FLINT SFmin();
FLINT SFweightedmin();
FLINT SFmax();
FLINT SFweightedmax();
FLINT SFsum();
FLINT SFweightedsum();
FLINT UFsum();

/*
int BPendrev();
int BPendfwd();
FLINT UFh_o();
FLINT SFbpsigmoid();
FLINT SFerror();
FLINT UFcontrol();
FLINT UFfire();
FLINT UFone();
int BPcycle();
int BPsetteach();
int BPsetinput();
int BPendmod();
int BPfire();
int BPteach();
int BPoutput();
int BPhidden();
int BPinput();
int BPlink();
Site * BPGetSite();
char * BPname();
int BPmodule();
*/

si_user_bind_func()
{
   si_User_Unit_Display = NULL;
   si_User_Site_Display = NULL;
   si_User_Link_Display = NULL;
   si_User_Link_List = NULL;
   si_User_Unit_Checkpoint = NULL;
   si_User_Site_Checkpoint = NULL;
   si_User_Link_Checkpoint = NULL;
   si_User_Unit_Restore = NULL;
   si_User_Site_Restore = NULL;
   si_User_Link_Restore = NULL;
   si_User_Unit_Save = NULL;
   si_User_Site_Save = NULL;
   si_User_Link_Save = NULL;
   si_User_Unit_Load = NULL;
   si_User_Site_Load = NULL;
   si_User_Link_Load = NULL;
   si_User_Help_Info = NULL;
    AddItemToTable("build",build,FUNC_SYM,-1,NULL,-1);
	AddItemToTable("UFcolor",UFcolor,FUNC_SYM,-1,NULL,-1);
	
/* user functions from userfunc.c */

    AddItemToTable("SiteValue",SiteValue,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("LFsimple",LFsimple,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFprod",SFprod,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFxor",SFxor,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFand",SFand,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFmin",SFmin,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFweightedmin",SFweightedmin,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFmax",SFmax,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFweightedmax",SFweightedmax,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFsum",SFsum,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFweightedsum",SFweightedsum,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("UFsum",UFsum,FUNC_SYM,-1,NULL,-1);

/* Backpropagation module functions	- remove the comments if using the backprop package */

/*    AddItemToTable("BPendrev",BPendrev,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPendfwd",BPendfwd,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("UFh_o",UFh_o,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFbpsigmoid",SFbpsigmoid,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("SFerror",SFerror,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("UFcontrol",UFcontrol,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("UFfire",UFfire,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("UFone",UFone,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPcycle",BPcycle,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPsetteach",BPsetteach,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPsetinput",BPsetinput,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPendmod",BPendmod,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPfire",BPfire,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPteach",BPteach,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPoutput",BPoutput,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPhidden",BPhidden,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPinput",BPinput,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPlink",BPlink,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPGetSite",BPGetSite,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPname",BPname,FUNC_SYM,-1,NULL,-1);
    AddItemToTable("BPmodule",BPmodule,FUNC_SYM,-1,NULL,-1);	*/
}





