static char rcsid[] = "$Header: commoncmds.c,v 1.2 87/10/21 17:37:18 goddard Exp $";
/*----------------------------------------------------------------------------

  The Rochester Connectionist Simulator.  Copyright Univesity of Rochester.
  Author: Nigel Goddard
  Date: May 1 1987

----------------------------------------------------------------------------*/

#pragma segment SEGM1

/*  This file contains the commands that are used by all versions of the
    simulator.  It also contains the Lexical analyzing routines and
    variables, which are declared external in other command function files
*/

#ifdef BFLY
# ifdef BSIM
#  include "bflysim.h"
# else
#  include "bflycontrol.h"
# endif
#else
#  include "Stupido:rochester:include:uniproc.h"
#  include <time.h>
#endif

#include <ctype.h>
#include "Stupido:rochester:include:lex.h"
#include <string.h>

static int Saved;                                 /* saves previous Lexed */
static int ufound;              				  /* used in GetUnit(s) */

#define GETCHAR cmd[Cmdindex++]                   /* get next character */
#define PUTCHAR Cmdindex--                        /* put it back */

/* Lex takes a string as argument, and tries to classify it as a known token,
   an unknown token (an identifier), or a number.  The strings are arguments
   to a command, previously parsed into argc, argv by LEX
*/

int Lex(cmd)
    char *cmd;
{
    int i,c;

BFLYCATCH
    if(Saved)                                     /* Saved in UnLex */
      {
        i = Saved;                                /* restore Saved token */
        Saved = 0;                                /* reset */
        if (Cmdindex == 0)                        /* if argument completely */
          Curarg++;                               /* parsed, move on to next */
        return i;                                 /* return Saved token */
      }
    for (i=0;                                     /* read identifier */
         (Yyarg[i] = c = GETCHAR) != '\0' &&      /* stop for end of string */
         c != '[' && c != ']';                    /* or square brackets */
         i++);                                    /* and store in Yyarg */
    if (c == '\0')                                /* if end of string found */
      EAT;                                        /* reset for next arg */
    else                                          /* must be [ or ] */
      if (i != 0)                                 /* found end of vector name*/
        {                                         /* and start of index, so */
          PUTCHAR;                                /* put [ or ] back */
          Yyarg[i] = '\0';                        /* terminate string */
        }
      else                                        /* single [ or ] */
        {
          if (GETCHAR == '\0')                    /* end of string */
            EAT;                                  /* move to next arg */
          else                                    /* more in this arg */
            PUTCHAR;                              /* so put back gotten char */
          if (c == ']')
            return CBRACK;                        /* return close bracket */
          else
            return OBRACK;                        /* return open bracket */
        }
    switch (Yyarg[0]) {                           /* on first letter foud */
        case '\0':                                /* empty string */
            return END_STRING;
        case 'a' :
            if(!strcmp(Yyarg,"all")) return ALL;  /* if all */
            if(!strcmp(Yyarg,"auto")) return AUTO; /* if auto */
            if(!strcmp(Yyarg,"array")) return ARR;
            return IDENT;                         /* else must be identifier */
        case 'c' :
            if(!strcmp(Yyarg,"c") || !strcmp(Yyarg,"connections"))
                return LINK;
            if(!strcmp(Yyarg,"clock")) return CLOCK;
            return IDENT;
        case 'd' :
            if(!strcmp(Yyarg,"def") || !strcmp(Yyarg,"default"))
                return DEFAULT;
            return IDENT;
        case 'f' :
            if(!strcmp(Yyarg,"func")) return FUNC;
            if(!strcmp(Yyarg,"from")) return FROM;
            return IDENT;
        case 'i' :
            if(!strcmp(Yyarg,"ipot")) return IPOT;
            if(!strcmp(Yyarg,"istate")) return ISTATE;
            return IDENT;
        case 'l' :
            if(!strcmp(Yyarg,"link") || !strcmp(Yyarg,"links")) return LINK;
            return IDENT;
        case 'n' :
            if(!strcmp(Yyarg,"name")) return NAME;
            return IDENT;
        case 'o' :
            if(!strcmp(Yyarg,"on")) return ON;
            if(!strcmp(Yyarg,"off")) return OFF;
            if(!strcmp(Yyarg,"out")) return OUT;
            return IDENT;
        case 'p':
            if(!strcmp(Yyarg,"pot")) return POT;
            return IDENT;
        case 'r':
            if(!strcmp(Yyarg,"random")) return RANDOM;
            return IDENT;
        case 's' :
            if(!strcmp(Yyarg,"sh") || !strcmp(Yyarg,"show")) return SHOW;
            if(!strcmp(Yyarg,"s") || !strcmp(Yyarg,"state") ||
               !strcmp(Yyarg,"states")) return STATE;
            if(!strcmp(Yyarg,"set") || !strcmp(Yyarg,"sets")) return SET;
            if(!strcmp(Yyarg,"site")) return SITE;
            if(!strcmp(Yyarg,"step")) return STEP;
            if(!strcmp(Yyarg,"scalar")) return SCAL;
            return IDENT;
        case 't':
            if(!strcmp(Yyarg,"type")) return TYPE;
            if(!strcmp(Yyarg,"to")) return TO;
            return IDENT;
        case 'u' :
            if(!strcmp(Yyarg,"u") || !strcmp(Yyarg,"unit") ||
               !strcmp(Yyarg,"units")) return UNIT;
            return IDENT;
        case 'v':
            if(!strcmp(Yyarg,"vector")) return VECT;
            return IDENT;
        case 'w' :
            if(!strcmp(Yyarg,"weight")) return WEIGHT;
            return IDENT;
        case '?' :
            return HELP;
        case '"' :                                /* a string */
            for(i = 0;(Yyarg[i] = Yyarg[i+1]) != '"';i++); /* get rid of */
            Yyarg[i] = '\0';                      /* quotes and terminate */
            return STRING;
        case '+' :
            if(Yyarg[1] == '\0') return PLUS;
            return IDENT;
        case '-' :
            if(Yyarg[1] == '\0') return MINUS;    /* - on its own */
            if(!isdigit(Yyarg[1])) return IDENT;  /* if not number */
        case '0': case '1': case '2': case '3': case '4': case '5':
        case '6': case '7': case '8': case '9':
            if(strchr(Yyarg,'.'))                  /* is this a float ? */
              {
                sscanf(Yyarg,"%f",&Yyfloat);      /* if so scan it in */
                return FLOAT;
              }
            sscanf(Yyarg,"%d",&Yyint);            /* else must be int */
            return INT;                           /* so scan it in */
        default : return IDENT;
    }
BFLYTHROW("Lex",MySimNumber)
}

/**** UnLex ****/

/*
    Next call to Lex will return tok.
    UnLex saves extra token
*/
UnLex(tok)

{
BFLYCATCH
    Saved = tok;                                  /* save token */
    if (Cmdindex == 0)                            /* if moved on to next */
      Curarg--;                                   /* arg, go back */
BFLYTHROW("UnLex",0)
}

/* getindex tries to read an index from <argv[argc]>, checking the
   index found against <limit> and using the <sym> string for error messages
*/

static getindex(argc,argv,sym,limit)
     int argc;                                    /* current argument */
     char ** argv;                                /* arguments string */
     char * sym;                                  /* name */
     int limit;                                   /* limit on index */

{ int tok,index;

BFLYCATCH
  if (Curarg < argc)                              /* more to read ? */
    if ((tok = Lex(argv[Curarg])) == OBRACK)      /* found a [ ? */
      {
        if (Curarg >= argc)                       /* no more to read ? */
          {                                       /* should be integer */
            LOGfprintf(stderr,
                    "expecting integer index, found nothing: %s[\n",
                    sym);
            EAT;                                  /* clean up */
            return INDEX_ERROR;                   /* return error */
          }
        if (Lex(argv[Curarg]) == INT)             /* integer index ? */
          index= Yyint;                           /* if so, save it */
        else
          {                                       /* error, should be int */
            LOGfprintf(stderr,
                    "expecting integer index, found %s[%s\n",
                    sym,Yyarg);
            EAT;                                  /* clean up */
            return INDEX_ERROR;                   /* return error */
          }
        if (Curarg >= argc)                       /* no more to read ? */
          {                                       /* but should be ] */
            LOGfprintf(stderr,
                    "expecting closing bracket, found nothing: %s[%d\n",
                    sym,index);
            EAT;                                  /* clean up */
            return INDEX_ERROR;                   /* return error */
          }
        if (Lex(argv[Curarg]) != CBRACK)          /* not right bracket ? */
          {                                       /* but should be */
            LOGfprintf(stderr,
                    "expecting closing bracket, found %s[%d%s\n",
                    sym,index,Yyarg);
            EAT;                                  /* clean up */
            return INDEX_ERROR;                   /* return error */
          }
        if(index >= limit)                        /* index within limit ? */
          {                                       /* if not, ignore */
            LOGfprintf(stderr,
                    "index out of range for name %s: [%1d]; limit is %d\n",
                    sym,index,limit-1);
            EAT;                                  /* clean up */
            return INDEX_ERROR;                   /* return error */
          }
        else                                      /* evertything ok */
          return index;                           /* return index found */
      }
    else                                          /* not a [ */
      {                                           /* must be \0 or ] */
        if (tok == CBRACK)                        /* is it ] ? */
          {
            LOGfprintf(stderr,"Syntax error: index starts with ]: %s]\n",sym);
            EAT;                                  /* clean up */
            return INDEX_ERROR;                   /* return error */
          }
        else                                      /* no index, must have */
          {                                       /* read next argument */
            UnLex(tok);                           /* replace token in stream */
            return NO_INDEX;                      /* return no index */
          }
      }                                           /* if no more to read */
  return NO_INDEX;                                /* return no index */
BFLYTHROW("getindex",MySimNumber)
}
  
/***** GetUnit *****/
/* Get Unit tries to read a unit identifier, which can be a unit number, the
   token ALL, a unit name (possibly indexed if a vector or array name) or a
   set name.  If a set name or an unindexed vector or a singly or non-indexed
   array, then a group of units has been specified (return RANGE), otherwise
   a single unit has been specified (return SINGLE)
*/

static int GetUnit(argc,argv)
     int argc;
     char ** argv;

{
    int u,tok,offset,row;
    char        buff[ARGLEN];
    char        indbuff[20];
    NameDesc *look,nte;
        
BFLYCATCH
    Uset = FALSE;                                 /* expect not a set */
    switch(Lex(argv[Curarg]))                     /* type of token */
      {
      case INT :                                  /* unit number */
        ufound = Yyint;                           /* remember */
        break;
      case ALL :                                  /* token ALL */
        Ulow = 0;                                 /* set low end of range */
        Uhigh = NoUnits - 1;                      /* and high end */
        return ALL;
      case END_LINE :
        EAT;                                      /* clean up */
        return FAIL;                              /* failed to find unit id */
      default:                                    /* must be identifier */
        strcpy(buff, Yyarg);                      /* Save name */
        look = FindName(buff,&nte);               /* look up in name table */
        if(look == NULL)                          /* not found ? */
          {
            LOGfprintf(stderr,                    /* print error */
                    "Name unknown: %s\n\n",buff);
            return FAIL;                          /* return fail */
          }
        switch (look->type)                       /* switch on name type */
          {
          case SET_SYM:                           /* if a set name... */
            if (Cmdindex != 0)                    /* can't be indexed */
              {
                LOGfprintf(stderr,"%s is a set, not a vector!\n\n");
                EAT;                              /* clean up */
                return FAIL;                      /* return fail */
              }
            else                                  /* set up range globals */
              {
                Uset = TRUE;                      /* means this id a set name*/
                Usetind = look->index;            /* set number */
                Ulow = 0;                         /* check all units  */
                Uhigh = NoUnits - 1;
                return SET;                       /* return unit id type */
              }
          case SCALAR:                            /* if a scalar, easy */
            break;                                /* continue later */
          case VECTOR:
            if ((offset = getindex(argc,          /* get index */
                                   argv,buff,
                                   look->size))
                == INDEX_ERROR)                   /* if error,  */
              return FAIL;                        /* return fail */
            if (offset == NO_INDEX)               /* if no index */
              {                                   /* must be whole vector */
                Ulow = look->index;               /* first unit in vector */
                Uhigh = Ulow + (look->size - 1);  /* last unit in vector */
                return RANGE;                     /* return unit id type */
              }
            break;
          case ARRAY:
            if ((row = getindex(argc,argv,        /* get first index */
                                buff,             /* if error... */
                                look->length)) == INDEX_ERROR )
              return FAIL;                        /* return fail */
            if (row == NO_INDEX)                  /* no index found, so.. */
              {                                   /* complete array */
                Ulow = look->index;               /* first unit in array */
                Uhigh = Ulow + look->size * look->length - 1; /* last unit */
                return RANGE;
              }
            sprintf(indbuff,"%s[%d]",buff,row);   /*for error msgs in getindx*/
            if ((offset = getindex(argc,          /* get second index */
                                   argv,indbuff,look->size)) == INDEX_ERROR)
              return FAIL;                        /* return fail if error */
            if (offset == NO_INDEX)               /* no second index ? */
              {                                   /* one complete row */
                Ulow = look->index + row * look->size; /* first in row */
                Uhigh = Ulow + (look->size - 1);  /* last unit in row */
                return RANGE;
              }
            break;
          default:                                /* func, state or type
            EAT;                                  /* clean up */
            LOGfprintf(stderr,
                       "%s is a %s, not a unit identifier!\n",
                       buff,NameToType(buff));    /* print type of name */
            return FAIL;                          /* return fail */
          } /*switch*/
        ufound = NameToInd(buff, offset, row);    /* must be single unit */
      } /*switch*/                                /* other options have */
    if(!LegalUnit(ufound))                        /* have returned already */
      {                                           /* check if unit no ok */
        LOGfprintf(stdout,"No such unit %s\n",argv[Curarg-1]);
        EAT;                                      /* clean up */
        return FAIL;                              /* return fail */
      }
    return SINGLE;                                /* if unit ok... */
BFLYTHROW("GetUnit",MySimNumber)
}

/* GetUnits attempts to read in a compLex unit identifier, which may
   be a simple unit identifier, a simple range idenifier, or two
   simple unit identifiers seperated by a '-'.  Returns FALSE if
   unsuccessful, or type of uid found if successful
*/

GetUnits(argc,argv)
     int argc;
     char ** argv;

{

BFLYCATCH
  switch(GetUnit(argc,argv))                      /* get the first uid */
    {
    case FAIL:                                    /* if failed */
      return FALSE;                               /* return false */
    case ALL:                                     /* all units - range */
      return ALL;
    case SET:                                     /* ok */
    case RANGE:
      return TRUE;
    case SINGLE:                                  /* single uid, so maybe 2nd*/
      Ulow = ufound;                              /* global range variable */
      if (Curarg >= argc)                         /* no more arguments */
        Uhigh = ufound;                           /* must be single unit id */
      else                                        /* else... */
        { int tok;                                /* check if range with - */
          if((tok = Lex(argv[Curarg])) == MINUS)  /* look for minus */
            {                                     /* range */
              if (Curarg >= argc)                 /* no more to read */
                {                                 /* but should be */
                  LOGfprintf(stderr,"Missing upper end of range\n\n");
                  EAT;                            /* clean up */
                  return FALSE;                   /* failed */
                }
              else                                /* get upper range */
                {
                  if (GetUnit(argc,argv) != SINGLE) /* must be single */
                    {
                      LOGfprintf(stderr,"illegal upper end of range\n\n");
                      return FALSE;
                    }
                  else
                    Uhigh = ufound;               /* set upper end of range */
                }
            }
          else
            {                                     /* for multi-unit cmds */
              UnLex(tok);                         /* put back in stream */
              Uhigh = Ulow;                       /* really single unit */
            }
        }
        return TRUE;
    }
BFLYTHROW("GetUnits",0)
}                                                 /* GetUnits */

/***** help *****/

Cmd_h(argc,argv)
     int argc;
     char ** argv;

{Cmd_help(argc,argv);
}

Cmd_help(argc,argv)             /* ??????????????? */
     int argc;
     char ** argv;

{
  func_ptr func;
  char buff[80];
  char * savearg0;
  int i;

BFLYCATCH

  PipeBegin();
  sprintf(buff,"%s","Cmd_");
  if (argc == 1)
    {
      char str1[60],str2[60];
#ifdef FSIM
      sprintf(str1,"floating point");
#else
      sprintf(str1,"integer");
#endif
#ifdef TSIM
      sprintf(str2,"does");
#else
      sprintf(str2,"does not");
#endif
      Format = TRUE;
      LOGfprintf(Dispf,"This is the Rochester Connectionist Simulator (version 4.1).  This simulator uses %s arithmetic and %s simulate link propagation delay.  Type ? to get a list of available commands.  Type `<command_name> ?' or `help <command name>' to get detailed help information for a particular command.  Type `help syntax' for a description of the command syntax language.\n\n",str1,str2);
      LOGfprintf(Dispf,"You are currently at the level %d interface.  Level 0 is the normal interface with debugging switched off.  Level 1 is the normal interface with debugging switched on.  Higher levels are reached by typing control_C, or making an error in specifying a unit, site or link when debugging is switced on.  Most commands available at levels 0 and 1 are also available at higher levels, but no commands that execute the network (go, etc).\n\n",Debug);
      LOGfprintf(Dispf,"You may return from a level n interface to the previous level with the `quit' command.  You may exit to the UNIX shell by typing control_D to any level interface.  Interrupt (control_C) processing may be delayed if the simulator is not in a safe state.  The level 0 prompt is `->'.  For levels 1 and above, it is a name (either `debug' or `interrupt') followed by the level number - for example, `debug[2]>'.\n\n");
      LOGfprintf(Dispf,"Commands are generally one of seven types: simulation commands (to run, or modify the parameters for running, the network); modification commands (to modify the structure or state of the network); display commands (to display a unit or link in more or less detail); file commands (to read or write files); set commands (to manipulate sets of units); compilation and linking commands; and miscellaneous commands (to call a function or execute a UNIX shell command).\n\n");
      Format = FALSE;
      if (si_User_Help_Info != NULL)
        si_User_Help_Info();
    }
  else
    if (argc == 2)
      if (!strcmp(argv[1],"UnitId"))
        {
          Format = TRUE;
          LOGfprintf(Dispf,"Units may be specified either single, as a range, in in a set.  The specificatiton of a single unit is either its index or name (e.g. RETINA[2][3]).  The specification of a range consists of the upper and lower units separated by ` - ', for instance 0 - 9 or RETINA[2][3] - RETINA[2][8], or may be an vector name, row of an array or entire array, for instance RETINA[2] or RETINA.  The specification by set is simply the set name, and indicates all the units in the set.\n\n");
          Format = FALSE;
        }
      else
        if (!strcmp(argv[1],"help"))
          LOGfprintf(stdout,"Typing simply `help' will get you help\n\n");
        else
          if (!strcmp(argv[1],"syntax"))
            {
              Format = TRUE;
              LOGfprintf(Dispf,"Command syntax descriptions use the following special symbols: `[', `]', `<', `>', `|', `*', and `+'.  An item enclosed in square braces is optional.  An item enclosed in angle braces is obligatory.  If a `*' follows an item enclosed in square braces, then there may be zero or more items of that nature.  If a `+' follows an item enclosed in square braces, then there must be at least one item of that nature.  For example:\n\n");
              LOGfprintf(Dispf,"Usage: nexus <type> [on | off] [value]+ [<min> <max>]*\n\n");
              LOGfprintf(Dispf,"would specify the fictional command `nexus'.  The command name must be followed by a type name.  Then one of the keywords `on' or `off' may appear (or neither).  Then a sequence of at least one values must appear.  Finally, an optional sequence of zero or more minimum/maximum pairs may be given.\n");
              Format = FALSE;
            }
          else
            if ((func = NameToFunc(strcat(buff,argv[1]))) == NULL)
              LOGfprintf(stderr,"No such command: %s\n",argv[1]);
            else
              {
                LOGfprintf(stdout,"\n");
                for (savearg0 = argv[1]; *(savearg0) != '\0'; savearg0++)
                  if (*savearg0 > 'Z')
                    LOGfprintf(stdout,"%c",(*savearg0) - 32);
                  else
                    LOGfprintf(stdout,"%c",(*savearg0));
                LOGfprintf(stdout,"\n\n");/*print command name in uppercase */
                savearg0 = argv[0]; /* so free works later */
                argv[0] = argv[1];
                argv[1][0] = '?';
                argv[1][1] = '\0';
                func(argc,argv);
                argv[1] = argv[0];
                argv[0] = savearg0;
              }
    else
      LOGfprintf(stderr,"\nUsage: help [command name]\n");
  PipeEnd();
  return 0;

BFLYTHROW("Cmd_help",0)
}


/***** pot ***/

Cmd_p(argc,argv)
     int argc;
     char ** argv;

{  Cmd_pot(argc,argv);
}

#ifndef LINDA								/* if not parallel version */
											/* (else the command will be in SIMAIN.C) */
Cmd_pot(argc,argv)
     int argc;
     char ** argv;

{
  register int u;
  register pot_type val;
  register Unit * up;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 3) goto synerror;
    Curarg = 1;

    while (argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case INT :
            val = Yyint;
            break;
          case FLOAT :
            val = Yyfloat;
            break;
          default:
            LOGfprintf(stdout,"Illegal potential: %s\n",argv[Curarg-1]);
            LOGfprintf(stdout,"Rest of line ignored\n\n");
            EAT;
            return 0;
          }
#ifdef BCNTL
        Set_Cmd_Numbs(3,Uset,Usetind,val);
        Send_Range_Cmd(POT_C,Ulow,Uhigh);
#else
        FOR_UNITS_P(u,up)
          up->potential = val;
#endif
      }
    return 0;

 helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The pot(ential) command is used to set the potential of one or more units.  It expects one or more unit-identifier/potential-value pairs.  The unit identifiers may specified in any of the usual ways (try `help UnitId' for information).\n\n");

  synerror:
    Format = FALSE;                                       /* print syntax */
    LOGfprintf(Dispf,"\nUsage: pot <UnitID> <value> [<UnitID> <value>]*\n\n");
    return 0;
BFLYTHROW("Cmd_pot",0)
} /* pot */

#endif

/***** out *****/
Cmd_o(argc,argv)
     int argc;
     char ** argv;

{Cmd_out(argc,argv);
}

#ifndef LINDA

Cmd_out(argc,argv)
     int argc;
     char ** argv;

{
  register int u;
  register Output val;
  register Unit * up;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 3) goto synerror;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case INT :
            val = Yyint;
            break;
          case FLOAT :
            val = Yyfloat;
            break;
          default:
            LOGfprintf(stdout,"Illegal output: %s\n",argv[Curarg-1]);
            LOGfprintf(stdout,"Rest of line ignored\n\n");
            EAT;
            return 0;
          }
#ifdef BCNTL
        Set_Cmd_Numbs(3,Uset,Usetind,val);
        Send_Range_Cmd(OUT_C,Ulow,Uhigh);
#else
        FOR_UNITS_P(u,up)
        {
#ifndef TSIM
            up->output = Outputs[u] = val;
#else
            up->output = *((Outputs[u])+1) = val;
#endif
        }
#endif
      }
    return 0;

  helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The out(put) command is used to set the output of one or more units.  It expects one or more unit-identifier/output-value pairs.  The unit identifiers may be specified in any of the usual ways (try `help UnitId' for information).\n\n");


  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: out <UnitID> <value> [<UnitID> <value>]*\n\n");
    return 0;
BFLYTHROW("Cmd_out",0)
} /* out */

#endif

#ifndef LINDA

/***** state *****/
Cmd_state(argc,argv)
     int argc;
     char ** argv;

{
    register int u;
    register int val;
    register Unit * up;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 3) goto synerror;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case INT :
            val = Yyint;
            break;
          default:      
            if((val = NameToState(Yyarg)) == -1)
              {
                LOGfprintf(stdout,"State not known: %s\n",argv[Curarg-1]);
                LOGfprintf(stdout,"Rest of line ignored\n\n");
                EAT;
                return 0;
              }
          }
#ifdef BCNTL
        Set_Cmd_Numbs(3,Uset,Usetind,val);
        Send_Range_Cmd(STATE_C,Ulow,Uhigh);
#else
        FOR_UNITS_P(u,up)
          up->state = val;
#endif
      }
    return 0;

  helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The state command is used to set the state of one or more units.  It expects one or more unit-identifier/state-value pairs.  The unit identifiers may be specified in any of the usual ways.  The state values may be a known state name or an integer.\n\n");

    
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: state <UnitID> <value> [<UnitID> <value>]*\n\n");
    return 0;
BFLYTHROW("Cmd_state",0)
} /* state */

#endif

#ifndef LINDA

/***** function *****/

Cmd_ufunc(argc,argv)
     int argc;
     char ** argv;

{
    register int u;
    register func_ptr func;
    register Unit * up;
    int flag;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 3) goto synerror;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case OFF:
            flag = -1;
            break;
          case ON:
            flag = 1;
            break;
          default:
            flag = 0;
            if((func = NameToFunc(Yyarg)) == NULL)
              {
                LOGfprintf(stdout,"function not known: %s\n",argv[Curarg-1]);
                LOGfprintf(stdout,"Rest of line ignored\n\n");
                EAT;
                return 0;
              }
          }
#ifdef BCNTL
            Set_Cmd_Numbs(2,Uset,Usetind);
            Set_Cmd_Names(1,Yyarg);
            Send_Range_Cmd(UFUNC_C,Ulow,Uhigh);
            /* see eg state command for impl */
#else
        if (flag == 0)
          {
            FOR_UNITS_P(u,up)
              up->unit_f = func; /* reset function */
          }
        else
          if (flag > 0)         /* turn unit on */
            FOR_UNITS_P(u,up)
              UnsetFlagP(up,NO_UNIT_FUNC_FLAG);
          else
            FOR_UNITS_P(u,up)   /* turn unit off */
              SetFlagP(up,NO_UNIT_FUNC_FLAG);
#endif
      }
    return 0;

  helpinfo:
  Format = TRUE;                /* print detailed help */
    LOGfprintf(Dispf,"The ufunc command is used to set the unit function of one or more units, or alternatively to turn the unit function on or off for one or more units.  It expects one or more unit-identifier/function-name pairs.  The unit identifiers may be specified in any of the usual ways.  The functions may be any global user function or a library function, or one of the keywords `on' or `off'.  If one of the keywords is given, the effect is to set the NO_UNIT_FUNC_FLAG on or off.\n\n");

    
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: ufunc <UnitID> <function> [<UnitID> <function>]*\n\n");
    return 0;
BFLYTHROW("Cmd_ufunc",0)
} /* function */

Cmd_sfunc(argc,argv)
     int argc;
     char ** argv;

{
    register int u;
    register func_ptr func;
    register Unit * up;
    char * sname;
    Site * sp;
    int flag;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 4) goto synerror;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        Lex(argv[Curarg]);      /* process it */
        sname = argv[Curarg-1];
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case OFF:
            flag = -1;
            break;
          case ON:
            flag = 1;
            break;
          default:
            flag = 0;
            if ((func = NameToFunc(Yyarg)) == NULL)
               {
                 LOGfprintf(stdout,"function not known: %s\n",argv[Curarg-1]);
                 LOGfprintf(stdout,"Rest of line ignored\n\n");
                 EAT;
                 return 0;
               }
          }

#ifdef BCNTL
        Set_Cmd_Numbs(2,Uset,Usetind);
        Set_Cmd_Names(2,sname,Yyarg);
        Send_Range_Cmd(FUNC_C,Ulow,Uhigh); /* see eg state command for impl */
#else
        if (!(strcmp(sname,"all")))
          {
            FOR_UNITS_P(u,up)
              for (sp = up->sites;
                   sp != NULL;
                   sp = sp->next)
                if (flag == 0)
                  sp->site_f = func;
                else
                  if (flag > 0)
                    UnsetFlagP(up,NO_SITE_FUNC_FLAG);
                  else
                    SetFlagP(up,NO_SITE_FUNC_FLAG);
          }
        else
          if (flag == 0)
            {
              FOR_UNITS_P(u,up)
                {
                  for (sp = up->sites;
                       sp != NULL && strcmp(sp->name,sname);
                       sp = sp->next);
                  if (sp != NULL)
                    sp->site_f = func;
                  else
                    LOGfprintf(stderr,
                               "Can't find site %s on unit %d\n",sname,u);
                }
            }
          else
            LOGfprintf(stderr,"Cannot turn single site function on or off\n");
#endif
      }
    return 0;

  helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The sfunc command is used to set the site function of one or all sites on one or more units, or alternatively to turn on or off all the site functions on one or more units  It expects one or more unit-identifier/site-name/function-name triples.  The unit identifiers may be specified in any of the usual ways.\n\n");
    LOGfprintf(Dispf,"The site name may be the name of any site on the unit, or `all' meaning all sites.  The functions may be any global user function or a library function, or the keywords `on' or `off'.  If one of the keywords is given, the sitename must be `all'.  The effect then is to set the NO_SITE_FUNC_FLAG on or off.\n\n");

    
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: sfunc <UnitID> <site> <func|on|off> [<UnitID> <site> <func|on|off>]*\n\n");
    return 0;
BFLYTHROW("Cmd_sfunc",0)
} /* function */

Cmd_lfunc(argc,argv)
     int argc;
     char ** argv;

{
    register int u;
    register func_ptr func;
    register Unit * up;
    char * sname;
    Site * sp;
    register Link * lp;
    int flag,ul,uh,us,usind;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 4) goto synerror;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        ul = Ulow; uh = Uhigh; us = Uset; usind = Usetind;
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        Lex(argv[Curarg]);      /* process it */
        sname = argv[Curarg-1];
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case OFF:
            flag = -1;
            break;
          case ON:
            flag = 1;
            break;
          default:
            flag = 0;
            if ((func = NameToFunc(Yyarg)) == NULL)
               {
                 LOGfprintf(stdout,"function not known: %s\n",argv[Curarg-1]);
                 LOGfprintf(stdout,"Rest of line ignored\n\n");
                 EAT;
                 return 0;
               }
          }

#ifdef BCNTL
        Set_Cmd_Numbs(6,Uset,Usetind,ul,uh,us,usind);
        Set_Cmd_Names(2,sname,Yyarg);
        Send_Range_Cmd(FUNC_C,Ulow,Uhigh); /* see eg state command for impl */
#else
        if (!(strcmp(sname,"all")))
          {
            if (flag == 0)      /* reset to new link function */
              FOR_UNITS_P(u,up) /* for all destination units */
                for (sp = up->sites; /* for all sites */
                     sp != NULL;
                     sp = sp->next)
                  for (lp = sp->inputs; lp != NULL; lp = lp->next)
                    {
                      if (lp->from_unit >= ul && lp->from_unit <= uh &&
                          (!us || (UnitList[lp->from_unit].sets >> usind) & 1))
                        lp->link_f = func;
                    }
            else
              if (ul == 0 && uh == NoUnits-1 && !us) /* source all units */
                if (flag > 0)
                  FOR_UNITS_P(u,up)     /* for all destination units */
                    UnsetFlagP(up,NO_LINK_FUNC_FLAG);
                else
                  FOR_UNITS_P(u,up)     /* for all destination units */
                    SetFlagP(up,NO_LINK_FUNC_FLAG);
              else
                LOGfprintf(stderr,"Can only turn linkfunction on or off for all links to a unit\n");
          }
        else                    /* a particular site */
          if (flag == 0)        /* so should be function change */
            {
              FOR_UNITS_P(u,up)
                {               /* for destination units */
                  for (sp = up->sites; /* find correct site */
                       sp != NULL && strcmp(sp->name,sname);
                       sp = sp->next);
                  if (sp != NULL) /* found ok */
                    for (lp = sp->inputs; lp != NULL; lp = lp->next)
                      {         /* for all links */
                        if (lp->from_unit >= ul && lp->from_unit <= uh &&
                          (!us || (UnitList[lp->from_unit].sets >> usind) & 1))
                         lp->link_f = func; /* change function */
                      }
                  else          /* can't find site */
                    LOGfprintf(stderr,
                               "Can't find site %s on unit %d\n",sname,u);
                }
            }
          else
            LOGfprintf(stderr,"Can only turn linkfunction on or off for all links to a unit\n");
#endif
      }
    return 0;

  helpinfo:
  Format = TRUE;                /* print detailed help */
    LOGfprintf(Dispf,"The lfunc command is used to set the link function of links from one or more units arriving at one or all sites on one or more units, or alternatively to turn on or off all the link functions on one or more units.  It expects one or more unit-identifier/unit-identifier/site-name/function-name quadruples.  The unit identifiers may be specified in any of the usual ways (type `help UnitId' for information).\n\n");
    LOGfprintf(Dispf,"The first unit-identifier specifies the units at which the affected links originate.  The second unit-identifier specifies the units at which the affected links arrive.  The site name is that of a site on the destination unit(s), or `all' meaning all sites.\n\n");
    LOGfprintf(Dispf,"The function may be any global user function or a library function, or the keywords `on' or `off'.  If one of the keywords is given, the sitename must be `all' and the source units (the first unit-identifier) must be all the units.  The effect then is to set the NO_LINK_FUNC_FLAG on or off.\n\n");
    
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: lfunc [<SourceUnitId> <DestinationUnitID> <site> <func|on|off>]+\n\n");
    return 0;
BFLYTHROW("Cmd_lfunc",0)
} /* function */

#endif

#ifdef TSIM
Cmd_delay(argc,argv)
     int argc;
     char ** argv;

{
    register int u;
    register func_ptr func;
    register Unit * up;
    char * sname;
    Site * sp;
    register Link * lp;
    int flag,ul,uh,us,usind,delay;

    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 4) goto synerror;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        ul = Ulow; uh = Uhigh; us = Uset; usind = Usetind;
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        Lex(argv[Curarg]);      /* process it */
        sname = argv[Curarg-1];
        if (argc <= Curarg) goto synerror;
        if (Lex(argv[Curarg]) != INT)
          goto synerror;
        if ((delay = Yyint) < 1 || delay > 255)
          {
            LOGfprintf(stderr,"delay must be between 1 and 255\n");
            return 0;
          }

        if (!(strcmp(sname,"all")))
          {
            FOR_UNITS_P(u,up)   /* for all destination units */
              for (sp = up->sites; /* for all sites */
                   sp != NULL;
                   sp = sp->next)
                for (lp = sp->inputs; lp != NULL; lp = lp->next)
                  {
                    if (lp->from_unit >= ul && lp->from_unit <= uh &&
                        (!us || (UnitList[lp->from_unit].sets >> usind) & 1))
                      {
                        if (*(Outputs[lp->from_unit]) < delay)
                          ChangeDelayBuffer(lp->from_unit,delay);
                        lp->value = Outputs[lp->from_unit] + delay;
                      }
                  }
          }
        else                    /* a particular site */
          {
            FOR_UNITS_P(u,up)
              {         /* for destination units */
                for (sp = up->sites; /* find correct site */
                     sp != NULL && strcmp(sp->name,sname);
                     sp = sp->next);
                if (sp != NULL) /* found ok */
                  for (lp = sp->inputs; lp != NULL; lp = lp->next)
                    {           /* for all links */
                      if (lp->from_unit >= ul && lp->from_unit <= uh &&
                          (!us || (UnitList[lp->from_unit].sets >> usind) & 1))
                      {
                        if (*(Outputs[lp->from_unit]) < delay)
                          ChangeDelayBuffer(lp->from_unit,delay);
                        lp->value = Outputs[lp->from_unit] + delay;
                      }
                    }
                else            /* can't find site */
                  LOGfprintf(stderr,
                             "Can't find site %s on unit %d\n",sname,u);
              }
          }
      }
    return 0;

  helpinfo:
  Format = TRUE;                /* print detailed help */
    LOGfprintf(Dispf,"The delay command is used to set the propagation delay on links from one or more units arriving at one or all sites on one or more units.  It expects one or more unit-identifier/unit-identifier/site-name/delay-value quadruples.  The unit identifiers may be specified in any of the usual ways (type `help UnitId' for information).\n");
    LOGfprintf(Dispf,"The first unit-identifier specifies the units at which the affected links originate.  The second unit-identifier specifies the units at which the affected links arrive.  The site name is that of a site on the destination unit(s), or `all' meaning all sites.  The delay may be any integer value from 1 to 255.  Note that increasing the delay on a link can take considerable time if your network is large.\n");
    
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: delay [<SourceUnitId> <DestinationUnitID> <site> <delay-value>]+\n\n");
    return 0;
} /* function */
#endif

/***** weight *****/

Cmd_w(argc,argv)
     int argc;
     char ** argv;

{
  Cmd_weight(argc,argv);
}

#ifndef LINDA

Cmd_weight(argc,argv)
     int argc;
     char ** argv;

{
    int u;
    weight_type val, pert;
    char * sitename;
    short ul,uh,us,usind,typ;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc == 2 && Lex(argv[1]) == RANDOM)
      {
#ifdef BFLY
        Set_Cmd_Numbs(2,500,100);
        Send_Range_Cmd(RWEIGHT_C,0,NoUnits-1);
#else
        RandomiseWeights(500,100);
#endif
        return(0);
      }
    if (argc < 5) goto synerror;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        ul = Ulow; uh = Uhigh; us = Uset; usind = Usetind;
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        sitename = argv[Curarg];
        u = Lex(argv[Curarg]);  /* read ident */
        if (u == HELP || u == INT || u == FLOAT || u == PLUS ||
            u == MINUS || u == CBRACK || u == OBRACK)
          goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case INT :
            typ = FALSE;
            val = Yyint;
            break;
          case FLOAT :
            typ = FALSE;
            val = Yyfloat;
            break;
          case RANDOM:
            typ = TRUE;
            if (argc <= Curarg)
              val = 500;
            else
              switch(Lex(argv[Curarg]))
                {
                case INT:
                  val = Yyint;
                  break;
                case FLOAT:
                  val = Yyfloat;
                  break;
                default:
                  goto synerror;
                }
            if (argc <= Curarg)
              pert = 200;
            else
              switch(Lex(argv[Curarg]))
                {
                case INT:
                  pert = Yyint;
                  break;
                case FLOAT:
                  pert = Yyfloat;
                  break;
                default:
                  goto synerror;
                }
            break;
          default:
            LOGfprintf(stdout,"Unknown weight value: %s\n",argv[--Curarg]);
            EAT;
            return 0;
          }
#ifdef BCNTL
        Set_Cmd_Numbs(9,ul,uh,us,usind,Uset,Usetind,typ,val,pert);
        Set_Cmd_Names(1,sitename);
        Send_Range_Cmd(WEIGHT_C,Ulow,Uhigh);
#else
        SetWeight(ul,uh,us,usind,Ulow,Uhigh,Uset,Usetind,
                  sitename,typ,val,pert);
#endif
      }
    return 0;

  helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The weight command sets the value of a weight on a link.");
	
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: weight [<From-UnitID> <To-UnitID> <To-site> <value| random [<mean> <deviation>]>]+\n\n");
    return 0;
BFLYTHROW("Cmd_weight",0)
} /* weight */

#endif

/***** list *****/
Cmd_l(argc,argv)                /* ?????????????????? */
     int argc;
     char ** argv;

{Cmd_list(argc,argv);
}

Cmd_list(argc,argv)
    int argc;
    char ** argv;

{
    int r,ct,i;
    register int u;
    register Unit * up;

BFLYCATCH
    switch(argc)
      {
      case 1:
        goto synerror;
      case 2:
        Curarg=1;
        switch(Lex(argv[Curarg]))
          {
          case HELP:
            goto helpinfo;
          case LINK :
            ListLinks();
            break;
          case SET :
            for(i = ct = 0;i < LastSet;i++) 
              if(SetNames[i] != NULL)
                {
                  LOGfprintf(stdout,"\t%s\n",SetNames[i]);
                  ct++;
                }
            LOGfprintf(stdout,"\n\t%d set slots available\n",LastSet-ct);
            break;
          default:
            goto synerror;
          }
        break;
      default:
        Curarg=1;
        if (Lex(argv[Curarg]) == UNIT)
          {
            if(!(r = GetUnits(argc,argv))) return 0;
#ifdef BCNTL
            if (r == ALL)
              {
                Set_Cmd_Numbs(2,FALSE,0);
                Send_Sync_Cmd(LISTUNIT_C,0,NoUnits-1);
              }
            else
              {
                Set_Cmd_Numbs(2,Uset,Usetind);
                Send_Sync_Cmd(LISTUNIT_C,Ulow,Uhigh);
              }
#else
            if(r == ALL)
              ListUnits(TRUE);
            else
              {
                FOR_UNITS_P(u,up)
                  SetFlagP(up,LIST_FLAG);
                ListUnits(FALSE);
                FOR_UNITS_P(u,up)
                  UnsetFlagP(up,LIST_FLAG);
              }
#endif
          }
        else
          goto synerror;
    }
    return 0;

  helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The l(ist) command is used to display information about links, units or sets.  `list link' displays the source unit index, the destination unit index, and the destination site, for each link.  `list set' displays the name and index of each current set, and the number of set slots remaining.  `list unit <UnitId>' displays the potential, output, data, and state of each unit specified by <UnitId> - type `help UnitId' for information on unit specification methods.\n");

  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: list <link|set|UnitId>\n\n");
    return 0;
BFLYTHROW("Cmd_list",0)
} /* list */



/***** disp *****/
Cmd_d(argc,argv)                /* ??????????????? */
     int argc;
     char ** argv;

{Cmd_disp(argc,argv);
}


Cmd_disp(argc,argv)
    int argc;
    char ** argv;

{
  register int u;
  register Unit * up;

BFLYCATCH
    if ((argc == 2) && (Lex(argv[1]) == HELP))
      goto helpinfo;
    if (argc > 2)
      {      
        Curarg=1;                                 /* first command argument */
        if(Lex(argv[Curarg]) == UNIT)
          {
            if(!GetUnits(argc,argv)) return 0;
#ifdef BCNTL
            Set_Cmd_Numbs(2,Uset,Usetind);
            Send_Sync_Cmd(DISP_C,Ulow,Uhigh);
#else
            PipeBegin();
            FOR_UNITS_P(u,up)
              DisplayUnitP(u,up);
            PipeEnd();
#endif
            return 0;
          }
        else
          EAT;
      }
    else
      goto synerror;

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The d(isp) command is used to display the values associated with one or more units, for instance the potential, output, state, functions, site names and values, link weights and values.\n");

  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: disp u <UnitID>\n\n");
    return 0;
BFLYTHROW("Cmd_disp",0)
} /* disp */


/*****  status *****/
/*
   Print key facts about the network and simulator state.
*/
Cmd_status(argc,argv)
    int argc;
    char ** argv;

{
  char str[256];
BFLYCATCH
    if ((argc == 2) && (Lex(argv[1]) == HELP))
      goto helpinfo;
    if (argc > 1)
      goto synerror;
    LOGfprintf(stdout,"Rochester Connectionist Simulator 4.1\n");
#ifdef FSIM
    sprintf(str,"floating point\t\t");
#else
    sprintf(str,"integer\t\t\t");
#endif
#ifdef TSIM
    sprintf(str+strlen(str),"with propagation delay simulation\n");
#else
    sprintf(str+strlen(str),"no propagation delay\n");
#endif
    LOGfprintf(stdout,"%s",str);
    LOGfprintf(stdout,"Clock: %d\t\tShow is %s\n",Clock,(Show ? "on" : "off"));
#ifdef FSIM
    LOGfprintf(stdout,"NoUnits: %d\t\tShowPot: %f\n",NoUnits,ShowPot);
#else
    LOGfprintf(stdout,"NoUnits: %d\t\tShowPot: %d\n",NoUnits,ShowPot);
#endif
    LOGfprintf(stdout,"NoLinks: %d\t\tNoSets: %d\n",NoLinks,NoSets);
    LOGfprintf(stdout,"Echo every %d steps\tPause is %s\n",EchoStep,(Pause?"on":"off"));
    LOGfprintf(stdout,"Pipe is %s\t\tPipeCommand is %s\n",(PipeFlag?"on":"off"),PipeCommand);
   switch (SyncFlag)
     {
     case ASYNC:
       LOGfprintf(stdout,"Simulation is asynchronous\n");
       break;
     case SYNC:
       LOGfprintf(stdout,"Simulation is synchronous\n");
       break;
     case FAIRASYNC:
       LOGfprintf(stdout,"Simulation is fair asynchronous\n");
       LOGfprintf(stdout,"%d%% of units per step, all units within %d steps\n",
                  ExecFract,ExecLimit);
       break;
     default:
       LOGfprintf(stdout,"internal error: simulation is of unknown type");
     }
#ifdef BCNTL
    Send_Sync_Cmd(STATUS_C,-1,-1);
#endif                
    return 0;

 helpinfo:
  Format = TRUE;
  LOGfprintf(Dispf,"The status command prints out information as to the current status of the simulation, e.g. Clock value, number of units, execution style, and much more.\n");

 synerror:
  Format = FALSE;
  LOGfprintf(Dispf,"\nUsage: status\n\n");
BFLYTHROW("Cmd_status",0)
}
