/*
 *
 * $Source: /filesv/usr/local/proj/sphinx/spx2/src/lib/cdb_gdbm/RCS/cdc_gdbm.c,v $
 *
 *
 *  MODULE NAME:    cdc_gdbm.c
 *
 *
 *  AUTHORS:
 *
 *	K. Alagappan
 *
 */


/*
 * COPYRIGHT (C) 1992 DIGITAL EQUIPMENT CORPORATION
 * ALL RIGHTS RESERVED
 *
 * "Digital Equipment Corporation authorizes the reproduction,
 * distribution and modification of this software subject to the following
 * restrictions:
 * 
 * 1.  Any partial or whole copy of this software, or any modification
 * thereof, must include this copyright notice in its entirety.
 *
 * 2.  This software is supplied "as is" with no warranty of any kind,
 * expressed or implied, for any purpose, including any warranty of fitness 
 * or merchantibility.  DIGITAL assumes no responsibility for the use or
 * reliability of this software, nor promises to provide any form of 
 * support for it on any basis.
 *
 * 3.  Distribution of this software is authorized only if no profit or
 * remuneration of any kind is received in exchange for such distribution. 
 * 
 * 4.  This software and all application programs are to be used only for
 * non-commercial purposes. However, media costs associated with the
 * distribution of the software or application programs may be recovered.
 *
 */


#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <strings.h>
#include <sys/file.h>

#include "gdbmdefs.h"
#include "systems.h"
#include "gdbmerrno.h"
#include "extern.h"

#include "cdc.h"
#include "cdc_db.h"
#include "BigNum.h"
#include "BigRSA.h"

#define CDC_DB_MAX_RETRY 5

#ifdef DEBUG
extern int debug;
extern long cdc_debug;
extern char *progname;
#endif
extern char *malloc();
extern int errno;

static  init = 0;
static char default_db_name[] = DBM_FILE;
static char *current_db_name = default_db_name;
static void encode_princ_key(), decode_princ_key();
static void encode_contents(), decode_contents();
static void cdc_dbl_fini();
static int cdc_dbl_lock();
static void cdc_dbl_unlock();
char keyed_name[FULLNAME_SZ];
char contents_buf[MAX_CDB_DATUM_SZ];

static struct timeval timestamp;/* current time of request */
static int non_blocking = 0;

extern gdbm_error gdbm_errno;

extern char *gdbm_version;

gdbm_file_info   *dbf;

/*
 * This module contains all of the code which directly interfaces to
 * the underlying representation of the certificate database; this
 * implementation uses a GDBM indexed "file"
 */


/*
 * initialization for data base routines.
 */

static char *gen_dbsuffix(db_name, sfx)
    char *db_name;
    char *sfx;
{
    char *dbsuffix;

    if (sfx == NULL)
        sfx = ".ok";

    dbsuffix = malloc (strlen(db_name) + strlen(sfx) + 1);
    strcpy(dbsuffix, db_name);
    strcat(dbsuffix, sfx);
    return dbsuffix;
}


cdc_db_init()
{
    init = 1;
    return (0);
}

/*
 * gracefully shut down database--must be called by ANY program that does
 * a cdc_db_init 
 */

cdc_db_fini()
{
}

/*
 * Set the "name" of the current database to some alternate value.
 *
 * Passing a null pointer as "name" will set back to the default.
 * If the alternate database doesn't exist, nothing is changed.
 */

cdc_db_set_name(name)
	char *name;
{

    if (name == NULL)
	name = default_db_name;
    dbf = gdbm_open(name, 512, GDBM_READER, 00664, NULL);
    if (dbf == NULL) {
      return errno;
    }
    gdbm_close(dbf);
    cdc_dbl_fini();
    current_db_name = name;
    return 0;
}

/*
 * Return the last modification time of the database.
 */

long cdc_get_db_age()
{
    struct stat st;
    char *okname;
    long age;
    
    okname = gen_dbsuffix(current_db_name, ".ok");

    if (stat (okname, &st) < 0)
	age = 0;
    else
	age = st.st_mtime;

    free (okname);
    return age;
}

/*
 * Remove the semaphore file; indicates that database is currently
 * under renovation.
 *
 * This is only for use when moving the database out from underneath
 * the server (for example, during slave updates).
 */

static long cdc_start_update(db_name)
    char *db_name;
{
    char *okname = gen_dbsuffix(db_name, ".ok");
    long age = cdc_get_db_age();
    
    if (unlink(okname) < 0
	&& errno != ENOENT) {
	    age = -1;
    }
    free (okname);
    return age;
}

static long cdc_end_update(db_name, age)
    char *db_name;
    long age;
{
    int fd;
    int retval = 0;
    char *new_okname = gen_dbsuffix(db_name, ".ok#");
    char *okname = gen_dbsuffix(db_name, ".ok");
    
    fd = open (new_okname, O_CREAT|O_RDWR|O_TRUNC, 0600);
    if (fd < 0)
	retval = errno;
    else {
	struct stat st;
	struct timeval tv[2];
	/* make sure that semaphore is "after" previous value. */
	if (fstat (fd, &st) == 0
	    && st.st_mtime <= age) {
	    tv[0].tv_sec = st.st_atime;
	    tv[0].tv_usec = 0;
	    tv[1].tv_sec = age;
	    tv[1].tv_usec = 0;
	    /* set times.. */
	    utimes (new_okname, tv);
	    fsync(fd);
	}
	close(fd);
	if (rename (new_okname, okname) < 0)
	    retval = errno;
    }

    free (new_okname);
    free (okname);

    return retval;
}

static long cdc_start_read()
{
    return cdc_get_db_age();
}

static long cdc_end_read(age)
    u_long age;
{
    if (cdc_get_db_age() != age || age == -1) {
	return -1;
    }
    return 0;
}

/*
 * Create the database, assuming it's not there.
 */

cdc_db_create(db_name)
    char *db_name;
{
    char *okname = gen_dbsuffix(db_name, ".ok");
    int fd;
    register int ret = 0;

    dbf = gdbm_open(db_name, 512, GDBM_NEWDB, 00664, NULL);
    if (dbf == NULL) {
	ret = errno;
    } else
	gdbm_close(dbf);

    if (ret == 0) {
	fd = open (okname, O_CREAT|O_RDWR|O_TRUNC, 0644);
	if (fd < 0)
	    ret = errno;
	close(fd);
    }
    free (okname);
    return ret;
}

/*
 * "Atomically" rename the database in a way that locks out read
 * access in the middle of the rename.
 *
 * Not perfect; if we crash in the middle of an update, we don't
 * necessarily know to complete the transaction the rename, but...
 */

cdc_db_rename(from, to)
    char *from;
    char *to;
{
    char *fromok = gen_dbsuffix(from, ".ok");
    long trans = cdc_start_update(to);
    int ok;
    
    if (rename(from, to) == 0) {
	(void) unlink (fromok);
	ok = 1;
    }

    free (fromok);
    if (ok)
	return cdc_end_update(to, trans);
    else
	return -1;
}

static void
encode_princ_key(key, name, type, index_to_use)
datum  *key;
char   *name;
int    type;
int    index_to_use;
{
  int  len, c_index = 0, i, c;
  char index_buf[4], upname[FULLNAME_SZ];

  len = strlen(name);
  bzero(keyed_name,sizeof(keyed_name));
  strcpy(keyed_name,name);
  for (i=0; i<len; i++) {
    c = keyed_name[i];
    if (islower(c) != 0) keyed_name[i] = c - 32;
  }
  keyed_name[len] = '\0';
  strcpy(upname, keyed_name);
  key->dptr = keyed_name;
  switch (type) {
  case ENC_PRIVKEY_ATTRIB :
    keyed_name[len++] = ',';
    keyed_name[len++] = 'E';
    keyed_name[len] = '\0';
    break;
  case FLAG_ATTRIB :
    keyed_name[len++] = ',';
    keyed_name[len++] = 'F';
    keyed_name[len] = '\0';
    break;
  case CERTIF_ATTRIB :
  case TA_CERTIF_ATTRIB :
    keyed_name[len++] = ',';
    if (index_to_use < 0) c_index = get_c_index(upname,type);
    else c_index = index_to_use;
    sprintf(index_buf, "%d", c_index);
    strcpy(&keyed_name[len],index_buf);
    len = len + strlen(index_buf);
    if (type == CERTIF_ATTRIB) keyed_name[len++] = 'C';
    else keyed_name[len++] = 'T';
    keyed_name[len] = '\0';
    break;
  }
  key->dsize = len;
}

int get_c_index(name, type)
char *name;
int  type;
{
  char  key_name[FULLNAME_SZ], base_name[FULLNAME_SZ], index_b[4];
  int   index = 0, len, base_len;
  datum contents, key;

  if ((type != CERTIF_ATTRIB) && (type != TA_CERTIF_ATTRIB))  return(0);

  if (!init)
    cdc_db_init();           /*  initialize database routines  */
  cdc_start_read();

  len = strlen(name);
  bzero(base_name,sizeof(base_name));
  strcpy(base_name,name);
  base_name[len++] = ',';
  base_name[len] = '\0';
  base_len = len;
  key.dptr = key_name;
  do {
    strcpy(key_name, base_name);
    sprintf(index_b, "%d", index);
    strcpy(&key_name[base_len],index_b);
    len = base_len + strlen(index_b);
    if (type == CERTIF_ATTRIB) key_name[len++] = 'C';
    else key_name[len++] = 'T';
    key_name[len] = '\0';
    key.dsize = len;
    contents = gdbm_fetch(dbf, key);
    if (contents.dptr != NULL)  index++;
  } while (contents.dptr != NULL);
  return(index);
}

static void
decode_princ_key(key, name, type)
    datum  *key;
    char   *name;
    int    *type;
{
  int len, i;

  len = key->dsize;
  bzero(keyed_name,sizeof(keyed_name));
  strncpy(keyed_name, key->dptr, len);

  *type = 0;
  strncpy(name, keyed_name, len-2);
  switch (keyed_name[len-1]) {
  case 'E' :
    *type = ENC_PRIVKEY_ATTRIB;
    break;
  case 'F' :
    *type = FLAG_ATTRIB;
    break;
  case 'C' :
    *type = CERTIF_ATTRIB;
    i = strlen(name);
    while (name[i] != ',')  i--;
    name[i] = '\0';
    break;
  case 'T' :
    *type = TA_CERTIF_ATTRIB;
    i = strlen(name);
    while (name[i] != ',')  i--;
    name[i] = '\0';
    break;
  default :
    break;
  }
}

static void
encode_contents(contents, data, sz)
    datum  *contents;
    char   *data;
    int    sz;
{
  bzero(contents_buf,MAX_CDB_DATUM_SZ);
  bcopy(data, contents_buf, sz);
  contents->dptr = contents_buf;
  contents->dsize = sz;
}

static void
decode_contents(contents, data, sz)
    datum  *contents;
    char   *data;
    int    sz;
{
    bcopy(contents->dptr, (char *) data, sz);
}

static int dblfd = -1;
static int mylock = 0;
static int inited = 0;

static cdc_dbl_init()
{
    if (!inited) {
	char *filename = gen_dbsuffix (current_db_name, ".ok");
	if ((dblfd = open(filename, 0)) < 0) {
	    fprintf(stderr, "cdc_dbl_init: couldn't open %s\n", filename);
	    fflush(stderr);
	    perror("open");
	    return(-1);
	}
	free (filename);
	inited++;
    }
    return (0);
}

static void cdc_dbl_fini()
{
    close(dblfd);
    dblfd = -1;
    inited = 0;
    mylock = 0;
}

static int cdc_dbl_lock(mode)
    int     mode;
{
    int flock_mode;
    
    if (!inited)
	cdc_dbl_init();
    if (mylock) {		/* Detect lock call when lock already
				 * locked */
	fprintf(stderr, "CDC locking error (mylock)\n");
	fflush(stderr);
	return(-1);
    }
    switch (mode) {
    case CDC_DBL_EXCLUSIVE:
	flock_mode = LOCK_EX;
	break;
    case CDC_DBL_SHARED:
	flock_mode = LOCK_SH;
	break;
    default:
	fprintf(stderr, "invalid lock mode %d\n", mode);
	abort();
    }
    if (non_blocking)
	flock_mode |= LOCK_NB;
    
    if (flock(dblfd, flock_mode) < 0) 
	return errno;
    mylock++;
    return 0;
}

static void cdc_dbl_unlock()
{
    if (!mylock) {		/* lock already unlocked */
	fprintf(stderr, "CDC database lock not locked when unlocking.\n");
	fflush(stderr);
	exit(-1);
    }
    if (flock(dblfd, LOCK_UN) < 0) {
	fprintf(stderr, "CDC database lock error. (unlocking)\n");
	fflush(stderr);
	perror("flock");
	exit(-1);
    }
    mylock = 0;
}

int cdc_db_set_lockmode(mode)
    int mode;
{
    int old = non_blocking;
    non_blocking = mode;
    return old;
}

print_datum(d)
datum *d;
{
  int i;

  for (i=0;i<d->dsize;i++)  printf("%c",d->dptr[i]);
}

/* ====================  database access routines =======================*/

int cdb_read_entry(name, type, value, sz, index)
char *name;
int type;
char *value;
int  *sz;
int  *index;   /* to recursively read certificates */
{
  datum    key, contents;
  u_long   trans;
  int     found = 0;
  extern int errorproc();
  int     try, local_index;

    if (!init)
	cdc_db_init();		/* initialize database routines */

    for (try = 0; try < CDC_DB_MAX_RETRY; try++) {
	trans = cdc_start_read();

	if ((cdc_dbl_lock(CDC_DBL_SHARED)) != 0)
	    return(CDB_ERROR);

	if ((dbf = gdbm_open(current_db_name, 512, GDBM_READER, 0664, NULL)) == NULL) {
	  printf("error opening %s (cdb_read_entry) - gdbm_error is %d\n",current_db_name, gdbm_errno);
	  return(-1);
	}

	local_index = *index;
	encode_princ_key(&key, name, type, *index);
	contents = gdbm_fetch(dbf, key);
	if (contents.dptr != NULL) {
	  decode_contents(&contents, value, contents.dsize);
	  *sz = contents.dsize;
	  local_index++;
	  found = CDB_SUCCESS;
	  free(contents.dptr);
	  encode_princ_key(&key, name, type, local_index);
	  contents = gdbm_fetch(dbf, key);
	  if (contents.dptr == NULL)  local_index = 0;
	  else free(contents.dptr);
	} else {
	  found = CDB_UNKNOWN_KEY;
	  local_index = 0;
	  *sz = local_index;   /*  set size to zero, since value not found  */
	}
	*index = local_index;
	cdc_dbl_unlock();	/* unlock read lock */
	gdbm_close(dbf);
	if (cdc_end_read(trans) == 0)
	    break;
	found = CDB_ERROR;
	if (!non_blocking)
	    sleep(1);
    }
    return (found);
}

int cdb_add_entry(name, type, value, sz)
char *name;
int type;
char *value;
int  sz;
{
  datum    key, contents;
  int     found = 0;
  extern int errorproc();

    gettimeofday(&timestamp, NULL);
    if (!init)
	cdc_db_init();

    if ((cdc_dbl_lock(CDC_DBL_EXCLUSIVE)) != 0)
	return(-1);
    dbf = gdbm_open(current_db_name, 512, GDBM_WRCREAT, 00664, NULL);
    if (dbf == NULL)  {
      printf("unable to write in dbm - permission denied\n");
      cdc_dbl_unlock();		/* unlock database */
      return(-1);
    }
    encode_princ_key(&key, name, type, -1);
    encode_contents(&contents, value, sz);
    gdbm_store(dbf, key, contents, GDBM_REPLACE);
    gdbm_close(dbf);
    cdc_dbl_unlock();		/* unlock database */
    return (found);
}

int cdb_replace_entry(name, type, value, sz, index_to_use)
char *name;
int type;
char *value;
int  sz;
int  index_to_use;
{
  datum    key, contents;
  int     found = 0;
  extern int errorproc();

    gettimeofday(&timestamp, NULL);
    if (!init)
	cdc_db_init();

    if ((cdc_dbl_lock(CDC_DBL_EXCLUSIVE)) != 0)
	return(-1);
    dbf = gdbm_open(current_db_name, 512, GDBM_WRCREAT, 00664, NULL);
    if (dbf == NULL)  {
      printf("unable to write in dbm - permission denied\n");
      cdc_dbl_unlock();		/* unlock database */
      return(-1);
    }
    encode_princ_key(&key, name, type, index_to_use);
    encode_contents(&contents, value, sz);
    gdbm_store(dbf, key, contents, GDBM_REPLACE);
    gdbm_close(dbf);
    cdc_dbl_unlock();		/* unlock database */
    return (found);
}

int cdb_load_entry(k, value, sz)
char *k;
char *value;
int  sz;
{
  datum    key, contents;
  int     found = 0;
  extern int errorproc();

    gettimeofday(&timestamp, NULL);
    if (!init)
	cdc_db_init();

    if ((cdc_dbl_lock(CDC_DBL_EXCLUSIVE)) != 0)
	return(-1);
    dbf = gdbm_open(current_db_name, 512, GDBM_WRCREAT, 00664, NULL);
    if (dbf == NULL)  {
      printf("unable to write in dbm - permission denied\n");
      cdc_dbl_unlock();		/* unlock database */
      return(-1);
    }

    key.dptr = k;
    key.dsize = strlen(k);

    contents.dptr = value;
    contents.dsize = sz;

    gdbm_store(dbf, key, contents, GDBM_REPLACE);
    gdbm_close(dbf);
    cdc_dbl_unlock();		/* unlock database */
    return (found);
}

cdb_read_contents(more, key_name, contents_val, contents_len, contents_type)
   int    *more;
   char key_name[MAX_CDB_KEY_SZ];
   char contents_val[PRIVATE_KEY_SIZE];
   int  *contents_len;
   int  *contents_type;
{
    int     found = 0;
    extern int errorproc();
    datum   key, contents;
    char    testname[FULLNAME_SZ];
    int     testtype;
    int try, i, local_more;
    u_long   trans;

    if (!init)
	cdc_db_init();		/* initialize database routines */

    for (try = 0; try < CDC_DB_MAX_RETRY; try++) {
	trans = cdc_start_read();

	if ((cdc_dbl_lock(CDC_DBL_SHARED)) != 0)
	    return -1;

	dbf = gdbm_open(current_db_name, 512, GDBM_READER, 0600, NULL);
	if (dbf == NULL) return(-1);

	local_more = *more;
	key = gdbm_firstkey(dbf);
	for (i = 0; i < local_more; i++) {
	  key = gdbm_nextkey(dbf,key);
	}

	if (key.dptr != NULL) {
	  contents = gdbm_fetch(dbf, key);
	  local_more++;
	  decode_princ_key(&key, testname, &testtype);
	  strncpy(key_name, key.dptr, key.dsize);
	  if (contents.dptr != NULL)
	    decode_contents(&contents, contents_val, contents.dsize);
	  *contents_len = contents.dsize;
	  *contents_type = testtype;
	  key = gdbm_nextkey(dbf, key);
	  if (key.dptr == NULL)  local_more = 0;
	} else local_more = 0;
	*more = local_more;
	cdc_dbl_unlock();	/* unlock read lock */
	gdbm_close(dbf);
	if (cdc_end_read(trans) == 0)
	    break;
	found = CDB_ERROR;
	if (!non_blocking)
	    sleep(1);
    }
    return (found);
}

int cdb_delete_entry(name)
char *name;
{
  datum    key, contents;
  int     found = 0;
  extern int errorproc();
  extern int errorproc();
  int     try;

  if (!init)
    cdc_db_init();		/* initialize database routines */

  if ((cdc_dbl_lock(CDC_DBL_EXCLUSIVE)) != 0)
    return(-1);

  dbf = gdbm_open(current_db_name, 512, GDBM_WRCREAT, 00664, NULL);
  if (dbf == NULL)  {
    printf("unable to write in dbm - permission denied\n");
    cdc_dbl_unlock();		/* unlock database */
    return(-1);
  }

  /*
   *  delete ,E entry
   */

  encode_princ_key(&key, name, ENC_PRIVKEY_ATTRIB, -1);
  gdbm_delete(dbf, key);

  /*
   *  delete ,F entry
   */
  encode_princ_key(&key, name, FLAG_ATTRIB, -1);
  gdbm_delete(dbf, key);

  /*
   *  delete ,T entry
   */

  for (try=0; try < 5; try++){
    encode_princ_key(&key, name, CERTIF_ATTRIB, try);
    if (gdbm_delete(dbf, key) < 0) break;
  }


  /*
   *  delete ,C entry
   */

  for (try=0; try < 5; try++) {
    encode_princ_key(&key, name, TA_CERTIF_ATTRIB, try);
    if (gdbm_delete(dbf, key) < 0) break;
  }

  gdbm_close(dbf);
  cdc_dbl_unlock();	/* unlock read lock */
  return(1);
}
