#############################################################################
##
#A  Matrix Group and G-module library                   Derek Holt
#A                                                      Charles Leedham-Green
#A                                                      Eamonn O'Brien
#A                                                      Sarah Rees 
##
#A  @(#)$Id$
##
#Y  Copyright 1994 -- School of Mathematical Sciences, ANU   
##
#H  $Log$
##
#############################################################################
#
# function to select various kinds of random elements 
#
#############################################################################
##
#F ElementOfOrder (Seed, RequiredOrder, NmrTries) . . . . . 
##
## try to find element of RequiredOrder in NmrTries attempts 
## 
ElementOfOrder := function (Seed, RequiredOrder, NmrTries)

   local g, o, i, rem;

   i := 0;
   repeat
      g := RandomElement (Seed); 
      o := Order (FiniteFieldMatrices, g); 
      i := i + 1;
      rem := Mod (o, RequiredOrder) = 0; 
   until rem = true or i > NmrTries;
   
   if rem then 
      return g^(o / RequiredOrder);
   else 
      return false;
   fi;

end; #ElementOfOrder

#############################################################################
##
#F LargestPrimeOrderElement (Seed, NmrTries) . . . . 
## 
## generate NmrTries random elements and check which element of prime order 
## can be obtained from powers of each; return largest and its order 
##
LargestPrimeOrderElement := function (Seed, NmrTries)

   local i, g, o, p, prime, max, pos, z;
   
   g := [];
   o := [];
   prime := [];

   for i in [1..NmrTries] do 
      g[i] := RandomElement (Seed);
      o[i] := Order (FiniteFieldMatrices, g[i]);
      #store maximum prime which occurs in factorisation of this order 
      prime[i] := Maximum (Set (FactorsInt (o[i])));
   od;

   #maximum prime encountered 
   p := Maximum (prime);
   pos := Position (prime, p);
    
   z := g[pos]^(o[pos] / p);
   return [z, p];

end; #LargestPrimeOrderElement
 
#############################################################################
##
#F LargestPrimePowerOrderElement (Seed, NmrTries) . . . . 
## 
## generate NmrTries random elements and check which element of prime power
## order can be obtained from powers of each; return largest and its order 
##
LargestPrimePowerOrderElement := function (Seed, NmrTries)

   local i, g, pow, powers, o, prime, max, pos, z, j;
   
   g := [];
   o := [];
   prime := [];

   for i in [1..NmrTries] do 
      g[i] := RandomElement (Seed);
      o[i] := Order (FiniteFieldMatrices, g[i]);

      powers := PrimePowersInt (o[i]);

      max := 1;
      for j in [1..Length (powers) / 2] do
         pow := powers[2 * j - 1]^powers[2 * j];
         if pow > max then 
            max := pow;
         fi;
      od;
         
      #store maximum prime power 
      prime[i] := max;

   od;

   #find position of element of largest prime-power order in list 
   pos := Position (prime, Maximum (prime));
    
   z := g[pos]^(o[pos] / prime[pos]);
   return [z, prime[pos]];

end; #LargestPrimePowerOrderElement
 
#############################################################################
##
#F ChooseRandomElements (Seed, NmrElts) . . . choose NmrElts random elements
##
ChooseRandomElements := function (Seed, NmrElts)

   local w, i;

   w := [];
   for i in [1..NmrElts] do 
      w[i] := RandomElement (Seed);
   od;

   return w;

end; #ChooseRandomElements

#############################################################################
##
#F ElementWithCharPol (Seed, o, pol, Limit) . . 
##
## find element of order o with characteristic polynomial pol; 
## Limit is number of tries to find such element
##
ElementWithCharPol := function (Seed, o, pol, Limit)

   local NmrTries, z, cf, ValidElement;

   NmrTries := 0;
   repeat
      z := ElementOfOrder (Seed, o, Limit);
      if z = false then
         Print (Limit, " tries failed to find element of order ", o, "\n");
         return false;
      fi;
      cf := CharacteristicPolynomial (z);
      ValidElement := (pol = cf);
      NmrTries := NmrTries + 1;
   until ValidElement or NmrTries > Limit;

   if ValidElement = false then
      Print (Limit, " attempts failed to find an element of order ", o,
                    "with given characteristic polynomial\n");
      return false;
   fi;

   return z;

end; #ElementWithCharPol
