#############################################################################
##
#A  Matrix Group and G-module library                   Derek Holt
#A                                                      Charles Leedham-Green
#A                                                      Eamonn O'Brien
#A                                                      Sarah Rees 
##
#A  @(#)$Id$
##
#Y  Copyright 1994 -- School of Mathematical Sciences, ANU   
##
#H  $Log$
##
#############################################################################
##
#F  WreathProd(G,P) . . wreath product of a group and a permutation group
## 
##  G can be either (i) a matrix group, GModule of list of matrices,
##  or              (ii) a permutation group or list of permutations.
##  P  must be a permutation group or list of permutations.
##  WreathProd constructs the wreath product of G by P in its natural
##  action. This is as a matrix or permutation group of degree d*t, where
##  d  is the degree of  G  and  t  the degree of  P.
##  NOTE: There is a GAP library function WreathProduct, which uses th
##  regular representation of P by default.
WreathProd := function(G,P)
  local i,j,k,error,matgroup,F,Ggens,nGgens,Pgens,nPgens,d,t,gen,Wgens,Wgen;

# First sort out the what the first argument is.
  if IsRec(G) and IsGModule(G) then
    matgroup:=true;
    Ggens := MatricesFlag(G);
    nGgens := Length(Ggens);
    F := FieldFlag(G);
    d := DimFlag(G);
  else
    if IsGroup(G) then
      Ggens := G.generators;
    else
      Ggens := G;
    fi;
    error := false;
    if IsList(Ggens)=false or Length(Ggens)=0 then
      error:=true;
    else
      nGgens := Length(Ggens);
      if IsMat(Ggens[1]) then
        matgroup := true;
        F := Field(Flat(Ggens));
        d := Length(Ggens[1]);
      elif IsPerm(Ggens[1]) then
        matgroup := false;
        d := PermDegree(Ggens);
      else
        error := true;
      fi;     
    fi;
    if error then
      return Error(
"First argument must be a perm. or mat. group or list of perms. or mats.");
    fi;
  fi;

# Now check and work out the degree of the second argument.
  if IsGroup(P) then
    Pgens := P.generators;
  else
    Pgens := P;
  fi;
  error := false;
  if IsList(Pgens)=false or Length(Pgens)=0 then
    error:=true;
  else
    nPgens := Length(Pgens);
    if IsPerm(Pgens[1]) then
      t := PermDegree(Pgens);
    else     
      error := true;
    fi;
  fi;
  if error then
    return Error(
"Second argument must be a permutation group or list of permutations.");
  fi;

# Now we can start the calculation. First do mat group case then perm group
  Wgens := [];
  if matgroup then
# The generators consist of those of G  filled out with 1's,
# and those of P as permutation matrices on blocks
    for gen in Ggens do
      Wgen := NullMat(d*t,d*t,F);
      for j in [1..d] do for k in [1..d] do
        Wgen[j][k] := gen[j][k];
      od; od;
      for i in [d+1..d*t] do
        Wgen[i][i] := F.one;
      od;
      Add(Wgens,Wgen);
    od;
    for gen in Pgens do
      Wgen := NullMat(d*t,d*t,F);
      for i in [1..t] do for j in [1..d] do
        Wgen[(i-1)*d+j][(i^gen-1)*d+j] := F.one;
      od; od;
      Add(Wgens,Wgen);
    od;
  else
# The generators consist of those of G and those of P on blocks
    for gen in Ggens do
      Add(Wgens,gen);
    od;
    for gen in Pgens do
      Wgen := [];
      for i in [1..t] do for j in [1..d] do
        Wgen[(i-1)*d+j] := (i^gen-1)*d+j;
      od; od;
      Add(Wgens,PermList(Wgen));
    od;
  fi;

  return Group(Wgens, Wgens[1]^0);

end;

#############################################################################
##
#F  WreathPower(G,P) . . wreath power of a group and a permutation group
## 
##  G can be either (i) a matrix group, GModule of list of matrices,
##  or              (ii) a permutation group or list of permutations.
##  P  must be a permutation group or list of permutations.
##  WreathPower constructs the wreath product of G by P in its power
##  action. This is as a matrix or permutation group of degree d^t, where
##  d  is the degree of  G  and  t  the degree of  P.
WreathPower := function(G,P)
  local i,j,k,error,matgroup,F,Ggens,nGgens,Pgens,nPgens,d,t,gen,Wgens,Wgen;

# First sort out the what the first argument is.
  if IsRec(G) and IsGModule(G) then
    matgroup:=true;
    Ggens := MatricesFlag(G);
    nGgens := Length(Ggens);
    F := FieldFlag(G);
    d := DimFlag(G);
  else
    if IsGroup(G) then
      Ggens := G.generators;
    else
      Ggens := G;
    fi;
    error := false;
    if IsList(Ggens)=false or Length(Ggens)=0 then
      error:=true;
    else
      nGgens := Length(Ggens);
      if IsMat(Ggens[1]) then
        matgroup := true;
        F := Field(Flat(Ggens));
        d := Length(Ggens[1]);
      elif IsPerm(Ggens[1]) then
        matgroup := false;
        d := PermDegree(Ggens);
      else
        error := true;
      fi;     
    fi;
    if error then
      return Error(
"First argument must be a perm. or mat. group or list of perms. or mats.");
    fi;
  fi;

# Now check and work out the degree of the second argument.
  if IsGroup(P) then
    Pgens := P.generators;
  else
    Pgens := P;
  fi;
  error := false;
  if IsList(Pgens)=false or Length(Pgens)=0 then
    error:=true;
  else
    nPgens := Length(Pgens);
    if IsPerm(Pgens[1]) then
      t := PermDegree(Pgens);
    else     
      error := true;
    fi;
  fi;
  if error then
    return Error(
"Second argument must be a permutation group or list of permutations.");
  fi;

# Now we can start the calculation. First do mat group case then perm group
  Wgens := [];
  if matgroup then
# The generators consist of those of G  repeated in blocks of d on the
# d^t basis vectors and those of P in the product action
    for gen in Ggens do
      Wgen := NullMat(d^t,d^t,F);
      for i in [1..d^(t-1)] do for j in [1..d] do for k in [1..d] do
        Wgen[(i-1)*d+j][(i-1)*d+k] := gen[j][k];
      od; od; od;
      Add(Wgens,Wgen);
    od;
    for gen in Pgens do
      Wgen := NullMat(d^t,d^t,F);
      for i in [1..d^t] do
        Wgen[i][PowerPerm(gen,d,t,i)] := F.one;
      od;
      Add(Wgens,Wgen);
    od;
  else
# The generators consist of those of G  repeated in blocks of d on the
# d^t points and those of P in the product action
    for gen in Ggens do
      Wgen := [];
      for i in [1..d^(t-1)] do for j in [1..d] do
        Wgen[(i-1)*d+j] := (i-1)*d+j^gen;
      od; od;
      Add(Wgens,PermList(Wgen));
    od;
    for gen in Pgens do
      Wgen := [];
# We construct the generator as a list. The images of the points are
# calculated using PowerPerm.
      for i in [1..d^t] do
        Wgen[i] := PowerPerm(gen,d,t,i);
      od;
      Add(Wgens,PermList(Wgen));
    od;
  fi;

  return Group(Wgens, Wgens[1]^0);

end;


#############################################################################
##
#F  PowerPerm(p,d,t,pt) . . induced power action of a permutation on a point
## 
##  p is a permutation on t point. Let pp be the induced action of p on
##  the set of t-tuples of d points (by permuting coordinates).
##  PowerPerm returns the action of pp on the point pt.
PowerPerm := function(p,d,t,pt)
  local  i,tup,ttup,x;
  if pt<=0 or pt>d^t then
    return Error("pt must be in range [1..d^t].");
  fi;
  # Find the expression of pt as a t-tuple of d points.
  tup := [];
  x := pt-1;
  for i in [1..t] do
    tup[i] := RemInt(x,d)+1;
    x := QuoInt(x,d);
  od;
  # Now apply the permutation to tup to get ttup
  ttup := [];
  for i in [1..t] do
    ttup[i^p]:=tup[i];
  od;
  # Finally calculate point to return.
  x := 1;
  for i in [1..t] do
    x := x + (ttup[i]-1)*d^(i-1);
  od;
  return x;

end;

#############################################################################
##
#F PermDegree(permlist) . . Largest moved point of a list or permutations
##
##  PermDegree(permlist) returns the largest moved point of any permutation
##  in the list permlist. i.e. the degree of the permutation group that
##  they generate.
##  I don't understand why there appears to be no standard GAP function
##  to do this.
PermDegree := function(permlist)
  local i,l,lmpi,lmp,pi;
  l := Length(permlist);
  lmp := 0;
  for i in [1..l] do
    pi := permlist[i];
    if pi<>() then
      lmpi := LargestMovedPointPerm(pi);
      if lmpi > lmp then
        lmp := lmpi;
      fi;
    fi;
  od;
  return lmp;
end;
