/*

Signed arbitrary length long int package.

	Copyright Arjen K. Lenstra, 1989

	Bugs, questions, suggestions, additions, whatever, to
		Arjen K. Lenstra
		Room 2Q334
		Bellcore
		445 South Street
		Morristown, NJ 07960

		email: lenstra@flash.bellcore.com
		tel: 201 829 4878
		fax: 201 829 2645

*/

#include <stdio.h>
#include <math.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>
#include "lip.h"

#ifdef vax
#include <sys/types.h>
#endif

#ifndef vax
#include <malloc.h>
#include <stddef.h>
#endif

#if defined(sparc) && defined(__svr4__)
#include <sys/rusage.h>
#endif /* SOLARIS */

#ifdef NO_ALLOCATE
# define ALLOCATE 0
#else
# define ALLOCATE 1
#endif

#ifdef PRT_REALLOC
# undef PRT_REALLOC
# define PRT_REALLOC 1
#else
# define PRT_REALLOC 0
#endif

#define BYTEL           8	/* 8 bits per byte */
#ifdef __alpha
#define SIZEOFLONG      8     /* set this to SIZEOFLONG */
/* 60 bits doesn't work (explicit failure because of precision loss;
   probably because IEEE double mantissas are 53 bits).
   Why 52 doesn't work, I dunno (zmod seems to loop forever.  argh!) */
#else
#define SIZEOFLONG      4	/* set this to SIZEOFLONG */
#endif
#define NBITS           30	/* must be even and 0 < NBITS <BITSOFLONG */
#define BITSOFLONG      (BYTEL*SIZEOFLONG)
#define NBITSH          (NBITS>>1)
#define RADIX           (1L<<NBITS)
#define RADIXM          (RADIX-1)
#define RADIXROOT       (1L<<NBITSH)
#define RADIXROOTM      (RADIXROOT-1)
#define LOG10RAD        (double)9.03089986991943585644	/* base 10 log(RADIX) */
#define MAXDEPTH        20
#define MULT_CROSS      30
#define SQUARE_CROSS    30

#define SIZE            20	/* SIZE*NBITS must be >= BITSOFLONG */

#if (!(NBITS&1)&&(NBITS>0)&&(NBITS<BITSOFLONG)&&(SIZE*NBITS>=BITSOFLONG))

#define LINE            68	/* approximate bound # digits per line */
#define BUFSIZE         2048	/* at most 2048 characters per line */

#define PRIMBND         16500	/* to generate primes <= (2*PRIMBND+1)^2 */

/*Function prototypes for static internal functions.*/
static void zsubmul(
	long r,
	verylong a,
	verylong b
	);

static void hidetimer(
	FILE *f,
	long w
	);

static void zhalt(
	char *c
	);

static void zaddls(
	verylong a,
	verylong b,
	verylong c
	);

static void zmstartint(
        verylong n
        );

static void zmkeep(
        verylong n
        );

static void zmback(
        );

static void kar_mul(
	verylong a,
	verylong b,
	verylong *c,
	long shi
	);

static void kar_sq(
	verylong a,
	verylong *c,
	long shi
	);

static long zxxeucl(
	verylong ain,
	verylong nin,
	verylong *invv,
	verylong *uu
	);

static void mont_init_odd_power(
	verylong a,
	verylong *a_sq,
	long m,
	verylong **odd_a_power
	);

static void init_odd_power(
	verylong a,
	verylong n,
	verylong *a_sq,
	long m,
	verylong **odd_a_power
	);

static void zsmexp(
	long a,
	verylong e,
	verylong *bb
	);

static void zpshift(
	void
	);

static void zrootnewton(
	verylong a,
	verylong *b,
	long n
	);

static long ph1set(
	verylong n,
	verylong *rap,
	verylong *alpha,
	verylong *mu,
	verylong *x,
	verylong *f
	);

static long ph1exp(
	verylong n,
	verylong *x,
	verylong e,
	verylong rap,
	verylong *f
	);

static long ph1(
	verylong n,
	verylong *x,
	long m,
	verylong rap,
	verylong *f
	);

static long ph1toph2(
	verylong n,
	verylong *x,
	verylong *y,
	verylong *ra,
	verylong alpha,
	verylong mu,
	verylong *f
	);

static long ph2squ(
	verylong n,
	verylong x1s,
	verylong y1s,
	verylong *x2,
	verylong *y2,
	verylong ra,
	verylong *f
	);

static long ph2mul(
	verylong n,
	verylong x1s,
	verylong y1s,
	verylong x2s,
	verylong y2s,
	verylong *x3,
	verylong *y3,
	verylong ra,
	verylong *f
	);

static long ph2exp(
	verylong n,
	verylong x1,
	verylong yy1,
	verylong *x2,
	verylong *y2,
	long e,
	verylong ra,
	verylong *f
	);

static long ph2set(
	verylong n,
	verylong x,
	verylong y,
	long te,
	verylong ra,
	verylong *f
	);

static void ph2prec(
	verylong n,
	long n1,
	long j
	);

static void ph2eval(
	verylong n,
	long ind,
	long n1,
	long j,
	verylong *result
	);

static long ph2(
	verylong n,
	verylong inx,
	verylong iny,
	long m,
	long te,
	verylong ra,
	verylong *f
	);

static void message(
	char *str,
	double t
	);

static long trial(
	verylong n,
	long m,
	verylong *f,
	long info
	);


#if 1
/*
	This definition of zaddmulp and zaddmulpsq presumes a two's complement
	machine in which integer overflow is ignored and where double
	precision arithmetic is fast. The 0.25 allows round to nearest
	or round towards zero (value being rounded should be integer
	except for roundoff.)
*/

static double fradix_inv = 1.0 / RADIX;	/* Global constant */

#define zaddmulp(_a, _b, _d, _t) \
{ \
	register at = *(_a) + *(_t); \
	register long aa = (at + (_b) * (_d)) & RADIXM; \
 \
	*(_t) = (long) (0.25 + fradix_inv * (((double) (at - aa)) \
	               + ((double) (_b)) * ((double) (_d)))); \
	*(_a) = aa; \
}

#define zaddmulpsq(_a, _b, _t) \
{ \
	register at = *(_a); \
	register long aa = (at + (_b) * (_b)) & RADIXM; \
 \
	*(_t) = (long) (0.25 + fradix_inv * (((double) (at - aa)) \
				      + ((double) (_b)) * ((double) (_b)))); \
	*(_a) = aa; \
}
#endif

#if 0
/*
	This definition of zaddmulp and zaddmulpsq assumes nothing
*/
#define zaddmulp(_a, _b, _d, _t) \
{ \
	register long lb = (_b); \
	register long ld = (_d); \
	register long b1 = (_b) & RADIXROOTM; \
	register long d1 = (_d) & RADIXROOTM; \
	register long aa = *(_a) + b1 * d1; \
 \
	b1 = b1 * (ld >>= NBITSH) + d1 * (lb >>= NBITSH) + (aa >> NBITSH); \
	aa = (aa & RADIXROOTM) + ((b1 & RADIXROOTM) << NBITSH) + *(_t); \
	*(_t) = ld * lb + (b1 >> NBITSH) + (aa >> NBITS); \
	*(_a) = (aa & RADIXM); \
}

#define zaddmulpsq(_a, _b, _t) \
{ \
	register long lb = (_b); \
	register long b1 = (_b) & RADIXROOTM; \
	register long aa = *(_a) + b1 * b1; \
 \
	b1 = (b1 * (lb >>= NBITSH) << 1) + (aa >> NBITSH); \
	aa = (aa & RADIXROOTM) + ((b1 & RADIXROOTM) << NBITSH); \
	*(_t) = lb * lb + (b1 >> NBITSH) + (aa >> NBITS); \
	*(_a) = (aa & RADIXM); \
}
#endif

#define zaddmulone(ama, amb) \
{ \
	register long lami; \
	register long lams = 0; \
	register verylong lama = (ama); \
	register verylong lamb = (amb); \
 \
	lams = 0; \
	for (lami = (*lamb++); lami > 0; lami--) \
	{ \
		lams += (*lama + *lamb++); \
		*lama++ = lams & RADIXM; \
		lams >>= NBITS; \
	} \
	*lama += lams; \
}

#define zaddmul(ams, ama, amb) \
{ \
	register long lami; \
	register long lams = (ams); \
	register verylong lama = (ama); \
	register verylong lamb = (amb); \
	long lamcarry = 0; \
 \
	for (lami = (*lamb++); lami > 0; lami--) \
	{ \
		zaddmulp(lama, *lamb, lams, &lamcarry); \
	/* Be careful, the last lama is unnormalized */ \
		lama++; \
		lamb++; \
	} \
        *lama += lamcarry; \
}

#define zaddmulsq(sql, sqa, sqb) \
{ \
	register long lsqi = (sql); \
	register long lsqs = *(sqb); \
	register verylong lsqa = (sqa); \
	register verylong lsqb = (sqb); \
	long lsqcarry = 0; \
 \
	lsqb++; \
	for (; lsqi > 0; lsqi--) \
	{ \
		zaddmulp(lsqa, *lsqb, lsqs, &lsqcarry); \
		lsqa++; \
		lsqb++; \
	} \
	*lsqa += lsqcarry; \
/* Be careful, the last lama is unnormalized */ \
}

#define zmmulp(mmpa, mmpb) \
{ \
	register verylong lmmpa = (mmpa); \
	register verylong lmmpb = (mmpb); \
	register long lmmi = (*lmmpa) >> NBITSH; \
	register long lmmd = (*lmmpa) & RADIXROOTM; \
	long zmmtemp = 0; \
 \
	lmmd = (zminv1 * lmmd + (((zminv1 * lmmi + zminv2 * lmmd) & RADIXROOTM) << NBITSH)) & RADIXM; \
	for (lmmi = *lmmpb++; lmmi > 0; lmmi--) \
	{ \
		zaddmulp(lmmpa, *lmmpb, lmmd, &zmmtemp); \
		lmmpa++; \
		*lmmpb++; \
	} \
	if (((*lmmpa += zmmtemp) & RADIX) > 0) \
	{ \
		(*lmmpa++) &= RADIXM; \
		(*lmmpa)++; \
	} \
}

#if 0
/*
	If you prefer you can replace the above macro's by C-routines ,
	(or better, replace them by assembly versions). Here are some
	examples, the rest follow easily.
*/

void
zaddmulp(
	verylong a,
	long b,
	long d,
	verylong t
	)
{
	register long b1 = b & RADIXROOTM;
	register long d1 = d & RADIXROOTM;
	register long aa = *a + b1 * d1;

	b1 = b1 * (d >>= NBITSH) + d1 * (b >>= NBITSH) + (aa >> NBITSH);
	aa = (aa & RADIXROOTM) + ((b1 & RADIXROOTM) << NBITSH) + *t;
	*t = d * b + (b1 >> NBITSH) + (aa >> NBITS);
	*a = (aa & RADIXM);
}

void
zaddmul(
	long d,
	verylong a,
	verylong b
	)
{
	register long i;
	long carry = 0;
	verylong carrya = &carry;

	for (i = (*b++); i > 0; i--)
		zaddmulp(a++, *b++, d, carrya);
	*a += carry;
}

void
zaddmulone(
	verylong a,
	verylong b
	)
{
	zaddmul(1, a, b);
}

void
zmmulp(
	verylong pa,
	verylong pb
	)
{
	register long i = (*pa) >> NBITSH;
	register long d = (*pa) & RADIXROOTM;
	long temp = 0;
	extern long zminv1;
	extern long zminv2;

	d = (zminv1 * d + (((zminv1 * i + zminv2 * d) & RADIXROOTM) << NBITSH)) & RADIXM;
	for (i = *pb++; i > 0; i--)
		zaddmulp(pa++, *pb++, d, &temp);
	if (((*pa += temp) & RADIX) > 0)
	{
		(*pa++) &= RADIXM;
		(*pa)++;
	}
}
#endif


/* a[s:] -= d*b, optimize this one, a and b not at overlapping addresses */

static void
zsubmul(
	long r,
	verylong a,
	verylong b
	)
{
	register long rd = RADIX - r;
	register long i;
	long carry = RADIX;

	for (i = (*b++); i > 0; i--)
	{
		zaddmulp(a, *b, rd, &carry);
		a++;
		carry += RADIXM - (*b++);
	}
	*a += carry - RADIX;	/* unnormalized */
}

/*
	zdiv21 returns quot, numhigh so

	quot = (numhigh*RADIX + numlow)/denom;
	numhigh  = (numhigh*RADIX + numlow)%denom;

Assumes 0 <= numhigh < denom < RADIX and 0 <= numlow < RADIX.
*/

#if 0

#define zdiv21(numhigh, numlow, denom, deninv, quot) \
{ \
	register long lr21; \
	register long lq21 = (long) (((fradix * (double) (numhigh)) \
			 + (double) (numlow)) * (deninv)); \

/* Following works in many two's complement architectures. */ \
	lr21 = (numhigh << NBITS) + numlow - lq21 * denom; \
	if (lr21 < 0) \
	{ \
		do \
		{ \
			lq21--; \
		} while ((lr21 += denom) < 0); \
	} \
	else \
	{ \
		while (lr21 >= denom) \
		{ \
			lr21 -= denom; \
			lq21++; \
		}; \
	} \
	quot = lq21; \
	numhigh = lr21; \
}
#else

#define zdiv21(numhigh, numlow, denom, deninv, quot) \
{ \
	register long lr21; \
	register long lq21 = (long) (((fradix * (double) (numhigh)) \
			 + (double) (numlow)) * (deninv)); \
	long lprodhigh = 0; \
	long lprodlow = 0; \
 \
	zaddmulp(&lprodlow, lq21, denom, &lprodhigh); \
	lr21 = RADIX * (numhigh - lprodhigh) + (numlow - lprodlow); \
	if (lr21 < 0) \
	{ \
		do \
		{ \
			lq21--; \
		} while ((lr21 += denom) < 0); \
	} \
	else \
	{ \
		while (lr21 >= denom) \
		{ \
			lr21 -= denom; \
			lq21++; \
		}; \
	} \
	quot = lq21; \
	numhigh = lr21; \
}
#endif

long
zmulmods(
	long a,
	long b,
	long n
	)
{
	register long lqmul = (long) (((double)a) * ((double)b) / ((double)n));
#if 0
	register long lr;
	long lprodhigh1 = 0;
	long lprodhigh2 = 0;
	long lprodlow1 = 0;
	long lprodlow2 = 0;

	zaddmulp(&lprodlow1, a, b, &lprodhigh1);
	zaddmulp(&lprodlow2, lqmul, n, &lprodhigh2);
	lr = RADIX * (lprodhigh1 - lprodhigh2) + (lprodlow1 - lprodlow2);
#else
/* Many machines compute the following modulo 2^32, which is OK */
	register long lr = a * b - lqmul * n;

#endif
	while (lr >= n)
		lr -= n;
	while (lr < 0)
		lr += n;
	return (lr);
}


#ifdef __hpux
extern void *calloc();
#else
extern void *calloc();
#endif

/* global variables */

/* for pollard time outs */
static int time_flag = 0;
/* for long division */
static double epsilon;
static double fradix = (double)RADIX, fudge = -1.0;
/* for random generator */
static verylong zseed = 0;
static verylong zranp = 0;
static verylong zprroot = 0;
/* for Montgomery multiplication */
static verylong zn = 0;
static verylong zoldzn = 0;
static verylong zr = 0;
static verylong zrr = 0;
static verylong zrrr = 0;
static verylong znm = 0;
static long znotinternal = 0;
static long zntop;
static long zminv1;
static long zminv2;
/* for small prime genaration */
static short *lowsieve = 0;
static short *movesieve = 0;
static long pindex;
static long pshift = -1;
static long lastp = 0;
/* for convenience */
static long oner[] = {1, 1, 1};
static long glosho[] = {1, 1, 0};
static verylong one = &oner[1];
/* for m_ary exponentiation */
static verylong **exp_odd_powers = 0;
/* for karatsuba */
static verylong kar_mem[5*MAXDEPTH];
static long kar_mem_initialized = 0;

double 
getutime()
{
	struct rusage used;

#ifndef __hpux
	getrusage(RUSAGE_SELF, &used);
#else	/* maybe delete in HPUX 9. */
#include <syscall.h>
	syscall(SYS_getrusage, RUSAGE_SELF, &used);
#endif
#if defined(sparc) && defined(__svr4__)
	return (used.ru_utime.tv_sec + (used.ru_utime.tv_nsec) / 1e9);
#else
	return (used.ru_utime.tv_sec + (used.ru_utime.tv_usec) / 1e6);
#endif /* SOLARIS */
}

double 
getstime()
{
	struct rusage used;

#ifndef __hpux
	getrusage(RUSAGE_SELF, &used);
#else	/* maybe delete in HPUX 9. */
#include <syscall.h>
	syscall(SYS_getrusage, RUSAGE_SELF, &used);
#endif
#if defined(sparc) && defined(__svr4__)
	return (used.ru_stime.tv_sec + (used.ru_stime.tv_nsec) / 1e9);
#else
	return (used.ru_stime.tv_sec + (used.ru_stime.tv_usec) / 1e6);
#endif /* SOLARIS */
}

double 
gettime()
{
	struct rusage used;

#ifndef __hpux
	getrusage(RUSAGE_SELF, &used);
#else	/* maybe delete in HPUX 9. */
#include <syscall.h>
	syscall(SYS_getrusage, RUSAGE_SELF, &used);
#endif
	return (used.ru_utime.tv_sec + used.ru_stime.tv_sec +
#if defined(sparc) && defined(__svr4__)
		(used.ru_utime.tv_nsec + used.ru_stime.tv_nsec) / 1e9);
#else
		(used.ru_utime.tv_usec + used.ru_stime.tv_usec) / 1e6);
#endif /* SOLARIS */
}

static void
hidetimer(
	FILE *f,
	long what
	)
{
	static double keep_time;
	if (what)
	{
		fprintf(f,"%8.5lf sec.\n",gettime()-keep_time);
		fflush(f);
	}
	else
		keep_time = gettime();
}

void
starttime()
{
	hidetimer(stderr,0);
}

void
printtime(
	FILE *f
	)
{
	hidetimer(f,1);
}

static void
zhalt(
        char *c
	)
{
#ifdef NOHALT
	fprintf(stderr,"error:\n   %s\ncontinue...\n",c);
	fflush(stderr);
#else
	fprintf(stderr,"fatal error:\n   %s\nexit...\n",c);
	fflush(stderr);
	(void)exit((int)0);
#endif
}

void
zsetlength(
	verylong *v,
	long len,
	char *str
	)
{
	verylong x = *v;

	if (x)
	{
		if (len <= x[-1])
			return;
		if (PRT_REALLOC)
		{
			fprintf(stderr,"%s reallocating to %ld\n", str, len);
			fflush(stderr);
		}
		x[-1] = len;
		if (!(x = (verylong)realloc((void*)(&(x[-1])), (size_t)(len + 2) * SIZEOFLONG)))
		{
			fprintf(stderr,"%d bytes realloc failed\n", ((int)len + 2) * SIZEOFLONG);
			zhalt("reallocation failed in zsetlength");
		}
	}
	else if (len >= 0)
	{
		if (len < SIZE)
			len = SIZE;
		if (PRT_REALLOC)
		{
			fprintf(stderr,"%s allocating to %ld\n", str, len);
			fflush(stderr);
		}
		if (!(x = (verylong)calloc((size_t)(len + 2), (size_t)SIZEOFLONG)))
		{
			fprintf(stderr,"%d bytes calloc failed\n", ((int)len + 2) * SIZEOFLONG);
			zhalt("allocation failed in zsetlength");
		}
		x[0] = len;
		x[1] = 1;
		x[2] = 0;
	}
	else
		zhalt("negative size allocation in zsetlength");
	x++;
	*v = x;
}

void
zfree(
	verylong *x
	)
{
	if (!(*x))
		return;
	{
		verylong y = (*x - 1);
		free((void*)y);
		*x = 0;
		return;
	}
}

double 
zdoub(
	verylong n
	)
{
	double res;
	register long i;

	if (ALLOCATE && !n)
		return ((double) 0);
	if ((i = n[0]) < 0)
		i = -i;
	res = (double) (n[i--]);
	for (; i; i--)
		res = res * fradix + (double) (n[i]);
	if (n[0] > 0)
		return (res);
	return (-res);
}

void
zstart()
{
	double local_one = (double) 1;
	double local_half = 1 / (double) 2;
	epsilon = local_one;
	fudge = local_one + epsilon;

 /* the following loop is sick, but we have to: */
 /* comparing one to one + epsilon does not have */
 /* the desired effect on some machines, because */
 /* one + epsilon is Not cast to double (as it should) */
	while (local_one != fudge)
	{
		epsilon = epsilon * local_half;
		fudge = local_one + epsilon;
	}
	epsilon += epsilon;
	if ((epsilon * RADIX) > local_one)
	{
		zhalt("recompile with smaller NBITS");
	}
	else
		epsilon *= 3;
	fudge = fradix + epsilon * fradix;
}

void
zzero(
	verylong *aa
	)
{
	if (!(*aa)) zsetlength(aa, (long)SIZE, "in zzero, first argument");
	(*aa)[0] = (long) 1;
	(*aa)[1] = (long) 0;
}

void
zone(
	verylong *aa
	)
{
	if (!(*aa)) zsetlength(aa, (long)SIZE, "in zone, first argument");
	(*aa)[0] = (long) 1;
	(*aa)[1] = (long) 1;
}

void
zcopy(
	verylong a,
	verylong *bb
	)
{
	register long i;
	verylong b = *bb;

	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	if (a != b)
	{
		if ((i = *a) < 0)
			i = (-i);
		zsetlength(&b, i, "in zcopy, second argument");
		*bb = b;
		for (; i >= 0; i--)
			*b++ = *a++;
	}
}

void
zintoz(
	long d,
	verylong *aa
	)
{
	register long i = 0;
	register long anegative = 0;
	verylong a = *aa;

	if (!(*aa)) zsetlength(&a, (long)SIZE, "in zintoz, second argument");
	*aa = a;
	if (d < 0)
	{
		anegative = 1;
		d = -d;
	}
	a[1] = 0;
	while (d > 0)
	{
		a[++i] = d & RADIXM;
		d >>= NBITS;
	}
	if (i > 0)
		a[0] = i;
	else
		a[0] = 1;
	if (anegative)
		a[0] = (-a[0]);
}

long 
ztoint(
	verylong a
	)
{
	register long d;
	register long sa;

	if (ALLOCATE && !a)
		return (0);
	if ((sa = *a) < 0)
		sa = -sa;
	d = *(a += sa);
	while (--sa)
	{
		d <<= NBITS;
		d += *(--a);
	}
	if ((*(--a)) < 0)
		return (-d);
	return (d);
}

void
zsbastoz(
	long base,
	long row[],
	long len,
	verylong *n
	)
{
	register long i;

	zintoz(row[len - 1], n);
	for (i = len - 1; i--;)
	{
		zsmul(*n, base, n);
		zsadd(*n, row[i], n);
	}
}

void
zbastoz(
	verylong base,
	verylong row[],
	long len,
	verylong *n
	)
{
	register long i;

	zcopy(row[len - 1], n);
	for (i = len - 1; i--;)
	{
		zmulin(base, n);
		zadd(*n, row[i], n);
	}
}

long 
zstobas(
	verylong n,
	long base,
	long row[],
	long *len
	)
{
 /* return 1 if it fits, 0 otherwise */
 /* n >= 0, base > 1 */
	register long i = 0;
	register long max = *len;
	static verylong q = 0;
	static verylong nn = 0;

	if (max < 1)
		return (0);
	if (base < 0)
		base = -base;
	if (base <= 1)
		return (0);
	*len = 0;
	row[0] = 0;
	zcopy(n, &nn);
	zabs(&nn);
	for (;;)
	{
		row[i] = zsdiv(nn, base, &q);
		i++;
		if ((q[1] == 0) && (q[0] == 1))
		{
			*len = i;
			return (1);
		}
		if (i == max)
			return (0);
		zswap(&q, &nn);
	}
}

long 
zstosymbas(
	verylong n,
	long base,
	long row[],
	long *len
	)
{
 /* return 1 if it fits, 0 otherwise */
 /* n >= 0, base > 1 */
	register long i = 0;
	register long max = *len;
	register long hbase;
	static verylong q = 0;
	static verylong nn = 0;

	if (max < 1)
		return (0);
	if (base < 0)
		base = -base;
	if (base <= 1)
		return (0);
	hbase = (base >> 1);
	*len = 0;
	row[0] = 0;
	zcopy(n, &nn);
	zabs(&nn);
	for (;;)
	{
		if ((row[i] = zsdiv(nn, base, &q)) > hbase)
		{
			row[i] -= base;
			zsadd(q, 1, &q);
		}
		i++;
		if ((q[1] == 0) && (q[0] == 1))
		{
			*len = i;
			return (1);
		}
		if (i == max)
			return (0);
		zswap(&q, &nn);
	}
}

long 
ztobas(
	verylong n,
	verylong base,
	verylong row[],
	long *len
	)
{
 /* return 1 if it fits, 0 otherwise */
 /* n >= 0, base > 1 */
	register long i = 0;
	register long max = *len;
	static verylong q = 0;
	static verylong nn = 0;
	static verylong lbase = 0;

	if (max < 1)
		return (0);
	zcopy(base, &lbase);
	zabs(&lbase);
	if (zscompare(base, 1) <= 0)
		return (0);
	*len = 0;
	zintoz(0, &(row[0]));
	zcopy(n, &nn);
	zabs(&nn);
	for (;;)
	{
		zdiv(nn, lbase, &q, &(row[i]));
		i++;
		if ((q[1] == 0) && (q[0] == 1))
		{
			*len = i;
			return (1);
		}
		if (i == max)
			return (0);
		zswap(&q, &nn);
	}
}

long 
ztosymbas(
	verylong n,
	verylong base,
	verylong row[],
	long *len
	)
{
 /* return 1 if it fits, 0 otherwise */
 /* n >= 0, base > 1 */
	register long i = 0;
	register long max = *len;
	static verylong q = 0;
	static verylong nn = 0;
	static verylong lbase = 0;
	static verylong hbase = 0;

	if (max < 1)
		return (0);
	zcopy(base, &lbase);
	zabs(&lbase);
	if (zscompare(base, 1) <= 0)
		return (0);
	z2div(lbase, &hbase);
	*len = 0;
	zintoz(0, &(row[0]));
	zcopy(n, &nn);
	zabs(&nn);
	for (;;)
	{
		zdiv(nn, lbase, &q, &(row[i]));
		if (zcompare(row[i], hbase) > 0)
		{
			zsub(row[i], lbase, &(row[i]));
			zsadd(q, 1, &q);
		}
		i++;
		if ((q[1] == 0) && (q[0] == 1))
		{
			*len = i;
			return (1);
		}
		if (i == max)
			return (0);
		zswap(&q, &nn);
	}
}

long 
zcompare(
	verylong a,
	verylong b
	)
{
	register long sa;
	register long sb;

	if (ALLOCATE && !a)
	{
		if (!b)
			return (0);
		if (b[0] < 0)
			return (1);
		if (b[0] > 1)
			return (-1);
		if (b[1])
			return (-1);
		return (0);
	}
	if (ALLOCATE && !b)
	{
		if (a[0] < 0)
			return (-1);
		if (a[0] > 1)
			return (1);
		if (a[1])
			return (1);
		return (0);
	}
 /* if (a==b) return (0); but who is comparing identical addresses? */
	if ((sa = *a) > (sb = *b))
		return (1);
	if (sa < sb)
		return (-1);
	if (sa < 0)
		sa = (-sa);
	a += sa;
	b += sa;
	for (; sa; sa--)
	{
		if (*a > *b)
		{
			if (sb < 0)
				return (-1);
			return (1);
		}
		if (*a-- < *b--)
		{
			if (sb < 0)
				return (1);
			return (-1);
		}
	/*
	 * depending on the relative speed of various operations it might be
	 * better to do the following: if ((aux=(*a--)-(*b--))>0) { if (sb<0)
	 * return (-1); return (1); } else if (aux<0) { if (sb<0) return (1);
	 * return (-1); } where aux is a register long
	 */
	}
	return (0);
}

void
znegate(
	verylong *aa
	)
{
	verylong a = *aa;

	if (!a)
		return;
	if (a[1] || a[0] != 1)
		a[0] = (-a[0]);
}

void
zsadd(
	verylong a,
	long d,
	verylong *b
	)
{
	static verylong x = 0;

	zintoz(d, &x);
	zadd(a, x, b);
}

static void
zaddls(
	verylong a,
	verylong b,
	verylong c
	)
{
/* low level c = a+b, b shorter than a, output can be input */
	register long sa = *a;
	register long sb = *b;
	register long carry = 0;
	register long i;
	register verylong pc;

 /* we know that sa and sb are both >0 or both <0 */
	pc = &c[0];
	if (sa < 0)
	{
		sa = -sa;
		sb = -sb;
	}
	for (i = 1; i <= sb; i++)
	{
		if ((*(++pc) = (*(++a)) + (*(++b)) + carry) < RADIX)
			carry = 0;
		else
		{
			*pc -= RADIX;
			carry = 1;
		}
	}
	for (; i <= sa; i++)
	{
		if ((*(++pc) = (*(++a)) + carry) < RADIX)
			carry = 0;
		else
		{
			*pc -= RADIX;
			carry = 1;
		}
	}
	if (carry)
	{
		c[0] = sa + 1;
		*(++pc) = 1;
	}
	else
		c[0] = sa;
}

void
zsubpos(
	verylong a,
	verylong b,
	verylong *cc
	)
{
	register long sa = a[0];
	register long sb = b[0];
	register long carry = 0;
	register long i;
	register verylong pc;
	verylong c = *cc;

	if (ALLOCATE && !b)
	{
		if (a)
			zcopy(a, cc);
		else
			zzero(cc);
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(cc);
		return;
	}
	zsetlength(&c, sa, "in zsubpos, third argument");
	/* if *cc == a, then nothing will happen */
	/* if *cc == b, then b might point to freed space, so */
	if (b == *cc) b = c;
	*cc = c;
	pc = &c[0];
	for (i = 1; i <= sb; i++)
	{
		if ((*(++pc) = (*(++a)) - (*(++b)) - carry) >= 0)
			carry = 0;
		else
		{
			*pc += RADIX;
			carry = 1;
		};
	}
	for (; i <= sa; i++)
	{
		if ((*(++pc) = (*(++a)) - carry) >= 0)
			carry = 0;
		else
		{
			*pc += RADIX;
			carry = 1;
		};
	}
	i = sa;
	while ((i > 1) && (!(*pc)))
	{
		i--;
		pc--;
	}
	c[0] = i;
}

void
zadd(
	verylong a,
	verylong b,
	verylong *cc
	)
{
	register long sa;
	register long sb;
	register long anegative;
	verylong c = *cc;

	if (ALLOCATE && !a)
	{
		if (b)
			zcopy(b, cc);
		else
			zzero(cc);
		return;
	}
	if (ALLOCATE && !b)
	{
		zcopy(a, cc);
		return;
	}
	if ((anegative = ((sa = a[0]) < 0)) == ((sb = b[0]) < 0))
	{
	/* signs a and b are the same */
		register long i;
		if (anegative)
		{
			sa = -sa;
			sb = -sb;
		}
		zsetlength(&c, (sa > sb ? sa : sb) + 1, "in zadd, third argument");
		if (a == *cc) a = c;
		if (b == *cc) b = c;
		*cc = c;
		if (sa == sb)
		{
			register verylong pc;
			pc = &c[0];
			i = 0;
			for (; sa; sa--)
			{
				if ((*(++pc) = (*(++a)) + (*(++b)) + i) < RADIX)
					i = 0;
				else
				{
					*pc -= RADIX;
					i = 1;
				}
			}
			if (i)
			{
				c[0] = sb + 1;
				*(++pc) = 1;
			}
			else
				c[0] = sb;
		}
		else if (sa > sb)
			zaddls(a, b, c);
		else
			zaddls(b, a, c);
		if (anegative)
			c[0] = -c[0];
	/* if anegative, then c cannot be zero */
	}
	else
	{
	/* signs a and b are different */
		verylong old;
		verylong oldc;

		oldc = c;
		if (anegative)
		{
			a[0] = -a[0];
			old = a;
		}
		else
		{
			b[0] = -b[0];
			old = b;
		}
	/* the one that's negative cannot be zero */
		if (!(sa = zcompare(a, b)))
		{
			zzero(&c);
			*cc = c;
			if (anegative)
			{
				if (old != oldc)
					a[0] = -a[0];
			}
			else if (old != oldc)
				b[0] = -b[0];
		}
		else if (sa > 0)
		{
			zsubpos(a, b, &c);
			*cc = c;
			if (anegative)
			{
				c[0] = -c[0];
				if (old != oldc)
					a[0] = -a[0];
			}
			else if (old != oldc)
				b[0] = -b[0];
		}
		else
		{
			zsubpos(b, a, &c);
			*cc = c;
			if (!anegative)
			{
				c[0] = -c[0];
				if (old != oldc)
					b[0] = -b[0];
			}
			else if (old != oldc)
				a[0] = -a[0];
		}
	}
}

void
zsub(
	verylong a,
	verylong b,
	verylong *cc
	)
{
	register long sa;
	register long sb;
	register long anegative;
	verylong c = *cc;

	if (ALLOCATE && !b)
	{
		if (a)
			zcopy(a, cc);
		else
			zzero(cc);
		return;
	}
	if (ALLOCATE && !a)
	{
		zcopy(b, cc);
		znegate(cc);
		return;
	}
	if ((anegative = ((sa = a[0]) < 0)) == ((sb = b[0]) < 0))
	{
	/* signs a and b are the same */
		register long carry = 0;
		register long i;
		register verylong pc;
		register long agrb;

		if (!(agrb = zcompare(a, b)))
		{
			zzero(cc);
			return;
		}
		if ((agrb > 0 && anegative) || (agrb < 0 && !anegative))
		{
			pc = a;
			a = b;
			b = pc;
			sa = *a;
			sb = *b;
		}
		if (anegative)
		{
			sa = -sa;
			sb = -sb;
		}
		zsetlength(&c, sa, "in zsub, third argument");
                if (b == *cc) b = c;
		*cc = c;
		pc = &c[0];
		for (i = 1; i <= sb; i++)
		{
			if ((*(++pc) = (*(++a)) - (*(++b)) - carry) >= 0)
				carry = 0;
			else
			{
				*pc += RADIX;
				carry = 1;
			};
		}
		for (; i <= sa; i++)
		{
			if ((*(++pc) = (*(++a)) - carry) >= 0)
				carry = 0;
			else
			{
				*pc += RADIX;
				carry = 1;
			};
		}
		i = sa;
		while ((i > 1) && (!(*pc)))
		{
			i--;
			pc--;
		}
		if (agrb > 0)
			c[0] = i;
		else
			c[0] = -i;
	}
	else
	{
	/* signs a and b are different */
		verylong old;
		verylong oldc;

		oldc = c;
		if (anegative)
		{
			a[0] = -a[0];
			old = a;
		}
		else
		{
			b[0] = -b[0];
			old = b;
		}
	/* the one that's negative cannot be zero */
		zadd(a, b, &c);
		*cc = c;
		if (anegative)
		{
			c[0] = -c[0];
			if (old != oldc)
				a[0] = -a[0];
		}
		else if (old != oldc)
			b[0] = -b[0];
	}
}

void
zsmul(
	verylong a,
	long d,
	verylong *bb
	)
{
	register long sa;
	register long i;
	register verylong pb;
	register long bnegative = 0;
	verylong b = *bb;
	static verylong x = 0;

	if (( d >= RADIX) || (-d >= RADIX)) {
		zintoz(d,&x);
		zmulin(a,&x);
		zcopy(x,bb);
		return;
	}

	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	if (ALLOCATE && !d)
	{
		zzero(bb);
		return;
	}
	if ((sa = a[0]) < 0)
	{
		sa = (-sa);
		if (d < 0)
			d = (-d);
		else
			bnegative = 1;
	}
	else if (bnegative = (d < 0))
		d = (-d);
	zsetlength(&b, sa + 1, "in zsmul, third argument");
        if (a == *bb) a = b;
	*bb = b;
	pb = &b[0];
	for (i = sa; i >= 0; i--)
		*pb++ = *a++;
	b[0] = sa;
	sa++;
	*pb = 0;
	zaddmul(d - 1, &b[1], &b[0]);
	while ((sa > 1) && (!(b[sa])))
		sa--;
	b[0] = sa;
	if (bnegative && (b[1] || b[0] != 1))
		b[0] = (-b[0]);
}

static void
kar_mul(
	verylong a,
	verylong b,
	verylong *c,
	long shi
	)
{
	register long al;
	register long hal;
	register long i;
	register long restoreb0 = b[0];
	register verylong pc;
	register long bbig = 1;
	verylong *a0;
	verylong *a1;
	verylong *a2;
	verylong *a3;
	verylong *a4;

	zsetlength(c, (hal = (al = a[0]) + (i = b[0])), "in kar_mul, third argument");
	if ((shi >= (5 * MAXDEPTH)) || (al < MULT_CROSS) || (i < MULT_CROSS))
	{
		pc = &(*c)[1];
		for (i = hal; i > 0; i--)
			*pc++ = 0;
		pc = &(*c)[1];
		if (al <= *b)
			for (i = al; i; i--)
			{
				zaddmul(*(++a), pc++, b);
			}
		else
			for (i = *b; i; i--)
			{
				zaddmul(*(++b), pc++, a);
			}
		while ((hal > 1) && (!((*c)[hal])))
			hal--;
		(*c)[0] = hal;
		return;
	}
	a0 = &(kar_mem[shi]);
	a1 = &(kar_mem[shi + 1]);
	a2 = &(kar_mem[shi + 2]);
	a3 = &(kar_mem[shi + 3]);
	a4 = &(kar_mem[shi + 4]);
	hal = ((al + 1) >> 1);
	zsetlength(a0, al, "in kar_mul, locals\n");
	zsetlength(a1, al, "");
	zsetlength(a2, al, "");
	zsetlength(a3, al + hal, "");
	zsetlength(a4, al + 2, "");
	i = hal;
	while ((i > 1) && (!(a[i])))
		i--;
	a[0] = i;
	if (hal >= b[0])
		bbig = 0;
	else
	{
		i = hal;
		while ((i > 1) && (!(b[i])))
			i--;
		b[0] = i;
	}
	for (i = hal + 1; i <= al; i++)
		(*a1)[i - hal] = a[i];
	(*a1)[0] = al - hal;
	if (bbig)
	{
		for (i = hal + 1; i <= restoreb0; i++)
			(*a3)[i - hal] = b[i];
		(*a3)[0] = restoreb0 - hal;
	}
	kar_mul(a, b, a4, shi + 5);
	zadd(a, (*a1), a0);
	a[0] = al;
	if (bbig)
	{
		kar_mul((*a1), (*a3), c, shi + 5);
		zadd(b, (*a3), a2);
		b[0] = restoreb0;
		kar_mul((*a0), (*a2), a3, shi + 5);
	}
	else
		kar_mul((*a0), b, a3, shi + 5);
	zsubpos((*a3), (*a4), a3);
	if (bbig)
		zsubpos((*a3), *c, a3);
	zlshift((*a3), hal * NBITS, a3);
	hal <<= 1;
	if (bbig)
	{
		for (i = (*c)[0]; i; i--)
			(*c)[i + hal] = (*c)[i];
		for (i = hal; i > (*a4)[0]; i--)
			(*c)[i] = 0;
		for (; i; i--)
			(*c)[i] = (*a4)[i];
		(*c)[0] += hal;
	}
	else
	{
		for (i = (*a4)[0]; i >= 0; i--)
			(*c)[i] = (*a4)[i];
	}
	zadd(*c, (*a3), c);
}

void
zmul(
	verylong a,
	verylong b,
	verylong *c
	)
{	/* output not input */
	register long aneg;
	register long bneg;
	verylong olda;
	verylong oldb;

	if (ALLOCATE && (!a || !b))
	{
		zzero(c);
		return;
	}
	if (a == b)
	{
		zsq(a, c);
		return;
	}
	if (!kar_mem_initialized)
	{
		kar_mem_initialized = 1;
		for (aneg = (5 * MAXDEPTH) - 1; aneg >= 0; aneg--)
			kar_mem[aneg] = (verylong) 0;
	}
	olda = a;
	oldb = b;
	if (aneg = (*a < 0))
		a[0] = -a[0];
	if (bneg = (*b < 0))
		b[0] = -b[0];
	if (*a > *b)
		kar_mul(a, b, c, (long) 0);
	else
		kar_mul(b, a, c, (long) 0);
	if (aneg != bneg && ((*c)[1] || (*c)[0] != 1))
		(*c)[0] = -(*c)[0];
	if (aneg)
		olda[0] = -olda[0];
	if (bneg)
		oldb[0] = -oldb[0];
}

static void
kar_sq(
	verylong a,
	verylong *c,
	long shi
	)
{
	register long al;
	register long hal;
	register long i;
	register verylong pc;
	verylong *a0;
	verylong *a1;
	verylong *a2;

	zsetlength(c, (i = ((al = a[0]) << 1)), "in kar_sq, second argument");
	if ((shi >= (3 * MAXDEPTH)) || (al < SQUARE_CROSS))
	{
		register unsigned long uncar;
		long carry = 0;
		pc = &(*c)[1];
		for (; i > 0; i--)
			*pc++ = 0;
		for (hal = 1; hal <= al; hal++)
		{
			i += 2;
			{
				zaddmulsq(al - hal, &((*c)[i]), &(a[hal]));
			}
			uncar = ((*c)[i - 1] << 1) + carry;
			(*c)[i - 1] = uncar & RADIXM;
			uncar = ((*c)[i] << 1) + (uncar >> NBITS);
			{
				zaddmulpsq(&(*c)[i - 1], a[hal], &carry);
			}
			uncar += carry;
			carry = uncar >> NBITS;
			(*c)[i] = uncar & RADIXM;
		}
		while ((i > 1) && (!((*c)[i])))
			i--;
		(*c)[0] = i;
		return;
	}
	a0 = &(kar_mem[shi]);
	a1 = &(kar_mem[shi + 1]);
	a2 = &(kar_mem[shi + 2]);
	hal = ((al + 1) >> 1);
	zsetlength(a0, al + hal + 2, "in kar_sq, locals\n");
	zsetlength(a1, al + 2, "");
	zsetlength(a2, al, "");
	i = hal;
	while ((i > 1) && (!(a[i])))
		i--;
	a[0] = i;
	for (i = hal + 1; i <= al; i++)
		(*a0)[i - hal] = a[i];
	(*a0)[0] = al - hal;
	kar_sq(a, a1, shi + 3);
	zadd(a, (*a0), a2);
	kar_sq((*a0), c, shi + 3);
	a[0] = al;
	kar_sq((*a2), a0, shi + 3);
	zsubpos((*a0), (*a1), a0);
	zsubpos((*a0), *c, a0);
	zlshift((*a0), hal * NBITS, a0);
	hal <<= 1;
	for (i = (*c)[0]; i; i--)
		(*c)[i + hal] = (*c)[i];
	for (i = hal; i > (*a1)[0]; i--)
		(*c)[i] = 0;
	for (; i; i--)
		(*c)[i] = (*a1)[i];
	(*c)[0] += hal;
	zadd(*c, (*a0), c);
}

void
zsq(
	verylong a,
	verylong *c
	)
{	/* output is not input */
	register long aneg;

	if (ALLOCATE && !a)
	{
		zzero(c);
		return;
	}
        if (!kar_mem_initialized)
        {
                kar_mem_initialized = 1;
                for (aneg = (5 * MAXDEPTH) - 1; aneg >= 0; aneg--)
                        kar_mem[aneg] = (verylong) 0;
        }
	if (aneg = (*a < 0))
		a[0] = -a[0];
	kar_sq(a, c, (long) 0);
	if (aneg)
		a[0] = -a[0];
}

void
zmul_plain(
	verylong a,
	verylong b,
	verylong *cc
	)
{
	register long i;
	register verylong pc;
	register long sc;
	verylong c = *cc;
	long anegative;
	long bnegative;
	verylong olda;
	verylong oldb;

	if (ALLOCATE && (!a || !b))
	{
		zzero(cc);
		return;
	}
	olda = a;
	oldb = b;
	if (anegative = (*a < 0))
		a[0] = -a[0];
	if (olda == oldb)
		bnegative = anegative;
	else if (bnegative = (*b < 0))
		b[0] = -b[0];
	zsetlength(&c, (sc = *a + *b), "in zmul_plain, third argument");
	*cc = c;
	pc = &c[1];
	for (i = sc; i > 0; i--)
		*pc++ = 0;
	pc = &c[1];
	if (*a <= *b)
		for (i = *a; i; i--)
		{
			zaddmul(*(++a), pc++, b);
		}
	else
		for (i = *b; i; i--)
		{
			zaddmul(*(++b), pc++, a);
		}
	while ((sc > 1) && (!(c[sc])))
		sc--;
	c[0] = sc;
	if (anegative != bnegative && (c[1] || c[0] != 1))
		c[0] = -c[0];
	if (anegative)
		olda[0] = -olda[0];
	if (bnegative && oldb != olda)
		oldb[0] = -oldb[0];
}

void
zmulin(
	verylong b,
	verylong *a
	)
{
	static verylong mem = 0;

	if (ALLOCATE && (!*a || !b))
	{
		zzero(a);
		return;
	}
	zcopy(*a, &mem);
	zmul(mem, b, a);
}

void
zsq_plain(
	verylong a,
	verylong *cc
	)
{
	register long i;
	register long sc;
	register verylong pc;
	register unsigned long uncar;
	long carry = 0;
	verylong c = *cc;
	long anegative;

	if (ALLOCATE && !a)
	{
		zzero(cc);
		return;
	}
	if ((anegative = (*a)) < 0)
		anegative = -anegative;
	zsetlength(&c, (sc = 2 * anegative), "in zsq_plain, second argument");
	*cc = c;
	pc = &c[1];
	for (i = sc; i > 0; i--)
		*pc++ = 0;
	for (sc = 1; sc <= anegative; sc++)
	{
		i += 2;
		{
			zaddmulsq(anegative - sc, &(c[i]), &(a[sc]));
		}
		uncar = (c[i - 1] << 1) + carry;
		c[i - 1] = uncar & RADIXM;
		uncar = (c[i] << 1) + (uncar >> NBITS);
		{
			zaddmulpsq(&c[i - 1], a[sc], &carry);
		}
		uncar += carry;
		carry = uncar >> NBITS;
		c[i] = uncar & RADIXM;
	}
	while ((i > 1) && (!(c[i])))
		i--;
	c[0] = i;
}

void
zsqin(
	verylong *a
	)
{
	static verylong mem = 0;

	if (ALLOCATE && !*a)
	{
		zzero(a);
		return;
	}
	zcopy(*a, &mem);
	zsq(mem, a);
}

long 
zsdiv(
	verylong a,
	long d,
	verylong *bb
	)
{
	register long sa;
	verylong b = *bb;

#ifndef START
	if (fudge < 0)
		zstart();
#endif
	if (ALLOCATE && !a)
	{
		zzero(bb);
		return (0);
	}
	if (!d)
        {
		zhalt("division by zero in zsdiv");
		return (0);
	}
	if ((sa = a[0]) < 0)
		sa = (-sa);
	zsetlength(&b, sa, "in zsdiv, third argument");
	if (a == *bb) a = b;
	*bb = b;
	if ((d >= RADIX) || (d <= -RADIX))
	{
		static verylong zd = 0;
		static verylong zb = 0;

		zintoz(d, &zb);
		zdiv(a, zb, &b, &zd);
		*bb = b;
		return (ztoint(zd));
	}
	else
	{
		register long den = d;
		register double deninv;
		register long carry = 0;
		register long i;
		long flag = (*a < 0 ? 2 : 0) | (den < 0 ? 1 : 0);

		if (den < 0)
			den = -den;
		deninv = (double)1/den;
		if (a[sa] < den && sa > 1)
			carry = a[sa--];
		for (i = sa; i; i--)
		{
			zdiv21(carry, a[i], den, deninv, b[i]);
		}
		while ((sa > 1) && (!(b[sa])))
			sa--;
		b[0] = sa;
		if (flag)
		{
			if (flag <= 2)
			{
				if (!carry)
					znegate(&b);
				else
				{
					zadd(b, one, &b);
					b[0] = -b[0];
					if (flag == 1)
						carry = carry - den;
					else
						carry = den - carry;
					*bb = b;
				}
			}
			else
				carry = -carry;
		}
		return (carry);
	}
}

long 
zsmod(
	verylong a,
	long b
	)
{
	static verylong q = 0;

	return (zsdiv(a, b, &q));
}

void
zdiv(
	verylong in_a,
	verylong in_b,
	verylong *qqq,
	verylong *rrr
	)
{
	register long i;
	register long qq;
	long sa;
	long sb;
	long sq;
	verylong p;
	verylong pc;
	long sign;
	static verylong a = 0;
	static verylong b = 0;
	static verylong c = 0;
	static verylong d = 0;
	double btopinv;
	double aux;
	verylong q = *qqq;
	verylong r = *rrr;

#ifndef START
	if (fudge < 0)
		zstart();
#endif
	if (ALLOCATE && !in_a)
	{
		zzero(qqq);
		zzero(rrr);
		return;
	}
	if ((!in_b) || (((sb = in_b[0]) == 1) && (!in_b[1])))
	{
		zhalt("division by zero in zdiv");
		return;
	}
	zcopy(in_a,&a);
	zcopy(in_b,&b);
	sign = (*a < 0 ? 2 : 0) | (*b < 0 ? 1 : 0);
	if (*a < 0)
		(*a) = (-(*a));
	sa = (*a);
	if (*b < 0)
		(*b) = (-(*b));
	sb = (*b);
	zsetlength(&c, (i = (sa > sb ? sa : sb) + 1), "in zdiv, locals\n");
	zsetlength(&d, i, "");
	zsetlength(&q, i, "in zdiv, third argument");
	*qqq = q;
	zsetlength(&r, sb + 1, "in zdiv, fourth argument");
	*rrr = r;
	p = &b[sb];
	if ((sb == 1) && (*p < RADIX))
		zintoz(zsdiv(a, *p, &q), &r);
	else
	{
		sq = sa - sb;
		btopinv = (double) (*p) * fradix;
		if (sb > 1)
			btopinv += (*(--p));
		btopinv *= fradix;
		if (sb > 2)
			btopinv += (*(p - 1));
		btopinv = fudge / btopinv;
		p = &a[1];
		pc = &c[0];
		for (i = sa; i > 0; i--)
			*pc++ = *p++;
		p = pc;
		sa = 1 - sb;
		for (i = (-sq); i > 0; i--)
			*p++ = 0;
		*pc = 0;
		d[1] = 0;
		p = &d[sq + 1];
		for (i = sq; i >= 0; i--)
		{
			aux = fradix * (fradix * (*pc) + (*(pc - 1))) + 1.0;
			if (i > sa)
				aux += (*(pc - 2));
			qq = (long) (btopinv * aux + 0.5);
		/* dirty, but safe. On most machines (long)(btopinv*aux) */
		/* is correct, or one too big; on some however it becomes */
		/* too small. Could change zstart, but +0.5 and a while */
		/* instead of one single if is safer */
			if (qq > 0)
			{
				zsubmul(qq, &c[i], &b[0]);
				while (*pc < 0)
				{
					qq--;
					{
						zaddmulone(&c[i], &b[0]);
					}
				}
			}
			pc--;
			*p-- = qq;
		}
		sb--;
		while ((sb > 0) && (!(c[sb])))
			sb--;
		sb++;
		r[0] = sb;
		p = &r[1];
		pc = &c[0];
		for (i = sb; i > 0; i--)
			*p++ = *pc++;
		if (sq < 0)
		{
			q[0] = 1;
			q[1] = d[1];
		}
		else
		{
			sq++;
			while ((sq > 1) && (!(d[sq])))
				sq--;
			q[0] = sq;
			p = &q[1];
			pc = &d[1];
			for (i = sq; i > 0; i--)
				*p++ = *pc++;
		}
	}	/* non-trivial case */

	if (sign)
	{
		if (sign <= 2)
		{
			if (!(r[1]) && (r[0] == 1))
				q[0] = -(q[0]);
			else
			{
				zadd(q, one, &q);
				q[0] = -q[0];
				if (sign == 1)
					zsub(r, b, &r);
				else
					zsub(b, r, &r);
			}
		}
		else
			znegate(&r);
	}
	*qqq = q;
	*rrr = r;
}

void
zmod(
	verylong in_a,
	verylong in_b,
	verylong *rr
	)
{
	register long i;
	register long qq;
	verylong r = *rr;
	static verylong a = 0;
	static verylong b = 0;
	static verylong c = 0;
	long sa;
	long sb;
	long sq;
	verylong p;
	verylong pc;
	long sign;
	double btopinv;
	double aux;

#ifndef START
	if (fudge < 0)
		zstart();
#endif
	if (ALLOCATE && !in_a)
	{
		zzero(rr);
		return;
	}
	if ((!in_b) || (((sb = in_b[0]) == 1) && (!in_b[1])))
	{
		zhalt("division by zero in zmod");
		return;
	}
	zcopy(in_a,&a);
	zcopy(in_b,&b);
	sign = (*a < 0 ? 2 : 0) | (*b < 0 ? 1 : 0);
	if (*a < 0)
		(*a) = (-(*a));
	sa = (*a);
	if (*b < 0)
		(*b) = (-(*b));
	sb = (*b);
	zsetlength(&c, (sa > sb ? sa : sb) + 1, "in zmod, local");
	zsetlength(&r, sb + 1, "in zmod, third argument");
	*rr = r;
	p = &b[sb];
	if ((sb == 1) && (*p < RADIX))
		zintoz(zsdiv(a, *p, &c), &r);
	else
	{
		sq = sa - sb;
		btopinv = (double) (*p) * fradix;
		if (sb > 1)
			btopinv += (*(--p));
		btopinv *= fradix;
		if (sb > 2)
			btopinv += (*(p - 1));
		btopinv = fudge / btopinv;
		p = &a[1];
		pc = &c[0];
		for (i = sa; i > 0; i--)
			*pc++ = *p++;
		p = pc;
		sa = 1 - sb;
		for (i = (-sq); i > 0; i--)
			*p++ = 0;
		*pc = 0;
		for (i = sq; i >= 0; i--)
		{
			aux = fradix * (fradix * (*pc) + (*(pc - 1))) + 1.0;
			if (i > sa)
				aux += (*(pc - 2));
			qq = (long) (btopinv * aux + 0.5);
		/* see comment in zdiv */
			if (qq > 0)
			{
				zsubmul(qq, &c[i], &b[0]);
				while (*pc < 0)
				{
					{
						zaddmulone(&c[i], &b[0]);
					}
				}
			}
			pc--;
		}	/* loop on i */
		sb--;
		while ((sb > 0) && (!(c[sb])))
			sb--;
		sb++;
		r[0] = sb;
		p = &r[1];
		pc = &c[0];
		for (i = sb; i > 0; i--)
			*p++ = *pc++;
	}	/* non-trivial case */
	if (sign)
	{
		if ((sign <= 2) && (!((r[0] == 1) && !(r[1]))))
		{
			if (sign == 1)
				zsub(r, b, &r);
			else
				zsub(b, r, &r);
		}
		else
			znegate(&r);	/* negating r==0 doesn't hurt */
	}
	*rr = r;
}

void
zaddmod(
	verylong a,
	verylong b,
	verylong n,
	verylong *c
	)
{
	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in zaddmod");
		return;
	}
	if (ALLOCATE && !a)
	{
		if (b)
			zcopy(b, c);
		else
			zzero(c);
		return;
	}
	if (ALLOCATE && !b)
	{
		zcopy(a, c);
		return;
	}
	zadd(a, b, c);
	if (zcompare(*c, n) >= 0)
		zsubpos(*c, n, c);
}

void
zsubmod(
	verylong a,
	verylong b,
	verylong n,
	verylong *c
	)
{
	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in zsubmod");
		return;
	}
	if (ALLOCATE && !b)
	{
		if (a)
			zcopy(a, c);
		else
			zzero(c);
		return;
	}
	if (ALLOCATE && !a)
	{
		if (b[1] || (b[0] != 1))
			zsubpos(n, b, c);
		else
			zzero(c);
		return;
	}
	zsub(a, b, c);
	if ((*c)[0] < 0)
		zadd(*c, n, c);
}

void
zsmulmod(
	verylong a,
	long d,
	verylong n,
	verylong *c
	)
{
	static verylong mem = 0;

	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in zsmulmod");
		return;
	}
	if (ALLOCATE && (!a || !d))
	{
		zzero(c);
		return;
	}
	zsetlength(&mem, a[0] + 3, "in zsmulmod, local");
	zsmul(a, d, &mem);
	zmod(mem, n, c);
}

void
zmulmod(
	verylong a,
	verylong b,
	verylong n,
	verylong *c
	)
{
	static verylong mem = 0;

	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in zmulmod");
		return;
	}
	if (ALLOCATE && (!a || !b))
	{
		zzero(c);
		return;
	}
	zsetlength(&mem, a[0] + b[0] + 2, "in zmulmod, local");
	zmul(a, b, &mem);
	zmod(mem, n, c);
}

void
zsqmod(
	verylong a,
	verylong n,
	verylong *c
	)
{
	static verylong mem = 0;

	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in zsqmod");
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(c);
		return;
	}
	zsetlength(&mem, 2 * a[0] + 2, "in zsqmod, local");
	zsq(a, &mem);
	zmod(mem, n, c);
}

void
zdivmod(
	verylong a,
	verylong b,
	verylong n,
	verylong *c
	)
{
	static verylong g = 0;
	static verylong na = 0;
	static verylong nb = 0;
	static verylong r = 0;

	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in zdivmod");
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(c);
		return;
	}
	if (ALLOCATE && !b)
	{
		zhalt("divisor zero in zdivmod");
		return;
	}
	zgcd(a, b, &g);
	if ((g[1] == 1) && (g[0] == 1))
	{
		if (zinv(b, n, &g))
		{
			zhalt("undefined quotient in zdivmod");
			zcopy(g,c);
		}
		else
			zmulmod(a, g, n, c);
	}
	else
	{
		zdiv(a, g, &na, &r);
		zdiv(b, g, &nb, &r);
		if (zinv(nb, n, &g))
		{
			zhalt("undefined quotient in zdivmod");
                        zcopy(g,c);
                }
		else
			zmulmod(na, g, n, c);
	}
}

void
zinvmod(
	verylong a,
	verylong n,
	verylong *c
	)
{
	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in zinvmod");
		return;
	}
	if (ALLOCATE && !a)
	{
		zhalt("division by zero in zinvmod");
		return;
	}
	if (zinv(a, n, c))
		zhalt("undefined inverse in zinvmod");
}

static void
zmstartint(
	verylong n
	)
{
	static verylong x = 0;
	long i;

	if (!n || !(n[1] & 1) || ((zntop = n[0]) < 1)
			|| ((zntop == 1) && (n[1] <= 1)))
	{
		zhalt("zero, or even, or negative modulus in zmstart");
		return;
	}
	if (n[zntop] >= (RADIX >> 1))
		zntop++;
	zsetlength(&x, (long) (zntop + 1), "in zmstart, local");
	zsetlength(&zn, (long) (zntop + 1), "in zmstart, globals\n");
	zsetlength(&zr, (long) (zntop + 1), "");
	zsetlength(&zrr, (long) (zntop + 1), "");
	zsetlength(&zrrr, (long) (zntop + 1), "");
	zsetlength(&znm, (long) (zntop + 1), "");
	if (zcompare(n, zn))
	{
		zcopy(n, &zn);
		zadd(zn, one, &zrr);
		zrshift(zrr, (long) 1, &x);	/* x = (n+1)/2 */
		zrr[0] = 1;
		zrr[1] = NBITS;
		zexpmod(x, zrr, zn, &x);	/* x = 2^(-NBITS) %n    */
		for (i = x[0]; i; i--)
			x[i + 1] = x[i];
		x[1] = 0;
		x[0]++;
		zsubpos(x, one, &x);
		zdiv(x, zn, &x, &zrr);
		zminv1 = x[1];	/* zminv1 = n[1]^(-1) modulo 2^NBITS */
		zminv2 = zminv1 >> NBITSH;
		zminv1 &= RADIXROOTM;
		for (i = 1; i <= zntop; i++)
			x[i] = 0;
		x[zntop + 1] = 1;
		x[0] = zntop + 1;
		zdiv(x, zn, &zrr, &zr);
		zsqmod(zr, zn, &zrr);
		zmulmod(zr, zrr, zn, &zrrr);
		zsubmod(zn, zr, zn, &znm);
	}
}

static void
zmkeep(
	verylong n
	)
{
	if (znotinternal)
		zcopy(zn, &zoldzn);
	zmstartint(n);
}

static void
zmback(
	)
{
	if (znotinternal)
		zmstartint(zoldzn);
}

void
zmstart(
	verylong n
	)
{
	zmstartint(n);
	znotinternal = 1;
}
void
zmfree(
	)
{
	znotinternal = 0;
}

void
ztom(
	verylong a,
	verylong *bb
	)
{
	if (!zn)
	{
		zhalt("undefined Montgomery modulus in ztom");
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	if (zcompare(zn, a) < 0)
	{
		zmod(a, zn, bb);
		zmontmul(*bb, zrr, bb);
	}
	else
		zmontmul(a, zrr, bb);
}

void
zmtoz(
	verylong a,
	verylong *bb
	)
{
	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zmtoz");
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	zmontmul(a, one, bb);
}

void
zmontadd(
	verylong a,
	verylong b,
	verylong *c
	)
{
	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zmontadd");
		return;
	}
	zaddmod(a, b, zn, c);
}

void
zmontsub(
	verylong a,
	verylong b,
	verylong *c
	)
{
	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zmontsub");
		return;
	}
	zsubmod(a, b, zn, c);
}

void
zsmontmul(
	verylong a,
	long d,
	verylong *c
	)
{
	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zsmontmul");
		return;
	}
	zsmulmod(a, d, zn, c);
}

void
zmontmul(
	verylong a,
	verylong b,
	verylong *cc
	)
{
	register long i;
	register long j;
	verylong c = *cc;
	static verylong x = 0;
	verylong px;
	verylong pc;

	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zmontmul");
		return;
	}
	if (ALLOCATE && (!a || !b))
	{
		zzero(cc);
		return;
	}
	zsetlength(&x, (i = (zntop << 1) + 1), "in zmontmul, local");
	zsetlength(&c, zntop, "in zmontmul, third argument");
	if (a == *cc) a = c;
	if (b == *cc) b = c;
	*cc = c;
        if (!kar_mem_initialized)
        {
                kar_mem_initialized = 1;
                for (j = (5 * MAXDEPTH) - 1; j >= 0; j--)
                        kar_mem[j] = (verylong) 0;
        }
	if (*a <= *b)
		kar_mul(b, a, &x, (long) 0);
	else
		kar_mul(a, b, &x, (long) 0);
	for (; i > x[0]; i--)
		x[i] = (long) 0;
	px = &x[1];
	for (i = zntop; i > 0; i--)
		zmmulp(px++, zn);
	pc = &c[1];
	for (i = zntop; i > 0; i--)
		*pc++ = *px++;
	i = zntop;
	while ((i > 1) && (!(*--pc)))
		i--;
	c[0] = i;
	if (zcompare(c, zn) >= 0)
		zsubpos(c, zn, &c);
	*cc = c;
}

void
zmontsq(
	verylong a,
	verylong *cc
	)
{
	register long i;
	register long j;
	verylong c = *cc;
	static verylong x = 0;
	verylong px;
	verylong pc;

	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zmontsq");
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(cc);
		return;
	}
	zsetlength(&x, (i = (zntop << 1) + 1), "in zmontsq, local");
	zsetlength(&c, zntop, "in zmontsq, third argument");
	if (a == *cc) a = c;
	*cc = c;
        if (!kar_mem_initialized)
        {
                kar_mem_initialized = 1;
                for (j = (5 * MAXDEPTH) - 1; j >= 0; j--)
                        kar_mem[j] = (verylong) 0;
        }
	kar_sq(a, &x, (long) 0);
	for (; i > x[0]; i--)
		x[i] = (long) 0;
	px = &x[1];
	for (i = zntop; i > 0; i--)
		zmmulp(px++, zn);
	pc = &c[1];
	for (i = zntop; i > 0; i--)
		*pc++ = *px++;
	i = zntop;
	while ((i > 1) && (!(*--pc)))
		i--;
	c[0] = i;
	if (zcompare(c, zn) >= 0)
		zsubpos(c, zn, &c);
	*cc = c;
}

void
zmontdiv(
	verylong a,
	verylong b,
	verylong *c
	)
{
	static verylong g = 0;
	static verylong na = 0;
	static verylong nb = 0;
	static verylong r = 0;

	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zmontdiv");
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(c);
		return;
	}
	if (ALLOCATE && !b)
	{
		zhalt("division by zero in zmontdiv");
		return;
	}
	zgcd(a, b, &g);
	if ((g[1] == 1) && (g[0] == 1))
	{
		if (zinv(b, zn, &g))
		{
			zhalt("undefined quotient in zmontdiv");
			zcopy(g,c);
			return;
		}
		else
			zmontmul(g, a, &g);
	}
	else
	{
		zdiv(a, g, &na, &r);
		zdiv(b, g, &nb, &r);
		if (zinv(nb, zn, &g))
		{
			zhalt("undefined quotient in zmontdiv");
			zcopy(g,c);
                        return;
		}
		else
			zmontmul(g, na, &g);
	}
	zmontmul(g, zrrr, c);
}

void
zmontinv(
	verylong a,
	verylong *c
	)
{
	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zmontinv");
		return;
	}
        if (ALLOCATE && !a)
	{
                zhalt("division by zero in zmontinv");
		return;
	}
	if (zinv(a, zn, c))
	{
		zhalt("undefined inverse in zmontinv");
		return;
	}
	zmontmul(*c, zrrr, c);
}


void
zrstarts(
	long s
	)
{
	register long i;
	register long ones;
	register long res;
	static verylong three = 0;

	zintoz(s, &zseed);
	if (!three)
	{
		zintoz((long) 3, &three);
		ones = 107 / NBITS;
		res = 107 % NBITS;
		zsetlength(&zranp, ones + 1, "in zrstarts, zranp");
		for (i = ones; i; i--)
			zranp[i] = RADIXM;
		if (res)
			zranp[++ones] = (RADIXM >> (NBITS - res));
		zranp[0] = ones;
	}
	zsexpmod(three, (long) 121, zranp, &zprroot);
}

void
zrstart(
	verylong s
	)
{
	if (!zseed)
		zrstarts((long) 7157891);
	zmod(s, zranp, &zseed);
}

long 
zrandom(
	long b
	)
{
	static verylong zb = 0;
	static verylong bound = 0;

	if (b <= 0)
		return (0);
	if (!zseed)
		zrstarts((long) 7157891);	/* guess how random 7157891
						 * is */
	zintoz(b, &zb);
	zmod(zranp, zb, &bound);
	zsubpos(zranp, bound, &bound);
	do
	{
		zmulmod(zseed, zprroot, zranp, &zseed);
	} while (zcompare(zseed, bound) >= 0);	/* I agree, this is sick */
	zmod(zseed, zb, &zb);
	return (ztoint(zb));
}

long 
zinvodds(
	long n,
	long p
	)
{
	register long n1 = n;
	register long n2 = p;
	register long preg = p;
	register long m1 = 1;
	register long m2 = 0;

 /* m1*n == n1 mod p,  m2*n == n2 mod p */
 /* n2 is odd, n1 is odd after initialization */
	while (!(n1 & 1))
	{
		n1 >>= 1;
		if (m1 & 1)
			m1 += preg;
		m1 >>= 1;
	}
	if (n1 == 1)
		return (m1);
	while (n1 != n2)
	{
		if (n1 > n2)
		{
			n1 -= n2;
			m1 -= m2;
			if (m1 < 0)
				m1 += preg;
			do
			{
				n1 >>= 1;
				if (m1 & 1)
					m1 += preg;
				m1 >>= 1;
			} while (!(n1 & 1));
			if (n1 == 1)
				return (m1);
		}
		else
		{
			n2 -= n1;
			m2 -= m1;
			if (m2 < 0)
				m2 += preg;
			do
			{
				n2 >>= 1;
				if (m2 & 1)
					m2 += preg;
				m2 >>= 1;
			} while (!(n2 & 1));
			if (n2 == 1)
				return (m2);
		}
	}
	zhalt("arguments not coprime in zinvodds");
	return (0);
}

long 
zinvs(
	long n,
	long p
	)
{
	register long q = n;
	register long r;
	register long nq;
	register long nr;
	register long u = 1;
	register long w;
	register long nw;
	register long par = 0;

	if (p < 0)
		p = -p;
	if (n < 0)
		n = -n;
	if (n > p)
	{
		w = 0;
		r = p;
	}
	else
		r = p - (w = p / n) * n;
	nr = r;
	while (nr != 0)
	{
		if ((nr = q - r) < r)
			nq = 1;
		else if ((nr -= r) < r)
			nq = 2;
		else
			nr = q - (nq = q / r) * r;
		nw = nq * w + u;
		u = w;
		w = nw;
		q = r;
		r = nr;
		par = 1 - par;
	}
	if (par == 0)
		return (u);
	else
		return (p - u);
}

static long 
zxxeucl(
	verylong ain,
	verylong nin,
	verylong *invv,
	verylong *uu
	)
{
	static verylong a = 0;
	static verylong n = 0;
	static verylong q = 0;
	static verylong w = 0;
	static verylong x = 0;
	static verylong y = 0;
	static verylong z = 0;
	verylong inv = *invv;
	verylong u = *uu;
	long diff;
	long ilo;
	long sa;
	long sn;
	long temp;
	long e;
	long fast;
	long parity;
	long gotthem;
	verylong pin;
	verylong p;
	long i;
	long try11;
	long try12;
	long try21;
	long try22;
	long got11;
	long got12;
	long got21;
	long got22;
	double hi;
	double lo;
	double dt;
	double fhi;
	double flo;
	double num;
	double den;
	double dirt;

#ifndef START
	if (fudge < 0)
		zstart();
#endif
	zsetlength(&a, (e = (ain[-1] > nin[-1] ? ain[-1] : nin[-1])), 
	           "in zxxeucl, locals\n");
	zsetlength(&n, e, "");
	zsetlength(&q, e, "");
	zsetlength(&w, e, "");
	zsetlength(&x, e, "");
	zsetlength(&y, e, "");
	zsetlength(&z, e, "");
	zsetlength(&inv, e, "in zxxeucl, third argument");
	*invv = inv;
	zsetlength(&u, e, "in zxxeucl, fourth argument");
	*uu = u;
	fhi = 1.0 + epsilon;
	flo = 1.0 - epsilon;
	pin = &ain[0];
	p = &a[0];
	for (i = (*pin); i >= 0; i--)
		*p++ = *pin++;
	pin = &nin[0];
	p = &n[0];
	for (i = (*pin); i >= 0; i--)
		*p++ = *pin++;
	inv[0] = 1;
	inv[1] = 1;
	w[0] = 1;
	w[1] = 0;
	while (n[0] > 1 || n[1] > 0)
	{
		gotthem = 0;
		sa = a[0];
		sn = n[0];
		diff = sa - sn;
		if (!diff || diff == 1)
		{
			sa = a[0];
			p = &a[sa];
			num = (double) (*p) * fradix;
			if (sa > 1)
				num += (*(--p));
			num *= fradix;
			if (sa > 2)
				num += (*(p - 1));
			sn = n[0];
			p = &n[sn];
			den = (double) (*p) * fradix;
			if (sn > 1)
				den += (*(--p));
			den *= fradix;
			if (sn > 2)
				den += (*(p - 1));
			hi = fhi * (num + 1.0) / den;
			lo = flo * num / (den + 1.0);
			if (diff > 0)
			{
				hi *= fradix;
				lo *= fradix;
			}
			try11 = 1;
			try12 = 0;
			try21 = 0;
			try22 = 1;
			parity = 1;
			fast = 1;
			while (fast > 0)
			{
				parity = 1 - parity;
				if (hi >= fradix)
					fast = 0;
				else
				{
					ilo = (long)lo;
					dirt = hi - ilo;
					if (!dirt || !ilo || ilo < (long)hi)
						fast = 0;
					else
					{
						dt = lo;
						lo = flo / dirt;
						if (dt > ilo)
							hi = fhi / (dt - ilo);
						else
							hi = fradix;
						temp = try11;
						try11 = try21;
						if ((RADIX - temp) / ilo < try21)
							fast = 0;
						else
							try21 = temp + ilo * try21;
						temp = try12;
						try12 = try22;
						if ((RADIX - temp) / ilo < try22)
							fast = 0;
						else
							try22 = temp + ilo * try22;
						if ((fast > 0) && (parity > 0))
						{
							gotthem = 1;
							got11 = try11;
							got12 = try12;
							got21 = try21;
							got22 = try22;
						}
					}
				}
			}
		}
		if (gotthem)
		{
			zsmul(inv, got11, &x);
			zsmul(w, got12, &y);
			zsmul(inv, got21, &z);
			zsmul(w, got22, &w);
			zadd(x, y, &inv);
			zadd(z, w, &w);
			zsmul(a, got11, &x);
			zsmul(n, got12, &y);
			zsmul(a, got21, &z);
			zsmul(n, got22, &n);
			zsub(x, y, &a);
			zsub(n, z, &n);
		}
		else
		{
			zdiv(a, n, &q, &a);
			zmul(q, w, &x);
			zadd(inv, x, &inv);
			if (a[0] > 1 || a[1] > 0)
			{
				zdiv(n, a, &q, &n);
				zmul(q, inv, &x);
				zadd(w, x, &w);
			}
			else
			{
				p = &a[0];
				pin = &n[0];
				for (i = (*pin); i >= 0; i--)
					*p++ = *pin++;
				n[0] = 1;
				n[1] = 0;
				zsub(nin, w, &inv);
			}
		}
	}
	if ((a[0] == 1) && (a[1] == 1))
		e = 0;
	else
		e = 1;
	p = &u[0];
	pin = &a[0];
	for (i = (*pin); i >= 0; i--)
		*p++ = *pin++;
	*invv = inv;
	*uu = u;
	return (e);
}

long 
zinv(
	verylong ain,
	verylong nin,
	verylong *invv
	)
{
	static verylong u = 0;

	if ((!nin) || (!ain) || (ain[0] < 0) || (nin[0] < 0))
	{
		zhalt("zero or negative argument(s) in zinv");
		return (0);
	}
	if (!(zxxeucl(ain, nin, invv, &u)))
		return (0);
	zcopy(u, invv);
	return (1);
}

void
zexteucl(
	verylong a,
	verylong *xa,
	verylong b,
	verylong *xb,
	verylong *d
	)
{
	static verylong modcon = 0;
	register long agrb;
	register long anegative = 0;
	register long bnegative = 0;

	if ((!a) || (!b))
	{
		zhalt("zero argument(s) in zexteucl");
		return;
	}
	if (anegative = (a[0] < 0))
		a[0] = -a[0];
	if (bnegative = (b[0] < 0))
		b[0] = -b[0];
	if (!a[1] && (a[0] == 1))
	{
		zzero(xa);
		zone(xb);
		zcopy(b, d);
		goto done;
	}
	if ((!b[1] && (b[0] == 1)) || !(agrb = zcompare(a, b)))
	{
		zone(xa);
		zzero(xb);
		zcopy(a, d);
		goto done;
	}
	if (agrb > 0)
	{
		zxxeucl(a, b, xa, d);
		zmul(a, *xa, xb);
		zsub(*d, *xb, xb);
		zdiv(*xb, b, xb, &modcon);
	}
	else
	{
		zxxeucl(b, a, xb, d);
		zmul(b, *xb, xa);
		zsub(*d, *xa, xa);
		zdiv(*xa, a, xa, &modcon);
	}
	if ((modcon[1]) || (modcon[0] != 1))
	{
		zhalt("non-zero remainder in zexteucl   BUG");
		return;
	}
done:
	if (anegative)
	{
		a[0] = -a[0];
		znegate(xa);
	}
	if (bnegative)
	{
		b[0] = -b[0];
		znegate(xb);
	}
}

void
zchirem(
	verylong aa,
	verylong maa,
	verylong bb,
	verylong mbb,
	verylong *d
	)
{
	static verylong u = 0;
	static verylong v = 0;
	static verylong w = 0;
	verylong a;
	verylong b;
	verylong ma;
	verylong mb;
	register long agrb;

	if ((!aa) || (!maa) || (!bb) || (!mbb))
	{
		zhalt("zero argument(s) in zchirem");
		return;
	}
	if ((!aa[1] && aa[0] == 1) || (!bb[1] && bb[0] == 1) ||
			(aa[0] < 0) || (bb[0] < 0))
	{
		zhalt("zero or negative argument(s) in zchirem");
return;
        }
	if ((agrb = aa[0]) < bb[0])
		agrb = bb[0];
	zsetlength(&u, agrb, "in zchirem, locals");
	zsetlength(&v, agrb, "");
	zsetlength(&w, agrb, "");
	if (!(agrb = zcompare(aa, bb)))
	{
		zsub(maa, mbb, &u);
		zmod(u, aa, &w);
		if (w[1] || w[0] != 1)
		{
			zhalt("same moduli with different remainders in zchirem");
			return;
		}
		zcopy(maa, d);
		return;
	}
	else if (agrb > 0)
	{
		a = aa;
		b = bb;
		ma = maa;
		mb = mbb;
	}
	else
	{
		a = bb;
		b = aa;
		ma = mbb;
		mb = maa;
	}
	if (zxxeucl(a, b, &u, &w))
	{
		zhalt("moduli not coprime in zchirem");
		return;
	}
	zmod(ma, a, &v);
	zsub(mb, v, &w);
	zmulmod(w, u, b, &w);
	zmul(w, a, &u);
	zadd(v, u, d);
}

void
zmontexp(
	verylong a,
	verylong e,
	verylong *bb
	)
{
	register long i;
	register long j;
	register long k = 0;
	verylong b = *bb;
	static verylong loca = 0;

	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zmontexp");
		return;
	}
	if (ALLOCATE && !e)
	{
		zcopy(zr,bb);
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	if ((i = e[0]) < 0)
		i = (-i);
	if ((i == 1) && (!(e[1])))
		zcopy(zr,&b);
	else
	{
		zcopy(a, &loca);
		zcopy(loca, &b);
		for (; i; i--)
		{
			j = e[i];
			if (!k)
			{
				k = 1;
				while ((k << 1) <= j)
					k <<= 1;
			}
			while (k >>= 1)
			{
				zmontsq(b, &b);
				if (j & k)
					zmontmul(loca, b, &b);
			}
			k = RADIX;
		}
	}
	if (e[0] < 0)
	{
		if (zinv(b, zn, &b))
		{
			zhalt("undefined quotient in zmontexp");
			return;
		}
		zmontmul(b, zrrr, &b);
	}
	*bb = b;
}

void
zsexpmod(
	verylong a,
	long e,
	verylong n,
	verylong *bb
	)
{
	static verylong le = 0;

	zintoz(e, &le);
	zexpmod(a, le, n, bb);
}

void
zexpmod(
	verylong a,
	verylong e,
	verylong n,
	verylong *bb
	)
{
	register long i;
	register long j;
	register long k = 0;
	verylong b = *bb;
	static verylong loca = 0;

	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in zexpmod");
		return;
	}
	if (ALLOCATE && !e)
	{
		zone(bb);
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	if ((i = e[0]) < 0)
		i = (-i);
	if ((i == 1) && (!(e[1])))
		zone(&b);
	else
	{
		zmod(a, n, &loca);
		zcopy(loca, &b);
		for (; i; i--)
		{
			j = e[i];
			if (!k)
			{
				k = 1;
				while ((k << 1) <= j)
					k <<= 1;
			}
			while (k >>= 1)
			{
				zsqmod(b, n, &b);
				if (j & k)
					zmulmod(loca, b, n, &b);
			}
			k = RADIX;
		}
	}
	if (e[0] < 0 && zinv(b, n, &b))
		zhalt("undefined quotient in zexpmod");
	*bb = b;
}

void
z2expmod(
	verylong e,
	verylong n,
	verylong *bb
	)
{
	register long i;
	register long j;
	register long k = 0;
	verylong b = *bb;

	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in z2expmod");
		return;
	}
	if (ALLOCATE && !e)
	{
		zone(bb);
		return;
	}
	if ((i = e[0]) < 0)
		i = (-i);
	if ((i == 1) && (!(e[1])))
		zone(&b);
	else
	{
		if ((j = n[0]) < 0)
			j = -j;
		zsetlength(&b, j, "in z2expmod, third argument");
		*bb = b;
		zintoz((long) 2, &b);
		for (; i; i--)
		{
			j = e[i];
			if (!k)
			{
				k = 1;
				while ((k << 1) <= j)
					k <<= 1;
			}
			while (k >>= 1)
			{
				zsqmod(b, n, &b);
				if (j & k)
				{
					z2mul(b, &b);
					if (zcompare(b, n) >= 0)
						zsubpos(b, n, &b);
				}
			}
			k = RADIX;
		}
	}
	if (e[0] < 0 && zinv(b, n, &b))
		zhalt("undefined quotient in z2expmod");
	*bb = b;
}

long 
zdefault_m(
	long l
	)
{
	if (l < 2)
		return (2);
	if (l < 3)
		return (3);
	if (l < 7)
		return (4);
	if (l < 16)
		return (5);
	if (l < 35)
		return (6);
	if (l < 75)
		return (7);
	if (l < 160)
		return (8);
	if (l < 340)
		return (9);
	return (10);
}

static void 
mont_init_odd_power(
	verylong a,
	verylong *a_sq,
	long m,
	verylong **odd_a_power
	)
{
 /* odd power setter for montgomery m-ary exponentiation */
 /* initialize odd_a_power with a^i mod n for odd positive i < 2^m */
 /* a on input is montgomery number */
	register long i;
	register long length = (1 << (m - 1));

	if (!(*odd_a_power))
	{
		*odd_a_power = (verylong *)malloc((size_t)(length * sizeof(verylong)));
		for (i = 0; i < length; i++)
			(*odd_a_power)[i] = (verylong) 0;
	}
	zcopy(a, &((*odd_a_power)[0]));
	zmontsq((*odd_a_power)[0], a_sq);
}

void
zmontexp_m_ary(
	verylong a,
	verylong e,
	verylong *bb,
	long m
	)
{
	register long i;
	register long ei;
	static verylong a_sq = 0;

	if (!zn)
	{
		zhalt("undefined Montgomery modulus in zmontexp_m_ary");
		return;
	}
	if (ALLOCATE && !e)
	{
		zcopy(zr,bb);
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	if ((i = e[0]) < 0)
		i = (-i);
	if (i == 1)
	{
		zmontexp(a, e, bb);
		return;
	}
	if (!exp_odd_powers)
	{
		exp_odd_powers = (verylong **)malloc((size_t)(NBITS * sizeof(verylong *)));
		for (ei = 0; ei < NBITS; ei++)
			exp_odd_powers[ei] = (verylong *) 0;
	}
	if (m <= 1)
		m = zdefault_m(i);
	else if (m >= NBITS)
		m = (NBITS - 1);
	{
		register long k = 0;
		register long od = 0;
		register long odlen;
		register long left;
		register long zeros = 0;
		register long rem = 0;
		register long first = 1;
		register long reach = 1;
		verylong **loc_pow = &(exp_odd_powers[m]);

		mont_init_odd_power(a, &a_sq, m, loc_pow);
		for (; i > 0; i--)
		{
			ei = e[i];
			if (!k)
			{
				k = 1;
				while (k <= ei)
					k <<= 1;
			}
			while (k >>= 1)
			{
				if (zeros)
				{
					if (ei & k)
					{
						odlen = od = 1;
						rem = zeros;
						left = zeros = 0;
					}
					else
						zeros++;
				}
				else if (!od)
				{
					if (ei & k)
					{
						odlen = od = 1;
						left = zeros = 0;
					}
					else
						zeros = 1;
				}
				else
				{
					left++;
					if (odlen + left == m)
					{
						if (ei & k)
						{
							od <<= left;
							odlen += left;
							left = 0;
							od++;
							od >>= 1;
							for (; reach <= od; reach++)
								zmontmul((*loc_pow)[reach - 1], a_sq, &((*loc_pow)[reach]));
							if (first)
							{
								zcopy((*loc_pow)[od], bb);
								first = rem = odlen = od = 0;
							}
							else
							{
								for (rem = rem + odlen; rem; rem--)
									zmontsq(*bb, bb);
								zmontmul(*bb, (*loc_pow)[od], bb);
								odlen = od = 0;
							}
						}
						else
						{
							if (first)
							{
								if (od == 1)
								{
									zcopy(a_sq, bb);
									left--;
								}
								else
								{
									od >>= 1;
									for (; reach <= od; reach++)
										zmontmul((*loc_pow)[reach - 1], a_sq, &((*loc_pow)[reach]));
									zcopy((*loc_pow)[od], bb);
								}
								zeros = left;
								first = left = odlen = od = rem = 0;
							}
							else
							{
								for (rem = rem + odlen; rem; rem--)
									zmontsq(*bb, bb);
								od >>= 1;
								for (; reach <= od; reach++)
									zmontmul((*loc_pow)[reach - 1], a_sq, &((*loc_pow)[reach]));
								zmontmul(*bb, (*loc_pow)[od], bb);
								zeros = left;
								left = odlen = od = 0;
							}
						}
					}
					else if (ei & k)
					{
						od <<= left;
						odlen += left;
						left = 0;
						od++;
					}
				}
			}
			k = RADIX;
		}
		if (od)
		{
			if (first)
			{
				if (od == 1)
				{
					zcopy(a_sq, bb);
					left--;
				}
				else
				{
					od >>= 1;
					for (; reach < od; reach++)
						zmontmul((*loc_pow)[reach - 1], a_sq, &((*loc_pow)[reach]));
					zmontmul((*loc_pow)[od - 1], a_sq, bb);
				}
			}
			else
			{
				for (rem = rem + odlen; rem; rem--)
					zmontsq(*bb, bb);
				od >>= 1;
				for (; reach <= od; reach++)
					zmontmul((*loc_pow)[reach - 1], a_sq, &((*loc_pow)[reach]));
				zmontmul(*bb, (*loc_pow)[od], bb);
			}
		}
		if (left || zeros)
		{
			for (rem = zeros + left; rem; rem--)
				zmontsq(*bb, bb);
		}
	}

	if (e[0] < 0)
	{
		if (zinv(*bb, zn, bb))
		{
			zhalt("undefined quotient in zmontexp_m_ary");
			return;
		}
		zmontmul(*bb, zrrr, bb);
	}
}

static void 
init_odd_power(
	verylong a,
	verylong n,
	verylong *a_sq,
	long m,
	verylong **odd_a_power
	)
{
 /* odd power setter for m-ary exponentiation */
 /* initialize odd_a_power with a^i mod n for odd positive i < 2^m */
 /* a on input is not necessarily reduced mod n */
	register long i;
	register long length = (1 << (m - 1));

	if (!(*odd_a_power))
	{
		*odd_a_power = (verylong *)malloc((size_t)(length * sizeof(verylong)));
		for (i = 0; i < length; i++)
			(*odd_a_power)[i] = (verylong) 0;
	}
	zmod(a, n, &((*odd_a_power)[0]));
	zsqmod((*odd_a_power)[0], n, a_sq);
}

void
zexpmod_m_ary(
	verylong a,
	verylong e,
	verylong n,
	verylong *bb,
	long m
	)
{
	register long i;
	register long ei;
	static verylong a_sq = 0;

	if (ALLOCATE && !n)
	{
		zhalt("modulus zero in zexpmod_m_ary");
		return;
	}
	if (ALLOCATE && !e)
	{
		zone(bb);
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	if ((i = e[0]) < 0)
		i = (-i);
	if (i == 1)
	{
		zexpmod(a, e, n, bb);
		return;
	}
	if (!exp_odd_powers)
	{
		exp_odd_powers = (verylong **)malloc((size_t)(NBITS * sizeof(verylong *)));
		for (ei = 0; ei < NBITS; ei++)
			exp_odd_powers[ei] = (verylong *) 0;
	}
	if (m <= 1)
		m = zdefault_m(i);
	else if (m >= NBITS)
		m = (NBITS - 1);
	{
		register long k = 0;
		register long od = 0;
		register long odlen;
		register long left;
		register long zeros = 0;
		register long rem = 0;
		register long first = 1;
		register long reach = 1;
		verylong **loc_pow = &(exp_odd_powers[m]);

		init_odd_power(a, n, &a_sq, m, loc_pow);
		for (; i > 0; i--)
		{
			ei = e[i];
			if (!k)
			{
				k = 1;
				while (k <= ei)
					k <<= 1;
			}
			while (k >>= 1)
			{
				if (zeros)
				{
					if (ei & k)
					{
						odlen = od = 1;
						rem = zeros;
						left = zeros = 0;
					}
					else
						zeros++;
				}
				else if (!od)
				{
					if (ei & k)
					{
						odlen = od = 1;
						left = zeros = 0;
					}
					else
						zeros = 1;
				}
				else
				{
					left++;
					if (odlen + left == m)
					{
						if (ei & k)
						{
							od <<= left;
							odlen += left;
							left = 0;
							od++;
							od >>= 1;
							for (; reach <= od; reach++)
								zmulmod((*loc_pow)[reach - 1], a_sq, n, &((*loc_pow)[reach]));
							if (first)
							{
								zcopy((*loc_pow)[od], bb);
								first = rem = odlen = od = 0;
							}
							else
							{
								for (rem = rem + odlen; rem; rem--)
									zsqmod(*bb, n, bb);
								zmulmod(*bb, (*loc_pow)[od], n, bb);
								odlen = od = 0;
							}
						}
						else
						{
							if (first)
							{
								if (od == 1)
								{
									zcopy(a_sq, bb);
									left--;
								}
								else
								{
									od >>= 1;
									for (; reach <= od; reach++)
										zmulmod((*loc_pow)[reach - 1], a_sq, n, &((*loc_pow)[reach]));
									zcopy((*loc_pow)[od], bb);
								}
								zeros = left;
								first = left = odlen = od = rem = 0;
							}
							else
							{
								for (rem = rem + odlen; rem; rem--)
									zsqmod(*bb, n, bb);
								od >>= 1;
								for (; reach <= od; reach++)
									zmulmod((*loc_pow)[reach - 1], a_sq, n, &((*loc_pow)[reach]));
								zmulmod(*bb, (*loc_pow)[od], n, bb);
								zeros = left;
								left = odlen = od = 0;
							}
						}
					}
					else if (ei & k)
					{
						od <<= left;
						odlen += left;
						left = 0;
						od++;
					}
				}
			}
			k = RADIX;
		}
		if (od)
		{
			if (first)
			{
				if (od == 1)
				{
					zcopy(a_sq, bb);
					left--;
				}
				else
				{
					od >>= 1;
					for (; reach < od; reach++)
						zmulmod((*loc_pow)[reach - 1], a_sq, n, &((*loc_pow)[reach]));
					zmulmod((*loc_pow)[od - 1], a_sq, n, bb);
				}
			}
			else
			{
				for (rem = rem + odlen; rem; rem--)
					zsqmod(*bb, n, bb);
				od >>= 1;
				for (; reach <= od; reach++)
					zmulmod((*loc_pow)[reach - 1], a_sq, n, &((*loc_pow)[reach]));
				zmulmod(*bb, (*loc_pow)[od], n, bb);
			}
		}
		if (left || zeros)
		{
			for (rem = zeros + left; rem; rem--)
				zsqmod(*bb, n, bb);
		}
	}

	if (e[0] < 0 && zinv(*bb, n, bb))
		zhalt("undefined result in zexpmod_m_ary");
}

void
zsexp(
	verylong a,
	long e,
	verylong *bb
	)
{
	static verylong le = 0;

	zintoz(e, &le);
	zexp(a, le, bb);
}

void
zexp(
	verylong a,
	verylong e,
	verylong *bb
	)
{
	register long i;
	register long j;
	register long k = 0;
	long sa;
	verylong b = (*bb);
	static verylong temp = 0;
/*	extern double log10(); */

	if (ALLOCATE && !e)
	{
		zone(bb);
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	if ((e[0] == 1) && (!(e[1])))
		zone(bb);
	else if ((a[1] == 1) && (a[0] == 1))
		zone(bb);
	else if ((a[1] == 1) && (a[0] == -1))
	{
		if (e[1] & 1)
			zintoz((long) (-1), bb);
		else
			zone(bb);
	}
	else if (e[0] < 0)
	{
		zhalt("negative exponent in zexp");
		return;
	}
	else if (!(a[1]) && (a[0] == 1))
		zzero(bb);
	else
	{
		if ((sa = a[0]) < 0)
			sa = (-sa);
		i = (long) ((sa - 1 + log10((double) (a[sa])) / LOG10RAD) * ztoint(e) + 4);
		zsetlength(&b, i, "in zexp, third argument");
		zsetlength(&temp, i, "in zexp, local");
		for (i = sa; i >= 0; i--)
			b[i] = a[i];
		for (i = e[0]; i; i--)
		{
			j = e[i];
			if (!k)
			{
				k = 1;
				while ((k << 1) <= j)
					k <<= 1;
			}
			while (k >>= 1)
			{
				zsqin(&b);
				if (j & k)
					zmulin(a, &b);
			}
			k = RADIX;
		}
		*bb = b;
	}
}

long 
zexpmods(
	long a,
	long e,
	long n
	)
{
	if (!e)
		return (1);
	else
	{
		register long aa = a % n;
		register long ipow2 = 1;
		long b = aa;

		if (e<0) e = -e;
		while ((ipow2 << 1) <= e)
			ipow2 <<= 1;
		while (ipow2 >>= 1)
		{
			b = zmulmods(b, b, n);
			if (e & ipow2)
				b = zmulmods(b, aa, n);
		}
		return (b);
	}
}

void
zexpmod_doub1(
	verylong xx1,
	verylong ee1,
	verylong xx2,
	verylong ee2,
	verylong n,
	verylong *r
	)
{
        register long bit11, bit21;
        register long i,j,mask,block;
        static verylong tab[2][2];
        static long non_ini = 1;
        static verylong e1 = 0;
        static verylong e2 = 0;
        if (ALLOCATE && !n)
        {
                zhalt("modulus zero in zexpmod_doub1");
                return;
        }
        if ((zscompare(ee1,0)<0) || (zscompare(ee2,0)<0)) {
                zhalt("negative exponent in zexpmod_doub1");
                zintoz(0,r);
                return;
        }
	if (non_ini) {
		for (i=0;i<2;i++) for (j=0;j<2;j++)
			tab[i][j] = 0;
		non_ini = 0;
	}
#define MOD_INITPAR \
        if (zcompare(ee1,ee2)>=0) { \
		zcopy(xx1,&(tab[1][0])); \
		zcopy(xx2,&(tab[0][1])); \
                zcopy(ee1,&e1); \
                zcopy(ee1,&e2); /* create space for e2 */ \
                zcopy(ee2,&e2); \
        } else { \
		zcopy(xx2,&(tab[1][0])); \
		zcopy(xx1,&(tab[0][1])); \
                zcopy(ee2,&e2); /* create space for e2 */ \
                zcopy(ee1,&e2); \
                zcopy(ee2,&e1); \
        } \
        if (!(zscompare(e1,0))) { \
		zintoz(1,r); \
                return; \
        } \
        for (i=e2[0]+1;i<=e1[0];i++) e2[i] = 0;

	MOD_INITPAR

	zmulmod(tab[1][0],tab[0][1],n,&(tab[1][1]));

	block = z2log(e1)-1;
        mask = ( 1<<(block % NBITS) );
        block = (block / NBITS) +1;
        bit11 = 1;
        bit21 = (e2[block]&mask) ? 1 : 0;
	zcopy(tab[1][bit21],r);

#define MOD_FIRST \
	mask >>= 1; \
	if (!mask) { \
		block --; \
		if (!block) return; \
		mask  = ( 1 <<(NBITS-1) ); \
	} \
	bit11 = (e1[block]&mask) ? 1 : 0; \
	bit21 = (e2[block]&mask) ? 1 : 0; 

	MOD_FIRST
	for (;;) {
		zsqmod(*r,n,r);
		if (bit11 || bit21) 
			zmulmod(tab[bit11][bit21],*r,n,r);
		MOD_FIRST
	}
}

void
zexpmod_doub2(
	verylong xx1,
	verylong ee1,
	verylong xx2,
	verylong ee2,
	verylong n,
	verylong *r
	)
{
	register long i,j,mask,block;
	register long bit11, bit12, bit21, bit22;
	static verylong tab[4][4];
	static long non_ini = 1;
	static verylong e1 = 0;
	static verylong e2 = 0;
        if (ALLOCATE && !n)
        {
                zhalt("modulus zero in zexpmod_doub2");
                return;
        }
	if ((zscompare(ee1,0)<0) || (zscompare(ee2,0)<0)) {
		zhalt("negative exponent in zexpmod_doub2");
		zintoz(0,r);
		return;
	}
	if (non_ini) {
		for (i=0;i<4;i++) for (j=0;j<4;j++)
			tab[i][j] = 0;
		non_ini = 0;
	}

	MOD_INITPAR

	zsqmod(tab[1][0],n,&(tab[2][0]));
	zsqmod(tab[0][1],n,&(tab[0][2]));
	zmulmod(tab[2][0],tab[1][0],n,&(tab[3][0]));
	zmulmod(tab[0][2],tab[0][1],n,&(tab[0][3]));
	for (j=1;j<4;j++) {
		for (i=1;i<4;i++) if ((i&1) | (j&1))
			zmulmod(tab[j][0],tab[0][i],n,&(tab[j][i]));
	}

	block = z2log(e1)-1;
	mask = ( 1<<(block % NBITS) );
	block = (block / NBITS) +1;
	bit11 = 1;
	bit21 = (e2[block]&mask) ? 1 : 0;
	mask >>= 1;
	if (!mask) {
		block --;
		if (!block) {
			zcopy(tab[1][bit21],r);
			return;
		}
		mask = ( 1<<(NBITS-1) );
	}
	bit12 = (e1[block]&mask) ? 1 : 0;
	bit22 = (e2[block]&mask) ? 1 : 0;
	if (bit12 || bit22) {
		zcopy(tab[2+bit12][(bit21<<1)+bit22],r);
	} else {
		zcopy(tab[bit11][bit21],r);
		zsqmod(*r,n,r);
	}

#define MOD_SECOND \
	mask >>= 1; \
	if (!mask) { \
		block--; \
		if (!block) { \
			zsqmod(*r,n,r); \
			if (bit11||bit21) \
				zmulmod(tab[bit11][bit21],*r,n,r); \
			return; \
		} \
		mask = ( 1<<(NBITS-1) ); \
	} \
        bit12 = (e1[block]&mask) ? 1 : 0; \
        bit22 = (e2[block]&mask) ? 1 : 0;

	MOD_FIRST
	MOD_SECOND
	for (;;) {
		zsqmod(*r,n,r);
		if (bit11 || bit21) {
			if (bit12 || bit22) {
			  zsqmod(*r,n,r);
			  zmulmod(tab[(bit11<<1)+bit12][(bit21<<1)+bit22],*r,n,r);
			} else {
			  zmulmod(tab[bit11][bit21],*r,n,r);
			  zsqmod(*r,n,r);
			}
			MOD_FIRST
		} else {
			bit11 = bit12;
			bit21 = bit22;
		}
		MOD_SECOND
	}
}

void
zexpmod_doub3(
	verylong xx1,
	verylong ee1,
	verylong xx2,
	verylong ee2,
	verylong n,
	verylong *r
	)
{
	register long i,j, mask, block;
	register long bit11, bit12, bit13, bit21, bit22, bit23;
	static verylong tab[8][8];
	static long non_ini = 1;
	static verylong e1 = 0;
	static verylong e2 = 0;
        if (ALLOCATE && !n)
        {
                zhalt("modulus zero in zexpmod_doub3");
                return;
        }
        if ((zscompare(ee1,0)<0) || (zscompare(ee2,0)<0)) {
                zhalt("negative exponent in zexpmod_doub3");
                zintoz(0,r);
                return;
        }
	if (non_ini) {
		for (i=0;i<8;i++) for (j=0;j<8;j++)
			tab[i][j] = 0;
		non_ini = 0;
	}

	MOD_INITPAR

	zsqmod(tab[1][0],n,&(tab[2][0]));
	zsqmod(tab[0][1],n,&(tab[0][2]));
	for (i=3;i<8;i+=2) zmulmod(tab[2][0],tab[i-2][0],n,&(tab[i][0]));
	for (i=3;i<8;i+=2) zmulmod(tab[0][2],tab[0][i-2],n,&(tab[0][i]));
	for (i=1;i<8;i++) zmulmod(tab[0][1],tab[1][i-1],n,&(tab[1][i]));
        for (i=1;i<8;i+=2) zmulmod(tab[0][i],tab[2][0],n,&(tab[2][i]));
        for (i=1;i<8;i++) zmulmod(tab[0][1],tab[3][i-1],n,&(tab[3][i]));
	for (j=4;j<8;j++) {
		zmulmod(tab[1][0],tab[j-1][1],n,&(tab[j][1]));
		for (i=3;i<8;i+=2) zmulmod(tab[0][2],tab[j][i-2],n,&(tab[j][i]));
		j++;
		for (i=1;i<8;i++) zmulmod(tab[0][1],tab[j][i-1],n,&(tab[j][i]));
	}
        block = z2log(e1)-1;
        mask = ( 1<<(block % NBITS) );
        block = (block / NBITS) +1;
        bit11 = 1;
        bit21 = (e2[block]&mask) ? 1 : 0;
        mask >>= 1;
        if (!mask) {
                block --;
                if (!block) {
                        zcopy(tab[1][bit21],r);
                        return;
                }
                mask = ( 1<<(NBITS-1) );
        }
        bit12 = (e1[block]&mask) ? 1 : 0;
        bit22 = (e2[block]&mask) ? 1 : 0;
	mask >>= 1;
	if (!mask) {
		block --;
		if (!block) {
			if (bit12 || bit22) {
				zcopy(tab[2+bit12][(bit21<<1)+bit22],r);
			} else {
                		zcopy(tab[bit11][bit21],r);
                		zsqmod(*r,n,r);
			}
			return;
		}
		mask = ( 1<<(NBITS-1) );
	}
	bit13 = (e1[block]&mask) ? 1 : 0;
        bit23 = (e2[block]&mask) ? 1 : 0;
	if (bit13 || bit23) {
		zcopy(tab[4+(bit12<<1)+bit13][(bit21<<2)+(bit22<<1)+bit23],r);
	} else {
		if (bit12 || bit22) {
			zcopy(tab[2+bit12][(bit21<<1)+bit22],r);
		} else {
			zcopy(tab[1][bit21],r);
			zsqmod(*r,n,r);
		}
		zsqmod(*r,n,r);
	}

#define MOD_THIRD \
	mask >>= 1; \
	if (!mask) { \
		block --; \
                if (!block) { \
			zsqmod(*r,n,r); \
			if (bit11 || bit21) { \
				if (bit12 || bit22) { \
				  zsqmod(*r,n,r); \
				  zmulmod(tab[(bit11<<1)+bit12][(bit21<<1)+bit22],*r,n,r); \
				} else { \
				  zmulmod(tab[bit11][bit21],*r,n,r); \
				  zsqmod(*r,n,r); \
				} \
			} else { \
				zsqmod(*r,n,r); \
				if (bit12 || bit22) { \
					zmulmod(tab[bit12][bit22],*r,n,r); \
				} \
			} \
			return; \
		} \
		mask = ( 1<<(NBITS-1) ); \
	} \
	bit13 = (e1[block]&mask) ? 1 : 0; \
        bit23 = (e2[block]&mask) ? 1 : 0;

	MOD_FIRST
	MOD_SECOND
	MOD_THIRD

	for (;;) {
		zsqmod(*r,n,r);
		if (bit11 || bit21) {
			if (bit13 || bit23) {
				zsqmod(*r,n,r);
				zsqmod(*r,n,r);
				zmulmod(tab[(bit11<<2)+(bit12<<1)+bit13]
					    [(bit21<<2)+(bit22<<1)+bit23],*r,n,r);
			} else {
				if (bit12 || bit22) {
					zsqmod(*r,n,r);
					zmulmod(tab[(bit11<<1)+bit12]
						    [(bit21<<1)+bit22],*r,n,r);
				} else {
					zmulmod(tab[bit11][bit21],*r,n,r);
					zsqmod(*r,n,r);
				}
				zsqmod(*r,n,r);
			}
			MOD_FIRST
			MOD_SECOND
		} else {
			bit11 = bit12;
			bit21 = bit22;
			bit12 = bit13;
			bit22 = bit23;
		}
		MOD_THIRD
	}

}

void
zexpmod_doub(
	verylong xx1,
	verylong ee1,
	verylong xx2,
	verylong ee2,
	verylong n,
	verylong *r
	)
{
        if (ALLOCATE && !n)
        {
                zhalt("modulus zero in zexpmod_doub");
                return;
        }
        if ((zscompare(ee1,0)<0) || (zscompare(ee2,0)<0)) {
                zhalt("negative exponent in zexpmod_doub");
                zintoz(0,r);
                return;
        }
	if (zscompare(ee1,1<<NBITS)<0 && zscompare(ee2,1<<NBITS)<0)
		zexpmod_doub1(xx1,ee1,xx2,ee2,n,r);
	else if (ee1[0]<10 && ee2[0]<10)
		zexpmod_doub2(xx1,ee1,xx2,ee2,n,r);
	else
		zexpmod_doub3(xx1,ee1,xx2,ee2,n,r);
}

void
zmontexp_doub1(
	verylong xx1,
	verylong ee1,
	verylong xx2,
	verylong ee2,
	verylong *r
	)
{
        register long bit11, bit21;
        register long i,j,mask,block;
        static verylong tab[2][2];
        static long non_ini = 1;
        static verylong e1 = 0;
        static verylong e2 = 0;
        if (!zn)
        {
                zhalt("undefined Montgomery modulus in zmontexp_doub1");
                return;
        }
        if ((zscompare(ee1,0)<0) || (zscompare(ee2,0)<0)) {
                zhalt("negative exponent in zmontexp_doub1");
                zintoz(0,r);
                return;
        }
	if (non_ini) {
		for (i=0;i<2;i++) for (j=0;j<2;j++)
			tab[i][j] = 0;
		non_ini = 0;
	}
#define MONT_INITPAR \
        if (zcompare(ee1,ee2)>=0) { \
		zcopy(xx1,&(tab[1][0])); \
		zcopy(xx2,&(tab[0][1])); \
                zcopy(ee1,&e1); \
                zcopy(ee1,&e2); /* create space for e2 */ \
                zcopy(ee2,&e2); \
        } else { \
		zcopy(xx2,&(tab[1][0])); \
		zcopy(xx1,&(tab[0][1])); \
                zcopy(ee2,&e2); /* create space for e2 */ \
                zcopy(ee1,&e2); \
                zcopy(ee2,&e1); \
        } \
        if (!(zscompare(e1,0))) { \
		zcopy(zr,r); \
                return; \
        } \
        for (i=e2[0]+1;i<=e1[0];i++) e2[i] = 0;

	MONT_INITPAR

	zmontmul(tab[1][0],tab[0][1],&(tab[1][1]));

	block = z2log(e1)-1;
        mask = ( 1<<(block % NBITS) );
        block = (block / NBITS) +1;
        bit11 = 1;
        bit21 = (e2[block]&mask) ? 1 : 0;
	zcopy(tab[1][bit21],r);

#define MONT_FIRST \
	mask >>= 1; \
	if (!mask) { \
		block --; \
		if (!block) return; \
		mask  = ( 1 <<(NBITS-1) ); \
	} \
	bit11 = (e1[block]&mask) ? 1 : 0; \
	bit21 = (e2[block]&mask) ? 1 : 0; \

	MONT_FIRST
	for (;;) {
		zmontsq(*r,r);
		if (bit11 || bit21) 
			zmontmul(tab[bit11][bit21],*r,r);
		MONT_FIRST
	}
}

void
zmontexp_doub2(
	verylong xx1,
	verylong ee1,
	verylong xx2,
	verylong ee2,
	verylong *r
	)
{
	register long i,j,mask,block;
	register long bit11, bit12, bit21, bit22;
	static verylong tab[4][4];
	static long non_ini = 1;
	static verylong e1 = 0;
	static verylong e2 = 0;
        if (!zn)
        {
                zhalt("undefined Montgomery modulus in zmontexp_doub2");
                return;
        }
	if ((zscompare(ee1,0)<0) || (zscompare(ee2,0)<0)) {
		zhalt("negative exponent in zmontexp_doub2");
		zintoz(0,r);
		return;
	}
	if (non_ini) {
		for (i=0;i<4;i++) for (j=0;j<4;j++)
			tab[i][j] = 0;
		non_ini = 0;
	}

	MONT_INITPAR

	zmontsq(tab[1][0],&(tab[2][0]));
	zmontsq(tab[0][1],&(tab[0][2]));
	zmontmul(tab[2][0],tab[1][0],&(tab[3][0]));
	zmontmul(tab[0][2],tab[0][1],&(tab[0][3]));
	for (j=1;j<4;j++) {
		for (i=1;i<4;i++) if ((i&1) | (j&1))
			zmontmul(tab[j][0],tab[0][i],&(tab[j][i]));
	}

	block = z2log(e1)-1;
	mask = ( 1<<(block % NBITS) );
	block = (block / NBITS) +1;
	bit11 = 1;
	bit21 = (e2[block]&mask) ? 1 : 0;
	mask >>= 1;
	if (!mask) {
		block --;
		if (!block) {
			zcopy(tab[1][bit21],r);
			return;
		}
		mask = ( 1<<(NBITS-1) );
	}
	bit12 = (e1[block]&mask) ? 1 : 0;
	bit22 = (e2[block]&mask) ? 1 : 0;
	if (bit12 || bit22) {
		zcopy(tab[2+bit12][(bit21<<1)+bit22],r);
	} else {
		zcopy(tab[bit11][bit21],r);
		zmontsq(*r,r);
	}

#define MONT_SECOND \
	mask >>= 1; \
	if (!mask) { \
		block--; \
		if (!block) { \
			zmontsq(*r,r); \
			if (bit11||bit21) \
				zmontmul(tab[bit11][bit21],*r,r); \
			return; \
		} \
		mask = ( 1<<(NBITS-1) ); \
	} \
        bit12 = (e1[block]&mask) ? 1 : 0; \
        bit22 = (e2[block]&mask) ? 1 : 0;

	MONT_FIRST
	MONT_SECOND
	for (;;) {
		zmontsq(*r,r);
		if (bit11 || bit21) {
			if (bit12 || bit22) {
			  zmontsq(*r,r);
			  zmontmul(tab[(bit11<<1)+bit12][(bit21<<1)+bit22],*r,r);
			} else {
			  zmontmul(tab[bit11][bit21],*r,r);
			  zmontsq(*r,r);
			}
			MONT_FIRST
		} else {
			bit11 = bit12;
			bit21 = bit22;
		}
		MONT_SECOND
	}
}

void
zmontexp_doub3(
	verylong xx1,
	verylong ee1,
	verylong xx2,
	verylong ee2,
	verylong *r
	)
{
	register long i,j, mask, block;
	register long bit11, bit12, bit13, bit21, bit22, bit23;
	static verylong tab[8][8];
	static long non_ini = 1;
	static verylong e1 = 0;
	static verylong e2 = 0;
        if (!zn)
        {
                zhalt("undefined Montgomery modulus in zmontexp_doub3");
                return;
        }
        if ((zscompare(ee1,0)<0) || (zscompare(ee2,0)<0)) {
                zhalt("negative exponent in zmontexp_doub3");
                zintoz(0,r);
                return;
        }
	if (non_ini) {
		for (i=0;i<8;i++) for (j=0;j<8;j++)
			tab[i][j] = 0;
		non_ini = 0;
	}

	MONT_INITPAR

	zmontsq(tab[1][0],&(tab[2][0]));
	zmontsq(tab[0][1],&(tab[0][2]));
	for (i=3;i<8;i+=2) zmontmul(tab[2][0],tab[i-2][0],&(tab[i][0]));
	for (i=3;i<8;i+=2) zmontmul(tab[0][2],tab[0][i-2],&(tab[0][i]));
	for (i=1;i<8;i++) zmontmul(tab[0][1],tab[1][i-1],&(tab[1][i]));
        for (i=1;i<8;i+=2) zmontmul(tab[0][i],tab[2][0],&(tab[2][i]));
        for (i=1;i<8;i++) zmontmul(tab[0][1],tab[3][i-1],&(tab[3][i]));
	for (j=4;j<8;j++) {
		zmontmul(tab[1][0],tab[j-1][1],&(tab[j][1]));
		for (i=3;i<8;i+=2) zmontmul(tab[0][2],tab[j][i-2],&(tab[j][i]));
		j++;
		for (i=1;i<8;i++) zmontmul(tab[0][1],tab[j][i-1],&(tab[j][i]));
	}
        block = z2log(e1)-1;
        mask = ( 1<<(block % NBITS) );
        block = (block / NBITS) +1;
        bit11 = 1;
        bit21 = (e2[block]&mask) ? 1 : 0;
        mask >>= 1;
        if (!mask) {
                block --;
                if (!block) {
                        zcopy(tab[1][bit21],r);
                        return;
                }
                mask = ( 1<<(NBITS-1) );
        }
        bit12 = (e1[block]&mask) ? 1 : 0;
        bit22 = (e2[block]&mask) ? 1 : 0;
	mask >>= 1;
	if (!mask) {
		block --;
		if (!block) {
			if (bit12 || bit22) {
				zcopy(tab[2+bit12][(bit21<<1)+bit22],r);
			} else {
                		zcopy(tab[bit11][bit21],r);
                		zmontsq(*r,r);
			}
			return;
		}
		mask = ( 1<<(NBITS-1) );
	}
	bit13 = (e1[block]&mask) ? 1 : 0;
        bit23 = (e2[block]&mask) ? 1 : 0;
	if (bit13 || bit23) {
		zcopy(tab[4+(bit12<<1)+bit13][(bit21<<2)+(bit22<<1)+bit23],r);
	} else {
		if (bit12 || bit22) {
			zcopy(tab[2+bit12][(bit21<<1)+bit22],r);
		} else {
			zcopy(tab[1][bit21],r);
			zmontsq(*r,r);
		}
		zmontsq(*r,r);
	}

#define MONT_THIRD \
	mask >>= 1; \
	if (!mask) { \
		block --; \
                if (!block) { \
			zmontsq(*r,r); \
			if (bit11 || bit21) { \
				if (bit12 || bit22) { \
				  zmontsq(*r,r); \
				  zmontmul(tab[(bit11<<1)+bit12][(bit21<<1)+bit22],*r,r); \
				} else { \
				  zmontmul(tab[bit11][bit21],*r,r); \
				  zmontsq(*r,r); \
				} \
			} else { \
				zmontsq(*r,r); \
				if (bit12 || bit22) { \
					zmontmul(tab[bit12][bit22],*r,r); \
				} \
			} \
			return; \
		} \
		mask = ( 1<<(NBITS-1) ); \
	} \
	bit13 = (e1[block]&mask) ? 1 : 0; \
        bit23 = (e2[block]&mask) ? 1 : 0;

	MONT_FIRST
	MONT_SECOND
	MONT_THIRD

	for (;;) {
		zmontsq(*r,r);
		if (bit11 || bit21) {
			if (bit13 || bit23) {
				zmontsq(*r,r);
				zmontsq(*r,r);
				zmontmul(tab[(bit11<<2)+(bit12<<1)+bit13]
					    [(bit21<<2)+(bit22<<1)+bit23],*r,r);
			} else {
				if (bit12 || bit22) {
					zmontsq(*r,r);
					zmontmul(tab[(bit11<<1)+bit12]
						    [(bit21<<1)+bit22],*r,r);
				} else {
					zmontmul(tab[bit11][bit21],*r,r);
					zmontsq(*r,r);
				}
				zmontsq(*r,r);
			}
			MONT_FIRST
			MONT_SECOND
		} else {
			bit11 = bit12;
			bit21 = bit22;
			bit12 = bit13;
			bit22 = bit23;
		}
		MONT_THIRD
	}

}

void
zmontexp_doub(
	verylong xx1,
	verylong ee1,
	verylong xx2,
	verylong ee2,
	verylong *r
	)
{
        if (!zn)
        {
                zhalt("undefined Montgomery modulus in zmontexp_doub");
                return;
        }
        if ((zscompare(ee1,0)<0) || (zscompare(ee2,0)<0)) {
                zhalt("negative exponent in zmontexp_doub");
                zintoz(0,r);
                return;
        }
	if (zscompare(ee1,1<<NBITS)<0 && zscompare(ee2,1<<NBITS)<0)
		zmontexp_doub1(xx1,ee1,xx2,ee2,r);
	else if (ee1[0]<10 && ee2[0]<10)
		zmontexp_doub2(xx1,ee1,xx2,ee2,r);
	else
		zmontexp_doub3(xx1,ee1,xx2,ee2,r);
}

void
z2mul(
	verylong n,
	verylong *rres
	)
{
	register long sn;
	register long i;
	register long carry = 0;
	verylong res = *rres;

	if (ALLOCATE && !n)
	{
		zzero(rres);
		return;
	}
	if ((!n[1]) && (n[0] == 1))
	{
		zzero(rres);
		return;
	}
	if ((sn = n[0]) < 0)
		sn = -sn;
	zsetlength(&res, sn + 1, "in z2mul, second argument");
	if (n == *rres) n = res;
	*rres = res;
	for (i = 1; i <= sn; i++)
	{
		if ((res[i] = (n[i] << 1) + carry) >= RADIX)
		{
			res[i] -= RADIX;
			carry = 1;
		}
		else
			carry = 0;
	}
	if (carry)
		res[++sn] = 1;
	if (n[0] < 0)
		res[0] = -sn;
	else
		res[0] = sn;
}

long 
z2div(
	verylong n,
	verylong *rres
	)
{
	register long sn;
	register long i;
	register long result;
	verylong res = *rres;

	if ((!n) || (!n[1]) && (n[0] == 1))
	{
		zzero(rres);
		return (0);
	}
	if ((sn = n[0]) < 0)
		sn = -sn;
	zsetlength(&res, sn, "in z2div, second argument");
	if (n == *rres) n = res;
	*rres = res;
	result = n[1] & 1;
	for (i = 1; i < sn; i++)
	{
		res[i] = (n[i] >> 1);
		if (n[i + 1] & 1)
			res[i] += (RADIX >> 1);
	}
	if (res[sn] = (n[sn] >> 1))
		res[0] = n[0];
	else if (sn == 1)
	{
		res[0] = 1;
	}
	else if (n[0] < 0)
		res[0] = -sn + 1;
	else
		res[0] = sn - 1;
	return (result);
}

long 
z2mod(
	verylong n
	)
{
	if (!n)
		return (0);
	return (n[1] & 1);
}

void
zlshift(
	verylong n,
	long k,
	verylong *rres
	)
{
	register long big;
	register long small;
	register long sn;
	register long i;
	register long cosmall;
	verylong res = *rres;

	if (ALLOCATE && !n)
	{
		zzero(rres);
		return;
	}
	if ((!n[1]) && (n[0] == 1))
	{
		zzero(rres);
		return;
	}
	if (!k)
	{
		if (n != *rres)
			zcopy(n, rres);
		return;
	}
	if (k < 0)
	{
		zrshift(n, -k, rres);
		return;
	}
	if (k == 1)
	{
		z2mul(n, rres);
		return;
	}
	if ((sn = n[0]) < 0)
		sn = -sn;
	i = sn + (big = k / NBITS);
	if (small = k - big * NBITS)
	{
		zsetlength(&res, i + 1, "in zlshift, third argument");
		if (n == *rres) n = res;
		*rres = res;
		res[i + 1] = n[sn] >> (cosmall = NBITS - small);
		for (i = sn; i > 1; i--)
			res[i + big] = ((n[i] << small) & RADIXM) + (n[i - 1] >> cosmall);
		res[big + 1] = (n[1] << small) & RADIXM;
		for (i = big; i; i--)
			res[i] = 0;
		if (res[sn + big + 1])
			big++;
	}
	else
	{
		zsetlength(&res, i, "in zlshift, third argument");
		if (n == *rres) n = res;
		*rres = res;
		for (i = sn; i; i--)
			res[i + big] = n[i];
		for (i = big; i; i--)
			res[i] = 0;
	}
	if (n[0] > 0)
		res[0] = n[0] + big;
	else
		res[0] = n[0] - big;
}

void
zrshift(
	verylong n,
	long k,
	verylong *rres
	)
{
	register long big;
	register long small;
	register long sn;
	register long i;
	register long cosmall;
	verylong res = *rres;

	if (ALLOCATE && !n)
	{
		zzero(rres);
		return;
	}
	if ((!n[1]) && (n[0] == 1))
	{
		zzero(rres);
		return;
	}
	if (!k)
	{
		if (n != *rres)
			zcopy(n, rres);
		return;
	}
	if (k < 0)
	{
		zlshift(n, -k, rres);
		return;
	}
	if (k == 1)
	{
		z2div(n, rres);
		return;
	}
	big = k / NBITS;
	small = k - big * NBITS;
	if ((sn = n[0]) < 0)
		sn = -sn;
	if ((big >= sn) || ((big == sn - 1) && (!(n[sn] >> small))))
	{
		zzero(rres);
		return;
	}
	sn -= big;
	zsetlength(&res, sn, "in zrshift, third argument");
	if (n == *rres) n = res;
	*rres = res;
	if (small)
	{
		cosmall = NBITS - small;
		for (i = 1; i < sn; i++)
			res[i] = (n[i + big] >> small) +
				((n[i + big + 1] << cosmall) & RADIXM);
		if (!(res[sn] = (n[sn + big] >> small)))
			sn--;
	}
	else
		for (i = 1; i <= sn; i++)
			res[i] = n[i + big];
	if (n[0] > 0)
		res[0] = sn;
	else
		res[0] = -sn;
}

long 
zmakeodd(
	verylong *nn
	)
{
	verylong n = *nn;
	register long i;
	long shift = 1;

	if (!n || (!n[1] && (n[0] == 1)))
		return (-1);
	while (!(n[shift]))
		shift++;
	i = n[shift];
	shift = NBITS * (shift - 1);
	while (!(i & 1))
	{
		shift++;
		i >>= 1;
	}
	zrshift(n, shift, &n);
	return (shift);
}

long 
zcomposite(
	verylong *mm,
	long t,
	long firstbase
	)
{
	register long sm = 0;
	register long i;
	register long j;
	register long s;
	register long loc_rand;
	verylong m = *mm;
	static verylong u = 0;
	static verylong m1 = 0;
	static verylong a = 0;

	if (!m || m[0] < 0)
		return (1);
	if ((m[0] == 1) && (m[1] <= RADIXROOT))
	{
		sm = m[1];
		if (sm < 2)
			return (1);
		if (sm == 2)
			return (0);
		if (!(sm & 1))
			return (1);
		i = 3;
		while (i * i <= sm)
		{
			if (!(sm % i))
				return (1);
			i++;
			i++;
		}
		return (0);
	}
	if (!(m[1] & 1))
		return (1);
	zsetlength(&u, (i = m[0]), "in zcomposite, locals\n");
	zsetlength(&m1, i, "");
	zsetlength(&a, i, "");
	zsubpos(m, one, &m1);
	zcopy(m1, &u);
	s = zmakeodd(&u) - 1;
	if (firstbase < 0)
		firstbase = -firstbase;
	if (!firstbase && (sm = (m[1]-3)) <= (RADIXROOT-3))
		sm = RADIXM - 3;
	if (t < 0)
		i = -t;
	else
		i = t;
	for (; i > 0; i--)
	{
		if (sm)
		{
			zintoz((loc_rand = (3 + zrandom(sm)) & RADIXM), &a);
			zexpmod(a, u, m, &a);
		}
		else
		{
			if (firstbase == 2)
				z2expmod(u, m, &a);
			else
			{
				zintoz(firstbase, &a);
				zexpmod(a, u, m, &a);
			}
			if ((sm = (m[1]-3)) <= (RADIXROOT-3))
				sm = RADIXM - 3;
		}
		if ((a[0] != 1 || a[1] != 1) && zcompare(m1, a))
		{
			for (j = s; j; j--)
			{
				zsqmod(a, m, &a);
				if (!zcompare(a, m1))
					goto nexti;
			}
			if (t > 0)
				return (1);
			else
			{
				zsqmod(a, m, &a);
				zsubmod(a, one, m, &a);
				zintoz(loc_rand, &u);
				zmulmod(a, u, m, &a);
				if (!a[1] && a[0] == 1)
				{
					zcopy(a, &m);
					return (-1);
				}
				zgcd(a, m, &u);
				if (u[1] == 1 && u[0] == 1)
					return (1);
				zcopy(u, &m);
				return (-2);
			}
		}
nexti:		;
	}
	return (0);
}

long 
zprime(
	verylong n,
	long t,
	long firstbase
	)
{
	if (t < 0)
		t = -t;
	return (1 - zcomposite(&n, t, firstbase));
}

static void
zsmexp(
	long a,
	verylong e,
	verylong *bb
	)
{
/* mixed arithmetic exponentiation, only for in zmcomposite */
 /* with mixed montgomery and ordinary arithmetic */
	register long i;
	register long j;
	register long k = 0;
	verylong b = *bb;

	if (ALLOCATE && !zn)
	{
		zhalt("wrong call to zsmexp   BUG");
		return;
	}
	if (ALLOCATE && !e)
	{
		zcopy(zr, bb);
		return;
	}
	if (ALLOCATE && !a)
	{
		zzero(bb);
		return;
	}
	if ((i = e[0]) < 0)
		i = (-i);	/* negative exponent made positive ... */
	if ((i == 1) && (!(e[1])))
		zcopy(zr, &b);
	else
	{
		zintoz(a, &b);
		ztom(b, &b);
		for (; i; i--)
		{
			j = e[i];
			if (!k)
			{
				k = 1;
				while ((k << 1) <= j)
					k <<= 1;
			}
			while (k >>= 1)
			{
				zmontsq(b, &b);
				if (j & k)
				{
					zsmul(b, a, &b);
					zmod(b, zn, &b);
				}
			}
			k = RADIX;
		}
	}
	*bb = b;
}

long 
zmcomposite(
	verylong m,
	long t
	)
{
	register long sm = 0;
	register long i;
	register long j;
	register long s;
	static verylong u = 0;
	static verylong m1 = 0;
	static verylong a = 0;

	if (!m || m[0] < 0)
		return (1);
	if ((m[0] == 1) && (m[1] <= RADIXROOT))
	{
		sm = m[1];
		if (sm < 2)
			return (1);
		if (sm == 2)
			return (0);
		if (!(sm & 1))
			return (1);
		i = 3;
		while (i * i <= sm)
		{
			if (!(sm % i))
				return (1);
			i++;
			i++;
		}
		return (0);
	}
	if (!(m[1] & 1))
		return (1);
	zsetlength(&u, (i = m[0]), "in zmcomposite, locals\n");
	zsetlength(&m1, i, "");
	zsetlength(&a, i, "");
	zmkeep(m);
	zsubpos(m, zr, &m1);	/* zr is montgomery-one, m1 is
				 * montgomery-(n-1) */
	zsubpos(m, one, &u);
	s = zmakeodd(&u) - 1;
	if ((sm = m[1]) <= RADIXROOT)
		sm = RADIXM - 3;
	if (t < 0)
		t = -t;
	for (i = t; i > 0; i--)
	{
		zsmexp((3 + zrandom(sm)) & RADIXM, u, &a);
		if (zcompare(zr, a) && zcompare(m1, a))
		{
			for (j = s; j; j--)
			{
				zmontsq(a, &a);
				if (!zcompare(a, m1))
					goto nexti;
			}
			zmback();
			return (1);
		}
nexti:		;
	}
	zmback();
	return (0);
}

long 
zsqrts(
	long n
	)
{
	register long a;
	register long ndiva;
	register long newa;
	static verylong ln=0;
	static verylong rr=0;
	static verylong dif=0;

	if (n <= 0)
		return (0);
	if (n <= 3)
		return (1);
	if (n <= 8)
		return (2);
	if (n >= RADIX)
	{
		zintoz(n,&ln);
		zsqrt(ln,&rr,&dif);
		return(ztoint(rr));
	}
	newa = 3 << (2 * (NBITSH - 1));
	a = 1 << NBITSH;
	while (!(n & newa))
	{
		newa >>= 2;
		a >>= 1;
	}
	while (1)
	{
		newa = ((ndiva = n / a) + a) / 2;
		if (newa - ndiva <= 1)
		{
			if (newa * newa <= n)
				return (newa);
			else
				return (ndiva);
		}
		a = newa;
	}
}

long 
zsqrt(
	verylong n_in,
	verylong *rr,
	verylong *ddif
	)
{
	static verylong n = 0;
	static verylong a = 0;
	static verylong ndiva = 0;
	static verylong diff = 0;
	register long i;
	verylong dif = *ddif;
	verylong r = *rr;

	if (ALLOCATE && !n_in)
	{
		zzero(rr);
		zzero(ddif);
		return (1);
	}
	if ((i = n_in[0]) < 0)
	{
		zhalt("negative argument in zsqrt");
		return (0);
	}
	zcopy(n_in,&n);
	zsetlength(&a, i, "in zsqrt, locals\n");
	zsetlength(&ndiva, i, "");
	zsetlength(&diff, i, "");
	if (i == 1)
	{
		zintoz(i = zsqrts(n[1]), &r);
		zintoz(i * i, &diff);
		goto done;
	}
	a[(a[0] = (i + 1) / 2)] = zsqrts(n[i]) + 1;
	if (!(i & 1))
		a[a[0]] <<= NBITSH;
	if (a[a[0]] & RADIX)
	{
		a[a[0]] = 0;
		a[0]++;
		a[a[0]] = 1;
	}
	for (i = a[0] - 1; i; i--)
		a[i] = 0;
	while (1)
	{
		zdiv(n, a, &ndiva, &r);
		zadd(a, ndiva, &r);
		zrshift(r, (long) 1, &r);
		if (zcompare(r, ndiva) <= 0)
		{
			zsq(r, &diff);
			goto done;
		}
		zsubpos(r, ndiva, &diff);
		if ((diff[0] == 1) && (diff[1] <= 1))
		{
			zsq(r, &diff);
			if (zcompare(diff, n) > 0)
			{
				zcopy(ndiva, &r);
				zsq(r, &diff);
			}
			goto done;
		}
		zcopy(r, &a);
	}
done:
	*rr = r;
	zsubpos(n, diff, &dif);
	*ddif = dif;
	if (!(dif[1]) && (dif[0] == 1))
		return (1);
	return (0);
}

void
zpstart()
{
	register long i;
	register long j;
	register long jstep;
	register long jstart;
	register long ibnd;
	register short int *p;

	if (!lowsieve)
	{
		lowsieve = (short int *)calloc((size_t)PRIMBND, sizeof(short int));
		p = &lowsieve[0];
		for (i = PRIMBND; i; i--)
			*p++ = 1;
		jstep = 1;
		jstart = -1;
		ibnd = (zsqrts((long) (2 * PRIMBND + 1)) - 3) / 2;
		for (i = 0; i <= ibnd; i++)
		{
			jstart += 2 * ((jstep += 2) - 1);
			if (lowsieve[i])
				for (j = jstart; j < PRIMBND; j += jstep)
					lowsieve[j] = 0;
		}
	}
	lastp = 0;
	pshift = 0;
	pindex = -1;
}

void
zpstart2()
{
	lastp = 0;
	pshift = -1;
}

static void
zpshift()
{
/* auxiliary routine for prime generator */
	register long i;
	register long j;
	register long jstep;
	register long jstart;
	register long ibound;
	register short *p;
	register short *pi;

	if (!movesieve)
		movesieve = (short int *)calloc((size_t)PRIMBND, sizeof(short int));
	pi = &movesieve[0];
	if (!pshift)
	{
		p = &lowsieve[0];
		for (i = PRIMBND; i; i--)
			*pi++ = *p++;
	}
	else
	{
		for (i = PRIMBND; i; i--)
			*pi++ = 1;
		jstep = 3;
		ibound = pshift + 2 * PRIMBND + 1;
		for (i = 0; jstep * jstep <= ibound; i++)
		{
			if (lowsieve[i])
			{
				if (!((jstart = (pshift + 2) / jstep + 1) & 1))
					jstart++;
				if (jstart <= jstep)
					jstart = jstep;
				jstart = (jstart * jstep - pshift - 3) / 2;
				for (j = jstart; j < PRIMBND; j += jstep)
					movesieve[j] = 0;
			}
			jstep += 2;
		}
	}
}

long 
zpnext()
{
	if (pshift < 0)
	{
		zpstart();
		return (lastp = 2);
	}
	if (pindex < 0)
	{
		pindex = 0;
		zpshift();
		return (lastp = 3);
	}
	for (;;)
	{
		while ((++pindex) < PRIMBND)
		{
			if (movesieve[pindex])
				return (lastp = pshift + 2 * pindex + 3);
		}
		if ((pshift += 2 * PRIMBND) > 2 * PRIMBND * (2 * PRIMBND + 1))
		{
			zpstart();
			return (lastp = 2);
		}
		zpshift();
		pindex = -1;
	}
}

long 
zp()
{
	return (lastp);
}

void
zgcd(
	verylong mm1,
	verylong mm2,
	verylong *rres
	)
{
	register long agrb;
	register long shibl;
	static verylong aa = 0;
	static verylong bb = 0;
	static verylong cc = 0;
	verylong a;
	verylong b;
	verylong c;
	verylong d;
	long m1negative;
	long m2negative;

	if (!mm1)
	{
		if (mm2 != *rres)
			zcopy(mm2,rres);
		zabs(rres);
		return;
	}
	if (!mm2)
	{
		if (mm1 != *rres)
			zcopy(mm1,rres);
		zabs(rres);
		return;
	}
	if (mm1 == mm2)
	{
		if (mm1 != *rres)
			zcopy(mm1, rres);
		zabs(rres);
		return;
	}
	if (m1negative = (mm1[0] < 0))
		mm1[0] = -mm1[0];
	if (m2negative = (mm2[0] < 0))
		mm2[0] = -mm2[0];
	if (!(mm1[1]) && (mm1[0] == 1))
	{
		a = mm2;
		goto done;
	}
	if (!(mm2[1]) && (mm2[0] == 1))
	{
		a = mm1;
		goto done;
	}
	if ((agrb = mm1[0]) < mm2[0])
		agrb = mm2[0];
	zsetlength(&aa, agrb, "in zgcd, locals\n");
	zsetlength(&bb, agrb, "");
	zsetlength(&cc, agrb, "");
	if (mm1[0] != mm2[0])
	{
		if (mm1[0] > mm2[0])
		{
			zcopy(mm2, &aa);
			zmod(mm1, aa, &bb);
		}
		else
		{
			zcopy(mm1, &aa);
			zmod(mm2, aa, &bb);
		}
		if (!(bb[1]) && (bb[0] == 1))
		{
			a = aa;
			goto done;
		}
	}
	else
	{
		zcopy(mm1, &aa);
		zcopy(mm2, &bb);
	}
	if ((agrb = zmakeodd(&aa)) < (shibl = zmakeodd(&bb)))
		shibl = agrb;
	if (!(agrb = zcompare(aa, bb)))
	{
		a = aa;
		goto endshift;
	}
	else if (agrb < 0)
	{
		a = bb;
		b = aa;
	}
	else
	{
		a = aa;
		b = bb;
	}
	c = cc;
	zsubpos(a, b, &c);
	do
	{
		zmakeodd(&c);
		if (!(agrb = zcompare(b, c)))
		{
			a = b;
			goto endshift;
		}
		else if (agrb > 0)
		{
			a = b;
			b = c;
			c = a;
		}
		else
		{
			d = a;
			a = c;
			c = d;
		}
		zsubpos(a, b, &c);
	} while (c[1] || c[0] != 1);
endshift:
	zlshift(a, shibl, &a);
done:
	if (m1negative)
		mm1[0] = -mm1[0];
	if (m2negative)
		mm2[0] = -mm2[0];
	zcopy(a, rres);
}

void
zgcdeucl(
	verylong mm1,
	verylong mm2,
	verylong *rres
	)
{
	register long agrb;
	static verylong aa = 0;
	static verylong bb = 0;
	verylong a;
	long m1negative;
	long m2negative;

	if (!mm1)
	{
		if (mm2 != *rres)
			zcopy(mm2, rres);
		zabs(rres);
		return;
	}
	if (!mm2)
	{
		if (mm1 != *rres)
			zcopy(mm1, rres);
		zabs(rres);
		return;
	}
	if (mm1 == mm2)
	{
		if (mm1 != *rres)
			zcopy(mm1, rres);
		zabs(rres);
		return;
	}
	if (m1negative = (mm1[0] < 0))
		mm1[0] = -mm1[0];
	if (m2negative = (mm2[0] < 0))
		mm2[0] = -mm2[0];
	if (!(mm1[1]) && (mm1[0] == 1))
	{
		a = mm2;
		goto done;
	}
	if (!(mm2[1]) && (mm2[0] == 1))
	{
		a = mm1;
		goto done;
	}
	if ((agrb = mm1[0]) < mm2[0])
		agrb = mm2[0];
	zsetlength(&aa, agrb, "in zgcdeucl, locals\n");
	zsetlength(&bb, agrb, "");
	zcopy(mm1, &aa);
	zcopy(mm2, &bb);
	while (bb[1] || (bb[0] != 1))
	{
		zmod(aa, bb, &aa);
		if (!(aa[1]) && (aa[0] == 1))
		{
			a = bb;
			goto done;
		}
		zmod(bb, aa, &bb);
	}
	a = aa;
done:
	if (m1negative)
		mm1[0] = -mm1[0];
	if (m2negative)
		mm2[0] = -mm2[0];
	zcopy(a, rres);
}

void
timer_handler()
{
	time_flag = 0;
}

long 
zpollardrho(
	verylong n,
	verylong *rres,
	verylong *ccof,
	long t
	)
{
	register long i;
	register long j = 10;
	static verylong xi = 0;
	static verylong x2i = 0;
	static verylong dif = 0;
	verylong res = *rres;
	verylong cof = *ccof;
	verylong adder = &glosho[1];

	time_flag = 1;
	adder[1] = 1;
	if (ALLOCATE && !n)
	{
		zzero(rres);
		zzero(ccof);
		return (0);
	}
	if (!n[1] && (n[0] == 1))
	{
		zzero(rres);
		zzero(ccof);
		return (0);
	}
	if (n[0] < 0)
	{
		zintoz((long) (-1), rres);
		zcopy(n, ccof);
		return (1);
	}
	if (!(n[1] & 1))
	{
		zintoz((long) 2, rres);
		zrshift(n, (long) 1, ccof);
		if (((*ccof)[1] != 1) || ((*ccof)[0] != 1))
			return (1);
		return (0);
	}
	zsetlength(&xi, (i = n[0] + 1), "in zpollardrho, locals\n");
	zsetlength(&x2i, i, "");
	zsetlength(&dif, i, "");
	i = 0;
	zmkeep(n);
	zcopy(one, &dif);
	ztom(dif, &xi);
	zcopy(xi, &x2i);
	while ((!t || i < t) && time_flag)
	{
		for (; i < j && time_flag; i++)
		{
			zmontsq(xi, &xi);
			zadd(xi, adder, &xi);
			zmontsq(x2i, &x2i);
			zadd(x2i, adder, &x2i);
			zmontsq(x2i, &x2i);
			zadd(x2i, adder, &x2i);
			zsub(xi, x2i, &res);
			if (res[0] < 0)
				res[0] = -res[0];
			if ((res[1]) || (res[0] != 1))
				zmontmul(dif, res, &dif);
			else
			{
				zcopy(one, &dif);
				ztom(dif, &xi);
				zcopy(xi, &x2i);
				if ((adder[1] += 2) > RADIX)
				{
					(void)signal(SIGALRM, SIG_IGN);
					zmback();
					return (0);
				}
			}
		}
		if ((dif[1]) || (dif[0] != 1))
		{
			zgcd(n, dif, &res);
			if (((res[1] != 1) || (res[0] != 1)) && zcompare(res, n))
			{
				zdiv(n, res, &cof, &xi);
				if (xi[1] || xi[0] != 1)
				{
					*rres = res;
					*ccof = cof;
					zhalt("wrong factor in zpollardrho   BUG");
					(void)signal(SIGALRM, SIG_IGN);
					zmback();
					return (0);
				}
				if (zcompare(res, cof) > 0)
				{
					*rres = cof;
					*ccof = res;
				}
				else
				{
					*rres = res;
					*ccof = cof;
				}
				(void)signal(SIGALRM, SIG_IGN);
				zmback();
				return (i + 1);
			}
		}
		else
			zcopy(one, &dif);
		j = i + 6;
	}
	if (!time_flag)
	{
		fprintf(stderr,"timed out in zpollardrho");
		fflush(stderr);
	}
	else
	{
		(void)signal(SIGALRM, SIG_IGN);
	}
	zmback();
	return (0);
}

long 
ztridiv(
	verylong n,
	verylong *cof,
	long b1,
	long b2
	)
{

	long old_lastp = lastp;
	long old_pshift = pshift;
	static long lasttried = 0;
	if (ALLOCATE && !n)
	{
		zzero(cof);
		return (0);
	}
	if (lasttried > b1)
	{
		zpstart2();
		lasttried = 0;
	}
	while (lasttried < b1)
	{
		lasttried = zpnext();
	}
	while (lasttried <= b2)
	{
		if (!zsdiv(n, lasttried, cof))
			goto done;
		lasttried = zpnext();
	}
	done:
	if (old_lastp != lastp)
	{
		if (old_pshift<0)
			zpstart2();
		else
			zpstart();
		if (old_lastp)
			while (zpnext() != old_lastp) ;
	}
	return (lasttried);
}

long 
zfread(
	FILE *f,
	verylong *aa
	)
{
 /* return 1 if success, 0 if not */
	static char *inmem = 0;
	char *in;
	register long d = 0;
	register long anegative = 0;
	register long return_value = 1;
	verylong a = *aa;
	verylong digit = &glosho[1];

	if (!inmem)
		inmem = (char *)calloc((size_t)BUFSIZE, sizeof(char));
	if (fscanf(f, "%s", inmem) == EOF)
		return (0);
	if ((inmem[0] == '-') || (inmem[0] == '_'))
	{
		anegative = 1;
		in = (inmem + 1);
	}
	else
		in = inmem;
	zsetlength(&a, (long) (strlen(in) / LOG10RAD), "in zfread, second argument");
	a[0] = 1;
	a[1] = 0;
	while (in[d] != '\0')
	{
		if (in[d] == '\\')
		{
			if (fscanf(f, "%s", in) == EOF)
				return (0);
			d = 0;
			if (strlen(in) > LOG10RAD * (a[-1] - a[0]))
			{
			/* to avoid multiple reallocations */
				zsetlength(&a, (long) (a[0] + 3 + strlen(in) / LOG10RAD), 
				           "in zfread, second argument");
			}
		}
		else
		{
			zsmul(a, (long) 10, &a);
			digit[1] = (long) (in[d++] - '0');
                        if (digit[1] < 0) {
                                digit[1] = 0;
                                return_value = 0;
                        } else if (digit[1] > 9) {
                                return_value = 0;
                        }
			zadd(a, digit, &a);
		};
	}
	if (anegative)
		znegate(&a);
	*aa = a;
	return (return_value);
}

long 
zread(
	verylong *a
	)
{
	return (zfread(stdin, a));
}

long 
zfwrite(
	FILE *f,
	verylong a,
	long linelen,
	char *str1,
	char *str2
	)
{
	static verylong out = 0;
	static verylong ca = 0;
	static long outsize = 0;
	static long div = 0;
	static long ldiv;
	register long i;
	long sa;
	long result;
	long zeros;
	long strlen1 = strlen(str1);
	long strlen2 = strlen(str2);

	if (ALLOCATE && !a)
	{
		fprintf(f, "0");
		return (1);
	}
	if (!div)
	{
		div = 10;
		ldiv = 1;
		while (div * 10 < RADIXROOT)
		{
			div *= 10;
			ldiv++;
		}
	}
	if (linelen && linelen < ldiv)
		linelen = LINE;
	if (!out)
		out = (verylong)calloc((size_t)(outsize = SIZE << 1), (size_t)SIZEOFLONG);
	if ((sa = a[0]) < 0)
	{
		sa = -sa;
		fprintf(f, "%s-", str1);
	}
	else
		fprintf(f, "%s", str1);
	if (ldiv * outsize < LOG10RAD * sa)
	{
		free((void*)out);
		if (!(out = (verylong)calloc((size_t)(outsize = (LOG10RAD * sa) / ldiv + 2), (size_t)SIZEOFLONG))) {
			zhalt("allocation failure in zfwrite");
			return (0);
		}
	}
	zsetlength(&ca, sa, "in zfwrite, local");
	for (i = (ca[0] = sa); i; i--)
		ca[i] = a[i];
	i = -1;
	do
	{
		if ((++i) >= outsize)
		{
			if (!(out = (verylong)realloc((void*)out, (size_t)((outsize += SIZE) * SIZEOFLONG))))
			{
				zhalt("reallocation failure in zfwrite");
				return (0);
			}
		}
		out[i] = zsdiv(ca, div, &ca);
	} while (ca[1] || ca[0] != 1);
	sa = 0;
	result = out[i];
	do
	{
		sa++;
	} while (result /= 10);
	result = sa + i * ldiv;;
	fprintf(f, "%ld", out[i--]);
	for (; i >= 0; i--)
	{
		if (zeros = 10 * out[i])
		{
			while (zeros < div)
			{
				fprintf(f, "0");
				zeros *= 10;
			}
		}
		else
		{
			for (zeros = ldiv - 1; zeros; zeros--)
				fprintf(f, "0");
		}
		fprintf(f, "%ld", out[i]);
		sa += ldiv;
		if (linelen && (sa > linelen - strlen1) && i)
		{
			fprintf(f, "\\\n%s", str2);
			sa = 0;
			strlen1 = strlen2;
		}
	}
	return (result);
}

long
zfwriteln(
        FILE *f,
        verylong a
        )
{
	long ret = zfwrite(f, a, (long)LINE, "", "");

        fprintf(f,"\n");
        return (ret);
}

long 
zwrite(
	verylong a
	)
{
	return (zfwrite(stdout, a, (long)LINE, "", ""));
}

long 
zwriteln(
	verylong a
	)
{
	long ret = zfwrite(stdout, a, (long)LINE, "", "");

	printf("\n");
	return (ret);
}

char 
eulav(
	long i
	)
{
	switch (i)
	{
		case 0:
			return ('0');
		case 1:
			return ('1');
		case 2:
			return ('2');
		case 3:
			return ('3');
		case 4:
			return ('4');
		case 5:
			return ('5');
		case 6:
			return ('6');
		case 7:
			return ('7');
		case 8:
			return ('8');
		case 9:
			return ('9');
#ifdef LOWER
		case 10:
			return ('a');
		case 11:
			return ('b');
		case 12:
			return ('c');
		case 13:
			return ('d');
		case 14:
			return ('e');
		case 15:
			return ('f');
#else
		case 10:
			return ('A');
		case 11:
			return ('B');
		case 12:
			return ('C');
		case 13:
			return ('D');
		case 14:
			return ('E');
		case 15:
			return ('F');
#endif
		default:
			return ('?');
	}
}

void
zhfwrite(
	FILE *fn,
	verylong a
	)
{
	static char *b = 0;
	static bl = 0;
	static verylong aa = 0;
	register long i;
	register long cnt = 0;
	register long mb = 0;
	register long lab = 0;

	if (!a)
		fprintf(fn, "0");
	zcopy(a, &aa);
	if (aa[0] < 0)
	{
		aa[0] = -aa[0];
		fprintf(fn, "-");
	}
	if (!b)
		b = (char *)malloc((size_t)(bl = (aa[0] << 3)));
	else if (bl < (aa[0] << 3))
		b = (char *)realloc((void*)b, (size_t)((bl = (aa[0] << 3)) * sizeof(char)));
	do
	{
		b[cnt] = eulav(aa[1] & 15);
		cnt++;
		zrshift(aa, (long) 4, &aa);
	} while ((aa[1] != 0) || (aa[0] != 1));
	for (i = cnt; i--;)
	{
		fprintf(fn, "%c", b[i]);
		mb++;
		if (mb == 8)
		{
			lab++;
			if (lab == 7)
			{
				if (i)
					fprintf(fn, "\\\n");
				lab = 0;
				mb = 0;
			}
			else
			{
				mb = 0;
				fprintf(fn, " ");
			}
		}
	}

}

void
zhfwriteln(
        FILE *f,
        verylong a
        )
{
        zhfwrite(f, a);
        fprintf(f, "\n");
}

void
zhwrite(
	verylong a
	)
{
	zhfwrite(stdout, a);
}

void
zhwriteln(
	verylong a
	)
{
	zhfwrite(stdout, a);
	printf("\n");
}

long 
value(
	char c
	)
{
	switch (c)
	{
		case '0':
			return (0);
		case '1':
			return (1);
		case '2':
			return (2);
		case '3':
			return (3);
		case '4':
			return (4);
		case '5':
			return (5);
		case '6':
			return (6);
		case '7':
			return (7);
		case '8':
			return (8);
		case '9':
			return (9);
		case 'A':
		case 'a':
			return (10);
		case 'B':
		case 'b':
			return (11);
		case 'C':
		case 'c':
			return (12);
		case 'D':
		case 'd':
			return (13);
		case 'E':
		case 'e':
			return (14);
		case 'F':
		case 'f':
			return (15);
		default:
			return(0);
	}
}

void
zhfread(
	FILE *fn,
	verylong *a
	)
{
	char prev;
	char c;
	long anegative = 0;

	zintoz((long) 0, a);
	do
	{
		prev = ' ';
		do
		{
			fscanf(fn, "%c", &c);
		} while (c == '\n');
		if (((c == '-') || (c =='_')) && (!anegative))
		{
			anegative = 1;
			fscanf(fn, "%c", &c);
		}
		while (c != '\n')
		{
			prev = c;
			fscanf(fn, "%c", &c);
			if ((prev != '\\') && (prev != ' '))
			{
				zlshift(*a, (long) 4, a);
				zsadd(*a, value(prev), a);
			}
		}
		if (!anegative)
			anegative = -1;
	} while (prev == '\\');
	if (anegative > 0)
		if (((*a)[1]) || ((*a)[0] != 1))
			(*a)[0] = -((*a)[0]);
}

void
zhread(
	verylong *a
	)
{
	zhfread(stdin, a);
}

long 
zsread(
	char *str,
	verylong *aa
	)
{
	static char *inmem = 0;
	char *in;
	register long d = 0;
	register long anegative = 0;
	register long return_value = 1;
	verylong a = *aa;
	verylong digit = &glosho[1];

	if (!inmem)
		inmem = (char *)calloc((size_t)BUFSIZE, sizeof(char));
	if (sscanf(str, "%s", inmem) == EOF)
		return (0);
	if ((inmem[0] == '-') || (inmem[0] == '_'))
	{
		anegative = 1;
		in = (inmem + 1);
	}
	else
		in = inmem;
	zsetlength(&a, (long) (strlen(in) / LOG10RAD), "in zsread, second argument");
	a[0] = 1;
	a[1] = 0;
	while (in[d] != '\0')
	{
		if (in[d] == '\\')
		{
			if (sscanf(str, "%s", in) == EOF)
				return (0);
			d = 0;
			if (strlen(in) > LOG10RAD * (a[-1] - a[0]))
			{
				zsetlength(&a, (long) (a[0] + 3 + strlen(in) / LOG10RAD), 
				           "in zsread, second argument");
			}
		}
		else
		{
			zsmul(a, (long) 10, &a);
			digit[1] = (long) (in[d++] - '0');
			if (digit[1] < 0) {
				digit[1] = 0;
				return_value = 0;
			} else if (digit[1] > 9) {
				return_value = 0;
			}
			zadd(a, digit, &a);
		};
	}
	if (anegative)
		znegate(&a);
	*aa = a;
	return (return_value);
}

long 
zswrite(
	char *str,
	verylong a
	)
{
	static verylong out = 0;
	static verylong ca = 0;
	static long outsize = 0;
	static long div = 0;
	static long ldiv;
	register long i;
	register long j;
	long sa;
	long result;
	long zeros;

	if (ALLOCATE && !a)
	{
		sprintf(str, "0");
		return (1);
	}
	if (!div)
	{
		div = 10;
		ldiv = 1;
		while (div * 10 < RADIXROOT)
		{
			div *= 10;
			ldiv++;
		}
	}
	if (!out)
		out = (verylong)calloc((size_t)(outsize = SIZE << 1), (size_t)SIZEOFLONG);
	if ((sa = a[0]) < 0)
	{
		sa = -sa;
		sprintf(str, "-");
		j = 1;
	}
	else
		j = 0;
	if (ldiv * outsize < LOG10RAD * sa)
	{
		free((void*)out);
		if (!(out = (verylong)calloc((size_t)(outsize = (LOG10RAD * sa) / ldiv + 2), (size_t)SIZEOFLONG))) {
			zhalt("allocation failure in zswrite");
			return (0);
		}
	}
	zsetlength(&ca, sa, "in zswrite, local");
	for (i = (ca[0] = sa); i; i--)
		ca[i] = a[i];
	i = -1;
	do
	{
		if ((++i) >= outsize)
		{
			if (!(out = (verylong)realloc((void*)out, (size_t)((outsize += SIZE) * SIZEOFLONG))))
			{
				zhalt("reallocation failure in zswrite");
				return (0);
			}
		}
		out[i] = zsdiv(ca, div, &ca);
	} while (ca[1] || ca[0] != 1);
	sa = 0;
	result = out[i];
	do
	{
		sa++;
	} while (result /= 10);
	result = sa + i * ldiv;;
	sprintf(str + j, "%ld", out[i--]);
	j += sa;
	for (; i >= 0; i--)
	{
		if (zeros = 10 * out[i])
		{
			while (zeros < div)
			{
				sprintf(str + j++, "0");
				zeros *= 10;
			}
		}
		else
		{
			for (zeros = ldiv - 1; zeros; zeros--)
				sprintf(str + j++, "0");
		}
		sprintf(str + j, "%ld", out[i]);
		sa += ldiv;
		j = sa;
	}
	return (result);
}

long 
zbfwrite(
	FILE *fn,
	verylong a
	)
{
	register long i;
	static verylong zero = 0;

	if (!zero)
		zintoz((long) 0, &zero);
	if (!a)
	{
		if ((long)fwrite((void*)zero, sizeof(long), 2, fn) < (long)2)
			return (0);
		return (1);
	}
	i = a[0];
	if (i < 0)
		i = -i + 1;
	else
		i++;
	if ((long)fwrite((void*)a, sizeof(long), (size_t)i, fn) < i)
		return (0);
	return (1);
}

long 
zbfread(
	FILE *fn,
	verylong *a
	)
{
	register long negative = 0;
	long i;

	if (feof(fn) || ferror(fn))
		return (0);
	if ((long)fread((void*)&i, sizeof(long), 1, fn) < (long)1)
		return (0);
	if (i < 0)
	{
		i = -i;
		negative = 1;
	}
	zsetlength(a, i, "in zbfread, second argument");
	if (negative)
		(*a)[0] = -i;
	else
		(*a)[0] = i;
	if (feof(fn) || ferror(fn))
		return (0);
	if ((long)fread((void*)&((*a)[1]), sizeof(long), (size_t)i, fn) < (long)i)
		return (0);
	return (1);
}

long
zfwrite_b(
	FILE *fff,
	verylong aa,
	verylong out_base,
	long sym_out
	)
{
	register long i;
	static long sl = 0;
	static long ll = 0;
	static long *s;
	static verylong *l;
	static verylong a;
	long negative = 0;
	long need;
	zcopy(aa,&a);
	if (zscompare(a,0)<0)
	{
		negative = 1;
		znegate(&a);
	}
	if (zscompare(out_base,1) <= 0)
		zhalt("output base < 2 in zfwrite_b");
	if (zscompare(out_base,RADIX) < 0)
	{
		long b;
		long cnt;
		long si;
		if (!zscompare(a,0))
		{
			fprintf(fff,"0");
			return (0);
		}
		b = ztoint(out_base);
		need = 5 + (long)zslog(a,b);
		if (!sl)
		{
			if (!(s = (long*)calloc(need,sizeof(long))))
			{
				zhalt("allocation failure in zfwrite_b\n");
				return (0);
			}
		} else if (need > sl)
		{
			if (!(s = (long*)realloc(s,need*sizeof(long))))
			{
				zhalt("reallocation failure in zfwrite_b\n");
				return (0);
			}
		}
		sl = need;
		if (sym_out)
			i = zstosymbas(a,b,s,&need);
		else
			i = zstobas(a,b,s,&need);
		if (!i)
		{
			zhalt("zfwrite_b   BUG\n");
			return (0);
		}
		if (sym_out && negative)
			for (i=0;i<need;i++) s[i] = -s[i];
		if (sym_out || (b>16))
		{ /* in blocks */
			long blockl;
			if ((!sym_out) && negative)
				fprintf(fff,"-");
			zsadd(out_base,-1,&out_base);
			blockl = (long)(zslog(out_base,10))+1;
			zsadd(out_base,1,&out_base);
			cnt = 0;
			for (i=need-1;i>=0;i--)
			{
				si = s[i];
				if (sym_out && b != 2 )
				{
					if (si<0)
					{
						fprintf(fff,"-");
						si = -si;
					} else
						fprintf(fff," ");
					cnt ++;
				}
				switch(blockl)
				{
					case 1:
						fprintf(fff,"%01d",si);
						break;
					case 2:
						fprintf(fff,"%02d",si);
						break;
					case 3:
						fprintf(fff,"%03d",si);
						break;
					case 4:
						fprintf(fff,"%04d",si);
						break;
					case 5:
						fprintf(fff,"%05d",si);
						break;
					case 6:
						fprintf(fff,"%06d",si);
						break;
					case 7:
						fprintf(fff,"%07d",si);
						break;
					case 8:
						fprintf(fff,"%08d",si);
						break;
					case 9:
						fprintf(fff,"%09d",si);
						break;
					case 10:
						fprintf(fff,"%010d",si);
						break;
				}
				cnt += (blockl+1);
				if (cnt > LINE - blockl)
				{
					if (i)
					{
						fprintf(fff,"\\\n");
						cnt = 0;
					} else
						fprintf(fff,"\n");
				} else if (i)
					fprintf(fff," ");
			}

		} else 
		{ /* without spaces */
			if (negative)
			{
				fprintf(fff,"-");
				cnt = 1;
			} else 
				cnt = 0;
			for (i=need-1;i>=0;i--)
			{
				fprintf(fff,"%c",eulav((s[i])&15));
				cnt ++;
				if (cnt == LINE)
				{
					if (i)
					{
						fprintf(fff,"\\\n");
						cnt = 0;
					} else
						fprintf(fff,"\n");
				} else if (b==16)
				{
					if (i)
					{
						if (cnt==56)
						{
							fprintf(fff,"\\\n");
							cnt = 0;
						} else if ((cnt&7)==0)
							fprintf(fff," ");
					} else 
						return (need);
				}
			}
		}
		return (need);
	} else
	{
		long b;
		long cnt;
		long blocks;
		if (!zscompare(a,0))
		{
			fprintf(fff,"0");
			return (0);
		}
		need = 5 + (long)zlog(a,out_base);
		if (!ll)
		{
			if (!(l = (verylong*)calloc(need,sizeof(verylong))))
			{
				zhalt("allocation failure in zfwrite_b\n");
				return (0);
			}
			for (i=0;i<need;i++)
				l[i] = 0;
		} else if (need > ll)
		{
			if (!(l = (verylong*)realloc(l,need*sizeof(verylong))))
			{
				zhalt("reallocation failure in zfwrite_b\n");
				return (0);
			}
			for (i=ll;i<need;i++)
				l[i] = 0;
		}
		ll = need;
		if (sym_out)
			i = ztosymbas(a,out_base,l,&need);
		else
			i = ztobas(a,out_base,l,&need);
		if (!i)
		{
			zhalt("zfwrite_b   BUG\n");
			return (0);
		}
		if (sym_out && negative)
			for (i=0;i<need;i++) znegate(&(l[i]));
		if ((!sym_out) && negative)
			fprintf(fff,"-");
		zsadd(out_base,-1,&out_base);
		blocks = LINE / ((long)(zslog(out_base,10))+1);
		zsadd(out_base,1,&out_base);
		if (blocks<=1)
		{
			for (i=need-1;i>=0;i--)
			{
				zfwrite(fff,l[i],LINE,"","");
				if (i)
					fprintf(fff," \\\n");
			}
		} else
		{
			cnt = 0;
			for (i=need-1;i>=0;i--)
			{
				zfwrite(fff,l[i],LINE,"","");
				cnt ++;
				if (i)
				{
					fprintf(fff," ");
					if (cnt == blocks)
					{
						fprintf(fff,"\\\n");
						cnt = 0;
					}
				}
			}
		}
		return (need);
	}
}

long
zfwriteln_b(
        FILE *fff,
        verylong aa,
        verylong out_base,
        long sym_out
        )
{
	long result = zfwrite_b(fff,aa,out_base,sym_out);
	fprintf(fff,"\n");
	return(result);
}

long
zfread_b(
	FILE *fff,
	verylong *a,
	verylong in_base,
	long sym_in
	)
{
	static verylong digit = 0;
	long negative = 0;
	long reach = 0;
	char y;
	if ((zscompare(in_base,16) > 0) || (sym_in))
	{
		reach = 1;
		zfread(fff,a);
		if ((zscompare(*a,0)<0) && (!sym_in))
		{
			znegate(a);
			negative = 1;
		}
		y=(char)fgetc(fff);
		while (y != '\n')
		{
			while (y == ' ')
				y=(char)fgetc(fff);
			if (y != '\\')
				ungetc((int)y,fff);
			zfread(fff,&digit);
			if (!sym_in) zabs(&digit);
			zmulin(in_base,a);
			zadd(*a,digit,a);
			y=(char)fgetc(fff);
		}
		if (sym_in) return (1);
	} else
	{
		register long short_in_base;
		register long non_hex;
		register long aftersl = 0;;
		short_in_base = ztoint(in_base);
		non_hex = (short_in_base != 16);
		while ((y=(char)fgetc(fff)) == '\n');
		if ((y == '-') || (y == '_'))
		{
			negative = 1;
			y=(char)fgetc(fff);
		}
		zintoz(0,a);
		while (strrchr("00123456789AaBbCcDdEeFf\\ ",y) != NULL)
		{
			if (non_hex && (y == ' ') && (!aftersl))
				goto done;
			if ((y != ' ') && (y != '\\'))
			{
				zsmul(*a,short_in_base,a);
				zsadd(*a,value(y),a);
				aftersl = 0;
			}
			if (y == '\\')
			{
				do
					y=(char)fgetc(fff);
				while (y != '\n');
				aftersl = 1;
			}
			y=(char)fgetc(fff);
		}
		if (y != '\n')
			ungetc((int)y,fff);
		else
			reach = 1;
	}
done:
	if (negative)
		znegate(a);
	return reach;
}

long 
zsign(
	verylong a
	)
{
	if (ALLOCATE && !a)
	{
		return (0);
	}
	if (a[0] < 0)
		return (-1);
	if (a[0] > 1)
		return (1);
	if (a[1])
		return (1);
	return (0);
}

void
zabs(
	verylong *pa
	)
{
	verylong a = *pa;

	if (!a)
		return;
	if (a[0] < 0)
		a[0] = (-a[0]);
}

long 
z2logs(
	long a
	)
{
	long i = 0;

	if (a < 0)
		a = -a;
	while (a>=256)
		i += 8, a >>= 8;
	if (a >=16)
		i += 4, a >>= 4;
	if (a >= 4)
		i += 2, a >>= 2;
	if (a >= 2)
		i += 2;
	else if (a >= 1)
		i++;
	return (i);
}

long 
z2log(
	verylong a
	)
{
	register long la;

	if (!a)
		return (0);
	la = (a[0] > 0 ? a[0] : -a[0]);
	return ( NBITS * (la - 1) + z2logs(a[la]) );
}

double 
zln(
	verylong a
	)
{
/*	extern double log(); */
	register long sa;

	if ((!a) || (a[0] <= 0))
	{
		zhalt("non-positive argument in zln");
		return (0.0);
	}
	if ((sa = a[0]) == 1)
		return (log((double) (a[1])));
	return ((sa - 2) * NBITS * log((double) 2) +
		log(((double)RADIX) * a[sa] + a[sa - 1]));
}

double 
zlog(
	verylong a,
	verylong b
	)
{
	return (zln(a) / zln(b));
}

double 
zslog(
	verylong a,
	long b
	)
{
/*	extern double log(); */

	return (zln(a) / log((double)b));
}

double 
zdlog(
	verylong a,
	double b
	)
{
/*	extern double log(); */

	return (zln(a) / log(b));
}

long
zscompare(
	verylong a,
	long b
	)
{
	static verylong c = 0;

	zintoz(b, &c);
	return (zcompare(a, c));
}

void
zswap(
	verylong *a,
	verylong *b
	)
{
	register verylong c;

	c = *a;
	*a = *b;
	*b = c;
}

long
ziszero(
	verylong a
	)
{
	if (!a) return (1);
	if (a[1]) return (0);
	if (a[0]==1) return (1);
	return (0);
}

long
zodd(
	verylong a
	)
{
	if (!a) return (0);
	return (a[1]&1);
}

long
zweights(
	long a
	)
{
	register long i;
	register long res = 0;
	for (i=NBITS;i;i--) {
		if (a & 1) res ++;
		a >>= 1;
	}
	return (res);
}

long
zweight(
        verylong a
        )
{
	register long i;
	register long res = 0;
	if (!a) return (0);
	i = a[0];
	if (i<0) i = -i;
	for (;i;i--)
		res += zweights(a[i]);
	return (res);
}

void
znot(
	verylong a,
	verylong *b
	)
{
	register long sa, i, m;
	if (ziszero(a)) {
		zintoz(1,b);
		return;
	}
	zcopy(a,b);
	sa = a[0];
        if (sa<0) sa = -sa;
        for (i=sa-1;i;i--)
		(*b)[i] = ((~((*b)[i])) & RADIXM);
	m = (RADIX >> 1);
	i = RADIXM;
	while (!(((*b)[sa])&m)) {
		m >>= 1;
		i >>= 1;
	}
	(*b)[sa] = ((~((*b)[sa])) & i);
	while ((sa>1) && (!((*b)[sa])))
		sa --;
	if (((*b)[0] < 0) && ((*b)[1])) (*b)[0] = -sa;
	else (*b)[0] = sa;
}

void
zand(
	verylong a,
	verylong b,
	verylong *cc
	)

{
	verylong c = *cc;
	register long sa;
	register long sb;
	register long sm;
	if (ziszero(a) || ziszero(b)) {
		zzero(cc);
		return;
	}
	sa = a[0];
	if (sa < 0) sa = -sa;
	sb = b[0];
	if (sb < 0) sb = -sb;
	sm = (sa > sb ? sb : sa );
	zsetlength(&c, sm, "in zand, third argument");
	if (a == *cc) a = c;
	if (b == *cc) b = c;
	*cc = c;
	for (sa = 1; sa <= sm; sa ++) 
		c[sa] = a[sa] & b[sa];
	while ((sm > 1) && (!(c[sm])))
		sm --;
	c[0] = sm;
}

void
zxor(
	verylong a,
	verylong b,
	verylong *cc
	)
{
        verylong c = *cc;
        register long sa;
        register long sb;
        register long sm;
        register long la;
        register long i;
	if (ziszero(a)) {
		zcopy(b,cc);
		zabs(cc);
		return;
	}
	if (ziszero(b)) {
		zcopy(a,cc);
		zabs(cc);
		return;
	}
        sa = a[0];
        if (sa < 0) sa = -sa;
        sb = b[0];
        if (sb < 0) sb = -sb;
	if (sa > sb) {
		la = sa;
		sm = sb;
	} else {
		la = sb;
		sm = sa;
	}
        zsetlength(&c, la, "in zxor, third argument");
        if (a == *cc) a = c;
        if (b == *cc) b = c;
        *cc = c;
        for (i = 1; i <= sm; i ++)
                c[i] = a[i] ^ b[i];
	if (sa > sb)
		for (;i <= la; i++) c[i] = a[i];
	else
		for (;i <= la; i++) c[i] = b[i];
        while ((la > 1) && (!(c[la])))
                la --;
        c[0] = la;
}

void
zor(
	verylong a,
	verylong b,
        verylong *cc
        )
{
        verylong c = *cc;
        register long sa;
        register long sb;
        register long sm;
        register long la;
        register long i;
	if (ziszero(a)) {
		zcopy(b,cc);
		zabs(cc);
		return;
	}
	if (ziszero(b)) {
		zcopy(a,cc);
		zabs(cc);
		return;
	}
        sa = a[0];
        if (sa < 0) sa = -sa;
        sb = b[0];
        if (sb < 0) sb = -sb;
        if (sa > sb) {
                la = sa;
                sm = sb;
        } else {
                la = sb;
                sm = sa;
        }
        zsetlength(&c, la, "in zor, third argument");
        if (a == *cc) a = c;
        if (b == *cc) b = c;
        *cc = c;
        for (i = 1; i <= sm; i ++)
                c[i] = a[i] | b[i];
        if (sa > sb)
                for (;i <= la; i++) c[i] = a[i];
        else
                for (;i <= la; i++) c[i] = b[i];
        c[0] = la;
}

long
zslowbits(
	verylong a,
	long b
	)
{
	if (ziszero(a) || (b<=0))
		return(0);
	if (b>NBITS)
		b = NBITS;
	return (a[1]&((1<<b)-1));
}

void
zlowbits(
	verylong a,
	long b,
        verylong *cc
        )
{
        verylong c = *cc;
	register long bl;
	register long wh;
	register long sa;
	if (ziszero(a) || (b<=0)) {
		zintoz(0,cc);
		return;
	}
        bl = (b/NBITS);
        wh = b - NBITS*bl;
        bl ++;
	sa = a[0];
	if (sa < 0) sa = -sa;
	if (sa < bl) {
		zcopy(a,cc);
		zabs(cc);
		return;
	}
        zsetlength(&c, bl, "in zlowbits, third argument");
        if (a == *cc) a = c;
        *cc = c;
	for (sa=1;sa<bl;sa++)
		c[sa] = a[sa];
	c[bl] = a[bl]&((1<<wh)-1);
	while ((bl>1) && (!c[bl]))
		bl --;
	c[0] = bl;
}

long
zshighbits(
	verylong a,
	long b
	)
{
	long l;
	static verylong aa=0;
        if (ziszero(a) || (b<0))
                return(0);
        if (b>NBITS)
                b = NBITS;
	zcopy(a,&aa);
	zabs(&aa);
	l=z2log(aa);
	if (l > b)
		zrshift(aa,l-b,&aa);
	return (ztoint(aa));
}

void
zhighbits(
	verylong a,
	long b,
	verylong *c
	)
{
	long l;
        if (ziszero(a) || (b<0)) {
		zzero(c);
		return;
	}
	zcopy(a,c);
	zabs(c);
	l = z2log(*c);
	if (l > b)
		zrshift(*c,l-b,c);
}

void
zcat(
	verylong a,
	verylong b,
	verylong *c
	)
{
	register long signa = 0;
	register long signb = 0;
	static verylong aux = 0; 
	if (ziszero(a)) {
		zcopy(b,c);
		return;
	}
	if (ziszero(b)) {
		zlshift(a,1,c);
		return;
	}
	if (a[0] < 0) {
		signa = 1;
		a[0] = -a[0];
	}
	if (b[0] < 0) {
		signb = 1;
		b[0] = -b[0];
	}
	zlshift(a,z2log(b),&aux);
	zadd(aux,b,c);
	if (signa && (a != *c)) a[0] = -a[0];
	if (signb && (b != *c)) b[0] = -b[0];
}

void
zgetbits(
	verylong a,
	long b,
	long p,
	verylong *c
	)
{
	register long bl;
	register long wh;
	if (b<=0) {
		zzero(c);
		return;
	}
	if (p<0) p = -p;
	zrshift(a,p,c);
	zabs(c);
	if (b>=z2log(*c)) return;
	bl = (b-1)/NBITS +1;
	for (wh=(*c)[0];wh>b;wh--) (*c)[wh] = 0;
	(*c)[wh] &= ((1<<(b%NBITS))-1);
	while ((wh>1) && (!((*c)[wh])))
		wh --;
	(*c)[0] = wh;
}

long
zbit(
	verylong a,
	long p
	)
{
        register long bl;
        register long wh;
        register long sa;
	if (ziszero(a)) return (0);
	if (p < 0) p = -p;
	bl = (p/NBITS);
        wh = 1 << (p - NBITS*bl);
	bl ++;
        sa = a[0];
        if (sa < 0) sa = -sa;
        if (sa < bl) return (0);
	if (a[bl] & wh) return (1);
	return (0);
}

long
zsetbit(
        verylong *a,
        long b
        )
{
	register long bl;
	register long wh;
	register long sa;
	if (b<0) b = -b;
	if (!(*a)) {
		zintoz(1,a);
		zlshift(*a,b,a);
		return (0);
	}
	bl = (b/NBITS);
	wh = 1 << (b - NBITS*bl);
	bl ++;
	sa = (*a)[0];
	if (sa<0) sa = -sa;
	if (sa >= bl) {
		sa = (*a)[bl] & wh;
		(*a)[bl] |= wh;
		if (sa) return (1);
		return (0);
	} else {
		zsetlength(a,bl,"in zsetbit, first argument");
		sa ++;
		for (;sa<=bl;sa++) (*a)[sa]=0;
		if ((*a)[0] < 0)
			(*a)[0] = -bl;
		else (*a)[0] = bl;
		(*a)[bl] |= wh;
		return (0);
	}
}

long
zswitchbit(
        verylong *a,
        long p
        )
{
        register long bl;
        register long wh;
        register long sa;
        if (p < 0) p = -p;
        if (ziszero(*a)) {
		zintoz(1,a);
		zlshift(*a,p,a);
		return (0);
	}
        bl = (p/NBITS);
        wh = 1 << (p - NBITS*bl);
        bl ++;
        sa = (*a)[0];
        if (sa < 0) sa = -sa;
        if ((sa < bl) || (!((*a)[bl] & wh))) {
		zsetbit(a,p);
		return (0);
	}
	(*a)[bl] ^= wh;
        while ((sa>1) && (!(*a)[sa]))
		sa --;
	if ((*a)[0] > 0) (*a)[0] = sa;
	else (*a)[0] = -sa;
        return (1);
}

long
zreverses(
	long aa
	)
{
	register long a = aa;
	register long result = 1;
	if (!a) return (0);
	if (a < 0) a = -a;
	while (!(a&1)) a >>= 1;
	a >>= 1;
	while (a) {
		result <<= 1;
		if (a & 1) result ++;
		a >>= 1;
	}
	return (result);
}

void
zreverse(
	verylong a,
	verylong *b
	)
{
	register long sa;
	register long i;
	register long aux;
	register long sh;
	register long ma;
	register long sr;
	static verylong result = 0;
	if (ziszero(a)) {
		zzero(b);
		return;
	}
	sa = a[0];
	if (sa < 0) sa = -sa;
	zsetlength(&result,sa,"in zreverse, local");
	result[sa] = 0;
	if (a[sa]&(RADIX>>1)) sr = sa;
	else sr = sa-1;
	for (i=1;i<=sr;i++) {
		aux = a[sr+1-i];
		if (!aux) 
			result[i] = 0;
		else {
			ma = (RADIX>>1);
			sh = 0;
			while (!(aux&ma)) {
				ma >>= 1;
				sh ++;
			}
			result[i] = (zreverses(aux) << sh);
		}
	}
	while ((sr>1) && (!(result[sr])))
		sr --;
	result[0] = sr;
	if (!(a[sa]&(RADIX>>1))) {
		zlshift(result,z2logs(a[sa]),&result);
		zsadd(result,zreverses(a[sa]),&result);
	}
	zcopy(result,b);
}

long 
zjacobi(
	verylong a,
	verylong n
	)
{
	long unit;
	long i;
	verylong temp = 0;

	if (!n || zsign(n) <= 0)
	{
		zhalt("non-positive second argument in zjacobi");
		return (0);
	}
	if (!a || !zsign(a))
	{
		return (0);
	}
	else if (a[0] == 1 && a[1] == 1)
	{
		return (1);
	}
	else if (n[0] == 1 && n[1] == 1)
	{
		return (1);
	}
	else if ((!(a[1] & 1)) && (!(n[1] & 1)))
	{
		return (0);
	}
	else if (!(n[1] & 1))
	{
		zcopy(n, &temp);
		i = zmakeodd(&temp);
		i = (i > 3 ? 8 : 1 << i);
		if (a[1] % i != 1)
			return (-1);
		return (zjacobi(a, temp));
	}
	else if (a[0] < 0)
	{
		zcopy(a, &temp);
		temp[0] = -temp[0];
	/* check (n-1)/2 parity */
		unit = (n[1] & 2 ? -1 : 1);
		return (zjacobi(temp, n) * unit);
	}
	else if (!(a[1] & 1))
	{
	/* check (n*n-1)/8 parity */
		i = n[1] & 7;
		unit = (i == 3 || i == 5 ? -1 : 1);
		zcopy(a, &temp);
		zrshift(temp, (long) 1, &temp);
		return (zjacobi(temp, n) * unit);
	}
	else
	{
	/* check ((a-1)*(n-1))/4 parity */
		unit = ((a[1] & 2) && (n[1] & 2) ? -1 : 1);
		zmod(n, a, &temp);	/* temp = n mod a */
		return (zjacobi(temp, a) * unit);
	}
}

static void
zrootnewton(
	verylong a,
	verylong *b,
	long n
	)
{
	verylong c = 0;
	verylong d = 0;

	zintoz((long) 1, b);
	zlshift(*b, (z2log(a) + n - 1) / n, b);
	do
	{
		zsexp(*b, n - 1, &c);
		zdiv(a, c, &c, &d);
		zsub(*b, c, &c);
		zsdiv(c, n, &c);
		zsub(*b, c, b);
	} while (zsign(c) > 0);
	zsexp(*b, n, &c);
	if (zcompare(c, a) > 0)
		zsadd(*b, (long) (-1), b);
}

void
zroot(
	verylong a_in,
	long n,
	verylong *b
	)
{
	static verylong a = 0;
	long signa = zsign(a_in);

	zcopy(a_in,&a);
	if ((signa < 0) && (n & 1))
		a[0] = -a[0];
	if ((signa > 0 && n > 1) || (signa < 0 && (n & 1)))
		zrootnewton(a, b, n);
	else if (n == 1 || signa == 0 && n > 0)
		zcopy(a, b);
	else if (n == 0)
	{
		zhalt("dth root with d=0 in zroot");
		return;
	}
	else if ((signa < 0) && (!(n & 1)))
	{
		zhalt("dth root with even d of negative number in zroot");
		return;
	}
	else if (signa == 0 && n < 0)
	{
		zhalt("dth root with d<0 of zero in zroot");
		return;
	}
	else
		zintoz((long) 0, b);
	if (signa < 0)
	{
		if (((*b)[1]) || ((*b)[0] > 1))
			(*b)[0] = -(*b)[0];
	}
}


long 
zjacobis(
	long m,
	long n
	)
{
	register long m2;
	register long m1;
	register long jcbi = 1;

	if (n < 0)
		m2 = -n;
	else
		m2 = n;
	if ((m2 & 1) == 0)
	{
		zhalt("even second argument in zjacobis");
		return (0);
	}

	m1 = (m >= 0) ? (m % m2) : (m2 - ((-m) % m2));

	while (1)
	{
		switch (m1 & 15)
		{
			case 0:
				if (m1 == 0)
					return ((m2 == 1) ? jcbi : 0);

				m1 >>= 4;	/* Divide numerator by 16 */
				continue;	/* continue while loop */

			case 8:
				m1 >>= 2;	/* Divide numerator by 4 */
				continue;	/* continue while loop */

			case 4:
				m1 >>= 2;	/* Divide numerator by 4 */
				break;

			case 1:
			case 5:
			case 9:
			case 13:
				break;

			case 2:
			case 10:
				m1 >>= 1;
				if ((m2 & 7) == 3 || (m2 & 7) == 5)
					jcbi = -jcbi;
				break;

			case 12:
				m1 >>= 2;	/* Divide numerator by 4, */
			/* fall through to 3 mod 4 cases */
			case 3:
			case 7:
			case 11:
			case 15:
				if ((m2 & 2) != 0)	/* negate if  m1 == m2
							 * == 3 mod 4 */
					jcbi = -jcbi;
				break;

			case 6:
			case 14:
				m1 >>= 1;
				if ((m2 & 4) != 0)
					jcbi = -jcbi;
				break;
		}	/* end switch */

		{	/* m1 and m2 are odd; invert them */
		/* First divide m2 by 4, mod m1 */
			register long m3;
			m3 = ((((m2 + m1) & 3) == 0) ? m2 + m1 : m2 - m1) >> 2;
			if (m3 >= m1)
				m3 %= m1;
			m2 = m1;
			m1 = m3;
		}
	}	/* end while(1) */
}

long 
zprobprime(
	verylong a,
	long number_of_tests
	)
{
	long result;
	static verylong cofactor_a = 0;

	if (!a || a[0] < 0)
		return (0);
	if ((a[0]==1) && (a[1]==2))
		return (1);
	if (!(a[1] & 1))
		return (0);
	if (a[0] == 1)
	{
		if ((a[1] == 3) || (a[1] == 5) || (a[1] == 7) || (a[1] == 11) )
			return (1);
		if (a[1] < 13) return (0);
		result = zsqrts(a[1]);
		if (ztridiv(a, &cofactor_a, (long) 3, result) <= result)
			return (0);
		return (1);
	}
	result = a[0] * 5 * NBITS;
	if (ztridiv(a, &cofactor_a, (long) 3, result) <= result)
		return (0);
	if (zcomposite(&a, (long) 1, (long) 2))
		return (0);
	result = !zmcomposite(a, number_of_tests);
 /*
  * if (!result) { printf("composite, but pseudo prime to the base 2:\n");
  * zwriteln(a); }
  */
	return (result);
}

void
zrandoml(
	long bitlength,
	verylong *a,
	long (*generator) (long)
	)
{
	register long i;
	register long neg = 0;

	if (bitlength == 0)
	{
		zintoz((long) 0, a);
		return;
	}
	if (bitlength < 0)
	{
		bitlength = -bitlength;
		neg = 1;
	}
	if (bitlength == 1)
	{
		zintoz((long) 1, a);
		return;
	}
	zintoz((long) 1, a);
	zlshift(*a, bitlength - 1, a);
	for (i = 1; i < (*a)[0]; i++)
		(*a)[i] = (*generator)((long)RADIX);
	(*a)[i] += (*generator)((*a)[i]);
	if (neg)
		(*a)[0] = -(*a)[0];
}

void
zrandomb(
	verylong bnd,
	verylong *a,
	long (*generator) (long)
	)
{
	static verylong big = 0;
	static verylong bnd_minus_one = 0;

	if (zscompare(bnd, (long) 1) <= 0)
	{
		zintoz((long) 0, a);
		return;
	}
	if (zscompare(bnd, (long) 2) == 0)
	{
		zintoz((long) 1, a);
		return;
	}
	zrandoml(z2log(bnd) + NBITS, &big, generator);
	zsadd(bnd, (long) (-1), &bnd_minus_one);
	zmod(big, bnd_minus_one, a);
	zsadd(*a, (long) 1, a);
}

long 
zrandomprime(
	long bitlength,
	long nbtests,
	verylong *a,
	long (*generator) (long)
	)
{
	static verylong t = 0;
	static verylong bnd = 0;
	register long adder = 2;
	long three_mod_four = 0;

	if (bitlength < 0)
	{
		bitlength = - bitlength;
		three_mod_four = 1;
		adder = 4;
	}
	if (bitlength < 2)
		return (0);
	if (bitlength == 2)
	{
		if (three_mod_four)
			zintoz(3,a);
		else
		{
			zintoz((*generator)((long) 2), a);
			zsadd(*a, (long) 2, a);
		}
		return (1);
	}
	zintoz((long) 1, &bnd);
	zlshift(bnd, bitlength, &bnd);
	while (1)
	{
		if (three_mod_four)
		{
			zrandoml(bitlength - 2, &t,generator);
			zlshift(t, (long) 2, a);
			zsadd(*a, (long) 3, a);
		}
		else
		{
			zrandoml(bitlength - 1, &t,generator);
			zlshift(t, (long) 1, a);
			zsadd(*a, (long) 1, a);
		}
		while (zcompare(*a, bnd) < 0)
		{
			if (zprobprime(*a, nbtests))
				return (1);
			zsadd(*a, adder, a);
		}
	}
}

long
zrandomqprime(
	long lp,
	long lq,
	long nbtests,
	verylong *p,
	verylong *q,
	verylong *frac,
	long (*generator) (long)
	)
{
	register long oddq = 1;
	register long cnt = 2 * lp;
	static verylong tpsizem = 0;
	static verylong tpsize = 0;
	static verylong lown = 0;
	static verylong upn = 0;
	static verylong twoq = 0;

	if (lq < 0)
	{
		lq = z2log(*q);
		if ((!lq) || ((*q)[0] < 0))
		{
			zhalt("wrong q in zrandomqprime");
			return (0);
		}
		if (((*q)[0] == 1) && ((*q)[1] == 1))
		{
			if (!(zrandomprime(lp, nbtests, p, generator)))
				return (0);
			zcopy(*p, frac);
			return (1);
		}
		if (-lq >= lp)
			return (0);
	}
	else if (!(zrandomprime(lq, nbtests, q, generator)))
		return (0);

	zintoz((long) 1, &tpsizem);
	zlshift(tpsizem, lp - 1, &tpsizem);
	zsadd(tpsizem, (long) (-1), &tpsizem);
	zintoz((long) 1, &tpsize);
	zlshift(tpsize, lp, &tpsize);
	zsadd(tpsize, (long) (-1), &tpsize);

	if ((*q)[1] & 1)
		zlshift(*q, (long) 1, &twoq);
	else
	{
		oddq = 0;
		zcopy(*q, &twoq);
	}
	zdiv(tpsizem, twoq, &lown, frac);
	zdiv(tpsize, twoq, &upn, frac);
	zsub(upn, lown, &upn);
	do
	{
		cnt--;
		if (!cnt)
		{
			if (zscompare(upn, lp) <= 0)
				return (0);
		}
		zrandomb(upn, p, generator);
		zadd(lown, *p, frac);
		zmul(twoq, *frac, p);
		zsadd(*p, (long) 1, p);
	} while (!zprobprime(*p, nbtests));
	if (oddq)
		zlshift(*frac, (long) 1, frac);
	return (1);
}

long
zrandomfprime(
	long length,
	long nbtests,
	verylong frac,
	verylong *p,
	verylong *q,
	long (*generator) (long)
	)
{
	register long pp;
	register long qmp;
	register long bound;
	register long small_frac = 0;

	static verylong aux = 0;

	if ((zscompare(frac,2)<0) || (frac[1]&1))
                return (0);

	if (length < (NBITS>>1))
	{
        	do
        	{
                	if (!zrandomprime(length,nbtests,q,generator))
                        	return (0);
                	zmul(*q,frac,p);
                	zsadd(*p,1,p);
        	} while (!zprobprime(*p,nbtests));
        	return (1);
	}

	if (zscompare(frac,RADIXROOT) <= 0)
		small_frac = ztoint(frac);
	if ((bound = (z2log(frac)+length)*5) > RADIXROOT)
		bound = RADIXROOT;
	for (;;)
	{
		zrandoml(length-1,q,generator);
		zlshift(*q,1,q);
		zsadd(*q,1,q);
		zpstart();
		while ((pp=zpnext()) <= bound) 
		{
			if (!(qmp = zsdiv(*q,pp,&aux)))
				goto next_try;
			if (small_frac)
				qmp *= small_frac;
			else
				qmp *= zsdiv(frac,pp,&aux);
			if (!((qmp+1)%pp))
				goto next_try;
		}
		if (zcomposite(q,1,2))
			goto next_try;
		zmul(*q,frac,p);
		zsadd(*p,1,p);
		if (zcomposite(p,1,2) || zmcomposite(*q,nbtests)
			|| zmcomposite(*p,nbtests))
			goto next_try;
		return (1);
		next_try:;
	}
}

long
zrandomgprime(
	long lq,
	long nbtests,
	long small,
	verylong *p,
	verylong *q,
	verylong *g,
	long (*generator) (long)
	)
{
	static verylong montone = 0;
	static verylong aux = 0;
	zintoz(2,&aux);
	if (!zrandomfprime(lq,nbtests,aux,p,q,generator))
		return 0;
	zmkeep(*p);
        zintoz((long)1, &montone);
        ztom(montone,&montone);
	if (small)
		zcopy(montone,g);
        for (;;)
	{
                if (small)
			zaddmod(*g,montone,*p,g);
		else
			zrandomb(*p,g,generator);
                zmontsq(*g,&aux);
                if (zcompare(aux,montone))
		{
                        zmontexp_m_ary(*g,*q,&aux,0);
                        if (zcompare(aux,montone))
			{
				zmtoz(*g,g);
				zmback();
				return 1;
			}
                }
        }
}

#define	MAXM		3000000
#define	PHASE1_BATCH	1
#define	MAXTEP		31
#define	MAXPWRP		12
#define	MAXRP		(1 << MAXPWRP)
#define	MINBOUND	100

static verylong ecm_tex[MAXTEP];
static verylong ecm_tey[MAXTEP];
static verylong ecm_coef[MAXRP];
static verylong ecm_power[MAXPWRP];
static verylong ecm_eval[MAXPWRP];

static long 
ph1set(
	verylong n,
	verylong *rap,
	verylong *alpha,
	verylong *mu,
	verylong *x,
	verylong *f
	)
{
 /* Set up elliptic curve */
 /* if 0, success, */
 /* if 1, n factored, factor in f */
	static verylong r = 0;
	static verylong s = 0;
	static verylong t1 = 0;
	static verylong t2 = 0;
	static verylong t3 = 0;
	static verylong u = 0;
	static verylong u2 = 0;
	static verylong u3 = 0;
	static verylong u4 = 0;

	zrandomb(n,&s,zrandom);
	zsqmod(s,n,&r);
	zsmulmod(s,6,n,&s);
	zsadd(r,6,&r);
	if (zinv(r, n, &t1))
	{
		zcopy(t1, f);
		return (1);
	}
	zmulmod(s, t1, n, &u);
	zsqmod(u, n, &u2);
	zmulmod(u, u2, n, &u3);
	zsqmod(u2, n, &u4);
	zsmul(u4, 3, &t1);
	zmod(t1, n, &t1);
	zsmul(u2, 6, &t2);
	zmod(t2, n, &t2);
	zaddmod(t1, t2, n, &t1);
	zone(&t2);
	zsubmod(t2, t1, n, &t1);
	zsmul(u3, 4, &t2);
	zmod(t2, n, &t2);
	if (zinv(t2, n, &t3))
	{
		zcopy(t3, f);
		return (1);
	}
	zmulmod(t1, t3, n, alpha);
	zintoz(2, &t1);
	zaddmod(*alpha, t1, n, &t1);
	zintoz(4, &t2);
	if (zinv(t2, n, &t3))
	{
		zcopy(t3, f);
		return (1);
	}
	zmulmod(t1, t3, n, rap);
	zsmul(u, 3, &t1);
	zmod(t1, n, &t1);
	zmulmod(t1, t3, n, x);
	zintoz(3, &t1);
	if (zinv(t1, n, &t2))
	{
		zcopy(t2, f);
		return (1);
	}
	zmulmod(*alpha, t2, n, &t1);
	zsubmod(n, t1, n, mu);
	ztom(*rap, rap);
	ztom(*alpha, alpha);
	ztom(*mu, mu);
	ztom(*x, x);
	return (0);
}

static long 
ph1exp(
	verylong n,
	verylong *x,
	verylong e,
	verylong rap,
	verylong *f
	)
{
 /* x = x^e on rap mod n */
 /* if 0, success */
 /* if 1, n factored, factor in f */
	static verylong x1 = 0;
	static verylong xn = 0;
	static verylong xn1 = 0;
	static verylong zzn = 0;
	static verylong zn1 = 0;
	static verylong s = 0;
	static verylong s1 = 0;
	static verylong d = 0;
	static verylong d1 = 0;
	static verylong t1 = 0;
	static verylong t2 = 0;
	static verylong t3 = 0;
	static verylong t4 = 0;
	static verylong longmask = 0;
	register long mask;
	register long maskcnt;
	register long eatmaskcnt;
	register long first = 1;

	if (zscompare(e,1) <= 0)
		return (0);
	zcopy(*x, &x1);
	zcopy(x1, &xn1);
	zcopy(zr, &zn1);
	zaddmod(xn1, zn1, n, &s1);
	zsubmod(xn1, zn1, n, &d1);
	zmontsq(s1, &t1);
	zmontsq(d1, &t2);
	zmontmul(t1, t2, &xn);
	zsubmod(t1, t2, n, &t3);
	zmontmul(rap, t3, &t4);
	zaddmod(t2, t4, n, &t4);
	zmontmul(t3, t4, &zzn);
	zintoz(1,&longmask);
	zlshift(longmask,z2log(e),&longmask);
	zsub(longmask,e,&e);
	maskcnt = longmask[0];
	mask = longmask[maskcnt] >> 1;
	if (!mask)
	{
		mask = (RADIX >> 2);
		maskcnt --;
	}
	else
	{
		mask >>= 1;
		if (!mask)
		{
			mask = (RADIX >> 1);
			maskcnt --;
		}
	}
	for (;maskcnt >0; maskcnt --)
	{
		if (maskcnt > e[0])
			eatmaskcnt = 0;
		else
			eatmaskcnt = e[maskcnt];
		while (mask)
		{
		  zaddmod(xn, zzn, n, &s);
		  zsubmod(xn, zzn, n, &d);
		  zaddmod(xn1, zn1, n, &s1);
		  zsubmod(xn1, zn1, n, &d1);
		  if (mask&eatmaskcnt)
		  {
			if ((maskcnt ) || ((!maskcnt) && (!(mask & 1))))
			{
				if (first)
				{
					zcopy(xn, &xn1);
					zcopy(zzn, &zn1);
				}
				else
				{
					zmontsq(s1, &t1);
					zmontsq(d1, &t2);
					zmontmul(t1, t2, &xn1);
					zsubmod(t1, t2, n, &t3);
					zmontmul(rap, t3, &t4);
					zaddmod(t2, t4, n, &t4);
					zmontmul(t3, t4, &zn1);
				}
			}
			zmontmul(d1, s, &t1);
			zmontmul(d, s1, &t2);
			zaddmod(t1, t2, n, &t3);
			zmontsq(t3, &xn);
			zsubmod(t1, t2, n, &t3);
			zmontsq(t3, &t3);
			zmontmul(x1, t3, &zzn);
		  }
		  else
		  {
			zmontsq(s, &t1);
			zmontsq(d, &t2);
			zmontmul(t1, t2, &xn);
			zsubmod(t1, t2, n, &t3);
			zmontmul(rap, t3, &t4);
			zaddmod(t2, t4, n, &t4);
			zmontmul(t3, t4, &zzn);
			if ((maskcnt ) || ((!maskcnt) && (!(mask & 1))))
			{
				zmontmul(d1, s, &t1);
				zmontmul(d, s1, &t2);
				zaddmod(t1, t2, n, &t3);
				zmontsq(t3, &xn1);
				zsubmod(t1, t2, n, &t3);
				zmontsq(t3, &t3);
				zmontmul(x1, t3, &zn1);
			}
		  }
		  mask >>= 1; first = 0;
		}
		mask = (RADIX >> 1);
	}
	if (zinv(zzn, n, f))
		return (1);
	zmontmul(*f, zrrr, x);
	zmontmul(*x, xn, x);
	return (0);
}

static long 
ph1(
	verylong n,
	verylong *x,
	long m,
	verylong rap,
	verylong *f
	)
{
 /* x = x^B, B is product prime powers <=m	 */
 /* if 0, success				 */
 /* if 1, n factored, factor in f		 */
	static verylong e = 0;
	register long p;
	register long q;
	register long mp;
	register long iter_cnt = 0;

	zpstart2();
	zintoz(1,&e);
	while ((p = zpnext()) <= m)
	{
		mp = m / p;
		q = p;
		while (q <= mp)
			q *= p;
#if (PHASE1_BATCH == 1)
		zintoz(q,&e);
		if (ph1exp(n, x, e, rap, f) > 0)
			return (1);
	}
#else

		zsmul(e,q,&e);
		iter_cnt ++;
		if (iter_cnt == PHASE1_BATCH)
		{
			if (ph1exp(n, x, e, rap, f) > 0)
				return (1);
			iter_cnt = 0;
			zintoz(1,&e);
		}
	}
	if (iter_cnt)
	{
		if (ph1exp(n, x, e, rap, f) > 0)
			return (1);
	}
#endif
	return (0);
}

static long 
ph1toph2(
	verylong n,
	verylong *x,
	verylong *y,
	verylong *ra,
	verylong alpha,
	verylong mu,
	verylong *f
	)
{
 /* tranform from first type of curve to second	 */
 /* x, alpha, mu input; x, y, ra output		 */
 /* if 0, success				 */
 /* if 1, n factored, factor in f		 */
	static verylong beta = 0;
	static verylong t = 0;

	zaddmod(*x, alpha, n, &t);
	zmontmul(*x, t, &t);
	zmontmul(*x, t, &t);
	zaddmod(*x, t, n, &beta);
	if (zinv(beta, n, y))
	{
		zcopy(*y, f);
		return (1);
	}
	zmontmul(*y, zrrr, y);
	zsubmod(*x, mu, n, &t);
	zmontmul(t, *y, x);
	zmontmul(alpha, mu, &t);
	zaddmod(t, zr, n, &t);
	zmontmul(*y, t, &t);
	zmontmul(*y, t, ra);
	return (0);
}

static long 
ph2squ(
	verylong n,
	verylong x1s,
	verylong y1s,
	verylong *x2,
	verylong *y2,
	verylong ra,
	verylong *f
	)
{
 /* (x2,y2)=(x1s,y1s)^2 on y^2=x^3+ra*x-b 	 */
 /* infinity if x1s[0] < 0			 */
 /* if 0, success				 */
 /* if 1, n factored, factor in f		 */
	static verylong x1 = 0;
	static verylong yy1 = 0;
	static verylong t = 0;

	zcopy(x1s, &x1);
	zcopy(y1s, &yy1);
	if (x1s[0] < 0)
	{
		(*x2)[0] = -1;
		zzero(y2);
		zzero(f);
		return (1);
	}
	zaddmod(yy1, yy1, n, &t);
	if (zinv(t, n, y2))
	{
		zcopy(*y2, f);
		return (1);
	}
	zmontmul(*y2, zrrr, y2);
	zmontsq(x1,x2);
	zaddmod(*x2,*x2,n,&t);
	zaddmod(*x2,t,n,&t);
	zaddmod(t, ra, n, &t);
	zmontmul(t, *y2, y2);
	zmontsq(*y2, &t);
	zsubmod(t, x1, n, &t);
	zsubmod(t, x1, n, x2);
	zsubmod(x1, *x2, n, &t);
	zmontmul(*y2, t, &t);
	zsubmod(t, yy1, n, y2);
	return (0);
}

static long 
ph2mul(
	verylong n,
	verylong x1s,
	verylong y1s,
	verylong x2s,
	verylong y2s,
	verylong *x3,
	verylong *y3,
	verylong ra,
	verylong *f
	)
{
 /* (x3,y3)=(x1s,y1s)*(x2s,y2s) on y^2=x^3+ra*x-b 	 */
 /* infinity if x1s[0] < 0 (or x2s[0]<0)			 */
 /* if 0, success					 */
 /* if 1, n factored, factor in f			 */
	static verylong x1 = 0;
	static verylong yy1 = 0;
	static verylong x2 = 0;
	static verylong y2 = 0;
	static verylong t = 0;

	zcopy(x1s, &x1);
	zcopy(y1s, &yy1);
	zcopy(x2s, &x2);
	zcopy(y2s, &y2);
	if (x1s[0] < 0)
	{
		zcopy(x2, x3);
		zcopy(y2, y3);
		zcopy(y2, f);
		return (1);
	}
	if (x2s[0] < 0)
	{
		zcopy(x1, x3);
		zcopy(yy1, y3);
		zcopy(yy1, f);
		return (1);
	}
	if (!(zcompare(x1, x2)))
	{
		zaddmod(yy1, y2, n, &t);
		if (!t[1] && t[0] == 1)
		{
			(*x3)[0] = -1;
			zzero(y3);
			zzero(f);
			return (1);
		}
		else
			return (ph2squ(n, x1, yy1, x3, y3, ra, f));
	}
	else
	{
		zsubmod(x1, x2, n, &t);
		if (zinv(t, n, y3))
		{
			zcopy(*y3, f);
			return (1);
		}
		zmontmul(*y3, zrrr, y3);
		zsubmod(yy1, y2, n, &t);
		zmontmul(t, *y3, y3);
		zmontsq(*y3, &t);
		zsubmod(t, x1, n, &t);
		zsubmod(t, x2, n, x3);
		zsubmod(x1, *x3, n, &t);
		zmontmul(*y3, t, &t);
		zsubmod(t, yy1, n, y3);
		return (0);
	}
}

static long 
ph2exp(
	verylong n,
	verylong x1,
	verylong yy1,
	verylong *x2,
	verylong *y2,
	long e,
	verylong ra,
	verylong *f
	)
{
 /* (x2,y2)=(x1,yy1)^e on y^2=x^3+ra*x-b 	 */
 /* if 0, success			 */
 /* if 1, n factored, factor in f	 */
	static verylong x1s = 0;
	static verylong y1s = 0;
	static verylong x3 = 0;
	static verylong y3 = 0;
	register long i;
	register long two;
	register long three;
	long ee[NBITSH];

	if (e == 1)
	{
		zcopy(x1, x2);
		zcopy(yy1, y2);
		return (0);
	}
	zcopy(x1, &x1s);
	zcopy(yy1, &y1s);
	if (e == 2)
		return (ph2squ(n, x1s, y1s, x2, y2, ra, f));
	i = -1;
	two = 0;
	three = 0;
	while (e)
	{
		ee[++i] = (e & 3);
		e >>= 2;
		if (ee[i] == 3)
			three = 1;
	}
	if (three || ee[i] == 2)
	{
		if (ph2squ(n, x1s, y1s, x2, y2, ra, f))
			return (1);
		two = 1;
		if (three && ph2mul(n, x1s, y1s, *x2, *y2, &x3, &y3, ra, f))
			return (1);
	}
	if (ee[i] == 3)
	{
		zcopy(x3, x2);
		zcopy(y3, y2);
	}
	if (ee[i] == 1)
	{
		if (two)
			two = 0;
		else
		{
			two = 1;
			zcopy(x1s, x2);
			zcopy(y1s, y2);
		}
	}
	for (i = i - 1; i >= 0; i--)
	{
		if (two)
		{
			if (ph2squ(n, *x2, *y2, x2, y2, ra, f))
				return (1);
		}
		else
			two = 1;
		switch (ee[i])
		{
			case 0:
				{
					if (ph2squ(n, *x2, *y2, x2, y2, ra, f))
						return (1);
					break;
				}
			case 1:
				{
					if (ph2squ(n, *x2, *y2, x2, y2, ra, f))
						return (1);
					if (ph2mul(n, x1s, y1s, *x2, *y2, x2, y2, ra, f))
						return (1);
					break;
				}
			case 2:
				{
					if (ph2mul(n, x1s, y1s, *x2, *y2, x2, y2, ra, f))
						return (1);
					if (ph2squ(n, *x2, *y2, x2, y2, ra, f))
						return (1);
					break;
				}
			case 3:
				{
					if (ph2squ(n, *x2, *y2, x2, y2, ra, f))
						return (1);
					if (ph2mul(n, x3, y3, *x2, *y2, x2, y2, ra, f))
						return (1);
					break;
				}
		}
	}
	return (0);
}

static long 
ph2set(
	verylong n,
	verylong x,
	verylong y,
	long te,
	verylong ra,
	verylong *f
	)
{
 /* set up table for table size te	 */
 /* if 0, success			 */
 /* if 1, n factored, factor in f	 */
	static verylong z = 0;
	register long i;
	register long j;
	long expi;
	long exp2;

	expi = RADIX / (te + 2);

	exp2 = 1 + zrandom(expi);
	expi = 1 + zrandom(expi);
	for (i = 0; i <= te; i++)
	{
		zcopy(x, &(ecm_tex[i]));
		zcopy(y, &(ecm_tey[i]));
		for (j = te; j > 0; j--)
			if (ph2exp(n, ecm_tex[i], ecm_tey[i], &(ecm_tex[i]), &(ecm_tey[i]), expi, ra, f))
				return (1);
		expi += exp2;
	}
	for (j = 1; j <= te; j++)
	{
		for (i = te; i >= j; i--)
		{
			zsubmod(n, ecm_tey[i - 1], n, &z);
			if (ph2mul(n, ecm_tex[i], ecm_tey[i], ecm_tex[i - 1], z, &(ecm_tex[i]), &(ecm_tey[i]), ra, f))
				return (1);
		}
	}
	return (0);
}

static void
ph2prec(
	verylong n,
	long n1,
	long j
	)
{
	static verylong mcn2j = 0;
	static verylong t = 0;
	register long i;
	register long n2;
	register long n2j;

	if (n1 > 1)
	{
		n2 = ((n1 - 1) >> 1);
		n2j = j + n2;
		zaddmod(ecm_coef[n2j], znm, n, &(ecm_coef[n2j]));
		zsubmod(n, ecm_coef[n2j], n, &mcn2j);
		n2j ++;
		for (i = 0; i < n2; i++)
		{
			zmontmul(ecm_coef[i + n2j], mcn2j, &t);
			zaddmod(ecm_coef[i + j], t, n, &(ecm_coef[i + j]));
		}
		ph2prec(n, n2, n2j);
		ph2prec(n, n2, j);
	}
}

static void
ph2eval(
	verylong n,
	long ind,
	long n1,
	long j,
	verylong *result
	)
{
	register long n2;

	if (n1 == 1)
		zaddmod(ecm_coef[j], ecm_power[0], n, result);

	else
	{
		n2 = ((n1 - 1) >> 1);
		zaddmod(ecm_coef[n2 + j], ecm_power[ind], n, &(ecm_eval[ind]));
		ph2eval(n, ind - 1, n2, n2 + j + 1, result);
		zmontmul((ecm_eval[ind]), *result, &(ecm_eval[ind]));
		ph2eval(n, ind - 1, n2, j, result);
		zaddmod((ecm_eval[ind]), *result, n, result);
	}
}

static long 
ph2(
	verylong n,
	verylong inx,
	verylong iny,
	long m,
	long te,
	verylong ra,
	verylong *f
	)
{
 /* does second phase for m 		 */
 /* if 0, no factor found		 */
 /* if 1, n factored, factor in f	 */
	static long non_initialized = 1;
	static verylong x = 0;
	static verylong y = 0;
	static verylong x1 = 0;
	static verylong x2 = 0;
	static verylong prod = 0;
	register long j;
	register long k;
	long r;
	long r2;
	long s;
	long ind;

	if (non_initialized)
	{
		for (j = 0; j < MAXRP; j++)
			ecm_coef[j] = 0;
		for (j = 0; j < MAXTEP; j++)
			ecm_tex[j] = ecm_tey[j] = 0;
		for (j = 0; j < MAXPWRP; j++)
			ecm_power[j] = 0;
		for (j = 0; j < MAXPWRP; j++)
			ecm_eval[j] = 0;
		non_initialized = 0;
	}
	zcopy(inx, &x);
	zcopy(iny, &y);
	if (ph2set(n, x, y, te, ra, f))
		return (1);
	if ((r = 100 * te) >= MAXRP)
		r = MAXRP - 1;
	r2 = 1;
	ind = 0;
	j = 3;
	while (j <= r)
	{
		ind++;
		r2 = j;
		j <<= 1;
		j++;
	}
	r = r2;
	s = m / r;
	while (r > 3 && s < 2 * r)
	{
		r >>= 1;
		ind--;
		s = m / r;
	}
	zcopy(zr, (&(ecm_coef[0])));
	for (j = 1; j <= r; j++)
	{
		for (k = 0; k < te; k++)
		{
			if (ph2mul(n, ecm_tex[k], ecm_tey[k], ecm_tex[k + 1], ecm_tey[k + 1], &(ecm_tex[k]), &(ecm_tey[k]), ra, f))
				return (1);
		}
		zsubmod(n, ecm_tex[0], n, &x1);
		zcopy(ecm_coef[j - 1], &ecm_coef[j]);
		for (k = j - 1; k > 0; k--)
		{
			zmontmul(x1, ecm_coef[k], &x2);
			zaddmod(x2, ecm_coef[k - 1], n, &(ecm_coef[k]));
		}
		zmontmul(x1, ecm_coef[0], &(ecm_coef[0]));
	}
	ph2prec(n, r, (long) 0);
	if (ph2set(n, x, y, te, ra, f))
		return (1);
	zone(&prod);
	for (j = s; j > 0; j--)
	{
		for (k = 0; k < te; k++)
		{
			if (ph2mul(n, ecm_tex[k], ecm_tey[k], ecm_tex[k + 1], ecm_tey[k + 1], &(ecm_tex[k]), &(ecm_tey[k]), ra, f))
				return (1);
		}
		zcopy(ecm_tex[0], &(ecm_power[0]));
		for (k = 0; k < ind; k++)
			zmontsq(ecm_power[k], &(ecm_power[k + 1]));
		ph2eval(n, ind, r, (long) 0, &x2);
		zmontmul(x2, prod, &prod);
	}
	if (zinv(prod, n, f))
		return (1);
	return (0);
}

static void
message(
	char *str,
	double t
	)
{
	printf("%s, %8.3lf seconds\n", str, gettime() - t);
	fflush(stdout);
}

static long 
trial(
	verylong n,
	long m,
	verylong *f,
	long info
	)
{
 /* attempt to factor n, smoothness bound phase 1 is m	 */
 /* if 0, no success					 */
 /* if 1, n factored in setup, factor in f 		 */
 /* if 2, n factored in phase 1, factor in f 		 */
 /* if 3, n factored in ph1toph2, factor in f 		 */
 /* if 4, n factored in phase 1, factor in f 		 */
	static verylong x = 0;
	static verylong y = 0;
	static verylong rap = 0;
	static verylong alpha = 0;
	static verylong mu = 0;
	static verylong ra = 0;
	static long te;
	double tcnt = gettime();

	if (ph1set(n, &rap, &alpha, &mu, &x, f) > 0)
		return (1);
	if (info >= 2)
		message("   curve initialized", tcnt);
	if (m > MAXM)
		m = MAXM;
	if (ph1(n, &x, m, rap, f))
		return (2);
	if (info >= 2)
		message("   phase 1 completed", tcnt);
	if (ph1toph2(n, &x, &y, &ra, alpha, mu, f))
		return (3);
 /*
  * if (m >= 4500000) te = 60; else
	 */ if (m >= 1125000)
		te = 30;
	else if (m >= 720000)
		te = 24;
	else if (m >= 405000)
		te = 18;
	else if (m >= 180000)
		te = 12;
	else if (m >= 45000)
		te = 6;
	else if (m >= 11250)
		te = 3;
	else if (m >= 5000)
		te = 2;
	else
		te = 1;
	m *= 10;
	if (ph2(n, x, y, m, te, ra, f))
		return (4);
	if (info >= 2)
		message("   phase 2 completed", tcnt);
	return (0);
}

long 
zecm(
	verylong n,
	verylong *f,
	long uni,
	long nb,
	long bound,
	long co,
	long info
	)
{
 /* attempt to factor n, factor in f */
 /* uni is seed for random number generator */
 /* uni=0 means random generator started already */
 /* nb is number of curves */
 /* bound is bound for phase 1 of first curve */
 /* |co| is maximum number of probabilistic compositeness tests */
 /* info = 0: no timings or information */
 /* |info|=1: timing per curve, and some additional info */
 /* |info|>1: timing per phase and per curve */
 /* side effect: restart prime number generator, */
 /* zpnext() undefined after call */

	static verylong q = 0;
	static verylong r = 0;
	register long i;
	register long j;
	double tcnt = gettime();

	if (info < 0)
		info = -info;
	if (bound < MINBOUND)
		bound = MINBOUND;
	if (uni)
	{
		if (uni < 0)
			uni = ((-uni) << 1);
		else
			uni = 2 * uni + 1;
		zrstarts(uni + 1);
	}
	if (!n || n[0] < 0 || (n[0] == 1 && n[1] <= RADIXROOT))
	{
		if (info)
			printf("wrong input to ecm\n");
		return (0);
	}
	if (!(n[1] & 1))
	{
		zintoz(2, f);
		if (info)
		{
			printf("ecm input is even\n");
			fflush(stdout);
		}
		return (1);
	}
	if (info)
	{
		i = zfwrite(stdout, n, 70, "ecm on ", "");
		printf("\n(%d digits)\n", i);
	}
	if (co)
	{
		if (co > 0)
			co = -co;
		zcopy(n, f);
		i = zcomposite(f, co, 0);
		if (!i)
		{
			if (info)
				message("ecm input is probably prime", tcnt);
			return (-1);
		}
		if (i == -2)
		{
			if (info)
				message("factor found in compositeness test", tcnt);
			return (1);
		}
		if (i == -1)
		{
			if (info)
				message("factor in f^n-f", tcnt);
			return (2);	/* factor in f^n-f */
		}
		if (i != 1)
		{
			zhalt("impossible return value of zcomposite in zecm   BUG");
			return (0);
		}
		if (info >= 2)
			message("ecm input composite, not a prime power", tcnt);
	}
	else if (info >= 2)
	{
		printf("ecm input assumed to be composite, and not a prime power\n");
		fflush(stdout);
	}

	zone(f);
	zmkeep(n);

	for (i = nb; i > 0; i--)
	{
		if (info)
			printf("%ld-th ecm trial with bound %ld\n", nb - i + 1, bound);
		j = trial(n, bound, f, info);
		bound = (long) (1.02 * bound);
		if (j)
		{
			if (((*f)[0] == 1 && (*f)[1] <= 1) || !zcompare(n, *f))
			{
				if (info)
					message("trivial factor in ecm", tcnt);
				continue;
			}
			if (info)
			{
				printf("factor in ");
				switch (j)
				{
					case 1:
						{
							message("curve set-up", tcnt);
							break;
						}
					case 2:
						{
							message("phase 1", tcnt);
							break;
						}
					case 3:
						{
							message("curve tranformation (phase 1 -> 2)", tcnt);
							break;
						}
					case 4:
						{
							message("phase 2", tcnt);
							break;
						}
				}
			}
			zdiv(n, *f, &q, &r);
			if (r[1] || r[0] != 1)
			{
				zfwriteln(stderr,*f);
				zhalt("this is a wrong factor, found in zecm   BUG");
				zmback();
				return (0);
			}
			zmback();
			return (1);
		}
		if (info)
		{
			message("no success", tcnt);
		}
	}
	if (info)
		printf("done\n");
	zmback();
	return (0);
}

#endif
