(*  :Title:	Digital Signal Processing Analysis  *)

(*  :Authors:	Brian Evans, James McClellan  *)

(*
    :Summary:	Perform basic analysis on a one-dimensional discrete-time
		function, including plotting f[n] and graphing the poles
		and zeroes of F[z].
 *)

(*  :Context:	SignalProcessing`Digital`DSPAnalyze`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*  :History:	*)

(*  :Keywords:	analysis, transforms, signals and systems	*)

(*  :Source:	*)

(*  :Warning:	*)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*  :Limitation:  *)

(*  :Discussion:  *)

(*  :Functions:	DSPAnalyze  *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ];


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Digital`DSPAnalyze`",
	      "SignalProcessing`Digital`DTFT`",
	      "SignalProcessing`Digital`DSupport`",
	      "SignalProcessing`Digital`ZTransform`",
	      "SignalProcessing`Digital`ZSupport`",
	      "SignalProcessing`Support`TransSupport`",
	      "SignalProcessing`Support`ROC`",
	      "SignalProcessing`Support`SigProc`",
	      "SignalProcessing`Support`SupCode`" ]


(*  U S A G E     I N F O R M A T I O N  *)

DSPAnalyze::usage =
	"DSPAnalyze[f, n] and DSPAnalyze[f, n, start, end] \
	will plot the real values of f[n] from n = Floor[start] to \
	n = Ceiling[end], print its z-transform, indicate conditions \
	for stability, display the pole-zero diagram, and plot the magnitude \
	and phase responses. \
	DSPAnalyze[f, {n1,n2}] and DSPAnalyze[f, {n1,n2}, {s1,s2}, {e1,e2}] \
	treat f as a function of two-variables n1 and n2. \
	The domain of the time-domain graph is n1 from s1 to e1 and \
	n2 from s2 to e2. \
	All non-variable symbols will be assigned a value of 1 when the \
	analyzer generates graphics (override this with options like a -> 2)."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin["`Private`"]


(*  M E S S A G E S  *)

ZTransform::notvalid = "The forward z-transform could not be found."
DTFTransform::notvalid = "The forward DTFT could not be found."


(*  S U P P O R T I N G     F U N C T I O N S  *)

(*  DSPAnalyzeQ  *)
DSPAnalyzeQ[start_, end_] :=
	NumberQ[N[start]] && NumberQ[N[end]] && N[end >= start]

(*  PrintStability  *)
PrintStability[ztrans_] :=
	Block [	{stable},
		stable = Stable[ztrans];
		Which [ SameQ[stable, True],
			  Print["The system is stable."],
			SameQ[stable, False],
			  Print["The system  is not stable."],
			True,
			   Print["The system is stable if ", stable] ] ]

(*  D S P A n a l y z e  *)

DSPAnalyze[f_, n_Symbol] := DSPAnalyze[f, n, 0]
DSPAnalyze[f_, n_Symbol, start_] := DSPAnalyze[f, n, start, start + 40]

DSPAnalyze[f_, n_Symbol, start_, end_, op___] :=
	Block [	{defaultvalues = {}, dtft, freqresp, ftemp, funct, mag,
		 oplist, varlist, zfreqresp, ztrans},

		oplist = ToList[op];

		funct = ToDiscrete[f];
		ftemp = N[ TheFunction[ funct ] ];
		varlist = Select[GetVariables[ftemp /. oplist],
				 Function[var, ! SameQ[var, n]]];
		If [ (! EmptyQ[varlist]) || (! EmptyQ[oplist]),
		     defaultvalues = oplist ~Join~ Map[(#1 -> 1)&, varlist];
		     If [ EmptyQ[oplist],
			  Print["For plotting only, these symbols will"];
			  Print["be assigned a value of 1:  ", varlist],
			  Print["For plotting only, these symbols will"];
			  Print["will be assigned default values:"];
			  Print[ defaultvalues ] ];
		     ftemp = ftemp /. defaultvalues ];

		(*  Plot "time"-domain	*)

		SequencePlot[ ftemp, {n, start, end},
			      PlotLabel -> "Discrete-Time Domain Analysis" ];

		(*    Find F[z]; if it exists, print it out, give stability  *)
		(*  information, and display a pole-zero plot.		     *)

		ztrans = ZTransform[funct, n, Global`z];
		If [ ! SameQ[Head[ztrans], ZTransData],
		     Message[ZTransform::notvalid];
		     mag = Linear,

		     Print[ funct ];
		     Print[ "has the following z-transform:" ];
		     Print[ TheFunction[ztrans] ];
		     Print[ "The region of convergence is:"];
		     Print[ GetRMinus[ztrans], " < |z| < ", GetRPlus[ztrans] ];
		     PrintStability[ztrans];
		     PoleZeroPlot[ ztrans /. defaultvalues ];
		     Print[ "By substituting into the z-transform," ];
		     Print[ "the frequency response is" ];
		     zfreqresp = TheFunction[ztrans] /. Global`z -> Exp[I w];
		     Print[ zfreqresp /. w -> Global`w ];
		     Print[ " " ];
		     mag = Log ];

		(*  Determine the frequency response  *)

		dtft = DTFTransform[funct, n, w];
		If [ ! SameQ[Head[dtft], DTFTData],
		     Message[DTFTransform::notvalid]; Return[ztrans] ];
		freqresp = TheFunction[dtft];

		(*  Plot the interesting section of the freq. response  *)

		If [ ! SameQ[freqresp, zfreqresp],
		     Print[ funct ];
		     Print[ "has the following frequency response:"];
		     Print[ freqresp /. w -> Global`w ] ];

		freqresp = ToContinuous[freqresp];
		MagPhasePlot[ Periodic[2 Pi, w][freqresp /. defaultvalues],
			      {w, - 2 Pi, 2 Pi},
			      Domain -> Continuous, DomainScale -> Linear,
			      MagRangeScale -> mag, PhaseRangeScale -> Degree ];

		(*  Return the z-transform  *)

		ztrans ] /;
	N [ DSPAnalyzeQ[start, end] ]


DSPAnalyze[f_, {n1_Symbol, n2_Symbol}] := DSPAnalyze[f, {n1, n2}, {0, 0}]
DSPAnalyze[f_, {n1_Symbol, n2_Symbol}, start_] :=
	DSPAnalyze[f, {n1, n2}, start, start + {9, 9}]

DSPAnalyze[f_, {n1_Symbol, n2_Symbol}, start_, end_, op___] :=
	Block [	{coordlist, defaultvalues = {}, dtft, freqresp, ftemp, funct,
		 mag, nlist, oplist, stable, varlist, w1, w2,
		 wlist, zfreqresp, zlist, ztrans},

		oplist = ToList[op];

		nlist = { n1, n2 };
		wlist = { w1, w2 };
		zlist = { Global`z1, Global`z2 };

		funct = ToDiscrete[f];
		ftemp = N[ TheFunction[ funct ] ];
		varlist = Select[GetVariables[ftemp /. oplist],
			 	 Function[var,
					  ! SameQ[var,n1] && ! SameQ[var,n2]]];
		If [ (! EmptyQ[varlist]) || (! EmptyQ[oplist]),
		     defaultvalues = oplist ~Join~ Map[(#1 -> 1)&, varlist];
		     If [ EmptyQ[oplist],
			  Print["For plotting only, these symbols will"];
			  Print["be assigned a value of 1:  ", varlist],
			  Print["For plotting only, these symbols will"];
			  Print["will be assigned default values:"];
			  Print[ defaultvalues ] ];
		     ftemp = ftemp /. defaultvalues ];

		(*  Generate three-dimensional coordinates for plot  *)

		SequencePlot[ ftemp,
			      { n1, start[[1]], end[[1]] },
			      { n2, start[[2]], end[[2]] },
			      PlotLabel -> "Discrete-Time Domain Analysis" ];

		(*  Find F[z] and print it out.  *)

		ztrans = ZTransform[funct, nlist, zlist];
		If [ ! SameQ[Head[ztrans], ZTransData],
		     Message[ZTransform::notvalid];
		     mag = Linear,

		     Print[ funct ];
		     Print[ "has the following z-transform:" ];
		     Print[ TheFunction[ztrans] ];
		     Print[ "The region of convergence is:"];
		     Print[ GetRMinus[ztrans] [[1]], " < |z1| < ",
			    GetRPlus[ztrans]  [[1]] ];
		     Print[ GetRMinus[ztrans] [[2]], " < |z2| < ",
			    GetRPlus[ztrans]  [[2]] ];
		     PrintStability[ztrans];
		     PoleZeroPlot[ztrans /. defaultvalues];
		     Print[ "By substituting into the z-transform," ];
		     Print[ "the frequency response is" ];
		     zfreqresp = TheFunction[ztrans] /.
			         ReplaceWith[zlist, {Exp[I w1], Exp[I w2]}];
		     Print[ zfreqresp /.
			    ReplaceWith[wlist, {Global`w1, Global`w2}] ];
		     Print[ " " ];
		     mag = Log ];

		(*  Determine the frequency response  *)

		dtft = DTFTransform[funct, nlist, wlist];
		If [ ! SameQ[Head[dtft], DTFTData],
		     Message[DTFT::notvalid]; Return[ztrans] ];
		freqresp = TheFunction[dtft];

		(*  Plot the interesting section of the freq. response  *)

		If [ ! SameQ[freqresp, zfreqresp],
		     Print[ funct ];
		     Print[ "has the following frequency response:"];
		     Print[ freqresp /.
			    ReplaceWith[wlist, {Global`w1, Global`w2}] ] ];

		(*  Making the freqresp periodic takes too long to plot: *)
		(*    Periodic[{2 Pi, 2 Pi}, {w1, w2}][freqresp]	 *)

		MagPhasePlot[ freqresp /. defaultvalues,
			      {w1, 0, 2 Pi}, {w2, 0, 2 Pi},
			      Domain -> Continuous, DomainScale -> Linear,
			      MagRangeScale -> mag, PhaseRangeScale -> Degree ];

		(*  Return the z-transform  *)

		ztrans ] /;
	N [ DSPAnalyzeQ[ start[[1]], end[[1]] ] &&
	    DSPAnalyzeQ[ start[[2]], end[[2]] ] ]


(*  E N D     P A C K A G E  *)

End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ];


(*  H E L P     I N F O R M A T I O N  *)

Combine[ SPfunctions, { DSPAnalyze } ]
Protect[ DSPAnalyze ]


(*  E N D I N G     M E S S A G E  *)

Print["The digital signal analyzer is now loaded."]
Null
