/* Declarations for C-SKY opcode table
   Copyright (C) 2007-2021 Free Software Foundation, Inc.
   Contributed by C-SKY Microsystems and Mentor Graphics.

   This file is part of the GNU opcodes library.

   This library is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   It is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
   License for more details.

   You should have received a copy of the GNU General Public License
   along with GAS; see the file COPYING.  If not, write to the Free
   Software Foundation, 51 Franklin Street - Fifth Floor, Boston, MA
   02110-1301, USA.  */

#include "opcode/csky.h"
#include "safe-ctype.h"

#define OP_TABLE_NUM    2
#define MAX_OPRND_NUM   5

enum operand_type
{
  OPRND_TYPE_NONE = 0,
  /* Control register.  */
  OPRND_TYPE_CTRLREG,
  /* r0 - r7.  */
  OPRND_TYPE_GREG0_7,
  /* r0 - r15.  */
  OPRND_TYPE_GREG0_15,
  /* r16 - r31.  */
  OPRND_TYPE_GREG16_31,
  /* r0 - r31.  */
  OPRND_TYPE_AREG,
  /* (rx).  */
  OPRND_TYPE_AREG_WITH_BRACKET,
  OPRND_TYPE_AREG_WITH_LSHIFT,
  OPRND_TYPE_AREG_WITH_LSHIFT_FPU,

  OPRND_TYPE_FREG_WITH_INDEX,
  OPRND_TYPE_VREG_WITH_INDEX,
  /* r1 only, for xtrb0(1)(2)(3) in csky v1 ISA.  */
  OPRND_TYPE_REG_r1a,
  /* r1 only, for divs/divu in csky v1 ISA.  */
  OPRND_TYPE_REG_r1b,
  /* r28.  */
  OPRND_TYPE_REG_r28,
  OPRND_TYPE_REGr4_r7,
  /* sp register with bracket.  */
  OPRND_TYPE_REGbsp,
  /* sp register.  */
  OPRND_TYPE_REGsp,
  /* Register with bracket.  */
  OPRND_TYPE_REGnr4_r7,
  /* Not sp register.  */
  OPRND_TYPE_REGnsp,
  /* Not lr register.  */
  OPRND_TYPE_REGnlr,
  /* Not sp/lr register.  */
  OPRND_TYPE_REGnsplr,
  /* hi/lo register.  */
  OPRND_TYPE_REGhilo,
  /* VDSP register.  */
  OPRND_TYPE_VREG,

  /* cp index.  */
  OPRND_TYPE_CPIDX,
  /* cp regs.  */
  OPRND_TYPE_CPREG,
  /* cp cregs.  */
  OPRND_TYPE_CPCREG,
  /* fpu regs.  */
  OPRND_TYPE_FREG,
  /* fpu even regs.  */
  OPRND_TYPE_FEREG,
  /* Float round mode.  */
  OPRND_TYPE_RM,
  /* PSR bits.  */
  OPRND_TYPE_PSR_BITS_LIST,

  /* Constant.  */
  OPRND_TYPE_CONSTANT,
  /* Floating Constant.  */
  OPRND_TYPE_FCONSTANT,
  /* Extern lrw constant.  */
  OPRND_TYPE_ELRW_CONSTANT,
  /* [label].  */
  OPRND_TYPE_LABEL_WITH_BRACKET,
  /* The operand is the same as first reg.  It is a dummy reg that doesn't
     appear in the binary code of the instruction.  It is also used by
     the disassembler.
     For example: bclri rz, rz, imm5 -> bclri rz, imm5.  */
  OPRND_TYPE_DUMMY_REG,
  /* The type of the operand is same as the first operand.  If the value
     of the operand is same as the first operand, we can use a 16-bit
     instruction to represent the opcode.
     For example: addc r1, r1, r2 -> addc16 r1, r2.  */
  OPRND_TYPE_2IN1_DUMMY,
  /* Output a reg same as the first reg.
     For example: addc r17, r1 -> addc32 r17, r17, r1.
     The old "addc" cannot be represented by a 16-bit instruction because
     16-bit "addc" only supports regs from r0 to r15.  So we use "addc32"
     which has 3 operands, and duplicate the first operand to the second.  */
  OPRND_TYPE_DUP_GREG0_7,
  OPRND_TYPE_DUP_GREG0_15,
  OPRND_TYPE_DUP_AREG,
  /* Immediate.  */
  OPRND_TYPE_IMM1b,
  OPRND_TYPE_IMM2b,
  OPRND_TYPE_IMM3b,
  OPRND_TYPE_IMM4b,
  OPRND_TYPE_IMM5b,
  OPRND_TYPE_IMM7b,
  OPRND_TYPE_IMM8b,
  OPRND_TYPE_IMM9b,
  OPRND_TYPE_IMM12b,
  OPRND_TYPE_IMM15b,
  OPRND_TYPE_IMM16b,
  OPRND_TYPE_IMM18b,
  OPRND_TYPE_IMM32b,
  /* Immediate left shift 2 bits.  */
  OPRND_TYPE_IMM7b_LS2,
  OPRND_TYPE_IMM8b_LS2,
  /* OPRND_TYPE_IMM5b_a_b means: Immediate in (a, b).  */
  OPRND_TYPE_IMM5b_1_31,
  OPRND_TYPE_IMM5b_7_31,
  /* OPRND_TYPE_IMM5b_LS means: Imm <= prev Imm.  */
  OPRND_TYPE_IMM5b_LS,
  /* Operand type for rori and rotri.  */
  OPRND_TYPE_IMM5b_RORI,
  OPRND_TYPE_IMM5b_VSH,
  OPRND_TYPE_IMM5b_POWER,
  OPRND_TYPE_IMM5b_7_31_POWER,
  OPRND_TYPE_IMM5b_BMASKI,
  OPRND_TYPE_IMM8b_BMASKI,
  /* For v2 movih.  */
  OPRND_TYPE_IMM16b_MOVIH,
  /* For v2 ori.  */
  OPRND_TYPE_IMM16b_ORI,
  /* For v2 ld/st.  */
  OPRND_TYPE_IMM_LDST,
  OPRND_TYPE_IMM_FLDST,
  OPRND_TYPE_IMM2b_JMPIX,
  /* Offset for bloop.  */
  OPRND_TYPE_BLOOP_OFF4b,
  OPRND_TYPE_BLOOP_OFF12b,
  /* Offset for jump.  */
  OPRND_TYPE_OFF8b,
  OPRND_TYPE_OFF10b,
  OPRND_TYPE_OFF11b,
  OPRND_TYPE_OFF16b,
  OPRND_TYPE_OFF16b_LSL1,
  OPRND_TYPE_OFF26b,
  /* An immediate or label.  */
  OPRND_TYPE_IMM_OFF18b,
  /* Offset immediate.  */
  OPRND_TYPE_OIMM3b,
  OPRND_TYPE_OIMM4b,
  OPRND_TYPE_OIMM5b,
  OPRND_TYPE_OIMM8b,
  OPRND_TYPE_OIMM12b,
  OPRND_TYPE_OIMM16b,
  OPRND_TYPE_OIMM18b,
  /* For csky v2 idly.  */
  OPRND_TYPE_OIMM5b_IDLY,
  /* For v2 bmaski.  */
  OPRND_TYPE_OIMM5b_BMASKI,
  /* Constants.  */
  OPRND_TYPE_CONST1,
  /* PC relative offset.  */
  OPRND_TYPE_PCR_OFFSET_16K,
  OPRND_TYPE_PCR_OFFSET_64K,
  OPRND_TYPE_PCR_OFFSET_64M,
  OPRND_TYPE_CPFUNC,
  OPRND_TYPE_GOT_PLT,
  OPRND_TYPE_REGLIST_LDM,
  OPRND_TYPE_REGLIST_DASH,
  OPRND_TYPE_FREGLIST_DASH,
  OPRND_TYPE_REGLIST_COMMA,
  OPRND_TYPE_REGLIST_DASH_COMMA,
  OPRND_TYPE_BRACKET,
  OPRND_TYPE_ABRACKET,
  OPRND_TYPE_JBTF,
  OPRND_TYPE_JBR,
  OPRND_TYPE_JBSR,
  OPRND_TYPE_UNCOND10b,
  OPRND_TYPE_UNCOND16b,
  OPRND_TYPE_COND10b,
  OPRND_TYPE_COND16b,
  OPRND_TYPE_JCOMPZ,
  OPRND_TYPE_LSB2SIZE,
  OPRND_TYPE_MSB2SIZE,
  OPRND_TYPE_LSB,
  OPRND_TYPE_MSB,
  /* Single float and double float.  */
  OPRND_TYPE_SFLOAT,
  OPRND_TYPE_DFLOAT,
  OPRND_TYPE_HFLOAT_FMOVI,
  OPRND_TYPE_SFLOAT_FMOVI,
  OPRND_TYPE_DFLOAT_FMOVI,
};

/* Operand descriptors.  */
struct operand
{
  /* Mask for suboperand.  */
  unsigned int mask;
  /* Suboperand type.  */
  enum operand_type type;
  /* Operand shift.  */
  int shift;
};

struct soperand
{
  /* Mask for operand.  */
  unsigned int mask;
  /* Operand type.  */
  enum operand_type type;
  /* Operand shift.  */
  int shift;
  /* Suboperand.  */
  struct operand subs[3];
};

union csky_operand
{
  struct operand oprnds[MAX_OPRND_NUM];
  struct suboperand1
  {
    struct operand oprnd;
    struct soperand soprnd;
  } soprnd1;
  struct suboperand2
  {
    struct soperand soprnd;
    struct operand oprnd;
  } soprnd2;
};

/* Describe a single instruction encoding.  */
struct csky_opcode_info
{
  /* How many operands.  */
  long operand_num;
  /* The instruction opcode.  */
  unsigned int opcode;
  /* Operand information.  */
  union csky_operand oprnd;
};

/* C-SKY instruction description.  Each mnemonic can have multiple
   16-bit and 32-bit encodings.  */
struct csky_opcode
{
  /* The instruction name.  */
  const char *mnemonic;
  /* Whether this is an unconditional control transfer instruction,
     for the purposes of placing literal pools after it.
     0 = no, 1 = within function, 2 = end of function.
     See check_literals in gas/config/tc-csky.c.  */
  int transfer;
  /* Encodings for 16-bit opcodes.  */
  struct csky_opcode_info op16[OP_TABLE_NUM];
  /* Encodings for 32-bit opcodes.  */
  struct csky_opcode_info op32[OP_TABLE_NUM];
  /* Instruction set flag.  */
  BFD_HOST_U_64_BIT isa_flag16;
  BFD_HOST_U_64_BIT isa_flag32;
  /* Whether this insn needs relocation, 0: no, !=0: yes.  */
  signed int reloc16;
  signed int reloc32;
  /* Whether this insn needs relaxation, 0: no, != 0: yes.  */
  signed int relax;
  /* Worker function to call when this instruction needs special assembler
     handling.  */
  bfd_boolean (*work)(void);
};

/* The following are the opcodes used in relax/fix process.  */
#define CSKYV1_INST_JMPI            0x7000
#define CSKYV1_INST_ADDI            0x2000
#define CSKYV1_INST_SUBI            0x2400
#define CSKYV1_INST_LDW             0x8000
#define CSKYV1_INST_STW             0x9000
#define CSKYV1_INST_BSR             0xf800
#define CSKYV1_INST_LRW             0x7000
#define CSKYV1_INST_ADDU            0x1c00
#define CSKYV1_INST_JMP             0x00c0
#define CSKYV1_INST_MOV_R1_RX       0x1201
#define CSKYV1_INST_MOV_RX_R1       0x1210

#define CSKYV2_INST_BT16            0x0800
#define CSKYV2_INST_BF16            0x0c00
#define CSKYV2_INST_BT32            0xe8600000
#define CSKYV2_INST_BF32            0xe8400000
#define CSKYV2_INST_BR32            0xe8000000
#define CSKYV2_INST_NOP             0x6c03
#define CSKYV2_INST_MOVI16          0x3000
#define CSKYV2_INST_MOVI32          0xea000000
#define CSKYV2_INST_MOVIH           0xea200000
#define CSKYV2_INST_LRW16           0x1000
#define CSKYV2_INST_LRW32           0xea800000
#define CSKYV2_INST_BSR32           0xe0000000
#define CSKYV2_INST_BR32            0xe8000000
#define CSKYV2_INST_FLRW            0xf4003800
#define CSKYV2_INST_JMPI32          0xeac00000
#define CSKYV2_INST_JSRI32          0xeae00000
#define CSKYV2_INST_JSRI_TO_LRW     0xea9a0000
#define CSKYV2_INST_JSR_R26         0xe8fa0000
#define CSKYV2_INST_MOV_R0_R0       0xc4004820

#define OPRND_SHIFT_0_BIT           0
#define OPRND_SHIFT_1_BIT           1
#define OPRND_SHIFT_2_BIT           2
#define OPRND_SHIFT_3_BIT           3
#define OPRND_SHIFT_4_BIT           4

#define OPRND_MASK_NONE             0x0
#define OPRND_MASK_0_1              0x3
#define OPRND_MASK_0_2              0x7
#define OPRND_MASK_0_3              0xf
#define OPRND_MASK_0_4              0x1f
#define OPRND_MASK_0_7              0xff
#define OPRND_MASK_0_8              0x1ff
#define OPRND_MASK_0_9              0x3ff
#define OPRND_MASK_0_10             0x7ff
#define OPRND_MASK_0_11             0xfff
#define OPRND_MASK_0_14             0x7fff
#define OPRND_MASK_0_15             0xffff
#define OPRND_MASK_0_17             0x3ffff
#define OPRND_MASK_0_25             0x3ffffff
#define OPRND_MASK_2_4              0x1c
#define OPRND_MASK_2_5              0x3c
#define OPRND_MASK_3_7              0xf8
#define OPRND_MASK_4                0x10
#define OPRND_MASK_4_5              0x30
#define OPRND_MASK_4_6              0x70
#define OPRND_MASK_4_7              0xf0
#define OPRND_MASK_4_8              0x1f0
#define OPRND_MASK_4_10             0x7f0
#define OPRND_MASK_5                0x20
#define OPRND_MASK_5_6              0x60
#define OPRND_MASK_5_7              0xe0
#define OPRND_MASK_5_8              0x1e0
#define OPRND_MASK_5_9              0x3e0
#define OPRND_MASK_6                0x40
#define OPRND_MASK_6_7              0xc0
#define OPRND_MASK_6_8              0x1c0
#define OPRND_MASK_6_9              0x3c0
#define OPRND_MASK_6_10             0x7c0
#define OPRND_MASK_7                0x80
#define OPRND_MASK_7_8              0x180
#define OPRND_MASK_8_9              0x300
#define OPRND_MASK_8_10             0x700
#define OPRND_MASK_8_11             0xf00
#define OPRND_MASK_9_10             0x600
#define OPRND_MASK_9_12             0x1e00
#define OPRND_MASK_10_11            0xc00
#define OPRND_MASK_10_14            0x7c00
#define OPRND_MASK_12_15            0xf000
#define OPRND_MASK_13_17            0x3e000
#define OPRND_MASK_16_19            0xf0000
#define OPRND_MASK_16_20            0x1f0000
#define OPRND_MASK_16_25            0x3ff0000
#define OPRND_MASK_17_24            0x1fe0000
#define OPRND_MASK_20               0x0100000
#define OPRND_MASK_20_21            0x0300000
#define OPRND_MASK_20_22            0x0700000
#define OPRND_MASK_20_23            0x0f00000
#define OPRND_MASK_20_24            0x1f00000
#define OPRND_MASK_20_25            0x3f00000
#define OPRND_MASK_21_24            0x1e00000
#define OPRND_MASK_21_25            0x3e00000
#define OPRND_MASK_25               0x2000000
#define OPRND_MASK_RSV              0xffffffff
#define OPRND_MASK_0_3or5_8         OPRND_MASK_0_3 | OPRND_MASK_5_8
#define OPRND_MASK_0_3or6_7         OPRND_MASK_0_3 | OPRND_MASK_6_7
#define OPRND_MASK_0_3or21_24       OPRND_MASK_0_3 | OPRND_MASK_21_24
#define OPRND_MASK_0_3or25          OPRND_MASK_0_3 | OPRND_MASK_25
#define OPRND_MASK_0_4or21_24       OPRND_MASK_0_4 | OPRND_MASK_21_24
#define OPRND_MASK_0_4or21_25       OPRND_MASK_0_4 | OPRND_MASK_21_25
#define OPRND_MASK_0_4or16_20       OPRND_MASK_0_4 | OPRND_MASK_16_20
#define OPRND_MASK_0_4or8_10        OPRND_MASK_0_4 | OPRND_MASK_8_10
#define OPRND_MASK_0_4or8_9         OPRND_MASK_0_4 | OPRND_MASK_8_9
#define OPRND_MASK_0_14or16_20      OPRND_MASK_0_14 | OPRND_MASK_16_20
#define OPRND_MASK_4or5_8           OPRND_MASK_4   | OPRND_MASK_5_8
#define OPRND_MASK_5or20_21         OPRND_MASK_5   | OPRND_MASK_20_21
#define OPRND_MASK_5or20_22         OPRND_MASK_5   | OPRND_MASK_20_22
#define OPRND_MASK_5or20_23         OPRND_MASK_5   | OPRND_MASK_20_23
#define OPRND_MASK_5or20_24         OPRND_MASK_5   | OPRND_MASK_20_24
#define OPRND_MASK_5or20_25         OPRND_MASK_5   | OPRND_MASK_20_25
#define OPRND_MASK_5or21_24         OPRND_MASK_5   | OPRND_MASK_21_24
#define OPRND_MASK_2_5or6_9         OPRND_MASK_2_5 | OPRND_MASK_6_9
#define OPRND_MASK_4_6or21_25       OPRND_MASK_4_6 | OPRND_MASK_21_25
#define OPRND_MASK_4_7or21_24       OPRND_MASK_4_7 | OPRND_MASK_21_24
#define OPRND_MASK_5_6or21_25       OPRND_MASK_5_6 | OPRND_MASK_21_25
#define OPRND_MASK_5_7or8_10        OPRND_MASK_5_7 | OPRND_MASK_8_10
#define OPRND_MASK_5_9or21_25       OPRND_MASK_5_9 | OPRND_MASK_21_25
#define OPRND_MASK_8_9or21_25       OPRND_MASK_8_9 | OPRND_MASK_21_25
#define OPRND_MASK_8_9or16_25       OPRND_MASK_8_9 | OPRND_MASK_16_20 | OPRND_MASK_21_25
#define OPRND_MASK_16_19or21_24     OPRND_MASK_16_19 | OPRND_MASK_21_24
#define OPRND_MASK_16_20or21_25     OPRND_MASK_16_20 | OPRND_MASK_21_25
#define OPRND_MASK_4or9_10or25      OPRND_MASK_4 | OPRND_MASK_9_10 | OPRND_MASK_25
#define OPRND_MASK_4_7or16_24       OPRND_MASK_4_7 | OPRND_MASK_16_20 | OPRND_MASK_21_24
#define OPRND_MASK_4_6or20          OPRND_MASK_4_6 | OPRND_MASK_20
#define OPRND_MASK_5_7or20          OPRND_MASK_5_7 | OPRND_MASK_20
#define OPRND_MASK_4_5or20or25      OPRND_MASK_4 | OPRND_MASK_5 | OPRND_MASK_20 | OPRND_MASK_25
#define OPRND_MASK_4_6or20or25      OPRND_MASK_4_6 | OPRND_MASK_20 | OPRND_MASK_25
#define OPRND_MASK_4_7or20or25      OPRND_MASK_4_7 | OPRND_MASK_20 | OPRND_MASK_25
#define OPRND_MASK_6_9or17_24       OPRND_MASK_6_9 | OPRND_MASK_17_24
#define OPRND_MASK_6_7or20          OPRND_MASK_6_7 | OPRND_MASK_20
#define OPRND_MASK_6or20            OPRND_MASK_6 | OPRND_MASK_20
#define OPRND_MASK_7or20            OPRND_MASK_7 | OPRND_MASK_20
#define OPRND_MASK_5or8_9or16_25    OPRND_MASK_5 | OPRND_MASK_8_9or16_25
#define OPRND_MASK_5or8_9or20_25    OPRND_MASK_5 | OPRND_MASK_8_9 | OPRND_MASK_20_25

#define OPERAND_INFO(mask, type, shift) \
  {OPRND_MASK_##mask, OPRND_TYPE_##type, shift}

#define OPCODE_INFO_NONE() \
  {-2, 0, \
      {{OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT), \
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT), \
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT), \
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT), \
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)}}}

/* Here and in subsequent macros, the "oprnd" arguments are the
   parenthesized arglist to the OPERAND_INFO macro above.  */
#define OPCODE_INFO(num, op, oprnd1, oprnd2, oprnd3, oprnd4, oprnd5) \
  {num, op, \
      {OPERAND_INFO oprnd1, OPERAND_INFO oprnd2, OPERAND_INFO oprnd3, \
	  OPERAND_INFO oprnd4, OPERAND_INFO oprnd5}}

#define OPCODE_INFO0(op) \
  {0, op,						\
      {{OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)}}}
#define OPCODE_INFO1(op, oprnd) \
  {1, op,						\
      {{OPERAND_INFO oprnd,				\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)}}}
#define OPCODE_INFO2(op, oprnd1, oprnd2) \
  {2, op,						\
      {{OPERAND_INFO oprnd1,				\
	OPERAND_INFO oprnd2,				\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)}}}
#define OPCODE_INFO3(op, oprnd1, oprnd2, oprnd3)	\
  {3, op,						\
      {{OPERAND_INFO oprnd1,				\
	OPERAND_INFO oprnd2,				\
	OPERAND_INFO oprnd3,				\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)}}}
#define OPCODE_INFO4(op, oprnd1, oprnd2, oprnd3, oprnd4) \
  {4, op,						\
      {{OPERAND_INFO oprnd1,				\
	OPERAND_INFO oprnd2,				\
	OPERAND_INFO oprnd3,				\
	OPERAND_INFO oprnd4,				\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)}}}
#define OPCODE_INFO_LIST(op, oprnd) \
  {-1, op,						\
      {{OPERAND_INFO oprnd,				\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT) ,	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT),	\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)}}}
#define OPCODE_INFO5(op, oprnd1, oprnd2, oprnd3, oprnd4, oprnd5) \
  {5, op,						\
      {{OPERAND_INFO oprnd1,				\
	OPERAND_INFO oprnd2,				\
	OPERAND_INFO oprnd3,				\
	OPERAND_INFO oprnd4,				\
	OPERAND_INFO oprnd5}}}

#define BRACKET_OPRND(oprnd1, oprnd2)			\
  OPERAND_INFO (RSV, BRACKET, OPRND_SHIFT_0_BIT),	\
  OPERAND_INFO oprnd1,					\
  OPERAND_INFO oprnd2,					\
  OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)
#define ABRACKET_OPRND(oprnd1, oprnd2)			\
  OPERAND_INFO (RSV, ABRACKET, OPRND_SHIFT_0_BIT),	\
  OPERAND_INFO oprnd1,					\
  OPERAND_INFO oprnd2,					\
  OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)

#define SOPCODE_INFO1(op, soprnd) \
  {1, op,						\
      {{soprnd,						\
	OPERAND_INFO (NONE, NONE, OPRND_SHIFT_0_BIT)}}}
#define SOPCODE_INFO2(op, oprnd, soprnd)		\
  {2, op,						\
      {{OPERAND_INFO oprnd, soprnd}}}


/* Before using the opcode-defining macros, there need to be
   #defines for _TRANSFER, _RELOC16, _RELOC32, and _RELAX.  See
   below.  */
/* FIXME:  it is a wart that these parameters are not explicit.  */

#define OP16(mnem, opcode16, isa)			\
  {mnem, _TRANSFER,					\
      {opcode16, OPCODE_INFO_NONE ()},			\
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},	\
      isa, 0, _RELOC16, 0, _RELAX, NULL}

#ifdef BUILD_AS

#define OP16_WITH_WORK(mnem, opcode16, isa, work)		\
  {mnem, _TRANSFER,						\
      {opcode16, OPCODE_INFO_NONE ()},				\
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},		\
      isa, 0, _RELOC16, 0, _RELAX, work}
#define OP32_WITH_WORK(mnem, opcode32, isa, work)		\
  {mnem, _TRANSFER,						\
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},		\
      {opcode32, OPCODE_INFO_NONE ()},				\
      0, isa, 0, _RELOC32, _RELAX, work}
#define OP16_OP32_WITH_WORK(mnem, opcode16, isa16, opcode32, isa32, work)  \
  {mnem, _TRANSFER,						\
      {opcode16, OPCODE_INFO_NONE ()},				\
      {opcode32, OPCODE_INFO_NONE ()},				\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, work}
#define DOP16_OP32_WITH_WORK(mnem, opcode16a, opcode16b, isa16, opcode32, isa32, work)  \
  {mnem, _TRANSFER,						\
      {opcode16a, opcode16b},					\
      {opcode32, OPCODE_INFO_NONE ()},				\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, work}
#define DOP16_DOP32_WITH_WORK(mnem, opcode16a, opcode16b, isa16, opcode32a, opcode32b, isa32, work) \
  {mnem, _TRANSFER,						\
      {opcode16a, opcode16b},					\
      {opcode32a, opcode32b},					\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, work}
#define DOP32_WITH_WORK(mnem, opcode32a, opcode32b, isa, work)	\
  {mnem, _TRANSFER,						\
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},		\
      {opcode32a, opcode32b},					\
      0, isa, 0, _RELOC32, _RELAX, work}

#else /* ifdef BUILD_AS */

#define OP16_WITH_WORK(mnem, opcode16, isa, work)		\
  {mnem, _TRANSFER,						\
      {opcode16, OPCODE_INFO_NONE ()},				\
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},		\
      isa, 0, _RELOC16, 0, _RELAX, NULL}
#define OP32_WITH_WORK(mnem, opcode32, isa, work)  \
  {mnem, _TRANSFER, \
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},		\
      {opcode32, OPCODE_INFO_NONE ()},				\
      0, isa, 0, _RELOC32, _RELAX, NULL}
#define OP16_OP32_WITH_WORK(mnem, opcode16, isa16, opcode32, isa32, work)  \
  {mnem, _TRANSFER, \
      {opcode16, OPCODE_INFO_NONE ()},				\
      {opcode32, OPCODE_INFO_NONE ()},				\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, NULL}
#define DOP16_OP32_WITH_WORK(mnem, opcode16a, opcode16b, isa16, opcode32, isa32, work) \
  {mnem, _TRANSFER,						\
      {opcode16a, opcode16b},					\
      {opcode32, OPCODE_INFO_NONE ()},				\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, NULL}
#define DOP16_DOP32_WITH_WORK(mnem, opcode16a, opcode16b, isa16, opcode32a, opcode32b, isa32, work)  \
  {mnem, _TRANSFER,						\
      {opcode16a, opcode16b},					\
      {opcode32a, opcode32b},					\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, NULL}
#define DOP32_WITH_WORK(mnem, opcode32a, opcode32b, isa, work)  \
  {mnem, _TRANSFER,						\
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},		\
      {opcode32a, opcode32b},					\
      0, isa, 0, _RELOC32, _RELAX, NULL}

#endif  /* ifdef BUILD_AS */

#define DOP16(mnem, opcode16_1, opcode16_2, isa)		\
  {mnem, _TRANSFER,						\
      {opcode16_1, opcode16_2},					\
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},		\
      isa, 0, _RELOC16, 0, _RELAX, NULL}
#define OP32(mnem, opcode32, isa)				\
  {mnem, _TRANSFER,						\
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},		\
      {opcode32, OPCODE_INFO_NONE ()},				\
      0, isa, 0, _RELOC32, _RELAX, NULL}
#define DOP32(mnem, opcode32a, opcode32b, isa)			\
  {mnem, _TRANSFER,						\
      {OPCODE_INFO_NONE (), OPCODE_INFO_NONE ()},		\
      {opcode32a, opcode32b},					\
       0, isa, 0, _RELOC32, _RELAX, NULL}
#define OP16_OP32(mnem, opcode16, isa16, opcode32, isa32)	\
  {mnem, _TRANSFER,						\
      {opcode16, OPCODE_INFO_NONE ()},				\
      {opcode32, OPCODE_INFO_NONE ()},				\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, NULL}
#define DOP16_OP32(mnem, opcode16a, opcode16b, isa16, opcode32, isa32) \
  {mnem, _TRANSFER,						\
      {opcode16a, opcode16b},					\
      {opcode32, OPCODE_INFO_NONE ()},				\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, NULL}
#define OP16_DOP32(mnem, opcode16, isa16, opcode32a, opcode32b, isa32) \
  {mnem, _TRANSFER,						\
      {opcode16, OPCODE_INFO_NONE ()},				\
      {opcode32a, opcode32b},					\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, NULL}
#define DOP16_DOP32(mnem, opcode16a, opcode16b, isa16, opcode32a, opcode32b, isa32) \
  {mnem, _TRANSFER,						\
      {opcode16a, opcode16b},					\
      {opcode32a, opcode32b},					\
      isa16, isa32, _RELOC16, _RELOC32, _RELAX, NULL}


/* Register names and numbers.  */
#define V1_REG_SP              0
#define V1_REG_LR             15

struct psrbit
{
  int value;
  int isa;
  const char *name;
};

const struct psrbit cskyv1_psr_bits[] =
{
  {1,    0, "ie"},
  {2,    0, "fe"},
  {4,    0, "ee"},
  {8,    0, "af"},
  {0, 0, NULL},
};

const struct psrbit cskyv2_psr_bits[] =
{
  {8, 0, "ee"},
  {4, 0, "ie"},
  {2, 0, "fe"},
  {1, 0, "af"},
  {0x10, CSKY_ISA_TRUST, "sie"},
  {0, 0, NULL},
};

#define GENERAL_REG_BANK      0x80000000
#define REG_SUPPORT_ALL 0xffffffff

/* CSKY register description.  */
struct csky_reg_def
{
  /* The group number for control registers,
     and set the bank of genaral registers to a special number.  */
  int bank;
  int regno;
  /* The name displayed by serial number.  */
  const char *name;
  /* The name displayed by ABI infomation,
     used when objdump add option -Mabi-names.  */
  const char *abi_name;
  /* The flags indicate which arches support the register.  */
  int arch_flag;
  /* Some registers depend on special features.  */
  char *features;
};

/* Arch flag.  */
#define ASH(a) (1 << CSKY_ARCH_##a)

/* All arches exclued 801.  */
#define REG_SUPPORT_A   (REG_SUPPORT_ALL & ~ASH(801))

/* All arches exclued 801 and 802.  */
#define REG_SUPPORT_B   (REG_SUPPORT_ALL & ~(ASH(801) | ASH(802)))

/* All arches exclued 801, 802, 803, 805.*/
#define REG_SUPPORT_C   (REG_SUPPORT_ALL & ~(ASH(801)			\
		       	| ASH(802) | ASH(803) | ASH(805)))

/* All arches exclued 801, 802, 803, 805, 807, 810.  */
#define REG_SUPPORT_D   (REG_SUPPORT_C & ~(ASH(807) | ASH(810)))

/* All arches exclued 807, 810, 860.  */
#define REG_SUPPORT_E   (REG_SUPPORT_ALL & ~(ASH(807) | ASH(810) |	\
		       	ASH(860)))

/* C-SKY V1 general registers table.  */
static struct csky_reg_def csky_abiv1_general_regs[] = 
{
#define DECLARE_REG(regno, abi_name, support)		\
  {GENERAL_REG_BANK, regno, "r"#regno, abi_name, support, NULL}

  DECLARE_REG (0, "sp", REG_SUPPORT_ALL),
  DECLARE_REG (1, NULL, REG_SUPPORT_ALL),
  DECLARE_REG (2, "a0", REG_SUPPORT_ALL),
  DECLARE_REG (3, "a1", REG_SUPPORT_ALL),
  DECLARE_REG (4, "a2", REG_SUPPORT_ALL),
  DECLARE_REG (5, "a3", REG_SUPPORT_ALL),
  DECLARE_REG (6, "a4", REG_SUPPORT_ALL),
  DECLARE_REG (7, "a5", REG_SUPPORT_ALL),
  DECLARE_REG (8, "fp", REG_SUPPORT_ALL),
  DECLARE_REG (8, "l0", REG_SUPPORT_ALL),
  DECLARE_REG (9, "l1", REG_SUPPORT_ALL),
  DECLARE_REG (10, "l2", REG_SUPPORT_ALL),
  DECLARE_REG (11, "l3", REG_SUPPORT_ALL),
  DECLARE_REG (12, "l4", REG_SUPPORT_ALL),
  DECLARE_REG (13, "l5", REG_SUPPORT_ALL),
  DECLARE_REG (14, "gb", REG_SUPPORT_ALL),
  DECLARE_REG (15, "lr", REG_SUPPORT_ALL),
#undef DECLARE_REG
  {-1, -1, NULL, NULL, 0, NULL},
};

/* C-SKY V1 control registers table.  */
static struct csky_reg_def csky_abiv1_control_regs[] = 
{
#define DECLARE_REG(regno, abi_name, support)		\
  {0, regno, "cr"#regno, abi_name, support, NULL}

  DECLARE_REG (0, "psr", REG_SUPPORT_ALL),
  DECLARE_REG (1, "vbr", REG_SUPPORT_ALL),
  DECLARE_REG (2, "epsr", REG_SUPPORT_ALL),
  DECLARE_REG (3, "fpsr", REG_SUPPORT_ALL),
  DECLARE_REG (4, "epc", REG_SUPPORT_ALL),
  DECLARE_REG (5, "fpc", REG_SUPPORT_ALL),
  DECLARE_REG (6, "ss0", REG_SUPPORT_ALL),
  DECLARE_REG (7, "ss1", REG_SUPPORT_ALL),
  DECLARE_REG (8, "ss2", REG_SUPPORT_ALL),
  DECLARE_REG (9, "ss3", REG_SUPPORT_ALL),
  DECLARE_REG (10, "ss4", REG_SUPPORT_ALL),
  DECLARE_REG (11, "gcr", REG_SUPPORT_ALL),
  DECLARE_REG (12, "gsr", REG_SUPPORT_ALL),
  DECLARE_REG (13, "cpid", REG_SUPPORT_ALL),
  DECLARE_REG (14, "dcsr", REG_SUPPORT_ALL),
  DECLARE_REG (15, "cwr", REG_SUPPORT_ALL),
  DECLARE_REG (16, NULL, REG_SUPPORT_ALL),
  DECLARE_REG (17, "cfr", REG_SUPPORT_ALL),
  DECLARE_REG (18, "ccr", REG_SUPPORT_ALL),
  DECLARE_REG (19, "capr", REG_SUPPORT_ALL),
  DECLARE_REG (20, "pacr", REG_SUPPORT_ALL),
  DECLARE_REG (21, "prsr", REG_SUPPORT_ALL),
  DECLARE_REG (22, "mir", REG_SUPPORT_ALL),
  DECLARE_REG (23, "mrr", REG_SUPPORT_ALL),
  DECLARE_REG (24, "mel0", REG_SUPPORT_ALL),
  DECLARE_REG (25, "mel1", REG_SUPPORT_ALL),
  DECLARE_REG (26, "meh", REG_SUPPORT_ALL),
  DECLARE_REG (27, "mcr", REG_SUPPORT_ALL),
  DECLARE_REG (28, "mpr", REG_SUPPORT_ALL),
  DECLARE_REG (29, "mwr", REG_SUPPORT_ALL),
  DECLARE_REG (30, "mcir", REG_SUPPORT_ALL),
#undef DECLARE_REG
  {-1, -1, NULL, NULL, 0, NULL},
};

/* C-SKY V2 general registers table.  */
static struct csky_reg_def csky_abiv2_general_regs[] = 
{
#ifdef DECLARE_REG
#undef DECLARE_REG
#endif
#define DECLARE_REG(regno, abi_name, support)		\
  {GENERAL_REG_BANK, regno, "r"#regno, abi_name, support, NULL}

  DECLARE_REG (0, "a0", REG_SUPPORT_ALL),
  DECLARE_REG (1, "a1", REG_SUPPORT_ALL),
  DECLARE_REG (2, "a2", REG_SUPPORT_ALL),
  DECLARE_REG (3, "a3", REG_SUPPORT_ALL),
  DECLARE_REG (4, "l0", REG_SUPPORT_ALL),
  DECLARE_REG (5, "l1", REG_SUPPORT_ALL),
  DECLARE_REG (6, "l2", REG_SUPPORT_ALL),
  DECLARE_REG (7, "l3", REG_SUPPORT_ALL),
  DECLARE_REG (8, "l4", REG_SUPPORT_ALL),
  DECLARE_REG (9, "l5", REG_SUPPORT_A),
  DECLARE_REG (10, "l6", REG_SUPPORT_A),
  DECLARE_REG (11, "l7", REG_SUPPORT_A),
  DECLARE_REG (12, "t0", REG_SUPPORT_A),
  DECLARE_REG (13, "t1", REG_SUPPORT_ALL),
  DECLARE_REG (14, "sp", REG_SUPPORT_ALL),
  DECLARE_REG (15, "lr", REG_SUPPORT_ALL),
  DECLARE_REG (16, "l8", REG_SUPPORT_B),
  DECLARE_REG (17, "l9", REG_SUPPORT_B),
  DECLARE_REG (18, "t2", REG_SUPPORT_B),
  DECLARE_REG (19, "t3", REG_SUPPORT_B),
  DECLARE_REG (20, "t4", REG_SUPPORT_B),
  DECLARE_REG (21, "t5", REG_SUPPORT_B),
  DECLARE_REG (22, "t6", REG_SUPPORT_B),
  DECLARE_REG (23, "t7", REG_SUPPORT_B),
  DECLARE_REG (24, "t8", REG_SUPPORT_B),
  DECLARE_REG (25, "t9", REG_SUPPORT_B),
  DECLARE_REG (26, NULL, REG_SUPPORT_B),
  DECLARE_REG (27, NULL, REG_SUPPORT_B),
  DECLARE_REG (28, "gb", REG_SUPPORT_B),
  DECLARE_REG (28, "rgb", REG_SUPPORT_B),
  DECLARE_REG (28, "rdb", REG_SUPPORT_B),
  DECLARE_REG (29, "tb", REG_SUPPORT_B),
  DECLARE_REG (29, "rtb", REG_SUPPORT_B),
  DECLARE_REG (30, "svbr", REG_SUPPORT_A),
  DECLARE_REG (31, "tls", REG_SUPPORT_B),

  /* The followings JAVA/BCTM's features.  */
  DECLARE_REG (23, "fp", REG_SUPPORT_ALL),
  DECLARE_REG (24, "top", REG_SUPPORT_ALL),
  DECLARE_REG (25, "bsp", REG_SUPPORT_ALL),

  {-1, -1, NULL, NULL, 0, NULL},
};

/* C-SKY V2 control registers table.  */
static struct csky_reg_def csky_abiv2_control_regs[] = 
{

#ifdef DECLARE_REG
#undef DECLARE_REG
#endif
  /* Bank0.  */
#define DECLARE_REG(regno, abi_name)		\
  {0, regno, "cr<"#regno", 0>", abi_name, REG_SUPPORT_ALL, NULL}
  DECLARE_REG (0, "psr"),
  DECLARE_REG (1, "vbr"),
  DECLARE_REG (2, "epsr"),
  DECLARE_REG (3, "fpsr"),
  DECLARE_REG (4, "epc"),
  DECLARE_REG (5, "fpc"),
  DECLARE_REG (6, "ss0"),
  DECLARE_REG (7, "ss1"),
  DECLARE_REG (8, "ss2"),
  DECLARE_REG (9, "ss3"),
  DECLARE_REG (10, "ss4"),
  DECLARE_REG (11, "gcr"),
  DECLARE_REG (12, "gsr"),
  DECLARE_REG (13, "cpid"),
  DECLARE_REG (14, "dcsr"),
  DECLARE_REG (15, NULL),
  DECLARE_REG (16, NULL),
  DECLARE_REG (17, "cfr"),
  DECLARE_REG (18, "ccr"),
  DECLARE_REG (19, "capr"),
  DECLARE_REG (20, "pacr"),
  DECLARE_REG (21, "prsr"),
  DECLARE_REG (22, "cir"),
  DECLARE_REG (23, "ccr2"),
  DECLARE_REG (24, NULL),
  DECLARE_REG (25, "cer2"),
  DECLARE_REG (26, NULL),
  DECLARE_REG (27, NULL),
  DECLARE_REG (28, "rvbr"),
  DECLARE_REG (29, "rmr"),
  DECLARE_REG (30, "mpid"),

#undef DECLARE_REG
#define DECLARE_REG(regno, abi_name, support)		\
  {0, regno, "cr<"#regno", 0>", abi_name, support, NULL}
  DECLARE_REG (31, "chr", REG_SUPPORT_E),
  DECLARE_REG (31, "hint", REG_SUPPORT_C),

  /* Bank1.  */
#undef DECLARE_REG
#define DECLARE_REG(regno, abi_name)		\
  {1, regno, "cr<"#regno", 1>", abi_name, REG_SUPPORT_ALL, NULL}

  DECLARE_REG (14, "usp"),
  DECLARE_REG (26, "cindex"),
  DECLARE_REG (27, "cdata0"),
  DECLARE_REG (28, "cdata1"),
  DECLARE_REG (29, "cdata2"),
  DECLARE_REG (30, "cdata3"),
  DECLARE_REG (31, "cins"),

  /* Bank2.  */
#undef DECLARE_REG
#define DECLARE_REG(regno, abi_name)		\
  {2, regno, "cr<"#regno", 2>", abi_name, REG_SUPPORT_ALL, NULL}

  DECLARE_REG (0, "fid"),
  DECLARE_REG (1, "fcr"),
  DECLARE_REG (2, "fesr"),

  /* Bank3.  */
#undef DECLARE_REG
#define DECLARE_REG(regno, abi_name)		\
  {3, regno, "cr<"#regno", 3>", abi_name, REG_SUPPORT_ALL, NULL}
  DECLARE_REG (8, "dcr"),
  DECLARE_REG (8, "sedcr"),
  DECLARE_REG (9, "pcr"),
  DECLARE_REG (9, "sepcr"),

  /* Bank15.  */
#undef DECLARE_REG
#define DECLARE_REG(regno, abi_name)		\
  {15, regno, "cr<"#regno", 15>", abi_name, REG_SUPPORT_ALL, NULL}

  DECLARE_REG (0, "mir"),
  DECLARE_REG (2, "mel0"),
  DECLARE_REG (3, "mel1"),
  DECLARE_REG (4, "meh"),
  DECLARE_REG (6, "mpr"),
  DECLARE_REG (8, "mcir"),
  DECLARE_REG (28, "mpgd0"),
  DECLARE_REG (29, "mpgd"),
  DECLARE_REG (29, "mpgd1"),
  DECLARE_REG (30, "msa0"),
  DECLARE_REG (31, "msa1"),
#undef DECLARE_REG
  {-1, -1, NULL, NULL, 0, NULL},
};

/* Get register name according to giving parameters,
   IS_ABI controls whether is ABI name or not.  */
static inline const char *
get_register_name (struct csky_reg_def *reg_table,
		   int arch, int bank, int regno, int is_abi)
{
  static char regname[64] = {0};
  unsigned int i = 0;
  while (reg_table[i].name != NULL)
    {
      if (reg_table[i].bank == bank
	  && reg_table[i].regno == regno
	  && (reg_table[i].arch_flag & (1u << (arch & CSKY_ARCH_MASK))))
	{
	  if (is_abi && reg_table[i].abi_name)
	    return reg_table[i].abi_name;
	  else
	    return reg_table[i].name;
	}
      i++;
    }

  if (bank & 0x80000000)
    return "unkown register";

  sprintf (regname, "cr<%d, %d>", regno, bank);

  return regname;
}

/* Get register number according to giving parameters.
   If not found, return -1.  */
static inline int
get_register_number (struct csky_reg_def *reg_table,
		     int arch, char *s, char **end, int *bank)
{
  unsigned int i = 0;
  int len = 0;
  while (reg_table[i].name != NULL)
    {
      len = strlen (reg_table[i].name);
      if ((strncasecmp (reg_table[i].name, s, len) == 0)
	  && !(ISDIGIT (s[len]))
	  && (reg_table[i].arch_flag & (1u << (arch & CSKY_ARCH_MASK))))
	{
	  *end = s + len;
	  *bank = reg_table[i].bank;
	  return reg_table[i].regno;
	}

      if (reg_table[i].abi_name == NULL)
	{
	  i++;
	  continue;
	}

      len = strlen (reg_table[i].abi_name);
      if ((strncasecmp (reg_table[i].abi_name, s, len) == 0)
	  && !(ISALNUM (s[len]))
	  && (reg_table[i].arch_flag & (1u << (arch & CSKY_ARCH_MASK))))
	{
	  *end = s + len;
	  *bank = reg_table[i].bank;
	  return reg_table[i].regno;
	}
      i++;
    }
  return -1;
}

/* Return general register's name.  */
static inline const char *
csky_get_general_reg_name (int arch, int regno, int is_abi)
{
  struct csky_reg_def *reg_table;

  if (IS_CSKY_ARCH_V1 (arch))
    reg_table = csky_abiv1_general_regs;
  else
    reg_table = csky_abiv2_general_regs;

  return get_register_name (reg_table, arch, GENERAL_REG_BANK, regno, is_abi);
}

/* Return general register's number.  */
static inline int
csky_get_general_regno (int arch, char *s, char **end)
{
  struct csky_reg_def *reg_table;
  int bank = 0;

  if (IS_CSKY_ARCH_V1 (arch))
    reg_table = csky_abiv1_general_regs;
  else
    reg_table = csky_abiv2_general_regs;

  return get_register_number (reg_table, arch, s, end, &bank);
}

/* Return control register's name.  */
static inline const char *
csky_get_control_reg_name (int arch, int bank, int regno, int is_abi)
{
  struct csky_reg_def *reg_table;

  if (IS_CSKY_ARCH_V1 (arch))
    reg_table = csky_abiv1_control_regs;
  else
    reg_table = csky_abiv2_control_regs;

  return get_register_name (reg_table, arch, bank, regno, is_abi);
}

/* Return control register's number.  */
static inline int
csky_get_control_regno (int arch, char *s, char **end, int *bank)
{
  struct csky_reg_def *reg_table;

  if (IS_CSKY_ARCH_V1 (arch))
    reg_table = csky_abiv1_control_regs;
  else
    reg_table = csky_abiv2_control_regs;

  return get_register_number (reg_table, arch, s, end, bank);
}

/* C-SKY V1 opcodes.  */
const struct csky_opcode csky_v1_opcodes[] =
{
#define _TRANSFER   0
#define _RELOC16    0
#define _RELOC32    0
#define _RELAX      0
  OP16 ("bkpt",
	OPCODE_INFO0 (0x0000),
	CSKYV1_ISA_E1),
  OP16 ("sync",
	OPCODE_INFO0 (0x0001),
	CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER   2
  OP16 ("rfi",
	OPCODE_INFO0 (0x0003),
	CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER   0
  OP16 ("stop",
	OPCODE_INFO0 (0x0004),
	CSKYV1_ISA_E1),
  OP16 ("wait",
	OPCODE_INFO0 (0x0005),
	CSKYV1_ISA_E1),
  OP16 ("doze",
	OPCODE_INFO0 (0x0006),
	CSKYV1_ISA_E1),
  OP16 ("idly4",
	OPCODE_INFO0 (0x0007),
	CSKYV1_ISA_E1),
  OP16 ("trap",
	OPCODE_INFO1 (0x0008,
		      (0_1, IMM2b, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("mvtc",
	OPCODE_INFO0 (0x000c),
	CSKY_ISA_DSP),
  OP16 ("cprc",
	OPCODE_INFO0 (0x000d),
	CSKY_ISA_CP),
  OP16 ("cpseti",
	OPCODE_INFO1 (0x0010,
		      (0_3, CPIDX, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_CP),
  OP16 ("mvc",
	OPCODE_INFO1 (0x0020,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("mvcv",
	OPCODE_INFO1 (0x0030,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("ldq",
	OPCODE_INFO2 (0x0040,
		      (NONE, REGr4_r7, OPRND_SHIFT_0_BIT),
		      (0_3, REGnr4_r7, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("stq",
	OPCODE_INFO2 (0x0050,
		      (NONE, REGr4_r7, OPRND_SHIFT_0_BIT),
		      (0_3, REGnr4_r7, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("ldm",
	OPCODE_INFO2 (0x0060,
		      (0_3, REGLIST_DASH, OPRND_SHIFT_0_BIT),
		      (NONE, REGbsp, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("stm",
	OPCODE_INFO2 (0x0070,
		      (0_3, REGLIST_DASH, OPRND_SHIFT_0_BIT),
		      (NONE, REGbsp, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("dect",
	 OPCODE_INFO3 (0x0080,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0080,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("decf",
	 OPCODE_INFO3 (0x0090,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0090,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("inct",
	 OPCODE_INFO3 (0x00a0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x00a0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("incf",
	 OPCODE_INFO3 (0x00b0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x00b0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER 2
  OP16 ("jmp",
	OPCODE_INFO1 (0x00c0,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER 0
  OP16 ("jsr",
	OPCODE_INFO1 (0x00d0,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("ff1",
	 OPCODE_INFO2 (0x00e0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x00e0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("brev",
	 OPCODE_INFO2 (0x00f0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x00f0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("xtrb3",
	 OPCODE_INFO2 (0x0100,
		       (NONE, REG_r1a, OPRND_SHIFT_0_BIT),
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0100,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("xtrb2",
	 OPCODE_INFO2 (0x0110,
		       (NONE, REG_r1a, OPRND_SHIFT_0_BIT),
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0110,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("xtrb1",
	 OPCODE_INFO2 (0x0120,
		       (NONE, REG_r1a, OPRND_SHIFT_0_BIT),
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0120,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("xtrb0",
	 OPCODE_INFO2 (0x0130,
		       (NONE, REG_r1a, OPRND_SHIFT_0_BIT),
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0130,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("zextb",
	 OPCODE_INFO2 (0x0140,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0140,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("sextb",
	 OPCODE_INFO2 (0x0150,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0150,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("zexth",
	 OPCODE_INFO2 (0x0160,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0160,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("sexth",
	 OPCODE_INFO2 (0x0170,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0170,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("declt",
	 OPCODE_INFO3 (0x0180,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x0180,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("tstnbz",
	OPCODE_INFO1 (0x0190,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("decgt",
	 OPCODE_INFO3 (0x01a0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x01a0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("decne",
	 OPCODE_INFO3 (0x01b0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x01b0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("clrt",
	OPCODE_INFO1 (0x01c0,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("clrf",
	OPCODE_INFO1 (0x01d0,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("abs",
	 OPCODE_INFO2 (0x01e0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x01e0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("not",
	 OPCODE_INFO2 (0x01f0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x01f0,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("movt",
	OPCODE_INFO2 (0x0200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("mult",
	 OPCODE_INFO3 (0x0300,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x0300,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("mac",
	OPCODE_INFO2 (0x0400,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC),
  DOP16 ("subu",
	 OPCODE_INFO3 (0x0500,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x0500,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("sub",
	 OPCODE_INFO3 (0x0500,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x0500,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("addc",
	 OPCODE_INFO3 (0x0600,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x0600,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("subc",
	 OPCODE_INFO3 (0x0700,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x0700,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("cprgr",
	OPCODE_INFO2 (0x0800,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, CPREG, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_CP),
  OP16 ("movf",
	OPCODE_INFO2 (0x0a00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("lsr",
	 OPCODE_INFO3 (0x0b00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x0b00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("cmphs",
	OPCODE_INFO2 (0x0c00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("cmplt",
	OPCODE_INFO2 (0x0d00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("tst",
	OPCODE_INFO2 (0x0e00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("cmpne",
	OPCODE_INFO2 (0x0f00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("mfcr",
	OPCODE_INFO2 (0x1000,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, CTRLREG, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("psrclr",
	OPCODE_INFO_LIST (0x11f0,
			  (0_2, PSR_BITS_LIST, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("psrset",
	OPCODE_INFO_LIST (0x11f8,
			  (0_2, PSR_BITS_LIST, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("mov",
	OPCODE_INFO2 (0x1200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("bgenr",
	OPCODE_INFO2 (0x1300,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("rsub",
	 OPCODE_INFO3 (0x1400,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1400,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("ixw",
	 OPCODE_INFO3 (0x1500,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1500,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("and",
	 OPCODE_INFO3 (0x1600,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1600,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("xor",
	 OPCODE_INFO3 (0x1700,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1700,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("mtcr",
	OPCODE_INFO2 (0x1800,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, CTRLREG, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("asr",
	 OPCODE_INFO3 (0x1a00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1a00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("lsl",
	 OPCODE_INFO3 (0x1b00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1b00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("addu",
	 OPCODE_INFO3 (0x1c00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1c00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("add",
	OPCODE_INFO2 (0x1c00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("ixh",
	 OPCODE_INFO3 (0x1d00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1d00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("or",
	 OPCODE_INFO3 (0x1e00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1e00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("andn",
	 OPCODE_INFO3 (0x1f00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x1f00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("addi",
	 OPCODE_INFO3 (0x2000,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, OIMM5b, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x2000,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, OIMM5b, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("cmplti",
	OPCODE_INFO2 (0x2200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, OIMM5b, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("subi",
	 OPCODE_INFO3 (0x2400,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, OIMM5b, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x2400,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, OIMM5b, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("cpwgr",
	OPCODE_INFO2 (0x2600,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, CPREG, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_CP),
  DOP16 ("rsubi",
	 OPCODE_INFO3 (0x2800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x2800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("cmpnei",
	OPCODE_INFO2 (0x2a00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("bmaski",
	OPCODE_INFO2 (0x2c00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b_BMASKI, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("divu",
	 OPCODE_INFO3 (0x2c10,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, REG_r1b, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x2c10,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, REG_r1b, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("mflos",
	OPCODE_INFO1 (0x2c20,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC_DSP),
  OP16 ("mfhis",
	OPCODE_INFO1 (0x2c30,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC_DSP),
  OP16 ("mtlo",
	OPCODE_INFO1 (0x2c40,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC_DSP),
  OP16 ("mthi",
	OPCODE_INFO1 (0x2c50,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC_DSP),
  OP16 ("mflo",
	OPCODE_INFO1 (0x2c60,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC_DSP),
  OP16 ("mfhi",
	OPCODE_INFO1 (0x2c70,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC_DSP),
  DOP16 ("andi",
	 OPCODE_INFO3 (0x2e00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x2e00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("bclri",
	 OPCODE_INFO3 (0x3000,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x3000,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("bgeni",
	OPCODE_INFO2 (0x3200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b_7_31, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("cpwir",
	OPCODE_INFO1 (0x3200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_CP),
  DOP16 ("divs",
	 OPCODE_INFO3 (0x3210,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, REG_r1b, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x3210,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, REG_r1b, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("cprsr",
	OPCODE_INFO1 (0x3220,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_CP),
  OP16 ("cpwsr",
	OPCODE_INFO1 (0x3230,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_CP),
  DOP16 ("bseti",
	 OPCODE_INFO3 (0x3400,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x3400,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("btsti",
	OPCODE_INFO2 (0x3600,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("rotli",
	 OPCODE_INFO3 (0x3800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b_1_31, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x3800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b_1_31, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("xsr",
	 OPCODE_INFO3 (0x3800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x3800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("asrc",
	 OPCODE_INFO3 (0x3a00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x3a00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("asri",
	 OPCODE_INFO3 (0x3a00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b_1_31, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x3a00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b_1_31, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("lslc",
	 OPCODE_INFO3 (0x3c00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x3c00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("lsli",
	 OPCODE_INFO3 (0x3c00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b_1_31, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x3c00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b_1_31, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("lsrc",
	 OPCODE_INFO3 (0x3e00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO1 (0x3e00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("lsri",
	 OPCODE_INFO3 (0x3e00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b_1_31, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x3e00,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_8, IMM5b_1_31, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("ldex",
	SOPCODE_INFO2 (0x4000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				      (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	CSKY_ISA_MP),
  OP16 ("ldex.w",
	SOPCODE_INFO2 (0x4000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				      (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	CSKY_ISA_MP),
  OP16 ("ldwex",
	SOPCODE_INFO2 (0x4000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				      (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	CSKY_ISA_MP),
  OP16 ("stex",
	SOPCODE_INFO2 (0x5000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				      (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	CSKY_ISA_MP),
  OP16 ("stex.w",
	SOPCODE_INFO2 (0x5000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				      (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	CSKY_ISA_MP),
  OP16 ("stwex",
	SOPCODE_INFO2 (0x5000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				      (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	CSKY_ISA_MP),
  OP16 ("omflip0",
	OPCODE_INFO2 (0x4000,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC),
  OP16 ("omflip1",
	OPCODE_INFO2 (0x4100,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC),
  OP16 ("omflip2",
	OPCODE_INFO2 (0x4200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC),
  OP16 ("omflip3",
	OPCODE_INFO2 (0x4300,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_MAC),
  OP16 ("muls",
	OPCODE_INFO2 (0x5000,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("mulsa",
	OPCODE_INFO2 (0x5100,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("mulss",
	OPCODE_INFO2 (0x5200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("mulu",
	OPCODE_INFO2 (0x5400,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("mulua",
	OPCODE_INFO2 (0x5500,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("mulus",
	OPCODE_INFO2 (0x5600,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("vmulsh",
	OPCODE_INFO2 (0x5800,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("vmulsha",
	OPCODE_INFO2 (0x5900,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("vmulshs",
	OPCODE_INFO2 (0x5a00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("vmulsw",
	OPCODE_INFO2 (0x5c00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("vmulswa",
	OPCODE_INFO2 (0x5d00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("vmulsws",
	OPCODE_INFO2 (0x5e00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("movi",
	OPCODE_INFO2 (0x6000,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_10, IMM7b, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  DOP16 ("mulsh",
	 OPCODE_INFO3 (0x6800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x6800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("mulsh.h",
	 OPCODE_INFO3 (0x6800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 OPCODE_INFO2 (0x6800,
		       (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		       (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  OP16 ("mulsha",
	OPCODE_INFO2 (0x6900,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("mulshs",
	OPCODE_INFO2 (0x6a00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("cprcr",
	OPCODE_INFO2 (0x6b00,
		      (0_2, GREG0_7, OPRND_SHIFT_0_BIT),
		      (3_7, CPCREG, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_CP),
  OP16 ("mulsw",
	OPCODE_INFO2 (0x6c00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("mulswa",
	OPCODE_INFO2 (0x6d00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("mulsws",
	OPCODE_INFO2 (0x6e00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_7, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_DSP),
  OP16 ("cpwcr",
	OPCODE_INFO2 (0x6f00,
		      (0_2, GREG0_7, OPRND_SHIFT_0_BIT),
		      (3_7, CPCREG, OPRND_SHIFT_0_BIT)),
	CSKY_ISA_CP),
#undef _RELOC16
#define _RELOC16 BFD_RELOC_CKCORE_PCREL_IMM8BY4
#undef _TRANSFER
#define _TRANSFER 1
  OP16 ("jmpi",
	OPCODE_INFO1 (0x7000,
		      (0_7, OFF8b, OPRND_SHIFT_2_BIT)),
	CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER 0
  OP16 ("jsri",
	OPCODE_INFO1 (0x7f00,
		      (0_7, OFF8b, OPRND_SHIFT_2_BIT)),
	CSKYV1_ISA_E1),
  OP16_WITH_WORK ("lrw",
		  OPCODE_INFO2 (0x7000,
				(8_11, REGnsplr, OPRND_SHIFT_0_BIT),
				(0_7, CONSTANT, OPRND_SHIFT_2_BIT)),
		  CSKYV1_ISA_E1,
		  v1_work_lrw),
#undef _RELOC16
#define _RELOC16 0
  DOP16 ("ld.w",
	 SOPCODE_INFO2 (0x8000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_2_BIT))),
	 OPCODE_INFO2 (0x8000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("ldw",
	 SOPCODE_INFO2 (0x8000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_2_BIT))),
	 OPCODE_INFO2 (0x8000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("ld",
	 SOPCODE_INFO2 (0x8000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_2_BIT))),
	 OPCODE_INFO2 (0x8000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("st.w",
	 SOPCODE_INFO2 (0x9000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_2_BIT))),
	 OPCODE_INFO2 (0x9000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("stw",
	 SOPCODE_INFO2 (0x9000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_2_BIT))),
	 OPCODE_INFO2 (0x9000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("st",
	 SOPCODE_INFO2 (0x9000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_2_BIT))),
	 OPCODE_INFO2 (0x9000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("ld.b",
	 SOPCODE_INFO2 (0xa000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	 OPCODE_INFO2 (0xa000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("ldb",
	 SOPCODE_INFO2 (0xa000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	 OPCODE_INFO2 (0xa000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("st.b",
	 SOPCODE_INFO2 (0xb000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	 OPCODE_INFO2 (0xb000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("stb",
	 SOPCODE_INFO2 (0xb000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_0_BIT))),
	 OPCODE_INFO2 (0xb000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("ld.h",
	 SOPCODE_INFO2 (0xc000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_1_BIT))),
	 OPCODE_INFO2 (0xc000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("ldh",
	 SOPCODE_INFO2 (0xc000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_1_BIT))),
	 OPCODE_INFO2 (0xc000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("st.h",
	 SOPCODE_INFO2 (0xd000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_1_BIT))),
	 OPCODE_INFO2 (0xd000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),
  DOP16 ("sth",
	 SOPCODE_INFO2 (0xd000,
			(8_11, GREG0_15, OPRND_SHIFT_0_BIT),
			BRACKET_OPRND ((0_3, GREG0_15, OPRND_SHIFT_0_BIT),
				       (4_7, IMM_LDST, OPRND_SHIFT_1_BIT))),
	 OPCODE_INFO2 (0xd000,
		       (8_11, GREG0_15, OPRND_SHIFT_0_BIT),
		       (0_3, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	 CSKYV1_ISA_E1),

#undef _RELOC16
#define _RELOC16    BFD_RELOC_CKCORE_PCREL_IMM11BY2
  OP16 ("bt",
	OPCODE_INFO1 (0xe000,
		      (0_10, OFF11b, OPRND_SHIFT_1_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("bf",
	OPCODE_INFO1 (0xe800,
		      (0_10, OFF11b, OPRND_SHIFT_1_BIT)),
	CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER 1
  OP16 ("br",
	OPCODE_INFO1 (0xf000,
		      (0_10, OFF11b, OPRND_SHIFT_1_BIT)),
	CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER 0
  OP16 ("bsr",
	OPCODE_INFO1 (0xf800,
		      (0_10, OFF11b, OPRND_SHIFT_1_BIT)),
	CSKYV1_ISA_E1),
#undef _RELOC16
#define _RELOC16    0

#undef _RELAX
#define _RELAX 1
  OP16 ("jbt",
	OPCODE_INFO1 (0xe000,
		      (0_10, JBTF, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("jbf",
	OPCODE_INFO1 (0xe800,
		      (0_10, JBTF, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER 1
  OP16 ("jbr",
	OPCODE_INFO1 (0xf000,
		      (0_10, JBR, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER 0
#undef _RELAX
#define _RELAX 0

  OP16_WITH_WORK ("jbsr",
		  OPCODE_INFO1 (0xf800,
				(0_10, JBSR, OPRND_SHIFT_0_BIT)),
		  CSKYV1_ISA_E1,
		  v1_work_jbsr),

  /* The following are aliases for other instructions.  */
  /* rts -> jmp r15.  */
#undef _TRANSFER
#define _TRANSFER 2
  OP16 ("rts",
	OPCODE_INFO0 (0x00CF),
	CSKYV1_ISA_E1),
  OP16 ("rte",
	OPCODE_INFO0 (0x0002),
	CSKYV1_ISA_E1),
  OP16 ("rfe",
	OPCODE_INFO0 (0x0002),
	CSKYV1_ISA_E1),
#undef _TRANSFER
#define _TRANSFER 0

  /* cmphs r0,r0 */
  OP16 ("setc",
	OPCODE_INFO0 (0x0c00),
	CSKYV1_ISA_E1),
  /* cmpne r0,r0 */
  OP16 ("clrc",
	OPCODE_INFO0 (0x0f00),
	CSKYV1_ISA_E1),
  /* cmplti rd,1 */
  OP16 ("tstle",
	OPCODE_INFO1 (0x2200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  /* cmplei rd,X -> cmplti rd,X+1 */
  OP16 ("cmplei",
	OPCODE_INFO2 (0x2200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  /* rsubi rd,0 */
  OP16 ("neg",
	OPCODE_INFO1 (0x2800,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  /* cmpnei rd,0.  */
  OP16 ("tstne",
	OPCODE_INFO1 (0x2a00,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  /* btsti rx,31.  */
  OP16 ("tstlt",
	OPCODE_INFO1 (0x37f0,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  /* bclri rx,log2(imm).  */
  OP16 ("mclri",
	OPCODE_INFO2 (0x3000,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b_POWER, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  /* bgeni rx,log2(imm).  */
  OP16 ("mgeni",
	OPCODE_INFO2 (0x3200,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b_7_31_POWER, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  /* bseti rx,log2(imm).  */
  OP16 ("mseti",
	OPCODE_INFO2 (0x3400,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b_POWER, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  /* btsti rx,log2(imm).  */
  OP16 ("mtsti",
	OPCODE_INFO2 (0x3600,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b_POWER, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("rori",
	OPCODE_INFO2 (0x3800,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b_RORI, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  OP16 ("rotri",
	OPCODE_INFO2 (0x3800,
		      (0_3, GREG0_15, OPRND_SHIFT_0_BIT),
		      (4_8, IMM5b_RORI, OPRND_SHIFT_0_BIT)),
	CSKYV1_ISA_E1),
  /* mov r0, r0.  */
  OP16 ("nop",
	OPCODE_INFO0 (0x1200),
	CSKYV1_ISA_E1),

  /* Float instruction with work.  */
  OP16_WITH_WORK ("fabss",
		  OPCODE_INFO3 (0xffe04400,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnegs",
		  OPCODE_INFO3 (0xffe04c00,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fsqrts",
		  OPCODE_INFO3 (0xffe05400,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("frecips",
		  OPCODE_INFO3 (0xffe05c00,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fadds",
		  OPCODE_INFO4 (0xffe38000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(10_14, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fsubs",
		  OPCODE_INFO4 (0xffe48000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(10_14, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1, v1_work_fpu_fo),
  OP16_WITH_WORK ("fmacs",
		  OPCODE_INFO4 (0xffe58000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(10_14, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmscs",
		  OPCODE_INFO4 (0xffe68000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(10_14, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmuls",
		  OPCODE_INFO4 (0xffe78000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(10_14, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fdivs",
		  OPCODE_INFO4 (0xffe88000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(10_14, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnmacs",
		  OPCODE_INFO4 (0xffe98000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(10_14, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnmscs",
		  OPCODE_INFO4 (0xffea8000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(10_14, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnmuls",
		  OPCODE_INFO4 (0xffeb8000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(10_14, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fabsd",
		  OPCODE_INFO3 (0xffe04000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnegd",
		  OPCODE_INFO3 (0xffe04800,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fsqrtd",
		  OPCODE_INFO3 (0xffe05000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("frecipd",
		  OPCODE_INFO3 (0xffe05800,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("faddd",
		  OPCODE_INFO4 (0xffe30000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fsubd",
		  OPCODE_INFO4 (0xffe40000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmacd",
		  OPCODE_INFO4 (0xffe50000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmscd",
		  OPCODE_INFO4 (0xffe60000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmuld",
		  OPCODE_INFO4 (0xffe70000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fdivd",
		  OPCODE_INFO4 (0xffe80000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnmacd",
		  OPCODE_INFO4 (0xffe90000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnmscd",
		  OPCODE_INFO4 (0xffea0000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnmuld",
		  OPCODE_INFO4 (0xffeb0000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fabsm",
		  OPCODE_INFO3 (0xffe06000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnegm",
		  OPCODE_INFO3 (0xffe06400,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("faddm",
		  OPCODE_INFO4 (0xffec0000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fsubm",
		  OPCODE_INFO4 (0xffec8000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmacm",
		  OPCODE_INFO4 (0xffed8000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmscm",
		  OPCODE_INFO4 (0xffee0000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmulm",
		  OPCODE_INFO4 (0xffed0000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnmacm",
		  OPCODE_INFO4 (0xffee8000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnmscm",
		  OPCODE_INFO4 (0xffef0000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fnmulm",
		  OPCODE_INFO4 (0xffef8000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(10_14, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fcmphsd",
		  OPCODE_INFO3 (0xffe00800,
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpltd",
		  OPCODE_INFO3 (0xffe00c00,
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpned",
		  OPCODE_INFO3 (0xffe01000,
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpuod",
		  OPCODE_INFO3 (0xffe01400,
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmphss",
		  OPCODE_INFO3 (0xffe01800,
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmplts",
		  OPCODE_INFO3 (0xffe01c00,
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpnes",
		  OPCODE_INFO3 (0xffe02000,
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpuos",
		  OPCODE_INFO3 (0xffe02400,
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpzhsd",
		  OPCODE_INFO2 (0xffe00400,
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpzltd",
		  OPCODE_INFO2 (0xffe00480,
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpzned",
		  OPCODE_INFO2 (0xffe00500,
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpzuod",
		  OPCODE_INFO2 (0xffe00580,
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpzhss",
		  OPCODE_INFO2 (0xffe00600,
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpzlts",
		  OPCODE_INFO2 (0xffe00680,
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpznes",
		  OPCODE_INFO2 (0xffe00700,
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fcmpzuos",
		  OPCODE_INFO2 (0xffe00780,
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo_fc),
  OP16_WITH_WORK ("fstod",
		  OPCODE_INFO3 (0xffe02800,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fdtos",
		  OPCODE_INFO3 (0xffe02c00,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fsitos",
		  OPCODE_INFO3 (0xffe03400,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fsitod",
		  OPCODE_INFO3 (0xffe03000,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fuitos",
		  OPCODE_INFO3 (0xffe03c00,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fuitod",
		  OPCODE_INFO3 (0xffe03800,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fstosi",
		  OPCODE_INFO4 (0xffe10000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(13_17, RM, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fdtosi",
		  OPCODE_INFO4 (0xffe08000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(13_17, RM, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fstoui",
		  OPCODE_INFO4 (0xffe20000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(13_17, RM, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fdtoui",
		  OPCODE_INFO4 (0xffe18000,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FEREG, OPRND_SHIFT_0_BIT),
				(13_17, RM, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmovd",
		  OPCODE_INFO3 (0xffe06800,
				(5_9, FEREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmovs",
		  OPCODE_INFO3 (0xffe06c00,
				(5_9, FREG, OPRND_SHIFT_0_BIT),
				(0_4, FREG, OPRND_SHIFT_0_BIT),
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_fo),
  OP16_WITH_WORK ("fmts",
		  OPCODE_INFO2 (0x00000000,
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT),
				(NONE, FREG, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_write),
  OP16_WITH_WORK ("fmfs",
		  OPCODE_INFO2 (0x00000000,
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT),
				(NONE, FREG, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_read),
  OP16_WITH_WORK ("fmtd",
		  OPCODE_INFO2 (0x00000000,
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT),
				(NONE, FEREG, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_writed),
  OP16_WITH_WORK ("fmfd",
		  OPCODE_INFO2 (0x00000000,
				(NONE, GREG0_15, OPRND_SHIFT_0_BIT),
				(NONE, FEREG, OPRND_SHIFT_0_BIT)),
		  CSKY_ISA_FLOAT_E1,
		  v1_work_fpu_readd),
  {NULL, 0, {}, {}, 0, 0, 0, 0, 0, NULL}
};

#undef _TRANSFER
#undef _RELOC16
#undef _RELOC32
#undef _RELAX

/* C-SKY v2 opcodes.  */
const struct csky_opcode csky_v2_opcodes[] =
  {
#define _TRANSFER   0
#define _RELOC16    0
#define _RELOC32    0
#define _RELAX      0
    OP16 ("bkpt",
	  OPCODE_INFO0 (0x0000),
	  CSKYV2_ISA_E1),
    OP16_WITH_WORK ("nie",
		    OPCODE_INFO0 (0x1460),
		    CSKYV2_ISA_E1,
		    v2_work_istack),
    OP16_WITH_WORK ("nir",
		    OPCODE_INFO0 (0x1461),
		    CSKYV2_ISA_E1,
		    v2_work_istack),
    OP16_WITH_WORK ("ipush",
		    OPCODE_INFO0 (0x1462),
		    CSKYV2_ISA_E1,
		    v2_work_istack),
    OP16_WITH_WORK ("ipop",
		    OPCODE_INFO0 (0x1463),
		    CSKYV2_ISA_E1,
		    v2_work_istack),
    OP16 ("bpop.h",
	  OPCODE_INFO1 (0x14a0,
			(2_4, GREG0_7, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_JAVA),
    OP16 ("bpop.w",
	  OPCODE_INFO1 (0x14a2,
			(2_4, GREG0_7, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_JAVA),
    OP16 ("bpush.h",
	  OPCODE_INFO1 (0x14e0,
			(2_4, GREG0_7, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_JAVA),
    OP16 ("bpush.w",
	  OPCODE_INFO1 (0x14e2,
			(2_4, GREG0_7, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_JAVA),
    OP32 ("bmset",
	  OPCODE_INFO0 (0xc0001020),
	  CSKY_ISA_JAVA),
    OP32 ("bmclr",
	  OPCODE_INFO0 (0xc0001420),
	  CSKY_ISA_JAVA),
    OP32 ("sce",
	  OPCODE_INFO1 (0xc0001820,
			(21_24, IMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_MP),
    OP32 ("trap",
	  OPCODE_INFO1 (0xc0002020,
			(10_11, IMM2b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_E1),
    /* Secure/nsecure world switch.  */
    OP32 ("wsc",
	  OPCODE_INFO0 (0xc0003c20),
	  CSKY_ISA_TRUST),
    OP32 ("mtcr",
	  OPCODE_INFO2 (0xc0006420,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_4or21_25, CTRLREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_E1),
    OP32 ("mfcr",
	  OPCODE_INFO2 (0xc0006020,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20or21_25, CTRLREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_E1),
#undef _TRANSFER
#define _TRANSFER   2
    OP32 ("rte",
	  OPCODE_INFO0 (0xc0004020),
	  CSKYV2_ISA_E1),
    OP32 ("rfi",
	  OPCODE_INFO0 (0xc0004420),
	  CSKYV2_ISA_2E3),
#undef _TRANSFER
#define _TRANSFER   0
    OP32 ("stop",
	  OPCODE_INFO0 (0xc0004820),
	  CSKYV2_ISA_E1),
    OP32 ("wait",
	  OPCODE_INFO0 (0xc0004c20),
	  CSKYV2_ISA_E1),
    OP32 ("doze",
	  OPCODE_INFO0 (0xc0005020),
	  CSKYV2_ISA_E1),
    OP32 ("we",
	  OPCODE_INFO0 (0xc0005420),
	  CSKY_ISA_MP_1E2),
    OP32 ("se",
	  OPCODE_INFO0 (0xc0005820),
	  CSKY_ISA_MP_1E2),
    OP32 ("psrclr",
	  OPCODE_INFO_LIST (0xc0007020,
			    (21_25, PSR_BITS_LIST, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_E1),
    OP32 ("psrset",
	  OPCODE_INFO_LIST (0xc0007420,
			    (21_25, PSR_BITS_LIST, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_E1),
    DOP32 ("abs",
	   OPCODE_INFO2 (0xc4000200,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO1 (0xc4000200,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_2E3),
    OP32 ("mvc",
	  OPCODE_INFO1 (0xc4000500,
			(0_4, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("incf",
	  OPCODE_INFO3 (0xc4000c20,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_4, IMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("movf",
	  OPCODE_INFO2 (0xc4000c20,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("inct",
	  OPCODE_INFO3 (0xc4000c40,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_4, IMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("movt",
	  OPCODE_INFO2 (0xc4000c40,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("decf",
	  OPCODE_INFO3 (0xc4000c80,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_4, IMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("dect",
	  OPCODE_INFO3 (0xc4000d00,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_4, IMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("decgt",
	  OPCODE_INFO3 (0xc4001020,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, IMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("declt",
	  OPCODE_INFO3 (0xc4001040,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, IMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("decne",
	  OPCODE_INFO3 (0xc4001080,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, IMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("clrf",
	  OPCODE_INFO1 (0xc4002c20,
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("clrt",
	  OPCODE_INFO1 (0xc4002c40,
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    DOP32 ("rotli",
	   OPCODE_INFO3 (0xc4004900,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xc4004900,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_1E2),
    OP32 ("lslc",
	  OPCODE_INFO3 (0xc4004c20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("lsrc",
	  OPCODE_INFO3 (0xc4004c40,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    DOP32 ("asrc",
	   OPCODE_INFO3 (0xc4004c80,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, OIMM5b, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO1 (0xc4004c80,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_1E2),
    OP32 ("xsr",
	  OPCODE_INFO3 (0xc4004d00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM5b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("bgenr",
	  OPCODE_INFO2 (0xc4005040,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    DOP32 ("brev",
	   OPCODE_INFO2 (0xc4006200,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO1 (0xc4006200,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_2E3),
    OP32 ("xtrb0",
	  OPCODE_INFO2 (0xc4007020,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("xtrb1",
	  OPCODE_INFO2 (0xc4007040,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("xtrb2",
	  OPCODE_INFO2 (0xc4007080,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("xtrb3",
	  OPCODE_INFO2 (0xc4007100,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("ff0",
	  OPCODE_INFO2 (0xc4007c20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    DOP32 ("ff1",
	   OPCODE_INFO2 (0xc4007c40,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO1 (0xc4007c40,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_1E2),
    OP32 ("mulu",
	  OPCODE_INFO2 (0xc4008820,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mulua",
	  OPCODE_INFO2 (0xc4008840,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mulus",
	  OPCODE_INFO2 (0xc4008880,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("muls",
	  OPCODE_INFO2 (0xc4008c20,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mulsa",
	  OPCODE_INFO2 (0xc4008c40,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mulss",
	  OPCODE_INFO2 (0xc4008c80,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mulsha",
	  OPCODE_INFO2 (0xc4009040,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mulshs",
	  OPCODE_INFO2 (0xc4009080,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mulswa",
	  OPCODE_INFO2 (0xc4009440,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mulsws",
	  OPCODE_INFO2 (0xc4009500,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mfhis",
	  OPCODE_INFO1 (0xc4009820,
			(0_4, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mflos",
	  OPCODE_INFO1 (0xc4009880,
			(0_4, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mvtc",
	  OPCODE_INFO0 (0xc4009a00),
	  CSKY_ISA_DSPE60),
    OP32 ("mfhi",
	  OPCODE_INFO1 (0xc4009c20,
			(0_4, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mthi",
	  OPCODE_INFO1 (0xc4009c40,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mflo",
	  OPCODE_INFO1 (0xc4009c80,
			(0_4, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("mtlo",
	  OPCODE_INFO1 (0xc4009d00,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP),
    OP32 ("vmulsh",
	  OPCODE_INFO2 (0xc400b020,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_1E2),
    OP32 ("vmulsha",
	  OPCODE_INFO2 (0xc400b040,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_1E2),
    OP32 ("vmulshs",
	  OPCODE_INFO2 (0xc400b080,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_1E2),
    OP32 ("vmulsw",
	  OPCODE_INFO2 (0xc400b420,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_1E2),
    OP32 ("vmulswa",
	  OPCODE_INFO2 (0xc400b440,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_1E2),
    OP32 ("vmulsws",
	  OPCODE_INFO2 (0xc400b480,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_1E2),
    OP32 ("ldr.b",
	  SOPCODE_INFO2 (0xd0000000,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_9or21_25, AREG_WITH_LSHIFT, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_2E3),
    OP32 ("ldr.bs",
	  SOPCODE_INFO2 (0xd0001000,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_9or21_25, AREG_WITH_LSHIFT, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_2E3),
    OP32 ("ldr.h",
	  SOPCODE_INFO2 (0xd0000400,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_9or21_25, AREG_WITH_LSHIFT, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_2E3),
    OP32 ("ldr.hs",
	  SOPCODE_INFO2 (0xd0001400,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_9or21_25, AREG_WITH_LSHIFT, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_2E3),
    OP32 ("ldr.w",
	  SOPCODE_INFO2 (0xd0000800,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_9or21_25, AREG_WITH_LSHIFT, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_2E3),
    OP32 ("ldm",
	  OPCODE_INFO2 (0xd0001c20,
			(0_4or21_25, REGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("ldq",
	  OPCODE_INFO2 (0xd0801c23,
			(NONE, REGr4_r7, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("str.b",
	  SOPCODE_INFO2 (0xd4000000,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_9or21_25, AREG_WITH_LSHIFT, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_2E3),
    OP32 ("str.h",
	  SOPCODE_INFO2 (0xd4000400,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_9or21_25, AREG_WITH_LSHIFT, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_2E3),
    OP32 ("str.w",
	  SOPCODE_INFO2 (0xd4000800,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_9or21_25, AREG_WITH_LSHIFT, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_2E3),
    OP32 ("stm",
	  OPCODE_INFO2 (0xd4001c20,
			(0_4or21_25, REGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("stq",
	  OPCODE_INFO2 (0xd4801c23,
			(NONE, REGr4_r7, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("ld.bs",
	  SOPCODE_INFO2 (0xd8004000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_1E2),
    OP32 ("ldbs",
	  SOPCODE_INFO2 (0xd8004000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_1E2),
    OP32 ("ld.hs",
	  SOPCODE_INFO2 (0xd8005000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_1_BIT))),
	  CSKYV2_ISA_1E2),
    OP32 ("ldhs",
	  SOPCODE_INFO2 (0xd8005000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_1_BIT))),
	  CSKYV2_ISA_1E2),
    OP32 ("ld.d",
	  SOPCODE_INFO2 (0xd8003000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKYV2_ISA_3E7),
    OP32 ("ldex.w",
	  SOPCODE_INFO2 (0xd8007000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_MP_1E2),
    OP32 ("ldexw",
	  SOPCODE_INFO2 (0xd8007000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_MP_1E2),
    OP32 ("ldex",
	  SOPCODE_INFO2 (0xd8007000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_MP_1E2),
    OP32 ("st.d",
	  SOPCODE_INFO2 (0xdc003000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKYV2_ISA_3E7),
    OP32 ("stex.w",
	  SOPCODE_INFO2 (0xdc007000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_MP_1E2),
    OP32 ("stexw",
	  SOPCODE_INFO2 (0xdc007000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_MP_1E2),
    OP32 ("stex",
	  SOPCODE_INFO2 (0xdc007000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_MP_1E2),
    DOP32 ("andi",
	   OPCODE_INFO3 (0xe4002000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (0_11, IMM12b, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xe4002000,
			 (16_20or21_25, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (0_11, IMM12b, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_1E2),
    OP32 ("andni",
	  OPCODE_INFO3 (0xe4003000,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_11, IMM12b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("xori",
	  OPCODE_INFO3 (0xe4004000,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_11, IMM12b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("ins",
	  OPCODE_INFO4 (0xc4005c00,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(5_9, MSB2SIZE, OPRND_SHIFT_0_BIT),
			(0_4, LSB2SIZE, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
#undef _TRANSFER
#undef _RELOC32
#define _TRANSFER   1
#define _RELOC32  BFD_RELOC_CKCORE_PCREL_IMM16BY4
    OP32 ("jmpi",
	  OPCODE_INFO1 (0xeac00000,
			(0_15, OFF16b, OPRND_SHIFT_2_BIT)),
	  CSKYV2_ISA_2E3),
#undef _TRANSFER
#undef _RELOC32
#define _TRANSFER   0
#define _RELOC32  0

    OP32 ("fadds",
	  OPCODE_INFO3 (0xf4000000,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fsubs",
	  OPCODE_INFO3 (0xf4000020,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fmovs",
	  OPCODE_INFO2 (0xf4000080,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fabss",
	  OPCODE_INFO2 (0xf40000c0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fnegs",
	  OPCODE_INFO2 (0xf40000e0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fcmpzhss",
	  OPCODE_INFO1 (0xf4000100,
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fcmpzlss",
	  OPCODE_INFO1 (0xf4000120,
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fcmpznes",
	  OPCODE_INFO1 (0xf4000140,
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fcmpzuos",
	  OPCODE_INFO1 (0xf4000160,
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fcmphss",
	  OPCODE_INFO2 (0xf4000180,
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fcmplts",
	  OPCODE_INFO2 (0xf40001a0,
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fcmpnes",
	  OPCODE_INFO2 (0xf40001c0,
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fcmpuos",
	  OPCODE_INFO2 (0xf40001e0,
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fmuls",
	  OPCODE_INFO3 (0xf4000200,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fmacs",
	  OPCODE_INFO3 (0xf4000280,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fmscs",
	  OPCODE_INFO3 (0xf40002a0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fnmacs",
	  OPCODE_INFO3 (0xf40002c0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fnmscs",
	  OPCODE_INFO3 (0xf40002e0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fnmuls",
	  OPCODE_INFO3 (0xf4000220,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fdivs",
	  OPCODE_INFO3 (0xf4000300,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("frecips",
	  OPCODE_INFO2 (0xf4000320,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fsqrts",
	  OPCODE_INFO2 (0xf4000340,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("faddd",
	  OPCODE_INFO3 (0xf4000800,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fsubd",
	  OPCODE_INFO3 (0xf4000820,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmovd",
	  OPCODE_INFO2 (0xf4000880,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fabsd",
	  OPCODE_INFO2 (0xf40008c0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fnegd",
	  OPCODE_INFO2 (0xf40008e0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fcmpzhsd",
	  OPCODE_INFO1 (0xf4000900,
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fcmpzlsd",
	  OPCODE_INFO1 (0xf4000920,
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fcmpzned",
	  OPCODE_INFO1 (0xf4000940,
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fcmpzuod",
	  OPCODE_INFO1 (0xf4000960,
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fcmphsd",
	  OPCODE_INFO2 (0xf4000980,
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fcmpltd",
	  OPCODE_INFO2 (0xf40009a0,
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fcmpned",
	  OPCODE_INFO2 (0xf40009c0,
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fcmpuod",
	  OPCODE_INFO2 (0xf40009e0,
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmuld",
	  OPCODE_INFO3 (0xf4000a00,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fnmuld",
	  OPCODE_INFO3 (0xf4000a20,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmacd",
	  OPCODE_INFO3 (0xf4000a80,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmscd",
	  OPCODE_INFO3 (0xf4000aa0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fnmacd",
	  OPCODE_INFO3 (0xf4000ac0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fnmscd",
	  OPCODE_INFO3 (0xf4000ae0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fdivd",
	  OPCODE_INFO3 (0xf4000b00,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("frecipd",
	  OPCODE_INFO2 (0xf4000b20,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fsqrtd",
	  OPCODE_INFO2 (0xf4000b40,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("faddm",
	  OPCODE_INFO3 (0xf4001000,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fsubm",
	  OPCODE_INFO3 (0xf4001020,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmovm",
	  OPCODE_INFO2 (0xf4001080,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fabsm",
	  OPCODE_INFO2 (0xf40010c0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fnegm",
	  OPCODE_INFO2 (0xf40010e0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmulm",
	  OPCODE_INFO3 (0xf4001200,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fnmulm",
	  OPCODE_INFO3 (0xf4001220,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmacm",
	  OPCODE_INFO3 (0xf4001280,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmscm",
	  OPCODE_INFO3 (0xf40012a0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fnmacm",
	  OPCODE_INFO3 (0xf40012c0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fnmscm",
	  OPCODE_INFO3 (0xf40012e0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT),
			(21_24, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fstosi.rn",
	  OPCODE_INFO2 (0xf4001800,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstosi.rz",
	  OPCODE_INFO2 (0xf4001820,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstosi.rpi",
	  OPCODE_INFO2 (0xf4001840,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstosi.rni",
	  OPCODE_INFO2 (0xf4001860,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstoui.rn",
	  OPCODE_INFO2 (0xf4001880,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstoui.rz",
	  OPCODE_INFO2 (0xf40018a0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstoui.rpi",
	  OPCODE_INFO2 (0xf40018c0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstoui.rni",
	  OPCODE_INFO2 (0xf40018e0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fdtosi.rn",
	  OPCODE_INFO2 (0xf4001900,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fdtosi.rz",
	  OPCODE_INFO2 (0xf4001920,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fdtosi.rpi",
	  OPCODE_INFO2 (0xf4001940,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fdtosi.rni",
	  OPCODE_INFO2 (0xf4001960,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fdtoui.rn",
	  OPCODE_INFO2 (0xf4001980,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fdtoui.rz",
	  OPCODE_INFO2 (0xf40019a0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fdtoui.rpi",
	  OPCODE_INFO2 (0xf40019c0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fdtoui.rni",
	  OPCODE_INFO2 (0xf40019e0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fsitos",
	  OPCODE_INFO2 (0xf4001a00,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fuitos",
	  OPCODE_INFO2 (0xf4001a20,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fsitod",
	  OPCODE_INFO2 (0xf4001a80,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fuitod",
	  OPCODE_INFO2 (0xf4001aa0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fdtos",
	  OPCODE_INFO2 (0xf4001ac0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fstod",
	  OPCODE_INFO2 (0xf4001ae0,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmfvrh",
	  OPCODE_INFO2 (0xf4001b00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmfvrl",
	  OPCODE_INFO2 (0xf4001b20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_19, FREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fmtvrh",
	  OPCODE_INFO2 (0xf4001b40,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fmtvrl",
	  OPCODE_INFO2 (0xf4001b60,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("flds",
	  SOPCODE_INFO2 (0xf4002000,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fldd",
	  SOPCODE_INFO2 (0xf4002100,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fldm",
	  SOPCODE_INFO2 (0xf4002200,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_3_BIT))),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fsts",
	  SOPCODE_INFO2 (0xf4002400,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstd",
	  SOPCODE_INFO2 (0xf4002500,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fstm",
	  SOPCODE_INFO2 (0xf4002600,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_3_BIT))),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fldrs",
	  SOPCODE_INFO2 (0xf4002800,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstrs",
	  SOPCODE_INFO2 (0xf4002c00,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fldrd",
	  SOPCODE_INFO2 (0xf4002900,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fldrm",
	  SOPCODE_INFO2 (0xf4002a00,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fstrd",
	  SOPCODE_INFO2 (0xf4002d00,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fstrm",
	  SOPCODE_INFO2 (0xf4002e00,
			 (0_3, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fldms",
	  OPCODE_INFO2 (0xf4003000,
			(0_3or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fldmd",
	  OPCODE_INFO2 (0xf4003100,
			(0_3or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fldmm",
	  OPCODE_INFO2 (0xf4003200,
			(0_3or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fstms",
	  OPCODE_INFO2 (0xf4003400,
			(0_3or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_E1),
    OP32 ("fstmd",
	  OPCODE_INFO2 (0xf4003500,
			(0_3or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    OP32 ("fstmm",
	  OPCODE_INFO2 (0xf4003600,
			(0_3or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_1E2),
    DOP32 ("idly",
	   OPCODE_INFO1 (0xc0001c20,
			 (21_25, OIMM5b_IDLY, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO0 (0xc0601c20),
	   CSKYV2_ISA_E1),

#undef _RELOC32
#define _RELOC32 BFD_RELOC_CKCORE_PCREL_IMM18BY2
    OP32 ("grs",
	  OPCODE_INFO2 (0xcc0c0000,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(0_17, IMM_OFF18b, OPRND_SHIFT_1_BIT)),
	  CSKYV2_ISA_2E3),
#undef _RELOC32
#define _RELOC32 0
    DOP32 ("ixh",
	   OPCODE_INFO3 (0xc4000820,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xc4000820,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_1E2),
    DOP32 ("ixw",
	   OPCODE_INFO3 (0xc4000840,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xc4000840,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_1E2),
    OP32 ("ixd",
	  OPCODE_INFO3 (0xc4000880,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    DOP32 ("divu",
	   OPCODE_INFO3 (0xc4008020,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xc4008020,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_2E3),
    DOP32 ("divs",
	   OPCODE_INFO3 (0xc4008040,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xc4008040,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_2E3),
    OP32 ("pldr",
	  SOPCODE_INFO1 (0xd8006000,
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_CACHE),
    OP32 ("pldw",
	  SOPCODE_INFO1 (0xdc006000,
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_CACHE),
    OP32 ("cprgr",
	  SOPCODE_INFO2 (0xfc000000,
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 ABRACKET_OPRND ((21_25, IMM5b, OPRND_SHIFT_0_BIT),
					 (0_11 , IMM12b, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_E1),
    OP32 ("cpwgr",
	  SOPCODE_INFO2 (0xfc001000,
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 ABRACKET_OPRND ((21_25, IMM5b, OPRND_SHIFT_0_BIT),
					 (0_11 , IMM12b, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_E1),
    OP32 ("cprcr",
	  SOPCODE_INFO2 (0xfc002000,
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 ABRACKET_OPRND ((21_25, IMM5b, OPRND_SHIFT_0_BIT),
					 (0_11 , IMM12b, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_E1),
    OP32 ("cpwcr",
	  SOPCODE_INFO2 (0xfc003000,
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 ABRACKET_OPRND ((21_25, IMM5b, OPRND_SHIFT_0_BIT),
					 (0_11 , IMM12b, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_E1),
    OP32 ("cprc",
	  SOPCODE_INFO1 (0xfc004000,
			 ABRACKET_OPRND ((21_25, IMM5b, OPRND_SHIFT_0_BIT),
					 (0_11, IMM12b, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_E1),
    OP32 ("cpop",
	  SOPCODE_INFO1 (0xfc008000,
			 ABRACKET_OPRND ((21_25, IMM5b, OPRND_SHIFT_0_BIT),
					 (0_14or16_20 , IMM15b, OPRND_SHIFT_0_BIT))),
	  CSKYV2_ISA_E1),

    OP16_OP32 ("push",
	       OPCODE_INFO_LIST (0x14c0,
				 (0_4, REGLIST_DASH_COMMA, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO_LIST (0xebe00000,
				 (0_8, REGLIST_DASH_COMMA, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
#undef _TRANSFER
#define _TRANSFER   2
    OP16_OP32 ("pop",
	       OPCODE_INFO_LIST (0x1480,
				 (0_4, REGLIST_DASH_COMMA, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO_LIST (0xebc00000,
				 (0_8, REGLIST_DASH_COMMA, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
#undef _TRANSFER
#define _TRANSFER   0
    OP16_OP32 ("movi",
	       OPCODE_INFO2 (0x3000,
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			     (0_7, IMM8b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xea000000,
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (0_15, IMM16b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
    /* bmaski will transfer to movi when imm < 17.  */
    OP16_OP32 ("bmaski",
	       OPCODE_INFO2 (0x3000,
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			     (0_7, IMM8b_BMASKI, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2,
	       OPCODE_INFO2 (0xc4005020,
			     (0_4, AREG, OPRND_SHIFT_0_BIT),
			     (21_25, OIMM5b_BMASKI, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
    OP16_OP32 ("cmphsi",
	       OPCODE_INFO2 (0x3800,
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			     (0_4, OIMM5b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xeb000000,
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (0_15, OIMM16b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
    OP16_OP32 ("cmplti",
	       OPCODE_INFO2 (0x3820,
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			     (0_4, OIMM5b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xeb200000,
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (0_15, OIMM16b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
    OP16_OP32 ("cmpnei",
	       OPCODE_INFO2 (0x3840,
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			     (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xeb400000,
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (0_15, IMM16b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
#undef _TRANSFER
#define _TRANSFER   1
    OP16_OP32 ("jmpix",
	       OPCODE_INFO2 (0x38e0,
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			     (0_1, IMM2b_JMPIX, OPRND_SHIFT_0_BIT)),
	       CSKY_ISA_JAVA,
	       OPCODE_INFO2 (0xe9e00000,
			     (16_20, GREG0_7, OPRND_SHIFT_0_BIT),
			     (0_1, IMM2b_JMPIX, OPRND_SHIFT_0_BIT)),
	       CSKY_ISA_JAVA),
#undef _TRANSFER
#define _TRANSFER   0
    DOP16_DOP32 ("bclri",
    		 OPCODE_INFO2 (0x3880,
			       (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			       (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x3880,
			       (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
			       (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4002820,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4002820,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    DOP16_DOP32 ("bseti",
    		 OPCODE_INFO2 (0x38a0,
			       (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			       (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x38a0,
			       (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
			       (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4002840,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4002840,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    OP16_OP32_WITH_WORK ("btsti",
			 OPCODE_INFO2 (0x38c0,
				       (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
				       (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
			 CSKYV2_ISA_E1,
			 OPCODE_INFO2 (0xc4002880,
				       (16_20, AREG, OPRND_SHIFT_0_BIT),
				       (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
			 CSKYV2_ISA_1E2, v2_work_btsti),
    DOP16_DOP32 ("lsli",
		 OPCODE_INFO3 (0x4000,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			       (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0x4000,
			       (5_7or8_10, DUP_GREG0_7, OPRND_SHIFT_0_BIT),
			       (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4004820,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4004820,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    DOP16_DOP32 ("lsri",
		 OPCODE_INFO3 (0x4800,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			       (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0x4800,
			       (5_7or8_10, DUP_GREG0_7, OPRND_SHIFT_0_BIT),
			       (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4004840,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4004840,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    OP16_OP32 ("asri",
	       OPCODE_INFO3 (0x5000,
			     (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			     (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO3 (0xc4004880,
			     (0_4, AREG, OPRND_SHIFT_0_BIT),
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
    DOP16_DOP32_WITH_WORK ("addc",
			   OPCODE_INFO2 (0x6001,
					 (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
					 (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO3 (0x6001,
					 (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
					 (2_5, GREG0_15, OPRND_SHIFT_0_BIT),
					 (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   OPCODE_INFO3 (0xc4000040,
					 (0_4, AREG, OPRND_SHIFT_0_BIT),
					 (16_20, AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO2 (0xc4000040,
					 (0_4or16_20, AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_1E2,
			   v2_work_addc),
    DOP16_DOP32 ("subc",
		 OPCODE_INFO2 (0x6003,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x6003,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4000100,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4000100,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    OP16_OP32 ("cmphs",
	       OPCODE_INFO2 (0x6400,
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT),
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xc4000420,
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (21_25, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    OP16_OP32 ("cmplt",
	       OPCODE_INFO2 (0x6401,
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT),
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xc4000440,
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (21_25, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    OP16_OP32 ("cmpne",
	       OPCODE_INFO2 (0x6402,
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT),
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xc4000480,
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (21_25, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    OP16_OP32 ("mvcv",
	       OPCODE_INFO1 (0x6403,
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO1 (0xc4000600,
			     (0_4, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    DOP16_DOP32 ("and",
		 OPCODE_INFO2 (0x6800,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x6800,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4002020,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4002020,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    DOP16_DOP32 ("andn",
		 OPCODE_INFO2 (0x6801,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x6801,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4002040,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4002040,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    OP16_OP32 ("tst",
	       OPCODE_INFO2 (0x6802,
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT),
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xc4002080,
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (21_25, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    OP16_OP32 ("tstnbz",
	       OPCODE_INFO1 (0x6803,
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO1 (0xc4002100,
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    DOP16_DOP32 ("or",
		 OPCODE_INFO2 (0x6c00,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x6c00,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4002420,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4002420,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    DOP16_DOP32 ("xor",
		 OPCODE_INFO2 (0x6c01,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x6c01,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4002440,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4002440,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    DOP16_DOP32 ("nor",
		 OPCODE_INFO2 (0x6c02,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x6c02,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4002480,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4002480,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    OP16_OP32 ("mov",
	       OPCODE_INFO2 (0x6c03,
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xc4004820,
			     (0_4, AREG, OPRND_SHIFT_0_BIT),
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
    OP16_OP32 ("nop",
	       OPCODE_INFO0 (0x6c03),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO0 (0xc4004820),
	       CSKYV2_ISA_E1),
    DOP16_DOP32 ("lsl",
		 OPCODE_INFO2 (0x7000,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x7000,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4004020,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4004020,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    DOP16_DOP32 ("lsr",
		 OPCODE_INFO2 (0x7001,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x7001,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4004040,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4004040,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    DOP16_DOP32 ("asr",
		 OPCODE_INFO2 (0x7002,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x7002,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4004080,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4004080,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    DOP16_DOP32 ("rotl",
		 OPCODE_INFO2 (0x7003,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x7003,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (NONE, DUMMY_REG, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4004100,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4004100,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    DOP16_DOP32 ("zextb",
		 OPCODE_INFO2 (0x7400,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO1 (0x7400,
			       (2_5or6_9, DUP_GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO2 (0xc40054e0,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO1 (0xc40054e0,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_2E3),
    DOP16_DOP32 ("zexth",
		 OPCODE_INFO2 (0x7401,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO1 (0x7401,
			       (2_5or6_9, DUP_GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO2 (0xc40055e0,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO1 (0xc40055e0,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_2E3),
    DOP16_DOP32 ("sextb",
		 OPCODE_INFO2 (0x7402,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO1 (0x7402,
			       (2_5or6_9, DUP_GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO2 (0xc40058e0,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO1 (0xc40058e0,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_2E3),
    DOP16_DOP32 ("sexth",
		 OPCODE_INFO2 (0x7403,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO1 (0x7403,
			       (2_5or6_9, DUP_GREG0_15, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO2 (0xc40059e0,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO1 (0xc40059e0,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_2E3),
    OP32 ("zext",
	  OPCODE_INFO4 (0xc4005400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(5_9, IMM5b, OPRND_SHIFT_0_BIT),
			(21_25, IMM5b_LS, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("sext",
	  OPCODE_INFO4 (0xc4005800,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(5_9, IMM5b, OPRND_SHIFT_0_BIT),
			(21_25, IMM5b_LS, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
#undef _TRANSFER
#define _TRANSFER   2
    OP16_OP32 ("rts",
	       OPCODE_INFO0 (0x783c),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO0 (0xe8cf0000),
	       CSKYV2_ISA_E1),
#undef _TRANSFER
#define _TRANSFER   1
    OP16_OP32 ("jmp",
	       OPCODE_INFO1 (0x7800,
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO1 (0xe8c00000,
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
#undef _TRANSFER
#define _TRANSFER   0
    OP16_OP32 ("revb",
	       OPCODE_INFO2 (0x7802,
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2,
	       OPCODE_INFO2 (0xc4006080,
			     (0_4, AREG, OPRND_SHIFT_0_BIT),
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    OP16_OP32 ("revh",
	       OPCODE_INFO2 (0x7803,
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2,
	       OPCODE_INFO2 (0xc4006100,
			     (0_4, AREG, OPRND_SHIFT_0_BIT),
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    OP16_OP32 ("jsr",
	       OPCODE_INFO1 (0x7bc1,
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO1 (0xe8e00000,
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    DOP16_DOP32 ("mult",
		 OPCODE_INFO2 (0x7c00,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x7c00,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_E1,
		 OPCODE_INFO3 (0xc4008420,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4008420,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_1E2),
    OP16 ("mul",
	  OPCODE_INFO2 (0x7c00,
			(6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			(2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_E1),
    DOP16_DOP32 ("mulsh",
		 OPCODE_INFO2 (0x7c01,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO3 (0x7c01,
			       (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT),
			       (2_5, 2IN1_DUMMY, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_2E3,
		 OPCODE_INFO3 (0xc4009020,
			       (0_4, AREG, OPRND_SHIFT_0_BIT),
			       (16_20, AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 OPCODE_INFO2 (0xc4009020,
			       (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			       (21_25, AREG, OPRND_SHIFT_0_BIT)),
		 CSKYV2_ISA_2E3),
    OP16 ("muls.h",
	  OPCODE_INFO2 (0x7c01,
			(6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			(2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    DOP32 ("mulsw",
	   OPCODE_INFO3 (0xc4009420,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xc4009420,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT)),
	   CSKY_ISA_DSPE60),
    OP16_OP32 ("ld.b",
	       SOPCODE_INFO2 (0x8000,
			      (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					     (0_4, IMM_LDST, OPRND_SHIFT_0_BIT))),
	       CSKYV2_ISA_E1,
	       SOPCODE_INFO2 (0xd8000000,
			      (21_25, AREG, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					     (0_11, IMM_LDST, OPRND_SHIFT_0_BIT))),
	       CSKYV2_ISA_E1),
    OP16_OP32 ("ldb",
	       SOPCODE_INFO2 (0x8000,
			      (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					     (0_4, IMM_LDST, OPRND_SHIFT_0_BIT))),
	       CSKYV2_ISA_E1,
	       SOPCODE_INFO2 (0xd8000000,
			      (21_25, AREG, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					     (0_11, IMM_LDST, OPRND_SHIFT_0_BIT))),
	       CSKYV2_ISA_E1),
    OP16_OP32 ("st.b",
	       SOPCODE_INFO2 (0xa000,
			      (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					     (0_4, IMM_LDST, OPRND_SHIFT_0_BIT))),
	       CSKYV2_ISA_E1,
	       SOPCODE_INFO2 (0xdc000000,
			      (21_25, AREG, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					     (0_11, IMM_LDST, OPRND_SHIFT_0_BIT))),
	       CSKYV2_ISA_E1),
    OP16_OP32 ("stb",
	       SOPCODE_INFO2 (0xa000,
			      (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					     (0_4, IMM_LDST, OPRND_SHIFT_0_BIT))),
	       CSKYV2_ISA_E1,
	       SOPCODE_INFO2 (0xdc000000,
			      (21_25, AREG, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					     (0_11, IMM_LDST, OPRND_SHIFT_0_BIT))),
	       CSKYV2_ISA_E1),

    OP16_OP32 ("ld.h",
	       SOPCODE_INFO2 (0x8800,
			      (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					     (0_4, IMM_LDST, OPRND_SHIFT_1_BIT))),
	       CSKYV2_ISA_E1,
	       SOPCODE_INFO2 (0xd8001000,
			      (21_25, AREG, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					     (0_11, IMM_LDST, OPRND_SHIFT_1_BIT))),
	       CSKYV2_ISA_E1),
    OP16_OP32 ("ldh",
	       SOPCODE_INFO2 (0x8800,
			      (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					     (0_4, IMM_LDST, OPRND_SHIFT_1_BIT))),
	       CSKYV2_ISA_E1,
	       SOPCODE_INFO2 (0xd8001000,
			      (21_25, AREG, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					     (0_11, IMM_LDST, OPRND_SHIFT_1_BIT))),
	       CSKYV2_ISA_E1),
    OP16_OP32 ("st.h",
	       SOPCODE_INFO2 (0xa800,
			      (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					     (0_4, IMM_LDST, OPRND_SHIFT_1_BIT))),
	       CSKYV2_ISA_E1,
	       SOPCODE_INFO2 (0xdc001000,
			      (21_25, AREG, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					     (0_11, IMM_LDST, OPRND_SHIFT_1_BIT))),
	       CSKYV2_ISA_E1),
    OP16_OP32 ("sth",
	       SOPCODE_INFO2 (0xa800,
			      (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					     (0_4, IMM_LDST, OPRND_SHIFT_1_BIT))),
	       CSKYV2_ISA_E1,
	       SOPCODE_INFO2 (0xdc001000,
			      (21_25, AREG, OPRND_SHIFT_0_BIT),
			      BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					     (0_11, IMM_LDST, OPRND_SHIFT_1_BIT))),
	       CSKYV2_ISA_E1),
    DOP16_OP32 ("ld.w",
		SOPCODE_INFO2 (0x9000,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					      (0_4, IMM_LDST, OPRND_SHIFT_2_BIT))),
		SOPCODE_INFO2 (0x9800,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((NONE, REGsp, OPRND_SHIFT_0_BIT),
					      (0_4or8_10, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1,
		SOPCODE_INFO2 (0xd8002000,
			       (21_25, AREG, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					      (0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1),
    DOP16_OP32 ("ldw",
		SOPCODE_INFO2 (0x9000,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					      (0_4, IMM_LDST, OPRND_SHIFT_2_BIT))),
		SOPCODE_INFO2 (0x9800,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((NONE, REGsp, OPRND_SHIFT_0_BIT),
					      (0_4or8_10, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1,
		SOPCODE_INFO2 (0xd8002000,
			       (21_25, AREG, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					      (0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1),
    DOP16_OP32 ("ld",
		SOPCODE_INFO2 (0x9000,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					      (0_4, IMM_LDST, OPRND_SHIFT_0_BIT))),
		SOPCODE_INFO2 (0x9800,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((NONE, REGsp, OPRND_SHIFT_0_BIT),
					      (0_4or8_10, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1,
		SOPCODE_INFO2 (0xd8002000,
			       (21_25, AREG, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					      (0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1),
    DOP16_OP32 ("st.w",
		SOPCODE_INFO2 (0xb000,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					      (0_4, IMM_LDST, OPRND_SHIFT_2_BIT))),
		SOPCODE_INFO2 (0xb800,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((NONE, REGsp, OPRND_SHIFT_0_BIT),
					      (0_4or8_10, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1,
		SOPCODE_INFO2 (0xdc002000,
			       (21_25, AREG, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					      (0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1),
    DOP16_OP32 ("stw",
		SOPCODE_INFO2 (0xb000,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					      (0_4, IMM_LDST, OPRND_SHIFT_2_BIT))),
		SOPCODE_INFO2 (0xb800,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((NONE, REGsp, OPRND_SHIFT_0_BIT),
					      (0_4or8_10, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1,
		SOPCODE_INFO2 (0xdc002000,
			       (21_25, AREG, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					      (0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1),
    DOP16_OP32 ("st",
		SOPCODE_INFO2 (0xb000,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					      (0_4, IMM_LDST, OPRND_SHIFT_2_BIT))),
		SOPCODE_INFO2 (0xb800,
			       (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((NONE, REGsp, OPRND_SHIFT_0_BIT),
					      (0_4or8_10, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1,
		SOPCODE_INFO2 (0xdc002000,
			       (21_25, AREG, OPRND_SHIFT_0_BIT),
			       BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					      (0_11, IMM_LDST, OPRND_SHIFT_2_BIT))),
		CSKYV2_ISA_E1),
#ifdef BUILD_AS
    DOP16_DOP32_WITH_WORK ("addi",
			   OPCODE_INFO2 (0x2000,
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, IMM32b, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO3 (0x2000,
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, IMM32b, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   OPCODE_INFO2 (0xe4000000,
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, IMM32b, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO3 (0xe4000000,
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, IMM32b, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_1E2,
			   v2_work_addi),
#else
    DOP16 ("addi",
	   OPCODE_INFO2 (0x2000,
			 (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			 (0_7, OIMM8b, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO3 (0x5802,
			 (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			 (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			 (2_4, OIMM3b, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_E1),
    DOP16 ("addi",
	   OPCODE_INFO3 (0x1800,
			 (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			 (NONE, REGsp, OPRND_SHIFT_0_BIT),
			 (0_7, IMM8b_LS2, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO3 (0x1400,
			 (NONE, REGsp, OPRND_SHIFT_0_BIT),
			 (NONE, REGsp, OPRND_SHIFT_0_BIT),
			 (0_4or8_9, IMM7b_LS2, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_E1),
    DOP32 ("addi",
	   OPCODE_INFO3 (0xe4000000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (0_11, OIMM12b, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO3 (0xcc1c0000,
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 (NONE, REG_r28, OPRND_SHIFT_0_BIT),
			 (0_17, OIMM18b, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_1E2),
#endif
#ifdef BUILD_AS
    DOP16_DOP32_WITH_WORK ("subi",
			   OPCODE_INFO2 (0x2800,
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, IMM32b, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO3 (0x2800,
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, IMM32b, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   OPCODE_INFO2 (0xe4001000,
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, IMM32b, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO3 (0xe4001000,
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, AREG, OPRND_SHIFT_0_BIT),
					 (NONE, IMM32b, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_1E2, v2_work_subi),
#else
    DOP16 ("subi",
	   OPCODE_INFO2 (0x2800,
			 (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			 (0_7, OIMM8b, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO3 (0x5803,
			 (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
			 (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			 (2_4, OIMM3b, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_E1),
    OP32 ("subi",
	  OPCODE_INFO3 (0xe4001000,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_11, OIMM12b, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP16 ("subi",
	  OPCODE_INFO3 (0x1420,
			(NONE, REGsp, OPRND_SHIFT_0_BIT),
			(NONE, REGsp, OPRND_SHIFT_0_BIT),
			(0_4or8_9, IMM7b_LS2, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_E1),
#endif
    DOP16_DOP32_WITH_WORK ("addu",
			   OPCODE_INFO2 (0x6000,
					 (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
					 (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO3 (0x5800,
					 (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
					 (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					 (2_4, GREG0_7, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   OPCODE_INFO3 (0xc4000020,
					 (0_4, AREG, OPRND_SHIFT_0_BIT),
					 (16_20, AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO2 (0xc4000020,
					 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   v2_work_add_sub),
    DOP16_DOP32_WITH_WORK ("add",
			   OPCODE_INFO2 (0x6000,
					 (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
					 (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO3 (0x5800,
					 (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
					 (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					 (2_4, GREG0_7, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   OPCODE_INFO3 (0xc4000020,
					 (0_4, AREG, OPRND_SHIFT_0_BIT),
					 (16_20, AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO2 (0xc4000020,
					 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   v2_work_add_sub),
    DOP16_DOP32_WITH_WORK ("subu",
			   OPCODE_INFO2 (0x6002,
					 (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
					 (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO3 (0x5801,
					 (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
					 (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					 (2_4, GREG0_7, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   OPCODE_INFO3 (0xc4000080,
					 (0_4, AREG, OPRND_SHIFT_0_BIT),
					 (16_20, AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO2 (0xc4000080,
					 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   v2_work_add_sub),
    DOP16_DOP32_WITH_WORK ("sub",
			   OPCODE_INFO2 (0x6002,
					 (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
					 (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO3 (0x5801,
					 (5_7, GREG0_7, OPRND_SHIFT_0_BIT),
					 (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
					 (2_4, GREG0_7, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   OPCODE_INFO3 (0xc4000080,
					 (0_4, AREG, OPRND_SHIFT_0_BIT),
					 (16_20, AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   OPCODE_INFO2 (0xc4000080,
					 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
					 (21_25, AREG, OPRND_SHIFT_0_BIT)),
			   CSKYV2_ISA_E1,
			   v2_work_add_sub),
    OP32_WITH_WORK ("fmovis",
		    OPCODE_INFO2 (0xf4001c00,
				  (0_3, FREG, OPRND_SHIFT_0_BIT),
				  (4_7or16_24, SFLOAT, OPRND_SHIFT_2_BIT)),
		    CSKY_ISA_FLOAT_1E3,
		    float_work_fmovi),
    OP32_WITH_WORK ("fmovid",
		    OPCODE_INFO2 (0xf4001e00,
				  (0_3, FREG, OPRND_SHIFT_0_BIT),
				  (4_7or16_24, DFLOAT, OPRND_SHIFT_2_BIT)),
		    CSKY_ISA_FLOAT_3E4,
		    float_work_fmovi),
#undef _RELOC32
#define _RELOC32 BFD_RELOC_CKCORE_PCREL_IMM26BY2
    OP32 ("bsr",
	  OPCODE_INFO1 (0xe0000000,
			(0_25, OFF26b, OPRND_SHIFT_1_BIT)),
	  CSKYV2_ISA_E1),
#undef _RELOC32
#define _RELOC32 BFD_RELOC_CKCORE_DOFFSET_IMM18
    OP32 ("lrs.b",
	  OPCODE_INFO2 (0xcc000000,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(0_17, LABEL_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("srs.b",
	  OPCODE_INFO2 (0xcc100000,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(0_17, LABEL_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
#undef _RELOC32
#define _RELOC32 BFD_RELOC_CKCORE_DOFFSET_IMM18BY2
    OP32 ("lrs.h",
	  OPCODE_INFO2 (0xcc040000,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(0_17, LABEL_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("srs.h",
	  OPCODE_INFO2 (0xcc140000,
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(0_17, LABEL_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
#undef _RELOC32
#define _RELOC32 BFD_RELOC_CKCORE_PCREL_FLRW_IMM8BY4
    OP32 ("flrws",
	  OPCODE_INFO2 (0xf4003800,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(4_7or21_24, FCONSTANT, OPRND_SHIFT_2_BIT)),
	  CSKY_ISA_FLOAT_1E3),
    OP32 ("flrwd",
	  OPCODE_INFO2 (0xf4003900,
			(0_3, FREG, OPRND_SHIFT_0_BIT),
			(4_7or21_24, FCONSTANT, OPRND_SHIFT_2_BIT)),
	  CSKY_ISA_FLOAT_3E4),
#undef _RELOC32
#define _RELOC32 BFD_RELOC_CKCORE_DOFFSET_IMM18BY4
    OP32_WITH_WORK ("lrs.w",
		    OPCODE_INFO2 (0xcc080000,
				  (21_25, AREG, OPRND_SHIFT_0_BIT),
				  (0_17, LABEL_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
		    CSKYV2_ISA_2E3,
		    v2_work_lrsrsw),
    OP32_WITH_WORK ("srs.w",
		    OPCODE_INFO2 (0xcc180000,
				  (21_25, AREG, OPRND_SHIFT_0_BIT),
				  (0_17, LABEL_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
		    CSKYV2_ISA_2E3,
		    v2_work_lrsrsw),

#undef _RELOC32
#define _RELOC32    BFD_RELOC_CKCORE_PCREL_IMM16BY4
    OP32_WITH_WORK ("jsri",
		    OPCODE_INFO1 (0xeae00000,
				  (0_15, OFF16b, OPRND_SHIFT_2_BIT)),
		    CSKYV2_ISA_2E3,
		    v2_work_jsri),
#undef _RELOC32
#define _RELOC32    BFD_RELOC_CKCORE_PCREL_IMM16BY2
    OP32 ("bez",
	  OPCODE_INFO2 (0xe9000000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, OFF16b_LSL1, OPRND_SHIFT_1_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("bnez",
	  OPCODE_INFO2 (0xe9200000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, OFF16b_LSL1, OPRND_SHIFT_1_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("bhz",
	  OPCODE_INFO2 (0xe9400000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, OFF16b_LSL1, OPRND_SHIFT_1_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("blsz",
	  OPCODE_INFO2 (0xe9600000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, OFF16b_LSL1, OPRND_SHIFT_1_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("blz",
	  OPCODE_INFO2 (0xe9800000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, OFF16b_LSL1, OPRND_SHIFT_1_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("bhsz",
	  OPCODE_INFO2 (0xe9a00000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, OFF16b_LSL1, OPRND_SHIFT_1_BIT)),
	  CSKYV2_ISA_2E3),
#undef _RELAX
#undef _RELOC16
#undef _TRANSFER
#define _TRANSFER   1
#define _RELAX      1
#define _RELOC16    BFD_RELOC_CKCORE_PCREL_IMM10BY2
    OP16_OP32 ("br",
	       OPCODE_INFO1 (0x0400,
			     (0_9, UNCOND10b, OPRND_SHIFT_1_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO1 (0xe8000000,
			     (0_15, UNCOND16b, OPRND_SHIFT_1_BIT)),
	       CSKYV2_ISA_E1),
#undef _TRANSFER
#define _TRANSFER   0
    OP16_OP32 ("bt",
	       OPCODE_INFO1 (0x0800,
			     (0_9, COND10b, OPRND_SHIFT_1_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO1 (0xe8600000,
			     (0_15, COND16b, OPRND_SHIFT_1_BIT)),
	       CSKYV2_ISA_1E2),
    OP16_OP32 ("bf",
	       OPCODE_INFO1 (0x0c00,
			     (0_9, COND10b, OPRND_SHIFT_1_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO1 (0xe8400000,
			     (0_15, COND16b, OPRND_SHIFT_1_BIT)),
	       CSKYV2_ISA_1E2),
#undef _RELAX
#undef _RELOC16
#define _RELAX      0
#define _RELOC16    0
    OP32 ("bnezad",
	  OPCODE_INFO2 (0xe8200000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, COND16b, OPRND_SHIFT_1_BIT)),
	  CSKYV2_ISA_3E3R2),
#undef _RELOC16
#undef _RELOC32
#define _RELOC16    0
#define _RELOC32    0
#undef _TRANSFER
#define _TRANSFER   1
    OP16_WITH_WORK ("jbr",
		    OPCODE_INFO1 (0x0400,
				  (0_10, UNCOND10b, OPRND_SHIFT_1_BIT)),
		    CSKYV2_ISA_E1,
		    v2_work_jbr),
#undef _TRANSFER
#define _TRANSFER   0
    OP16_WITH_WORK ("jbt",
		    OPCODE_INFO1 (0x0800,
				  (0_10, COND10b, OPRND_SHIFT_1_BIT)),
		    CSKYV2_ISA_E1,
		    v2_work_jbtf),
    OP16_WITH_WORK ("jbf",
		    OPCODE_INFO1 (0x0c00,
				  (0_10, COND10b, OPRND_SHIFT_1_BIT)),
		    CSKYV2_ISA_E1,
		    v2_work_jbtf),
    OP32_WITH_WORK ("jbsr",
		    OPCODE_INFO1 (0xe0000000,
				  (0_25, OFF26b, OPRND_SHIFT_1_BIT)),
		    CSKYV2_ISA_E1,
		    v2_work_jbsr),
    OP32_WITH_WORK ("movih",
		    OPCODE_INFO2 (0xea200000,
				  (16_20, AREG, OPRND_SHIFT_0_BIT),
				  (0_15, IMM16b_MOVIH, OPRND_SHIFT_0_BIT)),
		    CSKYV2_ISA_1E2,
		    v2_work_movih),
    OP32_WITH_WORK ("ori",
		    OPCODE_INFO3 (0xec000000,
				  (21_25, AREG, OPRND_SHIFT_0_BIT),
				  (16_20, AREG, OPRND_SHIFT_0_BIT),
				  (0_15, IMM16b_ORI, OPRND_SHIFT_0_BIT)),
		    CSKYV2_ISA_1E2,
		    v2_work_ori),
    DOP32_WITH_WORK ("bgeni",
		     OPCODE_INFO2 (0xea000000,
				   (16_20, AREG, OPRND_SHIFT_0_BIT),
				   (0_4, IMM4b, OPRND_SHIFT_0_BIT)),
		     OPCODE_INFO2 (0xea200000,
				   (16_20, AREG, OPRND_SHIFT_0_BIT),
				   (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
		     CSKYV2_ISA_E1,
		     v2_work_bgeni),
#undef _RELOC16
#undef _RELOC32
#define _RELOC16    BFD_RELOC_CKCORE_PCREL_IMM7BY4
#define _RELOC32    BFD_RELOC_CKCORE_PCREL_IMM16BY4
    DOP16_OP32_WITH_WORK ("lrw",
			  OPCODE_INFO2 (0x1000,
					(5_7, GREG0_7, OPRND_SHIFT_0_BIT),
					(0_4or8_9, CONSTANT, OPRND_SHIFT_2_BIT)),
			  OPCODE_INFO2 (0x0000,
					(5_7, GREG0_7, OPRND_SHIFT_0_BIT),
					(0_4or8_9, ELRW_CONSTANT, OPRND_SHIFT_2_BIT)),
			  CSKYV2_ISA_E1,
			  OPCODE_INFO2 (0xea800000,
					(16_20, AREG, OPRND_SHIFT_0_BIT),
					(0_15, CONSTANT, OPRND_SHIFT_2_BIT)),
			  CSKYV2_ISA_E1,
			  v2_work_lrw),
#undef _RELOC16
#undef _RELOC32
#define _RELOC16    0
#define _RELOC32    0

#undef _RELAX
#define _RELAX      1
    OP32 ("jbez",
	  OPCODE_INFO2 (0xe9000000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, JCOMPZ, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("jbnez",
	  OPCODE_INFO2 (0xe9200000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, JCOMPZ, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("jbhz",
	  OPCODE_INFO2 (0xe9400000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, JCOMPZ, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("jblsz",
	  OPCODE_INFO2 (0xe9600000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, JCOMPZ, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("jblz",
	  OPCODE_INFO2 (0xe9800000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, JCOMPZ, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
    OP32 ("jbhsz",
	  OPCODE_INFO2 (0xe9a00000,
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(0_15, JCOMPZ, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_2E3),
#undef _RELAX
#define _RELAX      0

  /* CK860 instructions.  */
    OP32 ("sync.is",
	  OPCODE_INFO0 (0xc2200420),
	  CSKYV2_ISA_10E60),
    OP32 ("sync.i",
	  OPCODE_INFO0 (0xc0200420),
	  CSKYV2_ISA_10E60),
    OP32 ("sync.s",
	  OPCODE_INFO0 (0xc2000420),
	  CSKYV2_ISA_10E60),
    OP32 ("bar.brwarw",
	  OPCODE_INFO0 (0xc000842f),
	  CSKYV2_ISA_10E60),
    OP32 ("bar.brwarws",
	  OPCODE_INFO0 (0xc200842f),
	  CSKYV2_ISA_10E60),
    OP32 ("bar.brar",
	  OPCODE_INFO0 (0xc0008425),
	  CSKYV2_ISA_10E60),
    OP32 ("bar.brars",
	  OPCODE_INFO0 (0xc2008425),
	  CSKYV2_ISA_10E60),
    OP32 ("bar.bwaw",
	  OPCODE_INFO0 (0xc000842a),
	  CSKYV2_ISA_10E60),
    OP32 ("bar.bwaws",
	  OPCODE_INFO0 (0xc200842a),
	  CSKYV2_ISA_10E60),
    OP32 ("icache.iall",
	  OPCODE_INFO0 (0xc1009020),
	  CSKYV2_ISA_10E60),
    OP32 ("icache.ialls",
	  OPCODE_INFO0 (0xc3009020),
	  CSKYV2_ISA_10E60),
    OP32 ("l2cache.iall",
	  OPCODE_INFO0 (0xc1009820),
	  CSKYV2_ISA_10E60),
    OP32 ("l2cache.call",
	  OPCODE_INFO0 (0xc0809820),
	  CSKYV2_ISA_10E60),
    OP32 ("l2cache.ciall",
	  OPCODE_INFO0 (0xc1809820),
	  CSKYV2_ISA_10E60),
    OP32 ("icache.iva",
	  OPCODE_INFO1 (0xc1609020,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.iall",
	  OPCODE_INFO0 (0xc1009420),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.iva",
	  OPCODE_INFO1 (0xc1609420,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.isw",
	  OPCODE_INFO1 (0xc1409420,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.call",
	  OPCODE_INFO0 (0xc0809420),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.cva",
	  OPCODE_INFO1 (0xc0e09420,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.cval1",
	  OPCODE_INFO1 (0xc2e09420,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.csw",
	  OPCODE_INFO1 (0xc0c09420,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.ciall",
	  OPCODE_INFO0 (0xc1809420),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.civa",
	  OPCODE_INFO1 (0xc1e09420,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("dcache.cisw",
	  OPCODE_INFO1 (0xc1c09420,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("tlbi.vaa",
	  OPCODE_INFO1 (0xc0408820,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("tlbi.vaas",
	  OPCODE_INFO1 (0xc2408820,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("tlbi.asid",
	  OPCODE_INFO1 (0xc0208820,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("tlbi.asids",
	  OPCODE_INFO1 (0xc2208820,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("tlbi.va",
	  OPCODE_INFO1 (0xc0608820,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("tlbi.vas",
	  OPCODE_INFO1 (0xc2608820,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_10E60),
    OP32 ("tlbi.all",
	  OPCODE_INFO0 (0xc0008820),
	  CSKYV2_ISA_10E60),
    OP32 ("tlbi.alls",
	  OPCODE_INFO0 (0xc2008820),
	  CSKYV2_ISA_10E60),
    DOP32 ("sync",
	   OPCODE_INFO0 (0xc0000420),
	   OPCODE_INFO1 (0xc0000420,
			 (21_25, IMM5b, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_E1),

    /* The followings are enhance DSP instructions.  */
    DOP32_WITH_WORK ("bloop",
		     OPCODE_INFO3 (0xe9c00000,
				  (16_20, AREG, OPRND_SHIFT_0_BIT),
				  (0_11, BLOOP_OFF12b, OPRND_SHIFT_1_BIT),
				  (12_15, BLOOP_OFF4b, OPRND_SHIFT_1_BIT)),
		     OPCODE_INFO2 (0xe9c00000,
				  (16_20, AREG, OPRND_SHIFT_0_BIT),
				  (0_11, BLOOP_OFF12b, OPRND_SHIFT_1_BIT)),
		     CSKY_ISA_DSP_ENHANCE,
		     dsp_work_bloop),
    /* The followings are ld/st instructions.  */
    OP32 ("ldbi.b",
	  OPCODE_INFO2 (0xd0008000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("ldbi.h",
	  OPCODE_INFO2 (0xd0008400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("ldbi.w",
	  OPCODE_INFO2 (0xd0008800,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pldbi.d",
	  OPCODE_INFO2 (0xd0008c00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("ldbi.hs",
	  OPCODE_INFO2 (0xd0009000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("ldbi.bs",
	  OPCODE_INFO2 (0xd0009400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("stbi.b",
	  OPCODE_INFO2 (0xd4008000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("stbi.h",
	  OPCODE_INFO2 (0xd4008400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("stbi.w",
	  OPCODE_INFO2 (0xd4008800,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("ldbir.b",
	  OPCODE_INFO3 (0xd000a000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("ldbir.h",
	  OPCODE_INFO3 (0xd000a400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("ldbir.w",
	  OPCODE_INFO3 (0xd000a800,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pldbir.d",
	  OPCODE_INFO3 (0xd000ac00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("ldbir.bs",
	  OPCODE_INFO3 (0xd000b000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("ldbir.hs",
	  OPCODE_INFO3 (0xd000b400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("stbir.b",
	  OPCODE_INFO3 (0xd400a000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("stbir.h",
	  OPCODE_INFO3 (0xd400a400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("stbir.w",
	  OPCODE_INFO3 (0xd400a800,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    /* The followings are add/sub instructions.  */
    OP32 ("padd.8",
	  OPCODE_INFO3 (0xf800c040,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("padd.16",
	  OPCODE_INFO3 (0xf800c000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("padd.u8.s",
	  OPCODE_INFO3 (0xf800c140,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("padd.s8.s",
	  OPCODE_INFO3 (0xf800c1c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("padd.u16.s",
	  OPCODE_INFO3 (0xf800c100,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("padd.s16.s",
	  OPCODE_INFO3 (0xf800c180,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("add.u32.s",
	  OPCODE_INFO3 (0xf800c120,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("add.s32.s",
	  OPCODE_INFO3 (0xf800c1a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psub.8",
	  OPCODE_INFO3 (0xf800c440,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psub.16",
	  OPCODE_INFO3 (0xf800c400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psub.u8.s",
	  OPCODE_INFO3 (0xf800c540,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psub.s8.s",
	  OPCODE_INFO3 (0xf800c5c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psub.u16.s",
	  OPCODE_INFO3 (0xf800c500,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psub.s16.s",
	  OPCODE_INFO3 (0xf800c580,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("sub.u32.s",
	  OPCODE_INFO3 (0xf800c520,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("sub.s32.s",
	  OPCODE_INFO3 (0xf800c5a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("paddh.u8",
	  OPCODE_INFO3 (0xf800c240,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("paddh.s8",
	  OPCODE_INFO3 (0xf800c2c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("paddh.u16",
	  OPCODE_INFO3 (0xf800c200,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("paddh.s16",
	  OPCODE_INFO3 (0xf800c280,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("addh.u32",
	  OPCODE_INFO3 (0xf800c220,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("addh.s32",
	  OPCODE_INFO3 (0xf800c2a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psubh.u8",
	  OPCODE_INFO3 (0xf800c640,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psubh.s8",
	  OPCODE_INFO3 (0xf800c6c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psubh.u16",
	  OPCODE_INFO3 (0xf800c600,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psubh.s16",
	  OPCODE_INFO3 (0xf800c680,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("subh.u32",
	  OPCODE_INFO3 (0xf800c620,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("subh.s32",
	  OPCODE_INFO3 (0xf800c6a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("add.64",
	  OPCODE_INFO3 (0xf800c060,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("sub.64",
	  OPCODE_INFO3 (0xf800c460,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("add.u64.s",
	  OPCODE_INFO3 (0xf800c160,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("add.s64.s",
	  OPCODE_INFO3 (0xf800c1e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("sub.u64.s",
	  OPCODE_INFO3 (0xf800c560,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("sub.s64.s",
	  OPCODE_INFO3 (0xf800c5e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    /* The following are comparison instructions.  */
    OP32 ("pasx.16",
	  OPCODE_INFO3 (0xf800c860,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psax.16",
	  OPCODE_INFO3 (0xf800cc60,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pasx.u16.s",
	  OPCODE_INFO3 (0xf800c960,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pasx.s16.s",
	  OPCODE_INFO3 (0xf800c9e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psax.u16.s",
	  OPCODE_INFO3 (0xf800cd60,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psax.s16.s",
	  OPCODE_INFO3 (0xf800cde0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pasxh.u16",
	  OPCODE_INFO3 (0xf800ca60,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pasxh.s16",
	  OPCODE_INFO3 (0xf800cae0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psaxh.u16",
	  OPCODE_INFO3 (0xf800ce60,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psaxh.s16",
	  OPCODE_INFO3 (0xf800cee0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmpne.8",
	  OPCODE_INFO3 (0xf800c840,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmpne.16",
	  OPCODE_INFO3 (0xf800c800,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmphs.u8",
	  OPCODE_INFO3 (0xf800c940,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmphs.s8",
	  OPCODE_INFO3 (0xf800c9c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmphs.u16",
	  OPCODE_INFO3 (0xf800c900,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmphs.s16",
	  OPCODE_INFO3 (0xf800c980,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmplt.u8",
	  OPCODE_INFO3 (0xf800ca40,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmplt.s8",
	  OPCODE_INFO3 (0xf800cac0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmplt.u16",
	  OPCODE_INFO3 (0xf800ca00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pcmplt.s16",
	  OPCODE_INFO3 (0xf800ca80,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmax.u8",
	  OPCODE_INFO3 (0xf800cc40,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmax.s8",
	  OPCODE_INFO3 (0xf800ccc0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmax.u16",
	  OPCODE_INFO3 (0xf800cc00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmax.s16",
	  OPCODE_INFO3 (0xf800cc80,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("max.u32",
	  OPCODE_INFO3 (0xf800cc20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("max.s32",
	  OPCODE_INFO3 (0xf800cca0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmin.u8",
	  OPCODE_INFO3 (0xf800cd40,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmin.s8",
	  OPCODE_INFO3 (0xf800cdc0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmin.u16",
	  OPCODE_INFO3 (0xf800cd00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmin.s16",
	  OPCODE_INFO3 (0xf800cd80,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("min.u32",
	  OPCODE_INFO3 (0xf800cd20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("min.s32",
	  OPCODE_INFO3 (0xf800cda0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("sel",
	  OPCODE_INFO4 (0xf8009000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(5_9, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    /* The followings are miscs.  */
    OP32 ("psabsa.u8",
	  OPCODE_INFO3 (0xf800e040,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("psabsaa.u8",
	  OPCODE_INFO3 (0xf800e140,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("divul",
	  OPCODE_INFO3 (0xf800e260,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_3E3R3),
    OP32 ("divsl",
	  OPCODE_INFO3 (0xf800e2e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_3E3R3),
    OP32 ("mulaca.s8",
	  OPCODE_INFO3 (0xf800e4c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    /* The followings are shift instructions.  */
    OP32 ("asri.s32.r",
	  OPCODE_INFO3 (0xf800d1a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM5b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("asr.s32.r",
	  OPCODE_INFO3 (0xf800d1e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("lsri.u32.r",
	  OPCODE_INFO3 (0xf800d320,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM5b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("lsr.u32.r",
	  OPCODE_INFO3 (0xf800d360,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("lsli.u32.s",
	  OPCODE_INFO3 (0xf800d520,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM5b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("lsli.s32.s",
	  OPCODE_INFO3 (0xf800d5a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM5b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("lsl.u32.s",
	  OPCODE_INFO3 (0xf800d560,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("lsl.s32.s",
	  OPCODE_INFO3 (0xf800d5e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pasri.s16",
	  OPCODE_INFO3 (0xf800d080,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pasr.s16",
	  OPCODE_INFO3 (0xf800d0c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pasri.s16.r",
	  OPCODE_INFO3 (0xf800d180,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pasr.s16.r",
	  OPCODE_INFO3 (0xf800d1c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsri.u16",
	  OPCODE_INFO3 (0xf800d200,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsr.u16",
	  OPCODE_INFO3 (0xf800d240,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsri.u16.r",
	  OPCODE_INFO3 (0xf800d300,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsr.u16.r",
	  OPCODE_INFO3 (0xf800d340,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsli.16",
	  OPCODE_INFO3 (0xf800d400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsl.16",
	  OPCODE_INFO3 (0xf800d440,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsli.u16.s",
	  OPCODE_INFO3 (0xf800d500,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsli.s16.s",
	  OPCODE_INFO3 (0xf800d580,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsl.u16.s",
	  OPCODE_INFO3 (0xf800d540,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("plsl.s16.s",
	  OPCODE_INFO3 (0xf800d5c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    /* The following are package & unpackage instructions.  */
    OP32 ("pkg",
	  OPCODE_INFO5 (0xf800a000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(5_8, IMM4b, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(9_12, OIMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("dexti",
	  OPCODE_INFO4 (0xf8009800,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(5_9, IMM5b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("dext",
	  OPCODE_INFO4 (0xf8009c00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT),
			(5_9, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pkgll",
	  OPCODE_INFO3 (0xf800d840,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pkghh",
	  OPCODE_INFO3 (0xf800d860,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pext.u8.e",
	  OPCODE_INFO2 (0xf800d900,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pext.s8.e",
	  OPCODE_INFO2 (0xf800d980,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pextx.u8.e",
	  OPCODE_INFO2 (0xf800d920,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pextx.s8.e",
	  OPCODE_INFO2 (0xf800d9a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("narl",
	  OPCODE_INFO3 (0xf800da00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("narh",
	  OPCODE_INFO3 (0xf800da20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("narlx",
	  OPCODE_INFO3 (0xf800da40,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("narhx",
	  OPCODE_INFO3 (0xf800da60,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("clipi.u32",
	  OPCODE_INFO3 (0xf800db00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, IMM5b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("clipi.s32",
	  OPCODE_INFO3 (0xf800db80,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM5b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("clip.u32",
	  OPCODE_INFO3 (0xf800db20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("clip.s32",
	  OPCODE_INFO3 (0xf800dba0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pclipi.u16",
	  OPCODE_INFO3 (0xf800db40,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, IMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pclipi.s16",
	  OPCODE_INFO3 (0xf800dbc0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, OIMM4b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pclip.u16",
	  OPCODE_INFO3 (0xf800db60,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pclip.s16",
	  OPCODE_INFO3 (0xf800dbe0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pabs.s8.s",
	  OPCODE_INFO2 (0xf800dc80,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pabs.s16.s",
	  OPCODE_INFO2 (0xf800dca0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("abs.s32.s",
	  OPCODE_INFO2 (0xf800dcc0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pneg.s8.s",
	  OPCODE_INFO2 (0xf800dd80,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pneg.s16.s",
	  OPCODE_INFO2 (0xf800dda0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("neg.s32.s",
	  OPCODE_INFO2 (0xf800ddc0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("dup.8",
	  OPCODE_INFO3 (0xf800de00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(5_6, IMM2b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("dup.16",
	  OPCODE_INFO3 (0xf800df00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(5_6, IMM1b, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    /* The followings are multiplication instructions.  */
    OP32 ("mul.u32",
	  OPCODE_INFO3 (0xf8008000,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_3E3R1),
    OP32 ("mul.s32",
	  OPCODE_INFO3 (0xf8008200,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_3E3R1),
    OP32 ("mula.u32",
	  OPCODE_INFO3 (0xf8008080,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_3E3R1),
    OP32 ("mula.s32",
	  OPCODE_INFO3 (0xf8008280,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_3E3R1),
    OP32 ("mula.32.l",
	  OPCODE_INFO3 (0xf8008440,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_3E3R1),
    OP32 ("mulall.s16.s",
	  OPCODE_INFO3 (0xf80081a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_3E3R1),
    OP32 ("muls.u32",
	  OPCODE_INFO3 (0xf80080c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("muls.s32",
	  OPCODE_INFO3 (0xf80082c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mula.u32.s",
	  OPCODE_INFO3 (0xf8008180,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mula.s32.s",
	  OPCODE_INFO3 (0xf8008380,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("muls.u32.s",
	  OPCODE_INFO3 (0xf80081c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("muls.s32.s",
	  OPCODE_INFO3 (0xf80083c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mul.s32.h",
	  OPCODE_INFO3 (0xf8008400,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mul.s32.rh",
	  OPCODE_INFO3 (0xf8008600,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("rmul.s32.h",
	  OPCODE_INFO3 (0xf8008500,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("rmul.s32.rh",
	  OPCODE_INFO3 (0xf8008700,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mula.s32.hs",
	  OPCODE_INFO3 (0xf8008580,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("muls.s32.hs",
	  OPCODE_INFO3 (0xf80085c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mula.s32.rhs",
	  OPCODE_INFO3 (0xf8008780,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("muls.s32.rhs",
	  OPCODE_INFO3 (0xf80087c0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulxl.s32",
	  OPCODE_INFO3 (0xf8008800,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulxl.s32.r",
	  OPCODE_INFO3 (0xf8008a00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulxh.s32",
	  OPCODE_INFO3 (0xf8008c00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulxh.s32.r",
	  OPCODE_INFO3 (0xf8008e00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("rmulxl.s32",
	  OPCODE_INFO3 (0xf8008900,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("rmulxl.s32.r",
	  OPCODE_INFO3 (0xf8008b00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("rmulxh.s32",
	  OPCODE_INFO3 (0xf8008d00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("rmulxh.s32.r",
	  OPCODE_INFO3 (0xf8008f00,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulaxl.s32.s",
	  OPCODE_INFO3 (0xf8008980,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulaxl.s32.rs",
	  OPCODE_INFO3 (0xf8008b80,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulaxh.s32.s",
	  OPCODE_INFO3 (0xf8008d80,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulaxh.s32.rs",
	  OPCODE_INFO3 (0xf8008f80,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulll.s16",
	  OPCODE_INFO3 (0xf8008020,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulhh.s16",
	  OPCODE_INFO3 (0xf8008260,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulhl.s16",
	  OPCODE_INFO3 (0xf8008220,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("rmulll.s16",
	  OPCODE_INFO3 (0xf8008120,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("rmulhh.s16",
	  OPCODE_INFO3 (0xf8008360,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("rmulhl.s16",
	  OPCODE_INFO3 (0xf8008320,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulahh.s16.s",
	  OPCODE_INFO3 (0xf80083e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulahl.s16.s",
	  OPCODE_INFO3 (0xf80083a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulall.s16.e",
	  OPCODE_INFO3 (0xf80080a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulahh.s16.e",
	  OPCODE_INFO3 (0xf80082e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulahl.s16.e",
	  OPCODE_INFO3 (0xf80080e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmul.u16",
	  OPCODE_INFO3 (0xf80084a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmulx.u16",
	  OPCODE_INFO3 (0xf80084e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmul.s16",
	  OPCODE_INFO3 (0xf8008420,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("pmulx.s16",
	  OPCODE_INFO3 (0xf8008460,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("prmul.s16",
	  OPCODE_INFO3 (0xf8008520,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("prmulx.s16",
	  OPCODE_INFO3 (0xf8008560,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("prmul.s16.h",
	  OPCODE_INFO3 (0xf80085a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("prmul.s16.rh",
	  OPCODE_INFO3 (0xf80087a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("prmulx.s16.h",
	  OPCODE_INFO3 (0xf80085e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("prmulx.s16.rh",
	  OPCODE_INFO3 (0xf80087e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulca.s16.s",
	  OPCODE_INFO3 (0xf8008920,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulcax.s16.s",
	  OPCODE_INFO3 (0xf8008960,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulcs.s16",
	  OPCODE_INFO3 (0xf8008a20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulcsr.s16",
	  OPCODE_INFO3 (0xf8008a60,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulcsx.s16",
	  OPCODE_INFO3 (0xf8008c20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulaca.s16.s",
	  OPCODE_INFO3 (0xf80089a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulacax.s16.s",
	  OPCODE_INFO3 (0xf80089e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulacs.s16.s",
	  OPCODE_INFO3 (0xf8008ba0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulacsr.s16.s",
	  OPCODE_INFO3 (0xf8008be0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulacsx.s16.s",
	  OPCODE_INFO3 (0xf8008da0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulsca.s16.s",
	  OPCODE_INFO3 (0xf8008de0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulscax.s16.s",
	  OPCODE_INFO3 (0xf8008fa0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulaca.s16.e",
	  OPCODE_INFO3 (0xf80088a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulacax.s16.e",
	  OPCODE_INFO3 (0xf80088e0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulacs.s16.e",
	  OPCODE_INFO3 (0xf8008aa0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulacsr.s16.e",
	  OPCODE_INFO3 (0xf8008ae0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulacsx.s16.e",
	  OPCODE_INFO3 (0xf8008ca0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulsca.s16.e",
	  OPCODE_INFO3 (0xf8008ce0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),
    OP32 ("mulscax.s16.e",
	  OPCODE_INFO3 (0xf8008ea0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT),
			(21_25, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_DSP_ENHANCE),

    /* The followings are vdsp instructions for ck810.  */
    OP32 ("vdup.8",
	  OPCODE_INFO2 (0xf8000e80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vdup.16",
	  OPCODE_INFO2 (0xf8100e80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vdup.32",
	  OPCODE_INFO2 (0xfa000e80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmfvr.u8",
	  OPCODE_INFO2 (0xf8001200,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmfvr.u16",
	  OPCODE_INFO2 (0xf8001220,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmfvr.u32",
	  OPCODE_INFO2 (0xf8001240,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmfvr.s8",
	  OPCODE_INFO2 (0xf8001280,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmfvr.s16",
	  OPCODE_INFO2 (0xf80012a0,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmtvr.u8",
	  OPCODE_INFO2 (0xf8001300,
			(0_3or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmtvr.u16",
	  OPCODE_INFO2 (0xf8001320,
			(0_3or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vins.8",
	  OPCODE_INFO2 (0xf8001400,
			(0_3or5_8, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vins.16",
	  OPCODE_INFO2 (0xf8101400,
			(0_3or5_8, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vins.32",
	  OPCODE_INFO2 (0xfa001400,
			(0_3or5_8, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT),
			(16_19or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmtvr.u32",
	  OPCODE_INFO2 (0xf8001340,
			(0_3or21_24, VREG_WITH_INDEX, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vldd.8",
	  SOPCODE_INFO2 (0xf8002000,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_3_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldd.16",
	  SOPCODE_INFO2 (0xf8002100,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_3_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldd.32",
	  SOPCODE_INFO2 (0xf8002200,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_3_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldq.8",
	  SOPCODE_INFO2 (0xf8002400,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_4_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldq.16",
	  SOPCODE_INFO2 (0xf8002500,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_4_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldq.32",
	  SOPCODE_INFO2 (0xf8002600,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_4_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstd.8",
	  SOPCODE_INFO2 (0xf8002800,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_3_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstd.16",
	  SOPCODE_INFO2 (0xf8002900,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_3_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstd.32",
	  SOPCODE_INFO2 (0xf8002a00,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_3_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstq.8",
	  SOPCODE_INFO2 (0xf8002c00,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_4_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstq.16",
	  SOPCODE_INFO2 (0xf8002d00,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_4_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstq.32",
	  SOPCODE_INFO2 (0xf8002e00,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(4_7or21_24, IMM_FLDST, OPRND_SHIFT_4_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldrd.8",
	  SOPCODE_INFO2 (0xf8003000,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldrd.16",
	  SOPCODE_INFO2 (0xf8003100,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldrd.32",
	  SOPCODE_INFO2 (0xf8003200,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldrq.8",
	  SOPCODE_INFO2 (0xf8003400,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldrq.16",
	  SOPCODE_INFO2 (0xf8003500,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vldrq.32",
	  SOPCODE_INFO2 (0xf8003600,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstrd.8",
	  SOPCODE_INFO2 (0xf8003800,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstrd.16",
	  SOPCODE_INFO2 (0xf8003900,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstrd.32",
	  SOPCODE_INFO2 (0xf8003a00,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstrq.8",
	  SOPCODE_INFO2 (0xf8003c00,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstrq.16",
	  SOPCODE_INFO2 (0xf8003d00,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vstrq.32",
	  SOPCODE_INFO2 (0xf8003e00,
			 (0_3, VREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20, AREG, OPRND_SHIFT_0_BIT),
					(5_6or21_25, AREG_WITH_LSHIFT_FPU, OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_VDSP),
    OP32 ("vmov",
	  OPCODE_INFO2 (0xf8000c00,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.eu8",
	  OPCODE_INFO2 (0xf8000060,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.eu16",
	  OPCODE_INFO2 (0xf8100060,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.es8",
	  OPCODE_INFO2 (0xf8000070,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.es16",
	  OPCODE_INFO2 (0xf8100070,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.eu8",
	  OPCODE_INFO2 (0xf8000c20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.eu16",
	  OPCODE_INFO2 (0xf8100c20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.es8",
	  OPCODE_INFO2 (0xf8000c30,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.es16",
	  OPCODE_INFO2 (0xf8100c30,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.u16.l",
	  OPCODE_INFO2 (0xf8100d00,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.u32.l",
	  OPCODE_INFO2 (0xfa000d00,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.s16.l",
	  OPCODE_INFO2 (0xf8100d10,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.s32.l",
	  OPCODE_INFO2 (0xfa000d10,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.u16.sl",
	  OPCODE_INFO2 (0xf8100d40,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.u32.sl",
	  OPCODE_INFO2 (0xfa000d40,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.s16.sl",
	  OPCODE_INFO2 (0xf8100d50,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.s32.sl",
	  OPCODE_INFO2 (0xfa000d50,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.u16.h",
	  OPCODE_INFO2 (0xf8100d60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.u32.h",
	  OPCODE_INFO2 (0xfa000d60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.s16.h",
	  OPCODE_INFO2 (0xf8100d70,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.s32.h",
	  OPCODE_INFO2 (0xfa000d70,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.u16.rh",
	  OPCODE_INFO2 (0xf8100d80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.u32.rh",
	  OPCODE_INFO2 (0xfa000d80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.s16.rh",
	  OPCODE_INFO2 (0xf8100d90,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmov.s32.rh",
	  OPCODE_INFO2 (0xfa000d90,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vstou.u16.sl",
	  OPCODE_INFO2 (0xf8100dc0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vstou.u32.sl",
	  OPCODE_INFO2 (0xfa000dc0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vstou.s16.sl",
	  OPCODE_INFO2 (0xf8100dd0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vstou.s32.sl",
	  OPCODE_INFO2 (0xfa000dd0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vrev.8",
	  OPCODE_INFO2 (0xf8000e60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vrev.16",
	  OPCODE_INFO2 (0xf8100e60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vrev.32",
	  OPCODE_INFO2 (0xfa000e60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcnt1.8",
	  OPCODE_INFO2 (0xf8000ea0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vclz.8",
	  OPCODE_INFO2 (0xf8000ec0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vclz.16",
	  OPCODE_INFO2 (0xf8100ec0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vclz.32",
	  OPCODE_INFO2 (0xfa000ec0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcls.u8",
	  OPCODE_INFO2 (0xf8000ee0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcls.u16",
	  OPCODE_INFO2 (0xf8100ee0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcls.u32",
	  OPCODE_INFO2 (0xfa000ee0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcls.s8",
	  OPCODE_INFO2 (0xf8000ef0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcls.s16",
	  OPCODE_INFO2 (0xf8100ef0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcls.s32",
	  OPCODE_INFO2 (0xfa000ef0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vabs.s8",
	  OPCODE_INFO2 (0xf8001010,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vabs.s16",
	  OPCODE_INFO2 (0xf8101010,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vabs.s32",
	  OPCODE_INFO2 (0xfa001010,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vabs.u8.s",
	  OPCODE_INFO2 (0xf8001040,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vabs.u16.s",
	  OPCODE_INFO2 (0xf8101040,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vabs.u32.s",
	  OPCODE_INFO2 (0xfa001040,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vabs.s8.s",
	  OPCODE_INFO2 (0xf8001050,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vabs.s16.s",
	  OPCODE_INFO2 (0xf8101050,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vabs.s32.s",
	  OPCODE_INFO2 (0xfa001050,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.u8",
	  OPCODE_INFO2 (0xf8001080,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.u16",
	  OPCODE_INFO2 (0xf8101080,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.u32",
	  OPCODE_INFO2 (0xfa001080,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.s8",
	  OPCODE_INFO2 (0xf8001090,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.s16",
	  OPCODE_INFO2 (0xf8101090,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.s32",
	  OPCODE_INFO2 (0xfa001090,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.u8.s",
	  OPCODE_INFO2 (0xf80010c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.u16.s",
	  OPCODE_INFO2 (0xf81010c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.u32.s",
	  OPCODE_INFO2 (0xfa0010c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.s8.s",
	  OPCODE_INFO2 (0xf80010d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.s16.s",
	  OPCODE_INFO2 (0xf81010d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vneg.s32.s",
	  OPCODE_INFO2 (0xfa0010d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphsz.u8",
	  OPCODE_INFO2 (0xf8000880,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphsz.u16",
	  OPCODE_INFO2 (0xf8100880,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphsz.u32",
	  OPCODE_INFO2 (0xfa000880,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphsz.s8",
	  OPCODE_INFO2 (0xf8000890,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphsz.s16",
	  OPCODE_INFO2 (0xf8100890,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphsz.s32",
	  OPCODE_INFO2 (0xfa000890,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpltz.u8",
	  OPCODE_INFO2 (0xf80008a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpltz.u16",
	  OPCODE_INFO2 (0xf81008a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpltz.u32",
	  OPCODE_INFO2 (0xfa0008a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpltz.s8",
	  OPCODE_INFO2 (0xf80008b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpltz.s16",
	  OPCODE_INFO2 (0xf81008b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpltz.s32",
	  OPCODE_INFO2 (0xfa0008b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpnez.u8",
	  OPCODE_INFO2 (0xf80008c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpnez.u16",
	  OPCODE_INFO2 (0xf81008c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpnez.u32",
	  OPCODE_INFO2 (0xfa0008c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpnez.s8",
	  OPCODE_INFO2 (0xf80008d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpnez.s16",
	  OPCODE_INFO2 (0xf81008d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpnez.s32",
	  OPCODE_INFO2 (0xfa0008d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vtrch.8",
	  OPCODE_INFO3 (0xf8000f40,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vtrch.16",
	  OPCODE_INFO3 (0xf8100f40,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vtrch.32",
	  OPCODE_INFO3 (0xfa000f40,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vtrcl.8",
	  OPCODE_INFO3 (0xf8000f60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vtrcl.16",
	  OPCODE_INFO3 (0xf8100f60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vtrcl.32",
	  OPCODE_INFO3 (0xfa000f60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.u8",
	  OPCODE_INFO3 (0xf8000000,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.u16",
	  OPCODE_INFO3 (0xf8100000,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.u32",
	  OPCODE_INFO3 (0xfa000000,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.s8",
	  OPCODE_INFO3 (0xf8000010,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.s16",
	  OPCODE_INFO3 (0xf8100010,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.s32",
	  OPCODE_INFO3 (0xfa000010,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.eu8",
	  OPCODE_INFO3 (0xf8000020,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.eu16",
	  OPCODE_INFO3 (0xf8100020,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.es8",
	  OPCODE_INFO3 (0xf8000030,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.es16",
	  OPCODE_INFO3 (0xf8100030,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.u8",
	  OPCODE_INFO3 (0xf8000040,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.u16",
	  OPCODE_INFO3 (0xf8100040,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.u32",
	  OPCODE_INFO3 (0xfa000040,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.s8",
	  OPCODE_INFO3 (0xf8000050,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.s16",
	  OPCODE_INFO3 (0xf8100050,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcadd.s32",
	  OPCODE_INFO3 (0xfa000050,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.xu16.sl",
	  OPCODE_INFO3 (0xf8100140,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.xu32.sl",
	  OPCODE_INFO3 (0xfa000140,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.xs16.sl",
	  OPCODE_INFO3 (0xf8100150,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.xs32.sl",
	  OPCODE_INFO3 (0xfa000150,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.xu16",
	  OPCODE_INFO3 (0xf8100160,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.xu32",
	  OPCODE_INFO3 (0xfa000160,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.xs16",
	  OPCODE_INFO3 (0xf8100170,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.xs32",
	  OPCODE_INFO3 (0xfa000170,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.u8",
	  OPCODE_INFO3 (0xf8000180,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.u16",
	  OPCODE_INFO3 (0xf8100180,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.u32",
	  OPCODE_INFO3 (0xfa000180,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.s8",
	  OPCODE_INFO3 (0xf8000190,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.s16",
	  OPCODE_INFO3 (0xf8100190,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.s32",
	  OPCODE_INFO3 (0xfa000190,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.u8.r",
	  OPCODE_INFO3 (0xf80001a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.u16.r",
	  OPCODE_INFO3 (0xf81001a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.u32.r",
	  OPCODE_INFO3 (0xfa0001a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.s8.r",
	  OPCODE_INFO3 (0xf80001b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.s16.r",
	  OPCODE_INFO3 (0xf81001b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vaddh.s32.r",
	  OPCODE_INFO3 (0xfa0001b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.u8.s",
	  OPCODE_INFO3 (0xf80001c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.u16.s",
	  OPCODE_INFO3 (0xf81001c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.u32.s",
	  OPCODE_INFO3 (0xfa0001c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.s8.s",
	  OPCODE_INFO3 (0xf80001d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.s16.s",
	  OPCODE_INFO3 (0xf81001d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vadd.s32.s",
	  OPCODE_INFO3 (0xfa0001d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.u8",
	  OPCODE_INFO3 (0xf8000200,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.u16",
	  OPCODE_INFO3 (0xf8100200,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.u32",
	  OPCODE_INFO3 (0xfa000200,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.s8",
	  OPCODE_INFO3 (0xf8000210,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.s16",
	  OPCODE_INFO3 (0xf8100210,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.s32",
	  OPCODE_INFO3 (0xfa000210,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.eu8",
	  OPCODE_INFO3 (0xf8000220,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.eu16",
	  OPCODE_INFO3 (0xf8100220,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.eu32",
	  OPCODE_INFO3 (0xfa000220,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.es8",
	  OPCODE_INFO3 (0xf8000230,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.es16",
	  OPCODE_INFO3 (0xf8100230,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.es32",
	  OPCODE_INFO3 (0xfa000230,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.u8",
	  OPCODE_INFO3 (0xf8000240,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.u16",
	  OPCODE_INFO3 (0xf8100240,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.u32",
	  OPCODE_INFO3 (0xfa000240,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.s8",
	  OPCODE_INFO3 (0xf8000250,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.s16",
	  OPCODE_INFO3 (0xf8100250,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.s32",
	  OPCODE_INFO3 (0xfa000250,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.eu8",
	  OPCODE_INFO3 (0xf8000260,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.eu16",
	  OPCODE_INFO3 (0xf8100260,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.es8",
	  OPCODE_INFO3 (0xf8000270,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabs.es16",
	  OPCODE_INFO3 (0xf8100270,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.u8",
	  OPCODE_INFO3 (0xf8000280,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.u16",
	  OPCODE_INFO3 (0xf8100280,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.u32",
	  OPCODE_INFO3 (0xfa000280,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.s8",
	  OPCODE_INFO3 (0xf8000290,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.s16",
	  OPCODE_INFO3 (0xf8100290,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.s32",
	  OPCODE_INFO3 (0xfa000290,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.eu8",
	  OPCODE_INFO3 (0xf80002a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.eu16",
	  OPCODE_INFO3 (0xf81002a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.es8",
	  OPCODE_INFO3 (0xf80002b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsabsa.es16",
	  OPCODE_INFO3 (0xf81002b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.xu16",
	  OPCODE_INFO3 (0xf8100360,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.xu32",
	  OPCODE_INFO3 (0xfa000360,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.xs16",
	  OPCODE_INFO3 (0xf8100370,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.xs32",
	  OPCODE_INFO3 (0xfa000370,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.u8",
	  OPCODE_INFO3 (0xf8000380,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.u16",
	  OPCODE_INFO3 (0xf8100380,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.u32",
	  OPCODE_INFO3 (0xfa000380,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.s8",
	  OPCODE_INFO3 (0xf8000390,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.s16",
	  OPCODE_INFO3 (0xf8100390,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.s32",
	  OPCODE_INFO3 (0xfa000390,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.u8.r",
	  OPCODE_INFO3 (0xf80003a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.u16.r",
	  OPCODE_INFO3 (0xf81003a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.u32.r",
	  OPCODE_INFO3 (0xfa0003a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.s8.r",
	  OPCODE_INFO3 (0xf80003b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.s16.r",
	  OPCODE_INFO3 (0xf81003b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsubh.s32.r",
	  OPCODE_INFO3 (0xfa0003b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.u8.s",
	  OPCODE_INFO3 (0xf80003c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.u16.s",
	  OPCODE_INFO3 (0xf81003c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.u32.s",
	  OPCODE_INFO3 (0xfa0003c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.s8.s",
	  OPCODE_INFO3 (0xf80003d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.s16.s",
	  OPCODE_INFO3 (0xf81003d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vsub.s32.s",
	  OPCODE_INFO3 (0xfa0003d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.u8",
	  OPCODE_INFO3 (0xf8000400,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.u16",
	  OPCODE_INFO3 (0xf8100400,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.u32",
	  OPCODE_INFO3 (0xfa000400,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.s8",
	  OPCODE_INFO3 (0xf8000410,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.s16",
	  OPCODE_INFO3 (0xf8100410,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.s32",
	  OPCODE_INFO3 (0xfa000410,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.eu8",
	  OPCODE_INFO3 (0xf8000420,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.eu16",
	  OPCODE_INFO3 (0xf8100420,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.es8",
	  OPCODE_INFO3 (0xf8000430,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmul.es16",
	  OPCODE_INFO3 (0xf8100430,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.u8",
	  OPCODE_INFO3 (0xf8000440,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.u16",
	  OPCODE_INFO3 (0xf8100440,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.u32",
	  OPCODE_INFO3 (0xfa000440,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.s8",
	  OPCODE_INFO3 (0xf8000450,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.s16",
	  OPCODE_INFO3 (0xf8100450,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.s32",
	  OPCODE_INFO3 (0xfa000450,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.eu8",
	  OPCODE_INFO3 (0xf8000460,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.eu16",
	  OPCODE_INFO3 (0xf8100460,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.eu32",
	  OPCODE_INFO3 (0xfa000460,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.es8",
	  OPCODE_INFO3 (0xf8000470,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.es16",
	  OPCODE_INFO3 (0xf8100470,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmula.es32",
	  OPCODE_INFO3 (0xfa000470,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.u8",
	  OPCODE_INFO3 (0xf8000480,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.u16",
	  OPCODE_INFO3 (0xf8100480,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.u32",
	  OPCODE_INFO3 (0xfa000480,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.s8",
	  OPCODE_INFO3 (0xf8000490,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.s16",
	  OPCODE_INFO3 (0xf8100490,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.s32",
	  OPCODE_INFO3 (0xfa000490,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.eu8",
	  OPCODE_INFO3 (0xf80004a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.eu16",
	  OPCODE_INFO3 (0xf81004a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.es8",
	  OPCODE_INFO3 (0xf80004b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmuls.es16",
	  OPCODE_INFO3 (0xf81004b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.u8",
	  OPCODE_INFO3 (0xf8000680,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.u16",
	  OPCODE_INFO3 (0xf8100680,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.u32",
	  OPCODE_INFO3 (0xfa000680,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.s8",
	  OPCODE_INFO3 (0xf8000690,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.s16",
	  OPCODE_INFO3 (0xf8100690,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.s32",
	  OPCODE_INFO3 (0xfa000690,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.u8.r",
	  OPCODE_INFO3 (0xf80006c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.u16.r",
	  OPCODE_INFO3 (0xf81006c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.u32.r",
	  OPCODE_INFO3 (0xfa0006c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.s8.r",
	  OPCODE_INFO3 (0xf80006d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.s16.r",
	  OPCODE_INFO3 (0xf81006d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.u8",
	  OPCODE_INFO3 (0xf8000600,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.u16",
	  OPCODE_INFO3 (0xf8100600,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.u32",
	  OPCODE_INFO3 (0xfa000600,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.s8",
	  OPCODE_INFO3 (0xf8000610,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.s16",
	  OPCODE_INFO3 (0xf8100610,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.s32",
	  OPCODE_INFO3 (0xfa000610,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.u8.r",
	  OPCODE_INFO3 (0xf8000640,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.u16.r",
	  OPCODE_INFO3 (0xf8100640,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.u32.r",
	  OPCODE_INFO3 (0xfa000640,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.s8.r",
	  OPCODE_INFO3 (0xf8000650,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.s16.r",
	  OPCODE_INFO3 (0xf8100650,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshri.s32.r",
	  OPCODE_INFO3 (0xfa000650,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.s32.r",
	  OPCODE_INFO3 (0xfa0006d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshr.s32.r",
	  OPCODE_INFO3 (0xfa0006d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.u8",
	  OPCODE_INFO3 (0xf8000780,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.u16",
	  OPCODE_INFO3 (0xf8100780,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.u32",
	  OPCODE_INFO3 (0xfa000780,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.s8",
	  OPCODE_INFO3 (0xf8000790,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.s16",
	  OPCODE_INFO3 (0xf8100790,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.s32",
	  OPCODE_INFO3 (0xfa000790,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.u8.s",
	  OPCODE_INFO3 (0xf80007c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.u16.s",
	  OPCODE_INFO3 (0xf81007c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.u32.s",
	  OPCODE_INFO3 (0xfa0007c0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.s8.s",
	  OPCODE_INFO3 (0xf80007d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.s16.s",
	  OPCODE_INFO3 (0xf81007d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshl.s32.s",
	  OPCODE_INFO3 (0xfa0007d0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.u8",
	  OPCODE_INFO3 (0xf8000700,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.u16",
	  OPCODE_INFO3 (0xf8100700,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.u32",
	  OPCODE_INFO3 (0xfa000700,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.s8",
	  OPCODE_INFO3 (0xf8000710,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.s16",
	  OPCODE_INFO3 (0xf8100710,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.s32",
	  OPCODE_INFO3 (0xfa000710,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.u8.s",
	  OPCODE_INFO3 (0xf8000740,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.u16.s",
	  OPCODE_INFO3 (0xf8100740,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.u32.s",
	  OPCODE_INFO3 (0xfa000740,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.s8.s",
	  OPCODE_INFO3 (0xf8000750,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.s16.s",
	  OPCODE_INFO3 (0xf8100750,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vshli.s32.s",
	  OPCODE_INFO3 (0xfa000750,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(5or21_24, IMM5b_VSH, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphs.u8",
	  OPCODE_INFO3 (0xf8000800,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphs.u16",
	  OPCODE_INFO3 (0xf8100800,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphs.u32",
	  OPCODE_INFO3 (0xfa000800,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphs.s8",
	  OPCODE_INFO3 (0xf8000810,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphs.s16",
	  OPCODE_INFO3 (0xf8100810,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmphs.s32",
	  OPCODE_INFO3 (0xfa000810,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmplt.u8",
	  OPCODE_INFO3 (0xf8000820,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmplt.u16",
	  OPCODE_INFO3 (0xf8100820,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmplt.u32",
	  OPCODE_INFO3 (0xfa000820,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmplt.s8",
	  OPCODE_INFO3 (0xf8000830,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmplt.s16",
	  OPCODE_INFO3 (0xf8100830,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmplt.s32",
	  OPCODE_INFO3 (0xfa000830,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpne.u8",
	  OPCODE_INFO3 (0xf8000840,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpne.u16",
	  OPCODE_INFO3 (0xf8100840,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpne.u32",
	  OPCODE_INFO3 (0xfa000840,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpne.s8",
	  OPCODE_INFO3 (0xf8000850,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpne.s16",
	  OPCODE_INFO3 (0xf8100850,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmpne.s32",
	  OPCODE_INFO3 (0xfa000850,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmax.u8",
	  OPCODE_INFO3 (0xf8000900,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmax.u16",
	  OPCODE_INFO3 (0xf8100900,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmax.u32",
	  OPCODE_INFO3 (0xfa000900,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmax.s8",
	  OPCODE_INFO3 (0xf8000910,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmax.s16",
	  OPCODE_INFO3 (0xf8100910,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmax.s32",
	  OPCODE_INFO3 (0xfa000910,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmin.u8",
	  OPCODE_INFO3 (0xf8000920,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmin.u16",
	  OPCODE_INFO3 (0xf8100920,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmin.u32",
	  OPCODE_INFO3 (0xfa000920,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmin.s8",
	  OPCODE_INFO3 (0xf8000930,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmin.s16",
	  OPCODE_INFO3 (0xf8100930,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vmin.s32",
	  OPCODE_INFO3 (0xfa000930,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmax.u8",
	  OPCODE_INFO3 (0xf8000980,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmax.u16",
	  OPCODE_INFO3 (0xf8100980,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmax.u32",
	  OPCODE_INFO3 (0xfa000980,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmax.s8",
	  OPCODE_INFO3 (0xf8000990,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmax.s16",
	  OPCODE_INFO3 (0xf8100990,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmax.s32",
	  OPCODE_INFO3 (0xfa000990,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmin.u8",
	  OPCODE_INFO3 (0xf80009a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmin.u16",
	  OPCODE_INFO3 (0xf81009a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmin.u32",
	  OPCODE_INFO3 (0xfa0009a0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmin.s8",
	  OPCODE_INFO3 (0xf80009b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmin.s16",
	  OPCODE_INFO3 (0xf81009b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vcmin.s32",
	  OPCODE_INFO3 (0xfa0009b0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vand.8",
	  OPCODE_INFO3 (0xf8000a00,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vand.16",
	  OPCODE_INFO3 (0xf8100a00,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vand.32",
	  OPCODE_INFO3 (0xfa000a00,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vandn.8",
	  OPCODE_INFO3 (0xf8000a20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vandn.16",
	  OPCODE_INFO3 (0xf8100a20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vandn.32",
	  OPCODE_INFO3 (0xfa000a20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vor.8",
	  OPCODE_INFO3 (0xf8000a40,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vor.16",
	  OPCODE_INFO3 (0xf8100a40,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vor.32",
	  OPCODE_INFO3 (0xfa000a40,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vnor.8",
	  OPCODE_INFO3 (0xf8000a60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vnor.16",
	  OPCODE_INFO3 (0xf8100a60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vnor.32",
	  OPCODE_INFO3 (0xfa000a60,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vxor.8",
	  OPCODE_INFO3 (0xf8000a80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vxor.16",
	  OPCODE_INFO3 (0xf8100a80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vxor.32",
	  OPCODE_INFO3 (0xfa000a80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vtst.8",
	  OPCODE_INFO3 (0xf8000b20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vtst.16",
	  OPCODE_INFO3 (0xf8100b20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vtst.32",
	  OPCODE_INFO3 (0xfa000b20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vbpermz.8",
	  OPCODE_INFO3 (0xf8000f00,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vbpermz.16",
	  OPCODE_INFO3 (0xf8100f00,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vbpermz.32",
	  OPCODE_INFO3 (0xfa000f00,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vbperm.8",
	  OPCODE_INFO3 (0xf8000f20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vbperm.16",
	  OPCODE_INFO3 (0xf8100f20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vbperm.32",
	  OPCODE_INFO3 (0xfa000f20,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vdch.8",
	  OPCODE_INFO3 (0xf8000fc0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vdch.16",
	  OPCODE_INFO3 (0xf8100fc0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vdch.32",
	  OPCODE_INFO3 (0xfa000fc0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vdcl.8",
	  OPCODE_INFO3 (0xf8000fe0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vdcl.16",
	  OPCODE_INFO3 (0xf8100fe0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vdcl.32",
	  OPCODE_INFO3 (0xfa000fe0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vich.8",
	  OPCODE_INFO3 (0xf8000f80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vich.16",
	  OPCODE_INFO3 (0xf8100f80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vich.32",
	  OPCODE_INFO3 (0xfa000f80,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vicl.8",
	  OPCODE_INFO3 (0xf8000fa0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vicl.16",
	  OPCODE_INFO3 (0xf8100fa0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),
    OP32 ("vicl.32",
	  OPCODE_INFO3 (0xfa000fa0,
			(0_3, VREG, OPRND_SHIFT_0_BIT),
			(16_19, VREG, OPRND_SHIFT_0_BIT),
			(21_24, VREG, OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_VDSP),

#define OPRND_SHIFT0(mask, type) (mask, type, OPRND_SHIFT_0_BIT)
#define OPRND_SHIFT1(mask, type) (mask, type, OPRND_SHIFT_1_BIT)
#define OPRND_SHIFT2(mask, type) (mask, type, OPRND_SHIFT_2_BIT)
#define OPRND_SHIFT3(mask, type) (mask, type, OPRND_SHIFT_3_BIT)
#define OPRND_SHIFT4(mask, type) (mask, type, OPRND_SHIFT_4_BIT)

/* The followings are 860 floating instructions.  */
    OP32 ("fadd.16",
	  OPCODE_INFO3 (0xf400c800,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("faddh",
	  OPCODE_INFO3 (0xf400c800,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsub.16",
	  OPCODE_INFO3 (0xf400c820,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsubh",
	  OPCODE_INFO3 (0xf400c820,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmov.16",
	  OPCODE_INFO2 (0xf400c880,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmovh",
	  OPCODE_INFO2 (0xf400c880,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fabs.16",
	  OPCODE_INFO2 (0xf400c8c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fabsh",
	  OPCODE_INFO2 (0xf400c8c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fneg.16",
	  OPCODE_INFO2 (0xf400c8e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnegh",
	  OPCODE_INFO2 (0xf400c8e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphsz.16",
	  OPCODE_INFO1 (0xf400c900,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzhsh",
	  OPCODE_INFO1 (0xf400c900,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpltz.16",
	  OPCODE_INFO1 (0xf400c920,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzlth",
	  OPCODE_INFO1 (0xf400c920,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpnez.16",
	  OPCODE_INFO1 (0xf400c940,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzneh",
	  OPCODE_INFO1 (0xf400c940,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpuoz.16",
	  OPCODE_INFO1 (0xf400c960,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzuoh",
	  OPCODE_INFO1 (0xf400c960,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphs.16",
	  OPCODE_INFO2 (0xf400c980,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphsh",
	  OPCODE_INFO2 (0xf400c980,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmplt.16",
	  OPCODE_INFO2 (0xf400c9a0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpne.16",
	  OPCODE_INFO2 (0xf400c9c0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpneh",
	  OPCODE_INFO2 (0xf400c9c0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpuo.16",
	  OPCODE_INFO2 (0xf400c9e0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpuoh",
	  OPCODE_INFO2 (0xf400c9e0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmaxnm.16",
	  OPCODE_INFO3 (0xf400cd00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fminnm.16",
	  OPCODE_INFO3 (0xf400cd20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphz.16",
	  OPCODE_INFO1 (0xf400cd40,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmplsz.16",
	  OPCODE_INFO1 (0xf400cd60,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmul.16",
	  OPCODE_INFO3 (0xf400ca00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmulh",
	  OPCODE_INFO3 (0xf400ca00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmul.16",
	  OPCODE_INFO3 (0xf400ca20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmulh",
	  OPCODE_INFO3 (0xf400ca20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmula.16",
	  OPCODE_INFO3 (0xf400ca80,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmach",
	  OPCODE_INFO3 (0xf400ca80,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmuls.16",
	  OPCODE_INFO3 (0xf400caa0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmsch",
	  OPCODE_INFO3 (0xf400caa0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmuls.16",
	  OPCODE_INFO3 (0xf400cac0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmach",
	  OPCODE_INFO3 (0xf400cac0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmula.16",
	  OPCODE_INFO3 (0xf400cae0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmsch",
	  OPCODE_INFO3 (0xf400cae0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffmula.16",
	  OPCODE_INFO3 (0xf400ce00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffmuls.16",
	  OPCODE_INFO3 (0xf400ce20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffnmula.16",
	  OPCODE_INFO3 (0xf400ce40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffnmuls.16",
	  OPCODE_INFO3 (0xf400ce60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdivh",
	  OPCODE_INFO3 (0xf400cb00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdiv.16",
	  OPCODE_INFO3 (0xf400cb00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("freciph",
	  OPCODE_INFO2 (0xf400cb20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("frecip.16",
	  OPCODE_INFO2 (0xf400cb20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsqrt.16",
	  OPCODE_INFO2 (0xf400cb40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsqrth",
	  OPCODE_INFO2 (0xf400cb40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsel.16",
	  OPCODE_INFO3 (0xf400cf20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
  /* Single floating.  */
    OP32 ("fadd.32",
	  OPCODE_INFO3 (0xf4000000,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fadds",
	  OPCODE_INFO3 (0xf4000000,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsub.32",
	  OPCODE_INFO3 (0xf4000020,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsubs",
	  OPCODE_INFO3 (0xf4000020,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmov.32",
	  OPCODE_INFO2 (0xf4000080,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmovs",
	  OPCODE_INFO2 (0xf4000080,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fabs.32",
	  OPCODE_INFO2 (0xf40000c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fabss",
	  OPCODE_INFO2 (0xf40000c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fneg.32",
	  OPCODE_INFO2 (0xf40000e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnegs",
	  OPCODE_INFO2 (0xf40000e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphsz.32",
	  OPCODE_INFO1 (0xf4000100,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzhss",
	  OPCODE_INFO1 (0xf4000100,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpltz.32",
	  OPCODE_INFO1 (0xf4000120,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzlts",
	  OPCODE_INFO1 (0xf4000120,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpnez.32",
	  OPCODE_INFO1 (0xf4000140,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpznes",
	  OPCODE_INFO1 (0xf4000140,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpuoz.32",
	  OPCODE_INFO1 (0xf4000160,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzuos",
	  OPCODE_INFO1 (0xf4000160,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphs.32",
	  OPCODE_INFO2 (0xf4000180,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphss",
	  OPCODE_INFO2 (0xf4000180,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmplt.32",
	  OPCODE_INFO2 (0xf40001a0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmplts",
	  OPCODE_INFO2 (0xf40001a0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpne.32",
	  OPCODE_INFO2 (0xf40001c0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpnes",
	  OPCODE_INFO2 (0xf40001c0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpuo.32",
	  OPCODE_INFO2 (0xf40001e0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpuos",
	  OPCODE_INFO2 (0xf40001e0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmaxnm.32",
	  OPCODE_INFO3 (0xf4000500,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fminnm.32",
	  OPCODE_INFO3 (0xf4000520,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphz.32",
	  OPCODE_INFO1 (0xf4000540,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmplsz.32",
	  OPCODE_INFO1 (0xf4000560,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmul.32",
	  OPCODE_INFO3 (0xf4000200,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmuls",
	  OPCODE_INFO3 (0xf4000200,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmul.32",
	  OPCODE_INFO3 (0xf4000220,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmuls",
	  OPCODE_INFO3 (0xf4000220,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmula.32",
	  OPCODE_INFO3 (0xf4000280,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmacs",
	  OPCODE_INFO3 (0xf4000280,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmuls.32",
	  OPCODE_INFO3 (0xf40002a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmscs",
	  OPCODE_INFO3 (0xf40002a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmuls.32",
	  OPCODE_INFO3 (0xf40002c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmacs",
	  OPCODE_INFO3 (0xf40002c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmula.32",
	  OPCODE_INFO3 (0xf40002e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmscs",
	  OPCODE_INFO3 (0xf40002e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffmula.32",
	  OPCODE_INFO3 (0xf4000600,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffmuls.32",
	  OPCODE_INFO3 (0xf4000620,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffnmula.32",
	  OPCODE_INFO3 (0xf4000640,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffnmuls.32",
	  OPCODE_INFO3 (0xf4000660,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdiv.32",
	  OPCODE_INFO3 (0xf4000300,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdivs",
	  OPCODE_INFO3 (0xf4000300,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("frecip.32",
	  OPCODE_INFO2 (0xf4000320,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("frecips",
	  OPCODE_INFO2 (0xf4000320,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsqrt.32",
	  OPCODE_INFO2 (0xf4000340,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsqrts",
	  OPCODE_INFO2 (0xf4000340,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsel.32",
	  OPCODE_INFO3 (0xf4000720,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
  /* Double floating.  */
    OP32 ("fadd.64",
	  OPCODE_INFO3 (0xf4000800,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("faddd",
	  OPCODE_INFO3 (0xf4000800,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsub.64",
	  OPCODE_INFO3 (0xf4000820,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsubd",
	  OPCODE_INFO3 (0xf4000820,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmov.64",
	  OPCODE_INFO2 (0xf4000880,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmovd",
	  OPCODE_INFO2 (0xf4000880,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmovx.32",
	  OPCODE_INFO2 (0xf40008a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fabs.64",
	  OPCODE_INFO2 (0xf40008c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fabsd",
	  OPCODE_INFO2 (0xf40008c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fneg.64",
	  OPCODE_INFO2 (0xf40008e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnegd",
	  OPCODE_INFO2 (0xf40008e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphsz.64",
	  OPCODE_INFO1 (0xf4000900,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzhsd",
	  OPCODE_INFO1 (0xf4000900,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpltz.64",
	  OPCODE_INFO1 (0xf4000920,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzltd",
	  OPCODE_INFO1 (0xf4000920,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpnez.64",
	  OPCODE_INFO1 (0xf4000940,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzned",
	  OPCODE_INFO1 (0xf4000940,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpuoz.64",
	  OPCODE_INFO1 (0xf4000960,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpzuod",
	  OPCODE_INFO1 (0xf4000960,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphs.64",
	  OPCODE_INFO2 (0xf4000980,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphsd",
	  OPCODE_INFO2 (0xf4000980,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmplt.64",
	  OPCODE_INFO2 (0xf40009a0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpltd",
	  OPCODE_INFO2 (0xf40009a0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpne.64",
	  OPCODE_INFO2 (0xf40009c0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpned",
	  OPCODE_INFO2 (0xf40009c0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpuo.64",
	  OPCODE_INFO2 (0xf40009e0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmpuod",
	  OPCODE_INFO2 (0xf40009e0,
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmaxnm.64",
	  OPCODE_INFO3 (0xf4000d00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fminnm.64",
	  OPCODE_INFO3 (0xf4000d20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmphz.64",
	  OPCODE_INFO1 (0xf4000d40,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fcmplsz.64",
	  OPCODE_INFO1 (0xf4000d60,
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmul.64",
	  OPCODE_INFO3 (0xf4000a00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmuld",
	  OPCODE_INFO3 (0xf4000a00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmul.64",
	  OPCODE_INFO3 (0xf4000a20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmuld",
	  OPCODE_INFO3 (0xf4000a20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmula.64",
	  OPCODE_INFO3 (0xf4000a80,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmacd",
	  OPCODE_INFO3 (0xf4000a80,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmuls.64",
	  OPCODE_INFO3 (0xf4000aa0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmscd",
	  OPCODE_INFO3 (0xf4000aa0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmuls.64",
	  OPCODE_INFO3 (0xf4000ac0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmacd",
	  OPCODE_INFO3 (0xf4000ac0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmula.64",
	  OPCODE_INFO3 (0xf4000ae0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmscd",
	  OPCODE_INFO3 (0xf4000ae0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffmula.64",
	  OPCODE_INFO3 (0xf4000e00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffmuls.64",
	  OPCODE_INFO3 (0xf4000e20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffnmula.64",
	  OPCODE_INFO3 (0xf4000e40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("ffnmuls.64",
	  OPCODE_INFO3 (0xf4000e60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdiv.64",
	  OPCODE_INFO3 (0xf4000b00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdivd",
	  OPCODE_INFO3 (0xf4000b00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("frecip.64",
	  OPCODE_INFO2 (0xf4000b20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("frecipd",
	  OPCODE_INFO2 (0xf4000b20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsqrt.64",
	  OPCODE_INFO2 (0xf4000b40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsqrtd",
	  OPCODE_INFO2 (0xf4000b40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fins.32",
	  OPCODE_INFO2 (0xf4000360,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsel.64",
	  OPCODE_INFO3 (0xf4000f20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
  /* SIMD floating.  */
    OP32 ("fadd.f32",
	  OPCODE_INFO3 (0xf4001000,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("faddm",
	  OPCODE_INFO3 (0xf4001000,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsub.f32",
	  OPCODE_INFO3 (0xf4001020,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsubm",
	  OPCODE_INFO3 (0xf4001020,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmov.f32",
	  OPCODE_INFO2 (0xf4001080,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmovm",
	  OPCODE_INFO2 (0xf4001080,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fabs.f32",
	  OPCODE_INFO2 (0xf40010c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fabsm",
	  OPCODE_INFO2 (0xf40010c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fneg.f32",
	  OPCODE_INFO2 (0xf40010e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnegm",
	  OPCODE_INFO2 (0xf40010e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmul.f32",
	  OPCODE_INFO3 (0xf4001200,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmulm",
	  OPCODE_INFO3 (0xf4001200,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmula.f32",
	  OPCODE_INFO3 (0xf4001280,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmuls.f32",
	  OPCODE_INFO3 (0xf40012c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fnmacm",
	  OPCODE_INFO3 (0xf40012c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG),
			OPRND_SHIFT0 (21_25, FREG)),
	  CSKY_ISA_FLOAT_7E60),
  /* floating formate.  */
    OP32 ("fftoi.f32.s32.rn",
	  OPCODE_INFO2 (0xf4001800,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstosi.rn",
	  OPCODE_INFO2 (0xf4001800,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f32.s32.rz",
	  OPCODE_INFO2 (0xf4001820,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstosi.rz",
	  OPCODE_INFO2 (0xf4001820,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f32.s32.rpi",
	  OPCODE_INFO2 (0xf4001840,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstosi.rpi",
	  OPCODE_INFO2 (0xf4001840,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f32.s32.rni",
	  OPCODE_INFO2 (0xf4001860,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstosi.rni",
	  OPCODE_INFO2 (0xf4001860,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f32.u32.rn",
	  OPCODE_INFO2 (0xf4001880,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstoui.rn",
	  OPCODE_INFO2 (0xf4001880,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f32.u32.rz",
	  OPCODE_INFO2 (0xf40018a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstoui.rz",
	  OPCODE_INFO2 (0xf40018a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f32.u32.rpi",
	  OPCODE_INFO2 (0xf40018c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstoui.rpi",
	  OPCODE_INFO2 (0xf40018c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f32.u32.rni",
	  OPCODE_INFO2 (0xf40018e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstoui.rni",
	  OPCODE_INFO2 (0xf40018e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.s32.rn",
	  OPCODE_INFO2 (0xf4001900,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtosi.rn",
	  OPCODE_INFO2 (0xf4001900,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.s32.rz",
	  OPCODE_INFO2 (0xf4001920,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtosi.rz",
	  OPCODE_INFO2 (0xf4001920,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.s32.rpi",
	  OPCODE_INFO2 (0xf4001940,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtosi.rpi",
	  OPCODE_INFO2 (0xf4001940,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.s32.rni",
	  OPCODE_INFO2 (0xf4001960,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtosi.rni",
	  OPCODE_INFO2 (0xf4001960,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.u32.rn",
	  OPCODE_INFO2 (0xf4001980,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtoui.rn",
	  OPCODE_INFO2 (0xf4001980,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.u32.rz",
	  OPCODE_INFO2 (0xf40019a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtoui.rz",
	  OPCODE_INFO2 (0xf40019a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.u32.rpi",
	  OPCODE_INFO2 (0xf40019c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtoui.rpi",
	  OPCODE_INFO2 (0xf40019c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.u32.rni",
	  OPCODE_INFO2 (0xf40019e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtoui.rni",
	  OPCODE_INFO2 (0xf40019e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.s32.rn",
	  OPCODE_INFO2 (0xf4001c00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtosi.rn",
	  OPCODE_INFO2 (0xf4001c00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.s32.rz",
	  OPCODE_INFO2 (0xf4001c20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtosi.rz",
	  OPCODE_INFO2 (0xf4001c20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.s32.rpi",
	  OPCODE_INFO2 (0xf4001c40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtosi.rpi",
	  OPCODE_INFO2 (0xf4001c40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.s32.rni",
	  OPCODE_INFO2 (0xf4001c60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtosi.rni",
	  OPCODE_INFO2 (0xf4001c60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.u32.rn",
	  OPCODE_INFO2 (0xf4001c80,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtoui.rn",
	  OPCODE_INFO2 (0xf4001c80,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.u32.rz",
	  OPCODE_INFO2 (0xf4001ca0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtoui.rz",
	  OPCODE_INFO2 (0xf4001ca0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.u32.rpi",
	  OPCODE_INFO2 (0xf4001cc0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtoui.rpi",
	  OPCODE_INFO2 (0xf4001cc0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.u32.rni",
	  OPCODE_INFO2 (0xf4001ce0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtoui.rni",
	  OPCODE_INFO2 (0xf4001ce0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtos",
	  OPCODE_INFO2 (0xf4001a40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fhtos.f16",
	  OPCODE_INFO2 (0xf4001a40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstoh",
	  OPCODE_INFO2 (0xf4001a60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstoh.f32",
	  OPCODE_INFO2 (0xf4001a60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtos",
	  OPCODE_INFO2 (0xf4001ac0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fdtos.f64",
	  OPCODE_INFO2 (0xf4001ac0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fstod",
	  OPCODE_INFO2 (0xf4001ae0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmfvrh",
	  OPCODE_INFO2 (0xf4001b00,
			OPRND_SHIFT0 (0_4, AREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmfvr.32.1",
	  OPCODE_INFO2 (0xf4001b20,
			OPRND_SHIFT0 (0_4, AREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmfvrl",
	  OPCODE_INFO2 (0xf4001b20,
			OPRND_SHIFT0 (0_4, AREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmtvr.16",
	  OPCODE_INFO2 (0xf4001fa0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, AREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmfvr.16",
	  OPCODE_INFO2 (0xf4001f20,
			OPRND_SHIFT0 (0_4, AREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmtvrh",
	  OPCODE_INFO2 (0xf4001b40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, AREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmtvr.32.1",
	  OPCODE_INFO2 (0xf4001b60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, AREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmtvrl",
	  OPCODE_INFO2 (0xf4001b60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, AREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmtvr.64",
	  OPCODE_INFO3 (0xf4001f80,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, AREG),
			OPRND_SHIFT0 (21_25, AREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmfvr.64",
	  OPCODE_INFO3 (0xf4001f00,
			OPRND_SHIFT0 (0_4, AREG),
			OPRND_SHIFT0 (21_25, AREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmtvr.32.2",
	  OPCODE_INFO3 (0xf4001fc0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, AREG),
			OPRND_SHIFT0 (21_25, AREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fmfvr.32.2",
	  OPCODE_INFO3 (0xf4001f40,
			OPRND_SHIFT0 (0_4, AREG),
			OPRND_SHIFT0 (21_25, AREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    /* flsu.  */
    OP32 ("fld.16",
	  SOPCODE_INFO2 (0xf4002300,
			 (0_3or25, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(4_7or21_24,
					 IMM_FLDST,
					 OPRND_SHIFT_1_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldh",
	  SOPCODE_INFO2 (0xf4002300,
			 (0_3or25, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(4_7or21_24,
					 IMM_FLDST,
					 OPRND_SHIFT_1_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fst.16",
		    SOPCODE_INFO2 (0xf4002700,
				   (0_3or25, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (4_7or21_24,
						   IMM_FLDST,
						   OPRND_SHIFT_1_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fsth",
		    SOPCODE_INFO2 (0xf4002700,
				   (0_3or25, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (4_7or21_24,
						   IMM_FLDST,
						   OPRND_SHIFT_1_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fldr16",
	  SOPCODE_INFO2 (0xf4002b00,
			 (0_4, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(5_6or21_25,
					 AREG_WITH_LSHIFT_FPU,
					 OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldrh",
	  SOPCODE_INFO2 (0xf4002b00,
			 (0_4, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(5_6or21_25,
					 AREG_WITH_LSHIFT_FPU,
					 OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstr.16",
		    SOPCODE_INFO2 (0xf4002f00,
				   (0_4, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (5_6or21_25,
						   AREG_WITH_LSHIFT_FPU,
						   OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60,
	  float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstrh",
		    SOPCODE_INFO2 (0xf4002f00,
				   (0_4, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (5_6or21_25,
						   AREG_WITH_LSHIFT_FPU,
						   OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60,
	  float_work_fpuv3_fstore),
    OP32 ("fldm.16",
	  OPCODE_INFO2 (0xf4003300,
			(0_4or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET,OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldmh",
	  OPCODE_INFO2 (0xf4003300,
			(0_4or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET,OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstm.16",
		    OPCODE_INFO2 (0xf4003700,
				  (0_4or21_24,
				   FREGLIST_DASH,
				   OPRND_SHIFT_0_BIT),
				  (16_20,
				   AREG_WITH_BRACKET,
				   OPRND_SHIFT_0_BIT)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstmh",
		    OPCODE_INFO2 (0xf4003700,
				  (0_4or21_24,
				   FREGLIST_DASH,
				   OPRND_SHIFT_0_BIT),
				  (16_20,
				   AREG_WITH_BRACKET,
				   OPRND_SHIFT_0_BIT)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fldmu.16",
	  OPCODE_INFO2 (0xf4003380,
			OPRND_SHIFT0 (0_4or21_24, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldmu.h",
	  OPCODE_INFO2 (0xf4003380,
			OPRND_SHIFT0 (0_4or21_24, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstmu.16",
		    OPCODE_INFO2 (0xf4003780,
				  OPRND_SHIFT0 (0_4or21_24, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstmu.h",
		    OPCODE_INFO2 (0xf4003780,
				  OPRND_SHIFT0 (0_4or21_24, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		   CSKY_ISA_FLOAT_7E60,
		   float_work_fpuv3_fstore),
    OP32 ("fld.32",
	  SOPCODE_INFO2 (0xf4002000,
			 (0_3or25, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(4_7or21_24,
					 IMM_FLDST,
					 OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("flds",
	  SOPCODE_INFO2 (0xf4002000,
			 (0_3or25, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(4_7or21_24,
					 IMM_FLDST,
					 OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fst.32",
		    SOPCODE_INFO2 (0xf4002400,
				   (0_3or25, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (4_7or21_24,
						   IMM_FLDST,
						   OPRND_SHIFT_2_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fsts",
		    SOPCODE_INFO2 (0xf4002400,
				   (0_3or25, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (4_7or21_24,
						   IMM_FLDST,
						   OPRND_SHIFT_2_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fldr.32",
	  SOPCODE_INFO2 (0xf4002800,
			 (0_4, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(5_6or21_25,
					 AREG_WITH_LSHIFT_FPU,
					 OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldrs",
	  SOPCODE_INFO2 (0xf4002800,
			 (0_4, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(5_6or21_25,
					 AREG_WITH_LSHIFT_FPU,
					 OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstr.32",
		    SOPCODE_INFO2 (0xf4002c00,
				   (0_4, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (5_6or21_25,
						   AREG_WITH_LSHIFT_FPU,
						   OPRND_SHIFT_0_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstrs",
		    SOPCODE_INFO2 (0xf4002c00,
				   (0_4, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (5_6or21_25,
						   AREG_WITH_LSHIFT_FPU,
						   OPRND_SHIFT_0_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fldm.32",
	  OPCODE_INFO2 (0xf4003000,
			OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldms",
	  OPCODE_INFO2 (0xf4003000,
			OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstm.32",
		    OPCODE_INFO2 (0xf4003400,
				  OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstms",
		    OPCODE_INFO2 (0xf4003400,
				  OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fldmu.32",
	  OPCODE_INFO2 (0xf4003080,
			OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldmu.s",
	  OPCODE_INFO2 (0xf4003080,
			OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstmu.32",
		    OPCODE_INFO2 (0xf4003480,
				  OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstmu.s",
		    OPCODE_INFO2 (0xf4003480,
				  OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fld.64",
	  SOPCODE_INFO2 (0xf4002100,
			 (0_3or25, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(4_7or21_24,
					 IMM_FLDST,
					 OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldd",
	  SOPCODE_INFO2 (0xf4002100,
			 (0_3or25, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(4_7or21_24,
					 IMM_FLDST,
					 OPRND_SHIFT_2_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fst.64",
		    SOPCODE_INFO2 (0xf4002500,
				   (0_3or25, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (4_7or21_24,
						   IMM_FLDST,
						   OPRND_SHIFT_2_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstd",
		    SOPCODE_INFO2 (0xf4002500,
				   (0_3or25, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (4_7or21_24,
						   IMM_FLDST,
						   OPRND_SHIFT_2_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fldr.64",
	  SOPCODE_INFO2 (0xf4002900,
			 (0_4, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(5_6or21_25,
					 AREG_WITH_LSHIFT_FPU,
					 OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldrd",
	  SOPCODE_INFO2 (0xf4002900,
			 (0_4, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(5_6or21_25,
					 AREG_WITH_LSHIFT_FPU,
					 OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstr.64",
		    SOPCODE_INFO2 (0xf4002d00,
				   (0_4, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (5_6or21_25,
						   AREG_WITH_LSHIFT_FPU,
						   OPRND_SHIFT_0_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstrd",
		    SOPCODE_INFO2 (0xf4002d00,
				   (0_4, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (5_6or21_25,
						   AREG_WITH_LSHIFT_FPU,
						   OPRND_SHIFT_0_BIT))),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fldm.64",
	  OPCODE_INFO2 (0xf4003100,
			OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldmd",
	  OPCODE_INFO2 (0xf4003100,
			OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstm.64",
		    OPCODE_INFO2 (0xf4003500,
				  OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstmd",
		    OPCODE_INFO2 (0xf4003500,
				  OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fldmu.64",
	  OPCODE_INFO2 (0xf4003180,
			OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fldmu.d",
	  OPCODE_INFO2 (0xf4003180,
			OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
			OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstmu.64",
		    OPCODE_INFO2 (0xf4003580,
				  OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32_WITH_WORK ("fstmu.d",
		    OPCODE_INFO2 (0xf4003580,
				  OPRND_SHIFT0 (0_4or21_25, FREGLIST_DASH),
				  OPRND_SHIFT0 (16_20, AREG_WITH_BRACKET)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fldrm",
	  SOPCODE_INFO2 (0xf4002a00,
			 (0_4, FREG, OPRND_SHIFT_0_BIT),
			 BRACKET_OPRND ((16_20,
					 AREG,
					 OPRND_SHIFT_0_BIT),
					(5_6or21_25,
					 AREG_WITH_LSHIFT_FPU,
					 OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstrm",
		    SOPCODE_INFO2 (0xf4002e00,
				   (0_4, FREG, OPRND_SHIFT_0_BIT),
				   BRACKET_OPRND ((16_20,
						   AREG,
						   OPRND_SHIFT_0_BIT),
						  (5_6or21_25,
						   AREG_WITH_LSHIFT_FPU,
						   OPRND_SHIFT_0_BIT))),
	  CSKY_ISA_FLOAT_7E60,
	  float_work_fpuv3_fstore),
    OP32 ("fldmm",
	  OPCODE_INFO2 (0xf4003200,
			(0_4or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
			(16_20, AREG_WITH_BRACKET,OPRND_SHIFT_0_BIT)),
	  CSKY_ISA_FLOAT_7E60),
    OP32_WITH_WORK ("fstmm",
		    OPCODE_INFO2 (0xf4003600,
				  (0_4or21_24, FREGLIST_DASH, OPRND_SHIFT_0_BIT),
				  (16_20, AREG_WITH_BRACKET,OPRND_SHIFT_0_BIT)),
		    CSKY_ISA_FLOAT_7E60,
		    float_work_fpuv3_fstore),
    OP32 ("fftox.f16.u16",
	  OPCODE_INFO2 (0xf4004000,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftox.f16.s16",
	  OPCODE_INFO2 (0xf4004020,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftox.f16.u32",
	  OPCODE_INFO2 (0xf4004100,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftox.f16.s32",
	  OPCODE_INFO2 (0xf4004120,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftox.f32.u32",
	  OPCODE_INFO2 (0xf4004140,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftox.f32.s32",
	  OPCODE_INFO2 (0xf4004160,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftox.f64.u32",
	  OPCODE_INFO2 (0xf4004180,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftox.f64.s32",
	  OPCODE_INFO2 (0xf40041a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fxtof.u16.f16",
	  OPCODE_INFO2 (0xf4004800,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fxtof.s16.f16",
	  OPCODE_INFO2 (0xf4004820,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fxtof.u32.f16",
	  OPCODE_INFO2 (0xf4004900,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fxtof.s32.f16",
	  OPCODE_INFO2 (0xf4004920,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fxtof.u32.f32",
	  OPCODE_INFO2 (0xf4004940,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fxtof.s32.f32",
	  OPCODE_INFO2 (0xf4004960,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fxtof.u32.f64",
	  OPCODE_INFO2 (0xf4004980,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fxtof.s32.f64",
	  OPCODE_INFO2 (0xf40049a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.s16",
	  OPCODE_INFO2 (0xf4004220,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.u16",
	  OPCODE_INFO2 (0xf4004200,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.s32",
	  OPCODE_INFO2 (0xf4004320,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f16.u32",
	  OPCODE_INFO2 (0xf4004300,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f32.s32",
	  OPCODE_INFO2 (0xf4004360,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f32.u32",
	  OPCODE_INFO2 (0xf4004340,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.s32",
	  OPCODE_INFO2 (0xf40043a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftoi.f64.u32",
	  OPCODE_INFO2 (0xf4004380,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fitof.s16.f16",
	  OPCODE_INFO2 (0xf4004a20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fitof.u16.f16",
	  OPCODE_INFO2 (0xf4004a00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fitof.s32.f16",
	  OPCODE_INFO2 (0xf4004b20,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fitof.u32.f16",
	  OPCODE_INFO2 (0xf4004b00,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fitof.s32.f32",
	  OPCODE_INFO2 (0xf4004b60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsitos",
	  OPCODE_INFO2 (0xf4004b60,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fitof.u32.f32",
	  OPCODE_INFO2 (0xf4004b40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fuitos",
	  OPCODE_INFO2 (0xf4004b40,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fitof.s32.f64",
	  OPCODE_INFO2 (0xf4004ba0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fsitod",
	  OPCODE_INFO2 (0xf4004ba0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fitof.u32.f64",
	  OPCODE_INFO2 (0xf4004b80,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fuitod",
	  OPCODE_INFO2 (0xf4004b80,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f16.rn",
	  OPCODE_INFO2 (0xf4004400,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f16.rz",
	  OPCODE_INFO2 (0xf4004420,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f16.rpi",
	  OPCODE_INFO2 (0xf4004440,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f16.rni",
	  OPCODE_INFO2 (0xf4004460,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f32.rn",
	  OPCODE_INFO2 (0xf4004480,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f32.rz",
	  OPCODE_INFO2 (0xf40044a0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f32.rpi",
	  OPCODE_INFO2 (0xf40044c0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f32.rni",
	  OPCODE_INFO2 (0xf40044e0,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f64.rn",
	  OPCODE_INFO2 (0xf4004500,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f64.rz",
	  OPCODE_INFO2 (0xf4004520,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f64.rpi",
	  OPCODE_INFO2 (0xf4004540,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("fftofi.f64.rni",
	  OPCODE_INFO2 (0xf4004560,
			OPRND_SHIFT0 (0_4, FREG),
			OPRND_SHIFT0 (16_20, FREG)),
	  CSKY_ISA_FLOAT_7E60),
    DOP32_WITH_WORK ("fmovi.16",
		     OPCODE_INFO2 (0xf400e400,
				   OPRND_SHIFT0 (0_4, FREG),
				   OPRND_SHIFT0 (5or8_9or16_25, HFLOAT_FMOVI)),
		     OPCODE_INFO3 (0xf400e400,
				   OPRND_SHIFT0 (0_4, FREG),
				   OPRND_SHIFT0 (5or8_9or20_25, IMM9b),
				   OPRND_SHIFT0 (16_19, IMM4b)),
		     CSKY_ISA_FLOAT_7E60,
		     float_work_fpuv3_fmovi),
    DOP32_WITH_WORK ("fmovi.32",
		     OPCODE_INFO2 (0xf400e440,
				   OPRND_SHIFT0 (0_4, FREG),
				   OPRND_SHIFT0 (5or8_9or16_25, SFLOAT_FMOVI)),
		     OPCODE_INFO3 (0xf400e440,
				   OPRND_SHIFT0 (0_4, FREG),
				   OPRND_SHIFT0 (5or8_9or20_25, IMM9b),
				   OPRND_SHIFT0 (16_19, IMM4b)),
		     CSKY_ISA_FLOAT_7E60,
		     float_work_fpuv3_fmovi),
    DOP32_WITH_WORK ("fmovi.64",
		     OPCODE_INFO2 (0xf400e480,
				   OPRND_SHIFT0 (0_4, FREG),
				   OPRND_SHIFT0 (5or8_9or16_25, DFLOAT_FMOVI)),
		     OPCODE_INFO3 (0xf400e480,
				   OPRND_SHIFT0 (0_4, FREG),
				   OPRND_SHIFT0 (5or8_9or20_25, IMM9b),
				   OPRND_SHIFT0 (16_19, IMM4b)),
		     CSKY_ISA_FLOAT_7E60,
		     float_work_fpuv3_fmovi),
#undef _RELOC32
#define _RELOC32 BFD_RELOC_CKCORE_PCREL_FLRW_IMM8BY4
    OP32 ("flrw.32",
	  OPCODE_INFO2 (0xf4003800,
			(0_3or25, FREG, OPRND_SHIFT_0_BIT),
			(4_7or21_24, FCONSTANT, OPRND_SHIFT_2_BIT)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("flrws",
	  OPCODE_INFO2 (0xf4003800,
			(0_3or25, FREG, OPRND_SHIFT_0_BIT),
			(4_7or21_24, FCONSTANT, OPRND_SHIFT_2_BIT)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("flrw.64",
	  OPCODE_INFO2 (0xf4003900,
			(0_3or25, FREG, OPRND_SHIFT_0_BIT),
			(4_7or21_24, FCONSTANT, OPRND_SHIFT_2_BIT)),
	  CSKY_ISA_FLOAT_7E60),
    OP32 ("flrwd",
	  OPCODE_INFO2 (0xf4003900,
			(0_3or25, FREG, OPRND_SHIFT_0_BIT),
			(4_7or21_24, FCONSTANT, OPRND_SHIFT_2_BIT)),
	  CSKY_ISA_FLOAT_7E60),
#undef _RELOC32
#define _RELOC32 0

    /* The following are aliases for other instructions.  */
    /* setc -> cmphs r0, r0  */
    OP16 ("setc",
	  OPCODE_INFO0 (0x6400),
	  CSKYV2_ISA_E1),
    /* clrc -> cmpne r0, r0  */
    OP16 ("clrc",
	  OPCODE_INFO0 (0x6402),
	  CSKYV2_ISA_E1),
    /* tstlt rd -> btsti rd,31  */
    OP32 ("tstlt",
	  OPCODE_INFO1 (0xc7e02880,
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    /* idly4 -> idly 4  */
    OP32 ("idly4",
	  OPCODE_INFO0 (0xc0601c20),
	  CSKYV2_ISA_E1),
    /* rsub rz, ry, rx -> subu rz, rx, ry  */
    DOP32 ("rsub",
	   OPCODE_INFO3 (0xc4000080,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xc4000080,
			 (0_4or21_25, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT)), CSKYV2_ISA_1E2),
    /* cmplei rd,X -> cmplti rd,X+1  */
    OP16_OP32 ("cmplei",
	       OPCODE_INFO2 (0x3820,
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT),
			     (0_4, IMM5b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xeb200000,
			     (16_20, AREG, OPRND_SHIFT_0_BIT),
			     (0_15, IMM16b, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
    /* cmpls -> cmphs  */
    OP16_OP32 ("cmpls",
	       OPCODE_INFO2 (0x6400,
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xc4000420,
			     (21_25, AREG, OPRND_SHIFT_0_BIT),
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    /* cmpgt -> cmplt  */
    OP16_OP32 ("cmpgt",
	       OPCODE_INFO2 (0x6401,
			     (6_9, GREG0_15, OPRND_SHIFT_0_BIT),
			     (2_5, GREG0_15, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO2 (0xc4000440,
			     (21_25, AREG, OPRND_SHIFT_0_BIT),
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_2E3),
    /* tstle rd -> cmplti rd,1  */
    OP16_OP32 ("tstle",
	       OPCODE_INFO1 (0x3820,
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO1 (0xeb200000,
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
    /* tstne rd -> cmpnei rd,0  */
    OP16_OP32 ("tstne",
	       OPCODE_INFO1 (0x3840,
			     (8_10, GREG0_7, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_E1,
	       OPCODE_INFO1 (0xeb400000,
			     (16_20, AREG, OPRND_SHIFT_0_BIT)),
	       CSKYV2_ISA_1E2),
    /* rotri rz, rx, imm5 -> rotli rz, rx, 32-imm5  */
    DOP32 ("rotri",
	   OPCODE_INFO3 (0xc4004900,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, IMM5b_RORI, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xc4004900,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (21_25, IMM5b_RORI, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_2E3),
    DOP32 ("rori",
	   OPCODE_INFO3 (0xc4004900,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT),
			 (21_25, IMM5b_RORI, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO2 (0xc4004900,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT),
			 (21_25, IMM5b_RORI, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_2E3),

    /* rotlc rd -> addc rd, rd/ addc rd, rd, rd  */
    OP16_OP32_WITH_WORK ("rotlc",
			 OPCODE_INFO2 (0x6001,
				       (NONE, GREG0_15, OPRND_SHIFT_0_BIT),
				       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
			 CSKYV2_ISA_E1,
			 OPCODE_INFO2 (0xc4000040,
				       (NONE, AREG, OPRND_SHIFT_0_BIT),
				       (NONE, CONST1, OPRND_SHIFT_0_BIT)),
			 CSKYV2_ISA_2E3,
			 v2_work_rotlc),
    /* not rd -> nor rd, rd, not rz, rx -> nor rz, rx, rx  */
    OP16_OP32_WITH_WORK ("not",
			 OPCODE_INFO1 (0x6c02,
				       (NONE, AREG, OPRND_SHIFT_0_BIT)),
			 CSKYV2_ISA_E1,
			 OPCODE_INFO2 (0xc4002480,
				       (NONE, AREG, OPRND_SHIFT_0_BIT),
				       (NONE, AREG, OPRND_SHIFT_0_BIT)),
			 CSKYV2_ISA_E1, v2_work_not),

    /* Special force 32 bits instruction.  */
    OP32 ("xtrb0.32",
	  OPCODE_INFO2 (0xc4007020,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("xtrb1.32",
	  OPCODE_INFO2 (0xc4007040,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("xtrb2.32",
	  OPCODE_INFO2 (0xc4007080,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("xtrb3.32",
	  OPCODE_INFO2 (0xc4007100,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    OP32 ("ff0.32",
	  OPCODE_INFO2 (0xc4007c20,
			(0_4, AREG, OPRND_SHIFT_0_BIT),
			(16_20, AREG, OPRND_SHIFT_0_BIT)),
	  CSKYV2_ISA_1E2),
    DOP32 ("ff1.32",
	   OPCODE_INFO2 (0xc4007c40,
			 (0_4, AREG, OPRND_SHIFT_0_BIT),
			 (16_20, AREG, OPRND_SHIFT_0_BIT)),
	   OPCODE_INFO1 (0xc4007c40,
			 (0_4or16_20, DUP_AREG, OPRND_SHIFT_0_BIT)),
	   CSKYV2_ISA_1E2),

    {NULL, 0, {}, {}, 0, 0, 0, 0, 0, NULL}
  };
