/*
 * Z-star vc0321 library
 *
 * Copyright (C) 2009-2010 Jean-François Moine <http://moinejf.free.fr>
 * Copyright (C) 2006 Koninski Artur takeshi87@o2.pl
 * Copyright (C) 2006 Michel Xhaard
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#define MODULE_NAME "vc032x"

#include "gspca.h"

MODULE_AUTHOR("Jean-François Moine <http://moinejf.free.fr>");
MODULE_DESCRIPTION("GSPCA/VC032X USB Camera Driver");
MODULE_LICENSE("GPL");

/* specific webcam descriptor */
struct sd {
	struct gspca_dev gspca_dev;	/* !! must be the first item */

	u8 brightness;
	u8 contrast;
	u8 colors;
	u8 hflip;
	u8 vflip;
	u8 lightfreq;
	s8 sharpness;
	u16 exposure;
	u8 gain;
	u8 autogain;
	u8 backlight;

	u8 image_offset;

	u8 bridge;
	u8 sensor;
	u8 flags;
#define FL_SAMSUNG 0x01		/* SamsungQ1 (2 sensors) */
#define FL_HFLIP 0x02		/* mirrored by default */
#define FL_VFLIP 0x04		/* vertical flipped by default */
};
enum bridges {
	BRIDGE_VC0321,
	BRIDGE_VC0323,
};
enum sensors {
	SENSOR_HV7131R,
	SENSOR_MI0360,
	SENSOR_MI1310_SOC,
	SENSOR_MI1320,
	SENSOR_MI1320_SOC,
	SENSOR_OV7660,
	SENSOR_OV7670,
	SENSOR_PO1200,
	SENSOR_PO3130NC,
	SENSOR_POxxxx,
	NSENSORS
};

/* V4L2 controls supported by the driver */
static int sd_setbrightness(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getbrightness(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setcontrast(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getcontrast(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setcolors(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getcolors(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_sethflip(struct gspca_dev *gspca_dev, __s32 val);
static int sd_gethflip(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setvflip(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getvflip(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setfreq(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getfreq(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setsharpness(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getsharpness(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setgain(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getgain(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setexposure(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getexposure(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setautogain(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getautogain(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setbacklight(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getbacklight(struct gspca_dev *gspca_dev, __s32 *val);

static const struct ctrl sd_ctrls[] = {
#define BRIGHTNESS_IDX 0
	{
	    {
		.id      = V4L2_CID_BRIGHTNESS,
		.type    = V4L2_CTRL_TYPE_INTEGER,
		.name    = "Brightness",
		.minimum = 0,
		.maximum = 255,
		.step    = 1,
#define BRIGHTNESS_DEF 128
		.default_value = BRIGHTNESS_DEF,
	    },
	    .set = sd_setbrightness,
	    .get = sd_getbrightness,
	},
#define CONTRAST_IDX 1
	{
	    {
		.id      = V4L2_CID_CONTRAST,
		.type    = V4L2_CTRL_TYPE_INTEGER,
		.name    = "Contrast",
		.minimum = 0,
		.maximum = 255,
		.step    = 1,
#define CONTRAST_DEF 127
		.default_value = CONTRAST_DEF,
	    },
	    .set = sd_setcontrast,
	    .get = sd_getcontrast,
	},
#define COLORS_IDX 2
	{
	    {
		.id      = V4L2_CID_SATURATION,
		.type    = V4L2_CTRL_TYPE_INTEGER,
		.name    = "Saturation",
		.minimum = 1,
		.maximum = 127,
		.step    = 1,
#define COLOR_DEF 63
		.default_value = COLOR_DEF,
	    },
	    .set = sd_setcolors,
	    .get = sd_getcolors,
	},
/* next 2 controls work with some sensors only */
#define HFLIP_IDX 3
	{
	    {
		.id      = V4L2_CID_HFLIP,
		.type    = V4L2_CTRL_TYPE_BOOLEAN,
		.name    = "Mirror",
		.minimum = 0,
		.maximum = 1,
		.step    = 1,
#define HFLIP_DEF 0
		.default_value = HFLIP_DEF,
	    },
	    .set = sd_sethflip,
	    .get = sd_gethflip,
	},
#define VFLIP_IDX 4
	{
	    {
		.id      = V4L2_CID_VFLIP,
		.type    = V4L2_CTRL_TYPE_BOOLEAN,
		.name    = "Vflip",
		.minimum = 0,
		.maximum = 1,
		.step    = 1,
#define VFLIP_DEF 0
		.default_value = VFLIP_DEF,
	    },
	    .set = sd_setvflip,
	    .get = sd_getvflip,
	},
#define LIGHTFREQ_IDX 5
	{
	    {
		.id	 = V4L2_CID_POWER_LINE_FREQUENCY,
		.type    = V4L2_CTRL_TYPE_MENU,
		.name    = "Light frequency filter",
		.minimum = 0,
		.maximum = 2,	/* 0: No, 1: 50Hz, 2:60Hz */
		.step    = 1,
#define FREQ_DEF 1
		.default_value = FREQ_DEF,
	    },
	    .set = sd_setfreq,
	    .get = sd_getfreq,
	},
#define SHARPNESS_IDX 6
	{
	 {
	  .id = V4L2_CID_SHARPNESS,
	  .type = V4L2_CTRL_TYPE_INTEGER,
	  .name = "Sharpness",
	  .minimum = -1,
	  .maximum = 2,
	  .step = 1,
#define SHARPNESS_DEF -1
	  .default_value = SHARPNESS_DEF,
	  },
	 .set = sd_setsharpness,
	 .get = sd_getsharpness,
	 },
#define GAIN_IDX 7
	{
	    {
		.id      = V4L2_CID_GAIN,
		.type    = V4L2_CTRL_TYPE_INTEGER,
		.name    = "Gain",
		.minimum = 0,
		.maximum = 78,
		.step    = 1,
#define GAIN_DEF 0
		.default_value = GAIN_DEF,
	    },
	    .set = sd_setgain,
	    .get = sd_getgain,
	},
#define EXPOSURE_IDX 8
	{
		{
			.id = V4L2_CID_EXPOSURE,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Exposure",
#define EXPOSURE_DEF 450
			.minimum = 0,
			.maximum = 4095,
			.step = 1,
			.default_value = EXPOSURE_DEF,
		},
		.set = sd_setexposure,
		.get = sd_getexposure,
	},
#define AUTOGAIN_IDX 9
	{
		{
			.id = V4L2_CID_AUTOGAIN,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Automatic Gain and Exposure",
			.minimum = 0,
			.maximum = 1,
			.step = 1,
#define AUTOGAIN_DEF 1
			.default_value = AUTOGAIN_DEF,
		},
		.set = sd_setautogain,
		.get = sd_getautogain,
	},
#define BACKLIGHT_IDX 10
	{
		{
			.id = V4L2_CID_BACKLIGHT_COMPENSATION,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Backlight Compensation",
			.minimum = 0,
			.maximum = 15,
			.step = 1,
#define BACKLIGHT_DEF 15
			.default_value = BACKLIGHT_DEF,
		},
		.set = sd_setbacklight,
		.get = sd_getbacklight,
	},
};

/* table of the disabled controls */
static u32 ctrl_dis[NSENSORS] = {
    [SENSOR_HV7131R] =
	(1 << BRIGHTNESS_IDX) | (1 << CONTRAST_IDX) | (1 << COLORS_IDX)
		| (1 << HFLIP_IDX) | (1 << VFLIP_IDX) | (1 << LIGHTFREQ_IDX)
		| (1 << SHARPNESS_IDX)
		| (1 << GAIN_IDX) | (1 << EXPOSURE_IDX)
		| (1 << AUTOGAIN_IDX) | (1 << BACKLIGHT_IDX),
    [SENSOR_MI0360] =
	(1 << BRIGHTNESS_IDX) | (1 << CONTRAST_IDX) | (1 << COLORS_IDX)
		| (1 << HFLIP_IDX) | (1 << VFLIP_IDX) | (1 << LIGHTFREQ_IDX)
		| (1 << SHARPNESS_IDX)
		| (1 << GAIN_IDX) | (1 << EXPOSURE_IDX)
		| (1 << AUTOGAIN_IDX) | (1 << BACKLIGHT_IDX),
    [SENSOR_MI1310_SOC] =
	(1 << BRIGHTNESS_IDX) | (1 << CONTRAST_IDX) | (1 << COLORS_IDX)
		| (1 << LIGHTFREQ_IDX) | (1 << SHARPNESS_IDX)
		| (1 << GAIN_IDX) | (1 << EXPOSURE_IDX)
		| (1 << AUTOGAIN_IDX) | (1 << BACKLIGHT_IDX),
    [SENSOR_MI1320] =
	(1 << BRIGHTNESS_IDX) | (1 << CONTRAST_IDX) | (1 << COLORS_IDX)
		| (1 << LIGHTFREQ_IDX) | (1 << SHARPNESS_IDX)
		| (1 << GAIN_IDX) | (1 << EXPOSURE_IDX)
		| (1 << AUTOGAIN_IDX) | (1 << BACKLIGHT_IDX),
    [SENSOR_MI1320_SOC] =
	(1 << BRIGHTNESS_IDX) | (1 << CONTRAST_IDX) | (1 << COLORS_IDX)
		| (1 << LIGHTFREQ_IDX) | (1 << SHARPNESS_IDX)
		| (1 << GAIN_IDX) | (1 << EXPOSURE_IDX)
		| (1 << AUTOGAIN_IDX) | (1 << BACKLIGHT_IDX),
    [SENSOR_OV7660] =
	(1 << BRIGHTNESS_IDX) | (1 << CONTRAST_IDX) | (1 << COLORS_IDX)
		| (1 << LIGHTFREQ_IDX) | (1 << SHARPNESS_IDX)
		| (1 << GAIN_IDX) | (1 << EXPOSURE_IDX)
		| (1 << AUTOGAIN_IDX) | (1 << BACKLIGHT_IDX),
    [SENSOR_OV7670] =
	(1 << BRIGHTNESS_IDX) | (1 << CONTRAST_IDX) | (1 << COLORS_IDX)
		| (1 << SHARPNESS_IDX)
		| (1 << GAIN_IDX) | (1 << EXPOSURE_IDX)
		| (1 << AUTOGAIN_IDX) | (1 << BACKLIGHT_IDX),
    [SENSOR_PO1200] =
	(1 << BRIGHTNESS_IDX) | (1 << CONTRAST_IDX) | (1 << COLORS_IDX)
		| (1 << LIGHTFREQ_IDX)
		| (1 << GAIN_IDX) | (1 << EXPOSURE_IDX)
		| (1 << AUTOGAIN_IDX) | (1 << BACKLIGHT_IDX),
    [SENSOR_PO3130NC] =
	(1 << BRIGHTNESS_IDX) | (1 << CONTRAST_IDX) | (1 << COLORS_IDX)
		| (1 << HFLIP_IDX) | (1 << VFLIP_IDX) | (1 << LIGHTFREQ_IDX)
		| (1 << SHARPNESS_IDX)
		| (1 << GAIN_IDX) | (1 << EXPOSURE_IDX)
		| (1 << AUTOGAIN_IDX) | (1 << BACKLIGHT_IDX),
    [SENSOR_POxxxx] =
	(1 << HFLIP_IDX) | (1 << VFLIP_IDX) | (1 << LIGHTFREQ_IDX),
};

static const struct v4l2_pix_format vc0321_mode[] = {
	{320, 240, V4L2_PIX_FMT_YVYU, V4L2_FIELD_NONE,
		.bytesperline = 320,
		.sizeimage = 320 * 240 * 2,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 1},
	{640, 480, V4L2_PIX_FMT_YVYU, V4L2_FIELD_NONE,
		.bytesperline = 640,
		.sizeimage = 640 * 480 * 2,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 0},
};
static const struct v4l2_pix_format vc0323_mode[] = {
	{320, 240, V4L2_PIX_FMT_JPEG, V4L2_FIELD_NONE,
		.bytesperline = 320,
		.sizeimage = 320 * 240 * 3 / 8 + 590,
		.colorspace = V4L2_COLORSPACE_JPEG,
		.priv = 1},
	{640, 480, V4L2_PIX_FMT_JPEG, V4L2_FIELD_NONE,
		.bytesperline = 640,
		.sizeimage = 640 * 480 * 3 / 8 + 590,
		.colorspace = V4L2_COLORSPACE_JPEG,
		.priv = 0},
	{1280, 960, V4L2_PIX_FMT_JPEG, V4L2_FIELD_NONE, /* mi1310_soc only */
		.bytesperline = 1280,
		.sizeimage = 1280 * 960 * 3 / 8 + 590,
		.colorspace = V4L2_COLORSPACE_JPEG,
		.priv = 2},
};
static const struct v4l2_pix_format bi_mode[] = {
	{320, 240, V4L2_PIX_FMT_YUYV, V4L2_FIELD_NONE,
		.bytesperline = 320,
		.sizeimage = 320 * 240 * 2,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 2},
	{640, 480, V4L2_PIX_FMT_YUYV, V4L2_FIELD_NONE,
		.bytesperline = 640,
		.sizeimage = 640 * 480 * 2,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 1},
	{1280, 1024, V4L2_PIX_FMT_YUYV, V4L2_FIELD_NONE,
		.bytesperline = 1280,
		.sizeimage = 1280 * 1024 * 2,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 0},
};
static const struct v4l2_pix_format svga_mode[] = {
	{800, 600, V4L2_PIX_FMT_JPEG, V4L2_FIELD_NONE,
		.bytesperline = 800,
		.sizeimage = 800 * 600 * 1 / 4 + 590,
		.colorspace = V4L2_COLORSPACE_JPEG,
		.priv = 0},
};

/* OV7660/7670 registers */
#define OV7660_REG_MVFP 0x1e
#define OV7660_MVFP_MIRROR	0x20
#define OV7660_MVFP_VFLIP	0x10

static const u8 mi0360_matrix[9] = {
	0x50, 0xf8, 0xf8, 0xf5, 0x50, 0xfb, 0xff, 0xf1, 0x50
};

static const u8 mi0360_initVGA_JPG[][4] = {
	{0xb0, 0x03, 0x19, 0xcc},
	{0xb0, 0x04, 0x02, 0xcc},
	{0xb3, 0x00, 0x24, 0xcc},
	{0xb3, 0x00, 0x25, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x03, 0xcc},
	{0xb3, 0x03, 0x0a, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},
	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x35, 0xdd, 0xcc},	/* i2c add: 5d */
	{0xb3, 0x34, 0x02, 0xcc},
	{0xb3, 0x00, 0x25, 0xcc},
	{0xbc, 0x00, 0x71, 0xcc},
	{0xb8, 0x00, 0x13, 0xcc},
	{0xb8, 0x27, 0x20, 0xcc},
	{0xb8, 0x2c, 0x50, 0xcc},
	{0xb8, 0x2d, 0xf8, 0xcc},
	{0xb8, 0x2e, 0xf8, 0xcc},
	{0xb8, 0x2f, 0xf8, 0xcc},
	{0xb8, 0x30, 0x50, 0xcc},
	{0xb8, 0x31, 0xf8, 0xcc},
	{0xb8, 0x32, 0xf8, 0xcc},
	{0xb8, 0x33, 0xf8, 0xcc},
	{0xb8, 0x34, 0x50, 0xcc},
	{0xb8, 0x35, 0x00, 0xcc},
	{0xb8, 0x36, 0x00, 0xcc},
	{0xb8, 0x37, 0x00, 0xcc},
	{0xb8, 0x01, 0x79, 0xcc},
	{0xb8, 0x08, 0xe0, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0xb8, 0x01, 0x79, 0xcc},
	{0xb8, 0x14, 0x18, 0xcc},
	{0xb8, 0xb2, 0x0a, 0xcc},
	{0xb8, 0xb4, 0x0a, 0xcc},
	{0xb8, 0xb5, 0x0a, 0xcc},
	{0xb8, 0xfe, 0x00, 0xcc},
	{0xb8, 0xff, 0x28, 0xcc},
	{0xb9, 0x00, 0x28, 0xcc},
	{0xb9, 0x01, 0x28, 0xcc},
	{0xb9, 0x02, 0x28, 0xcc},
	{0xb9, 0x03, 0x00, 0xcc},
	{0xb9, 0x04, 0x00, 0xcc},
	{0xb9, 0x05, 0x3c, 0xcc},
	{0xb9, 0x06, 0x3c, 0xcc},
	{0xb9, 0x07, 0x3c, 0xcc},
	{0xb9, 0x08, 0x3c, 0xcc},
	{0xb8, 0x8e, 0x00, 0xcc},
	{0xb8, 0x8f, 0xff, 0xcc},
	{0xb8, 0x81, 0x09, 0xcc},
	{0x31, 0x00, 0x00, 0xbb},
	{0x09, 0x01, 0xc7, 0xbb},
	{0x34, 0x01, 0x00, 0xbb},
	{0x2b, 0x00, 0x28, 0xbb},
	{0x2c, 0x00, 0x30, 0xbb},
	{0x2d, 0x00, 0x30, 0xbb},
	{0x2e, 0x00, 0x28, 0xbb},
	{0x62, 0x04, 0x11, 0xbb},
	{0x03, 0x01, 0xe0, 0xbb},
	{0x2c, 0x00, 0x2c, 0xbb},
	{0x20, 0xd0, 0x00, 0xbb},
	{0x01, 0x00, 0x08, 0xbb},
	{0x06, 0x00, 0x10, 0xbb},
	{0x05, 0x00, 0x20, 0xbb},
	{0x20, 0x00, 0x00, 0xbb},
	{0xb6, 0x00, 0x00, 0xcc},
	{0xb6, 0x03, 0x02, 0xcc},
	{0xb6, 0x02, 0x80, 0xcc},
	{0xb6, 0x05, 0x01, 0xcc},
	{0xb6, 0x04, 0xe0, 0xcc},
	{0xb6, 0x12, 0x78, 0xcc},
	{0xb6, 0x18, 0x02, 0xcc},
	{0xb6, 0x17, 0x58, 0xcc},
	{0xb6, 0x16, 0x00, 0xcc},
	{0xb6, 0x22, 0x12, 0xcc},
	{0xb6, 0x23, 0x0b, 0xcc},
	{0xb3, 0x02, 0x02, 0xcc},
	{0xbf, 0xc0, 0x39, 0xcc},
	{0xbf, 0xc1, 0x04, 0xcc},
	{0xbf, 0xcc, 0x10, 0xcc},
	{0xb9, 0x12, 0x00, 0xcc},
	{0xb9, 0x13, 0x0a, 0xcc},
	{0xb9, 0x14, 0x0a, 0xcc},
	{0xb9, 0x15, 0x0a, 0xcc},
	{0xb9, 0x16, 0x0a, 0xcc},
	{0xb9, 0x18, 0x00, 0xcc},
	{0xb9, 0x19, 0x0f, 0xcc},
	{0xb9, 0x1a, 0x0f, 0xcc},
	{0xb9, 0x1b, 0x0f, 0xcc},
	{0xb9, 0x1c, 0x0f, 0xcc},
	{0xb8, 0x8e, 0x00, 0xcc},
	{0xb8, 0x8f, 0xff, 0xcc},
	{0xb6, 0x12, 0xf8, 0xcc},
	{0xb8, 0x0c, 0x20, 0xcc},
	{0xb8, 0x0d, 0x70, 0xcc},
	{0xb6, 0x13, 0x13, 0xcc},
	{0x35, 0x00, 0x60, 0xbb},
	{0xb3, 0x5c, 0x01, 0xcc},
	{}
};
static const u8 mi0360_initQVGA_JPG[][4] = {
	{0xb0, 0x03, 0x19, 0xcc},
	{0xb0, 0x04, 0x02, 0xcc},
	{0xb3, 0x00, 0x24, 0xcc},
	{0xb3, 0x00, 0x25, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x03, 0xcc},
	{0xb3, 0x03, 0x0a, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},
	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x35, 0xdd, 0xcc},
	{0xb3, 0x34, 0x02, 0xcc},
	{0xb3, 0x00, 0x25, 0xcc},
	{0xbc, 0x00, 0xd1, 0xcc},
	{0xb8, 0x00, 0x13, 0xcc},
	{0xb8, 0x27, 0x20, 0xcc},
	{0xb8, 0x2c, 0x50, 0xcc},
	{0xb8, 0x2d, 0xf8, 0xcc},
	{0xb8, 0x2e, 0xf8, 0xcc},
	{0xb8, 0x2f, 0xf8, 0xcc},
	{0xb8, 0x30, 0x50, 0xcc},
	{0xb8, 0x31, 0xf8, 0xcc},
	{0xb8, 0x32, 0xf8, 0xcc},
	{0xb8, 0x33, 0xf8, 0xcc},
	{0xb8, 0x34, 0x50, 0xcc},
	{0xb8, 0x35, 0x00, 0xcc},
	{0xb8, 0x36, 0x00, 0xcc},
	{0xb8, 0x37, 0x00, 0xcc},
	{0xb8, 0x01, 0x79, 0xcc},
	{0xb8, 0x08, 0xe0, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0xb8, 0x01, 0x79, 0xcc},
	{0xb8, 0x14, 0x18, 0xcc},
	{0xb8, 0xb2, 0x0a, 0xcc},
	{0xb8, 0xb4, 0x0a, 0xcc},
	{0xb8, 0xb5, 0x0a, 0xcc},
	{0xb8, 0xfe, 0x00, 0xcc},
	{0xb8, 0xff, 0x28, 0xcc},
	{0xb9, 0x00, 0x28, 0xcc},
	{0xb9, 0x01, 0x28, 0xcc},
	{0xb9, 0x02, 0x28, 0xcc},
	{0xb9, 0x03, 0x00, 0xcc},
	{0xb9, 0x04, 0x00, 0xcc},
	{0xb9, 0x05, 0x3c, 0xcc},
	{0xb9, 0x06, 0x3c, 0xcc},
	{0xb9, 0x07, 0x3c, 0xcc},
	{0xb9, 0x08, 0x3c, 0xcc},
	{0xb8, 0x8e, 0x00, 0xcc},
	{0xb8, 0x8f, 0xff, 0xcc},
	{0xb8, 0x81, 0x09, 0xcc},
	{0x31, 0x00, 0x00, 0xbb},
	{0x09, 0x01, 0xc7, 0xbb},
	{0x34, 0x01, 0x00, 0xbb},
	{0x2b, 0x00, 0x28, 0xbb},
	{0x2c, 0x00, 0x30, 0xbb},
	{0x2d, 0x00, 0x30, 0xbb},
	{0x2e, 0x00, 0x28, 0xbb},
	{0x62, 0x04, 0x11, 0xbb},
	{0x03, 0x01, 0xe0, 0xbb},
	{0x2c, 0x00, 0x2c, 0xbb},
	{0x20, 0xd0, 0x00, 0xbb},
	{0x01, 0x00, 0x08, 0xbb},
	{0x06, 0x00, 0x10, 0xbb},
	{0x05, 0x00, 0x20, 0xbb},
	{0x20, 0x00, 0x00, 0xbb},
	{0xb6, 0x00, 0x00, 0xcc},
	{0xb6, 0x03, 0x01, 0xcc},
	{0xb6, 0x02, 0x40, 0xcc},
	{0xb6, 0x05, 0x00, 0xcc},
	{0xb6, 0x04, 0xf0, 0xcc},
	{0xb6, 0x12, 0x78, 0xcc},
	{0xb6, 0x18, 0x00, 0xcc},
	{0xb6, 0x17, 0x96, 0xcc},
	{0xb6, 0x16, 0x00, 0xcc},
	{0xb6, 0x22, 0x12, 0xcc},
	{0xb6, 0x23, 0x0b, 0xcc},
	{0xb3, 0x02, 0x02, 0xcc},
	{0xbf, 0xc0, 0x39, 0xcc},
	{0xbf, 0xc1, 0x04, 0xcc},
	{0xbf, 0xcc, 0x10, 0xcc},
	{0xb9, 0x12, 0x00, 0xcc},
	{0xb9, 0x13, 0x0a, 0xcc},
	{0xb9, 0x14, 0x0a, 0xcc},
	{0xb9, 0x15, 0x0a, 0xcc},
	{0xb9, 0x16, 0x0a, 0xcc},
	{0xb9, 0x18, 0x00, 0xcc},
	{0xb9, 0x19, 0x0f, 0xcc},
	{0xb9, 0x1a, 0x0f, 0xcc},
	{0xb9, 0x1b, 0x0f, 0xcc},
	{0xb9, 0x1c, 0x0f, 0xcc},
	{0xb8, 0x8e, 0x00, 0xcc},
	{0xb8, 0x8f, 0xff, 0xcc},
	{0xb6, 0x12, 0xf8, 0xcc},
	{0xb6, 0x13, 0x13, 0xcc},
	{0xbc, 0x02, 0x18, 0xcc},
	{0xbc, 0x03, 0x50, 0xcc},
	{0xbc, 0x04, 0x18, 0xcc},
	{0xbc, 0x05, 0x00, 0xcc},
	{0xbc, 0x06, 0x00, 0xcc},
	{0xbc, 0x08, 0x30, 0xcc},
	{0xbc, 0x09, 0x40, 0xcc},
	{0xbc, 0x0a, 0x10, 0xcc},
	{0xb8, 0x0c, 0x20, 0xcc},
	{0xb8, 0x0d, 0x70, 0xcc},
	{0xbc, 0x0b, 0x00, 0xcc},
	{0xbc, 0x0c, 0x00, 0xcc},
	{0x35, 0x00, 0xef, 0xbb},
	{0xb3, 0x5c, 0x01, 0xcc},
	{}
};

static const u8 mi1310_socinitVGA_JPG[][4] = {
	{0xb0, 0x03, 0x19, 0xcc},
	{0xb0, 0x04, 0x02, 0xcc},
	{0xb3, 0x00, 0x64, 0xcc},
	{0xb3, 0x00, 0x65, 0xcc},
	{0xb3, 0x05, 0x00, 0xcc},
	{0xb3, 0x06, 0x00, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x34, 0x02, 0xcc},
	{0xb3, 0x35, 0xdd, 0xcc},	/* i2c add: 5d */
	{0xb3, 0x02, 0x00, 0xcc},
	{0xb3, 0x03, 0x0a, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x03, 0xcc},
	{0xb3, 0x23, 0xc0, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x04, 0xcc},
	{0xb3, 0x17, 0xff, 0xcc},
	{0xb3, 0x00, 0x65, 0xcc},
	{0xb8, 0x00, 0x00, 0xcc},
	{0xbc, 0x00, 0xd0, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},
	{0xf0, 0x00, 0x02, 0xbb},
	{0xc8, 0x9f, 0x0b, 0xbb},
	{0x5b, 0x00, 0x01, 0xbb},
	{0x2f, 0xde, 0x20, 0xbb},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x20, 0x03, 0x02, 0xbb},	/* h/v flip */
	{0xf0, 0x00, 0x01, 0xbb},
	{0x05, 0x00, 0x07, 0xbb},
	{0x34, 0x00, 0x00, 0xbb},
	{0x35, 0xff, 0x00, 0xbb},
	{0xdc, 0x07, 0x02, 0xbb},
	{0xdd, 0x3c, 0x18, 0xbb},
	{0xde, 0x92, 0x6d, 0xbb},
	{0xdf, 0xcd, 0xb1, 0xbb},
	{0xe0, 0xff, 0xe7, 0xbb},
	{0x06, 0xf0, 0x0d, 0xbb},
	{0x06, 0x70, 0x0e, 0xbb},
	{0x4c, 0x00, 0x01, 0xbb},
	{0x4d, 0x00, 0x01, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x2e, 0x0c, 0x55, 0xbb},
	{0x21, 0xb6, 0x6e, 0xbb},
	{0x36, 0x30, 0x10, 0xbb},
	{0x37, 0x00, 0xc1, 0xbb},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x07, 0x00, 0x84, 0xbb},
	{0x08, 0x02, 0x4a, 0xbb},
	{0x05, 0x01, 0x10, 0xbb},
	{0x06, 0x00, 0x39, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x58, 0x02, 0x67, 0xbb},
	{0x57, 0x02, 0x00, 0xbb},
	{0x5a, 0x02, 0x67, 0xbb},
	{0x59, 0x02, 0x00, 0xbb},
	{0x5c, 0x12, 0x0d, 0xbb},
	{0x5d, 0x16, 0x11, 0xbb},
	{0x39, 0x06, 0x18, 0xbb},
	{0x3a, 0x06, 0x18, 0xbb},
	{0x3b, 0x06, 0x18, 0xbb},
	{0x3c, 0x06, 0x18, 0xbb},
	{0x64, 0x7b, 0x5b, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x36, 0x30, 0x10, 0xbb},
	{0x37, 0x00, 0xc0, 0xbb},
	{0xbc, 0x0e, 0x00, 0xcc},
	{0xbc, 0x0f, 0x05, 0xcc},
	{0xbc, 0x10, 0xc0, 0xcc},
	{0xbc, 0x11, 0x03, 0xcc},
	{0xb6, 0x00, 0x00, 0xcc},
	{0xb6, 0x03, 0x02, 0xcc},
	{0xb6, 0x02, 0x80, 0xcc},
	{0xb6, 0x05, 0x01, 0xcc},
	{0xb6, 0x04, 0xe0, 0xcc},
	{0xb6, 0x12, 0xf8, 0xcc},
	{0xb6, 0x13, 0x25, 0xcc},
	{0xb6, 0x18, 0x02, 0xcc},
	{0xb6, 0x17, 0x58, 0xcc},
	{0xb6, 0x16, 0x00, 0xcc},
	{0xb6, 0x22, 0x12, 0xcc},
	{0xb6, 0x23, 0x0b, 0xcc},
	{0xbf, 0xc0, 0x39, 0xcc},
	{0xbf, 0xc1, 0x04, 0xcc},
	{0xbf, 0xcc, 0x00, 0xcc},
	{0xbc, 0x02, 0x18, 0xcc},
	{0xbc, 0x03, 0x50, 0xcc},
	{0xbc, 0x04, 0x18, 0xcc},
	{0xbc, 0x05, 0x00, 0xcc},
	{0xbc, 0x06, 0x00, 0xcc},
	{0xbc, 0x08, 0x30, 0xcc},
	{0xbc, 0x09, 0x40, 0xcc},
	{0xbc, 0x0a, 0x10, 0xcc},
	{0xbc, 0x0b, 0x00, 0xcc},
	{0xbc, 0x0c, 0x00, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xf0, 0x00, 0x01, 0xbb},
	{0x80, 0x00, 0x03, 0xbb},
	{0x81, 0xc7, 0x14, 0xbb},
	{0x82, 0xeb, 0xe8, 0xbb},
	{0x83, 0xfe, 0xf4, 0xbb},
	{0x84, 0xcd, 0x10, 0xbb},
	{0x85, 0xf3, 0xee, 0xbb},
	{0x86, 0xff, 0xf1, 0xbb},
	{0x87, 0xcd, 0x10, 0xbb},
	{0x88, 0xf3, 0xee, 0xbb},
	{0x89, 0x01, 0xf1, 0xbb},
	{0x8a, 0xe5, 0x17, 0xbb},
	{0x8b, 0xe8, 0xe2, 0xbb},
	{0x8c, 0xf7, 0xed, 0xbb},
	{0x8d, 0x00, 0xff, 0xbb},
	{0x8e, 0xec, 0x10, 0xbb},
	{0x8f, 0xf0, 0xed, 0xbb},
	{0x90, 0xf9, 0xf2, 0xbb},
	{0x91, 0x00, 0x00, 0xbb},
	{0x92, 0xe9, 0x0d, 0xbb},
	{0x93, 0xf4, 0xf2, 0xbb},
	{0x94, 0xfb, 0xf5, 0xbb},
	{0x95, 0x00, 0xff, 0xbb},
	{0xb6, 0x0f, 0x08, 0xbb},
	{0xb7, 0x3d, 0x16, 0xbb},
	{0xb8, 0x0c, 0x04, 0xbb},
	{0xb9, 0x1c, 0x07, 0xbb},
	{0xba, 0x0a, 0x03, 0xbb},
	{0xbb, 0x1b, 0x09, 0xbb},
	{0xbc, 0x17, 0x0d, 0xbb},
	{0xbd, 0x23, 0x1d, 0xbb},
	{0xbe, 0x00, 0x28, 0xbb},
	{0xbf, 0x11, 0x09, 0xbb},
	{0xc0, 0x16, 0x15, 0xbb},
	{0xc1, 0x00, 0x1b, 0xbb},
	{0xc2, 0x0e, 0x07, 0xbb},
	{0xc3, 0x14, 0x10, 0xbb},
	{0xc4, 0x00, 0x17, 0xbb},
	{0x06, 0x74, 0x8e, 0xbb},
	{0xf0, 0x00, 0x01, 0xbb},
	{0x06, 0xf4, 0x8e, 0xbb},
	{0x00, 0x00, 0x50, 0xdd},
	{0x06, 0x74, 0x8e, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x24, 0x50, 0x20, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x34, 0x0c, 0x50, 0xbb},
	{0xb3, 0x01, 0x41, 0xcc},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x03, 0x03, 0xc0, 0xbb},
	{},
};
static const u8 mi1310_socinitQVGA_JPG[][4] = {
	{0xb0, 0x03, 0x19, 0xcc},	{0xb0, 0x04, 0x02, 0xcc},
	{0xb3, 0x00, 0x64, 0xcc},	{0xb3, 0x00, 0x65, 0xcc},
	{0xb3, 0x05, 0x00, 0xcc},	{0xb3, 0x06, 0x00, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x34, 0x02, 0xcc},	{0xb3, 0x35, 0xdd, 0xcc},
	{0xb3, 0x02, 0x00, 0xcc},	{0xb3, 0x03, 0x0a, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},	{0xb3, 0x22, 0x03, 0xcc},
	{0xb3, 0x23, 0xc0, 0xcc},	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},	{0xb3, 0x16, 0x04, 0xcc},
	{0xb3, 0x17, 0xff, 0xcc},	{0xb3, 0x00, 0x65, 0xcc},
	{0xb8, 0x00, 0x00, 0xcc},	{0xbc, 0x00, 0xf0, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},	{0xf0, 0x00, 0x02, 0xbb},
	{0xc8, 0x9f, 0x0b, 0xbb},	{0x5b, 0x00, 0x01, 0xbb},
	{0x2f, 0xde, 0x20, 0xbb},	{0xf0, 0x00, 0x00, 0xbb},
	{0x20, 0x03, 0x02, 0xbb},	/* h/v flip */
	{0xf0, 0x00, 0x01, 0xbb},
	{0x05, 0x00, 0x07, 0xbb},	{0x34, 0x00, 0x00, 0xbb},
	{0x35, 0xff, 0x00, 0xbb},	{0xdc, 0x07, 0x02, 0xbb},
	{0xdd, 0x3c, 0x18, 0xbb},	{0xde, 0x92, 0x6d, 0xbb},
	{0xdf, 0xcd, 0xb1, 0xbb},	{0xe0, 0xff, 0xe7, 0xbb},
	{0x06, 0xf0, 0x0d, 0xbb},	{0x06, 0x70, 0x0e, 0xbb},
	{0x4c, 0x00, 0x01, 0xbb},	{0x4d, 0x00, 0x01, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x2e, 0x0c, 0x55, 0xbb},
	{0x21, 0xb6, 0x6e, 0xbb},	{0x36, 0x30, 0x10, 0xbb},
	{0x37, 0x00, 0xc1, 0xbb},	{0xf0, 0x00, 0x00, 0xbb},
	{0x07, 0x00, 0x84, 0xbb},	{0x08, 0x02, 0x4a, 0xbb},
	{0x05, 0x01, 0x10, 0xbb},	{0x06, 0x00, 0x39, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x58, 0x02, 0x67, 0xbb},
	{0x57, 0x02, 0x00, 0xbb},	{0x5a, 0x02, 0x67, 0xbb},
	{0x59, 0x02, 0x00, 0xbb},	{0x5c, 0x12, 0x0d, 0xbb},
	{0x5d, 0x16, 0x11, 0xbb},	{0x39, 0x06, 0x18, 0xbb},
	{0x3a, 0x06, 0x18, 0xbb},	{0x3b, 0x06, 0x18, 0xbb},
	{0x3c, 0x06, 0x18, 0xbb},	{0x64, 0x7b, 0x5b, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x36, 0x30, 0x10, 0xbb},
	{0x37, 0x00, 0xc0, 0xbb},	{0xbc, 0x0e, 0x00, 0xcc},
	{0xbc, 0x0f, 0x05, 0xcc},	{0xbc, 0x10, 0xc0, 0xcc},
	{0xbc, 0x11, 0x03, 0xcc},	{0xb6, 0x00, 0x00, 0xcc},
	{0xb6, 0x03, 0x01, 0xcc},	{0xb6, 0x02, 0x40, 0xcc},
	{0xb6, 0x05, 0x00, 0xcc},	{0xb6, 0x04, 0xf0, 0xcc},
	{0xb6, 0x12, 0xf8, 0xcc},	{0xb6, 0x13, 0x25, 0xcc},
	{0xb6, 0x18, 0x00, 0xcc},	{0xb6, 0x17, 0x96, 0xcc},
	{0xb6, 0x16, 0x00, 0xcc},	{0xb6, 0x22, 0x12, 0xcc},
	{0xb6, 0x23, 0x0b, 0xcc},	{0xbf, 0xc0, 0x39, 0xcc},
	{0xbf, 0xc1, 0x04, 0xcc},	{0xbf, 0xcc, 0x00, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},	{0xf0, 0x00, 0x01, 0xbb},
	{0x80, 0x00, 0x03, 0xbb},	{0x81, 0xc7, 0x14, 0xbb},
	{0x82, 0xeb, 0xe8, 0xbb},	{0x83, 0xfe, 0xf4, 0xbb},
	{0x84, 0xcd, 0x10, 0xbb},	{0x85, 0xf3, 0xee, 0xbb},
	{0x86, 0xff, 0xf1, 0xbb},	{0x87, 0xcd, 0x10, 0xbb},
	{0x88, 0xf3, 0xee, 0xbb},	{0x89, 0x01, 0xf1, 0xbb},
	{0x8a, 0xe5, 0x17, 0xbb},	{0x8b, 0xe8, 0xe2, 0xbb},
	{0x8c, 0xf7, 0xed, 0xbb},	{0x8d, 0x00, 0xff, 0xbb},
	{0x8e, 0xec, 0x10, 0xbb},	{0x8f, 0xf0, 0xed, 0xbb},
	{0x90, 0xf9, 0xf2, 0xbb},	{0x91, 0x00, 0x00, 0xbb},
	{0x92, 0xe9, 0x0d, 0xbb},	{0x93, 0xf4, 0xf2, 0xbb},
	{0x94, 0xfb, 0xf5, 0xbb},	{0x95, 0x00, 0xff, 0xbb},
	{0xb6, 0x0f, 0x08, 0xbb},	{0xb7, 0x3d, 0x16, 0xbb},
	{0xb8, 0x0c, 0x04, 0xbb},	{0xb9, 0x1c, 0x07, 0xbb},
	{0xba, 0x0a, 0x03, 0xbb},	{0xbb, 0x1b, 0x09, 0xbb},
	{0xbc, 0x17, 0x0d, 0xbb},	{0xbd, 0x23, 0x1d, 0xbb},
	{0xbe, 0x00, 0x28, 0xbb},	{0xbf, 0x11, 0x09, 0xbb},
	{0xc0, 0x16, 0x15, 0xbb},	{0xc1, 0x00, 0x1b, 0xbb},
	{0xc2, 0x0e, 0x07, 0xbb},	{0xc3, 0x14, 0x10, 0xbb},
	{0xc4, 0x00, 0x17, 0xbb},	{0x06, 0x74, 0x8e, 0xbb},
	{0xf0, 0x00, 0x01, 0xbb},	{0x06, 0xf4, 0x8e, 0xbb},
	{0x00, 0x00, 0x50, 0xdd},	{0x06, 0x74, 0x8e, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x24, 0x50, 0x20, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x34, 0x0c, 0x50, 0xbb},
	{0xb3, 0x01, 0x41, 0xcc},	{0xf0, 0x00, 0x00, 0xbb},
	{0x03, 0x03, 0xc0, 0xbb},
	{},
};
static const u8 mi1310_soc_InitSXGA_JPG[][4] = {
	{0xb0, 0x03, 0x19, 0xcc},
	{0xb0, 0x04, 0x02, 0xcc},
	{0xb3, 0x00, 0x64, 0xcc},
	{0xb3, 0x00, 0x65, 0xcc},
	{0xb3, 0x05, 0x00, 0xcc},
	{0xb3, 0x06, 0x00, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x34, 0x02, 0xcc},
	{0xb3, 0x35, 0xdd, 0xcc},
	{0xb3, 0x02, 0x00, 0xcc},
	{0xb3, 0x03, 0x0a, 0xcc},
	{0xb3, 0x04, 0x0d, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x03, 0xcc},
	{0xb3, 0x23, 0xc0, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x04, 0xcc},
	{0xb3, 0x17, 0xff, 0xcc},
	{0xb3, 0x00, 0x65, 0xcc},
	{0xb8, 0x00, 0x00, 0xcc},
	{0xbc, 0x00, 0x70, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},
	{0xf0, 0x00, 0x02, 0xbb},
	{0xc8, 0x9f, 0x0b, 0xbb},
	{0x5b, 0x00, 0x01, 0xbb},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x20, 0x03, 0x02, 0xbb},	/* h/v flip */
	{0xf0, 0x00, 0x01, 0xbb},
	{0x05, 0x00, 0x07, 0xbb},
	{0x34, 0x00, 0x00, 0xbb},
	{0x35, 0xff, 0x00, 0xbb},
	{0xdc, 0x07, 0x02, 0xbb},
	{0xdd, 0x3c, 0x18, 0xbb},
	{0xde, 0x92, 0x6d, 0xbb},
	{0xdf, 0xcd, 0xb1, 0xbb},
	{0xe0, 0xff, 0xe7, 0xbb},
	{0x06, 0xf0, 0x0d, 0xbb},
	{0x06, 0x70, 0x0e, 0xbb},
	{0x4c, 0x00, 0x01, 0xbb},
	{0x4d, 0x00, 0x01, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x2e, 0x0c, 0x60, 0xbb},
	{0x21, 0xb6, 0x6e, 0xbb},
	{0x37, 0x01, 0x40, 0xbb},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x07, 0x00, 0x84, 0xbb},
	{0x08, 0x02, 0x4a, 0xbb},
	{0x05, 0x01, 0x10, 0xbb},
	{0x06, 0x00, 0x39, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x58, 0x02, 0x67, 0xbb},
	{0x57, 0x02, 0x00, 0xbb},
	{0x5a, 0x02, 0x67, 0xbb},
	{0x59, 0x02, 0x00, 0xbb},
	{0x5c, 0x12, 0x0d, 0xbb},
	{0x5d, 0x16, 0x11, 0xbb},
	{0x39, 0x06, 0x18, 0xbb},
	{0x3a, 0x06, 0x18, 0xbb},
	{0x3b, 0x06, 0x18, 0xbb},
	{0x3c, 0x06, 0x18, 0xbb},
	{0x64, 0x7b, 0x5b, 0xbb},
	{0xb6, 0x00, 0x00, 0xcc},
	{0xb6, 0x03, 0x05, 0xcc},
	{0xb6, 0x02, 0x00, 0xcc},
	{0xb6, 0x05, 0x03, 0xcc},
	{0xb6, 0x04, 0xc0, 0xcc},
	{0xb6, 0x12, 0xf8, 0xcc},
	{0xb6, 0x13, 0x29, 0xcc},
	{0xb6, 0x18, 0x09, 0xcc},
	{0xb6, 0x17, 0x60, 0xcc},
	{0xb6, 0x16, 0x00, 0xcc},
	{0xb6, 0x22, 0x12, 0xcc},
	{0xb6, 0x23, 0x0b, 0xcc},
	{0xbf, 0xc0, 0x39, 0xcc},
	{0xbf, 0xc1, 0x04, 0xcc},
	{0xbf, 0xcc, 0x00, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0x00, 0x00, 0x80, 0xdd},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x00, 0x00, 0x10, 0xdd},
	{0x22, 0xa0, 0x78, 0xbb},
	{0x23, 0xa0, 0x78, 0xbb},
	{0x24, 0x7f, 0x00, 0xbb},
	{0x28, 0xea, 0x02, 0xbb},
	{0x29, 0x86, 0x7a, 0xbb},
	{0x5e, 0x52, 0x4c, 0xbb},
	{0x5f, 0x20, 0x24, 0xbb},
	{0x60, 0x00, 0x02, 0xbb},
	{0x02, 0x00, 0xee, 0xbb},
	{0x03, 0x39, 0x23, 0xbb},
	{0x04, 0x07, 0x24, 0xbb},
	{0x09, 0x00, 0xc0, 0xbb},
	{0x0a, 0x00, 0x79, 0xbb},
	{0x0b, 0x00, 0x04, 0xbb},
	{0x0c, 0x00, 0x5c, 0xbb},
	{0x0d, 0x00, 0xd9, 0xbb},
	{0x0e, 0x00, 0x53, 0xbb},
	{0x0f, 0x00, 0x21, 0xbb},
	{0x10, 0x00, 0xa4, 0xbb},
	{0x11, 0x00, 0xe5, 0xbb},
	{0x15, 0x00, 0x00, 0xbb},
	{0x16, 0x00, 0x00, 0xbb},
	{0x17, 0x00, 0x00, 0xbb},
	{0x18, 0x00, 0x00, 0xbb},
	{0x19, 0x00, 0x00, 0xbb},
	{0x1a, 0x00, 0x00, 0xbb},
	{0x1b, 0x00, 0x00, 0xbb},
	{0x1c, 0x00, 0x00, 0xbb},
	{0x1d, 0x00, 0x00, 0xbb},
	{0x1e, 0x00, 0x00, 0xbb},
	{0xf0, 0x00, 0x01, 0xbb},
	{0x00, 0x00, 0x20, 0xdd},
	{0x06, 0xf0, 0x8e, 0xbb},
	{0x00, 0x00, 0x80, 0xdd},
	{0x06, 0x70, 0x8e, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x00, 0x00, 0x20, 0xdd},
	{0x5e, 0x6a, 0x53, 0xbb},
	{0x5f, 0x40, 0x2c, 0xbb},
	{0xf0, 0x00, 0x01, 0xbb},
	{0x00, 0x00, 0x20, 0xdd},
	{0x58, 0x00, 0x00, 0xbb},
	{0x53, 0x09, 0x03, 0xbb},
	{0x54, 0x31, 0x18, 0xbb},
	{0x55, 0x8b, 0x5f, 0xbb},
	{0x56, 0xc0, 0xa9, 0xbb},
	{0x57, 0xe0, 0xd2, 0xbb},
	{0xe1, 0x00, 0x00, 0xbb},
	{0xdc, 0x09, 0x03, 0xbb},
	{0xdd, 0x31, 0x18, 0xbb},
	{0xde, 0x8b, 0x5f, 0xbb},
	{0xdf, 0xc0, 0xa9, 0xbb},
	{0xe0, 0xe0, 0xd2, 0xbb},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xf0, 0x00, 0x01, 0xbb},
	{0x06, 0xf0, 0x8e, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x2f, 0xde, 0x20, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x24, 0x50, 0x20, 0xbb},
	{0xbc, 0x0e, 0x00, 0xcc},
	{0xbc, 0x0f, 0x05, 0xcc},
	{0xbc, 0x10, 0xc0, 0xcc},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x34, 0x0c, 0x50, 0xbb},
	{0xbc, 0x11, 0x03, 0xcc},
	{0xf0, 0x00, 0x01, 0xbb},
	{0x80, 0x00, 0x03, 0xbb},
	{0x81, 0xc7, 0x14, 0xbb},
	{0x82, 0xeb, 0xe8, 0xbb},
	{0x83, 0xfe, 0xf4, 0xbb},
	{0x84, 0xcd, 0x10, 0xbb},
	{0x85, 0xf3, 0xee, 0xbb},
	{0x86, 0xff, 0xf1, 0xbb},
	{0x87, 0xcd, 0x10, 0xbb},
	{0x88, 0xf3, 0xee, 0xbb},
	{0x89, 0x01, 0xf1, 0xbb},
	{0x8a, 0xe5, 0x17, 0xbb},
	{0x8b, 0xe8, 0xe2, 0xbb},
	{0x8c, 0xf7, 0xed, 0xbb},
	{0x8d, 0x00, 0xff, 0xbb},
	{0x8e, 0xec, 0x10, 0xbb},
	{0x8f, 0xf0, 0xed, 0xbb},
	{0x90, 0xf9, 0xf2, 0xbb},
	{0x91, 0x00, 0x00, 0xbb},
	{0x92, 0xe9, 0x0d, 0xbb},
	{0x93, 0xf4, 0xf2, 0xbb},
	{0x94, 0xfb, 0xf5, 0xbb},
	{0x95, 0x00, 0xff, 0xbb},
	{0xb6, 0x0f, 0x08, 0xbb},
	{0xb7, 0x3d, 0x16, 0xbb},
	{0xb8, 0x0c, 0x04, 0xbb},
	{0xb9, 0x1c, 0x07, 0xbb},
	{0xba, 0x0a, 0x03, 0xbb},
	{0xbb, 0x1b, 0x09, 0xbb},
	{0xbc, 0x17, 0x0d, 0xbb},
	{0xbd, 0x23, 0x1d, 0xbb},
	{0xbe, 0x00, 0x28, 0xbb},
	{0xbf, 0x11, 0x09, 0xbb},
	{0xc0, 0x16, 0x15, 0xbb},
	{0xc1, 0x00, 0x1b, 0xbb},
	{0xc2, 0x0e, 0x07, 0xbb},
	{0xc3, 0x14, 0x10, 0xbb},
	{0xc4, 0x00, 0x17, 0xbb},
	{0x06, 0x74, 0x8e, 0xbb},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x03, 0x03, 0xc0, 0xbb},
	{}
};

static const u8 mi1320_gamma[17] = {
	0x00, 0x13, 0x38, 0x59, 0x79, 0x92, 0xa7, 0xb9, 0xc8,
	0xd4, 0xdf, 0xe7, 0xee, 0xf4, 0xf9, 0xfc, 0xff
};
static const u8 mi1320_matrix[9] = {
	0x54, 0xda, 0x06, 0xf1, 0x50, 0xf4, 0xf7, 0xea, 0x52
};
static const u8 mi1320_initVGA_data[][4] = {
	{0xb3, 0x01, 0x01, 0xcc},	{0x00, 0x00, 0x33, 0xdd},
	{0xb0, 0x03, 0x19, 0xcc},	{0x00, 0x00, 0x33, 0xdd},
	{0xb0, 0x04, 0x02, 0xcc},	{0x00, 0x00, 0x33, 0xdd},
	{0xb3, 0x00, 0x64, 0xcc},	{0xb3, 0x00, 0x65, 0xcc},
	{0xb0, 0x16, 0x03, 0xcc},	{0xb3, 0x05, 0x00, 0xcc},
	{0xb3, 0x06, 0x00, 0xcc},	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},	{0xb3, 0x34, 0x02, 0xcc},
	{0xb3, 0x35, 0xc8, 0xcc},	/* i2c add: 48 */
	{0xb3, 0x02, 0x00, 0xcc},
	{0xb3, 0x03, 0x0a, 0xcc},	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x03, 0xcc},	{0xb3, 0x23, 0xc0, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x04, 0xcc},	{0xb3, 0x17, 0xff, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},	{0xbc, 0x00, 0xd0, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},	{0xf0, 0x00, 0x00, 0xbb},
	{0x0d, 0x00, 0x09, 0xbb},	{0x00, 0x01, 0x00, 0xdd},
	{0x0d, 0x00, 0x08, 0xbb},	{0xf0, 0x00, 0x01, 0xbb},
	{0xa1, 0x05, 0x00, 0xbb},	{0xa4, 0x03, 0xc0, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x00, 0x00, 0x10, 0xdd},
	{0xc8, 0x9f, 0x0b, 0xbb},	{0x00, 0x00, 0x10, 0xdd},
	{0xf0, 0x00, 0x00, 0xbb},	{0x00, 0x00, 0x10, 0xdd},
	{0x20, 0x01, 0x00, 0xbb},	{0x00, 0x00, 0x10, 0xdd},
	{0xf0, 0x00, 0x01, 0xbb},	{0x9d, 0x3c, 0xa0, 0xbb},
	{0x47, 0x30, 0x30, 0xbb},	{0xf0, 0x00, 0x00, 0xbb},
	{0x0a, 0x80, 0x11, 0xbb},	{0x35, 0x00, 0x22, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x9d, 0xc5, 0x05, 0xbb},
	{0xdc, 0x0f, 0xfc, 0xbb},	{0xf0, 0x00, 0x01, 0xbb},
	{0x06, 0x74, 0x0e, 0xbb},	{0x80, 0x00, 0x06, 0xbb},
	{0x81, 0x04, 0x00, 0xbb},	{0x82, 0x01, 0x02, 0xbb},
	{0x83, 0x03, 0x02, 0xbb},	{0x84, 0x05, 0x00, 0xbb},
	{0x85, 0x01, 0x00, 0xbb},	{0x86, 0x03, 0x02, 0xbb},
	{0x87, 0x05, 0x00, 0xbb},	{0x88, 0x01, 0x00, 0xbb},
	{0x89, 0x02, 0x02, 0xbb},	{0x8a, 0xfd, 0x04, 0xbb},
	{0x8b, 0xfc, 0xfd, 0xbb},	{0x8c, 0xff, 0xfd, 0xbb},
	{0x8d, 0x00, 0x00, 0xbb},	{0x8e, 0xfe, 0x05, 0xbb},
	{0x8f, 0xfc, 0xfd, 0xbb},	{0x90, 0xfe, 0xfd, 0xbb},
	{0x91, 0x00, 0x00, 0xbb},	{0x92, 0xfe, 0x03, 0xbb},
	{0x93, 0xfd, 0xfe, 0xbb},	{0x94, 0xff, 0xfd, 0xbb},
	{0x95, 0x00, 0x00, 0xbb},	{0xb6, 0x07, 0x05, 0xbb},
	{0xb7, 0x13, 0x06, 0xbb},	{0xb8, 0x08, 0x06, 0xbb},
	{0xb9, 0x14, 0x08, 0xbb},	{0xba, 0x06, 0x05, 0xbb},
	{0xbb, 0x13, 0x06, 0xbb},	{0xbc, 0x03, 0x01, 0xbb},
	{0xbd, 0x03, 0x04, 0xbb},	{0xbe, 0x00, 0x02, 0xbb},
	{0xbf, 0x03, 0x01, 0xbb},	{0xc0, 0x02, 0x04, 0xbb},
	{0xc1, 0x00, 0x04, 0xbb},	{0xc2, 0x02, 0x01, 0xbb},
	{0xc3, 0x01, 0x03, 0xbb},	{0xc4, 0x00, 0x04, 0xbb},
	{0xf0, 0x00, 0x00, 0xbb},	{0x05, 0x01, 0x13, 0xbb},
	{0x06, 0x00, 0x11, 0xbb},	{0x07, 0x00, 0x85, 0xbb},
	{0x08, 0x00, 0x27, 0xbb},
	{0x20, 0x01, 0x00, 0xbb},	/* h/v flips - was 03 */
	{0x21, 0x80, 0x00, 0xbb},	{0x22, 0x0d, 0x0f, 0xbb},
	{0x24, 0x80, 0x00, 0xbb},	{0x59, 0x00, 0xff, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x39, 0x03, 0x0d, 0xbb},
	{0x3a, 0x06, 0x1b, 0xbb},	{0x3b, 0x00, 0x95, 0xbb},
	{0x3c, 0x04, 0xdb, 0xbb},	{0x57, 0x02, 0x00, 0xbb},
	{0x58, 0x02, 0x66, 0xbb},	{0x59, 0x00, 0xff, 0xbb},
	{0x5a, 0x01, 0x33, 0xbb},	{0x5c, 0x12, 0x0d, 0xbb},
	{0x5d, 0x16, 0x11, 0xbb},	{0x64, 0x5e, 0x1c, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x2f, 0xd1, 0x00, 0xbb},
	{0x5b, 0x00, 0x01, 0xbb},	{0xf0, 0x00, 0x02, 0xbb},
	{0x36, 0x68, 0x10, 0xbb},	{0x00, 0x00, 0x30, 0xdd},
	{0x37, 0x82, 0x00, 0xbb},	{0xbc, 0x0e, 0x00, 0xcc},
	{0xbc, 0x0f, 0x05, 0xcc},	{0xbc, 0x10, 0xc0, 0xcc},
	{0xbc, 0x11, 0x03, 0xcc},	{0xb6, 0x00, 0x00, 0xcc},
	{0xb6, 0x03, 0x05, 0xcc},	{0xb6, 0x02, 0x00, 0xcc},
	{0xb6, 0x05, 0x04, 0xcc},	{0xb6, 0x04, 0x00, 0xcc},
	{0xb6, 0x12, 0xf8, 0xcc},	{0xb6, 0x13, 0x29, 0xcc},
	{0xb6, 0x18, 0x0a, 0xcc},	{0xb6, 0x17, 0x00, 0xcc},
	{0xb6, 0x16, 0x00, 0xcc},	{0xb6, 0x22, 0x12, 0xcc},
	{0xb6, 0x23, 0x0b, 0xcc},	{0xbf, 0xc0, 0x26, 0xcc},
	{0xbf, 0xc1, 0x02, 0xcc},	{0xbf, 0xcc, 0x04, 0xcc},
	{0xbc, 0x02, 0x18, 0xcc},	{0xbc, 0x03, 0x50, 0xcc},
	{0xbc, 0x04, 0x18, 0xcc},	{0xbc, 0x05, 0x00, 0xcc},
	{0xbc, 0x06, 0x00, 0xcc},	{0xbc, 0x08, 0x30, 0xcc},
	{0xbc, 0x09, 0x40, 0xcc},	{0xbc, 0x0a, 0x10, 0xcc},
	{0xbc, 0x0b, 0x00, 0xcc},	{0xbc, 0x0c, 0x00, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},	{0xb3, 0x01, 0x41, 0xcc},
	{}
};
static const u8 mi1320_initQVGA_data[][4] = {
	{0xb3, 0x01, 0x01, 0xcc},	{0x00, 0x00, 0x33, 0xdd},
	{0xb0, 0x03, 0x19, 0xcc},	{0x00, 0x00, 0x33, 0xdd},
	{0xb0, 0x04, 0x02, 0xcc},	{0x00, 0x00, 0x33, 0xdd},
	{0xb3, 0x00, 0x64, 0xcc},	{0xb3, 0x00, 0x65, 0xcc},
	{0xb0, 0x16, 0x03, 0xcc},	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},	{0xb3, 0x34, 0x02, 0xcc},
	{0xb3, 0x35, 0xc8, 0xcc},	{0xb3, 0x02, 0x00, 0xcc},
	{0xb3, 0x03, 0x0a, 0xcc},	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x01, 0xcc},	{0xb3, 0x23, 0xe0, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x00, 0x65, 0xcc},	{0xb8, 0x00, 0x00, 0xcc},
	{0xbc, 0x00, 0xd0, 0xcc},	{0xbc, 0x01, 0x01, 0xcc},
	{0xf0, 0x00, 0x00, 0xbb},	{0x0d, 0x00, 0x09, 0xbb},
	{0x00, 0x01, 0x00, 0xdd},	{0x0d, 0x00, 0x08, 0xbb},
	{0xf0, 0x00, 0x00, 0xbb},	{0x02, 0x00, 0x64, 0xbb},
	{0x05, 0x01, 0x78, 0xbb},	{0x06, 0x00, 0x11, 0xbb},
	{0x07, 0x01, 0x42, 0xbb},	{0x08, 0x00, 0x11, 0xbb},
	{0x20, 0x01, 0x00, 0xbb},	{0x21, 0x80, 0x00, 0xbb},
	{0x22, 0x0d, 0x0f, 0xbb},	{0x24, 0x80, 0x00, 0xbb},
	{0x59, 0x00, 0xff, 0xbb},	{0xf0, 0x00, 0x01, 0xbb},
	{0x9d, 0x3c, 0xa0, 0xbb},	{0x47, 0x30, 0x30, 0xbb},
	{0xf0, 0x00, 0x00, 0xbb},	{0x0a, 0x80, 0x11, 0xbb},
	{0x35, 0x00, 0x22, 0xbb},	{0xf0, 0x00, 0x02, 0xbb},
	{0x9d, 0xc5, 0x05, 0xbb},	{0xdc, 0x0f, 0xfc, 0xbb},
	{0xf0, 0x00, 0x01, 0xbb},	{0x06, 0x74, 0x0e, 0xbb},
	{0x80, 0x00, 0x06, 0xbb},	{0x81, 0x04, 0x00, 0xbb},
	{0x82, 0x01, 0x02, 0xbb},	{0x83, 0x03, 0x02, 0xbb},
	{0x84, 0x05, 0x00, 0xbb},	{0x85, 0x01, 0x00, 0xbb},
	{0x86, 0x03, 0x02, 0xbb},	{0x87, 0x05, 0x00, 0xbb},
	{0x88, 0x01, 0x00, 0xbb},	{0x89, 0x02, 0x02, 0xbb},
	{0x8a, 0xfd, 0x04, 0xbb},	{0x8b, 0xfc, 0xfd, 0xbb},
	{0x8c, 0xff, 0xfd, 0xbb},	{0x8d, 0x00, 0x00, 0xbb},
	{0x8e, 0xfe, 0x05, 0xbb},	{0x8f, 0xfc, 0xfd, 0xbb},
	{0x90, 0xfe, 0xfd, 0xbb},	{0x91, 0x00, 0x00, 0xbb},
	{0x92, 0xfe, 0x03, 0xbb},	{0x93, 0xfd, 0xfe, 0xbb},
	{0x94, 0xff, 0xfd, 0xbb},	{0x95, 0x00, 0x00, 0xbb},
	{0xb6, 0x07, 0x05, 0xbb},	{0xb7, 0x13, 0x06, 0xbb},
	{0xb8, 0x08, 0x06, 0xbb},	{0xb9, 0x14, 0x08, 0xbb},
	{0xba, 0x06, 0x05, 0xbb},	{0xbb, 0x13, 0x06, 0xbb},
	{0xbc, 0x03, 0x01, 0xbb},	{0xbd, 0x03, 0x04, 0xbb},
	{0xbe, 0x00, 0x02, 0xbb},	{0xbf, 0x03, 0x01, 0xbb},
	{0xc0, 0x02, 0x04, 0xbb},	{0xc1, 0x00, 0x04, 0xbb},
	{0xc2, 0x02, 0x01, 0xbb},	{0xc3, 0x01, 0x03, 0xbb},
	{0xc4, 0x00, 0x04, 0xbb},	{0xf0, 0x00, 0x02, 0xbb},
	{0xc8, 0x00, 0x00, 0xbb},	{0x2e, 0x00, 0x00, 0xbb},
	{0x2e, 0x0c, 0x5b, 0xbb},	{0x2f, 0xd1, 0x00, 0xbb},
	{0x39, 0x03, 0xca, 0xbb},	{0x3a, 0x06, 0x80, 0xbb},
	{0x3b, 0x01, 0x52, 0xbb},	{0x3c, 0x05, 0x40, 0xbb},
	{0x57, 0x01, 0x9c, 0xbb},	{0x58, 0x01, 0xee, 0xbb},
	{0x59, 0x00, 0xf0, 0xbb},	{0x5a, 0x01, 0x20, 0xbb},
	{0x5c, 0x1d, 0x17, 0xbb},	{0x5d, 0x22, 0x1c, 0xbb},
	{0x64, 0x1e, 0x1c, 0xbb},	{0x5b, 0x00, 0x01, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},	{0x36, 0x68, 0x10, 0xbb},
	{0x00, 0x00, 0x30, 0xdd},	{0x37, 0x81, 0x00, 0xbb},
	{0xbc, 0x02, 0x18, 0xcc},	{0xbc, 0x03, 0x50, 0xcc},
	{0xbc, 0x04, 0x18, 0xcc},	{0xbc, 0x05, 0x00, 0xcc},
	{0xbc, 0x06, 0x00, 0xcc},	{0xbc, 0x08, 0x30, 0xcc},
	{0xbc, 0x09, 0x40, 0xcc},	{0xbc, 0x0a, 0x10, 0xcc},
	{0xbc, 0x0b, 0x00, 0xcc},	{0xbc, 0x0c, 0x00, 0xcc},
	{0xbf, 0xc0, 0x26, 0xcc},	{0xbf, 0xc1, 0x02, 0xcc},
	{0xbf, 0xcc, 0x04, 0xcc},	{0xb3, 0x5c, 0x01, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{}
};

static const u8 mi1320_soc_InitVGA[][4] = {
	{0xb3, 0x01, 0x01, 0xcc},
	{0xb0, 0x03, 0x19, 0xcc},
	{0xb0, 0x04, 0x02, 0xcc},
	{0x00, 0x00, 0x30, 0xdd},
	{0xb3, 0x00, 0x64, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x34, 0x02, 0xcc},
	{0xb3, 0x35, 0xc8, 0xcc},	/* i2c add: 48 */
	{0xb3, 0x02, 0x00, 0xcc},
	{0xb3, 0x03, 0x0a, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},
	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb8, 0x00, 0x00, 0xcc},
	{0xbc, 0x00, 0x71, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x00, 0x00, 0x10, 0xdd},
	{0xc8, 0x00, 0x00, 0xbb},
	{0x00, 0x00, 0x30, 0xdd},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x00, 0x00, 0x10, 0xdd},
	{0x07, 0x00, 0xe0, 0xbb},
	{0x08, 0x00, 0x0b, 0xbb},
	{0x21, 0x00, 0x0c, 0xbb},
	{0x20, 0x01, 0x03, 0xbb},	/* h/v flip */
	{0xbf, 0xc0, 0x26, 0xcc},
	{0xbf, 0xc1, 0x02, 0xcc},
	{0xbf, 0xcc, 0x04, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x05, 0x01, 0x78, 0xbb},
	{0x06, 0x00, 0x11, 0xbb},
	{0x07, 0x01, 0x42, 0xbb},
	{0x08, 0x00, 0x11, 0xbb},
	{0x20, 0x01, 0x03, 0xbb},	/* h/v flip */
	{0x21, 0x80, 0x00, 0xbb},
	{0x22, 0x0d, 0x0f, 0xbb},
	{0x24, 0x80, 0x00, 0xbb},
	{0x59, 0x00, 0xff, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x39, 0x03, 0xca, 0xbb},
	{0x3a, 0x06, 0x80, 0xbb},
	{0x3b, 0x01, 0x52, 0xbb},
	{0x3c, 0x05, 0x40, 0xbb},
	{0x57, 0x01, 0x9c, 0xbb},
	{0x58, 0x01, 0xee, 0xbb},
	{0x59, 0x00, 0xf0, 0xbb},
	{0x5a, 0x01, 0x20, 0xbb},
	{0x5c, 0x1d, 0x17, 0xbb},
	{0x5d, 0x22, 0x1c, 0xbb},
	{0x64, 0x1e, 0x1c, 0xbb},
	{0x5b, 0x00, 0x00, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x22, 0xa0, 0x78, 0xbb},
	{0x23, 0xa0, 0x78, 0xbb},
	{0x24, 0x7f, 0x00, 0xbb},
	{0x28, 0xea, 0x02, 0xbb},
	{0x29, 0x86, 0x7a, 0xbb},
	{0x5e, 0x52, 0x4c, 0xbb},
	{0x5f, 0x20, 0x24, 0xbb},
	{0x60, 0x00, 0x02, 0xbb},
	{0x02, 0x00, 0xee, 0xbb},
	{0x03, 0x39, 0x23, 0xbb},
	{0x04, 0x07, 0x24, 0xbb},
	{0x09, 0x00, 0xc0, 0xbb},
	{0x0a, 0x00, 0x79, 0xbb},
	{0x0b, 0x00, 0x04, 0xbb},
	{0x0c, 0x00, 0x5c, 0xbb},
	{0x0d, 0x00, 0xd9, 0xbb},
	{0x0e, 0x00, 0x53, 0xbb},
	{0x0f, 0x00, 0x21, 0xbb},
	{0x10, 0x00, 0xa4, 0xbb},
	{0x11, 0x00, 0xe5, 0xbb},
	{0x15, 0x00, 0x00, 0xbb},
	{0x16, 0x00, 0x00, 0xbb},
	{0x17, 0x00, 0x00, 0xbb},
	{0x18, 0x00, 0x00, 0xbb},
	{0x19, 0x00, 0x00, 0xbb},
	{0x1a, 0x00, 0x00, 0xbb},
	{0x1b, 0x00, 0x00, 0xbb},
	{0x1c, 0x00, 0x00, 0xbb},
	{0x1d, 0x00, 0x00, 0xbb},
	{0x1e, 0x00, 0x00, 0xbb},
	{0xf0, 0x00, 0x01, 0xbb},
	{0x06, 0xe0, 0x0e, 0xbb},
	{0x06, 0x60, 0x0e, 0xbb},
	{0xb3, 0x5c, 0x01, 0xcc},
	{}
};
static const u8 mi1320_soc_InitQVGA[][4] = {
	{0xb3, 0x01, 0x01, 0xcc},
	{0xb0, 0x03, 0x19, 0xcc},
	{0xb0, 0x04, 0x02, 0xcc},
	{0x00, 0x00, 0x30, 0xdd},
	{0xb3, 0x00, 0x64, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x34, 0x02, 0xcc},
	{0xb3, 0x35, 0xc8, 0xcc},
	{0xb3, 0x02, 0x00, 0xcc},
	{0xb3, 0x03, 0x0a, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},
	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb8, 0x00, 0x00, 0xcc},
	{0xbc, 0x00, 0xd1, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x00, 0x00, 0x10, 0xdd},
	{0xc8, 0x00, 0x00, 0xbb},
	{0x00, 0x00, 0x30, 0xdd},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x00, 0x00, 0x10, 0xdd},
	{0x07, 0x00, 0xe0, 0xbb},
	{0x08, 0x00, 0x0b, 0xbb},
	{0x21, 0x00, 0x0c, 0xbb},
	{0x20, 0x01, 0x03, 0xbb},	/* h/v flip */
	{0xbf, 0xc0, 0x26, 0xcc},
	{0xbf, 0xc1, 0x02, 0xcc},
	{0xbf, 0xcc, 0x04, 0xcc},
	{0xbc, 0x02, 0x18, 0xcc},
	{0xbc, 0x03, 0x50, 0xcc},
	{0xbc, 0x04, 0x18, 0xcc},
	{0xbc, 0x05, 0x00, 0xcc},
	{0xbc, 0x06, 0x00, 0xcc},
	{0xbc, 0x08, 0x30, 0xcc},
	{0xbc, 0x09, 0x40, 0xcc},
	{0xbc, 0x0a, 0x10, 0xcc},
	{0xbc, 0x0b, 0x00, 0xcc},
	{0xbc, 0x0c, 0x00, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x05, 0x01, 0x78, 0xbb},
	{0x06, 0x00, 0x11, 0xbb},
	{0x07, 0x01, 0x42, 0xbb},
	{0x08, 0x00, 0x11, 0xbb},
	{0x20, 0x01, 0x03, 0xbb},	/* h/v flip */
	{0x21, 0x80, 0x00, 0xbb},
	{0x22, 0x0d, 0x0f, 0xbb},
	{0x24, 0x80, 0x00, 0xbb},
	{0x59, 0x00, 0xff, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x39, 0x03, 0xca, 0xbb},
	{0x3a, 0x06, 0x80, 0xbb},
	{0x3b, 0x01, 0x52, 0xbb},
	{0x3c, 0x05, 0x40, 0xbb},
	{0x57, 0x01, 0x9c, 0xbb},
	{0x58, 0x01, 0xee, 0xbb},
	{0x59, 0x00, 0xf0, 0xbb},
	{0x5a, 0x01, 0x20, 0xbb},
	{0x5c, 0x1d, 0x17, 0xbb},
	{0x5d, 0x22, 0x1c, 0xbb},
	{0x64, 0x1e, 0x1c, 0xbb},
	{0x5b, 0x00, 0x00, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x22, 0xa0, 0x78, 0xbb},
	{0x23, 0xa0, 0x78, 0xbb},
	{0x24, 0x7f, 0x00, 0xbb},
	{0x28, 0xea, 0x02, 0xbb},
	{0x29, 0x86, 0x7a, 0xbb},
	{0x5e, 0x52, 0x4c, 0xbb},
	{0x5f, 0x20, 0x24, 0xbb},
	{0x60, 0x00, 0x02, 0xbb},
	{0x02, 0x00, 0xee, 0xbb},
	{0x03, 0x39, 0x23, 0xbb},
	{0x04, 0x07, 0x24, 0xbb},
	{0x09, 0x00, 0xc0, 0xbb},
	{0x0a, 0x00, 0x79, 0xbb},
	{0x0b, 0x00, 0x04, 0xbb},
	{0x0c, 0x00, 0x5c, 0xbb},
	{0x0d, 0x00, 0xd9, 0xbb},
	{0x0e, 0x00, 0x53, 0xbb},
	{0x0f, 0x00, 0x21, 0xbb},
	{0x10, 0x00, 0xa4, 0xbb},
	{0x11, 0x00, 0xe5, 0xbb},
	{0x15, 0x00, 0x00, 0xbb},
	{0x16, 0x00, 0x00, 0xbb},
	{0x17, 0x00, 0x00, 0xbb},
	{0x18, 0x00, 0x00, 0xbb},
	{0x19, 0x00, 0x00, 0xbb},
	{0x1a, 0x00, 0x00, 0xbb},
	{0x1b, 0x00, 0x00, 0xbb},
	{0x1c, 0x00, 0x00, 0xbb},
	{0x1d, 0x00, 0x00, 0xbb},
	{0x1e, 0x00, 0x00, 0xbb},
	{0xf0, 0x00, 0x01, 0xbb},
	{0x06, 0xe0, 0x0e, 0xbb},
	{0x06, 0x60, 0x0e, 0xbb},
	{0xb3, 0x5c, 0x01, 0xcc},
	{}
};
static const u8 mi1320_soc_InitSXGA[][4] = {
	{0xb3, 0x01, 0x01, 0xcc},
	{0xb0, 0x03, 0x19, 0xcc},
	{0x00, 0x00, 0x30, 0xdd},
	{0xb3, 0x00, 0x64, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x34, 0x02, 0xcc},
	{0xb3, 0x35, 0xc8, 0xcc},
	{0xb3, 0x02, 0x00, 0xcc},
	{0xb3, 0x03, 0x0a, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x04, 0xcc},
	{0xb3, 0x23, 0x00, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x04, 0xcc},
	{0xb3, 0x17, 0xff, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xbc, 0x00, 0x71, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x00, 0x00, 0x30, 0xdd},
	{0xc8, 0x9f, 0x0b, 0xbb},
	{0x00, 0x00, 0x20, 0xdd},
	{0x5b, 0x00, 0x01, 0xbb},
	{0x00, 0x00, 0x20, 0xdd},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x00, 0x00, 0x30, 0xdd},
	{0x20, 0x01, 0x03, 0xbb},	/* h/v flip */
	{0x00, 0x00, 0x20, 0xdd},
	{0xbf, 0xc0, 0x26, 0xcc},
	{0xbf, 0xc1, 0x02, 0xcc},
	{0xbf, 0xcc, 0x04, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x05, 0x01, 0x78, 0xbb},
	{0x06, 0x00, 0x11, 0xbb},
	{0x07, 0x01, 0x42, 0xbb},
	{0x08, 0x00, 0x11, 0xbb},
	{0x20, 0x01, 0x03, 0xbb},	/* h/v flip */
	{0x21, 0x80, 0x00, 0xbb},
	{0x22, 0x0d, 0x0f, 0xbb},
	{0x24, 0x80, 0x00, 0xbb},
	{0x59, 0x00, 0xff, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x39, 0x03, 0xca, 0xbb},
	{0x3a, 0x06, 0x80, 0xbb},
	{0x3b, 0x01, 0x52, 0xbb},
	{0x3c, 0x05, 0x40, 0xbb},
	{0x57, 0x01, 0x9c, 0xbb},
	{0x58, 0x01, 0xee, 0xbb},
	{0x59, 0x00, 0xf0, 0xbb},
	{0x5a, 0x01, 0x20, 0xbb},
	{0x5c, 0x1d, 0x17, 0xbb},
	{0x5d, 0x22, 0x1c, 0xbb},
	{0x64, 0x1e, 0x1c, 0xbb},
	{0x5b, 0x00, 0x00, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x22, 0xa0, 0x78, 0xbb},
	{0x23, 0xa0, 0x78, 0xbb},
	{0x24, 0x7f, 0x00, 0xbb},
	{0x28, 0xea, 0x02, 0xbb},
	{0x29, 0x86, 0x7a, 0xbb},
	{0x5e, 0x52, 0x4c, 0xbb},
	{0x5f, 0x20, 0x24, 0xbb},
	{0x60, 0x00, 0x02, 0xbb},
	{0x02, 0x00, 0xee, 0xbb},
	{0x03, 0x39, 0x23, 0xbb},
	{0x04, 0x07, 0x24, 0xbb},
	{0x09, 0x00, 0xc0, 0xbb},
	{0x0a, 0x00, 0x79, 0xbb},
	{0x0b, 0x00, 0x04, 0xbb},
	{0x0c, 0x00, 0x5c, 0xbb},
	{0x0d, 0x00, 0xd9, 0xbb},
	{0x0e, 0x00, 0x53, 0xbb},
	{0x0f, 0x00, 0x21, 0xbb},
	{0x10, 0x00, 0xa4, 0xbb},
	{0x11, 0x00, 0xe5, 0xbb},
	{0x15, 0x00, 0x00, 0xbb},
	{0x16, 0x00, 0x00, 0xbb},
	{0x17, 0x00, 0x00, 0xbb},
	{0x18, 0x00, 0x00, 0xbb},
	{0x19, 0x00, 0x00, 0xbb},
	{0x1a, 0x00, 0x00, 0xbb},
	{0x1b, 0x00, 0x00, 0xbb},
	{0x1c, 0x00, 0x00, 0xbb},
	{0x1d, 0x00, 0x00, 0xbb},
	{0x1e, 0x00, 0x00, 0xbb},
	{0xf0, 0x00, 0x01, 0xbb},
	{0x06, 0xe0, 0x0e, 0xbb},
	{0x06, 0x60, 0x0e, 0xbb},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xf0, 0x00, 0x00, 0xbb},
	{0x05, 0x01, 0x13, 0xbb},
	{0x06, 0x00, 0x11, 0xbb},
	{0x07, 0x00, 0x85, 0xbb},
	{0x08, 0x00, 0x27, 0xbb},
	{0x20, 0x01, 0x03, 0xbb},	/* h/v flip */
	{0x21, 0x80, 0x00, 0xbb},
	{0x22, 0x0d, 0x0f, 0xbb},
	{0x24, 0x80, 0x00, 0xbb},
	{0x59, 0x00, 0xff, 0xbb},
	{0xf0, 0x00, 0x02, 0xbb},
	{0x39, 0x03, 0x0d, 0xbb},
	{0x3a, 0x06, 0x1b, 0xbb},
	{0x3b, 0x00, 0x95, 0xbb},
	{0x3c, 0x04, 0xdb, 0xbb},
	{0x57, 0x02, 0x00, 0xbb},
	{0x58, 0x02, 0x66, 0xbb},
	{0x59, 0x00, 0xff, 0xbb},
	{0x5a, 0x01, 0x33, 0xbb},
	{0x5c, 0x12, 0x0d, 0xbb},
	{0x5d, 0x16, 0x11, 0xbb},
	{0x64, 0x5e, 0x1c, 0xbb},
	{}
};
static const u8 po3130_gamma[17] = {
	0x00, 0x13, 0x38, 0x59, 0x79, 0x92, 0xa7, 0xb9, 0xc8,
	0xd4, 0xdf, 0xe7, 0xee, 0xf4, 0xf9, 0xfc, 0xff
};
static const u8 po3130_matrix[9] = {
	0x5f, 0xec, 0xf5, 0xf1, 0x5a, 0xf5, 0xf1, 0xec, 0x63
};

static const u8 po3130_initVGA_data[][4] = {
	{0xb0, 0x4d, 0x00, 0xcc},	{0xb3, 0x01, 0x01, 0xcc},
	{0x00, 0x00, 0x50, 0xdd},	{0xb0, 0x03, 0x01, 0xcc},
	{0xb3, 0x00, 0x04, 0xcc},	{0xb3, 0x00, 0x24, 0xcc},
	{0xb3, 0x00, 0x25, 0xcc},	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},	{0xb3, 0x05, 0x00, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},	{0xb3, 0x03, 0x1a, 0xcc},
	{0xb3, 0x04, 0x15, 0xcc},	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe8, 0xcc},	{0xb8, 0x08, 0xe8, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x34, 0x01, 0xcc},
	{0xb3, 0x35, 0xf6, 0xcc},	/* i2c add: 76 */
	{0xb3, 0x00, 0x27, 0xcc},	{0xbc, 0x00, 0x71, 0xcc},
	{0xb8, 0x00, 0x21, 0xcc},	{0xb8, 0x27, 0x20, 0xcc},
	{0xb8, 0x01, 0x79, 0xcc},	{0xb8, 0x81, 0x09, 0xcc},
	{0xb8, 0x2c, 0x50, 0xcc},	{0xb8, 0x2d, 0xf8, 0xcc},
	{0xb8, 0x2e, 0xf8, 0xcc},	{0xb8, 0x2f, 0xf8, 0xcc},
	{0xb8, 0x30, 0x50, 0xcc},	{0xb8, 0x31, 0xf8, 0xcc},
	{0xb8, 0x32, 0xf8, 0xcc},	{0xb8, 0x33, 0xf8, 0xcc},
	{0xb8, 0x34, 0x50, 0xcc},	{0xb8, 0x35, 0x00, 0xcc},
	{0xb8, 0x36, 0x00, 0xcc},	{0xb8, 0x37, 0x00, 0xcc},
	{0x00, 0x1e, 0xc6, 0xaa},	{0x00, 0x20, 0x44, 0xaa},
	{0x00, 0xad, 0x02, 0xaa},	{0x00, 0xae, 0x2c, 0xaa},
	{0x00, 0x12, 0x08, 0xaa},	{0x00, 0x17, 0x41, 0xaa},
	{0x00, 0x19, 0x41, 0xaa},	{0x00, 0x1e, 0x06, 0xaa},
	{0x00, 0x21, 0x00, 0xaa},	{0x00, 0x36, 0xc0, 0xaa},
	{0x00, 0x37, 0xc8, 0xaa},	{0x00, 0x3b, 0x36, 0xaa},
	{0x00, 0x4b, 0xfe, 0xaa},	{0x00, 0x51, 0x1c, 0xaa},
	{0x00, 0x52, 0x01, 0xaa},	{0x00, 0x55, 0x0a, 0xaa},
	{0x00, 0x59, 0x02, 0xaa},	{0x00, 0x5a, 0x04, 0xaa},
	{0x00, 0x5c, 0x10, 0xaa},	{0x00, 0x5d, 0x10, 0xaa},
	{0x00, 0x5e, 0x10, 0xaa},	{0x00, 0x5f, 0x10, 0xaa},
	{0x00, 0x61, 0x00, 0xaa},	{0x00, 0x62, 0x18, 0xaa},
	{0x00, 0x63, 0x30, 0xaa},	{0x00, 0x70, 0x68, 0xaa},
	{0x00, 0x80, 0x71, 0xaa},	{0x00, 0x81, 0x08, 0xaa},
	{0x00, 0x82, 0x00, 0xaa},	{0x00, 0x83, 0x55, 0xaa},
	{0x00, 0x84, 0x06, 0xaa},	{0x00, 0x85, 0x06, 0xaa},
	{0x00, 0x86, 0x13, 0xaa},	{0x00, 0x87, 0x18, 0xaa},
	{0x00, 0xaa, 0x3f, 0xaa},	{0x00, 0xab, 0x44, 0xaa},
	{0x00, 0xb0, 0x68, 0xaa},	{0x00, 0xb5, 0x10, 0xaa},
	{0x00, 0xb8, 0x20, 0xaa},	{0x00, 0xb9, 0xa0, 0xaa},
	{0x00, 0xbc, 0x04, 0xaa},	{0x00, 0x8b, 0x40, 0xaa},
	{0x00, 0x8c, 0x91, 0xaa},	{0x00, 0x8d, 0x8f, 0xaa},
	{0x00, 0x8e, 0x91, 0xaa},	{0x00, 0x8f, 0x43, 0xaa},
	{0x00, 0x90, 0x92, 0xaa},	{0x00, 0x91, 0x89, 0xaa},
	{0x00, 0x92, 0x9d, 0xaa},	{0x00, 0x93, 0x46, 0xaa},
	{0x00, 0xd6, 0x22, 0xaa},	{0x00, 0x73, 0x00, 0xaa},
	{0x00, 0x74, 0x10, 0xaa},	{0x00, 0x75, 0x20, 0xaa},
	{0x00, 0x76, 0x2b, 0xaa},	{0x00, 0x77, 0x36, 0xaa},
	{0x00, 0x78, 0x49, 0xaa},	{0x00, 0x79, 0x5a, 0xaa},
	{0x00, 0x7a, 0x7f, 0xaa},	{0x00, 0x7b, 0x9b, 0xaa},
	{0x00, 0x7c, 0xba, 0xaa},	{0x00, 0x7d, 0xd4, 0xaa},
	{0x00, 0x7e, 0xea, 0xaa},	{0x00, 0xd6, 0x62, 0xaa},
	{0x00, 0x73, 0x00, 0xaa},	{0x00, 0x74, 0x10, 0xaa},
	{0x00, 0x75, 0x20, 0xaa},	{0x00, 0x76, 0x2b, 0xaa},
	{0x00, 0x77, 0x36, 0xaa},	{0x00, 0x78, 0x49, 0xaa},
	{0x00, 0x79, 0x5a, 0xaa},	{0x00, 0x7a, 0x7f, 0xaa},
	{0x00, 0x7b, 0x9b, 0xaa},	{0x00, 0x7c, 0xba, 0xaa},
	{0x00, 0x7d, 0xd4, 0xaa},	{0x00, 0x7e, 0xea, 0xaa},
	{0x00, 0xd6, 0xa2, 0xaa},	{0x00, 0x73, 0x00, 0xaa},
	{0x00, 0x74, 0x10, 0xaa},	{0x00, 0x75, 0x20, 0xaa},
	{0x00, 0x76, 0x2b, 0xaa},	{0x00, 0x77, 0x36, 0xaa},
	{0x00, 0x78, 0x49, 0xaa},	{0x00, 0x79, 0x5a, 0xaa},
	{0x00, 0x7a, 0x7f, 0xaa},	{0x00, 0x7b, 0x9b, 0xaa},
	{0x00, 0x7c, 0xba, 0xaa},	{0x00, 0x7d, 0xd4, 0xaa},
	{0x00, 0x7e, 0xea, 0xaa},
	{0x00, 0x4c, 0x07, 0xaa},
	{0x00, 0x4b, 0xe0, 0xaa},	{0x00, 0x4e, 0x77, 0xaa},
	{0x00, 0x59, 0x02, 0xaa},	{0x00, 0x4d, 0x0a, 0xaa},
/*	{0x00, 0xd1, 0x00, 0xaa},	{0x00, 0x20, 0xc4, 0xaa},
	{0xb8, 0x8e, 0x00, 0xcc},	{0xb8, 0x8f, 0xff, 0xcc}, */
	{0x00, 0xd1, 0x3c, 0xaa},	{0x00, 0x20, 0xc4, 0xaa},
	{0xb8, 0x8e, 0x00, 0xcc},	{0xb8, 0x8f, 0xff, 0xcc},
	{0xb8, 0xfe, 0x00, 0xcc},	{0xb8, 0xff, 0x28, 0xcc},
	{0xb9, 0x00, 0x28, 0xcc},	{0xb9, 0x01, 0x28, 0xcc},
	{0xb9, 0x02, 0x28, 0xcc},	{0xb9, 0x03, 0x00, 0xcc},
	{0xb9, 0x04, 0x00, 0xcc},	{0xb9, 0x05, 0x3c, 0xcc},
	{0xb9, 0x06, 0x3c, 0xcc},	{0xb9, 0x07, 0x3c, 0xcc},
	{0xb9, 0x08, 0x3c, 0xcc},	{0x00, 0x05, 0x00, 0xaa},
	{0xb3, 0x5c, 0x00, 0xcc},	{0xb3, 0x01, 0x41, 0xcc},
	{}
};
static const u8 po3130_rundata[][4] = {
	{0x00, 0x47, 0x45, 0xaa},	{0x00, 0x48, 0x9b, 0xaa},
	{0x00, 0x49, 0x3a, 0xaa},	{0x00, 0x4a, 0x01, 0xaa},
	{0x00, 0x44, 0x40, 0xaa},
/*	{0x00, 0xd5, 0x7c, 0xaa}, */
	{0x00, 0xad, 0x04, 0xaa},	{0x00, 0xae, 0x00, 0xaa},
	{0x00, 0xb0, 0x78, 0xaa},	{0x00, 0x98, 0x02, 0xaa},
	{0x00, 0x94, 0x25, 0xaa},	{0x00, 0x95, 0x25, 0xaa},
	{0x00, 0x59, 0x68, 0xaa},	{0x00, 0x44, 0x20, 0xaa},
	{0x00, 0x17, 0x50, 0xaa},	{0x00, 0x19, 0x50, 0xaa},
	{0x00, 0xd1, 0x3c, 0xaa},	{0x00, 0xd1, 0x3c, 0xaa},
	{0x00, 0x1e, 0x06, 0xaa},	{0x00, 0x1e, 0x06, 0xaa},
	{}
};

static const u8 po3130_initQVGA_data[][4] = {
	{0xb0, 0x4d, 0x00, 0xcc},	{0xb3, 0x01, 0x01, 0xcc},
	{0x00, 0x00, 0x50, 0xdd},	{0xb0, 0x03, 0x09, 0xcc},
	{0xb3, 0x00, 0x04, 0xcc},	{0xb3, 0x00, 0x24, 0xcc},
	{0xb3, 0x00, 0x25, 0xcc},	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},	{0xb3, 0x05, 0x00, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},	{0xb3, 0x03, 0x1a, 0xcc},
	{0xb3, 0x04, 0x15, 0xcc},	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},	{0xb8, 0x08, 0xe0, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x34, 0x01, 0xcc},	{0xb3, 0x35, 0xf6, 0xcc},
	{0xb3, 0x00, 0x27, 0xcc},	{0xbc, 0x00, 0xd1, 0xcc},
	{0xb8, 0x00, 0x21, 0xcc},	{0xb8, 0x27, 0x20, 0xcc},
	{0xb8, 0x01, 0x79, 0xcc},	{0xb8, 0x81, 0x09, 0xcc},
	{0xb8, 0x2c, 0x50, 0xcc},	{0xb8, 0x2d, 0xf8, 0xcc},
	{0xb8, 0x2e, 0xf8, 0xcc},	{0xb8, 0x2f, 0xf8, 0xcc},
	{0xb8, 0x30, 0x50, 0xcc},	{0xb8, 0x31, 0xf8, 0xcc},
	{0xb8, 0x32, 0xf8, 0xcc},	{0xb8, 0x33, 0xf8, 0xcc},
	{0xb8, 0x34, 0x50, 0xcc},	{0xb8, 0x35, 0x00, 0xcc},
	{0xb8, 0x36, 0x00, 0xcc},	{0xb8, 0x37, 0x00, 0xcc},
	{0x00, 0x1e, 0xc6, 0xaa},	{0x00, 0x20, 0x44, 0xaa},
	{0x00, 0xad, 0x02, 0xaa},	{0x00, 0xae, 0x2c, 0xaa},
	{0x00, 0x12, 0x08, 0xaa},	{0x00, 0x17, 0x41, 0xaa},
	{0x00, 0x19, 0x41, 0xaa},	{0x00, 0x1e, 0x06, 0xaa},
	{0x00, 0x21, 0x00, 0xaa},	{0x00, 0x36, 0xc0, 0xaa},
	{0x00, 0x37, 0xc8, 0xaa},	{0x00, 0x3b, 0x36, 0xaa},
	{0x00, 0x4b, 0xfe, 0xaa},	{0x00, 0x51, 0x1c, 0xaa},
	{0x00, 0x52, 0x01, 0xaa},	{0x00, 0x55, 0x0a, 0xaa},
	{0x00, 0x59, 0x6f, 0xaa},	{0x00, 0x5a, 0x04, 0xaa},
	{0x00, 0x5c, 0x10, 0xaa},	{0x00, 0x5d, 0x10, 0xaa},
	{0x00, 0x5e, 0x10, 0xaa},	{0x00, 0x5f, 0x10, 0xaa},
	{0x00, 0x61, 0x00, 0xaa},	{0x00, 0x62, 0x18, 0xaa},
	{0x00, 0x63, 0x30, 0xaa},	{0x00, 0x70, 0x68, 0xaa},
	{0x00, 0x80, 0x71, 0xaa},	{0x00, 0x81, 0x08, 0xaa},
	{0x00, 0x82, 0x00, 0xaa},	{0x00, 0x83, 0x55, 0xaa},
	{0x00, 0x84, 0x06, 0xaa},	{0x00, 0x85, 0x06, 0xaa},
	{0x00, 0x86, 0x13, 0xaa},	{0x00, 0x87, 0x18, 0xaa},
	{0x00, 0xaa, 0x3f, 0xaa},	{0x00, 0xab, 0x44, 0xaa},
	{0x00, 0xb0, 0x68, 0xaa},	{0x00, 0xb5, 0x10, 0xaa},
	{0x00, 0xb8, 0x20, 0xaa},	{0x00, 0xb9, 0xa0, 0xaa},
	{0x00, 0xbc, 0x04, 0xaa},	{0x00, 0x8b, 0x40, 0xaa},
	{0x00, 0x8c, 0x91, 0xaa},	{0x00, 0x8d, 0x8f, 0xaa},
	{0x00, 0x8e, 0x91, 0xaa},	{0x00, 0x8f, 0x43, 0xaa},
	{0x00, 0x90, 0x92, 0xaa},	{0x00, 0x91, 0x89, 0xaa},
	{0x00, 0x92, 0x9d, 0xaa},	{0x00, 0x93, 0x46, 0xaa},
	{0x00, 0xd6, 0x22, 0xaa},	{0x00, 0x73, 0x00, 0xaa},
	{0x00, 0x74, 0x10, 0xaa},	{0x00, 0x75, 0x20, 0xaa},
	{0x00, 0x76, 0x2b, 0xaa},	{0x00, 0x77, 0x36, 0xaa},
	{0x00, 0x78, 0x49, 0xaa},	{0x00, 0x79, 0x5a, 0xaa},
	{0x00, 0x7a, 0x7f, 0xaa},	{0x00, 0x7b, 0x9b, 0xaa},
	{0x00, 0x7c, 0xba, 0xaa},	{0x00, 0x7d, 0xd4, 0xaa},
	{0x00, 0x7e, 0xea, 0xaa},	{0x00, 0xd6, 0x62, 0xaa},
	{0x00, 0x73, 0x00, 0xaa},	{0x00, 0x74, 0x10, 0xaa},
	{0x00, 0x75, 0x20, 0xaa},	{0x00, 0x76, 0x2b, 0xaa},
	{0x00, 0x77, 0x36, 0xaa},	{0x00, 0x78, 0x49, 0xaa},
	{0x00, 0x79, 0x5a, 0xaa},	{0x00, 0x7a, 0x7f, 0xaa},
	{0x00, 0x7b, 0x9b, 0xaa},	{0x00, 0x7c, 0xba, 0xaa},
	{0x00, 0x7d, 0xd4, 0xaa},	{0x00, 0x7e, 0xea, 0xaa},
	{0x00, 0xd6, 0xa2, 0xaa},	{0x00, 0x73, 0x00, 0xaa},
	{0x00, 0x74, 0x10, 0xaa},	{0x00, 0x75, 0x20, 0xaa},
	{0x00, 0x76, 0x2b, 0xaa},	{0x00, 0x77, 0x36, 0xaa},
	{0x00, 0x78, 0x49, 0xaa},	{0x00, 0x79, 0x5a, 0xaa},
	{0x00, 0x7a, 0x7f, 0xaa},	{0x00, 0x7b, 0x9b, 0xaa},
	{0x00, 0x7c, 0xba, 0xaa},	{0x00, 0x7d, 0xd4, 0xaa},
	{0x00, 0x7e, 0xea, 0xaa},	{0x00, 0x4c, 0x07, 0xaa},
	{0x00, 0x4b, 0xe0, 0xaa},	{0x00, 0x4e, 0x77, 0xaa},
	{0x00, 0x59, 0x66, 0xaa},	{0x00, 0x4d, 0x0a, 0xaa},
	{0x00, 0xd1, 0x00, 0xaa},	{0x00, 0x20, 0xc4, 0xaa},
	{0xb8, 0x8e, 0x00, 0xcc},	{0xb8, 0x8f, 0xff, 0xcc},
	{0xb8, 0xfe, 0x00, 0xcc},	{0xb8, 0xff, 0x28, 0xcc},
	{0xb9, 0x00, 0x28, 0xcc},	{0xb9, 0x01, 0x28, 0xcc},
	{0xb9, 0x02, 0x28, 0xcc},	{0xb9, 0x03, 0x00, 0xcc},
	{0xb9, 0x04, 0x00, 0xcc},	{0xb9, 0x05, 0x3c, 0xcc},
	{0xb9, 0x06, 0x3c, 0xcc},	{0xb9, 0x07, 0x3c, 0xcc},
	{0xb9, 0x08, 0x3c, 0xcc},	{0xbc, 0x02, 0x18, 0xcc},
	{0xbc, 0x03, 0x50, 0xcc},	{0xbc, 0x04, 0x18, 0xcc},
	{0xbc, 0x05, 0x00, 0xcc},	{0xbc, 0x06, 0x00, 0xcc},
	{0xbc, 0x08, 0x30, 0xcc},	{0xbc, 0x09, 0x40, 0xcc},
	{0xbc, 0x0a, 0x10, 0xcc},	{0xbc, 0x0b, 0x00, 0xcc},
	{0xbc, 0x0c, 0x00, 0xcc},	{0x00, 0x05, 0x00, 0xaa},
	{0xb3, 0x5c, 0x00, 0xcc},	{0xb3, 0x01, 0x41, 0xcc},
	{}
};

static const u8 hv7131r_gamma[17] = {
	0x00, 0x13, 0x38, 0x59, 0x79, 0x92, 0xa7, 0xb9, 0xc8,
	0xd4, 0xdf, 0xe7, 0xee, 0xf4, 0xf9, 0xfc, 0xff
};
static const u8 hv7131r_matrix[9] = {
	0x5f, 0xec, 0xf5, 0xf1, 0x5a, 0xf5, 0xf1, 0xec, 0x63
};
static const u8 hv7131r_initVGA_data[][4] = {
	{0xb3, 0x01, 0x01, 0xcc},
	{0xb0, 0x03, 0x19, 0xcc},
	{0xb0, 0x04, 0x02, 0xcc},
	{0x00, 0x00, 0x20, 0xdd},
	{0xb3, 0x00, 0x24, 0xcc},
	{0xb3, 0x00, 0x25, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x03, 0xcc},
	{0xb3, 0x01, 0x45, 0xcc},
	{0xb3, 0x03, 0x0b, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x02, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},
	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x34, 0x01, 0xcc},
	{0xb3, 0x35, 0x91, 0xcc},	/* i2c add: 11 */
	{0xb3, 0x00, 0x27, 0xcc},
	{0xbc, 0x00, 0x73, 0xcc},
	{0xb8, 0x00, 0x23, 0xcc},
	{0xb8, 0x2c, 0x50, 0xcc},
	{0xb8, 0x2d, 0xf8, 0xcc},
	{0xb8, 0x2e, 0xf8, 0xcc},
	{0xb8, 0x2f, 0xf8, 0xcc},
	{0xb8, 0x30, 0x50, 0xcc},
	{0xb8, 0x31, 0xf8, 0xcc},
	{0xb8, 0x32, 0xf8, 0xcc},
	{0xb8, 0x33, 0xf8, 0xcc},
	{0xb8, 0x34, 0x58, 0xcc},
	{0xb8, 0x35, 0x00, 0xcc},
	{0xb8, 0x36, 0x00, 0xcc},
	{0xb8, 0x37, 0x00, 0xcc},
	{0xb8, 0x27, 0x20, 0xcc},
	{0xb8, 0x01, 0x7d, 0xcc},
	{0xb8, 0x81, 0x09, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0xb8, 0x8e, 0x00, 0xcc},
	{0xb8, 0x8f, 0xff, 0xcc},
	{0x00, 0x01, 0x0c, 0xaa},
	{0x00, 0x14, 0x01, 0xaa},
	{0x00, 0x15, 0xe6, 0xaa},
	{0x00, 0x16, 0x02, 0xaa},
	{0x00, 0x17, 0x86, 0xaa},
	{0x00, 0x23, 0x00, 0xaa},
	{0x00, 0x25, 0x03, 0xaa},
	{0x00, 0x26, 0xa9, 0xaa},
	{0x00, 0x27, 0x80, 0xaa},
	{0x00, 0x30, 0x18, 0xaa},
	{0xb6, 0x00, 0x00, 0xcc},
	{0xb6, 0x03, 0x02, 0xcc},
	{0xb6, 0x02, 0x80, 0xcc},
	{0xb6, 0x05, 0x01, 0xcc},
	{0xb6, 0x04, 0xe0, 0xcc},
	{0xb6, 0x12, 0x78, 0xcc},
	{0xb6, 0x18, 0x02, 0xcc},
	{0xb6, 0x17, 0x58, 0xcc},
	{0xb6, 0x16, 0x00, 0xcc},
	{0xb6, 0x22, 0x12, 0xcc},
	{0xb6, 0x23, 0x0b, 0xcc},
	{0xb3, 0x02, 0x02, 0xcc},
	{0xbf, 0xc0, 0x39, 0xcc},
	{0xbf, 0xc1, 0x04, 0xcc},
	{0xbf, 0xcc, 0x10, 0xcc},
	{0xb6, 0x12, 0xf8, 0xcc},
	{0xb6, 0x13, 0x13, 0xcc},
	{0xb9, 0x12, 0x00, 0xcc},
	{0xb9, 0x13, 0x0a, 0xcc},
	{0xb9, 0x14, 0x0a, 0xcc},
	{0xb9, 0x15, 0x0a, 0xcc},
	{0xb9, 0x16, 0x0a, 0xcc},
	{0xb8, 0x0c, 0x20, 0xcc},
	{0xb8, 0x0d, 0x70, 0xcc},
	{0xb9, 0x18, 0x00, 0xcc},
	{0xb9, 0x19, 0x0f, 0xcc},
	{0xb9, 0x1a, 0x0f, 0xcc},
	{0xb9, 0x1b, 0x0f, 0xcc},
	{0xb9, 0x1c, 0x0f, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{}
};

static const u8 hv7131r_initQVGA_data[][4] = {
	{0xb3, 0x01, 0x01, 0xcc},
	{0xb0, 0x03, 0x19, 0xcc},
	{0xb0, 0x04, 0x02, 0xcc},
	{0x00, 0x00, 0x20, 0xdd},
	{0xb3, 0x00, 0x24, 0xcc},
	{0xb3, 0x00, 0x25, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x03, 0xcc},
	{0xb3, 0x01, 0x45, 0xcc},
	{0xb3, 0x03, 0x0b, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x02, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},
	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x34, 0x01, 0xcc},
	{0xb3, 0x35, 0x91, 0xcc},
	{0xb3, 0x00, 0x27, 0xcc},
	{0xbc, 0x00, 0xd3, 0xcc},
	{0xb8, 0x00, 0x23, 0xcc},
	{0xb8, 0x2c, 0x50, 0xcc},
	{0xb8, 0x2d, 0xf8, 0xcc},
	{0xb8, 0x2e, 0xf8, 0xcc},
	{0xb8, 0x2f, 0xf8, 0xcc},
	{0xb8, 0x30, 0x50, 0xcc},
	{0xb8, 0x31, 0xf8, 0xcc},
	{0xb8, 0x32, 0xf8, 0xcc},
	{0xb8, 0x33, 0xf8, 0xcc},
	{0xb8, 0x34, 0x58, 0xcc},
	{0xb8, 0x35, 0x00, 0xcc},
	{0xb8, 0x36, 0x00, 0xcc},
	{0xb8, 0x37, 0x00, 0xcc},
	{0xb8, 0x27, 0x20, 0xcc},
	{0xb8, 0x01, 0x7d, 0xcc},
	{0xb8, 0x81, 0x09, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0xb8, 0x8e, 0x00, 0xcc},
	{0xb8, 0x8f, 0xff, 0xcc},
	{0x00, 0x01, 0x0c, 0xaa},
	{0x00, 0x14, 0x01, 0xaa},
	{0x00, 0x15, 0xe6, 0xaa},
	{0x00, 0x16, 0x02, 0xaa},
	{0x00, 0x17, 0x86, 0xaa},
	{0x00, 0x23, 0x00, 0xaa},
	{0x00, 0x25, 0x03, 0xaa},
	{0x00, 0x26, 0xa9, 0xaa},
	{0x00, 0x27, 0x80, 0xaa},
	{0x00, 0x30, 0x18, 0xaa},
	{0xb6, 0x00, 0x00, 0xcc},
	{0xb6, 0x03, 0x01, 0xcc},
	{0xb6, 0x02, 0x40, 0xcc},
	{0xb6, 0x05, 0x00, 0xcc},
	{0xb6, 0x04, 0xf0, 0xcc},
	{0xb6, 0x12, 0x78, 0xcc},
	{0xb6, 0x18, 0x00, 0xcc},
	{0xb6, 0x17, 0x96, 0xcc},
	{0xb6, 0x16, 0x00, 0xcc},
	{0xb6, 0x22, 0x12, 0xcc},
	{0xb6, 0x23, 0x0b, 0xcc},
	{0xb3, 0x02, 0x02, 0xcc},
	{0xbf, 0xc0, 0x39, 0xcc},
	{0xbf, 0xc1, 0x04, 0xcc},
	{0xbf, 0xcc, 0x10, 0xcc},
	{0xbc, 0x02, 0x18, 0xcc},
	{0xbc, 0x03, 0x50, 0xcc},
	{0xbc, 0x04, 0x18, 0xcc},
	{0xbc, 0x05, 0x00, 0xcc},
	{0xbc, 0x06, 0x00, 0xcc},
	{0xbc, 0x08, 0x30, 0xcc},
	{0xbc, 0x09, 0x40, 0xcc},
	{0xbc, 0x0a, 0x10, 0xcc},
	{0xbc, 0x0b, 0x00, 0xcc},
	{0xbc, 0x0c, 0x00, 0xcc},
	{0xb9, 0x12, 0x00, 0xcc},
	{0xb9, 0x13, 0x0a, 0xcc},
	{0xb9, 0x14, 0x0a, 0xcc},
	{0xb9, 0x15, 0x0a, 0xcc},
	{0xb9, 0x16, 0x0a, 0xcc},
	{0xb9, 0x18, 0x00, 0xcc},
	{0xb9, 0x19, 0x0f, 0xcc},
	{0xb8, 0x0c, 0x20, 0xcc},
	{0xb8, 0x0d, 0x70, 0xcc},
	{0xb9, 0x1a, 0x0f, 0xcc},
	{0xb9, 0x1b, 0x0f, 0xcc},
	{0xb9, 0x1c, 0x0f, 0xcc},
	{0xb6, 0x12, 0xf8, 0xcc},
	{0xb6, 0x13, 0x13, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{}
};

static const u8 ov7660_gamma[17] = {
	0x00, 0x13, 0x38, 0x59, 0x79, 0x92, 0xa7, 0xb9, 0xc8,
	0xd4, 0xdf, 0xe7, 0xee, 0xf4, 0xf9, 0xfc, 0xff
};
static const u8 ov7660_matrix[9] = {
	0x5a, 0xf0, 0xf6, 0xf3, 0x57, 0xf6, 0xf3, 0xef, 0x62
};
static const u8 ov7660_initVGA_data[][4] = {
	{0xb0, 0x4d, 0x00, 0xcc},	{0xb3, 0x01, 0x01, 0xcc},
	{0x00, 0x00, 0x50, 0xdd},
	{0xb0, 0x03, 0x01, 0xcc},
	{0xb3, 0x00, 0x21, 0xcc},	{0xb3, 0x00, 0x26, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x03, 0xcc},
	{0xb3, 0x03, 0x1f, 0xcc},	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x05, 0x00, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},/* 0xb315  <-0 href startl */
	{0xb3, 0x16, 0x02, 0xcc},	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},	{0xb3, 0x1d, 0x01, 0xcc},
	{0xb3, 0x1f, 0x02, 0xcc},
	{0xb3, 0x34, 0x01, 0xcc},
	{0xb3, 0x35, 0xa1, 0xcc},	/* i2c add: 21 */
	{0xb3, 0x00, 0x26, 0xcc},
	{0xb8, 0x00, 0x33, 0xcc}, /* 13 */
	{0xb8, 0x01, 0x7d, 0xcc},
	{0xbc, 0x00, 0x73, 0xcc},	{0xb8, 0x81, 0x09, 0xcc},
	{0xb8, 0x27, 0x20, 0xcc},
	{0xb8, 0x8f, 0x50, 0xcc},
	{0x00, 0x01, 0x80, 0xaa},	{0x00, 0x02, 0x80, 0xaa},
	{0x00, 0x12, 0x80, 0xaa},
	{0x00, 0x12, 0x05, 0xaa},
	{0x00, 0x1e, 0x01, 0xaa},	/* MVFP */
	{0x00, 0x3d, 0x40, 0xaa}, /* 0x3d <-40 gamma 01 */
	{0x00, 0x41, 0x00, 0xaa}, /* edge 00 */
	{0x00, 0x0d, 0x48, 0xaa},	{0x00, 0x0e, 0x04, 0xaa},
	{0x00, 0x13, 0xa7, 0xaa},
	{0x00, 0x40, 0xc1, 0xaa},	{0x00, 0x35, 0x00, 0xaa},
	{0x00, 0x36, 0x00, 0xaa},
	{0x00, 0x3c, 0x68, 0xaa},	{0x00, 0x1b, 0x05, 0xaa},
	{0x00, 0x39, 0x43, 0xaa},
	{0x00, 0x8d, 0xcf, 0xaa},
	{0x00, 0x8b, 0xcc, 0xaa},	{0x00, 0x8c, 0xcc, 0xaa},
	{0x00, 0x0f, 0x62, 0xaa},
	{0x00, 0x35, 0x84, 0xaa},
	{0x00, 0x3b, 0x08, 0xaa}, /* 0 * Nightframe 1/4 + 50Hz -> 0xC8 */
	{0x00, 0x3a, 0x00, 0xaa}, /* mx change yuyv format 00, 04, 01; 08, 0c*/
	{0x00, 0x14, 0x2a, 0xaa}, /* agc ampli */
	{0x00, 0x9e, 0x40, 0xaa},	{0xb8, 0x8f, 0x50, 0xcc},
	{0x00, 0x01, 0x80, 0xaa},
	{0x00, 0x02, 0x80, 0xaa},
	{0xb8, 0xfe, 0x00, 0xcc},	{0xb8, 0xff, 0x28, 0xcc},
	{0xb9, 0x00, 0x28, 0xcc},
	{0xb9, 0x01, 0x28, 0xcc},	{0xb9, 0x02, 0x28, 0xcc},
	{0xb9, 0x03, 0x00, 0xcc},
	{0xb9, 0x04, 0x00, 0xcc},
	{0xb9, 0x05, 0x3c, 0xcc},	{0xb9, 0x06, 0x3c, 0xcc},
	{0xb9, 0x07, 0x3c, 0xcc},
	{0xb9, 0x08, 0x3c, 0xcc},

	{0xb8, 0x8e, 0x00, 0xcc},	{0xb8, 0x8f, 0xff, 0xcc},

	{0x00, 0x29, 0x3c, 0xaa},	{0xb3, 0x01, 0x45, 0xcc},
	{}
};
static const u8 ov7660_initQVGA_data[][4] = {
	{0xb0, 0x4d, 0x00, 0xcc},	{0xb3, 0x01, 0x01, 0xcc},
	{0x00, 0x00, 0x50, 0xdd},	{0xb0, 0x03, 0x01, 0xcc},
	{0xb3, 0x00, 0x21, 0xcc},	{0xb3, 0x00, 0x26, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},	{0xb3, 0x06, 0x03, 0xcc},
	{0xb3, 0x03, 0x1f, 0xcc},	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x05, 0x00, 0xcc},	{0xb3, 0x06, 0x01, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},/* 0xb315  <-0 href startl */
	{0xb3, 0x16, 0x02, 0xcc},	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},	{0xb3, 0x1d, 0x01, 0xcc},
	{0xb3, 0x1f, 0x02, 0xcc},	{0xb3, 0x34, 0x01, 0xcc},
	{0xb3, 0x35, 0xa1, 0xcc},	{0xb3, 0x00, 0x26, 0xcc},
	{0xb8, 0x00, 0x33, 0xcc}, /* 13 */
	{0xb8, 0x01, 0x7d, 0xcc},
/* sizer */
	{0xbc, 0x00, 0xd3, 0xcc},
	{0xb8, 0x81, 0x09, 0xcc},	{0xb8, 0x81, 0x09, 0xcc},
	{0xb8, 0x27, 0x20, 0xcc},	{0xb8, 0x8f, 0x50, 0xcc},
	{0x00, 0x01, 0x80, 0xaa},	{0x00, 0x02, 0x80, 0xaa},
	{0x00, 0x12, 0x80, 0xaa},	{0x00, 0x12, 0x05, 0xaa},
	{0x00, 0x1e, 0x01, 0xaa},	/* MVFP */
	{0x00, 0x3d, 0x40, 0xaa}, /* 0x3d <-40 gamma 01 */
	{0x00, 0x41, 0x00, 0xaa}, /* edge 00 */
	{0x00, 0x0d, 0x48, 0xaa},	{0x00, 0x0e, 0x04, 0xaa},
	{0x00, 0x13, 0xa7, 0xaa},
	{0x00, 0x40, 0xc1, 0xaa},	{0x00, 0x35, 0x00, 0xaa},
	{0x00, 0x36, 0x00, 0xaa},
	{0x00, 0x3c, 0x68, 0xaa},	{0x00, 0x1b, 0x05, 0xaa},
	{0x00, 0x39, 0x43, 0xaa},	{0x00, 0x8d, 0xcf, 0xaa},
	{0x00, 0x8b, 0xcc, 0xaa},	{0x00, 0x8c, 0xcc, 0xaa},
	{0x00, 0x0f, 0x62, 0xaa},	{0x00, 0x35, 0x84, 0xaa},
	{0x00, 0x3b, 0x08, 0xaa}, /* 0  * Nightframe 1/4 + 50Hz -> 0xC8 */
	{0x00, 0x3a, 0x00, 0xaa}, /* mx change yuyv format 00, 04, 01; 08, 0c*/
	{0x00, 0x14, 0x2a, 0xaa}, /* agc ampli */
	{0x00, 0x9e, 0x40, 0xaa},	{0xb8, 0x8f, 0x50, 0xcc},
	{0x00, 0x01, 0x80, 0xaa},
	{0x00, 0x02, 0x80, 0xaa},
/* sizer filters */
	{0xbc, 0x02, 0x08, 0xcc},
	{0xbc, 0x03, 0x70, 0xcc},
	{0xb8, 0x35, 0x00, 0xcc},
	{0xb8, 0x36, 0x00, 0xcc},
	{0xb8, 0x37, 0x00, 0xcc},
	{0xbc, 0x04, 0x08, 0xcc},
	{0xbc, 0x05, 0x00, 0xcc},
	{0xbc, 0x06, 0x00, 0xcc},
	{0xbc, 0x08, 0x3c, 0xcc},
	{0xbc, 0x09, 0x40, 0xcc},
	{0xbc, 0x0a, 0x04, 0xcc},
	{0xbc, 0x0b, 0x00, 0xcc},
	{0xbc, 0x0c, 0x00, 0xcc},
/* */
	{0xb8, 0xfe, 0x00, 0xcc},
	{0xb8, 0xff, 0x28, 0xcc},
/* */
	{0xb9, 0x00, 0x28, 0xcc},	{0xb9, 0x01, 0x28, 0xcc},
	{0xb9, 0x02, 0x28, 0xcc},	{0xb9, 0x03, 0x00, 0xcc},
	{0xb9, 0x04, 0x00, 0xcc},	{0xb9, 0x05, 0x3c, 0xcc},
	{0xb9, 0x06, 0x3c, 0xcc},	{0xb9, 0x07, 0x3c, 0xcc},
	{0xb9, 0x08, 0x3c, 0xcc},
/* */
	{0xb8, 0x8e, 0x00, 0xcc},
	{0xb8, 0x8f, 0xff, 0xcc}, /* ff */
	{0x00, 0x29, 0x3c, 0xaa},
	{0xb3, 0x01, 0x45, 0xcc}, /* 45 */
	{}
};

static const u8 ov7660_50HZ[][4] = {
	{0x00, 0x3b, 0x08, 0xaa},
	{0x00, 0x9d, 0x40, 0xaa},
	{0x00, 0x13, 0xa7, 0xaa},
	{}
};

static const u8 ov7660_60HZ[][4] = {
	{0x00, 0x3b, 0x00, 0xaa},
	{0x00, 0x9e, 0x40, 0xaa},
	{0x00, 0x13, 0xa7, 0xaa},
	{}
};

static const u8 ov7660_NoFliker[][4] = {
	{0x00, 0x13, 0x87, 0xaa},
	{}
};

static const u8 ov7670_InitVGA[][4] = {
	{0xb3, 0x01, 0x05, 0xcc},
	{0x00, 0x00, 0x30, 0xdd},
	{0xb0, 0x03, 0x19, 0xcc},
	{0x00, 0x00, 0x10, 0xdd},
	{0xb0, 0x04, 0x02, 0xcc},
	{0x00, 0x00, 0x10, 0xdd},
	{0xb3, 0x00, 0x66, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb0, 0x16, 0x01, 0xcc},
	{0xb3, 0x35, 0xa1, 0xcc},	/* i2c add: 21 */
	{0xb3, 0x34, 0x01, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x02, 0x02, 0xcc},
	{0xb3, 0x03, 0x1f, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},
	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},
	{0xbc, 0x00, 0x41, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},
	{0x00, 0x12, 0x80, 0xaa},
	{0x00, 0x00, 0x20, 0xdd},
	{0x00, 0x12, 0x00, 0xaa},
	{0x00, 0x11, 0x40, 0xaa},
	{0x00, 0x6b, 0x0a, 0xaa},
	{0x00, 0x3a, 0x04, 0xaa},
	{0x00, 0x40, 0xc0, 0xaa},
	{0x00, 0x8c, 0x00, 0xaa},
	{0x00, 0x7a, 0x29, 0xaa},
	{0x00, 0x7b, 0x0e, 0xaa},
	{0x00, 0x7c, 0x1a, 0xaa},
	{0x00, 0x7d, 0x31, 0xaa},
	{0x00, 0x7e, 0x53, 0xaa},
	{0x00, 0x7f, 0x60, 0xaa},
	{0x00, 0x80, 0x6b, 0xaa},
	{0x00, 0x81, 0x73, 0xaa},
	{0x00, 0x82, 0x7b, 0xaa},
	{0x00, 0x83, 0x82, 0xaa},
	{0x00, 0x84, 0x89, 0xaa},
	{0x00, 0x85, 0x96, 0xaa},
	{0x00, 0x86, 0xa1, 0xaa},
	{0x00, 0x87, 0xb7, 0xaa},
	{0x00, 0x88, 0xcc, 0xaa},
	{0x00, 0x89, 0xe1, 0xaa},
	{0x00, 0x13, 0xe0, 0xaa},
	{0x00, 0x00, 0x00, 0xaa},
	{0x00, 0x10, 0x00, 0xaa},
	{0x00, 0x0d, 0x40, 0xaa},
	{0x00, 0x14, 0x28, 0xaa},
	{0x00, 0xa5, 0x05, 0xaa},
	{0x00, 0xab, 0x07, 0xaa},
	{0x00, 0x24, 0x95, 0xaa},
	{0x00, 0x25, 0x33, 0xaa},
	{0x00, 0x26, 0xe3, 0xaa},
	{0x00, 0x9f, 0x88, 0xaa},
	{0x00, 0xa0, 0x78, 0xaa},
	{0x00, 0x55, 0x90, 0xaa},
	{0x00, 0xa1, 0x03, 0xaa},
	{0x00, 0xa6, 0xe0, 0xaa},
	{0x00, 0xa7, 0xd8, 0xaa},
	{0x00, 0xa8, 0xf0, 0xaa},
	{0x00, 0xa9, 0x90, 0xaa},
	{0x00, 0xaa, 0x14, 0xaa},
	{0x00, 0x13, 0xe5, 0xaa},
	{0x00, 0x0e, 0x61, 0xaa},
	{0x00, 0x0f, 0x4b, 0xaa},
	{0x00, 0x16, 0x02, 0xaa},
	{0x00, 0x1e, 0x07, 0xaa},	/* MVFP */
	{0x00, 0x21, 0x02, 0xaa},
	{0x00, 0x22, 0x91, 0xaa},
	{0x00, 0x29, 0x07, 0xaa},
	{0x00, 0x33, 0x0b, 0xaa},
	{0x00, 0x35, 0x0b, 0xaa},
	{0x00, 0x37, 0x1d, 0xaa},
	{0x00, 0x38, 0x71, 0xaa},
	{0x00, 0x39, 0x2a, 0xaa},
	{0x00, 0x3c, 0x78, 0xaa},
	{0x00, 0x4d, 0x40, 0xaa},
	{0x00, 0x4e, 0x20, 0xaa},
	{0x00, 0x74, 0x19, 0xaa},
	{0x00, 0x8d, 0x4f, 0xaa},
	{0x00, 0x8e, 0x00, 0xaa},
	{0x00, 0x8f, 0x00, 0xaa},
	{0x00, 0x90, 0x00, 0xaa},
	{0x00, 0x91, 0x00, 0xaa},
	{0x00, 0x96, 0x00, 0xaa},
	{0x00, 0x9a, 0x80, 0xaa},
	{0x00, 0xb0, 0x84, 0xaa},
	{0x00, 0xb1, 0x0c, 0xaa},
	{0x00, 0xb2, 0x0e, 0xaa},
	{0x00, 0xb3, 0x82, 0xaa},
	{0x00, 0xb8, 0x0a, 0xaa},
	{0x00, 0x43, 0x14, 0xaa},
	{0x00, 0x44, 0xf0, 0xaa},
	{0x00, 0x45, 0x45, 0xaa},
	{0x00, 0x46, 0x63, 0xaa},
	{0x00, 0x47, 0x2d, 0xaa},
	{0x00, 0x48, 0x46, 0xaa},
	{0x00, 0x59, 0x88, 0xaa},
	{0x00, 0x5a, 0xa0, 0xaa},
	{0x00, 0x5b, 0xc6, 0xaa},
	{0x00, 0x5c, 0x7d, 0xaa},
	{0x00, 0x5d, 0x5f, 0xaa},
	{0x00, 0x5e, 0x19, 0xaa},
	{0x00, 0x6c, 0x0a, 0xaa},
	{0x00, 0x6d, 0x55, 0xaa},
	{0x00, 0x6e, 0x11, 0xaa},
	{0x00, 0x6f, 0x9e, 0xaa},
	{0x00, 0x69, 0x00, 0xaa},
	{0x00, 0x6a, 0x40, 0xaa},
	{0x00, 0x01, 0x40, 0xaa},
	{0x00, 0x02, 0x40, 0xaa},
	{0x00, 0x13, 0xe7, 0xaa},
	{0x00, 0x5f, 0xf0, 0xaa},
	{0x00, 0x60, 0xf0, 0xaa},
	{0x00, 0x61, 0xf0, 0xaa},
	{0x00, 0x27, 0xa0, 0xaa},
	{0x00, 0x28, 0x80, 0xaa},
	{0x00, 0x2c, 0x90, 0xaa},
	{0x00, 0x4f, 0x66, 0xaa},
	{0x00, 0x50, 0x66, 0xaa},
	{0x00, 0x51, 0x00, 0xaa},
	{0x00, 0x52, 0x22, 0xaa},
	{0x00, 0x53, 0x5e, 0xaa},
	{0x00, 0x54, 0x80, 0xaa},
	{0x00, 0x58, 0x9e, 0xaa},
	{0x00, 0x41, 0x08, 0xaa},
	{0x00, 0x3f, 0x00, 0xaa},
	{0x00, 0x75, 0x85, 0xaa},
	{0x00, 0x76, 0xe1, 0xaa},
	{0x00, 0x4c, 0x00, 0xaa},
	{0x00, 0x77, 0x0a, 0xaa},
	{0x00, 0x3d, 0x88, 0xaa},
	{0x00, 0x4b, 0x09, 0xaa},
	{0x00, 0xc9, 0x60, 0xaa},
	{0x00, 0x41, 0x38, 0xaa},
	{0x00, 0x62, 0x30, 0xaa},
	{0x00, 0x63, 0x30, 0xaa},
	{0x00, 0x64, 0x08, 0xaa},
	{0x00, 0x94, 0x07, 0xaa},
	{0x00, 0x95, 0x0b, 0xaa},
	{0x00, 0x65, 0x00, 0xaa},
	{0x00, 0x66, 0x05, 0xaa},
	{0x00, 0x56, 0x50, 0xaa},
	{0x00, 0x34, 0x11, 0xaa},
	{0x00, 0xa4, 0x88, 0xaa},
	{0x00, 0x96, 0x00, 0xaa},
	{0x00, 0x97, 0x30, 0xaa},
	{0x00, 0x98, 0x20, 0xaa},
	{0x00, 0x99, 0x30, 0xaa},
	{0x00, 0x9a, 0x84, 0xaa},
	{0x00, 0x9b, 0x29, 0xaa},
	{0x00, 0x9c, 0x03, 0xaa},
	{0x00, 0x78, 0x04, 0xaa},
	{0x00, 0x79, 0x01, 0xaa},
	{0x00, 0xc8, 0xf0, 0xaa},
	{0x00, 0x79, 0x0f, 0xaa},
	{0x00, 0xc8, 0x00, 0xaa},
	{0x00, 0x79, 0x10, 0xaa},
	{0x00, 0xc8, 0x7e, 0xaa},
	{0x00, 0x79, 0x0a, 0xaa},
	{0x00, 0xc8, 0x80, 0xaa},
	{0x00, 0x79, 0x0b, 0xaa},
	{0x00, 0xc8, 0x01, 0xaa},
	{0x00, 0x79, 0x0c, 0xaa},
	{0x00, 0xc8, 0x0f, 0xaa},
	{0x00, 0x79, 0x0d, 0xaa},
	{0x00, 0xc8, 0x20, 0xaa},
	{0x00, 0x79, 0x09, 0xaa},
	{0x00, 0xc8, 0x80, 0xaa},
	{0x00, 0x79, 0x02, 0xaa},
	{0x00, 0xc8, 0xc0, 0xaa},
	{0x00, 0x79, 0x03, 0xaa},
	{0x00, 0xc8, 0x40, 0xaa},
	{0x00, 0x79, 0x05, 0xaa},
	{0x00, 0xc8, 0x30, 0xaa},
	{0x00, 0x79, 0x26, 0xaa},
	{0x00, 0x11, 0x40, 0xaa},
	{0x00, 0x3a, 0x04, 0xaa},
	{0x00, 0x12, 0x00, 0xaa},
	{0x00, 0x40, 0xc0, 0xaa},
	{0x00, 0x8c, 0x00, 0xaa},
	{0x00, 0x17, 0x14, 0xaa},
	{0x00, 0x18, 0x02, 0xaa},
	{0x00, 0x32, 0x92, 0xaa},
	{0x00, 0x19, 0x02, 0xaa},
	{0x00, 0x1a, 0x7a, 0xaa},
	{0x00, 0x03, 0x0a, 0xaa},
	{0x00, 0x0c, 0x00, 0xaa},
	{0x00, 0x3e, 0x00, 0xaa},
	{0x00, 0x70, 0x3a, 0xaa},
	{0x00, 0x71, 0x35, 0xaa},
	{0x00, 0x72, 0x11, 0xaa},
	{0x00, 0x73, 0xf0, 0xaa},
	{0x00, 0xa2, 0x02, 0xaa},
	{0x00, 0xb1, 0x00, 0xaa},
	{0x00, 0xb1, 0x0c, 0xaa},
	{0x00, 0x1e, 0x37, 0xaa},	/* MVFP */
	{0x00, 0xaa, 0x14, 0xaa},
	{0x00, 0x24, 0x80, 0xaa},
	{0x00, 0x25, 0x74, 0xaa},
	{0x00, 0x26, 0xd3, 0xaa},
	{0x00, 0x0d, 0x00, 0xaa},
	{0x00, 0x14, 0x18, 0xaa},
	{0x00, 0x9d, 0x99, 0xaa},
	{0x00, 0x9e, 0x7f, 0xaa},
	{0x00, 0x64, 0x08, 0xaa},
	{0x00, 0x94, 0x07, 0xaa},
	{0x00, 0x95, 0x06, 0xaa},
	{0x00, 0x66, 0x05, 0xaa},
	{0x00, 0x41, 0x08, 0xaa},
	{0x00, 0x3f, 0x00, 0xaa},
	{0x00, 0x75, 0x07, 0xaa},
	{0x00, 0x76, 0xe1, 0xaa},
	{0x00, 0x4c, 0x00, 0xaa},
	{0x00, 0x77, 0x00, 0xaa},
	{0x00, 0x3d, 0xc2, 0xaa},
	{0x00, 0x4b, 0x09, 0xaa},
	{0x00, 0xc9, 0x60, 0xaa},
	{0x00, 0x41, 0x38, 0xaa},
	{0xbf, 0xc0, 0x26, 0xcc},
	{0xbf, 0xc1, 0x02, 0xcc},
	{0xbf, 0xcc, 0x04, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xb3, 0x01, 0x45, 0xcc},
	{0x00, 0x77, 0x05, 0xaa},
	{},
};

static const u8 ov7670_InitQVGA[][4] = {
	{0xb3, 0x01, 0x05, 0xcc},
	{0x00, 0x00, 0x30, 0xdd},
	{0xb0, 0x03, 0x19, 0xcc},
	{0x00, 0x00, 0x10, 0xdd},
	{0xb0, 0x04, 0x02, 0xcc},
	{0x00, 0x00, 0x10, 0xdd},
	{0xb3, 0x00, 0x66, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb0, 0x16, 0x01, 0xcc},
	{0xb3, 0x35, 0xa1, 0xcc},	/* i2c add: 21 */
	{0xb3, 0x34, 0x01, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x02, 0x02, 0xcc},
	{0xb3, 0x03, 0x1f, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},
	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x04, 0x05, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x01, 0xcc},
	{0xb3, 0x23, 0xe0, 0xcc},
	{0xbc, 0x00, 0xd1, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},
	{0x00, 0x12, 0x80, 0xaa},
	{0x00, 0x00, 0x20, 0xdd},
	{0x00, 0x12, 0x00, 0xaa},
	{0x00, 0x11, 0x40, 0xaa},
	{0x00, 0x6b, 0x0a, 0xaa},
	{0x00, 0x3a, 0x04, 0xaa},
	{0x00, 0x40, 0xc0, 0xaa},
	{0x00, 0x8c, 0x00, 0xaa},
	{0x00, 0x7a, 0x29, 0xaa},
	{0x00, 0x7b, 0x0e, 0xaa},
	{0x00, 0x7c, 0x1a, 0xaa},
	{0x00, 0x7d, 0x31, 0xaa},
	{0x00, 0x7e, 0x53, 0xaa},
	{0x00, 0x7f, 0x60, 0xaa},
	{0x00, 0x80, 0x6b, 0xaa},
	{0x00, 0x81, 0x73, 0xaa},
	{0x00, 0x82, 0x7b, 0xaa},
	{0x00, 0x83, 0x82, 0xaa},
	{0x00, 0x84, 0x89, 0xaa},
	{0x00, 0x85, 0x96, 0xaa},
	{0x00, 0x86, 0xa1, 0xaa},
	{0x00, 0x87, 0xb7, 0xaa},
	{0x00, 0x88, 0xcc, 0xaa},
	{0x00, 0x89, 0xe1, 0xaa},
	{0x00, 0x13, 0xe0, 0xaa},
	{0x00, 0x00, 0x00, 0xaa},
	{0x00, 0x10, 0x00, 0xaa},
	{0x00, 0x0d, 0x40, 0xaa},
	{0x00, 0x14, 0x28, 0xaa},
	{0x00, 0xa5, 0x05, 0xaa},
	{0x00, 0xab, 0x07, 0xaa},
	{0x00, 0x24, 0x95, 0xaa},
	{0x00, 0x25, 0x33, 0xaa},
	{0x00, 0x26, 0xe3, 0xaa},
	{0x00, 0x9f, 0x88, 0xaa},
	{0x00, 0xa0, 0x78, 0xaa},
	{0x00, 0x55, 0x90, 0xaa},
	{0x00, 0xa1, 0x03, 0xaa},
	{0x00, 0xa6, 0xe0, 0xaa},
	{0x00, 0xa7, 0xd8, 0xaa},
	{0x00, 0xa8, 0xf0, 0xaa},
	{0x00, 0xa9, 0x90, 0xaa},
	{0x00, 0xaa, 0x14, 0xaa},
	{0x00, 0x13, 0xe5, 0xaa},
	{0x00, 0x0e, 0x61, 0xaa},
	{0x00, 0x0f, 0x4b, 0xaa},
	{0x00, 0x16, 0x02, 0xaa},
	{0x00, 0x1e, 0x07, 0xaa},	/* MVFP */
	{0x00, 0x21, 0x02, 0xaa},
	{0x00, 0x22, 0x91, 0xaa},
	{0x00, 0x29, 0x07, 0xaa},
	{0x00, 0x33, 0x0b, 0xaa},
	{0x00, 0x35, 0x0b, 0xaa},
	{0x00, 0x37, 0x1d, 0xaa},
	{0x00, 0x38, 0x71, 0xaa},
	{0x00, 0x39, 0x2a, 0xaa},
	{0x00, 0x3c, 0x78, 0xaa},
	{0x00, 0x4d, 0x40, 0xaa},
	{0x00, 0x4e, 0x20, 0xaa},
	{0x00, 0x74, 0x19, 0xaa},
	{0x00, 0x8d, 0x4f, 0xaa},
	{0x00, 0x8e, 0x00, 0xaa},
	{0x00, 0x8f, 0x00, 0xaa},
	{0x00, 0x90, 0x00, 0xaa},
	{0x00, 0x91, 0x00, 0xaa},
	{0x00, 0x96, 0x00, 0xaa},
	{0x00, 0x9a, 0x80, 0xaa},
	{0x00, 0xb0, 0x84, 0xaa},
	{0x00, 0xb1, 0x0c, 0xaa},
	{0x00, 0xb2, 0x0e, 0xaa},
	{0x00, 0xb3, 0x82, 0xaa},
	{0x00, 0xb8, 0x0a, 0xaa},
	{0x00, 0x43, 0x14, 0xaa},
	{0x00, 0x44, 0xf0, 0xaa},
	{0x00, 0x45, 0x45, 0xaa},
	{0x00, 0x46, 0x63, 0xaa},
	{0x00, 0x47, 0x2d, 0xaa},
	{0x00, 0x48, 0x46, 0xaa},
	{0x00, 0x59, 0x88, 0xaa},
	{0x00, 0x5a, 0xa0, 0xaa},
	{0x00, 0x5b, 0xc6, 0xaa},
	{0x00, 0x5c, 0x7d, 0xaa},
	{0x00, 0x5d, 0x5f, 0xaa},
	{0x00, 0x5e, 0x19, 0xaa},
	{0x00, 0x6c, 0x0a, 0xaa},
	{0x00, 0x6d, 0x55, 0xaa},
	{0x00, 0x6e, 0x11, 0xaa},
	{0x00, 0x6f, 0x9e, 0xaa},
	{0x00, 0x69, 0x00, 0xaa},
	{0x00, 0x6a, 0x40, 0xaa},
	{0x00, 0x01, 0x40, 0xaa},
	{0x00, 0x02, 0x40, 0xaa},
	{0x00, 0x13, 0xe7, 0xaa},
	{0x00, 0x5f, 0xf0, 0xaa},
	{0x00, 0x60, 0xf0, 0xaa},
	{0x00, 0x61, 0xf0, 0xaa},
	{0x00, 0x27, 0xa0, 0xaa},
	{0x00, 0x28, 0x80, 0xaa},
	{0x00, 0x2c, 0x90, 0xaa},
	{0x00, 0x4f, 0x66, 0xaa},
	{0x00, 0x50, 0x66, 0xaa},
	{0x00, 0x51, 0x00, 0xaa},
	{0x00, 0x52, 0x22, 0xaa},
	{0x00, 0x53, 0x5e, 0xaa},
	{0x00, 0x54, 0x80, 0xaa},
	{0x00, 0x58, 0x9e, 0xaa},
	{0x00, 0x41, 0x08, 0xaa},
	{0x00, 0x3f, 0x00, 0xaa},
	{0x00, 0x75, 0x85, 0xaa},
	{0x00, 0x76, 0xe1, 0xaa},
	{0x00, 0x4c, 0x00, 0xaa},
	{0x00, 0x77, 0x0a, 0xaa},
	{0x00, 0x3d, 0x88, 0xaa},
	{0x00, 0x4b, 0x09, 0xaa},
	{0x00, 0xc9, 0x60, 0xaa},
	{0x00, 0x41, 0x38, 0xaa},
	{0x00, 0x62, 0x30, 0xaa},
	{0x00, 0x63, 0x30, 0xaa},
	{0x00, 0x64, 0x08, 0xaa},
	{0x00, 0x94, 0x07, 0xaa},
	{0x00, 0x95, 0x0b, 0xaa},
	{0x00, 0x65, 0x00, 0xaa},
	{0x00, 0x66, 0x05, 0xaa},
	{0x00, 0x56, 0x50, 0xaa},
	{0x00, 0x34, 0x11, 0xaa},
	{0x00, 0xa4, 0x88, 0xaa},
	{0x00, 0x96, 0x00, 0xaa},
	{0x00, 0x97, 0x30, 0xaa},
	{0x00, 0x98, 0x20, 0xaa},
	{0x00, 0x99, 0x30, 0xaa},
	{0x00, 0x9a, 0x84, 0xaa},
	{0x00, 0x9b, 0x29, 0xaa},
	{0x00, 0x9c, 0x03, 0xaa},
	{0x00, 0x78, 0x04, 0xaa},
	{0x00, 0x79, 0x01, 0xaa},
	{0x00, 0xc8, 0xf0, 0xaa},
	{0x00, 0x79, 0x0f, 0xaa},
	{0x00, 0xc8, 0x00, 0xaa},
	{0x00, 0x79, 0x10, 0xaa},
	{0x00, 0xc8, 0x7e, 0xaa},
	{0x00, 0x79, 0x0a, 0xaa},
	{0x00, 0xc8, 0x80, 0xaa},
	{0x00, 0x79, 0x0b, 0xaa},
	{0x00, 0xc8, 0x01, 0xaa},
	{0x00, 0x79, 0x0c, 0xaa},
	{0x00, 0xc8, 0x0f, 0xaa},
	{0x00, 0x79, 0x0d, 0xaa},
	{0x00, 0xc8, 0x20, 0xaa},
	{0x00, 0x79, 0x09, 0xaa},
	{0x00, 0xc8, 0x80, 0xaa},
	{0x00, 0x79, 0x02, 0xaa},
	{0x00, 0xc8, 0xc0, 0xaa},
	{0x00, 0x79, 0x03, 0xaa},
	{0x00, 0xc8, 0x40, 0xaa},
	{0x00, 0x79, 0x05, 0xaa},
	{0x00, 0xc8, 0x30, 0xaa},
	{0x00, 0x79, 0x26, 0xaa},
	{0x00, 0x11, 0x40, 0xaa},
	{0x00, 0x3a, 0x04, 0xaa},
	{0x00, 0x12, 0x00, 0xaa},
	{0x00, 0x40, 0xc0, 0xaa},
	{0x00, 0x8c, 0x00, 0xaa},
	{0x00, 0x17, 0x14, 0xaa},
	{0x00, 0x18, 0x02, 0xaa},
	{0x00, 0x32, 0x92, 0xaa},
	{0x00, 0x19, 0x02, 0xaa},
	{0x00, 0x1a, 0x7a, 0xaa},
	{0x00, 0x03, 0x0a, 0xaa},
	{0x00, 0x0c, 0x00, 0xaa},
	{0x00, 0x3e, 0x00, 0xaa},
	{0x00, 0x70, 0x3a, 0xaa},
	{0x00, 0x71, 0x35, 0xaa},
	{0x00, 0x72, 0x11, 0xaa},
	{0x00, 0x73, 0xf0, 0xaa},
	{0x00, 0xa2, 0x02, 0xaa},
	{0x00, 0xb1, 0x00, 0xaa},
	{0x00, 0xb1, 0x0c, 0xaa},
	{0x00, 0x1e, 0x37, 0xaa},	/* MVFP */
	{0x00, 0xaa, 0x14, 0xaa},
	{0x00, 0x24, 0x80, 0xaa},
	{0x00, 0x25, 0x74, 0xaa},
	{0x00, 0x26, 0xd3, 0xaa},
	{0x00, 0x0d, 0x00, 0xaa},
	{0x00, 0x14, 0x18, 0xaa},
	{0x00, 0x9d, 0x99, 0xaa},
	{0x00, 0x9e, 0x7f, 0xaa},
	{0x00, 0x64, 0x08, 0xaa},
	{0x00, 0x94, 0x07, 0xaa},
	{0x00, 0x95, 0x06, 0xaa},
	{0x00, 0x66, 0x05, 0xaa},
	{0x00, 0x41, 0x08, 0xaa},
	{0x00, 0x3f, 0x00, 0xaa},
	{0x00, 0x75, 0x07, 0xaa},
	{0x00, 0x76, 0xe1, 0xaa},
	{0x00, 0x4c, 0x00, 0xaa},
	{0x00, 0x77, 0x00, 0xaa},
	{0x00, 0x3d, 0xc2, 0xaa},
	{0x00, 0x4b, 0x09, 0xaa},
	{0x00, 0xc9, 0x60, 0xaa},
	{0x00, 0x41, 0x38, 0xaa},
	{0xbc, 0x02, 0x18, 0xcc},
	{0xbc, 0x03, 0x50, 0xcc},
	{0xbc, 0x04, 0x18, 0xcc},
	{0xbc, 0x05, 0x00, 0xcc},
	{0xbc, 0x06, 0x00, 0xcc},
	{0xbc, 0x08, 0x30, 0xcc},
	{0xbc, 0x09, 0x40, 0xcc},
	{0xbc, 0x0a, 0x10, 0xcc},
	{0xbc, 0x0b, 0x00, 0xcc},
	{0xbc, 0x0c, 0x00, 0xcc},
	{0xbf, 0xc0, 0x26, 0xcc},
	{0xbf, 0xc1, 0x02, 0xcc},
	{0xbf, 0xcc, 0x04, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xb3, 0x01, 0x45, 0xcc},
	{0x00, 0x77, 0x05, 0xaa},
	{},
};

/* PO1200 - values from usbvm326.inf and ms-win trace */
static const u8 po1200_gamma[17] = {
	0x00, 0x13, 0x38, 0x59, 0x79, 0x92, 0xa7, 0xb9, 0xc8,
	0xd4, 0xdf, 0xe7, 0xee, 0xf4, 0xf9, 0xfc, 0xff
};
static const u8 po1200_matrix[9] = {
	0x60, 0xf9, 0xe5, 0xe7, 0x50, 0x05, 0xf3, 0xe6, 0x5e
};
static const u8 po1200_initVGA_data[][4] = {
	{0xb0, 0x03, 0x19, 0xcc},	/* reset? */
	{0xb0, 0x03, 0x19, 0xcc},
/*	{0x00, 0x00, 0x33, 0xdd}, */
	{0xb0, 0x04, 0x02, 0xcc},
	{0xb0, 0x02, 0x02, 0xcc},
	{0xb3, 0x5d, 0x00, 0xcc},
	{0xb3, 0x01, 0x01, 0xcc},
	{0xb3, 0x00, 0x64, 0xcc},
	{0xb3, 0x00, 0x65, 0xcc},
	{0xb3, 0x05, 0x01, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb3, 0x02, 0xb2, 0xcc},
	{0xb3, 0x03, 0x18, 0xcc},
	{0xb3, 0x04, 0x15, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x02, 0xcc},
	{0xb3, 0x23, 0x58, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x03, 0xcc},
	{0xb3, 0x17, 0x1f, 0xcc},
	{0xbc, 0x00, 0x71, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},
	{0xb0, 0x54, 0x13, 0xcc},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb3, 0x34, 0x01, 0xcc},
	{0xb3, 0x35, 0xdc, 0xcc},	/* i2c add: 5c */
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0x12, 0x05, 0xaa},
	{0x00, 0x13, 0x02, 0xaa},
	{0x00, 0x1e, 0xc6, 0xaa},	/* h/v flip */
	{0x00, 0x21, 0x00, 0xaa},
	{0x00, 0x25, 0x02, 0xaa},
	{0x00, 0x3c, 0x4f, 0xaa},
	{0x00, 0x3f, 0xe0, 0xaa},
	{0x00, 0x42, 0xff, 0xaa},
	{0x00, 0x45, 0x34, 0xaa},
	{0x00, 0x55, 0xfe, 0xaa},
	{0x00, 0x59, 0xd3, 0xaa},
	{0x00, 0x5e, 0x04, 0xaa},
	{0x00, 0x61, 0xb8, 0xaa},	/* sharpness */
	{0x00, 0x62, 0x02, 0xaa},
	{0x00, 0xa7, 0x31, 0xaa},
	{0x00, 0xa9, 0x66, 0xaa},
	{0x00, 0xb0, 0x00, 0xaa},
	{0x00, 0xb1, 0x00, 0xaa},
	{0x00, 0xb3, 0x11, 0xaa},
	{0x00, 0xb6, 0x26, 0xaa},
	{0x00, 0xb7, 0x20, 0xaa},
	{0x00, 0xba, 0x04, 0xaa},
	{0x00, 0x88, 0x42, 0xaa},
	{0x00, 0x89, 0x9a, 0xaa},
	{0x00, 0x8a, 0x88, 0xaa},
	{0x00, 0x8b, 0x8e, 0xaa},
	{0x00, 0x8c, 0x3e, 0xaa},
	{0x00, 0x8d, 0x90, 0xaa},
	{0x00, 0x8e, 0x87, 0xaa},
	{0x00, 0x8f, 0x96, 0xaa},
	{0x00, 0x90, 0x3d, 0xaa},
	{0x00, 0x64, 0x00, 0xaa},
	{0x00, 0x65, 0x10, 0xaa},
	{0x00, 0x66, 0x20, 0xaa},
	{0x00, 0x67, 0x2b, 0xaa},
	{0x00, 0x68, 0x36, 0xaa},
	{0x00, 0x69, 0x49, 0xaa},
	{0x00, 0x6a, 0x5a, 0xaa},
	{0x00, 0x6b, 0x7f, 0xaa},
	{0x00, 0x6c, 0x9b, 0xaa},
	{0x00, 0x6d, 0xba, 0xaa},
	{0x00, 0x6e, 0xd4, 0xaa},
	{0x00, 0x6f, 0xea, 0xaa},
	{0x00, 0x70, 0x00, 0xaa},
	{0x00, 0x71, 0x10, 0xaa},
	{0x00, 0x72, 0x20, 0xaa},
	{0x00, 0x73, 0x2b, 0xaa},
	{0x00, 0x74, 0x36, 0xaa},
	{0x00, 0x75, 0x49, 0xaa},
	{0x00, 0x76, 0x5a, 0xaa},
	{0x00, 0x77, 0x7f, 0xaa},
	{0x00, 0x78, 0x9b, 0xaa},
	{0x00, 0x79, 0xba, 0xaa},
	{0x00, 0x7a, 0xd4, 0xaa},
	{0x00, 0x7b, 0xea, 0xaa},
	{0x00, 0x7c, 0x00, 0xaa},
	{0x00, 0x7d, 0x10, 0xaa},
	{0x00, 0x7e, 0x20, 0xaa},
	{0x00, 0x7f, 0x2b, 0xaa},
	{0x00, 0x80, 0x36, 0xaa},
	{0x00, 0x81, 0x49, 0xaa},
	{0x00, 0x82, 0x5a, 0xaa},
	{0x00, 0x83, 0x7f, 0xaa},
	{0x00, 0x84, 0x9b, 0xaa},
	{0x00, 0x85, 0xba, 0xaa},
	{0x00, 0x86, 0xd4, 0xaa},
	{0x00, 0x87, 0xea, 0xaa},
	{0x00, 0x57, 0x2a, 0xaa},
	{0x00, 0x03, 0x01, 0xaa},
	{0x00, 0x04, 0x10, 0xaa},
	{0x00, 0x05, 0x10, 0xaa},
	{0x00, 0x06, 0x10, 0xaa},
	{0x00, 0x07, 0x10, 0xaa},
	{0x00, 0x08, 0x13, 0xaa},
	{0x00, 0x0a, 0x00, 0xaa},
	{0x00, 0x0b, 0x10, 0xaa},
	{0x00, 0x0c, 0x20, 0xaa},
	{0x00, 0x0d, 0x18, 0xaa},
	{0x00, 0x22, 0x01, 0xaa},
	{0x00, 0x23, 0x60, 0xaa},
	{0x00, 0x25, 0x08, 0xaa},
	{0x00, 0x26, 0x82, 0xaa},
	{0x00, 0x2e, 0x0f, 0xaa},
	{0x00, 0x2f, 0x1e, 0xaa},
	{0x00, 0x30, 0x2d, 0xaa},
	{0x00, 0x31, 0x3c, 0xaa},
	{0x00, 0x32, 0x4b, 0xaa},
	{0x00, 0x33, 0x5a, 0xaa},
	{0x00, 0x34, 0x69, 0xaa},
	{0x00, 0x35, 0x78, 0xaa},
	{0x00, 0x36, 0x87, 0xaa},
	{0x00, 0x37, 0x96, 0xaa},
	{0x00, 0x38, 0xa5, 0xaa},
	{0x00, 0x39, 0xb4, 0xaa},
	{0x00, 0x3a, 0xc3, 0xaa},
	{0x00, 0x3b, 0xd2, 0xaa},
	{0x00, 0x3c, 0xe1, 0xaa},
	{0x00, 0x3e, 0xff, 0xaa},
	{0x00, 0x3f, 0xff, 0xaa},
	{0x00, 0x40, 0xff, 0xaa},
	{0x00, 0x41, 0xff, 0xaa},
	{0x00, 0x42, 0xff, 0xaa},
	{0x00, 0x43, 0xff, 0xaa},
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0x20, 0xc4, 0xaa},
	{0x00, 0x13, 0x03, 0xaa},
	{0x00, 0x3c, 0x50, 0xaa},
	{0x00, 0x61, 0x6a, 0xaa},	/* sharpness? */
	{0x00, 0x51, 0x5b, 0xaa},
	{0x00, 0x52, 0x91, 0xaa},
	{0x00, 0x53, 0x4c, 0xaa},
	{0x00, 0x54, 0x50, 0xaa},
	{0x00, 0x56, 0x02, 0xaa},
	{0xb6, 0x00, 0x00, 0xcc},
	{0xb6, 0x03, 0x03, 0xcc},
	{0xb6, 0x02, 0x20, 0xcc},
	{0xb6, 0x05, 0x02, 0xcc},
	{0xb6, 0x04, 0x58, 0xcc},
	{0xb6, 0x12, 0xf8, 0xcc},
	{0xb6, 0x13, 0x21, 0xcc},
	{0xb6, 0x18, 0x03, 0xcc},
	{0xb6, 0x17, 0xa9, 0xcc},
	{0xb6, 0x16, 0x80, 0xcc},
	{0xb6, 0x22, 0x12, 0xcc},
	{0xb6, 0x23, 0x0b, 0xcc},
	{0xbf, 0xc0, 0x39, 0xcc},
	{0xbf, 0xc1, 0x04, 0xcc},
	{0xbf, 0xcc, 0x00, 0xcc},
	{0xb8, 0x06, 0x20, 0xcc},
	{0xb8, 0x07, 0x03, 0xcc},
	{0xb8, 0x08, 0x58, 0xcc},
	{0xb8, 0x09, 0x02, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0xd9, 0x0f, 0xaa},
	{0x00, 0xda, 0xaa, 0xaa},
	{0x00, 0xd9, 0x10, 0xaa},
	{0x00, 0xda, 0xaa, 0xaa},
	{0x00, 0xd9, 0x11, 0xaa},
	{0x00, 0xda, 0x00, 0xaa},
	{0x00, 0xd9, 0x12, 0xaa},
	{0x00, 0xda, 0xff, 0xaa},
	{0x00, 0xd9, 0x13, 0xaa},
	{0x00, 0xda, 0xff, 0xaa},
	{0x00, 0xe8, 0x11, 0xaa},
	{0x00, 0xe9, 0x12, 0xaa},
	{0x00, 0xea, 0x5c, 0xaa},
	{0x00, 0xeb, 0xff, 0xaa},
	{0x00, 0xd8, 0x80, 0xaa},
	{0x00, 0xe6, 0x02, 0xaa},
	{0x00, 0xd6, 0x40, 0xaa},
	{0x00, 0xe3, 0x05, 0xaa},
	{0x00, 0xe0, 0x40, 0xaa},
	{0x00, 0xde, 0x03, 0xaa},
	{0x00, 0xdf, 0x03, 0xaa},
	{0x00, 0xdb, 0x02, 0xaa},
	{0x00, 0xdc, 0x00, 0xaa},
	{0x00, 0xdd, 0x03, 0xaa},
	{0x00, 0xe1, 0x08, 0xaa},
	{0x00, 0xe2, 0x01, 0xaa},
	{0x00, 0xd6, 0x40, 0xaa},
	{0x00, 0xe4, 0x40, 0xaa},
	{0x00, 0xa8, 0x8f, 0xaa},
	{0x00, 0xb4, 0x16, 0xaa},
	{0xb0, 0x02, 0x06, 0xcc},
	{0xb0, 0x18, 0x06, 0xcc},
	{0xb0, 0x19, 0x06, 0xcc},
	{0xb3, 0x5d, 0x18, 0xcc},
	{0xb3, 0x05, 0x00, 0xcc},
	{0xb3, 0x06, 0x00, 0xcc},
	{0x00, 0xb4, 0x0e, 0xaa},
	{0x00, 0xb5, 0x49, 0xaa},
	{0x00, 0xb6, 0x1c, 0xaa},
	{0x00, 0xb7, 0x96, 0xaa},
/* end of usbvm326.inf - start of ms-win trace */
	{0xb6, 0x12, 0xf8, 0xcc},
	{0xb6, 0x13, 0x3d, 0xcc},
/*read b306*/
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0x1a, 0x09, 0xaa},
	{0x00, 0x1b, 0x8a, 0xaa},
/*read b827*/
	{0xb8, 0x27, 0x00, 0xcc},
	{0xb8, 0x26, 0x60, 0xcc},
	{0xb8, 0x26, 0x60, 0xcc},
/*gamma - to do?*/
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0xae, 0x84, 0xaa},
/*gamma again*/
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0x96, 0xa0, 0xaa},
/*matrix*/
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0x91, 0x35, 0xaa},
	{0x00, 0x92, 0x22, 0xaa},
/*gamma*/
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0x95, 0x85, 0xaa},
/*matrix*/
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0x4d, 0x20, 0xaa},
	{0xb8, 0x22, 0x40, 0xcc},
	{0xb8, 0x23, 0x40, 0xcc},
	{0xb8, 0x24, 0x40, 0xcc},
	{0xb8, 0x81, 0x09, 0xcc},
	{0x00, 0x00, 0x64, 0xdd},
	{0x00, 0x03, 0x01, 0xaa},
/*read 46*/
	{0x00, 0x46, 0x3c, 0xaa},
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0x16, 0x40, 0xaa},
	{0x00, 0x17, 0x40, 0xaa},
	{0x00, 0x18, 0x40, 0xaa},
	{0x00, 0x19, 0x41, 0xaa},
	{0x00, 0x03, 0x01, 0xaa},
	{0x00, 0x46, 0x3c, 0xaa},
	{0x00, 0x00, 0x18, 0xdd},
/*read bfff*/
	{0x00, 0x03, 0x00, 0xaa},
	{0x00, 0xb4, 0x1c, 0xaa},
	{0x00, 0xb5, 0x92, 0xaa},
	{0x00, 0xb6, 0x39, 0xaa},
	{0x00, 0xb7, 0x24, 0xaa},
/*write 89 0400 1415*/
	{}
};

static const u8 poxxxx_init_common[][4] = {
	{0xb3, 0x00, 0x04, 0xcc},
	{0x00, 0x00, 0x10, 0xdd},
	{0xb3, 0x00, 0x64, 0xcc},
	{0x00, 0x00, 0x10, 0xdd},
	{0xb3, 0x00, 0x65, 0xcc},
	{0x00, 0x00, 0x10, 0xdd},
	{0xb3, 0x00, 0x67, 0xcc},
	{0xb0, 0x03, 0x09, 0xcc},
	{0xb3, 0x05, 0x00, 0xcc},
	{0xb3, 0x06, 0x00, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0xb3, 0x08, 0x01, 0xcc},
	{0xb3, 0x09, 0x0c, 0xcc},
	{0xb3, 0x34, 0x01, 0xcc},
	{0xb3, 0x35, 0xf6, 0xcc},	/* i2c add: 76 */
	{0xb3, 0x02, 0xb0, 0xcc},
	{0xb3, 0x03, 0x18, 0xcc},
	{0xb3, 0x04, 0x15, 0xcc},
	{0xb3, 0x20, 0x00, 0xcc},
	{0xb3, 0x21, 0x00, 0xcc},
	{0xb3, 0x22, 0x04, 0xcc},	/* sensor height = 1024 */
	{0xb3, 0x23, 0x00, 0xcc},
	{0xb3, 0x14, 0x00, 0xcc},
	{0xb3, 0x15, 0x00, 0xcc},
	{0xb3, 0x16, 0x04, 0xcc},	/* sensor width = 1280 */
	{0xb3, 0x17, 0xff, 0xcc},
	{0xb3, 0x2c, 0x03, 0xcc},
	{0xb3, 0x2d, 0x56, 0xcc},
	{0xb3, 0x2e, 0x02, 0xcc},
	{0xb3, 0x2f, 0x0a, 0xcc},
	{0xb3, 0x40, 0x00, 0xcc},
	{0xb3, 0x41, 0x34, 0xcc},
	{0xb3, 0x42, 0x01, 0xcc},
	{0xb3, 0x43, 0xe0, 0xcc},
	{0xbc, 0x00, 0x71, 0xcc},
	{0xbc, 0x01, 0x01, 0xcc},
	{0xb3, 0x01, 0x41, 0xcc},
	{0xb3, 0x4d, 0x00, 0xcc},
	{0x00, 0x0b, 0x2a, 0xaa},
	{0x00, 0x0e, 0x03, 0xaa},
	{0x00, 0x0f, 0xea, 0xaa},
	{0x00, 0x12, 0x08, 0xaa},
	{0x00, 0x1e, 0x06, 0xaa},
	{0x00, 0x21, 0x00, 0xaa},
	{0x00, 0x31, 0x1f, 0xaa},
	{0x00, 0x33, 0x38, 0xaa},
	{0x00, 0x36, 0xc0, 0xaa},
	{0x00, 0x37, 0xc8, 0xaa},
	{0x00, 0x3b, 0x36, 0xaa},
	{0x00, 0x4b, 0xfe, 0xaa},
	{0x00, 0x4d, 0x2e, 0xaa},
	{0x00, 0x51, 0x1c, 0xaa},
	{0x00, 0x52, 0x01, 0xaa},
	{0x00, 0x55, 0x0a, 0xaa},
	{0x00, 0x56, 0x0a, 0xaa},
	{0x00, 0x57, 0x07, 0xaa},
	{0x00, 0x58, 0x07, 0xaa},
	{0x00, 0x59, 0x04, 0xaa},
	{0x00, 0x70, 0x68, 0xaa},
	{0x00, 0x71, 0x04, 0xaa},
	{0x00, 0x72, 0x10, 0xaa},
	{0x00, 0x80, 0x71, 0xaa},
	{0x00, 0x81, 0x08, 0xaa},
	{0x00, 0x82, 0x00, 0xaa},
	{0x00, 0x83, 0x55, 0xaa},
	{0x00, 0x84, 0x06, 0xaa},
	{0x00, 0x85, 0x06, 0xaa},
	{0x00, 0x8b, 0x25, 0xaa},
	{0x00, 0x8c, 0x00, 0xaa},
	{0x00, 0x8d, 0x86, 0xaa},
	{0x00, 0x8e, 0x82, 0xaa},
	{0x00, 0x8f, 0x2d, 0xaa},
	{0x00, 0x90, 0x8b, 0xaa},
	{0x00, 0x91, 0x81, 0xaa},
	{0x00, 0x92, 0x81, 0xaa},
	{0x00, 0x93, 0x23, 0xaa},
	{0x00, 0xa3, 0x2a, 0xaa},
	{0x00, 0xa4, 0x03, 0xaa},
	{0x00, 0xa5, 0xea, 0xaa},
	{0x00, 0xb0, 0x68, 0xaa},
	{0x00, 0xbc, 0x04, 0xaa},
	{0x00, 0xbe, 0x3b, 0xaa},
	{0x00, 0x4e, 0x40, 0xaa},
	{0x00, 0x06, 0x04, 0xaa},
	{0x00, 0x07, 0x03, 0xaa},
	{0x00, 0xcd, 0x18, 0xaa},
	{0x00, 0x28, 0x03, 0xaa},
	{0x00, 0x29, 0xef, 0xaa},
/* reinit on alt 2 (qvga) or alt7 (vga) */
	{0xb3, 0x05, 0x00, 0xcc},
	{0xb3, 0x06, 0x00, 0xcc},
	{0xb8, 0x00, 0x01, 0xcc},

	{0x00, 0x1d, 0x85, 0xaa},
	{0x00, 0x1e, 0xc6, 0xaa},
	{0x00, 0x00, 0x40, 0xdd},
	{0x00, 0x1d, 0x05, 0xaa},
	{}
};
static const u8 poxxxx_gamma[][4] = {
	{0x00, 0xd6, 0x22, 0xaa},	/* gamma 0 */
	{0x00, 0x73, 0x00, 0xaa},
	{0x00, 0x74, 0x0a, 0xaa},
	{0x00, 0x75, 0x16, 0xaa},
	{0x00, 0x76, 0x25, 0xaa},
	{0x00, 0x77, 0x34, 0xaa},
	{0x00, 0x78, 0x49, 0xaa},
	{0x00, 0x79, 0x5a, 0xaa},
	{0x00, 0x7a, 0x7f, 0xaa},
	{0x00, 0x7b, 0x9b, 0xaa},
	{0x00, 0x7c, 0xba, 0xaa},
	{0x00, 0x7d, 0xd4, 0xaa},
	{0x00, 0x7e, 0xea, 0xaa},

	{0x00, 0xd6, 0x62, 0xaa},	/* gamma 1 */
	{0x00, 0x73, 0x00, 0xaa},
	{0x00, 0x74, 0x0a, 0xaa},
	{0x00, 0x75, 0x16, 0xaa},
	{0x00, 0x76, 0x25, 0xaa},
	{0x00, 0x77, 0x34, 0xaa},
	{0x00, 0x78, 0x49, 0xaa},
	{0x00, 0x79, 0x5a, 0xaa},
	{0x00, 0x7a, 0x7f, 0xaa},
	{0x00, 0x7b, 0x9b, 0xaa},
	{0x00, 0x7c, 0xba, 0xaa},
	{0x00, 0x7d, 0xd4, 0xaa},
	{0x00, 0x7e, 0xea, 0xaa},

	{0x00, 0xd6, 0xa2, 0xaa},	/* gamma 2 */
	{0x00, 0x73, 0x00, 0xaa},
	{0x00, 0x74, 0x0a, 0xaa},
	{0x00, 0x75, 0x16, 0xaa},
	{0x00, 0x76, 0x25, 0xaa},
	{0x00, 0x77, 0x34, 0xaa},
	{0x00, 0x78, 0x49, 0xaa},
	{0x00, 0x79, 0x5a, 0xaa},
	{0x00, 0x7a, 0x7f, 0xaa},
	{0x00, 0x7b, 0x9b, 0xaa},
	{0x00, 0x7c, 0xba, 0xaa},
	{0x00, 0x7d, 0xd4, 0xaa},
	{0x00, 0x7e, 0xea, 0xaa},
	{}
};
static const u8 poxxxx_init_start_3[][4] = {
	{0x00, 0xb8, 0x28, 0xaa},
	{0x00, 0xb9, 0x1e, 0xaa},
	{0x00, 0xb6, 0x14, 0xaa},
	{0x00, 0xb7, 0x0f, 0xaa},
	{0x00, 0x5c, 0x10, 0xaa},
	{0x00, 0x5d, 0x18, 0xaa},
	{0x00, 0x5e, 0x24, 0xaa},
	{0x00, 0x5f, 0x24, 0xaa},
	{0x00, 0x86, 0x1a, 0xaa},
	{0x00, 0x60, 0x00, 0xaa},
	{0x00, 0x61, 0x1b, 0xaa},
	{0x00, 0x62, 0x30, 0xaa},
	{0x00, 0x63, 0x40, 0xaa},
	{0x00, 0x87, 0x1a, 0xaa},
	{0x00, 0x64, 0x00, 0xaa},
	{0x00, 0x65, 0x08, 0xaa},
	{0x00, 0x66, 0x10, 0xaa},
	{0x00, 0x67, 0x20, 0xaa},
	{0x00, 0x88, 0x10, 0xaa},
	{0x00, 0x68, 0x00, 0xaa},
	{0x00, 0x69, 0x08, 0xaa},
	{0x00, 0x6a, 0x0f, 0xaa},
	{0x00, 0x6b, 0x0f, 0xaa},
	{0x00, 0x89, 0x07, 0xaa},
	{0x00, 0xd5, 0x4c, 0xaa},
	{0x00, 0x0a, 0x00, 0xaa},
	{0x00, 0x0b, 0x2a, 0xaa},
	{0x00, 0x0e, 0x03, 0xaa},
	{0x00, 0x0f, 0xea, 0xaa},
	{0x00, 0xa2, 0x00, 0xaa},
	{0x00, 0xa3, 0x2a, 0xaa},
	{0x00, 0xa4, 0x03, 0xaa},
	{0x00, 0xa5, 0xea, 0xaa},
	{}
};
static const u8 poxxxx_initVGA[][4] = {
	{0x00, 0x20, 0x11, 0xaa},
	{0x00, 0x33, 0x38, 0xaa},
	{0x00, 0xbb, 0x0d, 0xaa},
	{0xb3, 0x22, 0x01, 0xcc},	/* change to 640x480 */
	{0xb3, 0x23, 0xe0, 0xcc},
	{0xb3, 0x16, 0x02, 0xcc},
	{0xb3, 0x17, 0x7f, 0xcc},
	{0xb3, 0x02, 0xb0, 0xcc},
	{0xb3, 0x06, 0x00, 0xcc},
	{0xb3, 0x5c, 0x01, 0xcc},
	{0x00, 0x04, 0x06, 0xaa},
	{0x00, 0x05, 0x3f, 0xaa},
	{0x00, 0x04, 0x00, 0xdd},	/* delay 1s */
	{}
};
static const u8 poxxxx_initQVGA[][4] = {
	{0x00, 0x20, 0x33, 0xaa},
	{0x00, 0x33, 0x38, 0xaa},
	{0x00, 0xbb, 0x0d, 0xaa},
	{0xb3, 0x22, 0x00, 0xcc},	/* change to 320x240 */
	{0xb3, 0x23, 0xf0, 0xcc},
	{0xb3, 0x16, 0x01, 0xcc},
	{0xb3, 0x17, 0x3f, 0xcc},
	{0xb3, 0x02, 0xb0, 0xcc},
	{0xb3, 0x06, 0x01, 0xcc},
	{0xb3, 0x5c, 0x00, 0xcc},
	{0x00, 0x04, 0x06, 0xaa},
	{0x00, 0x05, 0x3f, 0xaa},
	{0x00, 0x04, 0x00, 0xdd},	/* delay 1s */
	{}
};
static const u8 poxxxx_init_end_1[][4] = {
	{0x00, 0x47, 0x25, 0xaa},
	{0x00, 0x48, 0x80, 0xaa},
	{0x00, 0x49, 0x1f, 0xaa},
	{0x00, 0x4a, 0x40, 0xaa},
	{0x00, 0x44, 0x40, 0xaa},
	{0x00, 0xab, 0x4a, 0xaa},
	{0x00, 0xb1, 0x00, 0xaa},
	{0x00, 0xb2, 0x04, 0xaa},
	{0x00, 0xb3, 0x08, 0xaa},
	{0x00, 0xb4, 0x0b, 0xaa},
	{0x00, 0xb5, 0x0d, 0xaa},
	{}
};
static const u8 poxxxx_init_end_2[][4] = {
	{0x00, 0x1d, 0x85, 0xaa},
	{0x00, 0x1e, 0x06, 0xaa},
	{0x00, 0x1d, 0x05, 0xaa},
	{}
};

struct sensor_info {
	s8 sensorId;
	u8 I2cAdd;
	u8 IdAdd;
	u16 VpId;
	u8 m1;
	u8 m2;
	u8 op;
};

/* probe values */
static const struct sensor_info vc0321_probe_data[] = {
/*      sensorId,	   I2cAdd,	IdAdd,  VpId,  m1,    m2,  op */
/* 0 OV9640 */
	{-1,		    0x80 | 0x30, 0x0a, 0x0000, 0x25, 0x24, 0x05},
/* 1 ICM108T (may respond on IdAdd == 0x83 - tested in vc032x_probe_sensor) */
	{-1,		    0x80 | 0x20, 0x82, 0x0000, 0x24, 0x25, 0x01},
/* 2 PO2130 (may detect PO3130NC - tested in vc032x_probe_sensor)*/
	{-1,		    0x80 | 0x76, 0x00, 0x0000, 0x24, 0x25, 0x01},
/* 3 MI1310 */
	{-1,		    0x80 | 0x5d, 0x00, 0x0000, 0x24, 0x25, 0x01},
/* 4 MI360 - tested in vc032x_probe_sensor */
/*	{SENSOR_MI0360,	    0x80 | 0x5d, 0x00, 0x8243, 0x24, 0x25, 0x01}, */
/* 5 7131R */
	{SENSOR_HV7131R,    0x80 | 0x11, 0x00, 0x0209, 0x24, 0x25, 0x01},
/* 6 OV7649 */
	{-1,		    0x80 | 0x21, 0x0a, 0x0000, 0x21, 0x20, 0x05},
/* 7 PAS302BCW */
	{-1,		    0x80 | 0x40, 0x00, 0x0000, 0x20, 0x22, 0x05},
/* 8 OV7660 */
	{SENSOR_OV7660,     0x80 | 0x21, 0x0a, 0x7660, 0x26, 0x26, 0x05},
/* 9 PO3130NC - (tested in vc032x_probe_sensor) */
/*	{SENSOR_PO3130NC,   0x80 | 0x76, 0x00, 0x3130, 0x24, 0x25, 0x01}, */
/* 10 PO1030KC */
	{-1,		    0x80 | 0x6e, 0x00, 0x0000, 0x24, 0x25, 0x01},
/* 11 MI1310_SOC */
	{SENSOR_MI1310_SOC, 0x80 | 0x5d, 0x00, 0x143a, 0x24, 0x25, 0x01},
/* 12 OV9650 */
	{-1,		    0x80 | 0x30, 0x0a, 0x0000, 0x25, 0x24, 0x05},
/* 13 S5K532 */
	{-1,		    0x80 | 0x11, 0x39, 0x0000, 0x24, 0x25, 0x01},
/* 14 MI360_SOC - ??? */
/* 15 PO1200N */
	{SENSOR_PO1200,     0x80 | 0x5c, 0x00, 0x1200, 0x67, 0x67, 0x01},
/* 16 PO3030K */
	{-1,		    0x80 | 0x18, 0x00, 0x0000, 0x24, 0x25, 0x01},
/* 17 PO2030 */
	{-1,		    0x80 | 0x6e, 0x00, 0x0000, 0x24, 0x25, 0x01},
/* ?? */
	{-1,		    0x80 | 0x56, 0x01, 0x0000, 0x64, 0x67, 0x01},
	{SENSOR_MI1320,     0x80 | 0x48, 0x00, 0x148c, 0x64, 0x65, 0x01},
};
static const struct sensor_info vc0323_probe_data[] = {
/*      sensorId,	   I2cAdd,	IdAdd,  VpId,  m1,    m2,  op */
/* 0 OV9640 */
	{-1,		    0x80 | 0x30, 0x0a, 0x0000, 0x25, 0x24, 0x05},
/* 1 ICM108T (may respond on IdAdd == 0x83 - tested in vc032x_probe_sensor) */
	{-1,		    0x80 | 0x20, 0x82, 0x0000, 0x24, 0x25, 0x01},
/* 2 PO2130 (may detect PO3130NC - tested in vc032x_probe_sensor)*/
	{-1,		    0x80 | 0x76, 0x00, 0x0000, 0x24, 0x25, 0x01},
/* 3 MI1310 */
	{-1,		    0x80 | 0x5d, 0x00, 0x0000, 0x24, 0x25, 0x01},
/* 4 MI360 - tested in vc032x_probe_sensor */
/*	{SENSOR_MI0360,	    0x80 | 0x5d, 0x00, 0x8243, 0x24, 0x25, 0x01}, */
/* 5 7131R */
	{SENSOR_HV7131R,    0x80 | 0x11, 0x00, 0x0209, 0x24, 0x25, 0x01},
/* 6 OV7649 */
	{-1,		    0x80 | 0x21, 0x0a, 0x0000, 0x21, 0x20, 0x05},
/* 7 PAS302BCW */
	{-1,		    0x80 | 0x40, 0x00, 0x0000, 0x20, 0x22, 0x05},
/* 8 OV7660 */
	{SENSOR_OV7660,     0x80 | 0x21, 0x0a, 0x7660, 0x26, 0x26, 0x05},
/* 9 PO3130NC - (tested in vc032x_probe_sensor) */
/*	{SENSOR_PO3130NC,   0x80 | 0x76, 0x00, 0x3130, 0x24, 0x25, 0x01}, */
/* 10 PO1030KC */
	{-1,		    0x80 | 0x6e, 0x00, 0x0000, 0x24, 0x25, 0x01},
/* 11 MI1310_SOC */
	{SENSOR_MI1310_SOC, 0x80 | 0x5d, 0x00, 0x143a, 0x24, 0x25, 0x01},
/* 12 OV9650 */
	{-1,		    0x80 | 0x30, 0x0a, 0x0000, 0x25, 0x24, 0x05},
/* 13 S5K532 */
	{-1,		    0x80 | 0x11, 0x39, 0x0000, 0x24, 0x25, 0x01},
/* 14 MI360_SOC - ??? */
/* 15 PO1200N */
	{SENSOR_PO1200,     0x80 | 0x5c, 0x00, 0x1200, 0x67, 0x67, 0x01},
/* 16 ?? */
	{-1,		    0x80 | 0x2d, 0x00, 0x0000, 0x65, 0x67, 0x01},
/* 17 PO2030 */
	{-1,		    0x80 | 0x6e, 0x00, 0x0000, 0x24, 0x25, 0x01},
/* ?? */
	{-1,		    0x80 | 0x56, 0x01, 0x0000, 0x64, 0x67, 0x01},
	{SENSOR_MI1320_SOC, 0x80 | 0x48, 0x00, 0x148c, 0x64, 0x67, 0x01},
/*fixme: not in the ms-win probe - may be found before? */
	{SENSOR_OV7670,     0x80 | 0x21, 0x0a, 0x7673, 0x66, 0x67, 0x05},
};

/* read 'len' bytes in gspca_dev->usb_buf */
static void reg_r_i(struct gspca_dev *gspca_dev,
		  u16 req,
		  u16 index,
		  u16 len)
{
	int ret;

	if (gspca_dev->usb_err < 0)
		return;
	ret = usb_control_msg(gspca_dev->dev,
			usb_rcvctrlpipe(gspca_dev->dev, 0),
			req,
			USB_DIR_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			1,			/* value */
			index, gspca_dev->usb_buf, len,
			500);
	if (ret < 0) {
		pr_err("reg_r err %d\n", ret);
		gspca_dev->usb_err = ret;
	}
}
static void reg_r(struct gspca_dev *gspca_dev,
		  u16 req,
		  u16 index,
		  u16 len)
{
	reg_r_i(gspca_dev, req, index, len);
#ifdef GSPCA_DEBUG
	if (gspca_dev->usb_err < 0)
		return;
	if (len == 1)
		PDEBUG(D_USBI, "GET %02x 0001 %04x %02x", req, index,
				gspca_dev->usb_buf[0]);
	else
		PDEBUG(D_USBI, "GET %02x 0001 %04x %02x %02x %02x",
				req, index,
				gspca_dev->usb_buf[0],
				gspca_dev->usb_buf[1],
				gspca_dev->usb_buf[2]);
#endif
}

static void reg_w_i(struct gspca_dev *gspca_dev,
			    u16 req,
			    u16 value,
			    u16 index)
{
	int ret;

	if (gspca_dev->usb_err < 0)
		return;
	ret = usb_control_msg(gspca_dev->dev,
			usb_sndctrlpipe(gspca_dev->dev, 0),
			req,
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			value, index, NULL, 0,
			500);
	if (ret < 0) {
		pr_err("reg_w err %d\n", ret);
		gspca_dev->usb_err = ret;
	}
}
static void reg_w(struct gspca_dev *gspca_dev,
			    u16 req,
			    u16 value,
			    u16 index)
{
#ifdef GSPCA_DEBUG
	if (gspca_dev->usb_err < 0)
		return;
	PDEBUG(D_USBO, "SET %02x %04x %04x", req, value, index);
#endif
	reg_w_i(gspca_dev, req, value, index);
}

static u16 read_sensor_register(struct gspca_dev *gspca_dev,
				u16 address)
{
	u8 ldata, mdata, hdata;
	int retry = 50;

	reg_r(gspca_dev, 0xa1, 0xb33f, 1);
	if (!(gspca_dev->usb_buf[0] & 0x02)) {
		pr_err("I2c Bus Busy Wait %02x\n", gspca_dev->usb_buf[0]);
		return 0;
	}
	reg_w(gspca_dev, 0xa0, address, 0xb33a);
	reg_w(gspca_dev, 0xa0, 0x02, 0xb339);

	do {
		reg_r(gspca_dev, 0xa1, 0xb33b, 1);
		if (gspca_dev->usb_buf[0] == 0x00)
			break;
		msleep(40);
	} while (--retry >= 0);

	reg_r(gspca_dev, 0xa1, 0xb33e, 1);
	ldata = gspca_dev->usb_buf[0];
	reg_r(gspca_dev, 0xa1, 0xb33d, 1);
	mdata = gspca_dev->usb_buf[0];
	reg_r(gspca_dev, 0xa1, 0xb33c, 1);
	hdata = gspca_dev->usb_buf[0];
	if (hdata != 0 && mdata != 0 && ldata != 0)
		PDEBUG(D_PROBE, "Read Sensor %02x%02x %02x",
			hdata, mdata, ldata);
	reg_r(gspca_dev, 0xa1, 0xb334, 1);
	if (gspca_dev->usb_buf[0] == 0x02)
		return (hdata << 8) + mdata;
	return hdata;
}

static int vc032x_probe_sensor(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	int i, n;
	u16 value;
	const struct sensor_info *ptsensor_info;

/*fixme: should also check the other sensor (back mi1320_soc, front mc501cb)*/
	if (sd->flags & FL_SAMSUNG) {
		reg_w(gspca_dev, 0xa0, 0x01, 0xb301);
		reg_w(gspca_dev, 0x89, 0xf0ff, 0xffff);
						/* select the back sensor */
	}

	reg_r(gspca_dev, 0xa1, 0xbfcf, 1);
	PDEBUG(D_PROBE, "vc032%d check sensor header %02x",
		sd->bridge == BRIDGE_VC0321 ? 1 : 3, gspca_dev->usb_buf[0]);
	if (sd->bridge == BRIDGE_VC0321) {
		ptsensor_info = vc0321_probe_data;
		n = ARRAY_SIZE(vc0321_probe_data);
	} else {
		ptsensor_info = vc0323_probe_data;
		n = ARRAY_SIZE(vc0323_probe_data);
	}
	for (i = 0; i < n; i++) {
		reg_w(gspca_dev, 0xa0, 0x02, 0xb334);
		reg_w(gspca_dev, 0xa0, ptsensor_info->m1, 0xb300);
		reg_w(gspca_dev, 0xa0, ptsensor_info->m2, 0xb300);
		reg_w(gspca_dev, 0xa0, 0x01, 0xb308);
		reg_w(gspca_dev, 0xa0, 0x0c, 0xb309);
		reg_w(gspca_dev, 0xa0, ptsensor_info->I2cAdd, 0xb335);
		reg_w(gspca_dev, 0xa0, ptsensor_info->op, 0xb301);
		value = read_sensor_register(gspca_dev, ptsensor_info->IdAdd);
		if (value == 0 && ptsensor_info->IdAdd == 0x82)
			value = read_sensor_register(gspca_dev, 0x83);
		if (value != 0) {
			PDEBUG(D_ERR|D_PROBE, "Sensor ID %04x (%d)",
				value, i);
			if (value == ptsensor_info->VpId)
				return ptsensor_info->sensorId;

			switch (value) {
			case 0x3130:
				return SENSOR_PO3130NC;
			case 0x7673:
				return SENSOR_OV7670;
			case 0x8243:
				return SENSOR_MI0360;
			}
		}
		ptsensor_info++;
	}
	return -1;
}

static void i2c_write(struct gspca_dev *gspca_dev,
			u8 reg, const u8 *val,
			u8 size)		/* 1 or 2 */
{
	int retry;

#ifdef GSPCA_DEBUG
	if (gspca_dev->usb_err < 0)
		return;
	if (size == 1)
		PDEBUG(D_USBO, "i2c_w %02x %02x", reg, *val);
	else
		PDEBUG(D_USBO, "i2c_w %02x %02x%02x", reg, *val, val[1]);
#endif
	reg_r_i(gspca_dev, 0xa1, 0xb33f, 1);
/*fixme:should check if (!(gspca_dev->usb_buf[0] & 0x02)) error*/
	reg_w_i(gspca_dev, 0xa0, size, 0xb334);
	reg_w_i(gspca_dev, 0xa0, reg, 0xb33a);
	reg_w_i(gspca_dev, 0xa0, val[0], 0xb336);
	if (size > 1)
		reg_w_i(gspca_dev, 0xa0, val[1], 0xb337);
	reg_w_i(gspca_dev, 0xa0, 0x01, 0xb339);
	retry = 4;
	do {
		reg_r_i(gspca_dev, 0xa1, 0xb33b, 1);
		if (gspca_dev->usb_buf[0] == 0)
			break;
		msleep(20);
	} while (--retry > 0);
	if (retry <= 0)
		pr_err("i2c_write timeout\n");
}

static void put_tab_to_reg(struct gspca_dev *gspca_dev,
			const u8 *tab, u8 tabsize, u16 addr)
{
	int j;
	u16 ad = addr;

	for (j = 0; j < tabsize; j++)
		reg_w(gspca_dev, 0xa0, tab[j], ad++);
}

static void usb_exchange(struct gspca_dev *gspca_dev,
			const u8 data[][4])
{
	int i = 0;

	for (;;) {
		switch (data[i][3]) {
		default:
			return;
		case 0xcc:			/* normal write */
			reg_w(gspca_dev, 0xa0, data[i][2],
					(data[i][0]) << 8 | data[i][1]);
			break;
		case 0xaa:			/* i2c op */
			i2c_write(gspca_dev, data[i][1], &data[i][2], 1);
			break;
		case 0xbb:			/* i2c op */
			i2c_write(gspca_dev, data[i][0], &data[i][1], 2);
			break;
		case 0xdd:
			msleep(data[i][1] * 256 + data[i][2] + 10);
			break;
		}
		i++;
	}
	/*not reached*/
}


/* this function is called at probe time */
static int sd_config(struct gspca_dev *gspca_dev,
			const struct usb_device_id *id)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->bridge = id->driver_info >> 8;
	sd->flags = id->driver_info & 0xff;

	if (id->idVendor == 0x046d &&
	    (id->idProduct == 0x0892 || id->idProduct == 0x0896))
		sd->sensor = SENSOR_POxxxx;	/* no probe */

	sd->brightness = BRIGHTNESS_DEF;
	sd->contrast = CONTRAST_DEF;
	sd->colors = COLOR_DEF;
	sd->hflip = HFLIP_DEF;
	sd->vflip = VFLIP_DEF;
	sd->lightfreq = FREQ_DEF;
	sd->sharpness = SHARPNESS_DEF;
	sd->gain = GAIN_DEF;
	sd->exposure = EXPOSURE_DEF;
	sd->autogain = AUTOGAIN_DEF;
	sd->backlight = BACKLIGHT_DEF;

	return 0;
}

/* this function is called at probe and resume time */
static int sd_init(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	struct cam *cam;
	int sensor;
	/* number of packets per ISOC message */
	static u8 npkt[NSENSORS] = {
		[SENSOR_HV7131R] =	64,
		[SENSOR_MI0360] =	32,
		[SENSOR_MI1310_SOC] =	32,
		[SENSOR_MI1320] =	64,
		[SENSOR_MI1320_SOC] =	128,
		[SENSOR_OV7660] =	32,
		[SENSOR_OV7670] =	64,
		[SENSOR_PO1200] =	128,
		[SENSOR_PO3130NC] =	128,
		[SENSOR_POxxxx] =	128,
	};

	if (sd->sensor != SENSOR_POxxxx)
		sensor = vc032x_probe_sensor(gspca_dev);
	else
		sensor = sd->sensor;

	switch (sensor) {
	case -1:
		pr_err("Unknown sensor...\n");
		return -EINVAL;
	case SENSOR_HV7131R:
		PDEBUG(D_PROBE, "Find Sensor HV7131R");
		break;
	case SENSOR_MI0360:
		PDEBUG(D_PROBE, "Find Sensor MI0360");
		sd->bridge = BRIDGE_VC0323;
		break;
	case SENSOR_MI1310_SOC:
		PDEBUG(D_PROBE, "Find Sensor MI1310_SOC");
		break;
	case SENSOR_MI1320:
		PDEBUG(D_PROBE, "Find Sensor MI1320");
		break;
	case SENSOR_MI1320_SOC:
		PDEBUG(D_PROBE, "Find Sensor MI1320_SOC");
		break;
	case SENSOR_OV7660:
		PDEBUG(D_PROBE, "Find Sensor OV7660");
		break;
	case SENSOR_OV7670:
		PDEBUG(D_PROBE, "Find Sensor OV7670");
		break;
	case SENSOR_PO1200:
		PDEBUG(D_PROBE, "Find Sensor PO1200");
		break;
	case SENSOR_PO3130NC:
		PDEBUG(D_PROBE, "Find Sensor PO3130NC");
		break;
	case SENSOR_POxxxx:
		PDEBUG(D_PROBE, "Sensor POxxxx");
		break;
	}
	sd->sensor = sensor;

	cam = &gspca_dev->cam;
	if (sd->bridge == BRIDGE_VC0321) {
		cam->cam_mode = vc0321_mode;
		cam->nmodes = ARRAY_SIZE(vc0321_mode);
	} else {
		switch (sensor) {
		case SENSOR_PO1200:
			cam->cam_mode = svga_mode;
			cam->nmodes = ARRAY_SIZE(svga_mode);
			break;
		case SENSOR_MI1310_SOC:
			cam->cam_mode = vc0323_mode;
			cam->nmodes = ARRAY_SIZE(vc0323_mode);
			break;
		case SENSOR_MI1320_SOC:
			cam->cam_mode = bi_mode;
			cam->nmodes = ARRAY_SIZE(bi_mode);
			break;
		case SENSOR_OV7670:
			cam->cam_mode = bi_mode;
			cam->nmodes = ARRAY_SIZE(bi_mode) - 1;
			break;
		default:
			cam->cam_mode = vc0323_mode;
			cam->nmodes = ARRAY_SIZE(vc0323_mode) - 1;
			break;
		}
	}
	cam->npkt = npkt[sd->sensor];
	gspca_dev->ctrl_dis = ctrl_dis[sd->sensor];

	if (sd->sensor == SENSOR_OV7670)
		sd->flags |= FL_HFLIP | FL_VFLIP;

	if (sd->bridge == BRIDGE_VC0321) {
		reg_r(gspca_dev, 0x8a, 0, 3);
		reg_w(gspca_dev, 0x87, 0x00, 0x0f0f);
		reg_r(gspca_dev, 0x8b, 0, 3);
		reg_w(gspca_dev, 0x88, 0x00, 0x0202);
		if (sd->sensor == SENSOR_POxxxx) {
			reg_r(gspca_dev, 0xa1, 0xb300, 1);
			if (gspca_dev->usb_buf[0] != 0) {
				reg_w(gspca_dev, 0xa0, 0x26, 0xb300);
				reg_w(gspca_dev, 0xa0, 0x04, 0xb300);
			}
			reg_w(gspca_dev, 0xa0, 0x00, 0xb300);
		}
	}
	return gspca_dev->usb_err;
}

static void setbrightness(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 data;

	if (gspca_dev->ctrl_dis & (1 << BRIGHTNESS_IDX))
		return;
	data = sd->brightness;
	if (data >= 0x80)
		data &= 0x7f;
	else
		data = 0xff ^ data;
	i2c_write(gspca_dev, 0x98, &data, 1);
}

static void setcontrast(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (gspca_dev->ctrl_dis & (1 << CONTRAST_IDX))
		return;
	i2c_write(gspca_dev, 0x99, &sd->contrast, 1);
}

static void setcolors(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 data;

	if (gspca_dev->ctrl_dis & (1 << COLORS_IDX))
		return;
	data = sd->colors - (sd->colors >> 3) - 1;
	i2c_write(gspca_dev, 0x94, &data, 1);
	i2c_write(gspca_dev, 0x95, &sd->colors, 1);
}

static void sethvflip(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 data[2], hflip, vflip;

	hflip = sd->hflip;
	if (sd->flags & FL_HFLIP)
		hflip = !hflip;
	vflip = sd->vflip;
	if (sd->flags & FL_VFLIP)
		vflip = !vflip;
	switch (sd->sensor) {
	case SENSOR_MI1310_SOC:
	case SENSOR_MI1320:
	case SENSOR_MI1320_SOC:
		data[0] = data[1] = 0;		/* select page 0 */
		i2c_write(gspca_dev, 0xf0, data, 2);
		data[0] = sd->sensor == SENSOR_MI1310_SOC ? 0x03 : 0x01;
		data[1] = 0x02 * hflip
			| 0x01 * vflip;
		i2c_write(gspca_dev, 0x20, data, 2);
		break;
	case SENSOR_OV7660:
	case SENSOR_OV7670:
		data[0] = sd->sensor == SENSOR_OV7660 ? 0x01 : 0x07;
		data[0] |= OV7660_MVFP_MIRROR * hflip
			| OV7660_MVFP_VFLIP * vflip;
		i2c_write(gspca_dev, OV7660_REG_MVFP, data, 1);
		break;
	case SENSOR_PO1200:
		data[0] = 0;
		i2c_write(gspca_dev, 0x03, data, 1);
		data[0] = 0x80 * hflip
			| 0x40 * vflip
			| 0x06;
		i2c_write(gspca_dev, 0x1e, data, 1);
		break;
	}
}

static void setlightfreq(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	static const u8 (*ov7660_freq_tb[3])[4] =
		{ov7660_NoFliker, ov7660_50HZ, ov7660_60HZ};

	if (sd->sensor != SENSOR_OV7660)
		return;
	usb_exchange(gspca_dev, ov7660_freq_tb[sd->lightfreq]);
}

static void setsharpness(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 data;

	switch (sd->sensor) {
	case SENSOR_PO1200:
		data = 0;
		i2c_write(gspca_dev, 0x03, &data, 1);
		if (sd->sharpness < 0)
			data = 0x6a;
		else
			data = 0xb5 + sd->sharpness * 3;
		i2c_write(gspca_dev, 0x61, &data, 1);
		break;
	case SENSOR_POxxxx:
		if (sd->sharpness < 0)
			data = 0x7e;	/* def = max */
		else
			data = 0x60 + sd->sharpness * 0x0f;
		i2c_write(gspca_dev, 0x59, &data, 1);
		break;
	}
}
static void setgain(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (gspca_dev->ctrl_dis & (1 << GAIN_IDX))
		return;
	i2c_write(gspca_dev, 0x15, &sd->gain, 1);
}

static void setexposure(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 data;

	if (gspca_dev->ctrl_dis & (1 << EXPOSURE_IDX))
		return;
	data = sd->exposure >> 8;
	i2c_write(gspca_dev, 0x1a, &data, 1);
	data = sd->exposure;
	i2c_write(gspca_dev, 0x1b, &data, 1);
}

static void setautogain(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	static const u8 data[2] = {0x28, 0x3c};

	if (gspca_dev->ctrl_dis & (1 << AUTOGAIN_IDX))
		return;
	i2c_write(gspca_dev, 0xd1, &data[sd->autogain], 1);
}

static void setgamma(struct gspca_dev *gspca_dev)
{
/*fixme:to do */
	usb_exchange(gspca_dev, poxxxx_gamma);
}

static void setbacklight(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u16 v;
	u8 data;

	data = (sd->backlight << 4) | 0x0f;
	i2c_write(gspca_dev, 0xaa, &data, 1);
	v = 613 + 12 * sd->backlight;
	data = v >> 8;
	i2c_write(gspca_dev, 0xc4, &data, 1);
	data = v;
	i2c_write(gspca_dev, 0xc5, &data, 1);
	v = 1093 - 12 * sd->backlight;
	data = v >> 8;
	i2c_write(gspca_dev, 0xc6, &data, 1);
	data = v;
	i2c_write(gspca_dev, 0xc7, &data, 1);
	v = 342 + 9 * sd->backlight;
	data = v >> 8;
	i2c_write(gspca_dev, 0xc8, &data, 1);
	data = v;
	i2c_write(gspca_dev, 0xc9, &data, 1);
	v = 702 - 9 * sd->backlight;
	data = v >> 8;
	i2c_write(gspca_dev, 0xca, &data, 1);
	data = v;
	i2c_write(gspca_dev, 0xcb, &data, 1);
}

static void setwb(struct gspca_dev *gspca_dev)
{
/*fixme:to do - valid when reg d1 = 0x1c - (reg16 + reg15 = 0xa3)*/
	static const u8 data[2] = {0x00, 0x00};

	i2c_write(gspca_dev, 0x16, &data[0], 1);
	i2c_write(gspca_dev, 0x18, &data[1], 1);
}

static int sd_start(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	const u8 (*init)[4];
	const u8 *GammaT = NULL;
	const u8 *MatrixT = NULL;
	int mode;
	static const u8 (*mi1320_soc_init[])[4] = {
		mi1320_soc_InitSXGA,
		mi1320_soc_InitVGA,
		mi1320_soc_InitQVGA,
	};

/*fixme: back sensor only*/
	if (sd->flags & FL_SAMSUNG) {
		reg_w(gspca_dev, 0x89, 0xf0ff, 0xffff);
		reg_w(gspca_dev, 0xa9, 0x8348, 0x000e);
		reg_w(gspca_dev, 0xa9, 0x0000, 0x001a);
	}

	/* Assume start use the good resolution from gspca_dev->mode */
	if (sd->bridge == BRIDGE_VC0321) {
		reg_w(gspca_dev, 0xa0, 0xff, 0xbfec);
		reg_w(gspca_dev, 0xa0, 0xff, 0xbfed);
		reg_w(gspca_dev, 0xa0, 0xff, 0xbfee);
		reg_w(gspca_dev, 0xa0, 0xff, 0xbfef);
		sd->image_offset = 46;
	} else {
		if (gspca_dev->cam.cam_mode[gspca_dev->curr_mode].pixelformat
				== V4L2_PIX_FMT_JPEG)
			sd->image_offset = 0;
		else
			sd->image_offset = 32;
	}

	mode = gspca_dev->cam.cam_mode[(int) gspca_dev->curr_mode].priv;
	switch (sd->sensor) {
	case SENSOR_HV7131R:
		GammaT = hv7131r_gamma;
		MatrixT = hv7131r_matrix;
		if (mode)
			init = hv7131r_initQVGA_data;	/* 320x240 */
		else
			init = hv7131r_initVGA_data;	/* 640x480 */
		break;
	case SENSOR_OV7660:
		GammaT = ov7660_gamma;
		MatrixT = ov7660_matrix;
		if (mode)
			init = ov7660_initQVGA_data;	/* 320x240 */
		else
			init = ov7660_initVGA_data;	/* 640x480 */
		break;
	case SENSOR_MI0360:
		GammaT = mi1320_gamma;
		MatrixT = mi0360_matrix;
		if (mode)
			init = mi0360_initQVGA_JPG;	/* 320x240 */
		else
			init = mi0360_initVGA_JPG;	/* 640x480 */
		break;
	case SENSOR_MI1310_SOC:
		GammaT = mi1320_gamma;
		MatrixT = mi1320_matrix;
		switch (mode) {
		case 1:
			init = mi1310_socinitQVGA_JPG;	/* 320x240 */
			break;
		case 0:
			init = mi1310_socinitVGA_JPG;	/* 640x480 */
			break;
		default:
			init = mi1310_soc_InitSXGA_JPG;	/* 1280x1024 */
			break;
		}
		break;
	case SENSOR_MI1320:
		GammaT = mi1320_gamma;
		MatrixT = mi1320_matrix;
		if (mode)
			init = mi1320_initQVGA_data;	/* 320x240 */
		else
			init = mi1320_initVGA_data;	/* 640x480 */
		break;
	case SENSOR_MI1320_SOC:
		GammaT = mi1320_gamma;
		MatrixT = mi1320_matrix;
		init = mi1320_soc_init[mode];
		break;
	case SENSOR_OV7670:
		init = mode == 1 ? ov7670_InitVGA : ov7670_InitQVGA;
		break;
	case SENSOR_PO3130NC:
		GammaT = po3130_gamma;
		MatrixT = po3130_matrix;
		if (mode)
			init = po3130_initQVGA_data;	/* 320x240 */
		else
			init = po3130_initVGA_data;	/* 640x480 */
		usb_exchange(gspca_dev, init);
		init = po3130_rundata;
		break;
	case SENSOR_PO1200:
		GammaT = po1200_gamma;
		MatrixT = po1200_matrix;
		init = po1200_initVGA_data;
		break;
	default:
/*	case SENSOR_POxxxx: */
		usb_exchange(gspca_dev, poxxxx_init_common);
		setgamma(gspca_dev);
		setbacklight(gspca_dev);
		setbrightness(gspca_dev);
		setcontrast(gspca_dev);
		setcolors(gspca_dev);
		setsharpness(gspca_dev);
		setautogain(gspca_dev);
		setexposure(gspca_dev);
		setgain(gspca_dev);
		usb_exchange(gspca_dev, poxxxx_init_start_3);
		if (mode)
			init = poxxxx_initQVGA;
		else
			init = poxxxx_initVGA;
		usb_exchange(gspca_dev, init);
		reg_r(gspca_dev, 0x8c, 0x0000, 3);
		reg_w(gspca_dev, 0xa0,
				gspca_dev->usb_buf[2] & 1 ? 0 : 1,
				0xb35c);
		msleep(300);
/*fixme: i2c read 04 and 05*/
		init = poxxxx_init_end_1;
		break;
	}
	usb_exchange(gspca_dev, init);
	if (GammaT && MatrixT) {
		put_tab_to_reg(gspca_dev, GammaT, 17, 0xb84a);
		put_tab_to_reg(gspca_dev, GammaT, 17, 0xb85b);
		put_tab_to_reg(gspca_dev, GammaT, 17, 0xb86c);
		put_tab_to_reg(gspca_dev, MatrixT, 9, 0xb82c);

		switch (sd->sensor) {
		case SENSOR_PO1200:
		case SENSOR_HV7131R:
			reg_w(gspca_dev, 0x89, 0x0400, 0x1415);
			break;
		case SENSOR_MI1310_SOC:
			reg_w(gspca_dev, 0x89, 0x058c, 0x0000);
			break;
		}
		msleep(100);
		sethvflip(gspca_dev);
		setlightfreq(gspca_dev);
	}
	switch (sd->sensor) {
	case SENSOR_OV7670:
		reg_w(gspca_dev, 0x87, 0xffff, 0xffff);
		reg_w(gspca_dev, 0x88, 0xff00, 0xf0f1);
		reg_w(gspca_dev, 0xa0, 0x0000, 0xbfff);
		break;
	case SENSOR_POxxxx:
		usb_exchange(gspca_dev, poxxxx_init_end_2);
		setwb(gspca_dev);
		msleep(80);		/* led on */
		reg_w(gspca_dev, 0x89, 0xffff, 0xfdff);
		break;
	}
	return gspca_dev->usb_err;
}

static void sd_stopN(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	switch (sd->sensor) {
	case SENSOR_MI1310_SOC:
		reg_w(gspca_dev, 0x89, 0x058c, 0x00ff);
		break;
	case SENSOR_POxxxx:
		return;
	default:
		if (!(sd->flags & FL_SAMSUNG))
			reg_w(gspca_dev, 0x89, 0xffff, 0xffff);
		break;
	}
	reg_w(gspca_dev, 0xa0, 0x01, 0xb301);
	reg_w(gspca_dev, 0xa0, 0x09, 0xb003);
}

/* called on streamoff with alt 0 and on disconnect */
static void sd_stop0(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (!gspca_dev->present)
		return;
/*fixme: is this useful?*/
	if (sd->sensor == SENSOR_MI1310_SOC)
		reg_w(gspca_dev, 0x89, 0x058c, 0x00ff);
	else if (!(sd->flags & FL_SAMSUNG))
		reg_w(gspca_dev, 0x89, 0xffff, 0xffff);

	if (sd->sensor == SENSOR_POxxxx) {
		reg_w(gspca_dev, 0xa0, 0x26, 0xb300);
		reg_w(gspca_dev, 0xa0, 0x04, 0xb300);
		reg_w(gspca_dev, 0xa0, 0x00, 0xb300);
	}
}

static void sd_pkt_scan(struct gspca_dev *gspca_dev,
			u8 *data,			/* isoc packet */
			int len)			/* iso pkt length */
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (data[0] == 0xff && data[1] == 0xd8) {
		PDEBUG(D_PACK,
			"vc032x header packet found len %d", len);
		gspca_frame_add(gspca_dev, LAST_PACKET, NULL, 0);
		data += sd->image_offset;
		len -= sd->image_offset;
		gspca_frame_add(gspca_dev, FIRST_PACKET, data, len);
		return;
	}

	/* The vc0321 sends some additional data after sending the complete
	 * frame, we ignore this. */
	if (sd->bridge == BRIDGE_VC0321) {
		int size, l;

		l = gspca_dev->image_len;
		size = gspca_dev->frsz;
		if (len > size - l)
			len = size - l;
	}
	gspca_frame_add(gspca_dev, INTER_PACKET, data, len);
}

static int sd_setbrightness(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->brightness = val;
	if (gspca_dev->streaming)
		setbrightness(gspca_dev);
	return gspca_dev->usb_err;
}

static int sd_getbrightness(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->brightness;
	return 0;
}

static int sd_setcontrast(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->contrast = val;
	if (gspca_dev->streaming)
		setcontrast(gspca_dev);
	return gspca_dev->usb_err;
}

static int sd_getcontrast(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->contrast;
	return 0;
}

static int sd_setcolors(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->colors = val;
	if (gspca_dev->streaming)
		setcolors(gspca_dev);
	return gspca_dev->usb_err;
}

static int sd_getcolors(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->colors;
	return 0;
}

static int sd_sethflip(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->hflip = val;
	if (gspca_dev->streaming)
		sethvflip(gspca_dev);
	return gspca_dev->usb_err;
}

static int sd_gethflip(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->hflip;
	return 0;
}

static int sd_setvflip(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->vflip = val;
	if (gspca_dev->streaming)
		sethvflip(gspca_dev);
	return gspca_dev->usb_err;
}

static int sd_getvflip(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->vflip;
	return 0;
}

static int sd_setfreq(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->lightfreq = val;
	if (gspca_dev->streaming)
		setlightfreq(gspca_dev);
	return gspca_dev->usb_err;
}

static int sd_getfreq(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->lightfreq;
	return 0;
}

static int sd_setsharpness(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->sharpness = val;
	if (gspca_dev->streaming)
		setsharpness(gspca_dev);
	return gspca_dev->usb_err;
}

static int sd_getsharpness(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->sharpness;
	return 0;
}

static int sd_setgain(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->gain = val;
	if (gspca_dev->streaming)
		setgain(gspca_dev);
	return gspca_dev->usb_err;
}

static int sd_getgain(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->gain;
	return 0;
}

static int sd_setexposure(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->exposure = val;
	if (gspca_dev->streaming)
		setexposure(gspca_dev);
	return gspca_dev->usb_err;
}

static int sd_getexposure(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->exposure;
	return 0;
}

static int sd_setautogain(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->autogain = val;
	if (gspca_dev->streaming)
		setautogain(gspca_dev);

	return gspca_dev->usb_err;
}

static int sd_getautogain(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->autogain;
	return 0;
}

static int sd_setbacklight(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->backlight = val;
	if (gspca_dev->streaming)
		setbacklight(gspca_dev);

	return gspca_dev->usb_err;
}

static int sd_getbacklight(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->backlight;
	return 0;
}

static int sd_querymenu(struct gspca_dev *gspca_dev,
			struct v4l2_querymenu *menu)
{
	static const char *freq_nm[3] = {"NoFliker", "50 Hz", "60 Hz"};

	switch (menu->id) {
	case V4L2_CID_POWER_LINE_FREQUENCY:
		if (menu->index >= ARRAY_SIZE(freq_nm))
			break;
		strcpy((char *) menu->name, freq_nm[menu->index]);
		return 0;
	}
	return -EINVAL;
}

/* sub-driver description */
static const struct sd_desc sd_desc = {
	.name = MODULE_NAME,
	.ctrls = sd_ctrls,
	.nctrls = ARRAY_SIZE(sd_ctrls),
	.config = sd_config,
	.init = sd_init,
	.start = sd_start,
	.stopN = sd_stopN,
	.stop0 = sd_stop0,
	.pkt_scan = sd_pkt_scan,
	.querymenu = sd_querymenu,
};

/* -- module initialisation -- */
#define BF(bridge, flags) \
	.driver_info = (BRIDGE_ ## bridge << 8) \
		| (flags)
static const struct usb_device_id device_table[] = {
	{USB_DEVICE(0x041e, 0x405b), BF(VC0323, FL_VFLIP)},
	{USB_DEVICE(0x046d, 0x0892), BF(VC0321, 0)},
	{USB_DEVICE(0x046d, 0x0896), BF(VC0321, 0)},
	{USB_DEVICE(0x046d, 0x0897), BF(VC0321, 0)},
	{USB_DEVICE(0x0ac8, 0x0321), BF(VC0321, 0)},
	{USB_DEVICE(0x0ac8, 0x0323), BF(VC0323, 0)},
	{USB_DEVICE(0x0ac8, 0x0328), BF(VC0321, 0)},
	{USB_DEVICE(0x0ac8, 0xc001), BF(VC0321, 0)},
	{USB_DEVICE(0x0ac8, 0xc002), BF(VC0321, 0)},
	{USB_DEVICE(0x0ac8, 0xc301), BF(VC0323, FL_SAMSUNG)},
	{USB_DEVICE(0x15b8, 0x6001), BF(VC0323, 0)},
	{USB_DEVICE(0x15b8, 0x6002), BF(VC0323, 0)},
	{USB_DEVICE(0x17ef, 0x4802), BF(VC0323, 0)},
	{}
};
MODULE_DEVICE_TABLE(usb, device_table);

/* -- device connect -- */
static int sd_probe(struct usb_interface *intf,
			const struct usb_device_id *id)
{
	return gspca_dev_probe(intf, id, &sd_desc, sizeof(struct sd),
				THIS_MODULE);
}

static struct usb_driver sd_driver = {
	.name = MODULE_NAME,
	.id_table = device_table,
	.probe = sd_probe,
	.disconnect = gspca_disconnect,
#ifdef CONFIG_PM
	.suspend = gspca_suspend,
	.resume = gspca_resume,
#endif
};

/* -- module insert / remove -- */
static int __init sd_mod_init(void)
{
	return usb_register(&sd_driver);
}
static void __exit sd_mod_exit(void)
{
	usb_deregister(&sd_driver);
}

module_init(sd_mod_init);
module_exit(sd_mod_exit);
