/*--------------------------------------------------------------------------*
 * 
 * This file contains the basic conversion utilities.
 *
 *  hdf_to_pixrect: Reads an hdf file and returns a pixrect.
 *  pixrect_to_hdf: Takes a pixrect and writes to an hdf file.
 *  hdf_to_rasterfile: Reads an hdf file and writes to a raster file stream.
 *  rasterfile_to_hdf: Reads a raster file stream and writes to an hdf file.
 *
 *
 * 				NO WARRANTY
 *  
 *  This software is distributed free of charge and is in the public domain.
 *  Anyone may use, duplicate or modify this program.  Thinking Machines
 *  Corporation does not restrict in any way the use of this software by
 *  anyone.
 *  
 *  Thinking Machines Corporation provides absolutely no warranty of any kind.
 *  The entire risk as to the quality and performance of this program is with
 *  you.  In no event will Thinking Machines Corporation be liable to you for
 *  damages, including any lost profits, lost monies, or other special,
 *  incidental or consequential damages arising out of the use of this program.
 * 
 * Jim Salem - Thinking Machines - 9/26/89 (Initial version)
 * Please keep this notice with the file.
 * 
 * Peter Webb - NCSA - 10/14/90
 *  - New error handling.
 *  - Removed dependencies on SUN libraries.
 *  - Integrated into Reformat
 *
 *--------------------------------------------------------------------------*/

#include <stdio.h>

/* Package include files */

#include "sun2ri8.h"
#include "types.h"      /* Package-wide type definitions */
#include "error.h"      /* Error codes */
#include "extern.h"     /* External routines and global variables */
#include "reformat.h"   /* Package-wide contants */


/* Convert a pixrect and colormap into an image and a palette */

static ErrorCode PixrectToHdf(pixrect, colormap, info, header)
     byte *pixrect;
     byte *colormap;
     FileInfo *info;
     SunRasterInfo *header;
{
  int xdim, ydim, ret;
  char *array, *temp_array;
  register int i, c, len;

/* Check depth */

  if (header->depth != 8)
    return(err_msg(Pixrect2HDF, ImageTooComplex));

/* initialize */

  xdim = header->width;
  ydim = header->height;

/* Handle colormap */

  if (colormap)
    switch (header->maptype)
      {
      case RMT_EQUAL_RGB:

        len = (header->maplen)/3;

        for (i = (MAX_PAL/3)-len; i < MAX_PAL/3; i++)
  	  for (c = 0; c < 3; ++c)
	    (info->palette)[(i*3)+c] = 
		*(colormap+(c*len)+(i-(MAX_PAL/3)+len));

/* Scan over the pixels in the image, adjusting them appropriately */

	for (i=0; i<xdim*ydim; i++)
	  *(pixrect+i) += (MAX_PAL/3)-len;

/* Fill in empty slots with gray-scale */

        for (i = 0; i < MAX_PAL-(len*3); i+=3)
	    info->palette[i] = info->palette[i+1] = 
		info->palette[i+2] = (255-len)-(i/3);

/* Set the palette */

        info->values |= (BitMask)CvtPalette;
	break;

      case RMT_NONE:  /* Make up a nice gray-scale */

	for (i=0; i < MAX_PAL; i+=3)
	  info->palette[i] = info->palette[i+1] = info->palette[i+2] = i;
        info->values |= (BitMask)CvtPalette;
	break;

      case RMT_RAW:
      default:        /* Error */
  	return(err_msg(Pixrect2HDF, NoPalette));
        break;
      }

/* Now fill in the info structure (used when writing out the image) */

  info->height = ydim;
  info->width = xdim;
  info->image = pixrect;

  info->values |= (BitMask)CvtImage;

/* Done ! */

  return (AllOk);
}

/* Read in the rasterfile */

static ErrorCode LoadPixrect(name, header, pixrect, colormap)
     char *name;
     SunRasterInfo *header;
     byte **pixrect, **colormap;
{
  FILE *fp;
  int ret;
  ErrorCode error;
  byte c, *temp;
  long i, j, k;

/* Open the sun raster file */

  fp = fopen(name, "r");
  if (fp == NULL) return(err_msg(LoadPix, InputOpenFailed));

/* Read in the header */

  ret = fread(header, sizeof(SunRasterInfo), 1, fp);
  if (ret != 1) return(err_msg(LoadPix, FileSystemError));

/* Check the magic number */

  if (header->magic != MAGIC)
    return(err_msg(LoadPix, BadFileData));

/* Get the colormap, if there is one. */

  if (header->type != RMT_NONE)
    if (header->maplen != 0)
      {
	*colormap = (byte *) malloc(header->maplen*sizeof(byte));
	ret = fread(*colormap, sizeof(byte), header->maplen, fp);
	if (ret != header->maplen) return(err_msg(LoadPix, FileSystemError));
      }
    else return(err_msg(LoadPix, CorruptedInputFile));

/* Snarf in the image */

  if (header->length == 0)
    if (header->type == RT_BYTE_ENCODED)  /* The file is corrupted */
      {
	free(*colormap);
	return(err_msg(LoadPix, CorruptedInputFile));
      }
    else header->length = header->width * header->height;

/* Different types of files are read in differently. */

  switch (header->type)
    {
    case RT_BYTE_ENCODED: 

      *pixrect = (byte *)malloc(header->width * header->height);
      if (*pixrect == NULL) return(err_msg(LoadPix, NoMemory));
      k=0;
      temp = (byte *)malloc(header->length);
      if (temp == NULL) return(err_msg(LoadPix, NoMemory));
      ret = fread(temp, sizeof(byte), header->length, fp);
      if (ret != header->length) return(err_msg(LoadPix, FileSystemError));
      i=0;
      while (i < header->length)
        if ((int)*(temp+i) == 128) /* ESC character - start of encoding */
          if ((int) *(temp+i+1) != 0)  /* Count not zero, expand encoding */
            {
              for (j=0; j <= (int) *(temp+i+1); j++, k++)
	        *(*pixrect+k) = *(temp+i+2); 
              i += 3;
            }
          else /* There is actually an escape char. in the input */
            {
              *(*pixrect+k) = (char)128;
	      k++;
              i += 2;
            }
        else  /* This byte not encoded */
          {
            *(*pixrect+k) = *(temp+i);
	    k++;
	    i++;
          }

      header->length = header->height * header->width;
      free(temp);
      break;

    case RT_OLD:  /* Length is sometimes 0 here, for historical reasons. */

	if (header->length == 0)
	  header->length = header->height * header->width;

    case RT_STANDARD:   /* No encoding, very easy. */
    
      *pixrect = (byte *)malloc(header->length*sizeof(byte));
      for (i=0; i<header->height; i++)
        {
          ret = fread(*pixrect+(i*header->width), sizeof(byte), 
		      header->width, fp);
          if (ret != header->width) 
            return(err_msg(LoadPix, FileSystemError));      
          if ((header->width)%2 != 0)   /* Read the extra byte, ignore it */
	    fread(&c, sizeof(byte), 1, fp);
        }
      break;

    default:
      return(err_msg(LoadPix, BadInputVersion));
    }

  return(AllOk);
}

/* This reads in a rasterfile and returns the image and color map in the 
 * info structure.
 */

ErrorCode SunToRI8 (ras_file, info)
char *ras_file;
FileInfo *info;
{
  SunRasterInfo header;
  byte *pixrect, *colormap;
  int ret;
  ErrorCode error;

/* Read in the pixrect file */

  if ((error = LoadPixrect(ras_file, &header, &pixrect, &colormap)) != AllOk)
    return(error);

/* Convert the pixrect data and color map to and HDF RI8 and palette */

  error = PixrectToHdf(pixrect, colormap, info, &header);

  return error;
}





