/*  Topic -- An assembler for the PIC microcontroller series.
    Copyright (C) 1994 Leonard Norrgard

    E-Mail: vinsci@nic.funet.fi

    Mail:   Leonard Norrgard
            Skolhusgatan 32 A 14
	    FIN-65100 VASA

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdarg.h>

enum true_or_false {false = 0, true};

/* fixme: ugly fixed length buffers.  */
char buffer[5001];
char buffer_copy[5001];
char tmpbuffer1[5001];
char tmpbuffer2[5001];

/* We keep track of the resulting program in an array of this struct.
   If an expression (or two) was present in the source for this memory
   location, EXPRESSION points to it.  */
struct memory_cell
{
  unsigned short instruction;
  struct expression *expression;
};

/* Program location we are assembling an instruction to.  */
struct memory_cell *pc = 0;

/* Number of errors encountered while assembling this file.  */
unsigned long errors_seen = 0;

/* Indicates that we have seen the "end" directive.  */
int assembly_ended = 0;

#if 1 || CORE12  /* fixme */
unsigned short core_size = 0x07ff;
unsigned short instruction_size;
struct memory_cell *mem_start, *mem_end;

/* 12 bit core.  */
/* Byte-oriented file register operations.  */
unsigned short in_addwf, in_andwf, in_clrf, in_clrw, in_comf, in_decf;
unsigned short in_decfsz, in_incf, in_incfsz, in_iorwf, in_movf, in_movwf;
unsigned short in_nop, in_rlf, in_rrf, in_subwf, in_swapf, in_xorwf;

/* Bit-oriented file register operations.  */
unsigned short in_bcf, in_bsf, in_btfsc, in_btfss;

/* Literal and control operations.  */
unsigned short in_andlw, in_call, in_clrwdt, in_goto, in_iorlw, in_movlw;
unsigned short in_option, in_retlw, in_sleep, in_tris, in_xorlw;
#endif

#if 0
/* fixme: macro functionality not completed yet (need to feed the result to the
   assembler rather than an output file; implement macro and endm directives
   so it isn't necessary to define the macros _here_).  */
char *vcall[] =
{
  /* We depend on the page select bits in the status register at
     this point.  If they're incorrect, the return from the call
     will arrive at the wrong place (ie., the wrong code page).  */
  "        movlw (vcall_return_#) >> .8",
  "        movwf push_pchi",
  "        movlw (vcall_return_#) & .127",
  "        movwf push_pclo",
/*  "        IF ($ >> .9) == (target_label >> .9)",
  "          ; a call to the current page",
  "        ELSE", */
#if 1
  "          IF ((target_label >> .9) == .0)",
  "            bcf STATUS,PA0",
  "            bcf STATUS,PA1",
  "          ENDIF",
  "          IF ((target_label >> .9) == .1)",
  "            bsf STATUS,PA0",
  "            bcf STATUS,PA1",
  "          ENDIF",
  "          IF ((target_label >> .9) == .2)",
  "            bcf STATUS,PA0",
  "            bsf STATUS,PA1",
  "          ENDIF",
  "          IF ((target_label >> .9) == .3)",
  "            bsf STATUS,PA0",
  "            bsf STATUS,PA1",
  "          ENDIF",
#endif
/*  "        ENDIF",  */
  "        goto  target_label",

  /* Since we don't push the LSB, we must return at an even address.
     Make sure we do by reserving an instruction if necessary.  */
  "        IF ($ & 1) == 1",
  "          nop",
  "        ENDIF",
  "vcall_return_#",
  0
};

char *vcall_args[] =
{
  "target_label",
  0
};
#endif

struct arglist
{
  char *argument;
  struct arglist *next;
};

struct macro
{
  char *name;
  char **macrotext;
  char **arguments;
  int invocations;
  struct macro *next;
};

#if 0
struct macro a_macro =
  {"vcall", vcall, vcall_args, 0, 0};

struct macro *first_macro = & a_macro;
#else
struct macro *first_macro = 0;
#endif

struct file_context
{
  char *file_name;

  FILE *file_handle;

  /* Current line number we are looking at.  */
  unsigned long line_number;

  /* Last character on line with include directive in this file.  */
  unsigned long file_position;

  /* File from which this file was included.  */
  struct file_context *parent_file;
};

struct file_context *current_file = 0;

struct expression
{
  /* The source text of the expression.  This is evaluated by
     evaluate() just before we write the output, or in the case of an
     expression as an argument to ORG, RES or IF immediately.  */
  char *text;

  /* This decides how many bits of the result we keep.  If there are any
     bits set to the left of this mask, we have an overflow condition.  */
  unsigned long bit_mask;

  /* If we need to shift the value of the expression before we or it into
     the final instruction bits, this indicates how many bits to shift.  */
  unsigned short left_shift;

  /* If true, the expression text has been evaluated and the value is in
     the value field.  */
  enum true_or_false is_evaluated;

  /* The value of the expression is found here after evaluation.  */
  unsigned long value;

  /* Maybe we have a second expression for this instruction.  */
  struct expression *next;

  /* Back pointer to the instruction (memory cell) where this
     expression was used.  */
  struct memory_cell *instruction;

  /* This expression written at the following place.  */
  char *file_name;
  unsigned long line_number;
};

void
warning (char *format, ...)
{
  va_list args;

  fprintf (stderr, "%s:%u:Warning: ", current_file->file_name,
	   current_file->line_number);
  va_start (args, format);
  vfprintf (stderr, format, args);
  va_end (args);
  (void) putc ('\n', stderr);
}

void
error (char *format, ...)
{
  va_list args;

  fprintf (stderr, "%s:%u:Error: ", current_file->file_name,
	   current_file->line_number);
  va_start (args, format);
  vfprintf (stderr, format, args);
  va_end (args);
  (void) putc ('\n', stderr);
  ++ errors_seen;
}

void
fatal (char *format, ...)
{
  va_list args;

  fprintf (stderr, "%s:%u:Fatal: ", current_file->file_name, current_file->line_number);
  va_start (args, format);
  vfprintf (stderr, format, args);
  va_end (args);
  (void) putc ('\n', stderr);
  exit (1);
}

char *
xcalloc (int n_bytes)
{
  char *p = calloc (1, n_bytes);

  if (p == 0)
     fatal ("couldn't allocate %d bytes.", n_bytes);
  return p;
}

struct arglist *
get_arguments (char *pos)
{
  struct arglist *args = (struct arglist *) xcalloc (sizeof (*args));
  struct arglist *new;
  struct arglist *last = args;
  char *p, *val;

  for (p = pos; p != NULL; )
    {
      while ((val = (char *) strsep (&p, " \t\n")) != 0 && *val == '\0')
	;
      if (val && strlen(val))
	{
	  new = (struct arglist *) xcalloc (sizeof (*new));
	  last->next = new;
	  new->argument = val;
	  last = new;
	}
    }
  new = args->next;
  free (args);
  return new;
}

void
insert_macro (struct macro *macro, struct arglist *args)
{
  char **line = macro->macrotext;
  struct arglist *argp;

  /* Keep track of how many times we've inserted this macro.  This
     can be used by the programmer to get this number inserted in
     a label (for example) instead of a "#" character in the macro text.  */
  macro->invocations += 1;

#if 0
  fprintf (stdout, "\t; %s ", macro->name);
  for (argp = args->next; argp; argp = argp->next)
    fprintf (stdout, "%s ", argp->argument);
  fprintf (stdout, "(prepic macro)\n");
#endif

  while (*line && **line)
    {
      char **replace;
      char *pos;

      sprintf (tmpbuffer1, "%s", *line);

      replace = macro->arguments;
      for (argp = args->next;
	   argp;
	   argp = argp->next, ++replace, ++replace)
	if (pos = strstr (tmpbuffer1, *replace))
	  {
	    char *rest_of_line = pos + strlen (*replace);

	    *pos = '\0';
	    sprintf (tmpbuffer2, "%s%s%s",
		     tmpbuffer1,
		     argp->argument,
		     rest_of_line ? rest_of_line : "");
	    sprintf (tmpbuffer1, "%s", tmpbuffer2);
	  }
	if (pos = strchr (tmpbuffer1, '#'))
	  {
	    char *rest_of_line = pos + 1;

	    *pos = '\0';
	    sprintf (tmpbuffer2, "%s%d%s",
		     tmpbuffer1,
		     macro->invocations,
		     rest_of_line);
	    sprintf (tmpbuffer1, "%s", tmpbuffer2);
	  }
      /* fprintf (stdout, "%s\t; prepic\n", tmpbuffer1); */
      ++line;
    }
}


struct symbol
{
  char *name;
  char *file_name;
  unsigned long line_number;

  /* Use for avoiding infinite recursion.  */
  enum true_or_false in_evaluate;

  struct expression *expression;
  struct symbol *next;
};
struct symbol *symbols = 0;

void
define_symbol (char *symbol_name, struct expression *expression)
{
  struct symbol *new;

  for (new = symbols; new; new = new->next)
    if (strcmp (symbol_name, new->name) == 0)
      {
	/* Ooops, duplicate symbol name.  */
	error ("Symbol `%s' already defined in file %s on line %d",
	       symbol_name, new->file_name, new->line_number);
	return;
      }

  new = (struct symbol *) xcalloc (sizeof (*new));
  new->name = strdup (symbol_name);
  new->file_name = current_file->file_name;
  new->line_number = current_file->line_number;
  new->next = symbols;
  new->expression = expression;
  symbols = new;

  /* printf ("%d:got new symbol %s = %s\n", current_file->line_number,
	  symbol_name, expression->text);  */
}

unsigned long
evaluate (struct expression *);

unsigned long
symbol_value (char *symbol_name)
{
  struct symbol *sym;

  /* printf ("looking up symbol \"%s\"\n", symbol_name);  */
  for (sym = symbols; sym; sym=sym->next)
    if (strcmp (symbol_name, sym->name) == 0)
      {
	if (!sym->expression->is_evaluated)
	  {
	    if (sym->in_evaluate == true)
	      {
		error ("Infinite recursion in symbol definition of `%s'.",
		       symbol_name);
		return 0;
	      }
	    else
	      {
		sym->in_evaluate = true;
		evaluate (sym->expression);
		sym->in_evaluate = false;
	      }
	  }
	return sym->expression->value;
      }
  error ("undefined symbol %s", symbol_name);
  return 0;
}

/* Can only shift one word at a time.  */
int shifts = 0;
char *shifted_word;

void
shift (char **line)
{
  /* Got an unrecognized word.  Remember it and continue,
     using it later on the line, or issuing an error message,
     as appropriate.  */
  if (++shifts > 1)
    {
      shifts = 0;
      error ("Only one label allowed before instruction or directive.");
    }

  shifted_word = strdup (*line);
  *(shifted_word + strcspn (shifted_word, " ,\t\r\n")) = '\0';

  /* Skip over the word.  */
  while (!isspace(**line) && **line != '\0')
    ++*line;
  **line = '\0';
  ++*line;

  /* fprintf(stderr, "shifted \"%s\"\n", shifted_word); */
}

char *
get_shifted_word ()
{
  if (shifts == 0)
    fatal ("internal error: file %s, line %d.", __FILE__, __LINE__);

  --shifts;
  return shifted_word;
}

/* Create an expression with the given value.  */
struct expression *
create_absolute_expression (unsigned long value)
{
  struct expression *new = (struct expression *) xcalloc (sizeof (*new));

  new->is_evaluated = true;
  new->value = value;
  new->text = "<absolute expression>";
  return new;
}

void
code (unsigned long instruction, struct expression *expression)
{
#define TOUCHED 0x8000  /* Indicate this memory position holds code.  */

  if (shifts)
    define_symbol (get_shifted_word (), create_absolute_expression (pc - mem_start));

  if (pc < mem_start || pc > mem_end)
    {
      /* Trying to assemble outside the microcontroller memory space
	 isn't allowed.  */
      fatal ("mpu memory not virtual");
    }

  if (pc->instruction & TOUCHED)
    {
      /* We already assembled something into this address,
	 don't do it again.  */
      error ("memory location %x already used.", pc - mem_start);

      /* We increment the pc anyway, so as to hopefully get away from
	 the duplicate memory use at some point -- otherwise we'll get
	 a warning for every assembled instruction after the first
	 overlapping one.  */
      ++pc;
      return;
    }
  pc->instruction = instruction | TOUCHED;
  pc->expression = expression;
  ++pc;
}

void
init_instructions ()
{
  if (instruction_size == 12)
    {
      /* Byte-oriented file register operations.  */
      in_addwf = 0x1c0;
      in_andwf = 0x140;
      in_clrf  = 0x060;
      in_clrw  = 0x040;
      in_comf  = 0x240;
      in_decf  = 0x0c0;
      in_decfsz= 0x2c0;
      in_incf  = 0x280;
      in_incfsz= 0x3c0;
      in_iorwf = 0x100;
      in_movf  = 0x200;
      in_movwf = 0x020;
      in_nop   = 0x000;
      in_rlf   = 0x340;
      in_rrf   = 0x300;
      in_subwf = 0x080;
      in_swapf = 0x380;
      in_xorwf = 0x180;

      /* Bit-oriented file register operations.  */
      in_bcf   = 0x400;
      in_bsf   = 0x500;
      in_btfsc = 0x600;
      in_btfss = 0x700;

      /* Literal and control operations.  */
      in_andlw = 0xe00;
      in_call  = 0x900;
      in_clrwdt= 0x004;
      in_goto  = 0xa00;
      in_iorlw = 0xd00;
      in_movlw = 0xc00;
      in_option= 0x002;
      in_retlw = 0x800;
      in_sleep = 0x003;
      in_tris  = 0x00f;
      in_xorlw = 0xf00;
    }
  else
    {
      fatal ("you loose.");
    }
}

void
get_comma (char **text)
{
  /* Get a comma (possible surrounded by whitespace) or indicate
     a syntax error.  */
  while (isspace (**text))
    ++*text;
  if (**text != ',')
    {
      error ("expected comma.");
      return;
    }
  while (isspace (**text))
    ++*text;
}

char *
next_word (char *line)
{
  /* First character must be a alphabetic or an underline.  */
  if (isalpha (*line) || *line == '_')
    ++line;

  /* After that, we also allow digits.  */
  while (isalnum (*line) || *line == '_')
    ++line;

  /* We want to skip over a possible comma after the current word.  */
  if (*line == ',')
    ++line;

  while (isspace (*line))
    ++line;

  return line;
}

struct expression *
get_expression (char **text, unsigned short bit_mask, unsigned short left_shift)
{
  /* Get an expression, ended by either a comma, CR or LF.  */
  struct expression *new = (struct expression *) xcalloc (sizeof (*new));
  char *next;
  char *tmp;

  *text = next_word (*text);

  /* Don't add a space in the complement string, an expression is allowed
     to include spaces.  Also, a character constant like ' ' is allowed.  */
  new->text = strdup(*text);
  *(new->text + strcspn (new->text, ",\t\r\n")) = '\0';
  next = *text + strcspn (*text, ",\t\r\n");

  /* Strip spaces at the end.  */
  tmp = new->text + strlen (new->text);
  while ((tmp - 1) >= new->text
	 && *(tmp-1) == ' ')
    *--tmp = '\0';

  new->bit_mask = bit_mask;
  new->left_shift = left_shift;
  *text = next;
  new->file_name = current_file->file_name;
  new->line_number = current_file->line_number;
  new->instruction = pc;
  /* printf ("got expression: \"%s\" in file %s on line %u\n",
	  new->text, new->file_name, new->line_number);  */
  return new;
}

struct expression *
get_file_and_bit (char **text)
{
  struct expression *file, *bit;

  file = get_expression (text, 0x1f, 0);
  get_comma (text);
  bit = get_expression (text, 0x07, 5);
  file->next = bit;
  return file;
}

struct expression *
get_file_and_dest (char **text)
{
  struct expression *file, *dest;

  file = get_expression (text, 0x1f, 0);
  get_comma (text);
  dest = get_expression (text, 0x01, 5);
  file->next = dest;
  return file;
}

struct expression *
get_8bits (char **text)
{
  struct expression *bits;

  bits = get_expression (text, 0xff, 0);
  return bits;
}

struct expression *
get_9bits (char **text)
{
  struct expression *bits;

  bits = get_expression (text, 0x1ff, 0);
  return bits;
}

struct expression *
get_32bits (char **text)
{
  struct expression *bits;
  bits = get_expression (text, 0xffffffff, 0);
  return bits;
}

struct expression *
get_11bits (char **text)
{
  struct expression *bits;
  bits = get_expression (text, 0x7ff, 0);
  return bits;
}

struct expression *
get_file (char **text)
{
  struct expression *bits;
  bits = get_expression (text, 0x1f, 0);
  return bits;
}

#if 0
unsigned long
evaluate_real (struct expression *expression)
#else
unsigned long
evaluate (struct expression *expression)
#endif
{
  unsigned long result = 0;

  if (expression->is_evaluated == true)
    fatal ("internal error: asked to evaluate already evaluated expression.");

  if (expression && expression->text)
    {
      /* printf ("actually evaluating \"%s\"\n", expression->text);  */
      while (isspace (*expression->text))
	++expression->text;
      if (isalpha (*expression->text)
	  && *(expression->text + 1)
	  && *(expression->text + 1) != '\'')
	{
	  result = symbol_value (expression->text);
	  goto evaluated;
	}
      else
	{
	  if (1 == sscanf (expression->text, "0x%x", &result))
	    goto evaluated;
	  else
	    {
	      /* Make sure we don't leave any garbage here.  */
	      result = 0;
	    }

	  if ((*expression->text == '0' && *(expression->text + 1) == 'b')
	      || ((*expression->text == 'b' || *expression->text == 'B')
		  && *(expression->text + 1) == '\''))
	    {
	      /* Handle a binary number expressed as 0b10110...  */
	      char *digit = expression->text + 2;
	      
	      /* fixme: we need to beware of overflows.  */
	      while (*digit == '0' || *digit == '1')
		result = (result << 1) | *digit++ - 48;
	      
	      goto evaluated;
	    }
	  if (*expression->text == '\'')
	    {
	      result = (unsigned long) *(expression->text + 1);
	      if (!(result && *(expression->text + 2) == '\''))
		error ("malformed character constant.");
	      goto evaluated;
	    }
	  if (strrchr (expression->text, 'h')
	      || strrchr (expression->text, 'H'))
	    {
	      if (1 != sscanf (expression->text, "%x", &result))
		error ("invalid hexadecimal constant");
	      goto evaluated;
	    }

	  /* MPALC allows a dot before a decimal number.  */
	  if (*expression->text == '.')
	    ++expression->text;

	  /* fixme: needs syntax error checking...  */
	  result = atoi (expression->text);
	  goto evaluated;
	}
      error ("internal error: can't happen :-)\n");
      return 0;

    evaluated:
      expression->is_evaluated = true;
      result = result & expression->bit_mask;
      result <<= expression->left_shift;
      
      if (expression->next)
	result |= evaluate (expression->next);

      expression->value = result;
      return result;
    }
  /* instructions without arguments (such as SLEEP) doesn't get
     an expression appended to them, so we return 0.  */
  return 0;
}

#if 0
unsigned long
evaluate (struct expression *expression)
{
  unsigned long result;

  if (expression)
    {
      printf ("evaluating: \"%s\" used @ %u\n", expression->text,
	      expression->instruction - mem_start);
      result = evaluate_real (expression);
      printf ("            \"%s\" =  %u\n", expression->text, result);
      return result;
    }
  printf ("expression pointer is 0!!!\n");
  return 0;
}
#endif

struct expression *
get_empty (char **text)
{
  *text = next_word (*text);

  while (isspace (**text))
    ++*text;

  if (**text)
    warning ("Ignoring garbage \"%s\" at end of line.", *text);
  return 0;
}

void
do_include_file (char *line)
{
  struct file_context *new = (struct file_context *) xcalloc (sizeof (struct file_context));
  char *tmp;

  /* Strip possible quotes.  */
  if (*line == '"')
    ++ line;
  tmp = line;
  while (isalpha(*tmp) || *tmp == '.')
    ++ tmp;
  *tmp = '\0';

  if (*(line + strlen(line) - 1) == '"')
    *(line + strlen(line) - 1) = '\0';

  new->file_handle = fopen (line, "r");
  if (!new->file_handle)
    fatal ("couldn't open %s for read", line);
  new->file_name = strdup (line);

  new->parent_file = current_file;
  if (current_file && current_file->file_handle)
    {
      current_file -> file_position = ftell (current_file->file_handle);
      fclose (current_file->file_handle);
    }
  current_file = new;
}

void
assemble (char *line)
{
  char *comment1 = strchr (line, ';');
  char *comment2 = strchr (line, '!');
  char *label_end;
  char *first_char = line;
#define past_first_column() (first_char != line)

  /* Watch out for command characers in single quotes.  */
  if (comment1
      && !(comment1 > line
	   && *(comment1 - 1) == '\''))
    *comment1 = '\0';
  if (comment2
      && !(comment2 > line
	   && *(comment2 - 1) == '\''))
    *comment2 = '\0';

#define match(word) ((strnicmp(line,(word), strlen((word)))==0) && strcspn (line, " \t\r\n") == strlen ((word)))

  while (*line)
    {
      switch (tolower(*line))
	{
	case 'a':
	  if (match ("addwf"))
	    code (in_addwf, get_file_and_dest (&line));
	  else if (match ("andwf"))
	    code (in_andwf, get_file_and_dest (&line));
	  else if (match ("andlw"))
	    code (in_andlw, get_8bits (&line));
	  else
	    goto do_shift;
	  return;

	case 'b':
	  if (match ("bcf"))
	    code (in_bcf, get_file_and_bit (&line));
	  else if (match ("bsf"))
	    code (in_bsf, get_file_and_bit (&line));
	  else if (match ("btfsc"))
	    code (in_btfsc, get_file_and_bit (&line));
	  else if (match ("btfss"))
	    code (in_btfss, get_file_and_bit (&line));
	  else
	    goto do_shift;
	  return;

	case 'c':
	  if (match ("clrf"))
	    code (in_clrf, get_file (&line));
	  else if (match ("clrw"))
	    code (in_clrw, get_empty (&line));
	  else if (match ("call"))
	    code (in_call, get_8bits (&line));
	  else if (match ("clrwdt"))
	    code (in_clrwdt, get_empty (&line));
	  else if (match ("comf"))
	    code (in_comf, get_file_and_dest (&line));
	  else
	    goto do_shift;
	  return;

	case 'd':
	  if (match ("decf"))
	    code (in_decf, get_file_and_dest (&line));
	  else if (match ("decfsz"))
	    code (in_decfsz, get_file_and_dest (&line));
	  else
	    goto do_shift;
	  return;

	case 'e':
	case '=':
	  if (match ("equ") || match ("="))
	    define_symbol (get_shifted_word (), get_32bits (&line));
	  else if (match ("end"))
	    assembly_ended = 1;
	  else if (match ("else"))
	    error ("`else' directive not supported yet.");
	  else if (match ("endif"))
	    error ("`endif' directive not supported yet.");
	  else if (match ("endm"))
	    error ("`endm' directive not supported yet.");
	  else
	    goto do_shift;
	  return;

	case 'g':
	  if (match ("goto"))
	    code (in_goto, get_9bits (&line));
	  else
	    goto do_shift;
	  return;

	case 'i':
	  if (match ("incf"))
	    code (in_incf, get_file_and_dest (&line));
	  else if (match ("incfsz"))
	    code (in_incfsz, get_file_and_dest (&line));
	  else if (match ("iorwf"))
	    code (in_iorwf, get_file_and_dest (&line));
	  else if (match ("iorlw"))
	    code (in_movf, get_file_and_dest (&line));
	  else if (match ("include"))
	    do_include_file (next_word (line));
	  else if (match ("if"))
	    error ("`if' directive not supported yet.");
	  else
	    goto do_shift;
	  return;

	case 'l':
	  if (match ("list"))
	    warning ("`list' directive not supported yet.");
	  else if (match ("local"))
	    warning ("`local' directive not supported yet.");
	  else
	    goto do_shift;
	  return;

	case 'm':
	  if (match ("movf"))
	    code (in_movf, get_file_and_dest (&line));
	  else if (match ("movwf"))
	    code (in_movwf, get_file (&line));
	  else if (match ("movlw"))
	    code (in_movlw, get_8bits (&line));
	  else if (match ("macro"))
	    error ("`macro' directive not supported yet.");
	  else
	    goto do_shift;
	  return;

	case 'n':
	  if (match ("nop"))
	    code (in_nop, get_empty (&line));
	  else
	    goto do_shift;
	  return;

	case 'o':
	  if (match ("option"))
	    code (in_option, get_empty (&line));
	  else if (match ("org"))
	    pc = mem_start + evaluate (get_11bits (&line));
	  else
	    goto do_shift;
	  return;

	case 'r':
	  if (match ("rlf"))
	    code (in_rlf, get_file_and_dest (&line));
	  else if (match ("rrf"))
	    code (in_rrf, get_file_and_dest (&line));
	  else if (match ("retlw"))
	    code (in_retlw, get_8bits (&line));
	  else if (match ("res"))
	    pc += evaluate (get_32bits (&line));
	  else
	    goto do_shift;
	  return;

	case 's':
	  if (match ("subwf"))
	    code (in_subwf, get_file_and_dest (&line));
	  else if (match ("swapf"))
	    code (in_swapf, get_file_and_dest (&line));
	  else if (match ("sleep"))
	    code (in_sleep, get_empty (&line));
	  else
	    goto do_shift;
	  return;

	case 't':
	  if (match ("tris"))
	    code (in_tris, get_file (&line));
	  else if (match ("title"))
	    warning ("`title' directive not supported yet.");
	  else
	    goto do_shift;
	  return;

	case 'x':
	  if (match ("xorwf"))
	    code (in_xorwf, get_file_and_dest (&line));
	  else if (match ("xorlw"))
	    code (in_xorlw, get_8bits (&line));
	  else
	    goto do_shift;
	  return;

	case '\n':
	case '\r':
	  /* At end of this line.  */
	  if (shifts)
	    {
	      /* This happens when a label has white space after it, but no
		 instruction.  */
	      define_symbol (get_shifted_word (), create_absolute_expression (pc - mem_start));
	    }
	  return;

	default:
	  goto do_shift;
	}

      /* We're not interested in spaces, tabs or dots */
    do_shift:
      if (*line == ' ' || *line == '\t' || *line == '.')
	++ line;
      else
	if (!past_first_column ())
	  shift (&line);
	else
	  {
	    *(line + strcspn (line, " ,\t\r\n")) = '\0';
	    error ("unrecognized instruction/directive `%s'.", line);
	    return;
	  }
    }
  if (shifts && *line == '\0')
    define_symbol (get_shifted_word (), create_absolute_expression (pc - mem_start));
}

void
process_line ()
{
  /* Insert the macro text, replacing the arguments with their
     real names for this macro call.  */
  char *pos;
  char *arg, *replarg;
  int go_on = 1;
  struct arglist *arguments, *argp, *found = 0;
  struct macro *macro;
  int length = strlen (buffer);

  sprintf (buffer_copy, "%s", buffer);
  arguments = get_arguments (buffer_copy);

  for (argp = arguments; !found && argp; argp = argp->next)
    {
      for (macro = first_macro; !found && macro; macro = macro->next)
	{
	  if (strcasecmp (macro->name, argp->argument) == 0)
	    {
	      found = argp;
	      /* Found a macro.  Substitute its text, replacing
		 argument placeholders with real names.  */
	      insert_macro (macro, argp);
	    }
	}
    }
  /* if (!found)
    fprintf (stdout, "%s", buffer); */

  assemble (buffer);
}

unsigned char checksum;

unsigned long
do_checksum (unsigned short data)
{
  checksum += data>>8;
  checksum += data & 255;
  return data;
}

void
write_object_file (char *file_name)
{
  FILE *out;
#if 0
  /* Fill the memory cells with the address of each cell for testing.  */
  struct memory_cell *mem;
  int address = 0;

  for (mem = mem_start; mem <= mem_end; ++ mem)
    {
      mem->instruction = address++ | TOUCHED;
      mem->expression = 0;
    }
#endif

  out = fopen (file_name, "w");
  if (!out)
    fatal ("couldn't open object file for writing.");

  /* Write an object file in "Merged 8-bit Intellec hex format".
     See MPALC Assembler user's guide p. 106, by Microchip Technology Inc.  */
  for (pc = mem_start; pc <= mem_end; ++pc)
    {
      if (pc->instruction & TOUCHED)
	{
	  /* Found something we need to write out, gather the data.  */
	  struct memory_cell *mem, *mem2;

	  for (mem = pc; (mem->instruction & TOUCHED) && (mem < pc + 8); ++mem)
	    ;

	  /* Write byte count for this line.  */
	  fprintf (out, ":%02X", do_checksum (2 * (mem - pc)));

	  /* Write start address for the bytes on this line.  */
	  fprintf (out, "%04X", do_checksum (2 * (pc - mem_start)));

	  /* Write two zeroes.  (fixme: Anybody know what these are good for?)  */
	  fprintf (out, "00");
	  
	  for (mem2 = pc; mem2 < mem; ++mem2)
	    fprintf (out, "%04X",
		    do_checksum ((mem2->instruction & ~TOUCHED)
				 | evaluate (mem2->expression)));

	  /* And finally, write the checksum for this line.  */
	  fprintf (out, "%02X\n", (256 - (checksum & 255)) & 255);

	  pc = mem - 1;
	}
    }

  /* fixme: Why is the last line in the file always constant?  (or is it?)  */
  fprintf (out, ":00000001FF\n");
  fclose (out);
}

main (int argc, char **argv)
{
  char *pos;
  char *source_filename;

  fprintf (stderr,
	   "Topic version 0.1, Copyright (C) 1994 Leonard Norrgard\n"
	   "Topic comes with ABSOLUTELY NO WARRANTY; for details see file COPYING\n"
	   "This is free software, and you are welcome to redistribute it under\n"
	   "certain conditions; see file COPYING for details.\n\n");

  core_size = 0x07ff;
  instruction_size = 12;  /* bits */
  init_instructions ();
  mem_start = (struct memory_cell *) xcalloc (core_size * sizeof (*mem_start));
  mem_end = mem_start + core_size;
  pc = mem_start;

  source_filename = argv[1];

  do_include_file (source_filename);

 continue_file:
  while (!assembly_ended && !feof(current_file->file_handle))
    {
      fgets (buffer, 1000, current_file->file_handle);
      ++ current_file->line_number;
      shifts = 0;
      process_line ();
    }

  if (current_file->parent_file)
    {
      struct file_context *parent;

      fclose (current_file->file_handle);

      parent = current_file->parent_file;
      free (current_file);

      current_file = parent;

      current_file->file_handle = fopen (current_file->file_name, "r");
      if (!current_file->file_handle)
	fatal ("couldn't reopen parent file");
      fseek (current_file->file_handle, current_file->file_position, SEEK_SET);

      goto continue_file;
    }

  /* If it is useful, write an object file.  */
  if (!errors_seen)
    {
      char *file_name, *tmp;
      int len;
      
      len = strlen (source_filename);
      if (tmp = strrchr (source_filename, '.'))
	*tmp = '\0';
      file_name = xcalloc (strlen (source_filename) + 5);
      sprintf (file_name, "%s.obj", source_filename);
      write_object_file (file_name);
      free (file_name);
    }

#if 1
  /* Dump symbol table.  */
  {
    struct symbol *sym;
    enum true_or_false saw_unused_symbol = false;

    printf ("\nSymbol table:\n\nDefined and used symbols\n\n");
    for (sym = symbols; sym; sym = sym->next)
      {
	if (sym->expression->is_evaluated)
	  printf ("%8u 0x%04x %-40.40s\n",
		  sym->expression->value, sym->expression->value, sym->name);
	else
	  saw_unused_symbol = true;
      }

    if (saw_unused_symbol)
      {
	printf ("\nDefined but unused symbols:\n\n");
	for (sym = symbols; sym; sym = sym->next)
	  {
	    if (!sym->expression->is_evaluated)
	      {
		evaluate (sym->expression);
		printf ("%8u 0x%04x %-40.40s\n",
			sym->expression->value, sym->expression->value, sym->name);
	      }
	  }
      }
  }
#endif
}
