


/*
 * Copyright (c) 1990, Geometry Supercomputer Project
 *                     University of Minnesota 
 *                     1200 Washington Ave. S
 *                     Minneapolis, MN  55415
 *
 * email address: software@geom.umn.edu
 *
 * This software is copyrighted as noted above.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the authors, who may or may not act on them as they desire.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */



/* $Header: /usr2/gcg/grap/src/lib/oogl/libvect/RCS/vect.h,v 1.9 91/04/05 09:27:15 sullivan Exp $ */
/* $Source: /usr2/gcg/grap/src/lib/oogl/libvect/RCS/vect.h,v $ */


 /*
  * Geometry Routines
  * 
  * Geometry Supercomputer Project
  * 
  * DESCRIPTION:  Header file for the vector routines
  * 
  */


#ifndef VECTDEF
#define VECTDEF

#ifndef NULL
#define NULL 0
#endif

/* Cube.h is included since one of the routines returns a cube */
#include	"oogl/bbox.h"
#include	"geometry/point.h"
#include	"color/color.h"
#include        "geometry/point3.h"

#define	VECT_MAGIC	0x4ac95203	/* For binary external format */


/*
 * A Vect is a collection of polylines.
 * It's represented by:
 *  p,  an array of vertices (for all the polylines in sequence),
 *  c,  an array of r/g/b/alpha colors (not necessarily one per vertex),
 *  vnvert, an array of sizes (vertex counts) of each polyline,
 *  vncolor, an array of numbers of colors (c[] entry counts) of each polyline,
 *  nvec, giving the number of polylines (entries in vnvert and vncolor),
 *  nvert, giving the total number of vertices (entries in p),
 *  ncolor, the total number of colors (entries in c).
 *
 * Here's the association:
 *	Let vN = sum from i = 0 to N-1 of vnvert[i],
 *	and cN = sum from i = 0 to N-1 of vncolor[i],
 *    then polyline N comprises the vnvert[N] vertices
 *	from p[vN] through p[vN + vnvert[N] - 1]
 *    and is drawn using the vncolor[N] different colors
 *	from c[cN] through c[cN + vncolor[N] - 1].
 *
 * This encoding implies colors and vertices may not be reused from line to
 * line (but the previous color persists for lines with 0 colors, see below).
 *
 * Closed polylines:
 *	A polyline is drawn closed (the last element connected to the first)
 *	if its vnvert[N] is negative.  It's then considered to have
 *	abs(vnvert[N]) vertices in the p[] array.
 *
 * A polyline with 1 vertex is a point.  0 vertices are illegal.
 *
 * Coloring:
 *	It's intended that polylines will be specified with either
 *	0, 1, or abs(vnvert[N]) colors.  The effects in each case are:
 *	 0: the polyline is drawn with the same color as the previous polyline.
 *	 1: the polyline is drawn with the specified single color.
 *	 abs(vnvert[N]): each vertex is colored with the specified color.
 *	    intermediate points on the line are interpolated.
 */

typedef struct vect
{
	int	nvec, nvert, ncolor;
	short	*vnvert;	/* vnvert[nvec] (# p[]'s per vec) */
	short	*vncolor;	/* vncolor[nvec] (# c[]'s per vec) */
	    /* NOTE: vnvert and vncolor lie in a *single* malloc'ed area */
	    /* To free, GeomFree(vnvert). */
	Point3	*p;		/* p[nvert] */
	ColorA	*c;		/* c[ncolor] */
}               Vect;


#define vcount(code)	(abs(code))
#define	vwrapped(code)	((code) < 0)

#define	vSane(v)   (0 <= (v)->ncolor && (v)->ncolor <= (v)->nvert && \
		     (v)->nvec <= (v)->nvert && (v)->nvert < 9999999)


GeomClass      *VectMethods();
char           *VectName();

Vect           *VectCreateNoCopy(
			   int nvect,	short	*vectcounts,
					short	*colorcounts,
			   int nvert,	Point3	*vertices,
			   int ncolor,	ColorA	*colors);

Vect           *VectCreate(int nvect,	short	*vectcounts,
					short	*colorcounts,
			   int nvert,	Point3	*vertices,
			   int ncolor,	ColorA	*colors);

GeomRef         VectDelete();
Vect           *VectFLoad();
Vect           *VectLoad();
Vect           *VectSave();
Vect           *VectFSave();
BBox           *VectBound();
Vect           *VectDraw();
Vect           *VectDrawIrisGl();
Vect           *VectDrawRenderman();
Vect           *VectDrawXWindows();
GeomRef         VectShare();
GeomRef         VectPrivate();
GeomRef         VectCopy();
void            RendLine();

#endif VECTDEF
